(* (c) Microsoft Corporation. All rights reserved *)
(*-------------------------------------------------------------------------
 * A fairly simple optimizer. The main aim is to inline simple, known functions
 * and constant values, and to eliminate non-side-affecting bindings that 
 * are never used.
 *------------------------------------------------------------------------- *)

(*F#
module Microsoft.FSharp.Compiler.Opt
open Microsoft.Research.AbstractIL
open Microsoft.Research.AbstractIL.Internal
open Microsoft.Research.AbstractIL.Extensions.ILX
open Microsoft.FSharp.Compiler

module Il = Microsoft.Research.AbstractIL.IL
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics
F#*)
open Ildiag
open List
open Range
open Ast
open Tast
open Tastops
open Env
open Lib
open Layout
open Il
open Typrelns

let verbose = false

(*-------------------------------------------------------------------------
!* Settings
 *------------------------------------------------------------------------- *)

let abstract_big_targets = ref false
let big_target_size = ref 100  (* size after which we start chopping methods in two, though only at match targets *)
let very_very_big_expr_size = ref 3000 (* size after which we start enforcing splitting sub-expressions to new methods, to avoid hitting .NET IL limitations *)
let jitopt_default = true
let jitopt_user = ref None
let jitopt () = (match !jitopt_user with Some f -> f | None -> jitopt_default)

let localopt_default = true
let localopt_user = ref None
let localopt () = (match !localopt_user with Some f -> f | None -> localopt_default)

let crossmoduleopt_default = true
let crossmoduleopt_user = ref None
let crossmoduleopt () = localopt () && (match !crossmoduleopt_user with Some f -> f | None -> crossmoduleopt_default)

let keep_opt_values () = crossmoduleopt ()

let lambda_inline_threshold = ref 4 (* size after which we don't inline *)
let do_inline_lambdas () = localopt ()  (* inline calls *) 
let do_opt_unused () = localopt () (* eliminate unused bindings with no effect *) 
let do_opt_try () = localopt () (* eliminate try around expr with no effect *)
let do_opt_seq () = localopt () (* eliminate first part of seq if no effect *)
let do_opt_switch () = localopt () (* determine branches *)
let do_opt_recd_field_get () = localopt () (* determine branches *)                                                           
let do_opt_tup_field_get () = localopt () (* determine branches *)
let do_opt_constr_field_get () = localopt () (* determine branches *)
let do_opt_immediate_bind() = localopt () (* eliminate non-copiler generated immediate bindings *)
let do_opt_expand_structural_bind() = localopt () (* expand "let x = (exp1,exp2,...)" bind fields as prior tmps *)


(*-------------------------------------------------------------------------
!* Info returned up the tree by optimization.
 * Partial information about an expression.
 * Some value_info can 
 * 
 * We store one of these for each value in the environment, including values 
 * which we know little or nothing about. 
 *------------------------------------------------------------------------- *)

type type_value_info =
  | ConstTypeValue of typ
  | UnknownTypeValue

type value_info =
  | UnknownValue
  | SizeValue   of int * value_info        (* record size info (max_depth) for value_info *)
  (* RECURSIVE cases *)
  | ValValue    of val_ref * value_info    (* "equal to another identifier about which we know some further detail" *)
  | ModuleValue of modul_info              (* used for  when optimizing module expressions *)
  | TupleValue  of value_info array 
  | RecdValue   of tycon_ref * value_info array         (* INVARIANT: in field definition order *)
  | ConstrValue of unionconstr_ref * value_info array 
  | LdlenValue  of value_info
  | DecrValue   of value_info
  (* NON RECURSIVE cases *)
  | ConstValue of tconst * Tast.typ
  | CurriedLambdaValue of
      ( int        (* id *)
      * int        (* arities, i.e. number of bunches of untupled args, and number of args in each bunch. INCLUDE TYPE ARGS. *)
      * int        (* size *)
      * Tast.expr  (* value, a lambda term *)
      * Tast.typ   (* type of lamba term *))
  | ConstExprValue of
      ( int        (* size *)
      * Tast.expr  (* value, a term *))

and modul_info = 
    { val_infos: (val_ref * value_info) namemap;
      modul_infos: modul_info namemap }

let braceL x = leftL "{" $$ x $$ rightL "}"  
    
let namemapL xL xmap = Namemap.fold (fun nm x z -> z @@ xL nm x) xmap emptyL
let rec value_infoL = function
  | ConstValue (x,_)                  -> NicePrint.tconstL x
  | UnknownValue             -> wordL "?"
  | SizeValue (_,vinfo)      -> value_infoL vinfo
  | ValValue (vr,vinfo)      -> bracketL ((vrefL vr $$ wordL "alias") --- value_infoL vinfo)
  | ModuleValue minfo        -> wordL "struct<...>"
  | TupleValue vinfos        -> bracketL (values_infoL vinfos)
  | RecdValue (_,vinfos)     -> braceL   (values_infoL vinfos)
  | ConstrValue (ucr,vinfos) -> ucrefL ucr $$ bracketL (values_infoL vinfos)
  | CurriedLambdaValue(lambda_id,arities,bsize,expr',ety) -> wordL "lam" ++ exprL expr' (* (Printf.sprintf "lam(size=%d)" bsize) *)
  | ConstExprValue (size,x)  -> exprL x
  | LdlenValue vinfo         -> bracketL (wordL "ldlen" --- value_infoL vinfo)
  | DecrValue vinfo          -> bracketL (wordL "decr" --- value_infoL vinfo)
and values_infoL vinfos = commaListL (map value_infoL (Array.to_list vinfos))
and modul_infoL x = braceL ((wordL "Modules: " @@ namemapL (fun nm x -> wordL nm $$ modul_infoL x) x.modul_infos) 
                             @@ (wordL "Values:" @@ namemapL (fun nm (vref,x) -> vrefL vref $$ value_infoL x) x.val_infos))

let info_of_struct = function 
  | ModuleValue s -> s
  | _ -> failwith "info_of_struct"

type 'a summary =
    { ivalue: 'a; 
      fsize: int; (* what's the contribution to the size of this function *)
      tsize: int; (* what's the total contribution to the size of the assembly, including closure classes etc. *)
      effect: bool  (* Meaning: could mutate, could non-terminate, could raise exception 
                      * One use: an effect expr can not be eliminated as dead code (e.g. sequencing)
                      * One use: an effect=false expr can not throw an exception? so try-catch is removed. *)
    }

type expr_summary = value_info summary
type modul_summary = modul_info summary
    
(*-------------------------------------------------------------------------
!* bound_value_info
 *------------------------------------------------------------------------- *)

let rec size_value_infos arr = 
    let n = Array.length arr in 
    let rec go i acc = if i >= n then acc else max acc (size_value_info arr.(i)) in
    go 0 0
and size_value_info x =
    match x with
    | SizeValue (vdepth,v)     -> vdepth (* terminate recursion at CACHED size nodes *)
    | ConstValue (x,_)                  -> 1
    | UnknownValue             -> 1
    | ValValue (vr,vinfo)      -> size_value_info vinfo + 1
    | ModuleValue minfo        -> 1 (* do not care about size of these, they do not nest heavily... *)
    | TupleValue vinfos        
    | RecdValue (_,vinfos)
    | ConstrValue (_,vinfos)   -> 1 + size_value_infos vinfos
    | CurriedLambdaValue(lambda_id,arities,bsize,expr',ety) -> 1
    | ConstExprValue (size,_)  -> 1
    | LdlenValue vinfo         
    | DecrValue vinfo          -> 1 + size_value_info vinfo

let rec mk_known_size_value vdepth v =
    match v with
      | SizeValue(_,v) -> mk_known_size_value vdepth v
      | _ -> let minDepthForASizeNode = 5 in (* for small vinfos do not record size info, save space *)
             if vdepth > minDepthForASizeNode then SizeValue(vdepth,v) else v (* add nodes to stop recursion *)
    
let mk_size_value v =
    let vdepth = size_value_info v in
    mk_known_size_value vdepth v

let bound_value_info vinfo =
    let rec bound depth x =
        if depth<0 then UnknownValue else
        match x with
        | SizeValue (vdepth,vinfo) -> if vdepth < depth then x else mk_size_value (bound depth vinfo)
        | ValValue (vr,vinfo)      -> ValValue (vr,bound (depth-1) vinfo)
        | TupleValue vinfos        -> TupleValue (Array.map (bound (depth-1)) vinfos)
        | RecdValue (tcref,vinfos) -> RecdValue  (tcref,Array.map (bound (depth-1)) vinfos)
        | ConstrValue (ucr,vinfos) -> ConstrValue (ucr,Array.map (bound (depth-1)) vinfos)
        | ModuleValue minfo        -> x
        | ConstValue _                  -> x
        | UnknownValue             -> x
        | CurriedLambdaValue(lambda_id,arities,bsize,expr',ety) -> x
        | ConstExprValue (size,_)  -> x
        | LdlenValue vinfo         -> LdlenValue (bound (depth-1) vinfo)
        | DecrValue vinfo          -> DecrValue  (bound (depth-1) vinfo) in
    let max_depth  = 6 in  (* beware huge constants! *)
    let trim_depth = 3 in
    let vdepth = size_value_info vinfo in
    if vdepth > max_depth 
    then mk_size_value (bound trim_depth vinfo)
    else mk_known_size_value vdepth vinfo

(*-------------------------------------------------------------------------
!* What we know about the world 
 *------------------------------------------------------------------------- *)

type cenv =
    { g: Env.tcGlobals;
      amap: Import.importMap;
      optimizing: bool;
      scope: ccu; }

type env =
    { latestBoundId: ident option;
      syntacticArgs: local_val_ref list;  (* for late checking of the no-escape-conditions for basev's and mutable locals *)
      dont_inline: int Zset.t;  (* Prevent recursive inlining - list of lambda ids already being inlined. *)
      typar_infos: (typar_spec * type_value_info) list; 
      local_val_infos: value_info vspec_map;
      global_modul_infos: modul_info namemap;   }

let empty_env = 
    { latestBoundId=None; 
      syntacticArgs=[];
      dont_inline=Zset.empty int_ord;
      typar_infos=[]; 
      local_val_infos=vspec_map_empty(); 
      global_modul_infos=Map.empty }


(*-------------------------------------------------------------------------
!* Bind information about values 
 *------------------------------------------------------------------------- *)

let empty_modul_info = { val_infos = Map.empty; modul_infos = Map.empty }
let rec modul_info_union m1 m2 = 
    { val_infos =  Namemap.layer m1.val_infos m2.val_infos;
      modul_infos = Namemap.union modul_info_union  m1.modul_infos m2.modul_infos }

let modul_info_Union minfos = fold_right modul_info_union minfos empty_modul_info

let find_or_create_modul n ss = 
    match Map.tryfind n ss with 
    | Some res -> res
    | None -> empty_modul_info

let rec bind_val_in_submodul mp v vval ss =
    match mp with 
    | [] -> {ss with val_infos = Map.add (name_of_val v) (mk_local_vref v,vval) ss.val_infos }
    | h::t -> {ss with modul_infos = bind_val_in_modul h t v vval ss.modul_infos }

and bind_val_in_modul n mp v vval ss =
    let old =  find_or_create_modul n ss in 
    Map.add n (bind_val_in_submodul mp v vval old) ss
let bind_val_in_env (NLPath(mv,mp)) v vval env =
    {env with global_modul_infos = bind_val_in_modul (name_of_ccu mv) mp v vval env.global_modul_infos }

let rec  bind_top_module_in_modul n mp mval ss =
    match mp with 
    | [] -> 
        begin match Map.tryfind n ss with 
        | Some res -> Map.add n (modul_info_union mval res) ss
        | None -> Map.add n mval ss
        end
    | h::t -> 
        let old =  find_or_create_modul n ss in
        Map.add n {old with modul_infos =  bind_top_module_in_modul h t mval old.modul_infos} ss

let bind_top_module_in_env (NLPath(mv,mp)) mval env =
    {env with global_modul_infos = bind_top_module_in_modul (name_of_ccu mv) mp mval env.global_modul_infos }

(* Bind a value *)
let bind_vspec cenv v vval env = 

    if verbose then dprint_endline ("*** Binding "^name_of_val v); 
    let vval = if (mutability_of_val v <> Immutable) then UnknownValue else vval in
    match vval with 
(*     | UnknownValue -> env  *)
    | _ -> 
        let env = { env with local_val_infos=vspec_map_add v vval env.local_val_infos } in 
        (* If we're compiling fslib then also bind the value as a non-local path to allow us to resolve the compiler-non-local-refereneces *)
        let env = 
            if cenv.g.compilingFslib && isSome (pubpath_of_val v) 
            then bind_val_in_env (enclosing_nlpath_of_pubpath cenv.g.fslibCcu (the (pubpath_of_val v))) v vval env 
            else env in 
        env

let rec bind_module_vspecs cenv mval env =
    Namemap.fold_range (fun (v,vval) env -> bind_vspec cenv (deref_val v) vval env) mval.val_infos
      (Namemap.fold_range  (bind_module_vspecs cenv) mval.modul_infos env)


let bind_vspec_to_unknown cenv v env = bind_vspec cenv v UnknownValue env
let bind_vspecs_to_unknown cenv vs env = fold_right (bind_vspec_to_unknown cenv) vs env

let bind_tyval tyv typeinfo env = { env with typar_infos= (tyv,typeinfo)::env.typar_infos } 

let bind_tyvals_to_unknown tps env = 
    (* The optimizer doesn't use the type values it could track. *)
    (* However here is a hack to provide better names for generalized type parameters *)
    let nms = PrettyTypes.prettyTyparNames (fun _ -> true) 'T' (List.map (fun (tp,_) -> (name_of_typar tp)) env.typar_infos) tps in 
    List.iter2 
        (fun tp nm -> 
            if compgen_of_typar tp then 
                (data_of_typar tp).typar_id <- ident (nm,range_of_typar tp))
        tps
        nms;      
    fold_left (fun sofar arg -> bind_tyval arg UnknownTypeValue sofar) env tps 

let bind_ccu (mv:Tast.ccu) mval env = 
    { env with global_modul_infos=Map.add (name_of_ccu mv) mval env.global_modul_infos }

let mk_cenv scope g amap = 
    { scope=scope; 
      g=g; 
      amap=amap;
      optimizing=true }

(*-------------------------------------------------------------------------
!* partialExprVal - is the expr fully known?
 *------------------------------------------------------------------------- *)

let rec partialExprVal x = (* partialExprVal can not rebuild to an expr *)
  match x with
  | UnknownValue -> true
  | ModuleValue ss -> partialStructVal ss
  | TupleValue args | RecdValue (_,args) | ConstrValue (_,args) -> array_exists partialExprVal args
  | ConstValue _ | CurriedLambdaValue _ | ConstExprValue _ -> false
  | ValValue (_,a) | DecrValue a | LdlenValue a | SizeValue(_,a) -> partialExprVal a

and partialStructVal ss =
     Map.exists (fun _ x -> partialStructVal x) ss.modul_infos or
     Map.exists (fun _ (_,x) -> partialExprVal x) ss.val_infos

let check msg m vref res  =
  if mustinline(inlineFlag_of_vref vref) && partialExprVal res then (
    errorR(Error("The value '"^full_name_of_vref vref^"' was marked inline but "^msg, m));
  );
  (vref,res)

(*-------------------------------------------------------------------------
!* Lookup information about values 
 *------------------------------------------------------------------------- *)

let lookup_locval env v m = 
    (* Abstract slots do not have values *)
    match vspr_of_lvref v with 
    | Some(vspr) when vspr.vspr_flags.memFlagsAbstract -> UnknownValue
    | _ -> 
        match vspec_map_tryfind v env.local_val_infos with 
        | Some vval -> vval
        | None -> 
            if mustinline(inlineFlag_of_val v) then (
              errorR(Error("The value '"^full_name_of_vref (mk_local_vref v) ^"' was marked inline but was not bound in the optimization environment", m));
            );
           warning(Error ("*** Local value "^(name_of_val v)^" not found during optimization. Please report this problem",m)); 
          UnknownValue 

let can_lookup_ccu env mv = Map.mem (name_of_ccu mv) env.global_modul_infos
let lookup_ccu env mv = Map.find (name_of_ccu mv) env.global_modul_infos

let rec can_lookup_modpath sv p = 
    match p with 
    | [] -> true
    | h::t -> 
        match Map.tryfind h sv.modul_infos with 
        | Some info -> can_lookup_modpath info t
        | None -> (if verbose then dprint_endline ("\n\n*** Optimization info for submodule "^h^" not found in parent module which contains submodules: "^String.concat "," (Namemap.domainL sv.modul_infos)); false)

let rec lookup_modpath sv p = 
  match p with 
    [] -> sv
  | h::t -> lookup_modpath (Map.find h sv.modul_infos) t

let can_lookup_submodul env (NLPath(ccu,p)) = 
  if can_lookup_ccu env ccu then
    can_lookup_modpath (lookup_ccu env ccu) p
  else 
    (if verbose then dprint_endline ("\n\n*** CCU "^name_of_ccu ccu^" not found, available ccus are: "^String.concat "," (map fst (Namemap.to_list env.global_modul_infos))); false)

let lookup_submodul env (NLPath(ccu,p)) = lookup_modpath (lookup_ccu env ccu) p

let lookup_nonlocal_val env nlr =
  if (* in_this: REVIEW: optionally turn x-module on/off on per-module basis  or  *)
    crossmoduleopt () || 
    mustinline(inlineFlag_of_val (deref_nlval nlr)) then 
   begin
    let smv = nlpath_of_nlref nlr in
    let n = item_of_nlref nlr in
    if not (can_lookup_ccu env (ccu_of_nlpath smv)) then
      (if verbose then dprint_endline ("\n\n*** CCU "^name_of_ccu (ccu_of_nlpath smv)^" not found when looking for value "^n^" from module "^(full_name_of_nlpath smv)^", available ccus are: "^String.concat "," (map fst (Namemap.to_list env.global_modul_infos))));

    if can_lookup_submodul env smv then  
      let struct_info =  lookup_submodul env smv in
      match Map.tryfind n struct_info.val_infos with 
      | Some ninfo -> snd ninfo
      | None -> 
           (if verbose then dprint_endline ("\n\n*** Optimization info for value "^n^" from module "^(full_name_of_nlpath smv)^" not found, module contains values: "^String.concat "," (Namemap.domainL struct_info.val_infos));  
            UnknownValue)
    else (if verbose then dprintf2 "\n\n*** Optimization info for module %s from ccu %s not found." (full_name_of_nlpath smv) (name_of_ccu (ccu_of_nlpath smv));  
          UnknownValue)
   end
  else UnknownValue

let lookup_vref cenv env m v =  
    let res = 
        match v with 
        | Ref_private v -> lookup_locval env v m
        | Ref_nonlocal nlr -> lookup_nonlocal_val env nlr in
    check "its value was incomplete" m v res |> ignore;
    res

(*-------------------------------------------------------------------------
!* Try to get information about values of particular types
 *------------------------------------------------------------------------- *)

let rec strip_value = function
  | ValValue(_,details) -> strip_value details (* step through ValValue "aliases" *) 
  | SizeValue(_,details) -> strip_value details (* step through SizeValue "aliases" *) 
  | vinfo               -> vinfo

let dest_const_value ev = 
  match strip_value ev with
  | ConstValue(c,_) -> Some c
  | _ -> None

let dest_lambda_value ev = 
  match strip_value ev with 
  | CurriedLambdaValue info -> Some info
  | _ -> None

let dest_const_expr_value ev = 
  match strip_value ev with
  | ConstExprValue info -> Some info
  | _ -> None

let dest_tuple_value ev = 
  match strip_value ev with 
  | TupleValue info -> Some info
  | _ -> None

let dest_recd_value ev = 
  match strip_value ev with 
  | RecdValue (tcref,info) -> Some info
  | _ -> None

let dest_constr_value ev = 
  match strip_value ev with 
  | ConstrValue (c,info) -> Some (c,info)
  | _ -> None

let dest_decr_value ev = 
  match strip_value ev with 
  | DecrValue a -> Some a
  | _ -> None

let dest_ldlen_value ev = 
  match strip_value ev with 
  | LdlenValue a -> Some a
  | _ -> None

let mk_int32_value g n = ConstValue(TConst_int32 n, g.int32_ty)
let mk_int64_value g n = ConstValue(TConst_int64 n, g.int64_ty)
let dest_int32_value ev =
  match strip_value ev with
  | ConstValue(TConst_int32 n,_) -> Some n
  | _ -> None

let dest_int64_value ev =
  match strip_value ev with
  | ConstValue(TConst_int64 n,_) -> Some n
  | _ -> None

      
(*-------------------------------------------------------------------------
!* mk value_infos
 *------------------------------------------------------------------------- *)

let mk_val_value g m vref vinfo            = 
    let rec check x = 
        match x with 
        | ValValue (vref2,detail)  -> if g.vref_eq vref vref2 then error(Error("recursive ValValue "^showL(value_infoL vinfo),m)) else check detail
        | SizeValue (n,detail) -> check detail
        | _ -> () in 
    check vinfo;
    ValValue (vref,vinfo)       |> bound_value_info
let mk_recd_value tcref tyargs argvals = RecdValue (tcref,argvals)   |> bound_value_info
let mk_tuple_value argvals             = TupleValue argvals          |> bound_value_info
let mk_constr_value cspec argvals      = ConstrValue (cspec,argvals) |> bound_value_info
let mk_const_value c ty                = ConstValue(c,ty)

let binop_integral g f32 f64 a b = 
     match dest_int32_value a,dest_int32_value b with
     | Some a,Some b -> Some(mk_int32_value g (f32 a b))
     | _ -> 
     match dest_int64_value a,dest_int64_value b with
     | Some a,Some b -> Some(mk_int64_value g (f64 a b))
     | _ -> 
     None
         
(* Highly conservative analysis to fix up array bounds *)
let mk_asm_value g instrs argvals =
  match instrs,argvals with
    | [ I_ldlen; I_arith (AI_conv DT_I4) ],[a] -> LdlenValue a  (* see NOTE: ldlen *)
    | [ I_arith AI_add ],[t1;t2] -> 
       begin 
         match binop_integral g Int32.add Int64.add t1 t2 with 
         | Some res -> res
         | _ -> 
         match dest_int32_value t1, dest_decr_value t2 with 
         | Some 1l,Some a ->  a  (* 1 + (a - 1) = a *)
         | _ -> 
         match dest_decr_value t1, dest_int32_value t2 with 
         | Some a,Some 1l ->  a  (* (a - 1) + 1 = a *)
         | _ ->
         UnknownValue
       end
    | [ I_arith AI_sub ],[t1;t2] -> 
       begin 
         match binop_integral g Int32.sub Int64.sub t1 t2 with 
         | Some res -> res
         | _ -> 
         match dest_int32_value t2 with 
         | Some 1l -> DecrValue t1  (* t1 - 1 = DecrValue(t1) *)
         | _ -> 
         UnknownValue
       end
    | [ I_arith AI_mul ],[a;b] -> (match binop_integral g Int32.mul Int64.mul a b with Some res -> res | None -> UnknownValue)
    | [ I_arith AI_and ],[a;b] -> (match binop_integral g Int32.logand Int64.logand a b  with Some res -> res | None -> UnknownValue)
    | [ I_arith AI_or ],[a;b] -> (match binop_integral g Int32.logor Int64.logor a b  with Some res -> res | None -> UnknownValue)
    | [ I_arith AI_xor ],[a;b] -> (match binop_integral g Int32.logxor Int64.logxor a b  with Some res -> res | None -> UnknownValue)
    | [ I_arith AI_not ],[a] -> 
       begin 
         match dest_int32_value a with
         | Some a -> (mk_int32_value g (Int32.lognot a))
         | _ -> 
         match dest_int64_value a with
         | Some a -> (mk_int64_value g (Int64.lognot a))
         | _ -> 
         UnknownValue
       end
    | [ I_arith AI_shl ],[a;n] -> 
       begin match dest_int32_value a,dest_int32_value n with
       | Some a,Some n when n >= 0l && n <= 31l -> (mk_int32_value g (a <<< (??? n)))
       | _ -> UnknownValue
       end
    | [ I_arith AI_shr ],[a;n] -> 
       begin match dest_int32_value a,dest_int32_value n with
       | Some a,Some n when n >= 0l && n <= 31l -> (mk_int32_value g (a asr (??? n)))
       | _ -> UnknownValue
       end
    | _ -> UnknownValue


(*-------------------------------------------------------------------------
!* Size constants and combinators
 *------------------------------------------------------------------------- *)

let local_var_size = 1

        
let rec add_tsizes_aux acc l = match l with [] -> acc | h::t -> add_tsizes_aux (h.tsize + acc) t
let add_tsizes l = add_tsizes_aux 0 l
let rec add_fsizes_aux acc l = match l with [] -> acc | h::t -> add_fsizes_aux (h.fsize + acc) t
let add_fsizes l = add_fsizes_aux 0 l

(*-------------------------------------------------------------------------
!* opt list/array combinators - zipping (_,_) return type
 *------------------------------------------------------------------------- *)
let rec or_effects l = match l with [] -> false | h::t -> h.effect || or_effects t
        
let rec opt_list_aux f l acc1 acc2 = 
  match l with 
  | [] -> List.rev acc1, List.rev acc2
  | (h ::t) -> 
      let (x1,x2)  = f h in
      opt_list_aux f t (x1::acc1) (x2::acc2) 

let opt_list f l = opt_list_aux f l [] [] 

(* let opt_array f l = let l1,l2 = opt_list f (Array.to_list l) in Array.of_list l1, l2 *)

let no_exprs = [],[]

(*-------------------------------------------------------------------------
!* Common ways of building new value_infos
 *------------------------------------------------------------------------- *)

let combine_vinfos einfos res = 
      { tsize  = add_tsizes einfos;
        fsize  = add_fsizes einfos;
        effect = or_effects einfos; 
        ivalue = res }

let combine_vinfos_unknown einfos = combine_vinfos einfos UnknownValue

(*-------------------------------------------------------------------------
!* Hide information because of a signature
 *------------------------------------------------------------------------- *)

let abstractModulInfoByHiding m mhi =

(* This code is not sound when abstracting at the assembly boundary. 
       1. The MHI is not looking at 'internal' access attributes
               
*)
    let hiddenTycon,hiddenTyconRepr,hiddenVal, hiddenRfield, hiddenUconstr = 
        Zset.mem_of mhi.mhiTycons, 
        Zset.mem_of mhi.mhiTyconReprs, 
        Zset.mem_of mhi.mhiVals, 
        Zset.mem_of mhi.mhiRecdFields, 
        Zset.mem_of mhi.mhiUnionConstrs in 

    let rec abstractExprInfo ivalue = 
        if verbose then dprintf0 "remapAndAbstractExprValBySig\n"; 
        match ivalue with 
        (* Check for escaping value. Revert to old info if possible  *)
        | ValValue (vref2,detail) ->
            let detail' = abstractExprInfo detail in 
            let v2 = (deref_val vref2) in
            if Zset.exists hiddenTycon (free_in_val v2).free_loctycons  or hiddenVal v2
            then detail'
            else ValValue (vref2,detail')
        (* Check for escape in lambda *)
        | CurriedLambdaValue (_,_,_,expr,_) | ConstExprValue(_,expr) when 
            (let fvs = free_in_expr expr in  
             (*dprintf2 "abstractModulInfoByHiding, #fvs.free_locvals = %d, #fvs.free_rfields = %d\n" (length (Zset.elements fvs.free_locvals)) (length (Zset.elements fvs.free_rfields)); *)
             Zset.exists hiddenVal fvs.free_locvals or
             Zset.exists hiddenTycon fvs.free_tyvars.free_loctycons or
             Zset.exists hiddenTyconRepr fvs.free_loctycon_reprs or
             Zset.exists hiddenRfield    fvs.free_rfields  or
             Zset.exists hiddenUconstr    fvs.free_uconstrs ) ->
                UnknownValue
        (* Check for escape in constant *)
        | ConstValue(_,ty) when 
            (let ftyvs = free_in_type ty in  
             Zset.exists hiddenTycon ftyvs.free_loctycons) ->
                UnknownValue
        | TupleValue vinfos         -> TupleValue (Array.map abstractExprInfo vinfos)
        | RecdValue (tcref,vinfos)  -> 
            if hiddenTyconRepr (deref_tycon tcref) || array_exists (rfref_of_rfield tcref >> hiddenRfield) (rfields_array_of_tcref tcref)
            then UnknownValue 
            else RecdValue (tcref,Array.map abstractExprInfo vinfos)
        | ConstrValue(ucref,vinfos) -> 
            let tcref = tcref_of_ucref ucref in
            if hiddenTyconRepr (deref_tycon (tcref_of_ucref ucref)) || array_exists (ucref_of_uconstr tcref >> hiddenUconstr) (uconstrs_array_of_tcref tcref)
            then UnknownValue 
            else ConstrValue (ucref,Array.map abstractExprInfo vinfos)
        | ModuleValue sinfo         -> ModuleValue (abstractModulInfo sinfo)
        | DecrValue vinfo           -> DecrValue (abstractExprInfo vinfo)
        | LdlenValue vinfo          -> LdlenValue (abstractExprInfo vinfo)      
        | SizeValue(vdepth,vinfo) -> mk_size_value (abstractExprInfo vinfo)
        | UnknownValue  
        | ConstExprValue _   
        | CurriedLambdaValue _ 
        | ConstValue _  -> ivalue
      and abstractModulInfo ss =
         if verbose then dprintf0 "abstractModulInfo\n"; 
         { modul_infos = Namemap.map abstractModulInfo ss.modul_infos;
           val_infos = 
               ss.val_infos 
               |> Namemap.filter_range (fst >> deref_val >> hiddenVal >> not)
               |> Namemap.map (fun (vref,e) -> 
                check "its implementation uses a binding hidden by a signature" m vref (abstractExprInfo e) )  } in 
           
      abstractModulInfo

(*-------------------------------------------------------------------------
!* Hide information because of a signature
 *------------------------------------------------------------------------- *)

let abstractModulInfoToEssentials =

    let rec abstractExprInfo ivalue = 
        match ivalue with 
        | ValValue (vref2,detail) -> abstractExprInfo detail 
        | TupleValue vinfos         -> TupleValue (Array.map abstractExprInfo vinfos)
        | RecdValue (tcref,vinfos)  -> RecdValue (tcref,Array.map abstractExprInfo vinfos)
        | ConstrValue(cspec,vinfos) -> ConstrValue (cspec,Array.map abstractExprInfo vinfos)
        | ModuleValue sinfo         -> ModuleValue (abstractModulInfo sinfo)
        | DecrValue vinfo           -> DecrValue (abstractExprInfo vinfo)
        | LdlenValue vinfo          -> LdlenValue (abstractExprInfo vinfo)      
        | SizeValue(vdepth,vinfo) -> mk_size_value (abstractExprInfo vinfo)
        | UnknownValue  | ConstExprValue _   | CurriedLambdaValue _ | ConstValue _  -> ivalue
      and abstractModulInfo ss =
         { modul_infos = Namemap.map abstractModulInfo ss.modul_infos;
           val_infos =  ss.val_infos  |> Namemap.filter_range (fst >> inlineFlag_of_vref  >> mustinline)  } in            
      abstractModulInfo


(*-------------------------------------------------------------------------
!* Hide information because of a "let ... in ..." or "let rec  ... in ... "
 *------------------------------------------------------------------------- *)

let abstractExprInfoByVars m (bvs,btyvs) ivalue =
  (* Module and member bindings can be skipped when checking abstraction, since abstraction of these values has already been done when *)
  (* we hit the end of the module and called abstractModulInfoByHiding. If we don't skip these then we end up quadtratically retraversing  *)
  (* the inferred optimization data, i.e. at each binding all the way up a sequences of 'lets' in a module. *)
  let bvs = bvs |> filter (modbind_of_val >> not) in 
  match bvs,btyvs with 
  | [],[] -> ivalue
  | _ -> 
(*
      let dump() = 
                  bvs |> List.iter (fun v -> dprintf3 "  -- bv  %s @ %a\n" (name_of_val v) output_range (range_of_val v));
                  btyvs |> List.iter (fun v -> dprintf3 "  -- btyv  %s @ %a\n" (name_of_typar v) output_range (range_of_typar v)) in 
*)      
      let rec abstractExprInfo ivalue =
          match ivalue with 
        (* Check for escaping value. Revert to old info if possible  *)
          | ValValue (Ref_private v2,detail) when  
            (nonNil bvs && exists (local_vref_eq v2) bvs) || 
            (nonNil btyvs &&
             let ftyvs = free_in_val v2 in 
             exists (Zset.mem_of ftyvs.free_loctypars) btyvs) -> 

              if verbose then (
                  dprintf3 "hiding value '%s' when used in expression (see %a)\n" (name_of_val v2) output_range (range_of_val v2);
                  let ftyvs = free_in_val v2 in 
                  ftyvs.free_loctypars |> Zset.iter (fun v -> dprintf3 "  -- ftyv  %s @ %a\n" (name_of_typar v) output_range (range_of_typar v));
                  bvs |> List.iter (fun v -> dprintf3 "  -- bv  %s @ %a\n" (name_of_val v) output_range (range_of_val v));
                  btyvs |> List.iter (fun v -> dprintf3 "  -- btyv  %s @ %a\n" (name_of_typar v) output_range (range_of_typar v))
              );
              abstractExprInfo detail
          | ValValue (v2,detail) -> 
              let detail' = abstractExprInfo detail in 
              ValValue (v2,detail')
        (* Check for escape in lambda *)
          | CurriedLambdaValue (_,_,_,expr,_) | ConstExprValue(_,expr)  when 
            let fvs = free_in_expr expr in
            (nonNil bvs   && exists (Zset.mem_of fvs.free_locvals) bvs) or
            (nonNil btyvs && exists (Zset.mem_of fvs.free_tyvars.free_loctypars) btyvs) or
            (fvs.uses_method_local_constructs = true) ->
              if verbose then (
                  let fvs = free_in_expr expr in  
                  dprintf4 "Trimming lambda @ %a, uses_method_local_constructs = %b, exprL = %s\n"  output_range (range_of_expr expr) fvs.uses_method_local_constructs (showL (exprL expr));
                  fvs.free_locvals |> Zset.iter (fun v -> dprintf3 "fv  %s @ %a\n" (name_of_val v) output_range (range_of_val v));
                  fvs.free_tyvars.free_loctypars |> Zset.iter (fun v -> dprintf3 "ftyv  %s @ %a\n" (name_of_typar v) output_range (range_of_typar v));
                  bvs |> List.iter (fun v -> dprintf3 "bv  %s @ %a\n" (name_of_val v) output_range (range_of_val v));
                  btyvs |> List.iter (fun v -> dprintf3 "btyv  %s @ %a\n" (name_of_typar v) output_range (range_of_typar v))
              );
              UnknownValue
        (* Check for escape in lambda *)
          | ConstValue(_,ty) when 
            let ftyvs = free_in_type ty in
            (nonNil btyvs && exists (Zset.mem_of ftyvs.free_loctypars) btyvs) ->
              UnknownValue
        (* Otherwise check all sub-values *)
          | TupleValue vinfos -> TupleValue (Array.map (abstractExprInfo) vinfos)
          | RecdValue (tcref,vinfos) -> RecdValue (tcref,Array.map (abstractExprInfo) vinfos)
          | ConstrValue (cspec,vinfos) -> ConstrValue(cspec,Array.map (abstractExprInfo) vinfos)
          | ModuleValue sinfo -> ModuleValue (abstractModulInfo sinfo)
          | DecrValue vinfo -> DecrValue (abstractExprInfo vinfo)
          | LdlenValue vinfo -> LdlenValue (abstractExprInfo vinfo)
          | CurriedLambdaValue _ 
          | ConstValue _ 
          | ConstExprValue _ 
          | UnknownValue -> ivalue
          | SizeValue (vdepth,vinfo) -> mk_size_value (abstractExprInfo vinfo)

      and abstractModulInfo ss =
         { modul_infos  = ss.modul_infos  |> Namemap.map (abstractModulInfo) ;
           val_infos = ss.val_infos |> Namemap.map (fun (vref,e) -> 
               if verbose then dprintf3 "checking %s @ %a\n" (name_of_vref vref) output_range (range_of_vref vref); 
               check "its implementation uses a private binding" m vref (abstractExprInfo e) ) }

      in abstractExprInfo ivalue

(*-------------------------------------------------------------------------
!* Rewrite optimization, e.g. to use public stable references so we can pickle it
 * to disk.
 *------------------------------------------------------------------------- *)
let remapModulInfo g tmenv =

    let rec remapExprInfo ivalue = 
        if verbose then dprintf0 "remapExprInfo\n"; 
        match ivalue with 
        | ValValue (v,detail)      -> ValValue (remap_vref tmenv v,remapExprInfo detail)
        | TupleValue vinfos         -> TupleValue (Array.map remapExprInfo vinfos)
        | RecdValue (tcref,vinfos)  -> RecdValue (remap_tcref tmenv.tyenv.tcref_remap tcref, Array.map remapExprInfo vinfos)
        | ConstrValue(cspec,vinfos) -> ConstrValue (remap_ucref tmenv.tyenv.tcref_remap cspec,Array.map remapExprInfo vinfos)
        | ModuleValue sinfo         -> ModuleValue (remapModulInfo sinfo)
        | DecrValue vinfo           -> DecrValue (remapExprInfo vinfo)
        | LdlenValue vinfo          -> LdlenValue (remapExprInfo vinfo)      
        | SizeValue(vdepth,vinfo) -> mk_size_value (remapExprInfo vinfo)
        | UnknownValue              -> UnknownValue
        | CurriedLambdaValue (uniq,arity,sz,expr,typ)  -> CurriedLambdaValue (uniq,arity,sz,remap_expr g false tmenv expr,remap_type tmenv.tyenv typ)  
        | ConstValue (c,ty)  -> ConstValue (c,remap_type tmenv.tyenv ty)
        | ConstExprValue (sz,expr)  -> ConstExprValue (sz,remap_expr g false tmenv expr)
    and remapModulInfo ss =
         if verbose then dprintf0 "remapModulInfo\n"; 
         { modul_infos = Namemap.map remapModulInfo ss.modul_infos;
           val_infos = Namemap.map (fun (vref,e) -> (remap_vref tmenv vref,remapExprInfo e)) ss.val_infos } in 
           
    remapModulInfo

(*-------------------------------------------------------------------------
!* Hide information when a value is no longer visible
 *------------------------------------------------------------------------- *)

let abstractAndRemapModulInfo msg g m (repackage,hidden) info =
    if verbose then dprintf4 "%s - %a - Optimization data prior to trim: \n%s\n" msg output_range m (Layout.showL (Layout.squashTo 192 (modul_infoL info)));
    let info = abstractModulInfoByHiding m hidden info in 
    if verbose then dprintf4 "%s - %a - Optimization data after trim:\n%s\n" msg output_range m (Layout.showL (Layout.squashTo 192 (modul_infoL info)));
    let info = remapModulInfo g (mk_repackage_remapping repackage) info in 
    if verbose then dprintf4 "%s - %a - Optimization data after remap:\n%s\n" msg output_range m (Layout.showL (Layout.squashTo 192 (modul_infoL info)));
    info

(*-------------------------------------------------------------------------
!* Misc helerps
 *------------------------------------------------------------------------- *)

(* Mark some variables (the ones we introduce via abstract_big_targets) as don't-eliminate *)
let continuedName = "$cont"
let dontElimVarsWithThisCharInName = '$'

(* Type applications of F# "type functions" may cause side effects, e.g. *)
(* let x<'a> = printfn "hello"; typeof<'a> *)
(* In this case do not treat them as constants. *)
let is_tyfunc_vref_expr = function 
          | TExpr_val (fv,_,_) -> is_tyfunc_of_vref fv
          | _ -> false

let rec small_const_expr x =
  match x with
  | TExpr_val (v,_,m) -> (mutability_of_vref v = Immutable)
  (* Type applications of existing functions are always simple constants, with the exception of F# 'type functions' *)
  (* REVIEW: we could also include any under-applied application here. *)
  | TExpr_app(fe,_,tyargs,args,_) -> isNil(args) && not (is_tyfunc_vref_expr fe) && small_const_expr fe
  | _ -> false

let ivalue_of_expr expr = 
  if small_const_expr expr then 
    ConstExprValue(0,expr)
  else UnknownValue

(*-------------------------------------------------------------------------
!* Dead binding elimination 
 *------------------------------------------------------------------------- *)
 
let is_used_or_has_effect m fvs (b,binfo) =
    let v = var_of_bind b in 
    not (do_opt_unused()) or
    (member_info_of_val v <> None) or
    binfo.effect || 
    Zset.mem v fvs.free_locvals 

let rec split_is_used_or_has_effect m fvs x = 
  match x with 
    [] -> [],[]
  | ((h1,h2) as p ::t) -> 
      let (l1,l2) as p2 = split_is_used_or_has_effect m fvs t in
      if is_used_or_has_effect m fvs p then (h1::l1, h2::l2) else p2

(*-------------------------------------------------------------------------
!* 
 *------------------------------------------------------------------------- *)

let il_instr_has_effect i = 
  match i with 
  | I_arith (AI_nop | AI_ldc _ | AI_add | AI_sub | AI_mul | AI_xor | AI_and | AI_or | 
             AI_ceq | AI_cgt | AI_cgt_un | AI_clt | AI_clt_un | AI_conv _ | AI_shl | 
             AI_shr | AI_shr_un | AI_neg | AI_not | AI_ldnull)
  | I_ldstr _ | I_ldtoken _  -> false
  | _ -> true
  
let il_instrs_have_effect instrs = List.exists il_instr_has_effect instrs

(*-------------------------------------------------------------------------
!* Effects
 *
 * note: allocating an object with observable identity (i.e. a name) 
 * or reading from a mutable field counts as an 'effect', i.e. in 
 * this context 'effect' has it's usual meaning in the effect analysis literature of 
 *   read-from-mutable 
 *   write-to-mutable 
 *   name-generation
 *   arbitrary-side-effect (e.g. 'non-termination' or 'fire the missiles')
 *------------------------------------------------------------------------- *)

let rec expr_has_effect g expr = 
    match expr with 
    | TExpr_val (vref,_,_) -> (is_tyfunc_of_vref vref) or (mutability_of_vref vref <> Immutable)
    | TExpr_quote _ 
    | TExpr_lambda _
    | TExpr_tlambda _ 
    | TExpr_const _ -> false
    (* type applications do not have effects with the exception of type functions *)
    | TExpr_app(f0,_,_,[],_) -> (is_tyfunc_vref_expr f0) or expr_has_effect g f0
    | TExpr_op(op,_,args,_) -> exprs_have_effect g args || op_has_effect g op
    | TExpr_letrec(binds,body,_,_) -> binds_have_effect g binds || expr_has_effect g body
    | TExpr_let(bind,body,_,_) -> bind_has_effect g bind || expr_has_effect g body
    (* REVIEW: could add TExpr_obj on an interface type - these are similar to records of lambda expressions *)
    | _ -> true
and exprs_have_effect g exprs = List.exists (expr_has_effect g) exprs
and binds_have_effect g binds = List.exists (bind_has_effect g) binds
and bind_has_effect g bind = bind |> rhs_of_bind |> expr_has_effect g
and op_has_effect g op = 
    match op with 
    | TOp_tuple -> false
    | TOp_recd (ctor,tcref) -> 
        begin match ctor with 
        | RecdExprIsObjInit -> true
        | RecdExpr -> tcref_alloc_observable tcref
        end
    | TOp_uconstr ucref -> tcref_alloc_observable (tcref_of_ucref ucref)
    | TOp_exnconstr ecref -> ecref_alloc_observable ecref
    | TOp_bytes _ | TOp_array -> true (* alloc observable *)
    | TOp_constr_tag_get _ -> false
    | TOp_constr_field_get (ucref,n) -> ucref_rfield_mutable g ucref n 
    | TOp_asm(instrs,_) -> il_instrs_have_effect instrs
    | TOp_tuple_field_get(_) -> false
    | TOp_exnconstr_field_get(ecref,n) -> ecref_rfield_mutable ecref n 
    | TOp_get_ref_lval -> false
    | TOp_field_get rfref  -> (rfield_of_rfref rfref).rfield_mutable
    | TOp_field_get_addr rfref  -> true (* check *)
    | TOp_constr_field_set _
    | TOp_exnconstr_field_set _
    | TOp_coerce
    | TOp_for _ | TOp_while  | TOp_try_catch | TOp_try_finally (* note: these really go through a different path anyway *)
    | TOp_trait_call _
    | TOp_ilcall _ (* conservative *)
    | TOp_lval_op _  (* conservative *)
    | TOp_field_set _ -> true


let try_elim_assign cenv env (TBind(vspec1,e1)) e2 m  =
    (* Only apply this to compiler generated values unless optimization is on. *)
    (* Improves the debug experience. *)
    if not cenv.optimizing || not (compgen_of_val vspec1 || do_opt_immediate_bind()) then None else
     
    (* Peephole on immediate consumption of single bindings, e.g. "let x = e in x" --> "e" *)
    (* REVIEW: enhance this by general elimination of bindings to *)
    (* non-side-effecting expressions that are used only once. *)
    (* But note the cases below cover side-effecting expressions as well.... *)
    let unique_use vspec2 args = 
           local_vref_eq vspec1 vspec2  
       && (not (String.contains (name_of_val vspec2) dontElimVarsWithThisCharInName))
       (* REVIEW: this looks slow. Look only for one variable instead *)
       && (let fvs = acc_free_in_exprs args empty_freevars in 
           not (Zset.mem vspec1 fvs.free_locvals)) in

     (* Immediate consumption of value as 2nd argument to a construction || projection operation *)
    let rec immediate_unique_use_context rargsl argsr = 
          match argsr with 
          | (TExpr_val(Ref_private vspec2,_,_)) :: argsr2
             when local_vref_eq vspec1 vspec2 && unique_use vspec2 (List.rev rargsl@argsr2) -> Some(List.rev rargsl,argsr2)
          | argsrh :: argsrt when not (expr_has_effect cenv.g argsrh) -> immediate_unique_use_context (argsrh::rargsl) argsrt 
          | _ -> None in 

    match strip_expr e2 with 

     (* Immediate consumption of value as itself *)
     | TExpr_val(Ref_private vspec2,_,_) 
         when unique_use vspec2 [] -> 
                (* if verbose then dprintf1 "Simplifying let x = e in x near %a\n" output_range m;*)
           Some e1

     (* Immediate consumption of value by a pattern match *)
     | TExpr_match(exprm,TDSwitch(TExpr_val(Ref_private vspec2,_,_),cases,dflt,_),targets,m,ty2,_)
         when  local_vref_eq vspec1 vspec2 && 
              let fvs = acc_free_in_targets targets (acc_free_in_switch_cases cases dflt empty_freevars) in 
              not (Zset.mem vspec1 fvs.free_locvals) -> 
                (* if verbose then dprintf1 "Simplifying let x = e in match x with ... near %a\n" output_range m;*)
          Some (TExpr_match(range_of_expr e1,TDSwitch(e1,cases,dflt,m),targets,m,ty2,new_cache()))
           
     (* Immediate consumption of value as a function *)
     (* note: functions are evaluated before args *)
     (* note: do not include functions with a single arg of unit type, introduced by abstract_big_targets *)
     | TExpr_app(f,f0ty,tyargs,args,m) 
           when (not (String.contains (name_of_val vspec1) dontElimVarsWithThisCharInName)) ->
         begin match immediate_unique_use_context [] (f::args) with 
         | Some([],rargs) -> Some (beta_mk_appl cenv.g nng (e1,f0ty,[tyargs],rargs ,m))
         | Some(f::largs,rargs) -> Some (beta_mk_appl cenv.g nng (f,f0ty,[tyargs],largs @ (e1::rargs),m))
         | None -> None
         end

     (* Immediate consumption of value as first non-effectful argument to a construction or projection operation *)
      | TExpr_op (c,tyargs,args,m) -> 
         begin match immediate_unique_use_context [] args with 
         | Some(largs,rargs) -> Some (TExpr_op (c,tyargs,(largs @ (e1:: rargs)),m))
         | None -> None
         end

     | _ ->  
        None

let try_elim_let cenv env bind e2 m = 
  match try_elim_assign cenv env bind e2 m with 
  | Some e2' -> e2',-local_var_size  (* eliminated a let *)
  | None -> mk_let_bind m bind e2 ,0

(*-------------------------------------------------------------------------
!* expand_structural_bind
 *
 * Expand bindings to tuple expressions by factoring sub-expression out as prior bindings.
 * Similarly for other structural constructions, like records...
 * If the item is only projected from then the construction (allocation) can be eliminated.
 * This transform encourages that by allowing projections to be simplified.
 *------------------------------------------------------------------------- *)

let is_val = function TExpr_val _ -> true | _ -> false
let expand_structural_bind cenv env expr =
  match expr with
    | TExpr_let (bind,body,m,_) when (is_tuple (rhs_of_bind bind) &&  
                                      let v      = var_of_bind bind in 
                                      isNone(arity_of_val v) &&  
                                      isNone(member_info_of_val v) && 
                                      not (is_tyfunc_of_val v) &&
                                      (mutability_of_val v = Immutable)) ->
          let v      = var_of_bind bind in
          let args   = try_dest_tuple (rhs_of_bind bind) in
          if for_all is_val args then
            expr (* avoid re-expanding when recursion hits original binding *)
          else
            let argTys = dest_tuple_typ (type_of_val v) in
            let argBind i arg argTy =
              let name = name_of_val v ^ "_" ^ string_of_int i in
              let v,ve = mk_compgen_local (range_of_expr arg) name argTy in
              ve,mk_bind v arg
            in
            let ves,binds = list_mapi2 argBind args argTys |> List.split in
            let tuple = mk_tupled cenv.g m ves argTys in
            mk_lets_bind m binds (mk_let m v tuple body)
            (* REVIEW: other cases - records, explicit lists etc. *)
    | expr -> expr
    
(*-------------------------------------------------------------------------
!* The traversal
 *------------------------------------------------------------------------- *)

let rec opt_expr cenv (env:env) expr =
    if verbose then dprintf2 "opt_expr@%a\n" output_range (range_of_expr expr);
    let expr = strip_expr expr in
    match expr with
    (* treat the common linear cases to avoid stack overflows, using an explicit continutation *)
    | TExpr_seq _ | TExpr_let _ ->  opt_linear cenv env expr (fun x -> x)

    | TExpr_const (c,m,ty) -> opt_const cenv env expr (c,m,ty)
    | TExpr_val (v,vFlags,m) -> opt_val cenv env expr (v,m)
    | TExpr_hole (m,_) -> expr,{ tsize = 10; fsize=10; effect = false;  ivalue=UnknownValue }
    | TExpr_quote(raw,ast,m,ty) -> 
          (* Do not rewrite at higher levels *)
          TExpr_quote(raw,ast,m,ty),
          { tsize = 10;
            fsize = 1;
            effect = false;  
            ivalue=UnknownValue }
    | TExpr_obj (_,typ,basev,expr,overrides,iimpls,m,_) -> opt_obj_expr cenv env (typ,basev,expr,overrides,iimpls,m)
    | TExpr_op (c,tyargs,args,m) -> opt_op cenv env (c,tyargs,args,m)
    | TExpr_app(f,fty,tyargs,argsl,m) -> opt_app cenv env (f,fty,tyargs,argsl,m) 
    (* REVIEW: fold the next two cases together *)
    | TExpr_lambda(lambda_id,_,argvs,body,m,rty,_) -> 
        let topValInfo = TopValInfo (0,[argvs |> map (fun _ -> TopValData.unnamedTopArg1)],TopValData.unnamedRetVal) in 
        let ty = mk_multi_lambda_ty argvs rty in 
        opt_lambdas false cenv env topValInfo expr ty
    | TExpr_tlambda(lambda_id,tps,body,m,rty,_)  -> 
        let topValInfo = TopValInfo (length tps,[],TopValData.unnamedRetVal) in
        let ty = try_mk_forall_ty tps rty in 
        opt_lambdas false cenv env topValInfo expr ty
    | TExpr_tchoose _  -> opt_expr cenv env (Typrelns.choose_typar_solutions_for_tchoose cenv.g cenv.amap expr)
    | TExpr_match(exprm,dtree,targets,m,ty,_) -> opt_match cenv env (exprm,dtree,targets,m,ty)
    | TExpr_letrec (binds,e,m,_) ->  opt_letrec cenv env (binds,e,m)
    | TExpr_static_optimization (constraints,e2,e3,m) ->
        let e2',e2info = opt_expr cenv env e2 in      
        let e3',e3info = opt_expr cenv env e3 in      
        TExpr_static_optimization(constraints,e2',e3',m), 
        { tsize = min e2info.tsize e3info.tsize;
          fsize = min e2info.fsize e3info.fsize;
          effect = e2info.effect || e3info.effect;
          ivalue= UnknownValue }
    | TExpr_link eref -> failwith "unexpected reclink"


(*-------------------------------------------------------------------------
!* Optimize/analyze an object expression
 *------------------------------------------------------------------------- *)

and opt_obj_expr cenv env (typ,basevopt,basecall,overrides,iimpls,m) =
    if verbose then dprintf0 "opt_obj_expr\n";
    let basecall',basecallinfo = opt_expr cenv env basecall in
    let overrides',overrideinfos = opt_methods cenv env basevopt overrides in
    let iimpls',iimplsinfos = opt_iimpls cenv env basevopt iimpls in
    let expr'=TExpr_obj(new_uniq(),typ,basevopt,basecall',overrides',iimpls',m,new_cache()) in 
    expr', { tsize=10 (* a class defn! *) + basecallinfo.tsize + add_tsizes overrideinfos + add_tsizes iimplsinfos;
             fsize=1 (* a newobj *) ;
             effect=true;
             ivalue=UnknownValue}

(*-------------------------------------------------------------------------
!* Optimize/analyze the methods that make up an object expression
 *------------------------------------------------------------------------- *)

and opt_methods cenv env basevopt l = opt_list (opt_method cenv env basevopt) l
and opt_method cenv env basevopt (TMethod(slotsig,tps,vs,e,m) as tmethod) = 
    if verbose then dprintf0 "opt_method\n";
    let env = {env with latestBoundId=Some (id_of_method tmethod)} in 
    let env = bind_tyvals_to_unknown tps env in 
    let env = bind_vspecs_to_unknown cenv vs env in 
    let env = Option.fold_right (bind_vspec_to_unknown cenv) basevopt env in   
    let e',einfo = opt_expr cenv env e in 
    (* REVIEW: if we ever change this from being UnknownValue then we should call abstractExprInfoByVars *)
    TMethod(slotsig,tps,vs,e',m),
    { tsize = einfo.tsize;
      fsize = 0;
      effect = false;
      ivalue=UnknownValue}

(*-------------------------------------------------------------------------
!* Optimize/analyze the interface implementations that form part of an object expression
 *------------------------------------------------------------------------- *)

and opt_iimpls cenv env basevopt l = opt_list (opt_iimpl cenv env basevopt) l
and opt_iimpl cenv env basevopt (ty,overrides) = 
    if verbose then dprintf0 "opt_iimpl\n";
    let overrides',overridesinfos = opt_methods cenv env basevopt overrides in
    (ty, overrides'), 
    { tsize = add_tsizes overridesinfos;
      fsize = 1;
      effect = false;
      ivalue=UnknownValue}

(*-------------------------------------------------------------------------
!* Optimize/analyze an application of an intrinsic operator to arguments
 *------------------------------------------------------------------------- *)

and opt_op cenv env (op,tyargs,args,m) =
    if verbose then dprintf0 "opt_op\n";
    (* Special cases *)
    match op,tyargs,args with 
    | TOp_coerce,[toty;fromty],[e] -> 
        let e',einfo = opt_expr cenv env e in      
        if type_equiv cenv.g toty fromty then e',einfo 
        else 
          mk_coerce(e',toty,m,fromty), 
          { tsize=einfo.tsize + 1;
            fsize=einfo.fsize + 1;
            effect = true;  
            ivalue=UnknownValue }
    (* Handle these as special cases since mutables are allowed inside their bodies *)
    | TOp_while,_,[TExpr_lambda(_,_,[_],e1,_,_,_);TExpr_lambda(_,_,[_],e2,_,_,_)]  -> opt_while cenv env (e1,e2,m) 
    | TOp_for(dir),_,[TExpr_lambda(_,_,[_],e1,_,_,_);TExpr_lambda(_,_,[_],e2,_,_,_);TExpr_lambda(_,_,[v],e3,_,_,_)]  -> opt_for cenv env (v,e1,dir,e2,e3,m) 
    | TOp_try_finally,[resty],[TExpr_lambda(_,_,[_],e1,_,_,_); TExpr_lambda(_,_,[_],e2,_,_,_)] -> opt_try_finally cenv env (e1,e2,m,resty)
    | TOp_try_catch,[resty],[TExpr_lambda(_,_,[_],e1,_,_,_); TExpr_lambda(_,_,[vf],ef,_,_,_); TExpr_lambda(_,_,[vh],eh,_,_,_)] -> opt_try_catch cenv env (e1,vf,ef,vh,eh,m,resty)
    | TOp_trait_call(ss),methTypeArgs,args -> opt_trait_call cenv env (ss,methTypeArgs, args, m) 

    | _ -> 
    (* Reductions *)
    let args',arginfos = opt_exprs_then_split_big cenv env args in
    let knownValue = 
      match op,arginfos with 
      | TOp_field_get (rf),[e1info] -> reduce_recd_field_get cenv env (e1info,rf,tyargs,m) 
      | TOp_tuple_field_get n,[e1info] -> reduce_tuple_field_get cenv env (e1info,tyargs,n,m)
      | TOp_constr_field_get (cspec,n),[e1info] -> reduce_constr_field_get cenv env (e1info,cspec,tyargs,n,m)
      | _ -> None in 
    match knownValue with 
    | Some valu -> 
      begin match try_opt_value cenv env (false,valu,m)  with 
      | Some res -> opt_expr cenv env res  (* discard e1 since guard ensures it has no effects *)
      | None -> noopt_op cenv env (op,tyargs,args',m) arginfos valu
      end
    | None -> noopt_op cenv env (op,tyargs,args',m) arginfos UnknownValue


and noopt_op cenv env (op,tyargs,args',m) arginfos valu =
    if verbose then dprintf0 "noopt_op\n";
    (* The generic case - we may collect information, but the construction/projection doesn't disappear *)
    let args_tsize = add_tsizes arginfos in 
    let args_fsize = add_fsizes arginfos in 
    let args_effect = or_effects arginfos in 
    let args_valus = List.map (fun x -> x.ivalue) arginfos in 
    let effect = op_has_effect cenv.g op in
    let cost,valu = 
      match op with
      | TOp_uconstr c              -> 2,mk_constr_value c (Array.of_list args_valus)
      | TOp_exnconstr _           -> 2,valu (* REVIEW: information collection possilbe here *)
      | TOp_tuple                 -> 1, mk_tuple_value (Array.of_list args_valus)
      | TOp_field_get _     
      | TOp_tuple_field_get _    
      | TOp_constr_field_get _   
      | TOp_exnconstr_field_get _
      | TOp_constr_tag_get _      -> 1,valu (* REVIEW: reduction possible here, and may be very effective *)
      | TOp_asm(instrs,_)         -> (let n = List.length instrs in min n 1), 
                                     mk_asm_value cenv.g instrs args_valus 
      | TOp_bytes bytes -> (Bytes.length bytes)/10 , valu
      | TOp_field_get_addr _     
      | TOp_array | TOp_for _ | TOp_while | TOp_try_catch | TOp_try_finally
      | TOp_ilcall _
      | TOp_trait_call _          
      | TOp_lval_op _    
      | TOp_field_set _
      | TOp_constr_field_set _
      | TOp_get_ref_lval 
      | TOp_coerce
      | TOp_exnconstr_field_set _ -> 1,valu
      | TOp_recd (ctorInfo,tcref) ->
          let finfos = instance_rfields_of_tcref tcref in
          (* REVIEW: this seems a little conservative: allocating a record with a mutable field *)
          (* is not an effect - only reading or writing the field is. *)
          let valu = 
              match ctorInfo with 
              | RecdExprIsObjInit -> UnknownValue
              | RecdExpr -> 
                   if List.length args_valus <> List.length finfos then valu 
                   else mk_recd_value tcref tyargs (Array.of_list (map2 (fun x f -> if f.rfield_mutable then UnknownValue else x) args_valus finfos)) in
          2,valu  in 
    
    let vinfo = { tsize=args_tsize + cost;
                  fsize=args_fsize + cost;
                  effect=args_effect || effect;
                  ivalue=valu } in 

    (* Replace entire expression with known value? *)
      match try_opt_vinfo cenv env m vinfo with 
      | Some res -> res,vinfo
      | None ->
            TExpr_op(op,tyargs,args',m),
            { tsize=args_tsize + cost;
              fsize=args_fsize + cost;
              effect=args_effect || effect;
              ivalue=valu }

(*-------------------------------------------------------------------------
!* Optimize/analyze a constant node
 *------------------------------------------------------------------------- *)

              
and opt_const cenv env expr (c,m,ty) = 
    match try_elim_bigint_bignum_constants cenv.g m c with 
    | Some(e) -> 
        opt_expr cenv env e
    | None ->
        if verbose then dprintf0 "opt_const\n";
        expr, { tsize=(match c with 
                       | TConst_string b -> (Bytes.length b)/10 
                       | _ -> 0);
                fsize=0;
                effect=false;
                ivalue=mk_const_value c ty}

(*-------------------------------------------------------------------------
!* Optimize/analyze a record lookup. 
 *------------------------------------------------------------------------- *)

and reduce_recd_field_get cenv env (e1info,r,tinst,m) =
  match dest_recd_value e1info.ivalue with
  | Some finfos when do_opt_recd_field_get() && not e1info.effect ->
      let n = (rfref_index r) in 
      if n >= Array.length finfos then errorR(Error( "reduce_recd_field_get: term argument out of range",m));
      Some(Array.get finfos n)   (* Uses INVARIANT on record val_infos that exprs are in defn order *)
  | _ -> None
  
and reduce_tuple_field_get cenv env (e1info,tys,n,m) =
  match dest_tuple_value e1info.ivalue with
  | Some tups when do_opt_tup_field_get() && not e1info.effect ->
      let len = Array.length tups in 
      if len <> length tys then errorR(Error("error: tuple lengths don't match",m));
      if n >= len then errorR(Error("reduce_tuple_field_get: tuple index out of range",m));
      Some (Array.get tups n)
  | _ -> None
      
and reduce_constr_field_get cenv env (e1info,cspec,tys,n,m) =
  match dest_constr_value e1info.ivalue with
  | Some (cspec2,args) when do_opt_constr_field_get() && not e1info.effect && cenv.g.ucref_eq cspec cspec2 ->
      if n >= Array.length args then errorR(Error( "reduce_constr_field_get: term argument out of range",m));
      Some (Array.get args n)
  | _ -> None


(*-------------------------------------------------------------------------
!* Optimize/analyze a for-loop
 *------------------------------------------------------------------------- *)

and opt_for cenv env (v,e1,dir,e2,e3,m) =
  if verbose then dprintf0 "opt_for\n";
  let e1',e1info = opt_expr cenv env e1 in 
  let e2',e2info = opt_expr cenv env e2 in
  let env = bind_vspec_to_unknown cenv v env in 
  let e3', e3info = opt_expr cenv env e3 in 
  let einfos = [e1info;e2info;e3info] in
  let eff = or_effects einfos in 
  (* neither bounds nor body has an effect, and loops always terminate, hence eliminate the loop *)
  if not eff then 
      mk_unit cenv.g m , { tsize=0; fsize=0; effect=false;ivalue=UnknownValue }
  else
      let expr' = mk_for cenv.g (v,e1',dir,e2',e3',m) in 
      expr', { tsize=add_tsizes einfos + 3;
               fsize=add_fsizes einfos + 3;
               effect=eff;
               ivalue=UnknownValue }

(*-------------------------------------------------------------------------
!* Optimize/analyze a set of recursive bindings
 *------------------------------------------------------------------------- *)

and opt_letrec cenv env (binds,bodyExpr,m) =
  if verbose then dprintf0 "opt_letrec\n";
  let env = bind_vspecs_to_unknown cenv (map var_of_bind binds) env in 
  let binds',env = opt_binds cenv env binds in 
  let bodyExpr',einfo = opt_expr cenv env bodyExpr in 
  (* REVIEW: graph analysis to determine which items are unused *)
  (* Eliminate any unused bindings, as in let case *)
  let binds'',bindinfos = let fvs0 = free_in_expr bodyExpr' in
                          let fvsN = map (fst >> free_in_rhs) binds' in
                          let fvs  = fold_left union_freevars fvs0 fvsN in
                          split_is_used_or_has_effect m fvs binds'
  in
  (* Trim out any optimization info that involves escaping values *)
  let evalue' = abstractExprInfoByVars m (map var_of_bind binds,[]) einfo.ivalue in
  (* REVIEW: size of constructing new closures - should probably add #freevars + #recfixups here *)
  let bodyExpr' = TExpr_letrec(binds'',bodyExpr',m,new_cache()) in
  let info = combine_vinfos (einfo :: bindinfos) evalue' in 
  bodyExpr', info

(*-------------------------------------------------------------------------
!* Optimize/analyze a linear sequence of sequentioanl execution or 'let' bindings.
 *------------------------------------------------------------------------- *)

and opt_linear cenv env expr contf =
  if verbose then dprintf0 "opt_linear\n";
  let expr = if do_opt_expand_structural_bind() then expand_structural_bind cenv env expr else expr in
  match expr with 
  | TExpr_seq (e1,e2,flag,m) -> 
    if verbose then dprintf0 "opt_linear: seq\n";
    let e1',e1info = opt_expr cenv env e1 in      
    opt_linear cenv env e2 (contf << (fun (e2',e2info) -> 
      if flag = NormalSeq && do_opt_seq () && not e1info.effect then 
          e2', e2info
      else 
          TExpr_seq(e1',e2',flag,m),
          { tsize = e1info.tsize + e2info.tsize;
            fsize = e1info.fsize + e2info.fsize;
            effect = flag <> NormalSeq || e1info.effect || e2info.effect;
            ivalue = UnknownValue (* can't propagate value: must access result of computation for its effects *) }))

  | TExpr_let (bind,body,m,_) ->  
    if verbose then dprintf0 "opt_linear: let\n";
    let (bind',binfo),env = opt_bind cenv env bind in 
    opt_linear cenv env body (contf << (fun (body',bodyInfo) ->  
      if is_used_or_has_effect m (free_in_expr body') (bind',binfo) then
          (* Eliminate let bindings on the way back up *)
          let expr',adjust = try_elim_let cenv env  bind' body' m in
          expr',
          { tsize = binfo.tsize + bodyInfo.tsize + adjust; 
            fsize = binfo.fsize + bodyInfo.fsize + adjust; 
            effect=binfo.effect || bodyInfo.effect;
            ivalue = UnknownValue }
      else 
          (* On the way back up: Trim out any optimization info that involves escaping values on the way back up *)
          let evalue' = abstractExprInfoByVars (range_of_val (var_of_bind bind')) ([var_of_bind bind'],[]) bodyInfo.ivalue in 
          body',
          { tsize = binfo.tsize + bodyInfo.tsize - local_var_size (* eliminated a local var *); 
            fsize = binfo.fsize + bodyInfo.fsize - local_var_size (* eliminated a local var *); 
            effect=binfo.effect || bodyInfo.effect;
            ivalue = evalue' } ))

  | _ -> contf (opt_expr cenv env expr)

(*-------------------------------------------------------------------------
!* Optimize/analyze a try/finally construct.
 *------------------------------------------------------------------------- *)
  
and opt_try_finally cenv env (e1,e2,m,ty) =
  if verbose then dprintf0 "opt_try_finally\n";
  let e1',e1info = opt_expr cenv env e1 in      
  let e2',e2info = opt_expr cenv env e2 in      
  let info = 
      { tsize = e1info.tsize + e2info.tsize + 5;
        fsize = e1info.fsize + e2info.fsize + 5;
        effect = e1info.effect || e2info.effect;
        ivalue = UnknownValue } in 
  (* try-finally, so no effect means no exception can be raised, so just sequence the finally *)
  if do_opt_try () && not e1info.effect then TExpr_seq(e1',e2',ThenDoSeq,m),info else
  mk_try_finally cenv.g (e1',e2',m,ty), 
  info

(*-------------------------------------------------------------------------
!* Optimize/analyze a try/catch construct.
 *------------------------------------------------------------------------- *)
  
and opt_try_catch cenv env (e1,vf,ef,vh,eh,m,ty) =
  if verbose then dprintf0 "opt_try_catch\n";
  let e1',e1info = opt_expr cenv env e1 in      
  (* try-catch, so no effect means no exception can be raised, so discard the catch *)
  if do_opt_try () && not e1info.effect then e1',e1info else
  let envinner = bind_vspec_to_unknown cenv vf (bind_vspec_to_unknown cenv vh env) in
  let ef',efinfo = opt_expr cenv envinner ef in      
  let eh',ehinfo = opt_expr cenv envinner eh in      
  let info = 
      { tsize = e1info.tsize + efinfo.tsize+ ehinfo.tsize  + 5;
        fsize = e1info.fsize + efinfo.fsize+ ehinfo.fsize  + 5;
        effect = e1info.effect || efinfo.effect || ehinfo.effect;
        ivalue = UnknownValue } in 
  mk_try_catch cenv.g (e1',vf,ef',vh,eh',m,ty), 
  info

(*-------------------------------------------------------------------------
!* Optimize/analyze a while loop
 *------------------------------------------------------------------------- *)
  
and opt_while cenv env  (e1,e2,m) =
  if verbose then dprintf0 "opt_while\n";
  let e1',e1info = opt_expr cenv env e1 in      
  let e2',e2info = opt_expr cenv env e2 in      
  mk_while cenv.g (e1',e2',m), 
  { tsize = e1info.tsize + e2info.tsize + 1;
    fsize = e1info.fsize + e2info.fsize + 1;
    effect = true; (* may not terminate *)
    ivalue = UnknownValue }

(*-------------------------------------------------------------------------
!* Optimize/analyze a call to a 'member' constraint. Try to resolve the call to 
 * a witness (should always be possible due to compulsory inlining of any
 * code that contains calls to member constraints, except when analyzing 
 * not-yet-inlined generic code)
 *------------------------------------------------------------------------- *)
 

and opt_trait_call cenv env   (traitInfo, methTypeArgs, args, m) =

  (* Resolve the static overloading early (during the compulsory rewrite phase) so we can inline. *)
  match Csolve.codegen_witnessThatTypSupportsTraitConstraint cenv.g cenv.amap m traitInfo with
         
  | OkResult (_,minfo) 
      (* Limitation related to bug 1281:   If we resolve to an instance method on a struct and we haven't yet taken the address of the object *)
      when not (Infos.minfo_is_struct minfo && Infos.minfo_is_instance minfo) -> 
      
      let (TSlotSig(nm1,typ1,ctps1,mtps1,ps1, rty1)) = Infos.slotsig_of_minfo cenv.g cenv.amap m minfo in
      let expr = Infos.mk_minfo_call cenv.g cenv.amap m minfo methTypeArgs args in 
      opt_expr cenv env expr

  (* resolution fails when optimizing generic code *)
  |  _ -> 
      let args',arginfos = opt_exprs_then_split_big cenv env args in
      noopt_op cenv env (TOp_trait_call(traitInfo),methTypeArgs,args,m) arginfos UnknownValue 

(*-------------------------------------------------------------------------
!* Make optimization decisions once we know the optimization information
 * for a value
 *------------------------------------------------------------------------- *)

and try_opt_value cenv env (mustinline,valInfoForVal,m) = 
    match valInfoForVal with 
    (* Inline constants immediately *)
    | ConstValue (c,ty) -> Some (TExpr_const (c,m,ty))
    | SizeValue (_,detail) -> try_opt_value cenv env (mustinline,detail,m) 
    | ValValue (v',detail) -> 
         if verbose then dprintf1 "try_opt_value, ValValue, valInfoForVal = %s\n" (showL(value_infoL valInfoForVal));
        (* Inline values bound to other values immediately *)
        (* if verbose then dprintf2 "Considering inlining value %a to value %a near %a\n" output_val_ref v output_locval_ref v' output_range m;  *)
        begin  match  try_opt_value cenv env (mustinline,detail,m) with 
          (* Prefer to inline using the more specific info if possible *)
          | Some e -> Some e
          (* If the more specific info didn't reveal an inline then use the value *)
          | None ->  Some(expr_for_vref m v')
        end
    | ConstExprValue(size,expr) ->
        if verbose then dprintf2 "Inlining constant expression value at %a\n"  output_range m;
        Some (remark_expr m (copy_expr cenv.g true expr))
    | CurriedLambdaValue (_,_,_,expr,_) when mustinline ->
        if verbose then dprintf2 "Inlining mustinline-lambda at %a\n"  output_range m;
        Some (remark_expr m (copy_expr cenv.g true expr))
    | LdlenValue v ->
        begin match try_opt_value cenv env (mustinline,v,m) with
          | None      -> None
          | Some arre -> Some (mk_ldlen cenv.g m arre)
        end
    | TupleValue _ | ConstrValue _ when mustinline -> failwith "tuple values cannot yet be marked 'inline'"
    | RecdValue _ when mustinline -> failwith "record values cannot yet be marked 'inline'"     
    | UnknownValue when mustinline -> warning(Error("a value marked as 'inline' has an unexpected value",m)); None
    | _ when mustinline -> warning(Error("a value marked as 'inline' could not be inlined",m)); None
    | _ -> None 
  
and try_opt_vinfo cenv env m vinfo = 
  if vinfo.effect then None else try_opt_value cenv env (false,vinfo.ivalue ,m)

(*-------------------------------------------------------------------------
!* Add 'v1 = v2' information into the information stored about a value
 *------------------------------------------------------------------------- *)
  
and addValEqualityInfo g m v info =
  match mutability_of_vref v with
  | Immutable -> {info with ivalue= mk_val_value g m v info.ivalue}
  | Mutable -> info  (* the env assumes known-values do not change *)

(*-------------------------------------------------------------------------
!* Optimize/analyze a use of a value
 *------------------------------------------------------------------------- *)

and opt_val cenv env expr (v,m) =
  let valInfoForVal = lookup_vref cenv env m v in 

  if verbose then dprintf2 "opt_val, v = %s, valInfoForVal = %s\n" (showL(vrefL v)) (showL(value_infoL valInfoForVal)) ; 

  let mustinline = mustinline(inlineFlag_of_vref v) in
  match try_opt_value cenv env (mustinline,valInfoForVal ,m) with
  | Some e -> 
     if verbose then dprintf0 "opt_val: optimized\n";
     let e,einfo = opt_expr cenv env e in (* reoptimize *)
     e,addValEqualityInfo cenv.g m v einfo 

  | None -> 
     if verbose then dprintf0 "opt_val: not optimized\n";
     if mustinline then errorR(Error("error: failed to inline the value "^(name_of_val (deref_val v))^" marked 'inline'",m));
     expr,(addValEqualityInfo cenv.g m v { ivalue=valInfoForVal; effect=false; fsize=1; tsize=1})

(*-------------------------------------------------------------------------
!* Attempt to replace an application of a value by an alternative value.
 *------------------------------------------------------------------------- *)

and strip_to_nominal_tcref cenv ty = 
      if is_stripped_tyapp_typ ty then dest_stripped_tyapp_typ ty 
      else if is_tuple_ty ty then 
        let tyargs = dest_tuple_typ ty in 
        compiled_tuple_tcref cenv.g tyargs, tyargs 
      else failwith "strip_to_nominal_tcref: unreachable" 
      

and can_devirtualize_app cenv v vref ty = 
       cenv.g.vref_eq v vref
       && not (is_unit_typ cenv.g ty)
       && (is_stripped_tyapp_typ ty || (is_tuple_ty ty && List.length (dest_tuple_typ ty) < maxTuple)) 
      (* Exclusion: Some unions have null as representations *)  
      (* Exclusion: for struct types this currently results in an incorrect call because we do not take the address of the argument *)
       && not (isUnionThatUsesNullAsRepresentation cenv.g (deref_tycon (fst(strip_to_nominal_tcref cenv ty))))  
       && not (is_struct_typ ty) 

and devirtualize_app cenv env vref tyargs args m =
 opt_expr cenv env (beta_mk_appl cenv.g nng (expr_for_vref m vref,type_of_vref vref,(if tyargs=[] then [] else [tyargs]),args,m))
 
and try_devirtualize_app cenv env (f,tyargs,args,m) =
  match f,tyargs,args with 

  (* Optimize/analyze calls to LanguagePrimitives.HashCompare.GenericComparisonIntrinsic when type is known *)
  (* to be augmented with a visible comparison value. *)
  | TExpr_val(v,_,_),[ty],_ when can_devirtualize_app cenv v cenv.g.poly_compare_inner_vref ty ->
       
      let tcref,tyargs = strip_to_nominal_tcref cenv ty in
      begin match (tcaug_of_tcref tcref).tcaug_compare with 
      | Some vref  -> Some (devirtualize_app cenv env vref tyargs args m)
      | _ -> None
      end
      
  (* Optimize/analyze calls to LanguagePrimitives.HashCompare.GenericEqualityIntrinsic when type is known *)
  (* to be augmented with a visible comparison value. *)
  | TExpr_val(v,_,_),[ty],_ when can_devirtualize_app cenv v cenv.g.poly_equals_inner_vref ty ->
       
      let tcref,tyargs = strip_to_nominal_tcref cenv ty in
      begin match (tcaug_of_tcref tcref).tcaug_equals with 
      | Some (_,vref)  -> Some (devirtualize_app cenv env vref tyargs args m)
      | _ -> None
      end
      
  (* Optimize/analyze calls to LanguagePrimitives.HashCompare.GenericHashParamIntrinsic when type is known *)
  (* to be augmented with a visible comparison value. *)
  | TExpr_val(v,_,_),[ty],_ when can_devirtualize_app cenv v cenv.g.poly_hash_param_inner_vref ty ->

      let tcref,tyargs = strip_to_nominal_tcref cenv ty in
      begin match (tcaug_of_tcref tcref).tcaug_structural_hash with 
      (* Exclusion: Some unions have null as representations *)  
      (* Exclusion: for struct types this currently results in an incorrect call because we do not take the address of the argument *)
      | Some vref -> Some (devirtualize_app cenv env vref tyargs args m)
      | _ -> None
      end

  (* Calls to LanguagePrimitives.IntrinsicFunctions.UnboxGeneric can be optimized to calls to UnboxFast when we know that the *)
  (* target type isn't 'NullNotLiked', i.e. that the target type is not an F# union, record etc. *)
  (* Note UnboxFast is just the .NET IL 'unbox.any' instruction. *)
  | TExpr_val(v,_,_),[ty],_ when cenv.g.vref_eq v cenv.g.unbox_vref && 
                                 can_use_unbox_fast cenv.g ty ->

      Some(devirtualize_app cenv env cenv.g.unbox_fast_vref tyargs args m)
      
  (* Calls to LanguagePrimitives.IntrinsicFunctions.TypeTestGeneric can be optimized to calls to TypeTestFast when we know that the *)
  (* target type isn't 'NullNotTrueValue', i.e. that the target type is not an F# union, record etc. *)
  (* Note TypeTestFast is just the .NET IL 'isinst' instruction followed by a non-null comparison *)
  | TExpr_val(v,_,_),[ty],_ when cenv.g.vref_eq v cenv.g.istype_vref && 
                                 can_use_istype_fast cenv.g ty ->

      Some(devirtualize_app cenv env cenv.g.istype_fast_vref tyargs args m)
      
  | _ -> None

(*-------------------------------------------------------------------------
!* Attempt to inline an application of a known value
 *------------------------------------------------------------------------- *)

and try_inline_app cenv env (f0',finfo) (tyargs,args,m) =
  (* Inline functions at application sites *)
  if verbose then dprintf2 "Considering inlining app near %a\n"  output_range m; 
  match dest_lambda_value finfo.ivalue with 
  | Some (lambda_id,arities,size,f2,f2ty) when

    (
    if verbose then dprintf4 "Considering inlining lambda near %a, size = %d, finfo.effect = %b\n"  output_range m size finfo.effect;
    cenv.optimizing &&
    do_inline_lambdas () &&
    not finfo.effect &&
    (* Don't inline recursively! *)
    not (Zset.mem lambda_id env.dont_inline) &&
     (  if verbose then dprintf3 "Recursion ok, #tyargs = %d, #args = %d, #arities=%d\n" (length tyargs) (length args) arities;
    (* Check the number of argument groups is enough to saturate the lambdas of the target. *)
    (if tyargs = [] then 0 else 1) + length args >= arities &&
    (if verbose then dprint_endline "Enough args"; 
    (if size > !lambda_inline_threshold + length args then
      begin
        if verbose then dprintf3 "Not inlining lambda near %a because size = %d\n" output_range m size; 
        false
      end
     else true)))) ->

       if verbose then dprintf2 "Inlining lambda near %a\n"  output_range m;
(* ----------       Printf.printf "Inlining lambda near %a = %s\n"  output_range m (showL (exprL f2));  (* JAMES: *) ----------*)
         (* REVIEW: this is a cheapshot way of optimizing the arg. *)
         (* expressions as well without the restriction of recursive  *)
         (* inlining kicking into effect *)
      let f2' = copy_expr cenv.g true f2 in
      if verbose then dprintf0 "--- try_inline_app, optimizing arguments\n";
      let args' = map (fun e -> let e',einfo = opt_expr cenv env e in e') args in
             (* Beta reduce. beta_mk_appl cenv.g does all the hard work. *)
      if verbose then dprintf0 "--- try_inline_app, beta reducing \n";
      let expr' = beta_mk_appl cenv.g nng (f2',f2ty,[tyargs],args',m) in 
      if verbose then dprintf0 "--- try_inline_app, reoptimizing\n";
      Some (opt_expr cenv {env with dont_inline= Zset.add lambda_id env.dont_inline} expr')
        
  | _ -> None

(*-------------------------------------------------------------------------
!* Optimize/analyze an application of a function to type and term arguments
 *------------------------------------------------------------------------- *)

and opt_app cenv env (f0,f0ty,tyargs,args,m) =
  if verbose then dprintf0 "--> opt_app\n";
  let f0',finfo = opt_expr cenv env f0 in
  if verbose then dprintf0 "--- opt_app, trying to devirtualize\n";
  match try_devirtualize_app cenv env (f0,tyargs,args,m) with 
  | Some res -> 
      if verbose then dprintf0 "<-- opt_app, devirtualized\n";
      res
  | None -> 

  match try_inline_app cenv env (f0',finfo) (tyargs,args,m) with 
  | Some res -> 
      if verbose then dprintf0 "<-- opt_app, inlined\n";
      res
  | None -> 

  let shapes = 
      match f0' with 
      | TExpr_val(vref,_,_) when isSome(arity_of_vref vref) -> 
          let (TopValInfo(ntps,detupArgsL,_)) = the(arity_of_vref vref) in
          let nargs = (length args) in 
          let nDetupArgsL = length detupArgsL in 
          let nShapes = min nargs nDetupArgsL in
          let detupArgsShapesL = 
            front  nShapes detupArgsL |> map (fun detupArgs -> 
              match detupArgs with 
              | [] | [_] -> UnknownValue
              | _ -> TupleValue(Array.of_list (map (fun _ -> UnknownValue) detupArgs))) in
          detupArgsShapesL @ replicate (nargs - nShapes) UnknownValue
          
      | _ -> args |> map (fun _ -> UnknownValue) in

  let args',arginfos = opt_exprs_reshape_and_consider_splits cenv env (combine shapes args) in
  if verbose then dprintf0 "<-- opt_app, beta reducing\n";
  let expr' = beta_mk_appl cenv.g nng (f0',f0ty, [tyargs],args',m) in 
  match f0' with 
  | TExpr_lambda _ | TExpr_tlambda _ -> 
     (* we beta-reduced, hence reoptimize *)
      if verbose then dprintf0 "<-- opt_app, beta reduced\n";
     opt_expr cenv env expr'
  | _ -> 
    if verbose then dprintf0 "<-- opt_app, regular\n";
    expr', { tsize=finfo.tsize + add_tsizes arginfos;
             fsize=finfo.fsize + add_fsizes arginfos;
             effect=true;
             ivalue=ivalue_of_expr expr' }

(*-------------------------------------------------------------------------
!* Optimize/analyze a lambda expression
 *------------------------------------------------------------------------- *)
        
and opt_lambdas isMeth cenv env topValInfo e ety = 
  if verbose then dprintf3 "opt_lambdas, #argsl = %d, %a\n" (TopValData.numCurriedArgs topValInfo) output_range (range_of_expr e) ;
  match e with 
  | TExpr_lambda (lambda_id,_,_,_,m,_,_)  
  | TExpr_tlambda(lambda_id,_,_,m,_,_) ->
      let tps,basevopt,vsl,body,bodyty = iterated_adjust_arity_of_lambda_body cenv.g cenv.amap nng topValInfo e in
(*      let tps,basevopt,vsl,body,bodyty = dest_top_lambda_upto cenv.g topValInfo (e, ety) in *)
      let env = Option.fold_right (bind_vspec_to_unknown cenv) basevopt env in   
      let env = bind_tyvals_to_unknown tps env in 
      let env = List.fold_right (bind_vspecs_to_unknown cenv) vsl env in
      let env = bind_vspecs_to_unknown cenv (Option.to_list basevopt) env in   
      let body',bodyinfo = opt_expr cenv env body in 
      let expr' = mk_basev_multi_lambdas m tps basevopt vsl (body',bodyty) in 
      let arities = length vsl in 
      let arities = if tps = [] then arities else 1+arities in 
      let bsize = bodyinfo.tsize in 
      if verbose then dprintf3 "lambda @ %a, bsize = %d\n" output_range m bsize;
      (* can't inline any values with semi-recursive object references to self or base *)
      let valu =   
        match basevopt with 
        | None -> CurriedLambdaValue (lambda_id,arities,bsize,expr',ety) 
        | _ -> UnknownValue in 

      expr', { tsize=bsize + (if isMeth then 1 else 10); (* estimate size of new syntactic closure - expensive, in contrast to a method *)
               fsize=1; 
               effect=false;
               ivalue= valu; }
  | _ -> opt_expr cenv env e 
      


(*-------------------------------------------------------------------------
!* Recursive calls that first try to make an expression "fit" the a shape
 * where it is about to be consumed.
 *------------------------------------------------------------------------- *)

and opt_exprs_reshape_and_consider_splits cenv env exprs = match exprs with [] -> no_exprs | _ -> opt_list (opt_expr_reshape_then_consider_split cenv env) exprs
and opt_exprs_then_split_big cenv env exprs = match exprs with [] -> no_exprs | _ -> opt_list (opt_expr_then_consider_split cenv env) exprs
and opt_targets cenv env m ty targets = opt_list (opt_target cenv env m ty) (Array.to_list targets)

and opt_expr_reshape_then_consider_split cenv env (shape,e) = 
    opt_expr_then_consider_split cenv env (reshape_expr cenv (shape,e))

and reshape_expr cenv (shape,e) = 
  match shape,e with 
  | TupleValue(subshapes), TExpr_val(vref,vFlags,m) ->
      let tinst = dest_tuple_typ (type_of_expr cenv.g e) in
      mk_tupled cenv.g m (list_mapi (fun i subshape -> reshape_expr cenv (subshape,mk_tuple_field_get(e,tinst,i,m))) (Array.to_list subshapes)) tinst
  | _ ->  
      e

and opt_expr_then_consider_split cenv env e = 
  let e',einfo = opt_expr cenv env e in 
  (* ALWAYS consider splits for enormous sub terms here - otherwise we will create invalid .NET programs  *)
  consider_split_to_method true !very_very_big_expr_size cenv env (e',einfo) 

(*-------------------------------------------------------------------------
!* Decide whether to split a sub-expression into a new method
 *------------------------------------------------------------------------- *)

and consider_split_to_method flag threshold cenv env (e,einfo) = 
    if not flag
       || !Msilxlib.tailcall_implementation = Ilxsettings.NoTailcalls (* don't mess with taking guaranteed tailcalls if used with --no-tailcalls! *)
       || (einfo.fsize < threshold) then e,einfo else 
    let fvs = free_in_expr e in  
    let ok = 
      (* We can only split an expression out as a method if certain conditions are met. *)
      (* It can't use any protected or base calls *)
      not fvs.uses_method_local_constructs &&
      fvs.free_locvals |> Zset.for_all (fun v -> 
          isSome(arity_of_val (deref_local_val v)) ||
              (* All the free variables (apart from things with an arity, i.e. compiled as methods) should be normal, i.e. not base/this etc. *)
              (base_of_lvref v = NormalVal && 
              (* None of them should be byrefs *)
               not (is_byref_ty cenv.g (type_of_lvref v)) && 
              (* None of them should be local polymorphic constrained values *)
               not (is_poly_constrained_val (deref_local_val v)) &&
              (* None of them should be mutable *)
               mutability_of_lvref v = Immutable))  in 
    if not ok then e,einfo else 
    let m = (range_of_expr e) in 
    let uv,ue = mk_compgen_local m "dummy" cenv.g.unit_ty in 
    let ty = type_of_expr cenv.g e in
    let fv,fe = mk_compgen_local m (match env.latestBoundId with Some id -> id.idText^continuedName | None -> continuedName) (cenv.g.unit_ty --> ty) in 
    mk_let m fv (mk_lambda m uv (e,ty)) 
      (prim_mk_app (fe,(cenv.g.unit_ty --> ty)) [] [mk_unit cenv.g m] m),
    {einfo with fsize=1 }

(*-------------------------------------------------------------------------
!* Optimize/analyze a pattern matching expression
 *------------------------------------------------------------------------- *)

and opt_match cenv env (exprm,dtree,targets,m, ty) =
  if verbose then dprintf0 "opt_match\n";
  (* REVIEW: collect, merge and use information flowing through each line of the decision tree to each target *)
  let dtree',minfo = opt_dtree cenv env dtree in
  let targets',tinfos = opt_targets cenv env m ty targets in
  let expr' = mk_and_optimize_match exprm m ty dtree' targets' in 
  expr', combine_vinfos_unknown (minfo :: tinfos)

(*-------------------------------------------------------------------------
!* Optimize/analyze a target of a decision tree
 *------------------------------------------------------------------------- *)

and opt_target cenv env m ty (TTarget(vs,e)) = 
  if verbose then dprintf0 "opt_target\n";
  (* REVIEW: this is where we should be using information collected for each target *)
  let env = bind_vspecs_to_unknown cenv vs env in 
  let e',einfo = opt_expr cenv env e in 
  let e',einfo = consider_split_to_method !abstract_big_targets !big_target_size cenv env (e',einfo) in
  let evalue' = abstractExprInfoByVars m (vs,[]) einfo.ivalue in 
  TTarget(vs,e'),
  { tsize=einfo.tsize; 
    fsize=einfo.fsize;
    effect=einfo.effect;
    ivalue=evalue' }

(*-------------------------------------------------------------------------
!* Optimize/analyze a decision tree
 *------------------------------------------------------------------------- *)

and opt_dtree cenv env x =
  match x with 
  | TDSuccess (es,n) -> 
      let es', einfos = opt_exprs_then_split_big cenv env es in 
      TDSuccess(es',n),combine_vinfos_unknown einfos
  | TDBind(bind,rest) -> 
      let (bind,binfo),envinner = opt_bind cenv env bind in 
      let rest,rinfo = opt_dtree cenv envinner rest in 
      TDBind(bind,rest),combine_vinfos_unknown [rinfo;binfo]
  | TDSwitch (e,cases,dflt,m) -> opt_switch cenv env (e,cases,dflt,m)

and dest_discrim_val vinfo = 
  match dest_constr_value vinfo with
  | Some (c,a)  -> Some(TTest_unionconstr(c,[])) (* note instantiation doesn't matter, as only comapring tags, and probably shouldn't be in this node *)
  | None -> 
  match dest_const_value vinfo with
  | Some (c)  -> Some(TTest_const(c))
  | None -> None

(*-------------------------------------------------------------------------
!* Optimize/analyze a switch construct from pattern matching 
 *------------------------------------------------------------------------- *)
  
and opt_switch cenv env (e,cases,dflt,m) =
  if verbose then dprintf0 "opt_switch\n";
  let e', einfo = opt_expr cenv env e in 

  match dest_discrim_val einfo.ivalue with
  | Some (d) when 
        do_opt_switch() && 
        einfo.effect = false  ->
      let case = 
        match (tryfind (function (TCase(d2,_)) when discrim_eq cenv.g d d2 -> true | _ -> false) cases) with 
        | Some(TCase(_,case)) -> case
        | _ -> 
        match dflt with 
        | Some case -> case  
        | None -> failwith "error during optimization: could not compute result of pattern match even though we knew the constructor" in 
      opt_dtree cenv env case
  | _ -> 
  match cases,dflt with 
  | [],Some case -> opt_dtree cenv env case
  | _ ->
      noopt_switch_cases cenv env (e', einfo, cases,dflt,m)

and noopt_switch_cases cenv env (e', einfo, cases,dflt,m) =
  let cases',cinfos = split (map (fun (TCase(discrim,e)) -> let e',einfo = opt_dtree cenv env e in TCase(discrim,e'),einfo) cases) in 
  let dflt',dinfos = match dflt with None -> None,[] | Some df -> let df',einfo = opt_dtree cenv env df in Some df',[einfo] in 
  let size = (length dinfos + length cinfos) * 4 in 
  TDSwitch (e',cases',dflt',m),combine_vinfos_unknown (einfo :: cinfos @ dinfos)

and opt_bind cenv env (TBind(v,e) as bind) =
  if verbose then dprintf0 "opt_bind\n";
  
  let repr',einfo = 
    let env = if compgen_of_val v && isSome env.latestBoundId then env else {env with latestBoundId=Some (id_of_val v)} in 
    let cenv = if inlineFlag_of_val v = PseudoValue then { cenv with optimizing=false} else cenv in 
    if verbose then dprintf0 "opt_bind --> opt_lambdas\n";
    let e',einfo = opt_lambdas (isSome (member_info_of_val v)) cenv env (infer_arity_of_expr_bind v e) e (type_of_val v) in
    let size = local_var_size in 
    e',{einfo with fsize=einfo.fsize+size; tsize = einfo.tsize+size} in
  (* Trim out optimization information for large lambdas we'll never inline *)
  (* Trim out optimization information for expressions that call protected members *)    
  let rec cut ivalue = 
    match ivalue with
    | CurriedLambdaValue (_, arities, size, body,_) -> 
        if size > (!lambda_inline_threshold + arities + 2) then (
          if verbose then dprintf4 "Discarding lambda for binding %s, size = %d, m = %a\n"  (name_of_val v) size output_range (range_of_expr body);
          UnknownValue (* trim large *)
        ) else
          let fvs = free_in_expr body in
          if fvs.uses_method_local_constructs then (
            if verbose then dprintf3 "Discarding lambda for binding %s because uses protected members, m = %a\n"  (name_of_val v) output_range (range_of_expr body);
            UnknownValue (* trim protected *)
          ) else
            ivalue
    | ValValue(v,x) -> ValValue(v,cut x)
    | ModuleValue _ -> UnknownValue
    | TupleValue a -> TupleValue(Array.map cut a)
    | RecdValue (tcref,a) -> RecdValue(tcref,Array.map cut a)       
    | ConstrValue (a,b) -> ConstrValue (a,Array.map cut b)
    | DecrValue a -> DecrValue (cut a)
    | LdlenValue a -> LdlenValue (cut a)        
    | UnknownValue | ConstValue _  | ConstExprValue _ -> ivalue
    | SizeValue(_,a) -> mk_size_value (cut a) in
  let mustinl = mustinline(inlineFlag_of_val v) in 
  let einfo = if mustinl then einfo else {einfo with ivalue = cut einfo.ivalue } in 
  let einfo = 
    if (not(mustinl) && not (keep_opt_values())) or
       (inlineFlag_of_val v = NeverInline) or
       (* These values are given a special going-over by the optimizer and *)
       (* ilxgen.ml, hence treat them as if no-inline *)
       (let nvref = mk_local_vref v in 
        cenv.g.vref_eq nvref cenv.g.poly_eq_inner_vref or
        cenv.g.vref_eq nvref cenv.g.poly_compare_inner_vref or
        cenv.g.vref_eq nvref cenv.g.poly_equals_inner_vref or
        cenv.g.vref_eq nvref cenv.g.poly_hash_param_inner_vref)
    then {einfo with ivalue=UnknownValue} 
    else einfo in 
  if mustinl && partialExprVal einfo.ivalue then 
    errorR(InternalError("the mustinline value '"^name_of_val v^"' was not inferred to have a known value",range_of_val v));
  if verbose then dprintf2 "val %s gets opt info %s\n" (showL(vspecL v)) (showL(value_infoL einfo.ivalue));
  let env = bind_vspec cenv v einfo.ivalue env in 
  (TBind(v,repr'), einfo), env
      
and opt_binds cenv env xs = map_acc_list (opt_bind cenv) env xs
    
and opt_mexpr cenv env x = 
    match x with   
    | TMTyped(mty,def,m) -> 
        let (def,info),(env,bindInfosColl) = opt_mdef cenv (env,[]) def  in
        let bindInfosColl = concat bindInfosColl in 
        
        (* The hidden set here must contain NOT MORE THAN the set of values made inaccessible by the application of the signature *)
         
        let (renaming, hidden) as rpi = mk_mdef_to_mtyp_remapping def mty in
        let def = 
            if localopt() then 
                let fvs = free_in_mdef def in
                let dead = bindInfosColl |> filter (fun (bind,binfo) -> not (is_used_or_has_effect m fvs (bind,binfo)) && Zset.mem (var_of_bind bind) hidden.mhiVals) in 
                if verbose then dead |> iter (fun (bind,_) -> dprintf1 "dead, hidden, buried, gone: %s\n" (showL (vspecAtBindL (var_of_bind bind))));
                let deadSet = Zset.addL (map (fun (bind,_) -> var_of_bind bind) dead) (Zset.empty val_spec_order) in

                let rec elim_mdef x =                  
                    match x with 
                    | TMDefRec(tycons,binds,m) -> TMDefRec(tycons, (binds |> filter (var_of_bind >> Zset.mem_of deadSet >> not)),m)
                    | TMDefLet(bind,m)  -> if Zset.mem (var_of_bind bind) deadSet then TMDefRec([],[],m) else x
                    | TMDefs(defs) -> TMDefs(map elim_mdef defs) 
                    | TMAbstract _ ->  x
                    | TMDefModul(TMBind(nm, d)) -> TMDefModul(TMBind(nm,elim_mdef d)) in 
                elim_mdef def
            else def in 
        let info = abstractAndRemapModulInfo "defs" cenv.g m rpi info in

        TMTyped(mty,def,m),info 

and mk_var_bind bind info =
    let v = var_of_bind bind in 
    (name_of_val v, (mk_local_ref v, info))

and opt_mdef cenv (env,bindInfosColl) x = 
    match x with 
    | TMDefRec(tycons,binds,m) -> 
        let env = bind_vspecs_to_unknown cenv (map var_of_bind binds) env in 
        let bindInfos,env = opt_binds cenv env binds in 
        let binds', binfos = split bindInfos in
          (* REVIEW: Eliminate let bindings on the way back up *)
        (TMDefRec(tycons,binds',m),
         { val_infos=Namemap.of_list (map2 mk_var_bind binds (map (fun v -> v.ivalue) binfos)); 
           modul_infos = Namemap.of_list []}),
         (env,(bindInfos::bindInfosColl))
    | TMAbstract(mexpr) -> 
        let mexpr,info = opt_mexpr cenv env mexpr in
        let env = bind_module_vspecs cenv info env in 
        (TMAbstract(mexpr),info),(env,bindInfosColl)
    | TMDefLet(bind,m)  ->
        let ((bind',binfo) as bindInfo),env = opt_bind cenv env bind in 
          (* REVIEW: Eliminate unused let bindings from modules *)
        (TMDefLet(bind',m),
         { val_infos=Namemap.of_list [mk_var_bind bind binfo.ivalue]; 
           modul_infos = Namemap.of_list []}),
        (env ,([bindInfo]::bindInfosColl))

    | TMDefModul(TMBind(tycon, def)) -> 
        let id = id_of_modul tycon in 
        let (def,info),(_,bindInfosColl) = opt_mdef cenv (env,bindInfosColl) def  in
        let env = bind_module_vspecs cenv info env in 
        (TMDefModul(TMBind(tycon,def)),
         { val_infos=Namemap.empty; 
           modul_infos = Namemap.of_list [(id.idText, info)]}), 
        (env,bindInfosColl)

    | TMDefs(defs) -> 
        let (defs,info),(env,bindInfosColl) = opt_mdefs cenv (env,bindInfosColl) defs  in
        (TMDefs(defs), info), (env,bindInfosColl)

and opt_mdefs cenv (env,bindInfosColl) defs = 
    if verbose then dprintf0 "opt_mdefs\n";
    let defs,(env,bindInfosColl) = map_acc_list (opt_mdef cenv) (env,bindInfosColl) defs in
    let defs,minfos = split defs in 
    (defs,modul_info_Union minfos),(env,bindInfosColl)
   
and opt_assembly cenv env isIncrementalFragment (TAssembly(mimpls)) =
    (* optimize a sequence of modul bindings left-to-right *)
    let env,mimpls',mvinfos = 
        fold_left (fun (env,l1,l2) (TImplFile(qname, (TMTyped(mty,_,m) as mexpr))) -> 
              let env,mexpr',minfo  = 
                  match mexpr with 
                  (* FSI: FSI compiles everything as if you're typing incrementally into one module *)
                  (* This means the fragment is not truly a constrained module as later fragments will be typechecked *)
                  (* against the internals of the module rather than the externals. Furthermore it would be wrong to apply *)
                  (* optimizations that did lots of reorganizing stuff to the internals of a module should we ever implement that. *)
                  | TMTyped(mty,def,m) when isIncrementalFragment -> 
                      let (def,minfo),(env,bindInfosColl) = opt_mdef cenv (env,[]) def  in
                      env, TMTyped(mty, def,m), minfo
                  |  _ -> 
                      let mexpr', minfo = opt_mexpr cenv env mexpr in 
                      let env = bind_module_vspecs cenv minfo env in 
                      env, mexpr', minfo in

              let hidden = mk_assembly_boundary_mhi mty in
              let minfo = abstractModulInfoByHiding m hidden minfo in 
              env, l1 @ [(TImplFile(qname,mexpr'))], l2 @ [minfo])

          (env,[],[])
          mimpls in 
    (* collect up the results for the public TopImpls *)
    let info = modul_info_Union mvinfos in 
    
    env,(TAssembly(mimpls')), info

(*-------------------------------------------------------------------------
!* Entry point
 *------------------------------------------------------------------------- *)

let optimize_assembly cenv optEnv isIncrementalFragment mimpls =
  let show = (* true || *) verbose in
(*  let statsA = if show then exprStats mimpls else "" in *)
  opt_assembly cenv optEnv isIncrementalFragment mimpls 
(*  let statsB = if show then exprStats mimpls else "" in
  if show then (dprintf1 "Stats before %s\n" statsA;
                dprintf1 "Stats after  %s\n" statsB);
*)  

(*-------------------------------------------------------------------------
!* Pickle to stable format for cross-module optimization data
 *------------------------------------------------------------------------- *)

open Pickle

let rec pvalue_info x st =
  match x with 
  | ConstValue (c,ty)   -> p_byte 0 st; p_tup2 p_const p_typ (c,ty) st 
  | UnknownValue   -> p_byte 1 st
  | ValValue (a,b) -> p_byte 2 st; p_tup2 (p_vref "optval") pvalue_info (a,b) st
  | ModuleValue a  -> p_byte 3 st; psubmodul_info a st
  | TupleValue a   -> p_byte 4 st; (p_array pvalue_info) a st
  | ConstrValue (a,b) -> p_byte 5 st; p_tup2 p_ucref (p_array pvalue_info) (a,b) st
  | CurriedLambdaValue (a,b,c,d,e) -> p_byte 6 st; p_tup5 p_int p_int p_int p_expr p_typ (a,b,c,d,e) st
  | ConstExprValue (a,b) -> p_byte 7 st; p_tup2 p_int p_expr (a,b) st
  | LdlenValue a         -> p_byte 8 st; pvalue_info a st
  | DecrValue a          -> p_byte 9 st; pvalue_info a st      
  | RecdValue (tcref,a)  -> p_byte 10 st; p_tup2 (p_tcref "opt data") (p_array pvalue_info) (tcref,a) st
  | SizeValue (adepth,a) -> pvalue_info a st

and psubmodul_info x st = p_tup2 (p_namemap (p_tup2 (p_vref "opttab") pvalue_info)) (p_namemap psubmodul_info) (x.val_infos, x.modul_infos) st
and pmodul_info info st = psubmodul_info info st

let rec uvalue_info st =
  let rec uvalue_info st =
    let tag = u_byte st in match tag with
    | 0 -> u_tup2 u_const u_typ       st |> (fun (c,ty) -> ConstValue(c,ty))
    | 1 -> u_void                   st |> (fun () -> UnknownValue)
    | 2 -> u_tup2 u_vref uvalue_info st |> (fun (a,b) -> ValValue (a,b))
    | 3 -> usubmodul_info          st |> (fun a -> ModuleValue a)
    | 4 -> u_array uvalue_info       st |> (fun a -> TupleValue a)
    | 5 -> u_tup2 u_ucref (u_array uvalue_info)  st |> (fun (a,b) -> ConstrValue (a,b))
    | 6 -> u_tup5 u_int u_int u_int u_expr u_typ st |> (fun (a,b,c,d,e) -> CurriedLambdaValue (a,b,c,d,e))
    | 7 -> u_tup2 u_int u_expr        st |> (fun (a,b) -> ConstExprValue (a,b))
    | 8 -> uvalue_info             st |> (fun a -> LdlenValue a)
    | 9 -> uvalue_info             st |> (fun a -> DecrValue a)
    | 10 -> u_tup2 u_tcref (u_array uvalue_info)      st |> (fun (a,b) -> RecdValue (a,b))
    | _ -> failwith "uvalue_info" in
  mk_size_value (uvalue_info st) (* calc size of unpicked value_info *)

and usubmodul_info st = 
  let a,b = u_tup2 (u_namemap (u_tup2 u_vref uvalue_info)) (u_namemap usubmodul_info) st in 
  { val_infos=a; modul_infos=b}
and umodul_info st = usubmodul_info st
