(* CONTENTS-INDEX-REGEXP = FROM>^\!\* <TO *)
(*----------------------------------------------------------------------------
CONTENTS-START-LINE: HERE=2 SEP=2
CONTENTS-END-LINE:
----------------------------------------------------------------------------*)

(* (c) Microsoft Corporation. All rights reserved *)

(*--------------------------------------------------------------------------
!* The ILX generator. 
 *
 * NOTE: unit args to a Method (occuring before "arity" position) are suppressed.
     Search for UNIT-ARGS.
 * NOTE: unit have NULL storage (no point storing units).
 *-------------------------------------------------------------------------- *)

(*F# 
module Microsoft.FSharp.Compiler.Ilxgen
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.Research.AbstractIL.Extensions.ILX
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilx = Microsoft.Research.AbstractIL.Extensions.ILX.Types 
module Ilmorph = Microsoft.Research.AbstractIL.Morphs 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
module Il = Microsoft.Research.AbstractIL.IL 
module Illib = Microsoft.Research.AbstractIL.Internal.Library
F#*) 
open Ildiag
open List
open Il
open Ilx
open Nums
open Tast
open Tastops
open Range
open Ast
open Env
open Layout
open Lib
open Illib
open Printf
open Typrelns

let verbose = false

(*--------------------------------------------------------------------------
!* misc
 *-------------------------------------------------------------------------- *)

let cga g = (if g.typeCheckerConfiguredToAssumeErasureOfGenerics then [] else [ mk_CompilerGeneratedAttribute g.ilg ])

let i_pop = I_arith AI_pop  
let i_dup = I_arith AI_dup
let i_ldnull = I_arith AI_ldnull
let mk_ldc_i64 i = (I_arith (AI_ldc (DT_I8,NUM_I8 i)))

let defaultHashNodes = 18  (* WARNING: CHANGE THIS IN prim-types.fs  TOO!! *) 

(* Approximation for purposes of optimization and giving a warning when compiling definition-only files as EXEs *)
let rec doesSomething code = 
  match code with 
  | BasicBlock bb -> Array.fold_left (fun x i -> x || match i with I_arith (AI_ldnull | AI_nop | AI_pop) | I_ret |  I_seqpoint _ -> false | _ -> true) false bb.bblockInstrs
  | GroupBlock (_,codes) -> List.exists doesSomething codes
  | RestrictBlock (_,code) -> doesSomething code
  | TryBlock (code,seh) -> true 

let choose_freevar_names takenNames ts =
    let tns = map (fun t -> (t,None)) ts in
    let rec chooseName names (t,nOpt) = 
        let tn = match nOpt with None -> t | Some n -> t^string_of_int n in
        if Zset.mem tn names then
          chooseName names (t,Some(match nOpt with None ->  0 | Some n -> (n+1)))
        else
          let names = Zset.add tn names in
          names,tn in
    let string_order = (Pervasives.compare : string -> string -> int) in
    let names    = Zset.empty string_order |> Zset.addL takenNames in
    let names,ts = fmap chooseName names tns in
    ts

let clo_name_generator = newStableNiceNameGenerator ()
let nng = newNiceNameGenerator ()

(* cannot tailcall to methods taking byrefs *)
let is_byref  = function Il.Type_byref _ -> true | _ -> false

let mainMethName = "_main"

let decodeAttribNamedArgs namedArgs = 
    let namemap = Namemap.of_list (map (fun (s,a,b,c) -> s,c) namedArgs) in 
    let find_const x = match Namemap.tryfind x namemap with  | Some(TExpr_const(c,_,_)) -> Some c | _ -> None in 
    let find_int32 x dflt = match find_const x with  | Some(TConst_int32 x) -> x | _ -> dflt in 
    let find_bool x dflt = match find_const x with  | Some(TConst_bool x) -> x | _ -> dflt in 
    let find_string x dflt = match find_const x with  | Some(TConst_string x) -> Bytes.unicode_bytes_as_string x | _ -> dflt in 
    find_int32,find_bool,find_string 

(*--------------------------------------------------------------------------
!* Statistics
 *-------------------------------------------------------------------------- *)

let report_ref = ref (fun oc -> ()) 
let add_report f = let old = report_ref.contents in report_ref := (fun oc -> old oc; f oc) 
let report oc = report_ref.contents oc

let new_counter(nm) = 
  let count = ref 0 in 
  add_report (fun oc -> if !count <> 0 then output_string oc (string_of_int !count ^ " "^nm^"\n"));
  (fun () -> incr count)

let closure_counter = new_counter "closures"
let method_def_counter = new_counter "IL method defintitions corresponding to values"
let static_field_def_counter = new_counter "IL field defintitions corresponding to values"
let callfunc_counter = new_counter "callfunc instructions (indirect calls)"

(*-------------------------------------------------------------------------
 * Part of the last-minute tranformation performed by this file
 * is to eliminate variables of static type "unit".  These are
 * utility functions related to this.
 *------------------------------------------------------------------------- *)

let bind_unit_vars g (mvs,body) = 
    let unit_mvs,nonunit_mvs = List.partition (fun v -> type_equiv g (type_of_val v) g.unit_ty) mvs in
    let body = 
      if isNil unit_mvs then body 
      else mk_lets (range_of_val (List.hd unit_mvs)) unit_mvs (List.map (range_of_val >> mk_unit g) unit_mvs) body in 
    nonunit_mvs,body

let filter_unit_types g tys = List.filter (fun (ty,_) -> not (is_unit_typ g ty)) tys  

(*--------------------------------------------------------------------------
!* Compilation environment for compiling a whole a module
 *-------------------------------------------------------------------------- *)

(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type cenv = 
    { g: Env.tcGlobals;
      viewCcu: ccu;
      fragName: string;
      generateFilterBlocks: bool;
      workAroundReflectionEmitMethodImplBug: bool;
      amap: Import.importMap;
      (* main_info: if this is set, then the last module becomes the "main" module and its toplevel bindings are executed at startup *)
      main_info: Tast.attrib list option; 
      optimize: bool;
      debug: bool;
      empty_ok : bool;
      manager: Il.manager option }


(*--------------------------------------------------------------------------
!* scope, cloc, visibility
 * Referencing other stuff, and descriptions of where items are to be placed
 * within the generated IL namespace/typespace.  A bit of a mess.
 *-------------------------------------------------------------------------- *)
      
type cloc = 
    (* cloc = compilation location = path to a ccu, namespace or class *)
    { clocScope: Il.scope_ref; 
      clocTopImplQualifiedName: string;
      clocNamespace: string option;  
      clocEncl: string list;
      clocQualifiedNameOfFile : string }

(*--------------------------------------------------------------------------
!* Access this and other assemblies
 *-------------------------------------------------------------------------- *)

let mk_il_name pos n = match pos with [] -> n | _ -> String.concat "." pos^"."^n
let mk_private_name n = ("__$"^n) 

let scoref_for_cloc cloc = cloc.clocScope

let cloc_for_fragment fragName ccu = 
   { clocQualifiedNameOfFile =fragName;
     clocTopImplQualifiedName= fragName; 
     clocScope=(scoref_of_ccu ccu); 
     clocNamespace=None; 
     clocEncl=[]} 

let cloc_for_ccu ccu =  cloc_for_fragment (name_of_ccu ccu) ccu

let mk_topname ns n = String.concat "." (match ns with Some x -> [x;n] | None -> [n])

let cloc_for_subnamespace cloc n istype = 
  match istype with 
  | AsMangledNamedType _ | AsNamedType -> { cloc with clocEncl= cloc.clocEncl @ [adjust_module_name istype n]}
  | Namespace -> {cloc with clocNamespace=Some (mk_topname cloc.clocNamespace n)}

let cloc_for_submodul cloc submod =
  cloc_for_subnamespace cloc (name_of_modul submod) (mtyp_of_modul submod).mtyp_kind 
  
let cloc_for_cpath fragName qname (CompPath(sref,cpath)) = 
  let ns,t = list_take_until (fun (_,istype) -> istype <> Namespace) cpath in 
  let ns = map fst ns in 
  let ns = text_of_path ns in 
  let encl = map (fun (s ,istype)-> adjust_module_name istype s) t in
  if verbose then dprint_endline ("cloc_for_cpath, ns = '"^ns^"', encl = '"^text_of_path encl^"'");
  let ns = if ns = "" then None else Some ns in 
  { clocQualifiedNameOfFile =fragName;
    clocTopImplQualifiedName=qname;
    clocScope=sref;
    clocNamespace=ns; 
    clocEncl=encl }

let mk_nested_tref_for_cloc cloc n = 
    match cloc.clocEncl with 
    | [] ->
        let tyname = mk_topname cloc.clocNamespace n in
        mk_tref(scoref_for_cloc cloc,tyname)
    | h::t -> mk_nested_tref(scoref_for_cloc cloc,mk_topname cloc.clocNamespace h :: t,n)
        
let mk_nested_tspec_for_cloc cloc n tinst = 
    mk_tspec (mk_nested_tref_for_cloc cloc n,tinst)

let tnameStartupCode cloc = "<StartupCode$"^cloc.clocQualifiedNameOfFile^">.$"^cloc.clocTopImplQualifiedName 
let tnamePrivateImplementationDetails cloc = "<PrivateImplementationDetails$"^cloc.clocQualifiedNameOfFile^">"

let cloc_for_startup_code cloc = 
    {cloc with clocEncl=[tnameStartupCode cloc];clocNamespace=None}

let cloc_for_PrivateImplementationDetails cloc = 
    {cloc with 
        clocEncl=[tnamePrivateImplementationDetails cloc];clocNamespace=None}

let rec mk_tref_for_cloc cloc  =
    match cloc.clocEncl with
    | [] ->  
      mk_tref(scoref_for_cloc cloc,tnamePrivateImplementationDetails cloc)
    | [h] -> 
      let tyname = mk_topname cloc.clocNamespace h in
      mk_tref(scoref_for_cloc cloc,tyname)
    | _ ->  
      let encl,n = frontAndBack cloc.clocEncl in
      mk_nested_tref_for_cloc {cloc with clocEncl=encl} n 

let mk_tspec_for_cloc cloc = 
    mk_nongeneric_tspec (mk_tref_for_cloc cloc)

let computeMemberAccess hidden = if hidden then MemAccess_assembly else MemAccess_public 
let computeTypeAccess tref hidden = 
    match enclosing_tnames_of_tref tref with 
    | [] -> if hidden then TypeAccess_private else TypeAccess_public 
    | _ -> TypeAccess_nested (computeMemberAccess hidden)


(*--------------------------------------------------------------------------
!* Representation of type constructors etc.
 *
 * How are module kinds, type parameters, local type constructors 
 * etc. are mapped to IL types and IL type variables 
 *-------------------------------------------------------------------------- *)

type 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  typeReprEnv = 
    { typar_reprs: (typar_spec * Nums.u16 (* static_item_repr *) ) list;
      typar_count: int; (* How many type variables are in scope? *)
      tyenv_nativeptr_as_nativeint: bool (* Do we compile the "'a nativeptr" type as a machine integer, e.g. in closures? *) }

(*--------------------------------------------------------------------------
!* Lookup tyenv
 *-------------------------------------------------------------------------- *)

let repr_of_typar m tp tyenv = 
    try gen_assoc typar_ref_eq tp tyenv.typar_reprs
    with Not_found -> 
      errorR(InternalError("Undefined or unsolved type variable: "^showL(typarL tp),m)); 
      int_to_u16 666  

(*--------------------------------------------------------------------------
!* Type parameters and the environment
 *-------------------------------------------------------------------------- *)

let add_typar_as tyenv tp y =  {tyenv with typar_reprs=(tp,y) :: tyenv.typar_reprs }
let add_typar tyenv tp = { (add_typar_as tyenv tp (int_to_u16 tyenv.typar_count)) with typar_count= tyenv.typar_count + 1 }
let add_typars tyenv tps = fold_left add_typar tyenv tps
let empty_tyenv =  { typar_count=0; 
                     typar_reprs=[]; 
                     tyenv_nativeptr_as_nativeint=false} 
let tyenv_for_typars tps = add_typars empty_tyenv tps

let tyenv_for_tycon tycon = tyenv_for_typars (typars_of_tycon tycon)
let tyenv_for_tcref tcref = tyenv_for_tycon (deref_tycon tcref) 

(*--------------------------------------------------------------------------
!* Generate type references
 *-------------------------------------------------------------------------- *)

let gen_tcref tcref = 
    assert(isNone (abbrev_of_tycon (deref_tycon tcref)));
    il_repr_of_tcref tcref

(* When generating parameter and return types generate precise .NET IL pointer types *)
(* These can't be generated for generic instantiations, since .NET generics doesn't *)
(* permit this. But for 'naked' values (locals, parameters, return values etc.) machine *)
(* integer values and native pointer values are compatible (though the code is unverifiable). *)
type ptrsOK = PtrTypesOK | PtrTypesNotOK

let rec gen_tyarg_aux m g tyenv tyarg =  mk_genactual (gen_type_aux m g tyenv PtrTypesNotOK tyarg)
and gen_tyargs_aux m g tyenv tyargs = map (gen_tyarg_aux m g tyenv) tyargs 

and gen_tyapp_aux m g tyenv ptrsOK repr tinst =  
    let il_tinst = gen_tyargs_aux m g tyenv tinst in 
    match repr with  
    | TyrepOpen ty -> Il.inst_typ il_tinst ty
    | TyrepNamed (tref,boxity) -> Il.mk_typ boxity (mk_tspec (tref,il_tinst))

and gen_tcref_tyapp_aux m g tyenv ptrsOK tcref tinst = 
    (* See above note on PtrsOK *)
    if ptrsOK = PtrTypesOK && g.tcref_eq tcref g.nativeptr_tcr then 
      gen_tcref_tyapp_aux m g tyenv ptrsOK g.ilsigptr_tcr tinst
    else
      gen_tyapp_aux m g tyenv ptrsOK (gen_tcref tcref) tinst

and gen_type_aux m g tyenv ptrsOK ty =
    (* if verbose then dprintf1 "generating type '%s'\n" ((DebugPrint.showType ty)); *)
    match strip_tpeqns_and_tcabbrevs ty with 
    | TType_app(tcref,tinst) -> gen_tcref_tyapp_aux m g tyenv ptrsOK tcref tinst
    | TType_tuple(args) -> gen_type_aux m g tyenv ptrsOK (compiled_tuple_ty g args)
    | TType_fun(dty,rty) -> Pubclo.typ_Func1 g.ilxPubCloEnv  (gen_tyarg_aux m g tyenv dty) (gen_tyarg_aux m g tyenv rty)
    | TType_forall(tps,tau) -> 
        let tyenv = (add_typars tyenv tps) in 
        fold_right (gen_gparam m g tyenv >> Pubclo.typ_TyFunc g.ilxPubCloEnv) tps (gen_type_aux m g tyenv ptrsOK tau)
    | TType_var(tp) -> Type_tyvar (repr_of_typar m tp tyenv)
    | _ -> failwith "gen_type_aux m: unexpected naked Unknown/Struct/Named type" 

and gen_gparam m g tyenv (tp:typar_spec) = 
    let constraints = (constraints_of_typar tp) in 
    let check p = constraints |> List.exists p  in
    let subTypeConstraints = constraints |> chooseList (function | TTyparCoercesToType(TTyparSubtypeConstraintFromFS ty,m) -> Some(ty) | _ -> None) |> List.map (gen_type_aux m g tyenv PtrTypesNotOK) in 
    let refTypeConstraint              = check (function TTyparIsReferenceType _ -> true | _ -> false) in 
    let notNullableValueTypeConstraint = check (function TTyparIsNotNullableValueType _ -> true | _ -> false) in 
    let defaultConstructorConstraint = check (function TTyparRequiresDefaultConstructor _ -> true | _ -> false) in 
    { gpName=(String.capitalize (name_of_typar tp)) (* ^(if compgen_of_typar tp then string_of_int (stamp_of_typar tp) else "") *); 
      gpConstraints=subTypeConstraints;
      gpVariance=NonVariant;
      gpReferenceTypeConstraint=refTypeConstraint;
      gpNotNullableValueTypeConstraint=notNullableValueTypeConstraint;
      gpDefaultConstructorConstraint= defaultConstructorConstraint }

let gen_type m g tyenv ty = (gen_type_aux m g tyenv PtrTypesNotOK ty)
let gen_types m g tyenv tys = map (gen_type m g tyenv) tys

let gen_tyapp m g tyenv repr tyargs = gen_tyapp_aux m g tyenv PtrTypesNotOK repr tyargs
let gen_tcref_tyapp m g tyenv tcref tinst = gen_tcref_tyapp_aux m g tyenv PtrTypesNotOK tcref tinst 

(* IL pointer types are only generated for DLL Import signatures *)
(* IL void types are only generated for return types *)
let ptrTypesOK isDllImport = (if isDllImport then PtrTypesOK else PtrTypesNotOK)
let gen_return_type m g tyenv isDllImport rty = if is_unit_typ g rty then Type_void else gen_type_aux m g tyenv (ptrTypesOK isDllImport) rty
let gen_param_types m g tyenv isDllImport tys = map (gen_type_aux m g tyenv (ptrTypesOK isDllImport)) tys

let gen_tyargs m g tyenv tyargs = gen_tyargs_aux m g tyenv tyargs

let gparam_has_constraint gp = 
     nonNil gp.gpConstraints or
     gp.gpVariance <> NonVariant or
     gp.gpReferenceTypeConstraint or
     gp.gpNotNullableValueTypeConstraint or
     gp.gpDefaultConstructorConstraint


(*--------------------------------------------------------------------------
!* Adding module kinds, local type constructors, etc to the type environment
 *-------------------------------------------------------------------------- *)

let gen_alt_spec m g tyenv i fspecs = 
    fspecs |> Array.mapi (fun j fspec -> 
      let fdef = Il.mk_instance_fdef(name_of_rfield fspec,gen_type m g tyenv fspec.rfield_type, None, MemAccess_public) in 
      { fdef with fdCustomAttrs = mk_custom_attrs [(mk_CompilationMappingAttrWithVariantNumAndSeqNum g (4 (* SourceLevelConstruct.Field *)) i j )] } )
   

let gen_curef m g tcref = 
    let tycon = (deref_tycon tcref) in 
    assert(abbrev_of_tycon tycon = None);
    match funion_of_tycon tycon with 
    | None -> failwith "gen_curef m"
    | Some funion -> 
      cached funion.funion_ilx_repr (fun () -> 
        let tyenvinner = tyenv_for_tycon tycon in 
        match il_repr_of_tcref tcref with
        | TyrepOpen _ -> failwith "gen_curef m: unexpected ASM tyrep"
        | TyrepNamed (tref,_) -> 
          let alternatives = 
              tycon |> uconstrs_array_of_tycon |> Array.mapi (fun i cspec -> 
                  { altName=cspec.uconstr_il_name;
                    altCustomAttrs=mk_custom_attrs [];
                    altFields=gen_alt_spec m g tyenvinner i (rfields_array_of_uconstr cspec) }) in 
          let nullPermitted = isUnionThatUsesNullAsRepresentation g tycon in
          Ilx.ClassunionRef(tref,alternatives,nullPermitted))

let repr_of_named_type cloc tycon_name boxity =
  TyrepNamed (mk_nested_tref_for_cloc cloc tycon_name,boxity)


(*--------------------------------------------------------------------------
!* Generate ILX references to closures, classunions etc. given a tyenv
 *-------------------------------------------------------------------------- *)

let gen_cuspec m cenv tyenv tcref tyargs = 
  let curef = gen_curef m cenv.g tcref in
  let tinst = gen_tyargs m cenv.g tyenv tyargs in  
  Ilx.ClassunionSpec(curef,tinst) 

let gen_cuspec_idx m cenv tyenv ucref tyargs = 
  let cuspec = gen_cuspec m cenv tyenv (tcref_of_ucref ucref) tyargs in 
  let idx = ucref_index ucref  in        
  cuspec, idx 

(* Static fields generally go in a private StartupCode section. This is to ensure all static 
   fields are initialized only in their class constructors (we generate one primary 
   cctor for each file to ensure initialization coherence across the file, regardless 
   of how many modules are in the file). This means F# passes an extra check applied by SQL Server when it
   verifies stored procedures: SQL Server checks that all 'initionly' static fields are only initialized from
   their own class constructor. 
   
   However, mutable static fields must be accessible across compilation units. This means we place them in their "natural" location
   which may be in a nested module etc. This means mutable static fields can't be used in code to be loaded by SQL Server. *)
   
let use_genuine_static_field g vspec =
    let mut = (mutability_of_val vspec <> Immutable) in
    let attribs = attribs_of_val vspec in
    let hasLiteralAttr = fsthing_has_attrib g g.attrib_LiteralAttribute attribs in
    mut || hasLiteralAttr 

let fspec_for_static_field g binding_tspec vspec cloc fieldName il_ty =
    let tspec = if use_genuine_static_field g vspec then binding_tspec else mk_tspec_for_cloc  (cloc_for_startup_code cloc) in 
    mk_fspec_in_boxed_tspec (tspec,fieldName, il_ty) 

(* REVIEW: this logic is also duplicated in tc.ml's attribute type checking code *)
let gen_field_name tycon f = 
    if use_genuine_field tycon f then f.rfield_id.idText
    else (* if f.rfield_mutable then *) "_"^f.rfield_id.idText (* else f.rfield_id.idText*)

let gen_recdfield_ref m cenv tyenv rfref tyargs = 
    let tycon,fld = deref_rfield rfref in 
    let tyenvinner = tyenv_for_tycon tycon in 
    mk_fspec_in_typ(gen_tyapp m cenv.g tyenv (il_repr_of_tcref (tcref_of_rfref rfref)) tyargs,
                    gen_field_name tycon fld,
                    gen_type m cenv.g tyenvinner fld.rfield_type)

let gen_exn_type m g tyenv ecref = gen_tyapp m g tyenv (il_repr_of_tcref ecref) []

let gen_exn_finfo m cenv tyenv exnc field_num = 
    let fspec = nth (instance_rfields_of_tycon exnc) field_num in 
    name_of_rfield fspec,gen_type m cenv.g tyenv fspec.rfield_type


(*--------------------------------------------------------------------------
!* Closure summaries
 *-------------------------------------------------------------------------- *)

type arity_info = int list
      

type 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  cloinfo = 
    { clo_name: string;
      clo_arity_info: arity_info;
      clo_formal_il_rty: Il.typ;
      clo_il_frees: Ilx.freevar list;
      clo_clospec: Ilx.closure_spec;
      clo_attribs: attrib list;
      clo_il_gparams: Il.genparams;
      clo_il_lambda_gparams_direct: Il.genparams;
      clo_freevars: val_spec list; (* nb. the freevars we actually close over *)
      clo_lambdas: Ilx.lambdas }


(*--------------------------------------------------------------------------
!* Representation of term declarations = Environments for compiling expressions.
 *-------------------------------------------------------------------------- *)

      
type 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  storage = 
    | Null (* always null *)
    | Unrealized (* not stored, no value created *)
    | StaticField of Il.field_spec * bool * Il.type_spec * string * string * Il.typ * Il.method_ref  * Il.method_ref (* value is stored in a static field. *)
    | Method of  val_arity * val_ref * Il.method_spec * Range.range * topArgInfo list * topArgInfo
        (* i.e. value is "stored" as a IL static method (in a "main" class for a F# *)
        (* compilation unit, or as a member) according to its inferred or specified arity.  *)
    | Env of int * namedLocalClosureInfo ref option  (* stored at given position in environment *)
    | Arg of int (* stored in argument of a method *)
    | Local of int * namedLocalClosureInfo ref option (* stored in local of a method *)

(* the representation of a NamedLocalClosure is based on a cloinfo.  However we can't generate a cloinfo until we've *)
(* decided the representations of other items in the recursive set. Hence we use two phases to decide representations in *)
(* a recursive set. Yuck. *)
and namedLocalClosureInfo = 
    | NamedLocalClosureInfoGenerator of (ilxGenEnv -> cloinfo)
    | NamedLocalClosureInfoGenerated of cloinfo
  
and struct_storage = 
    { storage_vals: storage namemap Lazy.t;
      storage_submoduls: struct_storage namemap Lazy.t; }

(* BranchCallItems are those where a call to the value can be implemented as *)
(* a branch. At the moment these are only used for generating branch calls back to *)
(* the entry label of the mehod currently being generated. *)
and branchCallItem = 
    | BranchCallClosure of arity_info
    | BranchCallMethod of arity_info * Tast.typ list list * Tast.typar_spec list * int
      
and mark = Mark of Il.code_label (* places we can branch to  *)

and ilxGenEnv =
    { tyenv: typeReprEnv; 
      someTspecInThisModule: type_spec;
      cloc: cloc; (* Where to place the stuff we're currently generating *) 
      mrmi: (expr_remap * module_hiding_information) list; (* hiding information down the signature chain, used to compute what's public to the assembly *)
      valsInScope: storage vspec_map; (* locals in scope *)
      innerVals: (val_ref * (branchCallItem * mark)) list; (* for optimizing direct tail recusion to a loop - mark says where to branch to.  REVIEW: generalize to arbitrary nested local loops?? *)
      letBoundVars: val_ref list; (* Full list of enclosing bound values.  First non-compiler-generated element is used to help give nice names for closures and other expressions.  *)
      liveLocals: unit Imap.t; (* set of local indexes currently in use, to allow reuse on different branches. *)
      withinSEH: bool; (* Are we under the scope of a try, catch or finally? If so we can't tailcall. *)  }

let replace_tyenv tyenv eenv = {eenv with tyenv = tyenv}
let env_for_typars tps eenv =  replace_tyenv (tyenv_for_typars tps) eenv
let add_typars_to_env typars eenv = {eenv with tyenv = add_typars eenv.tyenv typars}

let add_rmi msg (rpi,mhi) eenv = 
    if verbose then dprintf2 "add_rmi, %s, #tycons = %s\n" msg (showL (Layout.sepListL (wordL ";") (map tyconL (Zset.elements mhi.mhiTycons))));
    if verbose then dprintf3 "add_rmi, %s, #rpi.mrpiTycons = %d, #tyconReprs = %s\n" msg (length rpi.mrpiTycons) (showL (Layout.sepListL (wordL ";") (map tyconL (Zset.elements mhi.mhiTyconReprs))));
    if verbose then dprintf2 "add_rmi, %s, #vals = %s\n" msg (showL (Layout.sepListL (wordL ";") (map vspecL (Zset.elements mhi.mhiVals))));
    if verbose then dprintf2 "add_rmi, %s, #rfrefs = %s\n" msg (showL (Layout.sepListL (wordL ";") (map rfrefL (Zset.elements mhi.mhiRecdFields))));
    { eenv with mrmi = (mk_repackage_remapping rpi,mhi) :: eenv.mrmi }
let isHidden setF accessF remapF debugF = 
    let rec check mrmi x = 
        if verbose then dprintf1 "isHidden %s ??\n" (showL (debugF x));
        match mrmi with 
        | [] -> false (* Ah! we escaped to freedom! *)
        | (rpi,mhi) :: rest -> 
            (* Explicitly hidden? *)
            Zset.mem x (setF mhi) or 
            (* Internal/private? *)
            not (can_access_from_everywhere (accessF x)) or 
            (* Recurse... *)
            check rest (remapF rpi x) in 
    fun mrmi x -> 
        let res = check mrmi x in 
        if verbose then dprintf3 "isHidden, #mrmi = %d, %s = %b\n" (length mrmi) (showL (debugF x)) res;
        res
        
let isHiddenTycon     = isHidden (fun mhi -> mhi.mhiTycons)     access_of_tycon      (fun rpi x ->  deref_tycon (remap_tcref rpi.Tastops.tyenv.tcref_remap (mk_local_tcref x))) tyconL
let isHiddenTyconRepr = isHidden (fun mhi -> mhi.mhiTyconReprs) repr_access_of_tycon (fun rpi x ->  deref_tycon (remap_tcref rpi.Tastops.tyenv.tcref_remap (mk_local_tcref x))) tyconL
let isHiddenVal       = isHidden (fun mhi -> mhi.mhiVals)       access_of_val        (fun rpi x ->  deref_val (remap_vref rpi (mk_local_vref x))) vspecL
let isHiddenRecdField = isHidden (fun mhi -> mhi.mhiRecdFields) (rfield_of_rfref >> access_of_rfield)     (fun rpi x ->  remap_rfref rpi.Tastops.tyenv.tcref_remap x) rfrefL
     
(*--------------------------------------------------------------------------
!* Print eenv
 *-------------------------------------------------------------------------- *)

let output_storage pps s = 
    match s with 
    | StaticField _ -> output_string pps "(top)" 
    | Method _ -> output_string pps "(top)" 
    | Local _ -> output_string pps "(local)" 
    | Arg _ -> output_string pps "(arg)" 
    | Env _ -> output_string pps "(env)" 
    | Null -> output_string pps "(null)"
    | Unrealized -> output_string pps "(no real value required)"

(*--------------------------------------------------------------------------
!* Augment eenv with values
 *-------------------------------------------------------------------------- *)

let add_storage_for_val g (v,s) eenv = 
    if verbose then dprintf1 "adding %s to value table\n" (showL (vspecL v));
    let eenv = { eenv with valsInScope = vspec_map_add v s eenv.valsInScope } in 
    (* when compiling fslib add an entry under the results of a non-local lookup *)
    let eenv = 
        if g.compilingFslib then 
            match pubpath_of_val v with 
            | None -> eenv
            | Some pp -> 
                match try_deref_val (rescope_val_pubpath g.fslibCcu pp v) with
                | None -> eenv
                | Some gv -> 
                    if verbose then dprintf1 "adding remapped %s to value table\n" (showL (vspecL gv));
                    { eenv with valsInScope = vspec_map_add gv s eenv.valsInScope }
        else eenv in 
    eenv

let add_storage_for_locvals g vals eenv = fold_right (add_storage_for_val g) vals eenv

(*--------------------------------------------------------------------------
!* Lookup eenv 
 *-------------------------------------------------------------------------- *)
 
let storage_for_val m v eenv = 
    try vspec_map_find v eenv.valsInScope
    with Not_found -> errorR(Error(sprintf "undefined value: %s" (showL(vspecAtBindL v)),m)); Arg 668

let storage_for_vref m v eenv = storage_for_val m (deref_val v) eenv

(*--------------------------------------------------------------------------
!* Imported modules and the environment
 *
 * How a top level value is represented depends on its type.  If it's a 
 * function or is polymorphic, then it gets represented as a 
 * method (possibly and instance method).  Otherwise it gets represented as a 
 * static field.
 *-------------------------------------------------------------------------- *)

let vref_isDllImport g vref = 
    vref |> attribs_of_vref |> fsthing_has_attrib g g.attrib_DllImportAttribute 

let mspec_for_vspr_vref g vspr vref = 
    let m = range_of_vref vref in 
    if verbose then dprintf2 "mspec_for_vspr_vref %s, (type_of_val vref) = %s\n" (name_of_vref vref) (showL(vrefL vref));
    let tps,fargtysl,rty,retInfo = 
         assert(isSome(arity_of_vref vref));
         dest_top_type (the(arity_of_vref vref)) (type_of_vref vref) in
    let tyenv_under_typars = tyenv_for_typars tps in 
    let fargtys = (concat fargtysl) in 
    let isDllImport = vref_isDllImport g vref in 
    let il_rty = gen_return_type m g tyenv_under_typars isDllImport rty in 
    let ctor = (vspr.vspr_flags.memFlagsKind = MemberKindConstructor) in 
    let cctor = (vspr.vspr_flags.memFlagsKind = MemberKindClassConstructor) in 
    let il_actual_rty = if ctor || cctor then Type_void else il_rty in 
    let parent_tcref = actual_parent_of_vspr_vref vref in 
    let tctps = typars_of_tcref parent_tcref in
    let nctps = length tctps in 
    if List.length tps < nctps then error(Error("CodeGen check: type checking did not ensure that this method is sufficiently generic", m));
    let ctps,mtps = chop_at nctps tps in 
    let instance = vrefCompiledAsInstance g vref in 
    let il_typ = gen_type m g tyenv_under_typars (mk_tyapp_ty parent_tcref (map mk_typar_ty ctps)) in 
    if instance || ctor then begin
(*
        if (tycon_is_modul (deref_tycon parent_tcref)) then 
           failwith(Printf.sprintf "Assertion failed for '%s'" (name_of_vref vref));
*)        
        (* find the 'this' argument type if any *)
        let thisty,fargtys = 
          if ctor then rty,fargtys 
          else match fargtys with 
          | [] -> error(Error("This instance method '"^name_of_vref vref^"' has no arguments", m))
          | (h,_):: t -> h,t in  
        let thisty = if is_byref_ty g thisty then dest_byref_ty g thisty else thisty in 
        let this_argtys = (tinst_of_stripped_typ thisty) in 
        if List.length ctps <> length this_argtys then
           warning(Error(Printf.sprintf "CodeGen check: type checking did not quantify the correct number of type variables for this method, #tctps = %d, #ctps = %d, #mtps = %d, #this_argtys = %d" nctps (List.length ctps) (List.length mtps) (length this_argtys),m))
        else 
           List.iter2
              (fun gtp ty2 -> 
                if not (type_equiv g (mk_typar_ty gtp) ty2) then 
                  warning(Error("CodeGen check: type checking did not quantify the correct type variables for this method: generalization list contained "^name_of_typar gtp^"#"^string_of_int (stamp_of_typar gtp)^" and list from 'this' pointer contained "^ (showL(typeL ty2)), m)))
              ctps 
              this_argtys;
        let method_argtys = filter_unit_types g fargtys in 
        let method_argtys,paramInfos = split method_argtys in 
        let il_method_argtys = gen_param_types m g tyenv_under_typars isDllImport method_argtys in 
        let il_minst = gen_tyargs m g tyenv_under_typars (map mk_typar_ty mtps) in
        let mspec = mk_instance_mspec_in_typ (il_typ,vspr.vspr_il_name,il_method_argtys,il_actual_rty,il_minst) in 
        let mspec = (* intern_mspec cenv.manager *) mspec in
        mspec,ctps,mtps,paramInfos,retInfo
    end else begin
        let method_argtys = filter_unit_types g fargtys in 
        let method_argtys,paramInfos = split method_argtys in 
        let il_method_argtys = gen_param_types m g tyenv_under_typars isDllImport method_argtys in 
        let il_minst = gen_tyargs m g tyenv_under_typars (map mk_typar_ty mtps) in
        let mspec = mk_static_mspec_in_typ (il_typ,vspr.vspr_il_name,il_method_argtys,il_actual_rty,il_minst) in
        let mspec = (* intern_mspec cenv.manager *) mspec in
        mspec,ctps,mtps,paramInfos,retInfo
    end

let storage_for_topval g is_uniq vref cloc =
    (* This called via 2 routes.
     * (a) alloc_or_import_{ccu,modval,top_vspec} - for vref from modulespec mtyp_vals.
     * (b) alloc_val_for_bind - if arity specified for vref
     *
     * This function decides the storage for the val.
     * The decision is based on arity_info.
     *)

    if is_unit_typ g (type_of_vref vref)  && not is_uniq then  Null   else
    let arity_info = 
        match (arity_of_vref vref) with 
        | None -> error(InternalError("storage_for_topval: no arity found for "^showL(vrefL vref),range_of_vref vref))
        | Some a -> a in
        
    if inlineFlag_of_vref vref = PseudoValue then Unrealized else
    let nm = name_of_vref vref in
    let m = range_of_vref vref in 
    if verbose then dprintf1 "storage_for_topval %s\n" nm;
    let nm = if is_uniq then nm else nng.nngApply nm m in 

    match dest_top_type arity_info (type_of_vref vref) with 
    | [],[], rty,retInfo when isNone(member_info_of_vref vref) ->
        if verbose then dprintf1 "storage_for_topval %s as field\n" nm;
        (* Mutable and literal static fields must have stable names and live in the "public" location *)
        (* See notes on fspec_for_static_field above. *)
        let vspec = (deref_val vref) in 
        let fieldName = if use_genuine_static_field g vspec then nm else nng.nngApply nm m in
        let il_ty = gen_type m g empty_tyenv rty in (* empty_tyenv ok: not a field in a generic class *)
        let binding_tspec = mk_tspec_for_cloc cloc in
        let attribs = attribs_of_val vspec in
        let mut = (mutability_of_val vspec <> Immutable) in 
        let hasLiteralAttr = fsthing_has_attrib g g.attrib_LiteralAttribute attribs in

        let tref = tref_of_tspec binding_tspec in
        let get_mref = mk_mref(tref,static_callconv,"get_"^nm,0,[],il_ty) in 
        let set_mref = mk_mref(tref,static_callconv,"set_"^nm,0,[il_ty],Type_void) in
        let fspec = fspec_for_static_field g binding_tspec vspec cloc fieldName il_ty in
        if verbose then dprint_endline ("storage_for_topval: StaticField: "^fieldName);
        StaticField (fspec,hasLiteralAttr,binding_tspec,fieldName,nm,il_ty,get_mref,set_mref)
          
    | tps,fargtysl,rty,retInfo ->
        if verbose then dprintf1 "storage_for_topval %s as method\n" nm;
        match member_info_of_vref vref with 
        | Some(vspr) when not (isext_of_vref vref) -> 
            let mspec,_,_,paramInfos,retInfo = mspec_for_vspr_vref g vspr vref in
            Method (arity_info,vref,mspec, m,paramInfos,retInfo) 
        | _ -> 
            let tyenv_under_typars = tyenv_for_typars tps in 
            let fargtys = (concat fargtysl) in 
            let method_argtys = filter_unit_types g fargtys in 
            let method_argtys,paramInfos = split method_argtys in 
            let isDllImport = vref_isDllImport g vref in 
            let il_method_argtys = gen_param_types m g tyenv_under_typars isDllImport method_argtys in 
            let il_rty = gen_return_type m g tyenv_under_typars isDllImport rty in 
            let tspec = mk_tspec_for_cloc cloc in
            if verbose then dprintf3 "storage_for_topval, vref = %s, tspec = %a\n" nm Ilprint.output_tspec tspec;
            let il_minst = gen_tyargs m g tyenv_under_typars (map mk_typar_ty tps) in
            let mspec = mk_static_mspec_in_boxed_tspec (tspec,nm,il_method_argtys,il_rty,il_minst) in 
            let mspec = (* intern_mspec cenv.manager *) mspec in
            Method (arity_info,vref,mspec, m,paramInfos,retInfo)


      
let add_storage_for_local_topval g cloc is_uniq v eenv =
    let storage = storage_for_topval g (is_uniq && not (compgen_of_val v)) (mk_local_vref v) cloc in 
    add_storage_for_val g (v,storage) eenv

let storage_for_nonlocal_topval g cloc modref v =
    if inlineFlag_of_val v = PseudoValue then Unrealized
    else 
      match (arity_of_val v) with 
      | None -> error(InternalError("storage_for_nonlocal_topval, expected an arity for "^name_of_val v,range_of_val v))
      | Some _ -> storage_for_topval g true (mk_vref_in_modref modref v) cloc

let rec add_storage_for_nonlocal_modref g cloc (modref:modul_ref) modul acc = 
    if verbose then dprint_endline ("add_storage_for_nonlocal_modref for module "^name_of_modul modul);
    Namemap.fold_range 
        (fun v acc -> add_storage_for_val g (v, storage_for_nonlocal_topval g cloc modref v) acc) 
        (mtyp_of_modul modul).mtyp_vals
        (Namemap.fold_range 
            (fun smodul acc -> add_storage_for_nonlocal_modref g (cloc_for_submodul cloc smodul) (mk_modref_in_modref modref smodul) smodul acc) 
            (submoduls_of_mtyp (mtyp_of_modul modul))
            acc)

let add_storage_for_nonlocal_ccu g  eenv ccu = 
    if not (ccu_is_fsharp ccu) then eenv else
    let cloc = cloc_for_ccu ccu in
    if verbose then dprint_endline ("add_storage_for_nonlocal_ccu, ccu = "^(name_of_ccu ccu));
    let eenv = 
       Namemap.fold_range
           (fun smodul acc -> 
               let cloc = cloc_for_submodul cloc smodul in 
               let modref =  mk_nonlocal_ccu_top_modref ccu smodul in 
               add_storage_for_nonlocal_modref g cloc modref smodul acc)
           (top_moduls_of_ccu ccu)
           eenv in
    eenv
    
let rec add_local_mtyp g  cloc eenv mty = 
    let eenv = Namemap.fold_range (fun submodul eenv -> add_local_mtyp g (cloc_for_submodul cloc submodul) eenv (mtyp_of_modul submodul)) mty.mtyp_submoduls eenv in
    let eenv = Namemap.fold_range (add_storage_for_local_topval g cloc true) mty.mtyp_vals eenv in 
    eenv 

let add_external_ccus g eenv ccus = fold_left (add_storage_for_nonlocal_ccu g) eenv ccus

let add_binds_for_tycon g tycon eenv  =
    let unrealized_slots = 
        if is_fsobjmodel_tycon tycon
        then (tycon_objmodel_data_of_tycon tycon).fsobjmodel_vslots 
        else [] in 
    fold_right (deref_val >> add_storage_for_local_topval g eenv.cloc true) unrealized_slots eenv

let rec add_binds_for_mdefs g cloc eenv  mdefs = 
    fold_left (add_binds_for_mdef g cloc) eenv mdefs
and add_binds_for_mdef g cloc eenv x = 
    match x with 
    | TMDefRec(tycons,binds,m) -> 
        let eenv = add_bind_mdef_topvals  g cloc eenv (map var_of_bind binds)  in
        (* Virtual don't have 'let' bindings and must be added to the environment *)
        fold_right (add_binds_for_tycon g) tycons eenv 
    | TMDefLet(bind,m) -> 
        add_bind_mdef_topvals g cloc eenv [var_of_bind bind]
    | TMDefModul(TMBind(tycon, mdef)) -> 
        let cpath = cpath_of_modul tycon in 
        let cloc = 
            if (mkind_of_mtyp (mtyp_of_modul tycon) = Namespace) then cloc 
            else cloc_for_cpath cloc.clocQualifiedNameOfFile cloc.clocTopImplQualifiedName cpath in
            
        add_binds_for_mdef g cloc eenv  mdef
    | TMAbstract(TMTyped(mtyp,_,_)) -> 
        add_local_mtyp g cloc eenv  mtyp
    | TMDefs(mdefs) -> 
        add_binds_for_mdefs g cloc eenv  mdefs 
and add_bind_mdef_topvals g cloc eenv vs = fold_right (add_storage_for_local_topval g cloc true) vs eenv


(* Put the partial results for a generated fragment (i.e. a part of a CCU generated by FSI) *)
(* into the stored results for the whole CCU.  *)
(* isIncrementalExtension = true -->  "#use or typed input" *)
(* isIncrementalExtension = false -->  "#load" *)
let add_incremental_local_mimpls isIncrementalExtension  g ccu fragName eenv (TAssembly impls) = 
    let cloc = cloc_for_fragment fragName ccu in 
    fold_left (fun eenv (TImplFile(qname,mexpr)) -> 
        let cloc = { cloc with clocTopImplQualifiedName = text_of_qualNameOfFile qname } in
        if isIncrementalExtension then 
            match mexpr with
            | TMTyped(_,mdef,_) -> add_binds_for_mdef g  cloc eenv mdef
            (* | TMTyped(mtyp,_,m) -> error(Error("don't expect inner defs to have a constraint",m)) *)
        else
            add_local_mtyp g cloc eenv (mtyp_of_mexpr mexpr)
            
        ) eenv  impls

(*--------------------------------------------------------------------------
!* Buffers for compiling modules. Each F# module gets compiled into an mgbuf
 *-------------------------------------------------------------------------- *)

(** Information collected imperatively for each type definition ('Generated type definition') *)
type 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  gtdef = 
  { gbasic     : Il.type_def;
    gmethods   : Il.method_def ResizeArray.t;
    gfields    : Il.field_def  ResizeArray.t;
    gproperties: (string,Il.property_def) Hashtbl.t;
    gevents    : Il.event_def ResizeArray.t;
    gnested    : gtdefs }

and gtdefs = (string,gtdef) Hashtbl.t

let new_gtdef tdef = 
    { gbasic     = tdef;
      gmethods   = ResizeArray.create 0;
      gfields    = ResizeArray.create 0;
      gproperties= Hashtbl.create 0;
      gevents    = ResizeArray.create 0;
      gnested    = Hashtbl.create 0; }

(** Module generation buffers *)
type 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  mgbuf = 
    { (* cache for generating value types for big constant arrays  *)
      mutable vtgenerator: (cloc * int) -> Il.type_spec; 
      (* The Abstract IL table of types *)
      mutable gtdefs: gtdefs; 
      (* The definitions of top level values, as quotations. *)
      mutable quotedDefs: (Sreflect.Raw.topDefPath * Tast.typ * Tast.expr) list; 
      mutable explicitEntryPointInfo : type_ref option }

let find_gtdef gtdefs nm = 
    try Hashtbl.find gtdefs nm 
    with Not_found -> failwith ("find_gtdef: "^nm^" not found")

let find_gtdefs gtdefs path = 
    List.fold_left (fun acc x -> (find_gtdef acc x).gnested) gtdefs path

let find_nested_gtdef gtdefs tref =
    let gtdefs = find_gtdefs gtdefs (enclosing_tnames_of_tref tref) in 
    let gtdef = find_gtdef gtdefs (tname_of_tref tref) in 
    gtdef 

let insert_tdef mgbuf tref tdef =
    let gtdefs = find_gtdefs mgbuf.gtdefs (enclosing_tnames_of_tref tref) in 
    Hashtbl.add gtdefs tdef.tdName (new_gtdef tdef)

let new_mgbuf cenv = 
    let mgbuf = {  vtgenerator=(fun _ -> failwith ""); 
                   explicitEntryPointInfo=None;
                   gtdefs= Hashtbl.create 10; 
                   quotedDefs = [] } in 
    mgbuf.vtgenerator <- 
       memoize
         (fun (cloc,size) -> 
           let vtdef = mk_rawdata_vtdef cenv.g.ilg (mk_private_name (string_of_int(new_uniq())), int_to_i32 size, u16_zero) in 
           let vtspec = mk_nested_tspec_for_cloc cloc vtdef.tdName [] in 
           let vtref = tref_of_tspec vtspec in 
           let vtdef = {vtdef with tdAccess= computeTypeAccess vtref true} in
           insert_tdef mgbuf vtref vtdef;
           vtspec);
    mgbuf

let insert_renv_def mgbuf (PubPath(p,n)) ty e = 
    mgbuf.quotedDefs <- ((p,n),ty,e) :: mgbuf.quotedDefs

let insert_mdef mgbuf tref mdef = 
    ResizeArray.add (find_nested_gtdef mgbuf.gtdefs tref).gmethods mdef;
    if mdef.mdEntrypoint then 
        mgbuf.explicitEntryPointInfo <- Some(tref)

let prepend_instrs_to_specific_mdef cond mgbuf tref instrs tag = 
    let mdefs = (find_nested_gtdef mgbuf.gtdefs tref).gmethods in 
    match ResizeArray.tryfind_index cond mdefs with
    | Some idx -> ResizeArray.replace mdefs idx (prepend_instrs_to_mdef instrs (ResizeArray.get mdefs idx))
    | None -> ResizeArray.add mdefs (mk_cctor (mk_impl (false,[],1,nonbranching_instrs_to_code instrs,tag)))

let explicit_init_instrs fspec = 
    [ mk_ldc_i32 (Int32.of_int 0); 
      mk_normal_stsfld fspec; 
      mk_normal_ldsfld fspec; 
      i_pop]    

let add_explicit_init_to_cctor mgbuf tref fspec m = 
    prepend_instrs_to_specific_mdef (fun md -> string_eq md.mdName ".cctor") mgbuf tref (explicit_init_instrs fspec) m

let insert_edef mgbuf tref edef = 
    ResizeArray.add (find_nested_gtdef mgbuf.gtdefs tref).gevents edef

let insert_fdef mgbuf tref fdef = 
    ResizeArray.add (find_nested_gtdef mgbuf.gtdefs tref).gfields fdef

(* Merge Get and Set property nodes, whih we generate independently for F# code *)
(* when we come across their corresponding methods. *)
let merge_options o1 o2 = 
    match o1,o2 with
    | Some x, None | None, Some x -> Some x
    | None, None -> None
    | Some x, Some _ -> warning(Failure("merge_options: two values given")); Some x

let merge_pdefs pd pdef = 
            {pd with propGet=merge_options pd.propGet pdef.propGet;
                     propSet=merge_options pd.propSet pdef.propSet; }

let add_pdef_to_hash ht pdef = 
    let nm = pdef.propName in 
    if Hashtbl.mem ht nm then
      let pd = Hashtbl.find ht nm in 
      Hashtbl.replace ht nm (merge_pdefs pd pdef) 
    else 
      Hashtbl.add ht nm pdef

let hashtbl_range ht = 
    let res = ref [] in 
    Hashtbl.iter (fun _ x -> res := x :: !res) ht; 
    !res

(* Merge a whole group of properties all at once *)
let merge_pdef_list pdefs = 
    let ht = Hashtbl.create 0 in 
    pdefs |> List.iter (add_pdef_to_hash ht);  
    hashtbl_range ht
    
(* Merge properties one by one into an existing type def *)
let insert_or_merge_pdef mgbuf tref pdef = 
    let gtdef = find_nested_gtdef mgbuf.gtdefs tref in
    add_pdef_to_hash gtdef.gproperties pdef

let rec gtdef_to_tdef gtdef = 
    let tdef = gtdef.gbasic in 
    { tdef with 
        tdMethodDefs = mk_mdefs      (dest_mdefs tdef.tdMethodDefs @ ResizeArray.to_list gtdef.gmethods);
        tdFieldDefs  = mk_fdefs      (dest_fdefs tdef.tdFieldDefs  @ ResizeArray.to_list gtdef.gfields);
        tdProperties = mk_properties (dest_pdefs tdef.tdProperties @ hashtbl_range gtdef.gproperties);
        tdEvents     = mk_events     (dest_edefs tdef.tdEvents     @ ResizeArray.to_list gtdef.gevents);
        tdNested     = mk_tdefs      (dest_tdefs tdef.tdNested     @ gtdefs_to_tdefs gtdef.gnested) }

and gtdefs_to_tdefs gtdefs = List.map gtdef_to_tdef (hashtbl_range gtdefs)

let mgbuf_to_top_tdefs mgbuf = gtdefs_to_tdefs mgbuf.gtdefs

(*--------------------------------------------------------------------------
!* Buffers for IL code generation, including keeping track of the
 * current stack so we can spill stuff when we hit a "try" when some stuff
 * is on the stack.
 *-------------------------------------------------------------------------- *)

let code_label_of_mark (Mark(lab)) = lab

module CG = struct

    (* When generating a method we just write into a whole bunch of mutable  *)
    (* structures representing the contents of the method. *)
    type 
      (*F#
      [<StructuralEquality(false); StructuralComparison(false)>]
      F#*)
      cgbuf =
      { mgbuf: mgbuf; (* The bits of the current module being generated *)
        m : range;
        locals:  ((string * (mark * mark)) list * Il.typ) ResizeArray.t; 
        exn_specs:  Il.exception_spec ResizeArray.t;
        mutable stack : Il.typ list;
        mutable nstack: int;
        mutable maxstack: int;
        mutable last_was_seqpoint: bool;
        codebuf: Il.instr ResizeArray.t;
        methodName: string;
        code_labels: (code_label,int) Hashtbl.t;
        entrypoint_info: (val_ref * branchCallItem) list;
        already_used_args: int;
        preallocated_locals: int; }

    (* Record the types of the things on the evaluation stack. *)
    (* Used for the few times we have to flush the IL evaluation stack and to compute maxstack. *)
    type pushpop = Push of Il.typ | Pop
    let push ty = Push ty
    let doaction cgbuf a = 
        match a with 
        | Push ty -> 
           cgbuf.stack <- ty :: cgbuf.stack; 
           cgbuf.nstack <- cgbuf.nstack + 1;
           cgbuf.maxstack <- max cgbuf.maxstack cgbuf.nstack
        | Pop -> 
           match cgbuf.stack with
           | [] -> dprintf2 "***** WARNING: pop on empty stack near %a\n" output_range cgbuf.m
           | h::t -> cgbuf.stack <- t; cgbuf.nstack <- cgbuf.nstack - 1

    let curr_stack cgbuf = cgbuf.stack
    let write_codebuf cgbuf i = ResizeArray.add cgbuf.codebuf i
    let emit_instr cgbuf pps i = 
      cgbuf.last_was_seqpoint <- false;
      iter (doaction cgbuf) pps;
      write_codebuf cgbuf i 

    let emit_instrs cgbuf pps is = 
      cgbuf.last_was_seqpoint <- false;
      iter (doaction cgbuf) pps;
      iter (write_codebuf cgbuf) is 
       
    let emit_seqpoint cenv cgbuf src = 
      if cenv.debug (* && !stack = [] *) then begin
         (* Clobber a previous seqpoint with a better one *)
        let attr = gen_range src in
        assert(isSome(attr));
        let i = I_seqpoint (the attr) in 
        if cgbuf.last_was_seqpoint then 
          ResizeArray.replace cgbuf.codebuf (ResizeArray.length cgbuf.codebuf - 1) i
        else begin
          (* if !codebuf_size <> 0 then write_codebuf (I_arith AI_nop);*)
          write_codebuf cgbuf i;
        end;
        cgbuf.last_was_seqpoint <- true;
      end 
    let emit_delay_mark cgbuf nm = 
         let lab = generate_code_label() in 
         if verbose then dprintf2 " --> generated code label %s with name %s\n" (string_of_code_label lab) nm;
         Mark lab
    let here cgbuf = ResizeArray.length cgbuf.codebuf
    let set_code_label_pc cgbuf lab pc = 
         if verbose then dprintf2 " --> setting label %s to pc %d\n" (string_of_code_label lab) pc;
        if Hashtbl.mem cgbuf.code_labels lab then warning(Error(sprintf "internal error: two values for given for label %s" (string_of_code_label lab),cgbuf.m));
        Hashtbl.add cgbuf.code_labels lab pc 
    let set_mark cgbuf (Mark lab1) (Mark lab2) = 
        let pc = 
           try Hashtbl.find cgbuf.code_labels lab2 
           with Not_found -> failwith "CG.set_mark: code label has no pc specified yet..." in 
        set_code_label_pc cgbuf lab1 pc
    let set_mark_to_here cgbuf (Mark lab) =  set_code_label_pc cgbuf lab (here cgbuf) 
    let set_stack cgbuf s = cgbuf.stack <- s; cgbuf.nstack <- List.length s

    let mark cgbuf s = 
      let res = emit_delay_mark cgbuf s in 
      set_mark_to_here cgbuf res;
      res 

    let create m (mgbuf,methodName,entrypoint_info,already_used_args,preallocated_locals) = 
      { m=m;
        last_was_seqpoint=false;
        maxstack=0;
        mgbuf=mgbuf;
        locals=ResizeArray.create 10;
        codebuf=ResizeArray.create 200 ;
        exn_specs=ResizeArray.create 10;
        stack=[];
        nstack=0;
        code_labels=Hashtbl.create 10 ;
        methodName=methodName;
        entrypoint_info=entrypoint_info;
        already_used_args=already_used_args;
        preallocated_locals=preallocated_locals} 

end
open CG

(*--------------------------------------------------------------------------
!* Generate debugging marks 
 *-------------------------------------------------------------------------- *)

let gen_opt_range cenv m = if cenv.debug then gen_range m else None

(* These marks are used for closure constructors.  We avoid marking the whole expression. *)
let gen_closure_ctor_range cenv m = None (* gen_opt_range cenv (start_range_of_range m)  *)


(*--------------------------------------------------------------------------
!* Compile constants 
 *-------------------------------------------------------------------------- *)

let gen_string cenv cgbuf m s = 
  CG.emit_instrs cgbuf [Push cenv.g.ilg.typ_String] [ I_ldstr s ]

let gen_bytearray cenv cgbuf eenv m s = 
  let byte_array = mk_sdarray_ty cenv.g.ilg.typ_uint8 in 
  let len = Bytes.length s in 
  if len = 0 then 
    CG.emit_instrs cgbuf [Push cenv.g.ilg.typ_int32; Push byte_array; Pop] [ mk_ldc_i32 (Int32.of_int 0);  I_newarr (sdshape,cenv.g.ilg.typ_uint8); ]
  else 
    let nm = String.concat "_" ["bytes";string_of_int(new_uniq ()); 
                                 string_of_int (start_line_of_range m); 
                                 string_of_int (start_col_of_range m)] in 
    let vtspec = cgbuf.mgbuf.vtgenerator (eenv.cloc,len) in 
    let fname = mk_private_name ("field"^string_of_int(new_uniq())) in 
    let fty = Type_value vtspec in 
    let fdef = mk_static_fdef (fname,fty, None, Some s, MemAccess_assembly) in 
    let fspec = mk_fspec_in_boxed_tspec (mk_tspec_for_cloc eenv.cloc,fname, fty) in
    static_field_def_counter();
    insert_fdef cgbuf.mgbuf (tref_of_fref (fref_of_fspec fspec)) fdef; 
    CG.emit_instrs cgbuf 
      [ Push cenv.g.ilg.typ_int32; Pop; Push byte_array; 
        Push byte_array; Push cenv.g.ilg.typ_RuntimeFieldHandle; 
        Pop; Pop]
      [ mk_ldc_i32 (Int32.of_int len);
        I_newarr (sdshape,cenv.g.ilg.typ_uint8); 
        i_dup;
        I_ldtoken (Token_field fspec); 
        mk_normal_call (mspec_RuntimeHelpers_InitializeArray cenv.g.ilg) ]


(*--------------------------------------------------------------------------
!* We normally generate in the context of a "what to do next" continuation
 *-------------------------------------------------------------------------- *)

type sequel = 
  | EndFilter (* integer says which local to save result in *)
  | LeaveHandler of (bool (* finally? *) * int * mark)  (* integer says which local to save result in *)
  | Br of mark
  | CmpThenBrOrContinue of pushpop list * Il.instr
  | Continue
  | DiscardThen of sequel
  | Return
  | EndLocalScope of sequel * mark (* used at end of 'let' and 'let rec' blocks to get tail recursive setting of end-of-scope marks *)
(*
  | DiscardAndBr of mark
  | discardAndReturnVoid
*)
  | ReturnVoid

let discard = DiscardThen Continue
let discardAndReturnVoid = DiscardThen ReturnVoid


(*-------------------------------------------------------------------------
!* This is the main code generation routine.  It is used to generate 
 * the bodies of methods in a couple of places
 *------------------------------------------------------------------------- *)
 
let codegen_then cenv mgbuf (entrypoint_info,methodName,eenv,already_used_args,preallocated_locals,codegenf,m) = 
  let cgbuf = CG.create m (mgbuf,methodName,entrypoint_info,already_used_args,preallocated_locals) in 
  let start = CG.mark cgbuf "mstart" in 
  let innerVals = map (fun (v,kind) -> (v,(kind,start))) entrypoint_info in
  (* Make sure we always generate at least one sequence point *)
  CG.emit_seqpoint cenv cgbuf m;

  (* Call the given code generator *)
  codegenf cgbuf {eenv with withinSEH=false;
                            liveLocals=Imap.empty();  
                            innerVals = innerVals};

  let finish = CG.mark cgbuf "mfinish" in 
  
  let locals = ResizeArray.to_list cgbuf.locals in 
  let local_debug_specs = 
    locals
    |> list_mapi (fun i (nms,_) -> list_map (fun nm -> (i,nm)) nms)
    |> List.concat
    |> List.map (fun (i,(nm,(start,finish))) -> 
        { locRange=(code_label_of_mark start, code_label_of_mark finish);
          locInfos= [{ localNum=i; localName=nm }] }) in 

  if debug && length locals > 64 then dprint_endline ("Note, method "^methodName^" has "^string_of_int (length locals)^" locals (even before conversion from ILX to IL)."); 

  (map (snd >> Il.mk_local) locals, 
   cgbuf.maxstack,
   cgbuf.code_labels,
   ResizeArray.to_array cgbuf.codebuf,
   ResizeArray.to_list cgbuf.exn_specs,
   local_debug_specs)

let codegen_method cenv mgbuf (entrypoint_info,methodName,eenv,already_used_args,preallocated_locals,codegenf,m) = 
  (* Codegen the method. @todo: change this to generate the AbsIL code tree directly... *)
  if verbose then dprintf1 "----------\ncodegen method %s\n" methodName;
  let locals,maxstack,code_labels,instrs,exns,local_debug_specs = 
    codegen_then cenv mgbuf (entrypoint_info,methodName,eenv,already_used_args,preallocated_locals,codegenf,m) in
  if verbose then dprintf1 "----------\nbuilding AbsIL structured code for method %s\n" methodName;
  let dump() = 
     instrs |> Array.iteri (fun i instr -> dprintf4 "%s: %d: %a\n" methodName i Ilprint.output_instr instr);
  in 
  if verbose then dump();
  (* Build an Abstract IL code tree from the raw information *)
  let lab2pc lbl = try Hashtbl.find code_labels lbl with Not_found -> errorR(Error("label "^string_of_code_label lbl^" not found",m)); dump(); 676767 in 
  let code = build_code methodName lab2pc instrs exns local_debug_specs in 
  if verbose then dprintf1 "----------\nchecking AbsIL structured code for method %s\n" methodName;
  let code = check_code code in 

  (* Build an Abstract IL method *)
  mk_ilmbody (true,locals,maxstack,code, gen_opt_range cenv m )


let start_local_scope nm cgbuf =
  let start_scope = CG.mark cgbuf ("start_"^nm)  in
  let end_scope = CG.emit_delay_mark cgbuf ("end_"^nm) in 
  start_scope,end_scope
  
let local_scope nm cgbuf (f : (mark * mark) -> 'a) : 'a =
  let start_scope,end_scope as scopeMarks = start_local_scope nm cgbuf in 
  let res = f scopeMarks in 
  CG.set_mark_to_here cgbuf end_scope;
  res


(*-------------------------------------------------------------------------
!* Generate expressions
 *------------------------------------------------------------------------- *)

let rec gen_expr cenv cgbuf eenv expr sequel =
  if verbose then dprintf4 "gen_expr@%a, #stack = %d, sequel = %s\n" output_range (range_of_expr expr) (length cgbuf.stack) (string_of_sequel sequel);
  (* if verbose then dprintf5 "gen_expr@%a, #stack = %d, expr = %s, sequel = %s\n" output_range (range_of_expr expr) (length cgbuf.stack) (showL (exprL expr)) (string_of_sequel sequel); *)
  let expr =  strip_expr expr in 
  (* This is the list of expression types which do not in and of themselves *)
  (* get sequence points, because they have not side effects *)
  (* except via their sub-expressions. *)
  begin match expr with 
  | TExpr_obj _ | TExpr_match _ | TExpr_const _ | TExpr_val _ | TExpr_app(_,_,_,[],_)
  | TExpr_lambda _ | TExpr_tlambda _ 
  | TExpr_let _ | TExpr_letrec _ | TExpr_static_optimization _ 
  | TExpr_op((TOp_try_catch | TOp_try_finally | TOp_lval_op(LGetAddr,_) | TOp_uconstr _ | TOp_recd _ | TOp_exnconstr _ |
              TOp_tuple_field_get _ | TOp_constr_field_get _ | TOp_constr_tag_get _ | TOp_exnconstr_field_get _ |
              TOp_for _ | TOp_array | TOp_field_get _ | TOp_field_get_addr _ | TOp_tuple | TOp_while | TOp_coerce | TOp_get_ref_lval),_,_,_)
  | TExpr_seq _  |  TExpr_link _ -> ()
  | _ -> CG.emit_seqpoint cenv cgbuf (range_of_expr expr);
  end;
  match expr with 
  | TExpr_const(c,m,ty) -> 
      gen_const cenv cgbuf eenv (c,m,ty) sequel
  | TExpr_match (exprm,tree,targets,m,ty,_) -> 
      gen_match cenv cgbuf eenv (exprm,tree,targets,m,ty) sequel
  | TExpr_seq(e1,e2,dir,m) ->  
      gen_seq cenv cgbuf eenv (e1,e2,dir,m) sequel
  | TExpr_letrec (binds,body,m,_)  -> 
      gen_letrec cenv cgbuf eenv (binds,body,m) sequel
  | TExpr_let (bind,body,m,_)  -> 
     (* This case implemented here to get a guaranteed tailcall *)
     let _,end_scope as scopeMarks = start_local_scope "let" cgbuf in 
     let eenv = alloc_vals_for_binds cenv cgbuf scopeMarks eenv [bind] in
     gen_bind cenv cgbuf eenv bind;
     gen_expr cenv cgbuf eenv body (EndLocalScope(sequel,end_scope)) 

  | TExpr_lambda _  | TExpr_tlambda _  -> 
      gen_lambda cenv cgbuf eenv false None expr sequel
  | TExpr_app(f,fty,tyargs,args,m) -> 
      gen_app cenv cgbuf eenv (f,fty,tyargs,args,m) sequel
  | TExpr_val(v,flags,m) -> 
      gen_vref_get cenv cgbuf eenv (v,m) sequel
  | TExpr_op(op,tyargs,args,m) -> 
      begin match op,args,tyargs with 
      | TOp_exnconstr(c),_,_      -> 
          gen_exnconstr cenv cgbuf eenv (c,args,m) sequel
      | TOp_uconstr(c),_,_        -> 
          gen_constr cenv cgbuf eenv (c,tyargs,args,m) sequel
      | TOp_recd(ctor,tycon),_,_ -> 
          gen_recd cenv cgbuf eenv ctor (tycon,tyargs,args,m) sequel
      | TOp_tuple_field_get(n),[e],_ -> 
          gen_tuple_field_get cenv cgbuf eenv (e,tyargs,n,m) sequel
      | TOp_exnconstr_field_get(constr,n),[e],_ -> 
          gen_exnconstr_field_get cenv cgbuf eenv (e,constr,n,m) sequel
      | TOp_constr_field_get(constr,n),[e],_ -> 
          gen_constr_field_get cenv cgbuf eenv (e,constr,tyargs,n,m) sequel
      | TOp_constr_tag_get(constr),[e],_ -> 
          gen_constr_tag_get cenv cgbuf eenv (e,constr,tyargs,m) sequel
      | TOp_exnconstr_field_set(constr,n),[e;e2],_ -> 
          gen_exnconstr_field_set cenv cgbuf eenv (e,constr,n,e2,m) sequel 
      | TOp_constr_field_set(constr,n),[e;e2],_ -> 
          gen_constr_field_set cenv cgbuf eenv (e,constr,tyargs,n,e2,m) sequel
      | TOp_field_get(f),[e],_ -> 
         gen_recd_field_get cenv cgbuf eenv (e,f,tyargs,m) sequel
      | TOp_field_get(f),[],_ -> 
         gen_static_field_get cenv cgbuf eenv (f,tyargs,m) sequel
      | TOp_field_get_addr(f),[e],_ -> 
         gen_recd_field_get_addr cenv cgbuf eenv (e,f,tyargs,m) sequel
      | TOp_field_get_addr(f),[],_ -> 
         gen_static_field_get_addr cenv cgbuf eenv (f,tyargs,m) sequel
      | TOp_field_set(f),[e1;e2],_ -> 
         gen_recd_field_set cenv cgbuf eenv (e1,f,tyargs,e2,m) sequel
      | TOp_field_set(f),[e2],_ -> 
         gen_static_field_set cenv cgbuf eenv (f,tyargs,e2,m) sequel
      | TOp_tuple,_,_ -> 
         gen_tuple cenv cgbuf eenv (args,tyargs,m) sequel
      | TOp_asm(code,rtys),_,_ ->  
         gen_asm cenv cgbuf eenv (code,tyargs,args,rtys,m) sequel 
      | TOp_while,[TExpr_lambda(_,_,[_],e1,_,_,_);TExpr_lambda(_,_,[_],e2,_,_,_)],[]  -> 
         gen_while cenv cgbuf eenv (e1,e2,m) sequel 
      | TOp_for(dir),[TExpr_lambda(_,_,[_],e1,_,_,_);TExpr_lambda(_,_,[_],e2,_,_,_);TExpr_lambda(_,_,[v],e3,_,_,_)],[]  -> 
         gen_for cenv cgbuf eenv (v,e1,dir,e2,e3,m) sequel
      | TOp_try_finally,[TExpr_lambda(_,_,[_],e1,_,_,_); TExpr_lambda(_,_,[_],e2,_,_,_)],[resty] -> 
         gen_try_finally cenv cgbuf eenv (e1,e2,m,resty) sequel
      | TOp_try_catch,[TExpr_lambda(_,_,[_],e1,_,_,_); TExpr_lambda(_,_,[vf],ef,_,_,_);TExpr_lambda(_,_,[vh],eh,_,_,_)],[resty] -> 
         gen_try_catch cenv cgbuf eenv (e1,vf,ef,vh,eh,m,resty) sequel
      | TOp_ilcall(meth,enclTypeArgs,methTypeArgs,rtys),args,[] -> 
         gen_ilcall cenv cgbuf eenv (meth,enclTypeArgs,methTypeArgs,args,rtys,m) sequel
      | TOp_get_ref_lval,[e],[ty]       -> gen_get_ref_lval cenv cgbuf eenv (e,ty,m) sequel
      | TOp_coerce,[e],[tgty;srcty]    -> gen_coerce cenv cgbuf eenv (e,tgty,m,srcty) sequel
      | TOp_trait_call(ss),args,methTypeArgs -> gen_trait_call cenv cgbuf eenv (ss,methTypeArgs, args, m) sequel
      | TOp_lval_op(LSet,v),[e],[]      -> gen_val_set cenv cgbuf eenv (v,e,m) sequel
      | TOp_lval_op(LByrefGet,v),[],[]  -> gen_byref_get cenv cgbuf eenv (v,m) sequel
      | TOp_lval_op(LByrefSet,v),[e],[] -> gen_byref_set cenv cgbuf eenv (v,e,m) sequel
      | TOp_lval_op(LGetAddr,v),[],[]   -> gen_val_geta cenv cgbuf eenv (v,m) sequel
      | TOp_array,elems,[argty] ->  gen_array cenv cgbuf eenv (elems,argty,m) sequel
      | TOp_bytes bytes,[],[] -> gen_bytearray cenv cgbuf eenv m bytes
      | _ -> error(Error("unexpected operator node expression",range_of_expr expr))
     end 
  | TExpr_static_optimization(constraints,e2,e3,m) -> 
      gen_static_optimization cenv cgbuf eenv (constraints,e2,e3,m) sequel
  | TExpr_obj(uniq,typ,_,_,[meth],[],m,_) when is_delegate_typ typ -> 
      gen_delegate_expr cenv cgbuf eenv expr (meth,m) sequel
  | TExpr_obj(uniq,typ,basev,basecall,overrides,iimpls,m,_) -> 
      gen_obj_expr cenv cgbuf eenv expr (typ,basev,basecall,overrides,iimpls,m)  sequel

  | TExpr_quote(raw,ast,m,ty) -> 
      gen_quote cenv cgbuf eenv (raw,ast,m,ty) sequel
  | TExpr_hole(m,_) -> error(Error("unexpected TExpr_hole",m))

  | TExpr_link _ -> failwith "unexpected reclink"
  | TExpr_tchoose (_,_,m) -> error(Error("unexpected TExpr_tchoose",m))

and gen_exprs cenv cgbuf eenv es = iter (fun e -> gen_expr cenv cgbuf eenv e Continue) es

and codegen_method_for_expr cenv mgbuf (entrypoint_info,methodName,eenv,already_used_args,preallocated_locals,expr0,sequel0) = 
  codegen_method cenv mgbuf (entrypoint_info,methodName,eenv,already_used_args,preallocated_locals,
                             (fun cgbuf eenv -> gen_expr cenv cgbuf eenv expr0 sequel0),
                             (range_of_expr expr0))



(*--------------------------------------------------------------------------
!* Generate sequels
 *-------------------------------------------------------------------------- *)

(* does the sequel discard its result, and if so what does it do next? *)
and sequel_after_discard sequel = 
  match sequel with 
   | DiscardThen sequel -> Some(sequel)
   | EndLocalScope(sq,mark) -> sequel_after_discard sq |> Option.map (fun sq -> EndLocalScope(sq,mark))
   | _ -> None
and sequel_ignoring_end_scopes_and_discard sequel =
  let sequel = sequel_ignore_end_scopes sequel in 
  match sequel_after_discard sequel with 
  | Some sq -> sq
  | None ->  sequel 
and sequel_ignore_end_scopes  sequel = 
  match sequel with 
  | EndLocalScope(sq,m) -> sequel_ignore_end_scopes sq
  | sq -> sq
(* commit any 'EndLocalScope' nodes in the sequel and return the residue *)
and gen_sequel_end_scopes cgbuf sequel =
  match sequel with 
  | EndLocalScope(sq,m) -> CG.set_mark_to_here cgbuf m; gen_sequel_end_scopes cgbuf sq
  | sq -> ()

and string_of_sequel sequel =
  match sequel with
  | Continue -> "continue"
  | DiscardThen sequel -> "discard; "^string_of_sequel sequel
  | ReturnVoid -> "ReturnVoid"
  | CmpThenBrOrContinue(pushpops,bri) -> "CmpThenBrOrContinue"
  | Return -> "Return"
  | EndLocalScope (sq,Mark k) -> "EndLocalScope("^string_of_sequel sq^","^string_of_code_label k^")"
  | Br (Mark x) -> sprintf "Br L%s" (string_of_code_label x)
  | LeaveHandler (finlly, where_to_save_result,x) -> "LeaveHandler"
  | EndFilter -> "EndFilter"

and gen_sequel cenv cloc cgbuf sequel =
  let sq = sequel_ignore_end_scopes sequel in 
  if verbose then dprint_endline ("gen_sequel:" ^ string_of_sequel sequel);
  (match sq with 
  | Continue -> ()
  | DiscardThen sq -> 
      CG.emit_instr cgbuf [Pop] (i_pop);
      gen_sequel cenv cloc cgbuf sq 
  | ReturnVoid ->
      CG.emit_instr cgbuf [] I_ret 
  | CmpThenBrOrContinue(pushpops,bri) ->
      CG.emit_instr cgbuf pushpops bri
  | Return -> 
      CG.emit_instr cgbuf [Pop] I_ret 
  | EndLocalScope _ -> failwith "EndLocalScope unexpected"
  | Br x -> 
      CG.emit_instr cgbuf [] (I_br(code_label_of_mark x))  
  | LeaveHandler (finlly, where_to_save_result,x) ->
      if finlly then 
        CG.emit_instr cgbuf [Pop] (i_pop) 
      else
        gen_set_local cgbuf where_to_save_result;
      CG.emit_instr cgbuf [] (if finlly then I_endfinally else I_leave(code_label_of_mark x))
  | EndFilter ->
      CG.emit_instr cgbuf [Pop] I_endfilter
  );
  gen_sequel_end_scopes cgbuf sequel;
  if verbose then dprint_endline ("gen_sequel: done");


(*--------------------------------------------------------------------------
!* Generate constants
 *-------------------------------------------------------------------------- *)

and gen_const cenv cgbuf eenv (c,m,ty) sequel =
  let il_ty = gen_type m cenv.g eenv.tyenv ty in 
  (* Check if we need to generate the value at all! *)
  begin match sequel_after_discard sequel with 
  | None -> 
    if verbose then dprint_endline ("gen_const: generating ");
    begin match try_elim_bigint_bignum_constants cenv.g m c with 
    | Some e -> gen_expr cenv cgbuf eenv e Continue
    | None ->
        match c with 
        | TConst_bool b -> CG.emit_instr cgbuf [Push cenv.g.ilg.typ_bool] (mk_ldc_i32 (Int32.of_int (if b then 1 else 0)))

        | TConst_int8(i) -> CG.emit_instr cgbuf [Push il_ty] (mk_ldc_i32 (Nums.i8_to_i32 i))
        | TConst_int16(i) -> CG.emit_instr cgbuf [Push il_ty] (mk_ldc_i32 (Nums.i16_to_i32 i))
        | TConst_int32(i) -> CG.emit_instr cgbuf [Push il_ty] (mk_ldc_i32 i)
        | TConst_int64(i) -> CG.emit_instr cgbuf [Push il_ty] (mk_ldc_i64 i)
        | TConst_nativeint(i) -> CG.emit_instrs cgbuf [Push il_ty] [mk_ldc_i64 i; I_arith (AI_conv DT_I) ]
        | TConst_uint8(i) -> CG.emit_instr cgbuf [Push il_ty] (mk_ldc_i32 (Nums.u8_to_i32 i))
        | TConst_uint16(i) -> CG.emit_instr cgbuf [Push il_ty] (mk_ldc_i32 (Nums.u16_to_i32 i))
        | TConst_uint32(i) -> CG.emit_instr cgbuf [Push il_ty] (mk_ldc_i32 (Nums.u32_to_i32 i))
        | TConst_uint64(i) -> CG.emit_instr cgbuf [Push il_ty] (mk_ldc_i64 (Nums.u64_to_i64 i))
        | TConst_unativeint(i) -> CG.emit_instrs cgbuf [Push il_ty] [mk_ldc_i64 (Nums.u64_to_i64 i); I_arith (AI_conv DT_U) ]
        | TConst_float(f) -> CG.emit_instr cgbuf [Push il_ty] ( I_arith (AI_ldc (DT_R8,NUM_R8 f)) )
        | TConst_float32(f) -> CG.emit_instr cgbuf [Push il_ty] ( I_arith (AI_ldc (DT_R4,NUM_R4 f)) )
        | TConst_char(c) -> CG.emit_instr cgbuf [Push il_ty] ( mk_ldc_i32 (Nums.u16_to_i32 (Nums.unichar_to_u16 c)))
        | TConst_string(s) -> gen_string cenv cgbuf m s
        | TConst_unit -> gen_unit cenv cgbuf
        | TConst_default -> gen_ilzero cenv cgbuf eenv (ty,m) 
        | TConst_bignum _ | TConst_bigint _ | TConst_decimal _ -> failwith "unreachable"
    end;
    gen_sequel cenv eenv.cloc cgbuf sequel
  | Some sq -> 
    if verbose then dprint_endline ("gen_const: skipping");
  (* Even if we didn't need to generate the value then maybe we still have to branch or return *)
    gen_sequel cenv eenv.cloc cgbuf sq
  end

and gen_unit cenv cgbuf  = CG.emit_instr cgbuf [Push cenv.g.ilg.typ_Object] i_ldnull

and gen_unit_then_sequel cenv cloc cgbuf sequel =
  if verbose then dprint_endline ("gen_unit_then_sequel:");  
  match sequel_after_discard sequel with 
  | Some(sq) -> gen_sequel cenv cloc cgbuf sq
  | None -> gen_unit cenv cgbuf; gen_sequel cenv cloc cgbuf sequel


(*--------------------------------------------------------------------------
!* Generate simple data-related constructs
 *-------------------------------------------------------------------------- *)

and gen_tuple cenv cgbuf eenv (args,argtys,m) sequel =
  if verbose then dprint_endline ("gen_tuple:");
  gen_recd cenv cgbuf eenv RecdExpr (compiled_mk_tuple cenv.g (argtys,args,m)) sequel


and gen_tuple_field_get cenv cgbuf eenv (e,tys,n,m) sequel =
  gen_recd_field_get cenv cgbuf eenv (compiled_get_tuple_field cenv.g (e,tys,n,m)) sequel

and gen_exnconstr cenv cgbuf eenv (c,args,m) sequel =
  gen_exprs cenv cgbuf eenv args;
  let typ = gen_exn_type m cenv.g eenv.tyenv c in
  let mspec = 
    mk_ctor_mspec (tref_of_typ typ, AsObject,
                   map (formal_typ_of_rfield >> gen_type m cenv.g eenv.tyenv) (instance_rfields_of_tycon (strip_eqns_from_ecref c)),[]) in 
  let mspec = (* intern_mspec cenv.manager *) mspec in
  CG.emit_instr cgbuf
    (replicate (length args) Pop @ [ Push typ])
    (mk_normal_newobj mspec) ;
  gen_sequel cenv eenv.cloc cgbuf sequel

and gen_constr cenv cgbuf eenv  (c,tyargs,args,m) sequel =
  if verbose then dprint_endline ("gen_constr");  
  gen_exprs cenv cgbuf eenv args;
  let cuspec,idx = gen_cuspec_idx m cenv eenv.tyenv c tyargs in 
  CG.emit_instr cgbuf (replicate (length args) Pop @ [ Push (objtype_of_cuspec cuspec)]) (mk_IlxInstr (EI_newdata (cuspec,idx)));
  gen_sequel cenv eenv.cloc cgbuf sequel

and gen_recd cenv cgbuf eenv ctorInfo (tcref,argtys,args,m) sequel =
  let typ = gen_tcref_tyapp m cenv.g eenv.tyenv tcref argtys in 

  (* Filter out fields with default initialization *)
  let relevantFields = 
      instance_rfields_of_tcref tcref 
      |> List.filter (zero_init_of_rfield >> not) in

  match ctorInfo with 
  | RecdExprIsObjInit -> 
      if verbose then dprint_endline ("gen_recd: class constructor");  
      iter2 
          (fun e f -> 
              CG.emit_instr cgbuf [Push typ] ldarg_0; 
              gen_expr cenv cgbuf eenv e Continue;
              gen_field_store false cenv cgbuf eenv (mk_rfref tcref (name_of_rfield f),argtys,m) discard) 
          args
          relevantFields;
      (* Object construction doesn't generate a true value. *)
      (* Object constructions will always just get thrown away so this is safe *)      
      gen_sequel cenv eenv.cloc cgbuf sequel

  | RecdExpr -> 
      if verbose then dprintf1 "gen_recd: normal record, #args = %d\n" (length args);  
      gen_exprs cenv cgbuf eenv args;
          (* generate a reference to the record constructor *)
      let tyenvinner = tyenv_for_tcref tcref in   
      if verbose then dprintf1 "gen_recd: call, #args = %d" (length args);  
      CG.emit_instr cgbuf (replicate (length args) Pop @ [ Push typ; ])
        (mk_normal_newobj 
           (mk_ctor_mspec_for_typ (typ,map (fun f -> gen_type m cenv.g tyenvinner f.rfield_type) relevantFields)));
      gen_sequel cenv eenv.cloc cgbuf sequel

and gen_array cenv cgbuf eenv (elems,argty,m) sequel =
  let argty' = gen_type m cenv.g eenv.tyenv argty in        
  let arrty = mk_array_ty_old (sdshape,argty') in 
  
  CG.emit_instrs cgbuf [Push arrty] [ I_arith (AI_ldc (DT_I4,NUM_I4 (int_to_i32 (List.length elems)))); mk_IlxInstr (EI_newarr_erasable (sdshape,argty')) ];
  list_iteri
    (fun i e ->             
      CG.emit_instrs cgbuf [Push arrty; Push cenv.g.ilg.typ_int32] [ i_dup; 
                                                    I_arith (AI_ldc (DT_I4,NUM_I4 (int_to_i32 i))) ];
      gen_expr cenv cgbuf eenv e Continue;          
      CG.emit_instr cgbuf [Pop; Pop; Pop] (mk_IlxInstr (EI_stelem_any_erasable (sdshape,argty')))) 
    elems;
  gen_sequel cenv eenv.cloc cgbuf sequel

and gen_coerce cenv cgbuf eenv (e,tgty,m,srcty) sequel = 
  (* Is this an upcast? *)
  if Typrelns.type_definitely_subsumes_type_no_coercion 0 cenv.g cenv.amap m tgty srcty &&
     (* Do an extra check - should not be needed *)
     Typrelns.type_feasibly_subsumes_type 0 cenv.g cenv.amap m tgty Typrelns.NoCoerce srcty then
     begin 
       (* The .NET IL doesn't always support implict subsumption for interface types, e.g. at stack merge points *)
       (* Hence be conservative here and always cast explicitly. *)
       if (is_interface_typ tgty) then (
           gen_expr cenv cgbuf eenv e Continue;
           let il_toty = gen_type m cenv.g eenv.tyenv tgty in 
           CG.emit_instrs cgbuf [Pop; Push il_toty] [ I_unbox_any il_toty;  ];
           gen_sequel cenv eenv.cloc cgbuf sequel
       ) else (
           gen_expr cenv cgbuf eenv e sequel;
       )
     end       
  else  begin
    gen_expr cenv cgbuf eenv e Continue;          
    let il_fromty = gen_type m cenv.g eenv.tyenv srcty in 
    CG.emit_instrs cgbuf [Pop; Push cenv.g.ilg.typ_Object] [ I_box il_fromty;  ];
    let il_toty = gen_type m cenv.g eenv.tyenv tgty in 
    CG.emit_instrs cgbuf [Pop; Push il_toty] [ I_unbox_any il_toty;  ];
    gen_sequel cenv eenv.cloc cgbuf sequel
  end


and gen_exnconstr_field_get cenv cgbuf eenv (e,constr,n,m) sequel =
  gen_expr cenv cgbuf eenv e Continue;
  let finfo = gen_exn_finfo m cenv eenv.tyenv (strip_eqns_from_ecref constr) n in 
  let exn_typ = gen_exn_type m cenv.g eenv.tyenv constr in 
  let fspec = mk_fspec_in_typ (exn_typ,fst finfo,snd finfo) in 
  let rty = actual_typ_of_fspec fspec in 
  CG.emit_instrs cgbuf [Pop; Push rty] [ I_castclass exn_typ; mk_normal_ldfld fspec ];
  gen_sequel cenv eenv.cloc cgbuf sequel

and gen_constr_field_get cenv cgbuf eenv (e,constr,tyargs,n,m) sequel =
  gen_expr cenv cgbuf eenv e Continue;
  let cuspec,idx = gen_cuspec_idx m cenv eenv.tyenv constr tyargs in 
          (* ANALYSIS: don't use castdata where we've already done a typetest *)
  let fty = actual_typ_of_cuspec_field cuspec idx n in 
  CG.emit_instrs cgbuf [Pop; Push fty]
    [ mk_IlxInstr (EI_castdata(false,cuspec,idx));
      mk_IlxInstr (EI_lddata(cuspec,idx,n)) ];
  gen_sequel cenv eenv.cloc cgbuf sequel

and gen_constr_tag_get cenv cgbuf eenv (e,tycon,tyargs,m) sequel =
  gen_expr cenv cgbuf eenv e Continue;
  let cuspec = gen_cuspec m cenv eenv.tyenv tycon tyargs in 
  CG.emit_instrs cgbuf [Pop; Push cenv.g.ilg.typ_int32] [ mk_IlxInstr (EI_lddatatag(cuspec)) ];
  gen_sequel cenv eenv.cloc cgbuf sequel

and gen_exnconstr_field_set cenv cgbuf eenv (e,constr,n,e2,m) sequel = 
  gen_expr cenv cgbuf eenv e Continue;
  let finfo = gen_exn_finfo m cenv eenv.tyenv (strip_eqns_from_ecref constr) n in 
  let typ = gen_exn_type m cenv.g eenv.tyenv constr in 
  CG.emit_instrs cgbuf [] [ I_castclass typ ];
  gen_expr cenv cgbuf eenv e2 Continue;
  CG.emit_instr cgbuf [Pop; Pop] (mk_normal_stfld(mk_fspec_in_typ (typ,fst finfo,snd finfo)));
  gen_unit_then_sequel cenv eenv.cloc cgbuf sequel

and gen_constr_field_set cenv cgbuf eenv (e,constr,tyargs,n,e2,m) sequel = 
  gen_expr cenv cgbuf eenv e Continue;
  let cuspec,idx = gen_cuspec_idx m cenv eenv.tyenv constr tyargs in 
  CG.emit_instr cgbuf [Pop; Push (objtype_of_cuspec cuspec) ] (mk_IlxInstr (EI_castdata(false,cuspec,idx)));
  gen_expr cenv cgbuf eenv e2 Continue;
  CG.emit_instr cgbuf [Pop; Pop] (mk_IlxInstr (EI_stdata(cuspec,idx,n)) );
  gen_unit_then_sequel cenv eenv.cloc cgbuf sequel

and gen_recd_field_get_addr cenv cgbuf eenv (e,f,tyargs,m) sequel = (* follows gen_get_ref_lval code *)
  gen_expr cenv cgbuf eenv e Continue;
  let fref = gen_recdfield_ref m cenv eenv.tyenv f tyargs in 
  CG.emit_instrs cgbuf [Pop; Push (Type_byref (actual_typ_of_fspec fref))] [ I_ldflda fref ] ;
  gen_sequel cenv eenv.cloc cgbuf sequel
         
and gen_static_field_get_addr cenv cgbuf eenv (f,tyargs,m) sequel = (* follows gen_get_ref_lval code *)
  let fspec = gen_recdfield_ref m cenv eenv.tyenv f tyargs in 
  CG.emit_instrs cgbuf [Push (Type_byref (actual_typ_of_fspec fspec))] [ I_ldsflda fspec ] ;
  gen_sequel cenv eenv.cloc cgbuf sequel
         
and gen_recd_field_get cenv cgbuf eenv (e,f,tyargs,m) sequel =
  if verbose then dprint_endline ("gen_recd_field_get");    
  gen_expr cenv cgbuf eenv e Continue;
  gen_field_get false cenv cgbuf eenv (f,tyargs,m);
  gen_sequel cenv eenv.cloc cgbuf sequel
  
and gen_recd_field_set cenv cgbuf eenv (e1,f,tyargs,e2,m) sequel =
  gen_expr cenv cgbuf eenv e1 Continue;
  gen_expr cenv cgbuf eenv e2 Continue;
  gen_field_store false cenv cgbuf eenv (f,tyargs,m) sequel
  
and gen_static_field_get cenv cgbuf eenv (f,tyargs,m) sequel =
  gen_field_get true cenv cgbuf eenv (f,tyargs,m);
  gen_sequel cenv eenv.cloc cgbuf sequel
  
and gen_static_field_set cenv cgbuf eenv (f,tyargs,e2,m) sequel =
  gen_expr cenv cgbuf eenv e2 Continue;
  gen_field_store true cenv cgbuf eenv (f,tyargs,m) sequel

and mk_field_mspec isStatic = 
    (if isStatic then mk_static_nongeneric_mspec_in_typ else mk_nongeneric_instance_mspec_in_typ)
and mk_field_pops isStatic pops = if isStatic then pops else Pop::pops


and gen_field_get isStatic cenv cgbuf eenv (rfref,tyargs,m) =
    let fspec = gen_recdfield_ref m cenv eenv.tyenv rfref tyargs in 
    let tycon,fld = deref_rfield rfref in
    if use_genuine_field tycon fld then
        CG.emit_instrs cgbuf (mk_field_pops isStatic [ Push (actual_typ_of_fspec fspec)]) [ if isStatic then mk_normal_ldsfld fspec else mk_normal_ldfld fspec ] 
    else
        let mspec = mk_field_mspec isStatic (enclosing_typ_of_fspec fspec,"get_"^(snd (deref_rfield rfref)).rfield_id.idText, [], formal_typ_of_fspec fspec) in 
        CG.emit_instr cgbuf (mk_field_pops isStatic [Push (actual_typ_of_fspec fspec)]) (mk_normal_call mspec)

  
and gen_field_store isStatic cenv cgbuf eenv (rfref,tyargs,m) sequel =
    let fspec = gen_recdfield_ref m cenv eenv.tyenv rfref tyargs in
    let tycon, fld = deref_rfield rfref in 
    if fld.rfield_mutable && not (use_genuine_field tycon fld) then
        let mspec = mk_field_mspec isStatic (enclosing_typ_of_fspec fspec, "set_"^fld.rfield_id.idText, [formal_typ_of_fspec fspec],Type_void) in 
        let mspec = (* intern_mspec cenv.manager *) mspec in
        CG.emit_instr cgbuf (mk_field_pops isStatic [Pop]) (mk_normal_call mspec)
    else
        (* Within assemblies we do generate some set-field operations *)
        (* for immutable fields even when resolving recursive bindings. *)
        (* However we do not generate "set" properties for these. *)
        (* Hence we just set the field directly in this case. *)
        CG.emit_instr cgbuf (mk_field_pops isStatic [Pop]) (if isStatic then mk_normal_stsfld fspec else mk_normal_stfld fspec); 
    gen_unit_then_sequel cenv eenv.cloc cgbuf sequel

(*--------------------------------------------------------------------------
!* Codegen "unit" as "void" at interop points.  Discard unit 
 * values on return from methods, also ignore unit-typed 
 * values in arguments of methods.
 *-------------------------------------------------------------------------- *)

and gen_expr_discarding_unit cenv cgbuf eenv fargty expr sequel =
  if verbose then dprint_endline ("gen_expr_discarding_unit");    
  if is_unit_typ cenv.g fargty then 
    (gen_expr cenv cgbuf eenv expr discard;  gen_sequel cenv eenv.cloc cgbuf sequel)
  else gen_expr cenv cgbuf eenv expr sequel

and gen_untupled_expr_discarding_units cenv cgbuf eenv m fargtys expr sequel =
  let arity = length fargtys in 
  if arity = 1 then (
    if 1 <> length fargtys then error(InternalError("gen_untupled_expr_discarding_units (1)",m));
    gen_expr_discarding_unit cenv cgbuf eenv (hd fargtys) expr sequel
  ) else if is_tuple expr then (
    let es = try_dest_tuple expr in 
    if length es <> length fargtys then error(InternalError("gen_untupled_expr_discarding_units (2)",m));
    iter2 (fun x fargty -> gen_expr_discarding_unit cenv cgbuf eenv fargty x Continue) es fargtys;
    gen_sequel cenv eenv.cloc cgbuf sequel
  ) else (
    let ty = type_of_expr cenv.g expr in 
    let locv,loce = mk_compgen_local m (nng.nngApply "arg" m) ty in
    let bind = mk_bind locv expr in
    local_scope "untuple" cgbuf (fun scopeMarks ->
      let eenvinner = alloc_vals_for_binds cenv cgbuf scopeMarks eenv [bind] in 
      gen_bind cenv cgbuf eenvinner bind;
      if verbose then dprintf3 "expr = %s\nty = %s\narity = %d\n" (showL (exprL expr)) ((DebugPrint.showType ty)) arity;
      let tys = dest_tuple_typ_upto arity ty in 
      if length tys <> length fargtys then error(InternalError("gen_untupled_expr_discarding_units (3)",m));
      list_iteri
        (fun i (tup_ty,fargty) -> 
          let seq = if is_unit_typ cenv.g fargty then discard else Continue in 
          gen_tuple_field_get cenv cgbuf eenvinner (loce,tys,i,m) seq)
        (combine tys fargtys);
      gen_sequel cenv eenv.cloc cgbuf sequel
    );
  )

(*--------------------------------------------------------------------------
!* Generate calls (try to detect direct calls)
 *-------------------------------------------------------------------------- *)
 
and gen_app cenv cgbuf eenv (f,fty,tyargs,args,m) sequel =
 if verbose then dprint_endline ("gen_app:");
 match (f,tyargs,args) with 
   (* Look for tailcall to turn into branch *)
  | (TExpr_val(v,_,_),_,_) when  
    (gen_mem_assoc cenv.g.vref_eq v eenv.innerVals) && 
    let (kind,_) = gen_assoc cenv.g.vref_eq v eenv.innerVals in
    (* when branch-calling methods we must have the right type parameters *)
    begin match kind with
      | BranchCallClosure _ -> true
      | BranchCallMethod (_,_,tps,_)  ->  
          length tps = length tyargs &
          (for_all2 (fun ty tp -> type_equiv cenv.g ty (mk_typar_ty tp)) tyargs tps)
    end &&
    (* must be exact #args, ignoring tupling - we untuple if needed below *)
    (let arity_info = 
       match kind with
       | BranchCallClosure arity_info
       | BranchCallMethod (arity_info,_,_,_)  ->  arity_info in 
     length arity_info = length args
    ) &&
    (* no tailcall out of exception handler, etc. *)
    (match sequel_ignoring_end_scopes_and_discard sequel with Return | ReturnVoid -> true | _ -> false)
    -> 
        let (kind,mark) = gen_assoc cenv.g.vref_eq v eenv.innerVals in
        let ntmargs = 
          begin match kind with
          | BranchCallClosure arity_info ->
              let ntmargs = fold_right (+) arity_info 0 in 
              gen_exprs cenv cgbuf eenv args;
              ntmargs
          | BranchCallMethod (arity_info,fargtysl,tps,ntmargs)  ->
              if length fargtysl <> length arity_info then error(InternalError("length mismatch generating tailcall",m));
              if length fargtysl <> length args then error(InternalError("length mismatch generating tailcall (2)",m));
              list_iteri (fun i (fargtys,(arity,x)) -> gen_untupled_expr_discarding_units cenv cgbuf eenv m fargtys x Continue)  (combine fargtysl (combine arity_info args));
              ntmargs
          end in 
        for i = ntmargs - 1 downto 0 do 
          CG.emit_instrs cgbuf [Pop] [ I_starg (int_to_u16 (i+cgbuf.already_used_args)) ];
        done;
        CG.emit_instrs cgbuf [] [ I_br (code_label_of_mark mark) ];
        gen_sequel_end_scopes cgbuf sequel
        
  (* Similarly for '==' becomes cheap reference equality for non-value-types *)
  | (TExpr_val(v,_,_),[ty],[arg1;arg2]) when
    (cenv.g.vref_eq v cenv.g.poly_eq_inner_vref)  
    && (is_fsobjmodel_ref_typ ty || 
        (is_il_ref_typ cenv.g ty 
         && not (type_equiv cenv.g ty cenv.g.system_Object_typ) 
         && not (type_equiv cenv.g ty cenv.g.system_Value_typ)
         && not (type_equiv cenv.g ty cenv.g.system_Enum_typ)) or
        is_delegate_typ ty || 
        is_union_ty ty || 
        is_recd_ty ty || 
        is_abstract_ty ty || 
        is_tuple_ty ty) ->
      gen_expr cenv cgbuf eenv arg1 Continue;
      gen_expr cenv cgbuf eenv arg2 Continue;
      CG.emit_instr cgbuf [ Pop; Pop; Push cenv.g.ilg.typ_bool ] (I_arith AI_ceq);
      gen_sequel cenv eenv.cloc cgbuf sequel

    (* Optimize calls to top methods when given "enough" arguments. *)
  | (TExpr_val(vref,vFlags,_),_,_) when
    begin 
      let storage = storage_for_vref m vref eenv in
      match storage with   
      | Method (arity_info,vref,_,_,_,_) ->
          begin 
            let tps,argtys,_,_ = dest_top_type arity_info (type_of_vref vref) in
            length tps = length tyargs && 
            length argtys <= length args
          end
      | _ -> false
    end ->
      let storage = storage_for_vref m vref eenv in
      begin match storage with   
      | Method (arity_info,vref,mspec,_,_,_) ->
          if verbose then dprint_endline ("gen_app: Method");
          let _,fargtysl,rty,retInfo = dest_top_type arity_info (type_of_vref vref) in 
          let fargtysl = map (map fst) fargtysl in 
          let now_args,later_args = chop_at (length fargtysl) args in 
          let arty = apply_types (type_of_vref vref) (tyargs,now_args) in 
          if length fargtysl <> length now_args then error(InternalError("gen_app",m));
          let il_tyargs = gen_tyargs m cenv.g eenv.tyenv tyargs in 

          (* For instance method calls chop off some type arguments, which are already *)
          (* carried by the class.  Also work out if it's a virtual call. *)
          let numEnclTypeArgs,virtualCall,newobj,superInit,selfInit,instance,_,_ = get_member_call_info cenv.g (vref,vFlags) in
          let (il_ctyargs,il_mtyargs) = 
              if length il_tyargs  < numEnclTypeArgs then error(InternalError("length mismatch",m));
              chop_at numEnclTypeArgs il_tyargs in
          let boxity = boxity_of_typ (enclosing_typ_of_mspec mspec) in 
          let mspec = mk_mspec (formal_mref_of_mspec mspec, boxity,il_ctyargs,il_mtyargs) in 
          let mspec = (* intern_mspec cenv.manager *) mspec in
          (* "Unit" return types on static methods become "void" *)
          let mustGenerateUnitAfterCall = is_unit_typ cenv.g rty in 
          let tailcall = 
            if isNil later_args && not selfInit then 
                let isDllImport = vref_isDllImport cenv.g vref in 
                can_tailcall boxity eenv.withinSEH (List.exists (type_of_expr cenv.g >> is_byref_ty cenv.g) now_args) mustGenerateUnitAfterCall isDllImport sequel 
            else Normalcall in 
          let call_instr = 
            if virtualCall then I_callvirt (tailcall, mspec, None) 
            else if newobj then I_newobj (mspec, None) 
            else I_call (tailcall, mspec, None) in 

          (* ok, now we're ready to generate *)          
          if superInit || selfInit then CG.emit_instrs cgbuf [ Push (enclosing_typ_of_mspec mspec) ] [ ldarg_0 ] ;
          iter2 (fun fargtys x -> gen_untupled_expr_discarding_units cenv cgbuf eenv m fargtys x Continue) fargtysl  now_args;
          let nargs = length (formal_args_of_mspec mspec) in 
          if verbose then dprintf2 "gen_app: call, nargs = %d, is_static_callconv (callconv_of_mspec mspec) = %b\n" nargs (is_static_callconv (callconv_of_mspec mspec));
          CG.emit_instr cgbuf (replicate (nargs + (if is_static_callconv (callconv_of_mspec mspec) || newobj then 0 else 1)) Pop @ 
                               (if is_unit_typ cenv.g rty || superInit || selfInit then [] else [Push (gen_type m cenv.g eenv.tyenv arty)])) call_instr;
          if verbose then dprint_endline ("gen_app: after");
          (* load the 'this' pointer as the pretend 'result' of the operation.  It will be popped agin in most cases *)
          if superInit then CG.emit_instrs cgbuf [ Push (enclosing_typ_of_mspec mspec) ] [ ldarg_0 ] ;
          if isNil later_args then begin
            (* Generate the "unit" value if necessary *)
            gen_call_sequel cenv eenv.cloc cgbuf mustGenerateUnitAfterCall sequel 
          end else begin
            gen_indirect_call cenv cgbuf eenv (arty,[],later_args,m) sequel
          end;
          if verbose then dprint_endline ("gen_app: Method Done");
      | _ -> failwith "??"
      end
        
    (* This case is for getting/calling a value, when we can't call it directly. *)
    (* However, we know the type instantiation for the value.  *)
    (* In this case we can often generate a type-specific local expression for the value. *)
    (* This reduces the number of dynamic type applications. *)
  | (TExpr_val(vref,_,_),_,_)  -> 
     gen_vref_get_with_fetch_sequel cenv cgbuf eenv m vref (Some (tyargs,args,m,sequel))
        
  | _ ->
    (* worst case: generate a first-class function value and call *)
    gen_expr cenv cgbuf eenv f Continue;
    gen_indirect_call cenv cgbuf eenv (fty,tyargs,args,m) sequel
        
and can_tailcall boxity withinSEH hasByrefArg mustGenerateUnitAfterCall isDllImport sequel = 
    if boxity = AsObject && not withinSEH && not hasByrefArg && not isDllImport &&
        (* We can tailcall even if we need to generate "unit", as long as we're about to throw the value away anyway as par of the return. *)
        (* We can tailcall if we don't need to generate "unit", as long as we're about to return. *)
        (match sequel_ignore_end_scopes sequel with 
         | ReturnVoid | Return           -> not mustGenerateUnitAfterCall
         | DiscardThen ReturnVoid ->     mustGenerateUnitAfterCall
         | _                -> false) 
    then Tailcall 
    else Normalcall
        
and gen_named_local_tyfunc_cloinfo_call cenv cgbuf eenv actual_rty cloinfo tyargs m = 
    if verbose then dprint_endline ("Compiling local type func call in "^cgbuf.methodName);
    let il_tyargs = tyargs |> gen_tyargs m cenv.g eenv.tyenv in 
    let il_typ = gen_named_local_tyfunc_type cloinfo in 
    if not (length cloinfo.clo_il_lambda_gparams_direct = length tyargs) then errorR(Error("incorrect number of type arguments to local call",m));
    let il_mspec = mk_instance_mspec_in_typ(il_typ, "DirectInvoke", [], cloinfo.clo_formal_il_rty, il_tyargs) in 
    let il_actual_rty = gen_type m cenv.g eenv.tyenv actual_rty in
    callfunc_counter();
    CG.emit_instr cgbuf [Pop;Push il_actual_rty] (mk_normal_call il_mspec);
    if verbose then dprint_endline "Done local type func call..."

        
and gen_indirect_call cenv cgbuf eenv (functy,tyargs,args,m) sequel =
    if verbose then dprint_endline ("Compiling call in "^cgbuf.methodName);
    gen_exprs cenv cgbuf eenv args;
    if verbose then dprint_endline "Compiling call instruction...";
    (* Fold in the new types into the environment as we generate the formal types. *)
    let apps = 
        let typars,formal_functy = try_dest_forall_typ functy in 
        if verbose then dprintf2 "length args = %d, formal_functy = %s\n" (length args) (showL(typeL formal_functy));

        let feenv = add_typars eenv.tyenv typars in 
        let mk_ty_apps = fold_right (fun tyarg apps -> Apps_tyapp(gen_type m cenv.g eenv.tyenv tyarg,apps)) tyargs in 
        let formal_rty,mk_tm_apps = 
            fold_left 
              (fun (formal_functy,sofar) _ -> 
                let formal_dty,formal_rty = dest_fun_typ formal_functy in 
                (formal_rty,(fun apps -> sofar (Apps_app(gen_type m cenv.g feenv formal_dty,apps)))))
              (formal_functy,(fun x -> x))
              args in
        if verbose then dprint_endline "Compiling return type...";
        let ret_apps = Apps_done (gen_type m cenv.g feenv formal_rty) in 
        mk_ty_apps (mk_tm_apps ret_apps) in 
    let actual_rty = apply_types functy (tyargs, args) in 
    let il_actual_rty = gen_type m cenv.g eenv.tyenv actual_rty in
    let hasByrefArg = 
        let rec check x = 
          match x with 
          | Apps_tyapp(_,apps') -> check apps'
          | Apps_app(arg,apps') -> is_byref arg || check apps'
          | _ -> false in 
        check apps in 
        
    let tailcall = can_tailcall AsObject eenv.withinSEH hasByrefArg false false sequel in 
    callfunc_counter();
    CG.emit_instr cgbuf (replicate (1+length args) Pop @ [Push il_actual_rty]) (mk_IlxInstr (EI_callfunc(tailcall,apps)));
    if verbose then dprint_endline "Done compiling indirect call...";
    gen_sequel cenv eenv.cloc cgbuf sequel

(*--------------------------------------------------------------------------
!* Generate try expressions
 *-------------------------------------------------------------------------- *)

and gen_try cenv cgbuf eenv scopeMarks (e1,m,resty) =
    let stack,eenvinner = gen_stack_save cenv cgbuf eenv m scopeMarks in  
    let start_try = CG.mark cgbuf "start_try" in 
    let end_try = CG.emit_delay_mark cgbuf "end_try" in 
    let after_handler = CG.emit_delay_mark cgbuf "after_handler" in 
    let eenvinner = {eenvinner with withinSEH = true} in
    let il_resty = gen_type m cenv.g eenvinner.tyenv resty in 
    let where_to_save_expr,eenvinner = alloc_local cenv cgbuf eenvinner true (nng.nngApply "-tryres" m,il_resty) (start_try,end_try)in 
    gen_expr cenv cgbuf eenvinner e1 (LeaveHandler (false, where_to_save_expr,after_handler));
    CG.set_mark_to_here cgbuf end_try;
    let tryMarks = (code_label_of_mark start_try, code_label_of_mark end_try) in 
    where_to_save_expr,eenvinner,stack,tryMarks,after_handler,il_resty

and gen_try_catch cenv cgbuf eenv (e1,vf,ef,vh,eh,m,resty) sequel =
    if verbose then dprint_endline ("gen_try");      
    (* Save the stack - gross because IL flushes the stack at the exn. handler *)
    (* note: eenvinner notes spill vars are live *)
    local_scope "trystack" cgbuf (fun scopeMarks -> 
       let where_to_save_expr,eenvinner,stack,tryMarks,after_handler,il_resty = gen_try cenv cgbuf eenv scopeMarks (e1,m,resty)  in

       (* Now the filter and catch blocks *)

       let seh = 
           if cenv.generateFilterBlocks then begin 
               let start_filter = CG.mark cgbuf "start_filter"  in 
               let after_filter = CG.emit_delay_mark cgbuf "after_filter" in 
                 let (sequel_on_branches,after,stack_after,sequel_after) = gen_join_point cenv cgbuf "filter" eenv cenv.g.int_ty m EndFilter in 
                 CG.emit_seqpoint cenv cgbuf (start_range_of_range (range_of_expr ef));  
                 CG.set_stack cgbuf [cenv.g.ilg.typ_Object];
                 let _,eenvinner = alloc_local_val cenv cgbuf vf eenvinner None (start_filter,after_filter) in 
                 CG.emit_instr cgbuf [Pop; Push cenv.g.ilg.typ_Exception] (I_castclass cenv.g.ilg.typ_Exception);
                 gen_set_locval cenv cgbuf eenvinner m vf;
                 gen_expr cenv cgbuf eenvinner ef sequel_on_branches;
                 CG.set_mark_to_here cgbuf after;
                 CG.set_stack cgbuf stack_after;
                 gen_sequel cenv eenv.cloc cgbuf sequel_after;
               let end_filter = CG.mark cgbuf "end_filter" in 
               let filterMarks = (code_label_of_mark start_filter, code_label_of_mark end_filter) in 
               CG.set_mark_to_here cgbuf after_filter;

               let start_handler = CG.mark cgbuf "start_handler"  in 
                 CG.emit_seqpoint cenv cgbuf (start_range_of_range (range_of_expr eh));  
                 CG.set_stack cgbuf [cenv.g.ilg.typ_Object];
                 let _,eenvinner = alloc_local_val cenv cgbuf vh eenvinner None (start_handler,after_handler) in 
                 CG.emit_instr cgbuf [Pop; Push cenv.g.ilg.typ_Exception] (I_castclass cenv.g.ilg.typ_Exception);
                 gen_set_locval cenv cgbuf eenvinner m vh;
                 gen_expr cenv cgbuf eenvinner eh (LeaveHandler (false, where_to_save_expr,after_handler));
               let end_handler = CG.mark cgbuf "end_handler" in 
               let handlerMarks = (code_label_of_mark start_handler, code_label_of_mark end_handler) in 
               SEH_filter_catch(filterMarks, handlerMarks)
           end else begin
               let start_handler = CG.mark cgbuf "start_handler"  in 
                 CG.emit_seqpoint cenv cgbuf (start_range_of_range (range_of_expr eh));  
                 CG.set_stack cgbuf [cenv.g.ilg.typ_Object];
                 let _,eenvinner = alloc_local_val cenv cgbuf vh eenvinner None (start_handler,after_handler) in 
                 CG.emit_instr cgbuf [Pop; Push cenv.g.ilg.typ_Exception] (I_castclass cenv.g.ilg.typ_Exception);
                 gen_set_locval cenv cgbuf eenvinner m vh;
                 gen_expr cenv cgbuf eenvinner eh (LeaveHandler (false, where_to_save_expr,after_handler));
               let end_handler = CG.mark cgbuf "end_handler" in 
               let handlerMarks = (code_label_of_mark start_handler, code_label_of_mark end_handler) in 
               SEH_type_catch(cenv.g.ilg.typ_Object, handlerMarks)
           end in 
       ResizeArray.add cgbuf.exn_specs
         { exnClauses = [ seh ];
           exnRange= tryMarks } ;

       CG.set_mark_to_here cgbuf after_handler;
       CG.set_stack cgbuf [];

       (* Restore the stack and load the result *)
       gen_stack_restore cenv cgbuf stack; (* RESTORE *)
       gen_get_local cgbuf il_resty where_to_save_expr;
       gen_sequel cenv eenv.cloc cgbuf sequel
   ) 


and gen_try_finally cenv cgbuf eenv (e1,e2,m,resty) sequel =
    if verbose then dprint_endline ("gen_try");      
    (* Save the stack - gross because IL flushes the stack at the exn. handler *)
    (* note: eenvinner notes spill vars are live *)
    local_scope "trystack" cgbuf (fun scopeMarks -> 
       let where_to_save_expr,eenvinner,stack,tryMarks,after_handler,il_resty = gen_try cenv cgbuf eenv scopeMarks (e1,m,resty)  in

       (* Now the catch/finally block *)
       let start_handler = CG.mark cgbuf "start_handler"  in 
       CG.set_stack cgbuf [];
       gen_expr cenv cgbuf eenvinner e2 (LeaveHandler (true, where_to_save_expr,after_handler));
       let end_handler = CG.mark cgbuf "end_handler" in 
       let handlerMarks = (code_label_of_mark start_handler, code_label_of_mark end_handler) in 
       ResizeArray.add cgbuf.exn_specs
         { exnClauses = [ SEH_finally(handlerMarks) ];
           exnRange   = tryMarks } ;

       CG.set_mark_to_here cgbuf after_handler;
       CG.set_stack cgbuf [];

       (* Restore the stack and load the result *)
       gen_stack_restore cenv cgbuf stack; (* RESTORE *)
       gen_get_local cgbuf il_resty where_to_save_expr;
       gen_sequel cenv eenv.cloc cgbuf sequel
   ) 

(*--------------------------------------------------------------------------
!* Generate for-loop
 *-------------------------------------------------------------------------- *)
    
and gen_for cenv cgbuf eenv (v,e1,updown,e2,e3,m) sequel =
    (* The JIT/NGen eliminate array-bounds checks for C# loops of form:
     *   for(int i=0; i < (#ldlen arr#); i++) { ... arr[i] ... }
     * Notes:
     * - here, e2 generated for each test (so "ldlen arr" will be in place.
     * - tc.ml lifts F#-level upper bounds out of loop to preserve "effects".
     * - opt.ml inlines "ldlen arr" forms into upper bound position.
     * - following C# IL-layout exactly "prefix, jmp test, body, test, finish" for JIT/NGEN.
     *)
    let start = CG.mark cgbuf "for_start"  in
    let finish = CG.emit_delay_mark cgbuf "for_finish" in 
    let inner = CG.emit_delay_mark cgbuf "for_inner" in 
    let test = CG.emit_delay_mark cgbuf "for_test" in 
    let stack,eenvStack = gen_stack_save cenv cgbuf eenv m (start,finish) in

    let _,eenvinner = alloc_local_val cenv cgbuf v eenvStack None (start,finish) in (* not: eenvStack noted stack spill vars are live *)
    gen_expr cenv cgbuf eenv e1 Continue;
    gen_set_locval cenv cgbuf eenvinner m v;
    CG.emit_instr cgbuf [] (I_br (code_label_of_mark test));

    (* .inner - loop-body is e3 and index adjustment *)
    CG.set_mark_to_here cgbuf inner;
    CG.emit_seqpoint cenv cgbuf (range_of_expr e3);
    gen_expr cenv cgbuf eenvinner e3 discard;
    gen_get_locval cenv cgbuf eenvinner (range_of_expr e2) v None;
    CG.emit_instr cgbuf [Push cenv.g.ilg.typ_int32] (mk_ldc_i32 (Int32.of_int 1));
    CG.emit_instr cgbuf [Pop] (I_arith (if updown then AI_add else AI_sub));
    gen_set_locval cenv cgbuf eenvinner m v;
    CG.emit_instr cgbuf [] (I_br (code_label_of_mark test));

    (* .test - loop-test, e2 evaluated for each test *)
    let testm = union_ranges (range_of_expr e1) (range_of_expr e2) in
    CG.set_mark_to_here cgbuf test;
    CG.emit_seqpoint cenv cgbuf testm;
    gen_get_locval cenv cgbuf eenvinner testm v None;
    gen_expr cenv cgbuf eenvinner e2 
      (CmpThenBrOrContinue ( [Pop; Pop], I_brcmp((if updown then BI_blt else BI_bgt),code_label_of_mark inner,code_label_of_mark finish)));

    (* .finish - loop-exit here *)
    CG.set_mark_to_here cgbuf finish;

    (* Restore the stack and load the result *)  
    gen_stack_restore cenv cgbuf stack;
    gen_unit_then_sequel cenv eenv.cloc cgbuf sequel

(*--------------------------------------------------------------------------
!* Generate while-loop 
 *-------------------------------------------------------------------------- *)
    
and gen_while cenv cgbuf eenv (e1,e2,m) sequel =
    let finish = CG.emit_delay_mark cgbuf "while_finish"  in 
    let inner = CG.emit_delay_mark cgbuf "while_inner"  in     
    let start_test = CG.mark cgbuf "start_test" in 
    CG.emit_seqpoint cenv cgbuf (range_of_expr e1);
    gen_expr cenv cgbuf eenv e1 (CmpThenBrOrContinue ([Pop],(I_brcmp(BI_brfalse,code_label_of_mark finish,code_label_of_mark inner))));
    CG.set_mark_to_here cgbuf inner; 
    
    gen_expr cenv cgbuf eenv e2 (DiscardThen (Br start_test));
    CG.set_mark_to_here cgbuf finish; 
    CG.emit_seqpoint cenv cgbuf (end_range_of_range (range_of_expr e1));
    gen_unit_then_sequel cenv eenv.cloc cgbuf sequel

(*--------------------------------------------------------------------------
!* Generate seq
 *-------------------------------------------------------------------------- *)

and gen_seq cenv cgbuf eenv (e1,e2,specialSeqFlag,m) sequel =
    match specialSeqFlag with 
    | NormalSeq -> 
        if verbose then dprintf1 "gen_seq (normal), sequel = %s\n" (string_of_sequel sequel);
        gen_expr cenv cgbuf eenv e1 discard; 
        gen_expr cenv cgbuf eenv e2 sequel
    | ThenDoSeq ->
        gen_expr cenv cgbuf eenv e1 Continue;
        gen_expr cenv cgbuf eenv e2 discard;
        gen_sequel cenv eenv.cloc cgbuf sequel

(*--------------------------------------------------------------------------
!* Generate IL assembly code.
 * Polymorphic IL/ILX instructions may be instantiated when polymorphic code is inlined.
 * We must implement this for the few uses of polymorphic instructions 
 * in the standard libarary. 
 *-------------------------------------------------------------------------- *)

and gen_asm cenv cgbuf eenv (il,tyargs,args,rtys,m) sequel =
    if verbose then dprintf1 "gen_asm, #args = %d" (length args);  
    let il_tyargs = gen_types m cenv.g eenv.tyenv tyargs in 
    let il_rtys = gen_types m cenv.g eenv.tyenv rtys in 
    let il_after_inst = 
      il |> map (fun i -> 
          let err s  = 
              errorR(InternalError(s^": bad instruction: "^Ilprint.string_of_instr i,m)) in

          let mod_fspec fspec = 
                {fspec with fspecEnclosingType= 
                                   let ty = fspec.fspecEnclosingType in 
                                   let tspec = (tspec_of_typ ty) in 
                                   mk_typ (boxity_of_typ ty) { tspec with tspecInst=il_tyargs } } in
          match i,il_tyargs with   
            | I_unbox_any (Type_tyvar idx)           ,[tyarg] -> I_unbox_any (tyarg)
            | I_box (Type_tyvar idx)                 ,[tyarg] -> I_box (tyarg)
            | I_isinst (Type_tyvar idx)              ,[tyarg] -> I_isinst (tyarg)
            | I_castclass (Type_tyvar idx)           ,[tyarg] -> I_castclass (tyarg)
            | I_newarr (shape,Type_tyvar idx)        ,[tyarg] -> I_newarr (shape,tyarg)
            | I_ldelem_any (shape,Type_tyvar idx)    ,[tyarg] -> I_ldelem_any (shape,tyarg)
            | I_ldelema (ro,shape,Type_tyvar idx)    ,[tyarg] -> I_ldelema (ro,shape,tyarg)
            | I_stelem_any (shape,Type_tyvar idx)    ,[tyarg] -> I_stelem_any (shape,tyarg)
            | I_ldobj (a,b,Type_tyvar idx)           ,[tyarg] -> I_ldobj (a,b,tyarg)
            | I_stobj (a,b,Type_tyvar idx)           ,[tyarg] -> I_stobj (a,b,tyarg)
            | I_ldtoken (Token_type (Type_tyvar idx)),[tyarg] -> I_ldtoken (Token_type (tyarg))
            | I_sizeof (Type_tyvar idx)              ,[tyarg] -> I_sizeof (tyarg)
            | I_ldfld (al,vol,fspec)                 ,_       -> I_ldfld (al,vol,mod_fspec fspec)
            | I_ldflda (fspec)                       ,_       -> I_ldflda (mod_fspec fspec)
            | I_stfld (al,vol,fspec)                 ,_       -> I_stfld (al,vol,mod_fspec fspec)
            | I_ldsfld (vol,fspec)                   ,_       -> I_ldsfld (vol,mod_fspec fspec)
            | I_ldsflda (fspec)                      ,_       -> I_ldsflda (mod_fspec fspec)
            | I_other e,_ when is_ilx_ext_instr e -> 
                begin match (dest_ilx_ext_instr e),il_tyargs with 
                |  (EI_newarr_erasable (shape,Type_tyvar idx)),[tyarg] -> 
                    mk_IlxInstr (EI_newarr_erasable (shape,tyarg))
                |  (EI_ldelem_any_erasable (shape,Type_tyvar idx)),[tyarg] -> 
                    mk_IlxInstr (EI_ldelem_any_erasable (shape,tyarg))
                |  (EI_stelem_any_erasable (shape,Type_tyvar idx)),[tyarg] -> 
                    mk_IlxInstr (EI_stelem_any_erasable (shape,tyarg))
                |  (EI_ilzero(Type_tyvar idx)),[tyarg] -> 
                    mk_IlxInstr (EI_ilzero(tyarg))
                |  _ -> if not (isNil tyargs) then err "Bad polymorphic ILX instruction"; i
                end
            | I_arith AI_nop,_ -> i  
                (* These are embedded in the IL for a an initonly ldfld, i.e. *)
                (* here's the relevant comment from tc.ml *)
                (*     "Add an I_nop if this is an initonly field to make sure we never recognize it as an lvalue. See mk_expra_of_expr." *)

            | _ -> if not (isNil tyargs) then err "Bad polymorphic IL instruction"; i) in 
    match il_after_inst,args,sequel,il_rtys with 

      (* Strip off any "ldc.i4 1 xor" when the sequel is a comparison branch *)
      (* These are precisely the code for "not" *)
      (* For these we can just generate the argument and change the test (from a brfalse to a brtrue and vice versa) *)
    | [ I_arith (AI_ldc(DT_I4,NUM_I4 i32)); I_arith AI_ceq ], 
      [arg1],
      CmpThenBrOrContinue([Pop],I_brcmp (BI_brfalse, label1,label2)) ,
      _
        when i32 = int_to_i32 0
      ->
        gen_expr cenv cgbuf eenv arg1 (CmpThenBrOrContinue([Pop],I_brcmp (BI_brtrue, label1,label2)))

    | [ I_ret ], 
      [arg1],
      sequel,
      [il_rty]
      -> 
          gen_expr cenv cgbuf eenv arg1 Continue;
          CG.emit_instr cgbuf [Pop] I_ret;
          gen_sequel_end_scopes cgbuf sequel

    | [ I_ret ], 
      [],
      sequel,
      [il_rty]
      -> 
          CG.emit_instr cgbuf [Pop] I_ret;
          gen_sequel_end_scopes cgbuf sequel

        (* 'throw' instructions are a bit of a problem - e.g. let x = (throw ...) in ... expects a value *)
        (* to be left on the stack.  But dead-code checking by some versions of the .NET verifier *)
        (* mean that we can't just have fake code after the throw to generate the fake value *)
        (* (nb. a fake value can always be generated by a "ldnull unbox.any ty" sequence *)
        (* So in the worst case we generate a fake (never-taken) branch to a piece of code to generate *)
        (* the fake value *)
    | [ I_throw ], 
      [arg1],
      sequel,
      [il_rty]
      -> 
        begin match sequel_ignore_end_scopes sequel with 
        | s when sequel_is_immediate  s -> 
          if verbose then dprintf0 "gen_asm: throw: A\n";  
          (* In most cases we can avoid doing this... *)
          gen_expr cenv cgbuf eenv arg1 Continue;
          CG.emit_instr cgbuf [Pop] I_throw;
          gen_sequel_end_scopes cgbuf sequel
        | _ -> 
          if verbose then dprintf0 "gen_asm: throw: B\n";  
          let after1 = CG.emit_delay_mark cgbuf ("fake_join") in 
          let after2 = CG.emit_delay_mark cgbuf ("fake_join") in 
          let after3 = CG.emit_delay_mark cgbuf ("fake_join") in 
          CG.emit_instrs cgbuf [] [mk_ldc_i32 0l; 
                                   I_brcmp (BI_brfalse,code_label_of_mark after2,code_label_of_mark after1); ];
          CG.emit_seqpoint cenv cgbuf (end_range_of_range m);
          CG.set_mark_to_here cgbuf after1;
          CG.emit_instrs cgbuf [Push il_rty] [i_ldnull;  I_unbox_any il_rty; I_br (code_label_of_mark after3) ];
          
          CG.set_mark_to_here cgbuf after2;
          gen_expr cenv cgbuf eenv arg1 Continue;
          CG.emit_instr cgbuf [Pop] I_throw;
          CG.set_mark_to_here cgbuf after3;
          gen_sequel cenv eenv.cloc cgbuf sequel;
        end
    | _ -> 
    (* Otherwise generate the arguments, and see if we can use a I_brcmp rather than a comparison followed by an I_brfalse/I_brtrue *)
      gen_exprs cenv cgbuf eenv args;
      begin match il_after_inst,sequel with

      (* NOTE: THESE ARE NOT VALID ON FLOATING POINT DUE TO NaN.  Hence INLINE ASM ON FP. MUST BE CAREFULLY WRITTEN  *)

      | [ I_arith AI_clt ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brfalse, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_bge,label1,label2));
      | [ I_arith AI_cgt ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brfalse, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_ble,label1, label2));
      | [ I_arith AI_clt_un ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brfalse, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_bge_un,label1,label2));
      | [ I_arith AI_cgt_un ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brfalse, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_ble_un,label1, label2));
      | [ I_arith AI_ceq ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brfalse, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_bne_un,label1, label2));
        
      (* THESE ARE VALID ON FP w.r.t. NaN *)
        
      | [ I_arith AI_clt ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brtrue, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_blt,label1, label2));
      | [ I_arith AI_cgt ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brtrue, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_bgt,label1, label2));
      | [ I_arith AI_clt_un ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brtrue, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_blt_un,label1, label2));
      | [ I_arith AI_cgt_un ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brtrue, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_bgt_un,label1, label2));
      | [ I_arith AI_ceq ], CmpThenBrOrContinue([Pop],I_brcmp (BI_brtrue, label1,label2)) ->
        CG.emit_instr cgbuf [Pop; Pop] (I_brcmp(BI_beq,label1, label2));
      | _ -> 
      (* Failing that, generate the real IL leaving value(s) on the stack *)
        CG.emit_instrs cgbuf (replicate (length args) Pop @ map push il_rtys) il_after_inst;
      (* If no return values were specified generate a "unit" *)
        if isNil rtys then 
          gen_unit_then_sequel cenv eenv.cloc cgbuf sequel
        else 
          gen_sequel cenv eenv.cloc cgbuf sequel
      end

(*--------------------------------------------------------------------------
!* Generate expression quotations
 *-------------------------------------------------------------------------- *)

and gen_quote cenv cgbuf eenv (raw,ast,m,ety) sequel =
    let astSpec = 
        try Creflect.convExpr (Creflect.mk_cenv (cenv.g, cenv.amap, cenv.viewCcu, nng)) Creflect.empty_env ast 
        with Creflect.InvalidQuotedTerm e -> error(e) in 
    let astPickledBytes = Sreflect.Raw.pickle astSpec in 
    let someTypeInModuleExpr =  
        mk_asm ([  mk_normal_call (mspec_Type_GetTypeFromHandle cenv.g.ilg) ], [],
                       [mk_asm ([ I_ldtoken (Token_type (Type_boxed eenv.someTspecInThisModule)) ], [],[],[cenv.g.system_RuntimeTypeHandle_typ],m)],
                       [cenv.g.system_Type_typ],m) in 
    let bytesExpr = TExpr_op(TOp_bytes(astPickledBytes),[],[],m) in 
    let unpickled = 
        (if raw then mk_call_unpickle_raw_quotation_vref else mk_call_unpickle_quotation_vref) 
          cenv.g m (tinst_of_stripped_typ ety) someTypeInModuleExpr bytesExpr in 
    gen_expr cenv cgbuf eenv unpickled sequel

(*--------------------------------------------------------------------------
!* Generate calls to IL methods
 *-------------------------------------------------------------------------- *)

and gen_ilcall cenv cgbuf eenv ((virt,protect,valu,newobj,vFlags,_,isDllImport,_,mref),enclTypeArgs,methTypeArgs,args,rtys,m) sequel =
    if verbose then dprint_endline ("gen_ilcall");     
    let hasByrefArg  =  mref |> Il.args_of_mref |> List.exists is_byref in
    let superInit = (vFlags = CtorValUsedAsSuperInit) in
    let boxity = (if valu then AsValue else AsObject) in 
    let mustGenerateUnitAfterCall = (isNil rtys) in 
    (* REVIEW: avoid tailcalling when doing a simple "call" - note we don't guarantee to tailcall into other .NET languages *)
    let tail = can_tailcall boxity eenv.withinSEH hasByrefArg mustGenerateUnitAfterCall isDllImport sequel in
    let il_ttyargs = gen_tyargs m cenv.g eenv.tyenv enclTypeArgs in 
    let il_mtyargs = gen_tyargs m cenv.g eenv.tyenv methTypeArgs in 
    let il_rtys = gen_types m cenv.g eenv.tyenv rtys in 
    let mspec = mk_mspec (mref,boxity,il_ttyargs,il_mtyargs) in 
    (* Load the 'this' pointer to pass to the superclass constructor. This argument is not in the expression tree since it can't be treated like an ordinary value *)
    if superInit then CG.emit_instrs cgbuf [ Push (enclosing_typ_of_mspec mspec) ] [ ldarg_0 ] ;
    gen_exprs cenv cgbuf eenv args;
    let il = 
      if newobj then [ I_newobj(mspec,None) ] 
      else if virt then [ I_callvirt(tail,mspec,None) ] 
      else  [ I_call(tail,mspec,None) ] in 
    CG.emit_instrs cgbuf (replicate (length args + (if superInit then 1 else 0)) Pop @ (if superInit then [] else map push il_rtys)) il;
    (* Load the 'this' pointer as the pretend 'result' of the superInit operation.  It will be immediately popped in most cases, but may also be used as the target of ome "property set" oeprations. *)
    if superInit then CG.emit_instrs cgbuf [ Push (enclosing_typ_of_mspec mspec) ] [ ldarg_0 ] ;
    gen_call_sequel cenv eenv.cloc cgbuf mustGenerateUnitAfterCall sequel

and gen_call_sequel cenv cloc cgbuf mustGenerateUnitAfterCall sequel =
    if mustGenerateUnitAfterCall 
    then gen_unit_then_sequel cenv cloc cgbuf sequel
    else gen_sequel cenv cloc cgbuf sequel


and gen_trait_call cenv cgbuf eenv (traitInfo, methTypeArgs, args, m) sequel =
    let minfo = commitOperationResult (Csolve.codegen_witnessThatTypSupportsTraitConstraint cenv.g cenv.amap m traitInfo) in
    (* Fix bug 1281:  If we resolve to an instance method on a struct and we haven't yet taken the address of the object then go do that *)
    if Infos.minfo_is_struct minfo && Infos.minfo_is_instance minfo && (match args with [] -> false | h::t -> not (is_byref_ty cenv.g (type_of_expr cenv.g h))) then 
        let h,t = headAndTail args in
        let wrap,h' = mk_expra_of_expr cenv.g true PossiblyMutates h m  in 
        gen_expr cenv cgbuf eenv (wrap (TExpr_op(TOp_trait_call(traitInfo), methTypeArgs, (h' :: t), m))) sequel 
    else        
        let slotsig = Infos.slotsig_of_minfo cenv.g cenv.amap m minfo in
        let gty = Infos.typ_of_minfo minfo in 
        let il_gty,il_params,il_ret = gen_formal_slotsig m cenv eenv slotsig in
        let il_argtys = map typ_of_param il_params in 
        let il_rty = typ_of_return il_ret in 
        let mref = mk_mref(tref_of_typ il_gty, (if Infos.minfo_is_instance minfo then instance_callconv else static_callconv), Infos.name_of_minfo minfo, length methTypeArgs, il_argtys, il_rty) in 
        let tinst = snd(dest_stripped_tyapp_typ gty) in
        let rtys = if slotsig_has_void_rty cenv.g slotsig then [] else [actual_rty_of_slotsig tinst methTypeArgs slotsig] in 
        gen_ilcall cenv cgbuf eenv ((Infos.minfo_is_virt minfo,
                                     Infos.minfo_is_protected minfo,
                                     Infos.minfo_is_struct minfo,false,NormalValUse,false,false,None,mref),tinst,methTypeArgs,args,rtys,m) sequel

(*--------------------------------------------------------------------------
!* Generate byref-related operations
 *-------------------------------------------------------------------------- *)

and gen_get_ref_lval cenv cgbuf eenv (e,ty,m) sequel =
    if verbose then dprint_endline ("gen_get_ref_lval");     
    gen_expr cenv cgbuf eenv e Continue;
    let fref = gen_recdfield_ref m cenv eenv.tyenv (mk_refcell_contents_rfref cenv.g) [ty] in
    CG.emit_instrs cgbuf [Pop; Push (Type_byref (actual_typ_of_fspec fref))] [ I_ldflda fref ] ;
    gen_sequel cenv eenv.cloc cgbuf sequel

and gen_val_geta cenv cgbuf eenv (v,m) sequel =
    if verbose then dprint_endline ("gen_val_geta");     
    let vspec = deref_val v in 
    let il_ty = gen_type_of_val cenv eenv vspec in
    begin match storage_for_vref m v eenv with 
    | Local (idx,None) ->
        CG.emit_instrs cgbuf [ Push (Type_byref il_ty)] [ I_ldloca (int_to_u16 idx) ] ;
    | Arg idx ->
        CG.emit_instrs cgbuf [ Push (Type_byref il_ty)] [ I_ldarga (int_to_u16 idx) ] ;
    | StaticField (fspec,hasLiteralAttr,binding_tspec,fieldName,_,il_ty,_,_) ->  
        if hasLiteralAttr then errorR(Error("Taking the address of a literal field is invalid",m));
        gen_get_top_fielda cgbuf il_ty fspec
    | Local (_,Some _) | Method _ | Env _ | Unrealized | Null ->  
        errorR(Error( "This operation involves taking the address of a value '"^display_name_of_vref v^"' represented using a local variable or other special representation. This is invalid",m));
        CG.emit_instrs cgbuf [Pop; Push (Type_byref il_ty)] [ I_ldarga (int_to_u16 669) ] ;
    end;
    gen_sequel cenv eenv.cloc cgbuf sequel

and gen_byref_get cenv cgbuf eenv (v,m) sequel =
    if verbose then dprint_endline ("gen_byref_get");     
    gen_get_vref cenv cgbuf eenv m v None;
    let ilty = gen_type m cenv.g eenv.tyenv (dest_byref_ty cenv.g (type_of_vref v)) in
    CG.emit_instrs cgbuf [Pop; Push ilty] [ mk_normal_ldobj ilty ];
    gen_sequel cenv eenv.cloc cgbuf sequel

and gen_byref_set cenv cgbuf eenv (v,e,m) sequel =
    if verbose then dprint_endline ("gen_byref_set");     
    gen_get_vref cenv cgbuf eenv m v None;
    gen_expr cenv cgbuf eenv e Continue;
    let ilty = gen_type m cenv.g eenv.tyenv (dest_byref_ty cenv.g (type_of_vref v)) in
    CG.emit_instrs cgbuf [Pop; Pop] [ mk_normal_stobj ilty ];
    gen_unit_then_sequel cenv eenv.cloc cgbuf sequel

and gen_ilzero cenv cgbuf eenv (ty,m) =
    let il_ty = gen_type m cenv.g eenv.tyenv ty in 
    if is_ref_typ cenv.g ty then 
        CG.emit_instr cgbuf [Push il_ty] i_ldnull
    else
        let il_ty = gen_type m cenv.g eenv.tyenv ty in 
        local_scope "ilzero" cgbuf (fun scopeMarks ->
            let loc_idx,eenvinner = alloc_local cenv cgbuf eenv true (nng.nngApply "$default" m, il_ty) scopeMarks in 
            (* "initobj" (Generated by gen_init_local) doesn't work on byref types *)
            (* But ilzero(&ty) only gets generated in the built-in get-address function so *)
            (* we can just rely on zeroinit of all IL locals. *)
            begin match il_ty with 
            |  Type_byref _ -> ()
            | _ -> gen_init_local cgbuf il_ty loc_idx
            end;
            gen_get_local cgbuf il_ty loc_idx;
        )

(*--------------------------------------------------------------------------
!* Generate object expressions as ILX "closures"
 *-------------------------------------------------------------------------- *)

and gen_slotparam m cenv eenv (TSlotParam(nm,ty,inFlag,outFlag,optionalFlag,attribs)) = 
    let inFlag2,outFlag2,optionalFlag2,paramMarshal2 = gen_param_attribs cenv attribs in 
    
    { paramName=nm;
      paramType= gen_type m cenv.g eenv.tyenv ty;
      paramDefault=None;  
      paramMarshal=paramMarshal2; 
      paramIn=inFlag || inFlag2;
      paramOut=outFlag || outFlag2;
      paramOptional=optionalFlag || optionalFlag2;
      paramCustomAttrs= mk_custom_attrs (gen_attrs cenv eenv attribs) }
    
and gen_formal_slotsig m cenv eenv (TSlotSig(nm,typ,ctps,mtps,paraml,rty) as slotsig) = 
    let il_typ = gen_type m cenv.g eenv.tyenv typ in 
    let eenv_for_slotsig = env_for_typars (ctps @ mtps) eenv in 
    let il_params = map (gen_slotparam m cenv eenv_for_slotsig) paraml in 
    let il_rty = if slotsig_has_void_rty cenv.g slotsig then Type_void else gen_type m cenv.g eenv_for_slotsig.tyenv rty in 
    let il_ret = mk_return  il_rty in
    il_typ, il_params,il_ret

and inst_slotparam inst (TSlotParam(nm,ty,inFlag,fl2,fl3,attrs)) = TSlotParam(nm,inst_type inst ty,inFlag,fl2,fl3,attrs) 

and gen_actual_slotsig m cenv eenv (TSlotSig(nm,typ,ctps,mtps,paraml,rty) as tslotsig) ovby_mtps = 
    let slotsig_inst = mk_typar_inst (ctps@mtps) (tinst_of_stripped_typ typ @ generalize_typars ovby_mtps) in
    let il_params = map (inst_slotparam slotsig_inst >> gen_slotparam m cenv eenv) paraml in 
    let il_rty = if slotsig_has_void_rty cenv.g tslotsig then Type_void else gen_type m cenv.g eenv.tyenv (inst_type slotsig_inst rty) in 
    let il_ret = mk_return il_rty in 
    il_params,il_ret

and gen_mimpl cenv eenv (use_mimpl,(TSlotSig(ov_nm,ov_typ,_,_,_,_) as slotsig)) m =
    let ov_il_typ,ov_il_params,ov_il_ret = gen_formal_slotsig m cenv eenv slotsig in
    let use_mimpl = 
      if use_mimpl 
          && cenv.workAroundReflectionEmitMethodImplBug
          && inst_of_typ ov_il_typ <> [] 
          && scoref_of_tref (tref_of_typ ov_il_typ) = ScopeRef_local then 
        begin
          warning(Error("The implementation of a specified generic interface required a method implementation not fully supported by F# Interactive. In the unlikely event that the resulting class fails to load then compile the interface type into a statically-compiled DLL and reference it using '#r'",m));
          false
        end
      else use_mimpl in 

    let ovby_name = if use_mimpl then qualified_name_of_tcref (tcref_of_stripped_typ ov_typ) ov_nm else ov_nm in
    let ovinfo = Some(ov_il_typ,map typ_of_param ov_il_params, ov_il_ret.returnType) in 

    use_mimpl,ovby_name, ovinfo,
    (fun (ovby_tspec,ovby_mtps) -> 
        let ov_tref = tref_of_typ ov_il_typ in 
        let ov_mref = mk_mref(ov_tref, instance_callconv, ov_nm, length ovby_mtps, (map typ_of_param ov_il_params), typ_of_return ov_il_ret) in
        let eenv_for_ovby = add_typars_to_env ovby_mtps eenv in 
        let ovby_il_params,ovby_il_ret = gen_actual_slotsig m cenv eenv_for_ovby slotsig ovby_mtps in
        let ovby_mgparams = gen_static_formals m cenv eenv_for_ovby ovby_mtps in 
        let ovby_mgactuals = generalize_gparams ovby_mgparams in
        let ovby = mk_instance_mspec_in_boxed_tspec(ovby_tspec, ovby_name, map typ_of_param ovby_il_params, typ_of_return ovby_il_ret, ovby_mgactuals) in
        let ovby = (* intern_mspec cenv.manager *) ovby in
        { mimplOverrides = OverridesSpec(ov_mref,ov_il_typ);
          mimplOverrideBy = ovby })

and bind_basevopt cenv eenv basevopt = 
    match basevopt with 
    | None -> eenv
    | Some basev -> add_storage_for_val cenv.g (basev,Arg 0)  eenv  

and fixupVirtualSlotFlags mdef = 
    {mdef with
        mdHideBySig=true; 
        mdKind = (match mdef.mdKind with 
                   | MethodKind_virtual vinfo -> 
                      MethodKind_virtual
                         {vinfo with 
                             virtStrict=false }
                   | _ -> failwith "fixupVirtualSlotFlags") } 

and renameMethodDef ovby_name mdef = 
    {mdef with mdName=ovby_name }

and fixupMethodImplFlags mdef = 
    {mdef with mdAccess=MemAccess_private;
               mdHideBySig=true; 
               mdKind=(match mdef.mdKind with 
                         | MethodKind_virtual vinfo -> 
                            MethodKind_virtual
                               {vinfo with 
                                   virtStrict=false;
                                   virtFinal=true;
                                   virtNewslot=true;  }
                         | _ -> failwith "fixupMethodImpl") }

and gen_object_method cenv eenvinner cgbuf use_mimpl (TMethod((TSlotSig(ov_nm,ov_typ,_,_,_,_) as slotsig),ovby_mtps,vsl,e,m)) =
    (* Get the overrides methods's parameter specifications *)
    (* dprintf3 "gen_object_method (gen_formal_slotsig m), nm = %s, ov_typ = %s, #ovby_mtps = %d\n" ov_nm (NicePrint.pretty_string_of_typ (empty_denv cenv.g) ov_typ) (length ovby_mtps);  *)
    let eenv_under_typars = add_typars_to_env ovby_mtps eenvinner in
    (* dprintf1 "gen_object_method (gen_actual_slotsig m), nm = %s\n" ov_nm;  *)
    let ovby_il_params,ovby_il_ret = gen_actual_slotsig m cenv eenv_under_typars slotsig ovby_mtps in
    (* dprintf1 "gen_object_method (codegen_method_for_expr), nm = %s\n" ov_nm;  *)

    let meth_eenv = add_storage_for_locvals cenv.g (list_mapi (fun i v -> (v,Arg i)) vsl)  eenv_under_typars in 
    let ilmbody = codegen_method_for_expr cenv cgbuf.mgbuf ([],ov_nm,meth_eenv,0,0,e,(if slotsig_has_void_rty cenv.g slotsig then discardAndReturnVoid else Return)) in

    let really_use_mimpl,ovby_name,ovinfo,mimplf = gen_mimpl cenv eenvinner (use_mimpl,slotsig) (range_of_expr e) in
(*    dprintf4 "gen_object_method (gen_formal_slotsig m), nm = %s, use_mimpl = %b, ov_il_typ = %a\n" ov_nm use_mimpl Ilprint.NicePrint.output_typ ov_il_typ; *)
(*     dprintf2 "ov_il_ret = %a\n" Ilprint.NicePrint.output_typ ov_il_ret.returnType; *)

    let mdef = 
      mk_generic_virtual_mdef
        (ovby_name,
         MemAccess_public,
         gen_static_formals m cenv eenv_under_typars ovby_mtps,
         ovinfo,
         ovby_il_params,
         ovby_il_ret,
         MethodBody_il ilmbody) in
    (* fixup attributes to generate a method impl *)
    let mdef = if really_use_mimpl then fixupMethodImplFlags mdef else mdef in
    let mdef = fixupVirtualSlotFlags mdef in 
    (really_use_mimpl,mimplf,ovby_mtps),mdef

and gen_obj_expr cenv cgbuf eenvouter expr (base_typ,basevopt,basecall,overrides,iimpls,m)  sequel =
    if verbose then dprint_endline ("gen_obj_expr");     
    let cloinfo,body,eenvinner  = get_cloinfo cenv m false None eenvouter expr  in

    let attribs = cloinfo.clo_attribs in 
    let il_frees = cloinfo.clo_il_frees in
    let clo_freevars = cloinfo.clo_freevars in 
    let il_gparams = cloinfo.clo_il_gparams in
    assert(isNil cloinfo.clo_il_lambda_gparams_direct);
    let il_gactuals = inst_of_clospec cloinfo.clo_clospec in
    let ilty = cloinfo.clo_formal_il_rty in
    let lambdas = cloinfo.clo_lambdas in 
    let clospec = cloinfo.clo_clospec in 
    let clo_tref = tref_of_clospec cloinfo.clo_clospec in 
    let clo_name = cloinfo.clo_name in 
    let ovby_tspec = mk_tspec(clo_tref,il_gactuals) in

    let eenvinner = bind_basevopt cenv eenvinner basevopt in 
    let ctor_body = codegen_method_for_expr cenv cgbuf.mgbuf ([],clo_name,eenvinner,1,0,basecall,discardAndReturnVoid) in 

    let mdefs = map (gen_object_method cenv eenvinner cgbuf false >> snd) overrides in
    let ovby ((really_use_mimpl,mimplf,ovby_mtps),mdef) = 
      let mimpl = (if really_use_mimpl then Some(mimplf (ovby_tspec,ovby_mtps)) else None) in
      mimpl,mdef in 
    let mimpls,iimpl_mdefs = split (concat (map (snd >> map (gen_object_method cenv eenvinner cgbuf true >> ovby)) iimpls)) in
    let mimpls = chooseList (fun x -> x) mimpls in 
    let intftys = map (fst >> gen_type m cenv.g eenvinner.tyenv) iimpls in

    let attrs = gen_attrs cenv eenvinner attribs in 
    let super = (if is_interface_typ base_typ then cenv.g.ilg.typ_Object else ilty) in 
    let intfs = intftys @ (if is_interface_typ base_typ then [ilty] else []) in 
    let clo = mk_closure_tdef cenv (clo_tref,il_gparams,attrs,m,il_frees,lambdas,ctor_body,(iimpl_mdefs @ mdefs),mimpls,super,intfs) in

    insert_tdef cgbuf.mgbuf clo_tref clo;
    closure_counter();
    gen_get_locvals cenv cgbuf eenvouter m clo_freevars;
    CG.emit_instr cgbuf (replicate (length il_frees) Pop@ [ Push (Pubclo.typ_of_lambdas cenv.g.ilxPubCloEnv lambdas)]) (mk_IlxInstr (EI_newclo clospec));
    gen_sequel cenv eenvouter.cloc cgbuf sequel

(*-------------------------------------------------------------------------
!* Generate closure type_defs
 *------------------------------------------------------------------------- *)

and mk_closure_tdef cenv (tref,gparams,attrs,m,il_frees,lambdas,ctor_body,mdefs,mimpls,ext,intfs) =
      { tdName = tname_of_tref tref; 
        tdLayout = TypeLayout_auto;
        tdAccess =  computeTypeAccess tref true;
        tdGenericParams = gparams;
        tdCustomAttrs = mk_custom_attrs(attrs @ [mk_CompilationMappingAttr cenv.g (6 )(* SourceLevelConstruct.Closure *) ]);
        tdFieldDefs = mk_fdefs [];
        tdInitSemantics=TypeInit_beforefield;         
        tdSealed=true;
        tdAbstract=false;
        tdKind=mk_ilx_type_def_kind (ETypeDef_closure  { cloSource=gen_closure_ctor_range cenv m;
                                                         cloFreeVars=il_frees;  
                                                         cloStructure=lambdas;
                                                         cloCode=notlazy ctor_body });
        tdEvents= mk_events [];
        tdProperties = mk_properties [];
        tdMethodDefs= mk_mdefs mdefs; 
        tdMethodImpls= mk_mimpls mimpls; 
        tdSerializable= true; 
        tdComInterop=false;    
        tdSpecialName= false;
        tdNested=mk_tdefs [];
        tdEncoding= TypeEncoding_autochar;
        tdImplements= intfs;  
        tdExtends= Some ext;
        tdSecurityDecls= mk_security_decls [];
        tdHasSecurity=false; } 

(*-------------------------------------------------------------------------
!* Generate function closures
 *------------------------------------------------------------------------- *)
          
and gen_static_formals m cenv eenv tps = List.map (gen_gparam m cenv.g eenv.tyenv) tps
and gen_static_actuals m eenv tps = List.map (fun c -> mk_genactual (mk_tyvar_ty (repr_of_typar m c eenv))) tps

and gen_lambda_closure cenv cgbuf eenv is_ltyfunc selfv expr =
    if verbose then dprint_endline ("gen_lambda_closure:");      
    match expr with 
    | TExpr_lambda (_,_,_,_,m,_,_) 
    | TExpr_tlambda(_,_,_,m,_,_) -> 
          
        let cloinfo,body,eenvinner  = get_cloinfo cenv  m is_ltyfunc selfv eenv expr  in
          
        let entrypoint_info = 
          match selfv with 
          | Some v -> [(v, BranchCallClosure (cloinfo.clo_arity_info))]
          | _ -> [] in
        let clo_body = codegen_method_for_expr cenv cgbuf.mgbuf (entrypoint_info,cloinfo.clo_name,eenvinner,1,0,body,Return) in 
        let clo_tref = tref_of_clospec cloinfo.clo_clospec in 
        let clo_body,clo_meths = 
          if is_ltyfunc then 
            let ctor_body =  mk_ilmbody (true,[ ] ,8,nonbranching_instrs_to_code (mk_nongeneric_call_superclass_constructor([],cenv.g.ilg.tref_Object)), gen_opt_range cenv m ) in 
            ctor_body,
            [ mk_generic_instance_mdef("DirectInvoke",MemAccess_assembly,cloinfo.clo_il_lambda_gparams_direct,[],mk_return (cloinfo.clo_formal_il_rty), MethodBody_il clo_body) ]
          else 
            clo_body,[] in 
        let clo = mk_closure_tdef cenv (clo_tref,cloinfo.clo_il_gparams,[],m,cloinfo.clo_il_frees,cloinfo.clo_lambdas,clo_body,clo_meths,[],cenv.g.ilg.typ_Object,[]) in
        closure_counter();
        insert_tdef cgbuf.mgbuf clo_tref clo;
        cloinfo,m
    |     _ -> failwith "gen_lambda: not a lambda"
        
and mk_lambda_newclo eenv (cloinfo,m) = mk_IlxInstr (EI_newclo cloinfo.clo_clospec)

and gen_lambda_val cenv cgbuf eenv (cloinfo,m) =
    if verbose then dprint_endline ("Loading environment for "^cloinfo.clo_name ^" in "^cgbuf.methodName);
    gen_get_locvals cenv cgbuf eenv m cloinfo.clo_freevars;
    if verbose then dprint_endline ("Compiling newclo for "^cloinfo.clo_name ^" in "^cgbuf.methodName);
    CG.emit_instr cgbuf (replicate (length cloinfo.clo_il_frees) Pop@ [ Push (Pubclo.typ_of_lambdas cenv.g.ilxPubCloEnv cloinfo.clo_lambdas)]) (* REVIEW: more specific type when ILX supports them more explicitly  *) 
           (mk_lambda_newclo eenv (cloinfo,m))

and gen_lambda cenv cgbuf eenv is_ltyfunc selfv expr sequel =
    if verbose then dprint_endline ("gen_lambda:");   
    let cloinfo,m = gen_lambda_closure cenv cgbuf eenv is_ltyfunc selfv expr in
    gen_lambda_val cenv cgbuf eenv (cloinfo,m);
    if verbose then dprint_endline ("gen_lambda: done val");
    gen_sequel cenv eenv.cloc cgbuf sequel

and gen_type_of_val cenv eenv v = 
    if verbose then dprint_endline ("gen_type_of_val");
    gen_type (range_of_val v) cenv.g eenv.tyenv (type_of_val v)

and gen_type_of_val_as_freevar cenv m eenvouter eenvinner fv = 
    match storage_for_val m fv eenvouter with 
    | Local(_,Some _) | Env(_,Some _) -> cenv.g.ilg.typ_Object
    | (StaticField _ | Method _ |  Unrealized | Null) -> error(Error("gen_type_of_val_as_freevar: compiler error: unexpected unrealized value",range_of_val fv))
    | _ -> gen_type m cenv.g eenvinner.tyenv (type_of_val fv)

and get_closure_freevars cenv m selfv eenvouter expr =
    if verbose then dprint_endline ("get_closure_freevars:");
    let tref = 
      match expr with 
      | TExpr_obj (uniq,_,_,_,_,_,m,_) 
      | TExpr_lambda (uniq,_,_,_,m,_,_) 
      | TExpr_tlambda(uniq,_,_,m,_,_) -> 
          let boundv = tryfind (fun v -> not (compgen_of_val (deref_val v))) eenvouter.letBoundVars in
          let basename = 
            match boundv with
            | Some v -> compiled_name_of_val (deref_val v)
            | None -> "clo" in 
          let suffixmark = Some m in 

          let clo_name = clo_name_generator.snngApply basename suffixmark uniq in 
          mk_nested_tref_for_cloc eenvouter.cloc clo_name
      | _ -> error(Error("get_closure_freevars: not a lambda expression",range_of_expr expr)) in 
    let fvs =  free_in_expr expr in 

    (* Partition the free variables when some can be accessed from places besides the immediate environment *)
    (* Also filter out the current value being bound, if any, as it is available from the "this" *)
    (* pointer which gives the current closure itself. This is in the case e.g. let rec f = ... f ... *)
    let closed_fvs = 
        fvs.free_locvals
        |> Zset.elements 
        |> filter
          (fun fv -> 
            match storage_for_val m fv eenvouter with 
            | (StaticField _ | Method _ |  Unrealized | Null) -> false
            | _ -> 
                begin match selfv with 
                | Some v -> not (cenv.g.vref_eq (mk_local_vref fv) v) 
                | _ -> true
                end) in
    let closed_ftyvs = 
        fvs.free_tyvars.free_loctypars 
        |> Zset.elements in
    
    let attribs = [] in 
    let nfvs = length closed_fvs in

    if verbose then dprint_endline ("get_closure_freevars: compute eenvinner");
    (* If generating a named closure, add the closure itself as a var, available via "arg0" . *)
    (* The latter doesn't apply for the delegate implementation of closures. *)
    let eenvinner = 
        (env_for_typars closed_ftyvs eenvouter)
        |> (fun env -> { env with tyenv = { env.tyenv with tyenv_nativeptr_as_nativeint=true } } )
        |> add_storage_for_locvals cenv.g 
             (match selfv with
             | Some v  -> [(deref_val v,Arg 0)]
             | _ -> [])
        |> add_storage_for_locvals cenv.g 
            (list_mapi (fun i v -> 
                let repr = 
                  match storage_for_val m v eenvouter with 
                  | Local(_,localCloInfo) | Env(_,localCloInfo) -> localCloInfo
                  | _ -> None in 
                (v,Env(i,repr))) closed_fvs) in
    
    if verbose then dprint_endline ("get_closure_freevars: done");

    (attribs,closed_ftyvs,closed_fvs,tref,eenvinner)

and get_closure_il_frees cenv m takenNames eenvouter eenvinner closed_fvs =
    let il_freevar_names = choose_freevar_names takenNames (map name_of_val closed_fvs)    in
    let il_frees = map2 (fun fv nm -> mk_freevar (nm,gen_type_of_val_as_freevar cenv m eenvouter eenvinner fv)) closed_fvs il_freevar_names  in 
    il_frees

and get_cloinfo cenv m is_ltyfunc  selfv eenvouter expr =
    if verbose then dprint_endline ("get_cloinfo");     
    let (attribs,closed_ftyvs,closed_fvs,tref,eenvinner) = get_closure_freevars cenv m selfv eenvouter expr in 

    let clo_name = tname_of_tref tref in 

    if verbose then dprint_endline ("get_cloinfo: rty");     
    let rty = 
      match expr with 
      | TExpr_lambda (_,_,_,_,_,rty,_) | TExpr_tlambda(_,_,_,_,rty,_) -> rty
      | TExpr_obj(_,typ,_,_,_,_,_,_) -> typ
      | _ -> failwith "get_cloinfo: not a lambda expression" in 

    if verbose then dprint_endline ("get_cloinfo: get_args");     
    let rec get_args eenv ntmargs takenNames (e,rty) = 
        match e with 
        | TExpr_lambda (_,_,vs,body,m,bty,_) when not is_ltyfunc -> 
            (* Transform a lambda taking untupled arguments into one *)
            (* taking only a single tupled argument if necessary.  REVIEW: do this earlier *)
            let tupledv, body =  multi_lambda_to_tupled_lambda nng  vs body  in 
            let nm = name_of_val tupledv in
            let rty',l,arity_info,takenNames,(body',bty'),eenv = 
              let eenv = add_storage_for_val cenv.g (tupledv,Arg ntmargs) eenv in 
              get_args eenv (ntmargs + 1) (nm :: takenNames) (body,bty) in 
            rty',Lambdas_lambda (mk_named_param(nm,gen_type_of_val cenv eenv tupledv),l),1 :: arity_info,takenNames,(body',bty'),eenv
        | TExpr_tlambda(_,tvs,body,m,bty,_) -> 
            let rty',l,arity_info,takenNames,body,eenv = 
                  let eenv = add_typars_to_env tvs eenv in 
                  get_args eenv ntmargs takenNames (body,bty) in 
            rty',List.fold_right (fun tv sofar ->
                  let gp = gen_gparam m cenv.g eenv.tyenv tv in 
                  Lambdas_forall(gp,sofar)) tvs l,arity_info, takenNames,body,eenv
        | _ -> 
              let rty' = gen_type m cenv.g eenv.tyenv rty in 
              rty',Lambdas_return rty', [],takenNames,(e,rty),eenv in 

    (* start at arg number 1 as "this" pointer holds the current closure *) 
    let (rty',lambdas,narginfo,takenNames,(body,_),eenvinner) = get_args eenvinner 1 [] (expr,rty) in

    if verbose then dprintf0 "gen_static_formals m\n";
    let il_gparams = gen_static_formals m cenv eenvinner closed_ftyvs in 
    if verbose then dprintf0 "gen_static_actuals m\n";
    let il_gactuals = gen_static_actuals m eenvouter.tyenv closed_ftyvs in 
    if verbose then dprintf0 "done\n";

    let il_frees = get_closure_il_frees cenv m takenNames eenvouter eenvinner closed_fvs in
    
    let clospec = ClosureSpec(ClosureRef(tref, lambdas, il_frees), il_gactuals) in 
    let cloinfo = 
        { clo_name=clo_name;
          clo_arity_info =narginfo;
          clo_lambdas=lambdas;
          clo_il_frees = il_frees;
          clo_formal_il_rty=rty';
          clo_clospec = clospec;
          clo_il_gparams = il_gparams;
          clo_il_lambda_gparams_direct=[];
          clo_freevars=closed_fvs;
          clo_attribs=attribs } in 
    let cloinfo = if is_ltyfunc then adjust_named_local_tyfunc_cloinfo cloinfo else cloinfo in
    if verbose then dprint_endline ("<-- get_cloinfo");     
    cloinfo,body,eenvinner

(*--------------------------------------------------------------------------
!* Named local type functions
 *-------------------------------------------------------------------------- *)

and is_named_local_tyfunc_val v expr =
    isNone (arity_of_val v) &&
    is_forall_ty (type_of_val v) && 
    (let tps,_ = dest_forall_typ (type_of_val v) in tps |> List.exists (constraints_of_typar >> nonNil)) && 
    (match strip_expr expr with TExpr_tlambda _ -> true | _ -> false)
  
and gen_named_local_tyfunc_type cloinfo = 
    let clospec = cloinfo.clo_clospec in 
    let clo_tref = tref_of_clospec cloinfo.clo_clospec in 
    let clo_tspec = mk_tspec(clo_tref,inst_of_clospec clospec) in
    let il_typ = Type_boxed clo_tspec in
    il_typ
      
and adjust_named_local_tyfunc_cloinfo cloinfo = 
    let rec strip l = 
      match l with 
      | Lambdas_forall(gp,r) -> let gps, rty = strip r  in gp::gps,rty
      | Lambdas_return rty -> [],rty
      | _ -> failwith "adjust_named_local_tyfunc_cloinfo: local functions can currently only be type functions" in 
    let gps,rty = strip cloinfo.clo_lambdas in 
    if isNil gps then failwith "adjust_named_local_tyfunc_cloinfo: expected some type arguments on the type function";
    (* let rty' = gen_named_local_tyfunc_type cloinfo in  *)
    { cloinfo with 
            clo_il_lambda_gparams_direct=gps;
            clo_lambdas = Lambdas_return rty; 
            clo_formal_il_rty=rty }

(*--------------------------------------------------------------------------
!* Generate new delegate construction. This is a lot like generating closures
 * and object expressions, and most of the code is shared.
 *-------------------------------------------------------------------------- *)

and gen_delegate_expr cenv cgbuf eenvouter expr (TMethod((TSlotSig(_,delty, _,_,_, _) as slotsig),ovby_mtps,tmvs,body,implm),m) sequel =
    if verbose then dprint_endline ("gen_delegate_expr");     
    (* Get the instantiation of the delegate type *)
    let ctxt_il_delty = gen_type m cenv.g eenvouter.tyenv delty in 
    (* Yuck. TLBIMP.EXE generated APIs use UIntPtr for the delegate ctor. *)
    let useUIntPtrForDelegateCtor = 
        try 
            if is_il_named_typ delty then 
                let tcref = tcref_of_stripped_typ delty in 
                let _,_,tdef = dest_il_tcref  tcref in 
                match find_mdefs_by_name ".ctor" tdef.tdMethodDefs with 
                | [ctorMDef] -> 
                    begin match ctorMDef.mdParams with 
                    | [_;p2] -> (tname_of_tspec (tspec_of_typ p2.paramType) = "System.UIntPtr")
                    | _ -> false
                    end
                | _ -> false
            else false 
         with _ -> false in 
                
        
    (* Work out the free type variables for the morphing thunk *)
    let (attribs,closed_ftyvs,closed_fvs,delegee_tref,eenvinner) = get_closure_freevars cenv m None eenvouter expr in 
    let takenNames = map name_of_val tmvs in 
    let il_frees = get_closure_il_frees cenv m takenNames eenvouter eenvinner closed_fvs in
    let delegee_gparams = gen_static_formals m cenv eenvinner closed_ftyvs in 
    let delegee_tname = tname_of_tref delegee_tref in
    let delegee_gactuals_for_delegee = generalize_gparams delegee_gparams in

    (* Create a new closure class with a single "delegee" method that implements the delegate. *)
    let delegee_mname = "Invoke" in
    let delegee_tspec_for_delegee = mk_tspec (delegee_tref, delegee_gactuals_for_delegee) in 
    let attr = gen_closure_ctor_range cenv m in 

    let delegee_eenv_under_typars = add_typars_to_env ovby_mtps eenvinner in  
    (* The slot sig contains a formal instantiation.  When creating delegates we're only *)
    (* interested in the actual instantiation since we don't have to emit a method impl. *)
    let delegee_il_params,delegee_il_ret = gen_actual_slotsig m cenv delegee_eenv_under_typars slotsig ovby_mtps in

    let numthis = 1 in 
    let delegee_meth_env = add_storage_for_locvals cenv.g (list_mapi (fun i v -> (v,Arg (i+numthis))) tmvs)  delegee_eenv_under_typars in  
    let ilmbody = codegen_method_for_expr cenv cgbuf.mgbuf ([],delegee_mname,delegee_meth_env,1,0,body,(if slotsig_has_void_rty cenv.g slotsig then discardAndReturnVoid else Return)) in
    let delegee_mdef =
      mk_instance_mdef
        (delegee_mname,MemAccess_assembly, 
         delegee_il_params, 
         delegee_il_ret,
         MethodBody_il ilmbody) in
    let delegee_ctor = mk_simple_storage_ctor attr (Some cenv.g.ilg.tspec_Object) delegee_tspec_for_delegee [] in  
    let ctor_body = ilmbody_of_mdef delegee_ctor in

    let lambdas = Lambdas_return ctxt_il_delty in 
    let attrs = gen_attrs cenv eenvinner attribs in 
    let clo = mk_closure_tdef cenv (delegee_tref,delegee_gparams,attrs,m,il_frees,lambdas,ctor_body,[delegee_mdef],[],cenv.g.ilg.typ_Object,[]) in
    insert_tdef cgbuf.mgbuf delegee_tref clo;
    closure_counter();
    let ctxt_gactuals_for_delegee = gen_static_actuals m eenvouter.tyenv closed_ftyvs in 
    let clospec = ClosureSpec(ClosureRef(delegee_tref, lambdas, il_frees), ctxt_gactuals_for_delegee) in 
    gen_get_locvals cenv cgbuf eenvouter m closed_fvs;
    CG.emit_instr cgbuf (replicate (length il_frees) Pop@ [ Push (Pubclo.typ_of_lambdas cenv.g.ilxPubCloEnv lambdas)]) (mk_IlxInstr (EI_newclo clospec));
    let ctxt_tspec_for_delegee = mk_tspec (delegee_tref,ctxt_gactuals_for_delegee) in 
    let ctxt_mspec_for_delegee = mk_nongeneric_instance_mspec_in_boxed_tspec (ctxt_tspec_for_delegee,"Invoke",map typ_of_param delegee_il_params, delegee_il_ret.returnType) in 
    let delegate_ctor_mspec = mk_ctor_mspec_for_delegate cenv.g.ilg (tref_of_typ ctxt_il_delty,Il.inst_of_typ ctxt_il_delty,useUIntPtrForDelegateCtor) in 
    CG.emit_instrs cgbuf 
      [Push cenv.g.ilg.typ_int32; Pop; Pop; Push ctxt_il_delty]
      [ I_ldftn ctxt_mspec_for_delegee; 
        I_newobj(delegate_ctor_mspec,None) ];
    gen_sequel cenv eenvouter.cloc cgbuf sequel

(*-------------------------------------------------------------------------
!* Generate statically-resolved conditionals used for type-directed optimizations.
 *------------------------------------------------------------------------- *)
    
and gen_static_optimization cenv cgbuf eenv (constraints,e2,e3,m) sequel = 
    let e = 
      if for_all (static_optimization_constraint_definitely_satisfied cenv.g) constraints then e2 
      else e3 in 
    gen_expr cenv cgbuf eenv e sequel


(*-------------------------------------------------------------------------
!* Generate discrimination trees
 *------------------------------------------------------------------------- *)

and sequel_is_immediate  sequel = 
    match sequel with 
    (* All of these can be done at the end of each branch - we don't need a real join point *)
    | Return | ReturnVoid | Br _ | LeaveHandler _  -> true
    | DiscardThen sequel -> sequel_is_immediate  sequel
    | _ -> false

and gen_join_point cenv cgbuf pos eenv ty m sequel = 
    if verbose then dprint_endline ("gen_join_point");      
    match sequel with 
    (* All of these can be done at the end of each branch - we don't need a real join point *)
    | _ when sequel_is_immediate sequel -> 
        let stack_after = CG.curr_stack cgbuf in 
        let after = CG.emit_delay_mark cgbuf (pos^"_join")  in 
        sequel,after,stack_after,Continue
    (* We end scopes at the join point, if any *)
    | EndLocalScope(sq,mark) -> 
        let sequel_now,after,stack_after,sequel_after = gen_join_point cenv cgbuf pos eenv ty m sq  in 
        sequel_now,after,stack_after,EndLocalScope(sequel_after,mark)
    (* If something non-trivial happens after a discard then generate a join point, but first discard the value (often this means we won't generate it at all) *)
    | DiscardThen sequel -> 
        let stack_after =  CG.curr_stack cgbuf in 
        let after = CG.emit_delay_mark cgbuf (pos^"_join")  in 
        DiscardThen (Br after),after,stack_after,sequel
    (* The others (e.g. Continue, LeaveFilter and CmpThenBrOrContinue) can't be done at the end of each branch. We must create a join point. *)
    | _ -> 
        let pushed = gen_type m cenv.g eenv.tyenv ty in 
        let stack_after = (pushed :: (CG.curr_stack cgbuf)) in 
        let after = CG.emit_delay_mark cgbuf (pos^"_join")  in 
        (* go to the join point *)
        Br after, after,stack_after,sequel
        
and gen_match cenv cgbuf eenv (exprm,tree,targets,m,ty) sequel =
    if verbose then dprintf1 "gen_match, dtree = %s\n" (showL (dtreeL tree));      
(* First try the common cases where we don't need a join point. *)
    match tree with 
    | TDSuccess(es,n) -> failwith "internal error: matches that immediately succeed should have been normalized using mk_and_optimize_match"
    | _ -> 
        (* Create a join point *)
        let stack_at_dests = CG.curr_stack cgbuf in (* the stack at the r.h.s. of each clause *)
        let (sequel_on_branches,after,stack_after,sequel_after) = gen_join_point cenv cgbuf "match" eenv ty m sequel in 
        gen_dtree_and_targets cenv cgbuf stack_at_dests eenv tree targets sequel_on_branches; 
        CG.set_mark_to_here cgbuf after;
        CG.set_stack cgbuf stack_after;
        gen_sequel cenv eenv.cloc cgbuf sequel_after

and gen_dtree_and_targets cenv cgbuf stack_at_dests eenv tree targets sequel = 
    ignore (gen_dtree_and_targets_graph cenv cgbuf (CG.emit_delay_mark cgbuf "start_dtree") stack_at_dests eenv tree targets (Imap.empty()) sequel)

and get_prev_target rgraph n =  Imap.tryfind n rgraph 

and gen_dtree_and_targets_graph cenv cgbuf inplab stack_at_dests eenv tree targets rgraph sequel = 
    if verbose then dprintf1 "gen_dtree_and_targets_graph, dtree = %s\n" (showL (dtreeL tree));      
    match tree with 
    | TDBind(bind,rest) -> 
       CG.set_mark_to_here cgbuf inplab;
       let _,end_scope as scopeMarks = start_local_scope "dtree_bind" cgbuf in  
       let eenv = alloc_vals_for_binds cenv cgbuf scopeMarks eenv [bind] in
       gen_bind cenv cgbuf eenv bind;
       (* We don't get the scope marks quite right for dtree-bound variables.  This is because *)
       (* we effectively lose an EndLocalScope for all dtrees that go to the same target *)
       (* So we just pretend that the variable goes out of scope here. It's only debug info after all... *)
       CG.set_mark_to_here cgbuf end_scope;
       let body_label = CG.emit_delay_mark cgbuf "dtree_bind_body" in 
       CG.emit_instr cgbuf [] (I_br (code_label_of_mark body_label)); 
       gen_dtree_and_targets_graph cenv cgbuf body_label stack_at_dests eenv rest targets rgraph sequel

    | TDSuccess (es,n) ->  
       gen_success cenv cgbuf inplab stack_at_dests eenv es n targets rgraph sequel 

    | TDSwitch(e, cases, dflt,m)  -> 
       gen_switch cenv cgbuf inplab stack_at_dests eenv e cases dflt m targets rgraph sequel 

and get_target targets n =
    if n >= Array.length targets then failwith "get_target: target not found in decision tree";
    targets.(n)

and gen_success cenv cgbuf inplab stack_at_dests eenv es n targets rgraph sequel = 
    if verbose then dprint_endline ("gen_success");      
    let (TTarget(vs,success_rhs)) = get_target targets n in 
    match get_prev_target rgraph n with
    | Some (success,(eenvrhs,rhsm)) ->
      (* If not binding anything we can go directly to the success point *)
      (* This is useful to avoid lots of branches e.g. in match A | B | C -> e *)
      (* In this case each case will just go straight to "e" *)
        if isNil vs then begin
          CG.set_mark cgbuf inplab success;
          rgraph
        end else begin
          CG.set_mark_to_here cgbuf inplab;
          CG.emit_seqpoint cenv cgbuf rhsm;
          iter2 (gen_set_locval_to_expr cenv cgbuf eenvrhs eenv ) vs es;
          CG.emit_instr cgbuf [] (I_br (code_label_of_mark success)); 
          rgraph
        end
    | None -> 
        CG.set_mark_to_here cgbuf inplab;
        let rhsm = range_of_expr success_rhs in 
        CG.emit_seqpoint cenv cgbuf rhsm;
        let binds = mk_binds vs es in
        let _,end_scope as scopeMarks = start_local_scope "matchrhs" cgbuf in 
        let eenvrhs = alloc_vals_for_binds cenv cgbuf scopeMarks eenv binds in
        gen_binds cenv cgbuf eenvrhs binds;
        let success = CG.mark cgbuf "matching_rhs" in
        CG.set_stack cgbuf stack_at_dests;
        gen_expr cenv cgbuf eenvrhs success_rhs (EndLocalScope(sequel,end_scope));
        (* add the generated rhs. to the graph *)
        Imap.add n (success,(eenvrhs,rhsm)) rgraph

and gen_switch cenv cgbuf inplab stack_at_dests eenv e cases dflt_opt switchm targets rgraph sequel = 
    if verbose then dprint_endline ("gen_switch:");  
    let m = (range_of_expr e) in 
    CG.set_mark_to_here cgbuf inplab;
    CG.emit_seqpoint cenv cgbuf m;
    match cases with 
      (* optimize a test against a boolean value, i.e. the all-important if-then-else *)
      | TCase(TTest_const(TConst_bool b), success_dtree) :: _  ->  
       let failure_dtree = if dflt_opt = None then dest_of_case (hd (tl cases)) else the dflt_opt  in 
       gen_test cenv eenv.cloc cgbuf stack_at_dests e None eenv (if b then success_dtree else  failure_dtree) (if b then failure_dtree else success_dtree) targets rgraph sequel 

      (* optimize a single test for a type constructor to an "isdata" test - much more efficient code, and this case occurs in the generated equality testers where perf is important *)
      | TCase(TTest_unionconstr(c,tyargs), success_dtree) :: rest when List.length rest = (match dflt_opt with None -> 1 | Some x -> 0)  ->  
        let failure_dtree = if dflt_opt = None then dest_of_case (hd (tl cases)) else the dflt_opt  in 
        let cuspec = gen_cuspec m cenv eenv.tyenv (tcref_of_ucref c) tyargs in 
        let idx = ucref_index c in 
        gen_test cenv eenv.cloc cgbuf stack_at_dests e (Some ([Pop; Push cenv.g.ilg.typ_bool],(mk_IlxInstr (EI_isdata (cuspec, idx))))) eenv success_dtree failure_dtree targets rgraph sequel

      | _ ->  
        let case_labels = map (fun _ -> CG.emit_delay_mark cgbuf "switch_case") cases in 
        let dflt_label = match dflt_opt with None -> hd case_labels | Some _ -> CG.emit_delay_mark cgbuf "switch_dflt" in
        let fst_discrim =  discrim_of_case (hd cases) in 
        begin match fst_discrim with 
        (* Iterated tests, e.g. exception constructors, nulltests, typetests and active patterns - these should always have one positive and one negative branch *)
        | TTest_isinst _  
        | TTest_array_length _
        | TTest_isnull -> 
            if length cases <> 1 || isNone dflt_opt then failwith "internal error: gen_switch: TTest_isinst/isnull/query";
            let bi = 
              match fst_discrim with 
              | TTest_isnull -> 
                  gen_expr cenv cgbuf eenv e Continue; BI_brfalse
              | TTest_isinst (srcty,tgty) -> 
                  if cenv.g.typeCheckerConfiguredToAssumeErasureOfGenerics then (
                      gen_expr cenv cgbuf eenv e Continue;
                      let il_tgty = gen_type m cenv.g eenv.tyenv tgty in 
                      CG.emit_instr cgbuf [Pop; Push il_tgty] (I_isinst il_tgty)
                  ) else (
                      gen_expr cenv cgbuf eenv (mk_call_istype cenv.g m tgty e) Continue
                  );
                  BI_brtrue
              | _ -> failwith "internal error: gen_switch" in 
            CG.emit_instr cgbuf [Pop] (I_brcmp (bi,code_label_of_mark (hd case_labels),code_label_of_mark dflt_label));
            gen_cases cenv cgbuf stack_at_dests eenv targets rgraph case_labels cases dflt_opt dflt_label sequel
              
        | TTest_query _ -> error(Error("internal error in codegen: TTest_query",switchm))
        | TTest_unionconstr (hdc,tyargs) -> 
            gen_expr cenv cgbuf eenv e Continue;
            let tycon,constr = deref_uconstr hdc in 
            let cuspec = gen_cuspec m cenv eenv.tyenv (tcref_of_ucref hdc) tyargs in 
            let dests = 
              if List.length cases <> List.length case_labels then failwith "internal error: TTest_unionconstr";
              map2
                (fun case label  ->
                  match case with 
                  | TCase(TTest_unionconstr (c,_),_) -> (ucref_index c, code_label_of_mark label) 
                  | _ -> failwith "error: mixed constructor/const test?") 
                cases 
                case_labels in 
            
            CG.emit_instr cgbuf [Pop] (mk_IlxInstr (EI_datacase (false,cuspec,dests, code_label_of_mark dflt_label)));
            gen_cases cenv cgbuf stack_at_dests eenv  targets rgraph case_labels cases dflt_opt dflt_label sequel
              
        | TTest_const c ->
            gen_expr cenv cgbuf eenv e Continue;
            begin match c with (TConst_bool b) -> failwith "should have been done earlier"
            |  (TConst_int8 _)            
            |  (TConst_int16 _)           
            |  (TConst_int32 _)           
            |  (TConst_uint8 _)           
            |  (TConst_uint16 _)          
            |  (TConst_uint32 _)          
            |  (TConst_char _) ->
                if List.length cases <> List.length case_labels then failwith "internal error: ";
                let dests = 
                  map2
                    (fun case label  ->
                      let i = 
                        match discrim_of_case case with 
                          TTest_const c' ->
                            begin match c' with 
                            | TConst_int8 i -> Nums.i8_to_i32 i
                            | TConst_int16 i -> Nums.i16_to_i32 i
                            | TConst_int32 i -> i
                            | TConst_uint8 i -> Nums.u8_to_i32 i
                            | TConst_uint16 i -> Nums.u16_to_i32 i
                            | TConst_uint32 i -> Nums.u32_to_i32 i
                            | TConst_char c -> Nums.u16_to_i32 (Nums.unichar_to_u16 c)  
                            | _ -> failwith "internal error: badly formed const test"  
                            end
                        | _ -> failwith "internal error: badly formed const test"  in
                      (i,code_label_of_mark label))
                    cases 
                    case_labels in 
                let mn = fold_right (fst >> min) dests (fst(hd dests)) in 
                let mx = fold_right (fst >> max) dests (fst(hd dests)) in 
                (* Check if it's worth using a switch *)
                (* REVIEW: this is using switches even for single integer matches! *)
                if Int32.sub mx mn = Int32.of_int (length dests - 1) then
                  begin 
                    let dest_labels = map snd (sort (fun (i1,l1) (i2,l2) -> compare i1 i2) dests) in
                    if mn <> Int32.zero then 
                      CG.emit_instrs cgbuf [Push cenv.g.ilg.typ_int32; Pop] [ mk_ldc_i32 mn;I_arith AI_sub ];
                    CG.emit_instr cgbuf [Pop] (I_switch (dest_labels, code_label_of_mark dflt_label));
                  end
                else
                  error(InternalError("non-dense integer matches not implemented in codegen - these should have been removed by the pattern match compiler",switchm));
                gen_cases cenv cgbuf stack_at_dests eenv  targets rgraph case_labels cases dflt_opt dflt_label sequel
            | _ -> error(InternalError("these matches should never be needed",switchm))
            end
        end 

and gen_cases cenv cgbuf stack_at_dests eenv targets rgraph case_labels cases dflt_opt dflt_label sequel =
    if verbose then dprint_endline ("gen_cases:");
    let rgraph = 
      match dflt_opt with 
      | Some dflt_rhs -> gen_dtree_and_targets_graph cenv cgbuf dflt_label stack_at_dests eenv dflt_rhs targets rgraph sequel
      | None -> rgraph in 
    let rgraph = 
      fold_left2 
        (fun rgraph case_label (TCase(_,case_rhs)) -> 
          gen_dtree_and_targets_graph cenv cgbuf case_label stack_at_dests eenv case_rhs targets rgraph sequel)
        rgraph
        case_labels
        cases in 
    rgraph 
      
and gen_test cenv cloc cgbuf stack_at_dests e tester eenv success_dtree failure_dtree targets rgraph sequel =
    if verbose then dprint_endline ("gen_test:");      
    match success_dtree,failure_dtree with 
    (* Peephole: if generating a boolean value or its negation then just leave it on the stack *)
    (* This comes up in the generated equality functions.  REVIEW: do this as a peephole optimization elsewhere *)
    | TDSuccess([],n1), 
      TDSuccess([],n2) when 
         (match get_target targets n1, get_target targets n2 with 
           TTarget(_,TExpr_const(TConst_bool b1,_,_)),TTarget(_,TExpr_const(TConst_bool b2,_,_)) -> b1 = not b2
         | _ -> false) ->
             begin match get_target targets n1, get_target targets n2 with 
               TTarget(_,TExpr_const(TConst_bool b1,_,_)),_ -> 
                 gen_expr cenv cgbuf eenv e Continue;
                 (match tester with Some (pushpop,i) -> CG.emit_instr cgbuf pushpop i; | _ -> ());
                 if not b1 then 
                   CG.emit_instrs cgbuf [Push cenv.g.ilg.typ_bool; Pop] [mk_ldc_i32 (Int32.of_int 0); I_arith AI_ceq];
                 gen_sequel cenv cloc cgbuf sequel;
                 rgraph
             | _ -> failwith "internal error: gen_test during bool elim"
             end

    | _ ->
        let success = CG.emit_delay_mark cgbuf "test_success" in 
        let failure = CG.emit_delay_mark cgbuf "test_failure" in 
        (match tester with 
        | None -> 
            (* generate the expression, then test it for "false" *)
            gen_expr cenv cgbuf eenv e (CmpThenBrOrContinue([Pop],I_brcmp (BI_brfalse, code_label_of_mark failure,code_label_of_mark success)));

        (* Turn "EI_isdata" tests that branch into EI_brisdata tests *)
        | Some (_,I_other i) when is_ilx_ext_instr i && (match dest_ilx_ext_instr i with EI_isdata _ -> true | _ -> false) ->
            let (cuspec,idx) = match dest_ilx_ext_instr i with EI_isdata (cuspec,idx) -> (cuspec,idx) | _ -> failwith "??" in 
            gen_expr cenv cgbuf eenv e (CmpThenBrOrContinue([Pop],mk_IlxInstr (EI_brisdata (cuspec, idx, code_label_of_mark success,code_label_of_mark failure))));
        | Some (pushpop,i) ->
            gen_expr cenv cgbuf eenv e Continue;
            CG.emit_instr cgbuf pushpop i;
            CG.emit_instr cgbuf [Pop] (I_brcmp (BI_brfalse, code_label_of_mark failure,code_label_of_mark success)));
        let rgraph = gen_dtree_and_targets_graph cenv cgbuf success stack_at_dests eenv success_dtree targets rgraph sequel in 
        gen_dtree_and_targets_graph cenv cgbuf failure stack_at_dests eenv failure_dtree targets rgraph sequel 

(*-------------------------------------------------------------------------
!* Generate letrec bindings
 *------------------------------------------------------------------------- *)

and gen_setenv cenv cgbuf eenv (clospec,e,n,e2,m) =
    if verbose then dprint_endline ("gen_setenv:");   
    gen_expr cenv cgbuf eenv e Continue;
    CG.emit_instrs cgbuf [] [ mk_IlxInstr (EI_castclo clospec) ];
    gen_expr cenv cgbuf eenv e2 Continue;
    CG.emit_instrs cgbuf [Pop; Pop] [ mk_IlxInstr (EI_stclofld(clospec, n)) ]

and gen_letrec_binds cenv cgbuf eenv (allBinds,m) =
    if verbose then dprint_endline ("gen_letrec");
    (* Fix up recursion for non-toplevel recursive bindings *)
    let bindsPossiblyRequiringFixup = 
        allBinds |> filter (fun b -> 
            match (storage_for_val m (var_of_bind b) eenv) with  
            | Method _ 
            | Unrealized 
            (* Note: Recursive data stored in static fields may require fixups e.g. let x = C(x) *) 
            (* | StaticField _  *)
            | Null -> false 
            | _ -> true) in

    let computeFixupsForOneRecursiveVar boundv forwardReferenceSet fixups selfv access set e =
        match e with 
        | TExpr_lambda _ | TExpr_tlambda _ | TExpr_obj _ -> 
            let is_ltyfunc = (isSome selfv && (is_named_local_tyfunc_val (the selfv) e)) in
            let selfv = (match e with TExpr_obj _ -> None | _ when is_ltyfunc -> None | _ -> Option.map mk_local_vref selfv) in
            let clo,_,eenvclo =  get_cloinfo cenv m is_ltyfunc selfv {eenv with  letBoundVars=(mk_local_vref boundv)::eenv.letBoundVars}  e  in 
            clo.clo_freevars |> iter (fun fv -> 
                if Zset.mem fv forwardReferenceSet then 
                    match storage_for_val m fv eenvclo with
                    | Env (n,_) -> fixups := (boundv, fv, (fun () -> gen_setenv cenv cgbuf eenv (clo.clo_clospec,access,n,expr_for_val m fv,m))) :: !fixups
                    | _ -> error (Error("gen_letrec: "^name_of_val fv^" was not in the environment",m)) )
              
        | TExpr_val  (vref,_,m) -> 
            let fv = deref_val vref in 
            let needsFixup = Zset.mem fv forwardReferenceSet in 
            if verbose then dprintf2 "gen_letrec: needsFixup %s? %b\n" (showL(vrefL vref)) needsFixup;
            if needsFixup then fixups := (boundv, fv,(fun () -> gen_expr cenv cgbuf eenv (set e) discard)) :: !fixups
        | _ -> failwith "compute real fixup vars" in 


    let fixups = ref [] in 
    let recursiveVars = (Zset.addL (map var_of_bind bindsPossiblyRequiringFixup) (Zset.empty val_spec_order))  in 
    if verbose then dprintf0 "gen_letrec: compute fixups\n";
    fold_left 
        (fun forwardReferenceSet bind ->
            let valBeingDefined = var_of_bind bind in 
            if verbose then dprint_endline ("gen_letrec: compute fixups for "^showL(vspecL valBeingDefined));
            bind |> rhs_of_bind |> iter_letrec_fixups cenv.g (Some valBeingDefined)  (computeFixupsForOneRecursiveVar valBeingDefined forwardReferenceSet fixups) (expr_for_val m valBeingDefined, (fun e -> failwith ("internal error: should never need to set non-delayed recursive val: " ^ name_of_val valBeingDefined)));
            Zset.remove valBeingDefined forwardReferenceSet)
        recursiveVars
        bindsPossiblyRequiringFixup |> ignore;

    fold_left 
        (fun forwardReferenceSet bind ->
            let valBeingDefined = var_of_bind bind in 
            if verbose then dprint_endline ("gen_letrec: generate binding for "^showL(vspecL valBeingDefined));
            gen_bind cenv cgbuf eenv bind;
            (* execute and discard any fixups that can now be committed *)
            let forwardReferenceSet = Zset.remove valBeingDefined forwardReferenceSet in 
            fixups := !fixups |> filter (fun (boundv, fv, action) -> if (Zset.mem boundv forwardReferenceSet or Zset.mem fv forwardReferenceSet) then  true else (action(); false));
            forwardReferenceSet)
        recursiveVars
        allBinds |> ignore;


and gen_letrec cenv cgbuf eenv (binds,body,m) sequel =
    let _,end_scope as scopeMarks = start_local_scope "letrec" cgbuf in 
    if verbose then dprint_endline ("gen_letrec");
    let eenv = alloc_vals_for_binds cenv cgbuf scopeMarks eenv binds in
    gen_letrec_binds cenv cgbuf eenv (binds,m);
    if verbose then dprint_endline ("gen_letrec: body");      
    gen_expr cenv cgbuf eenv body (EndLocalScope(sequel,end_scope))

(*-------------------------------------------------------------------------
!* Generate simple bindings
 *------------------------------------------------------------------------- *)

and gen_bind cenv cgbuf eenv (TBind(vspec,e)) =
    if verbose then dprint_endline ("gen_bind");     
    (* Record the closed reflection definition if publishing *)
    begin match pubpath_of_val vspec, published_closed_defn_of_val vspec with 
    | Some p, Some e -> insert_renv_def cgbuf.mgbuf p (type_of_val vspec) e
    | _  -> ()
    end;

    if verbose then dprint_endline ("gen_bind: " ^ showL(vspecAtBindL vspec));      
    let eenv = {eenv with letBoundVars= (mk_local_vref vspec) :: eenv.letBoundVars} in 
    let access = computeMemberAccess (isHiddenVal eenv.mrmi vspec || not (modbind_of_val vspec)) in
    
    let m = range_of_val vspec in 
   
    match storage_for_val m vspec eenv with 
    | Unrealized -> ()
    | Method (arity_info,_,mspec,_,paramInfos,retInfo)  ->
        let tps,basevopt,vsl,body',bodyty = iterated_adjust_arity_of_lambda_body cenv.g cenv.amap nng arity_info e in
        let method_vs = concat vsl in 
        gen_bind_as_method cenv cgbuf eenv (vspec,mspec,access,paramInfos,retInfo) (arity_info,basevopt,tps,method_vs, body', bodyty)
    | StaticField (fspec,hasLiteralAttr,binding_tspec,fieldName,propName,fty,get_mref,set_mref) ->  

        let mut = (mutability_of_val vspec <> Immutable) in 
        
        begin match mut,hasLiteralAttr,e with 
        | _,false,_ -> ()
        | true,true,_ -> errorR(Error("Values marked with 'LiteralAttribute' may not be mutable",m)) 
        | _,true,TExpr_const _ -> ()
        | _,true,_ -> errorR(Error("Values marked with 'LiteralAttribute' must currently be simple integer, character, Boolean, string or floating point constants",m)) 
        end;          

        (* Generate a static field definition and the get/set properties to access it. *)
        
        let il_attribs = gen_attrs cenv eenv (attribs_of_val vspec) in 
        let fdef = 
          let access = computeMemberAccess (not hasLiteralAttr or isHiddenVal eenv.mrmi vspec) in
          let fdef = mk_static_fdef(fieldName,fty, None, None, access) in 
          { fdef with fdCustomAttrs = mk_custom_attrs il_attribs } in 
          

        let fdef =
            match hasLiteralAttr,e with 
            | false,_ -> fdef
            | true,TExpr_const(konst,m,_) -> { fdef with fdLiteral=true; fdInit= Some(gen_field_init m konst) }
            | true,_ -> fdef (* error given above *) in
          
        let curr_tref = tref_of_tspec (mk_tspec_for_cloc eenv.cloc) in 
        let binding_tref = tref_of_tspec binding_tspec in 
        let field_tref = tref_of_fspec fspec in 
        let is_cctor = (cgbuf.methodName = ".cctor") in
        let is_main = (cgbuf.methodName = mainMethName) in
        let fdef = if mut || (not is_cctor) || hasLiteralAttr then fdef else {fdef with fdInitOnly=true } in 
        insert_fdef cgbuf.mgbuf field_tref fdef;
        static_field_def_counter();
        if not hasLiteralAttr then begin
            let pdef = 
                         { propName=propName;
                           propRTSpecialName=false;
                           propSpecialName=false;
                           propSet=if mut then Some(set_mref) else None;
                           propGet=Some(get_mref);
                           propCallconv=CC_static;
                           propType=fty;          
                           propInit=None;
                           propArgs=[];
                           propCustomAttrs=mk_custom_attrs (il_attribs @ [mk_CompilationMappingAttr cenv.g (9 (* SourceLevelConstruct.Value *))]); } in
             let code_attr = gen_opt_range cenv (range_of_val vspec) in
             insert_or_merge_pdef cgbuf.mgbuf binding_tref pdef;
             let sp = match code_attr with None -> [] | Some attr -> [ I_seqpoint attr ] in
             let get_mdef = 
                 mk_static_mdef([],name_of_mref get_mref,access,[],mk_return fty,
                                MethodBody_il(mk_ilmbody(true,[],2,nonbranching_instrs_to_code(sp @ [ mk_normal_ldsfld fspec ]),code_attr))) in 
             let set_mdefs = 
                 if mut then 
                    Some(mk_static_mdef([],name_of_mref set_mref,access,[mk_named_param("value",fty)],mk_return Type_void,
                                        MethodBody_il(mk_ilmbody(true,[],2,nonbranching_instrs_to_code(sp @[ ldarg_0;mk_normal_stsfld fspec]),code_attr))))
                  else None in 
            get_mdef |> insert_mdef cgbuf.mgbuf binding_tref ;
            set_mdefs |> Option.iter (insert_mdef cgbuf.mgbuf binding_tref);
            gen_bind_rhs cenv cgbuf eenv vspec e;
            (* We MUST MUST MUST be in the cctor. *)
            gen_set_top_field cgbuf fspec
        end;

    | _ ->
        gen_set_locval_to_expr cenv cgbuf eenv eenv vspec e

(*-------------------------------------------------------------------------
!* Generate method bindings
 *------------------------------------------------------------------------- *)

(* Spectacularly gross table encoding P/Invoke and COM marshalling information *)
and gen_marshal cenv attribs = 
    match fsthing_tryfind_attrib cenv.g cenv.g.attrib_MarshalAsAttribute attribs with
    | Some (Attrib(_,[ TExpr_const (TConst_int32(unmanagedType),_,_) ],namedArgs))  -> 
        let find_int32,find_bool,find_string = decodeAttribNamedArgs namedArgs in 
        let rec decodeUnmanagedType unmanagedType = 
           (* enumeration values for System.Runtime.InteropServices.UnmanagedType taken from mscorlib.il *)
            match  unmanagedType with 
            | 0x0l -> NativeType_empty
            | 0x01l -> NativeType_void
            | 0x02l -> NativeType_bool
            | 0x03l -> NativeType_int8
            | 0x04l -> NativeType_unsigned_int8
            | 0x05l -> NativeType_int16
            | 0x06l -> NativeType_unsigned_int16
            | 0x07l -> NativeType_int32
            | 0x08l -> NativeType_unsigned_int32
            | 0x09l -> NativeType_int64
            | 0x0Al -> NativeType_unsigned_int64
            | 0x0Bl -> NativeType_float32
            | 0x0Cl -> NativeType_float64
            | 0x0Fl -> NativeType_currency
            | 0x13l -> NativeType_bstr
            | 0x14l -> NativeType_lpstr
            | 0x15l -> NativeType_lpwstr
            | 0x16l -> NativeType_lptstr
            | 0x17l -> NativeType_fixed_sysstring (find_int32 "SizeConst" 0x0l)
            | 0x19l -> NativeType_iunknown
            | 0x1Al -> NativeType_idsipatch
            | 0x1Bl -> NativeType_struct
            | 0x1Cl -> NativeType_interface
            | 0x1Dl -> 
                let safeArraySubType = 
                    match find_int32 "SafeArraySubType" 0x0l with 
                    (* enumeration values for System.Runtime.InteropServices.VarType taken from mscorlib.il *)
                    | 0x0l -> VariantType_empty
                    | 0x1l -> VariantType_null                  
                    | 0x02l -> VariantType_int16                
                    | 0x03l -> VariantType_int32                
                    | 0x0Cl -> VariantType_variant              
                    | 0x04l -> VariantType_float32              
                    | 0x05l -> VariantType_float64              
                    | 0x06l -> VariantType_currency             
                    | 0x07l -> VariantType_date                 
                    | 0x08l -> VariantType_bstr                 
                    | 0x09l -> VariantType_idispatch            
                    | 0x0al -> VariantType_error                
                    | 0x0bl -> VariantType_bool                 
                    | 0x0dl -> VariantType_iunknown             
                    | 0x0el -> VariantType_decimal              
                    | 0x10l -> VariantType_int8                 
                    | 0x11l -> VariantType_unsigned_int8        
                    | 0x12l -> VariantType_unsigned_int16       
                    | 0x13l -> VariantType_unsigned_int32       
                    | 0x15l -> VariantType_unsigned_int64       
                    | 0x16l -> VariantType_int                  
                    | 0x17l -> VariantType_unsigned_int         
                    | 0x18l -> VariantType_void                 
                    | 0x19l -> VariantType_hresult              
                    | 0x1al -> VariantType_ptr                  
                    | 0x1cl -> VariantType_carray               
                    | 0x1dl -> VariantType_userdefined          
                    | 0x1el -> VariantType_lpstr                
                    | 0x1Bl -> VariantType_safearray            
                    | 0x1fl -> VariantType_lpwstr               
                    | 0x24l -> VariantType_record               
                    | 0x40l -> VariantType_filetime             
                    | 0x41l -> VariantType_blob                 
                    | 0x42l -> VariantType_stream               
                    | 0x43l -> VariantType_storage              
                    | 0x44l -> VariantType_streamed_object      
                    | 0x45l -> VariantType_stored_object        
                    | 0x46l -> VariantType_blob_object          
                    | 0x47l -> VariantType_cf                   
                    | 0x48l -> VariantType_clsid                
                    | 0x14l -> VariantType_int64                
                    | _ -> VariantType_empty in
                let safeArrayUserDefinedSubType = 
                    match find_string "SafeArrayUserDefinedSubType" "" with 
                    | "" -> None
                    | res -> Some res in 
               NativeType_safe_array(safeArraySubType,safeArrayUserDefinedSubType)
            | 0x1El -> NativeType_fixed_array  (find_int32 "SizeConst" 0x0l)
            | 0x1Fl -> NativeType_int
            | 0x20l -> NativeType_unsigned_int
            | 0x22l -> NativeType_byvalstr
            | 0x23l -> NativeType_ansi_bstr
            | 0x24l -> NativeType_tbstr
            | 0x25l -> NativeType_variant_bool
            | 0x26l -> NativeType_method
            | 0x28l -> NativeType_as_any
            | 0x2Al -> 
               let sizeParamIndex = 
                    match find_int32 "SizeParamIndex" (-1l) with 
                    | -1l -> None
                    | res -> Some (res,None) in 
               let arraySubType = 
                    match find_int32 "ArraySubType" (-1l) with 
                    | -1l -> None
                    | res -> Some (decodeUnmanagedType res) in 
               NativeType_array(arraySubType,sizeParamIndex) 
            | 0x2Bl -> NativeType_lpstruct
            | 0x2Cl -> NativeType_empty
               (* failwith "Custom marshallers may not be consumed by F#. Consider using a C# helper function" *)
               (* NativeType_custom of bytes * string * string * bytes (* guid,nativeTypeName,custMarshallerName,cookieString *) *)
            | 0x2Dl -> NativeType_error  
            | _ -> NativeType_empty in
        Some(decodeUnmanagedType unmanagedType)
    | _ -> None 

and gen_param_attribs cenv attribs =
    let inFlag = isSome (fsthing_tryfind_attrib cenv.g cenv.g.attrib_InAttribute attribs) in
    let outFlag = isSome (fsthing_tryfind_attrib cenv.g cenv.g.attrib_OutAttribute attribs) in
    let optionalFlag = isSome (fsthing_tryfind_attrib cenv.g cenv.g.attrib_OptionalAttribute attribs) in
    let paramMarshal =  gen_marshal cenv attribs in 
    inFlag,outFlag,optionalFlag,paramMarshal

and gen_params cenv eenv mspec attribs implValsOpt =
    let il_argtys = (formal_args_of_mspec mspec) in 
    let argInfosAndTypes = 
        (* if List.length attribs <> List.length il_argtys then dprintf3 "Note: nm = %s, length il_argtys = %d, length attribs = %d\n" (name_of_mspec mspec) (length il_argtys) (length attribs);  *)
        if List.length attribs = List.length il_argtys then combine il_argtys attribs
        else map (fun il_ty -> il_ty,TopValData.unnamedTopArg1) il_argtys  in
    let argInfosAndTypes = 
        match implValsOpt with 
        | Some(implVals) when 
           ((* if List.length implVals <> List.length il_argtys then dprintf3 "Note: nm = %s, length implVals = %d, length il_argtys = %d\n" (name_of_mspec mspec) (length implVals) (length il_argtys); *)
            List.length implVals = List.length il_argtys)
            ->
            map2 (fun x y -> x,Some y) argInfosAndTypes implVals
        | _ -> 
            map (fun x -> x,None) argInfosAndTypes in 

    argInfosAndTypes 
    |> map (fun ((il_ty,TopArgData(attribs,nmOpt)),implValOpt) -> 
        let inFlag,outFlag,optionalFlag,paramMarshal = gen_param_attribs cenv attribs in 
           
        { paramName=(match nmOpt with 
                     | Some v -> Some(v.idText) 
                     | None -> match implValOpt with 
                               | Some v -> Some(name_of_val v) 
                               | None -> None);
          paramType= il_ty;  
          paramDefault=None; (* REVIEW: support "default" attributes *)   
          paramMarshal=paramMarshal; 
          paramIn=inFlag;    
          paramOut=outFlag;  
          paramOptional=optionalFlag; 
          paramCustomAttrs= mk_custom_attrs (gen_attrs cenv eenv attribs) })
    
and gen_returnv cenv eenv il_rty (TopArgData(attrs,_)) =
    if verbose then dprintf1 "length attrs = %d\n" (length attrs);
    { returnType=il_rty;
      returnMarshal=gen_marshal cenv attrs;
      returnCustomAttrs= mk_custom_attrs (gen_attrs cenv eenv attrs) }
       
and gen_property_for_mdef g propertyCompiledAsInstance propertyIsCompiledAsEvent tref mdef vspr il_argtys il_propty il_attrs =
    let basicName = get_property_name vspr in  (* chop "get_" *)
    let name = if propertyIsCompiledAsEvent then  basicName^"_IEvent" else basicName in 
    if verbose then dprintf1 "gen_property_for_mdef %s\n" name;
    
    { propName=name; 
      propRTSpecialName=false;
      propSpecialName=false;
      propSet=(if vspr.vspr_flags.memFlagsKind= MemberKindPropertySet then Some(mk_mref_to_mdef(tref,mdef)) else None);
      propGet=(if vspr.vspr_flags.memFlagsKind= MemberKindPropertyGet then Some(mk_mref_to_mdef(tref,mdef)) else None);
      propCallconv=(if propertyCompiledAsInstance then CC_instance else CC_static);
      propType=il_propty;          
      propInit=None;
      propArgs= il_argtys;
      propCustomAttrs=il_attrs; }  

and gen_bind_as_method cenv cgbuf eenv (v,mspec,access,paramInfos,retInfo) (arity_info,basevopt,tps,method_vs, body, rty) =
    let m = range_of_val v in
    let self_method_vs,non_self_method_vs,compileAsInstance =
        match member_info_of_val v with 
        | Some(vspr) when valCompiledAsInstance cenv.g v -> 
          begin match method_vs with 
          | [] -> error(Error("Internal error: empty argument list for instance method",range_of_val v))
          | h::t -> [h],t,true
          end
        |  _ -> [],method_vs,false in 

    let non_unit_non_self_method_vs,body = bind_unit_vars cenv.g (non_self_method_vs,body) in 
    let non_unit_method_vs = self_method_vs@non_unit_non_self_method_vs in
    let cmtps,farginfosl,_,_ = dest_top_type arity_info (type_of_val v) in 
    let fargtysl = map (map fst) farginfosl in 
    let eenv = bind_basevopt cenv eenv basevopt in 
    (* The type parameters of the method's type are different to the type parameters *)
    (* for the big lambda ("tlambda") of the implementation of the method. *)
    let eenv_under_meth_tlambda_typars = env_for_typars tps eenv in
    let eenv_under_meth_type_typars = env_for_typars cmtps eenv in
    (* Add the arguments to the environment.  We add an implicit 'this' argument to constructors *)
    let ctor = 
        match member_info_of_val v with 
        | Some(vspr) when not (isext_of_val v) && (vspr.vspr_flags.memFlagsKind = MemberKindConstructor) -> true
        | _ -> false in 
    let meth_eenv = 
      let meth_eenv = eenv_under_meth_tlambda_typars in 
      let numImplicitArgs = if ctor then 1 else 0 in 
      let meth_eenv = add_storage_for_locvals cenv.g (list_mapi (fun i v -> (v,Arg (numImplicitArgs+i))) non_unit_method_vs) meth_eenv in 
      meth_eenv in 
    let tailcall_info = [(mk_local_vref v,BranchCallMethod (TopValData.aritiesOfArgs arity_info,fargtysl, tps,length non_unit_method_vs))] in
    (* Discard the result on a 'void' return type. For a constructor just return 'void'  *)
    let seq = if is_unit_typ cenv.g rty then discardAndReturnVoid else if ctor then ReturnVoid else Return in 
    (* Now generate the code. *)        
    let attrs = (attribs_of_val v) in 
    let ilmbody,preservesig = 
      match fsthing_tryfind_attrib cenv.g cenv.g.attrib_DllImportAttribute attrs with
      | Some (Attrib(_,[ TExpr_const (TConst_string(dll),_,_) ],namedArgs))  -> 
          if tps <> [] then error(Error("The signature for this external function contains type parameters. Constrain the argument and return types to indicate the types of the corresponding C function",m)); 
          let preservesig = true in

          let find_int32,find_bool,find_string = decodeAttribNamedArgs namedArgs in 
          
          MethodBody_pinvoke 
            { pinvokeWhere=mk_simple_modref (Bytes.unicode_bytes_as_string dll);
              pinvokeName=find_string "EntryPoint" (compiled_name_of_val v);
              pinvokeCallconv=
              begin match find_int32 "CallingConvention" 0l with 
              | 1l -> PInvokeCallConvWinapi
              | 2l -> PInvokeCallConvCdecl
              | 3l -> PInvokeCallConvStdcall
              | 4l -> PInvokeCallConvThiscall
              | 5l -> PInvokeCallConvFastcall
              | _ -> PInvokeCallConvWinapi
              end;
              pinvokeEncoding=
              begin match find_int32 "CharSet" 0l with 
              | 1l -> PInvokeEncodingNone
              | 2l -> PInvokeEncodingAnsi
              | 3l -> PInvokeEncodingUnicode
              | 4l -> PInvokeEncodingAuto
              | _  -> PInvokeEncodingNone
              end;
              pinvokeNoMangle= find_bool "ExactSpelling" false;
              pinvokeLastErr= find_bool "SetLastError" false;
              pinvokeThrowOnUnmappableChar= if (find_bool "ThrowOnUnmappableChar" false) then PInvokeThrowOnUnmappableCharEnabled else PInvokeThrowOnUnmappableCharUseAssem;
              pinvokeBestFit=if (find_bool "BestFitMapping" false) then PInvokeBestFitEnabled else PInvokeBestFitUseAssem },
          preservesig
      | _ -> 
        (* This is the main code generation for most methods *)
          MethodBody_il(codegen_method_for_expr cenv cgbuf.mgbuf (tailcall_info, name_of_mspec mspec, meth_eenv, 0, 0, body, seq)),
          false in 
    if verbose then dprintf3 "gen_bind_as_method '%s': #non_unit_non_self_method_vs=%d, #formals=%d\n" (showL(vspecAtBindL v)) (length non_unit_non_self_method_vs) (length (formal_args_of_mspec mspec));
    (* Do not push the attributes to the method for events and properties *)
    (* However OverloadIDAttribute does get pushed to the methods as this is *)
    (* required by the F# quotation/reflection implementation. *)
    let attrsThatMustBeOnMethod, attrsThatGoOnPropertyIfItExists  = 
        List.partition (is_fs_attrib cenv.g cenv.g.attrib_OverloadIDAttribute) attrs  in

    let il_attrsThatGoOnMethod = (gen_attrs cenv eenv attrsThatMustBeOnMethod) in 
    let il_attrsThatGoOnPrimaryItem =  (gen_attrs cenv eenv attrsThatGoOnPropertyIfItExists) in 
    let il_tps = gen_static_formals m cenv eenv_under_meth_tlambda_typars tps in
    let parameters = gen_params cenv eenv mspec paramInfos (Some(non_unit_non_self_method_vs)) in 
    let returnv = gen_returnv cenv eenv (formal_ret_of_mspec mspec) retInfo in 
    let mname = name_of_mspec mspec in 
    let tref = tref_of_mref (formal_mref_of_mspec mspec) in

    if verbose then dprintf1 "gen_bind_as_method '%s': part b\n" (showL(vspecAtBindL v));

    (* don't generate unimplemented abstracts *)
    match member_info_of_val v with 
    | Some(vspr) when vspr.vspr_flags.memFlagsAbstract && not vspr.vspr_implemented -> 
       if verbose then dprintf1 "gen_bind_as_method '%s': skipping unimplemented abstract method\n" (showL(vspecAtBindL v));
       ()    
    | _ -> 

    let mdef =
      match member_info_of_val v with 
      | Some(vspr) when not (isext_of_val v) -> 
          if verbose then dprintf1 "gen_bind_as_method '%s': is a method\n" (showL(vspecAtBindL v));
         let il_ctps,il_mtps = chop_at (length(inst_of_typ (enclosing_typ_of_mspec mspec))) il_tps in 
         let mdef = 
           if not compileAsInstance then 
             if vspr.vspr_flags.memFlagsKind = MemberKindConstructor then 
               mk_ctor (access,parameters,ilmbody) 
             else if vspr.vspr_flags.memFlagsKind = MemberKindClassConstructor then 
               mk_cctor ilmbody 
             else 
               mk_static_mdef (il_mtps,vspr.vspr_il_name,access,parameters,returnv,ilmbody) 
           else if (vspr.vspr_flags.memFlagsVirtual || (vspr.vspr_flags.memFlagsAbstract && vspr.vspr_implemented) || vspr.vspr_flags.memFlagsOverride) then 
             (* Virtual methods are used to implement interfaces and hence must currently be public *)
             (* REVIEW: use method impls to implement the interfaces *)
             begin 
               if access <> MemAccess_public && not (compgen_of_val v) then warning(FullAbstraction("This method will currently be made public in the underlying IL because it may implement an interface or override a method",range_of_val v));
               (* Generate the ILX erasure information. *)
               (* Auto-generated CompareTo is treated differently, and *)
               (* doesn't need ILX erasure information *)
               if verbose then dprintf2 "gen_bind_as_method '%s', tps = %s: generating an abstract method implementation\n" (showL(vspecAtBindL v)) (showL(typarsL tps));
               let ovinfo,flagFixup = 
                 match vspr.vspr_implements_slotsig with 
                 | Some (TSlotSig(_,oty,_,_,_,_) as slotsig) -> 
                     let tcref = apparent_parent_of_vspr_val v in
                     let tcaug = tcaug_of_tcref tcref in
                     let isExtraCompareTo = 
                       vspr.vspr_il_name = "CompareTo" &&  
                       type_equiv cenv.g oty cenv.g.mk_IComparable_ty &&  
                       isSome tcaug.tcaug_compare in
                     if isExtraCompareTo then None ,fixupVirtualSlotFlags
                     else
                       let isHash = 
                           isSome tcaug.tcaug_structural_hash &&
                           type_equiv cenv.g oty cenv.g.mk_IStructuralHash_ty in
                       let memberParentTypars = 
                           match partition_val_typars v with
                           | Some(_,memberParentTypars,_,_,_) -> memberParentTypars
                           | None -> errorR(InternalError("partition_val_typars",range_of_val v)); [] in
                       let eenv_under_typars = env_for_typars memberParentTypars eenv in
                       let use_mimpl = is_interface_typ oty && not isHash in 
                       let really_use_mimpl,ovby_name, ovinfo,_ = gen_mimpl cenv eenv_under_typars (use_mimpl,slotsig) (range_of_val v) in
                       if verbose then dprintf2 "gen_bind_as_method '%s', tps = %s: generated a method impl\n" (showL(vspecAtBindL v)) (showL(typarsL tps));
                       ovinfo,
                       (if really_use_mimpl then fixupMethodImplFlags >> renameMethodDef ovby_name
                       else fixupVirtualSlotFlags >> renameMethodDef ovby_name)
                 | None -> None,fixupVirtualSlotFlags in 
               let mdef = mk_generic_virtual_mdef (vspr.vspr_il_name,MemAccess_public,il_mtps,ovinfo,parameters,returnv,ilmbody) in
               let mdef = flagFixup mdef in 
               mdef
             end
           else 
             mk_generic_instance_mdef (vspr.vspr_il_name,access,il_mtps,parameters,returnv,ilmbody)  in 
         let abstr = 
           vspr.vspr_flags.memFlagsAbstract && 
           let tcref =  apparent_parent_of_vspr_val v in
           not (is_fsobjmodel_delegate_tycon (deref_tycon tcref)) in 

         if verbose then dprintf1 "gen_bind_as_method : abstr = %b\n" abstr;

         let mdef = 
           {mdef with mdKind=match mdef.mdKind with 
                             | MethodKind_virtual vinfo -> 
                                 MethodKind_virtual {vinfo with virtFinal=vspr.vspr_flags.memFlagsFinal;
                                                                virtAbstract=abstr; } 
                             | k -> k } in
         begin match vspr.vspr_flags.memFlagsKind with 
         | MemberKindClassConstructor 
         | MemberKindConstructor -> 
             if nonNil il_mtps then 
               error(Error("A constructor may not be more generic than the enclosing type - constrain the polymorphishm in the expression",range_of_val v));
             { mdef with mdCustomAttrs= mk_custom_attrs (il_attrsThatGoOnPrimaryItem @ il_attrsThatGoOnMethod) }; 
             
         | MemberKindMember -> 
             { mdef with mdCustomAttrs= mk_custom_attrs (il_attrsThatGoOnPrimaryItem @ il_attrsThatGoOnMethod) }; 

         | MemberKindPropertyGetSet -> 
             error(Error("Unexpected GetSet annotation on a property",range_of_val v));
             
         | (MemberKindPropertySet | MemberKindPropertyGet)  as k->
             if il_mtps <> [] then 
               error(Error("A property may not be more generic than the enclosing type - constrain the polymorphism in the expression",range_of_val v));

             (* Properties compatible with type IPrimitiveDelegateEvent are special: we generate metadata and add/remove methods *)
             (* to make them into a .NET event, and mangle the name of a property.  *)
             (* We don't handle static, indexer or abstract properties correctly. *)
             (* Note the name mangling doesn't affect the name of the get/set methods for the property *)
             (* and so doesn't affect how we compile F# accesses to the property. *)
             let conforms_to_IPrimitiveDelegateEvent ty = 
                 is_fslib_IPrimitiveDelegateEvent_ty cenv.g ty && is_delegate_typ (dest_fslib_IPrimitiveDelegateEvent_ty cenv.g ty)  in
                 
                 
             let propertyIsCompiledAsEvent = 
               (not vspr.vspr_flags.memFlagsAbstract) && 
               compileAsInstance && 
               (k = MemberKindPropertyGet) && 
               (isNil parameters) && 
               (Infos.exists_in_entire_hierarchy_of_typ conforms_to_IPrimitiveDelegateEvent cenv.g cenv.amap m rty) in
             (* Emit the property.  Don't emit a property for method impls *)
             if mdef.mdAccess <> MemAccess_private then begin 
               let vtyp = vtyp_of_property_val cenv.g v in
               let il_propty = gen_type m cenv.g eenv_under_meth_type_typars.tyenv vtyp in 
               let il_argtys = v |> arginfos_of_propery_val cenv.g |> List.map fst |> gen_types m cenv.g eenv_under_meth_type_typars.tyenv in
               let pdef = gen_property_for_mdef cenv.g compileAsInstance propertyIsCompiledAsEvent tref mdef vspr il_argtys il_propty (mk_custom_attrs il_attrsThatGoOnPrimaryItem) in 
               insert_or_merge_pdef cgbuf.mgbuf tref pdef
             end;
             if propertyIsCompiledAsEvent then begin
                 (* 
                 // We don't treat abstract events correctly. These are some notes about
                 // how we might compile them:
                 //
                 // Currently: 
                 //   NV x.EP = e ~~> 
                 //      NV x.EP_IE 
                 //      NV x.get_EP() = e
                 //      NV x.add_EP(d) = x.get_EP().AddHandler(d)
                 //      NV x.remove_EP(d) = x.get_EP().RemoveHandler(d)
                 //
                 // IL currently:
                 //   ABS x.EP {add;remove} ~~> 
                 //      ABS x.add_EP(d)
                 //      ABS x.remove_EP(d)
                 //   OVD x.EP {add;remove} ~~> 
                 //      OVD x.add_EP(d) = e1
                 //      OVD x.remove_EP(d) = e2
                 // (derived: 
                 //   DEF x.EP {add=e1;remove=e2}~~> 
                 //      DEF x.add_EP(d) = e1
                 //      DEF x.remove_EP(d) = e2
                 // )
                 // 
                 // 
                 // propose:
                 //   ABS x.EP : IDelegateEvent<'del> ~~> 
                 //      PROP x.EP : IDelegateEvent<'del>
                 //      NV x.get_EP() = { new IDelegateEvent<'del> with y.AddHandler(h) = y.add_EP(h);
                 //                                                     y.RemoveHandler(h) = y.remove_EP(h) }
                 //      ABS x.add_EP(d)
                 //      ABS x.remove_EP(d)
                 //
                 //   OVD x.EP : IDelegateEvent<'del> = e ~~> 
                 //      OVD x.add_EP(d) = e.AddHandler(d)
                 //      OVD x.remove_EP(d) = e.RemoveHandler(d)
                 //              
                 // also gives:
                 //
                 //   DEF x.EP : IDelegateEvent<'del> = e ~~> 
                 //      PROP x.EP : IDelegateEvent<'del>
                 //      DEF x.add_EP(d) = e.AddHandler(d)
                 //      DEF x.remove_EP(d) = e.RemoveHandler(d)
                 //      NV x.get_EP() = { new IDelegateEvent<'del> with y.AddHandler(h) = y.add_EP(h);
                 //                                                     y.RemoveHandler(h) = y.remove_EP(h) }
                 // 
                 // Sanity check combinations: 
                 //   F# DEF + IL client -> 
                 //   F# ABS + F# OVD + IL client -> 
                 //   F# DEF + F# OVD + IL client -> 
                 //   IL ABS + F# OVD + IL client -> 
                 //   F# OVD of F# OVD behaves as expected, though the property still exists
                 //   IL override of F# OVD behaves as expected, though the property still exists
                 *)


               
               let evname = get_property_name vspr in 
               let delEvent_ty =
                   match Infos.first_in_entire_hierarchy_of_typ conforms_to_IPrimitiveDelegateEvent cenv.g cenv.amap m rty with
                   | None -> error(Error("unexpected: first_in_entire_hierarchy_of_typ failed",m))
                   | Some ty -> ty in
               let delty = dest_fslib_IPrimitiveDelegateEvent_ty cenv.g delEvent_ty in 
               let il_delty = gen_type m cenv.g meth_eenv.tyenv delty in
               let il_fdelty = mk_tyvar_ty u16_zero in 
               let il_rty = gen_type m cenv.g meth_eenv.tyenv rty in
               let il_delEvent_ty = gen_type m cenv.g meth_eenv.tyenv delEvent_ty in
               let il_thisty = enclosing_typ_of_mspec mspec in 
               let code_attr = gen_opt_range cenv m in
               let mk_add_remove_mdef nm1 nm2 =
                 mk_instance_mdef 
                   (nm1^evname,
                    access,
                    [mk_named_param("handler",il_delty)],
                    mk_return Type_void,
                    MethodBody_il
                      (mk_ilmbody
                        (true,[],2,
                         nonbranching_instrs_to_code 
                           [ ldarg_0; 
                             I_call(Normalcall,mk_nongeneric_instance_mspec_in_typ(il_thisty,"get_"^evname,[],il_rty),None);
                             I_ldarg (int_to_u16 1); 
                             I_callvirt(Normalcall,mk_nongeneric_instance_mspec_in_typ(il_delEvent_ty,nm2,[il_fdelty],Type_void),None) ],
                         code_attr))) in 
               let add_mdef = mk_add_remove_mdef "add_" "AddHandler" in 
               let remove_mdef = mk_add_remove_mdef "remove_" "RemoveHandler" in 
               let add_mref = mk_mref_to_mdef (tref_of_typ il_thisty,add_mdef) in 
               let remove_mref = mk_mref_to_mdef (tref_of_typ il_thisty,remove_mdef) in 
               let edef = 
                 { eventType = Some(il_delty); 
                   eventName= evname; 
                   eventRTSpecialName=false;
                   eventSpecialName=false;
                   eventAddOn = add_mref; 
                   eventRemoveOn = remove_mref;
                   eventFire= None;
                   eventOther= [];
                   eventCustomAttrs = mk_custom_attrs il_attrsThatGoOnPrimaryItem; } in
               insert_edef cgbuf.mgbuf tref edef;
               insert_mdef cgbuf.mgbuf tref add_mdef;
               insert_mdef cgbuf.mgbuf tref remove_mdef;

             end;
             
           (* Do not push the attributes to the method for events and properties *)
           (* However OverloadIDAttribute does get pushed to the methods as this is *)
           (* required by the F# quotation/reflection implementation. *)
           { mdef with mdCustomAttrs= mk_custom_attrs il_attrsThatGoOnMethod }; 
           
        end     
      | _ -> 
          if verbose then dprintf1 "gen_bind_as_method '%s': not a method\n" (showL(vspecAtBindL v));
          let mdef = mk_static_mdef (il_tps, mname, access,parameters,returnv,ilmbody) in
          { mdef with mdCustomAttrs= mk_custom_attrs (il_attrsThatGoOnPrimaryItem @ il_attrsThatGoOnMethod) }  in
          
    
    (* Does the function have an expllicit [<EntryPoint>] attribute? *)
    let isExplicitEntryPoint = fsthing_has_attrib cenv.g cenv.g.attrib_EntryPointAttribute attrs in
    
    let mdef = {mdef with 
                    mdPreserveSig=preservesig;
                    mdEntrypoint = isExplicitEntryPoint;
               } in

    let mdef = if has_prefix mdef.mdName "op_" then {mdef with mdSpecialName=true} else mdef in
    method_def_counter();
    insert_mdef cgbuf.mgbuf tref mdef

and gen_binds cenv cgbuf eenv binds = List.iter (gen_bind cenv cgbuf eenv) binds

(*-------------------------------------------------------------------------
!* Generate locals and other storage of values
 *------------------------------------------------------------------------- *)

and gen_val_set cenv cgbuf eenv (v,e,m) sequel =
    if verbose then dprint_endline ("gen_val_set");   
    gen_expr cenv cgbuf eenv e Continue;
    gen_set_vref cenv cgbuf eenv m v;
    gen_unit_then_sequel cenv eenv.cloc cgbuf sequel
      
and gen_vref_get_with_fetch_sequel cenv cgbuf eenv m v fetch_sequel =
    let ty = type_of_vref v in 
    gen_get_storage cenv cgbuf eenv m (ty, gen_type m cenv.g eenv.tyenv ty) (storage_for_vref m v eenv)  fetch_sequel

and gen_vref_get cenv cgbuf eenv (v,m) sequel =
    if verbose then dprint_endline ("gen_vref_get: " ^ (name_of_val (deref_val v)));
    gen_vref_get_with_fetch_sequel cenv cgbuf eenv m v None;
    gen_sequel cenv eenv.cloc cgbuf sequel
      
and gen_bind_rhs cenv cgbuf eenv vspec e =   
    if verbose then dprint_endline ("gen_bind_rhs, v = "^name_of_val vspec);   
    match e with 
    | TExpr_tlambda _ | TExpr_lambda _ -> 
        let is_ltyfunc = is_named_local_tyfunc_val vspec e in
        let selfv = if is_ltyfunc then None else Some (mk_local_vref vspec) in
        gen_lambda cenv cgbuf eenv is_ltyfunc selfv e Continue 
    | _ -> 
        gen_expr cenv cgbuf eenv e Continue;

and gen_set_locval_to_expr cenv cgbuf eenv eenv2 vspec e =   
    gen_bind_rhs cenv cgbuf eenv2 vspec e;
    if verbose then dprint_endline ("gen_set_locval_to_expr, v = "^name_of_val vspec);
    gen_set_locval cenv cgbuf eenv (range_of_val vspec) vspec
        
and gen_init_local cgbuf typ idx = CG.emit_instrs cgbuf []  [I_ldloca (int_to_u16 idx);  (I_initobj typ) ]
and gen_set_local cgbuf idx = CG.emit_instr cgbuf [Pop] (I_stloc (int_to_u16 idx))
and gen_get_local cgbuf typ idx = CG.emit_instr cgbuf [Push typ] (I_ldloc (int_to_u16 idx))
and gen_set_top_field cgbuf fspec = CG.emit_instr cgbuf [Pop] (mk_normal_stsfld fspec)
and gen_get_top_fielda cgbuf typ fspec = CG.emit_instr cgbuf [Push typ]  (I_ldsflda fspec)
and gen_get_top_field cgbuf typ fspec = CG.emit_instr cgbuf [Push typ]  (mk_normal_ldsfld fspec)

and gen_set_storage cenv cloc m cgbuf storage = 
    if verbose then dprint_endline ("gen_set_storage");        
    match storage with  
    | Local (idx,_)  ->   gen_set_local cgbuf idx
    | StaticField (_,hasLiteralAttr,tspec,_,_,_,_,set_mref) ->  
        if hasLiteralAttr then errorR(Error("Literal fields may not be set",m));
        CG.emit_instr cgbuf [Pop]  (I_call(Normalcall,mk_mref_mspec_in_typ(set_mref,mk_typ AsObject tspec,[]),None))
    | Method (_,_,mspec,m,_,_) -> 
        error(Error("gen_set_storage: "^name_of_mspec mspec^" was represented as a static method but was not an appropriate lambda expression",m))
    | Null ->  CG.emit_instr cgbuf [Pop] (i_pop)
    | Arg _ | Env _  -> error(Error("mutable variables may not escape their method",m))
    | Unrealized -> error(Error("compiler error: unexpected unrealized value",m))

and gen_get_storage_sequel cenv cgbuf eenv m typ localCloInfo store_sequel = 
    if verbose then dprint_endline ("gen_get_storage_sequel");          
    match localCloInfo,store_sequel with 
    | Some {contents =NamedLocalClosureInfoGenerator cloinfo},_ -> error(InternalError("unexpected generator",m))
    | Some {contents =NamedLocalClosureInfoGenerated cloinfo},Some (tyargs,args,m,sequel) when nonNil tyargs ->
        let clo_il_ty = gen_named_local_tyfunc_type cloinfo in 
        CG.emit_instr cgbuf [Pop;Push clo_il_ty]  (I_castclass clo_il_ty);          
        let actual_rty = apply_types typ (tyargs,[]) in 
        gen_named_local_tyfunc_cloinfo_call cenv cgbuf eenv actual_rty cloinfo tyargs m;
        gen_get_storage_sequel cenv cgbuf eenv m actual_rty None (Some ([],args,m,sequel))
    | _, None ->
            (if verbose then dprint_endline ("gen_get_storage_sequel: None");
             ())
    | _,Some ([],[],m,sequel) ->
        gen_sequel cenv eenv.cloc cgbuf sequel 
    | _,Some (tyargs,args,m,sequel) ->
        gen_indirect_call cenv cgbuf eenv (typ,tyargs,args,m) sequel 

and gen_get_storage cenv cgbuf eenv m (typ,il_typ) storage store_sequel =
    if verbose then dprint_endline ("gen_get_storage:");   
    match storage with  
    | Local (idx,localCloInfo) ->
        if verbose then dprint_endline ("gen_get_storage: Local...");            
        gen_get_local cgbuf il_typ idx;
        gen_get_storage_sequel cenv cgbuf eenv m typ localCloInfo store_sequel
    | StaticField (fspec,hasLiteralAttr,binding_tspec,fieldName,_,_,get_mref,_) ->  
        if verbose then dprint_endline ("gen_get_storage: StaticField...");         
        if hasLiteralAttr then 
            gen_get_top_field cgbuf il_typ fspec
        else
            CG.emit_instr cgbuf [Push il_typ]  (I_call(Normalcall,mk_mref_mspec_in_typ(get_mref,mk_typ AsObject binding_tspec,[]),None));
        gen_get_storage_sequel cenv cgbuf eenv m typ None store_sequel
    | Method (arity_info,vref,mspec,_,_,_) -> 
        (* Get a toplevel value as a first-class value. *)
        (* We generate a lambda expression and that simply calls *)
        (* the toplevel method. However we optimize the case where we are *)
        (* immediately applying the value anyway (to insufficient arguments). *)

        (* First build a lambda expression for the saturated use of the toplevel value... *)
        (* REVIEW: we should NOT be doing this in the backend... *)
        if verbose then dprint_endline ("gen_get_storage: Method...");      
        let expr,exprty = adjust_val_for_expected_arity cenv.g m vref NormalValUse arity_info in 

        (* Then reduce out any arguments (i.e. apply the sequel immediately if we can...) *)
        begin match store_sequel with 
        | None -> gen_lambda cenv cgbuf eenv false None expr Continue
        | Some (tyargs',args,m,sequel) -> 
            let specialized_expr = 
              if verbose && tyargs' <> [] then dprint_endline ("creating type-specialized lambda at use of method "^name_of_mspec mspec);
              if verbose && args <> [] then dprintf4 "creating term-specialized lambda at use of method %s\n--> expr = %s\n--> exprty = %s\n--> #args = %d\n" (name_of_mspec mspec) (showL (exprL expr)) ((DebugPrint.showType exprty)) (length args);
              if isNil args && isNil tyargs' then failwith ("non-lambda at use of method "^name_of_mspec mspec);
              beta_mk_appl cenv.g nng (expr,exprty,[tyargs'],args,m) in
            gen_expr cenv cgbuf eenv specialized_expr sequel
        end
    | Null  ->   
        CG.emit_instr cgbuf [Push il_typ] (i_ldnull); 
        gen_get_storage_sequel cenv cgbuf eenv m typ None store_sequel
    | Unrealized  ->
        dprintf3 "getting an unrealized value of type '%s' at or near %a\n" (showL(typeL typ)) output_range m;
        failwith "getting an unrealized value"
    | Arg i -> 
        if verbose then dprint_endline ("gen_get_storage: Arg...");      
        CG.emit_instr cgbuf [Push il_typ] (I_ldarg (int_to_u16 i)); 
        gen_get_storage_sequel cenv cgbuf eenv m typ None store_sequel
    | Env (i,localCloInfo) -> 
        if verbose then dprint_endline ("gen_get_storage: Env...");      
        CG.emit_instr cgbuf [Push il_typ] (mk_IlxInstr (EI_ldenv i)); 
        gen_get_storage_sequel cenv cgbuf eenv m typ localCloInfo store_sequel

and gen_get_locvals cenv cgbuf eenvouter m fvs = 
    iter (fun v -> gen_get_locval cenv cgbuf eenvouter m v None) fvs;
and gen_get_locval cenv cgbuf eenv m vspec fetch_sequel =
    gen_get_storage cenv cgbuf eenv m (type_of_val vspec, gen_type_of_val cenv eenv vspec) (storage_for_val m vspec eenv) fetch_sequel
and gen_get_vref cenv cgbuf eenv m vref fetch_sequel =
    gen_get_storage cenv cgbuf eenv m (type_of_vref vref, gen_type_of_val cenv eenv (deref_val vref)) (storage_for_vref m vref eenv) fetch_sequel
and gen_set_locval cenv cgbuf eenv m vspec =
    gen_set_storage cenv eenv.cloc (range_of_val vspec) cgbuf (storage_for_val m vspec eenv)
and gen_set_vref cenv cgbuf eenv m vref =
    gen_set_storage cenv eenv.cloc (range_of_vref vref) cgbuf (storage_for_vref m vref eenv)

(*--------------------------------------------------------------------------
!* Allocate locals for values
 *-------------------------------------------------------------------------- *)
 
and alloc_local cenv cgbuf eenv compgen (v,ty) (scopeMarks : mark * mark) = 
     let try_to_reuse i (v',ty') = 

         (* Reuse slots whenever they have become dead, also reuse OBJECT local slots *)
         (* in unverifiable code.  *)
         (* It also doesn't buy much in terms of speed of code produced, but *)
         (* surprisingly buys a big win in compiler speed because of the reduced *)
         (* number of locals produced... *)

         (* Note all ILX types (Type_other) are boxed types *)
         (* REVIEW: do this as a representation step. *)

         if not (Imap.mem i eenv.liveLocals) && 
           ((ty = ty') (*
           || 
            (!Msilxlib.unverifiable_ref  && 
             match ty,ty' with 
             | (Type_boxed _ | Type_other _ | Type_array _), 
               (Type_boxed _ | Type_other _ | Type_array _) -> true
             | _ -> false) *) ) then begin
           if verbose then dprint_endline "reusing local slot"; 
           Some (((if compgen then v' else (v,scopeMarks)::v'), ty'), i)
         end 
         else None in 
     let j =
         match (if not cenv.optimize then None  
                else ResizeArray.choosei try_to_reuse cgbuf.locals) with 
         | Some (entry,j) -> ResizeArray.replace cgbuf.locals j entry; j
         | None ->
             (* Could not reuse - allocate a new local *)
             let j = ResizeArray.length cgbuf.locals in 
             ResizeArray.add cgbuf.locals ((if compgen then [] else [(v,scopeMarks)]),ty);
             j in 
     let j = j + cgbuf.preallocated_locals in 
     if (Imap.mem j eenv.liveLocals) then failwith "local is still live";
     j, { eenv with liveLocals =  Imap.add j () eenv.liveLocals  }

and alloc_local_val cenv cgbuf v eenv repr scopeMarks = 
    if verbose then dprint_endline ("alloc_local_val: "^ name_of_val v ^ NicePrint.pretty_string_of_typ (empty_denv cenv.g) (type_of_val v));     
    let repr,eenv = 
      let ty = (type_of_val v) in 
      if is_unit_typ cenv.g ty then  Null,eenv
      else if inlineFlag_of_val v = PseudoValue then Unrealized,eenv
      else if isSome repr && is_named_local_tyfunc_val v (the repr) then 
        (* known, named, non-escaping type functions *)
        let cloinfoGenerate eenv = 
          let eenvinner = 
              {eenv with 
                   letBoundVars=(mk_local_vref v)::eenv.letBoundVars} in
          let cloinfo,_,_ = get_cloinfo cenv (range_of_val v) true None eenvinner (the repr) in 
          cloinfo in         
        
        let idx,eenv = alloc_local cenv cgbuf eenv (compgen_of_val v) (name_of_val v, cenv.g.ilg.typ_Object) scopeMarks in 
        Local (idx,Some(ref (NamedLocalClosureInfoGenerator cloinfoGenerate))),eenv
      else
        (* normal local *)
        let idx,eenv = alloc_local cenv cgbuf eenv (compgen_of_val v) (name_of_val v, gen_type_of_val cenv eenv v) scopeMarks in 
        Local (idx,None),eenv in 
    Some repr,add_storage_for_val cenv.g (v,repr) eenv

and alloc_vals_for_binds cenv cgbuf scopeMarks eenv binds = 
   (* phase 1 - decicde representations - most are very simple. *)
   let reps, eenv = map_acc_list (alloc_val_for_bind cenv cgbuf scopeMarks) eenv binds  in 
   (* phase 2 - run the cloinfo generators for NamedLocalClosure values against the environment recording the *)
   (* representation choices. *)
   reps |> List.iter (fun reprOpt -> 
     match reprOpt with 
     | Some repr -> 
       begin match repr with 
       | (Local(_,Some g) | Env(_,Some g)) -> 
           begin match !g with 
           | NamedLocalClosureInfoGenerator f -> g := NamedLocalClosureInfoGenerated (f eenv) 
           | NamedLocalClosureInfoGenerated _ -> ()
           end
       | _ -> ()
        end
     | _ -> ());
   eenv
   
and alloc_val_for_bind cenv cgbuf (scopeMarks:mark*mark) eenv (TBind(v,repr)) =
    match arity_of_val v with 
    | None -> alloc_local_val cenv cgbuf v eenv (Some repr) scopeMarks
    | Some _ -> None,add_storage_for_local_topval cenv.g eenv.cloc (modbind_of_val v) v eenv



(*--------------------------------------------------------------------------
!* Generate stack save/restore and assertions - pulled into letrec by alloc*
 *-------------------------------------------------------------------------- *)

and assert_stack_empty cgbuf =
    if (CG.curr_stack cgbuf) <> [] then failwith "huh - stack flush didn't work, or extraneous expressions left on stack before stack restore!";
    ()

and gen_stack_save cenv cgbuf eenv m scopeMarks =
    if verbose then dprint_endline ("gen_save_stack");
    (* Save the stack
     * - [gross] because IL flushes the stack at the exn. handler
     * - and     because IL requires empty stack following a forward br (jump).
     *)
    let stack_saved = (CG.curr_stack cgbuf) in 
    let where_stack_saved,eenvinner = map_acc_list (fun eenv ty -> alloc_local cenv cgbuf eenv true (nng.nngApply "$spill" m, ty) scopeMarks) eenv stack_saved in 
    iter (gen_set_local cgbuf) where_stack_saved;
    assert_stack_empty cgbuf;
    (stack_saved,where_stack_saved),eenvinner (* need to return, it marks locals "live" *)

and gen_stack_restore cenv cgbuf (stack_saved,where_stack_saved) =
    (* Restore the stack and load the result *)
    assert_stack_empty cgbuf;  
    iter2 (gen_get_local cgbuf) (rev stack_saved) (rev where_stack_saved)

(*-------------------------------------------------------------------------
 * gen_attr: custom attribute generation
 *------------------------------------------------------------------------- *)

and gen_attr_arg cenv eenv arg il_argty = 
    match arg,il_argty with 

    (* Detect standard constants *)
    | TExpr_const(c,m,_),_ -> 
        let tynm = (tname_of_tspec (tspec_of_typ il_argty) ) in 
        let isobj = (tynm = "System.Object") in 

        begin match c with 
        | TConst_bool b -> CustomElem_bool b
        | TConst_int32 i when isobj || tynm = "System.Int32" ->  CustomElem_int32 ( i)
        | TConst_int32 i when tynm = "System.SByte" ->  CustomElem_int8 (i32_to_i8 i)
        | TConst_int32 i when tynm = "System.Int16"  -> CustomElem_int16 (i32_to_i16 i)
        | TConst_int32 i when tynm = "System.Byte"  -> CustomElem_uint8 (i32_to_u8 i)
        | TConst_int32 i when tynm = "System.UInt16" ->CustomElem_uint16 (i32_to_u16 i)
        | TConst_int32 i when tynm = "System.UInt32" ->CustomElem_uint32 (i32_to_u32 i)
        | TConst_int32 i when tynm = "System.UInt64" ->CustomElem_uint64 (i64_to_u64 (i32_to_i64 i)) 
        | TConst_int8  i  ->  CustomElem_int8 i
        | TConst_int16  i  ->  CustomElem_int16 i
        | TConst_int32 i -> CustomElem_int32(i)
        | TConst_int64 i   ->  CustomElem_int64 i  
        | TConst_uint8  i  ->  CustomElem_uint8 i
        | TConst_uint16  i  ->  CustomElem_uint16 i
        | TConst_uint32  i  ->  CustomElem_uint32 i
        | TConst_uint64  i  ->  CustomElem_uint64 i
        | TConst_float i  ->  CustomElem_float64 i
        | TConst_float32 i ->  CustomElem_float32 i
        | TConst_char i ->   CustomElem_char i
        | TConst_default when  tynm = "System.String"  -> CustomElem_string None
        | TConst_string i  when isobj || tynm = "System.String" ->  CustomElem_string (Some(Bytes.unicode_bytes_as_string i))

        | _ -> error (Error ( "The type '"^tynm^"' may not be used as a custom attribute value",m))
        end

    (* Detect '[| ... |]' nodes *)
    | TExpr_op(TOp_array,[elemTy],args,m),Type_array _ ->
        let ilElemTy = gen_type m cenv.g eenv.tyenv elemTy in 
        CustomElem_array (List.map (fun arg -> gen_attr_arg cenv eenv arg ilElemTy) args)

    (* Detect 'typeof<ty>' calls *)
    | TExpr_app(TExpr_val(vref,_,_),_,[ty],[],m),_ when cenv.g.vref_eq vref cenv.g.typeof_vref  ->
        CustomElem_type (tref_of_typ (gen_type m cenv.g eenv.tyenv ty))    

    (* Detect 'typedefof<ty>' calls *)
    | TExpr_app(TExpr_val(vref,_,_),_,[ty],[],m),_ when cenv.g.vref_eq vref cenv.g.typedefof_vref  ->
        CustomElem_type (tref_of_typ (gen_type m cenv.g eenv.tyenv ty))    

    (* Ignore upcasts *)
    | TExpr_op(TOp_coerce,_,[arg2],_),_ ->
        gen_attr_arg cenv eenv arg2 il_argty

    (* Detect explicit enum values *)
    | TExpr_app(TExpr_val(vref,_,_),_,_,[arg1],_),_ when cenv.g.vref_eq vref cenv.g.enum_vref  ->
        gen_attr_arg cenv eenv arg1 il_argty
    

    (* Detect bitwise or of attribute flags: one case of constant folding (a more general treatment is needed *)
    
    | TExpr_app(TExpr_val(vref,_,_),_,_,[arg1;arg2],_),_ when cenv.g.vref_eq vref cenv.g.bitwise_or_vref  ->
        let v1 = gen_attr_arg cenv eenv arg1 il_argty  in
        let v2 = gen_attr_arg cenv eenv arg2 il_argty  in
        begin match v1,v2 with 
        | CustomElem_int8 i1, CustomElem_int8 i2 -> CustomElem_int8 ((i8_to_i32 i1 ||| i8_to_i32 i2) |> i32_to_i8)
        | CustomElem_int16 i1, CustomElem_int16 i2-> CustomElem_int16 ((i16_to_i32 i1 ||| i16_to_i32 i2) |> i32_to_i16)
        | CustomElem_int32 i1, CustomElem_int32 i2-> CustomElem_int32 ((i1 ||| i2))
        | CustomElem_int64 i1, CustomElem_int64 i2-> CustomElem_int64 ((i1 |||| i2))
        | CustomElem_uint8 i1, CustomElem_uint8 i2-> CustomElem_uint8 ((u8_to_i32 i1 ||| u8_to_i32 i2) |> i32_to_u8)
        | CustomElem_uint16 i1, CustomElem_uint16 i2-> CustomElem_uint16 ((u16_to_i32 i1 ||| u16_to_i32 i2) |> i32_to_u16)
        | CustomElem_uint32 i1, CustomElem_uint32 i2-> CustomElem_uint32 ((u32_to_i32 i1 ||| u32_to_i32 i2) |> i32_to_u32)
        | CustomElem_uint64 i1, CustomElem_uint64 i2-> CustomElem_uint64 ((u64_to_i64 i1 |||| u64_to_i64 i2) |> i64_to_u64)
        |  _ -> error (Error ("invalid custom attribute value (not a valid constant): "^showL (exprL arg),range_of_expr arg))
        end

    | _ -> error (Error ("invalid custom attribute value (not a constant): "^showL (exprL arg),range_of_expr arg))

and gen_attr cenv eenv (Attrib(k,args,props)) = 
  let props = props |> List.map (fun (s,ty,fld,expr) ->
    let m = (range_of_expr expr) in 
    let il_ty = gen_type m cenv.g eenv.tyenv ty in 
    let cval = gen_attr_arg cenv eenv expr il_ty in 
    (s,il_ty,fld,cval)) in 
  let mref = 
    match k with 
    | ILAttrib(mref) -> mk_mspec(mref,AsObject,[],[]) 
    | FSAttrib(vref) -> 
       assert(isSome(member_info_of_vref vref)); 
       let mspec,_,_,_,_ = mspec_for_vspr_vref cenv.g (the(member_info_of_vref vref)) vref in
       mspec in 
  mk_custom_attribute_mref cenv.g.ilg (mref,List.map2 (gen_attr_arg cenv eenv) args (formal_args_of_mspec mref), props)
    
and gen_attrs cenv eenv attrs = map (gen_attr cenv eenv) attrs

(*--------------------------------------------------------------------------
!* Generate the set of modules for an assembly, and the declarations in each module
 *-------------------------------------------------------------------------- *)

and gen_tdef_for_cloc cenv eenv mgbuf cloc hidden attribs = 
    let tref = mk_tref_for_cloc cloc in 
    let tdef = 
      mk_simple_tdef cenv.g.ilg
        (tname_of_tref tref, 
         computeTypeAccess tref hidden,
         mk_mdefs [], 
         mk_fdefs [],
         mk_properties [],
         mk_events [],
         mk_custom_attrs 
           (gen_attrs cenv eenv attribs @
            (if List.mem (tname_of_tref tref) [tnameStartupCode cloc; tnamePrivateImplementationDetails cloc]  
            then [ (* mk_CompilerGeneratedAttribute *) ] 
            else [mk_CompilationMappingAttr cenv.g (7 (* Module *) )]))) in 
    let tdef = { tdef with tdSealed=true; tdAbstract=true } in
    insert_tdef mgbuf tref tdef


and gen_mexpr cenv cgbuf qname lazyInitInfo eenv cloc x   = 
    if verbose then dprintf2 "gen_mexpr, tspec(cloc) = %a\n" Ilprint.output_tspec (mk_tspec_for_cloc cloc);
    match x with  
    | TMTyped(mty,def,m) -> 
        let mrmi = mk_mdef_to_mtyp_remapping def mty in
        let eenv = add_rmi "defs" mrmi eenv in 
        let eenv = add_binds_for_mdef cenv.g eenv.cloc eenv def in 
        gen_mdef cenv cgbuf qname lazyInitInfo eenv def

and gen_mdefs cenv cgbuf qname lazyInitInfo eenv  mdefs = 
    iter (gen_mdef cenv cgbuf qname lazyInitInfo eenv) mdefs
    
and gen_mdef cenv cgbuf qname lazyInitInfo eenv  x = 

    let checkForDeclsAfterEntryPoint m = 
        if isSome(cgbuf.mgbuf.explicitEntryPointInfo) then
            errorR(Error("A function labelled with the 'EntryPointAttribute' attribute must be the last declaration in the last file in the compilation sequence",m)) in
        
    if verbose then dprintf2 "gen_mdef, tspec(cloc) = %a\n" Ilprint.output_tspec (mk_tspec_for_cloc eenv.cloc);
    match x with 
    | TMDefRec(tycons,binds,m) -> 
        checkForDeclsAfterEntryPoint m;
        tycons |> iter (fun tc -> 
            if tycon_is_exnc tc
            then gen_exnconstr_def cenv cgbuf.mgbuf eenv m tc 
            else gen_type_def cenv cgbuf.mgbuf lazyInitInfo eenv m tc) ;
        gen_letrec_binds cenv cgbuf eenv (binds,m)

    | TMDefLet(bind,m) -> 
        checkForDeclsAfterEntryPoint m;
        gen_binds cenv cgbuf eenv [bind]

    | TMAbstract(mexpr) -> 
        gen_mexpr cenv cgbuf qname lazyInitInfo eenv eenv.cloc mexpr

    | TMDefModul(TMBind(tycon, mdef)) -> 
        let id = id_of_modul tycon in 
        let cpath = cpath_of_modul tycon in 
        let istype = mkind_of_modul tycon in 
        let attribs = attribs_of_modul tycon in
        let hidden = isHiddenTycon eenv.mrmi tycon in 

        checkForDeclsAfterEntryPoint id.idRange;
        if verbose then dprintf4 "--> TMDefModul %s, ?hidden: %b, ?namespace: %b, cpath = %s\n" id.idText hidden (istype = Namespace) (text_of_cpath cpath);
        let eenvinner = if istype = Namespace then eenv else {eenv with cloc = cloc_for_cpath cenv.fragName (text_of_qualNameOfFile qname) cpath } in 

        (* Create the class to hold the contents of this module.  No class needed if *)
        (* we're compiling it as a namespace *)
        if istype <> Namespace then 
            gen_tdef_for_cloc cenv eenvinner cgbuf.mgbuf eenvinner.cloc hidden attribs;
        gen_mdef cenv cgbuf qname lazyInitInfo eenvinner  mdef;

        (* Generate the declarations in the module and its initialization code *)
        if verbose then dprint_endline ("Generating initialization call for Module "^id.idText);
        if istype <> Namespace then 
            gen_initialization_cctor cenv cgbuf.mgbuf lazyInitInfo (mk_tref_for_cloc eenvinner.cloc) id.idRange;
        if verbose then dprint_endline (sprintf "<-- module %s" id.idText);

    | TMDefs(mdefs) -> 
        gen_mdefs cenv cgbuf qname lazyInitInfo eenv  mdefs


and gen_top_impl cenv mgbuf mainInfo eenv (TImplFile(qname,mexpr) as impl)  =
    let fragName = text_of_qualNameOfFile qname in 
    if verbose then dprintf1 "-----------------------------------------------------------------------------\ngen_top_impl %s\n" fragName;
    let eenv = {eenv with cloc = { eenv.cloc with clocTopImplQualifiedName = text_of_qualNameOfFile qname } } in

    (* This is used to point the inner classes back to the startup module for initialization purposes *)
    let clocStartup = cloc_for_startup_code eenv.cloc in 
    let startupTspec = mk_nongeneric_tspec (mk_tref_for_cloc clocStartup) in 

    (* create the class to hold the initialization code and static fields for this file.  *)
    gen_tdef_for_cloc cenv eenv mgbuf clocStartup true []; 
    
    let eenv = {eenv with cloc = clocStartup;
                          someTspecInThisModule=mk_nongeneric_tspec (mk_tref_for_cloc clocStartup) }  in 

    let  createStaticInitializerFieldInStartupClass() = 
        let fdef = mk_static_fdef ("_init",cenv.g.ilg.typ_int, None, None, MemAccess_assembly) in 
        let fdef = { fdef with fdAccess = computeMemberAccess true;
                               fdCustomAttrs = mk_custom_attrs (cga cenv.g) } in
        let fspec = mk_fspec_in_boxed_tspec (startupTspec,"_init",cenv.g.ilg.typ_int) in
        static_field_def_counter();
        insert_fdef mgbuf (tref_of_tspec startupTspec) fdef; 
        fspec in

    let  lazyInitInfo = 
      match mainInfo with 
      | Some _ -> None 
      | None -> 
        (* We keep an accumulator of the fragments needed to force the initialization semantics through the compiled code. *)
        (* These fragments only get executed/committed if we actually end up producing some code for the .cctor. *)
        (* NOTE: The existence of .cctors adds costs to execution so this is a half-sensible attempt to avoid adding them when possible. *)
        let initSemanticsAcc = ref [] in 
        let fspec = createStaticInitializerFieldInStartupClass() in
        (*initSemanticsAcc := addCCtor :: !initSemanticsAcc; *)
        Some(fspec,initSemanticsAcc) in 

    if verbose then dprint_endline ("gen_top_impl_expr: codegen .cctor/main for outer module");
    let m = range_of_qualNameOfFile qname in
    let clocCcu = (cloc_for_ccu cenv.viewCcu) in 
    let methodName = match mainInfo with None -> ".cctor" | _ -> mainMethName in 
    let topCode = codegen_method cenv mgbuf ([],methodName,eenv,0,0,(fun cgbuf eenv -> 
                        gen_mexpr cenv cgbuf qname lazyInitInfo eenv clocCcu mexpr;
                        CG.emit_instr cgbuf [] I_ret),m) in

    (* Make a .cctor method to run the top level bindings.  This initializes all modules. *)
    if verbose then dprint_endline ("Creating .cctor/main for outer module");
    let initmeths = 

        match mainInfo, lazyInitInfo with 

        | Some (main_attrs), None -> 

            (* Generate an explicit main method. If necessary, make a class constructor as *)
            (* well for the bindings earlier in the file containing the entrypoint.  *)
            begin match mgbuf.explicitEntryPointInfo with
            | Some(tref) ->           
                if (doesSomething topCode.ilCode) then
                    let fspec = createStaticInitializerFieldInStartupClass() in
                    prepend_instrs_to_specific_mdef (fun md -> md.mdEntrypoint) mgbuf tref (explicit_init_instrs fspec) (gen_opt_range cenv m);
                    [ mk_cctor (MethodBody_il topCode) ] 
                else 
                    []

            (* Generate an implicit main method *)
            | None ->

                let il_attrs = mk_custom_attrs (gen_attrs cenv eenv main_attrs) in 
                if not cenv.empty_ok && not (doesSomething topCode.ilCode) then 
                  warning (Error("Main module of program is empty: nothing will happen when it is run", m));
                let mdef = mk_static_nongeneric_mdef(mainMethName,MemAccess_public,[],mk_return Type_void, MethodBody_il topCode) in 
                [ {mdef with mdEntrypoint= true; mdCustomAttrs = il_attrs } ] 
            end

        (* Generate an on-demand .cctor for the file *)
        | None, Some(fspec, initSemanticsAcc) ->
        
            if (doesSomething topCode.ilCode) then (
                (* Run the imperative (yuck!) actions that force the generation of references to the cctor for nested modules etc. *)
                (List.rev !initSemanticsAcc) |> List.iter (fun f -> f());

                (* Return the generated cctor *)
                [ mk_cctor (MethodBody_il topCode) ] 
            ) else
                [] 

        | _ -> failwith "unreachable" in


    let eenvafter = add_local_mtyp cenv.g  clocCcu eenv (mtyp_of_mexpr mexpr) in
    if verbose then dprint_endline ("Adding .cctor/main for outer module");
    iter (insert_mdef mgbuf (mk_tref_for_cloc clocStartup)) initmeths;
    if verbose then dprintf1 "<-- gen_top_impl %s" fragName;
    eenvafter

and gen_initialization_cctor cenv mgbuf lazyInitInfo tref m =
    (* The cctor forces the cctor for the 'initialization' module by doing a dummy store & load of a field *)
    (* Doing both a store and load keeps FxCop happier because it thinks the field is useful *)
    match lazyInitInfo with 
    | Some (fspec,initSemanticsAcc) -> 
        initSemanticsAcc := (fun () -> add_explicit_init_to_cctor mgbuf tref fspec (gen_opt_range cenv m)) :: !initSemanticsAcc
    | None -> ()

(* Generate a CompareTo method.  *)
and gen_compare cenv mgbuf eenv m this_ty (this_ilty,that_ilty) =
    let mspec = mk_nongeneric_instance_mspec_in_typ (this_ilty, "CompareTo", (if is_unit_typ cenv.g this_ty then [] else [that_ilty]), cenv.g.ilg.typ_int32) in 
    
    mk_normal_virtual_mdef("CompareTo",MemAccess_public,[mk_named_param ("obj",cenv.g.ilg.typ_Object)], mk_return cenv.g.ilg.typ_int32,
         MethodBody_il(mk_ilmbody(true,[],2,
               nonbranching_instrs_to_code
                 ([ ldarg_0; ]
                  @ (if is_unit_typ cenv.g this_ty then [ ] else
                    [  I_ldarg (int_to_u16 1); 
                       I_unbox_any that_ilty; ])
                  @ [I_call ( (if is_struct_typ this_ty then Normalcall else Tailcall), mspec,None) ]), 
               gen_opt_range cenv m))) 

(* Generate an Equals method.  *)
and gen_equals_override_calling_IComparable cenv mgbuf eenv m (this_ilty,that_ilty) =
    let mspec = mk_nongeneric_instance_mspec_in_typ (cenv.g.ilg.typ_IComparable, "CompareTo", [cenv.g.ilg.typ_Object], cenv.g.ilg.typ_int32) in 
    
    mk_normal_virtual_mdef("Equals",MemAccess_public,[mk_named_param ("obj",cenv.g.ilg.typ_Object)], mk_return cenv.g.ilg.typ_bool,
         MethodBody_il(mk_ilmbody(true,[],2,
               nonbranching_instrs_to_code
                 ([ ldarg_0; ]
                  @ [ I_ldarg (int_to_u16 1); ]
                  @ [I_callvirt (Normalcall, mspec,None) ]
                  @ [ mk_ldc_i32 (Int32.of_int 0); I_arith AI_ceq ]), 
               gen_opt_range cenv m))) 

(* Generate a GetHashCode method.  *)
and gen_hash cenv mgbuf eenv m this_tcref this_ilty =
    let hashty = gen_type m cenv.g eenv.tyenv cenv.g.mk_IStructuralHash_ty in
    let mspec = mk_nongeneric_instance_mspec_in_typ (hashty, "GetStructuralHashCode", [Type_byref cenv.g.ilg.typ_int32 ], cenv.g.ilg.typ_int32) in 
    let noGenerics = cenv.g.typeCheckerConfiguredToAssumeErasureOfGenerics in
    
    
    mk_normal_virtual_mdef("GetHashCode",MemAccess_public,[], 
                           mk_return cenv.g.ilg.typ_int32,
                           MethodBody_il(mk_ilmbody(true,([Il.mk_local cenv.g.ilg.typ_int32] ),2,
                                                    nonbranching_instrs_to_code ([ mk_ldc_i32 (Int32.of_int defaultHashNodes); I_stloc (int_to_u16 0); ] @
                                                                                 (match is_struct_tcref this_tcref, noGenerics with 
                                                                                  | true,true -> [ldarg_0; mk_normal_ldobj this_ilty; I_box this_ilty] 
                                                                                  | _ -> [ldarg_0] ) @
                                                                                 [ I_ldloca (int_to_u16 0) ] @
                                                                                 [if not (is_struct_tcref this_tcref) or noGenerics 
                                                                                  then I_callvirt ( Normalcall, mspec,None) 
                                                                                  else I_callconstraint ( Normalcall, this_ilty,mspec,None) ]), 
                                                    gen_opt_range cenv m))) 

and gen_field_init m c =
    match c with 
    | TConst_int8 n -> FieldInit_int8 n
    | TConst_int16 n -> FieldInit_int16 n
    | TConst_int32 n -> FieldInit_int32 n
    | TConst_int64 n -> FieldInit_int64 n
    | TConst_uint8 n -> FieldInit_uint8 n
    | TConst_uint16 n -> FieldInit_uint16 n
    | TConst_uint32 n -> FieldInit_uint32 n
    | TConst_uint64 n -> FieldInit_uint64 n
    | TConst_bool n -> FieldInit_bool n
    | TConst_char n -> FieldInit_char (Nums.unichar_to_u16 n)
    | TConst_float32 n -> FieldInit_float32 n
    | TConst_float n -> FieldInit_float64 n
    | TConst_string s -> FieldInit_bytes s
    | TConst_default -> FieldInit_ref
    | _ -> error(Error("This type may not be used for a literal field",m))


and gen_type_def cenv mgbuf lazyInitInfo eenv m tycon =
    let tcref = mk_local_ref tycon in 
    if verbose then dprintf2 "gen_type_def '%s',  #%d attrs" (name_of_tycon tycon) (length (attribs_of_tycon tycon));
    if isSome(abbrev_of_tycon tycon) then () else
    match repr_of_tycon tycon with 
    | None -> ()
    | Some (TAsmRepr _) -> () 
    | Some (TIlObjModelRepr _ | TFsObjModelRepr _ | TRecdRepr _ | TFiniteUnionRepr _) -> 
        let eenvinner = replace_tyenv (tyenv_for_tycon tycon) eenv in 
        let code_attr = gen_opt_range cenv (range_of_tycon tycon) in
        let _,thisty = generalize_tcref tcref in 
        let ilty = gen_type m cenv.g eenvinner.tyenv thisty in
        let tref = tref_of_typ ilty in
        let tname = tname_of_tref tref in
        let hidden = isHiddenTycon eenv.mrmi tycon in 
        let hiddenRepr = hidden or isHiddenTyconRepr eenv.mrmi tycon in 
        let access = computeTypeAccess tref hidden in
        let gparams = gen_static_formals m cenv eenvinner (typars_of_tycon tycon) in 
        let aug = tcaug_of_tycon tycon in 
        let intfs =  map (p13 >> gen_type m cenv.g eenvinner.tyenv) aug.tcaug_implements in 
        (* The implicit augmentation with comparison methods doesn't actually create CompareTo(object) or Object.Equals *)
        let compare_vmeths = (if isSome (tcaug_of_tycon tycon).tcaug_compare then [ gen_compare cenv mgbuf eenv m thisty (ilty,ilty) ] else [])  in

        let tcaug = tcaug_of_tycon tycon in 

        let compare_vmeths = 
          (* Note: We should be able to get rid of this late-generated method *)
          (if isSome tcaug.tcaug_compare 
           then [ gen_compare cenv mgbuf eenv m thisty (ilty,ilty) ] 
           else []) @
           
          (* Note you only have to implement 'System.IComparable' to customize structural comparison AND equality on F# types *)
          (* See also checkAllImplemented in tc.ml *)
          
          (* Generate an Equals method implemented via IComparable if the type EXPLICITLY implements IComparable *)
          (* HOWEVER, if the type doesn't override Object.Equals already.  *)
          (if isNone tcaug.tcaug_compare &&
              tcaug_has_interface cenv.g tcaug cenv.g.mk_IComparable_ty && 
              not (tcaug_has_override cenv.g tcaug "Equals" [cenv.g.obj_ty])
           then 
              [ gen_equals_override_calling_IComparable cenv mgbuf eenv m (ilty,ilty) ] 
           else []) in

        (* The implicit augmentation with comparison methods doesn't actually create GetHashCode *)
        let hash_vmeths = 
          (if isSome (tcaug_of_tycon tycon).tcaug_structural_hash (* && not (tcref_alloc_observable tcref) *) && not (tcaug_of_tycon tycon).tcaug_hasObjectGetHashCode 
           then [ gen_hash cenv mgbuf eenv m tcref ilty ] 
           else []) in
   
   
        (* Generate the interface implementations and abstract slots.  *)
        let amdefs = 
          if is_fsobjmodel_delegate_tycon tycon then [] else
          Namemap.range_multi (tcaug_of_tycon tycon).tcaug_adhoc |> 
          chooseList (fun vref -> 
             assert(isSome(member_info_of_vref vref));
             let vspr = (the (member_info_of_vref vref)) in 
             let attribs = attribs_of_vref vref in 
             if vspr.vspr_flags.memFlagsAbstract && not vspr.vspr_implemented then 
                let il_attrs = mk_custom_attrs (gen_attrs cenv eenv attribs) in 
                
                let mspec,ctps,mtps,arginfos,retInfo = mspec_for_vspr_vref cenv.g vspr vref  in
                let eenv_for_meth = env_for_typars (ctps@mtps) eenv in 
                let il_mtps = gen_static_formals m cenv eenv_for_meth mtps in 
                let returnv = gen_returnv cenv eenv_for_meth (formal_ret_of_mspec mspec) retInfo in 
                let parameters = gen_params cenv eenv_for_meth mspec arginfos None in 
                let compileAsInstance = vrefCompiledAsInstance cenv.g vref in
                let mdef = mk_generic_virtual_mdef (vspr.vspr_il_name,MemAccess_public,il_mtps,None,parameters,returnv,MethodBody_abstract) in 
                let mdef = fixupVirtualSlotFlags mdef in 
                let mdef = 
                  {mdef with 
                    mdKind=match mdef.mdKind with 
                            | MethodKind_virtual vinfo -> 
                                MethodKind_virtual {vinfo with virtFinal=vspr.vspr_flags.memFlagsFinal;
                                                              virtAbstract=vspr.vspr_flags.memFlagsAbstract; } 
                            | k -> k } in
                
                let il_attrs,pdefOpt = 
                   match vspr.vspr_flags.memFlagsKind with 
                   | MemberKindClassConstructor 
                   | MemberKindConstructor 
                   | MemberKindMember -> il_attrs,None
                   | MemberKindPropertyGetSet -> error(Error("Unexpected GetSet annotation on a property",m));
                   | MemberKindPropertySet | MemberKindPropertyGet ->
                        let v = deref_val vref in 
                        let vtyp = vtyp_of_property_val cenv.g v in
                        let il_propty = gen_type m cenv.g eenv_for_meth.tyenv vtyp in 
                        let il_argtys = v |> arginfos_of_propery_val cenv.g |> List.map fst |> gen_types m cenv.g eenv_for_meth.tyenv in
                        let pdef = gen_property_for_mdef cenv.g compileAsInstance false tref mdef vspr il_argtys il_propty il_attrs  in
                        mk_custom_attrs [], Some(pdef) in 

                let mdef = {mdef with mdCustomAttrs=il_attrs } in
                Some (mdef,pdefOpt)
             else None) in 

        let amdefs,apdefOpts = split amdefs in 
        let apdefs = chooseList (fun pdefOpt -> pdefOpt) apdefOpts |> merge_pdef_list in 
        let abstr =  is_partially_implemented_tycon tycon in 
(*
        if abstr && not (is_interface_typ thisty) then 
          warning(Tc.AbstractType(m));
*)

        let mimpls = 
          (* REVIEW: no method impl generated for IStructuralHash or ICompare *)
          Namemap.range_multi (tcaug_of_tycon tycon).tcaug_adhoc |> 
          chooseList (fun vref -> 
             assert(isSome(member_info_of_vref vref));
             let vspr = (the (member_info_of_vref vref)) in 
             if vspr.vspr_flags.memFlagsOverride then 
               begin
                 match vspr.vspr_implements_slotsig with 
                 | Some (TSlotSig(_,oty,_,_,_,_) as slotsig) when is_interface_typ oty -> 
                     begin match arity_of_vref vref with 
                     | None -> None
                     | Some arities -> 
                         let memberParentTypars,memberMethodTypars = 
                           match partition_vref_typars vref with
                           | Some(_,memberParentTypars,memberMethodTypars,_,_) -> memberParentTypars,memberMethodTypars
                           | None -> [],[] in
                         let use_mimpl = true in 
                         let eenv_under_typars = env_for_typars memberParentTypars eenv in
                         let use_mimpl,_,_,mimplf = gen_mimpl cenv eenv_under_typars (use_mimpl,slotsig) m in 
                         if use_mimpl then Some (mimplf (tspec_of_typ ilty,memberMethodTypars)) else None
                     end
                 | _ -> None
               end 
             else None) in
        
        let tdAttrs = attribs_of_tycon tycon in

        let defaultMemberAttrs = 
            Namemap.range_multi (tcaug_of_tycon tycon).tcaug_adhoc 
            |> Lib.choose (fun vref -> 
                let name = display_name_of_vref vref in
                match member_info_of_vref vref with 
                | None -> None
                | Some vspr -> 
                    match name, vspr.vspr_flags.memFlagsKind with 
                    | "Item", (MemberKindPropertyGet  | MemberKindPropertySet) when nonNil (arginfos_of_propery_val cenv.g (deref_val vref)) ->
                        Some( mk_custom_attribute cenv.g.ilg (mk_tref (cenv.g.ilg.mscorlib_scoref,"System.Reflection.DefaultMemberAttribute"),[cenv.g.ilg.typ_String],[CustomElem_string(Some(name))],[]) ) 
                    | _ -> None)
            |> Option.to_list in

        let tdCustomAttrs = defaultMemberAttrs @ gen_attrs cenv eenv tdAttrs in
        
        let tdef = 
           let tdSerializable = (fsthing_bool_attrib cenv.g cenv.g.attrib_AutoSerializableAttribute tdAttrs <> Some(false)) in
                                       
           match repr_of_tycon tycon with 
           | Some (TIlObjModelRepr (_,_,td)) ->
               let td = {td with tdAccess = access;
                               (* Do not mark this as an F# type yet. *)
                              tdCustomAttrs = mk_custom_attrs tdCustomAttrs;
                              tdGenericParams = gparams; } in 
               td

           | Some (TRecdRepr _ | TFsObjModelRepr _ as tyconRepr)  ->
               let super = super_of_tycon cenv.g tycon in
               let super_il = gen_type m cenv.g eenvinner.tyenv super in 
               
               (* Compute a bunch of useful thnigs for each field *)
               let fields = 
                   tycon 
                   |> all_rfields_of_tycon  
                   |> map (fun fspec -> 

                         let useGenuineField = use_genuine_field tycon fspec in 
                         
                         (* The property (or genuine IL field) is hidden in these circumstances:  *)
                         (*     - secret fields apart from "__value" fields for enums *)
                         (*     - the representation of the type is hidden *)
                         (*     - the F# field is hidden by a signature or private declaration *)
                         let propHidden = 
                             ((fspec.rfield_secret && not (is_enum_tycon tycon)) or 
                              hiddenRepr or 
                              isHiddenRecdField eenv.mrmi (rfref_of_rfield tcref fspec)) in
                         let propType = gen_type m cenv.g eenvinner.tyenv fspec.rfield_type in 
                         let propName = gen_field_name tycon fspec in
                              
                         (useGenuineField,
                          propName,
                          fspec.rfield_mutable, 
                          fspec.rfield_static, 
                          pattribs_of_rfield fspec,
                          propType,
                          propHidden,
                          fspec)) in
                          
               (* Work out those fields accessed via properties *)
               let fieldsAccessedViaProperties =  
                 fields |> filter (fun (useGenuineField,_,_,_,_,_,_,fspec) -> not useGenuineField) in
               
               (* Generate the IL fields *)
               let fdefs = 
                 fields |> map (fun (useGenuineField,nm,mut,stat,_,propType,propHidden,fspec) -> 

                      let literalValue = literal_value_of_rfield fspec in 
                      let fattribs = fattribs_of_rfield fspec in
                      let fdOffset = 
                         match fsthing_tryfind_attrib cenv.g cenv.g.attrib_FieldOffsetAttribute fattribs with
                         | Some (Attrib(_,[ TExpr_const (TConst_int32(fieldOffset),_,_) ],_))  -> 
                             Some fieldOffset
                         | _ -> None in 

                      let fdNotSerialized = fsthing_has_attrib cenv.g cenv.g.attrib_NonSerializedAttribute fattribs in

                      (* The IL field is hidden if the property/field is hidden OR we're using a property AND the field is not mutable (because we can take the address of a mutable field). *)
                      (* Otherwise fields are always accessed via their property getters/setters *)
                      let fdHidden = propHidden || (not useGenuineField && not mut) in

                      { fdName=nm;
                        fdType=propType;
                        fdStatic=stat;
                        fdAccess=computeMemberAccess  fdHidden;
                        fdData=None; (* TODO? *)
                        fdInit= Option.map (gen_field_init m) literalValue;
                        fdOffset=fdOffset;
                        fdSpecialName = (nm="value__" && is_enum_tycon tycon);
                        fdMarshal=None;   (* TODO? *)
                        fdNotSerialized=fdNotSerialized; 
                        fdInitOnly = false;  (* TODO? *)
                        fdLiteral =isSome(literalValue); 
                        fdCustomAttrs=mk_custom_attrs (gen_attrs cenv eenv fattribs) }) in 
               
               (* Generate property definitions for the fields compiled as properties *)
               let pdefs = 
                 fieldsAccessedViaProperties
                 |> list_mapi (fun i (_,nm,mut,stat,pattribs,propType,propHidden,fspec) -> 
                     let cc = if stat then static_callconv else instance_callconv in 
                     let fname = name_of_rfield fspec in 
                     { propName=fname;
                       propRTSpecialName=false;
                       propSpecialName=false;
                       propSet=(if mut then Some(mk_mref(tref,cc,"set_"^fname,0,[propType],Type_void)) else None);
                       propGet=Some(mk_mref(tref,cc,"get_"^fname,0,[],propType));
                       propCallconv=(if stat then CC_static else CC_instance);
                       propType=propType;          
                       propInit=None;
                       propArgs=[];
                       propCustomAttrs=mk_custom_attrs (gen_attrs cenv eenv pattribs @ [mk_CompilationMappingAttrWithSeqNum cenv.g (4 (* SourceLevelConstruct.Field *)) i]); })  in 
               
               (* Compute the sequence point data that goes with some of the generated methods *)
               (* Note: a lot of these should probably have COmpilerStepThrough attributes or something *)
               let sp = match code_attr with None -> [] | Some attr -> [ I_seqpoint attr ] in
               
               (* Generate property getter methods for those fields that have properties *)
               let get_mdefs = 
                 fieldsAccessedViaProperties
                 |> map (fun (_,nm,_,stat,_,propType,propHidden,fspec) -> 
                   let fname = name_of_rfield fspec in 
                   let methnm = "get_"^fname in 
                   let ret = mk_return propType in 
                   let il_fspec = mk_fspec_in_typ(ilty,nm,propType) in 
                   let access = computeMemberAccess propHidden in
                   if stat then 
                       mk_static_nongeneric_mdef (methnm,access,[],ret,MethodBody_il(mk_ilmbody(true,[],2,nonbranching_instrs_to_code(sp @ [mk_normal_ldsfld il_fspec]),code_attr)))
                   else 
                       mk_instance_mdef (methnm,access,[],ret,MethodBody_il(mk_ilmbody (true,[],2,nonbranching_instrs_to_code (sp @ [ ldarg_0; mk_normal_ldfld il_fspec]),code_attr)))) in 

               (* Generate property setter methods for the mutable fields *)
               let set_mdefs = 
                 fieldsAccessedViaProperties
                 |> filter (fun (_,_,mut,_,_,_,_,_) -> mut)
                 |> map (fun (_,nm,mut,stat,_,propType,propHidden,fspec) -> 
                   let fname = name_of_rfield fspec in 
                   let il_fspec = mk_fspec_in_typ(ilty,nm,propType) in 
                   let methnm = "set_"^fname in 
                   let parms = [mk_named_param("value",propType)] in
                   let ret = mk_return Type_void in
                   let access = computeMemberAccess propHidden in
                   if stat then 
                       mk_static_nongeneric_mdef
                         (methnm,access,parms,ret,MethodBody_il
                            (mk_ilmbody(true,[],2,nonbranching_instrs_to_code (sp @[ ldarg_0;mk_normal_stsfld il_fspec]),code_attr)))
                   else 
                       mk_instance_mdef
                         (methnm,access,parms,ret,MethodBody_il
                            (mk_ilmbody(true,[],2,nonbranching_instrs_to_code (sp @[ ldarg_0;I_ldarg (int_to_u16 1);mk_normal_stfld il_fspec]),code_attr)))) in 

               (* Build record constructors and the funky methods that go with delegate types. *)
               let ctor_mdefs = 
                 match tyconRepr with 
                 | TRecdRepr _ ->
                     (* No constructor for enum types *)
                     if is_enum_tycon tycon then [] else
                     (* Otherwise find all the non-static, non zero-init fields and build a constructor *)
                     let fieldNamesAndTypes = 
                          fields 
                          |> filter (fun (_,_,_,stat,_,_,_,fspec) -> not stat && not (zero_init_of_rfield fspec))
                          |> map (fun (_,nm,_,_,_,propType,_,fspec) -> nm,propType) in
                          
                     [ mk_simple_storage_ctor code_attr (Some cenv.g.ilg.tspec_Object)  (tspec_of_typ ilty) fieldNamesAndTypes ] 

                 | TFsObjModelRepr r when is_fsobjmodel_delegate_tycon tycon ->

                     (* Build all the funcky methods that go with a delegate type *)
                     begin match r.tycon_objmodel_kind with 
                     | TTyconDelegate ss ->
                         let p,r = 
                             (* When "type delagateTy = delegate of unit -> returnTy",
                              * suppress the unit arg from delagate .Invoke vslot. *)
                             let (TSlotSig(nm,typ,ctps,mtps,paraml,rty)) = ss in
                             let paraml = match paraml with
                               | [tsp] when is_unit_typ cenv.g (typ_of_slotparam tsp) -> [] (* suppress unit arg *)
                               | paraml -> paraml in
                             gen_actual_slotsig m cenv eenvinner (TSlotSig(nm,typ,ctps,mtps,paraml,rty)) [] in 
                         mk_delegate_mdefs cenv.g.ilg (p,r)
                     | _ -> [] 
                     end
                 | _ -> [] in 

               (* Build a basic type definition *)
               let isObjectType = 
                   hiddenRepr or 
                   (match tyconRepr with TFsObjModelRepr _ -> true | _ -> false) in
               let tdef = 
                 mk_generic_class
                   (tname,
                    access,
                    gparams,
                    super_il, 
                    intfs,
                    mk_mdefs (compare_vmeths @ hash_vmeths @ ctor_mdefs @ get_mdefs @ set_mdefs @ amdefs),
                    mk_fdefs fdefs,
                    mk_properties (pdefs @ apdefs),
                    mk_events [],
                    mk_custom_attrs (tdCustomAttrs @ 
                                     [mk_CompilationMappingAttr cenv.g
                                         (if isObjectType
                                         then 3 (* SourceLevelConstruct.ObjectType *)
                                         else 2 (* SourceLevelConstruct.RecordType *)  )])) in
            (* Set some the extra entries in the definition *)
               let tdef = { tdef with  tdSealed = is_sealed_typ cenv.g thisty; 
                                       tdSerializable = tdSerializable;
                                       tdMethodImpls=mk_mimpls mimpls; 
                                       tdAbstract=abstr } in 
               let tdKind = 
                   match  tyconRepr with 
                   | TFsObjModelRepr o -> 
                       begin match o.tycon_objmodel_kind with 
                       | TTyconClass      -> TypeDef_class
                       | TTyconStruct     -> TypeDef_valuetype
                       | TTyconInterface  -> TypeDef_interface
                       | TTyconEnum       -> TypeDef_enum 
                       | TTyconDelegate _ -> TypeDef_delegate 
                       end 
                   | _ -> TypeDef_class in
               let tdLayout,tdEncoding = 
                  match fsthing_tryfind_attrib cenv.g cenv.g.attrib_StructLayoutAttribute tdAttrs with
                  | Some (Attrib(_,[ TExpr_const (TConst_int32(layoutKind),_,_) ],namedArgs))  -> 
                      if not (match  tyconRepr with | TFsObjModelRepr {tycon_objmodel_kind=(TTyconClass|TTyconStruct)} -> true | _ -> false) then
                          errorR(Error("The StructLayout attribute may only be used with classes and structs",m));
                      let find_int32,_,_ = decodeAttribNamedArgs namedArgs in 
                      let typePack = find_int32 "Pack" 0x0l in
                      let typeSize = find_int32 "Size" 0x0l in
                      let tdEncoding = 
                          match (find_int32 "CharSet" 0x0l) with
                          (* enumeration values for System.Runtime.InteropServices.CharSet taken from mscorlib.il *)
                          | 0x03l -> TypeEncoding_unicode
                          | 0x04l -> TypeEncoding_autochar
                          | _ -> TypeEncoding_ansi in
                      let layoutInfo = 
                          if typePack = 0x0l && typeSize = 0x0l 
                          then { typeSize=None; typePack=None } 
                          else { typeSize = Some typeSize; typePack = Some (Nums.i32_to_u16 typePack) } in
                      let tdLayout = 
                        match layoutKind with
                        (* enumeration values for System.Runtime.InteropServices.LayoutKind taken from mscorlib.il *)
                        | 0x0l -> TypeLayout_sequential layoutInfo
                        | 0x2l -> TypeLayout_explicit layoutInfo
                        | _ -> TypeLayout_auto in
                      tdLayout,tdEncoding
                  | _ when (match tdKind with TypeDef_valuetype -> true | _ -> false) ->
                       (* All structs are sequential by default *) 
                      TypeLayout_sequential { typeSize=None; typePack=None }, TypeEncoding_ansi 
                  | _ -> 
                      TypeLayout_auto, TypeEncoding_ansi in 

               let tdef = { tdef with tdKind =  tdKind; tdLayout=tdLayout; tdEncoding=tdEncoding } in 
               let tdef = match tdKind with TypeDef_interface -> { tdef with tdExtends = None; tdAbstract=true } | _ -> tdef in 
               tdef
           | Some (TFiniteUnionRepr cty) -> 
               let alternatives = Array.mapi (fun i ucspec -> { altName=ucspec.uconstr_il_name;
                                                                altFields=gen_alt_spec m cenv.g eenvinner.tyenv i (rfields_array_of_uconstr ucspec);
                                                                altCustomAttrs= mk_custom_attrs (gen_attrs cenv eenv ucspec.uconstr_attribs @ [mk_CompilationMappingAttrWithSeqNum cenv.g (8 (* SourceLevelConstruct.Alternative *) ) i]) }) 
                                  (uconstrs_array_of_funion cty) in 
               let repr_access = computeMemberAccess (isHiddenTyconRepr eenv.mrmi tycon) in
               let tdef = 
                 { tdName = tname;
                   tdLayout = TypeLayout_auto;
                   tdAccess = access;
                   tdGenericParams = gparams;
                   tdCustomAttrs = 
                       mk_custom_attrs (tdCustomAttrs @ 
                                        [mk_CompilationMappingAttr cenv.g
                                            (if hiddenRepr
                                            then 3 (* SourceLevelConstruct.ObjectType *)
                                            else 1 (* SourceLevelConstruct.SumType *)  )]);
                   tdInitSemantics=TypeInit_beforefield;      
                   tdSealed=true;
                   tdAbstract=false;
                   tdKind=
                   mk_ilx_type_def_kind
                     (ETypeDef_classunion
                        { cudReprAccess=repr_access;
                          cudNullPermitted=isUnionThatUsesNullAsRepresentation cenv.g tycon;
                          cudHelpersAccess=repr_access;
                          cudHelpers=
                            not (cenv.g.tcref_eq tcref cenv.g.unit_tcr_canon) &&
                            begin match fsthing_tryfind_attrib cenv.g cenv.g.attrib_DefaultAugmentationAttribute (attribs_of_tycon tycon) with
                            | Some(Attrib(_,[ TExpr_const (TConst_bool(b),_,_) ],_)) -> b
                            | _ -> not hiddenRepr
                            end;
                          cudAlternatives= alternatives;
                          cudWhere = gen_opt_range cenv (range_of_tycon tycon)});
                   tdFieldDefs = mk_fdefs [];
                   tdEvents= mk_events [];
                   tdProperties = mk_properties [];
                   tdMethodDefs= mk_mdefs (compare_vmeths @ hash_vmeths @ amdefs); 
                   tdMethodImpls= mk_mimpls mimpls; 
                   tdComInterop=false;    
                   tdSerializable= tdSerializable; 
                   tdSpecialName= false;
                   tdNested=mk_tdefs [];
                   tdEncoding= TypeEncoding_autochar;
                   tdImplements= intfs;
                   tdExtends= Some cenv.g.ilg.typ_Object;
                   tdSecurityDecls= mk_security_decls [];
                   tdHasSecurity=false; } in
               tdef
           | _ -> failwith "??" in 
        insert_tdef mgbuf tref tdef;

        (* Types do not by default have .cctors.  However *)
        (* we could end up running a method without having run the initialization code for the scope/file *)
        (* that the thing belongs to.  Hence generate a .cctor in these cases. *)
      
        if (not (Namemap.is_empty aug.tcaug_adhoc) || exists (fun (_,cg,_) -> not cg) aug.tcaug_implements) &&
           not (is_fsobjmodel_interface_tycon tycon) &&
           not (is_fsobjmodel_struct_tycon tycon) &&
           not (is_fsobjmodel_delegate_tycon tycon)  then 
          gen_initialization_cctor cenv mgbuf lazyInitInfo tref m

        
and gen_exnconstr_def cenv mgbuf eenv m exnc =
    let exncref  = mk_local_ecref exnc in
    if verbose then dprintf2 "gen_exnconstr_def %s: #attrs = %d \n" (name_of_tycon exnc) (length (attribs_of_tycon exnc));
    match exn_repr_of_tycon exnc with 
    | TExnAbbrevRepr _ | TExnAsmRepr _ | TExnNone -> ()
    | TExnFresh _ ->
        let ilty = gen_exn_type m cenv.g eenv.tyenv exncref in
        let tref = tref_of_typ ilty in
        let repr_access = computeMemberAccess (isHiddenTyconRepr eenv.mrmi exnc) in
        let access = computeTypeAccess tref (isHiddenTycon eenv.mrmi exnc) in
        let fields = list_mapi (fun i ty -> gen_exn_finfo m cenv eenv.tyenv exnc i) (instance_rfields_of_tycon exnc) in 
        let fdefs = fields |> map (fun (fname,fty) -> Il.mk_instance_fdef(fname,fty, None, repr_access)) in 
        let tname = tname_of_tref tref in
        let compare_vmeth = gen_compare cenv mgbuf eenv m cenv.g.exn_ty (ilty,cenv.g.ilg.typ_Exception) in
        let hash_vmeths = 
          (if isSome (tcaug_of_tycon exnc).tcaug_structural_hash (* && not (tcref_alloc_observable tcref) *) && not (tcaug_of_tycon exnc).tcaug_hasObjectGetHashCode 
           then [ gen_hash cenv mgbuf eenv m exncref ilty ] 
           else []) in

        (* The Message property is used in the report for uncaught exceptions. *)
        (* When the exception wraps a string (e.g. Failure), define a Message property that returns it. *)
        let extra_meths,props =
          let contains_one_string = (match instance_rfields_of_tycon exnc with [rfield] -> type_equiv cenv.g rfield.rfield_type cenv.g.string_ty | _ -> false) in
          if not contains_one_string then
            [],[]
          else
            (* NOTE: could factor out building of properties from read/write il-code *)
            let finfo   = List.hd fields in 
            let fspec   = mk_fspec_in_typ (ilty,fst finfo,snd finfo) in
            let string_concat_mspec = Il.mk_static_mspec_in_typ(cenv.g.ilg.typ_String,"Concat", [cenv.g.ilg.typ_String;cenv.g.ilg.typ_String], cenv.g.ilg.typ_String, [] ) in
            let string_concat_mspec = (* intern_mspec cenv.manager *) string_concat_mspec in
            let prefix = "" in                       (*     So for now, not including name in prefix *)
            let get_Message_meth =
              mk_normal_virtual_mdef
                ("get_Message",
                 MemAccess_public,
                 [], 
                 mk_return cenv.g.ilg.typ_String,
                 MethodBody_il
                   (mk_ilmbody (true,[],  (* init locals and locals *)
                                8,        (* maxstack *)
                                nonbranching_instrs_to_code [
                                  I_ldstr (Bytes.string_as_unicode_bytes prefix);  (* constant  : string *)
                                  ldarg_0;                                                (* this *)
                                  mk_normal_ldfld fspec;                                           (* this.data : String *)
                                  mk_normal_call string_concat_mspec;                              (* concat *)
                                ],
                                gen_opt_range cenv m)))
            in
            let message_prop =
              { propName="Message";
                propRTSpecialName=false;
                propSpecialName=false;
                propSet=None;
                propGet=Some(mk_mref(tref,instance_callconv,"get_Message",0,[],cenv.g.ilg.typ_String));
                propCallconv=CC_instance;
                propType=cenv.g.ilg.typ_String; 
                propInit=None;
                propArgs=[];
                propCustomAttrs=mk_custom_attrs []}
            in
            [get_Message_meth],[message_prop]
        in
        let interfaces = gen_types m cenv.g eenv.tyenv [ cenv.g.mk_IStructuralHash_ty; cenv.g.mk_IComparable_ty ] in
        let tdef = 
          mk_generic_class
            (tname,access,[],cenv.g.ilg.typ_Exception, 
             interfaces,  
             mk_mdefs 
               ([ mk_simple_storage_ctor 
                    (gen_opt_range cenv (range_of_tycon exnc) ) 
                    (Some cenv.g.ilg.tspec_Exception) 
                    (tspec_of_typ ilty)
                    fields; 
                  compare_vmeth;
                ] @ 
                extra_meths @ hash_vmeths
            ),
             mk_fdefs fdefs,
             mk_properties props,
             mk_events [],
             mk_custom_attrs [mk_CompilationMappingAttr cenv.g (5 (* SourceLevelConstruct.Exception *))]) in 
        if verbose then dprintf0 "gen_exnconstr_def:  writing results\n";
        insert_tdef mgbuf tref tdef  

and codegen_assembly cenv eenv mgbuf mimpls = 
    if List.length mimpls > 0 then 
      let a,b = frontAndBack mimpls in 
      let eenv = List.fold_left (gen_top_impl cenv mgbuf None) eenv a in
      let eenv = gen_top_impl cenv mgbuf cenv.main_info eenv b in 
      () 

(*-------------------------------------------------------------------------
!* When generating a module we just write into mutable 
 * structures representing the contents of the module. 
 *------------------------------------------------------------------------- *)

let empty_ilxGenEnv ccu = 
    let this_cloc = cloc_for_ccu ccu in 
    { tyenv=empty_tyenv;
      cloc = this_cloc;
      valsInScope=vspec_map_empty(); 
      someTspecInThisModule=ecma_mscorlib_refs.tspec_Object; (* dummy value *)
      letBoundVars=[];
      liveLocals=Imap.empty();
      innerVals = [];
      mrmi = []; (* "module remap info" *)
      withinSEH = false;}

type codegenResults = 
    { ilTypeDefs: Il.type_def list;
      ilAssemAttrs : Il.custom_attr list;
      ilNetModuleAttrs: Il.custom_attr list;
      quotationResourceBytes: Bytes.bytes list }


let codegen cenv eenv (TAssembly mimpls) (assemA,moduleA) =
  
    (* Generate the implementations into the mgbuf *)
    let mgbuf=new_mgbuf cenv in
    let eenv = { eenv with cloc = cloc_for_fragment cenv.fragName cenv.viewCcu } in 
    gen_tdef_for_cloc cenv eenv mgbuf (cloc_for_PrivateImplementationDetails eenv.cloc) true [];
    codegen_assembly cenv eenv mgbuf mimpls;
    let ilAssemAttrs = gen_attrs cenv eenv assemA in 
    let quotationResourceBytes = 
        match mgbuf.quotedDefs with 
        | [] -> []
        | _ -> 
            if verbose then dprintf0 "creating quotation resource";
            let defnsResource = 
              mgbuf.quotedDefs
              |> chooseList (fun (tdp,ety,e) -> 
                    try 
                      let tps,taue,tauty = 
                        match e with 
                        | TExpr_tlambda (_,tps,b,_,_,_) -> tps,b,reduce_forall_typ ety (map mk_typar_ty tps)
                        | _ -> [],e,ety in 
                      let cenv = Creflect.mk_cenv (cenv.g,cenv.amap,cenv.viewCcu, nng) in 
                      let env = Creflect.bind_typars Creflect.empty_env tps in 
                      let tpsR = map (Creflect.conv_typar cenv env) tps in 
                      Some(tdp,tpsR,Creflect.conv_type cenv env (range_of_expr e) tauty,
                           Creflect.convExpr cenv env taue) 
                    with 
                    | Creflect.InvalidQuotedTerm e -> warning(e); None)
              |> Sreflect.Raw.pickleDefns in
            [ defnsResource ] in
    let ilNetModuleAttrs = gen_attrs cenv eenv moduleA in 

    if verbose then dprintf0 "codegen complete";
    { ilTypeDefs= mgbuf_to_top_tdefs mgbuf;
      ilAssemAttrs = ilAssemAttrs;
      ilNetModuleAttrs = ilNetModuleAttrs;
      quotationResourceBytes = quotationResourceBytes }

    
