(* (c) Microsoft Corporation. All rights reserved *)

(*F# 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.Research.AbstractIL.Extensions.ILX
open Microsoft.FSharp.Compiler 
module Ilsupp = Microsoft.Research.AbstractIL.Internal.Support 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Il = Microsoft.Research.AbstractIL.IL 
module Unilex = Microsoft.FSharp.Compiler.UnicodeLexing 

#if CLI_AT_MOST_1_1
#else
open System.Runtime.CompilerServices
[<Dependency("Microsoft.Research.AbstractIL",LoadHint.Always)>] do ()
[<Dependency("FSharp.Core",LoadHint.Always)>] do ()
[<Dependency("FSharp.Compatibility",LoadHint.Always)>] do ()
#endif
F#*)

open List
open Build
open Ildiag
open Env
open Tc
open Tast
open Tastops 
open Ast
open Il
open Lib
open Range
open Lexhelp
open Ilxgen

let product = ref "MSR F# Compiler"
let maxErrors = ref 10
let baseAddress = ref None
let showTimes = ref false (* show times between passes? *)
let lexFilterVerbose = (*IF-OCAML*) false (*ENDIF-OCAML*) (*F# false F#*)
  
(* Backend configuration *)
let typeCheckOnly = ref false
let parseOnly = ref false
let printAst = ref false
let tokenizeOnly = ref false
let testInteractionParser = ref false
let printSignature = ref false
let printSignatureFile = ref ""
let xmldoc = ref None
let generateHtmlDocs = ref false
let htmlDocDirectory = ref None
let htmlDocCssFile = ref None
let htmlDocNamespaceFile = ref None
let htmldoc_append = ref false
let stats = ref false
let debuginfo = ref false
let filterBlocks = ref false (* don't generate filter blocks due to bugs on Mono *)
let outputFileRef = ref None
let debugSymbolFileRef = ref None
type target = WinExe | Exe | Dll | Module
let target = ref Exe

type signer = KeyFile of string | PublicKeyFile of string
let signer = ref None

let delaysign = ref false
let version = ref None
let desiredCLIMetadataVersionOpt = ref None
let standalone = ref false
let extraStaticLinkRoots = ref []
let noSignatureData = ref false
let onlyEssentialOptimizationData = ref false
let useOptimizationDataFile = ref false
let jitTracking = ref true
let retargetable = ref []
let internConstantStrings = ref true
let generateConfigFile = ref false
let extraOptimizationIterations = ref 0

let win32res = ref []
let embedResources = ref []
let linkResources = ref []

(* REVIEW: tidy up *)
let ilfiles       = ref false (* write il files? *)  
let showTerms     = ref false (* show terms between passes? *)
let showTermFileP = ref false (* show terms to files? *)
let showOptimizationData = ref false
let doDetuple     = ref false (* run detuple pass? *)
let doTLR         = ref false (* run TLR     pass? - not by default *)
let noEnvCCU      = ref false (* include env ccu? *)
type platform = X86
let platform = ref (None : platform option)
let setFlag r = function
    | 0 -> r := false
    | 1 -> r := true
    | _ -> raise (Failure "expected 0/1")
 
let opt_Ooff () = Opt.jitopt_user := Some false;                       Opt.localopt_user := Some false; doDetuple := false; doTLR := false; Opt.crossmoduleopt_user := Some false
let opt_O0() =    Opt.jitopt_user := Some true; jitTracking := false;  Opt.localopt_user := Some false; doDetuple := false; doTLR := false; Opt.crossmoduleopt_user := Some false
let opt_O1() =    Opt.jitopt_user := Some true; jitTracking := false;  Opt.localopt_user := Some true;  doDetuple := true;  doTLR := false; Opt.crossmoduleopt_user := Some false
let opt_O2() =    Opt.jitopt_user := Some true; jitTracking := false;  Opt.localopt_user := Some true;  doDetuple := true;  doTLR := false; Opt.crossmoduleopt_user := Some true
let opt_O3() =    Opt.jitopt_user := Some true; jitTracking := false;  Opt.localopt_user := Some true;  doDetuple := true;  doTLR := true;  Opt.crossmoduleopt_user := Some true; Opt.lambda_inline_threshold := !Opt.lambda_inline_threshold + 2

let (++) x s = x @ [s]

(*----------------------------------------------------------------------------
!* resources
 *--------------------------------------------------------------------------*)
 
let splitResourceInfo ri = 
    if String.contains ri ',' then 
        let p = String.index ri ',' in 
        let file = String.sub ri 0 p in 
        let rest = String.sub ri (p+1) (String.length ri - p - 1) in 
        if String.contains rest ',' then 
            let p = String.index rest ',' in 
            let name = String.sub rest 0 p^".resources" in 
            let pubpri = String.sub rest (p+1) (String.length rest - p - 1) in 
            if pubpri = "public" then file,name,Resource_public 
            else if pubpri = "private" then file,name,Resource_private
            else failwith ("unrecognized privacy setting "^pubpri^" for managed resource")
        else file,rest^".resources",Resource_public
    else ri,Filename.basename ri,Resource_public 

let addEmbeddedResource tcConfig filename =
    let file,name,pub = splitResourceInfo filename in 
    let file = resolveSourceFile tcConfig rangeStartup file in 
    let bytes = readBinaryFile file in       
    embedResources := !embedResources ++ (bytes,name,pub)
    
(*----------------------------------------------------------------------------
!* cmd line - --options
 *--------------------------------------------------------------------------*)

let set_cli_version_10(tcConfig) = 
        (Msilxlib.maximum_cli_library_version_to_depend_upon := Ilxsettings.CLI_Library_v1_0;
         desiredCLIMetadataVersionOpt := Some "v1.0.3705";
         tcConfig.desiredCLILibraryVersionOpt <- Some (Il.parse_version "1.0.3300"))

let set_cli_version_11(tcConfig) = 
        (Msilxlib.maximum_cli_library_version_to_depend_upon := Ilxsettings.CLI_Library_v1_0;
         desiredCLIMetadataVersionOpt := Some "v1.1.4322";
         tcConfig.desiredCLILibraryVersionOpt <- Some (Il.parse_version "1.0.5000"))

let set_cli_version_20(tcConfig) = 
        (Msilxlib.maximum_cli_library_version_to_depend_upon := Ilxsettings.CLI_Library_v2_0rc;
         desiredCLIMetadataVersionOpt := Some "v2.0.50727";
         tcConfig.desiredCLILibraryVersionOpt <- Some (Il.parse_version "2.0.0"))

let isCLIBuildVersionTag(s) = String.length s > 0 && s.[0] = 'v' 
let setCLIBuildTag(tcConfig,s) = 
    assert(isCLIBuildVersionTag(s));
    let v2 = (has_prefix s "v2") in 
    (Msilxlib.maximum_cli_library_version_to_depend_upon := (if v2 then Ilxsettings.CLI_Library_v1_0 else Ilxsettings.CLI_Library_v2_0rc);
     desiredCLIMetadataVersionOpt := Some s;
     tcConfig.desiredCLILibraryVersionOpt <- Some (if v2 then Il.parse_version "2.0.0" else Il.parse_version "1.0.3300"))

let advancedUsageFlags tcConfig = 
  let frontEndBasicUsage, frontEndAdvancedUsage =  frontEndArgSpecs tcConfig in 
  frontEndAdvancedUsage @ 
  [ 
    "--generate-interface-file", Arg.String (fun s -> printSignature := true; printSignatureFile := s), "\n\tPrint the inferred interface of the\n\tassembly to a file.";    

    "--target-exe",  Arg.Unit (fun () -> target := Exe), "Produce an executable with a console";   
    "--target-winexe",  Arg.Unit (fun () -> target := WinExe), "Produce an executable which does not have a\n\tstdin/stdout/stderr";  
    "--target-dll",  Arg.Unit (fun () -> target := Dll), "Produce a DLL"; 
    "--target-module",  Arg.Unit (fun () -> target := Module), "Produce a module that can be added to another assembly"; 
    "--auto-resolve",  Arg.Unit (fun () -> tcConfig.autoResolveOpenDirectivesToDlls <- true), "Automatically reference DLLs whose name matches a required\n\tF# module or namespace. Experimental."; 
    "-c",  Arg.Unit (fun () -> target := Module; tcConfig.autoResolveOpenDirectivesToDlls <- true), "Compile to a module (--target-module) and\n\tauto-resolve (--auto-resolve). Experimental.";

    "--standalone", Arg.Unit (fun s -> tcConfig.openDebugInformationForLaterStaticLinking <- true; standalone := true), "\n\tStatically link the F# library and all transitively referenced DLLs\n\tthat depend on the F# library into the assembly being generated.\n\tDisables the generation of the F# interface and optimization\n\tresources and renames all types beginning with Microsoft.FSharp,\n\tmaking them private to the target assembly. There must be no\n\tduplicate type names across any statically linked\n\tassemblies, no code may use reflection on any of the\n\ttypes in those assemblies, and no code may export F# library types\n\tas part of an API.";
    "--static-link", Arg.String (fun s -> extraStaticLinkRoots := !extraStaticLinkRoots @ [s]), "\n\tStatically link the given assembly and all transitively referenced DLLs\n\tthat depend on this assembly.  NOTE: use an assembly name e.g. mylib\n\t, not a DLL name, e.g. mylib.dll.";
    
    "--keyfile", Arg.String (fun s -> signer := Some(KeyFile(s))), "\n\tSign the assembly the given keypair file, as produced\n\tby the .NET Framework SDK 'sn.exe' tool. This produces\n\tan assembly with a strong name. This is only relevant if producing\n\tan assembly to be shared amongst programs from different\n\tdirectories, e.g. to be installed in the Global Assembly Cache.";
    "--public-keyfile", Arg.String (fun s -> signer := Some(PublicKeyFile(s))), "\n\tDelay sign with the given public key file. The assembly can\n\tbe fully signed later using 'sn.exe' from the .NET Framework SDK.";
    "--retargetable", Arg.String (fun s -> retargetable := !retargetable ++ s), "Specify that an assembly should be referenced in a\n\tretargetable fashion. Used to build binaries which are independent of\n\tthe publisher of the referenced DLLs.";
    "--win32res", Arg.String (fun s -> win32res := !win32res ++ s), "Specify a Win32 resource file (.res)";
    "--platform", Arg.String (fun s -> platform := match s with "x86" -> Some X86 | "anycpu" -> None | _ -> failwith ("unrecognized platform '"^s^"'")), "Limit which platforms this code can run on:\n\t\tx86\n\t\tanycpu\n\tThe default is anycpu. Other platform limitation specifications\n\tsuch as x64 and Itanium are not supported.";
    "--resource", Arg.String (fun s -> addEmbeddedResource tcConfig s), "Embed the specified managed resources (.resource).\n\tProduce .resource files from .resx files using resgen.exe or resxc.exe.";
    "--link-resource", Arg.String (fun s -> linkResources := !linkResources ++ s), "\n\tLink the specified managed resource file (.resource).\n\tFormat may be <file>[,<stringname>[,public|private]].\n\tYou can produce .resource files from .resx files using resgen.exe.";
    "--base-address", Arg.String (fun s -> baseAddress := Some(Int32.of_string s)), "Base address for the library to be built.";
    
    "--version-file",  Arg.String (fun s -> version := Some (Il.parse_version (input_line (open_in s)))), "";
    "--version",  Arg.String (fun s -> version := Some (Il.parse_version s)), "\n\tSpecify the version numbers for the generated .NET assembly\n\t(e.g. 1.2.0.1).";
    "--cli-version",  
    Arg.String (fun s -> 
      if s = "1.0" then set_cli_version_10(tcConfig)
      else if s = "1.1" then set_cli_version_11(tcConfig)
      else if s = "2.0rc" or s = "2.0" then set_cli_version_20(tcConfig)
      else if isCLIBuildVersionTag(s) then setCLIBuildTag(tcConfig,s)
      else failwith("invalid CLI version "^s)), 
    "\n\tSpecify the CLI version to generate code for, regardless of the CLI you\n\thave installed at compile time. \n\tValid values for this version of F# are:\n\t\t1.0\n\t\t1.1\n\t\t2.0\n\t\tbuild tags such as v2.0.x86chk\n\tYou may have to add the right Framework directory to your include path\n\t\t--clr-root \\WINDOWS\\Microsoft.NET\\Framework\\v1.1.4322\n\tYou should also run the appropriate version of peverify on your\n\tbinaries, e.g C:\\Program Files\\Microsoft Visual Studio .NET 2003\\\n\t\\SDK\\v1.1\\Bin\\PEVerify.exe\n\tto ensure neither you nor any of the libraries you import\n\thave used features specific to other versions of the .NET Framework.";
    "--clr-mscorlib", Arg.String (fun s -> tcConfig.Build.mscorlib_assembly_name <- s), "\n\tThe name of mscorlib on the target CLR";    
    "--debug-file", Arg.String (fun s -> debugSymbolFileRef := Some s), "\n\tName the debug output file.";
    "--generate-debug-file",  Arg.Set debuginfo, "";
    "--no-debug-file",  Arg.Clear debuginfo, "\n\tControl whether debug information is produced or not. Unlike -g\n\tthis does not by default turn off optimizations.";
    "--no-debug-file",  Arg.Clear debuginfo, "\n\tControl whether debug information is produced or not. Unlike -g\n\tthis does not by default turn off optimizations.";
    "--generate-html", Arg.Set generateHtmlDocs, "\n\tGenerate HTML documentation.";
    "--html-output-directory", Arg.String (fun s ->  htmlDocDirectory := Some s) , "\n\tOutput directory for HTML documentation.";
    "--html-css", Arg.String (fun s ->  htmlDocCssFile := Some s), "\n\tSet the name of the Cascading Style Sheet for HTML docs.";
    "--html-namespace-file", Arg.String (fun s ->  htmlDocNamespaceFile := Some s), "\n\tSet the name of the master namespaces.html file\n\tassumed to be in the output directory.";
    "--html-namespace-file-append", Arg.Set htmldoc_append , "\n\tAppend to the master namespace file when generating HTML documentation.";
    "--no-optimization-data", Arg.Set onlyEssentialOptimizationData, "\n\tOnly include optimization information essential for implementing \n\tinlined constructs. Inhibits cross-module inlining but improves\n\tbinary compatibility.";
    "--generate-optimization-data-file", Arg.Set useOptimizationDataFile, "\n\tPlace optional information in a .optdata file";
    "--no-interface-data", Arg.Set noSignatureData, "\n\tDon't add a resource to the generated assembly containing the\n\tinterface data for use by other F# assemblies. The assembly can then\n\tonly be accessed as a .NET assembly.";
    "--quotation-data", Arg.Unit (fun () -> ()), "\n\tThis flag is now obsolete. Consider using the\n\t'ReflectedDefinition' attribute instead.";
    "--generate-config-file",  Arg.Set generateConfigFile, "\n\tGenerate a config file to ensure that a .exe runs on the\n\tversion of the CLR specified by --cli-version.  The\n\tfile will contain a 'supportedRuntime' XML element.";
    
    "--jit-optimize",               Arg.Unit (fun _ -> Opt.jitopt_user := Some true), "";
    "--no-jit-optimize",            Arg.Unit (fun _ -> Opt.jitopt_user := Some false), "";
    "--local-optimize",             Arg.Unit (fun _ -> Opt.localopt_user := Some true), "";
    "--no-local-optimize",          Arg.Unit (fun _ -> Opt.localopt_user := Some false), "";
    "--cross-optimize",             Arg.Unit (fun _ -> Opt.crossmoduleopt_user := Some true), "";
    "--no-cross-optimize",          Arg.Unit (fun _ -> Opt.crossmoduleopt_user := Some false), "\n\tFine grained control for JIT, local and cross-module optimizations.";


    "--fast-sublanguage-only",        Arg.Clear generalize_inner_polymorphism, "\n\tRestrict the F# langauge slightly to rule out constructs that are\n\tmore difficult to compile efficiently, leading to more optimization\n\topportunities for the compiler. This can significantly improve\n\tperformance. Combines all the --no-ABC flags that follow.";
    "--no-inner-polymorphism", Arg.Clear generalize_inner_polymorphism, "\n\tDo not permit inner bindings to be genuinely polymorphic.";
    "--permit-inner-polymorphism", Arg.Set generalize_inner_polymorphism, "";
    
    "--no-string-interning", Arg.Clear internConstantStrings, "Do not intern compiled string constants at runtime.";
    
    "--statistics",  Arg.Set stats, "Show statistics of compiler.";
    "--progress",  Arg.Set progress, "Show progress of compiler.";
    
    "--no-framework", Arg.Unit (fun () -> tcConfig.framework <- false), "\n\tDo not reference the .NET Framework assemblies by default.";
    "--sscli", Arg.Unit (fun s -> Ilsupp.configure Ilsupp.SSCLI; tcConfig.sscli <- true), "\n\tForce the compiler to look use the installation\n\tof the SSCLI found on the path rather than an installed Microsoft CLI.\n\tAlso produces debug information in the SSCLI format.";
    "--max-errors", Arg.Int (fun n -> maxErrors := n), "<internal use>";
    
    "--generate-filter-blocks",        Arg.Set   filterBlocks,          "\n\tGenerate filter blocks to allow first-chance\n\texception catching for non-matching exceptions"; 
    "--stamps",        Arg.Set   Tastops.DebugPrint.layout_stamps,      "<internal use>"; 
    "--ranges",        Arg.Set   Tastops.DebugPrint.layout_ranges,      "<internal use>";   
    "--terms" ,        Arg.Set   showTerms,                             "<internal use>";
    "--termsfile" ,    Arg.Set   showTermFileP,                         "<internal use>";
    "--ilfiles" ,      Arg.Set   ilfiles,                               "<internal use>";
    "--times" ,        Arg.Unit  (fun () -> showTimes := true),         "<internal use>";       
    "--detuple",       Arg.Int   (setFlag  doDetuple),                  "<internal use>";
    "--tlr",           Arg.Int   (setFlag  doTLR),                      "<internal use>";
    "--tlrlift",       Arg.Int   (setFlag  Tlr.liftTLR),                "<internal use>";
    "--no-env",        Arg.Set   noEnvCCU,                              "<internal use>";

    "--parse-only",    Arg.Set parseOnly,                              "<internal use only>";
    "--typecheck-only",Arg.Set typeCheckOnly,                          "<internal use only>";
    "--ast",           Arg.Set printAst,                               "<internal use only>";    
    "--tokenize",  Arg.Set tokenizeOnly,                               "<internal use only>";
    "--testInteractionParser",  Arg.Set testInteractionParser,         "<internal use only>";
    "--light",  Arg.Unit (fun () -> tcConfig.light <- true),           "<experimental use only>";
    "--lightoff",  Arg.Unit (fun () -> tcConfig.light <- false),       "<experimental use only>";
    "--inline-threshold", Arg.Int (fun n -> Opt.lambda_inline_threshold := n),"<internal use only>";
    "--extra-optimization-loops", Arg.Int (fun n -> extraOptimizationIterations := n),"<internal use only>";
  ] 
  @ Ilxerase.usage

let basicUsageFlags tcConfig =
    let frontEndBasicUsage, frontEndAdvancedUsage =  frontEndArgSpecs tcConfig in 
    frontEndBasicUsage @
    [ "-o", Arg.String (fun s -> outputFileRef := Some s), "Name the output file.";
      "-a",  Arg.Unit (fun () -> target := Dll), "Produce a DLL.";
      
      "-g",  Arg.Unit (fun _ -> debuginfo := true; 
                                (* only turn off optimizations if earlier flags haven't explicitly enabled them *)
                                if !Opt.jitopt_user = None then Opt.jitopt_user := Some false; 
                                if !Opt.localopt_user = None then Opt.localopt_user := Some false; 
                                if !Opt.crossmoduleopt_user = None then Opt.crossmoduleopt_user := Some false), "Produce debug file. Disables optimizations if a -O flag is not given.";
      "--define", Arg.String (fun s -> tcConfig.ifdefs <- s :: tcConfig.ifdefs), "Define the given conditional compilation symbol.";
      "-i",     Arg.Set printSignature,      "Print the inferred interface of the assembly.";
      "-doc", Arg.String (fun s -> xmldoc := Some s), "Write the xmldoc of the assembly to the given file.";
      
      "-Ooff",      Arg.Unit opt_Ooff, "Disable all optimizations, including JIT.";
      "-O0",        Arg.Unit opt_O0  , "Enable JIT optimizations.";
      "-O1",        Arg.Unit opt_O1  , "Enable JIT and local optimizations.";
      "-O",         Arg.Unit opt_O2  , "Same as -O2 (default unless -g)";
      "-O2",        Arg.Unit opt_O2  , "Same as -O1 but also enables cross module optimizations.";
      "-O3",        Arg.Unit opt_O3  , "Same as -O2 but with increased inlining and lambda lifting.";  ]

let fullUsageFlags tcConfig = basicUsageFlags tcConfig @ advancedUsageFlags tcConfig

let showUsage (extra,full,basic) showFull =  
    let useline = Printf.sprintf "%s <options> [imported.dll ... ] [file1.ml [file2.ml ...] ] \n" (try Sys.executable_name |> Filename.basename |> Filename.chop_extension with _ -> "fsc") in
    dprint_endline ("Usage: "^useline);
    dprint_endline ("Basic options:");
    List.iter (function 
      | (s, (Arg.Unit _ | Arg.Set _ | Arg.Clear _), x) -> dprintf2 "  %s: %s\n" s x
      | (s, Arg.String f, x) -> dprintf2 "  %s <string>: %s\n" s x
      | (s, Arg.Int f, x) -> dprintf2 "  %s <int>: %s\n" s x
      | (s, Arg.Float f, x) ->  dprintf2 "  %s <float>: %s\n" s x 
      | (s, Arg.Rest f, x) -> dprintf2 "  %s ... %s\n" s x)
      (extra @ (if showFull then full else basic));
    if not showFull then 
        dprintf0 "Further options are available: use --full-help to see them\n"

let helpUsageFlags (extra,full,basic) = 
  let help showFull = Arg.Unit (fun () -> dprintf3 "\n%s, (c) Microsoft Corporation, All Rights Reserved\nF# Version %s, compiling for .NET Framework Version %s\n\n" !product Ilxconfig.version (Ilsupp.clrVersion()); showUsage (extra,full,basic) showFull; exit 0) in 
  [ "/?", help false, "";
    "-help", help false, "";
    "-?", help false, "";
    "--help", help false, "";
    "--full-help", help true, ""; ]

let specs  tcConfig extra = 
    let full = fullUsageFlags  tcConfig in 
    let basic= basicUsageFlags tcConfig in 
    extra @ full @ helpUsageFlags (extra,full,basic) 

(*----------------------------------------------------------------------------
!* parsing - processInput
 * Filename is either (ml/mli/fs/fsi source) or (.resx file).
 * For source file, parse it to AST. For .resx compile to .resource and
 * read.
 *--------------------------------------------------------------------------*)

let processInput tcConfig ifdefs n filename =
    try 
       let lower = String.lowercase filename in 
       if List.exists (Filename.check_suffix lower) [".resx"]  then (
           let filename = resolveSourceFile  tcConfig rangeStartup filename in 
           let outfile = (filename |> Filename.chop_extension) ^ ".resources" in
           
           (*IF-OCAML*)
          (* Fix: 977 *)
          (   let exe = Filename.concat tcConfig.fsharpBinariesDir "resxc.exe" in
              let quote s = Printf.sprintf "\"%s\"" s in
              let cmd = Printf.sprintf "%s -o %s %s" (quote exe) (quote outfile) (quote filename) in
              if n = 0 then dprintf2 "--> Executing '%s'... in directory '%s'\n" cmd (Sys.getcwd());
              let scriptFile = Filename.temp_file "script" ".bat" in
              if n = 0 then dprintf1 "--> using scriptFile '%s'\n" scriptFile;
              let os = open_out scriptFile in 
              if n <> 0 then Printf.fprintf os "@echo off\n";
              Printf.fprintf os "%s" cmd;
              close_out os;
              let res = Sys.command scriptFile in 
              Sys.remove scriptFile; 
              if res <> 0 then error(Error("Resource compilation failed\n",rangeStartup));
          );
           (*ENDIF-OCAML*)
           (*F#
           let readResX(f:string) = 
               use rsxr = new System.Resources.ResXResourceReader(f) in
               rsxr |> Seq.untyped_to_list 
                    |> List.map (fun (d:System.Collections.DictionaryEntry) -> (d.Key :?> string), d.Value) in
           let writeResources((r:(string * obj) list),(f:string)) = 
               use writer = new System.Resources.ResourceWriter(f) in
               r |> List.iter (fun (k,v) -> writer.AddResource(k,v)) in 
           writeResources(readResX(filename),outfile);
           F#*)
           addEmbeddedResource tcConfig outfile;
           Sys.remove outfile;
           None

       ) else if List.exists (Filename.check_suffix lower) (sigSuffixes@implSuffixes)  then  
           Unilex.usingUnicodeFileAsUTF8Lexbuf filename tcConfig.inputCodePage (fun lexbuf -> 
              let skip = true in (* don't report whitespace from lexer *)
              let lightSyntaxStatus = LightSyntaxStatus (ref tcConfig.light) in 
              let lexargs = mkLexargs ((fun () -> tcConfig.implicitIncludeDir),filename,ifdefs@tcConfig.ifdefs,lightSyntaxStatus) in
              let input = 
                  Lexhelp.usingLexbuf lexbuf filename None (fun lexbuf ->
                      if verbose then dprint_endline ("Parsing... "^filename);
                      let tokenizer = Lexfilter.create lightSyntaxStatus (Lex.token lexargs skip) lexbuf in 
                      if !tokenizeOnly then begin 
                          while true do 
                              Printf.eprintf "tokenize - getting one token from %s\n" filename;
                              let t = Lexfilter.get_lexer tokenizer lexbuf in 
                              (*IF-OCAML*) 
                              Printf.eprintf "tokenize - got "; 
                              begin match t with 
                              | Pars.EOF _ -> Printf.eprintf "EOF"
                              | Pars.INT32 n -> Printf.eprintf "INT32 %ld" n
                              | Pars.IEEE64 _ -> Printf.eprintf "IEEE64 "
                              | Pars.IDENT s -> Printf.eprintf "IDENT %s" s
                              | Pars.STRING s -> Printf.eprintf "STRING" 
                              | Pars.OLET _ -> Printf.eprintf "OLET" 
                              | Pars.MUTABLE -> Printf.eprintf "MUTABLE" 
                              | Pars.EQUALS -> Printf.eprintf "EQUALS" 
                              | Pars.OBLOCKBEGIN -> Printf.eprintf "OBLOCKBEGIN" 
                              | Pars.OBLOCKEND -> Printf.eprintf "OBLOCKEND" 
                              | Pars.OBLOCKSEP -> Printf.eprintf "OBLOCKSEP" 
                              | Pars.LPAREN -> Printf.eprintf "LPAREN" 
                              | Pars.RPAREN -> Printf.eprintf "RPAREN" 
                              | Pars.ODECLEND -> Printf.eprintf "ODECLEND" 
                              | Pars.END -> Printf.eprintf "END" 
                              | Pars.OEND -> Printf.eprintf "OEND" 
                              | Pars.WITH -> Printf.eprintf "WITH" 
                              | Pars.OWITH -> Printf.eprintf "OWITH" 
                              | Pars.INTERFACE -> Printf.eprintf "INTERFACE" 
                              | Pars.OINTERFACE_MEMBER -> Printf.eprintf "OINTERFACE_MEMBER" 
                              | Pars.AND -> Printf.eprintf "AND" 
                              | Pars.LARROW -> Printf.eprintf "LARROW" 
                              | Pars.MINUS -> Printf.eprintf "MINUS"  
                              | Pars.PLUS_MINUS_OP s -> Printf.eprintf "PLUS_MINUS_OP %s" s 
                              | Pars.ADJACENT_PREFIX_PLUS_MINUS_OP s -> Printf.eprintf "ADJACENT_PREFIX_PLUS_MINUS_OP %s" s 
                              | _ -> Printf.eprintf "???" 
                              end;
                              Printf.eprintf " @ %a\n" output_range (get_lex_range lexbuf);
                              (*ENDIF-OCAML*)
                              (*F# Printf.eprintf "tokenize - got %s\n" (Pars.token_to_string t); F#*)
                              (match t with Pars.EOF _ -> exit 0 | _ -> ());
                              (*F# if lexbuf.IsPastEndOfStream then  Printf.eprintf "!!! at end of stream\n" F#*)
                          done;
                      end;
                      if !testInteractionParser then begin
                          while true do 
                              match (Pars.interaction (Lexfilter.get_lexer tokenizer) lexbuf) with
                              | IDefns(l,m) -> dprintf3 "Parsed OK, got %d defs @ %a\n" (length l) output_range m;
                              | IHash (_,_,m) -> dprintf2 "Parsed OK, got hash @ %a\n" output_range m;
                          done;
                          exit 0;
                      end;
                      parseInput (Lexfilter.get_lexer tokenizer) lexbuf tcConfig.defaultNamespace filename
                  ) in
             if verbose then dprint_endline ("Parsed "^filename);
             Some input 
         )
       else failwith ("unknown input "^filename)
    with e -> (* errorR(Failure("parse failed")); *) errorRecoveryPoint e; None 

(*----------------------------------------------------------------------------
!* showTerm, reportPhase
 *--------------------------------------------------------------------------*)

let showTermN = ref 0    
let showTerm fL outfile header expr =
    if !showTerms then
        if !showTermFileP then (
          let showTermFile = outfile ^ ".terms" in                
          let n = !showTermN in
          showTermN := n+1;
          let f = open_out (showTermFile ^ "-" ^ string_of_int n ^ "-" ^ header) in
          Layout.outL f (Layout.squashTo 192 (fL expr));
          close_out f
        ) else (
          dprintf1 "\n------------------\nshowTerm: %s:\n" header;
          Layout.outL stderr (Layout.squashTo 192 (fL expr));
          dprintf0 "\n------------------\n";
        )

let noShowTerm f =
    let x = !showTerms in
    showTerms := false;
    let res = f () in
    showTerms := x;
    res

(*----------------------------------------------------------------------------
!* reportPhase
 *--------------------------------------------------------------------------*)

let reportPhase =
    let t0    = Sys.time() in
    let tPrev = ref t0     in
    let reportPhase descr =
        let t = Sys.time() in
        if (!showTimes || verbose) then dprintf3 "TIME %10.3f (total)   %10.3f (delta) - %s\n" t (t -. !tPrev) descr;
        tPrev := t in
    reportPhase

let _ = reportPhase "Started" (* maybe after JIT delay *)

(*----------------------------------------------------------------------------
!* OPTIMIZATION - support - addDllToOptEnv
 *--------------------------------------------------------------------------*)

let add_external_ccu_to_optEnv optEnv ccuinfo =
    match Lazy.force ccuinfo.ccuinfo_optdata with 
    | None -> optEnv
    | Some(data) -> Opt.bind_ccu ccuinfo.ccuinfo_ccu data optEnv 

(*----------------------------------------------------------------------------
!* OPTIMIZATION - support - optimize
 *--------------------------------------------------------------------------*)


let optEnvInit tcImports =
  let ccuinfos = ccuinfosOfTcImports tcImports in    
  let optEnv = Opt.empty_env in
  let optEnv = fold_left add_external_ccu_to_optEnv optEnv ccuinfos in
  optEnv
     
let applyAllOptimizations (tcGlobals,outfile,importMap,isIncrementalFragment) optEnv (ccu,(mimpls:typedAssembly)) =
    (* NOTE: optEnv - threads through *)
    (*---*)
    (* Always optimize once - the results of this step give the x-module optimization *)
    (* info.  Subsequent optimization steps choose representations etc. which we don't *)
    (* want to save in the x-module info (i.e. x-module info is currently "high level"). *)
    reportPhase ("Starting optimization");
    showTerm mimplsL outfile "pass-start" mimpls;
    if !showOptimizationData then dprintf1 "Expression prior to optimization:\n%s\n" (Layout.showL (Layout.squashTo 192 (mimplsL mimpls)));
    if !showOptimizationData then dprintf1 "CCU prior to optimization:\n%s\n" (Layout.showL (Layout.squashTo 192 (mspecL (top_modul_of_ccu ccu))));
    reportPhase ("Printed term");

    (* HACK: Only do abstract_big_targets on the first pass!  Only do it when TLR is on!  *)
    let tlr = !doTLR  in 

    let cenv = (Opt.mk_cenv ccu tcGlobals importMap) in 
    Opt.abstract_big_targets := tlr;
    
    let incrementalOptEnv,mimpls,data = Opt.optimize_assembly cenv optEnv isIncrementalFragment mimpls in
    (* HACK: Only do this on the first pass! *)
    Opt.abstract_big_targets := false;
    if !showOptimizationData then dprintf1 "Optimization data:\n%s\n" (Layout.showL (Layout.squashTo 192 (Opt.modul_infoL data)));
    showTerm mimplsL outfile "post-opt" mimpls;
    reportPhase ("Initial optimization");


    let rec loop n mimpls = 
      if n <= 0 then mimpls
      else begin
        let _,mimpls, _ = Opt.optimize_assembly cenv optEnv isIncrementalFragment mimpls in
        showTerm mimplsL outfile (Printf.sprintf "extra-loop-%d" n) mimpls;
        reportPhase ("Extra optimised optimization");
        mimpls
      end
    in 
    let mimpls = loop !extraOptimizationIterations mimpls in


    let mimpls = 
      if not tlr then mimpls else 
      let mimpls = Detuple.transformApps ccu tcGlobals mimpls in
      showTerm mimplsL outfile "post-detuple" mimpls;
      reportPhase ("Detupled optimization");
      mimpls in 

    let mimpls = 
      if not tlr then mimpls else 
      let mimpls = Tlr.makeTLRDecisions ccu tcGlobals mimpls in
      showTerm mimplsL outfile "post-tlr" mimpls;
      reportPhase ("TLR done optimization");
      mimpls in
      
    let mimpls = 
      let mimpls = Lowertop.lower_assembly tcGlobals mimpls in
      showTerm mimplsL outfile "post-lowertop" mimpls;
      reportPhase ("Lowertop done optimization");
      mimpls in 
      
    let mimpls =
        if not tlr then mimpls else 
        let _,mimpls, _ = Opt.optimize_assembly cenv optEnv isIncrementalFragment mimpls in
        showTerm mimplsL outfile "post-rec-opt" mimpls;
        reportPhase ("Final post TLR/detuple cleanup optimization pass");
        mimpls in

    mimpls, data,incrementalOptEnv

(*----------------------------------------------------------------------------
!* ILX generation 
 *--------------------------------------------------------------------------*)

let ilxgenEnvInit (tcConfig:tcConfig) tcImports tcGlobals generatedCcu = 
  let ccus = ccusOfTcImportsInDeclOrder tcImports in 
  let ilxGenEnv = Ilxgen.empty_ilxGenEnv generatedCcu in 
  Ilxgen.add_external_ccus tcGlobals ilxGenEnv ccus


let generateILX (isIncrementalFragment,isInteractive) tcGlobals importMap topAttrs optimizedImpls generatedCcu fragName ilxGenEnv =
    if !progress then dprintf0 "Generating ILX code...\n";
    let cenv = { g=tcGlobals;
                 viewCcu = generatedCcu;
                 generateFilterBlocks = !filterBlocks;
                 workAroundReflectionEmitMethodImplBug=isInteractive;
                 debug= !debuginfo;
                 fragName = fragName;
                 optimize= Opt.localopt ();
                 Ilxgen.manager=None;
                 Ilxgen.amap=importMap;
                 main_info= (if (!target = Dll or !target = Module) then None else Some topAttrs.mainMethodAttrs);
                 empty_ok = isInteractive;
               } in
    Ilxgen.codegen cenv ilxGenEnv optimizedImpls (topAttrs.assemblyAttrs,topAttrs.netModuleAttrs) 


        
(*----------------------------------------------------------------------------
!* Assembly ref normalization: make sure all assemblies are referred to
 * by the same references.
 *--------------------------------------------------------------------------*)

let normalizeAssemblyRefs tcConfig tcImports = 
(*  if !ilfiles  then (let _ = print_module (outpath outfile "il-pre-fixup") ilxMainModule in ()); *)
  if verbose then dprint_endline "Normalizing assembly references in generated IL code...";
  let assemFinder = findDllInfo tcImports Range.rangeStartup in 
  (fun scoref ->
     match scoref with 
     | ScopeRef_local 
     | ScopeRef_module _ -> scoref
     | ScopeRef_assembly aref -> (assemFinder (aref.assemRefName)).dllinfo_il_scoref)

let fsharpModuleName s = "F#-Module-"^s

(*----------------------------------------------------------------------------
!* Configure based on --cli-version switch and auto-detection of
 * most recent CLR version installed on the machine.  
 *
 * OCAML VERSION: 
 *   IF A CLI VERSION IS GIVEN EXPLICITLY WE ATTEMPT TO LOAD THE CLR FOR THAT
 *   VERSION.  THIS IS NEEDED TO ENSURE WE LOAD THE CORRECT PDB WRITER AND
 *   OTHER COM COMPONENTS FOR THAT VERSION.  
 *   NOTE THE CLR SHOULD NOT HAVE BEEN LOADED BEFORE THIS POINT!!!!
 * 
 * .NET VERSION: 
 *   We have to accept the PDB writer that comes with the version
 *   of the CLR we are already running.  Using a .exe.config file on the compiler DLL
 *   should work if any adjustments are needed.
 *--------------------------------------------------------------------------*)

(*IF-OCAML*) external pCorBindToRuntimeEx: string -> bool -> unit  = "pCorBindToRuntimeExProcCaml" (*ENDIF-OCAML*)

let configureCLIVersion(isFSC,tcConfig) = 
    (* When running fscp10.exe on Mono we lead everyone to believe we're doing .NET 2.0 compilation *)
    (* by default. This will set desiredCLIMetadataVersionOpt if it isn't set already. *)
    if isNone !desiredCLIMetadataVersionOpt && isFSC && compilerProcessRunsMono10() then 
        set_cli_version_20(tcConfig);
        
    let desiredCLIMetadataVersion = 
      match !desiredCLIMetadataVersionOpt with 
      | None -> 
          latestCLRVersionOnMachine_or_PretendNET20ForMono10_or_CLRVersionAlreadyLoaded()
      | Some v -> 
          (*IF-OCAML*) (try pCorBindToRuntimeEx v tcConfig.sscli with Failure s -> Printf.eprintf "Warning: %s\n" s; flush stderr); (*ENDIF-OCAML*)
          Il.parse_version v in 
    configureIlxDefaultsForCLRVersion(desiredCLIMetadataVersion);
    (* Generalization of type variables that flow to .NET array types *)
    (* depends on whether we are compiling using generics or not. *)
    let fullGenericsSupported = (!(Msilxlib.pp_implementation) = Ilxsettings.FullGenerics) in 
    tcConfig.fullGenericsSupported <- fullGenericsSupported; 
    desiredCLIMetadataVersion,fullGenericsSupported
