(* (c) Microsoft Corporation. All rights reserved *)
(*F# 
module Microsoft.FSharp.Compiler.Ast
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
open Microsoft.FSharp.Compiler 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilpars = Microsoft.Research.AbstractIL.Internal.AsciiParser 
module Illex = Microsoft.Research.AbstractIL.Internal.AsciiLexer 
module Il = Microsoft.Research.AbstractIL.IL 
F#*)  
open Ildiag
open Lib
open Printf
open Range

(*----------------------------------------------------------------------
 * General error recovery mechanism
 *----------------------------------------------------------------------*)

exception WrappedError of exn * range
exception ReportedError

(* common error kinds *)
exception Error of string * range
exception InternalError of string * range
exception OCamlCompatibility of string * range
exception LibraryUseOnly of range
exception Deprecated of string * range
exception Experimental of string * range


let warningHandler = ref (fun (e:exn) -> dprintf0 "no warning handler installed\n" ; ())

let errorHandler = ref (fun (e:exn) -> dprintf0 "no error handler installed\n" ; ())

let errorR exn = 
    match exn with 
    | ReportedError -> raise exn 
    | _ -> !errorHandler exn

let warning exn = 
    match exn with 
    | ReportedError -> raise exn 
    | _ -> !warningHandler exn

let errorRecoveryPoint (exn:exn) =
    match exn with
    (* Don't send ThreadAbortException down the error channel *)
    (*F# 
    |  :? System.Threading.ThreadAbortException 
    |  WrappedError((:? System.Threading.ThreadAbortException),_) ->  () 
    F#*)
    | ReportedError 
    | WrappedError(ReportedError,_) -> ()
    | e -> errorR exn

let error exn = 
    match exn with 
    | ReportedError -> raise exn 
    | _ -> !errorHandler exn; raise ReportedError

(*----------------------------------------------------------------------
 *  Errors as data: Sometimes we have to reify errors as data, e.g. if backtracking 
 *----------------------------------------------------------------------*)

type warning = exn
type error = exn
type 'a operationResult = 
    | OkResult of warning list * 'a
    | ErrorResult of warning list * error
    
type imperativeOperationResult = unit operationResult

let commitOperationResult res = 
    match res with 
    | OkResult (warns,res) -> List.iter warning warns; res
    | ErrorResult (warns,err) -> List.iter warning warns; error err

let raiseOperationResult res : unit = commitOperationResult res

let errorD err = ErrorResult([],err)
let warnD err = OkResult([err],())
let completeD = OkResult([],())
let resultD x = OkResult([],x)
let gaveNoErrors res = match res with OkResult _ -> true | ErrorResult _ -> false

(* The bind in the monad. Stop on first error. Accumulate warnings and continue. *)
let (++) res f = 
    match res with 
    | OkResult([],res) -> (* tailcall *) f res 
    | OkResult(warns,res) -> 
        begin match f res with 
        | OkResult(warns2,res2) -> OkResult(warns@warns2, res2)
        | ErrorResult(warns2,err) -> ErrorResult(warns@warns2, err)
        end
    | ErrorResult(warns,err) -> 
        ErrorResult(warns,err)

(* Stop on first error. Accumulate warnings and continue. *)
let rec iterD f xs = 
    match xs with 
    | [] -> completeD 
    | h :: t -> f h ++ (fun () -> iterD f t)

(* Stop on first error. Report index *)
let iteriD f xs = 
    let rec loop xs i = match xs with [] -> completeD | h :: t -> f i h ++ (fun () -> loop t (i+1)) in 
    loop xs 0

(* Stop on first error. Accumulate warnings and continue. *)
let rec iter2D f xs ys = 
    match xs,ys with 
    | [],[] -> completeD 
    | h1 :: t1, h2::t2 -> f h1 h2 ++ (fun () -> iter2D f t1 t2) | _ -> failwith "iter2D"

let tryD f g = 
    match f() with
    | ErrorResult(warns,err) ->  (OkResult(warns,())) ++ (fun () -> g err)
    | res -> res

let rec whileD gd body = 
    if gd() then body() ++ (fun () -> whileD gd body) 
    else completeD

let rec repeatWhileD body = 
    body() ++ (function true -> repeatWhileD body | false -> completeD) 

let rec atLeastOneD f l = 
    match l with 
    | [] -> resultD(false) 
    | h::t -> f h ++ (fun res1 -> atLeastOneD f t ++ (fun res2 -> resultD (res1 || res2)))

(*----------------------------------------------------------------------
 *  deprecation errors
 *----------------------------------------------------------------------*)

let deprecated s m = 
    warning(Deprecated(s,m))

let libraryOnly m = 
    warning(LibraryUseOnly(m))

let deprecated_op m = 
    deprecated "the treatment of this operator is now handled directly by the F# compiler and its meaning may not be redefined" m

let ocamlCompat s m = 
    warning(OCamlCompatibility(s,m))

(*----------------------------------------------------------------------
 *  AST: main ast definitions
 *----------------------------------------------------------------------*)

type xmlDoc = XMLDoc of string list
let emptyXMLDoc = XMLDoc[]
  
type ident = { idText: string; idRange: range }
type module_name = ident 

type typar_name = ident 
type value_name = ident 
type exn_name = ident 
type constr_name = ident 
type field_name = ident 
type tycon_name = ident 

type module_path = module_name list
type field = module_path * field_name 
type long_ident = ident list
type access = | Access of int  (* 0 = public, 1 = assembly, 2 = outer module etc. *)

let accessPublic = Access 0
let accessInternal = Access 1
let accessPrivate = Access max_int

type 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  sconst = 
  | Const_unit
  | Const_bool of bool
  | Const_int8 of Nums.i8
  | Const_uint8 of Nums.u8
  | Const_int16 of Nums.i16
  | Const_uint16 of Nums.u16
  | Const_int32 of int32
  | Const_uint32 of Nums.u32
  | Const_int64 of int64
  | Const_uint64 of Nums.u64
  | Const_nativeint of int64
  | Const_unativeint of Nums.u64
  | Const_float32 of Nums.ieee32
  | Const_float of Nums.ieee64
  | Const_char of Nums.unichar
  | Const_decimal of Bytes.bytes (* in unicode *)
  | Const_bigint of Bytes.bytes (* in unicode *)
  | Const_bignum of Bytes.bytes (* in unicode *)
  | Const_string of Bytes.bytes * range (* unicode encoded, F#/Caml independent *)
  | Const_bytearray of Bytes.bytes * range (* F#/Caml independent  *)

and  simple_pat =
  | SPat_as of  value_name * bool (* true if 'this' variable in member *) * bool (* true if an optional parm. *) * range
  | SPat_typed of  simple_pat * typ * range
  | SPat_attrib of  simple_pat * attributes * range

and  simple_pats =
  | SPats of simple_pat list * range
  | SPats_typed of  simple_pats * typ * range

and  
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  synpat =
  | Pat_const of sconst * range
  | Pat_wild of range
  | Pat_as of  synpat * value_name * bool (* true if 'this' variable *)  * access option * range
  | Pat_instance_member of  value_name * value_name * access option * range (* adhoc overloaded method/property *)
  | Pat_typed of  synpat * typ * range
  | Pat_attrib of  synpat * attributes * range
  | Pat_disj of  synpat * synpat * range
  | Pat_conjs of  synpat list * range
  | Pat_lid of long_ident * valTyparDecls option (* usually None: temporary used to parse "f<'a> x = x"*) * synpat list  * access option * range
  | Pat_tuple of  synpat list * range
  | Pat_paren of  synpat * range
  | Pat_array_or_list of  bool * synpat list * range
  | Pat_recd of (field * synpat) list * range
  | Pat_range of Nums.unichar * Nums.unichar * range
  | Pat_null of range
  | Pat_opt_var of ident * range
  | Pat_isinst of typ * range
  | Pat_expr of synexpr * range

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  typ =
  | Type_app of long_ident * typ list * range
  | Type_proj_then_app of typ * long_ident * typ list * range
  | Type_tuple of typ list * range
  | Type_arr of  int * typ * range
  | Type_fun of  typ * typ * range
  | Type_forall of  typarDecl * typ * range
  | Type_var of typar * range
  | Type_anon of range
  | Type_with_global_constraints of typ * typeConstraintList list * range
  | Type_anon_constraint of typ * range

and  
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  synexpr =
  | Expr_paren of synexpr * range  (* parenthesized expressions kept in AST to distinguish A.M((x,y)) from A.M(x,y) *)
  | Expr_quote of synexpr * bool * synexpr * range (* bool indicates if it is a 'raw' quotation *)
  | Expr_const of sconst * range
  | Expr_typed of  synexpr * typ * range
  | Expr_tuple of  synexpr list * range
  | Expr_array_or_list of  bool * synexpr list * range 
  | Expr_recd of (typ * synexpr * range) option * synexpr option * (field * synexpr) list * range
  | Expr_new of bool * typ * synexpr * range (* bool true if known to be 'family' ('proected') scope *)
  | Expr_impl of typ * (synexpr * ident option) option * binding list * iimpl list * range
  | Expr_while of synexpr * synexpr * range
  | Expr_for of ident * synexpr * bool * synexpr * synexpr * range
  | Expr_foreach of synpat * synexpr * synexpr * range
  | Expr_list_of_seq of synexpr * range
  | Expr_array_of_seq of synexpr * range
  | Expr_comprehension of comprehension * range
  | Expr_lambda of  bool * simple_pats * synexpr * range (* bool indicates if lambda originates from a method. Patterns here are always "simple" *)
  | Expr_match of  synexpr * matching list * bool * range (* bool indicates if this is an exception match in a computation expression which throws unmatched exceptions *)
  | Expr_assert of synexpr * range
  | Expr_app of synexpr * synexpr * range
  | Expr_tyapp of synexpr * typ list * range
  | Expr_let of bool * bool * binding list * synexpr * range
  | Expr_try_catch of synexpr * range * matching list * range * range
  | Expr_try_finally of synexpr * synexpr * range
  | Expr_seq of bool * synexpr * synexpr * range (* false indicates "do a then b then return a" *)
  | Expr_cond of synexpr * synexpr * synexpr option * range
  | Expr_lid_get of bool * long_ident * range  (* bool true if preceded by a '?' for an optional named parameter *) 
  | Expr_lid_set of long_ident * synexpr * range
  | Expr_lid_indexed_set of long_ident * synexpr * synexpr * range
  | Expr_lvalue_get of synexpr * long_ident * range
  | Expr_lvalue_set of synexpr * long_ident * synexpr * range
  | Expr_lvalue_indexed_set of synexpr * long_ident * synexpr * synexpr * range
  | Expr_constr_field_get of synexpr * long_ident * int * range
  | Expr_constr_field_set of synexpr * long_ident * int * synexpr * range
  | Expr_asm of Il.instr array *  typ list * synexpr list * typ list * range (* Embedded IL assembly code *)
  | Expr_static_optimization of staticOptimizationConstraint list * synexpr * synexpr * range
  | Expr_isinst of  synexpr * typ * range
  | Expr_upcast of  synexpr * typ * range
  | Expr_addrof of  bool * synexpr * range
  | Expr_downcast of  synexpr * typ * range
  | Expr_arb_upcast of  synexpr * range
  | Expr_arb_downcast of  synexpr * range
  | Expr_null of range
  | Expr_ifnull of synexpr * synexpr * range
  | Expr_trait_call of typar list * classMemberSpfn * synexpr list * range
  | Expr_typeof of typ * range
  | Expr_lbrack_get of synexpr * synexpr * range
  | Expr_lbrack_set of synexpr * synexpr * synexpr * range
  | Expr_hole of (Obj.t option ref * synexpr option) * range

and iimpl = InterfaceImpl of typ * binding list * range

and matching = Clause of synpat * synexpr option *  synexpr * range

and comp_matching = CompClause of synpat * synexpr option *  comprehension * range

and attributes = attribute list

and attribute = 
  (* ident option are target specifiers, e.g. "assembly","module",etc. *)
  | Attr of long_ident * synexpr * ident option * range 

and memberInfo = (memberFlags * topValSynData * ident option)

and bindingKind = 
  | StandaloneExpression
  | NormalBinding
  | DoBinding
  
and binding = 
  | Binding of 
      access option *
      bindingKind *  
      (* mustinline: *) bool *  
      (* mutable: *) bool *  
      attributes * 
      xmlDoc *
      memberInfo option * 
      synpat * 
      bindingExpr *
      range 

and comprehension = 
  | Comp_zero
  | Comp_sum     of comprehension * comprehension * range * range
  | Comp_for     of bool * synpat * synexpr * comprehension
  | Comp_while   of synexpr * comprehension
  | Comp_try_finally of comprehension * synexpr
  | Comp_cond    of bool * synexpr * comprehension * comprehension 
  | Comp_yield   of (bool * bool) * synexpr
  | Comp_yieldm  of (bool * bool) * synexpr
  | Comp_bind    of bool * string option * synpat option * synexpr * comprehension
  | Comp_match of  synexpr * comp_matching list 
  | Comp_try_with of  comprehension * comp_matching list 
  
(* BindingExpr records the r.h.s. of a binding after some munging in the parser. *)
(* NOTE: This is a bit of a mess.  In the early implementation of F# we decided *)
(* to have the parser convert "let f x = e" into *)
(* "let f = fun x -> e".  This is called "pushing" a pattern across to the right hand side. Complex *)
(* patterns (e.g. non-tuple patterns) result in a computation on the right. *)
(* However, this approach really isn't that great - especially since *)
(* the language is now considerably more complex, e.g. we use *)
(* type information from the first (but not the second) form in *)
(* type inference for recursive bindings, and the first form *)
(* may specify .NET attributes for arguments. There are still many *)
(* relics of this approach around, e.g. the expression in BindingExpr *)
(* below is of the second form. However, to extract relevant information *)
(* we keep a record of the pats and optional explicit return type already pushed *)
(* into expression so we can use any user-given type information from these *)
and bindingExpr = 
  BindingExpr of 
      simple_pats list * 
      (typ *range * attributes) option * 
      synexpr 

and memberFlags =
  { memFlagsOverloadQualifier: Bytes.bytes option; 
    memFlagsInstance: bool;
    memFlagsVirtual: bool;
    memFlagsAbstract: bool;
    memFlagsOverride: bool;
    memFlagsFinal: bool;
    memFlagsKind: memberKind }

(* Note the member kind is actually computed partially by a syntax tree transformation "norm_pat" in tc.ml *)
and memberKind = 
  | MemberKindClassConstructor
  | MemberKindConstructor
  | MemberKindMember 
  | MemberKindPropertyGet 
  | MemberKindPropertySet    
  | MemberKindPropertyGetSet    

and signature =
  | Sign_named of long_ident 
  | Sign_explicit of moduleSpecDecls


and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  moduleImplDecl =
  | Def_module_abbrev of ident * long_ident * range
  | Def_module of componentInfo * moduleImplDecls * signature option * range
  | Def_let of bool * binding list * range (* first flag recursion, second flag must-inline *)
  | Def_tycon of tyconDefn list * range
(*
  | Def_named_compsig of componentInfo * moduleSpecDecls * range
  | Def_partial_inline_compsig of moduleSpecDecl * range
*)
  | Def_partial_tycon of componentInfo * classDefn * range
  | Def_exn of exconDefn * range
  | Def_open of module_path * range
  | Def_attributes of attributes * range
  | Def_hash of interaction * range
and exconCore = ExconCore of attributes * funionConstrDecl * long_ident option * xmlDoc * access option * range
and exconDefn = ExconDefn of exconCore * classDefn * range

and classKind = 
  | TyconUnspecified 
  | TyconClass 
  | TyconInterface 
  | TyconStruct 
  | TyconDelegate of typ * topValSynData 

and tyconDefnRepr =
  | TyconDefnRepr_class  of classKind * classDefn * range
  | TyconDefnRepr_simple of tyconSpfnOrDefnSimpleRepr * range

and tyconDefn =
  | TyconDefn of componentInfo * tyconDefnRepr * classDefn * range

and classDefn = classMemberDefn list

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  classMemberDefn = 
  | ClassMemberDefn_open of module_path * range
  | ClassMemberDefn_member_binding of binding * range                          
  (* implicit ctor args as a defn line, 'as' specification *)       
  | ClassMemberDefn_implicit_ctor of simple_pat list * ident option * range    
  (* inherit <typ>(args...) as base *)
  | ClassMemberDefn_implicit_inherit of typ * synexpr * ident option * range   
  (* localDefns *)
  | ClassMemberDefn_let_bindings of binding list * (* static: *) bool * (* recursive: *) bool * range                    
  | ClassMemberDefn_slotsig of valSpfn * memberFlags * range 
  | ClassMemberDefn_interface of typ * classDefn option  * range
  | ClassMemberDefn_inherit of typ  * ident option * range
  | ClassMemberDefn_field of fieldDecl  * range
  | ClassMemberDefn_tycon of tyconDefn * access option * range
      
and exconSpfn = 
  | ExconSpfn of exconCore * classSpfn * range

and classSpfn = classMemberSpfn list

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  classMemberSpfn = 
  | ClassMemberSpfn_binding of valSpfn  * memberFlags * range 
  | ClassMemberSpfn_interface of typ  * range
  | ClassMemberSpfn_inherit of typ * range
  | ClassMemberSpfn_field of fieldDecl  * range
  | ClassMemberSpfn_tycon  of tyconSpfn * range

and valSpfn = 
  | ValSpfn of 
      attributes * 
      value_name * 
      valTyparDecls * 
      typ * 
      topValSynData * 
      bool * 
      (* mutable: *) bool * 
      xmlDoc * 
      access option *
      synexpr option *
      range 

and topValSynData = TopValSynData of (*args:*) topArgSynData list list * (*return:*) topArgSynData 
and topArgSynData = TopArgSynData of attributes * (*optional:*) bool *  ident option

and valTyparDecls = ValTyparDecls of typarDecl list * bool * typeConstraintList list

and tyconSpfnRepr =
  | TyconSpfnRepr_class of classKind * classSpfn * range
  | TyconSpfnRepr_simple of tyconSpfnOrDefnSimpleRepr * range 

and componentInfo = 
  | ComponentInfo of attributes * componentKind * typarDecl list * typeConstraintList list * long_ident * xmlDoc * (* preferPostfix: *) bool * access option * range

and componentKind = 
  | TMK_Namespace 
  | TMK_Module 
  | TMK_Tycon 
  | TMK_Constraint

and tyconSpfn =
  | TyconSpfn of componentInfo * tyconSpfnRepr * classSpfn * range

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  moduleSpecDecl =
  | Spec_module_abbrev of ident * long_ident * range
  | Spec_module   of componentInfo * moduleSpecDecls * range
  | Spec_val      of valSpfn * range
  | Spec_tycon    of tyconSpfn list * range
  | Spec_exn      of exconSpfn * range
  | Spec_open     of module_path * range
  | Spec_hash     of interaction * range

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  tyconSpfnOrDefnSimpleRepr =
  | TyconCore_funion of funionConstrDecls * range
  | TyconCore_enum of enumConstrDecls * range
  | TyconCore_recd of fieldDecls * range
  | TyconCore_general of classKind * (typ * range * ident option) list * (valSpfn * memberFlags) list * fieldDecls * bool * bool * range 
  | TyconCore_asm of Il.typ * range
  | TyconCore_abbrev of typ * range
  | TyconCore_repr_hidden of range

and funionConstrDecls = funionConstrDecl list

and enumConstrDecls = enumConstrDecl list

and fieldDecls = fieldDecl list

and fieldDecl = 
  | Field of attributes * (* static: *) bool * field_name option * typ * bool * xmlDoc * access option * range

and funionConstrDecl = 
  | UnionConstr of attributes * constr_name * unionConstrTypeDecl * xmlDoc * access option * range

and enumConstrDecl =
  | EnumConstr of attributes * constr_name * sconst * xmlDoc * range

and unionConstrTypeDecl = 
  (* Normal ML-style declaration *)
  | ConstrFields of fieldDecl list      
  (* Full type spec given by 'UnionConstr : ty1 * tyN -> rty' *)
  | ConstrFullType of (typ * topValSynData) 

and typar = 
  | Typar of ident * typarStaticReq * (* compgen: *) bool 

and typarDecl = 
  | TyparDecl of attributes * typar 

and typarStaticReq = 
  | NoStaticReq 
  | HeadTypeStaticReq 
  (* Now obsolete, was for .NET 1.x *)
  | CompleteStaticReq

and staticOptimizationConstraint =
  | WhenTyparTyconEqualsTycon of typar *  typ * range

and 
  (*F#
  [<StructuralEquality(false); StructuralComparison(false)>]
  F#*)
  typeConstraintList =
  | WhereTyparIsValueType of typar * range
  (* | WhereTyparSupportsDefaultConstructor of typar * range *)
  | WhereTyparIsReferenceType of typar * range
  | WhereTyparSupportsNull of typar * range
  | WhereTyparDefaultsToType of typar * typ * range
  | WhereTyparEqualsType of typar *  typ * range
  | WhereTyparSubtypeOfType of typar *  typ * range
  | WhereTyparSupportsMember of typar list * classMemberSpfn * range
  | WhereTyparIsEnum of typar * typ list * range
  | WhereTyparIsDelegate of typar * typ list * range

and moduleSpecDecls = moduleSpecDecl list
and moduleImplDecls = moduleImplDecl list

(* qualifiedNameOfFile act to fully-qualify module specifications and implementations, *)
(* most importantly the ones that simply contribute fragments to a namespace (i.e. the AnonNamespaceFragmentSpec case) *)
(* There may be multiple such fragments in a single assembly, a major difference between traditional *)
(* ML and F#.  There may thus also be multiple matching pairs of these in an assembly, all contributing types to the same *)
(* namespace. These are matched up by the filename-rule. *)
and qualifiedNameOfFile = 
  | QualifiedNameOfFile of ident 

and moduleImpl = 
  | ModuleImpl of module_path * (*isModule:*) bool * moduleImplDecls * xmlDoc * attributes * access option * range 

and moduleSpec = 
  | ModuleSpec of module_path * (*isModule:*) bool * moduleSpecDecls * xmlDoc * attributes * access option * range 

and intf = 
  | AnonTopModuleSpec of moduleSpecDecls * range
  | NamedTopModuleSpec of moduleSpec
  | AnonNamespaceFragmentSpec of module_path * bool * moduleSpecDecls * xmlDoc * attributes * range

and impl = 
  | AnonTopModuleImpl of moduleImplDecls * range
  | NamedTopModuleImpl of moduleImpl
  | AnonNamespaceFragmentImpl of module_path * bool * moduleImplDecls * xmlDoc * attributes * range

and interaction =
  | IDefns of moduleImplDecl list * range
  | IHash  of string * string list * range

(*---------------------------------------------------------------------
 * AST and parsing utilities.
 *----------------------------------------------------------------------*)

type path = string list 
let ident (s,r) = {idText = s; idRange = r}
let text_of_id id = id.idText
let path_of_lid lid = List.map text_of_id lid
let text_of_path path = String.concat "." path
let text_of_lid lid = text_of_path (path_of_lid lid)

let text_of_qualNameOfFile (QualifiedNameOfFile(t)) = t.idText
let id_of_qualNameOfFile (QualifiedNameOfFile(t)) = t
let range_of_qualNameOfFile (QualifiedNameOfFile(t)) = t.idRange


type implFile = ImplFile of string * qualifiedNameOfFile * moduleImpl list
type sigFile = SigFile of string * qualifiedNameOfFile * moduleSpec list

type input = 
  | ImplFileInput of implFile
  | SigFileInput of sigFile

let range_of_input = function 
  | ImplFileInput (ImplFile(_,_,(ModuleImpl(_,_,_,_,_,_,m) :: _)))
  | SigFileInput (SigFile(_,_,(ModuleSpec(_,_,_,_,_,_,m) :: _))) -> m
  | _ -> failwith "range_of_input"

(*---------------------------------------------------------------------
 * Construct syntactic AST nodes
 *----------------------------------------------------------------------*)

let mksyn_id m s = ident(s,m)
let path_to_lid m p = List.map (mksyn_id m) p
let text_to_id0 n = mksyn_id range0 n

(* REVIEW: get rid of this name generator, which is used for the type inference *)
(* variables implicit in the #C syntax *)
let mksyn_new_uniq = let i = ref 0 in fun () -> incr i; !i
let mksyn_item m n = Expr_lid_get(false,[mksyn_id m n],m)

(* REVIEW: get rid of this state *)
let new_arg_uniq_ref = ref 0 
let mksyn_new_arg_uniq () = incr new_arg_uniq_ref; !new_arg_uniq_ref
let mksyn_spat_var isOpt id = SPat_as (id,false,isOpt,id.idRange)

let range_of_synpat p = 
  match p with 
  | Pat_const(_,m) | Pat_wild m | Pat_as (_,_,_,_,m) | Pat_disj (_,_,m) | Pat_conjs (_,m) 
  | Pat_lid (_,_,_,_,m) | Pat_array_or_list(_,_,m) | Pat_tuple (_,m) |Pat_typed(_,_,m) |Pat_attrib(_,_,m) 
  | Pat_recd (_,m) | Pat_range (_,_,m) | Pat_null m | Pat_isinst (_,m) | Pat_expr (_,m)
  | Pat_instance_member(_,_,_,m) | Pat_opt_var(_,m) | Pat_paren(_,m) -> m 

let range_of_syntype ty = 
  match ty with 
  | Type_app(_,_,m) | Type_proj_then_app(_,_,_,m) | Type_tuple(_,m) | Type_arr(_,_,m) | Type_fun(_,_,m)
  | Type_forall(_,_,m) | Type_var(_,m) | Type_anon m | Type_with_global_constraints(_,_,m)
  | Type_anon_constraint(_,m) -> m

let range_of_synconst c dflt = 
  match c with 
  | Const_string (c,m0) | Const_bytearray (c,m0) -> m0
  | _ -> dflt
  
let range_of_synexpr = function
  | Expr_paren(_,m) 
  | Expr_quote(_,_,_,m) 
  | Expr_const(_,m) 
  | Expr_typed (_,_,m)
  | Expr_tuple (_,m)
  | Expr_array_or_list (_,_,m)
  | Expr_recd (_,_,_,m)
  | Expr_new (_,_,_,m)
  | Expr_impl (_,_,_,_,m)
  | Expr_while (_,_,m)
  | Expr_for (_,_,_,_,_,m)
  | Expr_foreach (_,_,_,m)
  | Expr_comprehension (_,m)
  | Expr_list_of_seq (_,m)
  | Expr_array_of_seq (_,m)
  | Expr_lambda (_,_,_,m)
  | Expr_match (_,_,_,m)
  | Expr_assert (_,m)
  | Expr_app (_,_,m)
  | Expr_tyapp (_,_,m)
  | Expr_let (_,_,_,_,m)
  | Expr_try_catch (_,_,_,_,m)
  | Expr_try_finally (_,_,m)
  | Expr_seq (_,_,_,m)
  | Expr_cond (_,_,_,m)
  | Expr_lid_get (_,_,m)
  | Expr_lid_set (_,_,m)
  | Expr_lid_indexed_set (_,_,_,m)
  | Expr_lbrack_get (_,_,m)
  | Expr_lbrack_set (_,_,_,m)
  | Expr_lvalue_get (_,_,m)
  | Expr_lvalue_set (_,_,_,m)
  | Expr_lvalue_indexed_set (_,_,_,_,m)
  | Expr_constr_field_get (_,_,_,m)
  | Expr_constr_field_set (_,_,_,_,m)
  | Expr_asm (_,_,_,_,m)
  | Expr_static_optimization (_,_,_,m)
  | Expr_isinst (_,_,m)
  | Expr_upcast (_,_,m)
  | Expr_addrof (_,_,m)
  | Expr_downcast (_,_,m)
  | Expr_arb_upcast (_,m)
  | Expr_arb_downcast (_,m)
  | Expr_null m
  | Expr_trait_call(_,_,_,m)
  | Expr_typeof(_,m)
  | Expr_hole(_,m)
  | Expr_ifnull (_,_,m) -> m
let range_of_syndecl d = 
  match d with 
  | Def_module_abbrev(_,_,m) 
  | Def_module(_,_,_,m)
  | Def_let(_,_,m) 
  | Def_tycon(_,m)
(*  | Def_named_compsig(_,_,m) *)
  | Def_partial_tycon(_,_,m) 
  | Def_exn(_,m)
  | Def_open (_,m)
  | Def_hash (_,m)
(*  | Def_partial_inline_compsig(_,m) *)
  | Def_attributes(_,m) -> m

let anon_field_of_typ ty = Field([],false,None,ty,false,emptyXMLDoc,None,range_of_syntype ty)

let mksyn_pat_var vis id = Pat_as (Pat_wild id.idRange,id,false,vis,id.idRange)
let mksyn_this_pat_var id = Pat_as (Pat_wild id.idRange,id,true,None,id.idRange)
let mksyn_pat_maybe_var lid vis m =  Pat_lid (lid,None,[],vis,m) 

let new_arg_name() = 
  ("_arg"^string_of_int (mksyn_new_arg_uniq())) 

let mksyn_new_arg_var m  =
  let nm = new_arg_name() in
  let id = mksyn_id m nm in 
  mksyn_pat_var None id,mksyn_item m nm

(* Push non-simple parts of a patten match over onto the r.h.s. of a lambda *)
let rec spat_of_pat p =
  match p with 
  | Pat_typed(p',ty,m) -> 
      let p2,laterf = spat_of_pat p' in 
      SPat_typed(p2,ty,m), 
      laterf
  | Pat_attrib(p',attribs,m) -> 
      let p2,laterf = spat_of_pat p' in 
      SPat_attrib(p2,attribs,m), 
      laterf
  | Pat_as (Pat_wild _, v,thisv,_,m) -> 
      SPat_as (v,thisv,false,m), 
      (fun e -> e)
  | Pat_opt_var (v,m) -> 
      SPat_as (v,false,true,m), 
      (fun e -> e)
  | Pat_paren (p,m) -> spat_of_pat p 
  | _ -> 
      let m = range_of_synpat p in 
      (* 'nm' may be a real variable. Maintain its name. *)
      let nm = (match p with Pat_lid([id],None,[],None,_) -> id.idText | _ -> new_arg_name()) in
      let id = mksyn_id m nm in 
      let item = mksyn_item m nm in 
      mksyn_spat_var false id, 
      (fun e -> Expr_match(item,[Clause(p,None,e,m)],false,m)) 

let rec spats_of_pat p =
  match p with 
  | Pat_typed(p',ty,m) -> 
      let p2,laterf = spats_of_pat p' in 
      SPats_typed(p2,ty,m), 
      laterf
  | Pat_paren (p,m) -> spats_of_pat p 
  | Pat_tuple (ps,m) -> 
      let ps2,laterf = 
        List.fold_right 
          (fun (p',rhsf) (ps',rhsf') -> 
            p'::ps', 
            (fun x -> rhsf (rhsf' x)))
          (List.map spat_of_pat ps) 
          ([], (fun x -> x)) in 
      SPats (ps2,m),
      laterf
  | Pat_const (Const_unit,m) -> 
      SPats ([],m),
      (fun e -> e)
  | _ -> 
      let m = range_of_synpat p in 
      let sp,laterf = spat_of_pat p in 
      SPats ([sp],m),laterf

let push_one_pat isMember pat rhs =
  let nowpats,laterf = spats_of_pat pat in
  nowpats, Expr_lambda (isMember,nowpats, laterf rhs,range_of_synexpr rhs)
  
(* "fun (UnionConstr x) (UnionConstr y) -> body" ==> "fun tmp1 tmp2 -> let (UnionConstr x) = tmp1 in let (UnionConstr y) = tmp2 in body" *)
let push_many_pats isMember pats rhs =
  let nowf,spatsl,rhs2 = 
    List.fold_right 
      (fun arg (nowf,spatsl,body) -> 
        let nowpats,laterf = spats_of_pat arg in 
        (fun e -> Expr_lambda (isMember,nowpats, nowf e,range_of_synexpr rhs)),
        nowpats::spatsl,
        laterf body)
      pats
      ((fun e -> e),[],rhs) in 
  spatsl,nowf rhs2

let new_unit_uniq_ref = ref 0
let new_unit_uniq () = incr new_unit_uniq_ref; !new_unit_uniq_ref

(*----------------------------------------------------------------------
 * Inline IL
 *----------------------------------------------------------------------*)

(* Two helpers for parsing the inline IL fragments. *)
let parse_il_instrs s m = 
  try Ilpars.top_instrs Illex.token (Lexing.from_string (Bytes.unicode_bytes_as_string s))
  with Parsing.Parse_error -> 
    errorR(Error("error while parsing embedded IL",m)); [| |]

let parse_il_typ s m = 
  try Ilpars.top_typ Illex.token (Lexing.from_string (Bytes.unicode_bytes_as_string s))
  with Parsing.Parse_error -> 
    errorR(Error("error while parsing embedded IL type",m)); Il.ecma_mscorlib_refs.Il.typ_Object

(*----------------------------------------------------------------------
 * Count holes to help typecheck quotations (needed to 
 * propagate type information outside in)
 *----------------------------------------------------------------------*)

let rec holes_of_synexpr e acc =  
 match e with 
  | Expr_null _
  | Expr_typeof _
  | Expr_const _
  | Expr_lid_get _
  | Expr_quote _ -> acc
  | Expr_lid_set (_,e,_)
  | Expr_paren(e,_)
  | Expr_typed (e,_,_)
  | Expr_new (_,_,e,_)
  | Expr_lambda(_,_,e,_)
  | Expr_lvalue_get (e,_,_)
  | Expr_constr_field_get (e,_,_,_)
  | Expr_isinst (e,_,_)
  | Expr_upcast (e,_,_)
  | Expr_addrof (_,e,_)
  | Expr_downcast (e,_,_)
  | Expr_arb_upcast (e,_)
  | Expr_tyapp (e,_,_)
  | Expr_assert (e,_)
  | Expr_arb_downcast (e,_) -> holes_of_synexpr e acc
  | Expr_array_or_list (_,earr,_) -> list_fold_right holes_of_synexpr earr acc
  | Expr_app (e1,e2,_)
  | Expr_ifnull (e1,e2,_)
  | Expr_while (e1,e2,_)
  | Expr_lvalue_set (e1,_,e2,_)
  | Expr_lbrack_get(e1,e2,_)
  | Expr_try_finally (e1,e2,_)
  | Expr_lid_indexed_set(_,e1,e2,_)
  | Expr_constr_field_set (e1,_,_,e2,_)
  | Expr_static_optimization (_,e1,e2,_)
  | Expr_foreach(_,e1,e2,_)
  | Expr_seq (_,e1,e2,_) -> holes_of_synexpr e1 (holes_of_synexpr e2 acc)
  | Expr_comprehension(c,_) -> holes_of_comprehension c acc
  | Expr_list_of_seq(c,_) -> holes_of_synexpr c acc
  | Expr_array_of_seq(c,_) -> holes_of_synexpr c acc
  | Expr_lbrack_set(e1,e2,e3,_)
  | Expr_lvalue_indexed_set (e1,_,e2,e3,_)
  | Expr_for (_,e1,_,e2,e3,_)
      -> holes_of_synexpr e1 (holes_of_synexpr e2 (holes_of_synexpr e3 acc))
  | Expr_cond (e1,e2,e3,_) 
      -> holes_of_synexpr e1 (holes_of_synexpr e2 (Option.fold_right holes_of_synexpr e3 acc))
  
  | Expr_recd (a,b,c,_) -> 
     acc |> Option.fold_right (fun (_,e,_) acc-> holes_of_synexpr e acc) a 
         |> Option.fold_right holes_of_synexpr b 
         |> list_fold_right (snd >> holes_of_synexpr) c
  | Expr_impl (_,a,b,c,_) ->
     acc 
     |> Option.fold_right (fst >> holes_of_synexpr) a 
     |> list_fold_right holes_of_bind  b
     |> list_fold_right (fun (InterfaceImpl(_,binds,_)) ->list_fold_right holes_of_bind  binds) c
  | Expr_let (_,_,binds,e,m) -> list_fold_right holes_of_bind binds (holes_of_synexpr e acc)
  | Expr_match (e,ms,_,_)
  | Expr_try_catch (e,_,ms,_,_) -> acc |> holes_of_synexpr e |> holes_of_matchings ms
  | Expr_tuple (es,_)
  | Expr_asm (_,_,es,_,_)
  | Expr_trait_call (_,_,es,_) 
      -> list_fold_right holes_of_synexpr es acc
  | Expr_hole (filler,_) -> filler::acc

and holes_of_comprehension c acc = 
    match c with 
    | Comp_zero -> acc
    | Comp_sum   (e1,e2,_,_)  -> acc |> holes_of_comprehension e1 |> holes_of_comprehension e2 
    | Comp_match  (e,ms)  -> acc |> holes_of_synexpr e |> holes_of_comp_matchings ms
    | Comp_bind   (_,_,_,e,c) 
    | Comp_while   (e,c) 
    | Comp_for   (_,_,e,c)  -> acc |> holes_of_synexpr e |> holes_of_comprehension c  
    | Comp_cond  (_,e,c1,c2) -> acc |> holes_of_synexpr e |> holes_of_comprehension c1  |> holes_of_comprehension c2
    | Comp_try_finally (c1,e2) -> acc |> holes_of_comprehension c1  |> holes_of_synexpr e2
    | Comp_try_with (c1,ms) -> acc |> holes_of_comprehension c1  |> holes_of_comp_matchings ms
    | Comp_yield (_,e)
    | Comp_yieldm (_,e)     -> acc |> holes_of_synexpr e
and holes_of_matchings ms acc = acc |> list_fold_right holes_of_matching ms
and holes_of_matching (Clause(_,a,b,_)) acc = acc |> Option.fold_right holes_of_synexpr a |> holes_of_synexpr b
and holes_of_comp_matchings ms acc = acc |> list_fold_right holes_of_comp_matching ms
and holes_of_comp_matching (CompClause(_,a,b,_)) acc = acc |> Option.fold_right holes_of_synexpr a |> holes_of_comprehension b

and holes_of_bind b acc = 
  match b with 
  | Binding (_,_,_,_,_,_,_,_,BindingExpr (_,_,e),_) ->  holes_of_synexpr e acc 

(*----------------------------------------------------------------------
 * Operator name compilation
 *----------------------------------------------------------------------*)

let lbrack_get = ".[]"
let lbrack_get2 = ".[,]"
let lbrack_get3 = ".[,,]"
let lbrack_set = ".[]<-"
let lbrack_set2 = ".[,]<-"
let lbrack_set3 = ".[,,]<-"
let lparen_get = ".()"
let lparen_get2 = ".(,)"
let lparen_get3 = ".(,,)"
let lparen_set = ".()<-"
let lparen_set2 = ".(,)<-"
let lparen_set3 = ".(,,)<-"
let lbrack_slice_get = ".[..]"
let lbrack_slice2_get = ".[..,..]"
let lbrack_slice_set = ".[..]<-"
let lbrack_slice2_set = ".[..,..]<-"

let opNameTable = 
 [ ("[]", "op_Nil");
   ("::", "op_ColonColon");
   ("+", "op_Addition");
   ("++", "op_Increment");
   ("--", "op_Decrement");
   ("-", "op_Subtraction");
   ("*", "op_Multiply");
   ("**", "op_Exponentiation");
   ("/", "op_Division");
   ("@", "op_Append");
   ("^", "op_Concatenate");
   ("%", "op_Modulus");
   ("&&&", "op_BitwiseAnd");
   ("|||", "op_BitwiseOr");
   ("^^^", "op_ExclusiveOr");
   ("<<<", "op_LeftShift");
   ("~~~", "op_LogicalNot");
   (">>>", "op_RightShift");
   ("~+", "op_UnaryPlus");
   ("~-", "op_UnaryNegation");
   ("<=", "op_LessThanOrEqual");
   (">=", "op_GreaterThanOrEqual");
   ("<", "op_LessThan");
   (">", "op_GreaterThan");
   ("|>", "op_PipeRight");
   ("<|", "op_PipeLeft");
   ("!", "op_Dereference");
   (">>", "op_ComposeRight");
   ("<<", "op_ComposeLeft");
   ("<< >>", "op_Chevrons");
   ("<<| |>>", "op_ChevronsBar");
   ("<@ @>", "op_Quotation");
   ("<@@ @@>", "op_QuotationUntyped");
   ("+=", "op_AdditionAssignment");
   ("-=", "op_SubtractionAssignment");
   ("*=", "op_MultiplyAssignment");
   ("/=", "op_DivisionAssignment");
   ("..", "op_Range");
   (".. ..", "op_RangeStep"); 
   (lbrack_get, "op_IndexedLookup");
   (lbrack_slice_get, "GetSlice");
   (lbrack_slice2_get, "GetSlice2D");
   (lbrack_slice_set, "SetSlice");
   (lbrack_get2, "op_IndexedLookup2");
   (lbrack_get3, "op_IndexedLookup3");
   (lbrack_set, "op_IndexedAssign");
   (lbrack_set2, "op_IndexedAssign2");
   (lbrack_set3, "op_IndexedAssign3");
   (lparen_get, "op_ArrayLookup");
   (lparen_get2, "op_ArrayLookup2");
   (lparen_get3, "op_ArrayLookup3");
   (lparen_set, "op_ArrayAssign");
   (lparen_set2, "op_ArrayAssign2");
   (lparen_set3, "op_ArrayAssign3");
   ]

let opCharTranslateTable =
  [ ( '>', "Greater");
    ( '<', "Less"); 
    ( '+', "Plus");
    ( '-', "Minus");
    ( '*', "Multiply");
    ( '=', "Equals");
    ( '~', "Twiddle");
    ( '%', "Percent");
    ( '.', "Dot");
    ( '$', "Dollar");
    ( '&', "Amp");
    ( '|', "Bar");
    ( '@', "At");
    ( '#', "Hash");
    ( '^', "Hat");
    ( '!', "Bang");
    ( '?', "Qmark");
    ( '/', "Divide");
    ( '.', "Dot");
    ( ':', "Colon");
    ( '(', "LParen");
    ( ',', "Comma");
    ( ')', "RParen");
    ( ' ', "Space");
    ( '[', "LBrack");
    ( ']', "RBrack"); ]

let isOpName = 
  let t = Zset.addL  (List.map fst opCharTranslateTable) (Zset.empty compare) in 
  fun n -> 
    let r = ref false in 
    for i = 0 to String.length n - 1 do
      if Zset.mem n.[i] t then r := true;
    done;
    !r

let compileOpName =
  let t = Zmap.of_list (Zmap.empty compare) opNameTable in 
  let t2 = Zmap.of_list (Zmap.empty compare) opCharTranslateTable in 
  fun n -> match Zmap.tryfind n t with Some(x) -> x | None -> 
    if isOpName n then 
      let r = ref [] in 
      for i = 0 to String.length n - 1 do
        let c = n.[i] in 
        let c2 = match Zmap.tryfind c t2 with Some(x) -> x | None -> String.make 1 c in
        r := c2 :: !r 
      done;
      "op_"^(String.concat "" (List.rev !r))
    else n
                         
let decompileOpName = 
  let t = Zmap.of_list (Zmap.empty compare) (List.map (fun (x,y) -> (y,x)) opNameTable) in 
  let t2 = Zmap.of_list (Zmap.empty compare) (List.map (fun (x,y) -> (y,x)) opCharTranslateTable) in 
  fun n -> match Zmap.tryfind n t with Some(x) -> x | None -> 
    if String.length n >= 3 && String.sub n 0 3 = "op_" then 
      let rec loop remaining = 
        let l = String.length remaining in 
        if l = 0 then Some(remaining) else
        let choice = 
          choose
            (fun (a,b) -> 
              let bl = String.length b in 
              if bl <= l && String.sub remaining 0 bl = b then 
                Some(String.make 1 a, String.sub remaining bl (l - bl)) 
              else None) 
            opCharTranslateTable in 
        match choice with 
        | Some (a,remaining2) -> 
            begin match loop remaining2 with 
            | None -> None
            | Some a2 -> Some(a^a2)
            end
        | None -> None (* giveup *) in 
      match loop (String.sub n 3 (String.length n - 3)) with
      | Some res -> res
      | None -> n
    else n

let opname_Cons = compileOpName "::"
let opname_Nil = compileOpName "[]"
let opname_Equals = compileOpName "="

(*----------------------------------------------------------------------
 * AST constructors
 *----------------------------------------------------------------------*)

let lbrack_set_opname  = (compileOpName lbrack_set) 
let lbrack_set2_opname  = (compileOpName lbrack_set2) 
let lbrack_set3_opname  = (compileOpName lbrack_set3) 
let lparen_set_opname  = (compileOpName lparen_set) 
let lparen_set2_opname = (compileOpName lparen_set2) 
let lparen_set3_opname = (compileOpName lparen_set3) 
let lbrack_get_opname  = (compileOpName lbrack_get) 
let lbrack_get2_opname  = (compileOpName lbrack_get2) 
let lbrack_get3_opname  = (compileOpName lbrack_get3) 
let lparen_get_opname  = (compileOpName lparen_get) 
let lparen_get2_opname = (compileOpName lparen_get2) 
let lparen_get3_opname = (compileOpName lparen_get3) 
let mksyn_lid_get m path n = Expr_lid_get(false,path_to_lid m path @ [mksyn_id m n],m)
let mksyn_mod_item m modul n = mksyn_lid_get m [modul] n
let mk_oper opm oper = mksyn_item opm (compileOpName oper)
let mksyn_infix opm m l oper r = Expr_app (Expr_app (mk_oper opm oper,l,m), r,m)
let mksyn_bifix m oper l r = Expr_app (Expr_app (mk_oper m oper,l,m), r,m)
let mksyn_trifix m oper  x1 x2 x3 = Expr_app (Expr_app (Expr_app (mk_oper m oper,x1,m), x2,m), x3,m)
let mksyn_quadfix m oper  x1 x2 x3 x4 = Expr_app (Expr_app (Expr_app (Expr_app (mk_oper m oper,x1,m), x2,m), x3,m),x4,m)
let mksyn_quinfix m oper  x1 x2 x3 x4 x5 = Expr_app (Expr_app (Expr_app (Expr_app (Expr_app (mk_oper m oper,x1,m), x2,m), x3,m),x4,m),x5,m)
let mksyn_prefix m oper x = Expr_app (mk_oper m oper, x,m)
let mksyn_constr m n = [mksyn_id m (compileOpName n)]

let mksyn_dot_lbrack_set  m a b c = mksyn_trifix m lbrack_set a b c
let mksyn_dot_lbrack_set2  m a b c = mksyn_trifix m lbrack_set2 a b c
let mksyn_dot_lbrack_set3  m a b c = mksyn_trifix m lbrack_set3 a b c
let mksyn_dot_lparen_set  m a b c = mksyn_trifix m lparen_set a b c
let mksyn_dot_lparen_set2  m a b c = mksyn_trifix m lparen_set2 a b c
let mksyn_dot_lparen_set3  m a b c = mksyn_trifix m lparen_set3 a b c
let mksyn_dot_lbrack_get  m a b   = Expr_lbrack_get(a,b,m)

let mksyn_dot_lbrack_slice_get  m arr (x,y) = 
    mksyn_trifix m lbrack_slice_get arr x y

let mksyn_dot_lbrack_slice2_get  m arr (x1,y1) (x2,y2) = 
    mksyn_quinfix m lbrack_slice2_get arr x1 y1 x2 y2

let mksyn_dot_lbrack_slice_set  m arr x y v = mksyn_quadfix m lbrack_slice_set arr x y v
   
let mksyn_dot_lparen_get  m a b   = 
  match b with
  | Expr_tuple ([_;_],_)   -> mksyn_infix m m a lparen_get2 b
  | Expr_tuple ([_;_;_],_) -> mksyn_infix m m a lparen_get3 b
  | _ -> mksyn_infix m m a lparen_get b
let mksyn_unit m = Expr_const(Const_unit,m)
let mksyn_unit_pat m = Pat_const(Const_unit,m)
let mksyn_delay m e = Expr_lambda (false,SPats ([mksyn_spat_var false (mksyn_id m ("_delay"^string_of_int(new_unit_uniq())))],m), e, m)

let rec mksyn_assign m l r = 
    match l with 
    | Expr_paren(l2,m2)  -> mksyn_assign m l2 r
    | Expr_lid_get(false,v,m)  -> Expr_lid_set (v,r,m)
    | Expr_lvalue_get(e,v,m)  -> Expr_lvalue_set (e,v,r,m)
    | Expr_lbrack_get(e1,e2,m)  -> Expr_lbrack_set (e1,e2,r,m)
    | Expr_constr_field_get (x,y,z,_) -> Expr_constr_field_set (x,y,z,r,m) 
    | Expr_app (Expr_app(Expr_lid_get(false,[{ idText = nm; idRange=_}], _), a, _),b,_) when nm = lparen_get_opname -> 
        mksyn_dot_lparen_set m a b r
    | Expr_app (Expr_app(Expr_lid_get(false,[{ idText = nm; idRange=_}], _), a, _),b,_) when nm = lparen_get2_opname -> 
        mksyn_dot_lparen_set2 m a b r
    | Expr_app (Expr_app(Expr_lid_get(false,[{ idText = nm; idRange=_}], _), a, _),b,_) when nm = lparen_get3_opname -> 
        mksyn_dot_lparen_set3 m a b r
    | Expr_app (Expr_app(Expr_lid_get(false,[{ idText = nm; idRange=_}], _), a, _),b,_) when nm = lbrack_get_opname -> 
        mksyn_dot_lbrack_set m a b r
    | Expr_app (Expr_app(Expr_lid_get(false,[{ idText = nm; idRange=_}], _), a, _),b,_) when nm = lbrack_get2_opname -> 
        mksyn_dot_lbrack_set2 m a b r
    | Expr_app (Expr_app(Expr_lid_get(false,[{ idText = nm; idRange=_}], _), a, _),b,_) when nm = lbrack_get3_opname -> 
        mksyn_dot_lbrack_set3 m a b r
    | Expr_app (Expr_lid_get(false,v,_),x,m)  -> Expr_lid_indexed_set (v,x,r,m)
    | Expr_app (Expr_lvalue_get(e,v,_),x,m)  -> Expr_lvalue_indexed_set (e,v,x,r,m)
    | Expr_app (Expr_app(Expr_app(Expr_lid_get(false,[{ idText = nm; idRange=_}], _), a, _),b,_),c,_) when nm = compileOpName lbrack_slice_get -> 
        mksyn_dot_lbrack_slice_set m a b c r
    |   _ -> errorR(Error("invalid expression on left of assignment",m));  Expr_const(Const_unit,m)

let rec mksyn_dot m l r = 
  match l with 
  | Expr_paren(l2,m2)  -> mksyn_dot m l2 r
  | Expr_lid_get(isOpt,lid,_) -> Expr_lid_get(isOpt,lid@[r],m)
  | Expr_lvalue_get(e,lid,_) -> Expr_lvalue_get(e,lid@[r],m)
  | expr -> Expr_lvalue_get(expr,[r],m)

let rec mksyn_dotn m l r = 
  match l with 
  | Expr_paren(l2,m2)  -> mksyn_dotn m l2 r
  | Expr_app (Expr_app(Expr_lid_get(false,[{ idText = nm; idRange=_}], _), a, _),Expr_lid_get (false,cid,_),_) when nm = lparen_get_opname-> 
      Expr_constr_field_get (a,cid,Int32.to_int r,m)
  |   _ -> errorR(Error("array access or constructor field access expected",m));  Expr_const(Const_unit,m)
        
let mksyn_match_lambda isMember isExnMatch wholem mtch =
  let p,pe = mksyn_new_arg_var wholem in 
  let _,e = push_many_pats isMember [p] (Expr_match(pe,mtch,isExnMatch,wholem)) in
  e

let mksyn_match_lambdas isMember wholem ps e = 
  let _,e =  push_many_pats isMember ps e  in 
  e

let mksyn_cons x y =
  let xm = range_of_synexpr x in 
  Expr_app(Expr_lid_get(false,[mksyn_id xm opname_Cons],xm),Expr_tuple([x;y],xm),xm) 

let mksyn_list m l = 
  List.fold_right mksyn_cons l (Expr_lid_get(false,[mksyn_id m opname_Nil],m))

let mksyn_cons_pat x y =
  let xm = range_of_synpat x in 
  Pat_lid (mksyn_constr xm opname_Cons, None, [Pat_tuple ([x;y],xm)],None,xm)

let mksyn_list_pat m l =
  List.fold_right mksyn_cons_pat l (Pat_lid(mksyn_constr m opname_Nil, None, [], None,m))

(*----------------------------------------------------------------------
 * Arities of members
 * Members have strongly syntactically constrained arities.  We must infer
 * the arity from the syntax in order to have any chance of handling recursive 
 * cross references during type inference.
 *
 * So we record the arity for: 
   StaticProperty --> [1]               -- for unit arg
   this.StaticProperty --> [1;1]        -- for unit arg
   StaticMethod(args) --> map argdata_of_spat args
   this.InstanceMethod() --> 1 :: map argdata_of_spat args
   this.InstanceProperty with get(argpat) --> 1 :: [argdata_of_spat argpat]
   StaticProperty with get(argpat) --> [argdata_of_spat argpat]
   this.InstanceProperty with get() --> 1 :: [argdata_of_spat argpat]
   StaticProperty with get() --> [argdata_of_spat argpat]
   
   this.InstanceProperty with set(argpat)(v) --> 1 :: [argdata_of_spat argpat; 1]
   StaticProperty with set(argpat)(v) --> [argdata_of_spat argpat; 1]
   this.InstanceProperty with set(v) --> 1 :: [1]
   StaticProperty with set(v) --> [1] 
 *----------------------------------------------------------------------*)

module SynArgInfo = struct
  let unnamedTopArg1 = TopArgSynData([],false,None)
  let unnamedTopArg = [unnamedTopArg1]
  let unitArgData = unnamedTopArg
  let unnamedRetVal = TopArgSynData([],false,None)
  let selfMetadata = unnamedTopArg
  let emptyTopValData = TopValSynData([],unnamedRetVal)

  let hasNoArgs (TopValSynData(args,_)) = isNil args
  let hasOptionalArgs (TopValSynData(args,_)) = List.exists (List.exists (fun (TopArgSynData(_,isOptArg,_)) -> isOptArg)) args
  let incorporateEmptyTupledArg (TopValSynData(args,rmdata)) = TopValSynData([]::args,rmdata)
  let incorporateUnitArg (TopValSynData(args,rmdata)) = TopValSynData(unitArgData::args,rmdata)
  let incorporateSelfArg (TopValSynData(args,rmdata)) = TopValSynData(selfMetadata::args,rmdata)
  let incorporateSetterArg (TopValSynData(args,rmdata)) = TopValSynData(args@[unnamedTopArg],rmdata)
  let numCurriedArgs(TopValSynData(args,_)) = List.length args
  let getArgInfos (TopValSynData(args,_)) = args
  let aritiesOfArgs (TopValSynData(args,_)) = List.map List.length args
  let attribsOfArgData (TopArgSynData(attribs,_,_)) = attribs
  let isOptionalArg (TopArgSynData(_,isOpt,_)) = isOpt
  let rec argdata_of_spat attribs p = 
      match p with 
      | SPat_as(nm,_,isOpt,_) -> 
         (* if List.length attribs <> 0 then dprintf1 "List.length attribs = %d\n" (List.length attribs); *)
         TopArgSynData(attribs, isOpt, Some nm)
      | SPat_typed(a,_,_) -> argdata_of_spat attribs a
      | SPat_attrib(a,attribs2,_) -> argdata_of_spat (attribs @ attribs2) a
    
  let rec argdata_of_spats x = 
      match x with 
      | SPats(ps,_) -> List.map (argdata_of_spat []) ps
      | SPats_typed(ps,_,_) -> argdata_of_spats ps

  let argdata_of_pat p = 
      let sp,_ = spats_of_pat p in 
      argdata_of_spats sp

  let infer_sig_metadata bindm memberInfo pat tyopt = 
               (* non-indexer properties: record the arity *)
    match memberInfo with
    | None -> None
    | Some memInfo  -> 
        let arities_of_this_arg = 
            if memInfo.memFlagsInstance then [ selfMetadata ] else [] in 

        let arities_of_explicit_args = 
            match pat with 
            | Some(Pat_lid(_,_,args,_,m)) -> List.map argdata_of_pat args
            | _ -> [] in 
        (* Curried members (e.g. "this.f x1 x2 x3 = ...") are currently compiled and called as *)
        (* members returning functions, (e.g. "this.f x1 = (fun x2 x3 -> ...)") *)
        (* That is, before adding the 'this' pointer the arities of members are limited to [], [_], and also *)
        (* [_;_] in the indexed property setter case *)
        (* *)
        (* REVIEW: we have a design choice here.  Curried members could be given arity N (where N is the number of curried *)
        (* arguments).  This would allow us to compile them as non-curried, as with modules. Since curried members are in *)
        (* many ways the natural way to write code in F# there could be considerable advantages here. Of course signatures *)
        (* would have to reveal the "curriedness" of the member just as with module values. The consistence with modules *)
        (* is very appealing. *)
        let arities_of_explicit_args = 
            match arities_of_explicit_args with 
            | [] | [_] -> arities_of_explicit_args
            | [h1;h2] when memInfo.memFlagsKind=MemberKindPropertySet -> arities_of_explicit_args
            (* | args -> args in *)
            | h1 :: _ -> [h1] in
        
        let arity = arities_of_this_arg @ arities_of_explicit_args in 
        let rmdata = match tyopt with None -> unnamedRetVal | Some((_,rmdata),_) -> rmdata in
  (*      dprintf3 "%a: inferred arity [%s]\n" output_range bindm (String.concat ";" (List.map string_of_int arity));   *)
        Some(memInfo,TopValSynData(arity,rmdata),None)
end


let mksyn_binding_rhs tyconds rhsexpr rhsm tyopt =
    let rhs_after_tyconds = List.fold_right (fun (c,e1) e2 -> Expr_static_optimization (c,e1,e2,rhsm)) tyconds rhsexpr in 
    let rhs_after_typ,rtyopt = 
      match tyopt with 
      | Some ((ty,TopArgSynData(rattribs,_,_)),tym) -> Expr_typed(rhs_after_tyconds,ty,tym), Some(ty,tym,rattribs) 
      | None -> rhs_after_tyconds,None  in
    rhs_after_typ,rtyopt

let mksyn_binding (xmlDoc,headPat) vis pseudo mut bindm wholem tyopt rhsexpr rhsm tyconds attrs memberInfo =
    let e,rtyopt = mksyn_binding_rhs tyconds rhsexpr rhsm tyopt in 
    let info = SynArgInfo.infer_sig_metadata bindm memberInfo (Some headPat) tyopt in 
    Binding (vis,NormalBinding,pseudo,mut,attrs,xmlDoc,info,headPat,BindingExpr([],rtyopt,e),bindm) 

let nonVirtualFlags q k = { memFlagsKind=k; memFlagsOverloadQualifier=q;  memFlagsInstance=true;  memFlagsVirtual=false; memFlagsAbstract=false; memFlagsOverride=false; memFlagsFinal=false }
let ctorMemFlags q =      { memFlagsOverloadQualifier=q;memFlagsKind=MemberKindConstructor; memFlagsInstance=false; memFlagsVirtual=false; memFlagsAbstract=false; memFlagsOverride=false;  memFlagsFinal=false }
let cctorMemFlags =      { memFlagsOverloadQualifier=None;memFlagsKind=MemberKindClassConstructor; memFlagsInstance=false; memFlagsVirtual=false; memFlagsAbstract=false; memFlagsOverride=false;  memFlagsFinal=false }
let virtFlags q k =       { memFlagsKind=k; memFlagsOverloadQualifier=q;  memFlagsInstance=true;  memFlagsVirtual=true;  memFlagsAbstract=false; memFlagsOverride=false;  memFlagsFinal=false }
let overrideFlags q k =   { memFlagsKind=k; memFlagsOverloadQualifier=q;  memFlagsInstance=true;  memFlagsVirtual=false;  memFlagsAbstract=false; memFlagsOverride=true;  memFlagsFinal=false }
let abstractFlags q k =   { memFlagsKind=k; memFlagsOverloadQualifier=q;  memFlagsInstance=true;  memFlagsVirtual=false;  memFlagsAbstract=true;  memFlagsOverride=false;  memFlagsFinal=false }
let staticFlags q k = { memFlagsKind=k; memFlagsOverloadQualifier=q;  memFlagsInstance=false; memFlagsVirtual=false; memFlagsAbstract=false; memFlagsOverride=false;  memFlagsFinal=false }

let inferredTyparDecls = ValTyparDecls([],true,[])
let noInferredTypars = ValTyparDecls([],false,[])

(*----------------------------------------------------------------------
 * The parser defines a number of tokens for whitespace and
 * comments eliminated by the lexer.  These carry a specification of
 * a continuation for the lexer when used in scenarios where we don't
 * care about whitespace.
 *----------------------------------------------------------------------*)

type lexcont = 
    | AT_token
    | AT_ifdef_skip of int * range
    | AT_string of range
    | AT_vstring of range
    | AT_comment of int * range
    | AT_comment_string of int * range
    | AT_camlonly of range

(*----------------------------------------------------------------------
 * XML doc pre-processing
 *----------------------------------------------------------------------*)

(* chop leading spaces (well, this isn't very efficient, is it?) *) 
let rec trimSpaces str = if has_prefix str " " then trimSpaces (drop_prefix str " ") else str
   
let rec processLines lines =
    match lines with 
    | [] -> []
    | (lineA::rest) as lines ->
        let lineAT = trimSpaces lineA in
        if lineAT = "" then processLines rest
        else if has_prefix lineAT "<" then lines
        else ["<summary>"] @ lines @ ["</summary>"] 

let processXMLDoc (XMLDoc lines) = XMLDoc (processLines lines)


(*----------------------------------------------------------------------
 * Parser/Lexer state
 *----------------------------------------------------------------------*)

let currentLexbuf = ref None  (* global state *)

type concreteSyntaxSink = 
    { matchPair: (range -> range -> unit);
      startName: (range -> unit);
      qualifyName: (range -> range -> unit);
      startParameters: pos -> unit; 
      nextParameter: pos -> unit; 
      endParameters: pos -> unit;  }

let concreteSyntaxSinkRef = ref None

(*----------------------------------------------------------------------
 * Helpers for lexer/parser
 *----------------------------------------------------------------------*)

let pos_of_lexpos p = 
  mk_pos (p.Lexing.pos_lnum) (p.Lexing.pos_cnum - p.Lexing.pos_bol)

let mksyn_range p1 p2 = 
  mk_file_idx_range (decode_file_idx p1.Lexing.pos_fname) (pos_of_lexpos p1) (pos_of_lexpos p2)

let get_lex_range lexbuf = 
  mksyn_range (Lexing.lexeme_start_p lexbuf) (Lexing.lexeme_end_p lexbuf)

let curr_lex_range () = 
  match !currentLexbuf with 
  | Some lexbuf -> get_lex_range lexbuf
  | None -> range0 

let lhs () = mksyn_range (Parsing.symbol_start_pos()) (Parsing.symbol_end_pos()) 
let rhspos n = pos_of_lexpos (Parsing.rhs_start_pos n)
let rhs2 n m = mksyn_range (Parsing.rhs_start_pos n) (Parsing.rhs_end_pos m)
let rhs n = rhs2 n n

(*----------------------------------------------------------------------
 * Feedback about concrete syntax from parsing
 *----------------------------------------------------------------------*)

let matchPair p1 p2 = 
  match !concreteSyntaxSinkRef with 
  | None -> ()
  | Some snk -> snk.matchPair (rhs p1) (rhs p2)

let startName m = 
  match !concreteSyntaxSinkRef with 
  | None -> ()
  | Some snk -> snk.startName m

let qualifyName m1 m2 = 
  match !concreteSyntaxSinkRef with 
  | None -> ()
  | Some snk -> snk.qualifyName m1 m2

(* "System.   Name"  counts as two long names, "System.Name" counts as one *)
(* This gives better intellisense *)
let qualifyNameIfAlongside m1 m2 = 
  if pos_geq (end_of_range m1) (start_of_range m2) then 
    qualifyName m1 m2 
  else
    (qualifyName m1 m1; startName m2)
 
let startParameters p = 
  match !concreteSyntaxSinkRef with 
  | None -> ()
  | Some snk -> snk.startParameters p

let nextParameter p = 
  match !concreteSyntaxSinkRef with 
  | None -> ()
  | Some snk -> snk.nextParameter p

let endParameters p = 
  match !concreteSyntaxSinkRef with 
  | None -> ()
  | Some snk -> snk.endParameters p

(*----------------------------------------------------------------------
 * XMLDoc F# lexer/parser state (global)...
 *----------------------------------------------------------------------*)

(* ATTENTION: GLOBAL STATE (defined away from usage site)
 * 
 * The F# parser is not re-entrant, for various reasons.
 * The lexer "saves" XMLDoc comments (prefixed by ///) as it sees them.
 * The parser "grabs" them at key points in it's production rules.
 * The "lines" store is defined here (ast.ml).
 * [ASIDE: actually, a natural place to define it is in pars.mly and to pick it up in the lex.mll header].
 *)      
let revXMLs = ref [] (* reverse collect XML lines *)
let saveXMLDoc line = revXMLs := line :: !revXMLs  (* called from lex.mll *)
let grabXML () = let lines = List.rev !revXMLs in revXMLs := []; XMLDoc lines
let dumpXMLDoc note (XMLDoc lines) = printf "\nXMLDoc: %s\n" note; List.iter (printf "  %s\n") lines; XMLDoc lines
let xmlDocMerge (XMLDoc lines) (XMLDoc lines') = XMLDoc (lines @ lines')

(*-------------------------------------------------------------------------
!* Name generators.  It is hard to generate names that are stable, nice
 * and unique (where stable means they tend not to change as you make modifications
 * to the source code,and nice means you use simple identifiers where possible,
 * and unique means they are unique across the a compilation unit).
 *
 * In general these should only be used in the backend.
 *------------------------------------------------------------------------- *)

type stableNiceNameGenerator = 
  { snngApply: string -> range option -> int -> string; 
    snngReset: unit -> unit }

let newStableNiceNameGenerator () =
  (* First: For a given basename, create unique names for each uniq.
   * State: unames = (basename,(uniq,basename^suffix) table,nextN)
   *
   * For efficiency:
   * - storing the final string so it is not built on each lookup.
   * - lookups for existing entries do not modify tables.
   * 
   * For basename, Given uniq.
   * If uniq is known, get it's uname.
   * Otherwise,
   *   generate the next uname, update (mutate) state, and have uname.
   *)
  let new_unames basename = (basename,Hashtbl.create 10,ref 0) in      (* NB: creates state *)
  let suffixOfN = function 0 -> "" | n -> "_" ^ string_of_int n in
  let unameForUnique (basename,unamesHT,nextR) uniq =                  (* NB: unamesHT, nextR may mutate *)
      if Hashtbl.mem unamesHT uniq then
          let uname = Hashtbl.find unamesHT uniq in
          uname
      else
          (* create fresh uname for (basename,uniq) *)
          let n = !nextR in
          let uname = basename ^ suffixOfN n in
          (* mutate state (which is associated with basename) *)
          nextR := n+1;
          Hashtbl.add unamesHT uniq uname;
          uname  in
  (* Second: Extend to (basename -> unames) table.
   * basenameHT - (basename,unames) table, NB: "unames" mutates. *)
  let basenameHT = Hashtbl.create 100 in
  let unameForBasenameUnqie basename uniq =
      let unames =
          if Hashtbl.mem basenameHT basename then
              Hashtbl.find basenameHT basename
          else
              let unames = new_unames basename in      (* NB: mutable *)
              Hashtbl.add basenameHT basename unames;  (* NB: log it against basename *)
              unames in
      unameForUnique unames uniq in
  let reset () = Hashtbl.clear basenameHT in
  (* Third: actually basenames come from combining (base,mopt : range option) *)
  let snngApply base mopt uniq =
      let basename =
          match mopt with
          | Some m -> base ^ "@"^ string_of_int (start_line_of_range m)
          | None   -> base in
      unameForBasenameUnqie basename uniq in
  { snngApply = snngApply;
    snngReset = reset}
   
type niceNameGenerator = { nngApply: string -> range -> string; nngReset: unit -> unit }

let newNiceNameGenerator () = 
  let new_id = let i = ref 0 in fun () -> incr i; !i in 
  let stableNiceNameGenerator =  newStableNiceNameGenerator() in 
  { nngApply = (fun base m -> stableNiceNameGenerator.snngApply base (Some m) (new_id()));
    nngReset = (fun () -> stableNiceNameGenerator.snngReset ()) }

let nng = newNiceNameGenerator()













(* REMOVE_FROM_RELEASE_END *) let report f = f 0                                       (* REMOVE_FROM_RELEASE_END *)


