(* (c) Microsoft Corporation 2005-2007.  *)

(*F# 
module Microsoft.Research.AbstractIL.BinaryWriter 
open Microsoft.Research.AbstractIL 
open Microsoft.Research.AbstractIL.Internal 
module Ilx = Microsoft.Research.AbstractIL.Extensions.ILX.Types  
module Ilsupp = Microsoft.Research.AbstractIL.Internal.Support 
module Ildiag = Microsoft.Research.AbstractIL.Diagnostics 
module Ilbinary = Microsoft.Research.AbstractIL.Internal.BinaryConstants 
module Ilprint = Microsoft.Research.AbstractIL.AsciiWriter 
module Il = Microsoft.Research.AbstractIL.IL 
module Illib = Microsoft.Research.AbstractIL.Internal.Library 
F#*)  

open Illib
open Ildiag
open Il
open Ilbinary
open Nums
open Ilsupp

module Bytebuf = Bytes.Bytebuf

(*---------------------------------------------------------------------
 * The big writer.
 *---------------------------------------------------------------------*)

let checking = false let _ = if checking then dprint_endline "warning : Ilwrite.checking is on"
let logging = false
let showTimes = ref false

(*---------------------------------------------------------------------
 * Library
 *---------------------------------------------------------------------*)

let (@@) = Array.append  

let reportTime =
  let tFirst = ref None     in
  let tPrev = ref None     in
  fun descr ->
    if !showTimes then begin 
      let t = Sys.time() in
      let prev = match !tPrev with None -> 0.0 | Some t -> t in 
      let first = match !tFirst with None -> (tFirst := Some t; t) | Some t -> t in 
      dprintf3 "ilwrite: TIME %10.3f (total)   %10.3f (delta) - %s\n" (t -. first) (t -. prev) descr;
      tPrev := Some t
    end

(*---------------------------------------------------------------------
 * Byte, byte array fragments and other concrete representations
 * manipulations.
 *---------------------------------------------------------------------*)

(* Little-endian encoding of int32 *)
let b0 n = ??? (n &&& 0xFFl)
let b1 n = ??? ((n lsr 8) &&& 0xFFl)
let b2 n = ??? ((n lsr 16) &&& 0xFFl)
let b3 n = ??? ((n lsr 24) &&& 0xFFl)

(* Little-endian encoding of int64 *)
let dw7 n = Int64.to_int (Int64.logand (Int64.shift_right_logical n 56) 0xFFL)
let dw6 n = Int64.to_int (Int64.logand (Int64.shift_right_logical n 48) 0xFFL)
let dw5 n = Int64.to_int (Int64.logand (Int64.shift_right_logical n 40) 0xFFL)
let dw4 n = Int64.to_int (Int64.logand (Int64.shift_right_logical n 32) 0xFFL)
let dw3 n = Int64.to_int (Int64.logand (Int64.shift_right_logical n 24) 0xFFL)
let dw2 n = Int64.to_int (Int64.logand (Int64.shift_right_logical n 16) 0xFFL)
let dw1 n = Int64.to_int (Int64.logand (Int64.shift_right_logical n 8) 0xFFL)
let dw0 n = Int64.to_int (Int64.logand n 0xFFL)

let dW1 n = Int64.to_int32 (Int64.logand (Int64.shift_right_logical n 32) 0xFFFFFFFFL)
let dW0 n = Int64.to_int32 (Int64.logand n 0xFFFFFFFFL)

(* @todo: get rid of all of these in favour of writing directly to buffers *)
let u8_as_intarray i = [| b0 !!!(u8_to_int i) |]
let u16_as_intarray x =  let n = !!!(u16_to_int x) in [| b0 n; b1 n |]
let i32_as_intarray i = [| b0 i; b1 i; b2 i; b3 i |]
let i64_as_intarray i = [| dw0 i; dw1 i; dw2 i; dw3 i; dw4 i; dw5 i; dw6 i; dw7 i |]

let i8_as_intarray i = u8_as_intarray (i8_to_u8 i)
let i16_as_intarray i = u16_as_intarray (i16_to_u16 i)
let u32_as_intarray i = i32_as_intarray (u32_to_i32 i)
let u64_as_intarray i = i64_as_intarray (u64_to_i64 i)

let ieee32_as_intarray i = i32_as_intarray (ieee32_to_bits i)
let ieee64_as_intarray i = i64_as_intarray (ieee64_to_bits i)

(* -------------------------------------------------------------------- 
 * Expanding byte buffers
 * -------------------------------------------------------------------- *)

let emit_i32_as_u16 buf x = Bytebuf.emit_i32_as_u16 buf x
let bytebuf_emit_u16 buf x = Bytebuf.emit_i32_as_u16 buf (u16_to_i32 x)
let bytebuf_emit_i64 buf x = 
  Bytebuf.emit_i32 buf (dW0 x);
  Bytebuf.emit_i32 buf (dW1 x)

(*---------------------------------------------------------------------
 * Alignment and padding
 *---------------------------------------------------------------------*)

let align alignment n = ((n +++ alignment --- !!!1) /./ alignment) *** alignment

(*---------------------------------------------------------------------
 * Concrete token representations etc. used in PE files
 *---------------------------------------------------------------------*)

let z_u32_size n = 
  if n <= 0x7Fl then !!!1
  else if n <= 0x3FFFl then !!!2
  else !!!4

let z_u32 n = 
  if n >= 0l &&  n <= 0x7Fl then [| ??? n |] 
  else if n >= 0x80l && n <= 0x3FFFl then [| ??? (0x80l ||| (n lsr 8)); ??? (n &&& 0xFFl) |] 
  else [| ??? (0xc0l ||| (n lsr 24)); ??? ((n lsr 16) &&& 0xFFl); ??? ((n lsr 8) &&& 0xFFl); ??? (n &&& 0xFFl) |] 

let bytebuf_emit_pad buf n = Bytebuf.emit_intarray_as_bytes buf (Array.create n 0x0)
let bytebuf_emit_z_u32 buf x = Bytebuf.emit_intarray_as_bytes buf (z_u32 x)

let bytebuf_emit_z_untagged_index buf  big idx = 
  if big then Bytebuf.emit_i32 buf idx
  else if idx > 0xffffl then failwith "z_untagged_index: too big for small address or simple index"
  else Bytebuf.emit_i32_as_u16 buf idx

let bytebuf_emit_z_tagged_index buf tag nbits big idx =
  let idx2 = (!!!idx <<< nbits) ||| tag in
  if big then Bytebuf.emit_i32 buf idx2
  else Bytebuf.emit_i32_as_u16 buf idx2

let uncoded_token tab idx = ((!!!(tag_of_table tab) <<< 24) ||| !!!idx)

(* From ECMA for UserStrings:
This final byte holds the value 1 if and only if any UTF16 character within the string has any bit set in its top byte, or its low byte is any of the following:
0x010x08, 0x0E0x1F, 0x27, 0x2D,
0x7F. Otherwise, it holds 0. The 1 signifies Unicode characters that require handling beyond that normally provided for 8-bit encoding sets.
*)

let marker_for_unicode_bytes (b:Bytes.bytes) = 
  let len = Bytes.length b in
  let rec scan i = 
    i < len/2 && 
    if 
     (let b1 = Bytes.get b (i*2) in
      let b2 = Bytes.get b (i*2+1) in
      (b2 <> 0)
      or (b1 >= 0x01 && b1 <= 0x08) 
      or (b1 >= 0xE && b1 <= 0x1F)
      or (b1 = 0x27)
      or (b1 = 0x2D)) then true
    else scan (i+1) in 
  let marker = if scan 0 then 0x01 else 0x00 in
  marker


(* -------------------------------------------------------------------- 
 * Fixups
 * -------------------------------------------------------------------- *)

let check_fixup32 data offset exp = 
  if Bytes.get data (offset + 3) <> b3 exp then failwith "fixup sanity check failed";
  if Bytes.get data (offset + 2) <> b2 exp then failwith "fixup sanity check failed";
  if Bytes.get data (offset + 1) <> b1 exp then failwith "fixup sanity check failed";
  if Bytes.get data (offset) <> b0 exp then failwith "fixup sanity check failed"

let fixup32 data offset v = 
  Bytes.set data (offset)   (b0 v);
  Bytes.set data (offset+1) (b1 v);
  Bytes.set data (offset+2) (b2 v);
  Bytes.set data (offset+3) (b3 v)

(* -------------------------------------------------------------------- 
 * PDB data
 * -------------------------------------------------------------------- *)

type pdb_document = Il.source_document
(* type pdb_namespace = string  (* todo: do we need more here? *)*)
type pdb_var = 
    { pdbVarName: string;
      pdbVarSig: Bytes.bytes; 
      pdbVarAttributes: Nums.i32 (* this is essentially the local index the name corresponds to *) }

type pdb_method_scope = 
    { pdbScopeChildren: pdb_method_scope array;
      pdbScopeStartOffset: int;
      pdbScopeEndOffset: int;
      pdbScopeLocals: pdb_var array;
      (* @todo open_namespaces: pdb_namespace array; *) }

type pdb_source_loc = 
    { pdbLocDocument: int;
      pdbLocLine: int;
      pdbLocColumn: int; }
      
type pdb_sequence_point = 
    { pdbSeqPointDocument: int;
      pdbSeqPointOffset: int;
      pdbSeqPointLine: int;
      pdbSeqPointColumn: int;
      pdbSeqPointEndLine: int;
      pdbSeqPointEndColumn: int; }

type pdb_method_data = 
    { pdbMethToken: int32;
      pdbMethParams: pdb_var array;
      pdbMethRootScope: pdb_method_scope;
      pdbMethRange: (pdb_source_loc * pdb_source_loc) option;
      pdbMethSequencePoints: pdb_sequence_point array; }

let compare_seqpoints_by_source sp1 sp2 = 
  let c1 = compare sp1.pdbSeqPointDocument sp2.pdbSeqPointDocument in 
  if c1 <> 0 then c1 else 
  let c1 = compare sp1.pdbSeqPointLine sp2.pdbSeqPointLine in 
  if c1 <> 0 then c1 else 
  compare sp1.pdbSeqPointColumn sp2.pdbSeqPointColumn 
    
let compare_seqpoints_by_offset sp1 sp2 = 
  compare sp1.pdbSeqPointOffset sp2.pdbSeqPointOffset 

let sizeof_IMAGE_DEBUG_DIRECTORY = 28 (* 28 is the size of the IMAGE_DEBUG_DIRECTORY in ntimage.h *)

type pdb_data = 
    { pdbEntrypoint: int32 option;
      pdbDocuments: pdb_document array;
      pdbMethods: pdb_method_data array }

(*---------------------------------------------------------------------
 * PDB Writer.  The function [write_pdb_info] abstracts the 
 * impeative calls to the Symbol Writer API.
 *---------------------------------------------------------------------*)

let write_pdb_info desiredMetadataVersionOpt fixupOverlappingSequencePoints f fpdb info = 
  let versionHint = 
    match desiredMetadataVersionOpt with 
    | None -> 0
    | Some v -> 
        if (Il.version_compare v (Il.parse_version "1.0.3705") = 0) then 1
        else if (Il.version_compare v (Il.parse_version "1.1.4322") = 0) then 2
        else 0 in
  (try Sys.remove(fpdb) with _ -> ());
  let pdbw = pdbInitialize f fpdb versionHint in 

  begin match info.pdbEntrypoint with None -> () | Some x -> pdbSetUserEntryPoint pdbw x end;
  let docs = Array.map (fun doc -> pdbDefineDocument pdbw doc.sourceFile) info.pdbDocuments in 
  let get_doc i = 
    if i < 0 or i >= Array.length docs then failwith "get_doc: bad doc number";
    docs.(i) in 
  reportTime (Printf.sprintf "PDB: Defined %d documents" (Array.length info.pdbDocuments));
  Array.sort (fun x y -> compare x.pdbMethToken y.pdbMethToken) info.pdbMethods;
  reportTime (Printf.sprintf "PDB: Sorted %d methods" (Array.length info.pdbMethods));

        (* This next bit is a bit of a hack.  The sequence points we get *)
        (* from F# (which has nothing to do with this module) are actually expression *)
        (* marks, i.e. the source ranges they denote are typically *)
        (* nested, and each point indicates where the  *)
        (* code for an expression with a particular range begins.  *)
        (* This is in many ways a much more convenient form to emit. *)
        (* However, it is not the form that debug tools accept nicely. *)
        (* However, sequence points are really a non-overlapping, non-nested *)
        (* partition of the source code of a method.  So here we shorten the *)
        (* length of all sequence point marks so they do not go further than *)
        (* the next sequence point in the source. *)
  let spCounts = Array.map (fun x -> x.pdbMethSequencePoints |> Array.length) info.pdbMethods in 
  let allSps = Array.concat (Array.map (fun x -> x.pdbMethSequencePoints) info.pdbMethods |> Array.to_list) in
  let allSps = Array.mapi (fun i sp -> (i,sp)) allSps in 
  if fixupOverlappingSequencePoints then begin 
    (* sort the sequence points into source order *)
    Array.sort (fun (_,sp1) (_,sp2) -> compare_seqpoints_by_source sp1 sp2) allSps;
    (* shorten the ranges of any that overlap with following sequence points *)
    (* sort the sequence points back into offset order *)
    for i = 0 to Array.length allSps - 2 do
      let n,sp1 = allSps.(i) in 
      let _,sp2 = allSps.(i+1) in 
      if (sp1.pdbSeqPointDocument = sp2.pdbSeqPointDocument) && 
         (sp1.pdbSeqPointEndLine > sp2.pdbSeqPointLine or 
         (sp1.pdbSeqPointEndLine = sp2.pdbSeqPointLine &
          sp1.pdbSeqPointEndColumn >= sp2.pdbSeqPointColumn)) then
        let adjustToPrevLine = (sp1.pdbSeqPointLine < sp2.pdbSeqPointLine) in
        allSps.(i) <-  n,{sp1 with pdbSeqPointEndLine = (if adjustToPrevLine then sp2.pdbSeqPointLine-1 else sp2.pdbSeqPointLine);
                                   pdbSeqPointEndColumn = (if adjustToPrevLine then 80 else sp2.pdbSeqPointColumn); }
    done;
    Array.sort (fun (n1,_) (n2,_) -> compare n1 n2) allSps;
  end;


  
  let spOffset = ref 0 in 
  Array.iteri
    (fun i minfo ->
      begin 
        pdbOpenMethod pdbw minfo.pdbMethToken;

        let sps = Array.sub allSps !spOffset spCounts.(i) in
        spOffset := !spOffset + spCounts.(i);
        (* Partition the sequence points by document *)
        let spsets =
          let res = (Zmap.empty compare : (int,pdb_sequence_point list ref) Zmap.map) in
          let add res (_,sp) = 
            let k = sp.pdbSeqPointDocument in
            match Zmap.tryfind k res with
                Some xsR -> xsR := sp :: !xsR; res
              | None     -> Zmap.add k (ref [sp]) res
          in
          let res = Array.fold_left add res sps in
          let res = Zmap.to_list res in  (* ordering may bnot be stable *)
          List.map (fun (_,x) -> Array.of_list !x) res in

        List.iter
          (fun spset -> 
            if Array.length spset > 0 then begin
              Array.sort compare_seqpoints_by_offset spset;
              let sps = 
                Array.map (fun sp -> 
                     (* Ildiag.dprintf2 "token 0x%08lx has an sp at offset 0x%08x\n" minfo.pdbMethToken sp.pdbSeqPointOffset; *)
                     (sp.pdbSeqPointOffset, sp.pdbSeqPointLine, sp.pdbSeqPointColumn,sp.pdbSeqPointEndLine, sp.pdbSeqPointEndColumn)) spset in 
            (* Use of alloca in implementation of pdbDefineSequencePoints can give stack overflow here *)
            if Array.length sps < 10000 then 
              pdbDefineSequencePoints pdbw (get_doc spset.(0).pdbSeqPointDocument) sps;
            end) 
          spsets;

        begin match minfo.pdbMethRange with None -> () | Some (a,b) ->
          pdbSetMethodRange pdbw 
            (get_doc a.pdbLocDocument) a.pdbLocLine a.pdbLocColumn
            (get_doc b.pdbLocDocument) b.pdbLocLine b.pdbLocColumn
        end;

        (* Write the scopes *)
        let rec write_scope top sco = 
          if top or Array.length sco.pdbScopeLocals <> 0 or Array.length sco.pdbScopeChildren <> 0 then begin 
            pdbOpenScope pdbw sco.pdbScopeStartOffset;
            Array.iter (fun v -> pdbDefineLocalVariable pdbw v.pdbVarName v.pdbVarSig v.pdbVarAttributes) sco.pdbScopeLocals;
            Array.iter (write_scope false) sco.pdbScopeChildren;
            pdbCloseScope pdbw sco.pdbScopeEndOffset;
          end in
        write_scope true minfo.pdbMethRootScope; 

        pdbCloseMethod pdbw
      end)
    info.pdbMethods;
  reportTime "PDB: Wrote methods";
  let res = pdbGetDebugInfo pdbw in 
  pdbClose pdbw;
  reportTime "PDB: Closed";
  res

(*---------------------------------------------------------------------
 * Strong name signing
 *---------------------------------------------------------------------*)

type signer =  
  | PublicKeySigner of Ilsupp.pubkey
  | KeyPair of keyPair
  | KeyContainer of keyContainerName

let signerOpenPublicKeyFile s = 
  let pubkey = Ilsupp.signerOpenPublicKeyFile s in
  PublicKeySigner(pubkey)
  
let signerOpenPublicKey pubkey = 
  PublicKeySigner(pubkey)

let signerOpenKeyPairFile s = 
  let keypair = Ilsupp.signerOpenKeyPairFile s in
  KeyPair(keypair)

let signerOpenKeyContainer s = 
  KeyContainer(s)

let signerClose s = 
  match s with 
  | PublicKeySigner _
  | KeyPair _ -> ()
  | KeyContainer containerName -> Ilsupp.signerCloseKeyContainer(containerName)
  
let signerFullySigned s =
  match s with 
  | PublicKeySigner _ -> false
  | KeyPair _ | KeyContainer _ -> true

let signerPublicKey s = 
  match s with 
  | PublicKeySigner p -> p
  | KeyPair kp -> Ilsupp.signerGetPublicKeyForKeyPair kp
  | KeyContainer kn -> Ilsupp.signerGetPublicKeyForKeyContainer kn

let signerSignatureSize s = 
  try Ilsupp.signerSignatureSize(signerPublicKey s)
  with e -> 
    dprint_endline ("Warning: A call to StrongNameSignatureSize failed ("^Printexc.to_string e^")");
    0x80 

let signerSignFile file s = 
  match s with 
  | PublicKeySigner p -> ()
  | KeyPair kp -> Ilsupp.signerSignFileWithKeyPair file kp
  | KeyContainer kn -> Ilsupp.signerSignFileWithKeyContainer file kn


(*---------------------------------------------------------------------
 * TYPES FOR TABLES
 *---------------------------------------------------------------------*)

type row_element = 
  | UShort of Nums.u16
  | ULong of int32
  | Data of int * bool (* Index into cenv.data or cenv.resources.  Will be adjusted later in writing once we fix an overall location for the data section.  flag indicates if offset is relative to cenv.resources. *)
  | Guid of int (* pos. in guid array *)
  | Blob of int (* pos. in blob array *)
  | String of int (* pos. in string array *)
  | SimpleIndex of table * int (* pos. in some table *)
  | TypeDefOrRefOrSpec of typeDefOrRef_tag * int
  | TypeOrMethodDef of typeOrMethodDef_tag * int
  | HasConstant of hasConstant_tag * int
  | HasCustomAttribute of hasCustomAttribute_tag * int
  | HasFieldMarshal of hasFieldMarshal_tag * int
  | HasDeclSecurity of hasDeclSecurity_tag * int
  | MemberRefParent of memberRefParent_tag * int
  | HasSemantics of hasSemantics_tag * int
  | MethodDefOrRef of methodDefOrRef_tag * int
  | MemberForwarded of  memberForwarded_tag * int
  | Implementation of implementation_tag * int
  | CustomAttributeType of customAttributeType_tag * int
  | ResolutionScope of resolutionScope_tag * int

type row = Row of row_element array

(*=====================================================================
 *=====================================================================
 * IL --> TABLES+CODE
 *=====================================================================
 *=====================================================================*)

(* This environment keeps track of how many generic parameters are in scope. *)
(* This lets us translate AbsIL type variable number to IL type variable numbering *)
type env = 
    { envClassFormals: int }
let env_enter_tdef gparams ={envClassFormals=gparams }
let env_enter_msig tgparams mgparams =
  { envClassFormals=tgparams }
let env_enter_fspec fspec =
  { envClassFormals=List.length (inst_of_typ (enclosing_typ_of_fspec fspec)) }
let env_enter_ospec ospec =
  { envClassFormals=List.length (inst_of_typ (enclosing_typ_of_ospec ospec)) }

let env_enter_mref mref =
  { envClassFormals=max_int }

let mk_env = { envClassFormals=0 }

(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type 'a tbl = 
    { tblname: string;
      tbl: ('a, int) Hashtbl.t;
      mutable lookups: int;
      mutable rows: 'a list;
      mutable count: int }

(* inline to allow comparison function to be specialized for type of key *)
let (*F# inline F#*) new_tbl nm = 
  { tblname=nm;
    lookups=0;
    tbl = (Hashtbl.create 20);
    rows=[];
    count=0; }

let get_tbl tbl = 
  if !showTimes then dprintf3 "--> table %s had %d entries and %d lookups\n" tbl.tblname tbl.count tbl.lookups;
  List.rev tbl.rows

let add_entry tbl x =
  tbl.count <- tbl.count + 1;
  let n = tbl.count in 
  Hashtbl.add tbl.tbl x n;
  tbl.rows <- x :: tbl.rows;
  n

let find_or_add_entry tbl x =
  tbl.lookups <- tbl.lookups + 1; 
  if Hashtbl.mem tbl.tbl x then (Hashtbl.find tbl.tbl x)
  else add_entry tbl x

(* This is only used in one special place - see furthre below. *)
let set_rows_of_table tbl t = 
  tbl.rows <- List.rev t;  (* rows stored in reverse order *)
  let h = tbl.tbl in 
  Hashtbl.clear h;
  list_iteri (fun i x -> Hashtbl.add h x (i+1)) t

let add_uniq_entry nm geterr tbl x =
  if Hashtbl.mem tbl.tbl x then failwith ("duplicate entry '"^geterr x^"' in "^nm^" table")
  else add_entry tbl x

let tbl_find tbl x =
  if Hashtbl.mem tbl.tbl x then Hashtbl.find tbl.tbl x 
  else raise Not_found

type mdkey = MdKey of int (* type. def. idx. *) * int (* #generic formals *) * string * typ * typ list
type fdkey = FdKey of int (* type def idx. *) * string * typ
type propkey = PropKey of int (* type. def. idx. *) * string * typ * typ list
type eventkey = EventKey of int (* type. def. idx. *) * string
type tdkey = TdKey of string list * string

(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type cenv = 
    { mscorlib: scope_ref;
      ilg: mscorlib_refs;
      desiredMetadataVersion: Il.version_info;
      reqd_data_fixups: (int32 * (int * bool)) list ref;
      mutable reqd_string_fixups: (int32 * (int * int) list) list; (* References to strings in codestreams: offset of code and a (fixup-location , string token) list) *)
      code_chunks: Bytebuf.t; 
      mutable next_code_addr: int32;
      generate_pdb: bool;
      pdbinfo: pdb_method_data buf;
      pdbdocuments: pdb_document tbl;
      data: Bytebuf.t; (* Raw data, to go into the data section *)
      resources: Bytebuf.t; (* Raw resource data, to go into the data section *)
      mutable entrypoint: (bool * int) option; 

      (* The following are all used to generate unique items in the output *)
      tables: row tbl array;
      fieldDefs: fdkey tbl;
      methodDefs:  mdkey tbl;
      propertyDefs: propkey tbl;
      eventDefs: eventkey tbl;
      typeDefs: tdkey tbl; 
      guids: bytes tbl; 
      blobs: bytes tbl; 
      strings: string tbl; 
      userStrings: bytes tbl;

      (* These are possible memoization points *)
      tref_as_TypeRef_idx_memoized: type_ref -> int;
      typ_as_TypeSpec_idx_memoized: env * typ -> int;
      typ_as_tdor_memoized        : env * typ -> typeDefOrRef_tag * int;
      mrefinfo_as_mdor_memoized   : bool * env * (string * typ * callconv * typ list * typ * varargs * int) -> methodDefOrRef_tag * int;
      mspec_as_uncoded_memoized   : env * (method_spec * varargs) -> int32;
  }


let metadataSchemaVersionSupportedByCLRVersion v = 
    (* Whidbey Beta 1 version numbers are between 2.0.40520.0 and 2.0.40607.0 *)
    (* Later Whidbey versions are post 2.0.40607.0.. However we assume *)
    (* internal builds such as 2.0.x86chk are Whidbey Beta 2 or later *)
    if Il.version_compare v (parse_version ("2.0.40520.0")) >= 0 &&
       Il.version_compare  v (parse_version ("2.0.40608.0")) < 0 then 1,1
    else if Il.version_compare v (parse_version ("2.0.0.0")) >= 0 then 2,0
    else 1,0 

let headerVersionSupportedByCLRVersion v = 
   (* The COM20HEADER version number *)
   (* Whidbey version numbers are 2.5 *)
   (* Earlier are 2.0 *)
   (* From an email from jeffschw: "Be built with a compiler that marks the COM20HEADER with Major >=2 and Minor >= 5.  The V2.0 compilers produce images with 2.5, V1.x produces images with 2.0." *)
    if Il.version_compare v (parse_version ("2.0.0.0")) >= 0 then 2,5
    else 2,0 

let peOptionalHeaderByteByCLRVersion v = 
   (*  A flag in the PE file optional header seems to depend on CLI version *)
   (* Whidbey version numbers are 8 *)
   (* Earlier are 6 *)
   (* Tools are meant to ignore this, but the VS Profiler wants it to have the right value *)
    if Il.version_compare v (parse_version ("2.0.0.0")) >= 0 then 8
    else 6

(* returned by write_binary_internal *)
(*F#
[<StructuralEquality(false); StructuralComparison(false)>]
F#*)
type mappings =  
    { tdefMap: type_def list * type_def -> int32;
      fdefMap: type_def list * type_def -> field_def -> int32;
      mdefMap: type_def list * type_def -> method_def -> int32;
      propertyMap: type_def list * type_def -> property_def -> int32;
      eventMap: type_def list * type_def -> event_def -> int32 }

let record_reqd_data_fixup reqd_data_fixups buf pos lab =
  reqd_data_fixups :=  (pos,lab) :: !reqd_data_fixups;
  (* Write a special value in that we check later when applying the fixup *)
  Bytebuf.emit_i32 buf 0xdeadddddl

let table cenv (Table idx) = cenv.tables.(idx)

    
let add_code cenv ((reqd_string_fixups_offset,reqd_string_fixups),code) = 
  if align !!!4 cenv.next_code_addr <> cenv.next_code_addr then dprint_endline "warning: code not 4-byte aligned";
  cenv.reqd_string_fixups <- (cenv.next_code_addr +++ reqd_string_fixups_offset, reqd_string_fixups) :: cenv.reqd_string_fixups;
  Bytebuf.emit_bytes cenv.code_chunks code;
  cenv.next_code_addr <- cenv.next_code_addr +++ !!!(Bytes.length code)

let get_code cenv = Bytebuf.close cenv.code_chunks

let split_name_at nm idx = 
  let last = Pervasives.(-) (String.length nm) 1 in 
  (String.sub nm 0 idx),
  (if idx < last then String.sub nm (Pervasives.(+) idx 1) (Pervasives.(-) last idx) else "")

let split_namespace_aux nm = 
  if String.contains nm '.' then 
    let idx = String.rindex nm '.' in
    let s1,s2 = split_name_at nm idx in 
    Some s1,s2 
  else None, nm

let memoize_namespace_tab = Hashtbl.create 10
let split_namespace nm =
  if Hashtbl.mem memoize_namespace_tab nm then
    Hashtbl.find memoize_namespace_tab nm
  else
    let x = split_namespace_aux nm in
    (Hashtbl.add  memoize_namespace_tab nm x; x)

let string_as_utf8_intarray (s:string) = Bytes.to_intarray (Bytes.string_as_utf8_bytes s)
   
let bytes_as_UserString_idx cenv bytes = 
    find_or_add_entry cenv.userStrings bytes
let bytes_as_BlobHeap_idx cenv bytes = 
    if Bytes.length bytes = 0 then 0 
    else find_or_add_entry cenv.blobs bytes
(* todo: should really be ustring_as_StringHeap_idx - string should universally be unicode string in absil *)
let string_as_StringHeap_idx cenv s = 
    if s = "" then 0 
    else find_or_add_entry cenv.strings s

let guid_as_GuidHeap_idx cenv info = find_or_add_entry cenv.guids (Bytes.of_intarray info)
let intarray_as_BlobHeap_idx cenv blob = bytes_as_BlobHeap_idx cenv (Bytes.of_intarray blob)

let string_option_as_StringHeap_idx cenv sopt =
  match sopt with 
  | Some ns -> string_as_StringHeap_idx cenv ns
  | None -> 0

let name_as_elem_pair cenv n =
  let (n1,n2) = split_namespace n in 
  String (string_option_as_StringHeap_idx cenv n1),
  String (string_as_StringHeap_idx cenv n2)

(*=====================================================================
 * Pass 1 - allocate indexes for types 
 *=====================================================================*)

let name_of_tdkey (TdKey (enc,n)) = n

let rec tdef_pass1 enc cenv td = 
  ignore (add_uniq_entry "type index" name_of_tdkey cenv.typeDefs (TdKey (enc,name_of_tdef td)));
  tdefs_pass1 (enc@[name_of_tdef td]) cenv (dest_tdefs (nested_of_tdef td))
and tdefs_pass1 enc cenv tds = List.iter (tdef_pass1 enc cenv) tds


(*=====================================================================
 * Pass 2 - allocate indexes for methods and fields and write rows for types 
 *=====================================================================*)

let rec idx_alloced_for_tdef cenv (enc,n) = 
  try tbl_find cenv.typeDefs (TdKey (enc,n) )
  with Not_found -> failwith ("One of your modules expects the type '"^String.concat "." (enc@[n])^"' to be defined within the module being emitted.  Are you missing an object file?")
    
(* -------------------------------------------------------------------- 
 * Assembly and module references
 * -------------------------------------------------------------------- *)

let rec aref_as_AssemblyRef_row cenv aref =
  Row [| UShort (match aref.assemRefVersion with None -> u16_zero | Some (x,y,z,w) -> x);
        UShort (match aref.assemRefVersion with None -> u16_zero | Some (x,y,z,w) -> y);
        UShort (match aref.assemRefVersion with None -> u16_zero | Some (x,y,z,w) -> z);
        UShort (match aref.assemRefVersion with None -> u16_zero | Some (x,y,z,w) -> w);
        ULong ((match aref.assemRefPublicKeyInfo with Some (PublicKey _) -> 0x0001l | _ -> 0x0000l)
             ||| (if aref.assemRefRetargetable then 0x0100l else 0x0000l) 
               );  
        (match aref.assemRefPublicKeyInfo with None -> Blob 0 | Some (PublicKey b | PublicKeyToken b) -> Blob (bytes_as_BlobHeap_idx cenv b));
        String (string_as_StringHeap_idx cenv aref.assemRefName);
        (match aref.assemRefLocale with None -> String 0 | Some s -> String (string_as_StringHeap_idx cenv s));
        (match aref.assemRefHash with None -> Blob 0 | Some s -> Blob (bytes_as_BlobHeap_idx cenv s)); |]
  
and aref_as_AssemblyRef_idx cenv aref = 
  find_or_add_entry (table cenv tab_AssemblyRef) (aref_as_AssemblyRef_row cenv aref)

and modref_as_ModuleRef_row cenv mref =
  Row [| String (string_as_StringHeap_idx cenv mref.modulRefName) |]

and modref_as_File_row cenv mref =
  Row [|  ULong (if mref.modulRefNoMetadata then 0x0001l else 0x0000l);
        String (string_as_StringHeap_idx cenv mref.modulRefName);
        (match mref.modulRefHash with None -> Blob 0 | Some s -> Blob (bytes_as_BlobHeap_idx cenv s)); |]

and modref_as_ModuleRef_idx cenv mref = 
  find_or_add_entry (table cenv tab_ModuleRef) (modref_as_ModuleRef_row cenv mref)

and modref_as_File_idx cenv mref = 
  find_or_add_entry (table cenv tab_File) (modref_as_File_row cenv mref)

(* -------------------------------------------------------------------- 
 * Does a scope_ref point to this module?
 * -------------------------------------------------------------------- *)

let scoref_is_local scoref = (scoref = ScopeRef_local) 

(* -------------------------------------------------------------------- 
 * Scopes to Implementation elements.
 * -------------------------------------------------------------------- *)

let rec scoref_as_Implementation_elem cenv scoref = 
  match scoref with 
  | ScopeRef_local ->  (i_AssemblyRef, 0)
  | ScopeRef_assembly aref -> (i_AssemblyRef, aref_as_AssemblyRef_idx cenv aref)
  | ScopeRef_module mref -> (i_File, modref_as_File_idx cenv mref)
 
(* -------------------------------------------------------------------- 
 * Type references, types etc.
 * -------------------------------------------------------------------- *)

let rec tref_as_TypeRef_row cenv tref = 
  let nselem,nelem = name_as_elem_pair cenv (tname_of_tref tref) in 
  let rs1,rs2 = rscope_as_ResolutionScope_elem cenv (scoref_of_tref tref,enclosing_tnames_of_tref tref) in
  Row [| ResolutionScope (rs1,rs2); 
        nelem; 
        nselem |]

and tref_as_TypeRef_idx_unmemoized cenv tref = 
  find_or_add_entry (table cenv tab_TypeRef) (tref_as_TypeRef_row cenv tref)

and tref_as_TypeRef_idx cenv tref = cenv.tref_as_TypeRef_idx_memoized tref

and tdesc_as_TypeRef_idx cenv (scoref,enc,n) =  tref_as_TypeRef_idx cenv (mk_nested_tref (scoref,enc,n))

and rscope_as_ResolutionScope_elem cenv (scoref,enc) = 
  if isNil enc then 
    match scoref with 
    | ScopeRef_local -> (rs_Module, 1) 
    | ScopeRef_assembly aref -> (rs_AssemblyRef, aref_as_AssemblyRef_idx cenv aref)
    | ScopeRef_module mref -> (rs_ModuleRef, modref_as_ModuleRef_idx cenv mref)
  else
    let enc2,n2 = front_n_back enc in 
    (rs_TypeRef, tdesc_as_TypeRef_idx cenv (scoref,enc2,n2))
 

and tdesc_as_TypeDefOrRefEncoded cenv (scoref,enc,nm) = 
  if scoref_is_local scoref then 
    let idx = idx_alloced_for_tdef cenv (enc,nm) in 
    z_u32 (!!!idx <<< 2) (* ECMA 22.2.8 TypeDefOrRefEncoded - TypeDef *)
  else 
    let idx = tdesc_as_TypeRef_idx cenv (scoref,enc,nm) in 
    z_u32 ((!!!idx <<< 2) ||| 0x01l) (* ECMA 22.2.8 TypeDefOrRefEncoded - TypeRef *)

let  tref_is_local cenv tref = scoref_is_local (scoref_of_tref tref)

let typ_is_local cenv typ = 
  is_tref_typ typ && isNil (inst_of_typ typ) && tref_is_local cenv (tref_of_typ typ)


let tdor_as_uncoded (tag,idx) =
  let tab = 
    if tag = tdor_TypeDef then tab_TypeDef 
    else if tag = tdor_TypeRef then tab_TypeRef  
    else if tag = tdor_TypeSpec then tab_TypeSpec
    else failwith "tdor_as_uncoded" in 
  uncoded_token tab idx

let array_shape_as_intarray (ArrayShape shape) = 
  let sized = List.filter (function (_,Some _) -> true | _ -> false) shape in 
  let lobounded = List.filter (function (Some _,_) -> true | _ -> false) shape in 
  Array.concat 
    [ (* Rank *) z_u32 !!!(List.length shape);
      (* Num sizes *) z_u32 !!!(List.length sized);
      (* Sizes *) Array.concat (List.map (function (_,Some sz) -> z_u32 sz | _ -> failwith "?") sized);
      (* Num low bounds *) z_u32 !!!(List.length lobounded);
      (* Low bounds *) Array.concat (List.map (function (Some low,_) -> z_u32 low | _ -> failwith "?") lobounded) ]
        
let hasthis_as_byte hasthis =
   match hasthis with 
    CC_instance -> e_IMAGE_CEE_CS_CALLCONV_INSTANCE
  | CC_instance_explicit -> e_IMAGE_CEE_CS_CALLCONV_INSTANCE_EXPLICIT
  | CC_static -> 0x00

let callconv_as_byte ntypars (Callconv (hasthis,bcc)) = 
  hasthis_as_byte hasthis lor
  (if ntypars > 0 then e_IMAGE_CEE_CS_CALLCONV_GENERIC else 0x00) lor
  (match bcc with 
  | CC_fastcall -> e_IMAGE_CEE_CS_CALLCONV_FASTCALL
  | CC_stdcall -> e_IMAGE_CEE_CS_CALLCONV_STDCALL
  | CC_thiscall -> e_IMAGE_CEE_CS_CALLCONV_THISCALL
  | CC_cdecl -> e_IMAGE_CEE_CS_CALLCONV_CDECL
  | CC_default -> 0x00
  | CC_vararg -> e_IMAGE_CEE_CS_CALLCONV_VARARG)
  
let rec tspec_as_intarray cenv env (et,tspec) = 
  if isNil (inst_of_tspec tspec) then 
    [| et |] @@ tdesc_as_TypeDefOrRefEncoded cenv (scoref_of_tspec tspec,
                                                   enclosing_tnames_of_tspec tspec,
                                                   tname_of_tspec tspec)
  else 
    Array.concat
      [ [| et_WITH; et |];
        tdesc_as_TypeDefOrRefEncoded cenv (scoref_of_tspec tspec,
                                           enclosing_tnames_of_tspec tspec,
                                           tname_of_tspec tspec);
        z_u32 !!!(List.length (inst_of_tspec tspec));
        genactuals_as_intarray cenv env (inst_of_tspec tspec) ]

and typ_as_tdor cenv env ty = cenv.typ_as_tdor_memoized (env,ty)

and typ_as_tdor_unmemoized cenv (env,ty) =
  if (typ_is_local cenv ty) then 
    let tref = tref_of_typ ty in 
    (tdor_TypeDef, idx_alloced_for_tdef cenv (enclosing_tnames_of_tref tref,tname_of_tref tref))
  else if is_tref_typ ty && isNil (inst_of_typ ty) then
    let tref = tref_of_typ ty in 
    (tdor_TypeRef, tdesc_as_TypeRef_idx cenv (scoref_of_tref tref, enclosing_tnames_of_tref tref,tname_of_tref tref))
  else 
    (tdor_TypeSpec, typ_as_TypeSpec_idx cenv env ty)


and genactuals_as_intarray cenv env (genactuals: genactual list) = 
  Array.concat (List.map (genactual_as_intarray cenv env) genactuals)

and typ_as_TypeSpec_idx cenv env ty = cenv.typ_as_TypeSpec_idx_memoized (env,ty)
and typ_as_TypeSpec_idx_unmemoized cenv (env,ty) = find_or_add_entry (table cenv tab_TypeSpec) (typ_as_TypeSpec_row cenv env ty)
and typ_as_BlobHeap_idx cenv env (ty:typ) = intarray_as_BlobHeap_idx cenv (typ_as_intarray cenv env ty)
and typ_as_TypeSpec_row cenv env (ty:typ) = Row [| Blob (typ_as_BlobHeap_idx cenv env ty) |]

and genactual_as_intarray cenv env ty = typ_as_intarray cenv env ty

and typ_as_intarray cenv env ty =
  let ilg = cenv.ilg in
  match ty with 
(* @todo: what are these doing here? *)
  | Type_value tspec when tname_of_tspec tspec = "System.String" ->   [| et_STRING |]
  | Type_value tspec when tname_of_tspec tspec = "System.Object" ->   [| et_OBJECT |]
  | typ when typ_is_SByte ilg typ ->   [| et_I1 |]
  | typ when typ_is_Int16 ilg typ ->   [| et_I2 |]
  | typ when typ_is_Int32 ilg typ ->    [| et_I4 |]
  | typ when typ_is_Int64 ilg typ ->     [| et_I8 |]
  | typ when typ_is_Byte ilg typ ->     [| et_U1 |]
  | typ when typ_is_UInt16 ilg typ ->     [| et_U2 |]
  | typ when typ_is_UInt32 ilg typ ->     [| et_U4 |]
  | typ when typ_is_UInt64 ilg typ ->     [| et_U8 |]
  | typ when typ_is_Double ilg typ ->     [| et_R8 |]
  | typ when typ_is_Single ilg typ ->     [| et_R4 |]
  | typ when typ_is_Bool ilg typ ->     [| et_BOOLEAN |]
  | typ when typ_is_Char ilg typ ->     [| et_CHAR |]
  | typ when typ_is_String ilg typ ->     [| et_STRING |]
  | typ when typ_is_Object ilg typ ->     [| et_OBJECT |]
  | typ when typ_is_IntPtr ilg typ ->     [| et_I |]
  | typ when typ_is_UIntPtr ilg typ ->     [| et_U |]
  | typ when typ_is_TypedReference ilg typ ->     [| et_TYPEDBYREF |]

  | Type_boxed tspec ->  tspec_as_intarray cenv env (et_CLASS,tspec)
  | Type_value tspec ->  tspec_as_intarray cenv env (et_VALUETYPE,tspec)
  | Type_array (shape,ty) ->  
      if shape = sdshape then [| et_SZARRAY |] @@ typ_as_intarray cenv env ty
      else Array.concat [ [| et_ARRAY |];typ_as_intarray cenv env ty; array_shape_as_intarray shape ]
  | Type_tyvar tv ->  
      let cgparams = env.envClassFormals in 
      if u16_to_int tv <  cgparams then 
        [| et_VAR |] @@ z_u32 (int_to_i32 (u16_to_int tv))
      else [| et_MVAR |] @@ z_u32 (int_to_i32 (u16_to_int tv -  cgparams))

  | Type_byref typ -> [| et_BYREF |] @@ typ_as_intarray cenv env typ
  | Type_ptr typ ->  [| et_PTR |] @@ typ_as_intarray cenv env typ
  | Type_void ->   [| et_VOID |] 
  | Type_fptr x ->
        [| et_FNPTR |] @@ 
        callsig_as_intarray cenv env (callconv_of_callsig x,args_of_callsig x,ret_of_callsig x,None,0)
  | Type_modified (req,tref,ty) ->
      [| (if req then et_CMOD_REQD else et_CMOD_OPT) |] @@ 
      tdesc_as_TypeDefOrRefEncoded cenv (scoref_of_tref tref, enclosing_tnames_of_tref tref,tname_of_tref tref) @@ 
      typ_as_intarray cenv env ty
(* ILX: @todo erase me earlier *)
  | Type_other(e) when Ilx.is_ilx_ext_typ e -> 
      begin match Ilx.dest_ilx_ext_typ e with 
      | Ilx.EType_erasable_array(shape,ty) -> typ_as_intarray cenv env (Type_array(shape,ty))
      end
   | _ -> failwith "typ_as_intarray"

and callsig_as_intarray cenv env (callconv,args,ret,(varargs:varargs),genarity) = 
  Array.concat 
    ([ [| callconv_as_byte genarity callconv |] ] @
     (if genarity > 0 then [ z_u32 (int_to_i32 genarity) ] else []) @
     [ z_u32 (int_to_i32 (List.length args + (match varargs with None -> 0 | Some l -> List.length l)));
       typ_as_intarray cenv env ret;
       Array.concat (List.map (typ_as_intarray cenv env) args) ] @
     begin match varargs with 
     | None
     | Some [] -> [] (* no extra arg = no sentinel *)
     | Some tys -> [| et_SENTINEL |] :: List.map (typ_as_intarray cenv env) tys
     end)

let typ_as_MemberRef_parent cenv env ty =
  match typ_as_tdor cenv env ty with 
  | (tag,tok) when tag = tdor_TypeDef -> dprint_endline "typ_as_MemberRef_parent: mspec should have been encoded as mdtMethodDef?"; MemberRefParent (mrp_TypeRef, 1)
  | (tag,tok) when tag = tdor_TypeRef -> MemberRefParent (mrp_TypeRef, tok)
  | (tag,tok) when tag = tdor_TypeSpec -> MemberRefParent (mrp_TypeSpec, tok)
  | _ -> failwith "typ_as_MemberRef_parent"


(* -------------------------------------------------------------------- 
 * Native types
 * -------------------------------------------------------------------- *)

(* based on information in ECMA and asmparse.y in the CLR codebase *)
let rec native_typ_as_BlobHeap_idx cenv (ty:native_type) = 
  intarray_as_BlobHeap_idx cenv (native_typ_as_intarray ty)
and native_typ_as_intarray ty = 
  if List.mem_assoc ty (Lazy.force native_type_rmap) then 
    [| List.assoc ty (Lazy.force native_type_rmap) |]
  else match ty with 
  | NativeType_empty -> [| |]
  | NativeType_custom (guid,nativeTypeName,custMarshallerName,cookieString) ->
      let guid = Bytes.to_intarray guid in 
      let u1 = string_as_utf8_intarray nativeTypeName in 
      let u2 = string_as_utf8_intarray custMarshallerName in 
      let u3 = Bytes.to_intarray cookieString in 
     [| nt_CUSTOMMARSHALER; |] @@ 
      z_u32 !!!(Array.length guid) @@ guid @@
      z_u32 !!!(Array.length u1) @@ u1 @@
      z_u32 !!!(Array.length u2) @@ u2 @@
      z_u32 !!!(Array.length u3) @@ u3
  | NativeType_fixed_sysstring i -> 
      [| nt_FIXEDSYSSTRING; |] @@ 
      z_u32 i

  | NativeType_fixed_array i -> 
      [| nt_FIXEDARRAY; |] @@ 
      z_u32 i
  | (* COM interop *) NativeType_safe_array (vt,name) -> 
      Array.concat
        ([ [| nt_SAFEARRAY; |] ] @ 
         [ z_u32 (variant_typ_as_i32 vt) ] @
         (begin match name with None -> [] | Some n -> 
         (* REVIEW: surely this is utf8 not unicode?? check me *)
           let u1 = Bytes.to_intarray (Bytes.string_as_unicode_bytes_null_terminated n) in 
           [ z_u32 !!!(Array.length u1) ; u1 ]
         end))
  | NativeType_array (nt,sizeinfo) ->  (* @todo: check if this corresponds to the ECMA spec *)
      Array.concat
        ([ [| nt_ARRAY; |] ] @
         (match nt with 
           None -> []
         | Some ntt ->
             (if ntt = NativeType_empty then 
               [ z_u32 !!!nt_MAX ]
             else 
               [ native_typ_as_intarray ntt ]) @ 
             (begin match sizeinfo with 
             | None -> [ ]  (* chunk out with zeroes because some crappy tools (e.g. asmmeta) read these poorly and expect further elements. *)
             | Some (pnum,additive) ->
                 [ (* ParamNum *) z_u32 pnum;
               (* ElemMul *) (* z_u32 0x1l *) ] @
                 (match additive with 
                   None -> [ ]
                 |  Some n ->  (* NumElem *) [ z_u32 n ])
             end)))
  | _ -> failwith "unexpected native type"

and variant_typ_as_i32 ty = 
  if List.mem_assoc ty (Lazy.force variant_type_map) then 
    (List.assoc ty (Lazy.force variant_type_map ))
  else match ty with 
  | VariantType_array vt -> vt_ARRAY ||| variant_typ_as_i32  vt
  | VariantType_vector  vt -> vt_VECTOR ||| variant_typ_as_i32  vt
  | VariantType_byref vt -> vt_BYREF ||| variant_typ_as_i32  vt
  | _ -> failwith "unexpected variant type"

(* -------------------------------------------------------------------- 
 * Native types
 * -------------------------------------------------------------------- *)

and field_init_as_BlobHeap_idx cenv env (ty:field_init) = 
  intarray_as_BlobHeap_idx cenv (field_init_as_intarray cenv env ty)
and field_init_as_intarray cenv env i = 
  match i with 
  | FieldInit_bytes b -> Bytes.to_intarray b
  | FieldInit_bool b ->  [| if b then 0x01 else 0x00 |]
  | FieldInit_char x -> u16_as_intarray x
  | FieldInit_int8 x -> i8_as_intarray x
  | FieldInit_int16 x -> i16_as_intarray x
  | FieldInit_int32 x -> i32_as_intarray x
  | FieldInit_int64 x -> i64_as_intarray x
  | FieldInit_uint8 x -> u8_as_intarray x
  | FieldInit_uint16 x -> u16_as_intarray x
  | FieldInit_uint32 x -> u32_as_intarray x
  | FieldInit_uint64 x -> u64_as_intarray x
  | FieldInit_float32 x -> ieee32_as_intarray x
  | FieldInit_float64 x -> ieee64_as_intarray x
  | FieldInit_ref  -> [| 0x00; 0x00; 0x00; 0x00;  |]

and field_init_as_flags i = 
  UShort 
    (int_to_u16
       begin match i with 
       | FieldInit_bytes _ -> et_STRING
       | FieldInit_bool _ -> et_BOOLEAN
       | FieldInit_char _ -> et_CHAR
       | FieldInit_int8 _ -> et_I1
       | FieldInit_int16 _ -> et_I2
       | FieldInit_int32 _ -> et_I4
       | FieldInit_int64 _ -> et_I8
       | FieldInit_uint8 _ -> et_U1
       | FieldInit_uint16 _ -> et_U2
       | FieldInit_uint32 _ -> et_U4
       | FieldInit_uint64 _ -> et_U8
       | FieldInit_float32 _ -> et_R4
       | FieldInit_float64 _ -> et_R8
       | FieldInit_ref -> et_CLASS
       end)
                  
(* -------------------------------------------------------------------- 
 * Type definitions
 * -------------------------------------------------------------------- *)

let member_access_as_flags access = 
  match access with 
  | MemAccess_compilercontrolled -> 0x00000000l
  | MemAccess_public -> 0x00000006l
  | MemAccess_private  -> 0x00000001l
  | MemAccess_family  -> 0x00000004l
  | MemAccess_famandassem -> 0x00000002l
  | MemAccess_famorassem -> 0x00000005l
  | MemAccess_assembly -> 0x00000003l

let access_as_flags  access = 
  match access with 
  | TypeAccess_public -> 0x00000001l
  | TypeAccess_private  -> 0x00000000l
  | TypeAccess_nested MemAccess_public -> 0x00000002l
  | TypeAccess_nested MemAccess_private  -> 0x00000003l
  | TypeAccess_nested MemAccess_family  -> 0x00000004l
  | TypeAccess_nested MemAccess_famandassem -> 0x00000006l
  | TypeAccess_nested MemAccess_famorassem -> 0x00000007l
  | TypeAccess_nested MemAccess_assembly -> 0x00000005l
  | TypeAccess_nested MemAccess_compilercontrolled -> failwith "bad type acccess"

let rec tdef_as_TypeDef_row cenv env enc td = 
  let nselem,nelem = name_as_elem_pair cenv (name_of_tdef td) in 
  let flags = 
    if (is_toplevel_tname (name_of_tdef td)) then 0x00000000l
    else
      
      access_as_flags (access_of_tdef td) |||
      begin 
        match layout_of_tdef td with 
        | TypeLayout_auto ->  0x00000000l
        | TypeLayout_sequential _  -> 0x00000008l
        | TypeLayout_explicit _ -> 0x00000010l
      end |||
      begin 
        match td.tdKind with
        | TypeDef_interface -> 0x00000020l
        | _ -> 0x00000000l
      end |||
      (if abstract_of_tdef td then 0x00000080l else 0x00000000l) |||
      (if sealed_of_tdef td then 0x00000100l else 0x00000000l) ||| 
      (if td.tdComInterop then 0x00001000l else 0x00000000l)  |||
      (if serializable_of_tdef td then 0x00002000l else 0x00000000l) |||
      begin 
        match encoding_of_tdef td with 
        | TypeEncoding_ansi -> 0x00000000l
        | TypeEncoding_autochar -> 0x00020000l
        | TypeEncoding_unicode ->  0x00010000l
      end |||
      begin 
        match initsemantics_of_tdef td with
        |  TypeInit_beforefield when not (match td.tdKind with TypeDef_interface -> true | _ -> false) -> 0x00100000l 
        | _ -> 0x00000000l
      end |||
(*         @@todo    (if specialname_of_tdef td then 0x00000400l else 0x00000000l) |||
           @@todo    (if rtspecialname_of_tdef td then 0x00000800l else 0x00000000l) ||| *)
      (if td.tdHasSecurity or dest_security_decls td.tdSecurityDecls <> [] then 0x00040000l else 0x00000000l) in

  let tdor_tag, tdor_row = optional_typ_as_tdor cenv env (extends_of_tdef td) in
  Row [| ULong flags ; 
        nelem; 
        nselem; 
        TypeDefOrRefOrSpec (tdor_tag, tdor_row); 
        SimpleIndex (tab_Field, cenv.fieldDefs.count + 1); 
        SimpleIndex(tab_Method,cenv.methodDefs.count + 1) |]  

and optional_typ_as_tdor cenv env ty_opt = 
  match ty_opt with
  |  None -> (tdor_TypeDef, 0)
  | Some ty -> (typ_as_tdor cenv env ty)

and tdef_as_PropertyMap_row cenv tidx = 
  Row [| SimpleIndex (tab_TypeDef,  tidx);
    SimpleIndex (tab_Property, cenv.propertyDefs.count + 1) |]  

and tdef_as_EventMap_row cenv tidx = 
  Row [| SimpleIndex (tab_TypeDef,  tidx);
        SimpleIndex (tab_Event, cenv.eventDefs.count + 1) |]  
    
and key_for_fdef tidx fd = 
  FdKey (tidx,fd.fdName, fd.fdType)

and fdef_pass2 cenv tidx fd = 
  ignore (add_uniq_entry "field" (fun (FdKey (_,n,_)) -> n) cenv.fieldDefs (key_for_fdef tidx fd))

and key_for_mdef tidx md = 
  MdKey (tidx,List.length md.mdGenericParams, md.mdName, typ_of_return md.mdReturn, List.map typ_of_param md.mdParams)

and mdef_pass2 cenv tidx md = 
  ignore 
    (add_uniq_entry "method" 
       (fun (MdKey (tidx,arity,n,ret,args) as key) -> 
         dprint_endline "Duplicate in method table is:";
         dprint_endline ("  Type index: "^string_of_int tidx);
         dprint_endline ("  Method name: "^n);
         dprint_endline ("  Method arity (num generic params): "^string_of_int arity);
         dprint_string ("  Method return type: "); Ilprint.output_typ stderr ret; dprint_endline "";
         List.iter (fun arg -> 
           dprint_string ("  Method arg: "); Ilprint.output_typ stderr arg; dprint_endline "")
           args;
         n)
       cenv.methodDefs 
       (key_for_mdef tidx md))

and key_for_property tidx x = 
  PropKey (tidx, x.propName, x.propType, x.propArgs)

and property_pass2 cenv tidx x = 
  ignore (add_uniq_entry "property" (fun (PropKey (_,n,_,_)) -> n) cenv.propertyDefs (key_for_property tidx x))

and typ_as_Implements_Row cenv env tidx ty =
  let tdor_tag,tdor_row = typ_as_tdor cenv env ty in 
  Row [|SimpleIndex (tab_TypeDef, tidx); 
        TypeDefOrRefOrSpec (tdor_tag,tdor_row) |]
and implements_pass2 cenv env tidx ty =
  ignore (add_entry (table cenv tab_InterfaceImpl) (typ_as_Implements_Row cenv env tidx ty))
    
and key_for_event tidx x = 
  EventKey (tidx, x.eventName)

and event_pass2 cenv tidx x = 
  ignore (add_uniq_entry "event" (fun (EventKey(a,b)) -> b) cenv.eventDefs (key_for_event tidx x))

and tdef_pass2 pidx enc cenv td =
 try 
  let env = env_enter_tdef (List.length td.tdGenericParams) in 
  let tidx = idx_alloced_for_tdef cenv (enc,name_of_tdef td) in 
  let tidx2 = add_entry (table cenv tab_TypeDef) (tdef_as_TypeDef_row cenv env enc td) in 
  if tidx <> tidx2 then failwith "index of typedef on second pass does not match index on first pass";

  (* Add entries to auxiliary mapping tables, e.g. Nested, PropertyMap etc. *)
  (* Note Nested is organised differntly to the others... *)
  if enc <> [] then begin
    ignore (add_entry (table cenv tab_Nested) (Row [| SimpleIndex (tab_TypeDef, tidx); 
                                                     SimpleIndex (tab_TypeDef, pidx) |]));
  end;
  let props = dest_pdefs (properties_of_tdef td) in 
  if props <> [] then begin
    ignore (add_entry (table cenv tab_PropertyMap) (tdef_as_PropertyMap_row cenv tidx)); 
  end;
  let events = (dest_edefs (events_of_tdef td)) in 
  if events <> [] then begin
    ignore (add_entry (table cenv tab_EventMap) (tdef_as_EventMap_row cenv tidx)); 
  end;
  (* Now generate or assign index numbers for tables referenced by the maps. *)
  (* Don't yet generate contents of these tables - leave that to pass3, as *)
  (* code may need to embed these entries. *)
  List.iter (implements_pass2 cenv env tidx) (implements_of_tdef td);
  List.iter (property_pass2 cenv tidx) props;
  List.iter (event_pass2 cenv tidx) events;
  List.iter (fdef_pass2 cenv tidx) (dest_fdefs (fields_of_tdef td));
  List.iter (mdef_pass2 cenv tidx) (dest_mdefs (methods_of_tdef td));
  tdefs_pass2 tidx (enc@[name_of_tdef td]) cenv (dest_tdefs (nested_of_tdef td))
 with e ->
   dprint_endline ("Error in pass2 for type "^td.tdName^", error: "^Printexc.to_string e);
   (*F# rethrow(); F#*) raise e

and tdefs_pass2 pidx enc cenv tds =
  List.iter (tdef_pass2 pidx enc cenv) tds

(*=====================================================================
 * Pass 3 - write details of methods, fields, IL code, custom attrs etc.
 *=====================================================================*)

exception Mdef_not_found
let rec mdkey_as_MethodDef_idx cenv mdkey = 
  try tbl_find cenv.methodDefs mdkey
  with Not_found -> 
    let MdKey (tidx,ngparams,n,rty,argtys) = mdkey in 
    begin 
      let tname_of_tidx i = 
        match Hashtbl.fold (fun tkey2 tidx2 sofar -> if i = tidx2 then if sofar = None then Some tkey2 else failwith "mutiple type names map to index" else sofar) cenv.typeDefs.tbl None 
        with 
          Some x -> x
        | None -> raise Mdef_not_found in 
      let TdKey (tenc,tname) = tname_of_tidx tidx in 
      dprint_endline ("The local method '"^(String.concat "." (tenc@[tname]))^"'::'"^n^"' was referenced but not declared");
      dprint_endline ("generic arity: "^string_of_int ngparams);
      dprint_string "return type: "; Ilprint.output_typ stderr rty; dprint_endline "";
      List.iter (fun ty -> dprint_string "arg type: "; Ilprint.output_typ stderr ty; dprint_endline "") argtys;
      let res = ref None in 
      Hashtbl.iter (fun (MdKey (tidx2,ngparams2,n2,rty2,argtys2) as mdkey2) _ ->  
        if tidx2 = tidx && n = n2 then begin
          let TdKey (tenc2,tname2) = tname_of_tidx tidx2 in 
          dprint_endline ("A method in '"^(String.concat "." (tenc2@[tname2]))^"' had the right name but the wrong signature:");
          dprint_endline ("generic arity: "^string_of_int ngparams);
          dprint_string "return type: "; Ilprint.output_typ stderr rty2; dprint_endline "";
          List.iter (fun ty -> dprint_string "arg type: "; Ilprint.output_typ stderr ty; dprint_endline "") argtys2;
          res := Some (tbl_find cenv.methodDefs mdkey)
        end) cenv.methodDefs.tbl;
      raise Mdef_not_found
    end

let rec mdef_as_MethodDef_idx cenv tidx md = 
  try mdkey_as_MethodDef_idx cenv ( key_for_mdef tidx md) with Mdef_not_found -> 1

and fdef_key_as_FieldDef_idx cenv fdkey = 
  try tbl_find cenv.fieldDefs fdkey 
  with Not_found -> 
    let (FdKey (tidx,n,ty)) = fdkey in 
    dprint_endline ("The local field "^n^" was referenced but not declared");
    1

and fdef_as_FieldDef_idx cenv tidx fd = 
  fdef_key_as_FieldDef_idx cenv (key_for_fdef tidx fd) 

(* -------------------------------------------------------------------- 
 * Il.method_ref --> MethodDef.  
 * 
 * Only successfuly converts method_ref's referring to 
 * methods in the module being emitted.
 * -------------------------------------------------------------------- *)

let mref_as_MethodDef_idx cenv mref =
 try 
  let tref = tref_of_mref mref in 
  if not (tref_is_local cenv tref) then
   begin
      dprint_string ("tref = ");
      Ilprint.output_tref stderr tref;
      dprint_string (", mref = ");
      Ilprint.output_mref stderr mref;
      dprint_endline (".");
       failwith "method referred to by method impl, event or property is not in a type defined in this module";
   end;
  let tidx = idx_alloced_for_tdef cenv (enclosing_tnames_of_tref tref,tname_of_tref tref) in 
  let mdkey = MdKey (tidx,genarity_of_mref mref, name_of_mref mref, ret_of_mref mref, args_of_mref mref) in 
  mdkey_as_MethodDef_idx cenv mdkey
 with e ->
   dprint_endline ("Error in mref_as_MethodDef_idx for method "^name_of_mref mref^", error: "^Printexc.to_string e);
   (*F# rethrow(); F#*) raise e

let rec mrefinfo_as_MemberRef_row cenv env fenv (nm,typ,callconv,args,ret,varargs,genarity) =
  Row [| typ_as_MemberRef_parent cenv env typ;
        String (string_as_StringHeap_idx cenv nm);
        Blob (mrefinfo_as_BlobHeap_idx cenv fenv (callconv,args,ret,varargs,genarity)) |]

and mrefinfo_as_BlobHeap_idx cenv env info = 
  intarray_as_BlobHeap_idx cenv (callsig_as_intarray cenv env info)

let mrefinfo_as_MemberRef_idx cenv env  ((nm,typ,cc,args,ret,varargs,genarity) as minfo) = 
  let fenv = env_enter_msig (if is_array_ty typ then env.envClassFormals else List.length (inst_of_typ typ)) genarity in
  find_or_add_entry (table cenv tab_MemberRef) 
    (mrefinfo_as_MemberRef_row cenv env fenv  minfo)

let mrefinfo_as_mdor_unmemoized cenv (always_mdef,env,((nm,typ,cc,args,ret,varargs,genarity) as minfo)) =
  if varargs = None && (always_mdef or typ_is_local cenv typ) then
    begin
      if not (is_tref_typ typ) then failwith "mrefinfo_as_mdor: unexpected local tref-typ";
      try (mdor_MethodDef, mref_as_MethodDef_idx cenv (mk_mref(tref_of_typ typ, cc, nm, genarity, args,ret)))
      with Mdef_not_found -> (mdor_MemberRef, mrefinfo_as_MemberRef_idx cenv env minfo)
    end 
  else (mdor_MemberRef, mrefinfo_as_MemberRef_idx cenv env minfo)

let mrefinfo_as_mdor always_mdef cenv env mrefinfo = cenv.mrefinfo_as_mdor_memoized (always_mdef,env,mrefinfo)

(* -------------------------------------------------------------------- 
 * Il.method_spec --> MethodRef/MethodDef/MethodSpec
 * -------------------------------------------------------------------- *)

let rec mspecinfo_as_MethodSpec_idx cenv env (nm,typ,cc,args,ret,varargs,minst) = 
  let mdor_tag,mdor_row = mrefinfo_as_mdor false cenv env (nm,typ,cc,args,ret,varargs,List.length minst) in
  find_or_add_entry (table cenv tab_MethodSpec) 
    (Row [| MethodDefOrRef (mdor_tag,mdor_row);
           Blob (intarray_as_BlobHeap_idx cenv 
                   (Array.concat 
                      [ [| e_IMAGE_CEE_CS_CALLCONV_GENERICINST |];
                        z_u32 (int_to_i32 (List.length minst));
                        Array.concat (List.map (typ_as_intarray cenv env) minst) ])) |])

and mdor_as_uncoded (tag,idx) =
  let tab = 
    if tag = mdor_MethodDef then tab_Method
    else if tag = mdor_MemberRef then tab_MemberRef  
    else failwith "mdor_as_uncoded" in 
  uncoded_token tab idx

and mspecinfo_as_uncoded cenv env ((_,_,_,_,_,_,minst) as minfo) =
  if minst <> [] then 
    uncoded_token tab_MethodSpec (mspecinfo_as_MethodSpec_idx cenv env minfo)
  else 
    mdor_as_uncoded (mrefinfo_as_mdor false cenv env (mrefinfo_of_mspecinfo minfo))

and mspec_as_uncoded cenv env mspec = cenv.mspec_as_uncoded_memoized (env,mspec)
and mspec_as_uncoded_unmemoized cenv (env,mspec) = mspecinfo_as_uncoded cenv env (mspecinfo_of_mspec mspec)

and mrefinfo_of_mspecinfo (nm,typ,cc,args,ret,varargs,minst) = (nm,typ,cc,args,ret,varargs,List.length minst)

and mspec_as_mdor cenv env (mspec,varargs) =
 mrefinfo_as_mdor false cenv env (mrefinfo_of_mspecinfo (mspecinfo_of_mspec (mspec,varargs)))
and mspec_as_mdef cenv env (mspec,varargs) =
 mrefinfo_as_mdor true cenv env (mrefinfo_of_mspecinfo (mspecinfo_of_mspec (mspec,varargs)))

and mspecinfo_of_mspec (mspec,varargs) = 
      (name_of_mspec mspec,
       enclosing_typ_of_mspec mspec,
       callconv_of_mspec mspec,
       formal_args_of_mspec mspec,
       formal_ret_of_mspec mspec,
       varargs,
       minst_of_mspec mspec)

(* -------------------------------------------------------------------- 
 * Il.method_in_parent --> MethodRef/MethodDef
 * 
 * Used for MethodImpls.
 * -------------------------------------------------------------------- *)

let rec ospec_as_MemberRef_idx cenv env ospec = 
  let fenv = env_enter_ospec ospec in
  find_or_add_entry (table cenv tab_MemberRef) 
    (mrefinfo_as_MemberRef_row cenv env fenv  (name_of_ospec ospec,
                                               enclosing_typ_of_ospec ospec,
                                               callconv_of_ospec ospec,
                                               formal_args_of_ospec ospec,
                                               formal_ret_of_ospec ospec,
                                               None,
                                               genarity_of_mref (formal_mref_of_ospec ospec)))
    
and ospec_as_mdor cenv env ospec =
   let typ = enclosing_typ_of_ospec ospec in 
   if typ_is_local cenv typ then 
    begin
      if not (is_tref_typ typ) then failwith "ospec_as_mdor: unexpected local tref-typ"; 
      try (mdor_MethodDef, mref_as_MethodDef_idx cenv (formal_mref_of_ospec ospec))
      with Mdef_not_found ->  (mdor_MemberRef, ospec_as_MemberRef_idx cenv env ospec) 
    end  
  else (mdor_MemberRef, ospec_as_MemberRef_idx cenv env ospec) 

(* -------------------------------------------------------------------- 
 * Il.method_ref --> MethodRef/MethodDef
 * 
 * Used for Custom Attrs.
 * -------------------------------------------------------------------- *)

let rec mref_as_MemberRef_idx cenv env fenv mref = 
  find_or_add_entry (table cenv tab_MemberRef) 
    (mrefinfo_as_MemberRef_row cenv env fenv (name_of_mref mref,
                                              mk_nongeneric_boxed_typ (tref_of_mref mref),
                                              callconv_of_mref mref,
                                              args_of_mref mref,
                                              ret_of_mref mref,
                                              None,
                                              genarity_of_mref mref))

and mref_as_cat cenv mref =
  let fenv = env_enter_mref mref in
  let tref = tref_of_mref mref in 
  if tref_is_local cenv tref then
    try (cat_MethodDef, mref_as_MethodDef_idx cenv mref)
    with Mdef_not_found -> (cat_MemberRef, mref_as_MemberRef_idx cenv fenv fenv mref)
  else
    (cat_MemberRef, mref_as_MemberRef_idx cenv fenv fenv mref)

(* -------------------------------------------------------------------- 
 * Il.custom_attrs --> CustomAttribute rows
 * -------------------------------------------------------------------- *)

let rec custom_attr_data_as_BlobHeap_idx cenv data = 
  if Bytes.length data = 0 then 0 else bytes_as_BlobHeap_idx cenv data

and custom_attr_as_CustomAttribute_row cenv hca attr = 
  let cat = mref_as_cat cenv (formal_mref_of_mspec attr.customMethod) in 
  Row [| HasCustomAttribute (fst hca, snd hca);
        CustomAttributeType (fst cat, snd cat); 
        Blob (custom_attr_data_as_BlobHeap_idx cenv attr.customData); |]  

and custom_attr_pass3 cenv hca attr = 
  ignore (find_or_add_entry (table cenv tab_CustomAttribute) 
            (custom_attr_as_CustomAttribute_row cenv hca attr))

and custom_attrs_pass3 cenv hca attrs = 
  List.iter (custom_attr_pass3 cenv hca) (dest_custom_attrs attrs) 

(* -------------------------------------------------------------------- 
   * Il.security_decl --> DeclSecurity rows
 * -------------------------------------------------------------------- *)

let rec security_decl_as_DeclSecurity_row cenv hds attr = 
  let action = 
    match attr with 
    | Permission (act,_,_) 
    | PermissionSet (act, _) -> act in 

  let bytes = 
    match attr with 
    | Permission (_,ty,vals) -> failwith "ilwriter cannot yet serialize permission blobs" (* @todo *)
    | PermissionSet (_, s) -> Bytes.to_intarray s in 

  Row [| UShort (int_to_u16 (List.assoc action (Lazy.force secaction_map)));
        HasDeclSecurity (fst hds, snd hds);
        Blob (intarray_as_BlobHeap_idx cenv bytes); |]  

and security_decl_pass3 cenv hds attr = 
  ignore (find_or_add_entry (table cenv tab_Permission) 
            (security_decl_as_DeclSecurity_row cenv hds attr))

and security_decls_pass3 cenv hds attrs = 
  List.iter (security_decl_pass3 cenv hds) attrs 

(* -------------------------------------------------------------------- 
 * Il.field_spec --> FieldRef/FieldDef
 * -------------------------------------------------------------------- *)

let rec fspec_as_MemberRef_row cenv env fenv fspec = 
  Row [| typ_as_MemberRef_parent cenv env (enclosing_typ_of_fspec fspec);
        String (string_as_StringHeap_idx cenv (name_of_fspec fspec));
        Blob (fspec_sig_as_BlobHeap_idx cenv fenv fspec) |]

and fspec_as_MemberRef_idx cenv env fspec = 
  let fenv = env_enter_fspec fspec in
  find_or_add_entry (table cenv tab_MemberRef) (fspec_as_MemberRef_row cenv env fenv fspec)

and fspec_sig_as_intarray cenv env fspec = 
  [| e_IMAGE_CEE_CS_CALLCONV_FIELD  |] @@ 
  typ_as_intarray cenv env (formal_typ_of_fspec fspec)

and fspec_sig_as_BlobHeap_idx cenv env x = 
  intarray_as_BlobHeap_idx cenv (fspec_sig_as_intarray cenv env x)

(* fdor == FieldDefOrRef, only used here to make this code look like the method code above *)
and fspec_as_fdor cenv env fspec =
  let typ = enclosing_typ_of_fspec fspec in 
  if typ_is_local cenv typ then
    begin
      if not (is_tref_typ typ) then failwith "fspec_as_fdor: unexpected local tref-typ";
      let tref = tref_of_typ typ in 
      let tidx = idx_alloced_for_tdef cenv (enclosing_tnames_of_tref tref,tname_of_tref tref) in 
      let fdkey = FdKey (tidx,name_of_fspec fspec, formal_typ_of_fspec fspec) in 
      (true, fdef_key_as_FieldDef_idx cenv fdkey)
    end 
  else (false, fspec_as_MemberRef_idx cenv env fspec)

and fdor_as_uncoded (tag,idx) =
  let tab = if tag then tab_Field else tab_MemberRef in 
  uncoded_token tab idx

(* -------------------------------------------------------------------- 
 * Il.callsig --> StandAloneSig
 * -------------------------------------------------------------------- *)

let callsig_sig_as_BlobHeap_idx cenv env (callsig,varargs) = 
  intarray_as_BlobHeap_idx cenv 
    (callsig_as_intarray cenv env (callconv_of_callsig callsig,
                                    args_of_callsig callsig,
                                    ret_of_callsig callsig,varargs,0))
    
let callsig_as_StandAloneSig_row cenv env x = 
  Row [| Blob (callsig_sig_as_BlobHeap_idx cenv env x) |]

let callsig_as_StandAloneSig_idx cenv env info = 
  find_or_add_entry (table cenv tab_StandAloneSig) (callsig_as_StandAloneSig_row cenv env info)

(* -------------------------------------------------------------------- 
 * local signatures --> BlobHeap idx
 * -------------------------------------------------------------------- *)

let local_sig_as_intarray cenv env locals = 
  Array.concat 
    [ [| e_IMAGE_CEE_CS_CALLCONV_LOCAL_SIG |];
      z_u32 !!!(List.length locals);
      Array.concat (List.map (typ_of_local >> typ_as_intarray cenv env) locals) ]

let local_sig_as_BlobHeap_idx cenv env locals = 
  intarray_as_BlobHeap_idx cenv (local_sig_as_intarray cenv env locals)

let local_sig_as_StandAloneSig_row cenv env locals = 
  Row [| Blob ( local_sig_as_BlobHeap_idx cenv env locals ); |]


(* -------------------------------------------------------------------- 
 * Buffer to write results of emitting code into.  Also record:
 *   - branch sources (where fixups will occur)
 *   - possible branch destinations
 *   - locations of embedded handles into the string table
 *   - the exception table
 * -------------------------------------------------------------------- *)

type seh_kind = 
  | FinallyClause 
  | FaultClause 
  | TypeFilterClause of int32 
  | FilterClause of int

type seh_spec = (int * int * int * int * seh_kind)

module Codebuf = struct

    type t = 
        { code: Bytebuf.t; 
          mutable reqd_brfixups: ((int * int option) * int * code_label list) list; (* (instruction; optional short form); start of instr in code buffer; code loc for the end of the instruction the fixup resides in ; where is the destination of the fixup *)
          avail_brfixups: (code_label, int) Hashtbl.t;
          mutable reqd_string_fixups_in_method: (int * int) list; (* code loc to fixup in code buffer *)
          mutable seh: seh_spec list; (* data for exception handling clauses *)
          seqpoints: pdb_sequence_point buf;
        }

    let new_codebuf nm = 
        { seh = [];
          code= Bytebuf.create 200;
          reqd_brfixups=[];
          reqd_string_fixups_in_method=[];
          avail_brfixups = Hashtbl.create 0;
          seqpoints = new_buf "sequence points" 10 
            { pdbSeqPointDocument=0;
              pdbSeqPointOffset=0;
              pdbSeqPointLine=0;
              pdbSeqPointColumn=0;
              pdbSeqPointEndLine=0;
              pdbSeqPointEndColumn=0; }
       }

    let emit_seh_clause codebuf seh = codebuf.seh <- seh :: codebuf.seh

    let emit_seqpoint cenv codebuf m = 
        if cenv.generate_pdb then 
          let doc = (find_or_add_entry cenv.pdbdocuments m.sourceDocument) - 1 in  (* table indexes are 1-based, document array indexes are 0-based *)
          buf_emit_one codebuf.seqpoints 
            { pdbSeqPointDocument=doc;
              pdbSeqPointOffset= Bytebuf.length codebuf.code;
              pdbSeqPointLine=m.sourceLine;
              pdbSeqPointColumn=m.sourceColumn;
              pdbSeqPointEndLine=m.sourceEndLine;
              pdbSeqPointEndColumn=m.sourceEndColumn; }
              
    let emit_byte codebuf x = Bytebuf.emit_int_as_byte codebuf.code x
    let emit_intarray_as_bytes codebuf x = Bytebuf.emit_intarray_as_bytes codebuf.code x
    let emit_u16 codebuf x = bytebuf_emit_u16 codebuf.code x
    let emit_i32 codebuf x = Bytebuf.emit_i32 codebuf.code x
    let emit_i64 codebuf x = bytebuf_emit_i64 codebuf.code x

    let emit_uncoded codebuf u = emit_i32 codebuf u

    let record_reqd_stringfixup codebuf stringidx = 
        codebuf.reqd_string_fixups_in_method <- (Bytebuf.length codebuf.code, stringidx) :: codebuf.reqd_string_fixups_in_method;
        (* Write a special value in that we check later when applying the fixup *)
        emit_i32 codebuf 0xdeadbeefl

    let record_reqd_brfixups codebuf i tgs = 
        codebuf.reqd_brfixups <- (i, Bytebuf.length codebuf.code, tgs) :: codebuf.reqd_brfixups;
        (* Write a special value in that we check later when applying the fixup *)
        (* Value is 0x11 {deadbbbb}* where 11 is for the instruction and deadbbbb is for each target *)
        emit_byte codebuf 0x11; (* for the instruction *)
        (if fst i = i_switch then 
          emit_i32 codebuf (int_to_i32 (List.length tgs)));
        List.iter (fun _ -> emit_i32 codebuf 0xdeadbbbbl) tgs

    let record_reqd_brfixup codebuf i tg = record_reqd_brfixups codebuf i [tg]
    let record_avail_brfixup codebuf tg = 
        Hashtbl.add codebuf.avail_brfixups tg (Bytebuf.length codebuf.code)

    (* -------------------------------------------------------------------- 
     * Applying branch fixups.  Use short versions of instructions
     * wherever possible.  Sadly we can only determine if we can use a short
     * version after we've layed out the code for all other instructions.  
     * This in turn means that using a short version may change 
     * the various offsets into the code.
     * -------------------------------------------------------------------- *)

    let binchop p arr = 
      let rec go n m =
        if n > m then raise Not_found 
        else 
          let i = (n+m)/2 in 
          let c = p arr.(i) in if c = 0 then i else if c < 0 then go n (i-1) else go (i+1) m in 
      go 0 (Array.length arr)

    let apply_brfixups 
          orig_code 
          orig_seh 
          orig_reqd_string_fixups 
          orig_avail_brfixups 
          orig_reqd_brfixups 
          orig_seqpoints
          orig_scopes = 
      let ordered_orig_reqd_brfixups = List.sort (fun (_,fixuploc1,_) (_,fixuploc2,_) -> compare fixuploc1 fixuploc2) orig_reqd_brfixups in 
      let new_code = Bytebuf.create (Bytes.length orig_code) in

      (* Copy over all the code, working out whether the branches will be short *)
      (* or long and adjusting the branch destinations.  Record an adjust function to adjust all the other *)
      (* gumpf that refers to fixed offsets in the code stream. *)
      let new_code, new_reqd_brfixups,adjuster = 
        let remaining_reqd_brfixups = ref ordered_orig_reqd_brfixups in 
        let orig_where = ref 0 in 
        let new_where = ref 0 in 
        let done_last = ref false in 
        let new_reqd_brfixups = ref [] in 

        let adjustments = ref [] in 

        while (!remaining_reqd_brfixups <> [] or not !done_last) do
          let doing_last = isNil !remaining_reqd_brfixups   in 
          let orig_start_of_nobranch_block = !orig_where in 
          let new_start_of_nobranch_block = !new_where in 

          if logging then dprint_endline ("move chunk, doing_last = "^(if doing_last then "true" else "false"));

          let orig_end_of_nobranch_block = 
            if doing_last then Bytes.length orig_code 
            else 
              let (_,orig_start_of_instr,_) = List.hd !remaining_reqd_brfixups in 
              orig_start_of_instr in 

          (* Copy over a chunk of non-branching code *)
          let nobranch_len = orig_end_of_nobranch_block - orig_start_of_nobranch_block in 
          Bytebuf.emit_bytes new_code (Bytes.sub orig_code  orig_start_of_nobranch_block nobranch_len);
            
          (* Record how to adjust addresses in this range, including the branch instruction *)
          (* we write below, or the end of the method if we're doing the last bblock *)
          adjustments := (orig_start_of_nobranch_block,orig_end_of_nobranch_block,new_start_of_nobranch_block) :: !adjustments;
         
          (* Increment locations to the branch instruction we're really interested in  *)
          orig_where := orig_end_of_nobranch_block;
          new_where := !new_where + nobranch_len;
            
          (* Now do the branch instruction.  Decide whether the fixup will be short or long in the new code *)
          if doing_last then done_last := true
          else begin
            let (i,orig_start_of_instr,tgs) = List.hd !remaining_reqd_brfixups in 
            remaining_reqd_brfixups := List.tl !remaining_reqd_brfixups;
            if Bytes.get orig_code orig_start_of_instr <> 0x11 then failwith "br fixup sanity check failed (1)";
            let i_length = if fst i = i_switch then 5 else 1 in 
            orig_where := !orig_where + i_length;

            let orig_end_of_instr = orig_start_of_instr + i_length + 4 * List.length tgs in 
            let new_end_of_instr_if_small = !new_where + i_length + 1 in 
            let new_end_of_instr_if_big = !new_where + i_length + 4 * List.length tgs in 
            
            let short = 
              match i,tgs with 
              | (_,Some i_short),[tg] 
                  when
                begin 
                  (* Use the original offsets to compute if the branch is small or large.  This is *)
                  (* a safe approximation because code only gets smaller. *)
                  if not (Hashtbl.mem orig_avail_brfixups tg) then 
                    dprint_endline ("branch target "^string_of_code_label tg^" not found in code");
                  let orig_dest = try Hashtbl.find orig_avail_brfixups tg with Not_found -> 666666 in 
                  let orig_rel_offset = orig_dest - orig_end_of_instr in
                  -128 <= orig_rel_offset && orig_rel_offset <= 127
                end 
                ->
                  Bytebuf.emit_int_as_byte new_code i_short;
                  true
              | (i_long,_),_ ->
                  Bytebuf.emit_int_as_byte new_code i_long;
                  (if i_long = i_switch then 
                    Bytebuf.emit_i32 new_code (int_to_i32 (List.length tgs)));
                  false in 
            
            new_where := !new_where + i_length;
            if !new_where <> (Bytebuf.length new_code) then dprint_endline "mismatch between new_where and new_code";

            List.iter
              (fun tg ->
                let orig_fixuploc = !orig_where in 
                check_fixup32 orig_code orig_fixuploc 0xdeadbbbbl;
                
                if short then begin
                  new_reqd_brfixups := (!new_where, new_end_of_instr_if_small, tg, true) :: !new_reqd_brfixups;
                  Bytebuf.emit_int_as_byte new_code 0x98; (* sanity check *)
                  new_where := !new_where + 1;
                end else begin
                  new_reqd_brfixups := (!new_where, new_end_of_instr_if_big, tg, false) :: !new_reqd_brfixups;
                  Bytebuf.emit_i32 new_code 0xf00dd00fl; (* sanity check *)
                  new_where := !new_where + 4;
                end;
                if !new_where <> Bytebuf.length new_code then dprint_endline "mismatch between new_where and new_code";
                orig_where := !orig_where + 4)
              tgs;
            
            if !orig_where <> orig_end_of_instr then dprint_endline "mismatch between orig_where and orig_end_of_instr";
          end;
        done;

        let adjuster  = 
          let arr = Array.of_list (List.rev !adjustments) in 
          fun addr -> 
            let i = 
                try binchop (fun (a1,a2,_) -> if addr < a1 then -1 else if addr > a2 then 1 else 0) arr 
                with Not_found -> dprint_endline ("adjuster: address "^string_of_int addr^" is out of range"); addr in 
            let (orig_start_of_nobranch_block,orig_end_of_nobranch_block,new_start_of_nobranch_block) = arr.(i) in
            addr - (orig_start_of_nobranch_block - new_start_of_nobranch_block)  in

        Bytebuf.close new_code, 
        !new_reqd_brfixups, 
        adjuster in

      (* Now adjust everything *)
      let new_avail_brfixups = 
        let tab = Hashtbl.create 10 in 
        Hashtbl.iter (fun tglab orig_brdest -> Hashtbl.add tab tglab (adjuster orig_brdest)) orig_avail_brfixups;
        tab in 
      let new_reqd_string_fixups = List.map (fun (orig_fixuploc,stok) -> adjuster orig_fixuploc,stok) orig_reqd_string_fixups in 
      let new_seqpoints = Array.map (fun sp -> {sp with pdbSeqPointOffset=adjuster sp.pdbSeqPointOffset}) orig_seqpoints in 
      let new_seh = 
        List.map
          (fun (st1,sz1,st2,sz2,kind) ->
            (adjuster st1,(adjuster (st1 + sz1) - adjuster st1),
             adjuster st2,(adjuster (st2 + sz2) - adjuster st2),
             (match kind with 
             | FinallyClause | FaultClause | TypeFilterClause _ -> kind
             | FilterClause n -> FilterClause (adjuster n))))
          orig_seh in 
      let new_scopes =
        let rec remap scope =
          {scope with pdbScopeStartOffset = adjuster scope.pdbScopeStartOffset;
                      pdbScopeEndOffset = adjuster scope.pdbScopeEndOffset;
                      pdbScopeChildren = Array.map remap scope.pdbScopeChildren } in 
        List.map remap orig_scopes in 
      
      (* Now apply the adjusted fixups in the new code *)
      List.iter
        (fun (new_fixuploc,end_of_instr,tg, small) ->
          if not (Hashtbl.mem new_avail_brfixups tg) then 
            dprint_endline ("target "^string_of_code_label tg^" not found in new fixups");
          try 
            let n = Hashtbl.find new_avail_brfixups tg in 
            let rel_offset = !!!(n - end_of_instr) in
            if small then begin
              if Bytes.get new_code new_fixuploc <> 0x98 then failwith "br fixupsanity check failed";
              Bytes.set new_code new_fixuploc (b0 rel_offset);
            end else begin
              check_fixup32 new_code new_fixuploc 0xf00dd00fl;
              fixup32 new_code new_fixuploc rel_offset
            end
          with Not_found -> ())
        new_reqd_brfixups;
      new_code, new_reqd_string_fixups, new_seh, new_seqpoints, new_scopes


    (* -------------------------------------------------------------------- 
     * Structured residue of emitting instructions: SEH exception handling
     * and scopes for local variables.
     * -------------------------------------------------------------------- *)

    (* Emitting instructions generates a tree of seh specifications *)
    (* We then emit the exception handling specs separately. *)
    (* nb. ECMA spec says the SEH blocks must be returned inside-out *)
    type seh_tree = 
      | Tip 
      | Node of (seh_spec option * seh_tree list) list
        
    (* Emitting instructions also generates a tree of locals-in-use specifications *)
    (* i.e. scopes suitable for use to generate debugging info *)
    type scope = pdb_method_scope


    (* -------------------------------------------------------------------- 
     * Table of encodings for instructions without arguments, also indexes
     * for all instructions.
     * -------------------------------------------------------------------- *)

    let encoding_of_noarg_instr_table = Hashtbl.create 300;;
    let _ = List.iter 
        (fun (x,mk) -> Hashtbl.add encoding_of_noarg_instr_table mk x)
        (noarg_instrs());;
    let encoding_of_noarg_instr si = Hashtbl.find encoding_of_noarg_instr_table si
    (* let is_noarg_instr s = Hashtbl.mem encoding_of_noarg_instr_table s;; *)


    (* -------------------------------------------------------------------- 
     * Emit instructions
     * -------------------------------------------------------------------- *)

    let emit_instr_code codebuf i = 
      if i > 0xff then begin
        assert (!!!i lsr 8 = 0xfel); 
        emit_byte codebuf (??? ((!!!i lsr 8)  &&& 0xffl)); 
        emit_byte codebuf (??? (!!!i &&& 0xffl)); 
      end else begin
        emit_byte codebuf i
      end

    let emit_typ_instr cenv codebuf env i ty = 
      emit_instr_code codebuf i; 
      emit_uncoded codebuf (tdor_as_uncoded (typ_as_tdor cenv env ty))

    let emit_mspecinfo_instr cenv codebuf env i mspecinfo = 
      emit_instr_code codebuf i; 
      emit_uncoded codebuf (mspecinfo_as_uncoded cenv env mspecinfo)

    let emit_mspec_instr cenv codebuf env i mspec = 
      emit_instr_code codebuf i; 
      emit_uncoded codebuf (mspec_as_uncoded cenv env mspec)

    let emit_fspec_instr cenv codebuf env i fspec = 
      emit_instr_code codebuf i; 
      emit_uncoded codebuf (fdor_as_uncoded (fspec_as_fdor cenv env fspec))

    let emit_short_u16_instr codebuf (i_short,i) x = 
      let n = u16_to_int x in 
      if n <= 255 then begin
        emit_instr_code codebuf i_short; 
        emit_byte codebuf n;
      end else begin 
        emit_instr_code codebuf i; 
        emit_u16 codebuf x;
      end

    let emit_short_i32_instr codebuf (i_short,i) x = 
      if x >= !!!(-128) && x <= !!!127 then begin
        emit_instr_code codebuf i_short; 
        emit_byte codebuf (??? (if x < 0x0l then x +++ !!!256 else x));
      end else begin 
        emit_instr_code codebuf i; 
        emit_i32 codebuf x;
      end

    let emit_tailness codebuf tl = 
      if tl = Tailcall && !Ilprint.print_tailcall then emit_instr_code codebuf i_tail

    let emit_after_tailcall codebuf tl =
      if tl = Tailcall then emit_instr_code codebuf i_ret

    let emit_volatility codebuf tl = 
      if tl = Volatile then emit_instr_code codebuf i_volatile

    let emit_constrained cenv codebuf env ty = 
      emit_instr_code codebuf i_constrained;
      emit_uncoded codebuf (tdor_as_uncoded (typ_as_tdor cenv env ty))

    let emit_alignment codebuf tl = 
      match tl with 
      | Aligned -> ()
      | Unaligned_1 -> emit_instr_code codebuf i_unaligned; emit_byte codebuf 0x1
      | Unaligned_2 -> emit_instr_code codebuf i_unaligned; emit_byte codebuf 0x2
      | Unaligned_4 -> emit_instr_code codebuf i_unaligned; emit_byte codebuf 0x4


    let rec emit_instr cenv codebuf env instr =
      match instr with
      | si when is_noarg_instr si ->
           emit_instr_code codebuf (encoding_of_noarg_instr si)
      | I_brcmp (cmp,tg1,tg2)  -> 
          record_reqd_brfixup codebuf (List.assoc cmp (Lazy.force brcmp_map), Some (List.assoc cmp (Lazy.force brcmp_smap))) tg1
      | I_br tg -> ()
      | I_seqpoint s ->   emit_seqpoint cenv codebuf s
      | I_leave tg -> record_reqd_brfixup codebuf (i_leave,Some i_leave_s) tg
      | I_call  (tl,mspec,varargs)      -> 
          emit_tailness codebuf tl;
          emit_mspec_instr cenv codebuf env i_call (mspec,varargs);
          emit_after_tailcall codebuf tl
      | I_callvirt      (tl,mspec,varargs)      -> 
          emit_tailness codebuf tl;
          emit_mspec_instr cenv codebuf env i_callvirt (mspec,varargs);
          emit_after_tailcall codebuf tl
      | I_callconstraint        (tl,ty,mspec,varargs)   -> 
          emit_tailness codebuf tl;
          emit_constrained cenv codebuf env ty;
          emit_mspec_instr cenv codebuf env i_callvirt (mspec,varargs);
          emit_after_tailcall codebuf tl
      | I_newobj        (mspec,varargs) -> 
          emit_mspec_instr cenv codebuf env i_newobj (mspec,varargs)
      | I_ldftn mspec   -> 
          emit_mspec_instr cenv codebuf env i_ldftn (mspec,None)
      | I_ldvirtftn     mspec   -> 
          emit_mspec_instr cenv codebuf env i_ldvirtftn (mspec,None)

      | I_calli (tl,callsig,varargs)    -> 
          emit_tailness codebuf tl;
          emit_instr_code codebuf i_calli; 
          emit_uncoded codebuf (uncoded_token tab_StandAloneSig (callsig_as_StandAloneSig_idx cenv env (callsig,varargs)));
          emit_after_tailcall codebuf tl

      | I_ldarg u16 ->  emit_short_u16_instr codebuf (i_ldarg_s,i_ldarg) u16 
      | I_starg u16 ->  emit_short_u16_instr codebuf (i_starg_s,i_starg) u16 
      | I_ldarga u16 ->  emit_short_u16_instr codebuf (i_ldarga_s,i_ldarga) u16 
      | I_ldloc u16 ->  emit_short_u16_instr codebuf (i_ldloc_s,i_ldloc) u16 
      | I_stloc u16 ->  emit_short_u16_instr codebuf (i_stloc_s,i_stloc) u16 
      | I_ldloca u16 ->  emit_short_u16_instr codebuf (i_ldloca_s,i_ldloca) u16 

      | I_cpblk (al,vol)        -> 
          emit_alignment codebuf al; 
          emit_volatility codebuf vol;
          emit_instr_code codebuf i_cpblk
      | I_initblk       (al,vol)        -> 
          emit_alignment codebuf al; 
          emit_volatility codebuf vol;
          emit_instr_code codebuf i_initblk

      | I_arith (AI_ldc (DT_I4, NUM_I4 x)) -> 
          emit_short_i32_instr codebuf (i_ldc_i4_s,i_ldc_i4) x
      | I_arith (AI_ldc (DT_I8, NUM_I8 x)) -> 
          emit_instr_code codebuf i_ldc_i8; 
          emit_i64 codebuf x;
      | I_arith (AI_ldc (dt, NUM_R4 x)) -> 
          emit_instr_code codebuf i_ldc_r4; 
          emit_i32 codebuf (Nums.ieee32_to_bits x)
      | I_arith (AI_ldc (dt, NUM_R8 x)) -> 
          emit_instr_code codebuf i_ldc_r8; 
          emit_i64 codebuf (Nums.ieee64_to_bits x)

      | I_ldind (al,vol,dt)     -> 
          emit_alignment codebuf al; 
          emit_volatility codebuf vol;
          emit_instr_code codebuf 
            (match dt with 
            | DT_I -> i_ldind_i
            | DT_I1  -> i_ldind_i1     
            | DT_I2  -> i_ldind_i2     
            | DT_I4  -> i_ldind_i4     
            | DT_U1  -> i_ldind_u1     
            | DT_U2  -> i_ldind_u2     
            | DT_U4  -> i_ldind_u4     
            | DT_I8  -> i_ldind_i8     
            | DT_R4  -> i_ldind_r4     
            | DT_R8  -> i_ldind_r8     
            | DT_REF  -> i_ldind_ref
            | _ -> failwith "ldind")

      | I_stelem dt     -> 
          emit_instr_code codebuf 
            (match dt with 
            | DT_I | DT_U -> i_stelem_i
            | DT_U1 | DT_I1  -> i_stelem_i1     
            | DT_I2 | DT_U2  -> i_stelem_i2     
            | DT_I4 | DT_U4  -> i_stelem_i4     
            | DT_I8 | DT_U8  -> i_stelem_i8     
            | DT_R4  -> i_stelem_r4     
            | DT_R8  -> i_stelem_r8     
            | DT_REF  -> i_stelem_ref
            | _ -> failwith "stelem")

      | I_ldelem dt     -> 
          emit_instr_code codebuf 
            (match dt with 
            | DT_I -> i_ldelem_i
            | DT_I1  -> i_ldelem_i1     
            | DT_I2  -> i_ldelem_i2     
            | DT_I4  -> i_ldelem_i4     
            | DT_I8  -> i_ldelem_i8     
            | DT_U1  -> i_ldelem_u1     
            | DT_U2  -> i_ldelem_u2     
            | DT_U4  -> i_ldelem_u4     
            | DT_R4  -> i_ldelem_r4     
            | DT_R8  -> i_ldelem_r8     
            | DT_REF  -> i_ldelem_ref
            | _ -> failwith "ldelem")

      | I_stind (al,vol,dt)     -> 
          emit_alignment codebuf al; 
          emit_volatility codebuf vol;
          emit_instr_code codebuf 
            (match dt with 
            | DT_U | DT_I -> i_stind_i
            | DT_U1 | DT_I1  -> i_stind_i1     
            | DT_U2 | DT_I2  -> i_stind_i2     
            | DT_U4 | DT_I4  -> i_stind_i4     
            | DT_U8 | DT_I8  -> i_stind_i8     
            | DT_R4  -> i_stind_r4     
            | DT_R8  -> i_stind_r8     
            | DT_REF  -> i_stind_ref
            | _ -> failwith "stelem")

      | I_switch (labs,dflt)    ->  record_reqd_brfixups codebuf (i_switch,None) labs

      | I_ldfld (al,vol,fspec)  -> 
          emit_alignment codebuf al; 
          emit_volatility codebuf vol;
          emit_fspec_instr cenv codebuf env i_ldfld fspec
      | I_ldflda        fspec   -> 
          emit_fspec_instr cenv codebuf env i_ldflda fspec
      | I_ldsfld        (vol,fspec)     -> 
          emit_volatility codebuf vol;
          emit_fspec_instr cenv codebuf env i_ldsfld fspec
      | I_ldsflda       fspec   -> 
          emit_fspec_instr cenv codebuf env i_ldsflda fspec
      | I_stfld (al,vol,fspec)  -> 
          emit_alignment codebuf al; 
          emit_volatility codebuf vol;
          emit_fspec_instr cenv codebuf env i_stfld fspec
      | I_stsfld        (vol,fspec)     -> 
          emit_volatility codebuf vol;
          emit_fspec_instr cenv codebuf env i_stsfld fspec

      | I_ldtoken  tok  -> 
          emit_instr_code codebuf i_ldtoken;
          emit_uncoded codebuf 
            (match tok with 
            | Token_type typ -> 
                begin match typ_as_tdor cenv env typ with 
                | (tag,idx) when tag = tdor_TypeDef -> uncoded_token tab_TypeDef idx
                | (tag,idx) when tag = tdor_TypeRef -> uncoded_token tab_TypeRef idx
                | (tag,idx) when tag = tdor_TypeSpec -> uncoded_token tab_TypeSpec idx
                | _ -> failwith "?"
                end           
            | Token_method mspec ->
                begin match mspec_as_mdor cenv env (mspec,None) with 
                | (tag,idx) when tag = mdor_MethodDef -> uncoded_token tab_Method idx
                | (tag,idx) when tag = mdor_MemberRef -> uncoded_token tab_MemberRef idx
                | _ -> failwith "?"
                end           

            | Token_field fspec ->
                begin match fspec_as_fdor cenv env fspec with 
                | (tag,idx) when tag -> uncoded_token tab_Field idx
                | (tag,idx)  -> uncoded_token tab_MemberRef idx
                end)
      | I_ldstr s       -> 
          emit_instr_code codebuf i_ldstr;
          record_reqd_stringfixup codebuf (bytes_as_UserString_idx cenv s)

      | I_box  ty       -> emit_typ_instr cenv codebuf env i_box ty
      | I_unbox  ty     -> emit_typ_instr cenv codebuf env i_unbox ty
      | I_unbox_any  ty -> emit_typ_instr cenv codebuf env i_unbox_any ty 

      | I_newarr (shape,ty) -> 
          if (shape = sdshape) then   
            emit_typ_instr cenv codebuf env i_newarr ty
          else
            let rank = rank_of_array_shape shape in 
            let args = Array.to_list (Array.create (??? rank) (cenv.ilg.typ_int32)) in 
            emit_mspecinfo_instr cenv codebuf env i_newobj (".ctor",mk_array_ty(ty,shape),instance_callconv,args,Type_void,None,[])

      | I_stelem_any (shape,ty) -> 
          if (shape = sdshape) then   
            emit_typ_instr cenv codebuf env i_stelem_any ty  
          else 
            let rank = rank_of_array_shape shape in 
            let args = Array.to_list (Array.create (??? rank) (cenv.ilg.typ_int32)) @ [ty] in 
            emit_mspecinfo_instr cenv codebuf env i_call ("Set",mk_array_ty(ty,shape),instance_callconv,args,Type_void,None,[])

      | I_ldelem_any (shape,ty) -> 
          if (shape = sdshape) then   
            emit_typ_instr cenv codebuf env i_ldelem_any ty  
          else 
            let rank = rank_of_array_shape shape in 
            let args = Array.to_list (Array.create (??? rank) (cenv.ilg.typ_int32)) in 
            emit_mspecinfo_instr cenv codebuf env i_call ("Get",mk_array_ty(ty,shape),instance_callconv,args,ty,None,[])

      | I_ldelema  (ro,shape,ty) -> 
          if (ro = ReadonlyAddress) then
            emit_instr_code codebuf i_readonly;
          if (shape = sdshape) then   
            emit_typ_instr cenv codebuf env i_ldelema ty
          else 
            let rank = rank_of_array_shape shape in 
            let args = Array.to_list (Array.create (??? rank) (cenv.ilg.typ_int32)) in 
            emit_mspecinfo_instr cenv codebuf env i_call ("Address",mk_array_ty(ty,shape),instance_callconv,args,Type_byref ty,None,[])

      | I_castclass  ty -> emit_typ_instr cenv codebuf env i_castclass ty
      | I_isinst  ty -> emit_typ_instr cenv codebuf env i_isinst ty
      | I_refanyval  ty -> emit_typ_instr cenv codebuf env i_refanyval ty
      | I_mkrefany  ty -> emit_typ_instr cenv codebuf env i_mkrefany ty
      | I_initobj  ty -> emit_typ_instr cenv codebuf env i_initobj ty
      | I_ldobj  (al,vol,ty) -> 
          emit_alignment codebuf al; 
          emit_volatility codebuf vol;
          emit_typ_instr cenv codebuf env i_ldobj ty
      | I_stobj  (al,vol,ty) -> 
          emit_alignment codebuf al; 
          emit_volatility codebuf vol;
          emit_typ_instr cenv codebuf env i_stobj ty
      | I_cpobj  ty -> emit_typ_instr cenv codebuf env i_cpobj ty
      | I_sizeof  ty -> emit_typ_instr cenv codebuf env i_sizeof ty

    (* ILX: @todo erase me earlier *)
      | I_other e when Ilx.is_ilx_ext_instr e -> 
          begin match (Ilx.dest_ilx_ext_instr e) with 
          |  (Ilx.EI_ldftn_then_call (mr1,(tl,mr2,varargs)))    -> 
              emit_instr cenv codebuf env (I_ldftn mr1);
              emit_instr cenv codebuf env (I_call (tl,mr2,varargs))
          |  (Ilx.EI_ld_instance_ftn_then_newobj (mr1,_,(mr2,varargs))) -> 
              emit_instr cenv codebuf env (I_ldftn mr1);
              emit_instr cenv codebuf env (I_newobj (mr2,varargs))
          | Ilx.EI_stelem_any_erasable (shape,ty) -> 
              emit_instr cenv codebuf env (I_stelem_any(shape,ty))
          | Ilx.EI_ldelem_any_erasable (shape,ty) -> 
              emit_instr cenv codebuf env (I_ldelem_any(shape,ty))
          | Ilx.EI_newarr_erasable (shape, ty) -> 
              emit_instr cenv codebuf env (I_newarr(shape,ty))
          | Ilx.EI_ldlen_multi (n,m)    -> 
              emit_short_i32_instr codebuf (i_ldc_i4_s,i_ldc_i4) m;
              emit_instr cenv codebuf env (mk_normal_call(mk_nongeneric_mspec_in_typ(cenv.ilg.typ_Array, instance_callconv, "GetLength", [(cenv.ilg.typ_int32)], (cenv.ilg.typ_int32))))

          |  _ -> failwith "an ILX instruction cannot be emitted"
          end
      |  _ -> failwith "an IL instruction cannot be emitted"


    let mk_scope_node cenv localSigs (a,b,ls,ch) = 
      if (isNil ls or not cenv.generate_pdb) then ch
      else
        [ { pdbScopeChildren= Array.of_list ch;
            pdbScopeStartOffset=a;
            pdbScopeEndOffset=b;
            pdbScopeLocals=
            Array.of_list
              (List.map
                 (fun x -> { pdbVarName=x.localName;
                             pdbVarSig= (try List.nth localSigs x.localNum with _ -> failwith ("local variable index "^string_of_int x.localNum^"in debug info does not reference a valid local"));
                             pdbVarAttributes= int_to_i32 x.localNum } ) 
                 (List.filter (fun v -> v.localName <> "") ls)) } ]
            
    let rec emit_code cenv localSigs codebuf env (susp,code) = 
      match code with 
      | TryBlock (c,seh) -> 
          commit_susp codebuf susp (unique_entry_of_code c);
          let try_start = Bytebuf.length codebuf.code in 
          let susp,child1,scope1 = emit_code cenv localSigs codebuf env (None,c) in 
          commit_susp_no_dest codebuf susp;
          let try_finish = Bytebuf.length codebuf.code in 
          let exn_branches = 
            begin match seh with 
            | FaultBlock flt -> 
                let handler_start = Bytebuf.length codebuf.code in 
                let susp,child2,scope2 = emit_code cenv localSigs codebuf env (None,flt) in 
                commit_susp_no_dest codebuf susp;
                let handler_finish = Bytebuf.length codebuf.code in 
                [ Some (try_start,(try_finish - try_start),
                        handler_start,(handler_finish - handler_start),
                        FaultClause), 
                  [(child2,scope2)] ]
                  
            | FinallyBlock flt -> 
                let handler_start = Bytebuf.length codebuf.code in 
                let susp,child2,scope2 = emit_code cenv localSigs codebuf env (None,flt) in 
                commit_susp_no_dest codebuf susp;
                let handler_finish = Bytebuf.length codebuf.code in 
                [ Some (try_start,(try_finish - try_start),
                        handler_start,(handler_finish - handler_start),
                        FinallyClause),
                  [(child2,scope2)] ]
                  
            | FilterCatchBlock clauses -> 
                list_mapi 
                  (fun i (flt,ctch) -> 
                    match flt with 
                    | TypeFilter typ ->
                        let handler_start = Bytebuf.length codebuf.code in 
                        let susp,child2,scope2 = emit_code cenv localSigs codebuf env (None,ctch) in 
                        commit_susp_no_dest codebuf susp;
                        let handler_finish = Bytebuf.length codebuf.code in 
                        Some (try_start,(try_finish - try_start),
                              handler_start,(handler_finish - handler_start),
                              TypeFilterClause (tdor_as_uncoded (typ_as_tdor cenv env typ))),
                        [(child2,scope2)]
                    | CodeFilter fltcode -> 
                        
                        let filter_start = Bytebuf.length codebuf.code in 
                        let susp,child2,scope2 = emit_code cenv localSigs codebuf env (None,fltcode) in 
                        commit_susp_no_dest codebuf susp;
                        let handler_start = Bytebuf.length codebuf.code in 
                        let susp,child3,scope3 = emit_code cenv localSigs codebuf env (None,ctch) in 
                        commit_susp_no_dest codebuf susp;
                        let handler_finish = Bytebuf.length codebuf.code in 
                        
                        Some (try_start,
                              (try_finish - try_start),
                              handler_start,
                              (handler_finish - handler_start),
                              FilterClause filter_start),
                        [(child2,scope2); (child3,scope3)])
                  clauses
            end in 
          None,
          Node((None,[child1])::List.map (fun (a,b) -> (a,List.map fst b)) exn_branches), 
          scope1 @ List.concat (List.concat (List.map (fun (a,b) -> List.map snd b) exn_branches))

      | RestrictBlock (_,code2) -> emit_code cenv localSigs codebuf env (susp,code2)
      | GroupBlock (locs,codes) -> 
          let start = Bytebuf.length codebuf.code in 
          let new_susp = ref susp in
          let childseh = ref [] in
          let childscopes = ref [] in
          List.iter
            (fun c -> 
              let susp,seh,scopes = emit_code cenv localSigs codebuf env (!new_susp,c) in 
              new_susp := susp;
              childseh := seh :: !childseh;
              childscopes := !childscopes @ scopes) 
            codes;
          let fin = Bytebuf.length codebuf.code in 
          !new_susp, 
          Node([(None,(List.rev !childseh))]), 
          mk_scope_node cenv localSigs (start,fin,locs,!childscopes)
      | BasicBlock bb ->  
          commit_susp codebuf susp bb.bblockLabel;
          record_avail_brfixup codebuf bb.bblockLabel;
          let instrs = bb.bblockInstrs in 
          for i = 0 to Array.length instrs - 1 do
            emit_instr cenv codebuf env instrs.(i);
          done;
          (fallthrough_of_bblock bb), Tip, []
            
    and br_to_susp codebuf dest = record_reqd_brfixup codebuf (i_br,Some i_br_s) dest
              
    and commit_susp codebuf susp lab = 
      match susp with 
      | Some dest when dest <> lab -> br_to_susp codebuf dest
      | _ -> ()
    and commit_susp_no_dest codebuf susp = 
      match susp with 
      | Some dest -> br_to_susp codebuf dest
      | _ -> ()
     
    (* Flatten the SEH tree *)
    let rec emit_seh_tree codebuf seh_tree = 
      match seh_tree with 
      | Tip -> ()
      | Node clauses -> List.iter (emit_seh_tree2 codebuf) clauses
    and emit_seh_tree2 codebuf (x,childseh) = 
      List.iter (emit_seh_tree codebuf) childseh; (* internal first *)
      begin match x with 
      | None -> () 
      | Some clause -> emit_seh_clause codebuf clause
      end

    let emit_topcode cenv localSigs env nm code = 
     if logging then dprint_endline ("nm = "^nm);
      let codebuf = new_codebuf nm in 
      let final_susp, seh_tree, orig_scopes = 
        emit_code cenv localSigs codebuf env (Some (unique_entry_of_code code),code) in
      (match final_susp with Some dest  -> br_to_susp codebuf dest | _ -> ());
      emit_seh_tree codebuf seh_tree;
      let orig_code = Bytebuf.close codebuf.code in 
      let orig_seh = List.rev codebuf.seh in 
      let orig_reqd_string_fixups = codebuf.reqd_string_fixups_in_method in
      let orig_avail_brfixups = codebuf.avail_brfixups in
      let orig_reqd_brfixups = codebuf.reqd_brfixups in
      let orig_seqpoints = buf_close codebuf.seqpoints in 
      if logging then begin 
        dprint_endline ("length orig_seh = "^string_of_int (List.length orig_seh));
        List.iter
          (fun (st1,sz1,st2,sz2,kind) -> 
            dprint_endline ("st1 = "^string_of_int st1);
            dprint_endline ("sz1 = "^string_of_int sz1);
            dprint_endline ("st2 = "^string_of_int st2);
            dprint_endline ("sz2 = "^string_of_int sz2);) orig_seh;
      end;
      let new_code, new_reqd_string_fixups, new_seh, new_seqpoints, new_scopes = 
        apply_brfixups orig_code orig_seh orig_reqd_string_fixups orig_avail_brfixups orig_reqd_brfixups orig_seqpoints orig_scopes in 
      if logging then begin 
        dprint_endline ("length new_seh = "^string_of_int (List.length new_seh));
        List.iter
          (fun (st1,sz1,st2,sz2,kind) -> 
            dprint_endline ("st1 = "^string_of_int st1);
            dprint_endline ("sz1 = "^string_of_int sz1);
            dprint_endline ("st2 = "^string_of_int st2);
            dprint_endline ("sz2 = "^string_of_int sz2);) new_seh;
      end;
      let rootscope = 
        { pdbScopeChildren= Array.of_list new_scopes;
          pdbScopeStartOffset=0;
          pdbScopeEndOffset=Bytes.length new_code;
          pdbScopeLocals=[| |]; } in 

      new_reqd_string_fixups,new_seh, new_code, new_seqpoints, rootscope


end

(* -------------------------------------------------------------------- 
 * Il.il_method_body --> bytes
 * -------------------------------------------------------------------- *)

let ilmbody_as_intarray mname cenv env il =
  let localSigs = 
    if cenv.generate_pdb then 
      List.map
        (fun l -> 
          let dbgsig_as_StandAloneSig_row cenv s = 
            Row [| Blob (intarray_as_BlobHeap_idx cenv (Array.concat [ [| e_IMAGE_CEE_CS_CALLCONV_FIELD |]; s ])) |] in 
          
          let s = typ_as_intarray cenv env l.localType in 
          ignore (find_or_add_entry (table cenv tab_StandAloneSig) (dbgsig_as_StandAloneSig_row cenv s));
          Bytes.of_intarray s) il.ilLocals 
    else [] in 

  let reqd_string_fixups,seh,code,seqpoints, scopes = Codebuf.emit_topcode cenv localSigs env mname il.ilCode in 
  let code_size = Bytes.length code in 
  let methbuf = Bytebuf.create (code_size * 3) in 
  (* Do we use the tiny format? *)
  if isNil il.ilLocals && il.ilMaxStack <= !!!8 && not il.ilZeroInit  && isNil seh && code_size < 64 then
    (* Use Tiny format *)
    let aligned_code_size = align !!!4 (!!!code_size +++ !!!1) in 
    let code_padding = i32_to_int (aligned_code_size --- (!!!code_size +++ !!!1)) in
    let reqd_string_fixups' = (!!!1,reqd_string_fixups) in 
    Bytebuf.emit_int_as_byte methbuf (??? (!!!code_size <<< 2 ||| e_CorILMethod_TinyFormat));
    Bytebuf.emit_bytes methbuf code;
    bytebuf_emit_pad methbuf code_padding;
    (reqd_string_fixups', Bytebuf.close methbuf), seqpoints, scopes
  else
    (* Use Fat format *)
    let flags = 
      e_CorILMethod_FatFormat |||
      (if seh <> [] then e_CorILMethod_MoreSects else 0x0l) ||| 
      (if il.ilZeroInit then e_CorILMethod_InitLocals else 0x0l) in 
    let localToken = 
      if isNil il.ilLocals then 0x0l else 
      uncoded_token tab_StandAloneSig
        (find_or_add_entry (table cenv tab_StandAloneSig) (local_sig_as_StandAloneSig_row cenv env il.ilLocals)) in 
    let aligned_code_size = align 0x4l !!!code_size in 
    let code_padding = i32_to_int (aligned_code_size --- !!!code_size) in
    
    Bytebuf.emit_int_as_byte methbuf (b0 flags); 
    Bytebuf.emit_int_as_byte methbuf 0x30; (* last four bits record size of fat header in 4 byte chunks - this is always 12 bytes = 3 four word chunks *)
    bytebuf_emit_u16 methbuf (i32_to_u16 il.ilMaxStack);
    Bytebuf.emit_i32 methbuf !!!code_size;
    Bytebuf.emit_i32 methbuf localToken;
    Bytebuf.emit_bytes methbuf code;
    bytebuf_emit_pad methbuf code_padding;

    if seh <> [] then begin
      (* Can we use the small exception handling table format? *)
      let small_size = !!!(List.length seh * 12 + 4) in 
      let can_use_small = 
        small_size <= 0xffl &
        List.for_all
          (fun (st1,sz1,st2,sz2,_) -> 
            st1 <= 0xffff && st2 <= 0xffff && sz1 <= 0xff && sz2 <= 0xff) seh in 
      
      let kind_as_i32 k = 
        match k with 
          FinallyClause -> e_COR_ILEXCEPTION_CLAUSE_FINALLY
        | FaultClause -> e_COR_ILEXCEPTION_CLAUSE_FAULT
        | FilterClause _ -> e_COR_ILEXCEPTION_CLAUSE_FILTER
        | TypeFilterClause _ -> e_COR_ILEXCEPTION_CLAUSE_EXCEPTION in 
      let kind_as_extra_i32 k = 
        match k with 
          FinallyClause |FaultClause -> 0x0l
        | FilterClause i -> !!!i
        | TypeFilterClause uncoded -> uncoded in 
      
      if can_use_small then     begin 
        if logging then dprint_endline ("using small SEH format for method "^mname); 
        Bytebuf.emit_int_as_byte methbuf (??? e_CorILMethod_Sect_EHTable);
        Bytebuf.emit_int_as_byte methbuf (b0 small_size); 
        Bytebuf.emit_int_as_byte methbuf 0x00; 
        Bytebuf.emit_int_as_byte methbuf 0x00;
        List.iter
          (fun (st1,sz1,st2,sz2,kind) -> 
            let k32 = kind_as_i32 kind in 
            Bytebuf.emit_i32_as_u16 methbuf k32; 
            Bytebuf.emit_i32_as_u16 methbuf !!!st1; 
            Bytebuf.emit_int_as_byte methbuf (b0 !!!sz1); 
            Bytebuf.emit_i32_as_u16 methbuf !!!st2; 
            Bytebuf.emit_int_as_byte methbuf (b0 !!!sz2);
            Bytebuf.emit_i32 methbuf (kind_as_extra_i32 kind))
          seh
      end else begin
        if logging  then dprint_endline ("using fat SEH format for method "^mname); 
        let big_size = !!!(List.length seh * 24 + 4) in 
        Bytebuf.emit_int_as_byte methbuf (??? (e_CorILMethod_Sect_EHTable ||| e_CorILMethod_Sect_FatFormat));
        Bytebuf.emit_int_as_byte methbuf (b0 big_size);
        Bytebuf.emit_int_as_byte methbuf (b1 big_size);
        Bytebuf.emit_int_as_byte methbuf (b2 big_size);
        List.iter
          (fun (st1,sz1,st2,sz2,kind) -> 
            let k32 = kind_as_i32 kind in 
            Bytebuf.emit_i32 methbuf k32;
            Bytebuf.emit_i32 methbuf !!!st1;
            Bytebuf.emit_i32 methbuf !!!sz1;
            Bytebuf.emit_i32 methbuf !!!st2;
            Bytebuf.emit_i32 methbuf !!!sz2;
            Bytebuf.emit_i32 methbuf (kind_as_extra_i32 kind))
          seh
      end;
    end;
    
    let reqd_string_fixups' = (!!!12,reqd_string_fixups) in 

    (reqd_string_fixups', Bytebuf.close methbuf), seqpoints, scopes

(* -------------------------------------------------------------------- 
 * Il.field_def --> FieldDef Row
 * -------------------------------------------------------------------- *)

let rec fdef_as_FieldDef_row cenv env fd = 
  let flags = 
    member_access_as_flags fd.fdAccess |||
    (if fd.fdStatic then 0x0010l else 0x0l) |||
    (if fd.fdInitOnly then 0x0020l else 0x0l) |||
    (if fd.fdLiteral then 0x0040l else 0x0l) |||
    (if fd.fdNotSerialized then 0x0080l else 0x0l) |||
    (if fd.fdSpecialName then 0x0200l else 0x0l) |||
    (if fd.fdSpecialName then 0x0400l else 0x0l) ||| (* @todo: RTSpecialName *)
    (if (fd.fdInit <> None) then 0x8000l else 0x0l) |||
    (if (fd.fdMarshal <> None) then 0x1000l else 0x0l) |||
    (if (fd.fdData <> None) then 0x0100l else 0x0l) in 
  Row [| UShort (i32_to_u16 flags); 
        String (string_as_StringHeap_idx cenv fd.fdName);
        Blob ( fdef_sig_as_BlobHeap_idx cenv env fd ); |]

and fdef_sig_as_BlobHeap_idx cenv env fd = 
  intarray_as_BlobHeap_idx cenv ([| e_IMAGE_CEE_CS_CALLCONV_FIELD |] @@ 
                         typ_as_intarray cenv env fd.fdType)

and fdef_pass3 cenv env fd = 
  let fidx = add_entry (table cenv tab_Field) (fdef_as_FieldDef_row cenv env fd) in 
  custom_attrs_pass3 cenv (hca_FieldDef,fidx) fd.fdCustomAttrs;
  (* Write FieldRVA table - fixups into data section done later *)
  begin match fd.fdData with 
  | None -> () 
  | Some b -> 
      if logging then dprint_endline ("field data: size = "^string_of_int (Bytes.length b));
      let offs = Bytebuf.length cenv.data in 
      Bytebuf.emit_bytes cenv.data b;
      ignore (add_entry (table cenv tab_FieldRVA) 
                (Row [| Data (offs, false); 
                       SimpleIndex (tab_Field,fidx) |]))
  end;
  (* Write FieldMarshal table *)
  begin match fd.fdMarshal with 
  | None -> ()
  | Some ntyp -> 
      ignore (add_entry (table cenv tab_FieldMarshal) 
                (Row [| HasFieldMarshal (hfm_FieldDef, fidx);
                       Blob (native_typ_as_BlobHeap_idx cenv ntyp) |]))
  end;
  (* Write Contant table *)
  begin match fd.fdInit with 
  | None -> ()
  | Some i -> 
      ignore (add_entry (table cenv tab_Constant) 
                (Row [| field_init_as_flags i;
                       HasConstant (hc_FieldDef, fidx);
                       Blob (field_init_as_BlobHeap_idx cenv env i) |]))
  end;
  (* Write FieldLayout table *)
  begin match fd.fdOffset with 
  | None -> ()
  | Some offset -> 
      ignore (add_entry (table cenv tab_FieldLayout) 
                (Row [| ULong offset;
                       SimpleIndex (tab_Field, fidx) |]))
  end

                
(* -------------------------------------------------------------------- 
 * Il.genparam --> GenericParam Row
 * -------------------------------------------------------------------- *)

let rec gparam_as_GenericParam_row cenv env idx owner gp = 
  let flags = 
    (match  gp.gpVariance with 
       | NonVariant -> 0x0000l
       | CoVariant -> 0x0001l
       | ContraVariant -> 0x0002l) |||
    (if gp.gpReferenceTypeConstraint then 0x0004l else 0x0000l) |||
    (if gp.gpNotNullableValueTypeConstraint then 0x0008l else 0x0000l) |||
    (if gp.gpDefaultConstructorConstraint then 0x0010l else 0x0000l) in 

  let mdVersionMajor,mdVersionMinor = metadataSchemaVersionSupportedByCLRVersion cenv.desiredMetadataVersion in 
  if (mdVersionMajor = 1) then 
    Row [|UShort (int_to_u16 idx); 
          UShort (i32_to_u16 flags);   
          TypeOrMethodDef (fst owner, snd owner);
          String (string_as_StringHeap_idx cenv gp.gpName);
          TypeDefOrRefOrSpec (tdor_TypeDef, 0); (* empty kind field in deprecated metadata *) |]
  else
    Row [|UShort (int_to_u16 idx); 
          UShort (i32_to_u16 flags);   
          TypeOrMethodDef (fst owner, snd owner);
          String (string_as_StringHeap_idx cenv gp.gpName) |]

and typ_as_GenericParamConstraint_row cenv env gpidx ty = 
  let tdor_tag,tdor_row = typ_as_tdor cenv env ty in 
  Row [|SimpleIndex (tab_GenericParam, gpidx);
        TypeDefOrRefOrSpec (tdor_tag,tdor_row) |]

and gparam_constraint_pass4 cenv env gpidx ty =
  ignore (add_entry (table cenv tab_GenericParamConstraint) (typ_as_GenericParamConstraint_row cenv env gpidx ty))

and gparam_pass3 cenv env idx owner gp = 
  ignore (add_entry (table cenv tab_GenericParam) (gparam_as_GenericParam_row cenv env idx owner gp))

and gparam_pass4 cenv env idx owner gp = 
  let gpidx = find_or_add_entry (table cenv tab_GenericParam) (gparam_as_GenericParam_row cenv env idx owner gp) in
  List.iter (gparam_constraint_pass4 cenv env gpidx) gp.gpConstraints

(* -------------------------------------------------------------------- 
 * Il.param and Il.return --> Param Row
 * -------------------------------------------------------------------- *)

let rec param_as_Param_row cenv env seq param = 
  let flags = 
    (if  param.paramIn then 0x0001l else 0x0000l) |||
    (if  param.paramOut then 0x0002l else 0x0000l) |||
    (if  param.paramOptional then 0x0010l else 0x0000l) |||
    (if param.paramDefault <> None then 0x1000l else 0x0000l) |||
    (if param.paramMarshal <> None then 0x2000l else 0x0000l) in 
  
  Row [| UShort (i32_to_u16 flags); 
        UShort (int_to_u16 seq); 
        String (string_option_as_StringHeap_idx cenv param.paramName) |]  

and param_pass3 cenv env seq param = 
  if param.paramIn=false && param.paramOut=false && param.paramOptional=false && isNone param.paramDefault && isNone param.paramName && isNone param.paramMarshal 
  then ()
  else    
    let pidx = add_entry (table cenv tab_Param) (param_as_Param_row cenv env seq param) in 
    custom_attrs_pass3 cenv (hca_ParamDef,pidx) param.paramCustomAttrs;
    (* Write FieldRVA table - fixups into data section done later *)
    begin match param.paramMarshal with 
    | None -> ()
    | Some ntyp -> 
        ignore (add_entry (table cenv tab_FieldMarshal) 
                (Row [| HasFieldMarshal (hfm_ParamDef, pidx);
                       Blob (native_typ_as_BlobHeap_idx cenv ntyp) |]))
    end

(*  paramDefault: field_init option;  (* -- Optional parameter *)*)

let return_as_Param_row cenv env returnv = 
  let flags = (if returnv.returnMarshal <> None then 0x2000l else 0x0000l) in 
  Row [| UShort (i32_to_u16 flags); 
        UShort (u16_zero); (* sequence num. *)
        String 0 |]  

let return_pass3 cenv env returnv = 
  if isNone returnv.returnMarshal && isNil (dest_custom_attrs returnv.returnCustomAttrs)
  then ()
  else    
    let pidx = add_entry (table cenv tab_Param) (return_as_Param_row cenv env returnv) in 
    custom_attrs_pass3 cenv (hca_ParamDef,pidx) returnv.returnCustomAttrs;
    begin match returnv.returnMarshal with 
    | None -> ()
    | Some ntyp -> 
        ignore (add_entry (table cenv tab_FieldMarshal)   
                (Row [| HasFieldMarshal (hfm_ParamDef, pidx);
                       Blob (native_typ_as_BlobHeap_idx cenv ntyp) |]))
    end

(* -------------------------------------------------------------------- 
 * Il.method_def --> MethodDef Row
 * -------------------------------------------------------------------- *)

let mdef_sig_as_intarray cenv env mdef = 
  Array.concat 
    ([ [| callconv_as_byte (List.length mdef.mdGenericParams) mdef.mdCallconv |] ] @
     (if List.length mdef.mdGenericParams > 0 then [ z_u32 (int_to_i32 (List.length mdef.mdGenericParams)) ] else []) @
     [ z_u32 (int_to_i32 (List.length mdef.mdParams));
       typ_as_intarray cenv env (typ_of_return mdef.mdReturn);
       Array.concat (List.map (typ_of_param >> typ_as_intarray cenv env) mdef.mdParams) ])

let mdef_sig_as_BlobHeap_idx cenv env mdef = 
  intarray_as_BlobHeap_idx cenv (mdef_sig_as_intarray cenv env mdef)

let mdef_as_MethodDef_row cenv env midx md = 
  let flags = 
    member_access_as_flags md.mdAccess |||
    (if (match md.mdKind with
      | MethodKind_static | MethodKind_cctor -> true
      | _ -> false) then 0x0010l else 0x0l) |||
    (if (match md.mdKind with MethodKind_virtual vinfo -> vinfo.virtFinal | _ -> false) then 0x0020l else 0x0l) |||
    (if (match md.mdKind with MethodKind_virtual _ -> true | _ -> false) then 0x0040l else 0x0l) |||
    (if md.mdHideBySig then 0x0080l else 0x0l) |||
    (if (match md.mdKind with MethodKind_virtual vinfo -> vinfo.virtStrict | _ -> false) then 0x0200l else 0x0l) |||
    (if (match md.mdKind with MethodKind_virtual vinfo -> vinfo.virtNewslot | _ -> false) then 0x0100l else 0x0l) |||
    (if (match md.mdKind with MethodKind_virtual vinfo -> vinfo.virtAbstract | _ -> false) then 0x0400l else 0x0l) |||
    (if md.mdSpecialName then 0x0800l else 0x0l) |||
    (if (match dest_mbody md.mdBody with MethodBody_pinvoke _ -> true | _ -> false) then 0x2000l else 0x0l) |||
    (if md.mdUnmanagedExport then 0x0008l else 0x0l) |||
    (if 
      (match md.mdKind with
      | MethodKind_ctor | MethodKind_cctor -> true 
      | _ -> false) then 0x1000l else 0x0l) ||| (* RTSpecialName *)
    (if md.mdReqSecObj then 0x8000l else 0x0l) |||
    (if md.mdHasSecurity or dest_security_decls (md.mdSecurityDecls) <> [] then 0x4000l else 0x0l) in 
  let implflags = 
    (match  md.mdCodeKind with 
    | MethodCodeKind_native -> 0x0001l
    | MethodCodeKind_runtime -> 0x0003l
    | MethodCodeKind_il  -> 0x0000l) |||
    (if md.mdInternalCall then 0x1000l else 0x0000l) |||
    (if md.mdManaged then 0x0000l else 0x0004l) |||
    (if md.mdForwardRef then 0x0010l else 0x0000l) |||
    (if md.mdPreserveSig then 0x0080l else 0x0000l) |||
    (if md.mdSynchronized then 0x0020l else 0x0000l) |||
    (if md.mdMustRun then 0x0040l else 0x0000l) |||
    (if (match dest_mbody md.mdBody with MethodBody_il il -> il.ilNoInlining | _ -> false) then 0x0008l else 0x0000l) in

  if md.mdEntrypoint then 
    if cenv.entrypoint <> None then failwith "duplicate entrypoint"
    else cenv.entrypoint <- Some (true, midx);
  let code_addr = 
    (match dest_mbody md.mdBody with 
    | MethodBody_il ilmbody -> 
        let addr = cenv.next_code_addr in 
        if logging then dprint_endline ("start of code = "^Int32.to_string addr); 
        let code, seqpoints, rootscope = (ilmbody_as_intarray md.mdName cenv env ilmbody) in 

        (* Now record the PDB record for this method - we write this out later. *)
        begin if cenv.generate_pdb then 
          buf_emit_one cenv.pdbinfo  
            { pdbMethToken=uncoded_token tab_Method midx;
              pdbMethParams= [| |]; (* @todo *)
              pdbMethRootScope = rootscope;
              pdbMethRange=  
                begin match ilmbody.ilSource with 
                | Some m  when cenv.generate_pdb -> 
                    let doc = (find_or_add_entry cenv.pdbdocuments m.sourceDocument) - 1 in  (* table indexes are 1-based, document array indexes are 0-based *)

                    Some ({ pdbLocDocument=doc;
                            pdbLocLine=m.sourceLine;
                            pdbLocColumn=m.sourceColumn; },
                          { pdbLocDocument=doc;
                            pdbLocLine=m.sourceEndLine;
                            pdbLocColumn=m.sourceEndColumn; })
                | _ -> None
                end;
              pdbMethSequencePoints=seqpoints; }
        end;
       
        add_code cenv code;
        addr 
    | MethodBody_native -> 
        failwith "cannot write body of native method - Abstract IL cannot roundtrip mixed native/managed binaries";
    | _  -> 0x0000l) in 

  Row [| ULong  code_addr ; 
        UShort (i32_to_u16 implflags); 
        UShort (i32_to_u16 flags); 
        String (string_as_StringHeap_idx cenv md.mdName); 
        Blob (mdef_sig_as_BlobHeap_idx cenv env md); 
        SimpleIndex(tab_Param,(table cenv tab_Param).count + 1) |]  

let method_impl_pass3 cenv env tgparams tidx mimpl =
  let midx_tag, midx_row = mspec_as_mdef cenv env (mimpl.mimplOverrideBy,None) in 
  let midx2_tag, midx2_row = ospec_as_mdor cenv env mimpl.mimplOverrides in 
  ignore (add_entry (table cenv tab_MethodImpl)
            (Row [| SimpleIndex (tab_TypeDef, tidx);
                  MethodDefOrRef (midx_tag, midx_row);
                  MethodDefOrRef (midx2_tag, midx2_row) |]))
    
let mdef_pass3 cenv env tidx md = 
(*  dprint_endline ("pass3: "^md.mdName);*)
  let midx = mdef_as_MethodDef_idx cenv tidx md in 
  let idx2 = add_entry (table cenv tab_Method) (mdef_as_MethodDef_row cenv env midx md) in 
  if midx <> idx2 then failwith "index of method def on pass 3 does not match index on pass 2";
  return_pass3 cenv env md.mdReturn;  
  list_iteri (fun n param -> param_pass3 cenv env (n+1) param) md.mdParams;
  custom_attrs_pass3 cenv (hca_MethodDef,midx) md.mdCustomAttrs;
  security_decls_pass3 cenv (hds_MethodDef,midx) (dest_security_decls md.mdSecurityDecls);
  list_iteri (fun n gp -> gparam_pass3 cenv env n (tomd_MethodDef, midx) gp) md.mdGenericParams;
  begin match dest_mbody md.mdBody with 
  | MethodBody_pinvoke attr ->
      let flags = 
        begin match attr.pinvokeCallconv with 
        | PInvokeCallConvNone ->     0x0000l
        | PInvokeCallConvCdecl ->    0x0200l
        | PInvokeCallConvStdcall ->  0x0300l
        | PInvokeCallConvThiscall -> 0x0400l
        | PInvokeCallConvFastcall -> 0x0500l
        | PInvokeCallConvWinapi ->   0x0100l
        end |||
        begin match attr.pinvokeEncoding with 
        | PInvokeEncodingNone ->    0x0000l
        | PInvokeEncodingAnsi ->    0x0002l
        | PInvokeEncodingUnicode -> 0x0004l
        | PInvokeEncodingAuto ->    0x0006l
        end |||
        begin match attr.pinvokeBestFit with 
        | PInvokeBestFitUseAssem -> 0x0000l
        | PInvokeBestFitEnabled ->  0x0010l
        | PInvokeBestFitDisabled -> 0x0020l
        end |||
        begin match attr.pinvokeThrowOnUnmappableChar with 
        | PInvokeThrowOnUnmappableCharUseAssem -> 0x0000l
        | PInvokeThrowOnUnmappableCharEnabled ->  0x1000l
        | PInvokeThrowOnUnmappableCharDisabled -> 0x2000l
        end |||
        (if attr.pinvokeNoMangle then 0x0001l else 0x0000l) |||
        (if attr.pinvokeLastErr then 0x0040l else 0x0000l) in 
      ignore (add_entry (table cenv tab_ImplMap)
                (Row [| UShort (i32_to_u16 flags); 
                      MemberForwarded (mf_MethodDef,midx);
                      String (string_as_StringHeap_idx cenv attr.pinvokeName); 
                      SimpleIndex (tab_ModuleRef, modref_as_ModuleRef_idx cenv attr.pinvokeWhere); |]))
  | _ -> ()
  end

let mdef_pass4 cenv env tidx md = 
  let midx = mdef_as_MethodDef_idx cenv tidx md in 
  list_iteri (fun n gp -> gparam_pass4 cenv env n (tomd_MethodDef, midx) gp) md.mdGenericParams

(*       mdExport: (i32 * string option) option; @todo *)
(*      mdVtableEntry: (i32 * i32) option; @todo *)

(* -------------------------------------------------------------------- 
 * Il.property_def --> Property Row + MethodSemantics entries
 * -------------------------------------------------------------------- *)

let property_MethodSemantics_pass3 cenv pidx kind mref =
  let midx = try mref_as_MethodDef_idx cenv mref with Mdef_not_found -> 1 in  
  ignore (add_entry (table cenv tab_MethodSemantics)
            (Row [| UShort (int_to_u16 kind);
                  SimpleIndex (tab_Method,midx);
                  HasSemantics (hs_Property, pidx) |]))
    
let rec property_sig_as_BlobHeap_idx cenv env prop = 
  intarray_as_BlobHeap_idx cenv (property_sig_as_intarray cenv env prop)

and property_sig_as_intarray cenv env prop = 
  let b =  ((hasthis_as_byte prop.propCallconv) lor e_IMAGE_CEE_CS_CALLCONV_PROPERTY) in 
  Array.concat 
    [ [| b |];
      z_u32 (int_to_i32 (List.length prop.propArgs));
      typ_as_intarray cenv env prop.propType;
      Array.concat (List.map (typ_as_intarray cenv env) prop.propArgs) ]

and property_as_Property_row cenv env prop = 
  let flags = 
    (if prop.propSpecialName then 0x0200l else 0x0l) ||| 
    (if  prop.propRTSpecialName then 0x0400l else 0x0l) ||| 
    (if prop.propInit <> None then 0x1000l else 0x0l) in 
  Row [| UShort (i32_to_u16 flags); 
        String (string_as_StringHeap_idx cenv prop.propName); 
        Blob (property_sig_as_BlobHeap_idx cenv env prop); |]  

and property_pass3 cenv env prop = 
  let pidx = add_entry (table cenv tab_Property) (property_as_Property_row cenv env prop) in 
  oiter (property_MethodSemantics_pass3 cenv pidx 0x0001) prop.propSet;
  oiter (property_MethodSemantics_pass3 cenv pidx 0x0002) prop.propGet;
  (* Write Constant table *)
  begin match prop.propInit with 
  | None -> ()
  | Some i -> 
      ignore (add_entry (table cenv tab_Constant) 
                (Row [| field_init_as_flags i;
                       HasConstant (hc_Property, pidx);
                       Blob (field_init_as_BlobHeap_idx cenv env i) |]))
  end;
  custom_attrs_pass3 cenv (hca_Property,pidx) prop.propCustomAttrs

(* -------------------------------------------------------------------- 
 * Il.event_def --> Event Row + MethodSemantics entries
 * -------------------------------------------------------------------- *)

let rec event_MethodSemantics_pass3 cenv eidx kind mref =
  let add_idx = try mref_as_MethodDef_idx cenv mref with Mdef_not_found -> 1 in  
  ignore (add_entry (table cenv tab_MethodSemantics)
            (Row [| UShort (int_to_u16 kind);
                   SimpleIndex (tab_Method,add_idx);
                   HasSemantics (hs_Event, eidx) |]))

and event_as_Event_row cenv env md = 
  let flags = 
    (if md.eventSpecialName then 0x0200l else 0x0l) ||| 
    (if  md.eventRTSpecialName then 0x0400l else 0x0l) in 
  let tdor_tag, tdor_row = optional_typ_as_tdor cenv env md.eventType in 
  Row [| UShort (i32_to_u16 flags); 
        String (string_as_StringHeap_idx cenv md.eventName); 
        TypeDefOrRefOrSpec (tdor_tag,tdor_row) |]

and event_pass3 cenv env md = 
  let eidx = add_entry (table cenv tab_Event) (event_as_Event_row cenv env md) in 
  event_MethodSemantics_pass3 cenv eidx 0x0008 md.eventAddOn (* AddMethod *);
  event_MethodSemantics_pass3 cenv eidx 0x0010 md.eventRemoveOn (* RemoveMethod *);
  oiter (event_MethodSemantics_pass3 cenv eidx 0x0020) md.eventFire  (* Fire *);
  List.iter (event_MethodSemantics_pass3 cenv eidx 0x0004 (* Other *)) md.eventOther;
  custom_attrs_pass3 cenv (hca_Event,eidx) md.eventCustomAttrs


(* -------------------------------------------------------------------- 
 * Il.resource --> generate ...
 * -------------------------------------------------------------------- *)

let rec resource_as_ManifestResource_row cenv r = 
  let data,impl = 
    match r.resourceWhere with
    | Resource_local bf ->
        let b = bf() in 
        (* Embedded managed resources must be word-aligned.  At least I think so - resource format is not specified in ECMA.  But some mscorlib resources appear to be non-aligned - I think it doesn't matter.. *)
        if logging then dprint_endline ("resource data: size = "^string_of_int (Bytes.length b));
        let offs = Bytebuf.length cenv.resources in 
        let aligned_offs = ??? (align 0x8l !!!offs) in 
        let pad = aligned_offs - offs in 
        let resource_size = !!!(Bytes.length b) in 
        if logging then dprint_endline ("resource pad: "^string_of_int pad);
        bytebuf_emit_pad cenv.resources pad;
        Bytebuf.emit_i32 cenv.resources resource_size;
        Bytebuf.emit_bytes cenv.resources b;
        Data (aligned_offs,true),  (i_File, 0) 
    | Resource_file (mref,offs) -> ULong offs, (i_File, modref_as_File_idx cenv mref)
    | Resource_assembly aref -> ULong 0x0l, (i_AssemblyRef, aref_as_AssemblyRef_idx cenv aref) in
  Row [| data; 
        ULong (match r.resourceAccess with Resource_public -> 0x01l | Resource_private -> 0x02l);
        String (string_as_StringHeap_idx cenv r.resourceName);    
        Implementation (fst impl, snd impl); |]

and resource_pass3 cenv r = 
  let idx = find_or_add_entry (table cenv tab_ManifestResource) (resource_as_ManifestResource_row cenv r) in 
  custom_attrs_pass3 cenv (hca_ManifestResource,idx) r.resourceCustomAttrs

(* -------------------------------------------------------------------- 
 * Il.type_def --> generate FieldDef, MethodDef, PropertyDef etc. rows
 * -------------------------------------------------------------------- *)

let rec tdef_pass3 enc cenv td = 
 try
  let env = env_enter_tdef (List.length td.tdGenericParams) in 
  let tidx = idx_alloced_for_tdef cenv (enc,name_of_tdef td) in 
  List.iter (property_pass3 cenv env) (dest_pdefs (properties_of_tdef td));
  List.iter (event_pass3 cenv env) (dest_edefs (events_of_tdef td));
  List.iter (fdef_pass3 cenv env) (dest_fdefs (fields_of_tdef td));
  List.iter (mdef_pass3 cenv env tidx) (dest_mdefs (methods_of_tdef td));
  List.iter (method_impl_pass3 cenv env  (List.length td.tdGenericParams) tidx) (dest_mimpls (mimpls_of_tdef td));
(* ClassLayout entry if needed *)
  begin 
    match layout_of_tdef td with 
    | TypeLayout_auto -> ()
    | TypeLayout_sequential layout | TypeLayout_explicit layout ->  
        if layout.typePack = None && layout.typeSize = None then ()
        else
          ignore (add_entry (table cenv tab_ClassLayout)
                    (Row [| UShort (match layout.typePack with None -> int_to_u16 0x0 | Some p -> p);
                           ULong (match layout.typeSize with None -> 0x0l | Some p -> p);
                           SimpleIndex (tab_TypeDef, tidx) |]))
  end;
  security_decls_pass3 cenv (hds_TypeDef,tidx) (dest_security_decls td.tdSecurityDecls);
  custom_attrs_pass3 cenv (hca_TypeDef,tidx) (custom_attrs_of_tdef td);
  list_iteri (fun n gp -> gparam_pass3 cenv env n (tomd_TypeDef,tidx) gp) (gparams_of_tdef td); 
  tdefs_pass3 (enc@[name_of_tdef td]) cenv (dest_tdefs (nested_of_tdef td));
 with e ->
   dprint_endline ("Error in pass3 for type "^td.tdName^", error: "^Printexc.to_string e);
   (*F# rethrow(); F#*) raise e

and tdefs_pass3 enc cenv tds =
  List.iter (tdef_pass3 enc cenv) tds

(* -------------------------------------------------------------------- 
 * Il.type_def --> generate generic params on MethodDef: ensures
 * GenericParam table is built sorted by owner.
 * -------------------------------------------------------------------- *)

let rec tdef_pass4 enc cenv td = 
 try
  let env = env_enter_tdef (List.length td.tdGenericParams) in 
  let tidx = idx_alloced_for_tdef cenv (enc,name_of_tdef td) in 
  List.iter (mdef_pass4 cenv env tidx) (dest_mdefs (methods_of_tdef td));
  list_iteri (fun n gp -> gparam_pass4 cenv env n (tomd_TypeDef,tidx) gp) (gparams_of_tdef td); 
  tdefs_pass4 (enc@[name_of_tdef td]) cenv (dest_tdefs (nested_of_tdef td));
 with e ->
   dprint_endline ("Error in pass4 for type "^td.tdName^", error: "^Printexc.to_string e);
   (*F# rethrow(); F#*) raise e

and tdefs_pass4 enc cenv tds =
  List.iter (tdef_pass4 enc cenv) tds

(* -------------------------------------------------------------------- 
 * Il.exported_types --> ExportedType table 
 * -------------------------------------------------------------------- *)

let rec nested_exported_type_pass3 cenv cidx ce = 
  let flags =  member_access_as_flags ce.nestedExportedTypeAccess in 
  let nidx = 
    add_entry (table cenv tab_ExportedType) 
      (Row [| ULong flags ; 
             ULong 0x0l;
             String (string_as_StringHeap_idx cenv ce.nestedExportedTypeName); 
             String 0; 
             Implementation (i_ExportedType, cidx) |]) in 
  custom_attrs_pass3 cenv (hca_ExportedType,nidx) ce.nestedExportedTypeCustomAttrs;
  nested_exported_types_pass3 cenv nidx ce.nestedExportedTypeNested

and nested_exported_types_pass3 cenv nidx nce =
  List.iter
    (nested_exported_type_pass3 cenv nidx)
    (dest_nested_exported_types nce)

and exported_type_pass3 cenv ce = 
  let nselem,nelem = name_as_elem_pair cenv ce.exportedTypeName in 
  let flags =  access_as_flags ce.exportedTypeAccess in 
  let flags = if ce.exportedTypeForwarder then 0x00200000l ||| flags else flags in
  let impl = scoref_as_Implementation_elem cenv ce.exportedTypeScope in
  let cidx = 
    add_entry (table cenv tab_ExportedType) 
      (Row [| ULong flags ; 
             ULong 0x0l;
             nelem; 
             nselem; 
             Implementation (fst impl, snd impl); |]) in 
  custom_attrs_pass3 cenv (hca_ExportedType,cidx) ce.exportedTypeCustomAttrs;
  nested_exported_types_pass3 cenv cidx ce.exportedTypeNested

and exported_types_pass3 cenv ce = 
  List.iter (exported_type_pass3 cenv) (dest_exported_types ce);

(* -------------------------------------------------------------------- 
 * Il.manifest --> generate Assembly row
 * -------------------------------------------------------------------- *)

and manifest_as_Assembly_row cenv m = 
  Row [| ULong m.manifestAuxModuleHashAlgorithm;
        UShort (match m.manifestVersion with None -> u16_zero | Some (x,y,z,w) -> x);
        UShort (match m.manifestVersion with None -> u16_zero | Some (x,y,z,w) -> y);
        UShort (match m.manifestVersion with None -> u16_zero | Some (x,y,z,w) -> z);
        UShort (match m.manifestVersion with None -> u16_zero | Some (x,y,z,w) -> w);
        ULong 
          ( begin match m.manifestLongevity with 
            | LongevityUnspecified -> 0x0000l
            | LongevityLibrary -> 0x0002l 
            | LongevityPlatformAppDomain -> 0x0004l
            | LongevityPlatformProcess -> 0x0006l
            | LongevityPlatformSystem -> 0x0008l
            end |||
            (* Setting these causes peverify errors. Hence both ilread and ilwrite ignore them and refuse to set them. *)
            (* ANy debugging customattributes will automatically propagate *)

           (* (if m.manifestJitTracking then !!!0x8000 else !!!0x0) |||  *)
           (* (if m.manifestDisableJitOptimizations then !!!0x4000 else !!!0x0) |||  *)
           (match m.manifestPublicKey with None -> 0x0000l | Some _ -> 0x0001l) ||| 
           0x0000l);
        (match m.manifestPublicKey with None -> Blob 0 | Some x -> Blob (bytes_as_BlobHeap_idx cenv x));
        String (string_as_StringHeap_idx cenv m.manifestName);
        (match m.manifestLocale with None -> String 0 | Some x -> String (string_as_StringHeap_idx cenv x)); |]

and manifest_pass3 cenv m = 
  let aidx = add_entry (table cenv tab_Assembly) (manifest_as_Assembly_row cenv m) in 
  security_decls_pass3 cenv (hds_Assembly,aidx) (dest_security_decls m.manifestSecurityDecls);
  custom_attrs_pass3 cenv (hca_Assembly,aidx) m.manifestCustomAttrs;
  exported_types_pass3 cenv m.manifestExportedTypes;
  (* Record the entrypoint decl if needed. *)
  begin match m.manifestEntrypointElsewhere with
  | Some mref -> 
      if cenv.entrypoint <> None then failwith "duplicate entrypoint"
      else cenv.entrypoint <- Some (false, modref_as_ModuleRef_idx cenv mref);
  | None -> ()
  end

and new_guid modul = 
  let n = absilWriteGetTimeStamp () in
  let m = !!!(Hashtbl.hash n) in
  let m2 = !!!(Hashtbl.hash modul.modulName) in
  [| b0 m; b1 m; b2 m; b3 m; b0 m2; b1 m2; b2 m2; b3 m2; 0xa7; 0x45; 0x03; 0x83; b0 n; b1 n; b2 n; b3 n |]

and modul_as_Module_row cenv modul = 
  Row [| UShort (int_to_u16 0x00); 
        String (string_as_StringHeap_idx cenv modul.modulName); 
        Guid (guid_as_GuidHeap_idx cenv (new_guid modul)); 
        Guid 0; 
        Guid 0 |]


let row_elem_compare e1 e2 = 
  match e1,e2 with 
  | SimpleIndex (Table tab1,n1), SimpleIndex(Table tab2,n2) -> 
      let c1 = compare n1 n2 in 
      if c1 <> 0 then c1 else compare tab1 tab2 
  | TypeDefOrRefOrSpec(TypeDefOrRefOrSpecTag tag1,n1),
      TypeDefOrRefOrSpec(TypeDefOrRefOrSpecTag tag2,n2)
  | TypeOrMethodDef(TypeOrMethodDefTag tag1,n1),
      TypeOrMethodDef(TypeOrMethodDefTag tag2,n2)
  | HasConstant (HasConstantTag tag1,n1),
      HasConstant (HasConstantTag tag2,n2) 
  | HasCustomAttribute (HasCustomAttributeTag tag1,n1),
      HasCustomAttribute (HasCustomAttributeTag tag2,n2) 
  | HasFieldMarshal (HasFieldMarshalTag tag1,n1),
      HasFieldMarshal (HasFieldMarshalTag tag2,n2) 
  | HasDeclSecurity (HasDeclSecurityTag tag1,n1),
      HasDeclSecurity (HasDeclSecurityTag tag2,n2)
  | MemberRefParent (MemberRefParentTag tag1,n1),
      MemberRefParent (MemberRefParentTag tag2,n2) 
  | HasSemantics (HasSemanticsTag tag1,n1),
      HasSemantics (HasSemanticsTag tag2,n2) 
  | MethodDefOrRef (MethodDefOrRefTag tag1,n1),
      MethodDefOrRef (MethodDefOrRefTag tag2,n2) 
  | MemberForwarded (MemberForwardedTag tag1,n1),
      MemberForwarded (MemberForwardedTag tag2,n2)
  | Implementation (ImplementationTag tag1,n1),
      Implementation (ImplementationTag tag2,n2)
  | CustomAttributeType (CustomAttributeTypeTag tag1,n1),
      CustomAttributeType (CustomAttributeTypeTag tag2,n2) 
  | ResolutionScope (ResolutionScopeTag tag1,n1),
      ResolutionScope (ResolutionScopeTag tag2,n2) -> 
        let c1 = compare n1 n2 in 
        if c1 <> 0 then c1 else compare tag1 tag2  
  | ULong _,ULong _ 
  | UShort _, UShort _ 
  | Guid _,Guid _ 
  | Blob _, Blob _
  | String _, String _
  | Data _,Data _ -> failwith "should not have to sort tables on this element"
  | _ -> failwith "sorting on column where two rows have different kinds of element in this column" 

let sort_rows tab rows = 
  if List.mem_assoc tab sorted_table_info then
    let col = List.assoc tab sorted_table_info in 
    List.stable_sort (fun (Row r1) (Row r2) -> row_elem_compare r1.(col) r2.(col)) rows
  else 
    rows

let gen_module cenv modul = 
  let midx = add_entry (table cenv tab_Module) (modul_as_Module_row cenv modul) in 
  List.iter (resource_pass3 cenv) (dest_resources modul.modulResources); 
  let tds = dest_tdefs_with_toplevel_first cenv.ilg modul.modulTypeDefs in 
  reportTime "Module Generation Preperation";
  tdefs_pass1 [] cenv tds;
  reportTime "Module Generation Pass 1";
  tdefs_pass2 0 [] cenv tds;
  reportTime "Module Generation Pass 2";
  (match modul.modulManifest with None -> () | Some m -> manifest_pass3 cenv m);
  tdefs_pass3 [] cenv tds;
  reportTime "Module Generation Pass 3";
  custom_attrs_pass3 cenv (hca_Module,midx) modul.modulCustomAttrs;
  (* GenericParam is the only sorted table indexed by columns in other tables (GenericParamConstraint). *)
  (* Hence we need to sort it before we emit any entries in GenericParamConstraint. *)
  (* Note this mutates the rows in a table.  'set_rows_of_table' clears *) 
  (* the key --> index map since it is no longer valid *)
  set_rows_of_table cenv.tables.(tag_of_table tab_GenericParam) (sort_rows tab_GenericParam (get_tbl (table cenv tab_GenericParam)));
  tdefs_pass4 [] cenv tds;
  reportTime "Module Generation Pass 4"

let gen_il manager reqd_data_fixups (desiredMetadataVersion,generate_pdb,mscorlib)  (m : Il.modul) cil_addr =
  let is_dll = m.modulDLL in 
  if logging then dprint_endline ("cil_addr = "^Int32.to_string cil_addr);

  let  cenvHole = ref None in 
  let the = function Some x -> x | None -> failwith "initialization hole failure" in
  let cenv = 
    { mscorlib=mscorlib;
      ilg = mk_MscorlibRefs mscorlib None; (* assumes mscorlib is Scope_assembly _ scope_ref *)
      desiredMetadataVersion=desiredMetadataVersion;
      reqd_data_fixups= reqd_data_fixups;
      reqd_string_fixups = [];
      code_chunks=Bytebuf.create 40000;
      next_code_addr = cil_addr;
      data = Bytebuf.create 200;
      resources = Bytebuf.create 200;
      tables= Array.init 64 (fun i -> new_tbl ("row table "^string_of_int i));
      pdbdocuments=new_tbl "pdbdocs";
      pdbinfo=new_buf "pdb method infos" 200     { pdbMethToken= 0l;
                                                   pdbMethParams=[| |];
                                                   pdbMethRootScope=
                                                   { pdbScopeChildren= [| |]; 
                                                     pdbScopeStartOffset=0;
                                                     pdbScopeEndOffset=0;
                                                     pdbScopeLocals=[| |];  };
                                                   pdbMethRange=None;
                                                   pdbMethSequencePoints=[| |] };
      fieldDefs= new_tbl "field defs";
      methodDefs = new_tbl "method defs";
      propertyDefs = new_tbl "property defs";
      eventDefs = new_tbl "event defs";
      typeDefs = new_tbl "type defs";
      guids=new_tbl "guids";
      entrypoint=None;
      generate_pdb=generate_pdb;
      blobs= new_tbl "blobs";
      strings= new_tbl "strings"; 
      userStrings= new_tbl "user strings"; 
      tref_as_TypeRef_idx_memoized = memoize (fun x -> tref_as_TypeRef_idx_unmemoized (the !cenvHole) x);
      typ_as_TypeSpec_idx_memoized = memoize (fun x -> typ_as_TypeSpec_idx_unmemoized (the !cenvHole) x);
      mspec_as_uncoded_memoized =    
       (* (match manager with 
        | None -> (fun f -> f) 
        | Some m -> memoize_on (fun (env,(mspec,va)) -> env,mspec_idx m mspec,va)) *)
          (fun x -> mspec_as_uncoded_unmemoized    (the !cenvHole) x);
     (* Memoizing these two doesn't seem to give any pay back, indeed a loss *)
     (* I don't quite understand why - the default hash function may be poor *)
      typ_as_tdor_memoized =         (* memoize *) (fun x -> typ_as_tdor_unmemoized         (the !cenvHole) x);
      mrefinfo_as_mdor_memoized =    (* memoize *) (fun x -> mrefinfo_as_mdor_unmemoized    (the !cenvHole) x); } in
  cenvHole := Some cenv;
     

  (* Now the main compilation step *)
  gen_module cenv  m;

  (* Fetch out some of the results  *)
  let eptoken = 
    match cenv.entrypoint with 
    | Some (ep_here,tok) -> 
        if logging then dprint_endline ("ep idx is "^string_of_int tok);
        uncoded_token (if ep_here then tab_Method else tab_File) tok 
    | None -> 
        if not is_dll then dprint_endline "warning: no entrypoint specified in executable binary";
        0x0l in 

  let pdb_data = 
    { pdbEntrypoint= (if is_dll then None else Some eptoken);
      pdbDocuments = Array.of_list (get_tbl cenv.pdbdocuments);
      pdbMethods= buf_close cenv.pdbinfo } in

  let tidx_for_nested_tdef (tds, td) =
    idx_alloced_for_tdef cenv (List.map name_of_tdef tds, name_of_tdef td) in

  let strings =     Array.map Bytes.string_as_utf8_bytes_null_terminated (Array.of_list (get_tbl cenv.strings)) in 
  let userStrings = Array.of_list (get_tbl cenv.userStrings) in 
  let blobs =       Array.of_list (get_tbl cenv.blobs) in 
  let guids =       Array.of_list (get_tbl cenv.guids) in 
  let tables =      Array.map get_tbl cenv.tables in 
  let code =        get_code cenv in 
  (* turn idx tbls into token maps *)
  let mappings =
   { tdefMap = (fun t ->
      uncoded_token tab_TypeDef (tidx_for_nested_tdef t));
     fdefMap = (fun t fd ->
      let tidx = tidx_for_nested_tdef t in
      uncoded_token tab_Field (fdef_as_FieldDef_idx cenv tidx fd));
     mdefMap = (fun t md ->
      let tidx = tidx_for_nested_tdef t in
      uncoded_token tab_Method (mdkey_as_MethodDef_idx cenv (key_for_mdef tidx md)));
     propertyMap = (fun t pd ->
      let tidx = tidx_for_nested_tdef t in
      uncoded_token tab_Property (tbl_find cenv.propertyDefs (key_for_property tidx pd)));
     eventMap = (fun t ed ->
      let tidx = tidx_for_nested_tdef t in
      uncoded_token tab_Event (tbl_find cenv.eventDefs (EventKey (tidx, ed.eventName)))) } in 
  reportTime "Finalize Module Generation Results";
  (* New return the results *)
  strings,
  userStrings,
  blobs,
  guids,
  tables,
  eptoken,
  code,
  cenv.reqd_string_fixups,
  Bytebuf.close cenv.data,
  Bytebuf.close cenv.resources,
  pdb_data,
  mappings


(*=====================================================================
 * TABLES+BLOBS --> PHYSICAL METADATA+BLOBS
 *=====================================================================*)

type chunk = 
    { size: int32; 
      addr: int32 }

let chunk sz next = ({addr=next; size=sz},next +++ sz) 
let nochunk next = ({addr= 0x0l;size= 0x0l; } ,next)

let count f arr = 
  Array.fold_left (fun x y -> x +++ f y) 0x0l arr 

let write_binary_il manager (generate_pdb,desiredMetadataVersion,compileTimeVersion,mscorlib) modul cil_addr = 

  let is_dll = modul.modulDLL in 
  (* When we know the real RVAs of the data section we fixup the references for the FieldRVA table. *)
  (* These references are stored as offsets into the metadata we return from this function *)
  let reqd_data_fixups = ref [] in 

  if logging then dprint_endline ("calling absilWriteGetMetadataVersion()");

  let next = cil_addr in 

  let strings,userStrings,blobs,guids,tables,eptoken,code,reqd_string_fixups,data,resources,pdb_data,mappings = 
    gen_il manager reqd_data_fixups (desiredMetadataVersion,generate_pdb,mscorlib) modul cil_addr in

  reportTime "Generated Tables and Code";
  let table_size (Table idx) = List.length tables.(idx) in

  (* Compute a minimum version if generics were present: we give warnings if *)
  (* the version is not sufficient to support the constructs being emitted *)
  let minVersion = 
    if table_size tab_GenericParam > 0 or
      table_size tab_MethodSpec > 0 or
      table_size tab_GenericParamConstraint > 0 
    then parse_version ("2.0.0.0")  (* Whidbey Minumum *)
    else parse_version ("1.0.3705.0") in 

  (* Entrypoint is coded as an uncoded token *)
  if logging then dprint_endline ("ep token is "^Int32.to_string eptoken);

 (* Now place the code *)  
  let code_size = !!!(Bytes.length code) in
  let aligned_code_size = align !!!0x4 code_size in 
  let codep,next = chunk code_size next in 
  let code_padding = Array.create (i32_to_int (aligned_code_size --- code_size)) 0x0 in
  let code_paddingp,next = chunk !!!(Array.length code_padding) next in 

  if logging then dprint_endline ("codep.size = "^Int32.to_string codep.size);

 (* Now layout the chunks of metadata and IL *)  
  let metadata_header_startp,next = chunk !!!0x10 next in 

  if logging then dprint_endline ("metadata_header_startp.addr = "^Int32.to_string metadata_header_startp.addr);

  let num_streams = !!!0x05 in 
  let num_tables = count (fun rows -> if rows <> [] then 0l else !!!1) tables in 

  
  let actualVersion = Il.version_max desiredMetadataVersion minVersion in 
  if (Il.version_compare desiredMetadataVersion minVersion < 0) then begin
    dprint_endline ("*** Warning, your binary requires .NET metadata version "^version_to_string minVersion^" or later.");
    dprint_endline ("You are compiling using .NET metadata version "^version_to_string compileTimeVersion^".");
    if version_compare desiredMetadataVersion compileTimeVersion <> 0 then 
      dprint_endline ("You requested to produce a binary using .NET metadata version "^version_to_string desiredMetadataVersion^".");
    dprint_endline ("The actual binary produced will require .NET metadata version "^version_to_string actualVersion^" or later.");
  end;

  let (mdtable_version_major, mdtable_version_minor) = metadataSchemaVersionSupportedByCLRVersion actualVersion in

(*   let version = string_as_utf8_intarray ("v"^(Il.version_to_string actualVersion)) in  *)
  let version = 
    let (a,b,c,d) = actualVersion in 
    string_as_utf8_intarray (Printf.sprintf "v%d.%d.%d" (u16_to_int a) (u16_to_int b) (u16_to_int c)) in


  if logging then dprint_endline ("called absilWriteGetMetadataVersion()");
  let padded_version_length = align !!!0x4 !!!(Array.length version) in 

  (* Most addresses after this point are measured from the MD root *)
  (* Switch to md-rooted addresses *)
  let next = metadata_header_startp.size in 
  let metadata_header_versionp,next = chunk padded_version_length next in 
  let metadata_header_endp,next = chunk !!!0x04 next in 
  let tables_stream_headerp,next = chunk (!!!0x08 +++ (align !!!4 (!!!(String.length "#~") +++ !!!0x01))) next in 
  let strings_stream_headerp,next = chunk (!!!0x08 +++ (align !!!4 (!!!(String.length "#Strings") +++ !!!0x01))) next in 
  let userStrings_stream_headerp,next = chunk (!!!0x08 +++ (align !!!4 (!!!(String.length "#US") +++ !!!0x01))) next in 
  let guids_stream_headerp,next = chunk (!!!0x08 +++ (align !!!4 (!!!(String.length "#GUID") +++ !!!0x01))) next in 
  let blobs_stream_headerp,next = chunk (!!!0x08 +++ (align !!!4 (!!!(String.length "#Blob") +++ !!!0x01))) next in 

  let tables_stream_start = next in 

  let strings_stream_unpadded_size = count (fun s -> !!!(Bytes.length s)) strings +++ !!!1 in 
  let strings_stream_padded_size = align !!!4 strings_stream_unpadded_size in 
  
  let userStrings_stream_unpadded_size = count (fun s -> let n = !!!(Bytes.length s) +++ !!!1 in n +++ z_u32_size n) userStrings +++ !!!1 in 
  let userStrings_stream_padded_size = align !!!4 userStrings_stream_unpadded_size in 
  
  let guids_stream_unpadded_size = !!!(Array.length guids) *** !!!0x10 in 
  let guids_stream_padded_size = align !!!4 guids_stream_unpadded_size in 
  
  let blobs_stream_unpadded_size = count (fun blob -> let n = !!!(Bytes.length blob) in n +++ z_u32_size n) blobs +++ !!!1 in 
  let blobs_stream_padded_size = align !!!4 blobs_stream_unpadded_size in 

  let guids_big = guids_stream_padded_size >= !!!0x10000 in 
  let strings_big = strings_stream_padded_size >= !!!0x10000 in 
  let blobs_big = blobs_stream_padded_size >= !!!0x10000 in 

  (* 64bit bitvector indicating which tables are in the metadata. *)
  let (valid1,valid2),_ = 
    Array.fold_left 
      (fun ((valid1,valid2) as valid,n) rows -> 
        let valid = 
          if isNil rows then valid else
          ( (if n < 32 then  valid1 ||| (!!!1 <<< n     ) else valid1),
            (if n >= 32 then valid2 ||| (!!!1 <<< (n-32)) else valid2) ) in
         (valid,n+1))
      ((0l,0l), 0)
      tables in 
  (* 64bit bitvector indicating which tables are sorted. *)
  (* Constant - @todo: make symbolic! compute from sorted table info! *)
  let sorted1 = !!!0x3301fa00 in
  let sorted2 = 
    (* If there are any generic parameters in the binary we're emitting then mark that *)
    (* table as sorted, otherwise don't.  This maximizes the number of assemblies we emit *)
    (* which have an ECMA-v.1. compliant set of sorted tables. *)
    (if table_size (tab_GenericParam) > 0 then !!!0x00000400 else !!!0x00000000) ||| 
    (if table_size (tab_GenericParamConstraint) > 0 then !!!0x00001000 else !!!0x00000000) ||| 
    !!!0x00000200 in 
  
  reportTime "Layout Header of Tables";

  if logging then dprint_endline ("building string address table...");

  let guid_address n =   (if n = 0 then 0l else (!!!n --- !!!1) *** !!!0x10 +++ !!!0x01) in 

  let string_address_tab = 
    let tab = Array.create (Array.length strings + 1) 0l in
    let pos = ref !!!1 in 
    for i = 1 to Array.length strings do
      tab.(i) <- !pos;
      let s = strings.(i - 1) in 
      pos := !pos +++ !!!(Bytes.length s)
    done;
    tab in 
  let string_address n = 
    if n >= Array.length string_address_tab then failwith ("string index "^string_of_int n^" out of range");
    string_address_tab.(n) in 
  
  let userString_address_tab = 
    let tab = Array.create (Array.length userStrings + 1) 0l in
    let pos = ref !!!1 in 
    for i = 1 to Array.length userStrings do
      tab.(i) <- !pos;
      let s = userStrings.(i - 1) in 
      let n = !!!(Bytes.length s) +++ !!!1 in 
      pos := !pos +++ n +++ z_u32_size n
    done;
    tab in 
  let userString_address n = 
    if n >= Array.length userString_address_tab then failwith "userString index out of range";
    userString_address_tab.(n) in 
  
  let blob_address_tab = 
    let tab = Array.create (Array.length blobs + 1) 0l in
    let pos = ref !!!1 in 
    for i = 1 to Array.length blobs do
      tab.(i) <- !pos;
      let blob = blobs.(i - 1) in 
      pos := !pos +++ !!!(Bytes.length blob) +++ z_u32_size !!!(Bytes.length blob)
    done;
    tab in 
  let blob_address n = 
    if n >= Array.length blob_address_tab then failwith "blob index out of range";
    blob_address_tab.(n) in 
  
  reportTime "Build String/Blob Address Tables";

  if logging then dprint_endline ("done building string/blob address table...");

  
  if logging then dprint_endline ("sorting tables...");

  let sorted_tables = 
    let res = Array.create 64 [| |] in
    for i = 0 to 63 do 
      res.(i) <- Array.of_list (sort_rows (Table i) tables.(i))
    done;
    res in 
    
  reportTime "Sort Tables";

  if logging then dprint_endline ("encoding tables...");

  let coded_tables = 
        
    let table_big rows = Array.length rows >= 0x10000 in 
    let bigness_tab = Array.map table_big sorted_tables in  
    let bigness (Table idx) = bigness_tab.(idx) in  
    
    let coded_bigness nbits tab =
      !!!(table_size tab) >= (!!!0x10000 lsr nbits) in 
    
    let tdor_bigness = 
      coded_bigness 2 tab_TypeDef || 
      coded_bigness 2 tab_TypeRef || 
      coded_bigness 2 tab_TypeSpec in 
    
    let tomd_bigness = 
      coded_bigness 1 tab_TypeDef || 
      coded_bigness 1 tab_Method in 
    
    let hc_bigness = 
      coded_bigness 2 tab_Field ||
      coded_bigness 2 tab_Param ||
      coded_bigness 2 tab_Property in 
    
    let hca_bigness = 
      coded_bigness 5 tab_Method ||
      coded_bigness 5 tab_Field ||
      coded_bigness 5 tab_TypeRef  ||
      coded_bigness 5 tab_TypeDef ||
      coded_bigness 5 tab_Param ||
      coded_bigness 5 tab_InterfaceImpl ||
      coded_bigness 5 tab_MemberRef ||
      coded_bigness 5 tab_Module ||
      coded_bigness 5 tab_Permission ||
      coded_bigness 5 tab_Property ||
      coded_bigness 5 tab_Event ||
      coded_bigness 5 tab_StandAloneSig ||
      coded_bigness 5 tab_ModuleRef ||
      coded_bigness 5 tab_TypeSpec ||
      coded_bigness 5 tab_Assembly ||
      coded_bigness 5 tab_AssemblyRef ||
      coded_bigness 5 tab_File ||
      coded_bigness 5 tab_ExportedType ||
      coded_bigness 5 tab_ManifestResource  in
    
    let hfm_bigness = 
      coded_bigness 1 tab_Field || 
      coded_bigness 1 tab_Param in 
    
    let hds_bigness = 
      coded_bigness 2 tab_TypeDef || 
      coded_bigness 2 tab_Method ||
      coded_bigness 2 tab_Assembly in 
    
    let mrp_bigness = 
      coded_bigness 3 tab_TypeRef ||
      coded_bigness 3 tab_ModuleRef ||
      coded_bigness 3 tab_Method ||
      coded_bigness 3 tab_TypeSpec in 
    
    let hs_bigness = 
      coded_bigness 1 tab_Event || 
      coded_bigness 1 tab_Property  in 
    
    let mdor_bigness =
      coded_bigness 1 tab_Method ||    
      coded_bigness 1 tab_MemberRef  in 
    
    let mf_bigness =
      coded_bigness 1 tab_Field ||
      coded_bigness 1 tab_Method  in 
    
    let i_bigness =
      coded_bigness 2 tab_File || 
      coded_bigness 2 tab_AssemblyRef ||    
      coded_bigness 2 tab_ExportedType  in
    
    let cat_bigness =  
      coded_bigness 3 tab_Method ||    
      coded_bigness 3 tab_MemberRef  in 
    
    let rs_bigness = 
      coded_bigness 2 tab_Module ||    
      coded_bigness 2 tab_ModuleRef || 
      coded_bigness 2 tab_AssemblyRef  ||
      coded_bigness 2 tab_TypeRef in
    
    let tablesbuf =  Bytebuf.create 20000 in 
    
(* Now the coded tables themselves  - first the schemata header *)
    Bytebuf.emit_intarray_as_bytes tablesbuf    
      [| 0x00; 0x00; 0x00; 0x00; 
        mdtable_version_major; (* major version of table schemata *)
        mdtable_version_minor; (* minor version of table schemata *)
        ??? 
          ((if strings_big then !!!0x01 else !!!0x00) |||  (* bit vector for heap sizes *)
          (if guids_big then !!!0x02 else !!!0x00) |||  (* bit vector for heap sizes *)
          (if blobs_big then !!!0x04 else !!!0x00));
        0x01; (* reserved, always 1 *) |];

    Bytebuf.emit_i32 tablesbuf valid1;
    Bytebuf.emit_i32 tablesbuf valid2;
    Bytebuf.emit_i32 tablesbuf sorted1;
    Bytebuf.emit_i32 tablesbuf sorted2;
    
  (* Numbers of rows in various tables *)
    Array.iter
      (fun rows -> 
        if Array.length rows <> 0 then Bytebuf.emit_i32 tablesbuf !!!(Array.length rows)) 
      sorted_tables;
    
    
    let start_of_tables = Bytebuf.length tablesbuf in 
  reportTime "Write Header of tablebuf";

  (* The tables themselves *)
    for t = 0 to Array.length sorted_tables - 1 do
      (* dprintf2 "offset of start of table %d = 0x%08x\n" t (Bytebuf.length tablesbuf - start_of_tables) ; *)
      let rows = sorted_tables.(t) in
      for r = 0 to Array.length rows - 1 do
        let Row row = rows.(r) in 
        for c = 0 to Array.length row - 1 do
          (* Emit the coded token for the array element *)
          let x = row.(c) in
          match x with 
          | ULong n -> Bytebuf.emit_i32 tablesbuf n
          | UShort n -> bytebuf_emit_u16 tablesbuf n
          | Guid n -> bytebuf_emit_z_untagged_index tablesbuf guids_big (guid_address n)
          | Blob n -> bytebuf_emit_z_untagged_index tablesbuf blobs_big  (blob_address n)
          | Data (offset,kind) -> 
              record_reqd_data_fixup reqd_data_fixups tablesbuf (tables_stream_start +++ !!!(Bytebuf.length tablesbuf)) (offset, kind)
          | String n -> bytebuf_emit_z_untagged_index tablesbuf strings_big (string_address n)
          | SimpleIndex (tab,n) -> bytebuf_emit_z_untagged_index tablesbuf (bigness tab) !!!n
          | TypeDefOrRefOrSpec(TypeDefOrRefOrSpecTag tag,n) ->  
              bytebuf_emit_z_tagged_index tablesbuf tag 2 tdor_bigness n
          | TypeOrMethodDef(TypeOrMethodDefTag tag,n) ->  
              bytebuf_emit_z_tagged_index tablesbuf tag 1 tomd_bigness n
          | HasConstant (HasConstantTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 2 hc_bigness n
          | HasCustomAttribute (HasCustomAttributeTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 5 hca_bigness n
          | HasFieldMarshal (HasFieldMarshalTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 1   hfm_bigness n
          | HasDeclSecurity (HasDeclSecurityTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 2  hds_bigness n
          | MemberRefParent (MemberRefParentTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 3  mrp_bigness n 
          | HasSemantics (HasSemanticsTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 1  hs_bigness n 
          | MethodDefOrRef (MethodDefOrRefTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 1  mdor_bigness n
          | MemberForwarded (MemberForwardedTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 1  mf_bigness n
          | Implementation (ImplementationTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 2  i_bigness n
          | CustomAttributeType (CustomAttributeTypeTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag  3  cat_bigness n
          | ResolutionScope (ResolutionScopeTag tag,n) -> 
              bytebuf_emit_z_tagged_index tablesbuf tag 2  rs_bigness n
        done;
      done; 
    done;
    Bytebuf.close tablesbuf in 

  reportTime "Write Tables to tablebuf";

  if logging then dprint_endline ("laying out final metadata...");
  
  let tables_stream_unpadded_size = !!!(Bytes.length coded_tables) in 
  (* QUERY: extra 4 empty bytes in array.exe - why? Include some extra padding after the tables just in case there is a mistake in the ECMA spec. *)
  let tables_stream_padded_size = align !!!4 (tables_stream_unpadded_size +++ !!!4) in 
  let tables_streamp,next = chunk tables_stream_padded_size next in
  let tables_stream_padding = tables_streamp.size --- tables_stream_unpadded_size in 

  let strings_streamp,next = chunk strings_stream_padded_size next in 
  let strings_stream_padding = strings_streamp.size --- strings_stream_unpadded_size in 
  let userStrings_streamp,next = chunk userStrings_stream_padded_size next in 
  let userStrings_stream_padding = userStrings_streamp.size --- userStrings_stream_unpadded_size in 
  let guids_streamp,next = chunk (!!!0x10 *** !!!(Array.length guids)) next in 
  let blobs_streamp,next = chunk blobs_stream_padded_size next in 
  let blobs_stream_padding = blobs_streamp.size --- blobs_stream_unpadded_size in 
  
  reportTime "Layout Metadata";

  if logging then dprint_endline ("producing final metadata...");
  let metadata = 
    let mdbuf =  Bytebuf.create 500000 in 
    Bytebuf.emit_intarray_as_bytes mdbuf 
      [| 0x42; 0x53; 0x4a; 0x42; (* Magic signature *)
        0x01; 0x00; (* Major version *)
        0x01; 0x00; (* Minor version *)
      |];
    Bytebuf.emit_i32 mdbuf !!!0x0; (* Reservered *)

(*
    Bytebuf.emit_i32 mdbuf padded_version_length;
    Bytebuf.emit_intarray_as_bytes mdbuf version;
*)

    Bytebuf.emit_i32 mdbuf padded_version_length;
    Bytebuf.emit_intarray_as_bytes mdbuf version;
    for i = 1 to (??? padded_version_length - Array.length version) do 
      Bytebuf.emit_int_as_byte mdbuf 0x00;
    done;

    Bytebuf.emit_intarray_as_bytes mdbuf 
      [| 0x00; 0x00; (* flags, reserved *)
        b0 num_streams; b1 num_streams; |];
    Bytebuf.emit_i32 mdbuf tables_streamp.addr;
    Bytebuf.emit_i32 mdbuf tables_streamp.size;
    Bytebuf.emit_intarray_as_bytes mdbuf [| 0x23; 0x7e; 0x00; 0x00; (* #~00 *)|];
    Bytebuf.emit_i32 mdbuf strings_streamp.addr;
    Bytebuf.emit_i32 mdbuf strings_streamp.size;
    Bytebuf.emit_intarray_as_bytes mdbuf  [| 0x23; 0x53; 0x74; 0x72; 0x69; 0x6e; 0x67; 0x73; 0x00; 0x00; 0x00; 0x00 (* "#Strings0000" *)|];
    Bytebuf.emit_i32 mdbuf userStrings_streamp.addr;
    Bytebuf.emit_i32 mdbuf userStrings_streamp.size;
    Bytebuf.emit_intarray_as_bytes mdbuf [| 0x23; 0x55; 0x53; 0x00; (* #US0*) |];
    Bytebuf.emit_i32 mdbuf guids_streamp.addr;
    Bytebuf.emit_i32 mdbuf guids_streamp.size;
    Bytebuf.emit_intarray_as_bytes mdbuf [| 0x23; 0x47; 0x55; 0x49; 0x44; 0x00; 0x00; 0x00; (* #GUID000 *)|];
    Bytebuf.emit_i32 mdbuf blobs_streamp.addr;
    Bytebuf.emit_i32 mdbuf blobs_streamp.size;
    Bytebuf.emit_intarray_as_bytes mdbuf [| 0x23; 0x42; 0x6c; 0x6f; 0x62; 0x00; 0x00; 0x00; (* #Blob000 *)|];
    
  reportTime "Write Metadata Header";
   (* Now the coded tables themselves *)
    Bytebuf.emit_bytes mdbuf coded_tables;    
    for i = 1 to (??? tables_stream_padding) do 
      Bytebuf.emit_int_as_byte mdbuf 0x00;
    done;
  reportTime "Write Metadata Tables";

   (* The string stream *)
    Bytebuf.emit_intarray_as_bytes mdbuf [| 0x00 |];
    for i = 0 to Array.length strings - 1 do
      let s = strings.(i) in 
      Bytebuf.emit_bytes mdbuf s;
    done;
    for i = 1 to (??? strings_stream_padding) do 
      Bytebuf.emit_int_as_byte mdbuf 0x00;
    done;
  reportTime "Write Metadata Strings";
   (* The user string stream *)
    Bytebuf.emit_intarray_as_bytes mdbuf [| 0x00 |];
    for i = 0 to Array.length userStrings - 1 do
      let s = userStrings.(i) in 
      bytebuf_emit_z_u32 mdbuf (!!!(Bytes.length s) +++ !!!1);
      Bytebuf.emit_bytes mdbuf s;
      Bytebuf.emit_int_as_byte mdbuf (marker_for_unicode_bytes s)
    done;
    for i = 1 to (??? userStrings_stream_padding) do 
      Bytebuf.emit_int_as_byte mdbuf 0x00;
    done;

  reportTime "Write Metadata User Strings";
  (* The GUID stream *)
    Array.iter (Bytebuf.emit_bytes mdbuf) guids;
    
  (* The blob stream *)
    Bytebuf.emit_intarray_as_bytes mdbuf [| 0x00 |];
    for i = 0 to Array.length blobs - 1 do
      let s = blobs.(i) in 
      bytebuf_emit_z_u32 mdbuf (!!!(Bytes.length s));
      Bytebuf.emit_bytes mdbuf s
    done;
    for i = 1 to (??? blobs_stream_padding) do 
      Bytebuf.emit_int_as_byte mdbuf 0x00;
    done;
  reportTime "Write Blob Stream";
   (* Done - close the buffer and return the result. *)
    Bytebuf.close mdbuf in
  

  if logging then dprint_endline ("fixing up strings in final metadata...");

 (* Now we know the user string tables etc. we can fixup the *)
 (* uses of strings in the code *)
  begin 
    List.iter 
      (fun (code_start_addr, l) ->
        List.iter
          (fun (code_offset,userstring_idx) -> 
            if code_start_addr < codep.addr or code_start_addr >= codep.addr +++ codep.size  then failwith "strings-in-code fixup: a group of fixups is located outside the code array";
            let loc_in_code = ??? ((code_start_addr +++ !!!code_offset) --- codep.addr) in
            check_fixup32 code loc_in_code 0xdeadbeefl;
            let uncoded_userstring_token = 
              uncoded_token tab_UserStrings (??? (userString_address userstring_idx)) in  
            if (Bytes.get code (loc_in_code-1) <> i_ldstr) then failwith "strings-in-code fixup: not at ldstr instruction!";
            fixup32 code loc_in_code uncoded_userstring_token)
          l) 
      reqd_string_fixups;
  end;
  reportTime "Fixup Metadata";

  if logging then dprint_endline ("done metadata/code...");
  eptoken,
  code, 
  code_padding,
  metadata,
  data,
  resources,
  !reqd_data_fixups,
  pdb_data,
  mappings



(*---------------------------------------------------------------------
 * PHYSICAL METADATA+BLOBS --> PHYSICAL PE FORMAT
 *---------------------------------------------------------------------*)

(* THIS LAYS OUT A 2-SECTION .NET PE BINARY *)
(* SECTIONS *)
(* TEXT: physical 0x0200 --> RVA 0x00020000
           e.g. raw size 0x9600, 
           e.g. virt size 0x9584
   RELOC: physical 0x9800 --> RVA 0x0000c000
      i.e. phys_base --> rva_base
      where phys_base = text_base + text raw size
           phs_rva = roundup(0x2000, 0x0002000 + text virt size)

*)


let msdos_header = 
  [| 0x4d ; 0x5a ; 0x90 ; 0x00 ; 0x03 ; 0x00 ; 0x00 ; 0x00
      ; 0x04 ; 0x00 ; 0x00 ; 0x00 ; 0xFF ; 0xFF ; 0x00 ; 0x00
      ; 0xb8 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00
      ; 0x40 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00
      ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00
      ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00
      ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00
      ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x80 ; 0x00 ; 0x00 ; 0x00
      ; 0x0e ; 0x1f ; 0xba ; 0x0e ; 0x00 ; 0xb4 ; 0x09 ; 0xcd
      ; 0x21 ; 0xb8 ; 0x01 ; 0x4c ; 0xcd ; 0x21 ; 0x54 ; 0x68
      ; 0x69 ; 0x73 ; 0x20 ; 0x70 ; 0x72 ; 0x6f ; 0x67 ; 0x72
      ; 0x61 ; 0x6d ; 0x20 ; 0x63 ; 0x61 ; 0x6e ; 0x6e ; 0x6f
      ; 0x74 ; 0x20 ; 0x62 ; 0x65 ; 0x20 ; 0x72 ; 0x75 ; 0x6e
      ; 0x20 ; 0x69 ; 0x6e ; 0x20 ; 0x44 ; 0x4f ; 0x53 ; 0x20
      ; 0x6d ; 0x6f ; 0x64 ; 0x65 ; 0x2e ; 0x0d ; 0x0d ; 0x0a
      ; 0x24 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 ; 0x00 |];;


let write_i32_as_i32 os x = 
  Pervasives.output_byte os  (b0 x);
  Pervasives.output_byte os  (b1 x);
  Pervasives.output_byte os  (b2 x);
  Pervasives.output_byte os  (b3 x)  
let write_i32_as_u16 os x = 
  Pervasives.output_byte os  (b0 x);
  Pervasives.output_byte os  (b1 x)  
    
let write_directory os dict =
  write_i32_as_i32 os (if dict.size = !!!0x0 then !!!0x0 else dict.addr);
  write_i32_as_i32 os dict.size

let write_intarray os chunk = 
  for i = 0 to Array.length chunk - 1  do 
    let b = chunk.(i) in 
    if checking && (b < 0 or b > 255) then dprint_endline ("write: "^string_of_int b^" is not a byte at offset "^string_of_int i);
    Pervasives.output_byte os (b mod 256)
  done 

let write_bytes os chunk = Bytes.output os chunk 

let write_binary_internal manager outfile mscorlib (pdbfile: string option) (signer: signer option) desiredMetadataVersionOpt fixupOverlappingSequencePoints modul =
    (* Store the public key from the signer into the manifest.  This means it will be written *)
    (* to the binary and also acts as an indicator to leave space for delay sign *)

    reportTime "Write Started";
    let is_dll = modul.modulDLL in 
    
    let signer = 
        match signer,modul.modulManifest with
        | Some _, _ -> signer
        | _, None -> signer
        | None, Some {manifestPublicKey=Some pubkey} -> 
            (dprint_endline "Note: The output assembly will be delay-signed using the original public";
            dprint_endline "Note: key. In order to load it you will need to either sign it with";
            dprint_endline "Note: the original private key or to turn off strong-name verification";
            dprint_endline "Note: (use sn.exe from the .NET Framework SDK to do this, e.g. 'sn -Vr *').";
            dprint_endline "Note: Alternatively if this tool supports it you can provide the original";
            dprint_endline "Note: private key when converting the assembly, assuming you have access to";
            dprint_endline "Note: it.";
            Some (signerOpenPublicKey pubkey))
        | _ -> signer in 

    let modul = 
        let pubkey =
          match signer with 
          | None -> None
          | Some s -> 
             try Some (signerPublicKey s) 
             with e ->     
               dprint_endline ("Warning: A call to StrongNameGetPublicKey failed ("^Printexc.to_string e^")"); 
               None in 
        begin match modul.modulManifest with 
        | None -> () 
        | Some m -> 
           if m.manifestPublicKey <> None && m.manifestPublicKey <> pubkey then 
             dprint_endline "Warning: The output assembly is being signed or delay-signed with a strong name that is different to the original."
        end;
        { modul with modulManifest = match modul.modulManifest with None -> None | Some m -> Some {m with manifestPublicKey = pubkey} } in 

    if logging then dprintf1 "signerPublicKey (2): %s" outfile;

    let timestamp = absilWriteGetTimeStamp () in 

    let os = try  open_out_bin outfile
             with e -> failwith ("Could not open file for writing (binary mode): " ^ outfile)    
    in 

    let  pdb_data,debug_directoryp,debug_datap,textV2P,mappings =
        begin try 
      
          let image_base_real = modul.modulImageBase in (* FIXED CHOICE *) 
          let align_virt = modul.modulVirtAlignment in (* FIXED CHOICE *) 
          let align_phys = modul.modulPhysAlignment in (* FIXED CHOICE *) 
          
          let num_sections = !!!3 in (* .text, .sdata, .reloc *)

          (* HEADERS *)

          let next = !!!0x0 in 
          let header_phys_loc = !!!0x0 in 
          let header_addr = next in 
          let next = header_addr in 
          
          let msdos_header_size = 0x80l in 
          let msdos_headerp,next = chunk msdos_header_size next in 
          
          let pe_signature_size = !!!0x04 in 
          let pe_signaturep,next = chunk pe_signature_size next in 
          
          let pe_file_header_size = !!!0x14 in 
          let pe_file_headerp,next = chunk pe_file_header_size next in 
          
          let pe_optional_header_size = !!!0xe0 in 
          let pe_optional_headerp,next = chunk pe_optional_header_size next in 
          
          let text_section_header_size = !!!0x28 in 
          let text_section_headerp,next = chunk text_section_header_size next in 
          
          let data_section_header_size = !!!0x28 in 
          let data_section_headerp,next = chunk data_section_header_size next in 
          
          let reloc_section_header_size = !!!0x28 in 
          let reloc_section_headerp,next = chunk reloc_section_header_size next in 
          
          let header_size = next --- header_addr in 
          let next_phys = align align_phys (header_phys_loc +++ header_size) in 
          let header_phys_size = next_phys --- header_phys_loc in 
          let next = align align_virt (header_addr +++ header_size) in 
          
          (* TEXT SECTION:  8 bytes IAT table 72 bytes CLI header *)

          let text_phys_loc = next_phys in 
          let text_addr = next in 
          let next = text_addr in 
          
          let import_addr_tabp,next = chunk !!!0x08 next in 
          let cli_headerp,next = chunk !!!0x48 next in 
          
          let compileTimeVersion = 
            try 
              let v = absilWriteGetMetadataVersion () in 
              Il.parse_version (if String.get v 0 = 'v' then String.sub v 1 (String.length v - 1) else v)
            with Failure s -> 
              dprint_endline ("warning, could not determine CLR version, using 1.0.3705: "^s); 
              Il.parse_version "1.0.3705" in
          let desiredMetadataVersion =  
            match desiredMetadataVersionOpt with 
              None -> compileTimeVersion
            | Some v -> v in 

          let eptoken,code,code_padding,metadata,data,resources,reqd_data_fixups,pdb_data,mappings = 
            write_binary_il manager ((pdbfile <> None), desiredMetadataVersion,compileTimeVersion,mscorlib) modul next in 

          reportTime "Generated IL and metadata";
          let codep,next = chunk !!!(Bytes.length code) next in 
          let code_paddingp,next = chunk !!!(Array.length code_padding) next in 
          
          let metadatap,next = chunk !!!(Bytes.length metadata) next in 
          
          let strongnamep,next = 
            match signer with 
            | None -> nochunk next
            | Some s -> chunk !!!(signerSignatureSize s) next in

          let resourcesp,next = chunk !!!(Bytes.length resources) next in 
         
          let rawdatap,next = chunk !!!(Bytes.length data) next in 

          let vtfixupsp,next = nochunk next in  (* @todo *)
          
          let import_tabp,next = chunk !!!0x28 next in
          let import_lookup_tabp,next = chunk !!!0x14 next in 
          let import_name_hint_tabp,next = chunk !!!0x0e next in 
          let mscoree_stringp,next = chunk !!!0x0c next in 
          
          let next = align !!!0x10 (next +++ !!!0x05) --- !!!0x05 in 
          let import_tabp = { addr=import_tabp.addr; size = next --- import_tabp.addr} in
          let import_tabp_padding = import_tabp.size --- (!!!0x28 +++ !!!0x14 +++ !!!0x0e +++ !!!0x0c) in
          
          let next = next +++ !!!0x03 in 
          let entrypoint_codep,next = chunk !!!0x06 next in 
          
          let debug_directoryp,next = chunk (if pdbfile = None then !!!0x0 else !!!sizeof_IMAGE_DEBUG_DIRECTORY) next in 
          (* The debug data is given to us by the PDB writer and appears to typically be the type of the data plus the PDB file name.  We fill this in after we've written the binary. We approximate the size according to what PDB writers seem to require and leave extra space just in case... *)
          let debug_data_just_in_case = 40 in 
          let debug_datap,next = chunk (align !!!0x4 (match pdbfile with None -> !!!0x0 | Some f -> !!!(24 + String.length f + debug_data_just_in_case))) next in 


          let text_size = next --- text_addr in 
          let next_phys = align align_phys (text_phys_loc +++ text_size) in 
          let text_phys_size = next_phys --- text_phys_loc in 
          let next = align align_virt (text_addr +++ text_size) in 
          
          (* .RSRC SECTION (DATA) *)
          let data_phys_loc =  next_phys in 
          let data_addr = next in 
          let dataV2P v = v --- data_addr +++ data_phys_loc in 
          
          let native_resource_size = 
             match modul.modulNativeResources with 
             | Some f -> 
                 let unlinkedResource= Lazy.force f in 
                 begin 
                   try linkNativeResource unlinkedResource 0l (Bytes.of_intarray [| |])
                   with e -> 
                     (dprint_endline ("Warning: Linking a native resource failed ("^Printexc.to_string e^")");
                      dprint_endline "Warning: The output assembly will not contain the resource.";
                      0l)
                 end
             | None -> 0l in 

          let native_resourcesp,next = chunk native_resource_size next in 
        
          let dummydatap,next = chunk (if next = data_addr then !!!0x01 else !!!0x0) next in
          
          let data_size = next --- data_addr in 
          let next_phys = align align_phys (data_phys_loc +++ data_size) in 
          let data_phys_size = next_phys --- data_phys_loc in 
          let next = align align_virt (data_addr +++ data_size) in 
          
          (* .RELOC SECTION  base reloc table: 0x0c size *)
          let reloc_phys_loc =  next_phys in 
          let reloc_addr = next in 
          
          let base_reloc_tabp,next = chunk !!!0x0c next in 

          let reloc_size = next --- reloc_addr in 
          let next_phys = align align_phys (reloc_phys_loc +++ reloc_size) in 
          let reloc_phys_size = next_phys --- reloc_phys_loc in 
          let next = align align_virt (reloc_addr +++ reloc_size) in 
          
          if logging then dprint_endline ("fixup references into data section...");

         (* Now we know where the data section lies we can fix up the  *)
         (* references into the data section from the metadata tables. *)
          begin 
            reqd_data_fixups |> List.iter
              (fun (metadata_offset32,(data_offset,kind)) -> 
                let metadata_offset = ??? metadata_offset32 in 
                if metadata_offset < 0 or metadata_offset >= Bytes.length metadata - 4  then failwith "data RVA fixup: fixup located outside metadata";
                check_fixup32 metadata metadata_offset 0xdeadddddl;
                let data_rva = 
                  if kind then
                    let res = !!!data_offset in 
                    if res >= resourcesp.size then dprint_endline ("resource offset bigger than resource data section");
                    res
                  else 
                    let res = rawdatap.addr +++ !!!data_offset in 
                    if res < rawdatap.addr then dprint_endline ("data rva before data section");
                    if res >= rawdatap.addr +++ rawdatap.size then dprint_endline ("data rva after end of data section, data_rva = "^Int32.to_string res^", rawdatap.addr = "^Int32.to_string rawdatap.addr^", rawdatap.size = "^Int32.to_string rawdatap.size);
                    res in
                fixup32 metadata metadata_offset data_rva);
          end;
          
         (* IMAGE TOTAL SIZE *)
          let image_end_phys_loc =  next_phys in 
          let image_end_addr = next in 

          reportTime "Layout image";
          if logging then dprint_endline ("writing image...");

          let write p os string chunk = 
            begin 
              match p with 
              | None -> () 
              | Some p' -> 
                  if Int32.of_int (Pervasives.pos_out os) <> p' then 
                    dprint_endline("warning: "^string^" not where expected, pos_out = "^string_of_int (Pervasives.pos_out os)^", p.addr = "^Int32.to_string p') 
            end;
            write_intarray os chunk in
          
          let write_padding os string sz = 
            let sz2 = ??? sz in 
            if sz2 < 0 then failwith "write_padding: size < 0";
            for i = 0 to sz2 - 1 do 
              Pervasives.output_byte os 0
            done in 
          
          (* Now we've computed all the offsets, write the image *)
          
          write (Some msdos_headerp.addr) os "msdos header" msdos_header;
          
      (* PE HEADER *)
          write (Some pe_signaturep.addr) os "pe_signature" [| |];
          write_i32_as_i32 os !!!0x4550;
          
          write (Some pe_file_headerp.addr) os "pe_file_header" [| |];
          write_i32_as_u16 os !!!0x014c; (* Machine *)
          write_i32_as_u16 os num_sections; 
          write_i32_as_i32 os timestamp; (* date since 1970 *)
          write_i32_as_i32 os !!!0x00; (* Pointer to Symbol Table Always 0 *)
       (* 00000090 *) 
          write_i32_as_i32 os !!!0x00; (* Number of Symbols Always 0 *)
          write_i32_as_u16 os !!!0xe0; (* Size of the optional header, the format is described below. *)
          write_i32_as_u16 os ((if is_dll then !!!0x2000 else !!!0x0000) ||| !!!0x0002 ||| !!!0x0004 ||| !!!0x0008 ||| !!!0x0100);
          
     (* Now comes optional header *)

          let peOptionalHeaderByte = peOptionalHeaderByteByCLRVersion desiredMetadataVersion in 

          write (Some pe_optional_headerp.addr) os "pe_optional_header" [| |];
          write_i32_as_u16 os !!!0x010b; (* Always !!!0x10B (see Section 23.1). *)
          write_i32_as_u16 os !!!peOptionalHeaderByte; (* QUERY: ECMA spec says 6, some binaries, e.g. fscmanaged.exe say 7, Whidbey binaries say 8 *)
          write_i32_as_i32 os text_phys_size; (* Size of the code (text) section, or the sum of all code sections if there are multiple sections. *)
       (* 000000a0 *) 
          write_i32_as_i32 os data_phys_size; (* Size of the initialized data section, or the sum of all such sections if there are multiple data sections. *)
          write_i32_as_i32 os !!!0x00; (* Size of the uninitialized data section, or the sum of all such sections if there are multiple unitinitalized data sections. *)
          write_i32_as_i32 os entrypoint_codep.addr; (* RVA of entry point , needs to point to bytes 0xFFl !!!0x25 followed by the RVA+!0x4000000 in a section marked execute/read for EXEs or 0 for DLLs e.g. !!!0x0000b57e *)
          write_i32_as_i32 os text_addr; (* e.g. !!!0x0002000 *)
       (* 000000b0 *) 
          write_i32_as_i32 os data_addr; (* e.g. !!!0x0000c000 *)
          write_i32_as_i32 os image_base_real; (* Image Base Always !!!0x400000 (see Section 23.1). - QUERY : no it's not always !!!0x400000, e.g. !!!0x034f0000 *)
          write_i32_as_i32 os align_virt;  (*  Section Alignment Always !!!0x2000 (see Section 23.1). *)
          write_i32_as_i32 os align_phys; (* File Alignment Either !!!0x200 or !!!0x1000. *)
       (* 000000c0 *) 
          write_i32_as_u16 os !!!0x04; (*  OS Major Always 4 (see Section 23.1). *)
          write_i32_as_u16 os !!!0x00; (* OS Minor Always 0 (see Section 23.1). *)
          write_i32_as_u16 os !!!0x00; (* User Major Always 0 (see Section 23.1). *)
          write_i32_as_u16 os !!!0x00; (* User Minor Always 0 (see Section 23.1). *)
          write_i32_as_u16 os !!!0x04; (* SubSys Major Always 4 (see Section 23.1). *)
          write_i32_as_u16 os !!!0x00; (* SubSys Minor Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Reserved Always 0 (see Section 23.1). *)
       (* 000000d0 *) 
          write_i32_as_i32 os image_end_addr; (* Image Size: Size, in bytes, of image, including all headers and padding; shall be a multiple of Section Alignment. e.g. !!!0x0000e000 *)
          write_i32_as_i32 os header_phys_size; (* Header Size Combined size of MS-DOS Header, PE Header, PE Optional Header and padding; shall be a multiple of the file alignment. *)
          write_i32_as_i32 os !!!0x00; (* File Checksum Always 0 (see Section 23.1). QUERY: NOT ALWAYS ZERO *)
          write_i32_as_u16 os modul.modulSubSystem; (* SubSystem Subsystem required to run this image. Shall be either IMAGE_SUBSYSTEM_WINDOWS_CE_GUI (!!!0x3) or IMAGE_SUBSYSTEM_WINDOWS_GUI (!!!0x2). QUERY: Why is this 3 on the images ILASM produces??? *)
          write_i32_as_u16 os !!!0x00;  (*  DLL Flags Always 0 (see Section 23.1). *)
       (* 000000e0 *) 
          write_i32_as_i32 os !!!0x100000; (* Stack Reserve Size Always !!!0x100000 (1Mb) (see Section 23.1). *)
          write_i32_as_i32 os !!!0x1000; (* Stack Commit Size Always !!!0x1000 (4Kb) (see Section 23.1). *)
          write_i32_as_i32 os !!!0x100000; (* Heap Reserve Size Always !!!0x100000 (1Mb) (see Section 23.1). *)
          write_i32_as_i32 os !!!0x1000; (* Heap Commit Size Always !!!0x1000 (4Kb) (see Section 23.1). *)
       (* 000000f0 *) 
          write_i32_as_i32 os !!!0x00; (* Loader Flags Always 0 (see Section 23.1) *)
          write_i32_as_i32 os !!!0x10; (* Number of Data Directories: Always !!!0x10 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; 
          write_i32_as_i32 os !!!0x00; (* Export Table Always 0 (see Section 23.1). *)
       (* 00000100 *) 
          write_directory os import_tabp; (* Import Table RVA of Import Table, (see clause 24.3.1). e.g. 0000b530 *) 
          (* Native Resource Table: ECMA says Always 0 (see Section 23.1), but mscorlib and other files with resources bound into executable do not.  For the moment assume the resources table is always the first resource in the file. *)
          write_directory os native_resourcesp;

       (* 00000110 *) 
          write_i32_as_i32 os !!!0x00; (* Exception Table Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Exception Table Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Certificate Table Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Certificate Table Always 0 (see Section 23.1). *)
       (* 00000120 *) 
          write_directory os base_reloc_tabp; 
          write_directory os debug_directoryp; (* Debug Directory *)
       (* 00000130 *) 
          write_i32_as_i32 os !!!0x00; (*  Copyright Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (*  Copyright Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Global Ptr Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Global Ptr Always 0 (see Section 23.1). *)
       (* 00000140 *) 
          write_i32_as_i32 os !!!0x00; (* Load Config Table Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Load Config Table Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* TLS Table Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* TLS Table Always 0 (see Section 23.1). *)
       (* 00000150 *) 
          write_i32_as_i32 os !!!0x00; (* Bound Import Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Bound Import Always 0 (see Section 23.1). *)
          write_directory os import_addr_tabp; (* Import Addr Table, (see clause 24.3.1). e.g. !!!0x00002000 *) 
       (* 00000160 *) 
          write_i32_as_i32 os !!!0x00; (* Delay Import Descriptor Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Delay Import Descriptor Always 0 (see Section 23.1). *)
          write_directory os cli_headerp;
       (* 00000170 *) 
          write_i32_as_i32 os !!!0x00; (* Reserved Always 0 (see Section 23.1). *)
          write_i32_as_i32 os !!!0x00; (* Reserved Always 0 (see Section 23.1). *)
          
          write (Some text_section_headerp.addr) os "text_section_header" [| |];
          
       (* 00000178 *) 
          write_intarray os  [| 0x2e; 0x74; 0x65; 0x78; 0x74; 0x00; 0x00; 0x00; |]; (* ".text\000\000\000" *)
       (* 00000180 *) 
          write_i32_as_i32 os text_size; (* VirtualSize: Total size of the section when loaded into memory in bytes rounded to Section Alignment. If this value is greater than Size of Raw Data, the section is zero-padded. e.g. !!!0x00009584 *)
          write_i32_as_i32 os text_addr; (*  VirtualAddress For executable images this is the address of the first byte of the section, when loaded into memory, relative to the image base. e.g. !!!0x00020000 *)
          write_i32_as_i32 os text_phys_size; (*  SizeOfRawData Size of the initialized data on disk in bytes, shall be a multiple of FileAlignment from the PE header. If this is less than VirtualSize the remainder of the section is zero filled. Because this field is rounded while the VirtualSize field is not it is possible for this to be greater than VirtualSize as well. When a section contains only uninitialized data, this field should be 0. !!!0x00009600 *)
          write_i32_as_i32 os text_phys_loc; (* PointerToRawData RVA to sections first page within the PE file. This shall be a multiple of FileAlignment from the optional header. When a section contains only uninitialized data, this field should be 0. e.g. 00000200 *)
       (* 00000190 *) 
          write_i32_as_i32 os !!!0x00; (* PointerToRelocations RVA of Relocation section. *)
          write_i32_as_i32 os !!!0x00; (* PointerToLinenumbers Always 0 (see Section 23.1). *)
       (* 00000198 *) 
          write_i32_as_u16 os !!!0x00;(* NumberOfRelocations Number of relocations, set to 0 if unused. *)
          write_i32_as_u16 os !!!0x00;  (*  NumberOfLinenumbers Always 0 (see Section 23.1). *)
          write_intarray os [| 0x20; 0x00; 0x00; 0x60 |]; (*  Characteristics Flags describing sections characteristics, see below. IMAGE_SCN_CNT_CODE || IMAGE_SCN_MEM_EXECUTE || IMAGE_SCN_MEM_READ *)
          
          write (Some data_section_headerp.addr) os "data_section_header" [| |];
          
       (* 000001a0 *) 
          write_intarray os [| 0x2e; 0x72; 0x73; 0x72; 0x63; 0x00; 0x00; 0x00; |]; (* ".rsrc\000\000\000" *)
    (*  write_intarray os [| 0x2e; 0x73; 0x64; 0x61; 0x74; 0x61; 0x00; 0x00; |]; (* ".sdata\000\000" *) *)
          write_i32_as_i32 os data_size; (* VirtualSize: Total size of the section when loaded into memory in bytes rounded to Section Alignment. If this value is greater than Size of Raw Data, the section is zero-padded. e.g. !!!0x0000000c *)
          write_i32_as_i32 os data_addr; (*  VirtualAddress For executable images this is the address of the first byte of the section, when loaded into memory, relative to the image base. e.g. !!!0x0000c000*)
       (* 000001b0 *) 
          write_i32_as_i32 os data_phys_size; (*  SizeOfRawData Size of the initialized data on disk in bytes, shall be a multiple of FileAlignment from the PE header. If this is less than VirtualSize the remainder of the section is zero filled. Because this field is rounded while the VirtualSize field is not it is possible for this to be greater than VirtualSize as well. When a section contains only uninitialized data, this field should be 0. e.g. !!!0x00000200 *)
          write_i32_as_i32 os data_phys_loc; (* PointerToRawData QUERY: Why does ECMA say "RVA" here? Offset to sections first page within the PE file. This shall be a multiple of FileAlignment from the optional header. When a section contains only uninitialized data, this field should be 0. e.g. !!!0x00009800 *)
       (* 000001b8 *) 
          write_i32_as_i32 os !!!0x00; (* PointerToRelocations RVA of Relocation section. *)
          write_i32_as_i32 os !!!0x00; (* PointerToLinenumbers Always 0 (see Section 23.1). *)
       (* 000001c0 *) 
          write_i32_as_u16 os !!!0x00; (* NumberOfRelocations Number of relocations, set to 0 if unused. *)
          write_i32_as_u16 os !!!0x00;  (*  NumberOfLinenumbers Always 0 (see Section 23.1). *)
          write_intarray os [| 0x40; 0x00; 0x00; 0x40 |]; (*  Characteristics Flags: IMAGE_SCN_MEM_READ |  IMAGE_SCN_CNT_INITIALIZED_DATA *)
          
          write (Some reloc_section_headerp.addr) os "reloc_section_header" [| |];
       (* 000001a0 *) 
          write_intarray os [| 0x2e; 0x72; 0x65; 0x6c; 0x6f; 0x63; 0x00; 0x00; |]; (* ".reloc\000\000" *)
          write_i32_as_i32 os reloc_size; (* VirtualSize: Total size of the section when loaded into memory in bytes rounded to Section Alignment. If this value is greater than Size of Raw Data, the section is zero-padded. e.g. !!!0x0000000c *)
          write_i32_as_i32 os reloc_addr; (*  VirtualAddress For executable images this is the address of the first byte of the section, when loaded into memory, relative to the image base. e.g. !!!0x0000c000*)
       (* 000001b0 *) 
          write_i32_as_i32 os reloc_phys_size; (*  SizeOfRawData Size of the initialized reloc on disk in bytes, shall be a multiple of FileAlignment from the PE header. If this is less than VirtualSize the remainder of the section is zero filled. Because this field is rounded while the VirtualSize field is not it is possible for this to be greater than VirtualSize as well. When a section contains only uninitialized reloc, this field should be 0. e.g. !!!0x00000200 *)
          write_i32_as_i32 os reloc_phys_loc; (* PointerToRawData QUERY: Why does ECMA say "RVA" here? Offset to sections first page within the PE file. This shall be a multiple of FileAlignment from the optional header. When a section contains only uninitialized reloc, this field should be 0. e.g. !!!0x00009800 *)
       (* 000001b8 *) 
          write_i32_as_i32 os !!!0x00; (* b0 relocptr; b1 relocptr; b2 relocptr; b3 relocptr; *) (* PointerToRelocations RVA of Relocation section. *)
          write_i32_as_i32 os !!!0x00; (* PointerToLinenumbers Always 0 (see Section 23.1). *)
       (* 000001c0 *) 
          write_i32_as_u16 os !!!0x00; (* b0 numreloc; b1 numreloc; *) (* NumberOfRelocations Number of relocations, set to 0 if unused. *)
          write_i32_as_u16 os !!!0x00;  (*  NumberOfLinenumbers Always 0 (see Section 23.1). *)
          write_intarray os [| 0x40; 0x00; 0x00; 0x42 |]; (*  Characteristics Flags: IMAGE_SCN_CNT_INITIALIZED_DATA | IMAGE_SCN_MEM_READ | ??? *)
          
          write_padding os "pad to text begin" (text_phys_loc --- header_size);
          
          (* TEXT SECTION: e.g. 0x200 *)
          
          let textV2P v = v --- text_addr +++ text_phys_loc in 
          
          (* e.g. 0x0200 *)
          write (Some (textV2P import_addr_tabp.addr)) os "import_addr_table" [| |];
          write_i32_as_i32 os import_name_hint_tabp.addr; 
          write_i32_as_i32 os !!!0x00;  (* QUERY 4 bytes of zeros not 2 like ECMA  24.3.1 says *)
          
          (* e.g. 0x0208 *)

          let flags = 
            (if modul.modulILonly then !!!0x01 else !!!0x00) ||| 
            (if modul.modul32bit then !!!0x02 else !!!0x00) ||| 
            (if (match signer with None -> false | Some s -> signerFullySigned s) then !!!0x08 else !!!0x00) in 

          let headerVersionMajor,headerVersionMinor = headerVersionSupportedByCLRVersion desiredMetadataVersion in 


          write (Some (textV2P cli_headerp.addr)) os "cli_header"  [| |];
          write_i32_as_i32 os !!!0x48; (* size of header *)
          write_i32_as_u16 os !!!headerVersionMajor; (* Major part of minimum version of CLR reqd. *)
          write_i32_as_u16 os !!!headerVersionMinor; (* Minor part of minimum version of CLR reqd. ... *)
          (* e.g. 0x0210 *)
          write_directory os metadatap;
          write_i32_as_i32 os flags;
          
          write_i32_as_i32 os eptoken; 
          write None os "rest of cli header" [| |];
          
          (* e.g. 0x0220 *)
          write_directory os resourcesp;
          write_directory os strongnamep;
          (* e.g. 0x0230 *)
          write_i32_as_i32 os !!!0x00; (* code manager table, always 0 *)
          write_i32_as_i32 os !!!0x00; (* code manager table, always 0 *)
          write_directory os vtfixupsp; 
          (* e.g. 0x0240 *)
          write_i32_as_i32 os !!!0x00;  (* export addr table jumps, always 0 *)
          write_i32_as_i32 os !!!0x00;  (* export addr table jumps, always 0 *)
          write_i32_as_i32 os !!!0x00;  (* managed native header, always 0 *)
          write_i32_as_i32 os !!!0x00;  (* managed native header, always 0 *)
          
          write_bytes os code;
          write None os "code padding" code_padding;
          
          write_bytes os metadata;
          
          (* write 0x80 bytes of empty space for encrypted SHA1 hash, written by SN.EXE or call to signing API *)
          if signer <> None then 
            write (Some (textV2P strongnamep.addr)) os "strongname" (Array.create (Int32.to_int strongnamep.size) 0x0);
          
          write (Some (textV2P resourcesp.addr)) os "raw resources" [| |];
          write_bytes os resources;
          write (Some (textV2P rawdatap.addr)) os "raw data" [| |];
          write_bytes os data;

          (* vtfixups would go here *)
          write (Some (textV2P import_tabp.addr)) os "import_table" [| |];
          
          write_i32_as_i32 os import_lookup_tabp.addr;
          write_i32_as_i32 os !!!0x00;
          write_i32_as_i32 os !!!0x00;
          write_i32_as_i32 os mscoree_stringp.addr;
          write_i32_as_i32 os import_addr_tabp.addr;
          write_i32_as_i32 os !!!0x00;
          write_i32_as_i32 os !!!0x00;
          write_i32_as_i32 os !!!0x00;
          write_i32_as_i32 os !!!0x00;
          write_i32_as_i32 os !!!0x00; 
        
          write (Some (textV2P import_lookup_tabp.addr)) os "import_lookup_table" [| |];
          write_i32_as_i32 os import_name_hint_tabp.addr; 
          write_i32_as_i32 os !!!0x00; 
          write_i32_as_i32 os !!!0x00; 
          write_i32_as_i32 os !!!0x00; 
          write_i32_as_i32 os !!!0x00; 

          write (Some (textV2P import_name_hint_tabp.addr)) os "import_name_hint_table" [| |];
          (* Two zero bytes of hint, then Case sensitive, null-terminated ASCII string containing name to import. Shall _CorExeMain a .exe file _CorDllMain for a .dll file. *)
          (if  is_dll then 
            write_intarray os [| 0x00;  0x00;  
                                0x5f;  0x43 ;  0x6f;  0x72 ;  0x44;  0x6c ;  0x6c;  0x4d ;  0x61;  0x69 ;  0x6e;  0x00 |]
          else 
            write_intarray os [| 0x00;  0x00;  
                                0x5f;  0x43 ;  0x6f;  0x72 ;  0x45;  0x78 ;  0x65;  0x4d ;  0x61;  0x69 ;  0x6e;  0x00 |]);
          
          write (Some (textV2P mscoree_stringp.addr)) os "mscoree string"
            [| 0x6d;  0x73;  
              0x63;  0x6f ;  0x72;  0x65 ;  0x65;  0x2e ;  0x64;  0x6c ;  0x6c;  0x00 ; |];
          
          write_padding os "end of import tab" import_tabp_padding;
          
          write_padding os "head of entrypoint" !!!0x03;
          let ep = (image_base_real +++ text_addr) in 
          write (Some (textV2P entrypoint_codep.addr)) os " entrypoint code"
            [| 0xff; 0x25; (* x86 Instructions for entry *)
              b0 ep; b1 ep; b2 ep; b3 ep |];
          
           if pdbfile <> None then 
            write (Some (textV2P debug_directoryp.addr)) os "debug directory" (Array.create sizeof_IMAGE_DEBUG_DIRECTORY 0x0);
          
          if pdbfile <> None then 
            write (Some (textV2P debug_datap.addr)) os "debug data" (Array.create (??? (debug_datap.size)) 0x0);
          
          write_padding os "end of .text" (data_phys_loc --- text_phys_loc --- text_size);
          
          (* DATA SECTION *)
          begin match modul.modulNativeResources with 
              | Some f -> 
                  if (native_resourcesp.size > 0l) then 
                    begin
                      let buf = Bytes.zero_create (??? (native_resourcesp.size))  in 
                      ignore(linkNativeResource (Lazy.force f) native_resourcesp.addr buf);
                      write (Some (dataV2P native_resourcesp.addr)) os "raw native resources" [| |];
                      write_bytes os buf;
                    end
              | None ->  ()  
          end;

          if dummydatap.size <> !!!0x0 then
            write (Some (dataV2P dummydatap.addr)) os "dummy data" [| 0x0 |];
          
          write_padding os "end of .rsrc" (reloc_phys_loc --- data_phys_loc --- data_size);
          
          (* RELOC SECTION *)

          (* See ECMA 24.3.2 *)
          let relocV2P v = v --- reloc_addr +++ reloc_phys_loc in 
          
          let entrypoint_fixup_addr = entrypoint_codep.addr +++ !!!0x02 in 
          let entrypoint_fixup_block = (entrypoint_fixup_addr /./ !!!4096) *** !!!4096 in 
          let entrypoint_fixup_offset = entrypoint_fixup_addr --- entrypoint_fixup_block in
          let reloc = !!!0x3000 (* IMAGE_REL_BASED_HIGHLOW *) ||| entrypoint_fixup_offset in 
          write (Some (relocV2P base_reloc_tabp.addr)) os "base_reloc_table" 
            [| b0 entrypoint_fixup_block; b1 entrypoint_fixup_block; b2 entrypoint_fixup_block; b3 entrypoint_fixup_block;
              0x0c; 0x00; 0x00; 0x00;
              b0 reloc; b1 reloc; 
              0x00; 0x00; |];
          write_padding os "end of .reloc" (image_end_phys_loc --- reloc_phys_loc --- reloc_size);

          close_out os;
          pdb_data,debug_directoryp,debug_datap,textV2P,mappings
          
        with e -> (try close_out os; Sys.remove outfile with _ -> ()); (*F# rethrow(); F#*) raise e
        end 
    in 

    reportTime "Writing Image";
    if logging then dprint_endline ("Finished writing the binary...");
     
    (* Now we've done the bulk of the binary, do the PDB file and fixup the binary. *)
    begin match pdbfile with
    | None -> ()
    | Some fpdb -> 
        try 
            if logging then dprint_endline ("Now write debug info...");

            let idd = write_pdb_info desiredMetadataVersionOpt fixupOverlappingSequencePoints outfile fpdb pdb_data in 
            reportTime "Generate PDB Info";
            
          (* Now we have the debug data we can go back and fill in the debug directory in the image *)
            let os2 = open_out_gen [Open_binary; Open_wronly] 0x777 outfile in 
            try 
                (* write the IMAGE_DEBUG_DIRECTORY *)
                seek_out os2 (??? (textV2P debug_directoryp.addr));
                write_i32_as_i32 os2 idd.iddCharacteristics; (* IMAGE_DEBUG_DIRECTORY.Characteristics *)
                write_i32_as_i32 os2 timestamp;
                write_i32_as_u16 os2 idd.iddMajorVersion;
                write_i32_as_u16 os2 idd.iddMinorVersion;
                write_i32_as_i32 os2 idd.iddType;
                write_i32_as_i32 os2 !!!(Bytes.length idd.iddData);  (* IMAGE_DEBUG_DIRECTORY.SizeOfData *)
                write_i32_as_i32 os2 debug_datap.addr;  (* IMAGE_DEBUG_DIRECTORY.AddressOfRawData *)
                write_i32_as_i32 os2 (textV2P debug_datap.addr);(* IMAGE_DEBUG_DIRECTORY.PointerToRawData *)

                (* dprintf1 "idd.iddCharacteristics = %ld\n" idd.iddCharacteristics;
                dprintf1 "iddMajorVersion = %ld\n" idd.iddMajorVersion;
                dprintf1 "iddMinorVersion = %ld\n" idd.iddMinorVersion;
                dprintf1 "iddType = %ld\n" idd.iddType;
                dprintf3 "iddData = (%a) = %s\n" Ilprint.output_bytes idd.iddData (Bytes.utf8_bytes_as_string idd.iddData); *)
                  
                (* write the debug raw data as given us by the PDB writer *)
                seek_out os2 (??? (textV2P debug_datap.addr));
                if debug_datap.size < !!!(Bytes.length idd.iddData) then dprint_endline "*** WARNING: debug data area not big enough.  Debug info may not be usable.";
                let len = min (??? (debug_datap.size)) (Bytes.length idd.iddData) in
                write_bytes os2 idd.iddData;
                close_out os2;
            with e -> 
                dprint_endline ("*** Warning: Error while writing debug directory entry: "^Printexc.to_string e);
                (try close_out os2; Sys.remove outfile with _ -> ()); 
                (*F# rethrow(); F#*) raise e
        with e -> 
            dprint_endline ("*** Warning: Error while writing PDB file or debug directory entry: "^Printexc.to_string e);
            ()
    end;
    reportTime "Finalize PDB";

    (* Sign the binary.  No further changes to binary allowed past this point! *)
    begin match signer with 
    | None -> ()
    | Some s -> 
        if logging then dprint_endline ("Now sign the binary...");
        try 
            signerSignFile outfile s; signerClose s 
        with e -> 
            dprint_endline ("Warning: A call to StrongNameSignatureGeneration failed ("^Printexc.to_string e^")");
            (try signerClose s with _ -> ());
            (try Sys.remove outfile with _ -> ()); 
            ()
    end;
    reportTime "Signing Image";
    if logging then dprint_endline ("Finished writing and signing the binary and debug info...");

    mappings


type options =
 { mscorlib: scope_ref;
   manager: manager option;
   pdbfile: string option;
   signer: signer option;
   desiredMetadataVersionOpt : version_info option;
   fixupOverlappingSequencePoints: bool }

let defaults = 
 { mscorlib=Il.ecma_mscorlib_scoref ;
   manager=None;
   pdbfile=None;
   signer=None;
   desiredMetadataVersionOpt=None;
   fixupOverlappingSequencePoints=false }

let write_binary_ex outfile args modul =
  ignore (write_binary_internal  args.manager outfile args.mscorlib args.pdbfile args.signer args.desiredMetadataVersionOpt args.fixupOverlappingSequencePoints modul)

let write_binary outfile (pdbfile: string option) (signer: signer option) ( desiredMetadataVersionOpt : version_info option) modul =
  write_binary_ex outfile {defaults with manager=None; pdbfile=pdbfile;  signer=signer; desiredMetadataVersionOpt=desiredMetadataVersionOpt } modul

let write_binary_and_report_mappings outfile args modul =
  write_binary_internal  args.manager outfile Il.ecma_mscorlib_scoref args.pdbfile args.signer args.desiredMetadataVersionOpt args.fixupOverlappingSequencePoints modul


