// $Id$
//
// Earth System Modeling Framework
// Copyright 2002-2014, University Corporation for Atmospheric Research, 
// Massachusetts Institute of Technology, Geophysical Fluid Dynamics 
// Laboratory, University of Michigan, National Centers for Environmental 
// Prediction, Los Alamos National Laboratory, Argonne National Laboratory, 
// NASA Goddard Space Flight Center.
// Licensed under the University of Illinois-NCSA License.
//
//==============================================================================
#define ESMC_FILENAME "ESMCI_Array.C"
//==============================================================================
//
// Array class implementation (body) file
//
//-----------------------------------------------------------------------------
//
// !DESCRIPTION:
//
// The code in this file implements the C++ Array methods declared
// in the companion file ESMCI_Array.h
//
// Macros that affect the code:
//   * WORKAROUND_NONBLOCKPROGRESSBUG - when defined limits the outsanding 
//     non-blocking sends to one at a time, or in other places uses blocking
//     send calls instead. This work-around was introduced for the 
//     discover/pgi-14.1.0/mvapich2-2.0b combination that started hanging some
//     of the RegridWeightGen tests in the --check part.
//-----------------------------------------------------------------------------
// include associated header file
#include "ESMCI_Array.h"

// include higher level, 3rd party or system headers
#include <cstdio>
#include <cstring>
#include <vector>
#include <map>
#include <algorithm>

// include ESMF headers
#include "ESMCI_Macros.h"
#include "ESMCI_LogErr.h"
#include "ESMCI_IO.h"

using namespace std;

//-----------------------------------------------------------------------------
// leave the following line as-is; it will insert the cvs ident string
// into the object file for tracking purposes.
static const char *const version = "$Id$";
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#define VERBOSITY             (1)       // 0: off, 10: max
//-----------------------------------------------------------------------------


namespace ESMCI {

  //-----------------------------------------------------------------------------
//
// constructor and destructor
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::Array()"
//BOPI
// !IROUTINE:  ESMCI::Array::Array    - constructor
//
// !INTERFACE:
Array::Array(
//
// !RETURN VALUE:
//    
//
// !ARGUMENTS:
//
  ESMC_TypeKind_Flag typekindArg,         // (in)
  int rankArg,                            // (in)
  LocalArray **larrayListArg,             // (in)
  DistGrid *distgridArg,                  // (in)
  bool distgridCreatorArg,                // (in)
  int *exclusiveLBoundArg,                // (in)
  int *exclusiveUBoundArg,                // (in)
  int *computationalLBoundArg,            // (in)
  int *computationalUBoundArg,            // (in)
  int *totalLBoundArg,                    // (in)
  int *totalUBoundArg,                    // (in)
  int tensorCountArg,                     // (in)
  int tensorElementCountArg,              // (in)
  int *undistLBoundArray,                 // (in)
  int *undistUBoundArray,                 // (in)
  int *distgridToArrayMapArray,           // (in)
  int *arrayToDistGridMapArray,           // (in)
  int *distgridToPackedArrayMapArray,     // (in)
  ESMC_IndexFlag indexflagArg,            // (in)
  int *rc,                                // (out)
  VM *vm                                  // (in)
  ):ESMC_Base(vm){    // allow specific VM instead default
//
// !DESCRIPTION:
//    Construct the internal information structure of an ESMCI::Array object.
//    No error checking wrt consistency of input arguments is needed because
//    Array constructor is only to be called by Array::create() interfaces which
//    are responsible for providing consistent arguments to this layer.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code

  try{  

  // fill in the Array object
  typekind = typekindArg;
  rank = rankArg;
  distgrid = distgridArg;
  distgridCreator = distgridCreatorArg;
  delayout = distgrid->getDELayout();
  // copy the PET-local LocalArray pointers
  int localDeCount = delayout->getLocalDeCount();
  larrayList = new LocalArray*[localDeCount];
  memcpy(larrayList, larrayListArg, localDeCount*sizeof(LocalArray *));
  // determine the base addresses of the local arrays:
  larrayBaseAddrList = new void*[localDeCount];
  for (int i=0; i<localDeCount; i++)
    larrayBaseAddrList[i] = larrayList[i]->getBaseAddr();
  // copy the PET-local bound arrays
  int redDimCount = rank - tensorCountArg; // reduced dimCount w/o repl. dims
  exclusiveLBound = new int[redDimCount*localDeCount];
  memcpy(exclusiveLBound, exclusiveLBoundArg,
    redDimCount*localDeCount*sizeof(int));
  exclusiveUBound = new int[redDimCount*localDeCount];
  memcpy(exclusiveUBound, exclusiveUBoundArg,
    redDimCount*localDeCount*sizeof(int));
  computationalLBound = new int[redDimCount*localDeCount];
  memcpy(computationalLBound, computationalLBoundArg,
    redDimCount*localDeCount*sizeof(int));
  computationalUBound = new int[redDimCount*localDeCount];
  memcpy(computationalUBound, computationalUBoundArg,
    redDimCount*localDeCount*sizeof(int));
  totalLBound = new int[redDimCount*localDeCount];
  memcpy(totalLBound, totalLBoundArg,
    redDimCount*localDeCount*sizeof(int));
  totalUBound = new int[redDimCount*localDeCount];
  memcpy(totalUBound, totalUBoundArg,
    redDimCount*localDeCount*sizeof(int));
  // tensor dimensions
  tensorCount = tensorCountArg;
  tensorElementCount = tensorElementCountArg;
  undistLBound = new int[tensorCountArg];
  memcpy(undistLBound, undistLBoundArray, tensorCountArg * sizeof(int));
  undistUBound = new int[tensorCountArg];
  memcpy(undistUBound, undistUBoundArray, tensorCountArg * sizeof(int));
  // distgridToArrayMap, arrayToDistGridMap and distgridToPackedArrayMap
  int dimCount = distgrid->getDimCount();
  distgridToArrayMap = new int[dimCount];
  memcpy(distgridToArrayMap, distgridToArrayMapArray, dimCount * sizeof(int));
  arrayToDistGridMap = new int[rank];
  memcpy(arrayToDistGridMap, arrayToDistGridMapArray, rank * sizeof(int));
  distgridToPackedArrayMap = new int[dimCount];
  memcpy(distgridToPackedArrayMap, distgridToPackedArrayMapArray,
    dimCount * sizeof(int));
  // indexflag
  indexflag = indexflagArg;
  // exclusiveElementCountPDe
  int deCount = delayout->getDeCount();
  exclusiveElementCountPDe = new int[deCount];
  const int *indexCountPDimPDe = distgrid->getIndexCountPDimPDe();
  for (int i=0; i<deCount; i++){
    exclusiveElementCountPDe[i] = 1;   // prime exclusiveElementCountPDe element
    for (int jj=0; jj<rank; jj++){
      int j = arrayToDistGridMap[jj];// j is dimIndex basis 1, or 0 for tensor d
      if (j){
        // decomposed dimension 
        --j;  // shift to basis 0
        exclusiveElementCountPDe[i] *= indexCountPDimPDe[i*dimCount+j];
      }
    }
  }
  // totalElementCountPLocalDe
  totalElementCountPLocalDe = new int[localDeCount];
  for (int i=0; i<localDeCount; i++){
    totalElementCountPLocalDe[i] = 1;   // prime totalElementCountPLocalDe elem
    for (int j=0; j<redDimCount; j++){
      totalElementCountPLocalDe[i] *=
        totalUBound[i*redDimCount+j] - totalLBound[i*redDimCount+j] + 1;
    }
  }
  
  // contiguous flag
  contiguousFlag = new int[localDeCount];
  for (int i=0; i<localDeCount; i++)
    contiguousFlag[i] = -1;  // initialize as "not yet constructed"
  
  // Set up rim members and fill with canonical seqIndex values
  //TODO: rim setup has potentially sizable memory and performance impact???
  rimElementCount.resize(localDeCount);
  rimSeqIndex.resize(localDeCount);
  rimLinIndex.resize(localDeCount);
  for (int i=0; i<localDeCount; i++){
    ArrayElement arrayElement(this, i, true); // iterator through total w/o excl
    int element = 0;
    while(arrayElement.isWithin()){
      // obtain linear index for this element
      int linIndex = arrayElement.getLinearIndexExclusive();
      // obtain canonical seqIndex value according to DistGrid topology
      SeqIndex seqIndex;  // invalidated by default constructor
      if (arrayElement.hasValidSeqIndex()){
        // seqIndex is well defined for this arrayElement
        seqIndex = arrayElement.getSequenceIndexExclusive(3);
      }
      rimSeqIndex[i].push_back(seqIndex); // store seqIndex for this rim element
      rimLinIndex[i].push_back(linIndex); // store linIndex for this rim element
      arrayElement.next();  // next element
      ++element;
    } // multi dim index loop
    rimElementCount[i] = element; // store element count
  }
  
  localDeCountAux = localDeCount; // TODO: auxilary for garb until ref. counting
  
  // invalidate the name for this Array object in the Base class
  ESMC_BaseSetName(NULL, "Array");
   
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      rc);
    return;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, rc);
    return;
  }
  
  // return successfully
  if (rc!=NULL) *rc = ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::destruct()"
//BOPI
// !IROUTINE:  ESMCI::Array::destruct
//
// !INTERFACE:
void Array::destruct(bool followCreator){
//
// TODO: The followCreator flag is only needed until we have reference counting
// TODO: For now followCreator, which by default is true, will be coming in as
// TODO: false when calling through the native destructor. This prevents
// TODO: sequence problems during automatic garbage collection unitl reference
// TODO: counting comes in to solve this problem in the final manner.
//
// !DESCRIPTION:
//    Destruct the internal information structure of an ESMCI::Array object.
//
//EOPI
//-----------------------------------------------------------------------------
  if (ESMC_BaseGetStatus()==ESMF_STATUS_READY){
    // garbage collection
//    int localDeCount = delayout->getLocalDeCount();
    int localDeCount = localDeCountAux; // TODO: delayout may be gone already!
// TODO: replace the above line with the line before once ref. counting implem.
    for (int i=0; i<localDeCount; i++){
      int localrc = LocalArray::destroy(larrayList[i]);
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, NULL)) throw localrc;  // bail out with exception
    }
    if (larrayList != NULL)
      delete [] larrayList;
    if (larrayBaseAddrList != NULL)
      delete [] larrayBaseAddrList;
    if (exclusiveLBound != NULL)
      delete [] exclusiveLBound;
    if (exclusiveUBound != NULL)
      delete [] exclusiveUBound;
    if (computationalLBound != NULL)
      delete [] computationalLBound;
    if (computationalUBound != NULL)
      delete [] computationalUBound;
    if (totalLBound != NULL)
      delete [] totalLBound;
    if (totalUBound != NULL)
      delete [] totalUBound;
    if (undistLBound != NULL)
      delete [] undistLBound;
    if (undistUBound != NULL)
      delete [] undistUBound;
    if (distgridToArrayMap != NULL)
      delete [] distgridToArrayMap;
    if (arrayToDistGridMap != NULL)
      delete [] arrayToDistGridMap;
    if (distgridToPackedArrayMap != NULL)
      delete [] distgridToPackedArrayMap;
    if (contiguousFlag != NULL)
      delete [] contiguousFlag;
    if (exclusiveElementCountPDe != NULL)
      delete [] exclusiveElementCountPDe;
    if (totalElementCountPLocalDe != NULL)
      delete [] totalElementCountPLocalDe;
    if (distgridCreator && followCreator){
      int localrc = DistGrid::destroy(&distgrid);
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, NULL)) throw localrc;  // bail out with exception
    }
  }
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::constructContiguousFlag()"
//BOPI
// !IROUTINE:  ESMCI::Array::constructContiguousFlag
//
// !INTERFACE:
int Array::constructContiguousFlag(int redDimCount){
//
// !DESCRIPTION:
//    Fill the contiguousFlag member in an ESMCI::Array object.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  
  int const localDeCount = delayout->getLocalDeCount();
  const int *localDeToDeMap = delayout->getLocalDeToDeMap();
  const int *indexCountPDimPDe = distgrid->getIndexCountPDimPDe();
  int dimCount = distgrid->getDimCount();

  for (int i=0; i<localDeCount; i++){
    int de = localDeToDeMap[i];
    contiguousFlag[i] = 1;  // initialize as contiguous
    int pI = 0; // initialize packed index
    for (int jj=0; jj<rank; jj++){
      int j = arrayToDistGridMap[jj];// j is dimIndex basis 1, or 0 for tensor d
      if (j){
        // decomposed dimension 
        if (totalLBound[i*redDimCount+pI] != exclusiveLBound[i*redDimCount+pI]
          || totalUBound[i*redDimCount+pI] !=exclusiveUBound[i*redDimCount+pI]){
          contiguousFlag[i] = 0; // reset
          break;
        }
        // obtain indexList for this DE and dim
        --j;  // shift to basis 0
        const int *indexList =
          distgrid->getIndexListPDimPLocalDe(i, j+1, &localrc);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, NULL)) return localrc;
        // check if this dim has a contiguous index list
        for (int k=1; k<indexCountPDimPDe[de*dimCount+j]; k++){
          if (indexList[k] != indexList[k-1]+1){
            contiguousFlag[i] = 0; // reset
            break;
          }
        }
        ++pI;
      }
    } // jj
  }
  
  // return successfully
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// create() and destroy()
//
//-----------------------------------------------------------------------------

;//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::create()"
//BOPI
// !IROUTINE:  ESMCI::Array::create
//
// !INTERFACE:
Array *Array::create(
//
// !RETURN VALUE:
//    ESMC_Array * to newly allocated ESMC_Array
//
// !ARGUMENTS:
//
  LocalArray **larrayListArg,                 // (in)
  int larrayCount,                            // (in)
  DistGrid *distgrid,                         // (in)
  CopyFlag copyflag,                          // (in)
  InterfaceInt *distgridToArrayMap,           // (in)
  InterfaceInt *computationalEdgeLWidthArg,   // (in)
  InterfaceInt *computationalEdgeUWidthArg,   // (in)
  InterfaceInt *computationalLWidthArg,       // (in)
  InterfaceInt *computationalUWidthArg,       // (in)
  InterfaceInt *totalLWidthArg,               // (in)
  InterfaceInt *totalUWidthArg,               // (in)
  ESMC_IndexFlag *indexflagArg,               // (in)
  InterfaceInt *undistLBoundArg,              // (in)
  InterfaceInt *undistUBoundArg,              // (in)
  int *rc                                     // (out) return code
  ){
//
// !DESCRIPTION:
//    Create an {\tt ESMCI::Array} object from list if LocalArrays and DistGrid.
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code
  
  Array *array;
  try{
  
  // check the input and get the information together to call construct()
  // larrayListArg -> typekind/rank
  if (larrayListArg == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to larrayList argument", ESMC_CONTEXT, rc);
    return ESMC_NULL_POINTER;
  }
  if (larrayCount < 1){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
      "- The larrayList argument must provide at least one LocalArray object",
      ESMC_CONTEXT, rc);
    return ESMC_NULL_POINTER;
  }
  ESMC_TypeKind_Flag typekind = larrayListArg[0]->getTypeKind();
  int rank = larrayListArg[0]->getRank();
  for (int i=1; i<larrayCount; i++){
    if (larrayListArg[0]->getTypeKind() != typekind){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- TypeKind mismatch in the elements of larrayList argument",
        ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (larrayListArg[0]->getRank() != rank){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- Rank mismatch in the elements of larrayList argument", ESMC_CONTEXT,
        rc);
      return ESMC_NULL_POINTER;
    }
  }
  // distgrid -> delayout, dimCount
  if (distgrid == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to distgrid", ESMC_CONTEXT, rc);
    return ESMC_NULL_POINTER;
  }
  const DELayout *delayout = distgrid->getDELayout();
  int dimCount = distgrid->getDimCount();
  // check if distgridToArrayMap was provided and matches rest of arguments
  int *distgridToArrayMapArray = new int[dimCount];
  for (int i=0; i<dimCount; i++){
    if (i < rank)
      distgridToArrayMapArray[i] = i+1; // default (basis 1)
    else
      distgridToArrayMapArray[i] = 0;   // default (replicator dims beyond rank)
  }
  if (distgridToArrayMap != NULL){
    if (distgridToArrayMap->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- distgridToArrayMap array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (distgridToArrayMap->extent[0] != dimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- distgridToArrayMap and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    memcpy(distgridToArrayMapArray, distgridToArrayMap->array,
      dimCount*sizeof(int));
  }
  {
    // check distgridToArrayMapArray
    bool *check = new bool[rank];
    for (int i=0; i<rank; i++)
      check[i] = false; // initialize
    for (int i=0; i<dimCount; i++){
      if (distgridToArrayMapArray[i] < 0 || distgridToArrayMapArray[i] > rank){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- invalid distgridToArrayMap element", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      if (distgridToArrayMapArray[i] > 0){
        if(check[distgridToArrayMapArray[i]-1]){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
            "- invalid distgridToArrayMap element", ESMC_CONTEXT, rc);
          return ESMC_NULL_POINTER;
        }
        check[distgridToArrayMapArray[i]-1] = true;
      }
    }
    delete [] check;
  }
  // determine replicatorCount
  int replicatorCount = 0;  // initialize
  for (int i=0; i<dimCount; i++)
    if (distgridToArrayMapArray[i] == 0) ++replicatorCount;
  // determine reduced dimCount -> redDimCount
  int redDimCount = dimCount - replicatorCount;
  // determine tensorCount
  int tensorCount = rank - redDimCount;
  if (tensorCount < 0) tensorCount = 0;
  // generate arrayToDistGridMap
  int *arrayToDistGridMapArray = new int[rank];
  for (int i=0; i<rank; i++)
    arrayToDistGridMapArray[i] = 0; // reset  (basis 1), 0 indicates tensor dim
  for (int i=0; i<dimCount; i++)
    if (int j=distgridToArrayMapArray[i])
      arrayToDistGridMapArray[j-1] = i+1;
  // generate distgridToPackedArrayMap - labels the distributed Array dims 1,2,.
  int *distgridToPackedArrayMap = new int[dimCount];
  for (int i=0; i<dimCount; i++)
    distgridToPackedArrayMap[i] = 0; // reset  (basis 1), 0 indicates repl. dim
  {
    int k=1;  // reset
    for (int i=0; i<rank; i++){
      if (int j=arrayToDistGridMapArray[i]){
        distgridToPackedArrayMap[j-1] = k;
        ++k;
      }
    }
  }
  // check for undistLBound and undistUBound arguments and that they match
  // tensorCount
  int undistLBoundArrayAllocFlag = 0;  // reset
  int *undistLBoundArray = NULL; // reset
  if (undistLBoundArg != NULL){
    if (undistLBoundArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- undistLBound array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (undistLBoundArg->extent[0] != tensorCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- undistLBound, arrayspec, distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    undistLBoundArray = undistLBoundArg->array;
  }else if (tensorCount > 0){
    // tensor dimensions are present, but no explicit bounds provided'
    // -> set to bounds of incoming array at DE 0 (should be same across DEs!)
    const int *undistLBound = larrayListArg[0]->getLbounds();
    undistLBoundArrayAllocFlag = 1;  // set
    undistLBoundArray = new int[tensorCount];
    int tensorIndex = 0;  // reset
    for (int i=0; i<rank; i++)
      if (arrayToDistGridMapArray[i] == 0){
        undistLBoundArray[tensorIndex] = undistLBound[i];
        ++tensorIndex;
      }
  }
  int undistUBoundArrayAllocFlag = 0;  // reset
  int *undistUBoundArray = NULL; // reset
  if (undistUBoundArg != NULL){
    if (undistUBoundArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- undistUBound array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (undistUBoundArg->extent[0] != tensorCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- undistUBound, arrayspec, distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    undistUBoundArray = undistUBoundArg->array;
  }else if (tensorCount > 0){
    // tensor dimensions are present, but no explicit bounds provided
    // -> set to bounds of incoming array at DE 0
    // -> set to bounds of incoming array at DE 0 (should be same across DEs!)
    const int *undistUBound = larrayListArg[0]->getUbounds();
    undistUBoundArrayAllocFlag = 1;  // set
    undistUBoundArray = new int[tensorCount];
    int tensorIndex = 0;  // reset
    for (int i=0; i<rank; i++)
      if (arrayToDistGridMapArray[i] == 0){
        undistUBoundArray[tensorIndex] = undistUBound[i];
        ++tensorIndex;
      }
  }
  // tensorElementCount
  int tensorElementCount = 1;  // prime tensorElementCount
  for (int i=0; i<tensorCount; i++)
    tensorElementCount *= (undistUBoundArray[i] - undistLBoundArray[i] + 1);
  
  // delayout -> deCount, localDeCount, localDeToDeMap
  int deCount = delayout->getDeCount();
  int localDeCount = delayout->getLocalDeCount();
  if ((localDeCount > 0) && (localDeCount != larrayCount)){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
      "- Mismatch in localDeCount between larrayList argument and DELayout", 
      ESMC_CONTEXT, rc);
    return ESMC_NULL_POINTER;
  }
  const int *localDeToDeMap = delayout->getLocalDeToDeMap();
  // distgrid -> indexCountPDimPDe[]
  const int *indexCountPDimPDe = distgrid->getIndexCountPDimPDe();
  // check on indexflag
  ESMC_IndexFlag indexflag = ESMF_INDEX_DELOCAL;  // default
  if (indexflagArg != NULL)
    indexflag = *indexflagArg;
  // figure exclusive region
  int *exclusiveLBound = new int[redDimCount*localDeCount];
  int *exclusiveUBound = new int[redDimCount*localDeCount];
  for (int i=0; i<redDimCount*localDeCount; i++)
    exclusiveLBound[i] = 1; // excl. region starts at (1,1,1...) <- Fortran
  // exlc. region for each DE ends at indexCountPDimPDe of the associated
  // DistGrid
  for (int i=0; i<localDeCount; i++){
    int de = localDeToDeMap[i];
    for (int j=0; j<dimCount; j++)
      if (int k=distgridToPackedArrayMap[j])
        exclusiveUBound[i*redDimCount+k-1] = indexCountPDimPDe[de*dimCount+j];
  }
  // optionally shift origin of exclusive region to pseudo global index space
  if (indexflag == ESMF_INDEX_GLOBAL){
    for (int i=0; i<localDeCount; i++){
      int de = localDeToDeMap[i];
      for (int j=0; j<dimCount; j++){
        // check that this DE/dim has a contiguous index list
        const int *contigFlagPDimPDe = distgrid->getContigFlagPDimPDe();
        if (!contigFlagPDimPDe[de*dimCount+j]){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_NOT_VALID,
            "- Cannot use non-contiguous decomposition for pseudo global"
            " index space", ESMC_CONTEXT, rc);
          return ESMC_NULL_POINTER;
        }
        // obtain indexList for this DE and dim
        const int *indexList =
          distgrid->getIndexListPDimPLocalDe(i, j+1, &localrc);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, rc)) return ESMC_NULL_POINTER;
        // shift bounds of exclusive region to match indexList[0]
        if (indexCountPDimPDe[de*dimCount+j]){
          // only shift if there are indices associated
          if (int k=distgridToPackedArrayMap[j]){
            // only shift if this isn't a replicated dim
            int shift = indexList[0] - exclusiveLBound[i*redDimCount+k-1];
            exclusiveLBound[i*redDimCount+k-1] += shift;
            exclusiveUBound[i*redDimCount+k-1] += shift;
          }
        }
      } // j
    } // i
  }
  // deal with computationalEdge widths
  int *computationalEdgeLWidth = new int[redDimCount];
  int *computationalEdgeUWidth = new int[redDimCount];
  if (computationalEdgeLWidthArg != NULL){
    if (computationalEdgeLWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalEdgeLWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (computationalEdgeLWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- computationalEdgeLWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    memcpy(computationalEdgeLWidth, computationalEdgeLWidthArg->array,
      redDimCount*sizeof(int));
  }else{
    // set default
    for (int i=0; i<redDimCount; i++)
      computationalEdgeLWidth[i] = 0;
  }
  if (computationalEdgeUWidthArg != NULL){
    if (computationalEdgeUWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalEdgeUWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (computationalEdgeUWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- computationalEdgeUWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    memcpy(computationalEdgeUWidth, computationalEdgeUWidthArg->array,
      redDimCount*sizeof(int));
  }else{
    // set default
    for (int i=0; i<redDimCount; i++)
      computationalEdgeUWidth[i] = 0;
  }
  // deal with computational widths
  int *computationalLBound = new int[redDimCount*localDeCount];
  int *computationalUBound = new int[redDimCount*localDeCount];
  if (computationalLWidthArg != NULL){
    if (computationalLWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalLWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (computationalLWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- computationalLWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    for (int j=0; j<localDeCount; j++)
      for (int i=0; i<redDimCount; i++)
        computationalLBound[j*redDimCount+i] = exclusiveLBound[j*redDimCount+i]
          - computationalLWidthArg->array[i];
  }else{
    // set default
    memcpy(computationalLBound, exclusiveLBound,
      localDeCount*redDimCount*sizeof(int));
  }
  if (computationalUWidthArg != NULL){
    if (computationalUWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalUWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (computationalUWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- computationalUWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    for (int j=0; j<localDeCount; j++)
      for (int i=0; i<redDimCount; i++)
        computationalUBound[j*redDimCount+i] = exclusiveUBound[j*redDimCount+i]
          + computationalUWidthArg->array[i];
  }else{
    // set default
    memcpy(computationalUBound, exclusiveUBound,
      localDeCount*redDimCount*sizeof(int));
  }
  // modify computational bounds on tile edges
  for (int j=0; j<localDeCount; j++){
    for (int i=0; i<dimCount; i++){
      if (int k=distgridToPackedArrayMap[i]){
        if (computationalEdgeLWidth[k-1]){
          bool onEdgeL = distgrid->isLocalDeOnEdgeL(j, i+1, &localrc);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, rc)) return ESMC_NULL_POINTER;
          if (onEdgeL)
            computationalLBound[j*redDimCount+k-1]
              -= computationalEdgeLWidth[k-1];
        }
        if (computationalEdgeUWidth[k-1]){
          bool onEdgeU = distgrid->isLocalDeOnEdgeU(j, i+1, &localrc);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, rc)) return ESMC_NULL_POINTER;
          if (onEdgeU)
            computationalUBound[j*redDimCount+k-1]
              += computationalEdgeUWidth[k-1];
        }
      }
    }
  }
  // clean-up
  delete [] computationalEdgeLWidth;
  delete [] computationalEdgeUWidth;
  // deal with total widths
  int totalLBoundFlag = 0;  // reset
  int totalUBoundFlag = 0;  // reset
  int *totalLBound = new int[redDimCount*localDeCount];
  int *totalUBound = new int[redDimCount*localDeCount];
  if (totalLWidthArg != NULL){
    totalLBoundFlag = 1;  // set
    if (totalLWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- totalLWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (totalLWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- totalLWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    for (int i=0; i<redDimCount; i++){
      if (totalLWidthArg->array[i] < 0){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- totalLWidth may only contain positive values", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      for (int j=0; j<localDeCount; j++){
        totalLBound[j*redDimCount+i] = exclusiveLBound[j*redDimCount+i]
          - totalLWidthArg->array[i];
      }
    }
  }else{
    // set default
    for (int i=0; i<localDeCount*redDimCount; i++)
      totalLBound[i] = computationalLBound[i];
  }
  if (totalUWidthArg != NULL){
    totalUBoundFlag = 1;  // set
    if (totalUWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- totalUWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (totalUWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- totalUWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    for (int i=0; i<redDimCount; i++){
      if (totalUWidthArg->array[i] < 0){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- totalUWidth may only contain positive values", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      for (int j=0; j<localDeCount; j++){
        totalUBound[j*redDimCount+i] = exclusiveUBound[j*redDimCount+i]
          + totalUWidthArg->array[i];
      }
    }
  }else{
    // set default
    for (int i=0; i<localDeCount*redDimCount; i++)
      totalUBound[i] = computationalUBound[i];
  }
  // total region _must_ cover exclusive region -> adjust total region if not so
  for (int i=0; i<localDeCount*redDimCount; i++){
    if (exclusiveLBound[i] < totalLBound[i])
      totalLBound[i] = exclusiveLBound[i];
    if (exclusiveUBound[i] > totalUBound[i])
      totalUBound[i] = exclusiveUBound[i];
  }
  
  // check computational bounds against total bounds
  for (int i=0; i<localDeCount*redDimCount; i++){
    if (totalLBound[i] <= totalUBound[i]){
      // DE/dim is associated with DistGrid elements
      if (computationalLBound[i] < totalLBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationaLBound / totalLBound mismatch", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      if (computationalLBound[i] > totalUBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationaLBound / totalUBound mismatch", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      if (computationalUBound[i] < totalLBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationalUBound / totalLBound mismatch", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      if (computationalUBound[i] > totalUBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationalUBound / totalUBound mismatch", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
    }
  }

  // allocate LocalArray list that holds all PET-local DEs and adjust elements
  LocalArray **larrayList = new LocalArray*[localDeCount];
  int *temp_larrayLBound = new int[rank];
  int *temp_larrayUBound = new int[rank];
  for (int i=0; i<localDeCount; i++){
    if (indexflag == ESMF_INDEX_USER){
      // don't adjust dope vector, use F90 pointers directly and their bounds
      const int *temp_counts = larrayListArg[i]->getCounts();
      int j=0;    // reset distributed index
      int jjj=0;  // reset undistributed index
      for (int jj=0; jj<rank; jj++){
        if (arrayToDistGridMapArray[jj]){
          // distributed dimension
          if (temp_counts[jj] < 
            totalUBound[i*redDimCount+j] - totalLBound[i*redDimCount+j] + 1){
            ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
              "- LocalArray does not accommodate requested element count",
              ESMC_CONTEXT, rc);
            return ESMC_NULL_POINTER;
          }
          // move the total bounds according to input info
          if (totalLBoundFlag){
            // totalLBound fixed
            temp_larrayLBound[jj] = totalLBound[i*redDimCount+j];
            if (totalUBoundFlag){
              // totalUBound fixed
              temp_larrayUBound[jj] = totalUBound[i*redDimCount+j];
            }else{
              // totalUBound not fixed
              temp_larrayUBound[jj] = temp_counts[jj]
                + totalLBound[i*redDimCount+j] - 1;
            }
          }else{
            // totalLBound not fixed
            if (totalUBoundFlag){
              // totalUBound fixed
              temp_larrayUBound[jj] = totalUBound[i*redDimCount+j];
              temp_larrayLBound[jj] = totalUBound[i*redDimCount+j]
                - temp_counts[jj] + 1;
            }else{
              // totalLBound and totalUBound not fixed
              // -> shift computational/excl. region into center of total region
              int lBound = computationalLBound[i*redDimCount+j];
              if (exclusiveLBound[i*redDimCount+j] 
                < computationalLBound[i*redDimCount+j])
                lBound = exclusiveLBound[i*redDimCount+j];
              int uBound = computationalUBound[i*redDimCount+j];
              if (exclusiveUBound[i*redDimCount+j]
                > computationalUBound[i*redDimCount+j])
                uBound = exclusiveUBound[i*redDimCount+j];
              temp_larrayLBound[jj] = lBound
                - (int)(0.5 * (temp_counts[jj] - 1 + lBound - uBound));
              temp_larrayUBound[jj] = temp_counts[jj] + temp_larrayLBound[jj]
                - 1;
            }
          }
          totalLBound[i*redDimCount+j] = temp_larrayLBound[jj]; // write back
          totalUBound[i*redDimCount+j] = temp_larrayUBound[jj]; // write back
          // now bounds must match exactly or it's an error
          if (temp_counts[jj] !=
            totalUBound[i*redDimCount+j] - totalLBound[i*redDimCount+j] + 1){
            ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
              "- LocalArray does not match requested element count",
              ESMC_CONTEXT, rc);
            return ESMC_NULL_POINTER;
          }
          ++j;
        }else{
          // non-distributed dimension
          if (temp_counts[jj] !=
            undistUBoundArray[jjj] - undistLBoundArray[jjj] + 1){
            ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
              "- LocalArray does not match requested element count",
              ESMC_CONTEXT, rc);
            return ESMC_NULL_POINTER;
          }
          ++jjj;
        }
      }
      // adjust all of the bounds to match absolute bounds of F90 pointers
      const int *larrayLBound = larrayListArg[i]->getLbounds();
      j=0;    // reset distributed index
      for (int jj=0; jj<rank; jj++){
        if (arrayToDistGridMapArray[jj]){
          // distributed dimension
          int shift = larrayLBound[jj] - totalLBound[i*redDimCount+j];
          totalLBound[i*redDimCount+j] += shift;
          totalUBound[i*redDimCount+j] += shift;
          computationalLBound[i*redDimCount+j] += shift;
          computationalUBound[i*redDimCount+j] += shift;
          exclusiveLBound[i*redDimCount+j] += shift;
          exclusiveUBound[i*redDimCount+j] += shift;
          ++j;
        }
      }      
      // Use the incoming LocalArray objects directly. No need for a create
      // from copy here because the bounds and Fortran dope vector will stay
      // unchanged.
      larrayList[i] = larrayListArg[i];
    }else{
      // prepare to adjust dope vector
      const int *temp_counts = larrayListArg[i]->getCounts();
      int j=0;    // reset distributed index
      int jjj=0;  // reset undistributed index
      for (int jj=0; jj<rank; jj++){
        if (arrayToDistGridMapArray[jj]){
          // distributed dimension
          if (temp_counts[jj] < 
            totalUBound[i*redDimCount+j] - totalLBound[i*redDimCount+j] + 1){
            ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
              "- LocalArray does not accommodate requested element count",
              ESMC_CONTEXT, rc);
            return ESMC_NULL_POINTER;
          }
          // move the total bounds according to input info
          if (totalLBoundFlag){
            // totalLBound fixed
            temp_larrayLBound[jj] = totalLBound[i*redDimCount+j];
            if (totalUBoundFlag){
              // totalUBound fixed
              // this case allows total alloc. to be larger than total region,
              // but LocalArrayAdjust() method will ultimately prevent this by
              // returning an error if counts are not equal to
              // ubounds-lbounds+1.
              temp_larrayUBound[jj] = totalUBound[i*redDimCount+j];
            }else{
              // totalUBound not fixed
              temp_larrayUBound[jj] = temp_counts[jj]
                + totalLBound[i*redDimCount+j] - 1;
            }
          }else{
            // totalLBound not fixed
            if (totalUBoundFlag){
              // totalUBound fixed
              temp_larrayUBound[jj] = totalUBound[i*redDimCount+j];
              temp_larrayLBound[jj] = totalUBound[i*redDimCount+j]
                - temp_counts[jj] + 1;
            }else{
              // totalLBound and totalUBound not fixed
              // -> shift computational/excl. region into center of total region
              int lBound = computationalLBound[i*redDimCount+j];
              if (exclusiveLBound[i*redDimCount+j] 
                < computationalLBound[i*redDimCount+j])
                lBound = exclusiveLBound[i*redDimCount+j];
              int uBound = computationalUBound[i*redDimCount+j];
              if (exclusiveUBound[i*redDimCount+j]
                > computationalUBound[i*redDimCount+j])
                uBound = exclusiveUBound[i*redDimCount+j];
              temp_larrayLBound[jj] = lBound
                - (int)(0.5 * (temp_counts[jj] - 1 + lBound - uBound));
              temp_larrayUBound[jj] = temp_counts[jj] + temp_larrayLBound[jj]
                - 1;
            }
          }
          totalLBound[i*redDimCount+j] = temp_larrayLBound[jj]; // write back
          totalUBound[i*redDimCount+j] = temp_larrayUBound[jj]; // write back
          ++j;
        }else{
          // non-distributed dimension
          if (temp_counts[jj] !=
            undistUBoundArray[jjj] - undistLBoundArray[jjj] + 1){
            ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
              "- LocalArray does not match requested element count",
              ESMC_CONTEXT, rc);
            return ESMC_NULL_POINTER;
          }
          temp_larrayLBound[jj] = undistLBoundArray[jjj];
          temp_larrayUBound[jj] = undistUBoundArray[jjj];
          ++jjj;
        }
      }
      // Make a copy of the LocalArray object with adjusted undistLBound and
      // undistUBound values. The returned LocalArray object will contain a
      // correctly adjusted Fortran dope vector.
      // Depending on copyflag the original memory used for data storage will
      // either be referenced, or a copy of the data will have been made.
      larrayList[i] = LocalArray::create(larrayListArg[i], copyflag,
        temp_larrayLBound, temp_larrayUBound, &localrc);
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, rc)) return ESMC_NULL_POINTER;
    }        
  }
  delete [] temp_larrayLBound;
  delete [] temp_larrayUBound;
  
  // call class constructor
  try{
    array = new Array(typekind, rank, larrayList, distgrid, false,
      exclusiveLBound, exclusiveUBound, computationalLBound,
      computationalUBound, totalLBound, totalUBound, tensorCount,
      tensorElementCount, undistLBoundArray, undistUBoundArray, 
      distgridToArrayMapArray, arrayToDistGridMapArray,
      distgridToPackedArrayMap, indexflag, &localrc);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      rc)){
      array->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
      return NULL;
    }
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      rc);
    array->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
    return NULL;
  }catch(...){
    // allocation error
    ESMC_LogDefault.MsgAllocError("for new ESMCI::Array.", ESMC_CONTEXT, rc);  
    array->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
    return NULL;
  }
  
  // garbage collection
  delete [] larrayList;
  delete [] exclusiveLBound;
  delete [] exclusiveUBound;
  delete [] computationalLBound;
  delete [] computationalUBound;
  delete [] totalLBound;
  delete [] totalUBound;
  delete [] distgridToArrayMapArray;
  delete [] arrayToDistGridMapArray;
  delete [] distgridToPackedArrayMap;
  if (undistLBoundArrayAllocFlag) delete [] undistLBoundArray;
  if (undistUBoundArrayAllocFlag) delete [] undistUBoundArray;
  
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      rc);
    return NULL;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, rc);
    return NULL;
  }
  
  // return successfully
  if (rc!=NULL) *rc = ESMF_SUCCESS;
  return array;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::create()"
//BOPI
// !IROUTINE:  ESMCI::Array::create
//
// !INTERFACE:
Array *Array::create(
//
// !RETURN VALUE:
//    ESMCI::Array * to newly allocated ESMCI::Array
//
// !ARGUMENTS:
//
  ArraySpec *arrayspec,                       // (in)
  DistGrid *distgrid,                         // (in)
  InterfaceInt *distgridToArrayMap,           // (in)
  InterfaceInt *computationalEdgeLWidthArg,   // (in)
  InterfaceInt *computationalEdgeUWidthArg,   // (in)
  InterfaceInt *computationalLWidthArg,       // (in)
  InterfaceInt *computationalUWidthArg,       // (in)
  InterfaceInt *totalLWidthArg,               // (in)
  InterfaceInt *totalUWidthArg,               // (in)
  ESMC_IndexFlag *indexflagArg,               // (in)
  InterfaceInt *distLBoundArg,                // (in)
  InterfaceInt *undistLBoundArg,              // (in)
  InterfaceInt *undistUBoundArg,              // (in)
  int *rc,                                    // (out) return code
  VM *vm                                      // (in)
  ){
//
// !DESCRIPTION:
//    Create an {\tt ESMCI::Array} object from ArraySpec and DistGrid.
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code
  
  Array *array;
  try{
  
  // check the input and get the information together to call construct()
  // arrayspec -> typekind/rank
  if (arrayspec == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to arrayspec", ESMC_CONTEXT, rc);
    return ESMC_NULL_POINTER;
  }
  ESMC_TypeKind_Flag typekind = arrayspec->getTypeKind(&localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    rc)) return ESMC_NULL_POINTER;
  int rank = arrayspec->getRank(&localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    rc)) return ESMC_NULL_POINTER;
  // distgrid -> delayout, dimCount
  if (distgrid == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to distgrid", ESMC_CONTEXT, rc);
    return ESMC_NULL_POINTER;
  }
  const DELayout *delayout = distgrid->getDELayout();
  int dimCount = distgrid->getDimCount();
  // check if distgridToArrayMap was provided and matches rest of arguments
  int *distgridToArrayMapArray = new int[dimCount];
  for (int i=0; i<dimCount; i++){
    if (i < rank)
      distgridToArrayMapArray[i] = i+1; // default (basis 1)
    else
      distgridToArrayMapArray[i] = 0;   // default (replicator dims beyond rank)
  }
  if (distgridToArrayMap != NULL){
    if (distgridToArrayMap->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- distgridToArrayMap array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (distgridToArrayMap->extent[0] != dimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- distgridToArrayMap and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    memcpy(distgridToArrayMapArray, distgridToArrayMap->array,
      dimCount*sizeof(int));
  }
  {
    // check distgridToArrayMapArray
    bool *check = new bool[rank];
    for (int i=0; i<rank; i++)
      check[i] = false; // initialize
    for (int i=0; i<dimCount; i++){
      if (distgridToArrayMapArray[i] < 0 || distgridToArrayMapArray[i] > rank){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- invalid distgridToArrayMap element", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      if (distgridToArrayMapArray[i] > 0){
        if(check[distgridToArrayMapArray[i]-1]){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
            "- invalid distgridToArrayMap element", ESMC_CONTEXT, rc);
          return ESMC_NULL_POINTER;
        }
        check[distgridToArrayMapArray[i]-1] = true;
      }
    }
    delete [] check;
  }
  // determine replicatorCount
  int replicatorCount = 0;  // initialize
  for (int i=0; i<dimCount; i++)
    if (distgridToArrayMapArray[i] == 0) ++replicatorCount;
  // determine reduced dimCount -> redDimCount
  int redDimCount = dimCount - replicatorCount;
  // determine tensorCount
  int tensorCount = rank - redDimCount;
  if (tensorCount < 0) tensorCount = 0;
  // generate arrayToDistGridMap
  int *arrayToDistGridMapArray = new int[rank];
  for (int i=0; i<rank; i++)
    arrayToDistGridMapArray[i] = 0; // reset  (basis 1), 0 indicates tensor dim
  for (int i=0; i<dimCount; i++)
    if (int j=distgridToArrayMapArray[i])
      arrayToDistGridMapArray[j-1] = i+1;
  // generate distgridToPackedArrayMap - labels the distributed Array dims 1,2,.
  int *distgridToPackedArrayMap = new int[dimCount];
  for (int i=0; i<dimCount; i++)
    distgridToPackedArrayMap[i] = 0; // reset  (basis 1), 0 indicates repl. dim
  {
    int k=1;  // reset
    for (int i=0; i<rank; i++){
      if (int j=arrayToDistGridMapArray[i]){
        distgridToPackedArrayMap[j-1] = k;
        ++k;
      }
    }
  }
  // check for undistLBound and undistUBound arguments and that they match
  // tensorCount
  if (tensorCount > 0 && undistLBoundArg == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Valid undistLBound argument required to create Array with tensor dims",
      ESMC_CONTEXT, rc);
    return ESMC_NULL_POINTER;
  }
  if (tensorCount > 0 && undistUBoundArg == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Valid undistUBound argument required to create Array with tensor dims",
      ESMC_CONTEXT, rc);
    return ESMC_NULL_POINTER;
  }
  int *undistLBoundArray = NULL; // reset
  if (undistLBoundArg != NULL){
    if (undistLBoundArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- undistLBound array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (undistLBoundArg->extent[0] != tensorCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- undistLBound, arrayspec, distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    undistLBoundArray = undistLBoundArg->array;
  }
  int *undistUBoundArray = NULL; // reset
  if (undistUBoundArg != NULL){
    if (undistUBoundArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- undistUBound array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (undistUBoundArg->extent[0] != tensorCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- undistUBound, arrayspec, distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    undistUBoundArray = undistUBoundArg->array;
  }
  // tensorElementCount
  int tensorElementCount = 1;  // prime tensorElementCount
  for (int i=0; i<tensorCount; i++)
    tensorElementCount *= (undistUBoundArray[i] - undistLBoundArray[i] + 1);
  
  // delayout -> deCount, localDeCount, localDeToDeMap
  int deCount = delayout->getDeCount();
  int localDeCount = delayout->getLocalDeCount();
  const int *localDeToDeMap = delayout->getLocalDeToDeMap();
  // distgrid -> indexCountPDimPDe[]
  const int *indexCountPDimPDe = distgrid->getIndexCountPDimPDe();
  // check on indexflag
  ESMC_IndexFlag indexflag = ESMF_INDEX_DELOCAL;  // default
  if (indexflagArg != NULL)
    indexflag = *indexflagArg;
  // check that presence of distLBoundArg is consistent with indexflag
  if(indexflag == ESMF_INDEX_USER){
    if (distLBoundArg == NULL){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
        "- distLBoundArg required in ESMF_INDEX_USER mode", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
  }else{
    if (distLBoundArg != NULL){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
        "- distLBoundArg must only be specified in ESMF_INDEX_USER mode",
        ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
  }
  // figure exclusive region
  int *exclusiveLBound = new int[redDimCount*localDeCount];
  int *exclusiveUBound = new int[redDimCount*localDeCount];
  for (int i=0; i<redDimCount*localDeCount; i++)
    exclusiveLBound[i] = 1; // excl. region starts at (1,1,1...) <- Fortran
  // exlc. region for each DE ends at indexCountPDimPDe of the associated
  // DistGrid
  for (int i=0; i<localDeCount; i++){
    int de = localDeToDeMap[i];
    for (int j=0; j<dimCount; j++)
      if (int k=distgridToPackedArrayMap[j])
        exclusiveUBound[i*redDimCount+k-1] = indexCountPDimPDe[de*dimCount+j];
  }
  // optionally shift origin of exclusive region to pseudo global index space
  if (indexflag == ESMF_INDEX_GLOBAL){
    for (int i=0; i<localDeCount; i++){
      int de = localDeToDeMap[i];
      for (int j=0; j<dimCount; j++){
        // check that this DE/dim has a contiguous index list
        const int *contigFlagPDimPDe = distgrid->getContigFlagPDimPDe();
        if (!contigFlagPDimPDe[de*dimCount+j]){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_NOT_VALID,
            "- Cannot use non-contiguous decomposition for pseudo global"
            " index space", ESMC_CONTEXT, rc);
          return ESMC_NULL_POINTER;
        }
        // obtain indexList for this DE and dim
        const int *indexList =
          distgrid->getIndexListPDimPLocalDe(i, j+1, &localrc);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, rc)) return ESMC_NULL_POINTER;
        // shift bounds of exclusive region to match indexList[0]
        if (indexCountPDimPDe[de*dimCount+j]){
          // only shift if there are indices associated
          if (int k=distgridToPackedArrayMap[j]){
            // only shift if this isn't a replicated dim
            int shift = indexList[0] - exclusiveLBound[i*redDimCount+k-1];
            exclusiveLBound[i*redDimCount+k-1] += shift;
            exclusiveUBound[i*redDimCount+k-1] += shift;
          }
        }
      } // j
    } // i
  }
  // deal with computationalEdge widths
  int *computationalEdgeLWidth = new int[redDimCount];
  int *computationalEdgeUWidth = new int[redDimCount];
  if (computationalEdgeLWidthArg != NULL){
    if (computationalEdgeLWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalEdgeLWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (computationalEdgeLWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- computationalEdgeLWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    memcpy(computationalEdgeLWidth, computationalEdgeLWidthArg->array,
      redDimCount*sizeof(int));
  }else{
    // set default
    for (int i=0; i<redDimCount; i++)
      computationalEdgeLWidth[i] = 0;
  }
  if (computationalEdgeUWidthArg != NULL){
    if (computationalEdgeUWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalEdgeUWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (computationalEdgeUWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- computationalEdgeUWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    memcpy(computationalEdgeUWidth, computationalEdgeUWidthArg->array,
      redDimCount*sizeof(int));
  }else{
    // set default
    for (int i=0; i<redDimCount; i++)
      computationalEdgeUWidth[i] = 0;
  }
  // deal with computational widths
  int *computationalLBound = new int[redDimCount*localDeCount];
  int *computationalUBound = new int[redDimCount*localDeCount];
  if (computationalLWidthArg != NULL){
    if (computationalLWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalLWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (computationalLWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- computationalLWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    for (int j=0; j<localDeCount; j++)
      for (int i=0; i<redDimCount; i++)
        computationalLBound[j*redDimCount+i] = exclusiveLBound[j*redDimCount+i]
          - computationalLWidthArg->array[i];
  }else{
    // set default
    memcpy(computationalLBound, exclusiveLBound,
      localDeCount*redDimCount*sizeof(int));
  }
  if (computationalUWidthArg != NULL){
    if (computationalUWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalUWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (computationalUWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- computationalUWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    for (int j=0; j<localDeCount; j++)
      for (int i=0; i<redDimCount; i++)
        computationalUBound[j*redDimCount+i] = exclusiveUBound[j*redDimCount+i]
          + computationalUWidthArg->array[i];
  }else{
    // set default
    memcpy(computationalUBound, exclusiveUBound,
      localDeCount*redDimCount*sizeof(int));
  }
  // modify computational bounds on tile edges
  for (int j=0; j<localDeCount; j++){
    for (int i=0; i<dimCount; i++){
      if (int k=distgridToPackedArrayMap[i]){
        if (computationalEdgeLWidth[k-1]){
          bool onEdgeL = distgrid->isLocalDeOnEdgeL(j, i+1, &localrc);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, rc)) return ESMC_NULL_POINTER;
          if (onEdgeL)
            computationalLBound[j*redDimCount+k-1]
              -= computationalEdgeLWidth[k-1];
        }
        if (computationalEdgeUWidth[k-1]){
          bool onEdgeU = distgrid->isLocalDeOnEdgeU(j, i+1, &localrc);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, rc)) return ESMC_NULL_POINTER;
          if (onEdgeU)
            computationalUBound[j*redDimCount+k-1]
              += computationalEdgeUWidth[k-1];
        }
      }
    }
  }
  // clean-up
  delete [] computationalEdgeLWidth;
  delete [] computationalEdgeUWidth;
  // deal with total widths
  int *totalLBound = new int[redDimCount*localDeCount];
  int *totalUBound = new int[redDimCount*localDeCount];
  if (totalLWidthArg != NULL){
    if (totalLWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- totalLWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (totalLWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- totalLWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    for (int i=0; i<redDimCount; i++){
      if (totalLWidthArg->array[i] < 0){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- totalLWidth may only contain positive values", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      for (int j=0; j<localDeCount; j++){
        totalLBound[j*redDimCount+i] = exclusiveLBound[j*redDimCount+i]
          - totalLWidthArg->array[i];
      }
    }
  }else{
    // set default
    for (int i=0; i<localDeCount*redDimCount; i++)
      totalLBound[i] = computationalLBound[i];
  }
  if (totalUWidthArg != NULL){
    if (totalUWidthArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- totalUWidth array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (totalUWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- totalUWidth and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    for (int i=0; i<redDimCount; i++){
      if (totalUWidthArg->array[i] < 0){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- totalUWidth may only contain positive values", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      for (int j=0; j<localDeCount; j++){
        totalUBound[j*redDimCount+i] = exclusiveUBound[j*redDimCount+i]
          + totalUWidthArg->array[i];
      }
    }
  }else{
    // set default
    for (int i=0; i<localDeCount*redDimCount; i++)
      totalUBound[i] = computationalUBound[i];
  }
  // total region _must_ cover exclusive region -> adjust total region if not so
  for (int i=0; i<localDeCount*redDimCount; i++){
    if (exclusiveLBound[i] < totalLBound[i])
      totalLBound[i] = exclusiveLBound[i];
    if (exclusiveUBound[i] > totalUBound[i])
      totalUBound[i] = exclusiveUBound[i];
  }
  
  // check computational bounds against total bounds
  for (int i=0; i<localDeCount*redDimCount; i++){
    if (totalLBound[i] <= totalUBound[i]){
      // DE/dim is associated with DistGrid elements
      if (computationalLBound[i] < totalLBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationaLBound / totalLBound mismatch", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      if (computationalLBound[i] > totalUBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationaLBound / totalUBound mismatch", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      if (computationalUBound[i] < totalLBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationalUBound / totalLBound mismatch", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
      if (computationalUBound[i] > totalUBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationalUBound / totalUBound mismatch", ESMC_CONTEXT, rc);
        return ESMC_NULL_POINTER;
      }
    }
  }
  
  // optionally shift to supplied lower bounds
  if (distLBoundArg != NULL){
    if (distLBoundArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- distLBoundArg array must be of rank 1", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    if (distLBoundArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- distLBoundArg and distgrid mismatch", ESMC_CONTEXT, rc);
      return ESMC_NULL_POINTER;
    }
    for (int i=0; i<localDeCount; i++){
      int j=0;    // reset distributed index
      for (int jj=0; jj<rank; jj++){
        if (arrayToDistGridMapArray[jj]){
          // distributed dimension
          // shift all bounds to be consistent with supplied lower bounds
          int shift = distLBoundArg->array[jj] - totalLBound[i*redDimCount+j];
          totalLBound[i*redDimCount+j] += shift;
          totalUBound[i*redDimCount+j] += shift;
          computationalLBound[i*redDimCount+j] += shift;
          computationalUBound[i*redDimCount+j] += shift;
          exclusiveLBound[i*redDimCount+j] += shift;
          exclusiveUBound[i*redDimCount+j] += shift;
          ++j;
        }
      }
    }
  }
  
  // allocate LocalArray list that holds all PET-local DEs
  LocalArray **larrayList = new LocalArray*[localDeCount];
  int *temp_counts = new int[rank];
  int *temp_larrayLBound = new int[rank];
  int *temp_larrayUBound = new int[rank];
  for (int i=0; i<localDeCount; i++){
    int j=0;    // reset distributed index
    int jjj=0;  // reset undistributed index
    for (int jj=0; jj<rank; jj++){
      if (arrayToDistGridMapArray[jj]){
        // distributed dimension
        temp_counts[jj] =
          totalUBound[i*redDimCount+j] - totalLBound[i*redDimCount+j] + 1;
        temp_larrayLBound[jj] = totalLBound[i*redDimCount+j];
        temp_larrayUBound[jj] = totalUBound[i*redDimCount+j];
        ++j;
      }else{
        // non-distributed dimension
        temp_counts[jj] = undistUBoundArray[jjj] - undistLBoundArray[jjj] + 1;
        temp_larrayLBound[jj] = undistLBoundArray[jjj];
        temp_larrayUBound[jj] = undistUBoundArray[jjj];
        ++jjj;
      }
    }
    // allocate LocalArray object with specific undistLBound and undistUBound
    larrayList[i] = LocalArray::create(typekind, rank, temp_counts,
      temp_larrayLBound, temp_larrayUBound, NULL, DATA_REF, &localrc);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
      ESMC_CONTEXT, rc)) return ESMC_NULL_POINTER;
  }
  delete [] temp_counts;
  delete [] temp_larrayLBound;
  delete [] temp_larrayUBound;
  
  // call class constructor
  try{
    array = new Array(typekind, rank, larrayList, distgrid, false,
      exclusiveLBound, exclusiveUBound, computationalLBound,
      computationalUBound, totalLBound, totalUBound, tensorCount,
      tensorElementCount, undistLBoundArray, undistUBoundArray, 
      distgridToArrayMapArray, arrayToDistGridMapArray,
      distgridToPackedArrayMap, indexflag, &localrc, vm);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
      ESMC_CONTEXT, rc)){
      array->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
      return NULL;
    }
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      rc);
    array->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
    return NULL;
  }catch(...){
    // allocation error
    ESMC_LogDefault.MsgAllocError("for new ESMCI::Array.", ESMC_CONTEXT, rc);  
    array->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
    return NULL;
  }
  
  // garbage collection
  delete [] larrayList;
  delete [] exclusiveLBound;
  delete [] exclusiveUBound;
  delete [] computationalLBound;
  delete [] computationalUBound;
  delete [] totalLBound;
  delete [] totalUBound;
  delete [] distgridToArrayMapArray;
  delete [] arrayToDistGridMapArray;
  delete [] distgridToPackedArrayMap;
  
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,\
      rc);
    return NULL;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, rc);
    return NULL;
  }
  
  // return successfully
  if (rc!=NULL) *rc = ESMF_SUCCESS;
  return array;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::create()"
//BOPI
// !IROUTINE:  ESMCI::Array::create
//
// !INTERFACE:
Array *Array::create(
//
// !RETURN VALUE:
//    ESMC_Array * to newly allocated ESMC_Array
//
// !ARGUMENTS:
//
  Array *arrayIn,                             // (in) Array to copy
  int *rc                                     // (out) return code
  ){
//
// !DESCRIPTION:
//    Create an {\tt ESMCI::Array} object as copy of an existing
//    {\tt ESMCI::Array} object.
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code
  
  Array *arrayOut;
  
  try{
    // get an allocation for the new Array object
    try{
      arrayOut = new Array();
    }catch(int localrc){
      // catch standard ESMF return code
      ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
        rc);
      arrayOut->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
      return NULL;
    }catch(...){
      // allocation error
      ESMC_LogDefault.MsgAllocError("for new ESMCI::Array.", ESMC_CONTEXT, rc);  
      arrayOut->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
      return NULL;
    }
    // copy all scalar members and reference members
    arrayOut->typekind = arrayIn->typekind;
    int rank =
      arrayOut->rank = arrayIn->rank;
    arrayOut->indexflag = arrayIn->indexflag;
    int tensorCount =
      arrayOut->tensorCount = arrayIn->tensorCount;
    int tensorElementCount = 
      arrayOut->tensorElementCount = arrayIn->tensorElementCount;
    arrayOut->distgrid = arrayIn->distgrid; // copy reference
    arrayOut->distgridCreator = false;      // not a locally created object
    arrayOut->delayout = arrayIn->delayout; // copy reference
    // deep copy of members with allocations
    // copy the PET-local LocalArray pointers
    int localDeCount = arrayIn->delayout->getLocalDeCount();
    arrayOut->larrayList = new LocalArray*[localDeCount];
    for (int i=0; i<localDeCount; i++){
      arrayOut->larrayList[i] =
        LocalArray::create(arrayIn->larrayList[i], NULL, NULL, &localrc);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, rc)){
        arrayOut->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
        return ESMC_NULL_POINTER;
      }
    }
    // determine the base addresses of the local arrays:
    arrayOut->larrayBaseAddrList = new void*[localDeCount];
    for (int i=0; i<localDeCount; i++)
      arrayOut->larrayBaseAddrList[i] = arrayOut->larrayList[i]->getBaseAddr();
    // copy the PET-local bound arrays
    int redDimCount = rank - tensorCount;
    arrayOut->exclusiveLBound = new int[redDimCount*localDeCount];
    memcpy(arrayOut->exclusiveLBound, arrayIn->exclusiveLBound,
      redDimCount*localDeCount*sizeof(int));
    arrayOut->exclusiveUBound = new int[redDimCount*localDeCount];
    memcpy(arrayOut->exclusiveUBound, arrayIn->exclusiveUBound,
      redDimCount*localDeCount*sizeof(int));
    arrayOut->computationalLBound = new int[redDimCount*localDeCount];
    memcpy(arrayOut->computationalLBound, arrayIn->computationalLBound,
      redDimCount*localDeCount*sizeof(int));
    arrayOut->computationalUBound = new int[redDimCount*localDeCount];
    memcpy(arrayOut->computationalUBound, arrayIn->computationalUBound,
      redDimCount*localDeCount*sizeof(int));
    arrayOut->totalLBound = new int[redDimCount*localDeCount];
    memcpy(arrayOut->totalLBound, arrayIn->totalLBound,
      redDimCount*localDeCount*sizeof(int));
    arrayOut->totalUBound = new int[redDimCount*localDeCount];
    memcpy(arrayOut->totalUBound, arrayIn->totalUBound,
      redDimCount*localDeCount*sizeof(int));
    // tensor dimensions
    arrayOut->undistLBound = new int[tensorCount];
    memcpy(arrayOut->undistLBound, arrayIn->undistLBound,
      tensorCount * sizeof(int));
    arrayOut->undistUBound = new int[tensorCount];
    memcpy(arrayOut->undistUBound, arrayIn->undistUBound,
      tensorCount * sizeof(int));
    // distgridToArrayMap, arrayToDistGridMap and distgridToPackedArrayMap
    int dimCount = arrayIn->distgrid->getDimCount();
    arrayOut->distgridToArrayMap = new int[dimCount];
    memcpy(arrayOut->distgridToArrayMap, arrayIn->distgridToArrayMap,
      dimCount * sizeof(int));
    arrayOut->arrayToDistGridMap = new int[rank];
    memcpy(arrayOut->arrayToDistGridMap, arrayIn->arrayToDistGridMap,
      rank * sizeof(int));
    arrayOut->distgridToPackedArrayMap = new int[dimCount];
    memcpy(arrayOut->distgridToPackedArrayMap,
      arrayIn->distgridToPackedArrayMap, dimCount * sizeof(int));
    // contiguous flag
    arrayOut->contiguousFlag = new int[localDeCount];
    memcpy(arrayOut->contiguousFlag, arrayIn->contiguousFlag,
      localDeCount * sizeof(int));
    // exclusiveElementCountPDe
    int deCount = arrayIn->delayout->getDeCount();
    arrayOut->exclusiveElementCountPDe = new int[deCount];
    memcpy(arrayOut->exclusiveElementCountPDe,
      arrayIn->exclusiveElementCountPDe, deCount * sizeof(int));
    // totalElementCountPLocalDe
    arrayOut->totalElementCountPLocalDe = new int[localDeCount];
    memcpy(arrayOut->totalElementCountPLocalDe,
      arrayIn->totalElementCountPLocalDe, localDeCount * sizeof(int));
    // invalidate the name for this Array object in the Base class
    arrayOut->ESMC_BaseSetName(NULL, "Array");
    
    arrayOut->localDeCountAux = 
      arrayOut->delayout->getLocalDeCount(); // TODO: auxilary for garb
                                             // TODO: until ref. counting
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      rc);
    arrayOut->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
    return NULL;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, rc);
    arrayOut->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);  // mark invalid
    return NULL;
  }
  
  // reset the distgridCreator flag in the src Array, because the newly
  // created Array will now point to the same DistGrid by reference
  // -> leave it up to ESMF automatic garbage collection to clean up the
  // DistGrid when it is time
  arrayIn->distgridCreator = false; // drop ownership of the referenced DistGrid

  // return successfully
  if (rc!=NULL) *rc = ESMF_SUCCESS;
  return arrayOut;
}
//-----------------------------------------------------------------------------
    
    
//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::destroy()"
//BOPI
// !IROUTINE:  ESMCI::Array::destroy
//
// !INTERFACE:
int Array::destroy(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  Array **array){  // in - ESMC_Array to destroy
//
// !DESCRIPTION:
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  // return with errors for NULL pointer
  if (array == ESMC_NULL_POINTER || *array == ESMC_NULL_POINTER){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to Array", ESMC_CONTEXT, &rc);
    return rc;
  }

  try{
    // destruct Array object
    (*array)->destruct();
    // mark as invalid object
    (*array)->ESMC_BaseSetStatus(ESMF_STATUS_INVALID);
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// data copy()
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::copy()"
//BOPI
// !IROUTINE:  ESMCI::Array::copy
//
// !INTERFACE:
//
int Array::copy(
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  Array const *arrayIn                // (in) Array to copy data from
  ){
//
// !DESCRIPTION:
//    Copy data from one Array object to another
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  try{
  
    if (!match(this, arrayIn)){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- Arrays must match for data copy", ESMC_CONTEXT, &rc);
      return rc;
    }
    
    // do the actual data copy
    int const dataSize = ESMC_TypeKind_FlagSize(typekind);
    int const localDeCount = delayout->getLocalDeCount();
    for (int i=0; i<localDeCount; i++){
      int size =
        totalElementCountPLocalDe[i]*tensorElementCount*dataSize;  // bytes
      memcpy(larrayBaseAddrList[i], arrayIn->larrayBaseAddrList[i], size);
    }
      
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(exception &x){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      x.what(), ESMC_CONTEXT, &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// get() and set()
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::getLinearIndexExclusive()"
//BOPI
// !IROUTINE:  ESMCI::Array::getLinearIndexExclusive
//
// !INTERFACE:
int Array::getLinearIndexExclusive(
//
// !RETURN VALUE:
//    int linear index
//
// !ARGUMENTS:
//
  int localDe,                      // in - local DE
  int const *index,                 // in - DE-local index tuple in exclusive
                                    //      region basis 0
  int *rc                           // out - return code
  )const{
//
// !DESCRIPTION:
//    Get linear index - assuming index input to be basis 0 in excl. region
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code

  // determine the linearized index
  int redDimCount = rank - tensorCount;
  int joff = localDe*redDimCount;   // offset according to localDe index
  int linIndex = 0;                 // reset
  int tensorIndex = tensorCount-1;  // reset
  int packedIndex = redDimCount-1;  // reset
  for (int jj=rank-1; jj>=0; jj--){
    if (arrayToDistGridMap[jj]){
      // decomposed dimension 
      int j = packedIndex;
      // first time multiply with zero intentionally:
      linIndex *= totalUBound[joff+j] - totalLBound[joff+j] + 1;
      linIndex += exclusiveLBound[joff+j] - totalLBound[joff+j] + index[jj];    
      --packedIndex;
    }else{
      // tensor dimension
      // first time multiply with zero intentionally:
      linIndex *= undistUBound[tensorIndex] - undistLBound[tensorIndex] + 1;
      linIndex += index[jj];
      --tensorIndex;
    }
  }
   
  // return successfully
  if (rc!=NULL) *rc = ESMF_SUCCESS;
  return linIndex;    // leave this index basis 0
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::getSequenceIndexExclusive()"
//BOPI
// !IROUTINE:  ESMCI::Array::getSequenceIndexExclusive
//
// !INTERFACE:
SeqIndex Array::getSequenceIndexExclusive(
//
// !RETURN VALUE:
//    SeqIndex sequence index
//
// !ARGUMENTS:
//
  int localDe,                      // in  - local DE
  int const *index,                 // in  - DE-local index tuple in exclusive
                                    //       region basis 0
  int depth,                        // in  - topology recursions depth
  int *rc                           // out - return code
  )const{
//
// !DESCRIPTION:
//    Get sequential index - assuming index input to be basis 0 in excl. region,
//    but allowing the incoming index to be outside of the exclusive region
//    under special conditions. See the used DistGrid method for details.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code
  
  // initialize seqIndex
  SeqIndex seqIndex;  // invalidated by constructor

  // prepare decompIndex for decomposed dimensions in the DistGrid order
  int dimCount = distgrid->getDimCount();
  int *decompIndex = new int[dimCount];
  for (int i=0; i<dimCount; i++){
    if (distgridToArrayMap[i] > 0){
      // DistGrid dim associated with Array dim
      decompIndex[i] = index[distgridToArrayMap[i]-1];
    }else{
      // DistGrid dim _not_ associated with Array dim
      decompIndex[i] = 0; // use smallest seq index for replicated dims
    }
  }
  // determine the sequentialized index for decomposed dimensions
  int decompSeqIndex;
  decompSeqIndex = distgrid->getSequenceIndexLocalDe(localDe, decompIndex,
    depth, &localrc);  
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    rc)) return seqIndex;
  seqIndex.decompSeqIndex = decompSeqIndex;
  
  // garbage collection
  delete [] decompIndex;
  
  // determine sequentialized index for tensor dimensions
  seqIndex.tensorSeqIndex = getTensorSequenceIndex(index);
  
  // return successfully
  if (rc!=NULL) *rc = ESMF_SUCCESS;
  return seqIndex;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::getSequenceIndexTile()"
//BOPI
// !IROUTINE:  ESMCI::Array::getSequenceIndexTile
//
// !INTERFACE:
SeqIndex Array::getSequenceIndexTile(
//
// !RETURN VALUE:
//    SeqIndex sequence index
//
// !ARGUMENTS:
//
  int tile,                        // in - tile = {1,..., tileCount}
  const int *index,                 // in - index tuple within tile 
                                    //    - basis 0
  int *rc                           // out - return code
  )const{
//
// !DESCRIPTION:
//    Get sequential index - assuming index input to be basis 0 in tile region
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code
  
  // initialize seqIndex
  SeqIndex seqIndex;
  seqIndex.decompSeqIndex = seqIndex.tensorSeqIndex = -1;

  // prepare decompIndex for decomposed dimensions in the DistGrid order
  int dimCount = distgrid->getDimCount();
  int *decompIndex = new int[dimCount];
  for (int i=0; i<dimCount; i++){
    if (distgridToArrayMap[i] > 0){
      // DistGrid dim associated with Array dim
      decompIndex[i] = index[distgridToArrayMap[i]-1];
    }else{
      // DistGrid dim _not_ associated with Array dim
      decompIndex[i] = 0;
      // use smallest seq index for replicated dims
    }
  }
  // determine the sequentialized index for decomposed dimensions
  int decompSeqIndex;
  decompSeqIndex = distgrid->getSequenceIndexTileRelative(tile, decompIndex,
    0, &localrc);  
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    rc)) return seqIndex;
  seqIndex.decompSeqIndex = decompSeqIndex;
  
  // garbage collection
  delete [] decompIndex;
        
  // determine sequentialized index for tensor dimensions
  seqIndex.tensorSeqIndex = getTensorSequenceIndex(index);
  
  // return successfully
  if (rc!=NULL) *rc = ESMF_SUCCESS;
  return seqIndex;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::getTensorSequenceIndex()"
//BOPI
// !IROUTINE:  ESMCI::Array::getTensorSequenceIndex
//
// !INTERFACE:
int Array::getTensorSequenceIndex(
//
// !RETURN VALUE:
//    Tensor sequence index
//
// !ARGUMENTS:
//
  const int *index,                 // in - index tuple basis 0
  int *rc                           // out - return code
  )const{
//
// !DESCRIPTION:
//    Get tensor sequential index - assuming index input to be basis 0
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code
  
  // determine the sequentialized index for tensor dimensions
  int tensorSeqIndex = 0;             // reset
  int tensorIndex = tensorCount - 1;  // reset
  for (int jj=rank-1; jj>=0; jj--){
    if (arrayToDistGridMap[jj]==0){
      // tensor dimension
      // first time multiply with zero intentionally:
      tensorSeqIndex *= undistUBound[tensorIndex] - undistLBound[tensorIndex]
      + 1;
      tensorSeqIndex += index[jj];
      --tensorIndex;
    }
  }
  ++tensorSeqIndex; // shift tensor sequentialized index to basis 1 !!!!
  
  // return successfully
  if (rc!=NULL) *rc = ESMF_SUCCESS;
  return tensorSeqIndex;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::getArbSequenceIndexOffset()"
//BOPI
// !IROUTINE:  ESMCI::Array::getArbSequenceIndexOffset
//
// !INTERFACE:
int Array::getArbSequenceIndexOffset(
//
// !RETURN VALUE:
//    Arbitrary sequence index offset
//
// !ARGUMENTS:
//
  const int *index,                 // in - index tuple basis 0
  int *rc                           // out - return code
  )const{
//
// !DESCRIPTION:
//    Get offset into arb sequential index list - assuming index input to be
// basis 0
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code
  
  // determine the sequentialized index for tensor dimensions
  int arbSeqIndexOffset = 0;             // reset
  int arbIndex = (rank - tensorCount) - 1;  // reset
  for (int jj=rank-1; jj>=0; jj--){
    if (arrayToDistGridMap[jj]==1){
      // decomposed dimension
      // first time multiply with zero intentionally:
      arbSeqIndexOffset *= totalUBound[arbIndex] - totalLBound[arbIndex] + 1;
      arbSeqIndexOffset += index[jj];
      --arbIndex;
    }
  }
  
  // return successfully
  if (rc!=NULL) *rc = ESMF_SUCCESS;
  return arbSeqIndexOffset;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::setComputationalLWidth()"
//BOPI
// !IROUTINE:  ESMCI::Array::setComputationalLWidth
//
// !INTERFACE:
int Array::setComputationalLWidth(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  InterfaceInt *computationalLWidthArg        // (in)
  ){
//
// !DESCRIPTION:
//    Set computationalLWidth for all local DEs.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // check computationalLWidthArg input and process
  if (computationalLWidthArg != NULL){
    if (computationalLWidthArg->dimCount != 2){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalLWidth array must be of rank 2", ESMC_CONTEXT, &rc);
      return rc;
    }
    int redDimCount = rank - tensorCount;
    if (computationalLWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- 1st dim of computationalLWidth argument must be of size redDimCount",
        ESMC_CONTEXT, &rc);
      return rc;
    }
    int localDeCount = delayout->getLocalDeCount();
    if (computationalLWidthArg->extent[1] != localDeCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- 2nd dim of computationalLWidth argument must be of size "
        "localDeCount", ESMC_CONTEXT, &rc);
      return rc;
    }
    int *computationalLBoundNew = new int[redDimCount*localDeCount];
    for (int i=0; i<localDeCount*redDimCount; i++){
      computationalLBoundNew[i] = exclusiveLBound[i]
        - computationalLWidthArg->array[i];
      if (computationalLBoundNew[i] < totalLBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationaLWidth below totalLBound -> not updated", ESMC_CONTEXT,
          &rc);
        return rc;
      }
      if (computationalLBoundNew[i] > totalUBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationaLWidth above totalUBound -> not updated", ESMC_CONTEXT,
          &rc);
        return rc;
      }
    }
    // update computationalLBound
    memcpy(computationalLBound, computationalLBoundNew,
      redDimCount*localDeCount*sizeof(int));
    delete [] computationalLBoundNew;
  }

  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::setComputationalUWidth()"
//BOPI
// !IROUTINE:  ESMCI::Array::setComputationalUWidth
//
// !INTERFACE:
int Array::setComputationalUWidth(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  InterfaceInt *computationalUWidthArg        // (in)
  ){
//
// !DESCRIPTION:
//    Set computationalUWidth for all local DEs.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // check computationalUWidthArg input and process
  if (computationalUWidthArg != NULL){
    if (computationalUWidthArg->dimCount != 2){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- computationalUWidth array must be of rank 2", ESMC_CONTEXT, &rc);
      return rc;
    }
    int redDimCount = rank - tensorCount;
    if (computationalUWidthArg->extent[0] != redDimCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- 1st dim of computationalUWidth argument must be of size redDimCount",
        ESMC_CONTEXT, &rc);
      return rc;
    }
    int localDeCount = delayout->getLocalDeCount();
    if (computationalUWidthArg->extent[1] != localDeCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- 2nd dim of computationalUWidth argument must be of size "
        "localDeCount", ESMC_CONTEXT, &rc);
      return rc;
    }
    int *computationalLBoundNew = new int[redDimCount*localDeCount];
    for (int i=0; i<localDeCount*redDimCount; i++){
      computationalLBoundNew[i] = exclusiveLBound[i]
        - computationalUWidthArg->array[i];
      if (computationalLBoundNew[i] < totalLBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationaUWidth below totalLBound -> not updated", ESMC_CONTEXT,
          &rc);
        return rc;
      }
      if (computationalLBoundNew[i] > totalUBound[i]){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- computationaUWidth above totalUBound -> not updated", ESMC_CONTEXT,
          &rc);
        return rc;
      }
    }
    // update computationalLBound
    memcpy(computationalLBound, computationalLBoundNew,
      redDimCount*localDeCount*sizeof(int));
    delete [] computationalLBoundNew;
  }

  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::setRimSeqIndex()"
//BOPI
// !IROUTINE:  ESMCI::Array::setRimSeqIndex
//
// !INTERFACE:
int Array::setRimSeqIndex(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  int localDe,                        // (in)
  InterfaceInt *rimSeqIndexArg        // (in)
  ){
//
// !DESCRIPTION:
//    Set rimSeqIndex for localDe.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // ensure localDe is in range
  if (localDe < 0 || localDe >= delayout->getLocalDeCount()){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_OUTOFRANGE,
      "- localDe out of range", ESMC_CONTEXT, &rc);
    return rc;  // bail out
  }

  // check rimSeqIndexArg input and process
  if (rimSeqIndexArg != NULL){
    if (rimSeqIndexArg->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- rimSeqIndexArg array must be of rank 1", ESMC_CONTEXT, &rc);
      return rc;  // bail out
    }
    if (rimSeqIndexArg->extent[0]*tensorElementCount 
      != rimSeqIndex[localDe].size()){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- rimSeqIndexArg argument must be of size rimSeqIndex[localDe].size()",
        ESMC_CONTEXT, &rc);
      return rc;  // bail out
    }

    // fill Array rim elements with sequence indices provided in rimSeqIndexArg
    ArrayElement arrayElement(this, localDe, true);
    int i=0;
    int offStart = arrayElement.getArbSequenceIndexOffset();    
    while(arrayElement.isWithin()){
      int off = arrayElement.getArbSequenceIndexOffset() - offStart;
      rimSeqIndex[localDe][i].decompSeqIndex = rimSeqIndexArg->array[off];
      rimSeqIndex[localDe][i].tensorSeqIndex =
        arrayElement.getTensorSequenceIndex();
#if 0
printf("setRimSeqIndex(): %d, %d, %d, (%d, %d), %d\n", i, offStart, off, 
  rimSeqIndex[localDe][i].decompSeqIndex,
  rimSeqIndex[localDe][i].tensorSeqIndex,
  rimLinIndex[localDe][i]);
#endif 
      ++i;
      arrayElement.next();  // next element
    } // multi dim index loop
  }

  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// misc.
//
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::match()"
//BOPI
// !IROUTINE:  ESMCI::Array::match
//
// !INTERFACE:
bool Array::match(
//
// !RETURN VALUE:
//    bool according to match
//
// !ARGUMENTS:
//
  Array const *array1,                    // in
  Array const *array2,                    // in
  int *rc                                 // (out) return code
  ){
//
//TODO: 1) rename this method to congruent()
//TODO: 2) consider weakly and strongly congruent case
//
// !DESCRIPTION:
//    Determine if array1 and array2 match.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  if (rc!=NULL) *rc = ESMC_RC_NOT_IMPL;   // final return code

  // initialize return value
  bool matchResult = false;
  
  // return with errors for NULL pointer
  if (array1 == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to Array", ESMC_CONTEXT, rc);
    return matchResult;
  }
  if (array2 == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to Array", ESMC_CONTEXT, rc);
    return matchResult;
  }
  
  // check if Array pointers are identical
  if (array1 == array2){
    // pointers are identical -> nothing more to check
    matchResult = true;
    if (rc!=NULL) *rc = ESMF_SUCCESS; // bail out successfully
    return matchResult;
  }
  
  // check typekind, rank match
  if (array1->typekind != array2->typekind){
    matchResult = false;
    if (rc!=NULL) *rc = ESMF_SUCCESS; // bail out successfully
    return matchResult;
  }
  if (array1->rank != array2->rank){
    matchResult = false;
    if (rc!=NULL) *rc = ESMF_SUCCESS; // bail out successfully
    return matchResult;
  }

  // compare the distgrid members
  matchResult = DistGrid::match(array1->getDistGrid(), array2->getDistGrid(),
    &localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    rc)) return rc;
  if (matchResult==false){
    if (rc!=NULL) *rc = ESMF_SUCCESS; // bail out successfully
    return matchResult;
  }
  
  // compare Array members to ensure DE-local tiles are congruent
  if (array1->tensorCount != array2->tensorCount){
    matchResult = false;
    if (rc!=NULL) *rc = ESMF_SUCCESS; // bail out successfully
    return matchResult;
  }
  if (array1->tensorElementCount != array2->tensorElementCount){
    matchResult = false;
    if (rc!=NULL) *rc = ESMF_SUCCESS; // bail out successfully
    return matchResult;
  }
  int localDeCount = array1->getDistGrid()->getDELayout()->getLocalDeCount();
  int *int1 = array1->totalElementCountPLocalDe;
  int *int2 = array2->totalElementCountPLocalDe;
  for (int i=0; i<localDeCount; i++){
    if (int1[i] != int2[i]){
      matchResult = false;
      if (rc!=NULL) *rc = ESMF_SUCCESS; // bail out successfully
      return matchResult;
    }
  }

  // return successfully indicating match
  matchResult = true;
  if (rc!=NULL) *rc = ESMF_SUCCESS; // bail out successfully
  return matchResult;
}
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::read()"
//BOPI
// !IROUTINE:  ESMCI::Array::read
//
// !INTERFACE:
int Array::read(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  const char  *file,              // in    - name of file being read
  const char  *variableName,      // in    - optional variable name
  int   *timeslice,               // in    - timeslice option
  ESMC_IOFmt_Flag *iofmt          // in    - IO format flag
  ){
//
// !DESCRIPTION:
//   Read Array data from file and put it into an ESMF_Array object.
//   For this API to be functional, the environmentvariable ESMF_PIO
//   should be set to "internal" when the ESMF library is built.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  ESMC_IOFmt_Flag localiofmt;

  IO *newIO = IO::create(&localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  // For here on, we have to be sure to clean up before returning
  if (ESMF_SUCCESS == localrc) {
    localrc = newIO->addArray(this, variableName);
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
  }
  // Set optional parameters which are not optional at next layer
  if ((ESMC_IOFmt_Flag *)NULL != iofmt) {
    localiofmt = *iofmt;
  } else {
    localiofmt = ESMF_IOFMT_NETCDF;
  }
  // It is an error to supply a variable name if not in NetCDF mode
  if ((ESMF_IOFMT_NETCDF != localiofmt) &&
      ((char *)NULL != variableName) && (strlen(variableName) > 0)) {
    ESMC_LogDefault.Write("Array variable name not allowed in binary mode",
                          ESMC_LOGMSG_ERROR, ESMC_CONTEXT);
    return ESMF_RC_ARG_BAD;
  }
  if (ESMF_SUCCESS == localrc) {
    // Call the IO read function
    localrc = newIO->read(file, localiofmt, timeslice);
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
  }

  // cleanup
  IO::destroy(&newIO);
  newIO = (IO *)NULL;

  // return
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::write()"
//BOPI
// !IROUTINE:  ESMCI::Array::write
//
// !INTERFACE:
int Array::write(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  const char  *file,              // in    - name of file being written
  const char  *variableName,      // in    - optional variable name
  bool  *overwrite,               // in    - OK to overwrite file data
  ESMC_FileStatus_Flag *status,   // in    - file status flag
  int   *timeslice,               // in    - timeslice option
  ESMC_IOFmt_Flag *iofmt          // in    - IO format flag
  ){
//
// !DESCRIPTION:
//   Write Array data into a file. For this API to be functional, the 
//   environment variable {\tt ESMF\_PIO} should be set to "internal" when 
//   the ESMF library is built.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  ESMC_IOFmt_Flag localiofmt;              // For default handling
  bool localoverwrite;                    // For default handling
  ESMC_FileStatus_Flag localstatus;        // For default handling

  IO *newIO = IO::create(&rc);
  if (ESMC_LogDefault.MsgFoundError(rc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)){
    return rc;
  }
  // From now on, we have to be sure to clean up before returning
  if (ESMF_SUCCESS == rc) {
    rc = newIO->addArray(this, variableName);
    ESMC_LogDefault.MsgFoundError(rc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc);
  }

  // Handle format default
  if ((ESMC_IOFmt_Flag *)NULL == iofmt) {
    localiofmt = ESMF_IOFMT_NETCDF;
  } else {
    localiofmt = *iofmt;
  }
  // Handle overwrite default
  if ((bool *)NULL == overwrite) {
    localoverwrite = false;
  } else {
    localoverwrite = *overwrite;
  }
  // Handle status default
  if ((ESMC_FileStatus_Flag *)NULL == status) {
    localstatus = ESMC_FILESTATUS_UNKNOWN;
  } else {
    localstatus = *status;
  }
  if (ESMF_SUCCESS == rc) {
    // It is an error to supply a variable name in binary mode
    if ((ESMF_IOFMT_BIN == localiofmt) && (strlen(variableName) > 0)) {
      ESMC_LogDefault.Write("Array variable name not allowed in binary mode",
                            ESMC_LOGMSG_ERROR, ESMC_CONTEXT);
      rc = ESMF_RC_ARG_BAD;
    }
  }
  if (ESMF_SUCCESS == rc) {
    // Call the IO write function
    rc = newIO->write(file, localiofmt,
                      localoverwrite, localstatus, timeslice);
    ESMC_LogDefault.MsgFoundError(rc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc);
  }

  // cleanup
  IO::destroy(&newIO);
  newIO = (IO *)NULL;

  // return 
  return rc;
}
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::print()"
//BOPI
// !IROUTINE:  ESMCI::Array::print
//
// !INTERFACE:
int Array::print()const{
//
// !RETURN VALUE:
//    int return code
//
//
// !DESCRIPTION:
//    Print details of Array object 
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // return with errors for NULL pointer
  if (this == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to Array", ESMC_CONTEXT, &rc);
    return rc;
  }

  // print info about the ESMCI::Array object
  printf("--- ESMCI::Array::print start ---\n");
  ESMC_Print(); // print the Base class info
  printf("Array typekind/rank: %s / %d \n", ESMC_TypeKind_FlagString(typekind),
    rank);
  printf("~ lower class' values ~\n");
  int dimCount = distgrid->getDimCount();
  printf("DistGrid dimCount = %d\n", dimCount);
  int deCount = delayout->getDeCount();
  printf("deCount = %d\n", deCount);
  int localDeCount = delayout->getLocalDeCount();
  printf("localDeCount = %d\n", localDeCount);
  const int *localDeToDeMap = delayout->getLocalDeToDeMap();
  const int redDimCount = rank - tensorCount;
  for (int i=0; i<localDeCount; i++){
    int de = localDeToDeMap[i];
    printf("~ local data in LocalArray for DE %d ~\n", de);
    larrayList[i]->print();
    if (exclusiveElementCountPDe[de]){
      // associated DE
      int j=0;    // reset
      int jjj=0;  // reset
      for (int jj=0; jj<rank; jj++){
        if (arrayToDistGridMap[jj]){
          // distributed dimension
          printf("dim %d: [%d]: [%d [%d [%d, %d] %d] %d]\n", 
            jj+1, j, 
            totalLBound[i*redDimCount+j], computationalLBound[i*redDimCount+j],
            exclusiveLBound[i*redDimCount+j], exclusiveUBound[i*redDimCount+j],
            computationalUBound[i*redDimCount+j], totalUBound[i*redDimCount+j]);
          ++j;
        }else{
          // non-distributed dimension
          printf("dim %d: undistLBound[%d]=%d            undistUBound[%d]=%d\n",
            jj+1, jjj, undistLBound[jjj], jjj, undistUBound[jjj]);
          ++jjj;
        }
      }
    }else{
      // unassociated DE
      printf("this DE is not associated with DistGrid points\n");
    }
  }
  printf("--- ESMCI::Array::print end ---\n");
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::validate()"
//BOPI
// !IROUTINE:  ESMCI::Array::validate
//
// !INTERFACE:
int Array::validate()const{
//
// !RETURN VALUE:
//    int return code
//
//
// !DESCRIPTION:
//    Validate details of Array object 
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // check against NULL pointer
  if (this == ESMC_NULL_POINTER){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
       " - 'this' pointer is NULL.", ESMC_CONTEXT, &rc);
    return rc;
  }

  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::constructFileMap()"
//BOPI
// !IROUTINE:  ESMCI::Array::constructFileMap
//
// !INTERFACE:
//
int Array::constructFileMap(
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  int64_t *fileMapList,         // (in)  - Array of map elements to fill
  int mapListSize,              // (in)  - Number of elements in fileMapList
  int localDe,                  // (in)  - local DE = {0, ..., localDeCount-1}
  int64_t unmap_val             // (in)  - value to give to unmapped elements
  )const{
//
// !DESCRIPTION:
//    Construct the map between each local array element and its location
//    in a disk file for the total Array region on localDe.
//    Unmapped elements (those outside of exclusive region) are given a value
//    of of unmap_val (defaults to zero which is the PIO convention).
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  try{
  
    if (tensorElementCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- fileMapList is only defined for tensorElementCount == 1",
        ESMC_CONTEXT, &rc);
      return rc;
    }
    ArrayElement arrayElement(this, localDe, false);  // also checks localDe
    int elementCount = totalElementCountPLocalDe[localDe];
  
    if (fileMapList != NULL){
      // fileMapList provided -> error checking
      if (mapListSize < elementCount){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
          "- fileMapList array insufficiently sized", ESMC_CONTEXT, &rc);
        return rc;
      }
      // fill fileMapList
      //TODO: Make the following more efficient by obtaining the seqIndex list
      //TODO: from DistGrid for all the exclusive elements once, and then copy
      //TODO: from that list into fileMapList for each exclusive element.
      int element = 0;
      while(arrayElement.isWithin()){
        if (arrayElement.isWithinWatch()){
          // within exclusive Array region -> obtain seqIndex value
          SeqIndex seqIndex = arrayElement.getSequenceIndexExclusive(0);
          fileMapList[element] = seqIndex.decompSeqIndex;
        }else{
          // outside exclusive Array region -> mark this as unmapped element
          fileMapList[element] = unmap_val;
        }
        arrayElement.next();
        ++element;
      }
    }
  
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(exception &x){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      x.what(), ESMC_CONTEXT, &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
//
// serialize() and deserialize()
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::serialize()"
//BOPI
// !IROUTINE:  ESMCI::Array::serialize - Turn Array into a byte stream
//
// !INTERFACE:
int Array::serialize(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
  char *buffer,          // inout - byte stream to fill
  int *length,           // inout - buf length
  int *offset,           // inout - original offset
  const ESMC_AttReconcileFlag &attreconflag,     // in - attreconcile flag
  const ESMC_InquireFlag &inquireflag) const {   // in - inquiry flag
//
// !DESCRIPTION:
//    Turn info in array class into a stream of bytes.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // Prepare pointer variables of different types
  char *cp;
  int *ip;
  ESMC_TypeKind_Flag *dkp;
  ESMC_IndexFlag *ifp;
  int r;

  // Check if buffer has enough free memory to hold object
  if ((inquireflag != ESMF_INQUIREONLY) && (*length - *offset) < sizeof(Array)){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
      "Buffer too short to add an Array object", ESMC_CONTEXT, &rc);
    return rc;
  }

  // Serialize the Base class,
  r=*offset%8;
  if (r!=0) *offset += 8-r;  // alignment
  localrc = ESMC_Base::ESMC_Serialize(buffer, length, offset, attreconflag,
    inquireflag);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  // Serialize the DistGrid
  localrc = distgrid->serialize(buffer, length, offset, inquireflag);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  // Serialize Array meta data
  r=*offset%8;
  if (r!=0) *offset += 8-r;  // alignment
  dkp = (ESMC_TypeKind_Flag *)(buffer + *offset);
  if (inquireflag != ESMF_INQUIREONLY)
    *dkp++ = typekind;
  else
     dkp++;
  ip = (int *)dkp;
  if (inquireflag != ESMF_INQUIREONLY)
    *ip++ = rank;
  else
    ip++;
  ifp = (ESMC_IndexFlag *)ip;
  if (inquireflag != ESMF_INQUIREONLY)
    *ifp++ = indexflag;
  else
    ifp++;
  ip = (int *)ifp;
  if (inquireflag != ESMF_INQUIREONLY) {
    *ip++ = tensorCount;
    for (int i=0; i<tensorCount; i++){
      *ip++ = undistLBound[i];
      *ip++ = undistUBound[i];
    }
    for (int i=0; i<distgrid->getDimCount(); i++)
      *ip++ = distgridToArrayMap[i];
    for (int i=0; i<rank; i++)
      *ip++ = arrayToDistGridMap[i];
    for (int i=0; i<distgrid->getDimCount(); i++)
      *ip++ = distgridToPackedArrayMap[i];
    *ip++ = tensorElementCount;
    for (int i=0; i<delayout->getDeCount(); i++)
      *ip++ = exclusiveElementCountPDe[i];
  } else
    ip += 1 + 4*tensorCount + 2*distgrid->getDimCount () +
      rank + tensorElementCount +delayout->getDeCount ();
  
  // fix offset  
  cp = (char *)ip;
  *offset = (cp - buffer);
  
  if (inquireflag == ESMF_INQUIREONLY)
    if (*offset < sizeof (Array))
      *offset = sizeof (Array);
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::deserialize()"
//BOPI
// !IROUTINE:  ESMCI::Array::deserialize - Turn a byte stream into an Array
//
// !INTERFACE:
int Array::deserialize(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
  char *buffer,          // in - byte stream to read
  int *offset,           // inout - original offset
  const ESMC_AttReconcileFlag &attreconflag) {  // in - attreconcile flag
//
// !DESCRIPTION:
//    Turn a stream of bytes into an object.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // Prepare pointer variables of different types
  char *cp;
  int *ip;
  ESMC_TypeKind_Flag *dkp;
  ESMC_IndexFlag *ifp;
  int r;

  // Deserialize the Base class
  r=*offset%8;
  if (r!=0) *offset += 8-r;  // alignment
  localrc = ESMC_Base::ESMC_Deserialize(buffer, offset, attreconflag);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  // Deserialize the DistGrid
  distgrid = DistGrid::deserialize(buffer, offset);
  distgridCreator = true;  // deserialize creates a local object
  // Pull DELayout out of DistGrid
  delayout = distgrid->getDELayout();
  // Deserialize Array meta data
  r=*offset%8;
  if (r!=0) *offset += 8-r;  // alignment
  dkp = (ESMC_TypeKind_Flag *)(buffer + *offset);
  typekind = *dkp++;
  ip = (int *)dkp;
  rank = *ip++;
  ifp = (ESMC_IndexFlag *)ip;
  indexflag = *ifp++;
  ip = (int *)ifp;
  tensorCount = *ip++;
  undistLBound = new int[tensorCount];
  undistUBound = new int[tensorCount];
  for (int i=0; i<tensorCount; i++){
    undistLBound[i] = *ip++;
    undistUBound[i] = *ip++;
  }
  distgridToArrayMap = new int[distgrid->getDimCount()];
  for (int i=0; i<distgrid->getDimCount(); i++)
    distgridToArrayMap[i] = *ip++;
  arrayToDistGridMap = new int[rank];
  for (int i=0; i<rank; i++)
    arrayToDistGridMap[i] = *ip++;
  distgridToPackedArrayMap = new int[distgrid->getDimCount()];
  for (int i=0; i<distgrid->getDimCount(); i++)
    distgridToPackedArrayMap[i] = *ip++;
  tensorElementCount = *ip++;
  exclusiveElementCountPDe = new int[delayout->getDeCount()];
  for (int i=0; i<delayout->getDeCount(); i++)
    exclusiveElementCountPDe[i] = *ip++;
  
  // fix offset
  cp = (char *)ip;
  *offset = (cp - buffer);
  
  // set values with local dependency
  larrayList = new LocalArray*[0];     // no DE on proxy object
  larrayBaseAddrList = new void*[0];        // no DE on proxy object
  totalElementCountPLocalDe = NULL;         // no De on proxy object

  localDeCountAux = delayout->getLocalDeCount(); // TODO: auxilary for garb
                                                 // TODO: until ref. counting

  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}


//-----------------------------------------------------------------------------
//
// comms
//
//-----------------------------------------------------------------------------

//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::gather()"
//BOPI
// !IROUTINE:  ESMCI::Array::gather
//
// !INTERFACE:
int Array::gather(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  void *arrayArg,                       // out -
  ESMC_TypeKind_Flag typekindArg,       // in -
  int rankArg,                          // in -
  int *counts,                          // in -
  int *tileArg,                         // in -
  int rootPet,                          // in -
  VM *vm                                // in -
  ){
//
//
// !DESCRIPTION:
//    Gather Array object into native array.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // return with errors for NULL pointer
  if (this == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to Array", ESMC_CONTEXT, &rc);
    return rc;
  }

  // by default use the currentVM for vm
  if (vm == ESMC_NULL_POINTER){
    vm = VM::getCurrent(&localrc);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
  }

  // query the VM for localPet and petCount
  int localPet = vm->getLocalPet();
  int petCount = vm->getPetCount();
  
  // deal with optional tile argument
  int tile = 1;  // default
  if (tileArg)
    tile = *tileArg;
  int tileCount = distgrid->getTileCount();
  if (tile < 1 || tile > tileCount){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
      "- Specified tile out of bounds", ESMC_CONTEXT, &rc);
    return rc;
  }
  const int *tileListPDe = distgrid->getTileListPDe();

  // get minIndexPDim and maxIndexPDim for tile
  const int *minIndexPDim = distgrid->getMinIndexPDimPTile(tile, &localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  const int *maxIndexPDim = distgrid->getMaxIndexPDimPTile(tile, &localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  
  // only rootPet checks for consistency of input, others listen
  if (localPet == rootPet){
    // check consistency of input information: shape = (typekind, rank, extents)
    if (typekindArg != typekind){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
        "- TypeKind mismatch between array argument and Array object",
        ESMC_CONTEXT, &rc);
      vm->broadcast(&rc, sizeof(int), rootPet);
      return rc;
    }
    if (rankArg != rank){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
        "- Type mismatch between array argument and Array object", ESMC_CONTEXT,
        &rc);
      vm->broadcast(&rc, sizeof(int), rootPet);
      return rc;
    }
    int tensorIndex=0;  // reset
    for (int i=0; i<rank; i++){
      int j = arrayToDistGridMap[i];
      if (j){
        // decomposed dimension
        --j;  // shift to basis 0
        if (counts[i] != maxIndexPDim[j] - minIndexPDim[j] + 1){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
            "- Extent mismatch between array argument and Array object",
            ESMC_CONTEXT, &rc);
          vm->broadcast(&rc, sizeof(int), rootPet);
          return rc;
        }
      }else{
        // tensor dimension
        if (counts[i] != undistUBound[tensorIndex] - undistLBound[tensorIndex]
          + 1){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
            "- Extent mismatch between array argument and Array object",
            ESMC_CONTEXT, &rc);
          vm->broadcast(&rc, sizeof(int), rootPet);
          return rc;
        }
        ++tensorIndex;
      }
    }
    int success = ESMF_SUCCESS;
    vm->broadcast(&success, sizeof(int), rootPet);
  }else{
    // not rootPet receive status from rootPet
    vm->broadcast(&localrc, sizeof(int), rootPet);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      "- rootPet exited with error", ESMC_CONTEXT, &rc)) return rc;
  }

  // size in bytes of each piece of data
  int dataSize = ESMC_TypeKind_FlagSize(typekind);
  
  // distgrid and delayout values
  const int *indexCountPDimPDe = distgrid->getIndexCountPDimPDe();
  const int *contigFlagPDimPDe = distgrid->getContigFlagPDimPDe();
  const int *minIndexPDimPDe = distgrid->getMinIndexPDimPDe();
  int dimCount = distgrid->getDimCount();
  int deCount = delayout->getDeCount();
  int localDeCount = delayout->getLocalDeCount();
  const int *localDeToDeMap = delayout->getLocalDeToDeMap();
  
  int redDimCount = rank - tensorCount;
  
  // prepare for comms
  VMK::commhandle **commh = new VMK::commhandle*; // used by all comm calls
  VMK::commhandle **commhList = 
    new VMK::commhandle*[dimCount]; // used for indexList comm
  
  // the following code depends on the "contiguousFlag" -> may need to construct
  if (localDeCount && (contiguousFlag[0]==-1)){
    // has local DEs and contiguousFlag has no yet been constructed
    localrc = constructContiguousFlag(redDimCount);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
      ESMC_CONTEXT, &rc))
      return rc;
  }
  
  // all PETs may be senders of data, each PET issues a maximum of one
  // non-blocking send, so no problem with too many outstanding comms here
  char **sendBuffer = new char*[localDeCount];
  for (int i=0; i<localDeCount; i++){
    int de = localDeToDeMap[i];
    if (tileListPDe[de] != tile) continue; // skip to next local DE
    sendBuffer[i] = (char *)larrayBaseAddrList[i]; // default: contiguous
    int sendSize =
      exclusiveElementCountPDe[de]*tensorElementCount*dataSize;  // bytes
    if (!contiguousFlag[i]){
      // only if this DE has a non-contiguous decomposition the contiguous
      // send buffer must be compiled from DE-local array segment piece by p.
      sendBuffer[i] = new char[sendSize];
      char *larrayBaseAddr = (char *)larrayBaseAddrList[i];
      int contigLength = exclusiveUBound[i*redDimCount]
        - exclusiveLBound[i*redDimCount] + 1;
      ArrayElement arrayElement(this, i);
      arrayElement.setSkipDim(0); // next() will skip ahead to next contig. line
      // loop over all elements in exclusive region for this DE and memcpy data
      int sendBufferIndex = 0;  // reset
      while(arrayElement.isWithin()){
        // copy this element from excl. region into the contiguous sendBuffer
        int linearIndex = arrayElement.getLinearIndexExclusive();
        // contiguous data copy in 1st dim
        memcpy(sendBuffer[i]+sendBufferIndex*dataSize,
          larrayBaseAddr+linearIndex*dataSize, contigLength*dataSize);
        sendBufferIndex += contigLength;
        arrayElement.next();  // skip ahead to next contiguous line
      } // multi dim index loop
    } // !contiguousFlag
        
    // ready to send the sendBuffer to rootPet
    *commh = NULL; // invalidate
    localrc = vm->send(sendBuffer[i], sendSize, rootPet, commh);
    if (localrc){
      char *message = new char[160];
      sprintf(message, "VMKernel/MPI error #%d\n", localrc);
      ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
        message, ESMC_CONTEXT, &rc);
      delete [] message;
      return rc;
    }
  } // i -> de
  
  // rootPet is the only receiver for gather data
  char **recvBuffer;
  if (localPet == rootPet){
    int nbCount = 0;  // reset: count of outstanding non-blocking comms
    const int boostSize = 512;  // max number of posted non-blocking calls:
                                // stay below typical system limits
    VMK::commhandle **commhDataList = 
      new VMK::commhandle*[boostSize]; // used for data comms
    
    // for each DE of the Array receive a single contiguous recvBuffer
    // from the associated PET non-blocking and memcpy it into the right 
    // location in "array".
    recvBuffer = new char*[deCount]; // contiguous recvBuffer
    for (int i=0; i<deCount; i++){
      int de = i;
      if (tileListPDe[de] == tile){
        // this DE is located on sending tile
        // prepare contiguous recvBuffer for this DE and issue non-blocking recv
        int recvSize =
          exclusiveElementCountPDe[de]*tensorElementCount*dataSize;  // bytes
        recvBuffer[de] = new char[recvSize];
        int srcPet;
        delayout->getDEMatchPET(de, *vm, NULL, &srcPet, 1);
        commhDataList[nbCount] = NULL; // invalidate
        localrc = vm->recv(recvBuffer[de], recvSize, srcPet,
          &(commhDataList[nbCount]));
        ++nbCount;  // count this non-blocking recv
        if (localrc){
          char *message = new char[160];
          sprintf(message, "VMKernel/MPI error #%d\n", localrc);
          ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
            message, ESMC_CONTEXT, &rc);
          delete [] message;
          return rc;
        }
        
        // see if outstanding nb-recvs have reached boostSize limit
        if (nbCount >= boostSize){
//printf("clearing Gather() boost at nbCount = %d\n", nbCount);
          // wait for nb-recvs to finish before posting any more
          for (int j=0; j<nbCount; j++){
            vm->commwait(&(commhDataList[j]));
            delete commhDataList[j];
          }
          nbCount = 0;  // reset
        }
        
      } // DE on tile
    } // i -> de
//printf("clearing Gather() boost at nbCount = %d\n", nbCount);
    // wait for nb-recvs to finish before exiting
    for (int j=0; j<nbCount; j++){
      vm->commwait(&(commhDataList[j]));
      delete commhDataList[j];
    }
    nbCount = 0;  // reset
    delete [] commhDataList;
  }
  
  // wait until all the local sends are complete
  vm->commqueuewait();
  // - done waiting on sends -
  
  if (localPet == rootPet){
    char *array = (char *)arrayArg;
    // rootPet gathers information from _all_ DEs
    for (int i=0; i<deCount; i++){
      int de = i;
      if (tileListPDe[de] == tile){
        // this DE is located on sending tile
        int **indexList = new int*[dimCount];
        int tensorIndex=0;  // reset
        int commhListCount = 0;  // reset
        for (int j=0; j<dimCount; j++){
          if(distgridToArrayMap[j]!=0 && contigFlagPDimPDe[de*dimCount+j]==0){
            // associated and non-contiguous dimension
            // -> obtain indexList for this DE and dim
            indexList[j] = new int[indexCountPDimPDe[de*dimCount+j]];
            commhList[commhListCount] = NULL; // prime for later test
            localrc = distgrid->fillIndexListPDimPDe(indexList[j], de, j+1, 
              &(commhList[commhListCount]), localPet, vm);
            if (commhList[commhListCount] != NULL)
              ++commhListCount;
            if (ESMC_LogDefault.MsgFoundError(localrc,
              ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          }
        } // j

        // initialize multi dim index loop
        vector<int> sizes;
        tensorIndex=0;  // reset
        for (int jj=0; jj<rank; jj++){
          int j = arrayToDistGridMap[jj];// j is dimIndex basis 1, or 0 f tensor
          if (j){
            // decomposed dimension 
            --j;  // shift to basis 0
            sizes.push_back(indexCountPDimPDe[de*dimCount+j]);
          }else{
            // tensor dimension
            sizes.push_back(
              undistUBound[tensorIndex] - undistLBound[tensorIndex] + 1);
            ++tensorIndex;
          }
        }
        
        // wait for all outstanding receives issued by fillIndexListPDimPDe()
        for (int j=0; j<commhListCount; j++){
          vm->commwait(&(commhList[j]));
          delete commhList[j];
        }
        
        MultiDimIndexLoop multiDimIndexLoop(sizes);
        if (contigFlagPDimPDe[de*dimCount])
          multiDimIndexLoop.setSkipDim(0); // contiguous data in first dimension
        // loop over all elements in exclusive region for this DE 
        int recvBufferIndex = 0;  // reset
        while(multiDimIndexLoop.isWithin()){
          // determine linear index for this element into array
          int linearIndex = 0;  // reset
          for (int jj=rank-1; jj>=0; jj--){
            linearIndex *= counts[jj];  // first time zero o.k.
            int j = arrayToDistGridMap[jj];// j is dimIndex bas 1, or 0 f tensor
            if (j){
              // decomposed dimension 
              --j;  // shift to basis 0
              if (contigFlagPDimPDe[de*dimCount+j]){
                linearIndex += minIndexPDimPDe[de*dimCount+j]
                  + multiDimIndexLoop.getIndexTuple()[jj];
              }else{
                linearIndex +=
                  indexList[j][multiDimIndexLoop.getIndexTuple()[jj]];
              }
              // shift basis 1 -> basis 0
              linearIndex -= minIndexPDim[j];
            }else{
              // tensor dimension
              linearIndex += multiDimIndexLoop.getIndexTuple()[jj];
            }
          }
          // copy this element from the contiguous recvBuffer for this DE
          if (contigFlagPDimPDe[de*dimCount]){
            // contiguous data in first dimension
            memcpy(array+linearIndex*dataSize, 
              recvBuffer[de]+recvBufferIndex*dataSize,
              multiDimIndexLoop.getIndexTupleEnd()[0]*dataSize);
            multiDimIndexLoop.next(); // skip to next contiguous line
            recvBufferIndex += multiDimIndexLoop.getIndexTupleEnd()[0];
          }else{
            // non-contiguous data in first dimension
            memcpy(array+linearIndex*dataSize, 
              recvBuffer[de]+recvBufferIndex*dataSize, dataSize);
            multiDimIndexLoop.next(); // next element
            ++recvBufferIndex;
          }
        } // multi dim index loop
            
        // clean-up
        for (int j=0; j<dimCount; j++)
          if(contigFlagPDimPDe[de*dimCount+j]==0)
            delete [] indexList[j];
        delete [] indexList;
      } // DE on tile
    } // i -> de
  }else{
    // localPet is _not_ rootPet -> provide localIndexList to rootPet if nec.
    int commhListCount = 0;  // reset
    for (int i=0; i<localDeCount; i++){
      int de = localDeToDeMap[i];
      if (tileListPDe[de] == tile){
        // this DE is located on receiving tile -> must send info to rootPet
        for (int j=0; j<dimCount; j++){
          if(distgridToArrayMap[j]!=0 && contigFlagPDimPDe[de*dimCount+j]==0){
            // associated and non-contiguous dimension
            // -> send local indexList for this DE and dim to rootPet
            commhList[commhListCount] = NULL; // prime for later test
            localrc = distgrid->fillIndexListPDimPDe(NULL, de, j+1, 
              &(commhList[commhListCount]), rootPet, vm);
            if (commhList[commhListCount] != NULL)
              ++commhListCount;
            if (ESMC_LogDefault.MsgFoundError(localrc,
              ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          }
        } // j
      } // DE on tile
    } // i -> de
    // wait for all outstanding indexList sends issued by fillIndexListPDimPDe()
    for (int j=0; j<commhListCount; j++){
      vm->commwait(&(commhList[j]));
        delete commhList[j];
    }
  }
  
  // garbage collection
  if (localPet == rootPet){
    for (int i=0; i<deCount; i++){
      int de = i;
      if (tileListPDe[de] == tile)
        delete [] recvBuffer[de];
    }    
    delete [] recvBuffer;
  }
  for (int i=0; i<localDeCount; i++){
    int de = localDeToDeMap[i];
    if (tileListPDe[de] != tile) continue; // skip to next local DE
    if (!contiguousFlag[i])
      delete [] sendBuffer[i];
  }
  delete [] sendBuffer;
  delete commh;
  delete [] commhList;
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::scatter()"
//BOPI
// !IROUTINE:  ESMCI::Array::scatter
//
// !INTERFACE:
int Array::scatter(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  void *arrayArg,                       // in -
  ESMC_TypeKind_Flag typekindArg,       // in -
  int rankArg,                          // in -
  int *counts,                          // in -
  int *tileArg,                         // in -
  int rootPet,                          // in -
  VM *vm                                // in -
  ){
//
//
// !DESCRIPTION:
//    Scatter native array across Array object 
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // return with errors for NULL pointer
  if (this == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to Array", ESMC_CONTEXT, &rc);
    return rc;
  }

  // by default use the currentVM for vm
  if (vm == ESMC_NULL_POINTER){
    vm = VM::getCurrent(&localrc);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
  }

  // query the VM
  int localPet = vm->getLocalPet();
  
  // deal with optional tile argument
  int tile = 1;  // default
  if (tileArg)
    tile = *tileArg;
  int tileCount = distgrid->getTileCount();
  if (tile < 1 || tile > tileCount){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
      "- Specified tile out of bounds", ESMC_CONTEXT, &rc);
    return rc;
  }
  const int *tileListPDe = distgrid->getTileListPDe();

  // get minIndexPDim and maxIndexPDim for tile
  const int *minIndexPDim = distgrid->getMinIndexPDimPTile(tile, &localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  const int *maxIndexPDim = distgrid->getMaxIndexPDimPTile(tile, &localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  
  // only rootPet checks for consistency of input, others listen
  if (localPet == rootPet){
    // check consistency of input information: shape = (typekind, rank, extents)
    if (typekindArg != typekind){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
        "- TypeKind mismatch between array argument and Array object",
        ESMC_CONTEXT, &rc);
      vm->broadcast(&rc, sizeof(int), rootPet);
      return rc;
    }
    if (rankArg != rank){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
        "- Type mismatch between array argument and Array object", ESMC_CONTEXT,
        &rc);
      vm->broadcast(&rc, sizeof(int), rootPet);
      return rc;
    }
    int tensorIndex=0;  // reset
    for (int i=0; i<rank; i++){
      int j = arrayToDistGridMap[i];
      if (j){
        // decomposed dimension
        --j;  // shift to basis 0
        if (counts[i] != maxIndexPDim[j] - minIndexPDim[j] + 1){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
            "- Extent mismatch between array argument and Array object",
            ESMC_CONTEXT, &rc);
          vm->broadcast(&rc, sizeof(int), rootPet);
          return rc;
        }
      }else{
        // tensor dimension
        if (counts[i] != undistUBound[tensorIndex] - undistLBound[tensorIndex]
          + 1){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
            "- Extent mismatch between array argument and Array object",
            ESMC_CONTEXT, &rc);
          vm->broadcast(&rc, sizeof(int), rootPet);
          return rc;
        }
        ++tensorIndex;
      }
    }
    int success = ESMF_SUCCESS;
    vm->broadcast(&success, sizeof(int), rootPet);
  }else{
    // not rootPet receive status from rootPet
    vm->broadcast(&localrc, sizeof(int), rootPet);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      "- rootPet exited with error", ESMC_CONTEXT, &rc)) return rc;
  }

  // size in bytes of each piece of data
  int dataSize = ESMC_TypeKind_FlagSize(typekind);

  // distgrid and delayout values
  const int *indexCountPDimPDe = distgrid->getIndexCountPDimPDe();
  const int *contigFlagPDimPDe = distgrid->getContigFlagPDimPDe();
  const int *minIndexPDimPDe = distgrid->getMinIndexPDimPDe();
  int dimCount = distgrid->getDimCount();
  int deCount = delayout->getDeCount();
  int localDeCount = delayout->getLocalDeCount();
  const int *localDeToDeMap = delayout->getLocalDeToDeMap();
  
  int redDimCount = rank - tensorCount;
  
  // prepare for comms
  VMK::commhandle **commh = new VMK::commhandle*; // used by all comm calls
  VMK::commhandle **commhList = 
    new VMK::commhandle*[dimCount]; // used for indexList comm
  
  // the following code depends on the "contiguousFlag" -> may need to construct
  if (localDeCount && (contiguousFlag[0]==-1)){
    // has local DEs and contiguousFlag has no yet been constructed
    localrc = constructContiguousFlag(redDimCount);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
      ESMC_CONTEXT, &rc))
      return rc;
  }
  
  // all PETs may be receivers of data, each PET issues a maximum of one
  // non-blocking receive, so no problem with too many outstanding comms here
  char **recvBuffer = new char*[localDeCount];
  for (int i=0; i<localDeCount; i++){
    int de = localDeToDeMap[i];
    if (tileListPDe[de] != tile) continue; // skip to next local DE
    recvBuffer[i] = (char *)larrayBaseAddrList[i]; // default: contiguous
    int recvSize =
      exclusiveElementCountPDe[de]*tensorElementCount*dataSize; // bytes
    if (!contiguousFlag[i])
      recvBuffer[i] = new char[recvSize];
    *commh = NULL; // invalidate
    // receive data into recvBuffer
    localrc = vm->recv(recvBuffer[i], recvSize, rootPet, commh);
    if (localrc){
      char *message = new char[160];
      sprintf(message, "VMKernel/MPI error #%d\n", localrc);
      ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
        message, ESMC_CONTEXT, &rc);
      delete [] message;
      return rc;
    }
  }
  // - done issuing nb receives (potentially all Pets) -

  // rootPet is the only sender of scatter data,
  // but may need info from other PETs to construct sendBuffer
  if (localPet == rootPet){
    int nbCount = 0; // reset: count of outstanding non-blocking comms
    const int boostSize = 512;  // max number of posted non-blocking calls:
                                // stay below typical system limits
    VMK::commhandle **commhDataList = 
      new VMK::commhandle*[boostSize]; // used for data comms
    
    char *array = (char *)arrayArg;
    // rootPet scatters information to _all_ DEs
    // for each DE of the Array memcpy together a single contiguous sendBuffer
    // from "array" data and send it to the receiving PET non-blocking.
    char **sendBuffer = new char*[deCount]; // contiguous sendBuffer
    for (int i=0; i<deCount; i++){
      int de = i;
      if (tileListPDe[de] == tile){
        // this DE is located on receiving tile
        int **indexList = new int*[dimCount];
        int tensorIndex=0;  // reset
        int commhListCount = 0;  // reset
        for (int j=0; j<dimCount; j++){
          if(distgridToArrayMap[j]!=0 && contigFlagPDimPDe[de*dimCount+j]==0){
            // associated and non-contiguous dimension
            // -> obtain indexList for this DE and dim
            indexList[j] = new int[indexCountPDimPDe[de*dimCount+j]];
            commhList[commhListCount] = NULL; // prime for later test
            localrc = distgrid->fillIndexListPDimPDe(indexList[j], de, j+1, 
              &(commhList[commhListCount]), localPet, vm);
            if (commhList[commhListCount] != NULL)
              ++commhListCount;
            if (ESMC_LogDefault.MsgFoundError(localrc,
              ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          }
        } // j

        // prepare contiguous sendBuffer for this DE
        int sendSize =
          exclusiveElementCountPDe[de]*tensorElementCount*dataSize;  // bytes
        sendBuffer[de] = new char[sendSize];
        
        // initialize multi dim index loop
        vector<int> sizes;
        tensorIndex=0;  // reset
        for (int jj=0; jj<rank; jj++){
          int j = arrayToDistGridMap[jj];// j is dimIndex basis 1, or 0 f tensor
          if (j){
            // decomposed dimension 
            --j;  // shift to basis 0
            sizes.push_back(indexCountPDimPDe[de*dimCount+j]);
          }else{
            // tensor dimension
            sizes.push_back(
              undistUBound[tensorIndex] - undistLBound[tensorIndex] + 1);
            ++tensorIndex;
          }
        }
        
        // wait for all outstanding receives issued by fillIndexListPDimPDe()
        for (int j=0; j<commhListCount; j++){
          vm->commwait(&(commhList[j]));
          delete commhList[j];
        }
        
        MultiDimIndexLoop multiDimIndexLoop(sizes);
        if (contigFlagPDimPDe[de*dimCount])
          multiDimIndexLoop.setSkipDim(0); // contiguous data in first dimension
        // loop over all elements in exclusive region for this DE 
        int sendBufferIndex = 0;  // reset
        while(multiDimIndexLoop.isWithin()){
          // determine linear index for this element into array
          int linearIndex = 0;  // reset
          for (int jj=rank-1; jj>=0; jj--){
            linearIndex *= counts[jj];  // first time zero o.k.
            int j = arrayToDistGridMap[jj];// j is dimIndex bas 1, or 0 f tensor
            if (j){
              // decomposed dimension 
              --j;  // shift to basis 0
              if (contigFlagPDimPDe[de*dimCount+j]){
                linearIndex += minIndexPDimPDe[de*dimCount+j]
                  + multiDimIndexLoop.getIndexTuple()[jj];
              }else{
                linearIndex +=
                  indexList[j][multiDimIndexLoop.getIndexTuple()[jj]];
              }
              // shift basis 1 -> basis 0
              linearIndex -= minIndexPDim[j];
            }else{
              // tensor dimension
              linearIndex += multiDimIndexLoop.getIndexTuple()[jj];
            }
          }
          // copy this element into the contiguous sendBuffer for this DE
          if (contigFlagPDimPDe[de*dimCount]){
            // contiguous data in first dimension
            memcpy(sendBuffer[de]+sendBufferIndex*dataSize,
              array+linearIndex*dataSize,
              multiDimIndexLoop.getIndexTupleEnd()[0]*dataSize);
            multiDimIndexLoop.next(); // skip to next contiguous line
            sendBufferIndex += multiDimIndexLoop.getIndexTupleEnd()[0];
          }else{
            // non-contiguous data in first dimension
            memcpy(sendBuffer[de]+sendBufferIndex*dataSize,
              array+linearIndex*dataSize, dataSize);
            multiDimIndexLoop.next(); // next element
            ++sendBufferIndex;
          }
        } // multi dim index loop
    
        // ready to send the sendBuffer
        int dstPet;
        delayout->getDEMatchPET(de, *vm, NULL, &dstPet, 1);
        commhDataList[nbCount] = NULL;  // invalidate
        localrc = vm->send(sendBuffer[de], sendSize, dstPet,
          &(commhDataList[nbCount]));
        ++nbCount;  // count this non-blocking send
        if (localrc){
          char *message = new char[160];
          sprintf(message, "VMKernel/MPI error #%d\n", localrc);
          ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
            message, ESMC_CONTEXT, &rc);
          delete [] message;
          return rc;
        }
        
        // clean-up
        for (int j=0; j<dimCount; j++)
          if(contigFlagPDimPDe[de*dimCount+j]==0)
            delete [] indexList[j];
        delete [] indexList;
        
        // see if outstanding nb-sends have reached boostSize limit
        if (nbCount >= boostSize){
//printf("clearing Scatter() boost at nbCount = %d\n", nbCount);
          // wait for nb-sends to finish before posting any more
          for (int j=0; j<nbCount; j++){
            vm->commwait(&(commhDataList[j]));
            delete commhDataList[j];
          }
          nbCount = 0;  // reset
        }
        
      } // DE on tile
    } // i -> de
//printf("clearing Scatter() boost at nbCount = %d\n", nbCount);
    // wait for nb-sends to finish before exiting
    for (int j=0; j<nbCount; j++){
      vm->commwait(&(commhDataList[j]));
      delete commhDataList[j];
    }
    nbCount = 0;  // reset
    delete [] commhDataList;
    // TODO: move the delete [] sendBuffer[] into nb-wait loops to lower
    // TODO: memory foot print.
    for (int i=0; i<deCount; i++)
      if (tileListPDe[i] == tile)
        delete [] sendBuffer[i];
    delete [] sendBuffer;
  }
  // - done issuing nb sends (from rootPet) -
  
  // send localIndexList information to rootPet if necessary
  if (localPet != rootPet){
    // localPet is _not_ rootPet -> provide localIndexList to rootPet if nec.
    int commhListCount = 0;  // reset
    for (int i=0; i<localDeCount; i++){
      int de = localDeToDeMap[i];
      if (tileListPDe[de] == tile){
        // this DE is located on receiving tile -> must send info to rootPet
        for (int j=0; j<dimCount; j++){
          if(distgridToArrayMap[j]!=0 && contigFlagPDimPDe[de*dimCount+j]==0){
            // associated and non-contiguous dimension
            // -> send local indexList for this DE and dim to rootPet
            commhList[commhListCount] = NULL; // prime for later test
            localrc = distgrid->fillIndexListPDimPDe(NULL, de, j+1, 
              &(commhList[commhListCount]), rootPet, vm);
            if (commhList[commhListCount] != NULL)
              ++commhListCount;
            if (ESMC_LogDefault.MsgFoundError(localrc,
              ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          }
        } // j
      } // DE on tile
    } // i -> de
    // wait for all outstanding indexList sends issued by fillIndexListPDimPDe()
    for (int j=0; j<commhListCount; j++){
      vm->commwait(&(commhList[j]));
        delete commhList[j];
    }
  }

  // wait until all the local receives are complete
  vm->commqueuewait();
  // - done waiting on receives -
    
  // distribute received data into non-contiguous exclusive regions
  for (int i=0; i<localDeCount; i++){
    int de = localDeToDeMap[i];
    if (tileListPDe[de] != tile) continue; // skip to next local DE
    if (!contiguousFlag[i]){
      // only if this DE has a non-contiguous decomposition the contiguous
      // receive buffer must be copied into DE-local array segment piece by p.
      char *larrayBaseAddr = (char *)larrayBaseAddrList[i];
      int contigLength = exclusiveUBound[i*redDimCount]
        - exclusiveLBound[i*redDimCount] + 1;
      ArrayElement arrayElement(this, i);
      arrayElement.setSkipDim(0); // next() will skip ahead to next contig. line
      // loop over all elements in exclusive region for this DE and memcpy data
      int recvBufferIndex = 0;  // reset
      while(arrayElement.isWithin()){
        // copy this element from the contiguous recvBuffer into excl. region
        int linearIndex = arrayElement.getLinearIndexExclusive();
        // since the data in the recvBuffer was constructed to be contiguous
        // wrt data layout on destination DE -> contiguous data copy in 1st dim
        memcpy(larrayBaseAddr+linearIndex*dataSize,
          recvBuffer[i]+recvBufferIndex*dataSize, contigLength*dataSize);
        recvBufferIndex += contigLength;
        arrayElement.next();  // skip ahead to next contiguous line
      } // multi dim index loop

      // clean-up
      delete [] recvBuffer[i];
    } // !contiguousFlag
  } // i -> de

  // garbage collection
  delete [] recvBuffer;
  delete commh;
  delete [] commhList;
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::haloStore()"
//BOPI
// !IROUTINE:  ESMCI::Array::haloStore
//
// !INTERFACE:
int Array::haloStore(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  Array *array,                         // in    - Array
  RouteHandle **routehandle,            // inout - handle to precomputed comm
  ESMC_HaloStartRegionFlag halostartregionflag, // in - start of halo region
  InterfaceInt *haloLDepth,             // in    - lower corner halo depth
  InterfaceInt *haloUDepth,             // in    - upper corner halo depth
  int *pipelineDepthArg                 // in (optional)
  ){
//
// !DESCRIPTION:
//  Precompute and store communication pattern for halo
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code

#define HALOSTOREMEMLOG_off
  
    try{
    
    // every Pet must provide array argument
    if (array == NULL){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
        "- Not a valid pointer to array", ESMC_CONTEXT, &rc);
      return rc;
    }
    
    // get the current VM and VM releated information
    VM *vm = VM::getCurrent(&localrc);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
    int localPet = vm->getLocalPet();
    int petCount = vm->getPetCount();
    
#ifdef HALOSTOREMEMLOG_on
    VM::logMemInfo(std::string("HaloStore1"));
#endif
    
    // prepare haloOutsideLBound and haloInsideLBound arrays
    int localDeCount = array->getDELayout()->getLocalDeCount();
    int redDimCount = array->rank - array->tensorCount;
    vector<vector<int> > haloInsideLBound(localDeCount);
    vector<vector<int> > haloOutsideLBound(localDeCount);
    if (haloLDepth == NULL){
      // haloLDepth was not provided
      for (int i=0; i<localDeCount; i++){
        int kOff = i * (array->rank - array->tensorCount);
        int kPacked = 0;    // reset
        int kTensor = 0;    // reset
        haloInsideLBound[i].resize(array->rank);
        haloOutsideLBound[i].resize(array->rank);
        for (int k=0; k<array->rank; k++){
          if (array->getArrayToDistGridMap()[k]){
            // decomposed dimension
            if (halostartregionflag==ESMF_REGION_COMPUTATIONAL){
              haloInsideLBound[i][k] = array->computationalLBound[kOff+kPacked]
                - array->exclusiveLBound[kOff+kPacked];
            }else{
              haloInsideLBound[i][k] = 0;
            }
            haloOutsideLBound[i][k] = array->totalLBound[kOff+kPacked]
              - array->exclusiveLBound[kOff+kPacked];
            ++kPacked;
          }else{
            // tensor dimension
            haloOutsideLBound[i][k] = 0;
          }   
        }
      }
    }else{
      // haloLDepth was provided -> check and use
      if (haloLDepth->dimCount != 1){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
          "- haloLDepth array must be of rank 1", ESMC_CONTEXT, &rc);
        return rc;
      }
      if (haloLDepth->extent[0] != (array->rank - array->tensorCount)){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
          "- haloLDepth array has wrong size", ESMC_CONTEXT, &rc);
        return rc;
      }
      for (int i=0; i<localDeCount; i++){
        int kOff = i * (array->rank - array->tensorCount);
        int kPacked = 0;    // reset
        int kTensor = 0;    // reset
        haloInsideLBound[i].resize(array->rank);
        haloOutsideLBound[i].resize(array->rank);
        for (int k=0; k<array->rank; k++){
          if (array->getArrayToDistGridMap()[k]){
            // decomposed dimension
            if (halostartregionflag==ESMF_REGION_COMPUTATIONAL){
              haloInsideLBound[i][k] = array->computationalLBound[kOff+kPacked]
                - array->exclusiveLBound[kOff+kPacked];
              haloOutsideLBound[i][k] = haloInsideLBound[i][k]
                - haloLDepth->array[kPacked];
            }else{
              haloInsideLBound[i][k] = 0;
              haloOutsideLBound[i][k] = -haloLDepth->array[kPacked];
            }
            ++kPacked;
          }else{
            // tensor dimension
            haloOutsideLBound[i][k] = 0;
          }   
        }
      }
    }
    // prepare haloOutsideUBound and haloInsideUBound arrays
    vector<vector<int> > haloInsideUBound(localDeCount);
    vector<vector<int> > haloOutsideUBound(localDeCount);
    if (haloUDepth == NULL){
      // haloUDepth was not provided
      for (int i=0; i<localDeCount; i++){
        int kOff = i * (array->rank - array->tensorCount);
        int kPacked = 0;    // reset
        int kTensor = 0;    // reset
        haloInsideUBound[i].resize(array->rank);
        haloOutsideUBound[i].resize(array->rank);
        for (int k=0; k<array->rank; k++){
          if (array->getArrayToDistGridMap()[k]){
            // decomposed dimension
            if (halostartregionflag==ESMF_REGION_COMPUTATIONAL){
              haloInsideUBound[i][k] = array->computationalUBound[kOff+kPacked]
                - array->exclusiveLBound[kOff+kPacked];
            }else{
              haloInsideUBound[i][k] = array->exclusiveUBound[kOff+kPacked]
              - array->exclusiveLBound[kOff+kPacked];
            }
            haloOutsideUBound[i][k] = array->totalUBound[kOff+kPacked]
              - array->exclusiveLBound[kOff+kPacked];
            ++kPacked;
          }else{
            // tensor dimension
            haloOutsideUBound[i][k] = array->undistUBound[kTensor]
              - array->undistLBound[kTensor] + 1;
            ++kTensor;
          }   
        }
      }
    }else{
      // haloUDepth was provided -> check and use
      if (haloUDepth->dimCount != 1){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
          "- haloUDepth array must be of rank 1", ESMC_CONTEXT, &rc);
        return rc;
      }
      if (haloUDepth->extent[0] != (array->rank - array->tensorCount)){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
          "- haloUDepth array has wrong size", ESMC_CONTEXT, &rc);
        return rc;
      }
      for (int i=0; i<localDeCount; i++){
        int kOff = i * (array->rank - array->tensorCount);
        int kPacked = 0;    // reset
        int kTensor = 0;    // reset
        haloInsideUBound[i].resize(array->rank);
        haloOutsideUBound[i].resize(array->rank);
        for (int k=0; k<array->rank; k++){
          if (array->getArrayToDistGridMap()[k]){
            // decomposed dimension
            if (halostartregionflag==ESMF_REGION_COMPUTATIONAL){
              haloInsideUBound[i][k] = array->computationalUBound[kOff+kPacked]
                - array->exclusiveLBound[kOff+kPacked];
              haloOutsideUBound[i][k] = haloInsideUBound[i][k]
                + haloUDepth->array[kPacked];
            }else{
              haloInsideUBound[i][k] = array->exclusiveUBound[kOff+kPacked]
              - array->exclusiveLBound[kOff+kPacked];
              haloOutsideUBound[i][k] = array->exclusiveUBound[kOff+kPacked]
                - array->exclusiveLBound[kOff+kPacked]
                + haloUDepth->array[kPacked];
            }
            ++kPacked;
          }else{
            // tensor dimension
            haloOutsideUBound[i][k] = array->undistUBound[kTensor]
              - array->undistLBound[kTensor] + 1;
            ++kTensor;
          }   
        }
      }
    }
    
#ifdef HALOSTOREMEMLOG_on
    VM::logMemInfo(std::string("HaloStore2"));
#endif

#define HALOTENSORMIX_off
    // construct identity sparse matrix from rim elements with valid seqIndex
    vector<int> factorIndexList;
    int factorListCount = 0;  // init
    vector<vector<int> > rimMaskElement(localDeCount);
    vector<vector<SeqIndex> > rimMaskSeqIndex(localDeCount);
    for (int i=0; i<localDeCount; i++){
      ArrayElement arrayElement(array, i, true);
      int element = 0;
      while(arrayElement.isWithin()){
        SeqIndex seqIndex = array->rimSeqIndex[i][element];
        if (seqIndex.valid()){
          // this rim element holds a valid seqIndex
          int const *indexTuple = arrayElement.getIndexTuple();
          // check whether indexTuple is within halo bounds
          bool withinHalo = true;
          bool insideFlag = true;
          for (int k=0; k<array->rank; k++){
            if (indexTuple[k] < haloOutsideLBound[i][k] ||
              indexTuple[k] > haloOutsideUBound[i][k]){
              withinHalo = false; // index outside halo region
              break;
            }
            if (indexTuple[k] < haloInsideLBound[i][k] ||
              indexTuple[k] > haloInsideUBound[i][k]){
              insideFlag = false; // below halo start
            }
          }
          if (insideFlag) withinHalo = false; // element below halo start
          if (withinHalo){
            // add element to identity matrix
            factorIndexList.push_back(seqIndex.decompSeqIndex); // src
#ifdef HALOTENSORMIX_on
            factorIndexList.push_back(seqIndex.tensorSeqIndex); // src
#endif
            factorIndexList.push_back(seqIndex.decompSeqIndex); // dst
#ifdef HALOTENSORMIX_on
            factorIndexList.push_back(seqIndex.tensorSeqIndex); // dst
#endif
            ++factorListCount;  // count this element
          }else{
            // need to mask this element in the Array rim region in order to
            // prevent haloing due to the fact that another DE may still
            // add an entry for this seqIndex to the identity matrix
            rimMaskElement[i].push_back(element);
            rimMaskSeqIndex[i].push_back(seqIndex);
            array->rimSeqIndex[i][element].decompSeqIndex = -1;  // mask entry
          }
        }
        arrayElement.next();  // next element
        ++element;
      } // multi dim index loop
    }    
    
    // load type specific factorList with "1" for the identity matrix
    ESMC_TypeKind_Flag typekindFactor = array->getTypekind();
    void *factorList;
    if (typekindFactor == ESMC_TYPEKIND_R4){
      ESMC_R4 *factorListT = new ESMC_R4[factorListCount];
      for (int i=0; i<factorListCount; i++)
        factorListT[i] = 1.;
      factorList = (void *)factorListT;
    }else if (typekindFactor == ESMC_TYPEKIND_R8){
      ESMC_R8 *factorListT = new ESMC_R8[factorListCount];
      for (int i=0; i<factorListCount; i++)
        factorListT[i] = 1.;
      factorList = (void *)factorListT;
    }else if (typekindFactor == ESMC_TYPEKIND_I4){
      ESMC_I4 *factorListT = new ESMC_I4[factorListCount];
      for (int i=0; i<factorListCount; i++)
        factorListT[i] = 1;
      factorList = (void *)factorListT;
    }else if (typekindFactor == ESMC_TYPEKIND_I8){
      ESMC_I8 *factorListT = new ESMC_I8[factorListCount];
      for (int i=0; i<factorListCount; i++)
        factorListT[i] = 1;
      factorList = (void *)factorListT;
    }
    
    // prepare SparseMatrix vector with the constructed sparse matrix
    vector<SparseMatrix> sparseMatrix;
    int *factorIndexListPtr = NULL; // initialize
    if (factorListCount>0) factorIndexListPtr = &(factorIndexList[0]);
    sparseMatrix.push_back(SparseMatrix(typekindFactor, factorList,
#ifdef HALOTENSORMIX_on
      factorListCount, 2, 2, factorIndexListPtr));
#else
      factorListCount, 1, 1, factorIndexListPtr));
#endif
    
#ifdef HALOSTOREMEMLOG_on
    VM::logMemInfo(std::string("HaloStore3"));
#endif

    // precompute sparse matrix multiplication
    int srcTermProcessing = 0;  // no need to use auto-tuning to figure this out
    localrc = sparseMatMulStore(array, array, routehandle, sparseMatrix, true,
      &srcTermProcessing, pipelineDepthArg);
    
#ifdef HALOSTOREMEMLOG_on
    VM::logMemInfo(std::string("HaloStore4"));
#endif
    
    // remove seqIndex masking in Array rim region before evaluating return code
    for (int i=0; i<localDeCount; i++){
      for (int k=0; k<rimMaskElement[i].size(); k++){
        int element = rimMaskElement[i][k];
        array->rimSeqIndex[i][element] = rimMaskSeqIndex[i][k]; // restore
      }
    }
    
#ifdef HALOSTOREMEMLOG_on
    VM::logMemInfo(std::string("HaloStore5"));
#endif
    
    // garbage collection before evaluating return code
    if (typekindFactor == ESMC_TYPEKIND_R4){
      ESMC_R4 *factorListT = (ESMC_R4 *)factorList;
      delete [] factorListT;
    }else if (typekindFactor == ESMC_TYPEKIND_R8){
      ESMC_R8 *factorListT = (ESMC_R8 *)factorList;
      delete [] factorListT;
    }else if (typekindFactor == ESMC_TYPEKIND_I4){
      ESMC_I4 *factorListT = (ESMC_I4 *)factorList;
      delete [] factorListT;
    }else if (typekindFactor == ESMC_TYPEKIND_I8){
      ESMC_I8 *factorListT = (ESMC_I8 *)factorList;
      delete [] factorListT;
    }
    
#ifdef HALOSTOREMEMLOG_on
    VM::logMemInfo(std::string("HaloStore6"));
#endif
    
    // error handling
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
      ESMC_CONTEXT, &rc)) return rc;
    
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(exception &x){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      x.what(), ESMC_CONTEXT, &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------

  //-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::halo()"
//BOPI
// !IROUTINE:  ESMCI::Array::halo
//
// !INTERFACE:
int Array::halo(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  Array *array,                         // in    - Array
  RouteHandle **routehandle,            // inout - handle to precomputed comm
  ESMC_CommFlag commflag,               // in    - communication options
  bool *finishedflag,                   // out   - TEST ops finished or not
  bool *cancelledflag,                  // out   - any cancelled operations
  bool checkflag                        // in    - false: (def.) basic chcks
                                        //         true:  full input check
  ){
//
// !DESCRIPTION:
//    Execute an Array halo
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // implemented via sparseMatMul
  localrc = sparseMatMul(array, array, routehandle,
    commflag, finishedflag, cancelledflag, ESMC_REGION_SELECT, 
    ESMC_TERMORDER_FREE, checkflag, true);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::haloRelease()"
//BOPI
// !IROUTINE:  ESMCI::Array::haloRelease
//
// !INTERFACE:
int Array::haloRelease(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  RouteHandle *routehandle        // inout -
  ){
//
// !DESCRIPTION:
//    Release information for an Array halo
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  // implemented via sparseMatMul
  localrc = sparseMatMulRelease(routehandle);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;

  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::redistStore()"
//BOPI
// !IROUTINE:  ESMCI::Array::redistStore
//
// !INTERFACE:
int Array::redistStore(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  Array *srcArray,                      // in    - source Array
  Array *dstArray,                      // in    - destination Array
  RouteHandle **routehandle,            // inout - handle to precomputed comm
  InterfaceInt *srcToDstTransposeMap,   // in    - mapping src -> dst dims
  ESMC_TypeKind_Flag typekindFactor,    // in    - typekind of factor
  void *factor,                         // in    - redist factor
  int *pipelineDepthArg                 // in (optional)
  ){
//
// !DESCRIPTION:
//  Precompute and store communication pattern for redistribution
//  from srcArray to dstArray.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  try{
    
  // every Pet must provide srcArray and dstArray
  if (srcArray == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to srcArray", ESMC_CONTEXT, &rc);
    return rc;
  }
  if (dstArray == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to dstArray", ESMC_CONTEXT, &rc);
    return rc;
  }
  // srcArray and dstArray may not point to the identical Array object
  if (srcArray == dstArray){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
      "- srcArray and dstArray must not be identical", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // get the current VM and VM releated information
  VM *vm = VM::getCurrent(&localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  int localPet = vm->getLocalPet();
  int petCount = vm->getPetCount();

  // determine consistent typekindFactor and factor across all PETs
  if (factor != NULL){
    // must define a valid typekind
    if (typekindFactor == ESMF_NOKIND){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- must specify valid typekindFactor on PETs that provide factor",
        ESMC_CONTEXT, &rc);
      return rc;
    }
    if (typekindFactor != ESMC_TYPEKIND_I4
      && typekindFactor != ESMC_TYPEKIND_I8
      && typekindFactor != ESMC_TYPEKIND_R4
      && typekindFactor != ESMC_TYPEKIND_R8){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_NOT_IMPL,
        "- method not implemented for specified typekindFactor", ESMC_CONTEXT,
        &rc);
      return rc;
    }
  }else{
    // set typekindFactor to ESMF_NOKIND 
    // -> this Pet will not provide factor
    typekindFactor = ESMF_NOKIND;
  }
  
  // communicate typekindFactor across all Pets
  ESMC_TypeKind_Flag *typekindList = new ESMC_TypeKind_Flag[petCount];
  vm->allgather(&typekindFactor, typekindList, sizeof(ESMC_TypeKind_Flag));
  // Check that all non-ESMF_NOKIND typekindList elements match,
  // set local typekindFactor accordingly and keep track of Pets that have
  // factors.
  int *factorPetList = new int[petCount];
  int factorPetCount = 0; // reset
  typekindFactor = ESMF_NOKIND;  // initialize
  for (int i=0; i<petCount; i++){
    if (typekindList[i] != ESMF_NOKIND)
      factorPetList[factorPetCount++] = i;
    if (typekindFactor == ESMF_NOKIND){
      typekindFactor = typekindList[i];  // set to 1st element not ESMF_NOKIND
    }else{
      // check following elements against the typekindFactors and tensorMixFlag
      if (typekindList[i] != ESMF_NOKIND){
        if (typekindFactor != typekindList[i]){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
            "- TypeKind mismatch between PETs", ESMC_CONTEXT, &rc);
          return rc;
        }
      }
    }
  }
  delete [] typekindList;
  // set typekindFactor default in case there were no factors provided
  if (factorPetCount == 0)
    typekindFactor = srcArray->getTypekind();
  
  // set factorLocal
  void *factorLocal;
  if (typekindFactor == ESMC_TYPEKIND_R4){
    ESMC_R4 *factorLocalT = new ESMC_R4;
    if (factor)
      *factorLocalT = *(ESMC_R4 *)factor;
    else
      *factorLocalT = 1.;
    factorLocal = (void *)factorLocalT;
  }else if (typekindFactor == ESMC_TYPEKIND_R8){
    ESMC_R8 *factorLocalT = new ESMC_R8;
    if (factor)
      *factorLocalT = *(ESMC_R8 *)factor;
    else
      *factorLocalT = 1.;
    factorLocal = (void *)factorLocalT;
  }else if (typekindFactor == ESMC_TYPEKIND_I4){
    ESMC_I4 *factorLocalT = new ESMC_I4;
    if (factor)
      *factorLocalT = *(ESMC_I4 *)factor;
    else
      *factorLocalT = 1;
    factorLocal = (void *)factorLocalT;
  }else if (typekindFactor == ESMC_TYPEKIND_I8){
    ESMC_I8 *factorLocalT = new ESMC_I8;
    if (factor)
      *factorLocalT = *(ESMC_I8 *)factor;
    else
      *factorLocalT = 1;
    factorLocal = (void *)factorLocalT;
  }
  
  if (factorPetCount > 0){
    // communicate factorLocal variables and check for consistency
    int factorSize = ESMC_TypeKind_FlagSize(typekindFactor);
    char *factorLocalList = new char[petCount*factorSize];
    vm->allgather(factorLocal, factorLocalList, factorSize);
    // prime factorLocal with value from first Pet with factor
    memcpy(factorLocal, factorLocalList + factorSize * factorPetList[0],
      factorSize);
    // check against all other factorLocal entries
    for (int i=1; i<factorPetCount; i++){
      char *factorPtr = factorLocalList + factorSize * factorPetList[i];
      if (memcmp(factorLocal, factorPtr, factorSize) != 0){
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
          "- Factor mismatch between PETs", ESMC_CONTEXT, &rc);
        return rc;
      }
    }
    delete [] factorLocalList;
  }
  delete [] factorPetList;
  
  // set up local factorList and factorIndexList
  int factorListCount;
  int srcN;
  int dstN;
  int *factorIndexList;
  
  if (srcToDstTransposeMap == NULL){
    // srcToDstTransposeMap not specified -> default mode
    
    // src and dst Arrays must have identical number of exclusive elements
    int srcElementCount = 0; // init
    const int *srcElementCountPTile =
      srcArray->distgrid->getElementCountPTile();
    for (int i=0; i<srcArray->distgrid->getTileCount(); i++)
      srcElementCount += srcElementCountPTile[i];
    int dstElementCount = 0; // init
    const int *dstElementCountPTile =
      dstArray->distgrid->getElementCountPTile();
    for (int i=0; i<dstArray->distgrid->getTileCount(); i++)
      dstElementCount += dstElementCountPTile[i];
    if (srcElementCount != dstElementCount){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- srcArray and dstArray must provide identical number of exclusive"
        " elements", ESMC_CONTEXT, &rc);
      return rc;
    }
  
    // implemented via sparseMatMul using identity matrix
    const int *srcElementCountPDe = srcArray->distgrid->getElementCountPDe();
    int *const*srcArbSeqIndexCountPCollPLocalDe =
      srcArray->distgrid->getElementCountPCollPLocalDe();
    const int *srcLocalDeToDeMap = srcArray->delayout->getLocalDeToDeMap();
    int srcLocalDeCount = srcArray->delayout->getLocalDeCount();
    factorListCount = 0;  // init
    for (int i=0; i<srcLocalDeCount; i++)
      factorListCount += srcElementCountPDe[srcLocalDeToDeMap[i]];
    // set up factorIndexList
    srcN = 1; // 1 component seqIndex
    dstN = 1; // 1 component seqIndex
    factorIndexList = new int[(srcN+dstN)*factorListCount];
    int jj = 0; // reset
    for (int i=0; i<srcLocalDeCount; i++){
      //TODO: this is hardcoded for first collocation only
      int arbSeqIndexCount = srcArbSeqIndexCountPCollPLocalDe[0][i];
      const int *srcArbSeqIndexListPLocalDe =
        srcArray->distgrid->getArbSeqIndexList(i,1);
      if (srcArbSeqIndexListPLocalDe){
        for (int j=0; j<arbSeqIndexCount; j++){
          factorIndexList[2*jj] = factorIndexList[2*jj+1] =
            srcArbSeqIndexListPLocalDe[j];
          ++jj;
        }
      }else{
        // multi-dim loop object
        ArrayElement arrayElement(srcArray, i);
        // set up to skip over undistributed, i.e. tensor dimensions
        const int *srcArrayToDistGridMap = srcArray->getArrayToDistGridMap();
        for (int j=0; j<srcArray->getRank(); j++)
          if (srcArrayToDistGridMap[j]==0) arrayElement.setSkipDim(j);
        // fill in the factorIndexList
        while(arrayElement.isWithin()){
          SeqIndex seqIndex = arrayElement.getSequenceIndexExclusive();
          factorIndexList[2*jj] = factorIndexList[2*jj+1] =
            seqIndex.decompSeqIndex;
          ++jj; // increment counter
          arrayElement.next();
        } // end while over all exclusive elements
      }
    }  
    
  }else{
    // srcToDstTransposeMap specified -> transpose mode
    
    // src and dst Arrays must be of same rank
    int rank = srcArray->getRank();
    if (rank != dstArray->getRank()){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- in transpose mode srcArray and dstArray must be of same rank",
        ESMC_CONTEXT, &rc);
      return rc;
    }

    // src and dst Arrays must be have same number of tiles
    int tileCount = srcArray->distgrid->getTileCount();
    if (tileCount != dstArray->distgrid->getTileCount()){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- in transpose mode srcArray and dstArray must have same number of"
        " tiles", ESMC_CONTEXT, &rc);
      return rc;
    }

    // check srcToDstTransposeMap input
    if (srcToDstTransposeMap->dimCount != 1){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_RANK,
        "- srcToDstTransposeMap must be of rank 1", ESMC_CONTEXT, &rc);
      return rc;
    }
    if (srcToDstTransposeMap->extent[0] != rank){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_SIZE,
        "- srcToDstTransposeMap must provide rank values", ESMC_CONTEXT, &rc);
      return rc;
    }
    int *srcToDstTMap = new int[rank];
    for (int i=0; i<rank; i++){
      srcToDstTMap[i] = srcToDstTransposeMap->array[i] - 1; // shift to base 0
      int j;
      for (j=0; j<rank; j++)
        if (srcToDstTransposeMap->array[j] == i+1) break;
      if (j==rank){
        // did not find (i+1) value in transpose map
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_VALUE,
          "- srcToDstTransposeMap values must be unique and within range:"
          " [1,..,rank].", ESMC_CONTEXT, &rc);
        return rc;
      }
    }
    
    // size of dims in src and dst Arrays must pairwise match in each tile
    const int *srcArrayToDistGridMap = srcArray->getArrayToDistGridMap();
    const int *dstArrayToDistGridMap = dstArray->getArrayToDistGridMap();
    const int *srcMinIndexPDimPTile =
      srcArray->distgrid->getMinIndexPDimPTile();
    const int *dstMinIndexPDimPTile =
      dstArray->distgrid->getMinIndexPDimPTile();
    const int *srcMaxIndexPDimPTile =
      srcArray->distgrid->getMaxIndexPDimPTile();
    const int *dstMaxIndexPDimPTile =
      dstArray->distgrid->getMaxIndexPDimPTile();
    int srcDimCount = srcArray->distgrid->getDimCount();
    int dstDimCount = dstArray->distgrid->getDimCount();
    // prepare dstArrayToTensorMap
    int *dstArrayToTensorMap = new int[rank];
    int tensorIndex = 0;
    for (int jj=0; jj<rank; jj++){
      if (dstArrayToDistGridMap[jj]==0){
        // tensor dimension
        dstArrayToTensorMap[jj] = tensorIndex;
        ++tensorIndex;
      }
    }
    factorListCount = 0;
    int *localStart = new int[tileCount];  // localPet's start index
    int *localSize = new int[tileCount];   // localPet's number of elements
    for (int i=0; i<tileCount; i++){
      int tileFactorListCount = 1;
      // prepare decomposition along last distributed dim in srcArray
      //TODO: this assumes that srcArray has at least one distr. dim
      int lastDimSize = srcMaxIndexPDimPTile[i*srcDimCount+srcDimCount-1]
        - srcMinIndexPDimPTile[i*srcDimCount+srcDimCount-1] + 1;
      int intervalSize = lastDimSize/petCount;
      int extraElements = lastDimSize%petCount;
      localStart[i] = 0;  // initialize
      localStart[i] += localPet * intervalSize;
      localSize[i] = intervalSize;
      if (localPet < extraElements){
        localStart[i] += localPet;
        localSize[i] += 1;
      }
#if 0
fprintf(stderr, "%d start:%d, size:%d\n", localPet, localStart[i], localSize[i]);
#endif
      // check every tile
      int srcTensorIndex = 0;
      int dstTensorIndex = 0;
      for (int jj=0; jj<rank; jj++){
        int srcSize;
        int dstSize;
        int j = srcArrayToDistGridMap[jj];  // j is dimIndex bas 1, or 0 undist.
        if (j){
          // decomposed dimension 
          --j;  // shift to basis 0
          srcSize = srcMaxIndexPDimPTile[i*srcDimCount+j]
            - srcMinIndexPDimPTile[i*srcDimCount+j] + 1;
          if (j == srcDimCount-1)
            tileFactorListCount *= localSize[i];
          else
            tileFactorListCount *= srcSize;
        }else{
          // tensor dimension
          srcSize = srcArray->undistUBound[srcTensorIndex]
            - srcArray->undistLBound[srcTensorIndex] + 1;
          ++srcTensorIndex;
          tileFactorListCount *= srcSize;
        }
        int jjj = srcToDstTMap[jj];         // src -> dst dimension mapping
        j = dstArrayToDistGridMap[jjj];     // j is dimIndex bas 1, or 0 undist.
        if (j){
          // decomposed dimension 
          --j;  // shift to basis 0
          dstSize = dstMaxIndexPDimPTile[i*dstDimCount+j]
            - dstMinIndexPDimPTile[i*dstDimCount+j] + 1;
        }else{
          // tensor dimension
          dstTensorIndex = dstArrayToTensorMap[jjj];
          dstSize = dstArray->undistUBound[dstTensorIndex]
            - dstArray->undistLBound[dstTensorIndex] + 1;
        }
#if 0
fprintf(stderr, "%d, %d\n", srcSize, dstSize);
#endif
        if (srcSize != dstSize){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
            "- in transpose mode the size of srcArray and dstArray dimensions"
            " must pairwise match", ESMC_CONTEXT, &rc);
          return rc;
        }
      }
      factorListCount += tileFactorListCount;
    }
#if 0
fprintf(stderr, "factorListCount = %d\n", factorListCount);
#endif
    // set up factorIndexList
    srcN = 2; // 2 component seqIndex
    dstN = 2; // 2 component seqIndex
    factorIndexList = new int[(srcN+dstN)*factorListCount];
    // prepare to fill in factorIndexList elements
    int factorIndexListIndex = 0; // reset
    int *dstTuple = new int[rank];
    for (int i=0; i<tileCount; i++){
      // initialize multi dim index loop
      vector<int> offsets;
      vector<int> sizes;
      int tensorIndex = 0;  // reset
      for (int jj=0; jj<rank; jj++){
        int j = srcArrayToDistGridMap[jj];  // j is dimIndex bas 1, or 0 undist.
        if (j){
          // decomposed dimension 
          --j;  // shift to basis 0
          if (j == srcDimCount-1){
            // share work across PETs for last decomposed dim
            offsets.push_back(localStart[i]);
            sizes.push_back(localSize[i]);
          }else{
            offsets.push_back(0);
            sizes.push_back(srcMaxIndexPDimPTile[i*srcDimCount+j]
              - srcMinIndexPDimPTile[i*srcDimCount+j] + 1);
          }
        }else{
          // tensor dimension
          offsets.push_back(0);
          sizes.push_back(srcArray->undistUBound[tensorIndex]
            - srcArray->undistLBound[tensorIndex] + 1);
          ++tensorIndex;
        }
      }
      
      MultiDimIndexLoop multiDimIndexLoop(offsets, sizes);
      while(multiDimIndexLoop.isWithin()){
        const int *srcTuple = multiDimIndexLoop.getIndexTuple();
        // redist is identity mapping, but must consider srcToDstTMap
        // srcTuple --(srcToDstTMap)--> dstTuple
        for (int j=0; j<rank; j++)
          dstTuple[srcToDstTMap[j]] = srcTuple[j];
        // determine seq indices
        SeqIndex srcSeqIndex = srcArray->getSequenceIndexTile(i+1, srcTuple);
        SeqIndex dstSeqIndex = dstArray->getSequenceIndexTile(i+1, dstTuple);
        // fill this info into factorIndexList
        int fili = 4*factorIndexListIndex;
        factorIndexList[fili]   = srcSeqIndex.decompSeqIndex;
        factorIndexList[fili+1] = srcSeqIndex.tensorSeqIndex;
        factorIndexList[fili+2] = dstSeqIndex.decompSeqIndex;
        factorIndexList[fili+3] = dstSeqIndex.tensorSeqIndex;

        ++factorIndexListIndex;        
        multiDimIndexLoop.next();
      } // multi dim index loop
    }
    // garbage collection    
    delete [] srcToDstTMap;
    delete [] dstArrayToTensorMap;
    delete [] localStart;
    delete [] localSize;
    delete [] dstTuple;
  }
  
#if 0
fprintf(stderr, "factorListCount = %d\n", factorListCount);
for (int i=0; i<factorListCount; i++)
  fprintf(stderr, "%d, %d, %d\n", i, factorIndexList[2*i], 
    factorIndexList[2*i+1]);
#endif

  // load type specific factorList with "1"
  void *factorList;
  if (typekindFactor == ESMC_TYPEKIND_R4){
    ESMC_R4 *factorListT = new ESMC_R4[factorListCount];
    for (int i=0; i<factorListCount; i++)
      factorListT[i] = *(ESMC_R4 *)factorLocal;
    factorList = (void *)factorListT;
    delete (ESMC_R4 *)factorLocal;
  }else if (typekindFactor == ESMC_TYPEKIND_R8){
    ESMC_R8 *factorListT = new ESMC_R8[factorListCount];
    for (int i=0; i<factorListCount; i++)
      factorListT[i] = *(ESMC_R8 *)factorLocal;
    factorList = (void *)factorListT;
    delete (ESMC_R8 *)factorLocal;
  }else if (typekindFactor == ESMC_TYPEKIND_I4){
    ESMC_I4 *factorListT = new ESMC_I4[factorListCount];
    for (int i=0; i<factorListCount; i++)
      factorListT[i] = *(ESMC_I4 *)factorLocal;
    factorList = (void *)factorListT;
    delete (ESMC_I4 *)factorLocal;
  }else if (typekindFactor == ESMC_TYPEKIND_I8){
    ESMC_I8 *factorListT = new ESMC_I8[factorListCount];
    for (int i=0; i<factorListCount; i++)
      factorListT[i] = *(ESMC_I8 *)factorLocal;
    factorList = (void *)factorListT;
    delete (ESMC_I4 *)factorLocal;
  }
  
  // prepare SparseMatrix vector
  vector<SparseMatrix> sparseMatrix;
  sparseMatrix.push_back(SparseMatrix(typekindFactor, factorList,
    factorListCount, srcN, dstN, factorIndexList));
  
  // precompute sparse matrix multiplication
  int srcTermProcessing = 0;  // no need to use auto-tuning to figure this out
  localrc = sparseMatMulStore(srcArray, dstArray, routehandle, sparseMatrix,
    false, &srcTermProcessing, pipelineDepthArg);
  // garbage collection
  delete [] factorIndexList;
  if (typekindFactor == ESMC_TYPEKIND_R4){
    ESMC_R4 *factorListT = (ESMC_R4 *)factorList;
    delete [] factorListT;
  }else if (typekindFactor == ESMC_TYPEKIND_R8){
    ESMC_R8 *factorListT = (ESMC_R8 *)factorList;
    delete [] factorListT;
  }else if (typekindFactor == ESMC_TYPEKIND_I4){
    ESMC_I4 *factorListT = (ESMC_I4 *)factorList;
    delete [] factorListT;
  }else if (typekindFactor == ESMC_TYPEKIND_I8){
    ESMC_I8 *factorListT = (ESMC_I8 *)factorList;
    delete [] factorListT;
  }
  // error handling
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;

  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(exception &x){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      x.what(), ESMC_CONTEXT, &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------

  //-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::redist()"
//BOPI
// !IROUTINE:  ESMCI::Array::redist
//
// !INTERFACE:
int Array::redist(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  Array *srcArray,                      // in    - source Array
  Array *dstArray,                      // inout - destination Array
  RouteHandle **routehandle,            // inout - handle to precomputed comm
  ESMC_CommFlag commflag,               // in    - communication options
  bool *finishedflag,                   // out   - TEST ops finished or not
  bool *cancelledflag,                  // out   - any cancelled operations
  bool checkflag                        // in    - false: (def.) basic chcks
                                        //         true:  full input check
  ){
//
// !DESCRIPTION:
//    Execute an Array redistribution
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  // implemented via sparseMatMul
  localrc = sparseMatMul(srcArray, dstArray, routehandle,
    commflag, finishedflag, cancelledflag, ESMC_REGION_SELECT, 
    ESMC_TERMORDER_FREE, checkflag);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::redistRelease()"
//BOPI
// !IROUTINE:  ESMCI::Array::redistRelease
//
// !INTERFACE:
int Array::redistRelease(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  RouteHandle *routehandle        // inout -
  ){
//
// !DESCRIPTION:
//    Release information for an Array redistribution
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  // implemented via sparseMatMul
  localrc = sparseMatMulRelease(routehandle);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;

  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
bool operator==(SeqIndex a, SeqIndex b){
  if (a.decompSeqIndex != b.decompSeqIndex) return false;
  // decompSeqIndex was equal -> check tensorSeqIndex
  return (a.tensorSeqIndex == b.tensorSeqIndex);
}
bool operator!=(SeqIndex a, SeqIndex b){
  if (a.decompSeqIndex != b.decompSeqIndex) return true;
  // decompSeqIndex was equal -> check tensorSeqIndex
  return (a.tensorSeqIndex != b.tensorSeqIndex);
}
bool operator<(SeqIndex a, SeqIndex b){
  if (a.decompSeqIndex < b.decompSeqIndex) return true;
  if (a.decompSeqIndex > b.decompSeqIndex) return false;
  // decompSeqIndex must be equal
  return (a.tensorSeqIndex < b.tensorSeqIndex);
}
//-----------------------------------------------------------------------------


#define ASMMPROFILE___disable

#define ASMMSTOREPRINT___disable

#ifdef ASMMSTOREPRINT
  char asmmstoreprintfile[160];
  FILE *asmmstoreprintfp;
#endif


namespace ArrayHelper{
  
  struct Deflator{
    SeqIndex seqIndex;
    int index;
  };
  bool operator==(Deflator a, Deflator b){
    return (a.seqIndex == b.seqIndex);
  }
  bool operator!=(Deflator a, Deflator b){
    return (a.seqIndex != b.seqIndex);
  }
  bool operator<(Deflator a, Deflator b){
    return (a.seqIndex < b.seqIndex);
  }
  
#define MSG_DST_CONTIG

  struct DstInfo{
    int linIndex;               // if vector element then this is start
    int vectorLength;           // ==1 single element, > 1 vector element
    SeqIndex seqIndex;          // if vector element then this is start
    SeqIndex partnerSeqIndex;   // if vector element then this is start
    void *factor;               // if vector element then this factor for all
    int bufferIndex;            // index into the receive buffer
  };
  bool scalarOrderDstInfo(DstInfo a, DstInfo b){
#ifdef MSG_DST_CONTIG
    if (a.seqIndex == b.seqIndex)
      return (a.partnerSeqIndex < b.partnerSeqIndex);
    else
      return (a.seqIndex < b.seqIndex);
#else
    if (a.partnerSeqIndex == b.partnerSeqIndex)
      return (a.seqIndex < b.seqIndex);
    else
      return (a.partnerSeqIndex < b.partnerSeqIndex);
#endif
  }
  bool vectorOrderDstInfo(DstInfo a, DstInfo b){
    if (a.seqIndex.decompSeqIndex == b.seqIndex.decompSeqIndex)
      if (a.partnerSeqIndex.decompSeqIndex == b.partnerSeqIndex.decompSeqIndex)
        return (a.seqIndex.tensorSeqIndex < b.seqIndex.tensorSeqIndex);
      else
        return
          (a.partnerSeqIndex.decompSeqIndex < b.partnerSeqIndex.decompSeqIndex);
    else
      return (a.seqIndex.decompSeqIndex < b.seqIndex.decompSeqIndex);
  }

  struct DstInfoSrcSeqSort{
    vector<DstInfo>::iterator pp;
    int recvnbVectorIndex;
    DstInfoSrcSeqSort(vector<DstInfo>::iterator pp_, int recvnbVectorIndex_){
      pp=pp_;
      recvnbVectorIndex=recvnbVectorIndex_;
    }
  };
  bool operator<(DstInfoSrcSeqSort a, DstInfoSrcSeqSort b){
    return (a.pp->partnerSeqIndex < b.pp->partnerSeqIndex);
  }
  
  struct SrcInfo{
    int linIndex;               // if vector element then this is start
    int vectorLength;           // ==1 single element, > 1 vector element
    SeqIndex seqIndex;          // if vector element then this is start
    SeqIndex partnerSeqIndex;   // if vector element then this is start
    void *factor;               // if vector element then this factor for all
  };
  bool scalarOrderSrcInfo(SrcInfo a, SrcInfo b){
#ifdef MSG_DST_CONTIG
    if (a.partnerSeqIndex == b.partnerSeqIndex)
      return (a.seqIndex < b.seqIndex);
    else
      return (a.partnerSeqIndex < b.partnerSeqIndex);
#else
    if (a.seqIndex == b.seqIndex)
      return (a.partnerSeqIndex < b.partnerSeqIndex);
    else
      return (a.seqIndex < b.seqIndex);
#endif
  }
  bool vectorOrderSrcInfo(SrcInfo a, SrcInfo b){
    if (a.partnerSeqIndex.decompSeqIndex == b.partnerSeqIndex.decompSeqIndex)
      if (a.seqIndex.decompSeqIndex == b.seqIndex.decompSeqIndex)
        return
          (a.partnerSeqIndex.tensorSeqIndex < b.partnerSeqIndex.tensorSeqIndex);
      else
        return (a.seqIndex.decompSeqIndex < b.seqIndex.decompSeqIndex);
    else
      return
        (a.partnerSeqIndex.decompSeqIndex < b.partnerSeqIndex.decompSeqIndex);
  }
  
  struct RecvnbElement{
    int srcPet;
    int srcDe;          // global DE index of src DE in src DELayout
    int srcLocalDe;     // local enumeration of srcDe
    int dstDe;          // global DE index of dst DE in local DELayout
    int dstLocalDe;     // local enumeration of dstDe
    char **bufferInfo;  // indirect buffer pointer to support buffer resize
                        // during exec() via BufferInfo structure in XXE
    int partnerDeDataCount;
    int recvnbIndex;
    bool vectorFlag;  // control vectorization
    vector<DstInfo> dstInfoTable;
    int localPet;
    int petCount;
    //
    int appendRecvnb(XXE *xxe, int predicateBitField, int srcTermProcessing,
      int dataSizeSrc, int k);
    int appendRecv(XXE *xxe, int predicateBitField, int srcTermProcessing,
      int dataSizeSrc, int k);
    int appendZeroSuperScalar(XXE *xxe, int predicateBitField,
      int srcLocalDeCount, XXE::TKId elementTK);
    int appendProductSum(XXE *xxe, int predicateBitField, int srcTermProcessing,
      int srcLocalDeCount, XXE::TKId elementTK, XXE::TKId valueTK,
      XXE::TKId factorTK, int dataSizeDst, int dataSizeSrc, int dataSizeFactors,
      char **rraList, int rraCount);
    static int appendSingleProductSum(XXE *xxe,
      int predicateBitField, int srcTermProcessing,  int srcLocalDeCount,
      XXE::TKId elementTK, XXE::TKId valueTK, XXE::TKId factorTK, 
      int dataSizeDst, int dataSizeSrc, int dataSizeFactors, char **rraList,
      int rraCount, vector<RecvnbElement> recvnbVector);
    int appendTestWaitProductSum(XXE *xxe, int predicateBitField,
      int srcTermProcessing, int srcLocalDeCount, XXE::TKId elementTK,
      XXE::TKId valueTK, XXE::TKId factorTK, int dataSizeDst, int dataSizeSrc,
      int dataSizeFactors, char **rraList, int rraCount, int k);
  };
  bool operator<(RecvnbElement a, RecvnbElement b){
    int aSrcPet = (a.localPet - a.srcPet + a.petCount) % a.petCount;
    int bSrcPet = (b.localPet - b.srcPet + b.petCount) % b.petCount;
    if (aSrcPet == bSrcPet)
      if (a.srcDe == b.srcDe)
        return (a.dstDe < b.dstDe);
      else
        return (a.srcDe < b.srcDe);
    else
      return (aSrcPet < bSrcPet);
  }
  int RecvnbElement::appendRecvnb(XXE *xxe, int predicateBitField,
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayHelper::RecvnbElement::appendRecvnb()"
    int srcTermProcessing, int dataSizeSrc, int k){
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    int rc = ESMC_RC_NOT_IMPL;              // final return code
#ifdef ASMMSTOREPRINT
    fprintf(asmmstoreprintfp, "gjt: XXE::recvnb on localPet %d from Pet %d\n",
      localPet, srcPet);
#endif
    recvnbIndex = xxe->count;  // store index for the associated wait
    int tag = 0;  // no need for special tags - messages are ordered to match
    // determine bufferItemCount according to srcTermProcessing
    int bufferItemCount = 0; // reset
    if (srcTermProcessing == 0)
      bufferItemCount = partnerDeDataCount;
    else{
      vector<ArrayHelper::DstInfo>::iterator pp = dstInfoTable.begin();
      while (pp != dstInfoTable.end()){
        SeqIndex seqIndex = pp->seqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          ++pp;
          if ((pp == dstInfoTable.end()) || !(seqIndex == pp->seqIndex)) break;
        } // for srcTermProcessing
        ++bufferItemCount;
      }
    }
    // append the recvnb operation
    localrc = xxe->appendRecvnb(predicateBitField, bufferInfo, 
      bufferItemCount * dataSizeSrc, srcPet, tag, vectorFlag, true);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
    char *tempString = new char[160];
    sprintf(tempString, "Recvnb: vectorFlag=%d", vectorFlag);
    localrc = xxe->appendProfileMessage(predicateBitField, tempString);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    sprintf(tempString, "<(%04d/%04d)-Rnb(%d/%d)-(%04d/%04d)> ",
      srcDe, srcPet, k, recvnbIndex, dstDe, localPet);
    localrc = xxe->appendProfileMessage(predicateBitField, tempString);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    sprintf(tempString, "/Recvnb (%d/)", k);
    localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
      xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    delete [] tempString;
#endif
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }
  int RecvnbElement::appendRecv(XXE *xxe, int predicateBitField,
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayHelper::RecvnbElement::appendRecv()"
    int srcTermProcessing, int dataSizeSrc, int k){
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    int rc = ESMC_RC_NOT_IMPL;              // final return code
#ifdef ASMMSTOREPRINT
    fprintf(asmmstoreprintfp, "gjt: XXE::recv on localPet %d from Pet %d\n",
      localPet, srcPet);
#endif
    recvnbIndex = xxe->count;  // store index for the associated wait
    int tag = 0;  // no need for special tags - messages are ordered to match
    // determine bufferItemCount according to srcTermProcessing
    int bufferItemCount = 0; // reset
    if (srcTermProcessing == 0)
      bufferItemCount = partnerDeDataCount;
    else{
      vector<ArrayHelper::DstInfo>::iterator pp = dstInfoTable.begin();
      while (pp != dstInfoTable.end()){
        SeqIndex seqIndex = pp->seqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          ++pp;
          if ((pp == dstInfoTable.end()) || !(seqIndex == pp->seqIndex)) break;
        } // for srcTermProcessing
        ++bufferItemCount;
      }
    }
    // append the recv operation
    localrc = xxe->appendRecv(predicateBitField, bufferInfo, 
      bufferItemCount * dataSizeSrc, srcPet, tag, vectorFlag, true);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
    char *tempString = new char[160];
    sprintf(tempString, "Recv: vectorFlag=%d", vectorFlag);
    localrc = xxe->appendProfileMessage(predicateBitField, tempString);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    sprintf(tempString, "<(%04d/%04d)-Rcv(%d/%d)-(%04d/%04d)> ",
      srcDe, srcPet, k, recvnbIndex, dstDe, localPet);
    localrc = xxe->appendProfileMessage(predicateBitField, tempString);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    sprintf(tempString, "/Recv (%d/)", k);
    localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
      xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    delete [] tempString;
#endif
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }
  int RecvnbElement::appendZeroSuperScalar(XXE *xxe, int predicateBitField,
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayHelper::RecvnbElement::appendZeroSuperScalar()"
    int srcLocalDeCount, XXE::TKId elementTK){
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    int rc = ESMC_RC_NOT_IMPL;              // final return code
    int j = dstLocalDe;
    int rraIndex = srcLocalDeCount + j; // localDe index into dstArray shifted 
                                        // by srcArray localDeCount
#ifdef ASMMPROFILE
    localrc = xxe->appendWtimer(predicateBitField, "slct zero",
      xxe->count, xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#endif
    int xxeIndex = xxe->count;  // need this beyond the increment
    localrc = xxe->appendZeroSuperScalarRRA(predicateBitField, elementTK,
      rraIndex, dstInfoTable.size(), vectorFlag);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    XXE::ZeroSuperScalarRRAInfo *xxeZeroSuperScalarRRAInfo =
      (XXE::ZeroSuperScalarRRAInfo *)&(xxe->stream[xxeIndex]);
    // fill rraOffsetList[]
    int vectorLength = dstInfoTable.begin()->vectorLength;  // store time vLen
    int k=0;  // reset
    for (vector<ArrayHelper::DstInfo>::iterator pp = dstInfoTable.begin();
      pp != dstInfoTable.end(); ++pp){
      int linIndex = pp->linIndex;
      xxeZeroSuperScalarRRAInfo->rraOffsetList[k] = linIndex/vectorLength;
      ++k;
    }
#ifdef ASMMPROFILE
    localrc = xxe->appendWtimer(predicateBitField, "/slct zero",
      xxe->count, xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#endif
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }
  int RecvnbElement::appendProductSum(XXE *xxe, int predicateBitField,
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayHelper::RecvnbElement::appendProductSum()"
    int srcTermProcessing,  int srcLocalDeCount, XXE::TKId elementTK,
    XXE::TKId valueTK, XXE::TKId factorTK, int dataSizeDst, int dataSizeSrc,
    int dataSizeFactors, char **rraList, int rraCount){
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    int rc = ESMC_RC_NOT_IMPL;              // final return code
    int j = dstLocalDe;
    int vectorLength = dstInfoTable.begin()->vectorLength;  // store time vLen
    if (srcTermProcessing==0){
      // do all the processing on the dst side
      // use super-scalar "+=*" operation containing all terms
      int rraIndex = srcLocalDeCount + j; // localDe index into dstArray
                                          // shifted by srcArray localDeCount
      int termCount = dstInfoTable.size();
      int xxeIndex = xxe->count;  // need this beyond the increment
      localrc = xxe->appendProductSumSuperScalarDstRRA(predicateBitField,
        elementTK, valueTK, factorTK, rraIndex, termCount, bufferInfo,
        vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      XXE::ProductSumSuperScalarDstRRAInfo *xxeProductSumSuperScalarDstRRAInfo =
        (XXE::ProductSumSuperScalarDstRRAInfo *)&(xxe->stream[xxeIndex]);
      int *rraOffsetList = xxeProductSumSuperScalarDstRRAInfo->rraOffsetList;
      void **factorList = xxeProductSumSuperScalarDstRRAInfo->factorList;
      int *valueOffsetList =
        xxeProductSumSuperScalarDstRRAInfo->valueOffsetList;
      // fill in rraOffsetList, factorList, valueOffsetList
      vector<ArrayHelper::DstInfo>::iterator pp = dstInfoTable.begin();
      for (int kk=0; kk<termCount; kk++){
        rraOffsetList[kk] = pp->linIndex/vectorLength;
        factorList[kk] = (void *)(pp->factor);
        valueOffsetList[kk] = pp->bufferIndex;
        ++pp;
      } // for kk - termCount
      // need to fill in sensible elements and values or timing will be bogus
      switch (elementTK){ // elements in dstArray
      case XXE::R4:
        for (int kk=0; kk<termCount; kk++){
          ESMC_R4 *element =
            (ESMC_R4 *)rraList[rraIndex] + rraOffsetList[kk] * vectorLength;
          for (int k=0; k<vectorLength; k++)
            *(element+k) = (ESMC_R4)0.;
        }
        break;
      case XXE::R8:
        for (int kk=0; kk<termCount; kk++){
          ESMC_R8 *element =
            (ESMC_R8 *)rraList[rraIndex] + rraOffsetList[kk] * vectorLength;
          for (int k=0; k<vectorLength; k++)
            *(element+k) = (ESMC_R8)0.;
        }
        break;
      case XXE::I4:
        for (int kk=0; kk<termCount; kk++){
          ESMC_I4 *element =
            (ESMC_I4 *)rraList[rraIndex] + rraOffsetList[kk] * vectorLength;
          for (int k=0; k<vectorLength; k++)
            *(element+k) = (ESMC_I4)0;
        }
        break;
      case XXE::I8:
        for (int kk=0; kk<termCount; kk++){
          ESMC_I8 *element =
            (ESMC_I8 *)rraList[rraIndex] + rraOffsetList[kk] * vectorLength;
          for (int k=0; k<vectorLength; k++)
            *(element+k) = (ESMC_I8)0;
        }
        break;
      default:
        break;
      }
      char *buffer = *bufferInfo; // access buffer through layer of indirection
      switch (valueTK){ // values in buffer
      case XXE::R4:
        for (int kk=0; kk<termCount; kk++){
          ESMC_R4 *value =
            (ESMC_R4 *)buffer + valueOffsetList[kk] * vectorLength;
          for (int k=0; k<vectorLength; k++)
            *(value+k) = (ESMC_R4)0.01;
        }
        break;
      case XXE::R8:
        for (int kk=0; kk<termCount; kk++){
          ESMC_R8 *value =
            (ESMC_R8 *)buffer + valueOffsetList[kk] * vectorLength;
          for (int k=0; k<vectorLength; k++)
            *(value+k) = (ESMC_R8)0.01;
        }
        break;
      case XXE::I4:
        for (int kk=0; kk<termCount; kk++){
          ESMC_I4 *value =
            (ESMC_I4 *)buffer + valueOffsetList[kk] * vectorLength;
          for (int k=0; k<vectorLength; k++)
            *(value+k) = (ESMC_I4)1;
        }
        break;
      case XXE::I8:
        for (int kk=0; kk<termCount; kk++){
          ESMC_I8 *value =
            (ESMC_I8 *)buffer + valueOffsetList[kk] * vectorLength;
          for (int k=0; k<vectorLength; k++)
            *(value+k) = (ESMC_I8)1;
        }
        break;
      default:
        break;
      }
      xxe->optimizeElement(xxeIndex);
      
#define MSG_DEFLATE___disable
#define MSG_DEFLATE

#ifdef MSG_DEFLATE
      // cannot use ProductSumSuperScalarContigRRAInfo operation unless the
      // values even in the deflated message are stored in continuous order
      // --> extremely unlikely for a regridding operation, but possible for
      // redist.
      double dt_sScalar = 1.;
      double dt_sScalarC = 2.;  // force dt_sScalar to be picked below
#else
      double dt_sScalar;
      localrc = xxe->exec(rraCount, rraList, &vectorLength, 0x0, NULL, NULL,
        &dt_sScalar, xxeIndex, xxeIndex);
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, &rc)) return rc;
      // try super-scalar contig "+=*" operation in XXE stream
      xxe->stream[xxeIndex].opId = XXE::productSumSuperScalarContigRRA;
      XXE::ProductSumSuperScalarContigRRAInfo
        *xxeProductSumSuperScalarContigRRAInfo =
        (XXE::ProductSumSuperScalarContigRRAInfo *)&(xxe->stream[xxeIndex]);
      // only change members that are different wrt super-scalar operation
      xxeProductSumSuperScalarContigRRAInfo->valueList = bufferInfo;
      xxe->optimizeElement(xxeIndex);
      double dt_sScalarC;
      localrc = xxe->exec(rraCount, rraList, &vectorLength, 0x0, NULL, NULL,
        &dt_sScalarC, xxeIndex, xxeIndex);
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, &rc)) return rc;
#endif
#ifdef ASMMPROFILE
      char *tempString = new char[160];
      vector<int> sortOffsetList(termCount);
      copy(rraOffsetList, rraOffsetList+termCount, sortOffsetList.begin());
      sort(sortOffsetList.begin(), sortOffsetList.end());
      int diffElements = 1;
      for (int i=1; i<termCount; i++)
        if (sortOffsetList[i] != sortOffsetList[i-1]) ++diffElements;
#endif
      // decide for the fastest option
      
      if (dt_sScalar < dt_sScalarC){
        // use productSumSuperScalarDstRRA
#ifdef ASMMPROFILE
        sprintf(tempString, "use productSumSuperScalarDstRRA for %d terms"
          " (diffElements=%d), vectorFlag=%d", termCount, diffElements,
          vectorFlag);
#endif
        xxe->stream[xxeIndex].opId = XXE::productSumSuperScalarDstRRA;
        xxeProductSumSuperScalarDstRRAInfo->valueBase = bufferInfo;
      }else{
        // use productSumSuperScalarContigRRA
#ifdef ASMMPROFILE
        sprintf(tempString, "use productSumSuperScalarContigRRA for %d terms"
          " (diffElements=%d), vectorFlag=%d", termCount, diffElements,
          vectorFlag);
#endif
        // nothing to be done -> already set from last trial
      }
#ifdef ASMMPROFILE
      localrc = xxe->appendProfileMessage(predicateBitField, tempString);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      delete [] tempString;
#endif
    }else{
      // do some processing on the src side
      // use super-scalar "+=" operation containing all terms
      int rraIndex = srcLocalDeCount + j; // localDe index into dstArray
                                          // shifted by srcArray localDeCount
      // determine bufferItemCount according to srcTermProcessing
      int bufferItemCount = 0; // reset
      vector<ArrayHelper::DstInfo>::iterator pp = dstInfoTable.begin();
      while (pp != dstInfoTable.end()){
        SeqIndex seqIndex = pp->seqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          ++pp;
          if ((pp == dstInfoTable.end()) || !(seqIndex == pp->seqIndex)) break;
        } // for srcTermProcessing
        ++bufferItemCount;
      }
      int xxeIndex = xxe->count;  // need this beyond the increment
      localrc = xxe->appendSumSuperScalarDstRRA(predicateBitField, elementTK,
        valueTK, rraIndex, bufferItemCount, bufferInfo, vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      XXE::SumSuperScalarDstRRAInfo *xxeSumSuperScalarDstRRAInfo =
        (XXE::SumSuperScalarDstRRAInfo *)&(xxe->stream[xxeIndex]);
      int *rraOffsetList = xxeSumSuperScalarDstRRAInfo->rraOffsetList;
      int *valueOffsetList = xxeSumSuperScalarDstRRAInfo->valueOffsetList;
      // fill in rraOffsetList, valueList
      int bufferItem = 0; // reset
      pp = dstInfoTable.begin();  // reset
      while (pp != dstInfoTable.end()){
        rraOffsetList[bufferItem] = pp->linIndex/vectorLength;
        valueOffsetList[bufferItem] = bufferItem;
        // skip dstInfoTable elements that were summed up on the src side
        SeqIndex seqIndex = pp->seqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          ++pp;
          if ((pp == dstInfoTable.end()) || !(seqIndex == pp->seqIndex)) break;
        } // for srcTermProcessing
        ++bufferItem;
      }
#ifdef ASMMPROFILE
      char *tempString = new char[160];
      vector<int> sortOffsetList(bufferItem);
      copy(rraOffsetList, rraOffsetList+bufferItem, sortOffsetList.begin());
      sort(sortOffsetList.begin(), sortOffsetList.end());
      int diffElements = 1;
      for (int i=1; i<bufferItem; i++)
        if (sortOffsetList[i] != sortOffsetList[i-1]) ++diffElements;
      sprintf(tempString, "use sumSuperScalarDstRRA for termCount=%d -> reduced"
        " to %d terms (diffElements=%d)", partnerDeDataCount, bufferItem,
        diffElements);
      localrc = xxe->appendProfileMessage(predicateBitField, tempString);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      delete [] tempString;
#endif
    }
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }
  int RecvnbElement::appendTestWaitProductSum(XXE *xxe, int predicateBitField,
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayHelper::RecvnbElement::appendTestWaitProductSum()"
    int srcTermProcessing, int srcLocalDeCount, XXE::TKId elementTK,
    XXE::TKId valueTK, XXE::TKId factorTK, int dataSizeDst, int dataSizeSrc,
    int dataSizeFactors, char **rraList, int rraCount, int k){
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    int rc = ESMC_RC_NOT_IMPL;              // final return code
#ifdef ASMMPROFILE
    char *tempString = new char[160];
    sprintf(tempString, "WaitProductSum (%d/)", k);
    localrc = xxe->appendWtimer(predicateBitField|XXE::filterBitNbWaitFinish,
      tempString, xxe->count, xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    delete [] tempString;
#endif
    // use substream for doing the productSum
    XXE *xxeSub;
    try{
      xxeSub = new XXE(xxe->vm, 10, 10, 10);
    }catch (...){
      ESMC_LogDefault.AllocError(ESMC_CONTEXT, &rc);
      return rc;
    }
    localrc = xxe->storeXxeSub(xxeSub); // for XXE garbage collection
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
    localrc = appendProductSum(xxeSub, predicateBitField, srcTermProcessing,
      srcLocalDeCount, elementTK, valueTK, factorTK, dataSizeDst, dataSizeSrc,
      dataSizeFactors, rraList, rraCount);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    // attach to testOnIndexSub element with filterBitNbTestFinish
    localrc = xxe->appendTestOnIndexSub(
      predicateBitField|XXE::filterBitNbTestFinish, xxeSub, 0, 0, recvnbIndex);
    if (ESMC_LogDefault.MsgFoundError(localrc, 
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    // attach to waitOnIndexSub element with filterBitNbWaitFinish
    localrc = xxe->appendWaitOnIndexSub(
      predicateBitField|XXE::filterBitNbWaitFinish, xxeSub, 0, 0, recvnbIndex);
    if (ESMC_LogDefault.MsgFoundError(localrc, 
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    // cancel for this receive operations
    localrc = xxe->appendCancelIndex(     
      predicateBitField|XXE::filterBitCancel, recvnbIndex);
    if (ESMC_LogDefault.MsgFoundError(localrc, 
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
    tempString = new char[160];
    sprintf(tempString, "/WaitProductSum (%d/)", k);
    localrc = xxe->appendWtimer(predicateBitField|XXE::filterBitNbWaitFinish,
      tempString, xxe->count, xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    delete [] tempString;
#endif
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }
  int RecvnbElement::appendSingleProductSum(XXE *xxe,
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayHelper::RecvnbElement::appendSingleProductSum()"
    int predicateBitField, int srcTermProcessing,  int srcLocalDeCount,
    XXE::TKId elementTK, XXE::TKId valueTK, XXE::TKId factorTK, 
    int dataSizeDst, int dataSizeSrc, int dataSizeFactors, char **rraList,
    int rraCount, vector<RecvnbElement> recvnbVector){
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    int rc = ESMC_RC_NOT_IMPL;              // final return code
    if (recvnbVector.size() < 1){
      // bail out successfully
      rc = ESMF_SUCCESS;
      return rc;
    }
    // use first element in recvnbVector to access info that is the same for
    // all elements
    int vectorLength =
      recvnbVector[0].dstInfoTable.begin()->vectorLength;  // store time vLen
    bool vectorFlag = recvnbVector[0].vectorFlag;
    if (srcTermProcessing==0){
      // do all the processing on the dst side
      // use super-scalar "+=*" operation containing all terms
      // this single productSum operation supports multiple receive buffers:
      // -> construct the necessary helper variables
      int termCount = 0;
      vector<void *>bufferInfoList;
      vector<int> rraIndexList;
      for (int i=0; i<recvnbVector.size(); i++){
        termCount += recvnbVector[i].dstInfoTable.size();
        bufferInfoList.push_back(recvnbVector[i].bufferInfo);
        rraIndexList.push_back(srcLocalDeCount + recvnbVector[i].dstLocalDe);
      }
      int xxeIndex = xxe->count;  // need this beyond the increment
      localrc = xxe->appendProductSumSuperScalarListDstRRA(predicateBitField,
        elementTK, valueTK, factorTK, rraIndexList, termCount, bufferInfoList,
        vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      
      XXE::ProductSumSuperScalarListDstRRAInfo
        *xxeProductSumSuperScalarListDstRRAInfo =
        (XXE::ProductSumSuperScalarListDstRRAInfo *)&(xxe->stream[xxeIndex]);
      int *rraOffsetList =
        xxeProductSumSuperScalarListDstRRAInfo->rraOffsetList;
      void **factorList = xxeProductSumSuperScalarListDstRRAInfo->factorList;
      int *valueOffsetList =
        xxeProductSumSuperScalarListDstRRAInfo->valueOffsetList;
      int *baseListIndexList =
        xxeProductSumSuperScalarListDstRRAInfo->baseListIndexList;
      // set up a temporary vector for sorting for TERMORDER_SRCSEQ
      vector<DstInfoSrcSeqSort> dstInfoSort;
      vector<ArrayHelper::DstInfo>::iterator pp;
      for (int i=0; i<recvnbVector.size(); i++){
        // append terms from buffer "i"
        for (pp=recvnbVector[i].dstInfoTable.begin();
          pp!=recvnbVector[i].dstInfoTable.end(); ++pp){
          dstInfoSort.push_back(DstInfoSrcSeqSort(pp, i));
        }
      }
      // do the actual sort
      sort(dstInfoSort.begin(), dstInfoSort.end());
      // fill in rraOffsetList, factorList, valueOffsetList, baseListIndexList
      for (int i=0; i<dstInfoSort.size(); i++){
        rraOffsetList[i] = dstInfoSort[i].pp->linIndex/vectorLength;
        factorList[i] = (void *)(dstInfoSort[i].pp->factor);
        valueOffsetList[i] = dstInfoSort[i].pp->bufferIndex;
        baseListIndexList[i] = dstInfoSort[i].recvnbVectorIndex;
      }
    }else{
      // do some processing on the src side
      // use super-scalar "+=" operation containing all terms
      // this single productSum operation supports multiple receive buffers:
      // -> construct the necessary helper variables

      //TODO: need to fix this implementation to work with srcTermProcessing > 1
      
      int bufferItemCount = 0;
      vector<void *>bufferInfoList;
      vector<int> rraIndexList;
      vector<ArrayHelper::DstInfo>::iterator pp;
      for (int i=0; i<recvnbVector.size(); i++){
        pp = recvnbVector[i].dstInfoTable.begin();
        while (pp != recvnbVector[i].dstInfoTable.end()){
          SeqIndex seqIndex = pp->seqIndex;
          for (int term=0; term<srcTermProcessing; term++){
            ++pp;
            if ((pp == recvnbVector[i].dstInfoTable.end()) 
            || !(seqIndex == pp->seqIndex)) break;
          } // for srcTermProcessing
          ++bufferItemCount;
        }
        bufferInfoList.push_back(recvnbVector[i].bufferInfo);
        rraIndexList.push_back(srcLocalDeCount + recvnbVector[i].dstLocalDe);
      }
      int xxeIndex = xxe->count;  // need this beyond the increment
      localrc = xxe->appendSumSuperScalarListDstRRA(predicateBitField,
        elementTK, valueTK, rraIndexList, bufferItemCount, bufferInfoList,
        vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      
      XXE::SumSuperScalarListDstRRAInfo
        *xxeSumSuperScalarListDstRRAInfo =
        (XXE::SumSuperScalarListDstRRAInfo *)&(xxe->stream[xxeIndex]);
      int *rraOffsetList =
        xxeSumSuperScalarListDstRRAInfo->rraOffsetList;
      int *valueOffsetList =
        xxeSumSuperScalarListDstRRAInfo->valueOffsetList;
      int *baseListIndexList =
        xxeSumSuperScalarListDstRRAInfo->baseListIndexList;
      // set up a temporary vector for sorting for TERMORDER_SRCSEQ
      vector<DstInfoSrcSeqSort> dstInfoSort;
      for (int i=0; i<recvnbVector.size(); i++){
        // append terms from buffer "i"
        int bufferItem = 0; // reset
        pp = recvnbVector[i].dstInfoTable.begin();
        while (pp != recvnbVector[i].dstInfoTable.end()){
          dstInfoSort.push_back(DstInfoSrcSeqSort(pp, i));
          pp->bufferIndex = bufferItem; // adjust to modified buffer structure
          SeqIndex seqIndex = pp->seqIndex;
          for (int term=0; term<srcTermProcessing; term++){
            ++pp;
            if ((pp == recvnbVector[i].dstInfoTable.end()) 
            || !(seqIndex == pp->seqIndex)) break;
          } // for srcTermProcessing
          ++bufferItem;
        }
      }
      // sanity check to ensure srcTermProcessing was correctly considered
      if (bufferItemCount != dstInfoSort.size()) {
        ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_INCONS,
          "- inconsistent number of buffer elements", ESMC_CONTEXT, &rc);
        return rc;  // bail out
      }
      // do the actual sort
      sort(dstInfoSort.begin(), dstInfoSort.end());
      // fill in rraOffsetList, valueOffsetList, baseListIndexList
      for (int i=0; i<dstInfoSort.size(); i++){
        rraOffsetList[i] = dstInfoSort[i].pp->linIndex/vectorLength;
        valueOffsetList[i] = dstInfoSort[i].pp->bufferIndex;
        baseListIndexList[i] = dstInfoSort[i].recvnbVectorIndex;
      }
    }
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }

  
  struct LinIndexContigBlock{
    int linIndex;
    int linIndexCount;
  };
  struct SendnbElement{
    int dstPet;
    int dstDe;          // global DE index of dst DE in dst DELayout
    int dstLocalDe;     // local enumeration of dstDe
    int srcDe;          // global DE index of src DE in local DELayout
    int srcLocalDe;     // local enumeration of srcDe
    char **bufferInfo;  // indirect buffer pointer to support buffer resize
                        // during exec() via BufferInfo structure in XXE
    int partnerDeDataCount;
    int sendnbIndex;
    bool vectorFlag;  // control vectorization
    vector<SrcInfo> srcInfoTable;
    vector<LinIndexContigBlock> linIndexContigBlockList;
    int localPet;
    int petCount;
    //
    int appendSendnb(XXE *xxe, int predicateBitField, int srcTermProcessing,
      XXE::TKId elementTK, XXE::TKId valueTK, XXE::TKId factorTK,
      int dataSizeSrc, char **rraList, int rraCount, int k);
    int appendSend(XXE *xxe, int predicateBitField, int srcTermProcessing,
      XXE::TKId elementTK, XXE::TKId valueTK, XXE::TKId factorTK,
      int dataSizeSrc, char **rraList, int rraCount, int k);
    int appendSendRecv(XXE *xxe, int predicateBitField, int srcTermProcessing,
      XXE::TKId elementTK, XXE::TKId valueTK, XXE::TKId factorTK,
      int dataSizeSrc, char **rraList, int rraCount, int kSend,
      vector<RecvnbElement>::iterator pRecv, int kRecv);
  };
  bool operator<(SendnbElement a, SendnbElement b){
    int aDstPet = (a.dstPet - a.localPet + a.petCount) % a.petCount;
    int bDstPet = (b.dstPet - b.localPet + b.petCount) % b.petCount;
    if (aDstPet == bDstPet)
      if (a.srcDe == b.srcDe)
        return (a.dstDe < b.dstDe);
      else
        return (a.srcDe < b.srcDe);
    else
      return (aDstPet < bDstPet);
  }
  int SendnbElement::appendSendnb(XXE *xxe, int predicateBitField,
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayHelper::SendnbElement::appendSendnb()"
    int srcTermProcessing, XXE::TKId elementTK, XXE::TKId valueTK,
    XXE::TKId factorTK, int dataSizeSrc, char **rraList, int rraCount, int k){
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    int rc = ESMC_RC_NOT_IMPL;              // final return code
    int tag = 0;  // no need for special tags - messages are ordered to match
    int j = srcLocalDe;
    int vectorLength = srcInfoTable.begin()->vectorLength;  // store time vLen
    if (srcTermProcessing==0){
      // do all the processing on the dst side
      int count = linIndexContigBlockList.size();
#ifdef ASMMSTOREPRINT
      fprintf(asmmstoreprintfp, "gjt: XXE::sendnb from localPet %d to Pet %d\n",
        localPet, dstPet);
#endif
      if (count == 1){
        // sendnbRRA out of single contiguous linIndex run
#ifdef ASMMSTOREPRINT
        fprintf(asmmstoreprintfp, "gjt: single contiguous linIndex run "
          "on src side\n");
#endif
        sendnbIndex = xxe->count;  // store index for the associated wait
        localrc = xxe->appendSendnbRRA(predicateBitField,
          linIndexContigBlockList[0].linIndex/vectorLength * dataSizeSrc,
          partnerDeDataCount * dataSizeSrc, dstPet, j, tag, vectorFlag);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
        char *tempString = new char[160];
        sprintf(tempString, "Contiguous send: vectorFlag=%d", vectorFlag);
        localrc = xxe->appendProfileMessage(predicateBitField, tempString);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        delete [] tempString;
#endif
      }else{
        // use intermediate buffer
#ifdef ASMMSTOREPRINT
        fprintf(asmmstoreprintfp, "gjt: non-contiguous linIndex on "
          "src side -> need buffer \n");
#endif
        // use intermediate buffer
        // memGatherSrcRRA pieces into intermediate buffer
        int xxeIndex = xxe->count;  // need this beyond the increment
        localrc = xxe->appendMemGatherSrcRRA(predicateBitField, bufferInfo,
          valueTK, j, count, vectorFlag, true);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        XXE::MemGatherSrcRRAInfo *xxeMemGatherSrcRRAInfo =
          (XXE::MemGatherSrcRRAInfo *) &(xxe->stream[xxeIndex]);
        // try typekind specific memGatherSrcRRA
        for (int kk=0; kk<count; kk++){
          xxeMemGatherSrcRRAInfo->rraOffsetList[kk] =
            linIndexContigBlockList[kk].linIndex/vectorLength;
          xxeMemGatherSrcRRAInfo->countList[kk] =
            linIndexContigBlockList[kk].linIndexCount;
        }
        double dt_tk;
        localrc = xxe->exec(rraCount, rraList, &vectorLength, 0x0, NULL, NULL,
          &dt_tk, xxeIndex, xxeIndex);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, &rc)) return rc;
        // try byte option for memGatherSrcRRA
        xxeMemGatherSrcRRAInfo->dstBaseTK = XXE::BYTE;
        for (int kk=0; kk<count; kk++){
          // scale to byte
          xxeMemGatherSrcRRAInfo->rraOffsetList[kk] *= dataSizeSrc;
          xxeMemGatherSrcRRAInfo->countList[kk] *= dataSizeSrc;
        }
        double dt_byte;
        localrc = xxe->exec(rraCount, rraList, &vectorLength, 0x0, NULL, NULL,
          &dt_byte, xxeIndex, xxeIndex);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMSTOREPRINT
        fprintf(asmmstoreprintfp, "gjt - on localPet %d memGatherSrcRRA took "
          "dt_tk=%g s and dt_byte=%g s for count=%d\n", localPet, dt_tk,
          dt_byte, count);
#endif
        // decide for the fastest option
        if (dt_byte < dt_tk){
          // use byte option for memGatherSrcRRA
          // -> nothing to do because this was the last mode tested
        }else{
          // use typekind specific memGatherSrcRRA
          xxeMemGatherSrcRRAInfo->dstBaseTK = valueTK;
          for (int k=0; k<count; k++){
            // return to element units
            xxeMemGatherSrcRRAInfo->rraOffsetList[k] =
              linIndexContigBlockList[k].linIndex/vectorLength;
            xxeMemGatherSrcRRAInfo->countList[k] =
              linIndexContigBlockList[k].linIndexCount;
          }
        }
#ifdef ASMMPROFILE
        char *tempString = new char[160];
        sprintf(tempString, "MemGatherSrcRRA: vectorFlag=%d", vectorFlag);
        localrc = xxe->appendProfileMessage(predicateBitField, tempString);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        sprintf(tempString, "/MemGatherSrcRRA (%d/)", k);
        localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
          xxe->count);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        delete [] tempString;
#endif
        // sendnb out of contiguous intermediate buffer
        sendnbIndex = xxe->count;  // store index for the associated wait
        localrc = xxe->appendSendnb(predicateBitField, bufferInfo,
          partnerDeDataCount * dataSizeSrc, dstPet, tag, vectorFlag, true);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      }
    }else{
      // do some processing on the src side
      // determine bufferItemCount according to srcTermProcessing
      int bufferItemCount = 0; // reset
      vector<ArrayHelper::SrcInfo>::iterator pp = srcInfoTable.begin();
      while (pp != srcInfoTable.end()){
        SeqIndex partnerSeqIndex = pp->partnerSeqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          ++pp;
          if ((pp == srcInfoTable.end()) ||
            !(partnerSeqIndex == pp->partnerSeqIndex)) break;
        } // for srcTermProcessing
        ++bufferItemCount;
      }
      // zero out intermediate buffer
      localrc = xxe->appendZeroMemset(predicateBitField, bufferInfo,
        bufferItemCount * dataSizeSrc, vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
      char *tempString = new char[160];
      sprintf(tempString, "/ZeroVector (%d/)", k);
      localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
        xxe->count);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      delete [] tempString;
#endif
      // use super-scalar "+=*" operation containing all terms
      int xxeIndex = xxe->count;  // need this beyond the increment
      localrc = xxe->appendProductSumSuperScalarSrcRRA(predicateBitField,
        valueTK, valueTK, factorTK, j, srcInfoTable.size(), bufferInfo, 
        vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      XXE::ProductSumSuperScalarSrcRRAInfo *xxeProductSumSuperScalarSrcRRAInfo =
        (XXE::ProductSumSuperScalarSrcRRAInfo *)&(xxe->stream[xxeIndex]);
      int *rraOffsetList = xxeProductSumSuperScalarSrcRRAInfo->rraOffsetList;
      void **factorList = xxeProductSumSuperScalarSrcRRAInfo->factorList;
      int *elementOffsetList =
        xxeProductSumSuperScalarSrcRRAInfo->elementOffsetList;
      // fill in rraOffsetList, factorList, elementOffsetList
      int bufferItem = 0; // reset
      int kk = 0; // reset
      pp = srcInfoTable.begin();  // reset
      while (pp != srcInfoTable.end()){
        SeqIndex partnerSeqIndex = pp->partnerSeqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          rraOffsetList[kk] = pp->linIndex/vectorLength;
          factorList[kk] = (void *)(pp->factor);
          elementOffsetList[kk] = bufferItem;
          ++pp;
          ++kk;
          if ((pp == srcInfoTable.end()) ||
            !(partnerSeqIndex == pp->partnerSeqIndex)) break;
        } // for srcTermProcessing
        ++bufferItem;
      }
#ifdef ASMMPROFILE
      tempString = new char[160];
      sprintf(tempString, "use productSumSuperScalarSrcRRA for termCount=%d"
        " -> reducing it to %d terms", srcInfoTable.size(), bufferItem);
      localrc = xxe->appendProfileMessage(predicateBitField, tempString);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      sprintf(tempString, "productSumSuperScalarSrcRRA: vectorLength=%d",
        vectorLength);
      localrc = xxe->appendProfileMessage(predicateBitField, tempString);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      sprintf(tempString, "/PSSSSrcRRA (%d/)", k);
      localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
        xxe->count);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      delete [] tempString;
#endif
      // sendnb out of contiguous intermediate buffer
      sendnbIndex = xxe->count;  // store index for the associated wait
      localrc = xxe->appendSendnb(predicateBitField, bufferInfo,
        bufferItemCount * dataSizeSrc, dstPet, tag, vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    }
#ifdef ASMMPROFILE
    char *tempString = new char[160];
    sprintf(tempString, "<(%04d/%04d)-Snb(%d/%d)-(%04d/%04d)> ",
      srcDe, localPet, k, sendnbIndex, dstDe, dstPet);
    localrc = xxe->appendProfileMessage(predicateBitField, tempString);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    sprintf(tempString, "/Sendnb (%d/)", k);
    localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
      xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    delete [] tempString;
#endif
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }
      
  int SendnbElement::appendSend(XXE *xxe, int predicateBitField,
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayHelper::SendnbElement::appendSend()"
    int srcTermProcessing, XXE::TKId elementTK, XXE::TKId valueTK,
    XXE::TKId factorTK, int dataSizeSrc, char **rraList, int rraCount, int k){
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    int rc = ESMC_RC_NOT_IMPL;              // final return code
    int tag = 0;  // no need for special tags - messages are ordered to match
    int j = srcLocalDe;
    int vectorLength = srcInfoTable.begin()->vectorLength;  // store time vLen
    if (srcTermProcessing==0){
      // do all the processing on the dst side
      int count = linIndexContigBlockList.size();
#ifdef ASMMSTOREPRINT
      fprintf(asmmstoreprintfp, "gjt: XXE::send from localPet %d to Pet %d\n",
        localPet, dstPet);
#endif
      if (count == 1){
        // sendRRA out of single contiguous linIndex run
#ifdef ASMMSTOREPRINT
        fprintf(asmmstoreprintfp, "gjt: single contiguous linIndex run "
          "on src side\n");
#endif
        sendnbIndex = xxe->count;  // store index for the associated wait
        localrc = xxe->appendSendRRA(predicateBitField,
          linIndexContigBlockList[0].linIndex/vectorLength * dataSizeSrc,
          partnerDeDataCount * dataSizeSrc, dstPet, j, tag, vectorFlag);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
        char *tempString = new char[160];
        sprintf(tempString, "Contiguous send: vectorFlag=%d", vectorFlag);
        localrc = xxe->appendProfileMessage(predicateBitField, tempString);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        delete [] tempString;
#endif
      }else{
        // use intermediate buffer
#ifdef ASMMSTOREPRINT
        fprintf(asmmstoreprintfp, "gjt: non-contiguous linIndex on "
          "src side -> need buffer \n");
#endif
        // use intermediate buffer
        // memGatherSrcRRA pieces into intermediate buffer
        int xxeIndex = xxe->count;  // need this beyond the increment
        localrc = xxe->appendMemGatherSrcRRA(predicateBitField, bufferInfo,
          valueTK, j, count, vectorFlag, true);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        XXE::MemGatherSrcRRAInfo *xxeMemGatherSrcRRAInfo =
          (XXE::MemGatherSrcRRAInfo *) &(xxe->stream[xxeIndex]);
        // try typekind specific memGatherSrcRRA
        for (int kk=0; kk<count; kk++){
          xxeMemGatherSrcRRAInfo->rraOffsetList[kk] =
            linIndexContigBlockList[kk].linIndex/vectorLength;
          xxeMemGatherSrcRRAInfo->countList[kk] =
            linIndexContigBlockList[kk].linIndexCount;
        }
        double dt_tk;
        localrc = xxe->exec(rraCount, rraList, &vectorLength, 0x0, NULL, NULL,
          &dt_tk, xxeIndex, xxeIndex);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, &rc)) return rc;
        // try byte option for memGatherSrcRRA
        xxeMemGatherSrcRRAInfo->dstBaseTK = XXE::BYTE;
        for (int kk=0; kk<count; kk++){
          // scale to byte
          xxeMemGatherSrcRRAInfo->rraOffsetList[kk] *= dataSizeSrc;
          xxeMemGatherSrcRRAInfo->countList[kk] *= dataSizeSrc;
        }
        double dt_byte;
        localrc = xxe->exec(rraCount, rraList, &vectorLength, 0x0, NULL, NULL,
          &dt_byte, xxeIndex, xxeIndex);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMSTOREPRINT
        fprintf(asmmstoreprintfp, "gjt - on localPet %d memGatherSrcRRA took "
          "dt_tk=%g s and dt_byte=%g s for count=%d\n", localPet, dt_tk,
          dt_byte, count);
#endif
        // decide for the fastest option
        if (dt_byte < dt_tk){
          // use byte option for memGatherSrcRRA
          // -> nothing to do because this was the last mode tested
        }else{
          // use typekind specific memGatherSrcRRA
          xxeMemGatherSrcRRAInfo->dstBaseTK = valueTK;
          for (int k=0; k<count; k++){
            // return to element units
            xxeMemGatherSrcRRAInfo->rraOffsetList[k] =
              linIndexContigBlockList[k].linIndex/vectorLength;
            xxeMemGatherSrcRRAInfo->countList[k] =
              linIndexContigBlockList[k].linIndexCount;
          }
        }
#ifdef ASMMPROFILE
        char *tempString = new char[160];
        sprintf(tempString, "MemGatherSrcRRA: vectorFlag=%d", vectorFlag);
        localrc = xxe->appendProfileMessage(predicateBitField, tempString);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        sprintf(tempString, "/MemGatherSrcRRA (%d/)", k);
        localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
          xxe->count);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        delete [] tempString;
#endif
        // send out of contiguous intermediate buffer
        sendnbIndex = xxe->count;  // store index for the associated wait
        localrc = xxe->appendSend(predicateBitField, bufferInfo,
          partnerDeDataCount * dataSizeSrc, dstPet, tag, vectorFlag, true);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      }
    }else{
      // do some processing on the src side
      // determine bufferItemCount according to srcTermProcessing
      int bufferItemCount = 0; // reset
      vector<ArrayHelper::SrcInfo>::iterator pp = srcInfoTable.begin();
      while (pp != srcInfoTable.end()){
        SeqIndex partnerSeqIndex = pp->partnerSeqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          ++pp;
          if ((pp == srcInfoTable.end()) ||
            !(partnerSeqIndex == pp->partnerSeqIndex)) break;
        } // for srcTermProcessing
        ++bufferItemCount;
      }
      // zero out intermediate buffer
      localrc = xxe->appendZeroMemset(predicateBitField, bufferInfo,
        bufferItemCount * dataSizeSrc, vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
      char *tempString = new char[160];
      sprintf(tempString, "/ZeroVector (%d/)", k);
      localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
        xxe->count);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      delete [] tempString;
#endif
      // use super-scalar "+=*" operation containing all terms
      int xxeIndex = xxe->count;  // need this beyond the increment
      localrc = xxe->appendProductSumSuperScalarSrcRRA(predicateBitField,
        valueTK, valueTK, factorTK, j, srcInfoTable.size(), bufferInfo, 
        vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      XXE::ProductSumSuperScalarSrcRRAInfo *xxeProductSumSuperScalarSrcRRAInfo =
        (XXE::ProductSumSuperScalarSrcRRAInfo *)&(xxe->stream[xxeIndex]);
      int *rraOffsetList = xxeProductSumSuperScalarSrcRRAInfo->rraOffsetList;
      void **factorList = xxeProductSumSuperScalarSrcRRAInfo->factorList;
      int *elementOffsetList =
        xxeProductSumSuperScalarSrcRRAInfo->elementOffsetList;
      // fill in rraOffsetList, factorList, elementOffsetList
      int bufferItem = 0; // reset
      int kk = 0; // reset
      pp = srcInfoTable.begin();  // reset
      while (pp != srcInfoTable.end()){
        SeqIndex partnerSeqIndex = pp->partnerSeqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          rraOffsetList[kk] = pp->linIndex/vectorLength;
          factorList[kk] = (void *)(pp->factor);
          elementOffsetList[kk] = bufferItem;
          ++pp;
          ++kk;
          if ((pp == srcInfoTable.end()) ||
            !(partnerSeqIndex == pp->partnerSeqIndex)) break;
        } // for srcTermProcessing
        ++bufferItem;
      }
#ifdef ASMMPROFILE
      tempString = new char[160];
      sprintf(tempString, "use productSumSuperScalarSrcRRA for termCount=%d"
        " -> reducing it to %d terms", srcInfoTable.size(), bufferItem);
      localrc = xxe->appendProfileMessage(predicateBitField, tempString);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      sprintf(tempString, "productSumSuperScalarSrcRRA: vectorLength=%d",
        vectorLength);
      localrc = xxe->appendProfileMessage(predicateBitField, tempString);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      sprintf(tempString, "/PSSSSrcRRA (%d/)", k);
      localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
        xxe->count);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      delete [] tempString;
#endif
      // send out of contiguous intermediate buffer
      sendnbIndex = xxe->count;  // store index for the associated wait
      localrc = xxe->appendSend(predicateBitField, bufferInfo,
        bufferItemCount * dataSizeSrc, dstPet, tag, vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    }
#ifdef ASMMPROFILE
    char *tempString = new char[160];
    sprintf(tempString, "<(%04d/%04d)-Snb(%d/%d)-(%04d/%04d)> ",
      srcDe, localPet, k, sendnbIndex, dstDe, dstPet);
    localrc = xxe->appendProfileMessage(predicateBitField, tempString);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    sprintf(tempString, "/Send (%d/)", k);
    localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
      xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    delete [] tempString;
#endif
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }
      
  int SendnbElement::appendSendRecv(XXE *xxe, int predicateBitField,
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayHelper::SendnbElement::appendSendRecv()"
    int srcTermProcessing, XXE::TKId elementTK, XXE::TKId valueTK,
    XXE::TKId factorTK, int dataSizeSrc, char **rraList, int rraCount,
    int kSend, vector<RecvnbElement>::iterator pRecv, int kRecv){
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    int rc = ESMC_RC_NOT_IMPL;              // final return code
    int tag = 0;  // no need for special tags - messages are ordered to match
    int j = srcLocalDe;
    int vectorLength = srcInfoTable.begin()->vectorLength;  // store time vLen
    // determine recv side bufferItemCount according to srcTermProcessing
    int dstBufferItemCount = 0; // reset
    if (srcTermProcessing == 0)
      dstBufferItemCount = pRecv->partnerDeDataCount;
    else{
      vector<ArrayHelper::DstInfo>::iterator pp = pRecv->dstInfoTable.begin();
      while (pp != pRecv->dstInfoTable.end()){
        SeqIndex seqIndex = pp->seqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          ++pp;
          if ((pp == pRecv->dstInfoTable.end()) || 
            !(seqIndex == pp->seqIndex)) break;
        } // for srcTermProcessing
        ++dstBufferItemCount;
      }
    }
    if (srcTermProcessing==0){
      // do all the processing on the dst side
      int count = linIndexContigBlockList.size();
#ifdef ASMMSTOREPRINT
      fprintf(asmmstoreprintfp, "gjt: XXE::sendrecv from localPet %d to "
        " Pet %d\n", localPet, dstPet);
#endif
      if (count == 1){
        // sendRRArecv out of single contiguous linIndex run
#ifdef ASMMSTOREPRINT
        fprintf(asmmstoreprintfp, "gjt: single contiguous linIndex run "
          "on src side\n");
#endif
        sendnbIndex = xxe->count;  // store index for the associated wait
        pRecv->recvnbIndex = xxe->count;  // store index for the associated wait
        localrc = xxe->appendSendRRARecv(predicateBitField,
          linIndexContigBlockList[0].linIndex/vectorLength * dataSizeSrc,
          pRecv->bufferInfo, partnerDeDataCount * dataSizeSrc, 
          dstBufferItemCount * dataSizeSrc, pRecv->srcPet, dstPet, 
          j, tag, tag, vectorFlag, true);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
        char *tempString = new char[160];
        sprintf(tempString, "Contiguous sendrecv: vectorFlag=%d", vectorFlag);
        localrc = xxe->appendProfileMessage(predicateBitField, tempString);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        delete [] tempString;
#endif
      }else{
        // use intermediate buffer
#ifdef ASMMSTOREPRINT
        fprintf(asmmstoreprintfp, "gjt: non-contiguous linIndex on "
          "src side -> need buffer \n");
#endif
        // use intermediate buffer
        // memGatherSrcRRA pieces into intermediate buffer
        int xxeIndex = xxe->count;  // need this beyond the increment
        localrc = xxe->appendMemGatherSrcRRA(predicateBitField, bufferInfo,
          valueTK, j, count, vectorFlag, true);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        XXE::MemGatherSrcRRAInfo *xxeMemGatherSrcRRAInfo =
          (XXE::MemGatherSrcRRAInfo *) &(xxe->stream[xxeIndex]);
        // try typekind specific memGatherSrcRRA
        for (int kk=0; kk<count; kk++){
          xxeMemGatherSrcRRAInfo->rraOffsetList[kk] =
            linIndexContigBlockList[kk].linIndex/vectorLength;
          xxeMemGatherSrcRRAInfo->countList[kk] =
            linIndexContigBlockList[kk].linIndexCount;
        }
        double dt_tk;
        localrc = xxe->exec(rraCount, rraList, &vectorLength, 0x0, NULL, NULL,
          &dt_tk, xxeIndex, xxeIndex);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, &rc)) return rc;
        // try byte option for memGatherSrcRRA
        xxeMemGatherSrcRRAInfo->dstBaseTK = XXE::BYTE;
        for (int kk=0; kk<count; kk++){
          // scale to byte
          xxeMemGatherSrcRRAInfo->rraOffsetList[kk] *= dataSizeSrc;
          xxeMemGatherSrcRRAInfo->countList[kk] *= dataSizeSrc;
        }
        double dt_byte;
        localrc = xxe->exec(rraCount, rraList, &vectorLength, 0x0, NULL, NULL,
          &dt_byte, xxeIndex, xxeIndex);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMSTOREPRINT
        fprintf(asmmstoreprintfp, "gjt - on localPet %d memGatherSrcRRA took "
          "dt_tk=%g s and dt_byte=%g s for count=%d\n", localPet, dt_tk,
          dt_byte, count);
#endif
        // decide for the fastest option
        if (dt_byte < dt_tk){
          // use byte option for memGatherSrcRRA
          // -> nothing to do because this was the last mode tested
        }else{
          // use typekind specific memGatherSrcRRA
          xxeMemGatherSrcRRAInfo->dstBaseTK = valueTK;
          for (int k=0; k<count; k++){
            // return to element units
            xxeMemGatherSrcRRAInfo->rraOffsetList[k] =
              linIndexContigBlockList[k].linIndex/vectorLength;
            xxeMemGatherSrcRRAInfo->countList[k] =
              linIndexContigBlockList[k].linIndexCount;
          }
        }
#ifdef ASMMPROFILE
        char *tempString = new char[160];
        sprintf(tempString, "MemGatherSrcRRA: vectorFlag=%d", vectorFlag);
        localrc = xxe->appendProfileMessage(predicateBitField, tempString);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        sprintf(tempString, "/MemGatherSrcRRA (%d/)", k);
        localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
          xxe->count);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        delete [] tempString;
#endif
        // sendrecv out of contiguous intermediate buffer
        sendnbIndex = xxe->count;  // store index for the associated wait
        pRecv->recvnbIndex = xxe->count;  // store index for the associated wait
        localrc = xxe->appendSendRecv(predicateBitField, bufferInfo,
          pRecv->bufferInfo, partnerDeDataCount * dataSizeSrc, 
          dstBufferItemCount * dataSizeSrc, pRecv->srcPet, dstPet, tag, tag,
          vectorFlag, true, true);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      }
    }else{
      // do some processing on the src side
      // determine bufferItemCount according to srcTermProcessing
      int bufferItemCount = 0; // reset
      vector<ArrayHelper::SrcInfo>::iterator pp = srcInfoTable.begin();
      while (pp != srcInfoTable.end()){
        SeqIndex partnerSeqIndex = pp->partnerSeqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          ++pp;
          if ((pp == srcInfoTable.end()) ||
            !(partnerSeqIndex == pp->partnerSeqIndex)) break;
        } // for srcTermProcessing
        ++bufferItemCount;
      }
      // zero out intermediate buffer
      localrc = xxe->appendZeroMemset(predicateBitField, bufferInfo,
        bufferItemCount * dataSizeSrc, vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
      char *tempString = new char[160];
      sprintf(tempString, "/ZeroVector (%d/)", k);
      localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
        xxe->count);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      delete [] tempString;
#endif
      // use super-scalar "+=*" operation containing all terms
      int xxeIndex = xxe->count;  // need this beyond the increment
      localrc = xxe->appendProductSumSuperScalarSrcRRA(predicateBitField,
        valueTK, valueTK, factorTK, j, srcInfoTable.size(), bufferInfo, 
        vectorFlag, true);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      XXE::ProductSumSuperScalarSrcRRAInfo *xxeProductSumSuperScalarSrcRRAInfo =
        (XXE::ProductSumSuperScalarSrcRRAInfo *)&(xxe->stream[xxeIndex]);
      int *rraOffsetList = xxeProductSumSuperScalarSrcRRAInfo->rraOffsetList;
      void **factorList = xxeProductSumSuperScalarSrcRRAInfo->factorList;
      int *elementOffsetList =
        xxeProductSumSuperScalarSrcRRAInfo->elementOffsetList;
      // fill in rraOffsetList, factorList, elementOffsetList
      int bufferItem = 0; // reset
      int kk = 0; // reset
      pp = srcInfoTable.begin();  // reset
      while (pp != srcInfoTable.end()){
        SeqIndex partnerSeqIndex = pp->partnerSeqIndex;
        for (int term=0; term<srcTermProcessing; term++){
          rraOffsetList[kk] = pp->linIndex/vectorLength;
          factorList[kk] = (void *)(pp->factor);
          elementOffsetList[kk] = bufferItem;
          ++pp;
          ++kk;
          if ((pp == srcInfoTable.end()) ||
            !(partnerSeqIndex == pp->partnerSeqIndex)) break;
        } // for srcTermProcessing
        ++bufferItem;
      }
#ifdef ASMMPROFILE
      tempString = new char[160];
      sprintf(tempString, "use productSumSuperScalarSrcRRA for termCount=%d"
        " -> reducing it to %d terms", srcInfoTable.size(), bufferItem);
      localrc = xxe->appendProfileMessage(predicateBitField, tempString);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      sprintf(tempString, "productSumSuperScalarSrcRRA: vectorLength=%d",
        vectorLength);
      localrc = xxe->appendProfileMessage(predicateBitField, tempString);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      sprintf(tempString, "/PSSSSrcRRA (%d/)", k);
      localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
        xxe->count);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      delete [] tempString;
#endif
      // sendrecv out of contiguous intermediate buffer
      sendnbIndex = xxe->count;  // store index for the associated wait
      pRecv->recvnbIndex = xxe->count;  // store index for the associated wait
      localrc = xxe->appendSendRecv(predicateBitField, bufferInfo,
        pRecv->bufferInfo, bufferItemCount * dataSizeSrc, 
        dstBufferItemCount * dataSizeSrc, pRecv->srcPet, dstPet, tag, tag, 
        vectorFlag, true, true);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    }
#ifdef ASMMPROFILE
    char *tempString = new char[160];
    sprintf(tempString, "<(%04d/%04d)-Snb(%d/%d)-(%04d/%04d)> ",
      srcDe, localPet, k, sendnbIndex, dstDe, dstPet);
    localrc = xxe->appendProfileMessage(predicateBitField, tempString);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    sprintf(tempString, "/Send (%d/)", k);
    localrc = xxe->appendWtimer(predicateBitField, tempString, xxe->count,
      xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    delete [] tempString;
#endif
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }

  // more efficient allocation scheme for many little pieces of memory
  class MemHelper{
    class MemHelper *next;
    void *memAlloc;
    char *memPtr;
    int rest;
  public:
    MemHelper(){
      next = NULL;
      memAlloc = NULL;
      memPtr = NULL;
      rest = 0;
    }
    ~MemHelper(){
      if (memAlloc)
        std::free(memAlloc);
      if (next)
        delete next;
    }
    void *malloc(int bytes){
      const int limit=1024;     // 1kByte
      const int block=1048576;  // 1MByte
      if (bytes > limit){
        MemHelper *mh = this; // initialize to first element in list
        while (mh->next) mh = mh->next; // find last element in list
        mh->next = new MemHelper();
        mh = mh->next;
        mh->memAlloc = std::malloc(bytes);
        mh->memPtr = (char *)(mh->memAlloc);
        return mh->memAlloc;
      }
      if (memPtr==NULL){
        memAlloc = std::malloc(block);
        memPtr = (char *)memAlloc;
        rest = block;
      }
      if (rest >= bytes){
        rest -= bytes;
        void *mem = (void *)memPtr;
        memPtr += bytes;
        return mem;
      }else if(next)
        return next->malloc(bytes);
      else{
        next = new MemHelper();
        return next->malloc(bytes);
      }
    }
  };
  
} // ArrayHelper


namespace DD{
  struct Interval{
    int min;
    int max;
    int count;
  };
  
  struct FactorElement{
    SeqIndex partnerSeqIndex;
    vector<int> partnerDe;
    int padding;    // padding for 8-byte alignment
    char factor[8]; // large enough for R8 and I8
  };
  bool operator==(FactorElement a, FactorElement b){
    return (a.partnerSeqIndex == b.partnerSeqIndex);
  }
  bool operator!=(FactorElement a, FactorElement b){
    return (a.partnerSeqIndex != b.partnerSeqIndex);
  }
  bool operator<(FactorElement a, FactorElement b){
    return (a.partnerSeqIndex < b.partnerSeqIndex);
  }

  struct SeqIndexFactorLookup{
    vector<int> de;
    int factorCount;
    vector<FactorElement> factorList;
  public:
    SeqIndexFactorLookup(){
      factorCount = 0;
    }
  };
  
  // the following structs could be moved out of this namespace
  struct AssociationElement{
    int linIndex;
    SeqIndex seqIndex;
    int factorCount;
    vector<FactorElement> factorList;
  };
  struct FillLinSeqVectInfo{
    Array const *array;
    vector<AssociationElement> *linSeqVect;
    int localPet;
    int localDeCount;
    const int *localDeElementCount;
    const Interval *seqIndexInterval;
    vector<SeqIndexFactorLookup> &seqIndexFactorLookup;
    bool tensorMixFlag;
    bool haloRimFlag;   // true indicates that halo rim instead of excl reg used
  public:
    FillLinSeqVectInfo(
      vector<SeqIndexFactorLookup> &seqIndexFactorLookup_
    ):
      // members that need to be set on this level because of reference
      seqIndexFactorLookup(seqIndexFactorLookup_)
    {}
  };
  struct FillPartnerDeInfo{
    int localPet;
    const Interval *seqIndexIntervalIn;
    const Interval *seqIndexIntervalOut;
    vector<SeqIndexFactorLookup> &seqIndexFactorLookupIn;
    vector<SeqIndexFactorLookup> &seqIndexFactorLookupOut;
    bool tensorMixFlag;
  public:
    FillPartnerDeInfo(
      vector<SeqIndexFactorLookup> &seqIndexFactorLookupIn_,
      vector<SeqIndexFactorLookup> &seqIndexFactorLookupOut_
    ):
      // members that need to be set on this level because of reference
      seqIndexFactorLookupIn(seqIndexFactorLookupIn_),
      seqIndexFactorLookupOut(seqIndexFactorLookupOut_)
    {}
  };

// -- accessLookup()

template<typename T>
int requestSizeFactor(T *t);

template<typename T>
void clientRequest(T *t, int i, char **requestStreamClient);

template<typename T>
void localClientServerExchange(T *t);

template<typename T>
int serverResponseSize(T *t, int count, int i, char **requestStreamServer);

template<typename T>
void serverResponse(T *t, int count, int i, char **requestStreamServer,
  char **responseStreamServer);

template<typename T>
void clientProcess(T *t, char *responseStream, int responseStreamSize);

template<typename T>
void accessLookup(
  ESMCI::VM *vm,
  int petCount,
  int localPet,
  int *localIntervalPerPetCount,
  int *localElementsPerIntervalCount,
  T *t
  ){
  // access look up table
  VMK::commhandle **send1commhList = new VMK::commhandle*[petCount];
  VMK::commhandle **send2commhList = new VMK::commhandle*[petCount];
  VMK::commhandle **send3commhList = new VMK::commhandle*[petCount];
  VMK::commhandle **recv1commhList = new VMK::commhandle*[petCount];
  VMK::commhandle **recv2commhList = new VMK::commhandle*[petCount];
  VMK::commhandle **recv3commhList = new VMK::commhandle*[petCount];
  char **requestStreamClient = new char*[petCount];
  char **requestStreamServer = new char*[petCount];
  int *responseStreamSizeClient = new int[petCount];
  int *responseStreamSizeServer = new int[petCount];
  char **responseStreamClient = new char*[petCount];
  char **responseStreamServer = new char*[petCount];
  
//char msg[1024];
  
  // t-specific routine
  int requestFactor = requestSizeFactor(t);
  // localPet acts as server, posts non-blocking recvs for all client requests
//ESMC_LogDefault.Write("accessLookup: step 1", ESMC_LOGMSG_INFO); 
  for (int ii=localPet+1; ii<localPet+petCount; ii++){
    // localPet-dependent shifted loop reduces communication contention
    int i = ii%petCount;  // fold back into [0,..,petCount-1] range
    // receive request from Pet "i"
    int count = localIntervalPerPetCount[i];
    if (count>0){
//sprintf(msg, "posting nb-recv for message from PET %d size=%d", i, requestFactor*count);
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO); 
      requestStreamServer[i] = new char[requestFactor*count];
      recv3commhList[i] = NULL;
      vm->recv(requestStreamServer[i], requestFactor*count, i,
        &(recv3commhList[i]));
    }
  }
  // localPet acts as a client, sends its requests to the appropriate servers
//ESMC_LogDefault.Write("accessLookup: step 2", ESMC_LOGMSG_INFO); 
  for (int ii=localPet+petCount-1; ii>localPet; ii--){
    // localPet-dependent shifted loop reduces communication contention
    int i = ii%petCount;  // fold back into [0,..,petCount-1] range
    if (localElementsPerIntervalCount[i]>0){
      // localPet has elements that are located in interval of server Pet "i"
      requestStreamClient[i] =
        new char[requestFactor*localElementsPerIntervalCount[i]];
      // t-specific client routine
      clientRequest(t, i, requestStreamClient);
      // send information to the serving Pet
      send1commhList[i] = NULL;
//sprintf(msg, "posting nb-send to PET %d size=%d", i, requestFactor*localElementsPerIntervalCount[i]);
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO); 
#ifdef WORKAROUND_NONBLOCKPROGRESSBUG
      vm->send(requestStreamClient[i],
        requestFactor*localElementsPerIntervalCount[i], i); 
#else
      vm->send(requestStreamClient[i],
        requestFactor*localElementsPerIntervalCount[i], i, 
        &(send1commhList[i]));
#endif
      // post receive to obtain response size from server Pet
      recv1commhList[i] = NULL;
//sprintf(msg, "posting nb-recv for message from PET %d size=%d", i, sizeof(int));
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO); 

      vm->recv(&(responseStreamSizeClient[i]), sizeof(int), i,
        &(recv1commhList[i]));
    }
  }
  // localPet locally acts as server and client to fill its own request
  // t-specific client-server routine
//ESMC_LogDefault.Write("accessLookup: step 3", ESMC_LOGMSG_INFO); 
  localClientServerExchange(t);
  // localPet acts as server, processing requests from clients, send response sz
//ESMC_LogDefault.Write("accessLookup: step 4", ESMC_LOGMSG_INFO); 
  for (int ii=localPet+1; ii<localPet+petCount; ii++){
    // localPet-dependent shifted loop reduces communication contention
    int i = ii%petCount;  // fold back into [0,..,petCount-1] range
    int count = localIntervalPerPetCount[i];
    if (count>0){
      // wait for request from Pet "i"
//sprintf(msg, "waiting on nb-recv for message from PET %d", i);
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO); 
      vm->commwait(&(recv3commhList[i]));
      // t-specific server routine
      int responseStreamSize =
        serverResponseSize(t, count, i, requestStreamServer);
      // send response size to client Pet "i"
      responseStreamSizeServer[i] = responseStreamSize;
      send2commhList[i] = NULL;
//sprintf(msg, "posting nb-send to PET %d size=%d", i, sizeof(int));
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO); 

#ifdef WORKAROUND_NONBLOCKPROGRESSBUG
      vm->send(&(responseStreamSizeServer[i]), sizeof(int), i);
#else
      vm->send(&(responseStreamSizeServer[i]), sizeof(int), i,
        &(send2commhList[i]));
#endif
    }
  }
  // localPet acts as a client, waits for response size from server and posts
  // receive for response stream
//ESMC_LogDefault.Write("accessLookup: step 5", ESMC_LOGMSG_INFO); 
  for (int ii=localPet+petCount-1; ii>localPet; ii--){
    // localPet-dependent shifted loop reduces communication contention
    int i = ii%petCount;  // fold back into [0,..,petCount-1] range
    if (localElementsPerIntervalCount[i]>0){
      // localPet has elements that are located in interval of server Pet i
      // wait to receive response size from the serving Pet "i"
      vm->commwait(&(recv1commhList[i]));
      int responseStreamSize = responseStreamSizeClient[i];
      if (responseStreamSize>0){
        responseStreamClient[i] = new char[responseStreamSize];
        recv2commhList[i] = NULL;
//sprintf(msg, "posting nb-recv for message from PET %d size=%d", i, responseStreamSize);
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO); 

        vm->recv(responseStreamClient[i], responseStreamSize, i,
          &(recv2commhList[i]));
      }
    }
  }
  // localPet acts as server, send response stream
//ESMC_LogDefault.Write("accessLookup: step 6", ESMC_LOGMSG_INFO); 
  for (int ii=localPet+1; ii<localPet+petCount; ii++){
    // localPet-dependent shifted loop reduces communication contention
    int i = ii%petCount;  // fold back into [0,..,petCount-1] range
    int count = localIntervalPerPetCount[i];
    if (count>0){
      int responseStreamSize = responseStreamSizeServer[i];
      if (responseStreamSize>0){
        // construct response stream
        responseStreamServer[i] = new char[responseStreamSize];
        // t-specific server routine
        serverResponse(t, count, i, requestStreamServer, responseStreamServer);
        // send response stream to client Pet "i"
        send3commhList[i] = NULL;
//sprintf(msg, "posting nb-send to PET %d size=%d", i, responseStreamSize);
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO); 

#ifdef WORKAROUND_NONBLOCKPROGRESSBUG
        vm->send(responseStreamServer[i], responseStreamSize, i);
#else
        vm->send(responseStreamServer[i], responseStreamSize, i,
          &(send3commhList[i]));
#endif
      }
      // garbage collection
      delete [] requestStreamServer[i];
    }
  }
  // localPet acts as a client, waits for response stream from server, process
//ESMC_LogDefault.Write("accessLookup: step 7", ESMC_LOGMSG_INFO); 
  for (int ii=localPet+petCount-1; ii>localPet; ii--){
    // localPet-dependent shifted loop reduces communication contention
    int i = ii%petCount;  // fold back into [0,..,petCount-1] range
    if (localElementsPerIntervalCount[i]>0){
      // localPet has elements that are located in interval of server Pet i
      int responseStreamSize = responseStreamSizeClient[i];
      if (responseStreamSize>0){
        // wait to receive response stream from the serving Pet "i"
//sprintf(msg, "commwait for message from PET %d, of size %d, with localElements: %d", i, responseStreamSize, localElementsPerIntervalCount[i]);
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO); 
        vm->commwait(&(recv2commhList[i]));
        // process responseStream and complete t[][] info
        char *responseStream = responseStreamClient[i];
        // t-specific client routine
        clientProcess(t, responseStream, responseStreamSize);
        // garbage collection
        delete [] responseStreamClient[i];
      }
    }
  }
  // localPet acts as a client, wait for sends to complete and collect garbage
//ESMC_LogDefault.Write("accessLookup: step 8", ESMC_LOGMSG_INFO); 
  for (int ii=localPet+petCount-1; ii>localPet; ii--){
    // localPet-dependent shifted loop reduces communication contention
    int i = ii%petCount;  // fold back into [0,..,petCount-1] range
    if (localElementsPerIntervalCount[i]>0){
      // localPet has elements that are located in interval of server Pet i
#ifndef WORKAROUND_NONBLOCKPROGRESSBUG
      // wait for send
      vm->commwait(&(send1commhList[i]));
#endif
      // garbage collection
      delete [] requestStreamClient[i];
    }
  }
  // localPet acts as server, wait for sends to complete and collect garbage
//ESMC_LogDefault.Write("accessLookup: step 9", ESMC_LOGMSG_INFO); 
  for (int ii=localPet+1; ii<localPet+petCount; ii++){
    // localPet-dependent shifted loop reduces communication contention
    int i = ii%petCount;  // fold back into [0,..,petCount-1] range
    int count = localIntervalPerPetCount[i];
    if (count>0){
#ifndef WORKAROUND_NONBLOCKPROGRESSBUG
      vm->commwait(&(send2commhList[i]));
#endif
      if (responseStreamSizeServer[i]>0){
#ifndef WORKAROUND_NONBLOCKPROGRESSBUG
        vm->commwait(&(send3commhList[i]));
#endif
        // garbage collection
        delete [] responseStreamServer[i];
      }
    }
  }  
  // garbage collection
  delete [] requestStreamClient;
  delete [] requestStreamServer;
  delete [] responseStreamClient;
  delete [] responseStreamServer;
  delete [] responseStreamSizeClient;
  delete [] responseStreamSizeServer;
  delete [] send1commhList;
  delete [] send2commhList;
  delete [] send3commhList;
  delete [] recv1commhList;
  delete [] recv2commhList;
  delete [] recv3commhList;
//ESMC_LogDefault.Write("accessLookup: step 10", ESMC_LOGMSG_INFO); 
}

// --------------------------------------------------
// FillLinSeqVectInfo-specific accessLookup routines:

int requestSizeFactor(FillLinSeqVectInfo *fillLinSeqVectInfo){
  return 5*sizeof(int); // lookupIndex, j, decompSeqIndex, tensorSeqIndex,
                        // linIndex
}

void clientRequest(FillLinSeqVectInfo *fillLinSeqVectInfo, int dstPet,
  char **requestStreamClient){
  const int localDeCount = fillLinSeqVectInfo->localDeCount;
  const int *localDeElementCount = fillLinSeqVectInfo->localDeElementCount;
  const Interval *seqIndexInterval = fillLinSeqVectInfo->seqIndexInterval;
  const bool tensorMixFlag = fillLinSeqVectInfo->tensorMixFlag;
  // fill the requestStreamClient[dstPet] element
  int *requestStreamClientInt = (int *)requestStreamClient[dstPet];
  int seqIndMin = seqIndexInterval[dstPet].min;
  int seqIndMax = seqIndexInterval[dstPet].max;
  int seqIndCount = seqIndexInterval[dstPet].count;
  int jj = 0; // reset
  for (int j=0; j<localDeCount; j++){
    if (fillLinSeqVectInfo->haloRimFlag){
      // loop over the halo rim elements for localDe j
      for (int k=0; k<fillLinSeqVectInfo->array->getRimElementCount()[j]; k++){
        SeqIndex seqIndex = fillLinSeqVectInfo->array->getRimSeqIndex()[j][k];
        if (seqIndex.valid()){
          int seqInd = seqIndex.decompSeqIndex;
          if (seqInd >= seqIndMin && seqInd <= seqIndMax){
            int lookupIndex = seqInd - seqIndMin;
            if (tensorMixFlag)
              lookupIndex += (seqIndex.tensorSeqIndex - 1) * seqIndCount;
            requestStreamClientInt[5*jj]   = lookupIndex;
            requestStreamClientInt[5*jj+1] = j;
            requestStreamClientInt[5*jj+2] = seqIndex.decompSeqIndex;
            requestStreamClientInt[5*jj+3] = seqIndex.tensorSeqIndex;
            requestStreamClientInt[5*jj+4] =
              fillLinSeqVectInfo->array->getRimLinIndex()[j][k];
            ++jj; // increment counter
          }
        }
      }
    }else{
      // loop over all elements in the exclusive region for localDe j
      ArrayElement arrayElement(fillLinSeqVectInfo->array, j);
      while(arrayElement.isWithin()){
        SeqIndex seqIndex = arrayElement.getSequenceIndexExclusive();
        int seqInd = seqIndex.decompSeqIndex;
        if (seqInd >= seqIndMin && seqInd <= seqIndMax){
          int lookupIndex = seqInd - seqIndMin;
          if (tensorMixFlag)
            lookupIndex += (seqIndex.tensorSeqIndex - 1) * seqIndCount;
          requestStreamClientInt[5*jj]   = lookupIndex;
          requestStreamClientInt[5*jj+1] = j;
          requestStreamClientInt[5*jj+2] = seqIndex.decompSeqIndex;
          requestStreamClientInt[5*jj+3] = seqIndex.tensorSeqIndex;
          requestStreamClientInt[5*jj+4] =
            arrayElement.getLinearIndexExclusive();
          ++jj; // increment counter
        }
        arrayElement.next();
      } // end while over all exclusive elements
    }
  }
}

void localClientServerExchange(FillLinSeqVectInfo *fillLinSeqVectInfo){
  const int localPet = fillLinSeqVectInfo->localPet;
  const int localDeCount = fillLinSeqVectInfo->localDeCount;
  const int *localDeElementCount = fillLinSeqVectInfo->localDeElementCount;
  vector<AssociationElement> *linSeqVect = fillLinSeqVectInfo->linSeqVect;
  const Interval *seqIndexInterval = fillLinSeqVectInfo->seqIndexInterval;
  const bool tensorMixFlag = fillLinSeqVectInfo->tensorMixFlag;
  vector<SeqIndexFactorLookup> const &seqIndexFactorLookup =
    fillLinSeqVectInfo->seqIndexFactorLookup;
  // localPet locally acts as server and client
  int seqIndMin = seqIndexInterval[localPet].min;
  int seqIndMax = seqIndexInterval[localPet].max;
  int seqIndCount = seqIndexInterval[localPet].count;
  for (int j=0; j<localDeCount; j++){
    if (fillLinSeqVectInfo->haloRimFlag){
      // loop over the halo rim elements for localDe j
      for (int k=0; k<fillLinSeqVectInfo->array->getRimElementCount()[j]; k++){
        SeqIndex seqIndex = fillLinSeqVectInfo->array->getRimSeqIndex()[j][k];
        if (seqIndex.valid()){
          int seqInd = seqIndex.decompSeqIndex;
          if (seqInd >= seqIndMin && seqInd <= seqIndMax){
            int lookupIndex = seqInd - seqIndMin;
            if (tensorMixFlag)
              lookupIndex += (seqIndex.tensorSeqIndex - 1) * seqIndCount;
            int factorCount = seqIndexFactorLookup[lookupIndex].factorCount;
            if (factorCount > 0){
              AssociationElement element;
              element.factorCount = factorCount;
              element.factorList =
                seqIndexFactorLookup[lookupIndex].factorList;
              element.linIndex =
                fillLinSeqVectInfo->array->getRimLinIndex()[j][k];
              element.seqIndex = seqIndex;
              linSeqVect[j].push_back(element);
            }
          }
        }
      }
    }else{
      // loop over all elements in the exclusive region for localDe j
      ArrayElement arrayElement(fillLinSeqVectInfo->array, j);
      while(arrayElement.isWithin()){
        SeqIndex seqIndex = arrayElement.getSequenceIndexExclusive();
        int seqInd = seqIndex.decompSeqIndex;
        if (seqInd >= seqIndMin && seqInd <= seqIndMax){
          int lookupIndex = seqInd - seqIndMin;
          if (tensorMixFlag)
            lookupIndex += (seqIndex.tensorSeqIndex - 1) * seqIndCount;
          int factorCount = seqIndexFactorLookup[lookupIndex].factorCount;
          if (factorCount > 0){
            AssociationElement element;
            element.factorCount = factorCount;
            element.factorList = seqIndexFactorLookup[lookupIndex].factorList;
            element.linIndex = arrayElement.getLinearIndexExclusive();
            element.seqIndex = seqIndex;
            linSeqVect[j].push_back(element);
         }
        }
        arrayElement.next();
      } // end while over all exclusive elements
    }
  }
}

int serverResponseSize(FillLinSeqVectInfo *fillLinSeqVectInfo, int count,
  int srcPet, char **requestStreamServer){
  vector<SeqIndexFactorLookup> const &seqIndexFactorLookup =
    fillLinSeqVectInfo->seqIndexFactorLookup;
  // process requestStreamServer[srcPet] and return response stream size
  int indexCounter = 0; // reset
  int factorElementCounter = 0; // reset
  int partnerDeCounter = 0; // reset
  int *requestStreamServerInt = (int *)requestStreamServer[srcPet];
  for (int j=0; j<count; j++){
    int lookupIndex = requestStreamServerInt[5*j];
    int factorCount = seqIndexFactorLookup[lookupIndex].factorCount;
    if (factorCount > 0){
      ++indexCounter;
      factorElementCounter += factorCount;
      for (int jj=0; jj<factorCount; jj++)
        partnerDeCounter +=
          seqIndexFactorLookup[lookupIndex].factorList[jj].partnerDe.size();
    }
  }
  int responseStreamSize = 
    5*indexCounter*sizeof(int) +            // AssociationElement members
    3*factorElementCounter*sizeof(int) +    // FactorElement members
    partnerDeCounter*sizeof(int) +          // partnerDe elements
    8*factorElementCounter;                 // FactorElement factor
  return responseStreamSize;
}
      
void serverResponse(FillLinSeqVectInfo *fillLinSeqVectInfo, int count,
  int srcPet, char **requestStreamServer, char **responseStreamServer){
  vector<SeqIndexFactorLookup> const &seqIndexFactorLookup =
    fillLinSeqVectInfo->seqIndexFactorLookup;
  // construct response stream
  int *responseStreamInt = (int *)responseStreamServer[srcPet];
  int *requestStreamServerInt = (int *)requestStreamServer[srcPet];
  for (int jj=0; jj<count; jj++){
    int lookupIndex = requestStreamServerInt[5*jj];
    int factorCount = seqIndexFactorLookup[lookupIndex].factorCount;
    if (factorCount > 0){
      *responseStreamInt++ = requestStreamServerInt[5*jj+1];  // j
      *responseStreamInt++ = requestStreamServerInt[5*jj+2];  // decompSeqIndex
      *responseStreamInt++ = requestStreamServerInt[5*jj+3];  // tensorSeqIndex
      *responseStreamInt++ = requestStreamServerInt[5*jj+4];  // linIndex
      *responseStreamInt++ = factorCount;
      for (int k=0; k<factorCount; k++){
        *responseStreamInt++ = seqIndexFactorLookup[lookupIndex].factorList[k]
          .partnerSeqIndex.decompSeqIndex;
        *responseStreamInt++ = seqIndexFactorLookup[lookupIndex].factorList[k]
          .partnerSeqIndex.tensorSeqIndex;
        int size = seqIndexFactorLookup[lookupIndex]
          .factorList[k].partnerDe.size();
        *responseStreamInt++ = size;
        for (int kk=0; kk<size; kk++)
          *responseStreamInt++ = seqIndexFactorLookup[lookupIndex]
            .factorList[k].partnerDe[kk];
        char *responseStreamChar = (char *)responseStreamInt;
        for (int kk=0; kk<8; kk++)
          *responseStreamChar++ =
            seqIndexFactorLookup[lookupIndex].factorList[k].factor[kk];
        responseStreamInt = (int *)responseStreamChar;
      }
    }
  }
}        
        
void clientProcess(FillLinSeqVectInfo *fillLinSeqVectInfo,
  char *responseStream, int responseStreamSize){
  vector<AssociationElement> *linSeqVect = fillLinSeqVectInfo->linSeqVect;
  // process responseStream and fill linSeqVect[][]
  int *responseStreamInt = (int *)responseStream;
  while ((char *)responseStreamInt != responseStream+responseStreamSize){
    int j = *responseStreamInt++;
    int decompSeqIndex = *responseStreamInt++;
    int tensorSeqIndex = *responseStreamInt++;
    int linIndex = *responseStreamInt++;
    int factorCount = *responseStreamInt++;
    AssociationElement element;
    element.factorCount = factorCount;
    element.factorList.resize(factorCount);
    for (int jj=0; jj<factorCount; jj++){
      element.factorList[jj].partnerSeqIndex.decompSeqIndex =
        *responseStreamInt++;
      element.factorList[jj].partnerSeqIndex.tensorSeqIndex =
        *responseStreamInt++;
      int size = *responseStreamInt++;
      element.factorList[jj].partnerDe.resize(size);
      for (int kk=0; kk<size; kk++)
        element.factorList[jj].partnerDe[kk] = *responseStreamInt++;
      char *responseStreamChar = (char *)responseStreamInt;
      for (int kk=0; kk<8; kk++)
        element.factorList[jj].factor[kk] = *responseStreamChar++;
      responseStreamInt = (int *)responseStreamChar;
    }
    element.linIndex = linIndex;
    element.seqIndex.decompSeqIndex = decompSeqIndex;
    element.seqIndex.tensorSeqIndex = tensorSeqIndex;
    linSeqVect[j].push_back(element);
  }
}
        
// -------------------------------------------------
// FillPartnerDeInfo-specific accessLookup routines:

int requestSizeFactor(FillPartnerDeInfo *fillPartnerDeInfo){
  return 3*sizeof(int); // lookupIndex, localLookupIndex, k
}

void clientRequest(FillPartnerDeInfo *fillPartnerDeInfo, int dstPet,
  char **requestStreamClient){
  const int localPet = fillPartnerDeInfo->localPet;
  const Interval *seqIndexIntervalIn = fillPartnerDeInfo->seqIndexIntervalIn;
  const Interval *seqIndexIntervalOut = fillPartnerDeInfo->seqIndexIntervalOut;
  vector<SeqIndexFactorLookup> &seqIndexFactorLookupOut =
    fillPartnerDeInfo->seqIndexFactorLookupOut;
  const bool tensorMixFlag = fillPartnerDeInfo->tensorMixFlag;
  // fill the requestStreamClient[dstPet] element
  int seqIndMin = seqIndexIntervalIn[dstPet].min;
  int seqIndMax = seqIndexIntervalIn[dstPet].max;
  int seqIndCount = seqIndexIntervalIn[dstPet].count;
  int jj = 0; // reset
  int localLookupIndex = 0; // reset
  for (vector<DD::SeqIndexFactorLookup>::const_iterator
    j=seqIndexFactorLookupOut.begin(); j!=seqIndexFactorLookupOut.end(); ++j){
    for (int k=0; k<j->factorCount; k++){
      int partnerSeqInd = j->factorList[k]
        .partnerSeqIndex.decompSeqIndex;
      if (partnerSeqInd >= seqIndMin && partnerSeqInd <= seqIndMax){
        int lookupIndex = partnerSeqInd - seqIndMin;
        if (tensorMixFlag){
          lookupIndex += (j->factorList[k]
            .partnerSeqIndex.tensorSeqIndex - 1) * seqIndCount;
        }
        int *requestStreamClientInt = (int *)requestStreamClient[dstPet];
        requestStreamClientInt[3*jj] = lookupIndex;
        requestStreamClientInt[3*jj+1] = localLookupIndex;
        requestStreamClientInt[3*jj+2] = k;
        ++jj; // increment counter
      }
    }
    ++localLookupIndex;
  }
}

void localClientServerExchange(FillPartnerDeInfo *fillPartnerDeInfo){
  const int localPet = fillPartnerDeInfo->localPet;
  const Interval *seqIndexIntervalIn = fillPartnerDeInfo->seqIndexIntervalIn;
  const Interval *seqIndexIntervalOut = fillPartnerDeInfo->seqIndexIntervalOut;
  vector<SeqIndexFactorLookup> &seqIndexFactorLookupIn =
    fillPartnerDeInfo->seqIndexFactorLookupIn;
  vector<SeqIndexFactorLookup> &seqIndexFactorLookupOut =
    fillPartnerDeInfo->seqIndexFactorLookupOut;
  const bool tensorMixFlag = fillPartnerDeInfo->tensorMixFlag;
  // localPet locally acts as server and client
  int seqIndMin = seqIndexIntervalIn[localPet].min;
  int seqIndMax = seqIndexIntervalIn[localPet].max;
  int seqIndCount = seqIndexIntervalIn[localPet].count;
  for (vector<DD::SeqIndexFactorLookup>::iterator
    j=seqIndexFactorLookupOut.begin(); j!=seqIndexFactorLookupOut.end(); ++j){
    for (int k=0; k<j->factorCount; k++){
      int partnerSeqInd = j->factorList[k]
        .partnerSeqIndex.decompSeqIndex;
      if (partnerSeqInd >= seqIndMin && partnerSeqInd <= seqIndMax){
        int lookupIndex = partnerSeqInd - seqIndMin;
        if (tensorMixFlag){
          lookupIndex += (j->factorList[k]
            .partnerSeqIndex.tensorSeqIndex - 1) * seqIndCount;
        }
        j->factorList[k].partnerDe.insert(
          j->factorList[k].partnerDe.end(),
          seqIndexFactorLookupIn[lookupIndex].de.begin(),
          seqIndexFactorLookupIn[lookupIndex].de.end());
      }
    }
  }
}

int serverResponseSize(FillPartnerDeInfo *fillPartnerDeInfo, int count,
  int srcPet, char **requestStreamServer){
  vector<SeqIndexFactorLookup> &seqIndexFactorLookupIn =
    fillPartnerDeInfo->seqIndexFactorLookupIn;
  int *requestStreamServerInt = (int *)requestStreamServer[srcPet];
  int responseCount = 0;  // reset
  for (int i=0; i<count; i++){
    int lookupIndex = requestStreamServerInt[3*i];
    responseCount += seqIndexFactorLookupIn[lookupIndex].de.size();
    responseCount += 3; // localLookupIndex, k, size
  }
  int responseStreamSize = responseCount * sizeof(int);
  return responseStreamSize;
}
      
void serverResponse(FillPartnerDeInfo *fillPartnerDeInfo, int count,
  int srcPet, char **requestStreamServer, char **responseStreamServer){
  vector<SeqIndexFactorLookup> &seqIndexFactorLookupIn =
    fillPartnerDeInfo->seqIndexFactorLookupIn;
  // construct response stream
  int *responseStreamInt = (int *)responseStreamServer[srcPet];
  int *requestStreamServerInt = (int *)requestStreamServer[srcPet];
  for (int i=0; i<count; i++){
    int lookupIndex = requestStreamServerInt[3*i];
    *responseStreamInt++ = requestStreamServerInt[3*i+1];   // localLookupIndex
    *responseStreamInt++ = requestStreamServerInt[3*i+2];   // k
    int size = seqIndexFactorLookupIn[lookupIndex].de.size();
    *responseStreamInt++ = size;                            // size
    for (int j=0; j<size; j++)
      *responseStreamInt++ = seqIndexFactorLookupIn[lookupIndex].de[j]; // de
  }
}        
        
void clientProcess(FillPartnerDeInfo *fillPartnerDeInfo,
  char *responseStream, int responseStreamSize){
  vector<SeqIndexFactorLookup> &seqIndexFactorLookupOut =
    fillPartnerDeInfo->seqIndexFactorLookupOut;
  // process responseStream and complete seqIndexFactorLookupOut info
  int *responseStreamInt = (int *)responseStream;
  while ((char *)responseStreamInt != responseStream+responseStreamSize){
    int localLookupIndex = *responseStreamInt++;
    int k = *responseStreamInt++;
    int size = *responseStreamInt++;
    for (int j=0; j<size; j++){
      int de = *responseStreamInt++;
      seqIndexFactorLookupOut[localLookupIndex].factorList[k].partnerDe
        .push_back(de);
    }
  }
}

  // -------------------------------------------------
  // specialize ComPat class for total exchange during DE info fill.
  class FillSelfDeInfo:public ComPat{
    Array const *array;
    int localPet;
    int localDeCount;
    int const *localDeElementCount;
    int const *localDeToDeMap;
    Interval const *seqIndexInterval;
    vector<SeqIndexFactorLookup> &seqIndexFactorLookup;
    bool tensorMixFlag;
    int const *localIntervalPerPetCount;
    int const *localElementsPerIntervalCount;
    bool haloRimFlag;
   public:
    FillSelfDeInfo(
      Array const *array_,
      int localPet_,
      int localDeCount_,
      int const *localDeElementCount_,
      int const *localDeToDeMap_,
      Interval const *seqIndexInterval_,
      vector<SeqIndexFactorLookup> &seqIndexFactorLookup_,
      bool tensorMixFlag_,
      int const *localIntervalPerPetCount_,
      int const *localElementsPerIntervalCount_,
      bool haloRimFlag_
    ):
      // members that need to be set on this level because of reference
      seqIndexFactorLookup(seqIndexFactorLookup_)
    {
      array = array_;
      localPet = localPet_;
      localDeCount = localDeCount_;
      localDeElementCount = localDeElementCount_;
      localDeToDeMap = localDeToDeMap_;
      seqIndexInterval = seqIndexInterval_;
      tensorMixFlag = tensorMixFlag_;
      localIntervalPerPetCount = localIntervalPerPetCount_;
      localElementsPerIntervalCount = localElementsPerIntervalCount_;
      haloRimFlag = haloRimFlag_;
    }
   private:
    int messageSizeCount(int srcPet, int dstPet)const{
      if (localPet == srcPet)
        return localElementsPerIntervalCount[dstPet];
      else if (localPet == dstPet)
        return localIntervalPerPetCount[srcPet];
      else{
        return 0; // provoke MPI errors
      }
    }
    virtual int messageSize(int srcPet, int dstPet)const{
      return 2 * sizeof(int) * messageSizeCount(srcPet, dstPet);
    }
    virtual void messagePrepare(int srcPet, int dstPet, char *buffer)const{
      int seqIndMin = seqIndexInterval[dstPet].min;
      int seqIndMax = seqIndexInterval[dstPet].max;
      int seqIndCount = seqIndexInterval[dstPet].count;
      int *bufferInt = (int *)buffer;
      int jj = 0; // reset
      for (int j=0; j<localDeCount; j++){
        int de = localDeToDeMap[j];  // global DE number
        if (haloRimFlag){
          // loop over the halo rim elements for localDe j
          for (int k=0; k<array->getRimElementCount()[j]; k++){
            SeqIndex seqIndex = array->getRimSeqIndex()[j][k];
            if (seqIndex.valid()){
              int seqInd = seqIndex.decompSeqIndex;
              if (seqInd >= seqIndMin && seqInd <= seqIndMax){
                int lookupIndex = seqInd - seqIndMin;
                if (tensorMixFlag)
                  lookupIndex += (seqIndex.tensorSeqIndex - 1) * seqIndCount;
                bufferInt[2*jj] = lookupIndex;
                bufferInt[2*jj+1] = de;
                ++jj; // increment counter
              }
            }
          }
        }else{
          // loop over all elements in the exclusive region for localDe j
          ArrayElement arrayElement(array, j);
          while(arrayElement.isWithin()){
            SeqIndex seqIndex = arrayElement.getSequenceIndexExclusive();
            int seqInd = seqIndex.decompSeqIndex;
            if (seqInd >= seqIndMin && seqInd <= seqIndMax){
              int lookupIndex = seqInd - seqIndMin;
              if (tensorMixFlag)
                lookupIndex += (seqIndex.tensorSeqIndex - 1) * seqIndCount;
              bufferInt[2*jj] = lookupIndex;
              bufferInt[2*jj+1] = de;
              ++jj; // increment counter
            }
            arrayElement.next();
          } // end while over all exclusive elements
        }
      }
    }
    virtual void messageProcess(int srcPet, int dstPet, char *buffer){
      int count = messageSizeCount(srcPet, dstPet);
      int *bufferInt = (int *)buffer;    
      for (int jj=0; jj<count; jj++){
        int lookupIndex = bufferInt[2*jj];
        if (seqIndexFactorLookup[lookupIndex].factorCount > 0){
          // element with factors -> fill in the DE
          seqIndexFactorLookup[lookupIndex].de.push_back(bufferInt[2*jj+1]);
          // this will lead to duplicate de entries for cases with tensor
          // elements but no tensor mixing
          // -> duplicates must be eliminated by the calling code
        }
      }
    }
    virtual void localPrepareAndProcess(int localPet){
      int seqIndMin = seqIndexInterval[localPet].min;
      int seqIndMax = seqIndexInterval[localPet].max;
      int seqIndCount = seqIndexInterval[localPet].count;
      for (int j=0; j<localDeCount; j++){
        int de = localDeToDeMap[j];  // global DE number
        if (haloRimFlag){
          // loop over the halo rim elements for localDe j
          for (int k=0; k<array->getRimElementCount()[j]; k++){
            SeqIndex seqIndex = array->getRimSeqIndex()[j][k];
            if (seqIndex.valid()){
              int seqInd = seqIndex.decompSeqIndex;
              if (seqInd >= seqIndMin && seqInd <= seqIndMax){
                int lookupIndex = seqInd - seqIndMin;
                if (tensorMixFlag)
                  lookupIndex += (seqIndex.tensorSeqIndex - 1) * seqIndCount;
                if (seqIndexFactorLookup[lookupIndex].factorCount > 0){
                  // element with factors -> fill in the DE
                  seqIndexFactorLookup[lookupIndex].de.push_back(de);
                  // this will lead to duplicate de entries for cases with
                  // tensor elements but no tensor mixing
                  // -> duplicates must be eliminated by the calling code
                }
              }
            }
          }
        }else{
          // loop over all elements in the exclusive region for localDe j
          ArrayElement arrayElement(array, j);
          while(arrayElement.isWithin()){
            SeqIndex seqIndex = arrayElement.getSequenceIndexExclusive();
            int seqInd = seqIndex.decompSeqIndex;
            if (seqInd >= seqIndMin && seqInd <= seqIndMax){
              int lookupIndex = seqInd - seqIndMin;
              if (tensorMixFlag)
                lookupIndex += (seqIndex.tensorSeqIndex - 1) * seqIndCount;
              if (seqIndexFactorLookup[lookupIndex].factorCount > 0){
                // element with factors -> fill in the DE
                seqIndexFactorLookup[lookupIndex].de.push_back(de);
                // this will lead to duplicate de entries for cases with tensor
                // elements but no tensor mixing
                // -> duplicates must be eliminated by the calling code
              }
            }
            arrayElement.next();
          } // end while over all exclusive elements
        }
      }
    }
  };
        
  // -------------------------------------------------
  // abstract parent class for DE info fill stage1 and stage2
  class SetupSeqIndexFactorLookup:public ComPat{
   protected:
    vector<SeqIndexFactorLookup> &seqIndexFactorLookup;
    int localPet;
    int petCount;
    SparseMatrix const *sparseMatrix;
    vector<bool> const &factorPetFlag;
    Interval const *seqIndexInterval;
    vector<int> const &seqIntervFactorListCountToPet;
    vector<vector<int> > const &seqIntervFactorListIndexToPet;
    vector<vector<int> > const &seqIntervFactorListLookupIndexToPet;
    vector<int> const &seqIntervFactorListCountFromPet;
    bool tensorMixFlag;
    bool dstSetupFlag;
    ESMC_TypeKind_Flag typekindFactors;
   public:
    SetupSeqIndexFactorLookup(
      vector<SeqIndexFactorLookup> &seqIndexFactorLookup_,
      SparseMatrix const *sparseMatrix_,
      vector<bool> const &factorPetFlag_,
      vector<int> const &seqIntervFactorListCountToPet_,
      vector<vector<int> > const &seqIntervFactorListIndexToPet_,
      vector<vector<int> > const &seqIntervFactorListLookupIndexToPet_,
      vector<int> const &seqIntervFactorListCountFromPet_
    ):
      // members that need to be set on this level because of reference
      seqIndexFactorLookup(seqIndexFactorLookup_),
      sparseMatrix(sparseMatrix_),
      factorPetFlag(factorPetFlag_),
      seqIntervFactorListCountToPet(seqIntervFactorListCountToPet_),
      seqIntervFactorListIndexToPet(seqIntervFactorListIndexToPet_),
      seqIntervFactorListLookupIndexToPet(seqIntervFactorListLookupIndexToPet_),
      seqIntervFactorListCountFromPet(seqIntervFactorListCountFromPet_)
    {}
  };

  // -------------------------------------------------
  // specialize SetupSeqIndexFactorLookup for info fill stage1
  class SetupSeqIndexFactorLookupStage1:public SetupSeqIndexFactorLookup{
   public:
    SetupSeqIndexFactorLookupStage1(
      vector<SeqIndexFactorLookup> &seqIndexFactorLookup_,
      int localPet_,
      int petCount_,
      SparseMatrix const *sparseMatrix_,
      vector<bool> const &factorPetFlag_,
      Interval const *seqIndexInterval_,
      vector<int> const &seqIntervFactorListCountToPet_,
      vector<vector<int> > const &seqIntervFactorListIndexToPet_,
      vector<vector<int> > const &seqIntervFactorListLookupIndexToPet_,
      vector<int> const &seqIntervFactorListCountFromPet_,
      bool tensorMixFlag_,
      bool dstSetupFlag_,
      ESMC_TypeKind_Flag typekindFactors_
    ):SetupSeqIndexFactorLookup(
      seqIndexFactorLookup_,
      sparseMatrix_,
      factorPetFlag_,
      seqIntervFactorListCountToPet_,
      seqIntervFactorListIndexToPet_,
      seqIntervFactorListLookupIndexToPet_,
      seqIntervFactorListCountFromPet_
    ){
      localPet = localPet_;
      petCount = petCount_;
      seqIndexInterval = seqIndexInterval_;
      tensorMixFlag = tensorMixFlag_;
      dstSetupFlag = dstSetupFlag_;
      typekindFactors = typekindFactors_;
    }
   private:
    int messageSizeCount(int srcPet, int dstPet)const{
      if (localPet == srcPet)
        return seqIntervFactorListCountToPet[dstPet];
      else if (localPet == dstPet)
        return seqIntervFactorListCountFromPet[srcPet];
      else
        return 0; // provoke MPI errors
    }
    virtual int messageSize(int srcPet, int dstPet)const{
      return sizeof(int) * messageSizeCount(srcPet, dstPet);
    }
    virtual void messagePrepare(int srcPet, int dstPet, char *buffer)const{
      memcpy(buffer, &(seqIntervFactorListLookupIndexToPet[dstPet].front()),
        messageSizeCount(srcPet, dstPet)*sizeof(int));
    }
    virtual void messageProcess(int srcPet, int dstPet, char *buffer){
      int *bufferInt = (int *)buffer;
      for (int i=0; i<messageSizeCount(srcPet, dstPet); i++){
        // loop over buffer entries
        int lookupIndex = bufferInt[i];
        // count this factor
        ++(seqIndexFactorLookup[lookupIndex].factorCount);
      }
    }
    virtual void localPrepareAndProcess(int localPet){
      for (int i=0; i<messageSizeCount(localPet, localPet); i++){
        // loop over factorList entries in localPet's seqIndex interval
        int lookupIndex = seqIntervFactorListLookupIndexToPet[localPet][i];
        // count this factor
        ++(seqIndexFactorLookup[lookupIndex].factorCount);
      }
    }
    friend class SetupSeqIndexFactorLookupStage2;
  };
  
  // -------------------------------------------------
  // specialize SetupSeqIndexFactorLookup for info fill stage2
  class SetupSeqIndexFactorLookupStage2:public SetupSeqIndexFactorLookup{
   public:
    SetupSeqIndexFactorLookupStage2(SetupSeqIndexFactorLookupStage1 const &s1)
      :SetupSeqIndexFactorLookup(
      s1.seqIndexFactorLookup,
      s1.sparseMatrix,
      s1.factorPetFlag,
      s1.seqIntervFactorListCountToPet,
      s1.seqIntervFactorListIndexToPet,
      s1.seqIntervFactorListLookupIndexToPet,
      s1.seqIntervFactorListCountFromPet
    ){
      localPet = s1.localPet;
      petCount = s1.petCount;
      seqIndexInterval = s1.seqIndexInterval;
      tensorMixFlag = s1.tensorMixFlag;
      dstSetupFlag = s1.dstSetupFlag;
      typekindFactors = s1.typekindFactors;
    }
   private:
    int messageSizeCount(int srcPet, int dstPet)const{
      if (localPet == srcPet)
        return seqIntervFactorListCountToPet[dstPet];
      else if (localPet == dstPet)
        return seqIntervFactorListCountFromPet[srcPet];
      else{
        return 0; // provoke MPI errors
      }
    }
    virtual int messageSize(int srcPet, int dstPet)const{
      int dataSizeFactors = ESMC_TypeKind_FlagSize(typekindFactors);
      //todo: reduce waste of bandwidth in case there are _not_ 4-comp ind.
      return (4*sizeof(int)+dataSizeFactors) * messageSizeCount(srcPet, dstPet);
    }
    virtual void messagePrepare(int srcPet, int dstPet, char *buffer)const{
      if (typekindFactors == ESMC_TYPEKIND_R4)
        fillStream<ESMC_R4>(srcPet, dstPet, buffer);
      else if (typekindFactors == ESMC_TYPEKIND_R8)
        fillStream<ESMC_R8>(srcPet, dstPet, buffer);
      else if (typekindFactors == ESMC_TYPEKIND_I4)
        fillStream<ESMC_I4>(srcPet, dstPet, buffer);
      else if (typekindFactors == ESMC_TYPEKIND_I8)
        fillStream<ESMC_I8>(srcPet, dstPet, buffer);
    }
    virtual void messageProcess(int srcPet, int dstPet, char *buffer){
      if (typekindFactors == ESMC_TYPEKIND_R4)
        fillSeqIndexFactorLookupFromStream<ESMC_R4>(srcPet, dstPet, buffer);
      else if (typekindFactors == ESMC_TYPEKIND_R8)
        fillSeqIndexFactorLookupFromStream<ESMC_R8>(srcPet, dstPet, buffer);
      else if (typekindFactors == ESMC_TYPEKIND_I4)
        fillSeqIndexFactorLookupFromStream<ESMC_I4>(srcPet, dstPet, buffer);
      else if (typekindFactors == ESMC_TYPEKIND_I8)
        fillSeqIndexFactorLookupFromStream<ESMC_I8>(srcPet, dstPet, buffer);
    }
    virtual void localPrepareAndProcess(int localPet){
      if (typekindFactors == ESMC_TYPEKIND_R4)
        fillSeqIndexFactorLookupLocally<ESMC_R4>(localPet);
      else if (typekindFactors == ESMC_TYPEKIND_R8)
        fillSeqIndexFactorLookupLocally<ESMC_R8>(localPet);
      else if (typekindFactors == ESMC_TYPEKIND_I4)
        fillSeqIndexFactorLookupLocally<ESMC_I4>(localPet);
      else if (typekindFactors == ESMC_TYPEKIND_I8)
        fillSeqIndexFactorLookupLocally<ESMC_I8>(localPet);
    }
    template<typename T> void fillStream(int srcPet, int dstPet,
      char *stream)const{
      int *intStream;
      T *factorStream = (T *)stream;
      for (int i=0; i<messageSizeCount(srcPet, dstPet); i++){
        // loop over factorList entries in dstPet's seqIndex interval
        intStream = (int *)factorStream;
        int lookupIndex = seqIntervFactorListLookupIndexToPet[dstPet][i];
        int j = seqIntervFactorListIndexToPet[dstPet][i];
        SeqInd seqInd;
        if (dstSetupFlag)
          seqInd = sparseMatrix->getSrcSeqIndex(j);
          // reverse to lookupIndex construction b/c src side picks dst and rev.
        else
          seqInd = sparseMatrix->getDstSeqIndex(j);
          // reverse to lookupIndex construction b/c src side picks dst and rev.
        int seqIndex = seqInd.getIndex(0);
        int tensorSeqIndex = -1;  // dummy tensorSeqIndex
        if (tensorMixFlag)
          tensorSeqIndex = seqInd.getIndex(1);  // set actual tensor seqIndex
        *intStream++ = lookupIndex;     // index into distr. dir lookup table
        *intStream++ = seqIndex;        // seqIndex
        *intStream++ = tensorSeqIndex;  // dummy tensorSeqIndex
        *intStream++ = 0;               // padding for 8-byte alignment
        factorStream = (T *)intStream;
        *factorStream++ = ((T *)sparseMatrix->getFactorList())[j];  // factor
      }
    }
    template<typename T> void fillSeqIndexFactorLookupFromStream(int srcPet,
      int dstPet, char *stream){
      int *intStream;
      T *factorStream = (T *)stream;
      for (int i=0; i<messageSizeCount(srcPet, dstPet); i++){
        intStream = (int *)factorStream;
        int lookupIndex = *intStream++;   // index into lookup table
        FactorElement factorElement;
        factorElement.partnerSeqIndex.decompSeqIndex = *intStream++;
        factorElement.partnerSeqIndex.tensorSeqIndex = *intStream++;
        intStream++;  // skip padding
        factorStream = (T *)intStream;
        *((T *)factorElement.factor) = *factorStream++;
        seqIndexFactorLookup[lookupIndex].factorList.push_back(factorElement);
      }
    }
    template<typename T> void fillSeqIndexFactorLookupLocally(int localPet){
      for (int i=0; i<messageSizeCount(localPet, localPet); i++){
        // loop over factorList entries in localPet's seqIndex interval
        int j = seqIntervFactorListIndexToPet[localPet][i];
        SeqInd seqInd;
        if (dstSetupFlag)
          seqInd = sparseMatrix->getSrcSeqIndex(j);
          // reverse b/c src side picks dst and rev.
        else
          seqInd = sparseMatrix->getDstSeqIndex(j);
          // reverse b/c src side picks dst and rev.
        int seqIndex = seqInd.getIndex(0);
        int tensorSeqIndex = -1;  // dummy tensorSeqIndex
        if (tensorMixFlag)
          tensorSeqIndex = seqInd.getIndex(1);  // set actual tensor seqIndex
        int lookupIndex = seqIntervFactorListLookupIndexToPet[localPet][i];
        FactorElement factorElement;
        factorElement.partnerSeqIndex.decompSeqIndex = seqIndex;
        factorElement.partnerSeqIndex.tensorSeqIndex = tensorSeqIndex;
        *((T *)factorElement.factor) = ((T *)sparseMatrix->getFactorList())[j];
        seqIndexFactorLookup[lookupIndex].factorList.push_back(factorElement);
      }
    }
  };
  
  // -------------------------------------------------
  void sum(char *a, char *b, ESMC_TypeKind_Flag tk){
    if (tk == ESMC_TYPEKIND_R4)
      *((ESMC_R4 *)a) += *((ESMC_R4 *)b);
    else if (tk == ESMC_TYPEKIND_R8)
      *((ESMC_R8 *)a) += *((ESMC_R8 *)b);
    else if (tk == ESMC_TYPEKIND_I4)
      *((ESMC_I4 *)a) += *((ESMC_I4 *)b);
    else if (tk == ESMC_TYPEKIND_I8)
      *((ESMC_I8 *)a) += *((ESMC_I8 *)b);
  }

  // -------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::DD::setupSeqIndexFactorLookup()"
  int setupSeqIndexFactorLookup(VM *vm,
    vector<SeqIndexFactorLookup> &seqIndexFactorLookup,
    const int petCount, const int localPet, const int factorListCount,
    const bool dstSetupFlag, vector<SparseMatrix> const &sparseMatrix,
    const bool tensorMixFlag, DD::Interval const *seqIndexInterval,
    const int tensorElementCountEff, vector<bool> const &factorPetFlag,
    const ESMC_TypeKind_Flag typekindFactors, const bool haloFlag,
    Array const *array, const int localDeCount, const int *localDeElementCount,
    int const *localDeToDeMap, int const *localIntervalPerPetCount,
    int const *localElementsPerIntervalCount
    ){
    
    int localrc = ESMC_RC_NOT_IMPL;         // local return code
    
    // set up seqIntervFactorListCountToPet and seqIntervFactorListIndexToPet
    vector<int> seqIntervFactorListCountToPet(petCount);
    for (int i=0; i<petCount; i++)
      seqIntervFactorListCountToPet[i] = 0; // reset
    vector<vector<int> > seqIntervFactorListIndexToPet(petCount);
    vector<vector<int> > seqIntervFactorListLookupIndexToPet(petCount);
    
    for (int j=0; j<factorListCount; j++){
      // loop over all factorList entries, find matching interval via bisection
      // and count factor towards those that need to be sent to that PET.
      SeqInd seqInd;
      if (dstSetupFlag)
        seqInd = sparseMatrix[0].getDstSeqIndex(j);
      else
        seqInd = sparseMatrix[0].getSrcSeqIndex(j);
      int seqIndex = seqInd.getIndex(0);
      int tensorSeqIndex;
      if (tensorMixFlag)
        tensorSeqIndex = seqInd.getIndex(1);
      else
        tensorSeqIndex = 1;  // dummy
      int iMin=0, iMax=petCount-1;
      int i=petCount/2;
      bool foundFlag=false;     // reset
      do{
        if (seqIndex < seqIndexInterval[i].min){
          iMax = i;
          i = iMin + (iMax - iMin) / 2;
          continue; 
        }
        if (seqIndex > seqIndexInterval[i].max){
          iMin = i;
          i = iMin + 1 + (iMax - iMin) / 2;
          continue; 
        }
        // found interval -> check if tensorSeqIndex is within bounds
        if (tensorSeqIndex < 1 || tensorSeqIndex > tensorElementCountEff){
          // tensorSeqIndex outside Array bounds
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
            "- factorIndexList contains tensorSeqIndex outside Array bounds",
            ESMC_CONTEXT, &localrc);
          return localrc;
        }
        foundFlag = true;  // set
        ++seqIntervFactorListCountToPet[i]; // count this factor for this Pet
        seqIntervFactorListIndexToPet[i].push_back(j); // store factorList ind.
        // determine lookupIndex and store
        int lookupIndex = seqIndex - seqIndexInterval[i].min;
        if (tensorMixFlag)
          lookupIndex += (tensorSeqIndex - 1) * seqIndexInterval[i].count;
        seqIntervFactorListLookupIndexToPet[i].push_back(lookupIndex); // store
        break;
      }while (iMin != iMax);
      if (!foundFlag){
        // seqIndex lies outside Array bounds
        ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
          "- factorIndexList contains seqIndex outside Array bounds",
          ESMC_CONTEXT, &localrc);
        return localrc;
      }
    }

    // construct seqIntervFactorListCountFromPet
    vector<int> seqIntervFactorListCountFromPet(petCount);
    vm->alltoall(&(seqIntervFactorListCountToPet.front()), sizeof(int),
      &(seqIntervFactorListCountFromPet.front()), sizeof(int), vmBYTE);

    DD::SetupSeqIndexFactorLookupStage1 setupSeqIndexFactorLookupStage1(
      seqIndexFactorLookup,
      localPet,
      petCount,
      (sparseMatrix.size()==0) ? NULL : &(sparseMatrix[0]),
      factorPetFlag,
      seqIndexInterval,
      seqIntervFactorListCountToPet,
      seqIntervFactorListIndexToPet,
      seqIntervFactorListLookupIndexToPet,
      seqIntervFactorListCountFromPet,
      tensorMixFlag,
      dstSetupFlag,
      typekindFactors);
      
#define WITH_RESERVE
#ifdef WITH_RESERVE
    // Determine factorCount in stage1 and use to reserve memory of vector
    // before using it in stage2. Not sure this is really an improvement over
    // just dealing with memory allocation hit, b/c it does require extra
    // communication.
    
    setupSeqIndexFactorLookupStage1.totalExchange(vm);
    
    for (vector<SeqIndexFactorLookup>::iterator
      i=seqIndexFactorLookup.begin(); i!=seqIndexFactorLookup.end(); ++i){
      // obtain memory
      i->factorList.reserve(i->factorCount);  
    }
#endif
    
    DD::SetupSeqIndexFactorLookupStage2
      setupSeqIndexFactorLookupStage2(setupSeqIndexFactorLookupStage1);

    setupSeqIndexFactorLookupStage2.totalExchange(vm);
    
    // deal with duplicate sparse matrix entries in seqIndexFactorLookup
    for (vector<SeqIndexFactorLookup>::iterator
      i=seqIndexFactorLookup.begin(); i!=seqIndexFactorLookup.end(); ++i){
#ifdef ASMMSTOREPRINT
      fprintf(asmmstoreprintfp,
        "befr duplicate elimination seqIndexFactorLookup[%d].factorCount=%d\n",
        i-seqIndexFactorLookup.begin(), i->factorCount);
#endif
      sort(i->factorList.begin(), i->factorList.end());
      if (!haloFlag){
        // not halo: sum duplicate sparse matrix entries into the first occur.
        for (int j=0; j<i->factorCount; j++){
          int k;
          for (k=j+1; k<i->factorCount; k++){
            if (i->factorList[j] == i->factorList[k]){
              sum(i->factorList[j].factor,
                i->factorList[k].factor, typekindFactors);
            }else
              break;
          }
          j=k-1;  // skip over the duplicates
        }
      }
      // eliminate duplicates in factorList, only leaving first occurrences
      i->factorList.erase(
        unique(i->factorList.begin(),
        i->factorList.end()),
        i->factorList.end());
      i->factorCount = i->factorList.size();
#ifdef ASMMSTOREPRINT
      fprintf(asmmstoreprintfp,
        "aftr duplicate elimination seqIndexFactorLookup[%d].factorCount=%d\n",
        i-seqIndexFactorLookup.begin(), i->factorCount);
#endif
    }
    
    // communicate between Pets to set up "de" member in seqIndexFactorLookup[]
    {
      DD::FillSelfDeInfo fillSelfDeInfo(
        array,
        localPet,
        localDeCount,
        localDeElementCount,
        localDeToDeMap,
        seqIndexInterval,
        seqIndexFactorLookup,
        tensorMixFlag,
        localIntervalPerPetCount,
        localElementsPerIntervalCount,
        (dstSetupFlag & haloFlag));
      
      fillSelfDeInfo.totalExchange(vm);
    }
    
    // eliminate duplicate de entries in seqIndexFactorLookup
    for (vector<SeqIndexFactorLookup>::iterator
      i=seqIndexFactorLookup.begin(); i!=seqIndexFactorLookup.end(); ++i){
      sort(i->de.begin(), i->de.end());
      i->de.erase(unique(i->de.begin(),i->de.end()),i->de.end());
    }
  
    // return successfully
    return ESMF_SUCCESS;
  }
  
} // namespace DD


#define ASMMSTORETIMING___disable

int sparseMatMulStoreEncodeXXE(VM *vm, DELayout *srcDelayout,
  DELayout *dstDelayout, bool tensorMixFlag, 
  int srcTensorContigLength, int dstTensorContigLength,
  ESMC_TypeKind_Flag typekindFactors, ESMC_TypeKind_Flag typekindSrc,
  ESMC_TypeKind_Flag typekindDst,
  const int *srcLocalDeElementCount, const int *dstLocalDeElementCount,
  vector<DD::AssociationElement> *srcLinSeqVect,
  vector<DD::AssociationElement> *dstLinSeqVect,
  const int *dstLocalDeTotalElementCount,
  char **rraList, int rraCount, RouteHandle **routehandle,
#ifdef ASMMSTORETIMING
  double *t8, double *t9, double *t10, double *t11, double *t12, double *t13,
  double *t14,
#endif
  int *srcTermProcessingArg = NULL,       // in (optional)
  int *pipelineDepthArg = NULL            // in (optional)
  );

//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::sparseMatMulStore()"
//BOPI
// !IROUTINE:  ESMCI::Array::sparseMatMulStore
//
// !INTERFACE:
int Array::sparseMatMulStore(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  Array *srcArray,                          // in    - source Array
  Array *dstArray,                          // in    - destination Array
  RouteHandle **routehandle,                // inout - handle to precomp. comm
  vector<SparseMatrix> const &sparseMatrix, // in    - sparse matrix vector
  bool haloFlag,                            // in    - support halo conditions
  int *srcTermProcessingArg,                // inout - src term proc (optional)
                                // if (NULL) -> auto-tune, no pass back
                                // if (!NULL && -1) -> auto-tune, pass back
                                // if (!NULL && >=0) -> no auto-tune, use input
  int *pipelineDepthArg                     // inout - pipeline depth (optional)
                                // if (NULL) -> auto-tune, no pass back
                                // if (!NULL && -1) -> auto-tune, pass back
                                // if (!NULL && >=0) -> no auto-tune, use input
  ){
//
// !DESCRIPTION:
//  Precompute and store communication pattern for sparse matrix multiplication
//  from srcArray to dstArray.
//
//  The implementation consists of four main phases:
//
//  - Phase I:    Check input for consistency. The sparse matrix is provided in
//                "input" distribution 
//  - Phase II:   Construct two distributed directories of sparse matrix
//                elements, one indexed by srcSeqIndex, one indexed by
//                dstSeqIndex. This takes the matrix from "input" to "work"
//                distribution.
//  - Phase III:  Use the information in "work" distribution and take it into 
//                "run" distribution, i.e. src and dst DEs have access to all
//                the local data they may operate on.
//  - Phase IV:   Use the information in "run" distribution to encode an
//                optimized XXE stream, balancing src/dst work-loads and
//                pipelining overlapping communications and computation.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
#define ASMMSTOREMEMLOG_off
  try{
  
  //---------------------------------------------------------------------------
  // Phase I
  //---------------------------------------------------------------------------
  
  // get the current VM and VM releated information
  VM *vm = VM::getCurrent(&localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  int localPet = vm->getLocalPet();
  int petCount = vm->getPetCount();
  
#ifdef ASMMSTORETIMING
  double t0, t1, t2, t3, t4, t5, t6, t7;  //gjt - profile
  double t8, t9, t10, t11, t12, t13, t14, t15; //gjt - profile
  double t4a, t4b;  //gjt - profile
  double t4a1, t4a2, t4a3;  //gjt - profile  
  double t4b1, t4b2, t4b3;  //gjt - profile
  double t5c, t5f;  //gjt - profile
  vm->barrier();      //synchronize start time across all PETs
  VMK::wtime(&t0);    //gjt - profile
#endif
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore1.0"));
#endif

  // every Pet must provide srcArray and dstArray
  if (srcArray == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to srcArray", ESMC_CONTEXT, &rc);
    return rc;
  }
  if (dstArray == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- Not a valid pointer to dstArray", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // srcArray and dstArray must not point to the identical Array object,
  // unless in halo mode
  if (!haloFlag && (srcArray == dstArray)){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
      "- srcArray and dstArray must not be identical", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // check that sparseMatrix vector does not contain more than one element
  // TODO: this is a limitation of the current sparseMatMulStore() implement.
  if (sparseMatrix.size() > 1){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
      "- currently only a single sparseMatrix element is supported",
      ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // check that only supported seqIndex component modes are present
  // TODO: this is a limitation of the current sparseMatMulStore() implement.
  if (sparseMatrix.size() == 1){
    if (sparseMatrix[0].getSrcN() != sparseMatrix[0].getDstN()){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- src and dst sequence index tuples must have same number of"
        " components", ESMC_CONTEXT, &rc);
      return rc;
    }
    if ((sparseMatrix[0].getSrcN()!=1)&&(sparseMatrix[0].getSrcN()!=2)){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- only 1- or 2-component sequence index tuples supported",
        ESMC_CONTEXT, &rc);
      return rc;
    }
  }
  
  // set factorListCount variable (may be different on each PET)
  int const factorListCount = (sparseMatrix.size()==0) ?
    0 : sparseMatrix[0].getFactorListCount();
  
  // determine local tensorMixFlag and typekindFactors settings
  bool tensorMixFlag = false;                   // default
  ESMC_TypeKind_Flag typekindFactors = ESMF_NOKIND;  // default
  if (factorListCount > 0){
    // set typekindFactors
    typekindFactors = sparseMatrix[0].getTypekind();
    // check if tensorMixFlag must be set
    if (sparseMatrix[0].getSrcN() == 2)
      tensorMixFlag = true;
  }

  // communicate typekindFactors across all Pets
  vector<ESMC_TypeKind_Flag> typekindList(petCount);
  vm->allgather(&typekindFactors, &(typekindList[0]), 
    sizeof(ESMC_TypeKind_Flag));
  // communicate tensorMixFlag across all Pets
  bool *tensorMixFlagList = new bool[petCount]; // cannot use vector<bool> here
  vm->allgather(&tensorMixFlag, tensorMixFlagList, sizeof(bool));
  
  // Check that all non-ESMF_NOKIND typekindList elements match,
  // set local typekindFactors accordingly and keep track of Pets that have
  // factors. At the same time check that tensorMixFlag matches across Pets
  // that provide factors.
  int factorPetCount = 0;         // init: number of PETs that provide factors
  vector<bool> factorPetFlag(petCount); // flag whether factors on PET
  typekindFactors = ESMF_NOKIND;  // reset: typekind of factors in sparse matrix
  for (int i=0; i<petCount; i++){
    // simple counting section of the loop:
    if (typekindList[i] != ESMF_NOKIND){
      // factors on PET
      factorPetFlag[i] = true;
      ++factorPetCount;
    }else
      // no factors on PET
      factorPetFlag[i] = false;
    // consistency checking section of the loop:
    if (typekindFactors == ESMF_NOKIND){
      // still not found any factors -> use this element to initialize:
      typekindFactors = typekindList[i];    // typekindFactors
      tensorMixFlag = tensorMixFlagList[i]; // tensorMixFlag
    }else{
      // previously initialized consistency checking variables:
      // -> check following elements against the previously initialized ones
      if (typekindList[i] != ESMF_NOKIND){
        // only if i-th PET provides factors
        if (typekindFactors != typekindList[i]){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
            "- TypeKind mismatch between PETs", ESMC_CONTEXT, &rc);
          return rc;
        }
        if (tensorMixFlag != tensorMixFlagList[i]){
          ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
            "- Mismatch between PETs with respect to tensorMixFlag",
            ESMC_CONTEXT, &rc);
          return rc;
        }
      }
    }
  }
  // garbage collection
  delete [] tensorMixFlagList;
  
  // check for the special case of no factors provided on any PET
  if (factorPetCount < 1){
    // it is consistent to treat the factorless case as a special situation
    // where no error is returned but the routehandle simply holds a nop
    // create and initialize the RouteHandle
    *routehandle = RouteHandle::create(&localrc);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
    localrc = (*routehandle)->setType(ESMC_ARRAYXXE);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
    // mark storage pointer in RouteHandle as invalid/NOP
    localrc = (*routehandle)->setStorage(NULL);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
    // BREAK OUT EARLY: return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }
  
  // set dataSize for factors
  int dataSizeFactors = ESMC_TypeKind_FlagSize(typekindFactors);
  
  // check that if tensorMixFlag is not set that the tensorElementCount matches
  if (!tensorMixFlag){
    if (srcArray->getTensorElementCount() != dstArray->getTensorElementCount()){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
        "- w/o tensor mixing srcArray and dstArray tensorElementCount must"
        " match", ESMC_CONTEXT, &rc);
      return rc;
    }
  }
  
#ifdef ASMMSTORETIMING
  VMK::wtime(&t1);   //gjt - profile
#endif

//---DEBUG-------------------
// return successfully
//rc = ESMF_SUCCESS;
//return rc;
//---DEBUG-------------------
    
  //---------------------------------------------------------------------------
  // Phase II
  //---------------------------------------------------------------------------

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.0"));
#endif

  // determine local srcElementCount
  const int srcLocalDeCount = srcArray->delayout->getLocalDeCount();
  const int *srcLocalDeToDeMap = srcArray->delayout->getLocalDeToDeMap();
  int *srcLocalDeElementCount = new int[srcLocalDeCount];
  int srcElementCount = 0;   // initialize
  for (int i=0; i<srcLocalDeCount; i++){
    int de = srcLocalDeToDeMap[i];  // global DE index
    srcLocalDeElementCount[i] = srcArray->exclusiveElementCountPDe[de]
      * srcArray->tensorElementCount;
    srcElementCount += srcLocalDeElementCount[i];
  }
  // communicate srcElementCount across all Pets
  // todo: use nb-allgather and wait right before needed below
  int *srcElementCountList = new int[petCount];
  vm->allgather(&srcElementCount, srcElementCountList, sizeof(int));

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.1"));
#endif

  // determine local dstElementCount
  const int dstLocalDeCount = dstArray->delayout->getLocalDeCount();
  const int *dstLocalDeToDeMap = dstArray->delayout->getLocalDeToDeMap();
  int *dstLocalDeElementCount = new int[dstLocalDeCount];
  int dstElementCount = 0;   // initialize
  for (int i=0; i<dstLocalDeCount; i++){
    if (haloFlag){
      // for halo the dst elements are in the rim of dstArray
      dstLocalDeElementCount[i] = 0;  // init
      for (int k=0; k<dstArray->rimElementCount[i]; k++){
        SeqIndex seqIndex = dstArray->rimSeqIndex[i][k];
        if (seqIndex.valid()){
          // this rim element holds a valid seqIndex
          ++dstLocalDeElementCount[i];  // count this element
        }
      }
    }else{
      int de = dstLocalDeToDeMap[i];  // global DE index
      dstLocalDeElementCount[i] = dstArray->exclusiveElementCountPDe[de]
        * dstArray->tensorElementCount;
    }
    dstElementCount += dstLocalDeElementCount[i];
  }
  // communicate dstElementCount across all Pets
  // todo: use nb-allgather and wait right before needed below
  int *dstElementCountList = new int[petCount];
  vm->allgather(&dstElementCount, dstElementCountList, sizeof(int));
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.2"));
#endif

    // set the effective tensorElementCount for src and dst Arrays
  int srcTensorElementCountEff = srcArray->tensorElementCount;  // default
  int dstTensorElementCountEff = dstArray->tensorElementCount;  // default
  if (!tensorMixFlag){
    // if there is not tensor mixing then default into tensor for tensor mode
    srcTensorElementCountEff = 1;
    dstTensorElementCountEff = 1;
  }    

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.3"));
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(&t2);   //gjt - profile
#endif
  
//---DEBUG-------------------
// return successfully
//rc = ESMF_SUCCESS;
//return rc;
//---DEBUG-------------------
  
  // find local srcSeqIndex Min/Max
  int srcSeqIndexMinMax[2]; // [0]=min, [1]=max
  srcSeqIndexMinMax[0] = srcSeqIndexMinMax[1] = -1; // visibly invalidate
  bool firstMinMax = true;
  for (int i=0; i<srcLocalDeCount; i++){
    if (srcLocalDeElementCount[i]){
      // there are elements for local DE i
      ArrayElement arrayElement(srcArray, i);
      // loop over all elements in exclusive region for local DE i
      while(arrayElement.isWithin()){
        // determine the sequentialized index for the current Array element
        SeqIndex seqIndex = arrayElement.getSequenceIndexExclusive();
        // record seqIndex min and max
        if (firstMinMax){
          srcSeqIndexMinMax[0] = srcSeqIndexMinMax[1]
            = seqIndex.decompSeqIndex; // initialize
          firstMinMax = false;
        }else{
          if (seqIndex.decompSeqIndex < srcSeqIndexMinMax[0])
            srcSeqIndexMinMax[0] = seqIndex.decompSeqIndex;
          if (seqIndex.decompSeqIndex > srcSeqIndexMinMax[1])
            srcSeqIndexMinMax[1] = seqIndex.decompSeqIndex;
        }
        arrayElement.next();
      } // end while over all exclusive elements
    }
  }
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.4"));
#endif

  // communicate srcSeqIndexMinMax across all Pets
  // todo: use nb-allgather and wait right before needed below
  int *srcSeqIndexMinMaxList = new int[2*petCount];
  vm->allgather(srcSeqIndexMinMax, srcSeqIndexMinMaxList, 2*sizeof(int));  
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.5"));
#endif

//---DEBUG-------------------
//char msg[160];
//sprintf(msg, "srcLocalDeCount=%d, srcLocalDeElementCount[0]=%d,"
//  " sizeof(DD::AssociationElement)=%d", srcLocalDeCount,
//  srcLocalDeElementCount[0], sizeof(DD::AssociationElement));
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
// return successfully
//rc = ESMF_SUCCESS;
//return rc;
//---DEBUG-------------------

#ifdef ASMMSTOREPRINT
  sprintf(asmmstoreprintfile, "asmmstoreprint.%05d", localPet);
  asmmstoreprintfp = fopen(asmmstoreprintfile, "a");
  for (int i=0; i<petCount; i++)
    fprintf(asmmstoreprintfp, "gjt: srcSeqIndexMinMaxList[%d(0/1)] = %d/%d\n",
      i, srcSeqIndexMinMaxList[i*2], srcSeqIndexMinMaxList[i*2+1]);
#endif

  // find local dstSeqIndex Min/Max
  int dstSeqIndexMinMax[2]; // [0]=min, [1]=max
  dstSeqIndexMinMax[0] = dstSeqIndexMinMax[1] = -1; // visibly invalidate
  firstMinMax = true;
  for (int i=0; i<dstLocalDeCount; i++){
    if (dstLocalDeElementCount[i]){
      // there are elements for local DE i
      if (haloFlag){
        // for halo the dst elements are in the rim of dstArray
        for (int k=0; k<dstArray->rimElementCount[i]; k++){
          SeqIndex seqIndex = dstArray->rimSeqIndex[i][k];
          if (seqIndex.valid()){
            // this rim element holds a valid seqIndex
            // record seqIndex min and max
            if (firstMinMax){
              dstSeqIndexMinMax[0] = dstSeqIndexMinMax[1]
                = seqIndex.decompSeqIndex; // initialize
              firstMinMax = false;
            }else{
              if (seqIndex.decompSeqIndex < dstSeqIndexMinMax[0])
                dstSeqIndexMinMax[0] = seqIndex.decompSeqIndex;
              if (seqIndex.decompSeqIndex > dstSeqIndexMinMax[1])
                dstSeqIndexMinMax[1] = seqIndex.decompSeqIndex;
            }
          }
        }
      }else{
        ArrayElement arrayElement(dstArray, i);
        // loop over all elements in exclusive region for local DE i
        while(arrayElement.isWithin()){
          // determine the sequentialized index for the current Array element
          SeqIndex seqIndex = arrayElement.getSequenceIndexExclusive();
          // record seqIndex min and max
          if (firstMinMax){
            dstSeqIndexMinMax[0] = dstSeqIndexMinMax[1]
              = seqIndex.decompSeqIndex; // initialize
            firstMinMax = false;
          }else{
            if (seqIndex.decompSeqIndex < dstSeqIndexMinMax[0])
              dstSeqIndexMinMax[0] = seqIndex.decompSeqIndex;
            if (seqIndex.decompSeqIndex > dstSeqIndexMinMax[1])
              dstSeqIndexMinMax[1] = seqIndex.decompSeqIndex;
          }
          arrayElement.next();
        } // end while over all exclusive elements
      }
    }
  }

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.6"));
#endif

  // communicate dstSeqIndexMinMax across all Pets
  // todo: use nb-allgather and wait right before needed below
  int *dstSeqIndexMinMaxList = new int[2*petCount];
  vm->allgather(dstSeqIndexMinMax, dstSeqIndexMinMaxList, 2*sizeof(int));
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.7"));
#endif

//---DEBUG-------------------
//char msg[160];
//sprintf(msg, "dstLocalDeCount=%d, dstLocalDeElementCount[0]=%d,"
//  " sizeof(DD::AssociationElement)=%d", dstLocalDeCount,
//  dstLocalDeElementCount[0], sizeof(DD::AssociationElement));
//ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
// return successfully
//rc = ESMF_SUCCESS;
//return rc;
//---DEBUG-------------------

#ifdef ASMMSTOREPRINT
  for (int i=0; i<petCount; i++)
    fprintf(asmmstoreprintfp, "gjt: dstSeqIndexMinMaxList[%d(0/1)] = %d/%d\n",
      i, dstSeqIndexMinMaxList[i*2], dstSeqIndexMinMaxList[i*2+1]);
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(&t3);   //gjt - profile
#endif

//---DEBUG-------------------
// return successfully
//rc = ESMF_SUCCESS;
//return rc;
//---DEBUG-------------------

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.8"));
#endif

  // set up structure and intervals of src and dst distributed directories
  
  // determine the srcSeqIndexMinGlobal and MaxGlobal
  // todo: for nb-allgather(srcSeqIndexMinMaxList) here insert commwait()
  int srcSeqIndexMinGlobal, srcSeqIndexMaxGlobal;
  int pastInitFlag = 0; // reset
  for (int i=0; i<petCount; i++){
    if (srcElementCountList[i]){
      // Pet i holds elements in srcArray
      if (pastInitFlag){
        if (srcSeqIndexMinMaxList[2*i] < srcSeqIndexMinGlobal)
          srcSeqIndexMinGlobal = srcSeqIndexMinMaxList[2*i];
        if (srcSeqIndexMinMaxList[2*i+1] > srcSeqIndexMaxGlobal)
          srcSeqIndexMaxGlobal = srcSeqIndexMinMaxList[2*i+1];
      }else{
        // initialization
        srcSeqIndexMinGlobal = srcSeqIndexMinMaxList[2*i];
        srcSeqIndexMaxGlobal = srcSeqIndexMinMaxList[2*i+1];
        pastInitFlag = 1; // set
      }
    }
  }
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.9"));
#endif
  
#ifdef ASMMSTORETIMING
  VMK::wtime(&t4a1);   //gjt - profile
#endif

  // set up a distributed directory for srcArray seqIndex look-up
  DD::Interval *srcSeqIndexInterval = new DD::Interval[petCount];
  {
    srcSeqIndexInterval[0].min = srcSeqIndexMinGlobal;  // start
    int indicesPerPet = (srcSeqIndexMaxGlobal - srcSeqIndexMinGlobal + 1)
      / petCount;
    int extraIndices = (srcSeqIndexMaxGlobal - srcSeqIndexMinGlobal + 1)
      % petCount;
    for (int i=0; i<petCount-1; i++){
      srcSeqIndexInterval[i].max = srcSeqIndexInterval[i].min + indicesPerPet
        - 1;
      if (i<extraIndices)
        ++srcSeqIndexInterval[i].max;   // distribute extra indices evenly
      srcSeqIndexInterval[i].count = 
        srcSeqIndexInterval[i].max - srcSeqIndexInterval[i].min + 1;
      srcSeqIndexInterval[i+1].min = srcSeqIndexInterval[i].max + 1;
    }
    srcSeqIndexInterval[petCount-1].max = srcSeqIndexMaxGlobal;  // finish
    srcSeqIndexInterval[petCount-1].count = 
      srcSeqIndexInterval[petCount-1].max - srcSeqIndexInterval[petCount-1].min
      + 1;
  }
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.10"));
#endif

#ifdef ASMMSTOREPRINT
  fprintf(asmmstoreprintfp, "gjt: localPet %d, srcElementCountList[localPet] = "
    "%d, srcSeqIndexMinMax = %d / %d, srcSeqIndexMinGlobal/MaxGlobal = %d, %d, "
    "srcSeqIndexInterval[localPet].min/.max = %d, %d\n",
    localPet, srcElementCountList[localPet], srcSeqIndexMinMax[0],
    srcSeqIndexMinMax[1], srcSeqIndexMinGlobal, srcSeqIndexMaxGlobal,
    srcSeqIndexInterval[localPet].min, srcSeqIndexInterval[localPet].max);
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(&t4a2);   //gjt - profile
#endif
  
  // It is more efficient to first sort and then count in a sorted list than
  // it is to do the counting from an unsorted list.
  int *srcLocalElementsPerIntervalCount = new int[petCount];
  {
    // prepare temporary seqIndexList for sorting
    //TODO: this scales in memory as the srcArray size, does not consider sparse
    vector<int> seqIndexList(srcElementCount);
    int jj=0;
    for (int j=0; j<srcLocalDeCount; j++){
      // loop over all elements in the exclusive region for localDe j
      ArrayElement arrayElement(srcArray, j);
      while(arrayElement.isWithin()){
        seqIndexList[jj] =
          arrayElement.getSequenceIndexExclusive().decompSeqIndex;
        ++jj;
        arrayElement.next();
      } // end while over all exclusive elements
    }
    sort(seqIndexList.begin(), seqIndexList.end());
    jj=0;
    for (int i=0; i<petCount; i++){
      int seqIndexMax = srcSeqIndexInterval[i].max;
      int count = 0; // reset
      while (jj<srcElementCount && seqIndexList[jj]<=seqIndexMax){
        ++count;  // increment counter
        ++jj;
      }
      srcLocalElementsPerIntervalCount[i] = count;
    }
  }
  
  int *srcLocalIntervalPerPetCount = new int[petCount];
  
#ifdef ASMMSTORETIMING
  VMK::wtime(&t4a3);   //gjt - profile
#endif
  
  vm->alltoall(srcLocalElementsPerIntervalCount, sizeof(int),
    srcLocalIntervalPerPetCount, sizeof(int), vmBYTE);
  
#ifdef ASMMSTORETIMING
  VMK::wtime(&t4a);   //gjt - profile
#endif
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.11"));
#endif

  // determine the dstSeqIndexMinGlobal and MaxGlobal
  // todo: for nb-allgather(dstSeqIndexMinMaxList) here insert commwait()
  int dstSeqIndexMinGlobal, dstSeqIndexMaxGlobal;
  pastInitFlag = 0; // reset
  for (int i=0; i<petCount; i++){
    if (dstElementCountList[i]){
      // Pet i holds elements in dstArray
      if (pastInitFlag){
        if (dstSeqIndexMinMaxList[2*i] < dstSeqIndexMinGlobal)
          dstSeqIndexMinGlobal = dstSeqIndexMinMaxList[2*i];
        if (dstSeqIndexMinMaxList[2*i+1] > dstSeqIndexMaxGlobal)
          dstSeqIndexMaxGlobal = dstSeqIndexMinMaxList[2*i+1];
      }else{
        // initialization
        dstSeqIndexMinGlobal = dstSeqIndexMinMaxList[2*i];
        dstSeqIndexMaxGlobal = dstSeqIndexMinMaxList[2*i+1];
        pastInitFlag = 1; // set
      }
    }
  }
    
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.12"));
#endif
    
#ifdef ASMMSTORETIMING
  VMK::wtime(&t4b1);   //gjt - profile
#endif
  
////gjt  
//printf("srcMin/Max = %d/%d, dstMinMax = %d/%d\n", srcSeqIndexMinGlobal,
//  srcSeqIndexMaxGlobal, dstSeqIndexMinGlobal, dstSeqIndexMaxGlobal);

  // set up a distributed directory for dstArray seqIndex look-up
  DD::Interval *dstSeqIndexInterval = new DD::Interval[petCount];
  {
    dstSeqIndexInterval[0].min = dstSeqIndexMinGlobal;  // start
    int indicesPerPet = (dstSeqIndexMaxGlobal - dstSeqIndexMinGlobal + 1)
      / petCount;
    int extraIndices = (dstSeqIndexMaxGlobal - dstSeqIndexMinGlobal + 1)
      % petCount;
    for (int i=0; i<petCount-1; i++){
      dstSeqIndexInterval[i].max = dstSeqIndexInterval[i].min + indicesPerPet
        - 1;
      if (i<extraIndices)
        ++dstSeqIndexInterval[i].max;   // distribute extra indices evenly
      dstSeqIndexInterval[i].count = 
        dstSeqIndexInterval[i].max - dstSeqIndexInterval[i].min + 1;
      dstSeqIndexInterval[i+1].min = dstSeqIndexInterval[i].max + 1;
    }
    dstSeqIndexInterval[petCount-1].max = dstSeqIndexMaxGlobal;  // finish
    dstSeqIndexInterval[petCount-1].count = 
      dstSeqIndexInterval[petCount-1].max - dstSeqIndexInterval[petCount-1].min
      + 1;
  }
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.13"));
#endif

#ifdef ASMMSTOREPRINT
    fprintf(asmmstoreprintfp, "gjt: localPet %d, dstElementCountList[localPet] "
    "= %d, "
    "dstSeqIndexMinMax = %d / %d, dstSeqIndexMinGlobal/MaxGlobal = %d, %d, "
    "dstSeqIndexInterval[localPet].min/.max = %d, %d\n",
    localPet, dstElementCountList[localPet], dstSeqIndexMinMax[0],
    dstSeqIndexMinMax[1], dstSeqIndexMinGlobal, dstSeqIndexMaxGlobal,
    dstSeqIndexInterval[localPet].min, dstSeqIndexInterval[localPet].max);
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(&t4b2);   //gjt - profile
#endif

  // It is more efficient to first sort and then count in a sorted list than
  // it is to do the counting from an unsorted list.
  int *dstLocalElementsPerIntervalCount = new int[petCount];
  {
    // prepare temporary seqIndexList for sorting
    //TODO: this scales in memory as the dstArray size, does not consider sparse
    vector<int> seqIndexList(dstElementCount);
    int jj=0;
    for (int j=0; j<dstLocalDeCount; j++){
      if (haloFlag){
        // loop over the halo rim elements for localDe j
        for (int k=0; k<dstArray->rimElementCount[j]; k++){
          SeqIndex seqIndex = dstArray->rimSeqIndex[j][k];
          if (seqIndex.valid()){
            seqIndexList[jj] = seqIndex.decompSeqIndex;
            ++jj;
          }
        }
      }else{
        // loop over all elements in the exclusive region for localDe j
        ArrayElement arrayElement(dstArray, j);
        while(arrayElement.isWithin()){
          seqIndexList[jj] =
            arrayElement.getSequenceIndexExclusive().decompSeqIndex;
          ++jj;
          arrayElement.next();
        } // end while over all exclusive elements
      }
    }
    sort(seqIndexList.begin(), seqIndexList.end());
    jj=0;
    for (int i=0; i<petCount; i++){
      int seqIndexMax = dstSeqIndexInterval[i].max;
      int count = 0; // reset
      while (jj<dstElementCount && seqIndexList[jj]<=seqIndexMax){
        ++count;  // increment counter
        ++jj;
      }
      dstLocalElementsPerIntervalCount[i] = count;
    }
  }
  
  int *dstLocalIntervalPerPetCount = new int[petCount];
  
#ifdef ASMMSTORETIMING
  VMK::wtime(&t4b3);   //gjt - profile
#endif

  vm->alltoall(dstLocalElementsPerIntervalCount, sizeof(int),
    dstLocalIntervalPerPetCount, sizeof(int), vmBYTE);
  
#ifdef ASMMSTORETIMING
  VMK::wtime(&t4b);   //gjt - profile
#endif

  // garbage collection  
  delete [] srcElementCountList;
  delete [] dstElementCountList;
  delete [] srcSeqIndexMinMaxList;
  delete [] dstSeqIndexMinMaxList;

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.14"));
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(&t4);   //gjt - profile
#endif
  
  // allocate local look-up map indexed by srcSeqIndex, i.e. distributed dir.
  vector<DD::SeqIndexFactorLookup>
    srcSeqIndexFactorLookup(srcSeqIndexInterval[localPet].count
    * srcTensorElementCountEff);
  localrc = DD::setupSeqIndexFactorLookup(vm, 
    srcSeqIndexFactorLookup,
    petCount, localPet, factorListCount, 
    false,  // dstSetupFlag
    sparseMatrix, tensorMixFlag,
    srcSeqIndexInterval, srcTensorElementCountEff, factorPetFlag,
    typekindFactors, haloFlag, srcArray, srcLocalDeCount,
    srcLocalDeElementCount, srcLocalDeToDeMap, srcLocalIntervalPerPetCount,
    srcLocalElementsPerIntervalCount);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  
//vm->barrier();  /// only for profiling tests
      
#ifdef ASMMSTORETIMING
  VMK::wtime(&t5c);   //gjt - profile
#endif

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.15"));
#endif

  // allocate local look-up map indexed by dstSeqIndex, i.e. distributed dir.
  vector<DD::SeqIndexFactorLookup>
    dstSeqIndexFactorLookup(dstSeqIndexInterval[localPet].count
    * dstTensorElementCountEff);
  localrc = DD::setupSeqIndexFactorLookup(vm, 
    dstSeqIndexFactorLookup,
    petCount, localPet, factorListCount, 
    true,  // dstSetupFlag
    sparseMatrix, tensorMixFlag,
    dstSeqIndexInterval, dstTensorElementCountEff, factorPetFlag,
    typekindFactors, haloFlag, dstArray, dstLocalDeCount,
    dstLocalDeElementCount, dstLocalDeToDeMap, dstLocalIntervalPerPetCount,
    dstLocalElementsPerIntervalCount);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  
//vm->barrier();  /// only for profiling tests
      
#ifdef ASMMSTORETIMING
  VMK::wtime(&t5f);   //gjt - profile
#endif
  
#ifdef ASMMSTORETIMING
  VMK::wtime(&t5);   //gjt - profile
#endif
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.16"));
#endif
  
  // prepare count arrays for src partner look-up in dstSeqIndexFactorLookup
  int *srcLocalPartnerElementsPerIntervalCount = new int[petCount];
  for (int i=0; i<petCount; i++){
    // Pet "i" is the active srcSeqIndex interval
    int seqIndexMin = dstSeqIndexInterval[i].min;
    int seqIndexMax = dstSeqIndexInterval[i].max;
    int count = 0; // reset
    for (vector<DD::SeqIndexFactorLookup>::const_iterator
      j=srcSeqIndexFactorLookup.begin(); j!=srcSeqIndexFactorLookup.end(); ++j){
      for (int k=0; k<j->factorCount; k++){
        int partnerSeqIndex = j->factorList[k]
          .partnerSeqIndex.decompSeqIndex;
        if (partnerSeqIndex >= seqIndexMin && partnerSeqIndex <= seqIndexMax)
          ++count; // increment counter
      }
    }
    srcLocalPartnerElementsPerIntervalCount[i] = count;
  }
  int *dstLocalPartnerIntervalPerPetCount = new int[petCount];
  vm->alltoall(srcLocalPartnerElementsPerIntervalCount, sizeof(int),
    dstLocalPartnerIntervalPerPetCount, sizeof(int), vmBYTE);
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.17"));
#endif

  // fill partnerDe in srcSeqIndexFactorLookup using dstSeqIndexFactorLookup
  {
    DD::FillPartnerDeInfo *fillPartnerDeInfo =
      new DD::FillPartnerDeInfo(dstSeqIndexFactorLookup,
      srcSeqIndexFactorLookup);
      
    fillPartnerDeInfo->localPet = localPet;
    fillPartnerDeInfo->seqIndexIntervalIn = dstSeqIndexInterval;
    fillPartnerDeInfo->seqIndexIntervalOut = srcSeqIndexInterval;
    fillPartnerDeInfo->tensorMixFlag = tensorMixFlag;
      
    DD::accessLookup(vm, petCount, localPet, dstLocalPartnerIntervalPerPetCount,
      srcLocalPartnerElementsPerIntervalCount, fillPartnerDeInfo);
    delete fillPartnerDeInfo;
  }

  // garbage collection
  delete [] srcLocalPartnerElementsPerIntervalCount;
  delete [] dstLocalPartnerIntervalPerPetCount;
      
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.18"));
#endif

  // prepare count arrays for dst partner look-up in srcSeqIndexFactorLookup
  int *dstLocalPartnerElementsPerIntervalCount = new int[petCount];
  for (int i=0; i<petCount; i++){
    // Pet "i" is the active dstSeqIndex interval
    int seqIndexMin = srcSeqIndexInterval[i].min;
    int seqIndexMax = srcSeqIndexInterval[i].max;
    int count = 0; // reset
    for (vector<DD::SeqIndexFactorLookup>::const_iterator
      j=dstSeqIndexFactorLookup.begin(); j!=dstSeqIndexFactorLookup.end(); ++j){
      for (int k=0; k<j->factorCount; k++){
        int partnerSeqIndex = j->factorList[k]
          .partnerSeqIndex.decompSeqIndex;
        if (partnerSeqIndex >= seqIndexMin && partnerSeqIndex <= seqIndexMax)
          ++count; // increment counter
      }
    }
    dstLocalPartnerElementsPerIntervalCount[i] = count;
  }
  int *srcLocalPartnerIntervalPerPetCount = new int[petCount];
  vm->alltoall(dstLocalPartnerElementsPerIntervalCount, sizeof(int),
    srcLocalPartnerIntervalPerPetCount, sizeof(int), vmBYTE);
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.19"));
#endif

  // fill partnerDe in dstSeqIndexFactorLookup using srcSeqIndexFactorLookup
  {
    DD::FillPartnerDeInfo *fillPartnerDeInfo =
      new DD::FillPartnerDeInfo(srcSeqIndexFactorLookup,
      dstSeqIndexFactorLookup);
    fillPartnerDeInfo->localPet = localPet;
    fillPartnerDeInfo->seqIndexIntervalIn = srcSeqIndexInterval;
    fillPartnerDeInfo->seqIndexIntervalOut = dstSeqIndexInterval;
    fillPartnerDeInfo->tensorMixFlag = tensorMixFlag;
      
    DD::accessLookup(vm, petCount, localPet, srcLocalPartnerIntervalPerPetCount,
      dstLocalPartnerElementsPerIntervalCount, fillPartnerDeInfo);
    delete fillPartnerDeInfo;
  }

  // garbage collection
  delete [] dstLocalPartnerElementsPerIntervalCount;
  delete [] srcLocalPartnerIntervalPerPetCount;

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore2.20"));
#endif

#ifdef ASMMSTOREPRINT
  fprintf(asmmstoreprintfp, "\n========================================"
    "========================================\n");
  fprintf(asmmstoreprintfp, "========================================"
    "========================================\n\n");
  // some serious printing for src info
  for (vector<DD::SeqIndexFactorLookup>::const_iterator
    i=srcSeqIndexFactorLookup.begin(); i!=srcSeqIndexFactorLookup.end(); ++i){
    fprintf(asmmstoreprintfp, "gjt srcDistDir: localPet %d, srcSeqIndex = %d, "
      "srcSeqIndexFactorLookup[%d].factorCount = %d\n -> .de: ",
      localPet, (i-srcSeqIndexFactorLookup.begin())
      +srcSeqIndexInterval[localPet].min, i-srcSeqIndexFactorLookup.begin(),
      i->factorCount);
    for (int j=0; j<i->de.size(); j++)
      fprintf(asmmstoreprintfp, "%d, ", i->de[j]);
    fprintf(asmmstoreprintfp, "\n");
    for (int j=0; j<i->factorCount; j++){
      fprintf(asmmstoreprintfp, "\tfactorList[%d]\n"
        "\t\t.partnerSeqIndex.decompSeqIndex = %d\n"
        "\t\t.partnerDe = ", j,
        i->factorList[j].partnerSeqIndex
        .decompSeqIndex);
      for (int jj=0;
        jj<i->factorList[j].partnerDe.size(); jj++)
        fprintf(asmmstoreprintfp, "%d, ",
          i->factorList[j].partnerDe[jj]);
      fprintf(asmmstoreprintfp, "\n");
      fprintf(asmmstoreprintfp, "\t\t.factor = %d\n",
        *((int *)i->factorList[j].factor));
    }
  }
#endif
  
#ifdef ASMMSTOREPRINT
  // some serious printing for dst info
  for (vector<DD::SeqIndexFactorLookup>::const_iterator
    i=dstSeqIndexFactorLookup.begin(); i!=dstSeqIndexFactorLookup.end(); ++i){
    fprintf(asmmstoreprintfp, "gjt dstDistDir: localPet %d, dstSeqIndex = %d, "
      "dstSeqIndexFactorLookup[%d].factorCount = %d\n -> .de: ",
      localPet, (i-dstSeqIndexFactorLookup.begin())
      +dstSeqIndexInterval[localPet].min, i-dstSeqIndexFactorLookup.begin(),
      i->factorCount);
    for (int j=0; j<i->de.size(); j++)
      fprintf(asmmstoreprintfp, "%d, ", i->de[j]);
    fprintf(asmmstoreprintfp, "\n");
    for (int j=0; j<i->factorCount; j++){
      fprintf(asmmstoreprintfp, "\tfactorList[%d]\n"
        "\t\t.partnerSeqIndex.decompSeqIndex = %d\n"
        "\t\t.partnerDe = ", j,
        i->factorList[j].partnerSeqIndex
        .decompSeqIndex);
      for (int jj=0;
        jj<i->factorList[j].partnerDe.size(); jj++)
        fprintf(asmmstoreprintfp, "%d, ",
          i->factorList[j].partnerDe[jj]);
      fprintf(asmmstoreprintfp, "\n");
      fprintf(asmmstoreprintfp, "\t\t.factor = %d\n",
        *((int *)i->factorList[j].factor));
    }
  }
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(&t6);   //gjt - profile
#endif
  
//---DEBUG-------------------
// return successfully
//rc = ESMF_SUCCESS;
//return rc;
//---DEBUG-------------------
  
  //---------------------------------------------------------------------------
  // Phase III -> represent sparse matrix in "run distribution"
  //---------------------------------------------------------------------------
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore3.0"));
#endif

  vector<DD::AssociationElement> *srcLinSeqVect =
    new vector<DD::AssociationElement>[srcLocalDeCount];
  vector<DD::AssociationElement> *dstLinSeqVect =
    new vector<DD::AssociationElement>[dstLocalDeCount];

  // access srcSeqIndexFactorLookup to construct srcLinSeqVect
  {  
    DD::FillLinSeqVectInfo *fillLinSeqVectInfo =
      new DD::FillLinSeqVectInfo(srcSeqIndexFactorLookup);
    fillLinSeqVectInfo->array = srcArray;
    fillLinSeqVectInfo->linSeqVect = srcLinSeqVect;
    fillLinSeqVectInfo->localPet = localPet;
    fillLinSeqVectInfo->localDeCount = srcLocalDeCount;
    fillLinSeqVectInfo->localDeElementCount = srcLocalDeElementCount;
    fillLinSeqVectInfo->seqIndexInterval = srcSeqIndexInterval;
    fillLinSeqVectInfo->tensorMixFlag = tensorMixFlag;
    fillLinSeqVectInfo->haloRimFlag = false;

    DD::accessLookup(vm, petCount, localPet, srcLocalIntervalPerPetCount,
      srcLocalElementsPerIntervalCount, fillLinSeqVectInfo);
    delete fillLinSeqVectInfo;
  }
  
  // access dstSeqIndexFactorLookup to construct dstLinSeqVect
  {
    DD::FillLinSeqVectInfo *fillLinSeqVectInfo =
      new DD::FillLinSeqVectInfo(dstSeqIndexFactorLookup);
    fillLinSeqVectInfo->array = dstArray;
    fillLinSeqVectInfo->linSeqVect = dstLinSeqVect;
    fillLinSeqVectInfo->localPet = localPet;
    fillLinSeqVectInfo->localDeCount = dstLocalDeCount;
    fillLinSeqVectInfo->localDeElementCount = dstLocalDeElementCount;
    fillLinSeqVectInfo->seqIndexInterval = dstSeqIndexInterval;
    fillLinSeqVectInfo->tensorMixFlag = tensorMixFlag;
    fillLinSeqVectInfo->haloRimFlag = haloFlag; // forward HALO

    DD::accessLookup(vm, petCount, localPet, dstLocalIntervalPerPetCount,
      dstLocalElementsPerIntervalCount, fillLinSeqVectInfo);
    delete fillLinSeqVectInfo;
  }

  // garbage colletion
  delete [] srcLocalElementsPerIntervalCount;
  delete [] srcLocalIntervalPerPetCount;
  delete [] dstLocalElementsPerIntervalCount;
  delete [] dstLocalIntervalPerPetCount;
  delete [] srcSeqIndexInterval;
  delete [] dstSeqIndexInterval;
  
#ifdef GJT_DEBUG
/////////////////////////  
  for (vector<DD::SeqIndexFactorLookup>::const_iterator
    i=srcSeqIndexFactorLookup.begin(); i!=srcSeqIndexFactorLookup.end(); ++i){
    
    fprintf(stderr, "src: lookupIndex=%d, factorCount=%d\n",
      i-srcSeqIndexFactorLookup.begin(), i->factorCount);
  }
/////////////////////////  
/////////////////////////  
  for (vector<DD::SeqIndexFactorLookup>::const_iterator
    i=dstSeqIndexFactorLookup.begin(); i!=dstSeqIndexFactorLookup.end(); ++i){
    
    fprintf(stderr, "dst: lookupIndex=%d, factorCount=%d\n",
      i-dstSeqIndexFactorLookup.begin(), i->factorCount);
  }
/////////////////////////  
#endif
  
#ifdef ASMMSTOREPRINT
  fprintf(asmmstoreprintfp, "\n========================================"
    "========================================\n");
  fprintf(asmmstoreprintfp, "========================================"
    "========================================\n\n");
  // more serious printing
  for (int j=0; j<srcLocalDeCount; j++){
    for (int k=0; k<srcLinSeqVect[j].size(); k++){
      fprintf(asmmstoreprintfp, "localPet: %d, "
        "srcLinSeqVect[%d][%d].linIndex = %d, "
        ".seqIndex = %d/%d, .factorCount = %d\n",
        localPet, j, k, srcLinSeqVect[j][k].linIndex,
        srcLinSeqVect[j][k].seqIndex.decompSeqIndex,
        srcLinSeqVect[j][k].seqIndex.tensorSeqIndex,
        srcLinSeqVect[j][k].factorCount);
      for (int kk=0; kk<srcLinSeqVect[j][k].factorCount; kk++){
        fprintf(asmmstoreprintfp, "\tfactorList[%d]\n"
          "\t\t.partnerSeqIndex = %d/%d\n"
          "\t\t.partnerDe = ", kk,
          srcLinSeqVect[j][k].factorList[kk].partnerSeqIndex.decompSeqIndex,
          srcLinSeqVect[j][k].factorList[kk].partnerSeqIndex.tensorSeqIndex);
        for (int jj=0;
          jj<srcLinSeqVect[j][k].factorList[kk].partnerDe.size(); jj++)
          fprintf(asmmstoreprintfp, "%d, ",
            srcLinSeqVect[j][k].factorList[kk].partnerDe[jj]);
        fprintf(asmmstoreprintfp, "\n");
        fprintf(asmmstoreprintfp, "\t\t.factor = %d\n",
          *((int *)srcLinSeqVect[j][k].factorList[kk].factor));
      }
    }
  }
    
  // more serious printing
  for (int j=0; j<dstLocalDeCount; j++){
    for (int k=0; k<dstLinSeqVect[j].size(); k++){
      fprintf(asmmstoreprintfp, "localPet: %d, "
        "dstLinSeqVect[%d][%d].linIndex = %d, "
        ".seqIndex = %d/%d, .factorCount = %d\n",
        localPet, j, k, dstLinSeqVect[j][k].linIndex,
        dstLinSeqVect[j][k].seqIndex.decompSeqIndex,
        dstLinSeqVect[j][k].seqIndex.tensorSeqIndex,
        dstLinSeqVect[j][k].factorCount);
      for (int kk=0; kk<dstLinSeqVect[j][k].factorCount; kk++){
        fprintf(asmmstoreprintfp, "\tfactorList[%d]\n"
          "\t\t.partnerSeqIndex = %d/%d\n"
          "\t\t.partnerDe = ", kk,
          dstLinSeqVect[j][k].factorList[kk].partnerSeqIndex.decompSeqIndex,
          dstLinSeqVect[j][k].factorList[kk].partnerSeqIndex.tensorSeqIndex);
        for (int jj=0;
          jj<dstLinSeqVect[j][k].factorList[kk].partnerDe.size(); jj++)
          fprintf(asmmstoreprintfp, "%d, ",
            dstLinSeqVect[j][k].factorList[kk].partnerDe[jj]);
        fprintf(asmmstoreprintfp, "\n");
        fprintf(asmmstoreprintfp, "\t\t.factor = %d\n",
          *((int *)dstLinSeqVect[j][k].factorList[kk].factor));
      }
    }
  }
#endif
  
  if (haloFlag){
    // Phase IV below expects each FactorElement inside of srcLinSeqVect and
    // dstLinSeqVect to only reference a single partnerDe (- only partnerDe[0]
    // will be looked at!). Therefore transform FactorElements that have more
    // than one partnerDe entry into multiple FactorElements where each only
    // has a single partnerDe entry:
    // Doing this for the src side supports forward HALO:
    for (int j=0; j<srcLocalDeCount; j++){
      for (int k=0; k<srcLinSeqVect[j].size(); k++){
        for (int kk=0; kk<srcLinSeqVect[j][k].factorCount; kk++){
          if (srcLinSeqVect[j][k].factorList[kk].partnerDe.size() > 1){
            for (int jj=1;
              jj<srcLinSeqVect[j][k].factorList[kk].partnerDe.size(); jj++){
              // construct factorElement with one partnerDe entry
              DD::FactorElement factorElement =
                srcLinSeqVect[j][k].factorList[kk];
              factorElement.partnerDe.resize(1);
              factorElement.partnerDe[0] =
                srcLinSeqVect[j][k].factorList[kk].partnerDe[jj];
              srcLinSeqVect[j][k].factorList.push_back(factorElement);
            }
            // erase all of the replicated partnerDe entries
            srcLinSeqVect[j][k].factorList[kk].partnerDe.resize(1);
          }
        }
        // set new factorCount
        srcLinSeqVect[j][k].factorCount = srcLinSeqVect[j][k].factorList.size();
      }
    }
    // Doing this for the dst side supports backward HALO:
    for (int j=0; j<dstLocalDeCount; j++){
      for (int k=0; k<dstLinSeqVect[j].size(); k++){
        for (int kk=0; kk<dstLinSeqVect[j][k].factorCount; kk++){
          if (dstLinSeqVect[j][k].factorList[kk].partnerDe.size() > 1){
            for (int jj=1;
              jj<dstLinSeqVect[j][k].factorList[kk].partnerDe.size(); jj++){
              // construct factorElement with one partnerDe entry
              DD::FactorElement factorElement =
                dstLinSeqVect[j][k].factorList[kk];
              factorElement.partnerDe.resize(1);
              factorElement.partnerDe[0] =
                dstLinSeqVect[j][k].factorList[kk].partnerDe[jj];
              //TODO: It's certainly not correct to add up all the entries
              //TODO: from all of the halo elements in a backward HALO,
              //TODO: instead the average over all elements makes more sense.
              //TODO: However, this would require communication here in order
              //TODO: to set an adjusted factor for these elements also on the 
              //TODO: src side.
              dstLinSeqVect[j][k].factorList.push_back(factorElement);
            }
            // erase all of the replicated partnerDe entries
            dstLinSeqVect[j][k].factorList[kk].partnerDe.resize(1);
          }
        }
        // set new factorCount
        dstLinSeqVect[j][k].factorCount = dstLinSeqVect[j][k].factorList.size();
      }
    }
  }
  
#ifdef ASMMSTOREPRINT
  fprintf(asmmstoreprintfp, "\n========================================"
    "========================================\n");
  fprintf(asmmstoreprintfp, "========================================"
    "========================================\n\n");
  // more serious printing
  for (int j=0; j<srcLocalDeCount; j++){
    for (int k=0; k<srcLinSeqVect[j].size(); k++){
      fprintf(asmmstoreprintfp, "localPet: %d, "
        "srcLinSeqVect[%d][%d].linIndex = %d, "
        ".seqIndex = %d/%d, .factorCount = %d\n",
        localPet, j, k, srcLinSeqVect[j][k].linIndex,
        srcLinSeqVect[j][k].seqIndex.decompSeqIndex,
        srcLinSeqVect[j][k].seqIndex.tensorSeqIndex,
        srcLinSeqVect[j][k].factorCount);
      for (int kk=0; kk<srcLinSeqVect[j][k].factorCount; kk++){
        fprintf(asmmstoreprintfp, "\tfactorList[%d]\n"
          "\t\t.partnerSeqIndex = %d/%d\n"
          "\t\t.partnerDe = ", kk,
          srcLinSeqVect[j][k].factorList[kk].partnerSeqIndex.decompSeqIndex,
          srcLinSeqVect[j][k].factorList[kk].partnerSeqIndex.tensorSeqIndex);
        for (int jj=0;
          jj<srcLinSeqVect[j][k].factorList[kk].partnerDe.size(); jj++)
          fprintf(asmmstoreprintfp, "%d, ",
            srcLinSeqVect[j][k].factorList[kk].partnerDe[jj]);
        fprintf(asmmstoreprintfp, "\n");
        fprintf(asmmstoreprintfp, "\t\t.factor = %d\n",
          *((int *)srcLinSeqVect[j][k].factorList[kk].factor));
      }
    }
  }
    
  // more serious printing
  for (int j=0; j<dstLocalDeCount; j++){
    for (int k=0; k<dstLinSeqVect[j].size(); k++){
      fprintf(asmmstoreprintfp, "localPet: %d, "
        "dstLinSeqVect[%d][%d].linIndex = %d, "
        ".seqIndex = %d/%d, .factorCount = %d\n",
        localPet, j, k, dstLinSeqVect[j][k].linIndex,
        dstLinSeqVect[j][k].seqIndex.decompSeqIndex,
        dstLinSeqVect[j][k].seqIndex.tensorSeqIndex,
        dstLinSeqVect[j][k].factorCount);
      for (int kk=0; kk<dstLinSeqVect[j][k].factorCount; kk++){
        fprintf(asmmstoreprintfp, "\tfactorList[%d]\n"
          "\t\t.partnerSeqIndex = %d/%d\n"
          "\t\t.partnerDe = ", kk,
          dstLinSeqVect[j][k].factorList[kk].partnerSeqIndex.decompSeqIndex,
          dstLinSeqVect[j][k].factorList[kk].partnerSeqIndex.tensorSeqIndex);
        for (int jj=0;
          jj<dstLinSeqVect[j][k].factorList[kk].partnerDe.size(); jj++)
          fprintf(asmmstoreprintfp, "%d, ",
            dstLinSeqVect[j][k].factorList[kk].partnerDe[jj]);
        fprintf(asmmstoreprintfp, "\n");
        fprintf(asmmstoreprintfp, "\t\t.factor = %d\n",
          *((int *)dstLinSeqVect[j][k].factorList[kk].factor));
      }
    }
  }
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(&t7);   //gjt - profile
#endif
  
//---DEBUG-------------------
// return successfully
//rc = ESMF_SUCCESS;
//return rc;
//---DEBUG-------------------
  
  //---------------------------------------------------------------------------
  // Phase IV
  //---------------------------------------------------------------------------

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore4.0"));
#endif

  // prepare for relative run-time addressing (RRA)
  int rraCount = srcArray->delayout->getLocalDeCount();
  rraCount += dstArray->delayout->getLocalDeCount();
  char **rraList = new char*[rraCount];
  memcpy((char *)rraList, srcArray->larrayBaseAddrList,
    srcArray->delayout->getLocalDeCount() * sizeof(char *));
  memcpy((char *)rraList
    + srcArray->delayout->getLocalDeCount() * sizeof(char *),
    dstArray->larrayBaseAddrList,
    dstArray->delayout->getLocalDeCount() * sizeof(char *));
  // obtain typekindSrc
  ESMC_TypeKind_Flag typekindSrc = srcArray->getTypekind();
  // obtain typekindDst
  ESMC_TypeKind_Flag typekindDst = dstArray->getTypekind();
  
  // prepare dstLocalDeTotalElementCount
  int *dstLocalDeTotalElementCount = new int[dstLocalDeCount];
  for (int i=0; i<dstLocalDeCount; i++)
    dstLocalDeTotalElementCount[i] = 
      dstArray->totalElementCountPLocalDe[i] * dstArray->tensorElementCount;
  
  // prepare tensorContigLength arguments
  int srcTensorContigLength = 1;  // init
  for (int jj=0; jj<srcArray->rank; jj++){
    if (srcArray->arrayToDistGridMap[jj])
      // decomposed dimension
      break;
    else
      // tensor dimension
      srcTensorContigLength *= srcArray->undistUBound[jj]
        - srcArray->undistLBound[jj] + 1;
  }
  int dstTensorContigLength = 1;  // init
  for (int jj=0; jj<dstArray->rank; jj++){
    if (dstArray->arrayToDistGridMap[jj])
      // decomposed dimension
      break;
    else
      // tensor dimension
      dstTensorContigLength *= dstArray->undistUBound[jj]
        - dstArray->undistLBound[jj] + 1;
  }
  
  // encode sparseMatMul communication pattern into XXE stream
  localrc = sparseMatMulStoreEncodeXXE(vm,
    srcArray->delayout, dstArray->delayout,
    tensorMixFlag, srcTensorContigLength, dstTensorContigLength,
    typekindFactors, typekindSrc, typekindDst,
    srcLocalDeElementCount, dstLocalDeElementCount,
    srcLinSeqVect, dstLinSeqVect,
    dstLocalDeTotalElementCount,
    rraList, rraCount, routehandle,
#ifdef ASMMSTORETIMING
    &t8, &t9, &t10, &t11, &t12, &t13, &t14,
#endif
    srcTermProcessingArg,
    pipelineDepthArg
  );
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;

  // garbage collection
  delete [] rraList;
  delete [] srcLinSeqVect;
  delete [] srcLocalDeElementCount;
  delete [] dstLinSeqVect;
  delete [] dstLocalDeElementCount;
  delete [] dstLocalDeTotalElementCount;

#ifdef ASMMSTORETIMING
  VMK::wtime(&t15);   //gjt - profile
  char asmmstoretimingfile[160];
  sprintf(asmmstoretimingfile, "asmmstoretiming.%05d", localPet);
  FILE *asmmstoretimingfp = fopen(asmmstoretimingfile, "a");
  fprintf(asmmstoretimingfp, "\n========================================"
    "========================================\n");
  fprintf(asmmstoretimingfp, "========================================"
    "========================================\n\n");
  fprintf(asmmstoretimingfp, "gjt - profile for PET %d:\n"
    " t4a1=%g\n t4a2=%g\n t4a3=%g\n t4a=%g\n"
    " t4b1=%g\n t4b2=%g\n t4b3=%g\n t4b=%g\n",
    localPet, t4a1-t3, t4a2-t3, t4a3-t3, t4a-t3, t4b1-t3, t4b2-t3, t4b3-t3,
    t4b-t3);
  fprintf(asmmstoretimingfp, "gjt - profile for PET %d:\n"
    " t5c=%g\n"
    " t5f=%g\n",
    localPet, 
    t5c-t4,
    t5f-t4);
  fprintf(asmmstoretimingfp, "gjt - profile for PET %d:\n"
    " t1=%g\n t2=%g\n t3=%g\n t4=%g\n t5=%g\n t6=%g\n"
    " t7=%g\n t8=%g\n t9=%g\n t10=%g\n t11=%g\n t12=%g\n t13=%g\n t14=%g\n"
    " t15=%g\n",
    localPet, t1-t0, t2-t0, t3-t0, t4-t0, 
    t5-t0, t6-t0, t7-t0, t8-t0, t9-t0, t10-t0, t11-t0, t12-t0, t13-t0, t14-t0,
    t15-t0);
  fclose(asmmstoretimingfp);
#endif
  
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(exception &x){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      x.what(), ESMC_CONTEXT, &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
#ifdef ASMMSTOREPRINT
  fclose(asmmstoreprintfp);
#endif
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStore5.0"));
#endif

  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


int sparseMatMulStoreEncodeXXEStream(VM *vm,
  vector<ArrayHelper::RecvnbElement> &recvnbVector,
  vector<ArrayHelper::SendnbElement> &sendnbVector,
  int srcTermProcessing, int pipelineDepth, XXE::TKId elementTK,
  XXE::TKId valueTK, XXE::TKId factorTK,
  int dataSizeSrc, int dataSizeDst, int dataSizeFactors, int srcLocalDeCount,
  int dstLocalDeCount, const int *dstLocalDeTotalElementCount, char **rraList,
  int rraCount, int vectorLength, XXE *xxe);

//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::sparseMatMulStoreEncodeXXE()"
//BOPI
// !IROUTINE:  ESMCI::sparseMatMulStoreEncodeXXE
//
// !INTERFACE:
int sparseMatMulStoreEncodeXXE(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  VM *vm,                                 // in
  DELayout *srcDelayout,                  // in
  DELayout *dstDelayout,                  // in
  bool tensorMixFlag,                     // in
  int srcTensorContigLength,              // in
  int dstTensorContigLength,              // in
  ESMC_TypeKind_Flag typekindFactors,     // in
  ESMC_TypeKind_Flag typekindSrc,         // in
  ESMC_TypeKind_Flag typekindDst,         // in
  const int *srcLocalDeElementCount,      // in
  const int *dstLocalDeElementCount,      // in
  vector<DD::AssociationElement> *srcLinSeqVect, // in - sparse mat "run dist."
  vector<DD::AssociationElement> *dstLinSeqVect, // in - sparse mat "run dist."
  const int *dstLocalDeTotalElementCount, // in
  char **rraList,                         // in
  int rraCount,                           // in
  RouteHandle **routehandle,              // inout - handle to precomputed comm
#ifdef ASMMSTORETIMING
  double *t8, double *t9, double *t10, double *t11, double *t12, double *t13,
  double *t14,
#endif
  int *srcTermProcessingArg,    // inout (optional)
                                // if (NULL) -> auto-tune, no pass back
                                // if (!NULL && -1) -> auto-tune, pass back
                                // if (!NULL && >=0) -> no auto-tune, use input
  int *pipelineDepthArg         // inout (optional)
                                // if (NULL) -> auto-tune, no pass back
                                // if (!NULL && -1) -> auto-tune, pass back
                                // if (!NULL && >=0) -> no auto-tune, use input
  ){
//
// !DESCRIPTION:
//    Take the incoming sparse matrix information in "run distribution" and
//    use it to encode an XXE stream for the sparseMatMul. The generated XXE
//    object is attached to a new routehandle object through which it is
//    returned.
//
//    The XXE stream is generated by a two step process. First 
//    sparseMatMulStoreEncodeXXE() transforms the sparse matrix information from
//    "run distribution" into (srcDe, dstDe) pair specific SendnbElement and
//    RecvnbElement objects. These elements are stored in the two main local
//    data objects of this routine:
//
//      srcLinSeqVect -> sendnbVector
//      dstLinSeqVect -> recvnbVector
//
//    These two vectors contain as many objects as there are srcDe, dstDe on
//    the localPet, respectively.
//
//    In the second step sendnbVector and recvnbVector are used to encode the
//    actual XXE stream by calling sparseMatMulStoreEncodeXXEStream(). This
//    stream encode routine allows two stream parameters to be specified:
//    srcTermProcessing and pipelineDepth. sparseMatMulStoreEncodeXXE() calls
//    sparseMatMulStoreEncodeXXEStream() multiple times with different settings
//    for srcTermProcessing and pipelineDepth, scanning the parameter space to
//    find an optimum parameter setting for the current machine characteristic
//    and the communication pattern provided in sendnbVector and recvnbVector.
//
//EOPI
//-----------------------------------------------------------------------------
#define SMMSTOREENCODEXXEINFO_off
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  try{
  // create and initialize the RouteHandle
  *routehandle = RouteHandle::create(&localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  localrc = (*routehandle)->setType(ESMC_ARRAYXXE);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE1.0"));
#endif

  // allocate XXE and attach to RouteHandle
  XXE *xxe;
  try{
    xxe = new XXE(vm, 1000, 10000, 1000);
  }catch (...){
    ESMC_LogDefault.AllocError(ESMC_CONTEXT, &rc);
    return rc;
  }
  localrc = (*routehandle)->setStorage(xxe);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  // set typekind in xxe which is used to check Arrays before ASMM execution 
  xxe->typekind[0] = typekindFactors;
  xxe->typekind[1] = typekindSrc;
  xxe->typekind[2] = typekindDst;
  // prepare XXE type variables
  XXE::TKId elementTK;
  switch (typekindDst){
  case ESMC_TYPEKIND_R4:
    elementTK = XXE::R4;
    break;
  case ESMC_TYPEKIND_R8:
    elementTK = XXE::R8;
    break;
  case ESMC_TYPEKIND_I4:
    elementTK = XXE::I4;
    break;
  case ESMC_TYPEKIND_I8:
    elementTK = XXE::I8;
    break;
  default:
    break;
  }
  XXE::TKId valueTK;
  switch (typekindSrc){
  case ESMC_TYPEKIND_R4:
    valueTK = XXE::R4;
    break;
  case ESMC_TYPEKIND_R8:
    valueTK = XXE::R8;
    break;
  case ESMC_TYPEKIND_I4:
    valueTK = XXE::I4;
    break;
  case ESMC_TYPEKIND_I8:
    valueTK = XXE::I8;
    break;
  default:
    break;
  }
  XXE::TKId factorTK;
  switch (typekindFactors){
  case ESMC_TYPEKIND_R4:
    factorTK = XXE::R4;
    break;
  case ESMC_TYPEKIND_R8:
    factorTK = XXE::R8;
    break;
  case ESMC_TYPEKIND_I4:
    factorTK = XXE::I4;
    break;
  case ESMC_TYPEKIND_I8:
    factorTK = XXE::I8;
    break;
  default:
    break;
  }
  // prepare other local variables
  int dataSizeFactors = ESMC_TypeKind_FlagSize(typekindFactors);
  
  int dataSizeSrc = ESMC_TypeKind_FlagSize(typekindSrc);
  int srcLocalDeCount = srcDelayout->getLocalDeCount();
  const int *srcLocalDeToDeMap = srcDelayout->getLocalDeToDeMap();
  
  int dataSizeDst = ESMC_TypeKind_FlagSize(typekindDst);
  int dstLocalDeCount = dstDelayout->getLocalDeCount();
  const int *dstLocalDeToDeMap = dstDelayout->getLocalDeToDeMap();
  
  int localPet = vm->getLocalPet();
  int petCount = vm->getPetCount();
  
  bool vectorFlag = !(tensorMixFlag || 
    (srcTensorContigLength != dstTensorContigLength));

#ifdef ASMMSTORETIMING
  double t9a, t9b, t9d, t9e; //gjt - profile
  double t9c1, t9c2; //gjt - profile
  VMK::wtime(t8);   //gjt - profile
#endif
    
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE2.0"));
#endif

  // determine recv pattern for all localDEs on dst side
  vector<ArrayHelper::RecvnbElement> recvnbVector;
  for (int j=0; j<dstLocalDeCount; j++){
    int *index2Ref = new int[dstLocalDeElementCount[j]];  // large enough
    int localDeFactorCount = 0; // reset
    int iCount = 0; // reset
    for (int k=0; k<dstLinSeqVect[j].size(); k++){
      int factorCount = dstLinSeqVect[j][k].factorCount;
      if (factorCount){
        index2Ref[iCount] = k;   // store element index
        localDeFactorCount += factorCount;
        ++iCount; // increment counter
      }
    }

#ifdef ASMMSTORETIMING
    VMK::wtime(&t9a);   //gjt - profile
#endif
        
#ifdef ASMMSTOREPRINT
fprintf(asmmstoreprintfp, "iCount: %d, localDeFactorCount: %d\n", iCount,
  localDeFactorCount);
#endif
    int *index2Ref2 = new int[localDeFactorCount];  // large enough
    int *factorIndexRef = new int[localDeFactorCount];  // large enough
    int *partnerDeRef = new int[localDeFactorCount];  // large enough
    vector<int> recvnbPartnerDeList(localDeFactorCount);  // large enough
    vector<int> recvnbPartnerDeCount(localDeFactorCount);  // large enough
    int recvnbDiffPartnerDeCount = 0; // reset
    int count = 0; // reset
    for (int i=0; i<iCount; i++){
      int factorCount = dstLinSeqVect[j][index2Ref[i]].factorCount;
      for (int k=0; k<factorCount; k++){
        int partnerDe =
          dstLinSeqVect[j][index2Ref[i]].factorList[k].partnerDe[0];
        int kk;
        for (kk=0; kk<recvnbDiffPartnerDeCount; kk++)
          if (recvnbPartnerDeList[kk]==partnerDe) break;
        if (kk==recvnbDiffPartnerDeCount){
          // new entry
          recvnbPartnerDeList[kk] = partnerDe;
          recvnbPartnerDeCount[kk] = 1; // initialize
          ++recvnbDiffPartnerDeCount;
        }else
          ++recvnbPartnerDeCount[kk];   // increment
        index2Ref2[count] = index2Ref[i];
        factorIndexRef[count] = k;
        partnerDeRef[count] = kk;
        ++count;
      }
    }
    
#ifdef ASMMSTORETIMING
    VMK::wtime(&t9b);   //gjt - profile
#endif
        
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE3.0"));
#endif

    // invert the look-up direction
    vector<vector<ArrayHelper::DstInfo> >
      dstInfoTable(recvnbDiffPartnerDeCount);
    vector<int> dstInfoTableInit(recvnbDiffPartnerDeCount);
    for (int i=0; i<recvnbDiffPartnerDeCount; i++){
      dstInfoTable[i].resize(recvnbPartnerDeCount[i]);
      dstInfoTableInit[i] = 0;   // reset
    }
    char *localDeFactorBuffer = new char[localDeFactorCount * dataSizeFactors];
    localrc = xxe->storeStorage(localDeFactorBuffer); // XXE garbage collec.
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMSTORETIMING
    VMK::wtime(&t9c1);   //gjt - profile
#endif
    for (int i=0; i<localDeFactorCount; i++){
      int partnerDeListIndex = partnerDeRef[i];
      int index2 = dstInfoTableInit[partnerDeListIndex]++;
      dstInfoTable[partnerDeListIndex][index2].linIndex =
        dstLinSeqVect[j][index2Ref2[i]].linIndex;
      dstInfoTable[partnerDeListIndex][index2].vectorLength = 1;  // default
      dstInfoTable[partnerDeListIndex][index2].seqIndex =
        dstLinSeqVect[j][index2Ref2[i]].seqIndex;
      dstInfoTable[partnerDeListIndex][index2].partnerSeqIndex
        = dstLinSeqVect[j][index2Ref2[i]].factorList[factorIndexRef[i]]
        .partnerSeqIndex;
      if (!tensorMixFlag){
        // default into tensor for tensor src/dst mode
        dstInfoTable[partnerDeListIndex][index2].partnerSeqIndex
          .tensorSeqIndex
          = dstInfoTable[partnerDeListIndex][index2].seqIndex.tensorSeqIndex;
      }
      char *localDeFactorBufferEntry = localDeFactorBuffer + i*dataSizeFactors;
      memcpy(localDeFactorBufferEntry,
        dstLinSeqVect[j][index2Ref2[i]].factorList[factorIndexRef[i]]
        .factor, dataSizeFactors);
      dstInfoTable[partnerDeListIndex][index2].factor =
        (void *)(localDeFactorBufferEntry);
    }
    
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE4.0"));
#endif

      // garbage collection
    delete [] index2Ref;
    delete [] index2Ref2;
    delete [] factorIndexRef;
    delete [] partnerDeRef;
    
#ifdef ASMMSTORETIMING
    VMK::wtime(&t9c2);   //gjt - profile
#endif  
       
    // sort each "recvnbDiffPartnerDeCount group" (opposite of src)
    if (!vectorFlag){
      // no vectorization possible -> sort for scalar optimization
      for (int i=0; i<recvnbDiffPartnerDeCount; i++)
        sort(dstInfoTable[i].begin(), dstInfoTable[i].end(),
          ArrayHelper::scalarOrderDstInfo);
    }else if (dstTensorContigLength == 1){
      // support vectorization during execution, but nothing to deflate here
      // sort for scalar optimization
      for (int i=0; i<recvnbDiffPartnerDeCount; i++)
        sort(dstInfoTable[i].begin(), dstInfoTable[i].end(),
          ArrayHelper::scalarOrderDstInfo);
    }else{
      // vectorization
      // sort vector optimization
      for (int i=0; i<recvnbDiffPartnerDeCount; i++){
        sort(dstInfoTable[i].begin(), dstInfoTable[i].end(),
          ArrayHelper::vectorOrderDstInfo);
#ifdef ASMMSTOREPRINT
        for (int k=0; k<dstInfoTable[i].size(); k++)
          fprintf(asmmstoreprintfp, "dstInfoTable[%d][%d].seqIndex = %d/%d, "
            ".partnerSeqIndex = %d/%d, .factor = %p\n", i, k,
            dstInfoTable[i][k].seqIndex.decompSeqIndex, 
            dstInfoTable[i][k].seqIndex.tensorSeqIndex, 
            dstInfoTable[i][k].partnerSeqIndex.decompSeqIndex, 
            dstInfoTable[i][k].partnerSeqIndex.tensorSeqIndex,
            dstInfoTable[i][k].factor);
#endif
        // vectorize -> deflate dstInfoTable 
        vector<ArrayHelper::DstInfo>::iterator rangeStart =
          dstInfoTable[i].begin();
        vector<ArrayHelper::DstInfo>::iterator rangeStop = rangeStart;
        vector<ArrayHelper::DstInfo>::iterator rangeWrite = rangeStart;
        while (rangeStart != dstInfoTable[i].end()){
          int vectorLength = 1; // initialize
          int decompSeqIndex = rangeStart->seqIndex.decompSeqIndex;
          int linIndex = rangeStart->linIndex;
          rangeStop++;
          while((rangeStop != dstInfoTable[i].end())
            && (rangeStop->seqIndex.decompSeqIndex == decompSeqIndex)
            && (rangeStop->linIndex == (linIndex + 1))){
            linIndex = rangeStop->linIndex;
            ++vectorLength;
            rangeStop++;
          }
#ifdef ASMMSTOREPRINT
fprintf(asmmstoreprintfp, "dstTensorContigLength: %d, vectorLength: %d, decompSeqIndex: %d\n",
  dstTensorContigLength, vectorLength, decompSeqIndex);
fflush(asmmstoreprintfp);
#endif
          if ((rangeWrite != dstInfoTable[i].begin())
            && ((rangeWrite-1)->vectorLength != vectorLength)){
            ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_INCONS,
            "- vectorization failed", ESMC_CONTEXT, &rc);
            return rc;
          }
          if (vectorLength != dstTensorContigLength){
            ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_INCONS,
            "- vectorization failed", ESMC_CONTEXT, &rc);
            return rc;
          }
          *rangeWrite = *rangeStart;  // copy table element
          rangeWrite->vectorLength = vectorLength;
          rangeWrite++;
          rangeStart = rangeStop;
        }
        dstInfoTable[i].erase(rangeWrite, dstInfoTable[i].end());
      }
    }

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE5.0"));
#endif

#ifdef ASMMSTOREPRINT
    // print:
    fprintf(asmmstoreprintfp, "dstArray: %d, %d\n", j,
      recvnbDiffPartnerDeCount); 
    for (int i=0; i<recvnbDiffPartnerDeCount; i++)
      for (int k=0; k<dstInfoTable[i].size(); k++)
        fprintf(asmmstoreprintfp, "dstInfoTable[%d][%d].seqIndex = %d/%d, "
          ".partnerSeqIndex[][] = %d/%d\n", i, k,
          dstInfoTable[i][k].seqIndex.decompSeqIndex, 
          dstInfoTable[i][k].seqIndex.tensorSeqIndex, 
          dstInfoTable[i][k].partnerSeqIndex.decompSeqIndex, 
          dstInfoTable[i][k].partnerSeqIndex.tensorSeqIndex);
#endif
    
#ifdef ASMMSTORETIMING
    VMK::wtime(&t9d);   //gjt - profile
#endif

    // construct recv elements
    for (int i=0; i<recvnbDiffPartnerDeCount; i++){
      int vectorLength = dstInfoTable[i].begin()->vectorLength;
      // initialize the bufferIndex member in the dstInfoTable
#ifdef MSG_DEFLATE
      // construct bufferIndex member in dstInfoTable according to deflation
      vector<ArrayHelper::Deflator> deflator(dstInfoTable[i].size());
      for (int k=0; k<dstInfoTable[i].size(); k++){
        // fill
        deflator[k].seqIndex = dstInfoTable[i][k].partnerSeqIndex;
        deflator[k].index = k;
      }
      // sort by partnerSeqIndex
      sort(deflator.begin(), deflator.end());
      // record the buffer index in dstInfoTable
      int kk = 0;
      dstInfoTable[i][deflator[0].index].bufferIndex = kk;  // spin up
      for (int k=1; k<deflator.size(); k++){
        if (deflator[k].seqIndex != deflator[k-1].seqIndex)
          ++kk;
        dstInfoTable[i][deflator[k].index].bufferIndex = kk;
      }
      ++kk;
      
#ifdef MSG_DEFLATE_DEBUG
char msg[160];
      for (int k=0; k<kk; k++){
        
sprintf(msg, "recv: deflator[%d]: index=%d, bufferIndex=%d, seqIndex=%d",
  k, deflator[k].index, dstInfoTable[i][deflator[k].index].bufferIndex,
deflator[k].seqIndex.decompSeqIndex);
ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);

      }
#endif
        
#else
      //todo: remove this initialization once the deflator works right!!!!!!
      for (int k=0; k<dstInfoTable[i].size(); k++)
        dstInfoTable[i][k].bufferIndex = k;
      int kk = dstInfoTable[i].size();
#endif
      // large contiguous 1st level receive buffer
//alignment      char *buffer = new char[recvnbPartnerDeCount[i] * dataSizeSrc];
      int qwords = (recvnbPartnerDeCount[i] * dataSizeSrc) / 8;
      if ((recvnbPartnerDeCount[i] * dataSizeSrc) % 8)
        ++qwords;
      char *buffer = (char *)(new double[qwords]);
      localrc = xxe->storeStorage(buffer); // XXE garbage collec.
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      // store buffer information in BufferInfo for XXE buffer control
      localrc = xxe->storeBufferInfo(buffer,
        recvnbPartnerDeCount[i] * dataSizeSrc,
        recvnbPartnerDeCount[i] * dataSizeSrc / vectorLength);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      // prepare DE/PET info
      int srcDe = recvnbPartnerDeList[i];
      int srcPet;   //TODO: DE-based comms
      srcDelayout->getDEMatchPET(srcDe, *vm, NULL, &srcPet, 1);
      int dstDe = dstLocalDeToDeMap[j];
      // fill values into recvnbVector
      ArrayHelper::RecvnbElement recvnbElement;
      recvnbElement.srcPet = srcPet;
      recvnbElement.srcDe = srcDe;
      recvnbElement.srcLocalDe = i;
      recvnbElement.dstDe = dstDe;
      recvnbElement.dstLocalDe = j;
      recvnbElement.bufferInfo = (char **)xxe->getBufferInfoPtr();
      recvnbElement.partnerDeDataCount = kk;
      recvnbElement.vectorFlag = vectorFlag;
      recvnbElement.dstInfoTable = dstInfoTable[i];
      recvnbElement.localPet = localPet;
      recvnbElement.petCount = petCount;
      recvnbVector.push_back(recvnbElement);
#ifdef ASMMSTOREPRINT
      fprintf(asmmstoreprintfp, "gjt: recvnbElement localPet %d, srcPet %d, "
        "vectorLength=%d\n", localPet, srcPet, vectorLength);
#endif
    } // for i - recvnbDiffPartnerDeCount
    
#ifdef ASMMSTORETIMING
    VMK::wtime(&t9e);   //gjt - profile
//    printf("gjt - profile for PET %d, j-loop %d:\n"
//      " t9a=%g\n t9b=%g\n t9c1=%g\n t9c2=%g\n t9d=%g\n t9e=%g\n", localPet, j,
//      t9a-*t8, t9b-*t8, t9c1-*t8, t9c2-*t8, t9d-*t8, t9e-*t8);
#endif
    
  } // for j - dstLocalDeCount
    
#ifdef ASMMSTORETIMING
  VMK::wtime(t9);   //gjt - profile
#endif
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE6.0"));
#endif

  // determine send pattern for all localDEs on src side
  vector<ArrayHelper::SendnbElement> sendnbVector;
  for (int j=0; j<srcLocalDeCount; j++){
    int *index2Ref = new int[srcLocalDeElementCount[j]];  // large enough
    int localDeFactorCount = 0; // reset
    int iCount = 0; // reset
    for (int k=0; k<srcLinSeqVect[j].size(); k++){
      int factorCount = srcLinSeqVect[j][k].factorCount;
      if (factorCount){
        index2Ref[iCount] = k;   // store element index
        localDeFactorCount += factorCount;
        ++iCount; // increment counter
      }
    }
    int *index2Ref2 = new int[localDeFactorCount];  // large enough
    int *factorIndexRef = new int[localDeFactorCount];  // large enough
    int *partnerDeRef = new int[localDeFactorCount];  // large enough
    int *sendnbPartnerDeList = new int[localDeFactorCount];  // large enough
    int *sendnbPartnerDeCount = new int[localDeFactorCount];  // large enough
    int sendnbDiffPartnerDeCount = 0; // reset
    int count = 0; // reset
    for (int i=0; i<iCount; i++){
      int factorCount = srcLinSeqVect[j][index2Ref[i]].factorCount;
      for (int k=0; k<factorCount; k++){
        int partnerDe =
          srcLinSeqVect[j][index2Ref[i]].factorList[k].partnerDe[0];
        int kk;
        for (kk=0; kk<sendnbDiffPartnerDeCount; kk++)
          if (sendnbPartnerDeList[kk]==partnerDe) break;
        if (kk==sendnbDiffPartnerDeCount){
          // new entry
          sendnbPartnerDeList[kk] = partnerDe;
          sendnbPartnerDeCount[kk] = 1; // initialize
          ++sendnbDiffPartnerDeCount;
        }else
          ++sendnbPartnerDeCount[kk];   // increment
        index2Ref2[count] = index2Ref[i];
        factorIndexRef[count] = k;
        partnerDeRef[count] = kk;
        ++count;
      }
    }
    // invert the look-up direction
    vector<vector<ArrayHelper::SrcInfo> >
      srcInfoTable(sendnbDiffPartnerDeCount);
    vector<int> srcInfoTableInit(sendnbDiffPartnerDeCount);
    for (int i=0; i<sendnbDiffPartnerDeCount; i++){
      srcInfoTable[i].resize(sendnbPartnerDeCount[i]);
      srcInfoTableInit[i] = 0;   // reset
    }
    // alignment char *localDeFactorBuffer = new char[localDeFactorCount * dataSizeFactors];
    int qwords = (localDeFactorCount * dataSizeFactors) / 8;
    if ((localDeFactorCount * dataSizeFactors) % 8)
      ++qwords;
    char *localDeFactorBuffer = (char *)(new double[qwords]);
    localrc = xxe->storeStorage(localDeFactorBuffer); // XXE garbage collec.
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    for (int i=0; i<localDeFactorCount; i++){
      int partnerDeListIndex = partnerDeRef[i];
      int index2 = srcInfoTableInit[partnerDeListIndex]++;
      srcInfoTable[partnerDeListIndex][index2].linIndex =
        srcLinSeqVect[j][index2Ref2[i]].linIndex;
      srcInfoTable[partnerDeListIndex][index2].vectorLength = 1;
      srcInfoTable[partnerDeListIndex][index2].seqIndex =
        srcLinSeqVect[j][index2Ref2[i]].seqIndex;
      srcInfoTable[partnerDeListIndex][index2].partnerSeqIndex =
        srcLinSeqVect[j][index2Ref2[i]].factorList[factorIndexRef[i]]
        .partnerSeqIndex;
      if (!tensorMixFlag){
        // default into tensor for tensor src/dst mode
        srcInfoTable[partnerDeListIndex][index2].partnerSeqIndex
          .tensorSeqIndex =
          srcInfoTable[partnerDeListIndex][index2].seqIndex.tensorSeqIndex;
      }
      char *localDeFactorBufferEntry = localDeFactorBuffer + i*dataSizeFactors;
      memcpy(localDeFactorBufferEntry,
        srcLinSeqVect[j][index2Ref2[i]].factorList[factorIndexRef[i]]
        .factor, dataSizeFactors);
      srcInfoTable[partnerDeListIndex][index2].factor =
        (void *)(localDeFactorBufferEntry);
    }
    
    // garbage collection
    delete [] index2Ref;
    delete [] index2Ref2;
    delete [] factorIndexRef;
    delete [] partnerDeRef;
    
    // sort each "sendnbDiffPartnerDeCount group" (opposite of dst)
    if (!vectorFlag){
      // no vectorization possible -> sort for scalar optimization
      for (int i=0; i<sendnbDiffPartnerDeCount; i++)
        sort(srcInfoTable[i].begin(), srcInfoTable[i].end(),
          ArrayHelper::scalarOrderSrcInfo);
    }else if (srcTensorContigLength == 1){
      // support vectorization during execution, but nothing to deflate here
      // sort for scalar optimization
      for (int i=0; i<sendnbDiffPartnerDeCount; i++)
        sort(srcInfoTable[i].begin(), srcInfoTable[i].end(),
          ArrayHelper::scalarOrderSrcInfo);
    }else{
      // vectorization
      // sort vector optimization
      for (int i=0; i<sendnbDiffPartnerDeCount; i++){
        sort(srcInfoTable[i].begin(), srcInfoTable[i].end(),
          ArrayHelper::vectorOrderSrcInfo);
        // vectorize -> deflate srcInfoTable 
        vector<ArrayHelper::SrcInfo>::iterator rangeStart =
          srcInfoTable[i].begin();
        vector<ArrayHelper::SrcInfo>::iterator rangeStop = rangeStart;
        vector<ArrayHelper::SrcInfo>::iterator rangeWrite = rangeStart;
        while (rangeStart != srcInfoTable[i].end()){
          int vectorLength = 1; // initialize
          int decompSeqIndex = rangeStart->seqIndex.decompSeqIndex;
          int linIndex = rangeStart->linIndex;
          rangeStop++;
          while((rangeStop != srcInfoTable[i].end())
            && (rangeStop->seqIndex.decompSeqIndex == decompSeqIndex)
            && (rangeStop->linIndex == (linIndex + 1))){
            linIndex = rangeStop->linIndex;
            ++vectorLength;
            rangeStop++;
          }
          if ((rangeWrite != srcInfoTable[i].begin())
            && ((rangeWrite-1)->vectorLength != vectorLength)){
            ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_INCONS,
            "- vectorization failed", ESMC_CONTEXT, &rc);
            return rc;
          }
          if (vectorLength != srcTensorContigLength){
            ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_INCONS,
            "- vectorization failed", ESMC_CONTEXT, &rc);
            return rc;
          }
          *rangeWrite = *rangeStart;  // copy table element
          rangeWrite->vectorLength = vectorLength;
          rangeWrite++;
          rangeStart = rangeStop;
        }
        srcInfoTable[i].erase(rangeWrite, srcInfoTable[i].end());
      }
    }
        
#ifdef ASMMSTOREPRINT
    // print:
    for (int i=0; i<sendnbDiffPartnerDeCount; i++)
      for (int k=0; k<srcInfoTable[i].size(); k++)
        fprintf(asmmstoreprintfp, "srcInfoTable[%d][%d].seqIndex = %d/%d, "
          ".partnerSeqIndex[][] = %d/%d\n", i, k,
          srcInfoTable[i][k].seqIndex.decompSeqIndex, 
          srcInfoTable[i][k].seqIndex.tensorSeqIndex, 
          srcInfoTable[i][k].partnerSeqIndex.decompSeqIndex, 
          srcInfoTable[i][k].partnerSeqIndex.tensorSeqIndex);
#endif
    
    // construct send elements
    for (int i=0; i<sendnbDiffPartnerDeCount; i++){
      int vectorLength = srcInfoTable[i].begin()->vectorLength;
      // use a temporary vector to aide in deflating of the message to be sent
      vector<ArrayHelper::Deflator> deflator(srcInfoTable[i].size());
      for (int k=0; k<srcInfoTable[i].size(); k++){
        // fill
        deflator[k].seqIndex  = srcInfoTable[i][k].seqIndex;
        deflator[k].index     = srcInfoTable[i][k].linIndex;
      }
#ifdef MSG_DEFLATE
      // sort -> cut off duplic.
      sort(deflator.begin(), deflator.end());
      deflator.erase(unique(deflator.begin(),deflator.end()),deflator.end());
      
#ifdef MSG_DEFLATE_DEBUG
char msg[160];
      for (int k=0; k<deflator.size(); k++){
        
sprintf(msg, "send: deflator[%d]: index=%d, seqIndex=%d",
  k, deflator[k].index,
deflator[k].seqIndex.decompSeqIndex);
ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
      }
#endif      
      
#endif
      // determine contiguous runs in linIndex to minimize memcpy overhead
      vector<ArrayHelper::LinIndexContigBlock> linIndexContigBlockList;
      // initialize linIndexContigBlockList[]
      ArrayHelper::LinIndexContigBlock block;
      block.linIndex = deflator[0].index;
      block.linIndexCount = 1;
      linIndexContigBlockList.push_back(block);
      for (int k=1; k<deflator.size(); k++){
        if (deflator[k-1].index + vectorLength == deflator[k].index){
          // contiguous step in linIndex
          ++(linIndexContigBlockList.back().linIndexCount);
        }else{
          // discontiguous jump in linIndex
          block.linIndex = deflator[k].index;
          block.linIndexCount = 1;
          linIndexContigBlockList.push_back(block);
        }
      }
      // intermediate buffer (in case it is needed)
//alignment      char *buffer = new char[sendnbPartnerDeCount[i] * dataSizeSrc];
      int qwords = (sendnbPartnerDeCount[i] * dataSizeSrc) / 8;
      if ((sendnbPartnerDeCount[i] * dataSizeSrc) % 8)
        ++qwords;
      char *buffer = (char *)(new double[qwords]);
      localrc = xxe->storeStorage(buffer); // XXE garbage collec.
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      // store buffer information in BufferInfo for XXE buffer control
      localrc = xxe->storeBufferInfo(buffer,
        sendnbPartnerDeCount[i] * dataSizeSrc,
        sendnbPartnerDeCount[i] * dataSizeSrc / vectorLength);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      // prepare DE/PET info
      int srcDe = srcLocalDeToDeMap[j];
      int dstDe = sendnbPartnerDeList[i];
      int dstPet;   //TODO: DE-based comms
      dstDelayout->getDEMatchPET(dstDe, *vm, NULL, &dstPet, 1);
      // fill values into sendnbVector
      ArrayHelper::SendnbElement sendnbElement;
      sendnbElement.dstPet = dstPet;
      sendnbElement.dstDe = dstDe;
      sendnbElement.dstLocalDe = i;
      sendnbElement.srcDe = srcDe;
      sendnbElement.srcLocalDe = j;
      sendnbElement.partnerDeDataCount = deflator.size();
      sendnbElement.vectorFlag = vectorFlag;
      sendnbElement.srcInfoTable = srcInfoTable[i];
      sendnbElement.linIndexContigBlockList = linIndexContigBlockList;
      sendnbElement.bufferInfo = (char **)xxe->getBufferInfoPtr();
      sendnbElement.localPet = localPet;
      sendnbElement.petCount = petCount;
      sendnbVector.push_back(sendnbElement);
      
#ifdef MSG_DEFLATE_DEBUG
// debug      
      for (int k=0; k<linIndexContigBlockList.size(); k++){
        sprintf(msg, "linIndexContigBlockList[%d]: linIndex=%d, "
          "linIndexCount=%d", k, linIndexContigBlockList[k].linIndex,
          linIndexContigBlockList[k].linIndexCount);
ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
        
      }
#endif
      
#ifdef ASMMSTOREPRINT
      fprintf(asmmstoreprintfp, "gjt: sendnbElement localPet %d, dstPet %d, "
        "vectorLength=%d\n", localPet, dstPet, vectorLength);
      for (int k=0; k<linIndexContigBlockList.size(); k++){
        fprintf(asmmstoreprintfp, "linIndexContigBlockList[%d]: linIndex=%d, "
          "linIndexCount=%d\n", k, linIndexContigBlockList[k].linIndex,
          linIndexContigBlockList[k].linIndexCount);
      }
#endif
    } // for i - sendnbDiffPartnerDeCount
    // garbage collection
    delete [] sendnbPartnerDeList;
    delete [] sendnbPartnerDeCount;
  } // for j - srcLocalDeCount
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE7.0"));
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(t10);   //gjt - profile
#endif
  
  // --------------------------------------------------------------
  // recv and send patterns have been determined, ready to use them
  // --------------------------------------------------------------
  
  // sort recv and send vectors to lower communication contention
  // sorting also ensures correct ordering of sendnb and recvnb calls w/o tags
  sort(recvnbVector.begin(), recvnbVector.end());
  sort(sendnbVector.begin(), sendnbVector.end());
  
#define ASMMSTORECOMMMATRIX___disable
#ifdef ASMMSTORECOMMMATRIX
  // store the communication matrix in compact (sparse) distributed fashion,
  vector<int> *commMatrixDstPet        = new vector<int>(sendnbVector.size());
  vector<int> *commMatrixDstDataCount  = new vector<int>(sendnbVector.size());
  vector<int> *commMatrixSrcPet        = new vector<int>(recvnbVector.size());
  vector<int> *commMatrixSrcDataCount  = new vector<int>(recvnbVector.size());
  for (int i=0; i<sendnbVector.size(); i++){
    (*commMatrixDstPet)[i]       = sendnbVector[i].dstPet;
    (*commMatrixDstDataCount)[i] = sendnbVector[i].partnerDeDataCount;
  }
  for (int i=0; i<recvnbVector.size(); i++){
    (*commMatrixSrcPet)[i]       = recvnbVector[i].srcPet;
    (*commMatrixSrcDataCount)[i] = recvnbVector[i].partnerDeDataCount;
  }
  // attach the communication matrix to the routehandle
  localrc = (*routehandle)->setStorage(commMatrixDstPet, 1);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  localrc = (*routehandle)->setStorage(commMatrixDstDataCount, 2);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  localrc = (*routehandle)->setStorage(commMatrixSrcPet, 3);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  localrc = (*routehandle)->setStorage(commMatrixSrcDataCount, 4);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(t11);   //gjt - profile
#endif

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE8.0"));
#endif
  
  // store current XXE parameter in order to efficiently rewrite multiple times
  const int startCount = xxe->count;
  const int startStorageCount = xxe->storageCount;
  const int startCommhandleCount = xxe->commhandleCount;
  const int startXxeSubCount = xxe->xxeSubCount;
  const int startBufferInfoListSize = xxe->bufferInfoList.size();
  
  double dtMin;           // to find minimum time
  
#define ASMMSTOREOPTPRINT___disable
  
#ifdef ASMMSTOREOPTPRINT
  char asmmstoreoptprintfile[160];
  sprintf(asmmstoreoptprintfile, "asmmstoreoptprint.%05d", localPet);
  FILE *asmmstoreoptprintfp = fopen(asmmstoreoptprintfile, "a");
  fprintf(asmmstoreoptprintfp, "\n========================================"
    "========================================\n");
  fprintf(asmmstoreoptprintfp, "========================================"
    "========================================\n\n");
#endif
  
  // Need correct setting of vectorLength for the XXE exec() calls.
  int vectorLength = 0; // initialize
  
  // For the case that vectorLength is relevant, i.e. vectorFlag is set to
  // true, and the XXE stream contains elements that are vectorized,
  // srcTensorContigLength and dstTensorContigLength must be equivalent, and
  // define the vectorLength.
  if (vectorFlag)
    vectorLength = srcTensorContigLength; // consistent vectorLength
  
  //TODO: Implement a smarter optimization algorithm to optimize both
  //TODO: srcTermProcessing and pipelineDepth in a concurrent manner, rather
  //TODO: than the one-after-the-other approach below.
  
  // Optimize srcTermProcessing, finding srcTermProcessingOpt:
  int srcTermProcessingOpt; // optimium src term processing ... to be determined

#define FORCE_SRCTERMPROCESSING_off
#ifdef FORCE_SRCTERMPROCESSING_on
  int dummyVar = 0; // force to do all processing on the dst side
  if (srcTermProcessingArg && *srcTermProcessingArg < 0)
    *srcTermProcessingArg = dummyVar; // replace incoming value
  else
    srcTermProcessingArg = &dummyVar; // ignore incoming value
#endif
  
  if (srcTermProcessingArg && *srcTermProcessingArg >= 0){
    // use the provided srcTermProcessing
#ifdef SMMSTOREENCODEXXEINFO_on
    char msg[160];
    sprintf(msg, "srcTermProcessingArg = %d was provided -> do not tune",
      *srcTermProcessingArg);
    ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
#endif
    srcTermProcessingOpt = *srcTermProcessingArg;
  }else{
    // optimize srcTermProcessing
#ifdef SMMSTOREENCODEXXEINFO_on
    char msg[160];
    sprintf(msg, "srcTermProcessingArg was NOT provided -> tuning...");
    ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
#endif
#ifdef WORKAROUND_NONBLOCKPROGRESSBUG
    int pipelineDepth = 1;// only allow one outstanding connection as workaround
#else
    int pipelineDepth = petCount/2; // tricky to pick a good value here for all
                                    // cases (different interconnects!!!)
                                    // therefore need a concurrent opt scheme!!
#endif
    const int srcTermProcMax = 8; // 8 different values in the srcTermProcList[]
    const int srcTermProcList[] = {0, 1, 2, 3, 4, 6, 8, 20};  // trial settings
    for (int srcTermProc=0; srcTermProc<srcTermProcMax; srcTermProc++){
      int srcTermProcessing=srcTermProcList[srcTermProc];
      // start writing a fresh XXE stream
      xxe->clearReset(startCount, startStorageCount, startCommhandleCount,
        startXxeSubCount, startBufferInfoListSize);
      localrc = sparseMatMulStoreEncodeXXEStream(vm, recvnbVector, sendnbVector,
        srcTermProcessing, pipelineDepth, elementTK, valueTK, factorTK,
        dataSizeSrc, dataSizeDst, dataSizeFactors, srcLocalDeCount,
        dstLocalDeCount, dstLocalDeTotalElementCount, rraList, rraCount, 
        vectorLength, xxe);
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, &rc)) return rc;
#if 0  
      // optimize the XXE entire stream
      localrc = xxe->optimize();
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, &rc)) return rc;
#endif 
      // get XXE ready for execution
      localrc = xxe->execReady();
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, &rc)) return rc;
      // obtain timing
      double dtAverage = 0.;
      double dtStart, dtEnd;
      const int dtCount = 10;
      for (int i=0; i<dtCount; i++){
        vm->barrier();
        vm->wtime(&dtStart);
        localrc = xxe->exec(rraCount, rraList, &vectorLength,
          0x0|XXE::filterBitRegionTotalZero|XXE::filterBitNbTestFinish
          |XXE::filterBitCancel|XXE::filterBitNbWaitFinishSingleSum);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, &rc)) return rc;
        vm->barrier();
        vm->wtime(&dtEnd);
        dtAverage += dtEnd - dtStart;
      }
      dtAverage /= dtCount;
#ifdef ASMMSTOREOPTPRINT
      fprintf(asmmstoreoptprintfp, "localPet: %d, srcTermProcessing=%d -> dtAverage=%gs\n", 
        localPet, srcTermProcessing, dtAverage);
#endif
      // determine optimum srcTermProcessing  
      if (srcTermProcessing==0){
        // first time through -> initialize to find minimum
        dtMin = dtAverage;
        srcTermProcessingOpt = srcTermProcessing;
      }else{
        // compare to current minimum
        if (dtAverage < dtMin){
          // found better time
          dtMin = dtAverage;
          srcTermProcessingOpt = srcTermProcessing;
        }
      }
    } // srcTermProc
  
#ifdef ASMMSTOREOPTPRINT
    fprintf(asmmstoreoptprintfp, "localPet: %d, srcTermProcessingOpt=%d -> "
      "dtMin=%gs (local)\n", localPet, srcTermProcessingOpt, dtMin);
#endif
    
    // all PETs vote on srcTermProcessingOpt
    vector<int> srcTermProcessingOptList(petCount);
    vm->allgather(&srcTermProcessingOpt, &srcTermProcessingOptList[0],
      sizeof(int));
    sort(srcTermProcessingOptList.begin(), srcTermProcessingOptList.end());
    int votes = 1; // initialize
    int votesMax = 0; // initialize
    for (int i=1; i<petCount; i++){
      if (srcTermProcessingOptList[i-1] == srcTermProcessingOptList[i]){
        // same vote
        ++votes;
      }else{
        // different vote
        if (votes > votesMax){
          // new high vote found
          votesMax = votes;
          srcTermProcessingOpt = srcTermProcessingOptList[i-1];
        }
        votes = 1;
      }
    }
    // check last votes
    if (votes > votesMax){
      // new high vote found
      srcTermProcessingOpt = srcTermProcessingOptList[petCount-1];
    }
#ifdef SMMSTOREENCODEXXEINFO_on
    sprintf(msg, "... finished tuning, found srcTermProcessingOpt = %d",
      srcTermProcessingOpt);
    ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
#endif
    if (srcTermProcessingArg) *srcTermProcessingArg = srcTermProcessingOpt;
  
  } // finished finding srcTermProcessingOpt

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE9.0"));
#endif
  
#ifdef ASMMSTOREOPTPRINT
  fprintf(asmmstoreoptprintfp, "localPet: %d, srcTermProcessingOpt=%d "
    "(majority vote)\n", localPet, srcTermProcessingOpt);
#endif
    
#ifdef ASMMSTORETIMING
  VMK::wtime(t12);   //gjt - profile
#endif

  // Optimize pipelineDepth, finding pipelineDepthOpt:
  int pipelineDepthOpt;     // optimium pipeline depth ... to be determined

#define FORCE_PIPELINEDEPTH_off
#if (defined FORCE_PIPELINEDEPTH_on && !defined WORKAROUND_NONBLOCKPROGRESSBUG)
  int dummyVar2 = petCount; // force pipeline depth to be "petCount" deep
  if (pipelineDepthArg && *pipelineDepthArg < 0)
    *pipelineDepthArg = dummyVar2; // replace incoming value
  else
    pipelineDepthArg = &dummyVar2; // ignore incoming value
#endif

#ifdef WORKAROUND_NONBLOCKPROGRESSBUG
  int dummyVar2 = 1; // force pipeline depth to be only "1" deep as workaround
  if (pipelineDepthArg && *pipelineDepthArg < 0)
    *pipelineDepthArg = dummyVar2; // replace incoming value
  else
    pipelineDepthArg = &dummyVar2; // ignore incoming value
#endif

  if (pipelineDepthArg && *pipelineDepthArg >= 0){
    // use the provided pipelineDepthArg
#ifdef SMMSTOREENCODEXXEINFO_on
    char msg[160];
    sprintf(msg, "pipelineDepthArg = %d was provided -> do not tune",
      *pipelineDepthArg);
    ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
#endif
    pipelineDepthOpt = *pipelineDepthArg;
  }else{
    // optimize pipeline depth
#ifdef SMMSTOREENCODEXXEINFO_on
    char msg[160];
    sprintf(msg, "pipelineDepthArg was NOT provided -> tuning...");
    ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
#endif
    for (int pipelineDepth=1; pipelineDepth<=petCount; pipelineDepth*=2){
      // start writing a fresh XXE stream
      xxe->clearReset(startCount, startStorageCount, startCommhandleCount,
        startXxeSubCount, startBufferInfoListSize);
      localrc = sparseMatMulStoreEncodeXXEStream(vm, recvnbVector, sendnbVector,
        srcTermProcessingOpt, pipelineDepth, elementTK, valueTK, factorTK,
        dataSizeSrc, dataSizeDst, dataSizeFactors, srcLocalDeCount,
        dstLocalDeCount, dstLocalDeTotalElementCount, rraList, rraCount, 
        vectorLength, xxe);
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, &rc)) return rc;
#if 0  
      // optimize the XXE entire stream
      localrc = xxe->optimize();
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, &rc)) return rc;
#endif 
      // get XXE ready for execution
      localrc = xxe->execReady();
      if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
        ESMC_CONTEXT, &rc)) return rc;
      // obtain timing
      double dtAverage = 0.;
      double dtStart, dtEnd;
      const int dtCount = 10;
      for (int i=0; i<dtCount; i++){
        vm->barrier();
        vm->wtime(&dtStart);
          localrc = xxe->exec(rraCount, rraList, &vectorLength,
            0x0|XXE::filterBitRegionTotalZero|XXE::filterBitNbTestFinish
            |XXE::filterBitCancel|XXE::filterBitNbWaitFinishSingleSum);
          if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
            ESMC_CONTEXT, &rc)) return rc;
        vm->barrier();
        vm->wtime(&dtEnd);
        dtAverage += dtEnd - dtStart;
      }
      dtAverage /= dtCount;
#ifdef ASMMSTOREOPTPRINT
      fprintf(asmmstoreoptprintfp, "localPet: %d, pipelineDepth=%d -> "
        "dtAverage=%gs\n", localPet, pipelineDepth, dtAverage);
#endif
      // determine optimum pipelineDepth  
      if (pipelineDepth==1){
        // first time through -> initialize to find minimum
        dtMin = dtAverage;
        pipelineDepthOpt = pipelineDepth;
      }else{
        // compare to current minimum
        if (dtAverage < dtMin){
          // found better time
          dtMin = dtAverage;
          pipelineDepthOpt = pipelineDepth;
        }
      }
    } // pipelineDepth
    
#ifdef ASMMSTOREOPTPRINT
    fprintf(asmmstoreoptprintfp, "localPet: %d, pipelineDepthOpt=%d -> "
      "dtMin=%gs (local)\n", localPet, pipelineDepthOpt, dtMin);
#endif
      
    // all PETs vote on pipelineDepthOpt
    vector<int> pipelineDepthOptList(petCount);
    vm->allgather(&pipelineDepthOpt, &pipelineDepthOptList[0], sizeof(int));
    sort(pipelineDepthOptList.begin(), pipelineDepthOptList.end());
    int votes = 1; // initialize
    int votesMax = 0; // initialize
    for (int i=1; i<petCount; i++){
      if (pipelineDepthOptList[i-1] == pipelineDepthOptList[i]){
        // same vote
        ++votes;
      }else{
        // different vote
        if (votes > votesMax){
          // new high vote found
          votesMax = votes;
          pipelineDepthOpt = pipelineDepthOptList[i-1];
        }
        votes = 1;
      }
    }
    // check last votes
    if (votes > votesMax){
      // new high vote found
      pipelineDepthOpt = pipelineDepthOptList[petCount-1];
    }
#ifdef SMMSTOREENCODEXXEINFO_on
    sprintf(msg, "... finished tuning, found pipelineDepthOpt = %d",
      pipelineDepthOpt);
    ESMC_LogDefault.Write(msg, ESMC_LOGMSG_INFO);
#endif  
    if (pipelineDepthArg) *pipelineDepthArg = pipelineDepthOpt;
  
  } // finished finding pipelineDepthOpt

#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE10.0"));
#endif

#ifdef ASMMSTOREOPTPRINT
  fprintf(asmmstoreoptprintfp, "localPet: %d, pipelineDepthOpt=%d "
    "(majority vote)\n", localPet, pipelineDepthOpt);
  fclose(asmmstoreoptprintfp);
#endif
      
#ifdef ASMMSTORETIMING
  VMK::wtime(t13);   //gjt - profile
#endif

  // encode with the majority voted pipelineDepthOpt
  xxe->clearReset(startCount, startStorageCount, startCommhandleCount,
    startXxeSubCount, startBufferInfoListSize);
  localrc = sparseMatMulStoreEncodeXXEStream(vm, recvnbVector, sendnbVector,
    srcTermProcessingOpt, pipelineDepthOpt, elementTK, valueTK, factorTK,
    dataSizeSrc, dataSizeDst, dataSizeFactors, srcLocalDeCount,
    dstLocalDeCount, dstLocalDeTotalElementCount, rraList, rraCount,
    vectorLength, xxe);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  
#if 0  
  // optimize the XXE entire stream
  localrc = xxe->optimize();
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
#endif 
    
  // get XXE ready for execution
  localrc = xxe->execReady();
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  
  // execute final XXE stream for consistent profile data
  vm->barrier();  // ensure all PETs are present before profile run
  localrc = xxe->exec(rraCount, rraList, &vectorLength, 
    0x0|XXE::filterBitRegionTotalZero|XXE::filterBitNbTestFinish
    |XXE::filterBitCancel|XXE::filterBitNbWaitFinishSingleSum);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  
#ifdef ASMMSTOREMEMLOG_on
  VM::logMemInfo(std::string("ASMMStoreEncodeXXE11.0"));
#endif

#ifdef ASMMSTORETIMING
  VMK::wtime(t14);   //gjt - profile
#endif

//---DEBUG-------------------
//char file[160];
//sprintf(file, "xxe.%05d", localPet);
//FILE *fp = fopen(file, "a");
//xxe->print(fp);
//fclose(fp);
//---DEBUG-------------------

  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::sparseMatMulStoreEncodeXXEStream()"
//BOPI
// !IROUTINE:  ESMCI::sparseMatMulStoreEncodeXXEStream
//
// !INTERFACE:
int sparseMatMulStoreEncodeXXEStream(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  VM *vm,                                 // in
  vector<ArrayHelper::RecvnbElement> &recvnbVector, // in - exchange pattern
  vector<ArrayHelper::SendnbElement> &sendnbVector, // in - exchange pattern
  int srcTermProcessing,                  // in
  int pipelineDepth,                      // in
  XXE::TKId elementTK,                    // in
  XXE::TKId valueTK,                      // in
  XXE::TKId factorTK,                     // in
  int dataSizeSrc,                        // in
  int dataSizeDst,                        // in
  int dataSizeFactors,                    // in
  int srcLocalDeCount,                    // in
  int dstLocalDeCount,                    // in
  const int *dstLocalDeTotalElementCount, // in
  char **rraList,                         // in
  int rraCount,                           // in
  int vectorLength,                       // in
  XXE *xxe                                // inout - XXE stream
  ){
//
// !DESCRIPTION:
//    Encode a pipelined XXE stream with the specified degree of
//    srcTermProcessing for the sparseMatMul exchange pattern defined by
//    recvnbVector and sendnbVector.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  int localPet = vm->getLocalPet();
  int petCount = vm->getPetCount();
    
  try{

#ifdef ASMMPROFILE
    localrc = xxe->appendWtimer(0x0, "Wtimer 0", xxe->count, xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#endif
  
    // use pipeline approach
    vector<ArrayHelper::RecvnbElement>::iterator pRecv    =recvnbVector.begin();
    vector<ArrayHelper::RecvnbElement>::iterator pRecvWait=recvnbVector.begin();
    vector<ArrayHelper::SendnbElement>::iterator pSend    =sendnbVector.begin();
    vector<ArrayHelper::SendnbElement>::iterator pSendWait=sendnbVector.begin();
    
    
#define SMM_NONBLOCKINGSTYLE___disable
#define SMM_NONBLOCKINGSTYLE
    
#ifdef SMM_NONBLOCKINGSTYLE
    // prepare pipeline
#ifdef OLDSTYLEPIPELINEPREPARE
    for (int i=0; i<pipelineDepth; i++){
      if (pRecv != recvnbVector.end()){
        int k = pRecv - recvnbVector.begin();
        localrc = pRecv->appendRecvnb(xxe, 0x0|XXE::filterBitNbStart,
          srcTermProcessing, dataSizeSrc, k);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        ++pRecv;
      }
      if (pSend != sendnbVector.end()){
        int k = pSend - sendnbVector.begin();
        localrc = pSend->appendSendnb(xxe, 0x0|XXE::filterBitNbStart,
          srcTermProcessing, elementTK, valueTK, factorTK, dataSizeSrc, rraList,
          rraCount, k);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        ++pSend;
      }
    }
#else
    // fill in recvnb's first
    for (int i=0; i<pipelineDepth; i++){
      if (pRecv != recvnbVector.end()){
        int k = pRecv - recvnbVector.begin();
        localrc = pRecv->appendRecvnb(xxe, 0x0|XXE::filterBitNbStart,
          srcTermProcessing, dataSizeSrc, k);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        ++pRecv;
      }
    }
    // fill in the sendnb's next
    for (int i=0; i<pipelineDepth; i++){
      if (pSend != sendnbVector.end()){
        int k = pSend - sendnbVector.begin();
        localrc = pSend->appendSendnb(xxe, 0x0|XXE::filterBitNbStart,
          srcTermProcessing, elementTK, valueTK, factorTK, dataSizeSrc, rraList,
          rraCount, k);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        ++pSend;
      }
    }
#endif
#endif
    
    // append predicated zero operations for the total region
#ifdef ASMMPROFILE
    localrc = xxe->appendWtimer(
      XXE::filterBitRegionTotalZero|XXE::filterBitNbStart, "total zero",
      xxe->count, xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#endif
    for (int i=0; i<dstLocalDeCount; i++){
      int byteCount = dstLocalDeTotalElementCount[i] * dataSizeDst; // init
      bool vectorFlag = false;  // init
      if (vectorLength){
        // vectorLength is consistent between src and dst Array -> can be used
        byteCount /= vectorLength;  // scale with store time vectorLength
        vectorFlag = true;
      }
      localrc = xxe->appendZeroMemsetRRA(
        XXE::filterBitRegionTotalZero|XXE::filterBitNbStart,
        byteCount, srcLocalDeCount + i, vectorFlag);
      if (ESMC_LogDefault.MsgFoundError(localrc, 
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    }
#ifdef ASMMPROFILE
    localrc = xxe->appendWtimer(
      XXE::filterBitRegionTotalZero|XXE::filterBitNbStart,
      "/total zero", xxe->count, xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#endif
  
    // append predicated zero operations for targeted dst elements
    for(vector<ArrayHelper::RecvnbElement>::iterator pzero=recvnbVector.begin();
      pzero != recvnbVector.end(); ++pzero){
      localrc = pzero->appendZeroSuperScalar(xxe,
        XXE::filterBitRegionSelectZero|XXE::filterBitNbStart, srcLocalDeCount,
        elementTK);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    }
    
#ifdef SMM_NONBLOCKINGSTYLE
    // fill pipeline
    bool recvnbOK = true; // initialize
    bool sendnbOK = true; // initialize
    while ((pRecv != recvnbVector.end()) || (pSend != sendnbVector.end())){
      if ((pRecv != recvnbVector.end()) && recvnbOK){
        int k = pRecv - recvnbVector.begin();
        localrc = pRecv->appendRecvnb(xxe, 0x0|XXE::filterBitNbStart,
          srcTermProcessing, dataSizeSrc, k);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        ++pRecv;
      }
      if ((pSend != sendnbVector.end()) && sendnbOK){
        int k = pSend - sendnbVector.begin();
        localrc = pSend->appendSendnb(xxe, 0x0|XXE::filterBitNbStart,
          srcTermProcessing, elementTK, valueTK, factorTK, dataSizeSrc, rraList,
          rraCount, k);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        ++pSend;
      }
      if (pRecvWait != recvnbVector.end()){
        // prevent deadlock by staging waits correctly
        int recvnbStage = (localPet - pRecvWait->srcPet + petCount) % petCount;
        int sendnbStage = recvnbStage + 1;  // initialize to allow wait
        if (pSend != sendnbVector.end())
          sendnbStage = (pSend->dstPet - localPet + petCount) % petCount;//actu.
        if (recvnbStage < sendnbStage){
          // wait will not cause deadlock in the staggered Pet pattern
          int k = pRecvWait-recvnbVector.begin();
          // append test and wait calls that trigger productSum
          localrc = pRecvWait->appendTestWaitProductSum(xxe,
            0x0, srcTermProcessing,
            srcLocalDeCount, elementTK, valueTK, factorTK, dataSizeDst,
            dataSizeSrc, dataSizeFactors, rraList, rraCount, k);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          // append simple wait if summing to be done in a single sum at the end
          localrc = xxe->appendWaitOnIndex(
            0x0|XXE::filterBitNbWaitFinishSingleSum, pRecvWait->recvnbIndex);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          ++pRecvWait;
          recvnbOK = true;  // o.k. to post next recvnb call next iteration
        }else if (recvnbStage == sendnbStage)
          recvnbOK = true;  // this prevents multi DE/PET case from hanging
        else
          recvnbOK = false; // not o.k. to post next recvnb call next iteration
      }
      if (pSendWait != sendnbVector.end()){
        // prevent deadlock by staging waits correctly
        int sendnbStage = (pSendWait->dstPet - localPet + petCount) % petCount;
        int recvnbStage = sendnbStage + 1;  // initialize to allow wait
        if (pRecv != recvnbVector.end())
          recvnbStage = (localPet - pRecv->srcPet + petCount) % petCount;//actu.
        if (sendnbStage < recvnbStage){
          // wait will not cause deadlock in the staggered Pet pattern
          localrc = xxe->appendTestOnIndex(0x0|XXE::filterBitNbTestFinish,
            pSendWait->sendnbIndex);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          localrc = xxe->appendWaitOnIndex(0x0|XXE::filterBitNbWaitFinish,
            pSendWait->sendnbIndex);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          localrc = xxe->appendCancelIndex(0x0|XXE::filterBitCancel,
            pSendWait->sendnbIndex);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
          char *tempString = new char[160];
          sprintf(tempString, "done WaitOnIndex: %d",
            pSendWait->sendnbIndex);
          localrc = xxe->appendWtimer(0x0|XXE::filterBitNbWaitFinish,
            tempString, xxe->count, xxe->count);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          delete [] tempString;
#endif
          ++pSendWait;
          sendnbOK = true;  // o.k. to post next sendnb call next iteration
        }else if (sendnbStage == recvnbStage)
          sendnbOK = true;  // this prevents multi DE/PET case from hanging
        else
          sendnbOK = false; // not o.k. to post next sendnb call next iteration
      }
    }
    
    // drain pipeline
    while ((pRecvWait!=recvnbVector.end()) || (pSendWait!=sendnbVector.end())){
      if (pRecvWait != recvnbVector.end()){
        // prevent deadlock by staging waits correctly
        int recvnbStage = (localPet - pRecvWait->srcPet + petCount) % petCount;
        int sendnbStage = recvnbStage + 1;  // initialize to allow wait
        if (pSend != sendnbVector.end())
          sendnbStage = (pSend->dstPet - localPet + petCount) % petCount;//actu.
        if (recvnbStage < sendnbStage){
          // wait will not cause deadlock in the staggered Pet pattern
          int k = pRecvWait-recvnbVector.begin();
          // append test and wait calls that trigger productSum
          localrc = pRecvWait->appendTestWaitProductSum(xxe,
            0x0, srcTermProcessing, srcLocalDeCount,
            elementTK, valueTK, factorTK, dataSizeDst, dataSizeSrc,
            dataSizeFactors, rraList, rraCount, k);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          // append simple wait if summing to be done in a single sum at the end
          localrc = xxe->appendWaitOnIndex(
            0x0|XXE::filterBitNbWaitFinishSingleSum, pRecvWait->recvnbIndex);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          ++pRecvWait;
        }
      }
      if (pSendWait != sendnbVector.end()){
        // prevent deadlock by staging waits correctly
        int sendnbStage = (pSendWait->dstPet - localPet + petCount) % petCount;
        int recvnbStage = sendnbStage + 1;  // initialize to allow wait
        if (pRecv != recvnbVector.end())
          recvnbStage = (localPet - pRecv->srcPet + petCount) % petCount;//actu.
        if (sendnbStage < recvnbStage){
          // wait will not cause deadlock in the staggered Pet pattern
          localrc = xxe->appendTestOnIndex(0x0|XXE::filterBitNbTestFinish,
            pSendWait->sendnbIndex);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          localrc = xxe->appendWaitOnIndex(0x0|XXE::filterBitNbWaitFinish,
            pSendWait->sendnbIndex);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          localrc = xxe->appendCancelIndex(0x0|XXE::filterBitCancel,
            pSendWait->sendnbIndex);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#ifdef ASMMPROFILE
          char *tempString = new char[160];
          sprintf(tempString, "done WaitOnIndex: %d",
            pSendWait->sendnbIndex);
          localrc = xxe->appendWtimer(0x0|XXE::filterBitNbWaitFinish,
            tempString, xxe->count, xxe->count);
          if (ESMC_LogDefault.MsgFoundError(localrc,
            ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
          delete [] tempString;
#endif
          ++pSendWait;
        }
      }
    }
    // alternatively could post all XXE::waitOnAllSendnb at the end
    // -> not sure what gives better performance.
    //  localrc = xxe->appendWaitOnAllSendnb(0x0);
    //  if (ESMC_LogDefault.MsgFoundError(localrc, 
    //    ESMCI_ERR_PASSTHRU, &rc)) return rc;
    
    // append a single productSum operation that considers _all_ of the 
    // incoming elements and orders them accroding the strict canonical 
    // TERMORDER_SRCSEQ order
    localrc = ArrayHelper::RecvnbElement::appendSingleProductSum(xxe,
      0x0|XXE::filterBitNbWaitFinishSingleSum, srcTermProcessing,
      srcLocalDeCount, elementTK, valueTK, factorTK, dataSizeDst,
      dataSizeSrc, dataSizeFactors, rraList, rraCount, recvnbVector);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
    
#else
    
    // use BLOCKING comms instead!!!!!!!!
    // note that this will produce a routehandle that cannot be executed 
    // in non-blocking style.
    // TODO: the BLOCKING option should really be added with a special 
    // TODO: pedication bit set so it can be executed for blocking user calls,
    // TODO: of course only if it has been determined as being faster by the
    // TODO: tuning phase over the non-blocking couter part implementation.
    
    while ((pRecv != recvnbVector.end()) || (pSend != sendnbVector.end())){
      int recvStage = -1; // invalidate
      if (pRecv != recvnbVector.end()){
        recvStage = (localPet - pRecv->srcPet + petCount) % petCount;
      }
      int sendStage = -1; // invalidate
      if (pSend != sendnbVector.end()){
        sendStage = (pSend->dstPet - localPet + petCount) % petCount;
      }
      // deal with invalid stages here (only one can be invalid!!!)
      if (recvStage == -1)
        recvStage = sendStage + 1;  // allow the send to post
      else if (sendStage == -1)
        sendStage = recvStage + 1;  // allow the recv to post
      // ensure to schedule sends and receives correctly
      if (sendStage == recvStage){
        int kSend = pSend - sendnbVector.begin();
        int kRecv = pRecv - recvnbVector.begin();
        localrc = pSend->appendSendRecv(xxe, 0x0|XXE::filterBitNbStart,
          srcTermProcessing, elementTK, valueTK, factorTK, dataSizeSrc, rraList,
          rraCount, kSend, pRecv, kRecv);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        ++pSend;
        ++pRecv;
      }
      if (sendStage < recvStage){
        int k = pSend - sendnbVector.begin();
        localrc = pSend->appendSend(xxe, 0x0|XXE::filterBitNbStart,
          srcTermProcessing, elementTK, valueTK, factorTK, dataSizeSrc, rraList,
          rraCount, k);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        ++pSend;        
      }
      if (recvStage < sendStage){
        int k = pRecv - recvnbVector.begin();
        localrc = pRecv->appendRecv(xxe, 0x0|XXE::filterBitNbStart,
          srcTermProcessing, dataSizeSrc, k);
        if (ESMC_LogDefault.MsgFoundError(localrc,
          ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
        ++pRecv;        
      }
    }
    
    // here all blk. comms are done ... -> simple ProductSum() elements follow.
    
    while (pRecvWait!=recvnbVector.end()){
      localrc = pRecvWait->appendProductSum(xxe,
        0x0, srcTermProcessing, srcLocalDeCount,
        elementTK, valueTK, factorTK, dataSizeDst, dataSizeSrc,
        dataSizeFactors, rraList, rraCount);
      if (ESMC_LogDefault.MsgFoundError(localrc,
        ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
      ++pRecvWait;
    }
        
#endif
          
#ifdef ASMMPROFILE
    localrc = xxe->appendWtimer(0x0, "Wtimer End", xxe->count, xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
      ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#endif
      
#ifdef ASMMPROFILE
    localrc = xxe->appendWtimer(0x0, "Wtimer End2", xxe->count, xxe->count);
    if (ESMC_LogDefault.MsgFoundError(localrc,
    ESMCI_ERR_PASSTHRU, ESMC_CONTEXT, &rc)) return rc;
#endif
    
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::sparseMatMul()"
//BOPI
// !IROUTINE:  ESMCI::Array::sparseMatMul
//
// !INTERFACE:
int Array::sparseMatMul(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  Array *srcArray,                      // in    - source Array
  Array *dstArray,                      // inout - destination Array
  RouteHandle **routehandle,            // inout - handle to precomputed comm
  ESMC_CommFlag commflag,               // in    - communication options
  bool *finishedflag,                   // out   - TEST ops finished or not
  bool *cancelledflag,                  // out   - any cancelled operations
  ESMC_Region_Flag zeroflag,            // in    - ESMC_REGION_TOTAL:
                                        //          -> zero out total region
                                        //         ESMC_REGION_SELECT:
                                        //          -> zero out target points
                                        //         ESMC_REGION_EMPTY:
                                        //          -> don't zero out any points
  ESMC_TermOrder_Flag termorderflag,    // in    - ESMC_TERMORDER_SRCSEQ:
                                        //          -> strict srcSeqInd order
                                        //         ESMC_TERMORDER_SRCPET:
                                        //          -> order by src PET & seqInd
                                        //         ESMC_TERMORDER_FREE:
                                        //          -> free order
  bool checkflag,                       // in    - false: (def.) basic chcks
                                        //         true:  full input check
  bool haloFlag                         // in    - support halo conditions
  ){
//
// !DESCRIPTION:
//    Execute an Array sparse matrix multiplication operation
//
//EOPI
//-----------------------------------------------------------------------------
#define SMMINFO____disable
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  try{

#define ASMMTIMING___disable
    
#ifdef ASMMTIMING
    double t0, t1, t2, t3, t4, t5, t6, t7;            //gjt - profile
    VMK::wtime(&t0);      //gjt - profile
#endif    

  // basic input checking
  bool srcArrayFlag = false;
  if (srcArray != ESMC_NULL_POINTER) srcArrayFlag = true;
  bool dstArrayFlag = false;
  if (dstArray != ESMC_NULL_POINTER) dstArrayFlag = true;
  
  // srcArray and dstArray may not point to the identical Array object
  if (!haloFlag && (srcArrayFlag && dstArrayFlag)){
    if (srcArray == dstArray){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- srcArray and dstArray must not be identical", ESMC_CONTEXT, &rc);
      return rc;
    }
  }
  
#ifdef ASMMTIMING
  VMK::wtime(&t1);      //gjt - profile
#endif
  
  // get a handle on the XXE stored in routehandle
  XXE *xxe = (XXE *)(*routehandle)->getStorage();

  if (xxe == NULL){
    // NOP
    // return successfully
    rc = ESMF_SUCCESS;
    return rc;
  }

  // conditionally perform full input checks
  if (checkflag){
    // check that srcArray's typekind matches
    if (srcArrayFlag && (xxe->typekind[1] != srcArray->getTypekind())){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
        "- TypeKind mismatch between srcArray argument and precomputed XXE",
        ESMC_CONTEXT, &rc);
      return rc;
    }
    // check that dstArray's typekind matches
    if (dstArrayFlag && (xxe->typekind[2] != dstArray->getTypekind())){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP,
        "- TypeKind mismatch between dstArray argument and precomputed XXE",
        ESMC_CONTEXT, &rc);
      return rc;
    }
    //TODO: need to check DistGrid-conformance and congruence of local tiles
    //      between argument Array pair and Array pair used during XXE precomp.
    // use Array::match() method to perform this check. This however, would
    // require that a reference is kept inside the routehandle to the Array
    // pair used during store(). not sure how good an idea that is. The 
    // alternative is to keep a "fingerprint" of the Arrays that's just enough
    // info to perform the congurence check here.
  }
  
  // deal with finishedflag argument which can be NULL, but is also needed
  // locally -> therefore ensure that finishedflag locally is always valid
  bool finishedflagLocal;
  if (!finishedflag) finishedflag = &finishedflagLocal;
  
#ifdef ASMMTIMING
  VMK::wtime(&t2);      //gjt - profile
#endif
  
  // prepare for relative run-time addressing (RRA)
  int rraCount = 0; // init
  if (srcArrayFlag)
    rraCount += srcArray->delayout->getLocalDeCount();
  if (dstArrayFlag)
    rraCount += dstArray->delayout->getLocalDeCount();
  char **rraList = new char*[rraCount];
  char **rraListPtr = rraList;
  if (srcArrayFlag){
    memcpy((char *)rraListPtr, srcArray->larrayBaseAddrList,
      srcArray->delayout->getLocalDeCount() * sizeof(char *));
    rraListPtr += srcArray->delayout->getLocalDeCount();
  }
  if (dstArrayFlag)
    memcpy((char *)rraListPtr, dstArray->larrayBaseAddrList,
      dstArray->delayout->getLocalDeCount() * sizeof(char *));

#ifdef ASMMTIMING
  VMK::wtime(&t3);      //gjt - profile
#endif
  
  // set filterBitField  
  int filterBitField = 0x0; // init. to execute _all_ operations in XXE stream
  
  if (commflag==ESMF_COMM_BLOCKING){
    // blocking mode
    if (termorderflag == ESMC_TERMORDER_SRCSEQ){
      filterBitField |= XXE::filterBitNbWaitFinish; // set NbWaitFinish filter
      filterBitField |= XXE::filterBitNbTestFinish; // set NbTestFinish filter
      filterBitField |= XXE::filterBitCancel;       // set Cancel filter
#ifdef SMMINFO
      ESMC_LogDefault.Write("SMM exec: COMM_BLOCKING, TERMORDER_SRCSEQ",
        ESMC_LOGMSG_INFO);
#endif
    }else if (termorderflag == ESMC_TERMORDER_SRCPET){
      filterBitField |= XXE::filterBitNbTestFinish; // set NbTestFinish filter
      filterBitField |= XXE::filterBitCancel;       // set Cancel filter
      filterBitField |= XXE::filterBitNbWaitFinishSingleSum; // SingleSum filter
#ifdef SMMINFO
      ESMC_LogDefault.Write("SMM exec: COMM_BLOCKING, TERMORDER_SRCPET",
        ESMC_LOGMSG_INFO);
#endif
    }else if (termorderflag == ESMC_TERMORDER_FREE){
      filterBitField |= XXE::filterBitNbWaitFinish; // set NbWaitFinish filter
      filterBitField |= XXE::filterBitCancel;       // set Cancel filter    
      filterBitField |= XXE::filterBitNbWaitFinishSingleSum; // SingleSum filter
#ifdef SMMINFO
      ESMC_LogDefault.Write("SMM exec: COMM_BLOCKING, TERMORDER_FREE",
        ESMC_LOGMSG_INFO);
#endif
    }else{
      ESMC_LogDefault.MsgFoundError(ESMC_RC_NOT_IMPL, 
        "- termorderflag choice not supported under COMM_BLOCKING",
        ESMC_CONTEXT, &rc);
      return rc;  // bail out
    }
  }else if (commflag==ESMF_COMM_NBSTART){
    // non-blocking start -> independent of termorderflag
    filterBitField |= XXE::filterBitNbWaitFinish;     // set NbWaitFinish filter
    filterBitField |= XXE::filterBitNbTestFinish;     // set NbTestFinish filter
    filterBitField |= XXE::filterBitCancel;           // set Cancel filter
    filterBitField |= XXE::filterBitNbWaitFinishSingleSum; // SingleSum filter
#ifdef SMMINFO
    ESMC_LogDefault.Write("SMM exec: COMM_NBSTART",
      ESMC_LOGMSG_INFO);
#endif
  }else if(commflag==ESMF_COMM_NBTESTFINISH){
    // non-blocking test and finish
    //TODO: implement the other TERMORDER options
    if (termorderflag == ESMC_TERMORDER_FREE){
      filterBitField |= XXE::filterBitNbStart;          // set NbStart filter
      filterBitField |= XXE::filterBitNbWaitFinish;     // set NbWaitFinish filter
      filterBitField |= XXE::filterBitCancel;           // set Cancel filter
      filterBitField |= XXE::filterBitNbWaitFinishSingleSum; // SingleSum filter
#ifdef SMMINFO
      ESMC_LogDefault.Write("SMM exec: COMM_NBTESTFINISH, TERMORDER_FREE",
        ESMC_LOGMSG_INFO);
#endif
    }else{
      ESMC_LogDefault.MsgFoundError(ESMC_RC_NOT_IMPL, 
        "- termorderflag choice not supported under COMM_NBTESTFINISH",
        ESMC_CONTEXT, &rc);
      return rc;  // bail out
    }
  }else if(commflag==ESMF_COMM_NBWAITFINISH){
    // non-blocking wait and finish
    //TODO: implement the other TERMORDER options
    if (termorderflag == ESMC_TERMORDER_FREE){
      filterBitField |= XXE::filterBitNbStart;          // set NbStart filter
      filterBitField |= XXE::filterBitNbTestFinish;     // set NbTestFinish filter
      filterBitField |= XXE::filterBitCancel;           // set Cancel filter
      filterBitField |= XXE::filterBitNbWaitFinishSingleSum; // SingleSum filter
#ifdef SMMINFO
      ESMC_LogDefault.Write("SMM exec: COMM_NBWAITFINISH TERMORDER_FREE",
        ESMC_LOGMSG_INFO);
#endif
    }else{
      ESMC_LogDefault.MsgFoundError(ESMC_RC_NOT_IMPL, 
        "- termorderflag choice not supported under COMM_NBWAITFINISH",
        ESMC_CONTEXT, &rc);
      return rc;  // bail out
    }
  }else if(commflag==ESMF_COMM_CANCEL){
    // cancel
    filterBitField |= XXE::filterBitNbStart;          // set NbStart filter
    filterBitField |= XXE::filterBitNbWaitFinish;     // set NbWaitFinish filter
    filterBitField |= XXE::filterBitNbTestFinish;     // set NbTestFinish filter
    filterBitField |= XXE::filterBitNbWaitFinishSingleSum; // SingleSum filter
#ifdef SMMINFO
    ESMC_LogDefault.Write("SMM exec: COMM_CANCEL",
      ESMC_LOGMSG_INFO);
#endif
  }
  
  if (zeroflag!=ESMC_REGION_TOTAL)
    filterBitField |= XXE::filterBitRegionTotalZero;  // filter reg. total zero
  if (zeroflag!=ESMC_REGION_SELECT)
    filterBitField |= XXE::filterBitRegionSelectZero; // filter reg. select zero
  
#ifdef ASMMTIMING
  VMK::wtime(&t4);      //gjt - profile
#endif
  
  // set vectorLength
  int vectorLength = 0; // initialize
  
  // The vectorLength argument passed into xxe->exec() provides support
  // for flexible choice of vectorLength during execution time of the XXE
  // stream. Vectorization in the XXE stream is handled on a per operation
  // basis, i.e. it depends on each individual operation whether, and how the 
  // vectorLength argument is used.
  // On those PETs that don't call in with srcArray nor dstArray (unusual case,
  // but possible and supported!), the vectorLenght will be left at 0. In the
  // other cases (i.e. srcArray and/or dstArray are present) it is assumed that
  // the vectorLength can be determined from which ever Array is present (first
  // see about srcArray, and if not available then look at dstArray). This is
  // consistent because vectorization in the XXE stream is only meaningful under
  // the condition that the vectorLength determined from the srcArray equals
  // that determined from the dstArray. There is no check performed here on
  // whether srcArray and dstArray lead to the same vectorLength, because it
  // is not known here whether the XXE stream actually uses vectorization or
  // not. In the case that no vectorization is used there may well be
  // inconsistencies on this level, and the vectorLength passed into xxe->exec()
  // is bogus. However, the vectorLength is irrelevant under this condition, and
  // will be ignored by the XXE execution.

  if (srcArrayFlag){
    // use srcArray to determine vectorLength
    vectorLength = 1;  // prime
    int rank = srcArray->rank;
    for (int jj=0; jj<rank; jj++){
      if (srcArray->arrayToDistGridMap[jj])
        // decomposed dimension
        break;
      else
        // tensor dimension
        vectorLength *= srcArray->undistUBound[jj]
          - srcArray->undistLBound[jj] + 1;
    }
  }else if (dstArrayFlag){
    // use dstArray to determine vectorLength
    vectorLength = 1;  // prime
    int rank = dstArray->rank;
    for (int jj=0; jj<rank; jj++){
      if (dstArray->arrayToDistGridMap[jj])
        // decomposed dimension
        break;
      else
        // tensor dimension
        vectorLength *= dstArray->undistUBound[jj]
          - dstArray->undistLBound[jj] + 1;
    }
  }
  
#ifdef ASMMTIMING
  VMK::wtime(&t5);      //gjt - profile
#endif
  
  // execute XXE stream
  localrc = xxe->exec(rraCount, rraList, &vectorLength, filterBitField,
    finishedflag, cancelledflag);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;

  while (commflag==ESMF_COMM_BLOCKING && !(*finishedflag)){
    // must be a blocking call with TERMORDER_FREE -> free-order while
#ifdef SMMINFO
    ESMC_LogDefault.Write("...within free-order while",
      ESMC_LOGMSG_INFO);
#endif
    filterBitField = 0x0; // init. to execute _all_ operations in XXE stream
    // same as non-blocking test and finish
    filterBitField |= XXE::filterBitRegionTotalZero;  // filter reg. total zero
    filterBitField |= XXE::filterBitRegionSelectZero; // filter reg. select zero
    filterBitField |= XXE::filterBitNbStart;          // set NbStart filter
    filterBitField |= XXE::filterBitNbWaitFinish;     // set NbWaitFinish filter
    filterBitField |= XXE::filterBitCancel;           // set Cancel filter
    filterBitField |= XXE::filterBitNbWaitFinishSingleSum; // SingleSum filter
    localrc = xxe->exec(rraCount, rraList, &vectorLength, filterBitField,
      finishedflag, cancelledflag);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
  }
  
#ifdef ASMMTIMING
  VMK::wtime(&t6);      //gjt - profile
#endif
  
  // garbage collection
  delete [] rraList;
  
#ifdef ASMMTIMING
  VMK::wtime(&t7);      //gjt - profile
  VM *vm = VM::getCurrent(&localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    &rc)) return rc;
  int localPet = vm->getLocalPet();
  char file[160];
  sprintf(file, "asmmtiming.%05d", localPet);
  FILE *fp = fopen(file, "a");
  fprintf(fp, "\n=================================================="
    "==============================\n");
  fprintf(fp, "=================================================="
    "==============================\n\n");
  fprintf(fp, "gjt - exec() profile for PET %d:\n"
    "\tdt1 = %g\n"
    "\tdt2 = %g\n"
    "\tdt3 = %g\n"
    "\tdt4 = %g\n"
    "\tdt5 = %g\n"
    "\tdt6 = %g\n"
    "\tdt7 = %g\n",
    localPet, t1-t0, t2-t0, t3-t0, t4-t0, t5-t0, t6-t0, t7-t0);
  fclose(fp);
#endif
  
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::Array::sparseMatMulRelease()"
//BOPI
// !IROUTINE:  ESMCI::Array::sparseMatMulRelease
//
// !INTERFACE:
int Array::sparseMatMulRelease(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  RouteHandle *routehandle        // inout -
  ){
//
// !DESCRIPTION:
//    Release information for an Array sparse matrix multiplication operation
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code

  try{
  
    // get XXE from routehandle
    XXE *xxe = (XXE *)routehandle->getStorage();

    if (xxe == NULL){
      // NOP
      // return successfully
      rc = ESMF_SUCCESS;
      return rc;
    }
      
#define XXEPROFILEPRINT___disable
    
#ifdef XXEPROFILEPRINT
#ifdef ASMMPROFILE
    // print XXE stream profile
    VM *vm = VM::getCurrent(&localrc);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
    int localPet = vm->getLocalPet();
    int petCount = vm->getPetCount();
    char file[160];
    sprintf(file, "asmmprofile.%05d", localPet);
    FILE *fp = fopen(file, "a");
    fprintf(fp, "\n=================================================="
      "==============================\n");
    fprintf(fp, "=================================================="
      "==============================\n\n");
    for (int pet=0; pet<petCount; pet++){
      if (pet==localPet){
        localrc = xxe->printProfile(fp);
        if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU,
          ESMC_CONTEXT, &rc))
        return rc;
      }
      vm->barrier();
    }
    fclose(fp);
#endif
#endif
  
    // delete xxe
    delete xxe;

    // mark storage pointer in RouteHandle as invalid/NOP
    localrc = routehandle->setStorage(NULL);
    if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc)) return rc;
  
  }catch(int localrc){
    // catch standard ESMF return code
    ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
      &rc);
    return rc;
  }catch(...){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_INTNRL_BAD,
      "- Caught exception", ESMC_CONTEXT, &rc);
    return rc;
  }
  
  // return successfully
  rc = ESMF_SUCCESS;
  return rc;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayElement::ArrayElement()"
//BOPI
// !IROUTINE:  ESMCI::ArrayElement::ArrayElement
//
// !INTERFACE:
ArrayElement::ArrayElement(
//
// !RETURN VALUE:
//    ArrayElement*
//
// !ARGUMENTS:
//
  Array const *arrayArg,
  int localDeArg
  ){
//
// !DESCRIPTION:
//    Constructor of ArrayElement iterator through Array elements in exclusive
//    region.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  // check input arguments
  if (arrayArg == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- arrayArg must not be NULL", ESMC_CONTEXT, &rc);
    throw rc;  // bail out with exception
  }
  if (localDeArg < 0 || 
    localDeArg >= arrayArg->getDELayout()->getLocalDeCount()){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_OUTOFRANGE,
      "- localDeArg out of range", ESMC_CONTEXT, &rc);
    throw rc;  // bail out with exception
  }
  
  // set members
  array = arrayArg;
  localDe = localDeArg;
  int rank = array->getRank();
  indexTupleStart.resize(rank);
  indexTupleEnd.resize(rank);
  indexTuple.resize(rank);
  skipDim.resize(rank);
  indexTupleBlockStart.resize(rank);
  indexTupleBlockEnd.resize(rank);
  indexTupleWatchStart.resize(rank);
  indexTupleWatchEnd.resize(rank);
  
  // initialize tuple variables for iteration through Array elements within
  // exclusive region, with origin of exclusive region at tuple (0,0,..)
  int redDimCount = array->getRank() - array->getTensorCount();
  int iOff = localDe * redDimCount;
  int iPacked = 0;    // reset
  int iTensor = 0;    // reset
  for (int i=0; i<rank; i++){
    indexTupleStart[i] = indexTuple[i] = 0;   // reset
    skipDim[i] = false;                       // reset
    indexTupleBlockStart[i] = indexTupleBlockEnd[i] = 0;  // reset
    indexTupleWatchStart[i] = indexTupleWatchEnd[i] = 0;  // reset
    if (array->getArrayToDistGridMap()[i]){
      // decomposed dimension
      indexTupleEnd[i] = array->getExclusiveUBound()[iOff+iPacked]
        - array->getExclusiveLBound()[iOff+iPacked] + 1;
      ++iPacked;
    }else{
      // tensor dimension
      indexTupleEnd[i] = array->getUndistUBound()[iTensor]
        - array->getUndistLBound()[iTensor] + 1;
      ++iTensor;
    }   
  }
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayElement::ArrayElement()"
//BOPI
// !IROUTINE:  ESMCI::ArrayElement::ArrayElement
//
// !INTERFACE:
ArrayElement::ArrayElement(
//
// !RETURN VALUE:
//    ArrayElement*
//
// !ARGUMENTS:
//
  Array const *arrayArg,
  int localDeArg,
  bool blockExclusiveFlag   // in - block exclusive region if set to true
  ){
//
// !DESCRIPTION:
//    Constructor of ArrayElement iterator through Array elements in total
//    region, with the option to block the elements of the exclusive region.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  
  // check input arguments
  if (arrayArg == NULL){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
      "- arrayArg must not be NULL", ESMC_CONTEXT, &rc);
    throw rc;  // bail out with exception
  }
  if (localDeArg < 0 || 
    localDeArg >= arrayArg->getDELayout()->getLocalDeCount()){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_OUTOFRANGE,
      "- localDeArg out of range", ESMC_CONTEXT, &rc);
    throw rc;  // bail out with exception
  }
  
  // set members
  array = arrayArg;
  localDe = localDeArg;
  int rank = array->getRank();
  indexTupleStart.resize(rank);
  indexTupleEnd.resize(rank);
  indexTuple.resize(rank);
  skipDim.resize(rank);
  indexTupleBlockStart.resize(rank);
  indexTupleBlockEnd.resize(rank);
  indexTupleWatchStart.resize(rank);
  indexTupleWatchEnd.resize(rank);
  
  // initialize tuple variables for iteration through Array elements within
  // total region, with origin of exclusive region at tuple (0,0,..)
  int redDimCount = array->getRank() - array->getTensorCount();
  int iOff = localDe * redDimCount;
  int iPacked = 0;    // reset
  int iTensor = 0;    // reset
  for (int i=0; i<rank; i++){
    skipDim[i] = false;                       // reset
    indexTupleBlockStart[i] = indexTupleBlockEnd[i] = 0;  // reset
    indexTupleWatchStart[i] = indexTupleWatchEnd[i] = 0;  // reset
    if (array->getArrayToDistGridMap()[i]){
      // decomposed dimension
      indexTupleStart[i] = indexTuple[i] =
        array->getTotalLBound()[iOff+iPacked]
        - array->getExclusiveLBound()[iOff+iPacked];
      indexTupleEnd[i] = array->getTotalUBound()[iOff+iPacked]
        - array->getExclusiveLBound()[iOff+iPacked] + 1;
      indexTupleWatchEnd[i] = array->getExclusiveUBound()[iOff+iPacked]
        - array->getExclusiveLBound()[iOff+iPacked] + 1;
      ++iPacked;
    }else{
      // tensor dimension
      indexTupleStart[i] = indexTuple[i] = 0;
      indexTupleEnd[i] = array->getUndistUBound()[iTensor]
        - array->getUndistLBound()[iTensor] + 1;
      indexTupleWatchEnd[i] = array->getUndistUBound()[iTensor]
        - array->getUndistLBound()[iTensor] + 1;
      ++iTensor;
    }
    if (blockExclusiveFlag)
      indexTupleBlockEnd[i] =  indexTupleWatchEnd[i];
  }
  first();  // point indexTuple to the first element that is not blocked
  
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayElement::hasValidSeqIndex()"
//BOPI
// !IROUTINE:  ESMCI::ArrayElement::hasValidSeqIndex
//
// !INTERFACE:
bool ArrayElement::hasValidSeqIndex(
//
// !RETURN VALUE:
//    true or false
//
// !ARGUMENTS:
//
  )const{    
//
// !DESCRIPTION:
//    Indicate whether the ArrayElement is valid. Invalid elements are
//    those that are outside the exclusive region for discontiguous dimensions,
//    or arbitrarily decomposed dimensions.
//
//EOPI
//-----------------------------------------------------------------------------
  int rank = array->getRank();
  int de = array->getDELayout()->getLocalDeToDeMap()[localDe];
  int iOff = de * array->getDistGrid()->getDimCount();
  int iPacked = 0;    // reset
  for (int i=0; i<rank; i++){
    if (array->getArrayToDistGridMap()[i]){
      // decomposed dimension
      if (array->getDistGrid()->getContigFlagPDimPDe()[iOff+iPacked] == 0){
        // discontigous dimension -> check if within MultiDimIndexLoop block
        if (isWithinBlock(i) == false) return false;
      }
      int collocation = array->getDistGrid()->getCollocationPDim()[iPacked];
      if (array->getDistGrid()->getArbSeqIndexList(localDe, collocation)){
        // arbitrarily decomposed dimension
        // -> check if within MultiDimIndexLoop block
        if (isWithinBlock(i) == false) return false;
      }
      ++iPacked;
    }
  }
  return true;  // all dimensions are in valid range
}
//-----------------------------------------------------------------------------
    
    
//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayElement::getLinearIndexExclusive()"
//BOPI
// !IROUTINE:  ESMCI::ArrayElement::getLinearIndexExclusive
//
// !INTERFACE:
int ArrayElement::getLinearIndexExclusive(
//
// !RETURN VALUE:
//    linear index
//
// !ARGUMENTS:
//
  )const{    
//
// !DESCRIPTION:
//    Obtain linear index for ArrayElement
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  
  int linIndex = array->getLinearIndexExclusive(localDe, &indexTuple[0],
    &localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    NULL)) throw localrc;  // bail out with exception
  return linIndex;
}
//-----------------------------------------------------------------------------
    
    
//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayElement::getSequenceIndexExclusive()"
//BOPI
// !IROUTINE:  ESMCI::ArrayElement::getSequenceIndexExclusive
//
// !INTERFACE:
SeqIndex ArrayElement::getSequenceIndexExclusive(
//
// !RETURN VALUE:
//    sequence index
//
// !ARGUMENTS:
//
  int depth
  )const{    
//
// !DESCRIPTION:
//    Obtain sequence index for ArrayElement
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  
  SeqIndex seqIndex = array->getSequenceIndexExclusive(localDe, &indexTuple[0],
    depth, &localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    NULL)) throw localrc;  // bail out with exception
  return seqIndex;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayElement::getTensorSequenceIndex()"
//BOPI
// !IROUTINE:  ESMCI::ArrayElement::getTensorSequenceIndex
//
// !INTERFACE:
int ArrayElement::getTensorSequenceIndex(
//
// !RETURN VALUE:
//    linear index
//
// !ARGUMENTS:
//
  )const{    
//
// !DESCRIPTION:
//    Obtain linear index for ArrayElement
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  
  int tensorSeqIndex = array->getTensorSequenceIndex(&indexTuple[0], &localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    NULL)) throw localrc;  // bail out with exception
  return tensorSeqIndex;
}
//-----------------------------------------------------------------------------
    
    
//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayElement::getArbSequenceIndexOffset()"
//BOPI
// !IROUTINE:  ESMCI::ArrayElement::getArbSequenceIndexOffset
//
// !INTERFACE:
int ArrayElement::getArbSequenceIndexOffset(
//
// !RETURN VALUE:
//    linear index
//
// !ARGUMENTS:
//
  )const{    
//
// !DESCRIPTION:
//    Obtain linear index for ArrayElement
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  
  int arbSeqIndexOffset = array->getArbSequenceIndexOffset(&indexTuple[0],
    &localrc);
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    NULL)) throw localrc;  // bail out with exception
  return arbSeqIndexOffset;
}
//-----------------------------------------------------------------------------
    
    
//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::ArrayElement::print()"
//BOPI
// !IROUTINE:  ESMCI::ArrayElement::print
//
// !INTERFACE:
void ArrayElement::print(
//
// !ARGUMENTS:
//
  )const{    
//
// !DESCRIPTION:
//    Print internal information..
//
//EOPI
//-----------------------------------------------------------------------------
  cout << "array:   " << array << "  localDe: " << localDe << "\n";
  MultiDimIndexLoop::print();
}
//-----------------------------------------------------------------------------
    
    
//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMCI::SparseMatrix::SparseMatrix()"
//BOPI
// !IROUTINE:  ESMCI::SparseMatrix::SparseMatrix
//
// !INTERFACE:
SparseMatrix::SparseMatrix(
//
// !RETURN VALUE:
//    SparseMatrix*
//
// !ARGUMENTS:
//
  ESMC_TypeKind_Flag const typekind_,
  void const *factorList_,
  int const factorListCount_,
  int const srcN_,
  int const dstN_,
  int const *factorIndexList_
  ){
//
// !DESCRIPTION:
//    Constructor
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int rc = ESMC_RC_NOT_IMPL;              // final return code
  typekind = typekind_;
  factorList = factorList_;
  factorListCount = factorListCount_;
  srcN = srcN_;
  dstN = dstN_;
  factorIndexList = factorIndexList_;
  // check consistency
  if (factorListCount > 0){
    // must contain valid factorList and factorIndexList members
    if (factorList == NULL){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
        "- Not a valid pointer to factorList array", ESMC_CONTEXT, &rc);
      throw rc;  // bail out with exception
    }
    if (factorIndexList == NULL){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_PTR_NULL,
        "- Not a valid pointer to factorIndexList array", ESMC_CONTEXT, &rc);
      throw rc;  // bail out with exception
    }
    // must contain a valid typekind
    if (typekind == ESMF_NOKIND){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD,
        "- must specify valid typekind", ESMC_CONTEXT, &rc);
      throw rc;  // bail out with exception
    }
    if (typekind != ESMC_TYPEKIND_I4
      && typekind != ESMC_TYPEKIND_I8
      && typekind != ESMC_TYPEKIND_R4
      && typekind != ESMC_TYPEKIND_R8){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_NOT_IMPL,
        "- not a supported choice for typekind", ESMC_CONTEXT, &rc);
      throw rc;  // bail out with exception
    }
  }
}
//-----------------------------------------------------------------------------


} // namespace ESMCI





//-------------------------------------------------------------------------
//-------------------------------------------------------------------------
//-------------------------------------------------------------------------
//-------------------------------------------------------------------------
//-------------------------------------------------------------------------
//-------------------------------------------------------------------------
//-------------------------------------------------------------------------




//-------------------------------------------------------------------------
// The following code is for a first newArray prototype which I used
// to check out some communication ideas: DE-nonblocking paradigm!
//-------------------------------------------------------------------------


#ifdef FIRSTNEWARRAYPROTOTYPE


//-----------------------------------------------------------------------------
//
// This section includes all the newArray routines
//
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayConstruct()"
//BOPI
// !IROUTINE:  ESMC_newArrayConstruct
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayConstruct(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  ESMC_LocalArray *larray,  // pointer to ESMC_LocalArray object
  int *haloWidth,           // halo width
  ESMCI::DELayout *delayout,// DELayout
  int rootPET,              // root
  ESMCI::VM *vm){           // optional VM argument to speed up things
//
// !DESCRIPTION:
//    Construct the internal information structure in a new ESMC\_newArray
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  int petCount = vm->getPetCount();
  int localTid = vm->getTid(localPet);
  localVAS = vm->getVas(localPet);
  // get info about the LocalArray on rootPET and broadcast it to all PETs
  int laRank;
  if (localPet == rootPET)
    laRank = larray->ESMC_LocalArrayGetRank();
  vm->broadcast(&laRank, sizeof(int), rootPET);
  int *laLength = new int[laRank];
  if (localPet == rootPET)
    larray->ESMC_LocalArrayGetLengths(laRank, laLength);
  vm->broadcast(laLength, laRank * sizeof(int), rootPET);
  int *laLbound = new int[laRank];
  if (localPet == rootPET)
    larray->ESMC_LocalArrayGetLbounds(laRank, laLbound);
  vm->broadcast(laLbound, laRank * sizeof(int), rootPET);
  // set some newArray members
  rank = laRank;  // newArray's rank is equal to that of the origin LocalArray
  int decompRank;
  this->delayout = delayout;
  delayout->getDeprecated(&deCount, &decompRank, &localDeCount, NULL, 0,
    NULL, NULL, NULL, NULL, 0);
  localDeToDeMap = new int[localDeCount];
  delayout->getDeprecated(NULL, NULL, NULL, localDeToDeMap, localDeCount, NULL,
    NULL, NULL, NULL, 0);
  deVASList = new int[deCount];
  for (int de=0; de<deCount; de++)
    delayout->getDELocalInfo(de, NULL, 0, NULL, 0, NULL, 0, NULL,
      &(deVASList[de]));
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayConstruct: decompRank=%d, deCount=%d\n",
    decompRank, deCount);
#endif
  // determine the global min/max for all DELayout dimensions
  int *deCoordMin = new int[decompRank];
  int *deCoordMax = new int[decompRank];
  int *temp_deCoordMin = new int[decompRank];
  int *temp_deCoordMax = new int[decompRank];
//need to use the new DistGrid for this
//  delayout->getDELocalCoord(0, deCoordMin, deCoordMax);
  for (int i=0; i<deCount; i++){
//need to use the new DistGrid for this
//    delayout->getDELocalCoord(i, temp_deCoordMin, temp_deCoordMax);
    for (int j=0; j<decompRank; j++){
      if (temp_deCoordMin[j]<deCoordMin[j]) deCoordMin[j] = temp_deCoordMin[j];
      if (temp_deCoordMax[j]>deCoordMax[j]) deCoordMax[j] = temp_deCoordMax[j];
    }
  }
#if (VERBOSITY > 9)
  for (int j=0; j<decompRank; j++)
    printf("gjt in ESMC_newArrayConstruct: CoordMin=%d, CoordMax=%d\n",
      deCoordMin[j], deCoordMax[j]);
#endif
  // pre-chunk the decomposed larray dimensions according to DELayout axis
  int *deCoordLength = new int[decompRank];
  for (int i=0; i<decompRank; i++)
    deCoordLength[i] = deCoordMax[i] - deCoordMin[i] + 1;
  int ***arrayPreChunk = new int**[decompRank];
  for (int i=0; i<decompRank; i++){
    arrayPreChunk[i] = new int*[deCoordLength[i]];
    for (int j=0; j<deCoordLength[i]; j++){
      arrayPreChunk[i][j] = new int[2];
      arrayPreChunk[i][j][0] = deCoordMin[i] + j;
    }
  }
  
  int k=0;
  for (int i=0; i<laRank; i++){
    if (haloWidth[i]>=0){
      // this dimension may be decomposed
      for (int j=0; j<deCoordLength[k]; j++){
        // fill in evenly distributed preChunking
        arrayPreChunk[k][j][1] = laLength[i]/deCoordLength[k];
        if (j<laLength[i]%deCoordLength[k]) ++arrayPreChunk[k][j][1];
#if (VERBOSITY > 9)
        printf("gjt in ESMC_newArrayConstruct: arrayPreChunk[%d][%d][] = "
          "(%d, %d)\n", k, j, arrayPreChunk[k][j][0], arrayPreChunk[k][j][1]);
#endif
      }
      ++k;
    }
  }
  
  // fill in the newArray's meta info about the data bounds for each DE
  globalDataLBound = new int*[deCount];
  globalDataUBound = new int*[deCount];
  localFullLBound = new int*[deCount];
  localFullUBound = new int*[deCount];
  globalFullLBound = new int*[deCount];
  globalFullUBound = new int*[deCount];
  dataOffset = new int*[deCount];
  for (int de=0; de<deCount; de++){
    globalDataLBound[de] = new int[rank];
    globalDataUBound[de] = new int[rank];
    localFullLBound[de] = new int[rank];
    localFullUBound[de] = new int[rank];
    globalFullLBound[de] = new int[rank];
    globalFullUBound[de] = new int[rank];
    dataOffset[de] = new int[rank];
//need to use the new DistGrid for this
//    delayout->getDELocalCoord(de, deCoordMin, deCoordMax);
    int k=0;
    for (int i=0; i<laRank; i++){
      globalDataLBound[de][i] = 0;  // first use global frame starting at 0
      if (haloWidth[i]>=0){
        // this dimension may be decomposed
        globalDataUBound[de][i] = globalDataLBound[de][i];
        for (int j=0; j<deCoordLength[k]; j++){
          if (arrayPreChunk[k][j][0] < deCoordMin[k])
            globalDataLBound[de][i] += arrayPreChunk[k][j][1];
          if (arrayPreChunk[k][j][0] <= deCoordMax[k])
            globalDataUBound[de][i] += arrayPreChunk[k][j][1];
          else break;
        }
        ++k;
      }else{
        globalDataUBound[de][i] = globalDataLBound[de][i] + laLength[i];
      }
      globalDataUBound[de][i] -= 1; // adjustment of upper bound
      globalFullLBound[de][i] = globalDataLBound[de][i]; // prepare globalFullLB
      globalFullUBound[de][i] = globalDataUBound[de][i]; // prepare globalFullUB
      if (haloWidth[i]>=0){
        // valid halo width for this dimension
        // add halo region (assuming non-periodic boundaries in all directions)
        if (globalFullLBound[de][i] > 0)
          globalFullLBound[de][i] -= haloWidth[i];
        if (globalFullUBound[de][i] < laLength[i]-1)
          globalFullUBound[de][i] += haloWidth[i];
      }
      // determine dataOffset and shift to local bounds for full box
      dataOffset[de][i] = globalDataLBound[de][i] - globalFullLBound[de][i];
      localFullLBound[de][i] = 0;
      localFullUBound[de][i] = globalFullUBound[de][i]
        - globalFullLBound[de][i];
      globalDataLBound[de][i] += laLbound[i];  // shift into global frame
      globalDataUBound[de][i] += laLbound[i];  // shift into global frame
      globalFullLBound[de][i] += laLbound[i];  // shift into global frame
      globalFullUBound[de][i] += laLbound[i];  // shift into global frame
    }
  }
  
  // now the LocalArrays for all the DEs on this PET must be created
  if (localPet == rootPET){
    kind = larray->ESMC_LocalArrayGetTypeKind();
    vm->broadcast(&kind, sizeof(ESMC_TypeKind_Flag), rootPET);
  }else{
    vm->broadcast(&kind, sizeof(ESMC_TypeKind_Flag), rootPET);
  }
  if (localTid == 0){
    // this is the master thread of an ESMF-thread group
    localArrays = new ESMC_LocalArray*[localDeCount];
    commhArray = new ESMC_newArrayCommHandle[localDeCount];
    thargArray = new ESMC_newArrayThreadArg[localDeCount];
    // need to send localArrays/commhArray pointer to other PETs in threadGroup
    for (int pet=0; pet<petCount; pet++){
      int vas = vm->getVas(pet);
      if (pet != localPet && vas == localVAS){
        // send the localArrays pointer to this PET
        vm->send(&localArrays, sizeof(ESMC_LocalArray **), pet);
        vm->send(&commhArray, sizeof(ESMC_newArrayCommHandle *), pet);
        vm->send(&thargArray, sizeof(ESMC_newArrayThreadArg *), pet);
      }
    }
  }else{
    // localPet is part of a threadGroup but is not the master thread
    int pet;
    for (pet=0; pet<petCount; pet++){
      int tid = vm->getTid(pet);
      int vas = vm->getVas(pet);
      if (vas == localVAS && tid == 0) break; // found master thread
    }
    // receive the localArrays and commhArray pointer from master thread
    vm->recv(&localArrays, sizeof(ESMC_LocalArray **), pet);
    vm->recv(&commhArray, sizeof(ESMC_newArrayCommHandle *), pet);
    vm->recv(&thargArray, sizeof(ESMC_newArrayThreadArg *), pet);
  }
  int *temp_counts = new int[rank];
  int de;
  for (int i=0; i<localDeCount; i++){
    de = localDeToDeMap[i];
    if (delayout->serviceOffer(de, NULL) == ESMCI::SERVICEREPLY_ACCEPT){
      for (int j=0; j<rank; j++)
        temp_counts[j] = localFullUBound[de][j] - localFullLBound[de][j] + 1;
      localArrays[i] =
        ESMC_LocalArray::ESMC_LocalArrayCreate(rank, kind, temp_counts);
      commhArray[i].commhandleCount = 0;  // reset
      commhArray[i].pthidCount = 0;       // reset
      commhArray[i].buffer = NULL;        // reset
      delayout->serviceComplete(de);
    }
  }
  // scatter the larray across the newly constructed narray
  localrc = ESMC_newArrayScatter(larray, rootPET, vm);
  
  // garbage collection
  delete [] temp_counts;
  for (int i=0; i<decompRank; i++){
    for (int j=0; j<deCoordLength[i]; j++)
      delete [] arrayPreChunk[i][j];
    delete [] arrayPreChunk[i];
  }
  delete [] arrayPreChunk;
  delete [] deCoordLength;  
  delete [] temp_deCoordMin;
  delete [] temp_deCoordMax;
  delete [] deCoordMin;
  delete [] deCoordMax;
  delete [] laLbound;
  delete [] laLength;

  // error handling via LogErr
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, ESMC_CONTEXT,
    ESMC_NULL_POINTER)) return localrc;

  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------



//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayDestruct()"
//BOPI
// !IROUTINE:  ESMC_newArrayDestruct
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayDestruct(void){
//
// !RETURN VALUE:
//    int return code
//
//
// !DESCRIPTION:
//    Destruct the internal information structure in a new ESMC\_newArray
//
//EOPI
//-----------------------------------------------------------------------------
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayScatter()"
//BOPI
// !IROUTINE:  ESMC_newArrayScatter
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayScatter(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  ESMC_LocalArray *larray,  // pointer to ESMC_LocalArray object
  int rootPET,              // root
  ESMCI::VM *vm){             // optional VM argument to speed up things
//
// !DESCRIPTION:
//    Scatter the contents of an {\tt ESMC\_LocalArray} across the
//    {\tt ESMC\_newArray}. PET-based blocking paradigm.
//
//EOPI
//-----------------------------------------------------------------------------
  int localrc;
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  // check that there is a valid larray on rootPET 
  if (localPet == rootPET){
    if (larray == ESMC_NULL_POINTER){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD, 
        "- must supply a valid 'larray' argument on rootPET.", ESMC_CONTEXT,
        &localrc);
      return localrc;
    }
  }    
  // check that t/k/r matches
  if (localPet == rootPET){
    int laRank = larray->ESMC_LocalArrayGetRank();
    if (laRank != rank){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
        "- ranks don't match", ESMC_CONTEXT, &localrc);
      return localrc;
    }
    ESMC_TypeKind_Flag laTypeKind = larray->ESMC_LocalArrayGetTypeKind();
    if (laTypeKind != kind){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
        "- kinds don't match", ESMC_CONTEXT, &localrc);
      return localrc;
    }
  }
  // broadcast some of larray's meta info
  int *laLength = new int[rank];
  if (localPet == rootPET)
    larray->ESMC_LocalArrayGetLengths(rank, laLength);
  vm->broadcast(laLength, rank * sizeof(int), rootPET);
  int *laLbound = new int[rank];
  if (localPet == rootPET)
    larray->ESMC_LocalArrayGetLbounds(rank, laLbound);
  vm->broadcast(laLbound, rank * sizeof(int), rootPET);
  int laByteCount;
  if (localPet == rootPET)
    larray->ESMC_LocalArrayGetByteCount(&laByteCount);
  vm->broadcast(&laByteCount, sizeof(int), rootPET);
  // get info out of the associated localArrays
  void **localDeArrayBase = new void*[localDeCount];
  for (int i=0; i<localDeCount; i++)
    localArrays[i]->ESMC_LocalArrayGetBaseAddr(&localDeArrayBase[i]);
  // prepare a temporary buffer
  int blockCount = 1;
  for (int i=1; i<rank; i++)
    blockCount *= laLength[i];
  int blockSize = laByteCount / blockCount;
  int elementSize = blockSize / laLength[0];
#if (VERBOSITY > 1)
  printf("gjt in ESMC_newArrayScatter: elementSize = %d\n", elementSize);
#endif
  char *buffer;
  if (localPet == rootPET)
    larray->ESMC_LocalArrayGetBaseAddr((void **)&buffer); // start of data
  else
    buffer = new char[blockSize];   // buffer for first dim.
  // setup a blockID which basically is an array holding the higher dimensions
  int *blockID = new int[rank];   // blockID[0] is unused!
  for (int i=0; i<rank; i++)
    blockID[i] = 0;
  int *blockLocalIndex = new int[rank];
  int *blockGlobalIndex = new int[rank];
  blockGlobalIndex[0] = laLbound[0];
  // now loop over all blocks
  for (int blk=0; blk<blockCount; blk++){
    for (int i=1; i<rank; i++)
      blockGlobalIndex[i] = laLbound[i] + blockID[i];
#if (VERBOSITY > 1)
    printf("gjt in ESMC_newArrayScatter: blockID: ");
    for (int i=1; i<rank-1; i++)
      printf("%d, ", blockID[i]);
    printf("%d\n", blockID[rank-1]);
#endif
    // broadcast a block of data
    vm->broadcast(buffer, blockSize, rootPET);
    // loop over local DEs
    for (int ide=0; ide<localDeCount; ide++){
      int de = localDeToDeMap[ide];
      if (delayout->serviceOffer(de, NULL) == ESMCI::SERVICEREPLY_ACCEPT){
        // the localPet's offer was accepted by DELayout
        // check whether this DE's fullBox intersects the current block...
        int ii;
        for (ii=1; ii<rank; ii++){
#if (VERBOSITY > 1)
          printf("gjt in ESMC_newArrayScatter: ide=%d, blockGlobalIndex=%d\n"
            "globalXXX: %d, %d, %d, %d\n", ide, 
            blockGlobalIndex[ii], globalFullLBound[de][ii],
            globalDataLBound[de][ii], globalDataUBound[de][ii],
            globalFullUBound[de][ii]);
#endif    
          // check intersection with dataBox
          if (blockGlobalIndex[ii] >= globalDataLBound[de][ii] &&
            blockGlobalIndex[ii] <= globalDataUBound[de][ii]){
#if (VERBOSITY > 1)
            printf("gjt in ESMC_newArrayScatter: intersect dataBox\n");
#endif    
            // found intersection with dataBox
            blockLocalIndex[ii] = blockGlobalIndex[ii] 
              - globalDataLBound[de][ii]
              + dataOffset[de][ii];
            continue;
          }
          //check intersection with lower halo region
          if (globalFullLBound[de][ii] <= globalDataLBound[de][ii]){
            // this is a regular lower halo region
            if (blockGlobalIndex[ii] >= globalFullLBound[de][ii] &&
              blockGlobalIndex[ii] < globalDataLBound[de][ii]){
#if (VERBOSITY > 1)
              printf("gjt in ESMC_newArrayScatter: intersect regular lower"
                " halo\n");
#endif    
              // found intersection with regular lower halo region
              blockLocalIndex[ii] = blockGlobalIndex[ii] 
                - globalFullLBound[de][ii];
              continue;
            }
          }else{
            // this is a periodic lower halo region from the other end
            if (blockGlobalIndex[ii] >= globalFullLBound[de][ii] &&
              blockGlobalIndex[ii] <=
                (globalFullLBound[de][ii] + dataOffset[de][ii])){
#if (VERBOSITY > 1)
              printf("gjt in ESMC_newArrayScatter: intersect periodic lower"
                " halo\n");
#endif    
              // found intersection with periodic lower halo region
              blockLocalIndex[ii] = blockGlobalIndex[ii] 
                - globalFullLBound[de][ii];
              continue;
            }
          }
          //check intersection with upper halo region
          if (globalFullUBound[de][ii] >= globalDataUBound[de][ii]){
            // this is a regular upper halo region
            if (blockGlobalIndex[ii] > globalDataUBound[de][ii] &&
              blockGlobalIndex[ii] <= globalFullUBound[de][ii]){
#if (VERBOSITY > 1)
              printf("gjt in ESMC_newArrayScatter: intersect regular upper"
                " halo\n");
#endif    
              // found intersection with regular upper halo region
              blockLocalIndex[ii] = blockGlobalIndex[ii] 
                - globalDataLBound[de][ii] + dataOffset[de][ii];
              continue;
            }
          }else{
            // this is a periodic upper halo region from the other end
            int upperDataOffset = 
              localFullUBound[de][ii] - localFullLBound[de][ii]
              - dataOffset[de][ii]
              - (globalDataUBound[de][ii] - globalDataLBound[de][ii]);
            if (blockGlobalIndex[ii] >= 
              (globalFullUBound[de][ii] - upperDataOffset) &&
              blockGlobalIndex[ii] <= globalFullUBound[de][ii]){
#if (VERBOSITY > 1)
              printf("gjt in ESMC_newArrayScatter: intersect periodic upper"
                " halo\n");
#endif    
              // found intersection with periodic upper halo region
              blockLocalIndex[ii] = blockGlobalIndex[ii]
                - (globalFullUBound[de][ii] - upperDataOffset);
              continue;
            }
          }
          break;
        }        
#if (VERBOSITY > 1)
        if (ii==rank)
          printf("gjt in ESMC_newArrayScatter: block inside fullBox\n");
        else
          printf("gjt in ESMC_newArrayScatter: block outside fullBox\n");
#endif
        blockLocalIndex[0] = blockGlobalIndex[0] - globalFullLBound[de][0];
        if (ii==rank){
          // block intersects fullBox ->
          // find start address of DE's data column overlaping with block data
          char *base = (char *)localDeArrayBase[ide];
          int elementCount = 0;
          for (int i=rank-1; i>0; i--){
            elementCount += blockLocalIndex[i];
            elementCount *= (localFullUBound[de][i-1] 
              - localFullLBound[de][i-1] + 1);
          }
          base += elementCount * elementSize;
#if (VERBOSITY > 1)
          printf("gjt in ESMC_newArrayScatter: elementCount = %d, base = %p\n",
            elementCount, base);
#endif
          // determine start pointers of buffer and base overlap, overlapCount
          char *baseOverlap = base;
          char *blockOverlap = buffer;
          int overlapCount;
          int fullBoxCount = localFullUBound[de][0] - localFullLBound[de][0] 
            + 1;
          if (blockLocalIndex[0] < 0){
            blockOverlap += (-blockLocalIndex[0]) * elementSize;
            overlapCount = laLength[0] + blockLocalIndex[0];
            if (fullBoxCount < overlapCount)
              overlapCount = fullBoxCount;
          }else{
            baseOverlap += blockLocalIndex[0] * elementSize;
            overlapCount = fullBoxCount - blockLocalIndex[0];
            if (laLength[0] < overlapCount)
              overlapCount = laLength[0];
          }
          if (overlapCount < 0) 
            overlapCount = 0;
#if (VERBOSITY > 1)
          printf("gjt in ESMC_newArrayScatter: overlapCount = %d, blockOverlap"
            " = %p, baseOverlap = %p\n", overlapCount, blockOverlap,
            baseOverlap);
#endif
          // finally memcpy from buffer into DE's local array.
          memcpy(baseOverlap, blockOverlap, overlapCount * elementSize);
        }
        delayout->serviceComplete(de); // close window on DE
      }
    }
    // update the blockID
    ++blockID[1];
    for (int i=1; i<rank-1; i++)
      if (blockID[i] >= laLength[i]){
        blockID[i] = 0;
        ++blockID[i+1];
      }
    // shift the buffer start on rootPET to the next column
    if (localPet == rootPET)
      buffer += laLength[0] * elementSize;
  }
  
  // garbage collection
  delete [] blockGlobalIndex;
  delete [] blockLocalIndex;
  delete [] blockID;
  if (localPet != rootPET)
    delete [] buffer;
  delete [] laLength;
  delete [] localDeArrayBase;
    
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------



//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayScatter()"
//BOPI
// !IROUTINE:  ESMC_newArrayScatter
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayScatter(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  ESMC_LocalArray *larray,  // pointer to ESMC_LocalArray object
  int rootPET,              // root
  ESMC_newArrayCommHandle *commh, // commu handle for non-blocking mode
  ESMCI::VM *vm){             // optional VM argument to speed up things
//
// !DESCRIPTION:
//    Scatter the contents of an {\tt ESMC\_LocalArray} across the
//    {\tt ESMC\_newArray}. DE-based non-blocking paradigm.
//
//EOPI
//-----------------------------------------------------------------------------
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScatter(ROOT): DE-based nb paradigm\n");
#endif
  int localrc;
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  // if this is not the rootPET then exit because this is root side of scatter
  if (localPet != rootPET) return ESMF_SUCCESS;
  // check that the commhandle is clean, if not then exit with error
  int *cc = &(commh->commhandleCount); // to simplyfy usage during this method
  if (*cc != 0){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD, 
      "- commhandle still holds outstanding communication handles.", 
      ESMC_CONTEXT, &localrc);
    return localrc;
  }
  // check that there is a valid larray on rootPET 
  if (larray == ESMC_NULL_POINTER){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD, 
      "- must supply a valid 'larray' argument on rootPET.", ESMC_CONTEXT,
      &localrc);
    return localrc;
  }
  // check that t/k/r matches
  int laRank = larray->ESMC_LocalArrayGetRank();
  if (laRank != rank){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
      "- ranks don't match", ESMC_CONTEXT, &localrc);
    return localrc;
  }
  ESMC_TypeKind_Flag laTypeKind = larray->ESMC_LocalArrayGetTypeKind();
  if (laTypeKind != kind){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
      "- kinds don't match", ESMC_CONTEXT, &localrc);
    return localrc;
  }
  // query some of larray's meta info
  int *laLength = new int[rank];
  larray->ESMC_LocalArrayGetLengths(rank, laLength);
  int laByteCount;
  larray->ESMC_LocalArrayGetByteCount(&laByteCount);
  int *laLbound = new int[rank];
  larray->ESMC_LocalArrayGetLbounds(rank, laLbound);
  // determine blockCount and allocate according number of commhandles
  int blockCount = 1;
  for (int i=1; i<rank; i++)
    blockCount *= laLength[i];
  int blockSize = laByteCount / blockCount;
  int elementSize = blockSize / laLength[0];
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScatter(ROOT): blockCount = %d\n", blockCount);
  printf("gjt in ESMC_newArrayScatter(ROOT): elementSize = %d\n", elementSize);
#endif
  int totalHandles = deCount * (3 + blockCount);  // specific for this routine
  commh->vmk_commh = new VMK::commhandle*[totalHandles];
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScatter(ROOT): commh=%p, vmk_commh=%p\n",
    commh, commh->vmk_commh);
  printf("gjt in ESMC_newArrayScatter(ROOT): totalHandles = %d\n",
    totalHandles);
#endif
  // send some of larray's meta info to all the DEs
  for (int de=0; de<deCount; de++){
#if (VERBOSITY > 9)
    printf("gjt in ESMC_newArrayScatter(ROOT): sending info to de = %d\n", de);
#endif
    commh->vmk_commh[*cc] = NULL; // mark as invalid element
    vm->send(laLength, rank * sizeof(int), deVASList[de],
      &(commh->vmk_commh[(*cc)++]), de+3000);
    commh->vmk_commh[*cc] = NULL; // mark as invalid element
    vm->send(&laByteCount, sizeof(int), deVASList[de],
      &(commh->vmk_commh[(*cc)++]), de+3000);
    commh->vmk_commh[*cc] = NULL; // mark as invalid element
    vm->send(laLbound, rank * sizeof(int), deVASList[de],
      &(commh->vmk_commh[(*cc)++]), de+3000);  
#if (VERBOSITY > 9)
    printf("gjt in ESMC_newArrayScatter(ROOT): done sending info to de = %d\n",
      de);
#endif
  }
  // prepare buffer pointer into data block
  char *buffer;
  larray->ESMC_LocalArrayGetBaseAddr((void **)&buffer); // start of data
#if (VERBOSITY > 9)
    fprintf(stderr, "gjt in ESMC_newArrayScatter(ROOT): buffer = %p\n",
      buffer);
#endif
  // setup a blockID which basically is an array holding the higher dimensions
  int *blockID = new int[rank];   // blockID[0] is unused!
#if (VERBOSITY > 9)
    fprintf(stderr, "gjt in ESMC_newArrayScatter(ROOT): blockID = %p\n",
      blockID);
#endif
  for (int i=0; i<rank; i++)
    blockID[i] = 0;
  int *blockLocalIndex = new int[rank];
#if (VERBOSITY > 9)
    fprintf(stderr, "gjt in ESMC_newArrayScatter(ROOT): blockLocalIndex = %p\n",
      blockLocalIndex);
#endif
  int *blockGlobalIndex = new int[rank];
#if (VERBOSITY > 9)
    fprintf(stderr, "gjt in ESMC_newArrayScatter(ROOT): blockGlobalIndex ="
      " %p\n", blockGlobalIndex);
#endif

//  blockGlobalIndex[0] = laLbound[0];
  
  
#if (VERBOSITY > 9)
  fprintf(stderr, "gjt in ESMC_newArrayScatter(ROOT): start blk loop\n");
#endif
  // now loop over all blocks
  for (int blk=0; blk<blockCount; blk++){
    for (int i=1; i<rank; i++)
      blockGlobalIndex[i] = laLbound[i] + blockID[i];
#if (VERBOSITY > 9)
    fprintf(stderr, "gjt in ESMC_newArrayScatter(ROOT): blockID: ");
    for (int i=1; i<rank-1; i++)
      fprintf(stderr, "%d, ", blockID[i]);
    fprintf(stderr, "%d\n", blockID[rank-1]);
#endif
    // send block of data to all of the DEs
    for (int de=0; de<deCount; de++){
      commh->vmk_commh[*cc] = NULL; // mark as invalid element
      vm->send(buffer, blockSize, deVASList[de],
        &(commh->vmk_commh[(*cc)++]),  de+3000);
    }
    // update the blockID
    ++blockID[1];
    for (int i=1; i<rank-1; i++)
      if (blockID[i] >= laLength[i]){
        blockID[i] = 0;
        ++blockID[i+1];
      }
    // shift the buffer start to the next column in larray
    buffer += laLength[0] * elementSize;
  }
  
  // garbage collection
  delete [] blockGlobalIndex;
  delete [] blockLocalIndex;
  delete [] blockID;
  delete [] laLbound;
  delete [] laLength;

  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayScatter()"
//BOPI
// !IROUTINE:  ESMC_newArrayScatter
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayScatter(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  ESMC_LocalArray *larray,  // pointer to ESMC_LocalArray object
  int rootPET,              // root
  int de,                   // DE for DE-based non-blocking scatter
  ESMCI::VM *vm){             // optional VM argument to speed up things
//
// !DESCRIPTION:
//    Scatter the contents of an {\tt ESMC\_LocalArray} across the
//    {\tt ESMC\_newArray}. DE-based non-blocking paradigm.
//
//EOPI
//-----------------------------------------------------------------------------
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScatter(DE): DE-based non-blocking paradigm\n");
#endif
  int localrc;
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  // check that there is a valid larray on rootPET 
  if (localPet == rootPET){
    if (larray == ESMC_NULL_POINTER){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD, 
        "- must supply a valid 'larray' argument on rootPET.", ESMC_CONTEXT,
        &localrc);
      return localrc;
    }
  }    
  // check that t/k/r matches
  if (localPet == rootPET){
    int laRank = larray->ESMC_LocalArrayGetRank();
    if (laRank != rank){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
        "- ranks don't match", ESMC_CONTEXT, &localrc);
      return localrc;
    }
    ESMC_TypeKind_Flag laTypeKind = larray->ESMC_LocalArrayGetTypeKind();
    if (laTypeKind != kind){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
        "- kinds don't match", ESMC_CONTEXT, &localrc);
      return localrc;
    }
  }
  // determine localDE index
  int localDe;
  for (localDe=0; localDe<localDeCount; localDe++)
    if (localDeToDeMap[localDe] == de) break;
  // pack arguments to pass into ScatterThread
  thargArray[localDe].array = this;
  thargArray[localDe].vm = vm;
  thargArray[localDe].de = de;
  thargArray[localDe].rootPET = rootPET;
  // create the ScatterThread for this DE
  pthread_t *pthid =
    &(commhArray[localDe].pthid[(commhArray[localDe].pthidCount)++]);
// took the following call out so the rest would compile: 
// mpCC on AIX has trouble with friend func namespaces
//  pthread_create(pthid, NULL, ESMC_newArrayScatterThread,
//    &(thargArray[localDe]));
    
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayScalarReduce()"
//BOPI
// !IROUTINE:  ESMC_newArrayScalarReduce
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayScalarReduce(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  void *result,             // result value (scalar)
  ESMC_TypeKind_Flag dtk,   // data type kind
  ESMC_Operation op,        // reduce operation
  int rootPET,              // root
  ESMCI::VM *vm){           // optional VM argument to speed up things
//
// !DESCRIPTION:
//    Reduce the data of an {\tt ESMC\_newArray} into a single scalar value.
//
//EOPI
//-----------------------------------------------------------------------------
  // initialize return code; assume routine not implemented
  int localrc = ESMC_RC_NOT_IMPL;         // local return code
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  // check that there is a valid result argument on rootPET 
  if (localPet == rootPET){
    if (result == ESMC_NULL_POINTER){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD, 
        "- must supply a valid 'result' argument on rootPET.", ESMC_CONTEXT,
        &localrc);
      return localrc;
    }
  }    
  // check that t/k matches (on all PETs!)
  if (dtk != kind){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
      "- kinds don't match", &localrc);
    return localrc;
  }
  // prepeare PET-local temporary result variable
  void *localResult;
  switch (dtk){
  case ESMC_TYPEKIND_I4:
    localResult = new ESMC_I4;
    break;
  case ESMC_TYPEKIND_R4:
    localResult = new ESMC_R4;
    break;
  case ESMC_TYPEKIND_R8:
    localResult = new ESMC_R8;
    break;
  default:
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
      "- data type kind is unknown to VM", &localrc);
    return localrc;
  }
  // loop over local DEs
  int primeFlag = 1;
  for (int localDe=0; localDe<localDeCount; localDe++){
    int de = localDeToDeMap[localDe];
    if (delayout->serviceOffer(de, NULL) == ESMCI::SERVICEREPLY_ACCEPT){
      // the localPet's offer was accepted by DELayout
      // get info out of the associated localArray
      ESMC_LocalArray *localDeArray = localArrays[localDe];
      void *localDeArrayBase;
      localDeArray->ESMC_LocalArrayGetBaseAddr(&localDeArrayBase);
      int *laLength = new int[rank];
      localDeArray->ESMC_LocalArrayGetLengths(rank, laLength);
      int laByteCount;
      localDeArray->ESMC_LocalArrayGetByteCount(&laByteCount);
      // determine how many blocks (columns) this localDeArray has
      int blockCount = 1;
      for (int i=1; i<rank; i++)
        blockCount *= laLength[i];
      int blockSize = laByteCount / blockCount;
      int elementSize = blockSize / laLength[0];
#if (VERBOSITY > 1)
      printf("gjt in ESMC_newArrayScalarReduce: de = %d, blockCount = %d\n",
        de, blockCount);
#endif
      // setup a blockID which is an array holding the higher dimensions
      int *blockID = new int[rank];   // blockID[0] is unused!
      for (int i=0; i<rank; i++)
        blockID[i] = 0;
      // now loop over all blocks
      for (int blk=0; blk<blockCount; blk++){
#if (VERBOSITY > 1)
        printf("gjt in ESMC_newArrayScalarReduce: blockID: ");
        for (int i=1; i<rank-1; i++)
          printf("%d, ", blockID[i]);
        printf("%d\n", blockID[rank-1]);
#endif
        // find start address of DE's data column for this block
        int elementCount = 0;
        int skipFlag = 0;
        for (int i=rank-1; i>0; i--){
          if (blockID[i] < dataOffset[de][i] ||
            blockID[i] > (globalDataUBound[de][i] - globalFullLBound[de][i])){
            // this block is inside halo region -> skip
            skipFlag = 1;
            break;
          }
          elementCount += blockID[i];
          elementCount *= laLength[i-1];
        }
        // update the blockID (must be done before skipping!)
        ++blockID[1];
        for (int i=1; i<rank-1; i++)
          if (blockID[i] >= laLength[i]){
            blockID[i] = 0;
            ++blockID[i+1];
          }
        if (skipFlag) continue; 
        char *base = (char *)localDeArrayBase;
        base += (elementCount + dataOffset[de][0]) * elementSize;
        int itemCount = globalDataUBound[de][0] - globalDataLBound[de][0] + 1;
#if (VERBOSITY > 1)
        printf("gjt in ESMC_newArrayScalarReduce: base = %p, itemCount = %d\n",
          base, itemCount);
#endif
        // reduce the block into localResult
        switch (dtk){
        case ESMC_TYPEKIND_I4:
          {
            ESMC_I4 *tempResult = (ESMC_I4 *)localResult;
            ESMC_I4 *tempBase = (ESMC_I4 *)base;
            switch (op){
            case ESMF_SUM:
              if (primeFlag)
                *tempResult = 0;  // prime the result variable
              for (int i=0; i<itemCount; i++){
                *tempResult += tempBase[i];
              }
              break;
            case ESMF_MIN:
              if (primeFlag)
                *tempResult = tempBase[0];          // prime the result variable
              for (int i=0; i<itemCount; i++){
                if (tempBase[i] < *tempResult) *tempResult = tempBase[i];
              }
              break;
            case ESMF_MAX:
              if (primeFlag)
                *tempResult = tempBase[0];          // prime the result variable
              for (int i=0; i<itemCount; i++){
                if (tempBase[i] > *tempResult) *tempResult = tempBase[i];
              }
              break;
            }
          }
          break;
        case ESMC_TYPEKIND_R4:
          {
            ESMC_R4 *tempResult = (ESMC_R4 *)localResult;
            ESMC_R4 *tempBase = (ESMC_R4 *)base;
            switch (op){
            case ESMF_SUM:
              if (primeFlag)
                *tempResult = 0.;  // prime the result variable
              for (int i=0; i<itemCount; i++){
                *tempResult += tempBase[i];
              }
              break;
            case ESMF_MIN:
              if (primeFlag)
                *tempResult = tempBase[0];          // prime the result variable
              for (int i=0; i<itemCount; i++){
                if (tempBase[i] < *tempResult) *tempResult = tempBase[i];
              }
              break;
            case ESMF_MAX:
              if (primeFlag)
                *tempResult = tempBase[0];          // prime the result variable
              for (int i=0; i<itemCount; i++){
                if (tempBase[i] > *tempResult) *tempResult = tempBase[i];
              }
              break;
            }
          }
          break;
        case ESMC_TYPEKIND_R8:
          {
            ESMC_R8 *tempResult = (ESMC_R8 *)localResult;
            ESMC_R8 *tempBase = (ESMC_R8 *)base;
            switch (op){
            case ESMF_SUM:
              if (primeFlag)
                *tempResult = 0.;  // prime the result variable
              for (int i=0; i<itemCount; i++){
                *tempResult += tempBase[i];
              }
              break;
            case ESMF_MIN:
              if (primeFlag)
                *tempResult = tempBase[0];          // prime the result variable
              for (int i=0; i<itemCount; i++){
                if (tempBase[i] < *tempResult) *tempResult = tempBase[i];
              }
              break;
            case ESMF_MAX:
              if (primeFlag)
                *tempResult = tempBase[0];          // prime the result variable
              for (int i=0; i<itemCount; i++){
                if (tempBase[i] > *tempResult) *tempResult = tempBase[i];
              }
              break;
            }
          }
          break;
        }
        primeFlag = 0;  // priming of result has been accomplished
      }      
      // DE-local garbage collection
      delete [] laLength;
      delete [] blockID;
      delayout->serviceComplete(de); // close window on DE
    }
  }
  // prepare for VM operation
  vmType vmt;
  switch (dtk){
  case ESMC_TYPEKIND_I4:
    vmt = vmI4;
    break;
  case ESMC_TYPEKIND_R4:
    vmt = vmR4;
    break;
  case ESMC_TYPEKIND_R8:
    vmt = vmR8;
    break;
  default:
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
      "- data type kind is unknown to VM", &localrc);
    return localrc;
  }
  // reduce localResult across entire VM and put output into result on rootPET
  vm->reduce(localResult, result, 1, vmt, (vmOp)op, rootPET);
  
  // garbage collection
  // need to typcast back before: delete localResult;
  
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------



//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayScalarReduce()"
//BOPI
// !IROUTINE:  ESMC_newArrayScalarReduce
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayScalarReduce(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  void *result,             // result value (scalar)
  ESMC_TypeKind_Flag dtk,   // data type kind
  ESMC_Operation op,        // reduce operation
  int rootPET,              // root
  ESMC_newArrayCommHandle *commh, // commu handle for non-blocking mode
  ESMCI::VM *vm){           // optional VM argument to speed up things
//
// !DESCRIPTION:
//    Reduce the data of an {\tt ESMC\_newArray} into a single scalar value.
//
//EOPI
//-----------------------------------------------------------------------------
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScalarReduce(ROOT): DE-based nb paradigm\n");
#endif
  int localrc;
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  // check that there is a valid result argument on rootPET 
  if (localPet == rootPET){
    if (result == ESMC_NULL_POINTER){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD, 
        "- must supply a valid 'result' argument on rootPET.", &localrc);
      return localrc;
    }
  }    
  // check that t/k matches (on all PETs!)
  if (dtk != kind){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
      "- kinds don't match", &localrc);
    return localrc;
  }
  // pack arguments to pass into ScatterThread
  thargRoot.array = this;
  thargRoot.vm = vm;
  thargRoot.rootPET = rootPET;
  thargRoot.result = result;
  thargRoot.dtk = dtk;
  thargRoot.op = op;
  
  // create the ScalarReduceThread for rootPET
  pthread_t *pthid = &(commh->pthid[(commh->pthidCount)++]);
// took the following call out so the rest would compile: 
// mpCC on AIX has trouble with friend func namespaces
//  pthread_create(pthid, NULL, ESMC_newArrayScalarReduceThread, &thargRoot);
    
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayScalarReduce()"
//BOPI
// !IROUTINE:  ESMC_newArrayScalarReduce
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayScalarReduce(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  void *result,             // result value (scalar)
  ESMC_TypeKind_Flag dtk,   // data type kind
  ESMC_Operation op,        // reduce operation
  int rootPET,              // root
  int de,                   // DE for DE-based non-blocking reduce
  ESMCI::VM *vm){           // optional VM argument to speed up things
//
// !DESCRIPTION:
//    Reduce the data of an {\tt ESMC\_newArray} into a single scalar value.
//
//EOPI
//-----------------------------------------------------------------------------
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScalarReduce(DE): DE-based nb paradigm\n");
#endif
  int localrc;
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  // determine localDE index
  int localDe;
  for (localDe=0; localDe<localDeCount; localDe++)
    if (localDeToDeMap[localDe] == de) break;
  // determine VAS for rootPET
  int rootVAS = vm->getVas(rootPET);
  // check that t/k matches (on all PETs!)
  if (dtk != kind){
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
      "- kinds don't match", &localrc);
    return localrc;
  }
  // prepeare PET-local temporary result variable
  void *localResult;
  int size;
  switch (dtk){
  case ESMC_TYPEKIND_I4:
    localResult = new ESMC_I4;
    size = 4;
    break;
  case ESMC_TYPEKIND_R4:
    localResult = new ESMC_R4;
    size = 4;
    break;
  case ESMC_TYPEKIND_R8:
    localResult = new ESMC_R8;
    size = 8;
    break;
  default:
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
      "- data type kind is unknown", &localrc);
    return localrc;
  }
  // get info out of the associated localArray
  ESMC_LocalArray *localDeArray = localArrays[localDe];
  void *localDeArrayBase;
  localDeArray->ESMC_LocalArrayGetBaseAddr(&localDeArrayBase);
  int *laLength = new int[rank];
  localDeArray->ESMC_LocalArrayGetLengths(rank, laLength);
  int laByteCount;
  localDeArray->ESMC_LocalArrayGetByteCount(&laByteCount);
  // determine how many blocks (columns) this localDeArray has
  int blockCount = 1;
  for (int i=1; i<rank; i++)
    blockCount *= laLength[i];
  int blockSize = laByteCount / blockCount;
  int elementSize = blockSize / laLength[0];
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScalarReduce: de = %d, blockCount = %d\n",
    de, blockCount);
#endif
  // setup a blockID which is an array holding the higher dimensions
  int *blockID = new int[rank];   // blockID[0] is unused!
  for (int i=0; i<rank; i++)
    blockID[i] = 0;
  // now loop over all blocks
  int primeFlag = 1;
  for (int blk=0; blk<blockCount; blk++){
#if (VERBOSITY > 1)
    printf("gjt in ESMC_newArrayScalarReduce: blockID: ");
    for (int i=1; i<rank-1; i++)
      printf("%d, ", blockID[i]);
    printf("%d\n", blockID[rank-1]);
#endif
    // find start address of DE's data column for this block
    int elementCount = 0;
    int skipFlag = 0;
    for (int i=rank-1; i>0; i--){
      if (blockID[i] < dataOffset[de][i] ||
        blockID[i] > (globalDataUBound[de][i] - globalFullLBound[de][i])){
        // this block is inside halo region -> skip
        skipFlag = 1;
        break;
      }
      elementCount += blockID[i];
      elementCount *= laLength[i-1];
    }
    // update the blockID (must be done before skipping!)
    ++blockID[1];
    for (int i=1; i<rank-1; i++)
      if (blockID[i] >= laLength[i]){
        blockID[i] = 0;
        ++blockID[i+1];
      }
    if (skipFlag) continue; 
    char *base = (char *)localDeArrayBase;
    base += (elementCount + dataOffset[de][0]) * elementSize;
    int itemCount = globalDataUBound[de][0] - globalDataLBound[de][0] + 1;
#if (VERBOSITY > 1)
    printf("gjt in ESMC_newArrayScalarReduce: base = %p, itemCount = %d\n",
      base, itemCount);
#endif
    // reduce the block into localResult
    switch (dtk){
    case ESMC_TYPEKIND_I4:
      {
        ESMC_I4 *tempResult = (ESMC_I4 *)localResult;
        ESMC_I4 *tempBase = (ESMC_I4 *)base;
        switch (op){
        case ESMF_SUM:
          if (primeFlag)
            *tempResult = 0;                    // prime the result variable
          for (int i=0; i<itemCount; i++){
            *tempResult += tempBase[i];
          }
          break;
        case ESMF_MIN:
          if (primeFlag)
            *tempResult = tempBase[0];          // prime the result variable
          for (int i=0; i<itemCount; i++){
            if (tempBase[i] < *tempResult) *tempResult = tempBase[i];
          }
          break;
        case ESMF_MAX:
          if (primeFlag)
            *tempResult = tempBase[0];          // prime the result variable
          for (int i=0; i<itemCount; i++){
            if (tempBase[i] > *tempResult) *tempResult = tempBase[i];
          }
          break;
        }
      }
      break;
    case ESMC_TYPEKIND_R4:
      {
        ESMC_R4 *tempResult = (ESMC_R4 *)localResult;
        ESMC_R4 *tempBase = (ESMC_R4 *)base;
        switch (op){
        case ESMF_SUM:
          if (primeFlag)
            *tempResult = 0.;                   // prime the result variable
          for (int i=0; i<itemCount; i++){
            *tempResult += tempBase[i];
          }
          break;
        case ESMF_MIN:
          if (primeFlag)
            *tempResult = tempBase[0];          // prime the result variable
          for (int i=0; i<itemCount; i++){
            if (tempBase[i] < *tempResult) *tempResult = tempBase[i];
          }
          break;
        case ESMF_MAX:
          if (primeFlag)
            *tempResult = tempBase[0];          // prime the result variable
          for (int i=0; i<itemCount; i++){
            if (tempBase[i] > *tempResult) *tempResult = tempBase[i];
          }
          break;
        }
      }
      break;
    case ESMC_TYPEKIND_R8:
      {
        ESMC_R8 *tempResult = (ESMC_R8 *)localResult;
        ESMC_R8 *tempBase = (ESMC_R8 *)base;
        switch (op){
        case ESMF_SUM:
          if (primeFlag)
            *tempResult = 0.;                   // prime the result variable
          for (int i=0; i<itemCount; i++){
            *tempResult += tempBase[i];
          }
          break;
        case ESMF_MIN:
          if (primeFlag)
            *tempResult = tempBase[0];          // prime the result variable
          for (int i=0; i<itemCount; i++){
            if (tempBase[i] < *tempResult) *tempResult = tempBase[i];
          }
          break;
        case ESMF_MAX:
          if (primeFlag)
            *tempResult = tempBase[0];          // prime the result variable
          for (int i=0; i<itemCount; i++){
            if (tempBase[i] > *tempResult) *tempResult = tempBase[i];
          }
          break;
        }
      }
      break;
    }
    primeFlag = 0;  // priming of result has been accomplished
  }      
  // DE-local garbage collection
  delete [] laLength;
  delete [] blockID;
  // send DE-local reduction result to rootPET
  commhArray[localDe].commhandleCount = 1;
  commhArray[localDe].vmk_commh = new VMK::commhandle*[1];
  commhArray[localDe].vmk_commh[0] = NULL;  // mark as invalid
  vm->send(localResult, size, rootVAS,
    &(commhArray[localDe].vmk_commh[0]), de+5000);
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScalarReduce: sent localResult=%g for de = %d\n",
    *(ESMC_R8 *)localResult, de);
#endif

  // garbage collection must be done _after_ associated wait
  commhArray[localDe].buffer = localResult;

  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------




// ---- Wait methods ---  


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayWait()"
//BOPI
// !IROUTINE:  ESMC_newArrayWait
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayWait(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  int rootPET,              // root
  ESMC_newArrayCommHandle *commh, // commu handle specif. non-blocking op.
  ESMCI::VM *vm){             // optional VM argument to speed up things
//
// !DESCRIPTION:
//    Wait for a non-blocking newArray communication to be done with data 
//    object on rootPET.
//
//EOPI
//-----------------------------------------------------------------------------
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayWait: DE-based nb paradigm ROOT\n");
  printf("gjt in ESMC_newArrayWait(ROOT): commh=%p\n", commh);
#endif
  int localrc;
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  // if this is not the rootPET then exit because this is root side of scatter
  if (localPet != rootPET) return ESMF_SUCCESS;
  int *cc = &(commh->commhandleCount);  // to simplify usage during this method
  int *pc = &(commh->pthidCount);       // to simplify usage during this method
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayWait(ROOT): *cc=%d, *pc=%d\n", *cc, *pc);
#endif
  // join all of the root threads
  for (int i=0; i<*pc; i++){
    pthread_join(commh->pthid[i], NULL);
  }
  *pc = 0;  // reset
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayWait(ROOT): threads are joined\n");
#endif
  // wait for all the communication handles to clear
  for (int i=0; i<*cc; i++){
    vm->commwait(&(commh->vmk_commh[i]), NULL, 1);  // use nanopause=1ns to lower load
  }
  if (*cc) delete [] commh->vmk_commh;
  *cc = 0;  // reset
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayWait(ROOT): all commhandles are complete and deleted\n");
#endif
  
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayWait()"
//BOPI
// !IROUTINE:  ESMC_newArrayWait
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayWait(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  int de,                   // DE for which to wait
  ESMCI::VM *vm){             // optional VM argument to speed up things
//
// !DESCRIPTION:
//    Wait for a non-blocking newArray communication to finish for de.
//
//EOPI
//-----------------------------------------------------------------------------
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayWait: DE-based non-blocking paradigm\n");
#endif
  int localrc;
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  // determine localDE index
  int localDe;
  for (localDe=0; localDe<localDeCount; localDe++)
    if (localDeToDeMap[localDe] == de) break;
  int *cc = &(commhArray[localDe].commhandleCount);  // to simplify usage
  int *pc = &(commhArray[localDe].pthidCount);       // to simplify usage
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayWait(DE): *cc=%d, *pc=%d\n", *cc, *pc);
#endif
  // join this DE's pthreads
  for (int i=0; i<*pc; i++){
#if (VERBOSITY > 9)
    printf("gjt in ESMC_newArrayWait: pthread=%d\n",
      commhArray[localDe].pthid[i]);
#endif
    pthread_join(commhArray[localDe].pthid[i], NULL);
  }
  *pc = 0;  // reset
  // wait for all the communication handles to clear
  for (int i=0; i<*cc; i++){
    // use nanopause=1ns to lower load
    vm->commwait(&(commhArray[localDe].vmk_commh[i]), NULL, 1);
  }
  if (*cc) delete [] commhArray[localDe].vmk_commh;
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayWait: all commhandles deleted\n");
#endif
  *cc = 0;  // reset
  // destroy the buffer
  if (commhArray[localDe].buffer){
    delete commhArray[localDe].buffer;
    commhArray[localDe].buffer = NULL;  // mark invalid for next time... 
  }
  
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayGet()"
//BOPI
// !IROUTINE:  ESMC_newArrayGet
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayGet(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  int *rank_out,                      // out - rank of newArray
  ESMCI::DELayout **delayout,         // out - associated DELayout
  ESMC_LocalArray **localArrays,      // out - local arrays
  int len_localArrays,                // in  - length of localArrays array
  int *globalFullLBound,              // out - 2d array of bounds
  int *len_globalFullLBound,          // in  - lengths of the above array
  int *globalFullUBound,              // out - 2d array of bounds
  int *len_globalFullUBound,          // in  - lengths of the above array
  int *globalDataLBound,              // out - 2d array of bounds
  int *len_globalDataLBound,          // in  - lengths of the above array
  int *globalDataUBound,              // out - 2d array of bounds
  int *len_globalDataUBound,          // in  - lengths of the above array
  int *localDataLBound,               // out - 2d array of bounds
  int *len_localDataLBound,           // in  - lengths of the above array
  int *localDataUBound,               // out - 2d array of bounds
  int *len_localDataUBound            // in  - lengths of the above array
  ){
//
// !DESCRIPTION:
//    Get access to internals
//
//EOPI
//-----------------------------------------------------------------------------
  if (rank_out != ESMC_NULL_POINTER)
    *rank_out = this->rank;
  if (delayout != ESMC_NULL_POINTER)
    *delayout = this->delayout;
  for (int i=0; (i<localDeCount && i<len_localArrays); i++)
    localArrays[i] = this->localArrays[i];
  for (int i=0; (i<deCount && i<len_globalFullLBound[1]); i++)
    for (int j=0; (j<rank && j<len_globalFullLBound[0]); j++)
      *(globalFullLBound + i*rank + j) = this->globalFullLBound[i][j];
  for (int i=0; (i<deCount && i<len_globalFullUBound[1]); i++)
    for (int j=0; (j<rank && j<len_globalFullUBound[0]); j++)
      *(globalFullUBound + i*rank + j) = this->globalFullUBound[i][j];
  for (int i=0; (i<deCount && i<len_globalDataLBound[1]); i++)
    for (int j=0; (j<rank && j<len_globalDataLBound[0]); j++)
      *(globalDataLBound + i*rank + j) = this->globalDataLBound[i][j];
  for (int i=0; (i<deCount && i<len_globalDataUBound[1]); i++)
    for (int j=0; (j<rank && j<len_globalDataUBound[0]); j++)
      *(globalDataUBound + i*rank + j) = this->globalDataUBound[i][j];
  for (int i=0; (i<deCount && i<len_localDataLBound[1]); i++)
    for (int j=0; (j<rank && j<len_localDataLBound[0]); j++)
      *(localDataLBound + i*rank + j) =
        (this->localFullLBound[i][j] + dataOffset[i][j]) + 1;
  for (int i=0; (i<deCount && i<len_localDataUBound[1]); i++)
    for (int j=0; (j<rank && j<len_localDataUBound[0]); j++)
      *(localDataUBound + i*rank + j) =
        (this->localFullLBound[i][j] + dataOffset[i][j]) + 1
        + (this->globalDataUBound[i][j] - this->globalDataLBound[i][j]);
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------



//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayPrint()"
//BOPI
// !IROUTINE:  ESMC_newArrayPrint
//
// !INTERFACE:
int ESMC_newArray::ESMC_newArrayPrint(void){
//
// !RETURN VALUE:
//    int return code
//
//
// !DESCRIPTION:
//    Print ESMC\_newArray
//
//EOPI
//-----------------------------------------------------------------------------
  printf("====== <ESMC_newArrayPrint> ================================\n");
  printf("deCount = %d, rank = %d\n", deCount, rank);
  for (int de=0; de<deCount; de++){
    for (int i=0; i<rank; i++)
      printf("de = %d, dim = %d, globalDataLBound = %d, "
        "globalDataUBound = %d, dataOffset = %d, "
        "localFullLBound = %d, localFullUBound = %d\n",
        de, i, globalDataLBound[de][i], globalDataUBound[de][i],
        dataOffset[de][i], localFullLBound[de][i], localFullUBound[de][i]);
  }
  for (int de=0; de<localDeCount; de++){
    localArrays[de]->ESMC_LocalArrayPrint();
  }
  printf("====== </ESMC_newArrayPrint> ===============================\n");
  return ESMF_SUCCESS;
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayCreate()"
//BOPI
// !IROUTINE:  ESMC_newArrayCreate
//
// !INTERFACE:
ESMC_newArray *ESMC_newArrayCreate(
//
// !RETURN VALUE:
//    ESMC_newArray * to newly allocated ESMC_newArray
//
// !ARGUMENTS:
//
  ESMC_LocalArray *larray,  // pointer to ESMC_LocalArray object
  int *haloWidth,           // halo width
  int deCount,              // number of DEs
  int rootPET,              // root
  int *rc){                 // return code
//
// !DESCRIPTION:
//    This Create method generates a DELayout following a very simple 
//    algorithm that assumes:
//      1. that the DE vertix weight is proportional to the number of logical
//         grid points of the associated array domain - thus it is best to
//         chunk up the entire domain into equal pieces
//EOPI
//-----------------------------------------------------------------------------
  int localrc;
  // Initialize rc and localrc ; assume functions not implemented
  localrc = ESMC_RC_NOT_IMPL;
  if (*rc) *rc = ESMC_RC_NOT_IMPL;

  ESMC_newArray *array;
  // determine required information
  ESMCI::VM *vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  int petCount = vm->getPetCount();
  // check that there is a valid larray on rootPET 
  if (localPet == rootPET){
    if (larray == ESMC_NULL_POINTER){
      ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_BAD, 
        "- must supply a valid 'larray' argument on rootPET.", rc);
      return(ESMC_NULL_POINTER);
    }
  }    
  // allocate and initialize a newArray object via native constructor
  try {
    array = new ESMC_newArray;
  }
  catch (...) {
    ESMC_LogDefault.AllocError(rc);
    return(ESMC_NULL_POINTER);
  }
  // get info about the LocalArray on rootPET and broadcast it to all PETs
  int laRank, *laLength;
  if (localPet == rootPET){
    laRank = larray->ESMC_LocalArrayGetRank();
    vm->broadcast(&laRank, sizeof(int), rootPET);
    laLength = new int[laRank];
    larray->ESMC_LocalArrayGetLengths(laRank, laLength);
    vm->broadcast(laLength, laRank * sizeof(int), rootPET);
  }else{
    vm->broadcast(&laRank, sizeof(int), rootPET);
    laLength = new int[laRank];
    vm->broadcast(laLength, laRank * sizeof(int), rootPET);
  }
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayCreate: %d, %d\n", laRank, laLength[0]);
#endif
  // ensure a valid haloWidth array on rootPET
  int haloWidthAllocFlag = 0;
  if (localPet == rootPET){
    if (haloWidth == NULL){
      // need to fill in a valid haloWidth, default is a width of 0 in all dims
      haloWidthAllocFlag = 1;
      haloWidth = new int[laRank];
      for (int i=0; i<laRank; i++)
        haloWidth[i] = 0;
    }
  }    
  // broadcast rootPET's haloWidth array
  vm->broadcast(&haloWidth, sizeof(int *), rootPET);
  // don't worry, it is o.k. to overwrite local variables!
  if (haloWidth != NULL){
    if (localPet != rootPET)
      haloWidth = new int[laRank];
    vm->broadcast(haloWidth, laRank * sizeof(int), rootPET);
  }
  
  // determine the decomposition rank
  int decompRank = 0;
  if (haloWidth != NULL){
    // haloWidth array exists and needs to be evaluated
    for (int i=0; i<laRank; i++)
      if (haloWidth[i]>=0) ++decompRank;
  }else{
    // haloWidth array does not exist, by default all are decomp. dimensions
    decompRank = laRank;  // 
  }
  
  // find the haloVolume for all decomposition dimensions
  int *haloVolume = new int[decompRank];
  int *decompMap = new int[decompRank];
  if (haloWidth != NULL){
#if (VERBOSITY > 9)
    printf("gjt in ESMC_newArrayCreate: haloWidth available\n");
#endif
    // determine halo volumes of all ranks
    int k=0;
    for (int i=0; i<laRank; i++){
      if (haloWidth[i]>=0){
        decompMap[k] = i;
        haloVolume[k] = haloWidth[i];
        for (int j=0; j<i; j++)
          haloVolume[k] *= laLength[j];
        for (int j=i+1; j<laRank; j++)
          haloVolume[k] *= laLength[j];
        ++k;
      }
    }
  }else{
    // use a fake haloVolume that makes the decomposition algorithm work
    for (int i=0; i<laRank; i++){
      decompMap[i] = i;
      haloVolume[i] = laLength[i];
    }
  }
#if (VERBOSITY > 9)
  for (int i=0; i<decompRank; i++)
    printf("gjt in ESMC_newArrayCreate: haloVolume[%d]=%d\n", i, haloVolume[i]);
#endif
  // need to broadcast rootPET's deCount
  vm->broadcast(&deCount, sizeof(int), rootPET);
  // don't worry, it is o.k. to overwrite local variables!
  if (deCount == 0) deCount = petCount; // by default chose deCount = petCount

  // determine best DELayout to decompose larray data
  // todo: this is a trivial implementation that only works if the number of
  // DEs is smaller or equal the length of the dimension with the smallest
  // haloVolume. A more complete implementation would go through a factorization
  // of the deCount with as many factors as there are decompRank and pick the
  // decomposition with overall smallest sum of haloVolumes. 
  int *deLength = new int[decompRank];    // number of DEs along this dimension
  for (int i=0; i<decompRank; i++)
    deLength[i] = 1;        // reset to length one
  int minIndex=0;
  for (int i=0; i<decompRank; i++)
    if (haloVolume[i] < haloVolume[minIndex]) minIndex = i;
  if (laLength[decompMap[minIndex]]/deCount >= haloWidth[minIndex]){
    deLength[minIndex] = deCount; // set the deLength of the smallest haloVolume
  }else{
    printf("found error in ESMC_newArrayCreate\n");
  }
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayCreate: decompRank=%d\n", decompRank);
  for (int i=0; i<decompRank; i++)
    printf("gjt in ESMC_newArrayCreate: %d, deLength=%d, decompMap=%d\n", i,    
      deLength[i], decompMap[i]);
#endif
  // now there is enough information to create a DELayout
  ESMCI::DELayout *delayout = ESMCI::DELayout::create(*vm, deLength, decompRank,
    NULL, 0, NULL, &localrc);
#if (VERBOSITY > 9)
  delayout->ESMC_DELayoutPrint();
#endif
  
  // finally construct the newArray according to the DELayout
  localrc = array->ESMC_newArrayConstruct(larray, haloWidth, delayout, rootPET);

  // garbage collection
  delete [] laLength;
  if (haloWidth != NULL){
    if (localPet != rootPET || haloWidthAllocFlag)
      delete [] haloWidth;
  }
  delete [] haloVolume;
  delete [] decompMap;
  delete [] deLength;
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayCreate, array: %p, %d, %d\n", array, localPet, 
    petCount);
#endif
  
  // error handling via LogErr
  if (ESMC_LogDefault.MsgFoundError(localrc, ESMCI_ERR_PASSTHRU, rc))
    return ESMC_NULL_POINTER;
  // return handle to the newArray object
  return(array);
}
//-----------------------------------------------------------------------------


//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayDestroy()"
//BOPI
// !IROUTINE:  ESMC_newArrayDestroy
//
// !INTERFACE:
int ESMC_newArrayDestroy(
//
// !RETURN VALUE:
//    int return code
//
// !ARGUMENTS:
//
  ESMC_newArray **array){      // in - ESMC_newArray to destroy
//
// !DESCRIPTION:
//
//EOPI
//-----------------------------------------------------------------------------
int localrc;
//Initialize localrc; assume routine not implemented
localrc = ESMC_RC_NOT_IMPL;

  if (*array != ESMC_NULL_POINTER) {
#if 0
    (*array)->ESMC_newArrayDestruct();
#endif
    delete (*array);
    *array = ESMC_NULL_POINTER;
    return(ESMF_SUCCESS);
  }else{
    ESMC_LogDefault.Write("Cannot delete bad newArray object.", 
      ESMC_LOGMSG_ERROR);
    return(ESMC_RC_PTR_NULL);
  }
return localrc;
}
//-----------------------------------------------------------------------------








//--------------------------
// functions that provide thread support for non-blocking comms
//--------------------------




//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayScatterThread()"
//BOPI
// !IROUTINE:  ESMC_newArrayScatterThread
//
// !INTERFACE:
void *ESMC_newArrayScatterThread(
//
// !RETURN VALUE:
//    return value required by pthread standard
//
// !ARGUMENTS:
//
      void *arg){               // pointer to information structure
//
// !DESCRIPTION:
//    Scatter the contents of an {\tt ESMC\_LocalArray} across the
//    {\tt ESMC\_newArray}. DE-based non-blocking paradigm.
//
//EOPI
//-----------------------------------------------------------------------------
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScatter(THREAD): DE-based nb paradigm\n");
#endif
  int localrc;
  ESMC_newArrayThreadArg *tharg = (ESMC_newArrayThreadArg *)arg;
  // setup some useful local variables:
  ESMC_newArray *array = tharg->array;
  ESMCI::VM *vm = tharg->vm;
  int de = tharg->de;
  int rootPET = tharg->rootPET;
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScatter(THREAD): de=%d\n", de);
#endif
  // determine required information
  if (vm==NULL)
    vm = ESMCI::VM::getCurrent(&localrc);  // get current VM context
  int localPet = vm->getLocalPet();
  // determine localDE index
  int localDeCount = array->localDeCount;
  int *localDeToDeMap = array->localDeToDeMap;
  int localDe;
  for (localDe=0; localDe<localDeCount; localDe++)
    if (localDeToDeMap[localDe] == de) break;
  // determine VAS for rootPET
  int rootVAS = vm->getVas(rootPET);
  // receive some of larray's meta info from scatter root method
  int rank = array->rank;
  int *laLength = new int[rank];
  VMK::commhandle *ch_laLength = NULL; // mark as invalid
  vm->recv(laLength, rank * sizeof(int), rootVAS, &ch_laLength, de+3000);
  int laByteCount;
  VMK::commhandle *ch_laByteCount = NULL; // mark as invalid
  vm->recv(&laByteCount, sizeof(int), rootVAS, &ch_laByteCount, de+3000);
  int *laLbound = new int[rank];
  VMK::commhandle *ch_laLbound = NULL; // mark as invalid
  vm->recv(laLbound, rank * sizeof(int), rootVAS, &ch_laLbound, de+3000);
  // get info out of the associated localArrays
  ESMC_LocalArray **localArrays = array->localArrays;
  void **localDeArrayBase = new void*[localDeCount];
  for (int i=0; i<localDeCount; i++)
    localArrays[i]->ESMC_LocalArrayGetBaseAddr(&localDeArrayBase[i]);
  // prepare a temporary buffer
  vm->commwait(&ch_laLength, NULL, 1);     // need this variable in a couple of lines
  vm->commwait(&ch_laByteCount, NULL, 1);  // need this variable in a couple of lines
  int blockCount = 1;
  for (int i=1; i<rank; i++)
    blockCount *= laLength[i];
  int blockSize = laByteCount / blockCount;
  int elementSize = blockSize / laLength[0];
#if (VERBOSITY > 9)
  for (int i=0; i<rank; i++)
    printf("gjt in ESMC_newArrayScatter(THREAD): de=%d, laLenght[%d] = %d\n",
      de, i, laLength[i]);
  printf("gjt in ESMC_newArrayScatter(THREAD): elementSize = %d\n",
    elementSize);
#endif
  char *buffer;
  buffer = new char[blockSize];   // buffer for first dim.
  // setup a blockID which basically is an array holding the higher dimensions
  int *blockID = new int[rank];   // blockID[0] is unused!
  for (int i=0; i<rank; i++)
    blockID[i] = 0;
  int *blockLocalIndex = new int[rank];
  int *blockGlobalIndex = new int[rank];
  vm->commwait(&ch_laLbound, NULL, 1);
  blockGlobalIndex[0] = laLbound[0];
  // prepare for loop over all blocks
  VMK::commhandle *ch_buffer = NULL; // mark as invalid
  int **globalDataLBound = array->globalDataLBound;
  int **globalDataUBound = array->globalDataUBound;
  int **localFullLBound = array->localFullLBound; 
  int **localFullUBound = array->localFullUBound; 
  int **globalFullLBound = array->globalFullLBound;
  int **globalFullUBound = array->globalFullUBound;
  int **dataOffset = array->dataOffset;      
  // now loop over all blocks
  for (int blk=0; blk<blockCount; blk++){
    for (int i=1; i<rank; i++)
      blockGlobalIndex[i] = laLbound[i] + blockID[i];
#if (VERBOSITY > 9)
    printf("gjt in ESMC_newArrayScatter(THREAD): blockID: ");
    for (int i=1; i<rank-1; i++)
      printf("%d, ", blockID[i]);
    printf("%d\n", blockID[rank-1]);
#endif
    // use ide instead of localDe in the following lines...
    int ide = localDe;
    // receive block data from root
    vm->recv(buffer, blockSize, rootVAS, &ch_buffer, de+3000);
    // check whether this DE's fullBox intersects the current block...
    int ii;
    for (ii=1; ii<rank; ii++){
#if (VERBOSITY > 9)
      printf("gjt in ESMC_newArrayScatter(THREAD): ide=%d,"
        " blockGlobalIndex=%d\n globalXXX: %d, %d, %d, %d\n", ide, 
        blockGlobalIndex[ii], globalFullLBound[de][ii],
        globalDataLBound[de][ii], globalDataUBound[de][ii],
        globalFullUBound[de][ii]);
#endif    
      // check intersection with dataBox
      if (blockGlobalIndex[ii] >= globalDataLBound[de][ii] &&
        blockGlobalIndex[ii] <= globalDataUBound[de][ii]){
#if (VERBOSITY > 9)
        printf("gjt in ESMC_newArrayScatter(THREAD): intersect dataBox\n");
#endif    
        // found intersection with dataBox
        blockLocalIndex[ii] = blockGlobalIndex[ii] 
          - globalDataLBound[de][ii]
          + dataOffset[de][ii];
        continue;
      }
      //check intersection with lower halo region
      if (globalFullLBound[de][ii] <= globalDataLBound[de][ii]){
        // this is a regular lower halo region
        if (blockGlobalIndex[ii] >= globalFullLBound[de][ii] &&
          blockGlobalIndex[ii] < globalDataLBound[de][ii]){
#if (VERBOSITY > 9)
          printf("gjt in ESMC_newArrayScatter(THREAD): intersect regular lower"
                " halo\n");
#endif    
          // found intersection with regular lower halo region
          blockLocalIndex[ii] = blockGlobalIndex[ii] 
            - globalFullLBound[de][ii];
          continue;
        }
      }else{
        // this is a periodic lower halo region from the other end
        if (blockGlobalIndex[ii] >= globalFullLBound[de][ii] &&
          blockGlobalIndex[ii] <=
            (globalFullLBound[de][ii] + dataOffset[de][ii])){
#if (VERBOSITY > 9)
          printf("gjt in ESMC_newArrayScatter(THREAD): intersect periodic lower"
            " halo\n");
#endif    
          // found intersection with periodic lower halo region
          blockLocalIndex[ii] = blockGlobalIndex[ii] 
            - globalFullLBound[de][ii];
          continue;
        }
      }
      //check intersection with upper halo region
      if (globalFullUBound[de][ii] >= globalDataUBound[de][ii]){
        // this is a regular upper halo region
        if (blockGlobalIndex[ii] > globalDataUBound[de][ii] &&
          blockGlobalIndex[ii] <= globalFullUBound[de][ii]){
#if (VERBOSITY > 9)
          printf("gjt in ESMC_newArrayScatter(THREAD): intersect regular upper"
            " halo\n");
#endif    
          // found intersection with regular upper halo region
          blockLocalIndex[ii] = blockGlobalIndex[ii] 
            - globalDataLBound[de][ii] + dataOffset[de][ii];
          continue;
        }
      }else{
        // this is a periodic upper halo region from the other end
        int upperDataOffset = 
          localFullUBound[de][ii] - localFullLBound[de][ii]
          - dataOffset[de][ii]
          - (globalDataUBound[de][ii] - globalDataLBound[de][ii]);
        if (blockGlobalIndex[ii] >= 
          (globalFullUBound[de][ii] - upperDataOffset) &&
          blockGlobalIndex[ii] <= globalFullUBound[de][ii]){
#if (VERBOSITY > 9)
          printf("gjt in ESMC_newArrayScatter(THREAD): intersect periodic upper"
            " halo\n");
#endif    
          // found intersection with periodic upper halo region
          blockLocalIndex[ii] = blockGlobalIndex[ii]
            - (globalFullUBound[de][ii] - upperDataOffset);
          continue;
        }
      }
      break;
    }        
#if (VERBOSITY > 9)
    if (ii==rank)
      printf("gjt in ESMC_newArrayScatter(THREAD): block inside fullBox\n");
    else
      printf("gjt in ESMC_newArrayScatter(THREAD): block outside fullBox\n");
#endif
    blockLocalIndex[0] = blockGlobalIndex[0] - globalFullLBound[de][0];
    if (ii==rank){
      // block intersects fullBox ->
      // find start address of DE's data column overlaping with block data
      char *base = (char *)localDeArrayBase[ide];
      int elementCount = 0;
      for (int i=rank-1; i>0; i--){
        elementCount += blockLocalIndex[i];
        elementCount *= (localFullUBound[de][i-1] 
          - localFullLBound[de][i-1] + 1);
      }
      base += elementCount * elementSize;
#if (VERBOSITY > 9)
      printf("gjt in ESMC_newArrayScatter(THREAD): elementCount = %d, base "
        "=%p\n", elementCount, base);
#endif
      // determine start pointers of buffer and base overlap, overlapCount
      char *baseOverlap = base;
      char *blockOverlap = buffer;
      int overlapCount;
      int fullBoxCount = localFullUBound[de][0] - localFullLBound[de][0] 
        + 1;
      if (blockLocalIndex[0] < 0){
        blockOverlap += (-blockLocalIndex[0]) * elementSize;
        overlapCount = laLength[0] + blockLocalIndex[0];
        if (fullBoxCount < overlapCount)
          overlapCount = fullBoxCount;
      }else{
        baseOverlap += blockLocalIndex[0] * elementSize;
        overlapCount = fullBoxCount - blockLocalIndex[0];
        if (laLength[0] < overlapCount)
          overlapCount = laLength[0];
      }
      if (overlapCount < 0) 
        overlapCount = 0;
#if (VERBOSITY > 9)
      printf("gjt in ESMC_newArrayScatter(THREAD): overlapCount = %d,"
        " blockOverlap = %p, baseOverlap = %p\n", overlapCount, blockOverlap,
        baseOverlap);
#endif
      // going to need the data in buffer for the following memcpy call...
      vm->commwait(&ch_buffer, NULL, 1);
      // finally memcpy from buffer into DE's local array.
      memcpy(baseOverlap, blockOverlap, overlapCount * elementSize);
    }else{
      // block does not intersect fullBox
//gjt      vm->cancel(&ch_buffer);
      // gjt took the above line out again because canceling an MPI message is
      // much more involved than simply calling cancel on one side! In case the
      // cancel succeeds on the receiver side there is still the message queued
      // on the sender side and will mess up the message queue! In order to
      // cancel and remove the message correctly there will need to be
      // communication between sender and receiver! For this to be done behind
      // the scenes in a non-blocking approach will require extra threads!
      vm->commwait(&ch_buffer, NULL, 1);
    }
    // update the blockID
    ++blockID[1];
    for (int i=1; i<rank-1; i++)
      if (blockID[i] >= laLength[i]){
        blockID[i] = 0;
        ++blockID[i+1];
      }
  }
  // garbage collection
  delete [] blockGlobalIndex;
  delete [] blockLocalIndex;
  delete [] blockID;
  delete [] buffer;
  delete [] laLength;
  delete [] localDeArrayBase;

#if (VERBOSITY > 9)
      printf("gjt in ESMC_newArrayScatter(THREAD): returning\n");
#endif
    
  return NULL;
}
//-----------------------------------------------------------------------------




//-----------------------------------------------------------------------------
#undef  ESMC_METHOD
#define ESMC_METHOD "ESMC_newArrayScalarReduceThread()"
//BOPI
// !IROUTINE:  ESMC_newArrayScalarReduceThread
//
// !INTERFACE:
void *ESMC_newArrayScalarReduceThread(
//
// !RETURN VALUE:
//    return value required by pthread standard
//
// !ARGUMENTS:
//
      void *arg){               // pointer to information structure
//
// !DESCRIPTION:
//    Reduce the contents of an {\tt ESMC\_newArray}
//
//EOPI
//-----------------------------------------------------------------------------
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScalarReduce(THREAD): DE-based nb paradigm\n");
#endif
  int localrc;
  ESMC_newArrayThreadArg *tharg = (ESMC_newArrayThreadArg *)arg;
  // setup some useful local variables:
  ESMC_newArray *array = tharg->array;
  ESMCI::VM *vm = tharg->vm;
  int rootPET = tharg->rootPET;
  void *result = tharg->result;
  ESMC_TypeKind_Flag dtk = tharg->dtk;
  ESMC_Operation op = tharg->op;
  // prepeare PET-local temporary result variable
  int deCount = array->deCount;
  void *localResult;
  int size;
  switch (dtk){
  case ESMC_TYPEKIND_I4:
    localResult = new ESMC_I4[deCount];
    size = 4;
    break;
  case ESMC_TYPEKIND_R4:
    localResult = new ESMC_R4[deCount];
    size = 4;
    break;
  case ESMC_TYPEKIND_R8:
    localResult = new ESMC_R8[deCount];
    size = 8;
    break;
  default:
    ESMC_LogDefault.MsgFoundError(ESMC_RC_ARG_INCOMP, 
      "- data type kind is unknown to VM", &localrc);
    return NULL;
  }
  // loop over all DEs and issue nb receive localResult
  char *localResultChar = (char *)localResult;
  VMK::commhandle **vmk_commh = new VMK::commhandle*[deCount];
  int *deVASList = array->deVASList;
  for (int de=0; de<deCount; de++){
    vmk_commh[de] = NULL;    // mark as invalid
    vm->recv(localResultChar, size, deVASList[de], &(vmk_commh[de]),
      de+5000);
    
#if (VERBOSITY > 9)
  printf("gjt in ESMC_newArrayScalarReduce(THREAD): issued receive for de=%d"
    " in deVAS: %d, and obtained handle %p\n", de, deVASList[de],
        vmk_commh[de]);
#endif
  
    localResultChar += size;
  }
  // reduce to final result
  switch (dtk){
  case ESMC_TYPEKIND_I4:
    {
      ESMC_I4 *tempResult = (ESMC_I4 *)result;
      ESMC_I4 *tempBase = (ESMC_I4 *)localResult;
      switch (op){
      case ESMF_SUM:
        *tempResult = 0;  // prime the result variable
        for (int i=0; i<deCount; i++){
          vm->commwait(&(vmk_commh[i]), NULL, 1);  // complete receive, nanopause=1ns
          *tempResult += tempBase[i];
        }
        break;
      case ESMF_MIN:
        vm->commwait(&(vmk_commh[0]), NULL, 1);   // complete receive, nanopause=1ns
        *tempResult = tempBase[0];          // prime the result variable
        for (int i=1; i<deCount; i++){
          vm->commwait(&(vmk_commh[i]), NULL, 1);  // complete receive, nanopause=1ns
          if (tempBase[i] < *tempResult) *tempResult = tempBase[i];
        }
        break;
      case ESMF_MAX:
        vm->commwait(&(vmk_commh[0]), NULL, 1);   // complete receive, nanopause=1ns
        *tempResult = tempBase[0];          // prime the result variable
        for (int i=1; i<deCount; i++){
          vm->commwait(&(vmk_commh[i]), NULL, 1);  // complete receive, nanopause=1ns
          if (tempBase[i] > *tempResult) *tempResult = tempBase[i];
        }
        break;
      }
    }
    break;
  case ESMC_TYPEKIND_R4:
    {
      ESMC_R4 *tempResult = (ESMC_R4 *)result;
      ESMC_R4 *tempBase = (ESMC_R4 *)localResult;
      switch (op){
      case ESMF_SUM:
        *tempResult = 0.;  // prime the result variable
        for (int i=0; i<deCount; i++){
          vm->commwait(&(vmk_commh[i]), NULL, 1);  // complete receive, nanopause=1ns
          *tempResult += tempBase[i];
        }
        break;
      case ESMF_MIN:
        vm->commwait(&(vmk_commh[0]), NULL, 1);   // complete receive, nanopause=1ns
        *tempResult = tempBase[0];          // prime the result variable
        for (int i=1; i<deCount; i++){
          vm->commwait(&(vmk_commh[i]), NULL, 1);  // complete receive, nanopause=1ns
          if (tempBase[i] < *tempResult) *tempResult = tempBase[i];
        }
        break;
      case ESMF_MAX:
        vm->commwait(&(vmk_commh[0]), NULL, 1);   // complete receive, nanopause=1ns
        *tempResult = tempBase[0];          // prime the result variable
        for (int i=1; i<deCount; i++){
          vm->commwait(&(vmk_commh[i]), NULL, 1);  // complete receive, nanopause=1ns
          if (tempBase[i] > *tempResult) *tempResult = tempBase[i];
        }
        break;
      }
    }
    break;
  case ESMC_TYPEKIND_R8:
    {
      ESMC_R8 *tempResult = (ESMC_R8 *)result;
      ESMC_R8 *tempBase = (ESMC_R8 *)localResult;
      switch (op){
      case ESMF_SUM:
        *tempResult = 0.;  // prime the result variable
        for (int i=0; i<deCount; i++){
          vm->commwait(&(vmk_commh[i]), NULL, 1);  // complete receive, nanopause=1ns
          *tempResult += tempBase[i];
        }
        break;
      case ESMF_MIN:
        vm->commwait(&(vmk_commh[0]), NULL, 1);   // complete receive, nanopause=1ns
        *tempResult = tempBase[0];          // prime the result variable
        for (int i=1; i<deCount; i++){
          vm->commwait(&(vmk_commh[i]), NULL, 1);  // complete receive, nanopause=1ns
          if (tempBase[i] < *tempResult) *tempResult = tempBase[i];
        }
        break;
      case ESMF_MAX:
        vm->commwait(&(vmk_commh[0]), NULL, 1);   // complete receive, nanopause=1ns
        *tempResult = tempBase[0];          // prime the result variable
        for (int i=1; i<deCount; i++){
          vm->commwait(&(vmk_commh[i]), NULL, 1);  // complete receive, nanopause=1ns
          if (tempBase[i] > *tempResult) *tempResult = tempBase[i];
        }
        break;
      }
    }
    break;
  }

  // garbage collection
  delete [] localResult;
  
  return NULL;
}
//-----------------------------------------------------------------------------

#endif
