# Copyright 2018 The Emscripten Authors.  All rights reserved.
# Emscripten is available under two separate licenses, the MIT license and the
# University of Illinois/NCSA Open Source License.  Both these licenses can be
# found in the LICENSE file.

import os
import shutil
import logging

TAG = '1.7.5'
HASH = 'c2c13fc97bb74f0f13092b07804f7087e948bce49793f48b62c2c24a5792523acc0002840bebf21829172bb2e7c3df9f9625250aec6c786a55489667dd04d6a0'

deps = ['freetype']


def needed(settings):
  return settings.USE_HARFBUZZ


def get_lib_name(settings):
  return 'libharfbuzz' + ('-mt' if settings.USE_PTHREADS else '') + '.a'


def get(ports, settings, shared):
  ports.fetch_project('harfbuzz', 'https://github.com/harfbuzz/harfbuzz/releases/download/' +
                      TAG + '/harfbuzz-' + TAG + '.tar.bz2', 'harfbuzz-' + TAG, sha512hash=HASH)

  def create(final):
    logging.info('building port: harfbuzz')
    ports.clear_project_build('harfbuzz')

    source_path = os.path.join(ports.get_dir(), 'harfbuzz', 'harfbuzz-' + TAG)
    build_path = os.path.join(ports.get_build_dir(), 'harfbuzz')

    freetype_lib = shared.Cache.get_path(shared.Cache.get_lib_name('libfreetype.a'))
    freetype_include = os.path.join(ports.get_include_dir(), 'freetype2', 'freetype')
    freetype_include_dirs = freetype_include + ';' + os.path.join(freetype_include, 'config')

    cmake_cmd = [
      shared.EMCMAKE,
      'cmake',
      '-B' + build_path,
      '-H' + source_path,
      '-DCMAKE_BUILD_TYPE=Release',
      '-DCMAKE_INSTALL_PREFIX=' + build_path,
      '-DFREETYPE_INCLUDE_DIRS=' + freetype_include_dirs,
      '-DFREETYPE_LIBRARY=' + freetype_lib,
      '-DHB_HAVE_FREETYPE=ON'
    ]

    extra_cflags = []

    if settings.RELOCATABLE:
      extra_cflags.append('-fPIC')

    if settings.USE_PTHREADS:
      extra_cflags.append('-pthread')

    if len(extra_cflags):
      cmake_cmd += ['-DCMAKE_CXX_FLAGS="{}"'.format(' '.join(extra_cflags))]
      cmake_cmd += ['-DCMAKE_C_FLAGS="{}"'.format(' '.join(extra_cflags))]

    shared.run_process(cmake_cmd)
    shared.run_process(['cmake', '--build', build_path, '--target', 'install'])

    ports.install_header_dir(os.path.join(build_path, 'include', 'harfbuzz'))

    shutil.copyfile(os.path.join(build_path, 'libharfbuzz.a'), final)

  return [shared.Cache.get_lib(get_lib_name(settings), create, what='port')]


def clear(ports, settings, shared):
  shared.Cache.erase_lib(get_lib_name(settings))


def process_dependencies(settings):
  settings.USE_FREETYPE = 1


def process_args(ports):
  return ['-I' + os.path.join(ports.get_build_dir(), 'harfbuzz', 'include', 'harfbuzz')]


def show():
  return 'harfbuzz (USE_HARFBUZZ=1; MIT license)'
