static const char rcsid[] = "$Id: cvslock.c,v 1.7 2000/03/18 07:13:53 roessler Exp $";

/*
 *     Copyright (C) 1998,2000 Thomas Roessler <roessler@guug.de>
 * 
 *     This program is free software; you can redistribute it
 *     and/or modify it under the terms of the GNU General Public
 *     License as published by the Free Software Foundation; either
 *     version 2 of the License, or (at your option) any later
 *     version.
 * 
 *     This program is distributed in the hope that it will be
 *     useful, but WITHOUT ANY WARRANTY; without even the implied
 *     warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
 *     PURPOSE.  See the GNU General Public License for more
 *     details.
 * 
 *     You should have received a copy of the GNU General Public
 *     License along with this program; if not, write to the Free
 *     Software Foundation, Inc., 59 Temple Place - Suite 330, 
 *     Boston, MA  02111, USA.
 */

/*
 * This program is used to safely manipulate a CVS repository. The
 * locking algorithm has been taken from the "Locks" node of CVS
 * 1.9.6's texinfo documentation. -tlr
 */

#include <sys/stat.h>
#include <sys/types.h>
#include <sys/utsname.h>

#include <unistd.h>
#include <fcntl.h>
#include <dirent.h>
#include <limits.h>
#include <errno.h>
#include <signal.h>

#include <stdio.h>
#include <stdlib.h>
#include <string.h>

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#ifndef HAVE_SNPRINTF
extern int snprintf (char *, size_t, const char *,...);
#endif

const char CVS_MASTER_LOCK[] = "#cvs.lock";
const char CVS_READ_PREFIX[] = "#cvs.rfl";
const char CVS_WRITE_PREFIX[] = "#cvs.wfl";

#define CL_READ		(1 << 0)
#define CL_WRITE	(1 << 2)
#define CL_RELMASTER	(1 << 3)
#define CL_COMMAND	(1 << 4)
#define CL_UNLOCK	(1 << 5)
#define CL_LOCK		(1 << 6)
#define CL_QUIET	(1 << 7)

struct master_lock
{
  char *path;
  struct master_lock *next;
};

extern char *optarg;
extern int optind;

struct master_lock *LockList = NULL;
pid_t Pid;
char *Hostname;
char *CvsRoot;
char *Command;
int Signalled = 0;

int cl_register_master (const char *);
int cl_unregister_master (const char *);
int cl_have_master (const char *);
int cl_try_master (const char *);
int cl_release_master (const char *);
int cl_releas_all_masters (const char *);
int cl_unlock_dir (const char *, int);
int cl_lock_dir (const char *, int);
int cl_lock_tree (const char *, int);
int cl_unlock_tree (const char *, int);

void cl_release_all_masters (void);
void cl_make_lock_name (char *, size_t, const char *, int);


static void signal_handler (int);
static void set_signals (void);


int main (int argc, char *argv[])
{
  int i, flags = 0, count;
  struct utsname utb;
  char buffer[_POSIX_PATH_MAX];
  int (*lockfn) (const char *, int);
  int (*unlockfn) (const char *, int);

  Pid = getpid ();

  uname (&utb);
  Hostname = strdup (utb.nodename);

  CvsRoot = getenv ("CVSROOT");

  flags = CL_READ | CL_LOCK;

  /* by default, lock recursively */
  lockfn = cl_lock_tree;
  unlockfn = cl_unlock_tree;

  while ((i = getopt (argc, argv, "qp:d:RWusc:l")) != EOF)
  {
    switch (i)
    {
    case 'q':
      flags |= CL_QUIET;
      break;
    case 'p':
      Pid = atoi (optarg);
      break;
    case 'd':
      CvsRoot = optarg;
      break;
    case 'R':
      flags &= ~CL_WRITE;
      flags |= CL_READ;
      break;
    case 'W':
      flags &= ~CL_READ;
      flags |= CL_WRITE;
      break;
    case 's':
      flags |= CL_COMMAND;
      if (!(Command = getenv ("CVSLOCK_SHELL")))
	Command = getenv ("SHELL");
      break;
    case 'c':
      flags |= CL_COMMAND;
      Command = optarg;
      break;
    case 'u':
      flags &= ~CL_LOCK;
      flags |= CL_UNLOCK;
      break;
    case 'l':
      lockfn = cl_lock_dir;
      unlockfn = cl_unlock_dir;
      break;
    default:
      fprintf (stderr, "Usage: %s <parameters>\n", argv[0]);
      exit (1);
    }
  }

  if (optind == argc)
  {
    fprintf (stderr, "%s: There's no directory parameter\n", argv[0]);
    exit (1);
  }

  if (!CvsRoot)
  {
    fprintf (stderr, "%s: Don't know where to find your CVS root.\n", argv[0]);
    exit (1);
  }

  if ((flags & CL_COMMAND) && !Command)
  {
    fprintf (stderr, "%s: Don't know what to execute.\n", argv[0]);
    exit (1);
  }

  if ((flags & CL_COMMAND) && (flags & CL_UNLOCK))
  {
    fprintf (stderr, "%s: Bad command line parameter combination.\n", argv[0]);
    exit (1);
  }

  if (flags & CL_READ)
    flags |= CL_RELMASTER;

  snprintf (buffer, sizeof (buffer), "%s/%s", CvsRoot, argv[optind]);

  set_signals ();

  if (flags & (CL_LOCK | CL_COMMAND))
  {
    for (count = 5; !Signalled && count; count--)
    {
      if ((i = (*lockfn) (buffer, flags)) == 1)
      {
	(*unlockfn) (buffer, flags);
	cl_release_all_masters ();
	fprintf (stderr, "%s: Repository is locked, waiting. [%d]\n", argv[0], count);
	sleep (30);
      }
      else if (i == -1)
      {
	(*unlockfn) (buffer, flags);
	cl_release_all_masters ();
	fprintf (stderr, "%s: Error while locking repository.\n", argv[0]);
	exit (1);
      }
      else if (i == 0)
	break;
      else
      {
	fprintf (stderr, "%s: Internal error, dumping core.\n", argv[0]);
	abort ();
      }
    }

    if (i && !count)
    {
      fprintf (stderr, "%s: Time out while waiting for lock.\n", argv[0]);
      exit (1);
    }
  }

  if (!Signalled && flags & CL_COMMAND)
  {
    if (!(flags & CL_QUIET))
    {
      printf ("%s: `%s' locked successfully.\n", argv[0], buffer);
      printf ("%s: Starting `%s'...\n", argv[0], Command);
    }
    system (Command);
  }

  if (flags & (CL_UNLOCK | CL_COMMAND))
    (*unlockfn) (buffer, flags);

  if (flags & (CL_READ | CL_COMMAND))
    cl_release_all_masters ();

  if (Signalled)
    return 1;
  else
    return 0;
}

static void set_signals ()
{
  struct sigaction act;
  memset (&act, 0, sizeof (struct sigaction));

  act.sa_handler = signal_handler;
  sigaction (SIGTERM, &act, NULL);
  sigaction (SIGHUP, &act, NULL);
  sigaction (SIGQUIT, &act, NULL);
  sigaction (SIGINT, &act, NULL);
}

static void signal_handler (int sig)
{
  Signalled = 1;
}

int cl_register_master (const char *path)
{
  struct master_lock *p;

  if (!(p = malloc (sizeof (struct master_lock))))
  {
    fprintf (stderr, "cl_register_master: %s\n", strerror (errno));
    return -1;
  }

  if (!(p->path = strdup (path)))
  {
    fprintf (stderr, "cl_register_master: %s\n", strerror (errno));
    free (p);
    return -1;
  }

  p->next = LockList;
  LockList = p;
  return 0;
}

int cl_unregister_master (const char *path)
{
  struct master_lock *p, **q;

  q = &LockList;

  for (p = LockList; p; p = p->next)
  {
    if (!strcmp (p->path, path))
    {
      *q = p->next;
      free (p->path);
      free (p);
      return 0;
    }
    q = &p->next;
  }

  return 1;
}

int cl_have_master (const char *path)
{
  struct master_lock *p;

  for (p = LockList; p; p = p->next)
    if (!strcmp (p->path, path))
      return 1;

  return 0;
}

int cl_try_master (const char *path)
{
  char buff[_POSIX_PATH_MAX];

  if (cl_register_master (path) == -1)
    return -1;

  snprintf (buff, sizeof (buff), "%s/%s", path, CVS_MASTER_LOCK);

  if (mkdir (buff, 0700) == -1)
  {
    cl_unregister_master (path);
    if (errno == EEXIST)
      return 1;
    else
      return -1;
  }
  return 0;
}

int cl_release_master (const char *path)
{
  char fname[_POSIX_PATH_MAX];

  if (cl_unregister_master (path) != 0)
    return -1;

  snprintf (fname, sizeof (fname), "%s/%s", path, CVS_MASTER_LOCK);
  return rmdir (fname);
}

void cl_release_all_masters (void)
{
  struct master_lock *p, *q;
  char fname[_POSIX_PATH_MAX];

  for (p = LockList; p; p = q)
  {
    snprintf (fname, sizeof (fname), "%s/%s", p->path, CVS_MASTER_LOCK);

    if (rmdir (fname) == -1)
      fprintf (stderr, "cl_release_all_masters: Couldn't unlink `%s' (%s).\n", fname,
	       strerror (errno));

    q = p->next;
    free (p->path);
    free (p);
  }

  LockList = NULL;
}

void cl_make_lock_name (char *d, size_t l, const char *path, int flags)
{
  snprintf (d, l, "%s/%s.%s.%s.%d", path,
	    (flags & CL_WRITE) ? CVS_WRITE_PREFIX : CVS_READ_PREFIX,
	    "cvslock",
	    Hostname, (int) Pid);
}

int cl_unlock_dir (const char *path, int flags)
{
  char lockfile[_POSIX_PATH_MAX];
  int i;

  cl_make_lock_name (lockfile, sizeof (lockfile), path, flags);
  i = unlink (lockfile);

  /* special case: When removing a write lock from
   * the command line, we have to unlink the master
   * lock directories from here.
   */

  if ((i == 0) && (flags & CL_UNLOCK) && (flags & CL_WRITE))
  {
    snprintf (lockfile, sizeof (lockfile), "%s/%s", path, CVS_MASTER_LOCK);
    rmdir (lockfile);
  }

  return i;
}

int cl_lock_dir (const char *path, int flags)
{
  int i, fd;
  DIR *dir;
  struct dirent *de;
  char lockfile[_POSIX_PATH_MAX];

  if ((i = cl_try_master (path)))
    return i;

  if ((dir = opendir (path)) == NULL)
  {
    cl_release_master (path);
    return -1;
  }

  i = 0;			/* actually, that's guaranteed already. */

  while ((de = readdir (dir)))
  {
    if ((flags & CL_WRITE) &&
	!strncmp (de->d_name, CVS_READ_PREFIX, strlen (CVS_READ_PREFIX)))
    {
      i = 1;
      break;
    }

    if (!strncmp (de->d_name, CVS_WRITE_PREFIX, strlen (CVS_WRITE_PREFIX)))
    {
      i = 1;
      break;
    }
  }

  closedir (dir);

  if (i == 0)
  {
    cl_make_lock_name (lockfile, sizeof (lockfile), path, flags);
    if ((fd = open (lockfile, O_CREAT | O_RDWR, 0500)) == -1)
      i = -1;
    else
      close (fd);
  }

  if (i || flags & CL_RELMASTER)
    cl_release_master (path);

  return i;
}

int cl_lock_tree (const char *path, int flags)
{
  int i;
  struct stat sb;
  DIR *dir;
  struct dirent *de;
  char buff[_POSIX_PATH_MAX];

  if (Signalled)
    return -1;

  if ((i = cl_lock_dir (path, flags)))
    return i;

  if (!(dir = opendir (path)))
  {
    fprintf (stderr, "cl_lock_tree: Couldn't open `%s' (%s).\n",
	     path, strerror (errno));
    return -1;
  }

  while (i == 0 && (de = readdir (dir)))
  {
    snprintf (buff, sizeof (buff), "%s/%s", path, de->d_name);
    if (lstat (buff, &sb) == 0 && S_ISDIR (sb.st_mode)
	&& *de->d_name != '.' && *de->d_name != '#'
	&& strcmp (de->d_name, "Attic") && strcmp (de->d_name, "CVS"))
      i = cl_lock_tree (buff, flags);
  }

  closedir (dir);
  return i;

}

int cl_unlock_tree (const char *path, int flags)
{
  struct stat sb;
  DIR *dir;
  struct dirent *de;
  char buff[_POSIX_PATH_MAX];

  cl_unlock_dir (path, flags);

  if (!(dir = opendir (path)))
  {
    fprintf (stderr, "cl_unlock_tree: Couldn't open `%s' (%s).\n",
	     path, strerror (errno));
    return -1;
  }

  while ((de = readdir (dir)))
  {
    snprintf (buff, sizeof (buff), "%s/%s", path, de->d_name);
    if (lstat (buff, &sb) == 0 && S_ISDIR (sb.st_mode)
	&& *de->d_name != '.' && *de->d_name != '#'
	&& strcmp (de->d_name, "Attic") && strcmp (de->d_name, "CVS"))
      cl_unlock_tree (buff, flags);
  }

  closedir (dir);
  return 0;
}
