/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Writerandom writerandom
*/

#include <cdi.h>

#include "process_int.h"

void *
Writerandom(void *process)
{
  int nrecs;
  int varID, levelID;

  cdoInitialize(process);

  const auto streamID1 = cdoOpenRead(0);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID2 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID2, taxisID2);

  const auto streamID2 = cdoOpenWrite(1);

  cdoDefVlist(streamID2, vlistID2);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      taxisCopyTimestep(taxisID2, taxisID1);
      cdoDefTimestep(streamID2, tsID);

      Varray2D<double> recdata(nrecs);
      std::vector<int> recvarID(nrecs);
      std::vector<int> reclevelID(nrecs);
      std::vector<size_t> recnmiss(nrecs);
      std::vector<int> recindex(nrecs);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          auto gridsize = gridInqSize(vlistInqVarGrid(vlistID1, varID));
          recvarID[recID] = varID;
          reclevelID[recID] = levelID;
          recdata[recID].resize(gridsize);
          cdoReadRecord(streamID1, recdata[recID].data(), &recnmiss[recID]);
        }

      for (int recID = 0; recID < nrecs; recID++) recindex[recID] = -1;

      for (int rindex = nrecs - 1; rindex >= 0; rindex--)
        {
          const int index = (int) (rindex * ((double) rand()) / ((double) RAND_MAX));
          /*	printf("rindex %d %d\n", rindex, index); */
          int ipos = -1;
          for (int recID = 0; recID < nrecs; recID++)
            {
              if (recindex[recID] == -1) ipos++;
              if (recindex[recID] == -1 && ipos == index)
                {
                  recindex[recID] = rindex;
                  break;
                }
            }
        }

      /*
      for ( int recID = 0; recID < nrecs; recID++ )
        printf("recID %d %d\n", recID, recindex[recID]);
      */
      for (int recID = 0; recID < nrecs; recID++)
        if (recindex[recID] == -1) cdoAbort("Internal problem! Random initialize.");

      for (int recID = 0; recID < nrecs; recID++)
        {
          auto rindex = recindex[recID];
          varID = recvarID[rindex];
          levelID = reclevelID[rindex];
          cdoDefRecord(streamID2, varID, levelID);
          cdoWriteRecord(streamID2, recdata[rindex].data(), recnmiss[rindex]);
        }

      tsID++;
    }

  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return 0;
}
