/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2006 Brockmann Consult
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Hi      hi           Compute the humidity index
*/

#include <cdi.h>

#include "cdo_options.h"
#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "cdo_cdiWrapper.h"


static const char HI_NAME[] = "hum_index";
static const char HI_LONGNAME[] = "Humindex describes empirically in units of temperature how the "
                                  "temperature and humidity influence the wellness of a human being. HI = "
                                  "T + 5/9 * (A - 10) with A = e * (6.112 * 10 ** ((7.5 * T)/(237.7 + T)) "
                                  "* R), T  = air temperature in degree Celsius, R = relative humidity, e "
                                  "= vapour pressure. Humindex is only defined for temperatures of at "
                                  "least 26 degree Celsius and relative humidity of at least 40 percent.";
static const char HI_UNITS[] = "Celsius";

static const int FIRST_VAR = 0;

static double
humidityIndex(double t, double e, double r, double missval)
{
  static const double tmin = 26.0;
  static const double rmin = 40.0;

  if (t < tmin || r < rmin) return missval;

  return t + (5.0 / 9.0) * ((0.01 * r * e * 6.112 * std::pow(10.0, (7.5 * t) / (237.7 + t))) - 10.0);
}

static void
farexpr(Field &field1, const Field &field2, const Field &field3, double (*expression)(double, double, double, double))
{
  const int grid1 = field1.grid;
  const double missval1 = field1.missval;
  const int grid2 = field2.grid;
  const double missval2 = field2.missval;
  const int grid3 = field3.grid;
  const double missval3 = field3.missval;

  const auto len = gridInqSize(grid1);
  if (len != gridInqSize(grid2) || len != gridInqSize(grid3)) cdoAbort("Fields have different gridsize (%s)", __func__);

  if (field1.nmiss || field2.nmiss || field3.nmiss)
    {
      for (size_t i = 0; i < len; i++)
        if (DBL_IS_EQUAL(field1.vec[i], missval1) || DBL_IS_EQUAL(field2.vec[i], missval2) || DBL_IS_EQUAL(field3.vec[i], missval3))
          field1.vec[i] = missval1;
        else
          field1.vec[i] = expression(field1.vec[i], field2.vec[i], field3.vec[i], missval1);
    }
  else
    {
      for (size_t i = 0; i < len; i++) field1.vec[i] = expression(field1.vec[i], field2.vec[i], field3.vec[i], missval1);
    }

  field1.nmiss = fieldNumMiss(field1);
}

void *
Hi(void *process)
{
  int nrecs;
  int varID1, varID2, varID3;
  int levelID1, levelID2, levelID3;

  cdoInitialize(process);
  cdoOperatorAdd("hi", 0, 0, nullptr);

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);
  const auto streamID3 = cdoOpenRead(2);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = cdoStreamInqVlist(streamID3);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  // taxisID2 = vlistInqTaxis(vlistID2);
  // taxisID3 = vlistInqTaxis(vlistID3);

  vlistCompare(vlistID1, vlistID2, CMP_DIM);
  vlistCompare(vlistID1, vlistID3, CMP_DIM);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field2, field3;
  field1.resize(gridsizemax);
  field2.resize(gridsizemax);
  field3.resize(gridsizemax);

  if (Options::cdoVerbose)
    cdoPrint("Number of timesteps: file1 %d, file2 %d, file3 %d", vlistNtsteps(vlistID1), vlistNtsteps(vlistID2),
             vlistNtsteps(vlistID3));

  const auto vlistID4 = vlistCreate();
  const auto gridID = vlistInqVarGrid(vlistID1, FIRST_VAR);
  const auto zaxisID = vlistInqVarZaxis(vlistID1, FIRST_VAR);
  const auto varID4 = vlistDefVar(vlistID4, gridID, zaxisID, TIME_VARYING);

  const auto taxisID4 = cdoTaxisCreate(TAXIS_RELATIVE);
  taxisDefTunit(taxisID4, TUNIT_MINUTE);
  taxisDefCalendar(taxisID4, CALENDAR_STANDARD);
  taxisDefRdate(taxisID4, 19550101);
  taxisDefRtime(taxisID4, 0);
  vlistDefTaxis(vlistID4, taxisID4);

  vlistDefVarName(vlistID4, varID4, HI_NAME);
  vlistDefVarLongname(vlistID4, varID4, HI_LONGNAME);
  vlistDefVarUnits(vlistID4, varID4, HI_UNITS);

  const auto streamID4 = cdoOpenWrite(3);

  cdoDefVlist(streamID4, vlistID4);

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      const auto nrecs2 = cdoStreamInqTimestep(streamID2, tsID);
      const auto nrecs3 = cdoStreamInqTimestep(streamID3, tsID);
      if (nrecs2 == 0 || nrecs3 == 0) cdoAbort("Input streams have different number of timesteps!");

      taxisCopyTimestep(taxisID4, taxisID1);
      cdoDefTimestep(streamID4, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID1, &levelID1);
          cdoReadRecord(streamID1, field1.vec.data(), &field1.nmiss);

          cdoInqRecord(streamID2, &varID2, &levelID2);
          cdoReadRecord(streamID2, field2.vec.data(), &field2.nmiss);

          cdoInqRecord(streamID3, &varID3, &levelID3);
          cdoReadRecord(streamID3, field3.vec.data(), &field3.nmiss);

          if (varID1 != varID2 || varID1 != varID3 || levelID1 != levelID2 || levelID1 != levelID3)
            cdoAbort("Input streams have different structure!");

          if (varID1 != FIRST_VAR) continue;

          field1.grid = vlistInqVarGrid(vlistID1, varID1);
          field1.missval = vlistInqVarMissval(vlistID1, varID1);

          field2.grid = vlistInqVarGrid(vlistID2, varID2);
          field2.missval = vlistInqVarMissval(vlistID2, varID2);

          field3.grid = vlistInqVarGrid(vlistID3, varID3);
          field3.missval = vlistInqVarMissval(vlistID3, varID3);

          farexpr(field1, field2, field3, humidityIndex);

          cdoDefRecord(streamID4, varID4, levelID1);
          cdoWriteRecord(streamID4, field1.vec.data(), field1.nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID4);
  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
