/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

*/

#include <cdi.h>

#include "process_int.h"
#include <mpim_grid.h>


void *
Fldrms(void *process)
{
  int lastgrid = -1;
  int index;
  int nrecs;
  int varID, levelID;

  cdoInitialize(process);

  const auto needWeights = true;

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);
  const auto vlistID3 = vlistDuplicate(vlistID1);

  const auto taxisID1 = vlistInqTaxis(vlistID1);
  const auto taxisID3 = taxisDuplicate(taxisID1);
  vlistDefTaxis(vlistID3, taxisID3);

  double slon = 0;
  double slat = 0;
  const auto gridID3 = gridCreate(GRID_LONLAT, 1);
  gridDefXsize(gridID3, 1);
  gridDefYsize(gridID3, 1);
  gridDefXvals(gridID3, &slon);
  gridDefYvals(gridID3, &slat);

  const auto ngrids = vlistNgrids(vlistID1);
  int ndiffgrids = 0;
  for (index = 1; index < ngrids; index++)
    if (vlistGrid(vlistID1, 0) != vlistGrid(vlistID1, index)) ndiffgrids++;

  index = 0;
  const auto gridID1 = vlistGrid(vlistID1, index);
  const auto gridID2 = vlistGrid(vlistID2, index);

  if (gridInqSize(gridID1) != gridInqSize(gridID2)) cdoAbort("Fields have different grid size!");

  if (needWeights && gridInqType(gridID1) != GRID_LONLAT && gridInqType(gridID1) != GRID_GAUSSIAN)
    cdoAbort("Unsupported gridtype: %s", gridNamePtr(gridInqType(gridID1)));

  for (index = 0; index < ngrids; index++) vlistChangeGridIndex(vlistID3, index, gridID3);

  if (ndiffgrids > 0) cdoAbort("Too many different grids!");

  const auto streamID3 = cdoOpenWrite(2);
  cdoDefVlist(streamID3, vlistID3);

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Field field1, field2, field3;
  field1.resize(gridsizemax);
  if (needWeights) field1.weightv.resize(gridsizemax);

  field2.resize(gridsizemax);

  field3.resize(1);
  field3.grid = gridID3;

  int tsID = 0;
  while ((nrecs = cdoStreamInqTimestep(streamID1, tsID)))
    {
      nrecs = cdoStreamInqTimestep(streamID2, tsID);

      taxisCopyTimestep(taxisID3, taxisID1);

      cdoDefTimestep(streamID3, tsID);

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID, &levelID);
          cdoReadRecord(streamID1, field1.vec.data(), &field1.nmiss);
          cdoInqRecord(streamID2, &varID, &levelID);
          cdoReadRecord(streamID2, field2.vec.data(), &field2.nmiss);

          field1.grid = vlistInqVarGrid(vlistID1, varID);
          field2.grid = vlistInqVarGrid(vlistID2, varID);

          if (needWeights && field1.grid != lastgrid)
            {
              lastgrid = field1.grid;
              field1.weightv[0] = 1;
              if (field1.size > 1)
                {
                  const auto wstatus = gridWeights(field1.grid, field1.weightv.data());
                  if (wstatus != 0 && tsID == 0 && levelID == 0)
                    {
                      char varname[CDI_MAX_NAME];
                      vlistInqVarName(vlistID1, varID, varname);
                      cdoWarning("Grid cell bounds not available, using constant grid cell area weights for variable %s!", varname);
                    }
                }
            }

          field1.missval = vlistInqVarMissval(vlistID1, varID);
          field2.missval = vlistInqVarMissval(vlistID1, varID);
          field3.missval = vlistInqVarMissval(vlistID1, varID);

          vfldrms(field1, field2, field3);

          cdoDefRecord(streamID3, varID, levelID);
          cdoWriteRecord(streamID3, field3.vec.data(), field3.nmiss);
        }

      tsID++;
    }

  cdoStreamClose(streamID3);
  cdoStreamClose(streamID2);
  cdoStreamClose(streamID1);

  cdoFinish();

  return nullptr;
}
