/*
  This file is part of CDO. CDO is a collection of Operators to
  manipulate and analyse Climate model Data.

  Copyright (C) 2003-2019 Uwe Schulzweida, <uwe.schulzweida AT mpimet.mpg.de>
  See COPYING file for copying and redistribution conditions.

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; version 2 of the License.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
*/

/*
   This module contains the following operators:

      Diff       diff            Compare two datasets
*/

#include <map>

#include <cdi.h>

#include "functs.h"
#include "process_int.h"
#include "cdo_vlist.h"
#include "param_conversion.h"
#include "cdo_math.h"
#include "mpmo_color.h"
#include "cdo_options.h"
#include "printinfo.h"
#include "pmlist.h"
#include "cdo_zaxis.h"

static void
diffGetParameter(double &abslim, double &abslim2, double &rellim, int &mapflag)
{
  const auto pargc = operatorArgc();
  if (pargc)
    {
      const auto pargv = cdoGetOperArgv();

      KVList kvlist;
      kvlist.name = "DIFF";
      if (kvlist.parseArguments(pargc, pargv) != 0) cdoAbort("Parse error!");
      if (Options::cdoVerbose) kvlist.print();

      for (const auto &kv : kvlist)
        {
          const auto &key = kv.key;
          if (kv.nvalues > 1) cdoAbort("Too many values for parameter key >%s<!", key.c_str());
          if (kv.nvalues < 1) cdoAbort("Missing value for parameter key >%s<!", key.c_str());
          const auto &value = kv.values[0];

          // clang-format off
          if      (key == "abslim")  abslim = parameter2double(value);
          else if (key == "abslim2") abslim2 = parameter2double(value);
          else if (key == "rellim")  rellim = parameter2double(value);
          else if (key == "names")
            {
              if      (value == "left")      mapflag = 1;
              else if (value == "right")     mapflag = 2;
              else if (value == "intersect") mapflag = 3;
              else cdoAbort("Invalid value for key >%s< (names=<left/right/intersect>)", key.c_str(), value.c_str());
            }
          else cdoAbort("Invalid parameter key >%s<!", key.c_str());
          // clang-format on
        }
    }
}

void *
Diff(void *process)
{
  bool lhead = true;
  int nrecs, nrecs2;
  int varID1, varID2 = -1;
  int levelID;
  size_t nmiss1, nmiss2;
  int ndrec = 0, nd2rec = 0, ngrec = 0;
  char varname[CDI_MAX_NAME];
  char paramstr[32];
  double abslim = 0., abslim2 = 1.e-3, rellim = 1.0;

  cdoInitialize(process);

  // clang-format off
  const auto DIFF  = cdoOperatorAdd("diff",  0, 0, nullptr);
  const auto DIFFP = cdoOperatorAdd("diffp", 0, 0, nullptr);
  const auto DIFFN = cdoOperatorAdd("diffn", 0, 0, nullptr);
  const auto DIFFC = cdoOperatorAdd("diffc", 0, 0, nullptr);
  // clang-format on

  const int operatorID = cdoOperatorID();

  int mapflag = 0;

  diffGetParameter(abslim, abslim2, rellim, mapflag);

  if (rellim < -1.e33 || rellim > 1.e+33) cdoAbort("Rel. limit out of range!");
  if (abslim < -1.e33 || abslim > 1.e+33) cdoAbort("Abs. limit out of range!");
  if (abslim2 < -1.e33 || abslim2 > 1.e+33) cdoAbort("Abs2. limit out of range!");

  const auto streamID1 = cdoOpenRead(0);
  const auto streamID2 = cdoOpenRead(1);

  const auto vlistID1 = cdoStreamInqVlist(streamID1);
  const auto vlistID2 = cdoStreamInqVlist(streamID2);

  const auto nvars = vlistNvars(vlistID1);
  std::map<int, int> mapOfVarIDs;

  if (mapflag == 0)
    {
      vlistCompare(vlistID1, vlistID2, CMP_ALL);
      for (int varID = 0; varID < nvars; ++varID) mapOfVarIDs[varID] = varID;
    }
  else
    {
      vlistMap(vlistID1, vlistID2, CMP_ALL, mapflag, mapOfVarIDs);
    }

  const auto gridsizemax = vlistGridsizeMax(vlistID1);

  Varray<double> array1(gridsizemax);
  Varray<double> array2(gridsizemax);
  Varray<double> work(vlistNumber(vlistID1) != CDI_REAL ? 2 * gridsizemax : 0);

  const int taxisID = vlistInqTaxis(vlistID1);

  int indg = 0;
  int tsID = 0;
  while (true)
    {
      nrecs = cdoStreamInqTimestep(streamID1, tsID);
      const auto vdate = taxisInqVdate(taxisID);
      const auto vtime = taxisInqVtime(taxisID);
      const auto vdateString = dateToString(vdate);
      const auto vtimeString = timeToString(vtime);

      nrecs2 = cdoStreamInqTimestep(streamID2, tsID);

      if (nrecs == 0 || nrecs2 == 0) break;

      int recID2next = 0;

      for (int recID = 0; recID < nrecs; recID++)
        {
          cdoInqRecord(streamID1, &varID1, &levelID);

          auto it = mapOfVarIDs.find(varID1);
          if (it == mapOfVarIDs.end())
            {
              if (mapflag == 2 || mapflag == 3) continue;
              cdoAbort("Internal problem: varID1=%d not found!", varID1);
            }

          for (; recID2next < nrecs2; ++recID2next)
            {
              cdoInqRecord(streamID2, &varID2, &levelID);
              if (it->second == varID2)
                {
                  ++recID2next;
                  break;
                }
            }

          if (it->second != varID2 && recID2next == nrecs2)
            cdoAbort("Internal problem: varID2=%d not found in second stream!", it->second);

          indg += 1;

          const auto param = vlistInqVarParam(vlistID1, varID1);
          const auto code = vlistInqVarCode(vlistID1, varID1);
          const auto gridID = vlistInqVarGrid(vlistID1, varID1);
          const auto zaxisID = vlistInqVarZaxis(vlistID1, varID1);
          const auto gridsize = gridInqSize(gridID);
          const auto missval1 = vlistInqVarMissval(vlistID1, varID1);
          const auto missval2 = vlistInqVarMissval(vlistID2, varID2);

          // checkrel = gridInqType(gridID) != GRID_SPECTRAL;
          const bool checkrel = true;

          cdiParamToString(param, paramstr, sizeof(paramstr));

          if (vlistInqVarNumber(vlistID1, varID1) == CDI_COMP)
            {
              cdoReadRecord(streamID1, work.data(), &nmiss1);
              for (size_t i = 0; i < gridsize; ++i) array1[i] = work[i * 2];
            }
          else
            cdoReadRecord(streamID1, array1.data(), &nmiss1);

          if (vlistInqVarNumber(vlistID2, varID2) == CDI_COMP)
            {
              cdoReadRecord(streamID2, work.data(), &nmiss2);
              for (size_t i = 0; i < gridsize; ++i) array2[i] = work[i * 2];
            }
          else
            cdoReadRecord(streamID2, array2.data(), &nmiss2);

          int ndiff = 0;
          bool dsgn = false;
          bool zero = false;
          double absm = 0.0;
          double relm = 0.0;
          double absdiff;

          for (size_t i = 0; i < gridsize; i++)
            {
              if ((std::isnan(array1[i]) && !std::isnan(array2[i])) || (!std::isnan(array1[i]) && std::isnan(array2[i])))
                {
                  ndiff++;
                  relm = 1.0;
                }
              else if (!DBL_IS_EQUAL(array1[i], missval1) && !DBL_IS_EQUAL(array2[i], missval2))
                {
                  absdiff = std::fabs(array1[i] - array2[i]);
                  if (absdiff > 0.) ndiff++;

                  absm = cdo::max(absm, absdiff);

                  if (array1[i] * array2[i] < 0.)
                    dsgn = true;
                  else if (IS_EQUAL(array1[i] * array2[i], 0.))
                    zero = true;
                  else
                    relm = cdo::max(relm, absdiff / cdo::max(std::fabs(array1[i]), std::fabs(array2[i])));
                }
              else if ((DBL_IS_EQUAL(array1[i], missval1) && !DBL_IS_EQUAL(array2[i], missval2))
                       || (!DBL_IS_EQUAL(array1[i], missval1) && DBL_IS_EQUAL(array2[i], missval2)))
                {
                  ndiff++;
                  relm = 1.0;
                }
            }

          if (!Options::silentMode || Options::cdoVerbose)
            {
              if (absm > abslim || (checkrel && relm >= rellim) || Options::cdoVerbose)
                {
                  if (lhead)
                    {
                      lhead = false;

                      fprintf(stdout, "               Date     Time   Level Gridsize    Miss ");
                      fprintf(stdout, "   Diff ");
                      fprintf(stdout, ": S Z  Max_Absdiff Max_Reldiff : ");

                      if (operatorID == DIFFN)
                        fprintf(stdout, "Parameter name");
                      else if (operatorID == DIFF || operatorID == DIFFP)
                        fprintf(stdout, "Parameter ID");
                      else if (operatorID == DIFFC)
                        fprintf(stdout, "Code number");

                      fprintf(stdout, "\n");
                    }

                  if (operatorID == DIFFN) vlistInqVarName(vlistID1, varID1, varname);

                  fprintf(stdout, "%6d ", indg);
                  fprintf(stdout, ":");

                  set_text_color(stdout, MAGENTA);
                  fprintf(stdout, "%s %s ", vdateString.c_str(), vtimeString.c_str());
                  reset_text_color(stdout);
                  set_text_color(stdout, GREEN);
                  fprintf(stdout, "%7g ", cdoZaxisInqLevel(zaxisID, levelID));
                  fprintf(stdout, "%8zu %7zu ", gridsize, cdo::max(nmiss1, nmiss2));
                  fprintf(stdout, "%7d ", ndiff);
                  reset_text_color(stdout);

                  fprintf(stdout, ":");
                  fprintf(stdout, " %c %c ", dsgn ? 'T' : 'F', zero ? 'T' : 'F');
                  set_text_color(stdout, BLUE);
                  fprintf(stdout, "%#12.5g%#12.5g", absm, relm);
                  reset_text_color(stdout);
                  fprintf(stdout, " : ");

                  set_text_color(stdout, BRIGHT, GREEN);
                  if (operatorID == DIFFN)
                    fprintf(stdout, "%-11s", varname);
                  else if (operatorID == DIFF || operatorID == DIFFP)
                    fprintf(stdout, "%-11s", paramstr);
                  else if (operatorID == DIFFC)
                    fprintf(stdout, "%4d", code);
                  reset_text_color(stdout);

                  fprintf(stdout, "\n");
                }
            }

          ngrec++;
          if (absm > abslim || (checkrel && relm >= rellim)) ndrec++;
          if (absm > abslim2 || (checkrel && relm >= rellim)) nd2rec++;
        }
      tsID++;
    }

  if (ndrec > 0)
    {
      Options::cdoExitStatus = 1;

      set_text_color(stdout, BRIGHT, RED);
      fprintf(stdout, "  %d of %d records differ", ndrec, ngrec);
      reset_text_color(stdout);
      fprintf(stdout, "\n");

      if (ndrec != nd2rec && abslim < abslim2) fprintf(stdout, "  %d of %d records differ more than %g\n", nd2rec, ngrec, abslim2);
      //  fprintf(stdout, "  %d of %d records differ more then one thousandth\n", nprec, ngrec);
    }

  if (nrecs == 0 && nrecs2 > 0) cdoWarning("stream2 has more time steps than stream1!");
  if (nrecs > 0 && nrecs2 == 0) cdoWarning("stream1 has more time steps than stream2!");

  cdoStreamClose(streamID1);
  cdoStreamClose(streamID2);

  cdoFinish();

  return nullptr;
}
