//===-- ARMInstPrinter.cpp - Convert ARM MCInst to assembly syntax --------===//
//
//                     The LLVM Compiler Infrastructure
//
// This file is distributed under the University of Illinois Open Source
// License. See LICENSE.TXT for details.
//
//===----------------------------------------------------------------------===//
//
// This class prints an ARM MCInst to a .s file.
//
//===----------------------------------------------------------------------===//

/* Capstone Disassembler Engine */
/* By Nguyen Anh Quynh <aquynh@gmail.com>, 2013> */

#include <stdio.h>	// DEBUG
#include <stdbool.h>
#include <stdlib.h>
#include <string.h>
#include <inttypes.h> 

#include "ARMInstPrinter.h"
#include "ARMAddressingModes.h"
#include "ARMBaseInfo.h"
#include "ARMDisassembler.h"
#include "../../MCInst.h"
#include "../../SStream.h"
#include "../../MCRegisterInfo.h"
#include "../../utils.h"
#include "ARMMapping.h"

#define GET_SUBTARGETINFO_ENUM
#include "ARMGenSubtargetInfo.inc"

static void printRegName(SStream *OS, unsigned RegNo);

// Autogenerated by tblgen.
static void printInstruction(MCInst *MI, SStream *O, MCRegisterInfo *MRI);
static void printOperand(MCInst *MI, unsigned OpNo, SStream *O);
static void printSORegRegOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printSORegImmOperand(MCInst *MI, unsigned OpNum, SStream *O);

static void printAddrModeTBB(MCInst *MI, unsigned OpNum, SStream *O);
static void printAddrModeTBH(MCInst *MI, unsigned OpNum, SStream *O);
static void printAddrMode2Operand(MCInst *MI, unsigned OpNum, SStream *O);
static void printAM2PreOrOffsetIndexOp(MCInst *MI, unsigned OpNum, SStream *O);
static void printAddrMode2OffsetOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printAddrMode3Operand(MCInst *MI, unsigned OpNum, SStream *O, bool AlwaysPrintImm0);
static void printAddrMode3OffsetOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printAM3PostIndexOp(MCInst *MI, unsigned Op, SStream *O);
static void printAM3PreOrOffsetIndexOp(MCInst *MI, unsigned Op, SStream *O, bool AlwaysPrintImm0);
static void printPostIdxImm8Operand(MCInst *MI, unsigned OpNum, SStream *O);
static void printPostIdxRegOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printPostIdxImm8s4Operand(MCInst *MI, unsigned OpNum, SStream *O);
static void printAddrMode5Operand(MCInst *MI, unsigned OpNum, SStream *O, bool AlwaysPrintImm0);
static void printAddrMode6Operand(MCInst *MI, unsigned OpNum, SStream *O);
static void printAddrMode7Operand(MCInst *MI, unsigned OpNum, SStream *O);
static void printAddrMode6OffsetOperand(MCInst *MI, unsigned OpNum, SStream *O);

static void printBitfieldInvMaskImmOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printMemBOption(MCInst *MI, unsigned OpNum, SStream *O);
static void printShiftImmOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printPKHLSLShiftImm(MCInst *MI, unsigned OpNum, SStream *O);
static void printPKHASRShiftImm(MCInst *MI, unsigned OpNum, SStream *O);
static void printAdrLabelOperand(MCInst *MI, unsigned OpNum, SStream *O, unsigned);
static void printThumbS4ImmOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printThumbSRImm(MCInst *MI, unsigned OpNum, SStream *O);
static void printThumbITMask(MCInst *MI, unsigned OpNum, SStream *O);
static void printThumbAddrModeRROperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printThumbAddrModeImm5SOperand(MCInst *MI, unsigned OpNum, SStream *O, unsigned Scale);
static void printThumbAddrModeImm5S1Operand(MCInst *MI, unsigned OpNum, SStream *O);
static void printThumbAddrModeImm5S2Operand(MCInst *MI, unsigned OpNum, SStream *O);
static void printThumbAddrModeImm5S4Operand(MCInst *MI, unsigned OpNum, SStream *O);
static void printThumbAddrModeSPOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printT2SOOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printAddrModeImm12Operand(MCInst *MI, unsigned OpNum, SStream *O, bool AlwaysPrintImm0);
static void printT2AddrModeImm8Operand(MCInst *MI, unsigned OpNum, SStream *O, bool);
static void printT2AddrModeImm8s4Operand(MCInst *MI, unsigned OpNum, SStream *O, bool);
static void printT2AddrModeImm0_1020s4Operand(MCInst *MI, unsigned OpNum, SStream *O);
static void printT2AddrModeImm8OffsetOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printT2AddrModeImm8s4OffsetOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printT2AddrModeSoRegOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printSetendOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printCPSIMod(MCInst *MI, unsigned OpNum, SStream *O);
static void printCPSIFlag(MCInst *MI, unsigned OpNum, SStream *O);
static void printMSRMaskOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printPredicateOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printMandatoryPredicateOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printSBitModifierOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printRegisterList(MCInst *MI, unsigned OpNum, SStream *O);
static void printNoHashImmediate(MCInst *MI, unsigned OpNum, SStream *O);
static void printPImmediate(MCInst *MI, unsigned OpNum, SStream *O);
static void printCImmediate(MCInst *MI, unsigned OpNum, SStream *O);
static void printCoprocOptionImm(MCInst *MI, unsigned OpNum, SStream *O);
static void printFPImmOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printNEONModImmOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printImmPlusOneOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printRotImmOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printGPRPairOperand(MCInst *MI, unsigned OpNum, SStream *O, MCRegisterInfo *MRI);
static void printThumbLdrLabelOperand(MCInst *MI, unsigned OpNum, SStream *O);
static void printFBits16(MCInst *MI, unsigned OpNum, SStream *O);
static void printFBits32(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorIndex(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListOne(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListTwo(MCInst *MI, unsigned OpNum, SStream *O, MCRegisterInfo *MRI);
static void printVectorListTwoSpaced(MCInst *MI, unsigned OpNum, SStream *O, MCRegisterInfo *RI);
static void printVectorListThree(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListFour(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListOneAllLanes(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListTwoAllLanes(MCInst *MI, unsigned OpNum, SStream *O, MCRegisterInfo *RI);
static void printVectorListThreeAllLanes(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListFourAllLanes(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListTwoSpacedAllLanes(MCInst *MI, unsigned OpNum, SStream *O, MCRegisterInfo *MRI);
static void printVectorListThreeSpacedAllLanes(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListFourSpacedAllLanes(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListThreeSpaced(MCInst *MI, unsigned OpNum, SStream *O);
static void printVectorListFourSpaced(MCInst *MI, unsigned OpNum, SStream *O);

static void printInstSyncBOption(MCInst *MI, unsigned OpNum, SStream *O);

static void set_mem_access(MCInst *MI, bool status)
{
	if (MI->csh->detail != CS_OPT_ON)
		return;

	MI->csh->doing_mem = status;
	if (status) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_MEM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = ARM_REG_INVALID;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.index = ARM_REG_INVALID;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.scale = 1;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = 0;
	} else {
		// done, create the next operand slot
		MI->flat_insn.arm.op_count++;
	}
}

#define GET_INSTRINFO_ENUM
#include "ARMGenInstrInfo.inc"

//#define PRINT_ALIAS_INSTR
#include "ARMGenAsmWriter.inc"

#define UseMarkup false
#define markup(x) (UseMarkup? (x) : "")

/// translateShiftImm - Convert shift immediate from 0-31 to 1-32 for printing.
///
/// getSORegOffset returns an integer from 0-31, representing '32' as 0.
static unsigned translateShiftImm(unsigned imm)
{
	// lsr #32 and asr #32 exist, but should be encoded as a 0.
	//assert((imm & ~0x1f) == 0 && "Invalid shift encoding");
	if (imm == 0)
		return 32;
	return imm;
}

/// Prints the shift value with an immediate value.
static void printRegImmShift(MCInst *MI, SStream *O, ARM_AM_ShiftOpc ShOpc,
		unsigned ShImm, bool _UseMarkup)
{
	if (ShOpc == ARM_AM_no_shift || (ShOpc == ARM_AM_lsl && !ShImm))
		return;
	SStream_concat(O, ", ");

	//assert (!(ShOpc == ARM_AM_ror && !ShImm) && "Cannot have ror #0");
	SStream_concat(O, ARM_AM_getShiftOpcStr(ShOpc));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.type = (arm_shifter)ShOpc;

	if (ShOpc != ARM_AM_rrx) {
		SStream_concat(O, " ");
		if (_UseMarkup)
			SStream_concat(O, "<imm:");
		SStream_concat(O, "#%u", translateShiftImm(ShImm));
		if (_UseMarkup)
			SStream_concat(O, ">");
		if (MI->csh->detail)
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.value = translateShiftImm(ShImm);
	}
}

static void printRegName(SStream *OS, unsigned RegNo)
{
	SStream_concat(OS, markup("<reg:"));
	SStream_concat(OS, getRegisterName(RegNo));
	SStream_concat(OS, markup(">"));
}

static name_map insn_update_flgs[] = {
	{ ARM_INS_CMN, "cmn" },
	{ ARM_INS_CMP, "cmp" },
	{ ARM_INS_TEQ, "teq" },
	{ ARM_INS_TST, "tst" },

	{ ARM_INS_ADC, "adcs" },
	{ ARM_INS_ADD, "adds" },
	{ ARM_INS_AND, "ands" },
	{ ARM_INS_ASR, "asrs" },
	{ ARM_INS_BIC, "bics" },
	{ ARM_INS_EOR, "eors" },
	{ ARM_INS_LSL, "lsls" },
	{ ARM_INS_LSR, "lsrs" },
	{ ARM_INS_MLA, "mlas" },
	{ ARM_INS_MOV, "movs" },
	{ ARM_INS_MUL, "muls" },
	{ ARM_INS_MVN, "mvns" },
	{ ARM_INS_ORN, "orns" },
	{ ARM_INS_ORR, "orrs" },
	{ ARM_INS_ROR, "rors" },
	{ ARM_INS_RRX, "rrxs" },
	{ ARM_INS_RSB, "rsbs" },
	{ ARM_INS_RSC, "rscs" },
	{ ARM_INS_SBC, "sbcs" },
	{ ARM_INS_SMLAL, "smlals" },
	{ ARM_INS_SMULL, "smulls" },
	{ ARM_INS_SUB, "subs" },
	{ ARM_INS_UMLAL, "umlals" },
	{ ARM_INS_UMULL, "umulls" },
};

void ARM_post_printer(csh ud, cs_insn *insn, char *insn_asm)
{
	if (((cs_struct *)ud)->detail != CS_OPT_ON)
		return;

	// check if this insn requests write-back
	if (strrchr(insn_asm, '!') != NULL) {
		insn->detail->arm.writeback = true;
	}

	// check if this insn requests update flags
	if (insn->detail->arm.update_flags == false) {
		// some insn still update flags, regardless of tabgen info
		int i;

		for (i = 0; i < ARR_SIZE(insn_update_flgs); i++) {
			if (insn->id == insn_update_flgs[i].id &&
					!strncmp(insn_asm, insn_update_flgs[i].name,
						strlen(insn_update_flgs[i].name))) {
				insn->detail->arm.update_flags = true;
				// we have to update regs_write array as well
				int j;
				for (j = 0; j < ARR_SIZE(insn->detail->regs_write); j++) {
					if (insn->detail->regs_write[j] == 0) {
						insn->detail->regs_write[j] = ARM_REG_CPSR;
						break;
					}
				}
				break;
			}
		}
	}
}

void ARM_printInst(MCInst *MI, SStream *O, void *Info)
{
	MCRegisterInfo *MRI = (MCRegisterInfo *)Info;

	unsigned Opcode = MCInst_getOpcode(MI);

	switch(Opcode) {
		// Check for HINT instructions w/ canonical names.
		case ARM_HINT:
		case ARM_tHINT:
		case ARM_t2HINT:
			switch (MCOperand_getImm(MCInst_getOperand(MI, 0))) {
				case 0: SStream_concat(O, "nop"); break;
				case 1: SStream_concat(O, "yield"); break;
				case 2: SStream_concat(O, "wfe"); break;
				case 3: SStream_concat(O, "wfi"); break;
				case 4: SStream_concat(O, "sev"); break;
				case 5:
						// FIXME: HasV80Ops becomes a mode
						//if ((ARM_getFeatureBits(MI->csh->mode) & ARM_HasV8Ops)) {
						//	SStream_concat(O, "sevl");
						//	break;
						//}
						// Fallthrough for non-v8

						SStream_concat(O, "sevl");
						break;
				default:
						// Anything else should just print normally.
						printInstruction(MI, O, MRI);
						return;
			}
			printPredicateOperand(MI, 1, O);
			if (Opcode == ARM_t2HINT)
				SStream_concat(O, ".w");	// FIXME: expose this in register-size of insn?
			return;

			// Check for MOVs and print canonical forms, instead.
		case ARM_MOVsr: {
							// FIXME: Thumb variants?
							MCOperand *Dst = MCInst_getOperand(MI, 0);
							MCOperand *MO1 = MCInst_getOperand(MI, 1);
							MCOperand *MO2 = MCInst_getOperand(MI, 2);
							MCOperand *MO3 = MCInst_getOperand(MI, 3);

							SStream_concat(O, ARM_AM_getShiftOpcStr(ARM_AM_getSORegShOp((unsigned int)MCOperand_getImm(MO3))));
							printSBitModifierOperand(MI, 6, O);
							printPredicateOperand(MI, 4, O);

							SStream_concat(O, "\t");
							printRegName(O, MCOperand_getReg(Dst));
							if (MI->csh->detail) {
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(Dst);
								MI->flat_insn.arm.op_count++;
							}

							SStream_concat(O, ", ");
							printRegName(O, MCOperand_getReg(MO1));

							if (MI->csh->detail) {
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MO1);
								MI->flat_insn.arm.op_count++;
							}

							SStream_concat(O, ", ");
							printRegName(O, MCOperand_getReg(MO2));
							if (MI->csh->detail) {
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MO2);
								MI->flat_insn.arm.op_count++;
							}
							//assert(ARM_AM_getSORegOffset(MO3.getImm()) == 0);
							return;
						}

		case ARM_MOVsi: {
							// FIXME: Thumb variants?
							MCOperand *Dst = MCInst_getOperand(MI, 0);
							MCOperand *MO1 = MCInst_getOperand(MI, 1);
							MCOperand *MO2 = MCInst_getOperand(MI, 2);

							SStream_concat(O, ARM_AM_getShiftOpcStr(ARM_AM_getSORegShOp((unsigned int)MCOperand_getImm(MO2))));
							printSBitModifierOperand(MI, 5, O);
							printPredicateOperand(MI, 3, O);

							SStream_concat(O, "\t");
							printRegName(O, MCOperand_getReg(Dst));
							if (MI->csh->detail) {
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(Dst);
								MI->flat_insn.arm.op_count++;
							}

							SStream_concat(O, ", ");
							printRegName(O, MCOperand_getReg(MO1));
							if (MI->csh->detail) {
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MO1);
								MI->flat_insn.arm.op_count++;
							}

							if (ARM_AM_getSORegShOp((unsigned int)MCOperand_getImm(MO2)) == ARM_AM_rrx) {
								//printAnnotation(O, Annot);
								return;
							}

							SStream_concat(O, ", %s", markup("<imm:"));
							unsigned tmp = translateShiftImm(getSORegOffset((unsigned int)MCOperand_getImm(MO2)));
							if (tmp > HEX_THRESHOLD)
								SStream_concat(O, "#0x%x", tmp);
							else
								SStream_concat(O, "#%u", tmp);
							SStream_concat(O, markup(">"));
							if (MI->csh->detail) {
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.type =
									(arm_shifter)ARM_AM_getSORegShOp((unsigned int)MCOperand_getImm(MO2));
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.value = tmp;
							}
							return;
						}

						// A8.6.123 PUSH
		case ARM_STMDB_UPD:
		case ARM_t2STMDB_UPD:
						if (MCOperand_getReg(MCInst_getOperand(MI, 0)) == ARM_SP &&
								MCInst_getNumOperands(MI) > 5) {
							// Should only print PUSH if there are at least two registers in the list.
							SStream_concat(O, "push");
							printPredicateOperand(MI, 2, O);
							if (Opcode == ARM_t2STMDB_UPD)
								SStream_concat(O, ".w");
							SStream_concat(O, "\t");
							printRegisterList(MI, 4, O);
							return;
						}
						break;

		case ARM_STR_PRE_IMM:
						if (MCOperand_getReg(MCInst_getOperand(MI, 2)) == ARM_SP &&
								MCOperand_getImm(MCInst_getOperand(MI, 3)) == -4) {
							SStream_concat(O, "push");
							printPredicateOperand(MI, 4, O);
							SStream_concat(O, "\t{");
							printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, 1)));
							if (MI->csh->detail) {
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, 1));
								MI->flat_insn.arm.op_count++;
							}
							SStream_concat(O, "}");
							return;
						}
						break;

						// A8.6.122 POP
		case ARM_LDMIA_UPD:
		case ARM_t2LDMIA_UPD:
						if (MCOperand_getReg(MCInst_getOperand(MI, 0)) == ARM_SP &&
								MCInst_getNumOperands(MI) > 5) {
							// Should only print POP if there are at least two registers in the list.
							SStream_concat(O, "pop");
							printPredicateOperand(MI, 2, O);
							if (Opcode == ARM_t2LDMIA_UPD)
								SStream_concat(O, ".w");
							SStream_concat(O, "\t");
							printRegisterList(MI, 4, O);
							return;
						}
						break;

		case ARM_LDR_POST_IMM:
						if (MCOperand_getReg(MCInst_getOperand(MI, 2)) == ARM_SP &&
								MCOperand_getImm(MCInst_getOperand(MI, 4)) == 4) {
							SStream_concat(O, "pop");
							printPredicateOperand(MI, 5, O);
							SStream_concat(O, "\t{");
							printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, 0)));
							if (MI->csh->detail) {
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
								MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, 0));
								MI->flat_insn.arm.op_count++;
							}
							SStream_concat(O, "}");
							return;
						}
						break;

						// A8.6.355 VPUSH
		case ARM_VSTMSDB_UPD:
		case ARM_VSTMDDB_UPD:
						if (MCOperand_getReg(MCInst_getOperand(MI, 0)) == ARM_SP) {
							SStream_concat(O, "vpush");
							printPredicateOperand(MI, 2, O);
							SStream_concat(O, "\t");
							printRegisterList(MI, 4, O);
							return;
						}
						break;

						// A8.6.354 VPOP
		case ARM_VLDMSIA_UPD:
		case ARM_VLDMDIA_UPD:
						if (MCOperand_getReg(MCInst_getOperand(MI, 0)) == ARM_SP) {
							SStream_concat(O, "vpop");
							printPredicateOperand(MI, 2, O);
							SStream_concat(O, "\t");
							printRegisterList(MI, 4, O);
							return;
						}
						break;

		case ARM_tLDMIA: {
							 bool Writeback = true;	// FIXME: expose this
							 unsigned BaseReg = MCOperand_getReg(MCInst_getOperand(MI, 0));
							 unsigned i;
							 for (i = 3; i < MCInst_getNumOperands(MI); ++i) {
								 if (MCOperand_getReg(MCInst_getOperand(MI, i)) == BaseReg)
									 Writeback = false;
							 }

							 SStream_concat(O, "ldm");

							 printPredicateOperand(MI, 1, O);
							 SStream_concat(O, "\t");
							 printRegName(O, BaseReg);
							 if (MI->csh->detail) {
								 MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
								 MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = BaseReg;
								 MI->flat_insn.arm.op_count++;
							 }
							 if (Writeback)
								 SStream_concat(O, "!");
							 SStream_concat(O, ", ");
							 printRegisterList(MI, 3, O);
							 return;
						 }

						 // Combine 2 GPRs from disassember into a GPRPair to match with instr def.
						 // ldrexd/strexd require even/odd GPR pair. To enforce this constraint,
						 // a single GPRPair reg operand is used in the .td file to replace the two
						 // GPRs. However, when decoding them, the two GRPs cannot be automatically
						 // expressed as a GPRPair, so we have to manually merge them.
						 // FIXME: We would really like to be able to tablegen'erate this.
		case ARM_LDREXD:
		case ARM_STREXD:
		case ARM_LDAEXD:
		case ARM_STLEXD: {
				    MCRegisterClass* MRC = MCRegisterInfo_getRegClass(MRI, ARM_GPRRegClassID);
				    bool isStore = Opcode == ARM_STREXD || Opcode == ARM_STLEXD;

				    unsigned Reg = MCOperand_getReg(MCInst_getOperand(MI, isStore ? 1 : 0));
					if (MCRegisterClass_contains(MRC, Reg)) {
						MCInst NewMI;
						MCOperand *NewReg;
						MCInst_setOpcode(&NewMI, Opcode);

						if (isStore)
							MCInst_addOperand2(&NewMI, MCInst_getOperand(MI, 0));
						NewReg = MCOperand_CreateReg(MCRegisterInfo_getMatchingSuperReg(MRI, Reg, ARM_gsub_0,
									MCRegisterInfo_getRegClass(MRI, ARM_GPRPairRegClassID)));
						MCInst_addOperand2(&NewMI, NewReg);
						cs_mem_free(NewReg);

						// Copy the rest operands into NewMI.
						unsigned i;
						for(i= isStore ? 3 : 2; i < MCInst_getNumOperands(MI); ++i)
							MCInst_addOperand2(&NewMI, MCInst_getOperand(MI, i));
						printInstruction(&NewMI, O, MRI);
						return;
					}
				}
	}

	//if (printAliasInstr(MI, O, MRI))
	//	printInstruction(MI, O, MRI);
	printInstruction(MI, O, MRI);
}

static void printOperand(MCInst *MI, unsigned OpNo, SStream *O)
{
	MCOperand *Op = MCInst_getOperand(MI, OpNo);
	if (MCOperand_isReg(Op)) {
		unsigned Reg = MCOperand_getReg(Op);
		printRegName(O, Reg);
		if (MI->csh->detail) {
			if (MI->csh->doing_mem) {
				if (MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base == ARM_REG_INVALID)
					MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = Reg;
				else
					MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.index = Reg;
			} else {
				MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
				MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg;
				MI->flat_insn.arm.op_count++;
			}
		}
	} else if (MCOperand_isImm(Op)) {
		SStream_concat(O, markup("<imm:"));
		int32_t imm = (int32_t)MCOperand_getImm(Op);

		// relative branch only has relative offset, so we have to update it
		// to reflect absolute address. 
		// Note: in ARM, PC is always 2 instructions ahead, so we have to
		// add 8 in ARM mode, or 4 in Thumb mode
		if (ARM_rel_branch(MI->csh, MCInst_getOpcode(MI))) {
			// only do this for relative branch
			if (MI->csh->mode & CS_MODE_THUMB)
				imm += (int32_t)MI->address + 4;
			else
				imm += (int32_t)MI->address + 8;

			if (imm > HEX_THRESHOLD)
				SStream_concat(O, "#0x%x", imm);
			else
				SStream_concat(O, "#%u", imm);
		} else if (imm >= 0) {
			if (imm > HEX_THRESHOLD)
				SStream_concat(O, "#0x%x", imm);
			else
				SStream_concat(O, "#%u", imm);
		} else
			SStream_concat(O, "#-0x%x", -imm);

		SStream_concat(O, markup(">"));
		if (MI->csh->detail) {
			if (MI->csh->doing_mem)
				MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = imm;
			else {
				MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
				MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = imm;
				MI->flat_insn.arm.op_count++;
			}
		}
	}
}

static void printThumbLdrLabelOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);

	SStream_concat(O, "%s[pc, ", markup("<mem:"));

	int32_t OffImm = (int32_t)MCOperand_getImm(MO1);
	bool isSub = OffImm < 0;

	// Special value for #-0. All others are normal.
	if (OffImm == INT32_MIN)
		OffImm = 0;
	if (isSub) {
		SStream_concat(O, "%s#-0x%x%s", markup("<imm:"), -OffImm, markup(">"));
	} else {
		if (OffImm > HEX_THRESHOLD)
			SStream_concat(O, "%s#0x%x%s", markup("<imm:"), OffImm, markup(">"));
		else
			SStream_concat(O, "%s#%u%s", markup("<imm:"), OffImm, markup(">"));
	}

	SStream_concat(O, "]%s", markup(">"));
}

// so_reg is a 4-operand unit corresponding to register forms of the A5.1
// "Addressing Mode 1 - Data-processing operands" forms.  This includes:
//    REG 0   0           - e.g. R5
//    REG REG 0,SH_OPC    - e.g. R5, ROR R3
//    REG 0   IMM,SH_OPC  - e.g. R5, LSL #3
static void printSORegRegOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);
	MCOperand *MO3 = MCInst_getOperand(MI, OpNum+2);

	printRegName(O, MCOperand_getReg(MO1));

	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MO1);

		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].shift.type = (MCOperand_getImm(MO3) & 7) + ARM_SFT_ASR_REG - 1;
		MI->flat_insn.arm.op_count++;
	}

	// Print the shift opc.
	ARM_AM_ShiftOpc ShOpc = ARM_AM_getSORegShOp((unsigned int)MCOperand_getImm(MO3));
	SStream_concat(O, ", ");
	SStream_concat(O, ARM_AM_getShiftOpcStr(ShOpc));
	if (ShOpc == ARM_AM_rrx)
		return;

	SStream_concat(O, " ");
	printRegName(O, MCOperand_getReg(MO2));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.value = MCOperand_getReg(MO2);
	//assert(ARM_AM_getSORegOffset(MO3.getImm()) == 0);
}

static void printSORegImmOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);

	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MO1);
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].shift.type = MCOperand_getImm(MO2) & 7;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].shift.value = (unsigned int)MCOperand_getImm(MO2) >> 3;
		MI->flat_insn.arm.op_count++;
	}

	// Print the shift opc.
	printRegImmShift(MI, O, ARM_AM_getSORegShOp((unsigned int)MCOperand_getImm(MO2)),
			getSORegOffset((unsigned int)MCOperand_getImm(MO2)), UseMarkup);
}

//===--------------------------------------------------------------------===//
// Addressing Mode #2
//===--------------------------------------------------------------------===//

static void printAM2PreOrOffsetIndexOp(MCInst *MI, unsigned Op, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, Op);
	MCOperand *MO2 = MCInst_getOperand(MI, Op + 1);
	MCOperand *MO3 = MCInst_getOperand(MI, Op + 2);

	SStream_concat(O, "%s[", markup("<mem:"));
	set_mem_access(MI, true);

	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);
	}

	if (!MCOperand_getReg(MO2)) {
		unsigned tmp = getAM2Offset((unsigned int)MCOperand_getImm(MO3));
		if (tmp) { // Don't print +0.
			SStream_concat(O, ", %s", markup("<imm:"));
			if (tmp > HEX_THRESHOLD)
				SStream_concat(O, "#%s0x%x", ARM_AM_getAddrOpcStr(getAM2Op((unsigned int)MCOperand_getImm(MO3))), tmp);
			else
				SStream_concat(O, "#%s%u", ARM_AM_getAddrOpcStr(getAM2Op((unsigned int)MCOperand_getImm(MO3))), tmp);
			if (MI->csh->detail) {
				MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].shift.type = (arm_shifter)getAM2Op((unsigned int)MCOperand_getImm(MO3));
				MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].shift.value = tmp;
			}
			SStream_concat(O, markup(">"));
		}
		SStream_concat(O, "]%s", markup(">"));
		set_mem_access(MI, false);
		return;
	}

	SStream_concat(O, ", ");
	SStream_concat(O, ARM_AM_getAddrOpcStr(getAM2Op((unsigned int)MCOperand_getImm(MO3))));
	printRegName(O, MCOperand_getReg(MO2));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.index = MCOperand_getReg(MO2);
	}

	printRegImmShift(MI, O, getAM2ShiftOpc((unsigned int)MCOperand_getImm(MO3)),
			getAM2Offset((unsigned int)MCOperand_getImm(MO3)), UseMarkup);
	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printAddrModeTBB(MCInst *MI, unsigned Op, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, Op);
	MCOperand *MO2 = MCInst_getOperand(MI, Op+1);
	SStream_concat(O, "%s[", markup("<mem:"));
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MO2));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.index = MCOperand_getReg(MO2);
	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printAddrModeTBH(MCInst *MI, unsigned Op, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, Op);
	MCOperand *MO2 = MCInst_getOperand(MI, Op+1);
	SStream_concat(O, "%s[", markup("<mem:"));
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MO2));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.index = MCOperand_getReg(MO2);
	SStream_concat(O, ", lsl %s #1%s]%s", markup("<imm:"), markup(">"), markup(">"));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].shift.type = ARM_SFT_LSL;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].shift.value = 1;
	}
	set_mem_access(MI, false);
}

static void printAddrMode2Operand(MCInst *MI, unsigned Op, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, Op);

	if (!MCOperand_isReg(MO1)) {   // FIXME: This is for CP entries, but isn't right.
		printOperand(MI, Op, O);
		return;
	}

	printAM2PreOrOffsetIndexOp(MI, Op, O);
}

static void printAddrMode2OffsetOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);

	if (!MCOperand_getReg(MO1)) {
		unsigned ImmOffs = getAM2Offset((unsigned int)MCOperand_getImm(MO2));
		if (ImmOffs > HEX_THRESHOLD)
			SStream_concat(O, "%s#%s0x%x%s", markup("<imm:"),
					ARM_AM_getAddrOpcStr(getAM2Op((unsigned int)MCOperand_getImm(MO2))), ImmOffs,
					markup(">"));
		else
			SStream_concat(O, "%s#%s%u%s", markup("<imm:"),
					ARM_AM_getAddrOpcStr(getAM2Op((unsigned int)MCOperand_getImm(MO2))), ImmOffs,
					markup(">"));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = ImmOffs;
			MI->flat_insn.arm.op_count++;
		}
		return;
	}

	SStream_concat(O, ARM_AM_getAddrOpcStr(getAM2Op((unsigned int)MCOperand_getImm(MO2))));
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MO1);
		MI->flat_insn.arm.op_count++;
	}

	printRegImmShift(MI, O, getAM2ShiftOpc((unsigned int)MCOperand_getImm(MO2)),
			getAM2Offset((unsigned int)MCOperand_getImm(MO2)), UseMarkup);
}

//===--------------------------------------------------------------------===//
// Addressing Mode #3
//===--------------------------------------------------------------------===//

static void printAM3PostIndexOp(MCInst *MI, unsigned Op, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, Op);
	MCOperand *MO2 = MCInst_getOperand(MI, Op+1);
	MCOperand *MO3 = MCInst_getOperand(MI, Op+2);
	ARM_AM_AddrOpc op = getAM3Op((unsigned int)MCOperand_getImm(MO3));

	SStream_concat(O, "%s[", markup("<mem:"));
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);
	SStream_concat(O, "], %s", markup(">"));
	set_mem_access(MI, false);

	if (MCOperand_getReg(MO2)) {
		SStream_concat(O, "%c", (char)op);
		printRegName(O, MCOperand_getReg(MO2));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MO2);
			MI->flat_insn.arm.op_count++;
		}
		return;
	}

	unsigned ImmOffs = getAM3Offset((unsigned int)MCOperand_getImm(MO3));
	if (ImmOffs > HEX_THRESHOLD)
		SStream_concat(O, "%s#%s0x%x%s", markup("<imm:"),
				ARM_AM_getAddrOpcStr(op), ImmOffs,
				markup(">"));
	else
		SStream_concat(O, "%s#%s%u%s", markup("<imm:"),
				ARM_AM_getAddrOpcStr(op), ImmOffs,
				markup(">"));

	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;

		if (op)
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = ImmOffs;
		else
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = -(int)ImmOffs;

		MI->flat_insn.arm.op_count++;
	}
}

static void printAM3PreOrOffsetIndexOp(MCInst *MI, unsigned Op, SStream *O,
		bool AlwaysPrintImm0)
{
	MCOperand *MO1 = MCInst_getOperand(MI, Op);
	MCOperand *MO2 = MCInst_getOperand(MI, Op+1);
	MCOperand *MO3 = MCInst_getOperand(MI, Op+2);
	ARM_AM_AddrOpc op = getAM3Op((unsigned int)MCOperand_getImm(MO3));

	SStream_concat(O, "%s[", markup("<mem:"));
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);

	if (MCOperand_getReg(MO2)) {
		SStream_concat(O, ", %s", ARM_AM_getAddrOpcStr(op));
		printRegName(O, MCOperand_getReg(MO2));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.index = MCOperand_getReg(MO2);
			if (op)
				MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.scale = -1;
		}
		SStream_concat(O, "]%s", markup(">"));
		set_mem_access(MI, false);
		return;
	}

	//If the op is sub we have to print the immediate even if it is 0
	unsigned ImmOffs = getAM3Offset((unsigned int)MCOperand_getImm(MO3));

	if (AlwaysPrintImm0 || ImmOffs || (op == ARM_AM_sub)) {
		if (ImmOffs > HEX_THRESHOLD)
			SStream_concat(O, ", %s#%s0x%x%s", markup("<imm:"), ARM_AM_getAddrOpcStr(op),
					ImmOffs, markup(">"));
		else
			SStream_concat(O, ", %s#%s%u%s", markup("<imm:"), ARM_AM_getAddrOpcStr(op),
					ImmOffs, markup(">"));
	}

	if (MI->csh->detail) {
		if (op)
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = (int)MCOperand_getImm(MO3);
		else
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = (int)-MCOperand_getImm(MO3);
	}

	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printAddrMode3Operand(MCInst *MI, unsigned Op, SStream *O,
		bool AlwaysPrintImm0)
{
	MCOperand *MO1 = MCInst_getOperand(MI, Op);
	if (!MCOperand_isReg(MO1)) {   //  For label symbolic references.
		printOperand(MI, Op, O);
		return;
	}

	MCOperand *MO3 = MCInst_getOperand(MI, Op+2);
	unsigned IdxMode = getAM3IdxMode((unsigned int)MCOperand_getImm(MO3));

	if (IdxMode == ARMII_IndexModePost) {
		printAM3PostIndexOp(MI, Op, O);
		return;
	}

	printAM3PreOrOffsetIndexOp(MI, Op, O, AlwaysPrintImm0);
}

static void printAddrMode3OffsetOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);
	ARM_AM_AddrOpc op = getAM3Op((unsigned int)MCOperand_getImm(MO2));

	if (MCOperand_getReg(MO1)) {
		SStream_concat(O, ARM_AM_getAddrOpcStr(op));
		printRegName(O, MCOperand_getReg(MO1));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = MCOperand_getReg(MO1);
			MI->flat_insn.arm.op_count++;
		}
		return;
	}

	unsigned ImmOffs = getAM3Offset((unsigned int)MCOperand_getImm(MO2));
	if (ImmOffs > HEX_THRESHOLD)
		SStream_concat(O, "%s#%s0x%x%s", markup("<imm:"),
				ARM_AM_getAddrOpcStr(op), ImmOffs,
				markup(">"));
	else
		SStream_concat(O, "%s#%s%u%s", markup("<imm:"),
				ARM_AM_getAddrOpcStr(op), ImmOffs,
				markup(">"));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;

		if (op)
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = ImmOffs;
		else
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = -(int)ImmOffs;

		MI->flat_insn.arm.op_count++;
	}
}

static void printPostIdxImm8Operand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO = MCInst_getOperand(MI, OpNum);
	unsigned Imm = (unsigned int)MCOperand_getImm(MO);
	if ((Imm & 0xff) > HEX_THRESHOLD)
		SStream_concat(O, "%s#%s0x%x%s", markup("<imm:"), ((Imm & 256) ? "" : "-"),
				(Imm & 0xff), markup(">"));
	else
		SStream_concat(O, "%s#%s%u%s", markup("<imm:"), ((Imm & 256) ? "" : "-"),
				(Imm & 0xff), markup(">"));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = Imm & 0xff;
		MI->flat_insn.arm.op_count++;
	}
}

static void printPostIdxRegOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);

	SStream_concat(O, (MCOperand_getImm(MO2) ? "" : "-"));
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MO1);
		MI->flat_insn.arm.op_count++;
	}
}

static void printPostIdxImm8s4Operand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO = MCInst_getOperand(MI, OpNum);
	unsigned Imm = (unsigned int)MCOperand_getImm(MO);
	if (((Imm & 0xff) << 2) > HEX_THRESHOLD)
		SStream_concat(O, "%s#%s0x%x%s", markup("<imm:"), ((Imm & 256) ? "" : "-"),
				((Imm & 0xff) << 2), markup(">"));
	else
		SStream_concat(O, "%s#%s%u%s", markup("<imm:"), ((Imm & 256) ? "" : "-"),
				((Imm & 0xff) << 2), markup(">"));
}

static void printAddrMode5Operand(MCInst *MI, unsigned OpNum, SStream *O,
		bool AlwaysPrintImm0)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);

	if (!MCOperand_isReg(MO1)) {   // FIXME: This is for CP entries, but isn't right.
		printOperand(MI, OpNum, O);
		return;
	}

	SStream_concat(O, "%s[", markup("<mem:"));
	printRegName(O, MCOperand_getReg(MO1));

	unsigned ImmOffs = ARM_AM_getAM5Offset((unsigned int)MCOperand_getImm(MO2));
	unsigned Op = ARM_AM_getAM5Op((unsigned int)MCOperand_getImm(MO2));
	if (AlwaysPrintImm0 || ImmOffs || Op == ARM_AM_sub) {
		if (ImmOffs * 4 > HEX_THRESHOLD)
			SStream_concat(O, ", %s#%s0x%x%s", markup("<imm:"),
					ARM_AM_getAddrOpcStr(ARM_AM_getAM5Op((unsigned int)MCOperand_getImm(MO2))),
					ImmOffs * 4, markup(">"));
		else
			SStream_concat(O, ", %s#%s%u%s", markup("<imm:"),
					ARM_AM_getAddrOpcStr(ARM_AM_getAM5Op((unsigned int)MCOperand_getImm(MO2))),
					ImmOffs * 4, markup(">"));
	}
	SStream_concat(O, "]%s", markup(">"));
}

static void printAddrMode6Operand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);

	SStream_concat(O, "%s[", markup("<mem:"));
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);
	unsigned tmp = (unsigned int)MCOperand_getImm(MO2);
	if (tmp) {
		if (tmp << 3 > HEX_THRESHOLD)
			SStream_concat(O, ":0x%x", (tmp << 3));
		else
			SStream_concat(O, ":%u", (tmp << 3));
		if (MI->csh->detail)
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = tmp << 3;
	}
	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printAddrMode7Operand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	SStream_concat(O, "%s[", markup("<mem:"));
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);
	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printAddrMode6OffsetOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO = MCInst_getOperand(MI, OpNum);
	if (MCOperand_getReg(MO) == 0)
		SStream_concat(O, "!");
	else {
		SStream_concat(O, ", ");
		printRegName(O, MCOperand_getReg(MO));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MO);
			MI->flat_insn.arm.op_count++;
		}
	}
}

static void printBitfieldInvMaskImmOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO = MCInst_getOperand(MI, OpNum);
	uint32_t v = ~(uint32_t)MCOperand_getImm(MO);
	int32_t lsb = CountTrailingZeros_32(v);
	int32_t width = (32 - CountLeadingZeros_32 (v)) - lsb;

	//assert(MO.isImm() && "Not a valid bf_inv_mask_imm value!");
	if (lsb > HEX_THRESHOLD)
		SStream_concat(O, "%s#0x%x%s", markup("<imm:"), lsb, markup(">"));
	else
		SStream_concat(O, "%s#%u%s", markup("<imm:"), lsb, markup(">"));

	if (width > HEX_THRESHOLD)
		SStream_concat(O, ", %s#0x%x%s", markup("<imm:"), width, markup(">"));
	else
		SStream_concat(O, ", %s#%u%s", markup("<imm:"), width, markup(">"));

	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = lsb;
		MI->flat_insn.arm.op_count++;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = width;
		MI->flat_insn.arm.op_count++;
	}
}

static void printMemBOption(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned val = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	// FIXME: HasV80Ops becomes a mode
	// SStream_concat(O, ARM_MB_MemBOptToString(val,
	//			ARM_getFeatureBits(MI->csh->mode) & ARM_HasV8Ops));
	SStream_concat(O, ARM_MB_MemBOptToString(val, ARM_HasV8Ops));
}

void printInstSyncBOption(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned val = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	SStream_concat(O, "%s", ARM_ISB_InstSyncBOptToString(val));
}

static void printShiftImmOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned ShiftOp = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	bool isASR = (ShiftOp & (1 << 5)) != 0;
	unsigned Amt = ShiftOp & 0x1f;
	if (isASR) {
		unsigned tmp = Amt == 0 ? 32 : Amt;
		if (tmp > HEX_THRESHOLD)
			SStream_concat(O, ", asr %s#0x%x%s", markup("<imm:"), tmp, markup(">"));
		else
			SStream_concat(O, ", asr %s#%u%s", markup("<imm:"), tmp, markup(">"));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.type = ARM_SFT_ASR;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.value = tmp;
		}
	} else if (Amt) {
		if (Amt > HEX_THRESHOLD)
			SStream_concat(O, ", lsl %s#0x%x%s", markup("<imm:"), Amt, markup(">"));
		else
			SStream_concat(O, ", lsl %s#%u%s", markup("<imm:"), Amt, markup(">"));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.type = ARM_SFT_LSL;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.value = Amt;
		}
	}
}

static void printPKHLSLShiftImm(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned Imm = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	if (Imm == 0)
		return;
	//assert(Imm > 0 && Imm < 32 && "Invalid PKH shift immediate value!");
	if (Imm > HEX_THRESHOLD)
		SStream_concat(O, ", lsl %s#0x%x%s", markup("<imm:"), Imm, markup(">"));
	else
		SStream_concat(O, ", lsl %s#%u%s", markup("<imm:"), Imm, markup(">"));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.type = ARM_SFT_LSL;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.value = Imm;
	}
}

static void printPKHASRShiftImm(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned Imm = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	// A shift amount of 32 is encoded as 0.
	if (Imm == 0)
		Imm = 32;
	//assert(Imm > 0 && Imm <= 32 && "Invalid PKH shift immediate value!");
	if (Imm > HEX_THRESHOLD)
		SStream_concat(O, ", asr %s#0x%x%s", markup("<imm:"), Imm, markup(">"));
	else
		SStream_concat(O, ", asr %s#%u%s", markup("<imm:"), Imm, markup(">"));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.type = ARM_SFT_ASR;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.value = Imm;
	}
}

// FIXME: push {r1, r2, r3, ...} can exceed the number of operands in MCInst struct
static void printRegisterList(MCInst *MI, unsigned OpNum, SStream *O)
{
	SStream_concat(O, "{");
	unsigned i, e;
	for (i = OpNum, e = MCInst_getNumOperands(MI); i != e; ++i) {
		if (i != OpNum) SStream_concat(O, ", ");
		printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, i)));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, i));
			MI->flat_insn.arm.op_count++;
		}
	}
	SStream_concat(O, "}");
}

static void printGPRPairOperand(MCInst *MI, unsigned OpNum, SStream *O,
		MCRegisterInfo *MRI)
{
	unsigned Reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
	printRegName(O, MCRegisterInfo_getSubReg(MRI, Reg, ARM_gsub_0));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCRegisterInfo_getSubReg(MRI, Reg, ARM_gsub_0);
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCRegisterInfo_getSubReg(MRI, Reg, ARM_gsub_1));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCRegisterInfo_getSubReg(MRI, Reg, ARM_gsub_1);
		MI->flat_insn.arm.op_count++;
	}
}

// SETEND BE/LE
static void printSetendOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *Op = MCInst_getOperand(MI, OpNum);
	if (MCOperand_getImm(Op))
		SStream_concat(O, "be");
	else
		SStream_concat(O, "le");
}

static void printCPSIMod(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *Op = MCInst_getOperand(MI, OpNum);
	SStream_concat(O, "%s", ARM_PROC_IModToString((unsigned int)MCOperand_getImm(Op)));
}

static void printCPSIFlag(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *Op = MCInst_getOperand(MI, OpNum);
	unsigned IFlags = (unsigned int)MCOperand_getImm(Op);
	int i;
	for (i=2; i >= 0; --i)
		if (IFlags & (1 << i))
			SStream_concat(O, ARM_PROC_IFlagsToString(1 << i));

	if (IFlags == 0)
		SStream_concat(O, "none");

	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = IFlags;
		MI->flat_insn.arm.op_count++;
	}
}

// TODO
static void printMSRMaskOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *Op = MCInst_getOperand(MI, OpNum);
#if 0 // TODO once below is fixed
	unsigned SpecRegRBit = (unsigned int)MCOperand_getImm(Op) >> 4;
	unsigned Mask = (unsigned int)MCOperand_getImm(Op) & 0xf;
#endif

	// FIXME: FeatureMClass becomes mode??
	//if (ARM_getFeatureBits(MI->csh->mode) & ARM_FeatureMClass) {
	//if (true)
	{
		unsigned SYSm = (unsigned int)MCOperand_getImm(Op);
		unsigned Opcode = MCInst_getOpcode(MI);
		// For reads of the special registers ignore the "mask encoding" bits
		// which are only for writes.
		if (Opcode == ARM_t2MRS_M)
			SYSm &= 0xff;
		switch (SYSm) {
			default: return; //llvm_unreachable("Unexpected mask value!");
			case     0:
			case 0x800: SStream_concat(O, "apsr"); return; // with _nzcvq bits is an alias for aspr
			case 0x400: SStream_concat(O, "apsr_g"); return;
			case 0xc00: SStream_concat(O, "apsr_nzcvqg"); return;
			case     1:
			case 0x801: SStream_concat(O, "iapsr"); return; // with _nzcvq bits is an alias for iapsr
			case 0x401: SStream_concat(O, "iapsr_g"); return;
			case 0xc01: SStream_concat(O, "iapsr_nzcvqg"); return;
			case     2:
			case 0x802: SStream_concat(O, "eapsr"); return; // with _nzcvq bits is an alias for eapsr
			case 0x402: SStream_concat(O, "eapsr_g"); return;
			case 0xc02: SStream_concat(O, "eapsr_nzcvqg"); return;
			case     3:
			case 0x803: SStream_concat(O, "xpsr"); return; // with _nzcvq bits is an alias for xpsr
			case 0x403: SStream_concat(O, "xpsr_g"); return;
			case 0xc03: SStream_concat(O, "xpsr_nzcvqg"); return;
			case     5:
			case 0x805: SStream_concat(O, "ipsr"); return;
			case     6:
			case 0x806: SStream_concat(O, "epsr"); return;
			case     7:
			case 0x807: SStream_concat(O, "iepsr"); return;
			case     8:
			case 0x808: SStream_concat(O, "msp"); return;
			case     9:
			case 0x809: SStream_concat(O, "psp"); return;
			case  0x10:
			case 0x810: SStream_concat(O, "primask"); return;
			case  0x11:
			case 0x811: SStream_concat(O, "basepri"); return;
			case  0x12:
			case 0x812: SStream_concat(O, "basepri_max"); return;
			case  0x13:
			case 0x813: SStream_concat(O, "faultmask"); return;
			case  0x14:
			case 0x814: SStream_concat(O, "control"); return;
		}
	}
#if 0 // TODO once above is fixed
	// As special cases, CPSR_f, CPSR_s and CPSR_fs prefer printing as
	// APSR_nzcvq, APSR_g and APSRnzcvqg, respectively.
	if (!SpecRegRBit && (Mask == 8 || Mask == 4 || Mask == 12)) {
		SStream_concat(O, "apsr_");
		switch (Mask) {
			default: return; //llvm_unreachable("Unexpected mask value!");
			case 4:  SStream_concat(O, "g"); return;
			case 8:  SStream_concat(O, "nzcvq"); return;
			case 12: SStream_concat(O, "nzcvqg"); return;
		}
	}

	if (SpecRegRBit)
		SStream_concat(O, "spsr");
	else
		SStream_concat(O, "cpsr");

	if (Mask) {
		SStream_concat(O, "_");
		if (Mask & 8) SStream_concat(O, "f");
		if (Mask & 4) SStream_concat(O, "s");
		if (Mask & 2) SStream_concat(O, "x");
		if (Mask & 1) SStream_concat(O, "c");
	}
#endif
}

static void printPredicateOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	ARMCC_CondCodes CC = (ARMCC_CondCodes)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	// Handle the undefined 15 CC value here for printing so we don't abort().
	if ((unsigned)CC == 15) {
		SStream_concat(O, "<und>");
		if (MI->csh->detail)
			MI->flat_insn.arm.cc = ARM_CC_INVALID;
	} else {
		if (CC != ARMCC_AL) {
			SStream_concat(O, ARMCC_ARMCondCodeToString(CC));
		}
		if (MI->csh->detail)
			MI->flat_insn.arm.cc = CC + 1;
	}
}

// TODO: test this
static void printMandatoryPredicateOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	ARMCC_CondCodes CC = (ARMCC_CondCodes)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	SStream_concat(O, ARMCC_ARMCondCodeToString(CC));
	if (MI->csh->detail)
		MI->flat_insn.arm.cc = CC + 1;
}

static void printSBitModifierOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	if (MCOperand_getReg(MCInst_getOperand(MI, OpNum))) {
		//assert(MCOperand_getReg(MCInst_getOperand(MI, OpNum)) == ARM_CPSR &&
		//       "Expect ARM CPSR register!");
		if (MI->csh->detail)
			MI->flat_insn.arm.update_flags = true;
		SStream_concat(O, "s");
	}
}

static void printNoHashImmediate(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned tmp = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	if (tmp > HEX_THRESHOLD)
		SStream_concat(O, "0x%x", tmp);
	else
		SStream_concat(O, "%u", tmp);
	if (MI->csh->detail) {
		if (MI->csh->doing_mem) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = tmp;
		} else {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = tmp;
			MI->flat_insn.arm.op_count++;
		}
	}
}

static void printPImmediate(MCInst *MI, unsigned OpNum, SStream *O)
{
	SStream_concat(O, "p%u", MCOperand_getImm(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_PIMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
}

static void printCImmediate(MCInst *MI, unsigned OpNum, SStream *O)
{
	SStream_concat(O, "c%u", MCOperand_getImm(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_CIMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
}

static void printCoprocOptionImm(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned tmp = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	if (tmp > HEX_THRESHOLD)
		SStream_concat(O, "{0x%x}", tmp);
	else
		SStream_concat(O, "{%u}", tmp);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = tmp;
		MI->flat_insn.arm.op_count++;
	}
}

static void printAdrLabelOperand(MCInst *MI, unsigned OpNum, SStream *O, unsigned scale)
{
	MCOperand *MO = MCInst_getOperand(MI, OpNum);

	int32_t OffImm = (int32_t)MCOperand_getImm(MO) << scale;

	SStream_concat(O, markup("<imm:"));
	if (OffImm == INT32_MIN) {
		SStream_concat(O, "#-0");
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = 0;
			MI->flat_insn.arm.op_count++;
		}
	} else {
		if (OffImm < 0)
			SStream_concat(O, "#-0x%x", -OffImm);
		else {
			if (OffImm > HEX_THRESHOLD)
				SStream_concat(O, "#0x%x", OffImm);
			else
				SStream_concat(O, "#%u", OffImm);
		}
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = OffImm;
			MI->flat_insn.arm.op_count++;
		}
	}
	SStream_concat(O, markup(">"));
}

static void printThumbS4ImmOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned tmp = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum)) * 4;
	if (tmp > HEX_THRESHOLD)
		SStream_concat(O, "%s#0x%x", markup("<imm:"), tmp);
	else
		SStream_concat(O, "%s#%u", markup("<imm:"), tmp);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = tmp;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, markup(">"));
}

static void printThumbSRImm(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned Imm = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	unsigned tmp = Imm == 0 ? 32 : Imm;
	if (tmp > HEX_THRESHOLD)
		SStream_concat(O, "%s#0x%x", markup("<imm:"), tmp);
	else
		SStream_concat(O, "%s#%u", markup("<imm:"), tmp);

	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = tmp;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, markup(">"));
}

// TODO
static void printThumbITMask(MCInst *MI, unsigned OpNum, SStream *O)
{
	// (3 - the number of trailing zeros) is the number of then / else.
	unsigned Mask = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	unsigned Firstcond = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum-1));
	unsigned CondBit0 = Firstcond & 1;
	unsigned NumTZ = CountTrailingZeros_32(Mask);
	//assert(NumTZ <= 3 && "Invalid IT mask!");
	unsigned Pos, e;
	for (Pos = 3, e = NumTZ; Pos > e; --Pos) {
		bool T = ((Mask >> Pos) & 1) == CondBit0;
		if (T)
			SStream_concat(O, "t");
		else
			SStream_concat(O, "e");
	}
}

static void printThumbAddrModeRROperand(MCInst *MI, unsigned Op, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, Op);
	MCOperand *MO2 = MCInst_getOperand(MI, Op + 1);

	if (!MCOperand_isReg(MO1)) {   // FIXME: This is for CP entries, but isn't right.
		printOperand(MI, Op, O);
		return;
	}

	SStream_concat(O, markup("<mem:"));
	SStream_concat(O, "[");
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);
	unsigned RegNum = MCOperand_getReg(MO2);
	if (RegNum) {
		SStream_concat(O, ", ");
		printRegName(O, RegNum);
		if (MI->csh->detail)
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.index = RegNum;
	}
	SStream_concat(O, "]");
	set_mem_access(MI, false);
	SStream_concat(O, markup(">"));
}

static void printThumbAddrModeImm5SOperand(MCInst *MI, unsigned Op, SStream *O,
		unsigned Scale)
{
	MCOperand *MO1 = MCInst_getOperand(MI, Op);
	MCOperand *MO2 = MCInst_getOperand(MI, Op + 1);

	if (!MCOperand_isReg(MO1)) {   // FIXME: This is for CP entries, but isn't right.
		printOperand(MI, Op, O);
		return;
	}

	SStream_concat(O, markup("<mem:"));
	SStream_concat(O, "[");
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);
	unsigned ImmOffs = (unsigned int)MCOperand_getImm(MO2);
	if (ImmOffs) {
		unsigned tmp = ImmOffs * Scale;
		SStream_concat(O, ", %s", markup("<imm:"));
		if (tmp > HEX_THRESHOLD)
			SStream_concat(O, "#0x%x", tmp);
		else
			SStream_concat(O, "#%u", tmp);
		if (MI->csh->detail)
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = tmp;
		SStream_concat(O, markup(">"));
	}
	SStream_concat(O, "]");
	set_mem_access(MI, false);
	SStream_concat(O, markup(">"));
}

static void printThumbAddrModeImm5S1Operand(MCInst *MI, unsigned Op, SStream *O)
{
	printThumbAddrModeImm5SOperand(MI, Op, O, 1);
}

static void printThumbAddrModeImm5S2Operand(MCInst *MI, unsigned Op, SStream *O)
{
	printThumbAddrModeImm5SOperand(MI, Op, O, 2);
}

static void printThumbAddrModeImm5S4Operand(MCInst *MI, unsigned Op, SStream *O)
{
	printThumbAddrModeImm5SOperand(MI, Op, O, 4);
}

static void printThumbAddrModeSPOperand(MCInst *MI, unsigned Op, SStream *O)
{
	printThumbAddrModeImm5SOperand(MI, Op, O, 4);
}

// Constant shifts t2_so_reg is a 2-operand unit corresponding to the Thumb2
// register with shift forms.
// REG 0   0           - e.g. R5
// REG IMM, SH_OPC     - e.g. R5, LSL #3
static void printT2SOOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);

	unsigned Reg = MCOperand_getReg(MO1);
	printRegName(O, Reg);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg;
		MI->flat_insn.arm.op_count++;
	}

	// Print the shift opc.
	//assert(MO2.isImm() && "Not a valid t2_so_reg value!");
	printRegImmShift(MI, O, ARM_AM_getSORegShOp((unsigned int)MCOperand_getImm(MO2)),
			getSORegOffset((unsigned int)MCOperand_getImm(MO2)), UseMarkup);
}

static void printAddrModeImm12Operand(MCInst *MI, unsigned OpNum,
		SStream *O, bool AlwaysPrintImm0)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);
	int32_t OffImm;
	bool isSub;

	if (!MCOperand_isReg(MO1)) {   // FIXME: This is for CP entries, but isn't right.
		printOperand(MI, OpNum, O);
		return;
	}

	SStream_concat(O, markup("<mem:"));
	SStream_concat(O, "[");
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));

	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);

	OffImm = (int32_t)MCOperand_getImm(MO2);
	isSub = OffImm < 0;
	// Special value for #-0. All others are normal.
	if (OffImm == INT32_MIN)
		OffImm = 0;
	if (isSub) {
		SStream_concat(O, ", %s#-0x%x%s", markup("<imm:"), -OffImm, markup(">"));
	} else if (AlwaysPrintImm0 || OffImm > 0) {
		if (OffImm > HEX_THRESHOLD)
			SStream_concat(O, ", %s#0x%x%s", markup("<imm:"), OffImm, markup(">"));
		else
			SStream_concat(O, ", %s#%u%s", markup("<imm:"), OffImm, markup(">"));
	}
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = OffImm;
	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printT2AddrModeImm8Operand(MCInst *MI, unsigned OpNum, SStream *O,
		bool AlwaysPrintImm0)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);
	int32_t OffImm;
	bool isSub;

	SStream_concat(O, "%s[", markup("<mem:"));
	set_mem_access(MI, true);

	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);

	OffImm = (int32_t)MCOperand_getImm(MO2);
	isSub = OffImm < 0;
	// Don't print +0.
	if (OffImm == INT32_MIN)
		OffImm = 0;

	if (isSub)
		SStream_concat(O, ", %s#-0x%x%s", markup("<imm:"), -OffImm, markup(">"));
	else if (AlwaysPrintImm0 || OffImm > 0) {
		if (OffImm > HEX_THRESHOLD)
			SStream_concat(O, ", %s#0x%x%s", markup("<imm:"), OffImm, markup(">"));
		else
			SStream_concat(O, ", %s#%u%s", markup("<imm:"), OffImm, markup(">"));
	}

	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = OffImm;
	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printT2AddrModeImm8s4Operand(MCInst *MI,
		unsigned OpNum, SStream *O, bool AlwaysPrintImm0)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);
	int32_t OffImm;
	bool isSub;

	if (!MCOperand_isReg(MO1)) {   //  For label symbolic references.
		printOperand(MI, OpNum, O);
		return;
	}

	SStream_concat(O, markup("<mem:"));
	SStream_concat(O, "[");
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);

	OffImm = (int32_t)MCOperand_getImm(MO2);
	isSub = OffImm < 0;

	//assert(((OffImm & 0x3) == 0) && "Not a valid immediate!");

	// Don't print +0.
	if (OffImm == INT32_MIN)
		OffImm = 0;
	if (isSub) {
		SStream_concat(O, ", %s#-0x%x%s", markup("<imm:"), -OffImm, markup(">"));
	} else if (AlwaysPrintImm0 || OffImm > 0) {
		if (OffImm > HEX_THRESHOLD)
			SStream_concat(O, ", %s#0x%x%s", markup("<imm:"), OffImm, markup(">"));
		else
			SStream_concat(O, ", %s#%u%s", markup("<imm:"), OffImm, markup(">"));
	}
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = OffImm;

	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printT2AddrModeImm0_1020s4Operand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);

	SStream_concat(O, markup("<mem:"));
	SStream_concat(O, "[");
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);
	if (MCOperand_getImm(MO2)) {
		SStream_concat(O, ", ");
		SStream_concat(O, markup("<imm:"));
		unsigned tmp = (unsigned int)MCOperand_getImm(MO2) * 4;
		if (tmp > HEX_THRESHOLD)
			SStream_concat(O, "#0x%x", tmp);
		else
			SStream_concat(O, "#%u", tmp);
		SStream_concat(O, markup(">"));
		if (MI->csh->detail)
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.disp = tmp;
	}
	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printT2AddrModeImm8OffsetOperand(MCInst *MI,
		unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	int32_t OffImm = (int32_t)MCOperand_getImm(MO1);
	SStream_concat(O, ", %s", markup("<imm:"));
	if (OffImm == INT32_MIN) {
		SStream_concat(O, "#-0%s", markup(">"));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = 0;
			MI->flat_insn.arm.op_count++;
		}
	} else {
		if (OffImm < 0)
			SStream_concat(O, "#-0x%x%s", -OffImm, markup(">"));
		else {
			if (OffImm > HEX_THRESHOLD)
				SStream_concat(O, "#0x%x%s", OffImm, markup(">"));
			else
				SStream_concat(O, "#%u%s", OffImm, markup(">"));
		}
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = OffImm;
			MI->flat_insn.arm.op_count++;
		}
	}
}

static void printT2AddrModeImm8s4OffsetOperand(MCInst *MI,
		unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	int32_t OffImm = (int32_t)MCOperand_getImm(MO1);

	//assert(((OffImm & 0x3) == 0) && "Not a valid immediate!");

	SStream_concat(O, ", %s", markup("<imm:"));
	if (OffImm == INT32_MIN) {
		SStream_concat(O, "#-0%s", markup(">"));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = 0;
			MI->flat_insn.arm.op_count++;
		}
	} else {
		if (OffImm < 0)
			SStream_concat(O, "#-0x%x%s", -OffImm, markup(">"));
		else {
			if (OffImm > HEX_THRESHOLD)
				SStream_concat(O, "#0x%x%s", OffImm, markup(">"));
			else
				SStream_concat(O, "#%u%s", OffImm, markup(">"));
		}
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = OffImm;
			MI->flat_insn.arm.op_count++;
		}
	}
}

static void printT2AddrModeSoRegOperand(MCInst *MI,
		unsigned OpNum, SStream *O)
{
	MCOperand *MO1 = MCInst_getOperand(MI, OpNum);
	MCOperand *MO2 = MCInst_getOperand(MI, OpNum+1);
	MCOperand *MO3 = MCInst_getOperand(MI, OpNum+2);

	SStream_concat(O, "%s[", markup("<mem:"));
	set_mem_access(MI, true);
	printRegName(O, MCOperand_getReg(MO1));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.base = MCOperand_getReg(MO1);

	//assert(MCOperand_getReg(MO2.getReg() && "Invalid so_reg load / store address!");
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MO2));
	if (MI->csh->detail)
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].mem.index = MCOperand_getReg(MO2);

	unsigned ShAmt = (unsigned int)MCOperand_getImm(MO3);
	if (ShAmt) {
		//assert(ShAmt <= 3 && "Not a valid Thumb2 addressing mode!");
		SStream_concat(O, ", lsl ");
		SStream_concat(O, markup("<imm:"));
		SStream_concat(O, "#%d", ShAmt);
		SStream_concat(O, markup(">"));
		if (MI->csh->detail) {
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.type = ARM_SFT_LSL;
			MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.value = ShAmt;
		}
	}

	SStream_concat(O, "]%s", markup(">"));
	set_mem_access(MI, false);
}

static void printFPImmOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	MCOperand *MO = MCInst_getOperand(MI, OpNum);
	SStream_concat(O, "%s#%f%s", markup("<imm:"), getFPImmFloat((unsigned int)MCOperand_getImm(MO)), markup(">"));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_FP;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].fp = getFPImmFloat((unsigned int)MCOperand_getImm(MO));
		MI->flat_insn.arm.op_count++;
	}
}

static void printNEONModImmOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned EncodedImm = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	unsigned EltBits;
	uint64_t Val = ARM_AM_decodeNEONModImm(EncodedImm, &EltBits);
	if (Val > HEX_THRESHOLD)
		SStream_concat(O, "%s#0x%"PRIx64"%s", markup("<imm:"), Val, markup(">"));
	else
		SStream_concat(O, "%s#%"PRIu64"%s", markup("<imm:"), Val, markup(">"));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = (unsigned int)Val;
		MI->flat_insn.arm.op_count++;
	}
}

static void printImmPlusOneOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned Imm = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	if (Imm + 1 > HEX_THRESHOLD)
		SStream_concat(O, "%s#0x%x%s", markup("<imm:"), Imm + 1, markup(">"));
	else
		SStream_concat(O, "%s#%u%s", markup("<imm:"), Imm + 1, markup(">"));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = Imm + 1;
		MI->flat_insn.arm.op_count++;
	}
}

static void printRotImmOperand(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned Imm = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	if (Imm == 0)
		return;
	SStream_concat(O, ", ror %s#", markup("<imm:"));
	switch (Imm) {
		default: //assert (0 && "illegal ror immediate!");
		case 1: SStream_concat(O, "8"); break;
		case 2: SStream_concat(O, "16"); break;
		case 3: SStream_concat(O, "24"); break;
	}
	SStream_concat(O, markup(">"));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.type = ARM_SFT_ROR;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count - 1].shift.value = Imm * 8;
	}
}

static void printFBits16(MCInst *MI, unsigned OpNum, SStream *O)
{
	SStream_concat(O, markup("<imm:"));
	unsigned tmp = 16 - (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	if (tmp > HEX_THRESHOLD)
		SStream_concat(O, "#0x%x", tmp);
	else
		SStream_concat(O, "#%u", tmp);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = tmp;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, markup(">"));
}

static void printFBits32(MCInst *MI, unsigned OpNum, SStream *O)
{
	SStream_concat(O, markup("<imm:"));
	unsigned tmp = 32 - (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	if (tmp > HEX_THRESHOLD)
		SStream_concat(O, "#0x%x", tmp);
	else
		SStream_concat(O, "#%u", tmp);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = tmp;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, markup(">"));
}

static void printVectorIndex(MCInst *MI, unsigned OpNum, SStream *O)
{
	unsigned tmp = (unsigned int)MCOperand_getImm(MCInst_getOperand(MI, OpNum));
	if (tmp > HEX_THRESHOLD)
		SStream_concat(O, "[0x%x]",tmp);
	else
		SStream_concat(O, "[%u]",tmp);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_IMM;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].imm = tmp;
		MI->flat_insn.arm.op_count++;
	}
}

static void printVectorListOne(MCInst *MI, unsigned OpNum, SStream *O)
{
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "}");
}

static void printVectorListTwo(MCInst *MI, unsigned OpNum,
		SStream *O, MCRegisterInfo *MRI)
{
	unsigned Reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
	unsigned Reg0 = MCRegisterInfo_getSubReg(MRI, Reg, ARM_dsub_0);
	unsigned Reg1 = MCRegisterInfo_getSubReg(MRI, Reg, ARM_dsub_1);
	SStream_concat(O, "{");
	printRegName(O, Reg0);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg0;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, Reg1);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg1;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "}");
}

static void printVectorListTwoSpaced(MCInst *MI, unsigned OpNum,
		SStream *O, MCRegisterInfo *MRI)
{
	unsigned Reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
	unsigned Reg0 = MCRegisterInfo_getSubReg(MRI, Reg, ARM_dsub_0);
	unsigned Reg1 = MCRegisterInfo_getSubReg(MRI, Reg, ARM_dsub_2);
	SStream_concat(O, "{");
	printRegName(O, Reg0);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg0;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, Reg1);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg1;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "}");
}

static void printVectorListThree(MCInst *MI, unsigned OpNum, SStream *O)
{
	// Normally, it's not safe to use register enum values directly with
	// addition to get the next register, but for VFP registers, the
	// sort order is guaranteed because they're all of the form D<n>.
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 1);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 1;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "}");
}

static void printVectorListFour(MCInst *MI, unsigned OpNum, SStream *O)
{
	// Normally, it's not safe to use register enum values directly with
	// addition to get the next register, but for VFP registers, the
	// sort order is guaranteed because they're all of the form D<n>.
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 1);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 1;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 3);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 3;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "}");
}

static void printVectorListOneAllLanes(MCInst *MI, unsigned OpNum, SStream *O)
{
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[]}");
}

static void printVectorListTwoAllLanes(MCInst *MI, unsigned OpNum,
		SStream *O, MCRegisterInfo *MRI)
{
	unsigned Reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
	unsigned Reg0 = MCRegisterInfo_getSubReg(MRI, Reg, ARM_dsub_0);
	unsigned Reg1 = MCRegisterInfo_getSubReg(MRI, Reg, ARM_dsub_1);
	SStream_concat(O, "{");
	printRegName(O, Reg0);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg0;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, Reg1);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg1;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[]}");
}

static void printVectorListThreeAllLanes(MCInst *MI, unsigned OpNum, SStream *O)
{
	// Normally, it's not safe to use register enum values directly with
	// addition to get the next register, but for VFP registers, the
	// sort order is guaranteed because they're all of the form D<n>.
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 1);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 1;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[]}");
}

static void printVectorListFourAllLanes(MCInst *MI, unsigned OpNum, SStream *O)
{
	// Normally, it's not safe to use register enum values directly with
	// addition to get the next register, but for VFP registers, the
	// sort order is guaranteed because they're all of the form D<n>.
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 1);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 1;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 3);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 3;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[]}");
}

static void printVectorListTwoSpacedAllLanes(MCInst *MI,
		unsigned OpNum, SStream *O, MCRegisterInfo *MRI)
{
	unsigned Reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
	unsigned Reg0 = MCRegisterInfo_getSubReg(MRI, Reg, ARM_dsub_0);
	unsigned Reg1 = MCRegisterInfo_getSubReg(MRI, Reg, ARM_dsub_2);
	SStream_concat(O, "{");
	printRegName(O, Reg0);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg0;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, Reg1);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = Reg1;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[]}");
}

static void printVectorListThreeSpacedAllLanes(MCInst *MI,
		unsigned OpNum, SStream *O)
{
	// Normally, it's not safe to use register enum values directly with
	// addition to get the next register, but for VFP registers, the
	// sort order is guaranteed because they're all of the form D<n>.
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 4);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 4;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[]}");
}

static void printVectorListFourSpacedAllLanes(MCInst *MI,
		unsigned OpNum, SStream *O)
{
	// Normally, it's not safe to use register enum values directly with
	// addition to get the next register, but for VFP registers, the
	// sort order is guaranteed because they're all of the form D<n>.
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 4);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 4;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[], ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 6);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 6;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "[]}");
}

static void printVectorListThreeSpaced(MCInst *MI, unsigned OpNum, SStream *O)
{
	// Normally, it's not safe to use register enum values directly with
	// addition to get the next register, but for VFP registers, the
	// sort order is guaranteed because they're all of the form D<n>.
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 4);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 4;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "}");
}

static void printVectorListFourSpaced(MCInst *MI, unsigned OpNum, SStream *O)
{
	// Normally, it's not safe to use register enum values directly with
	// addition to get the next register, but for VFP registers, the
	// sort order is guaranteed because they're all of the form D<n>.
	SStream_concat(O, "{");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)));
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum));
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 2;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 4);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 4;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, ", ");
	printRegName(O, MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 6);
	if (MI->csh->detail) {
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].type = ARM_OP_REG;
		MI->flat_insn.arm.operands[MI->flat_insn.arm.op_count].reg = MCOperand_getReg(MCInst_getOperand(MI, OpNum)) + 6;
		MI->flat_insn.arm.op_count++;
	}
	SStream_concat(O, "}");
}
