// distribution boxbackup-0.11.1 (svn version: 2821_2827)
// Box Backup, http://www.boxbackup.org/
// 
// Copyright (c) 2003-2010, Ben Summers and contributors.
// All rights reserved.
// 
// Note that this project uses mixed licensing. Any file with this license
// attached, or where the code LICENSE-DUAL appears on the first line, falls
// under this license. See the file COPYING.txt for more information.
// 
// This file is dual licensed. You may use and distribute it providing that you
// comply EITHER with the terms of the BSD license, OR the GPL license. It is
// not necessary to comply with both licenses, only one.
// 
// The BSD license option follows:
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  
// 3. Neither the name of the Box Backup nor the names of its contributors may
//    be used to endorse or promote products derived from this software without
//    specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// [http://en.wikipedia.org/wiki/BSD_licenses#3-clause_license_.28.22New_BSD_License.22.29]
// 
// The GPL license option follows:
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
// 
// [http://www.gnu.org/licenses/old-licenses/gpl-2.0.html#SEC4]
// --------------------------------------------------------------------------
//
// File
//		Name:    LocalProcessStream.cpp
//		Purpose: Opens a process, and presents stdin/stdout as a stream.
//		Created: 12/3/04
//
// --------------------------------------------------------------------------

#include "Box.h"

#ifdef HAVE_SYS_SOCKET_H
	#include <sys/socket.h>
#endif

#ifdef HAVE_UNISTD_H
	#include <unistd.h>
#endif

#include "LocalProcessStream.h"
#include "autogen_ServerException.h"
#include "Utils.h"

#ifdef WIN32
	#include "FileStream.h"
#else
	#include "SocketStream.h"
#endif

#include "MemLeakFindOn.h"

#define MAX_ARGUMENTS	64

// --------------------------------------------------------------------------
//
// Function
//		Name:    LocalProcessStream(const char *, pid_t &)
//		Purpose: Run a new process, and return a stream giving access
//			 to its stdin and stdout (stdout and stderr on 
//			 Win32). Returns the PID of the new process -- this
//			 must be waited on at some point to avoid zombies
//			 (except on Win32).
//		Created: 12/3/04
//
// --------------------------------------------------------------------------
std::auto_ptr<IOStream> LocalProcessStream(const std::string& rCommandLine,
	pid_t &rPidOut)
{
#ifndef WIN32

	// Split up command
	std::vector<std::string> command;
	SplitString(rCommandLine, ' ', command);

	// Build arguments
	char *args[MAX_ARGUMENTS + 4];
	{
		int a = 0;
		std::vector<std::string>::const_iterator i(command.begin());
		while(a < MAX_ARGUMENTS && i != command.end())
		{
			args[a++] = (char*)(*(i++)).c_str();
		}
		args[a] = NULL;
	}

	// Create a socket pair to communicate over.
	int sv[2] = {-1,-1};
	if(::socketpair(AF_UNIX, SOCK_STREAM, PF_UNSPEC, sv) != 0)
	{
		THROW_EXCEPTION(ServerException, SocketPairFailed)
	}
	
	std::auto_ptr<IOStream> stream(new SocketStream(sv[0]));

	// Fork
	pid_t pid = 0;
	switch(pid = vfork())
	{
	case -1:	// error
		::close(sv[0]);
		::close(sv[1]);
		THROW_EXCEPTION(ServerException, ServerForkError)
		break;
		
	case 0:		// child
		// Close end of the socket not being used
		::close(sv[0]);
		// Duplicate the file handles to stdin and stdout
		if(sv[1] != 0) ::dup2(sv[1], 0);
		if(sv[1] != 1) ::dup2(sv[1], 1);
		// Close the now redundant socket
		if(sv[1] != 0 && sv[1] != 1)
		{
			::close(sv[1]);
		}
		// Execute command!
		::execv(args[0], args);
		::_exit(127);	// report error
		break;
	
	default:
		// just continue...
		break;
	}
	
	// Close the file descriptor not being used
	::close(sv[1]);

	// Return the stream object and PID
	rPidOut = pid;
	return stream;	

#else // WIN32

	SECURITY_ATTRIBUTES secAttr; 
	secAttr.nLength = sizeof(SECURITY_ATTRIBUTES); 
	secAttr.bInheritHandle = TRUE; 
	secAttr.lpSecurityDescriptor = NULL; 

	HANDLE writeInChild, readFromChild;
	if(!CreatePipe(&readFromChild, &writeInChild, &secAttr, 0))
	{
		BOX_ERROR("Failed to CreatePipe for child process: " <<
			GetErrorMessage(GetLastError()));
		THROW_EXCEPTION(ServerException, SocketPairFailed)
	}
	SetHandleInformation(readFromChild, HANDLE_FLAG_INHERIT, 0);

	PROCESS_INFORMATION procInfo; 
	STARTUPINFO startupInfo;

	ZeroMemory(&procInfo,    sizeof(procInfo));
	ZeroMemory(&startupInfo, sizeof(startupInfo));
	startupInfo.cb         = sizeof(startupInfo);
	startupInfo.hStdError  = writeInChild;
	startupInfo.hStdOutput = writeInChild;
	startupInfo.hStdInput  = INVALID_HANDLE_VALUE;
	startupInfo.dwFlags   |= STARTF_USESTDHANDLES;

	CHAR* commandLineCopy = (CHAR*)malloc(rCommandLine.size() + 1);
	strcpy(commandLineCopy, rCommandLine.c_str());

	BOOL result = CreateProcess(NULL, 
		commandLineCopy, // command line 
		NULL,          // process security attributes 
		NULL,          // primary thread security attributes 
		TRUE,          // handles are inherited 
		0,             // creation flags 
		NULL,          // use parent's environment 
		NULL,          // use parent's current directory 
		&startupInfo,  // STARTUPINFO pointer 
		&procInfo);    // receives PROCESS_INFORMATION 

	free(commandLineCopy);
   
	if(!result)
	{
		BOX_ERROR("Failed to CreateProcess: '" << rCommandLine <<
			"': " << GetErrorMessage(GetLastError()));
		CloseHandle(writeInChild);
		CloseHandle(readFromChild);
		THROW_EXCEPTION(ServerException, ServerForkError)
	}

	CloseHandle(procInfo.hProcess);
	CloseHandle(procInfo.hThread);
	CloseHandle(writeInChild);

	rPidOut = (int)(procInfo.dwProcessId);

	std::auto_ptr<IOStream> stream(new FileStream(readFromChild));
	return stream;

#endif // ! WIN32
}




