// distribution boxbackup-0.11.1 (svn version: 2821_2827)
// Box Backup, http://www.boxbackup.org/
// 
// Copyright (c) 2003-2010, Ben Summers and contributors.
// All rights reserved.
// 
// Note that this project uses mixed licensing. Any file with this license
// attached, or where the code LICENSE-DUAL appears on the first line, falls
// under this license. See the file COPYING.txt for more information.
// 
// This file is dual licensed. You may use and distribute it providing that you
// comply EITHER with the terms of the BSD license, OR the GPL license. It is
// not necessary to comply with both licenses, only one.
// 
// The BSD license option follows:
// 
// Redistribution and use in source and binary forms, with or without
// modification, are permitted provided that the following conditions are met:
// 
// 1. Redistributions of source code must retain the above copyright
//    notice, this list of conditions and the following disclaimer.
// 
// 2. Redistributions in binary form must reproduce the above copyright
//    notice, this list of conditions and the following disclaimer in the
//    documentation and/or other materials provided with the distribution.
//  
// 3. Neither the name of the Box Backup nor the names of its contributors may
//    be used to endorse or promote products derived from this software without
//    specific prior written permission.
// 
// THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS IS"
// AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
// IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
// ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER BE LIABLE FOR ANY
// DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
// (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES;
// LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND
// ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
// (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF
// THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.
// 
// [http://en.wikipedia.org/wiki/BSD_licenses#3-clause_license_.28.22New_BSD_License.22.29]
// 
// The GPL license option follows:
// 
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
// 
// [http://www.gnu.org/licenses/old-licenses/gpl-2.0.html#SEC4]
// --------------------------------------------------------------------------
//
// File
//		Name:    HTTPResponse.h
//		Purpose: Response object for HTTP connections
//		Created: 26/3/04
//
// --------------------------------------------------------------------------

#ifndef HTTPRESPONSE__H
#define HTTPRESPONSE__H

#include <string>
#include <vector>

#include "CollectInBufferStream.h"

class IOStreamGetLine;

// --------------------------------------------------------------------------
//
// Class
//		Name:    HTTPResponse
//		Purpose: Response object for HTTP connections
//		Created: 26/3/04
//
// --------------------------------------------------------------------------
class HTTPResponse : public CollectInBufferStream
{
public:
	HTTPResponse(IOStream* pStreamToSendTo);
	HTTPResponse();
	~HTTPResponse();

	// allow copying, but be very careful with the response stream,
	// you can only read it once! (this class doesn't police it).
	HTTPResponse(const HTTPResponse& rOther)
	: mResponseCode(rOther.mResponseCode),
	  mResponseIsDynamicContent(rOther.mResponseIsDynamicContent),
	  mKeepAlive(rOther.mKeepAlive),
	  mContentType(rOther.mContentType),
	  mExtraHeaders(rOther.mExtraHeaders),
	  mContentLength(rOther.mContentLength),
	  mpStreamToSendTo(rOther.mpStreamToSendTo)
	{
		Write(rOther.GetBuffer(), rOther.GetSize());
	}
		
	HTTPResponse &operator=(const HTTPResponse &rOther)
	{
		Reset();
		Write(rOther.GetBuffer(), rOther.GetSize());
		mResponseCode = rOther.mResponseCode;
		mResponseIsDynamicContent = rOther.mResponseIsDynamicContent;
		mKeepAlive = rOther.mKeepAlive;
		mContentType = rOther.mContentType;
		mExtraHeaders = rOther.mExtraHeaders;
		mContentLength = rOther.mContentLength;
		mpStreamToSendTo = rOther.mpStreamToSendTo;
		return *this;
	}
	
	typedef std::pair<std::string, std::string> Header;

	void SetResponseCode(int Code);
	int GetResponseCode() { return mResponseCode; }
	void SetContentType(const char *ContentType);
	const std::string& GetContentType() { return mContentType; }

	void SetAsRedirect(const char *RedirectTo, bool IsLocalURI = true);
	void SetAsNotFound(const char *URI);

	void Send(bool OmitContent = false);
	void SendContinue();
	void Receive(IOStream& rStream, int Timeout = IOStream::TimeOutInfinite);

	// void AddHeader(const char *EntireHeaderLine);
	// void AddHeader(const std::string &rEntireHeaderLine);
	void AddHeader(const char *Header, const char *Value);
	void AddHeader(const char *Header, const std::string &rValue);
	void AddHeader(const std::string &rHeader, const std::string &rValue);
	bool GetHeader(const std::string& rName, std::string* pValueOut) const
	{
		for (std::vector<Header>::const_iterator
			i  = mExtraHeaders.begin();
			i != mExtraHeaders.end(); i++)
		{
			if (i->first == rName)
			{
				*pValueOut = i->second;
				return true;
			}
		}
		return false;
	}
	std::string GetHeaderValue(const std::string& rName)
	{
		std::string value;
		if (!GetHeader(rName, &value))
		{
			THROW_EXCEPTION(CommonException, ConfigNoKey);
		}
		return value;
	}

	// Set dynamic content flag, default is content is dynamic
	void SetResponseIsDynamicContent(bool IsDynamic) {mResponseIsDynamicContent = IsDynamic;}
	// Set keep alive control, default is to mark as to be closed
	void SetKeepAlive(bool KeepAlive) {mKeepAlive = KeepAlive;}

	void SetCookie(const char *Name, const char *Value, const char *Path = "/", int ExpiresAt = 0);

	enum
	{
		Code_OK = 200,
		Code_NoContent = 204,
		Code_MovedPermanently = 301,
		Code_Found = 302,	// redirection
		Code_NotModified = 304,
		Code_TemporaryRedirect = 307,
		Code_MethodNotAllowed = 400,
		Code_Unauthorized = 401,
		Code_Forbidden = 403,
		Code_NotFound = 404,
		Code_InternalServerError = 500,
		Code_NotImplemented = 501
	};

	static const char *ResponseCodeToString(int ResponseCode);
	
	void WriteStringDefang(const char *String, unsigned int StringLen);
	void WriteStringDefang(const std::string &rString) {WriteStringDefang(rString.c_str(), rString.size());}

	// --------------------------------------------------------------------------
	//
	// Function
	//		Name:    HTTPResponse::WriteString(const std::string &)
	//		Purpose: Write a string to the response (simple sugar function)
	//		Created: 9/4/04
	//
	// --------------------------------------------------------------------------
	void WriteString(const std::string &rString)
	{
		Write(rString.c_str(), rString.size());
	}

	// --------------------------------------------------------------------------
	//
	// Function
	//		Name:    HTTPResponse::SetDefaultURIPrefix(const std::string &)
	//		Purpose: Set default prefix used to local redirections
	//		Created: 26/3/04
	//
	// --------------------------------------------------------------------------
	static void SetDefaultURIPrefix(const std::string &rPrefix)
	{
		msDefaultURIPrefix = rPrefix;
	}

private:
	int mResponseCode;
	bool mResponseIsDynamicContent;
	bool mKeepAlive;
	std::string mContentType;
	std::vector<Header> mExtraHeaders;
	int mContentLength; // only used when reading response from stream
	IOStream* mpStreamToSendTo; // nonzero only when constructed with a stream
	
	static std::string msDefaultURIPrefix;

	void ParseHeaders(IOStreamGetLine &rGetLine, int Timeout);
};

#endif // HTTPRESPONSE__H

