// distribution boxbackup-0.11.1 (svn version: 2821_2827)
// Box Backup, http://www.boxbackup.org/
// 
// Copyright (c) 2003-2010, Ben Summers and contributors.
// All rights reserved.
// 
// Note that this project uses mixed licensing. Any file with this license
// attached, or where the code LICENSE-GPL appears on the first line, falls
// under the "Box Backup GPL" license. See the file COPYING.txt for more
// information about this license.
// 
// ---------------------------------------------------------------------
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public License
// as published by the Free Software Foundation; either version 2
// of the License, or (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with this program; if not, write to the Free Software
// Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.
// 
// [http://www.gnu.org/licenses/old-licenses/gpl-2.0.html#SEC4]
// 
// As a special exception to the GPLv2, the Box Backup Project gives
// permission to link any code falling under this license (the Box Backup
// GPL) with any software that can be downloaded from
// the OpenSSL website [http://www.openssl.org] under either the
// "OpenSSL License" or the "Original SSLeay License", and to distribute
// the linked executables under the terms of the "Box Backup GPL" license.
// 
// As a special exception to the GPLv2, the Box Backup Project gives
// permission to link any code falling under this license (the Box Backup
// GPL) with any version of Microsoft's Volume Shadow Copy Service 7.2 SDK
// or Microsoft Windows Software Development Kit (SDK), including
// vssapi.lib, that can be downloaded from the Microsoft website
// [*.microsoft.com], and to distribute the linked executables under the
// terms of the "Box Backup GPL" license.
// --------------------------------------------------------------------------
//
// File
//		Name:    BoxBackupCompareParams.h
//		Purpose: Parameters and notifiers for a compare operation
//		Created: 2008/12/30
//
// --------------------------------------------------------------------------

#ifndef BOXBACKUPCOMPAREPARAMS__H
#define BOXBACKUPCOMPAREPARAMS__H

#include <memory>
#include <string>

#include "BoxTime.h"
#include "ExcludeList.h"
#include "BackupClientMakeExcludeList.h"

// --------------------------------------------------------------------------
//
// Class
//		Name:    BoxBackupCompareParams
//		Purpose: Parameters and notifiers for a compare operation
//		Created: 2003/10/10
//
// --------------------------------------------------------------------------
class BoxBackupCompareParams
{
private:
	std::auto_ptr<const ExcludeList> mapExcludeFiles, mapExcludeDirs;
	bool mQuickCompare;
	bool mIgnoreExcludes;
	bool mIgnoreAttributes;
	box_time_t mLatestFileUploadTime;
	
public:
	BoxBackupCompareParams(bool QuickCompare, bool IgnoreExcludes,
		bool IgnoreAttributes, box_time_t LatestFileUploadTime)
	: mQuickCompare(QuickCompare),
	  mIgnoreExcludes(IgnoreExcludes),
	  mIgnoreAttributes(IgnoreAttributes),
	  mLatestFileUploadTime(LatestFileUploadTime)
	{ }
	
	virtual ~BoxBackupCompareParams() { }
	
	bool QuickCompare() { return mQuickCompare; }
	bool IgnoreExcludes() { return mIgnoreExcludes; }
	bool IgnoreAttributes() { return mIgnoreAttributes; }
	box_time_t LatestFileUploadTime() { return mLatestFileUploadTime; }
		
	void LoadExcludeLists(const Configuration& rLoc)
	{
		mapExcludeFiles.reset(BackupClientMakeExcludeList_Files(rLoc));
		mapExcludeDirs.reset(BackupClientMakeExcludeList_Dirs(rLoc));
	}
	bool IsExcludedFile(const std::string& rLocalPath)
	{
		if (!mapExcludeFiles.get()) return false;
		return mapExcludeFiles->IsExcluded(rLocalPath);
	}
	bool IsExcludedDir(const std::string& rLocalPath)
	{
		if (!mapExcludeDirs.get()) return false;
		return mapExcludeDirs->IsExcluded(rLocalPath);
	}

	virtual void NotifyLocalDirMissing(const std::string& rLocalPath,
		const std::string& rRemotePath) = 0;
	virtual void NotifyLocalDirAccessFailed(const std::string& rLocalPath,
		const std::string& rRemotePath) = 0;
	virtual void NotifyStoreDirMissingAttributes(const std::string& rLocalPath,
		const std::string& rRemotePath) = 0;
	virtual void NotifyRemoteFileMissing(const std::string& rLocalPath,
		const std::string& rRemotePath,
		bool modifiedAfterLastSync) = 0;
	virtual void NotifyLocalFileMissing(const std::string& rLocalPath,
		const std::string& rRemotePath) = 0;
	virtual void NotifyExcludedFileNotDeleted(const std::string& rLocalPath,
		const std::string& rRemotePath) = 0;
	virtual void NotifyDownloadFailed(const std::string& rLocalPath,
		const std::string& rRemotePath, int64_t NumBytes,
		BoxException& rException) = 0;
	virtual void NotifyDownloadFailed(const std::string& rLocalPath,
		const std::string& rRemotePath, int64_t NumBytes,
		std::exception& rException) = 0;
	virtual void NotifyDownloadFailed(const std::string& rLocalPath,
		const std::string& rRemotePath, int64_t NumBytes) = 0;
	virtual void NotifyExcludedFile(const std::string& rLocalPath,
		const std::string& rRemotePath) = 0;
	virtual void NotifyExcludedDir(const std::string& rLocalPath,
		const std::string& rRemotePath) = 0;
	virtual void NotifyDirComparing(const std::string& rLocalPath,
		const std::string& rRemotePath) = 0;
	virtual void NotifyDirCompared(const std::string& rLocalPath,
		const std::string& rRemotePath,	bool HasDifferentAttributes,
		bool modifiedAfterLastSync) = 0;
	virtual void NotifyFileComparing(const std::string& rLocalPath,
		const std::string& rRemotePath) = 0;
	virtual void NotifyFileCompared(const std::string& rLocalPath,
		const std::string& rRemotePath, int64_t NumBytes,
		bool HasDifferentAttributes, bool HasDifferentContents,
		bool modifiedAfterLastSync, bool newAttributesApplied) = 0;
};

#endif // BOXBACKUPCOMPAREPARAMS__H
