/*==============================================================================

Project:   bdumpr
Module:    bdumpr
Filename:  bdumpr.c
Author:    Phil Braham (phil@braham.net)
Date:      Dec 2004

Web site:  www.braham.net/Phil/linux_utils.html

Description:
    Opposite of bdump. Takes bdump output and converts back to binary

==============================================================================*/
/*==============================================================================
Include Files
==============================================================================*/
#include <stdio.h>
#include <stdarg.h>
#include <string.h>
#include <errno.h>
#include <unistd.h>

/*==============================================================================
Local definitions
==============================================================================*/
typedef int bool;
#define true 1
#define false 0

#define LONGS_PER_LINE_ORD       8
#define LONGS_PER_LINE_WIDE     16
#define LONGS_PER_LINE_NARROW    4

#define FLAGS       "bhlmno:svw"

#define FI_LONG     "         "
#define FI_SHORT    "     "
#define FI_BYTE     "   "

#define MAX_LINE_LEN 1024
#define INT_MAX_OFFS 0X7FFFF000	/* Allows for width without overfolwong */

#define VERSION "bdumpr -  V1.1 22-Dec-2004. Written by P.L.Braham"

#define ITEMS_LONG_N	 4
#define ITEMS_LONG_M	 8
#define ITEMS_LONG_W	16
#define ITEMS_SHORT_N	 8
#define ITEMS_SHORT_M	16
#define ITEMS_SHORT_W	32
#define ITEMS_BYTE_N	16
#define ITEMS_BYTE_M	32
#define ITEMS_BYTE_W	64

#define ITEMS_LONG 	0
#define ITEMS_SHORT 	1
#define ITEMS_BYTE	2
#define ITEMS_NARROW	0
#define ITEMS_MED	1
#define ITEMS_WIDE	2

#define SCANF_LINE_4	"%x: %x %x %x %x"
#define SCANF_LINE_8	"%x: %x %x %x %x %x %x %x %x"
#define SCANF_LINE_16	"%x: %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x"
#define SCANF_LINE_32	"%x: %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x"
#define SCANF_LINE_64	"%x: %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x %x"
#define MAX_SIZES	3
#define MAX_WIDTHS	3

struct  SItemsCount
{
    int     count;
    int	    size;
    int     lineLen;
    char    *scanfLine;
} ItemCountArray[MAX_SIZES][MAX_WIDTHS] =
{
    {
        { ITEMS_LONG_N, sizeof (long), 62, SCANF_LINE_4 },
        { ITEMS_LONG_M, sizeof (long), 114, SCANF_LINE_8 },
        { ITEMS_LONG_W, sizeof (long), 218, SCANF_LINE_16 },
    },
    {
        { ITEMS_SHORT_N, sizeof (short), 66, SCANF_LINE_8 },
        { ITEMS_SHORT_M, sizeof (short), 122, SCANF_LINE_16 },
        { ITEMS_SHORT_W, sizeof (short), 234, SCANF_LINE_32 },
    },
    {
        { ITEMS_BYTE_N, sizeof (char), 74, SCANF_LINE_16 },
        { ITEMS_BYTE_M, sizeof (char), 138, SCANF_LINE_32 },
        { ITEMS_BYTE_W, sizeof (char), 266, SCANF_LINE_64 },
    },
};

/*==============================================================================
Function: main
Params:
p1          - Options
p2          - filename to dump
Return:
0 if ok, 1 if error
Description:
See module description
==============================================================================*/
int main(int argc, char **argv)
{
    int     i,
            j,
            inputSize = ITEMS_LONG,
            inputWidth = ITEMS_MED,
            itemsPerLine,
            nextFile,
            itemCount,
            itemSize,
            maxOut = 0,
            bytesOut = 0;

    unsigned long v[ITEMS_BYTE_W],
                addr;

    bool    exitStatusFlag = true,
            firstTimeInFlag = true,
            overrideLenFlag = false,
            exitFlag = false;

    char    c,
            pntr[MAX_LINE_LEN],
            *scanfLine;

    FILE    *inFile,
            *outFile = stdout,
            *helpOut = stderr;

    /*
    * Process arguments
    */
    while ((c = getopt (argc, argv, FLAGS)) != EOF)
    {
        switch ((char) c)
        {
        case 'b' :
        inputSize = ITEMS_BYTE;
        overrideLenFlag = true;
            break;

        case 's' :
            inputSize = ITEMS_SHORT;
            overrideLenFlag = true;
            break;

        case 'l' :
            inputSize = ITEMS_LONG;
            overrideLenFlag = true;
            break;

        case 'w' :
            inputWidth = ITEMS_WIDE;
            overrideLenFlag = true;
            break;

        case 'n' :
            inputWidth = ITEMS_NARROW;
            overrideLenFlag = true;
            break;

        case 'm' :
            inputWidth = ITEMS_MED;
            overrideLenFlag = true;
            break;

        case 'o' :
            maxOut = atoi (optarg);
            break;

        case 'v' :
            printf ("%s\n", VERSION);
            break;

        case 'h' :
            exitStatusFlag = false;
            helpOut = stdout;
            fprintf (helpOut, "%s\n", VERSION);

        default :
            fprintf (helpOut, "Converts the output of bdump back to a binary file.\n");
            fprintf (helpOut, "%s [-%s] [<input filename>] [-]\n", argv[0], FLAGS);
            fprintf (helpOut, "\tWhere <input filename> in the output of bdump. If no filename\n"
                    "\tis supplied or filename is - then stdin is used.\n\n");
            fprintf (helpOut, "\t-b Byte.     InputFile is in sets of 1 byte\n");
            fprintf (helpOut, "\t-s Short.    InputFile is in sets of 2 bytes\n");
            fprintf (helpOut, "\t-l Long.     InputFile is in sets of 4 bytes (default)\n");
            fprintf (helpOut, "\t-n Narrow.   InputFile is %d bytes per line\n", LONGS_PER_LINE_NARROW * sizeof (long));
            fprintf (helpOut, "\t-m Medium    InputFile is %d bytes per line (default)\n", LONGS_PER_LINE_ORD * sizeof (long));
            fprintf (helpOut, "\t-w Wide.     InputFile is %d bytes per line\n", LONGS_PER_LINE_WIDE * sizeof (long));
            fprintf (helpOut, "\t-o<bytes>    Specify max number of bytes to output\n");
            fprintf (helpOut, "\t-h Help.     Print this help\n");
            fprintf (helpOut, "\t-v Vers.     Print version number\n");
            fprintf (helpOut, "\nThe formatting command correspond to the commands in bdump so that bdumpr is able"
                "\nto determine the format of the input file.\n");
            fprintf (helpOut, "\nNote that if the file format is not -b (bytes) then it may not be possible\n"
                    "for bdumpr to determine the exact number of bytes in the file if the size is not\n"
                "an exact multiple of the format size (ie, 4 for for long, 2 for bytes). In this\n"
                "case bdumpr may output unnecessary null bytes at the end of the file. Using\n"
                "the -o option forces bdump to only output the necessary number of bytes.\n");
            exit (exitStatusFlag);
            break;
        }

    } /* end while */

    nextFile = optind;
    if ((nextFile >= argc) || (*(argv[nextFile]) == '-'))
    {
        inFile = stdin;
    }
    else
    {
        inFile = fopen (argv[nextFile], "r");
        if (inFile == NULL)
        {
            fprintf (stderr, "Can't open file %s\n", argv[nextFile]);
            exit (2);
        }
    }

    while ((exitFlag == false) && (fgets (pntr, MAX_LINE_LEN, inFile)) > 0)
    {
        if (firstTimeInFlag == true)
        {
            if (overrideLenFlag == false)
            {
                int len = strlen (pntr);
                bool foundFlag = false;
                for (i = 0; i < MAX_SIZES; i++)
                {
                    for (j = 0; j < MAX_WIDTHS; j++)
                    {
                        if (len == ItemCountArray[i][j].lineLen)
                        {
                            itemsPerLine = ItemCountArray[i][j].count;
                            scanfLine = ItemCountArray[i][j].scanfLine;
                            itemSize = ItemCountArray[i][j].size;
                            foundFlag = true;
                        }
                    }
                }
                if (foundFlag == false)
                {
                    fprintf (stderr, "Unable to determine format and no options supplied (line length = %d)\n", len);
                    exit (2);
                }
            }
            else
            {
                itemsPerLine = ItemCountArray[inputSize][inputWidth].count;
                scanfLine = ItemCountArray[inputSize][inputWidth].scanfLine;
                itemSize = ItemCountArray[inputSize][inputWidth].size;
            }
            firstTimeInFlag = false;
        }


        for (i = 0; i < strlen (pntr); i++)
        {
            if (!isascii (pntr[i]))
            {
                fprintf (stderr, "File is not in correct format (contains binary characters).\n");
    //            fprintf (stderr, "\"%s\"", pntr);
                exit (2);
            }
        }

        for (i = 0; i < itemsPerLine; i++)
        {
            v[i] = 0;
        }
        itemCount = sscanf (pntr, scanfLine, &addr,
                    &v[0], &v[1], &v[2], &v[3], &v[4], &v[5], &v[6], &v[7],
                    &v[8], &v[9], &v[10], &v[11], &v[12], &v[13], &v[14], &v[15],
                    &v[16], &v[17], &v[18], &v[19], &v[20], &v[21], &v[22], &v[23],
                    &v[24], &v[25], &v[26], &v[27], &v[28], &v[29], &v[30], &v[31],
                                    &v[32], &v[33], &v[34], &v[35], &v[36], &v[37], &v[38], &v[39],
                                    &v[40], &v[41], &v[42], &v[43], &v[44], &v[45], &v[46], &v[47],
                                    &v[48], &v[49], &v[50], &v[51], &v[52], &v[53], &v[54], &v[55],
                                    &v[56], &v[57], &v[58], &v[59], &v[60], &v[61], &v[62], &v[63]);
        if (itemCount < 1)
        {
            fprintf (stderr, "File is not in correct format.\n");
            exit (2);
        }
        for (i = 0; i < itemCount - 1; i++)
        {
            bool exitFlag = false;

            if ((maxOut > 0) && ((bytesOut + itemSize) > maxOut))
            {
                itemSize = maxOut - bytesOut;
                exitFlag = true;
            }
            fwrite (&v[i], 1, itemSize, outFile);
            if (exitFlag == true)
            {
                break;
            }
            bytesOut += itemSize;
        }
    }
    fclose (inFile);
/*   fclose (outFile); */
    exit (0);
}
