/* archive.h:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */

#ifndef INCLUDE__LIBARCH__ARCHIVE_H
#define INCLUDE__LIBARCH__ARCHIVE_H


#include "hackerlab/arrays/ar.h"
#include "hackerlab/mem/talloc.h"
#include "libawk/relational.h"
#include "libarch/hooks.h"
#include "libarch/archive-version.h"
#include "libarch/patch-id.h"
#include "libinifile/inifile.h"




struct arch_archive_vtable;

enum arch_revision_type
{
  arch_import_revision,
  arch_simple_revision,
  arch_continuation_revision
};

enum arch_revision_lock_state
{
  arch_revision_unlocked,
  arch_revision_user_locked,
  arch_revision_txn_locked,
  arch_revision_unknown_lock_state,
  arch_revision_illegal_lock_state,
};

typedef enum arch_fail_action_
{
    arch_fail_ignore,
    arch_fail_warn,
    arch_fail_error
} arch_fail_action_t;


struct arch_archive
{
  struct arch_archive_vtable * vtable;

  struct arch_archive * next;
  
  t_uchar * registered_name;
  t_uchar * official_name;
  t_uchar * location;
  t_uchar * version;
  enum arch_archive_access access;
  enum arch_archive_type type;
  t_uchar * mirror_of;
  t_uchar * client_anticipates_mirror;
  int http_blows; /* initialised during connection */
  int signed_archive;
  arch_fail_action_t when_unsigned;
  int in_registry; /* this is configured by ~/.arch-params/archives/archive-name */
};

struct arch_archive_vtable
{
  t_uchar * type;

  /**
   * \brief return the archive version string.
   *
   * if one is not available (i.e. its not an archive)
   * return NULL
   */
  t_uchar * (*archive_version) (struct arch_archive * a);

  rel_table (*categories) (struct arch_archive * a);
  rel_table (*branches) (struct arch_archive * a, t_uchar * category);
  rel_table (*versions) (struct arch_archive * a, t_uchar * package);
  rel_table (*revisions) (struct arch_archive * a, t_uchar * version);

  t_uchar * (*archive_log) (struct arch_archive *, t_uchar * revision);
  int (*revision_type) (enum arch_revision_type * type, int * is_cached,
			int *has_ancestry, 
                        struct arch_archive * a, t_uchar const * revision);
  void (*get_patch) (int out_fd, struct arch_archive * a, t_uchar * revision);
  void (*get_cached) (int out_fd, struct arch_archive * a, t_uchar * revision);
  void (*get_import) (int out_fd, struct arch_archive * a, t_uchar * revision);
  t_uchar * (*get_continuation) (struct arch_archive * a, t_uchar * revision);
  int (*set_meta_info)  (struct arch_archive * a, t_uchar * meta_info_name, t_uchar * meta_info_value);
  t_uchar * (*get_meta_info)  (struct arch_archive * a, t_uchar * meta_info_name);

  int (*make_category) (t_uchar ** errstr, struct arch_archive * a, t_uchar * category);
  int (*make_branch) (t_uchar ** errstr, struct arch_archive * a, t_uchar * branch);
  int (*make_version) (t_uchar ** errstr, struct arch_archive * a, t_uchar * version);

  int (*lock_revision) (t_uchar ** errstr, struct arch_archive * a,
                        t_uchar * version,
                        t_uchar * prev_level,
                        t_uchar * uid,
                        t_uchar * txn_id,
                        t_uchar * new_level);


  int (*revision_ready) (t_uchar ** errstr, struct arch_archive *a,
			 struct arch_archive * from_archive,
                         t_uchar * version,
                         t_uchar * prev_level,
                         t_uchar * uid, 
                         t_uchar * txn_id,
                         t_uchar * new_level);

  int (*finish_revision) (t_uchar ** errstr, struct arch_archive * a,
                          t_uchar * version,
                          t_uchar * prev_level,
                          t_uchar * uid,
                          t_uchar * txn_id,
                          t_uchar * new_level);

  int (*break_revision_lock) (t_uchar ** errstr, struct arch_archive * a,
                              t_uchar * version,
                              t_uchar * prev_level,
                              t_uchar * uid,
                              t_uchar * txn_id);

  enum arch_revision_lock_state (*lock_state) (t_uchar ** prev_level_ret,
                                               t_uchar ** uid_ret,
                                               t_uchar ** txn_id_ret,
                                               struct arch_archive * a,
                                               t_uchar * version);

  int (*put_log) (t_uchar ** errstr, struct arch_archive * a,
                  t_uchar * version,
                  t_uchar * prev_level,
                  t_uchar * uid,
                  t_uchar * txn_id,
                  t_uchar * log_text);
  int (*put_continuation) (t_uchar ** errstr, struct arch_archive * a,
                           t_uchar * version,
                           t_uchar * prev_level,
                           t_uchar * uid,
                           t_uchar * txn_id,
                           t_uchar * continuation);
  int (*put_changeset) (t_uchar ** errstr, struct arch_archive * a,
                        t_uchar * version,
                        t_uchar * prev_level,
                        t_uchar * uid,
                        t_uchar * txn_id,
                        t_uchar * level,
                        int in_fd);
  int (*put_import) (t_uchar ** errstr, struct arch_archive * a,
                     t_uchar * version,
                     t_uchar * prev_level,
                     t_uchar * uid,
                     t_uchar * txn_id,
                     t_uchar * level,
                     int in_fd);

  int (*put_cached) (t_uchar ** errstr, struct arch_archive * a,
		     struct arch_archive * from_archive,
                     t_uchar * revision,
                     int in_fd);
  int (*delete_cached) (t_uchar ** errstr, struct arch_archive * a, t_uchar * revision);

  void (*repair_non_txnal) (int chatter_fd, struct arch_archive * a);

  void (*get_ancestry) (int out_fd, struct arch_archive * a, t_uchar * revision);
  int (*put_ancestry) (t_uchar **errstr, struct arch_archive *a,
		       struct arch_archive * from_archive,
		       t_uchar * revision,
		       int in_fd);
  int (*delete_ancestry) (t_uchar ** errstr, struct arch_archive * a, t_uchar * revision);
  int (*set_mirror) (t_uchar ** errstr, struct arch_archive *archive, int enabled);
};



/* TODO this belongs as the underlying mechanism for arch_archive_connect.. name, 0, ?, ? */
/* A connection cache that can be used by the core code eventually,
 * and by any repetitive looping areas in the interim,
 * to simplify archive connection mgmt.
 */
AR_TYPEDEF (struct arch_archive *, arch_archive);
typedef struct _arch_archive_connection_cache
{
    ar_arch_archive cache;
} arch_archive_connection_cache;

extern int arch_archive_finalise (void *archive);
void arch_archive_connection_cache_init (arch_archive_connection_cache * cache);
void arch_archive_connection_cache_finalise (arch_archive_connection_cache * cache, struct arch_archive * dontclose);
struct arch_archive * arch_archive_connection_cache_find_or_maybe_connect (arch_archive_connection_cache * cache, t_uchar * name, int soft_errors);
void arch_archive_connection_cache_add (arch_archive_connection_cache * cache, struct arch_archive * arch);
extern void arch_make_archive (t_uchar * name, t_uchar * location, t_uchar * mirror_of, int dot_listing_lossage, int signed_archive, int tla_archive, int register_archive);
extern void arch_archive_register (struct arch_archive * archive);
extern struct arch_archive * arch_archive_connect_branch (t_uchar const * branch, t_uchar ** branch_out);
extern struct arch_archive * arch_archive_connect_writeable_branch (t_uchar const * branch, t_uchar ** branch_out);
extern struct arch_archive * arch_archive_connect_commitable_branch (t_uchar const * branch, t_uchar ** branch_out);
extern struct arch_archive * arch_archive_connect_readonly (t_uchar const * name);
extern struct arch_archive * arch_archive_connect_writable (t_uchar const * name);
extern struct arch_archive * arch_archive_connect_commitable (t_uchar const * name);
extern struct arch_archive * arch_archive_connect_ext (t_uchar const * const name, t_uchar const * const want_mirror_of, int soft_errors);
extern struct arch_archive * arch_archive_connect_location (t_uchar const * location, int soft_errors);
extern struct arch_archive * arch_archive_connect_location_ext (t_uchar const * name, t_uchar const * location, t_uchar const * want_mirror_of, int soft_errors, int override_signed);
extern int arch_archive_cmp_location (t_uchar const *left, t_uchar const *right);
extern int arch_archive_check_signed_status (struct arch_archive *archive, int status_fd);
extern arch_fail_action_t arch_archive_inifile_signed (struct arch_archive *archive, inifile_t *inifile, arch_fail_action_t default_value);
extern void arch_archive_inifile_set_signed (inifile_t *inifile, arch_fail_action_t value);
extern int arch_archive_set_mirror (t_uchar ** errstr, struct arch_archive *archive, int enabled);
extern int arch_archive_is_cached_connection (struct arch_archive *archive);
#define arch_archive_close(arch) talloc_free(arch)
extern t_uchar * arch_archive_version (struct arch_archive * arch);
extern rel_table arch_archive_categories (struct arch_archive * arch);
extern rel_table arch_archive_branches (struct arch_archive * arch, t_uchar * category);
extern rel_table arch_archive_versions (struct arch_archive * arch, t_uchar * package);
extern t_uchar * arch_archive_latest_revision (struct arch_archive * arch, t_uchar * version, int full);
extern rel_table arch_archive_revisions (struct arch_archive * arch, t_uchar const * version, int full);
extern t_uchar * arch_archive_log (struct arch_archive * arch, t_uchar * revision);
extern enum arch_revision_type arch_archive_get_revision_type (struct arch_archive * arch, t_uchar const * revision);
extern void arch_revision_type (enum arch_revision_type * type, int * is_cached, int *has_ancestry,
                                struct arch_archive * arch, arch_patch_id * revision);
extern int arch_revision_exists (struct arch_archive * arch, t_uchar * revision);
extern void arch_get_patch_targz (int out_fd, struct arch_archive * arch, t_uchar * revision);
extern void arch_get_patch (struct arch_archive * arch, arch_patch_id * const revision, t_uchar const * const dest_dir);
extern void arch_get_cached_revision_targz (int out_fd, struct arch_archive * arch, t_uchar * revision);
extern void arch_get_cached_revision (struct arch_archive * arch, arch_patch_id * const revision, t_uchar const * const dest_dir);
extern void arch_get_import_targz (int out_fd, struct arch_archive * arch, t_uchar * revision);
extern void arch_get_import_revision (struct arch_archive * arch, arch_patch_id * const revision, t_uchar const * const dest_dir);
extern arch_patch_id * arch_get_continuation (struct arch_archive * arch, arch_patch_id * revision);
extern int arch_set_meta_info (struct arch_archive * arch, t_uchar * meta_info_name, t_uchar * meta_info_value);
extern t_uchar * arch_get_meta_info (struct arch_archive * arch, t_uchar * meta_info_name);
extern t_uchar * arch_archive_not_writable (struct arch_archive * arch, int mirror_ok);
extern int arch_make_category (t_uchar ** errstr,
                               struct arch_archive * arch, t_uchar * category);
extern int arch_make_branch (t_uchar ** errstr,
                             struct arch_archive * arch, t_uchar * branch);
extern int arch_make_version (t_uchar ** errstr,
                              struct arch_archive * arch, t_uchar * version);
extern int arch_archive_lock_revision (t_uchar ** errstr, struct arch_archive * a,
                                       t_uchar * version,
                                       t_uchar * prev_level,
                                       t_uchar * uid,
                                       t_uchar * txn_id,
                                       t_uchar * new_level);
extern int arch_revision_ready (t_uchar ** errstr, struct arch_archive *a,
                                t_uchar * version,
                                t_uchar * prev_level,
                                t_uchar * uid,
                                t_uchar * txn_id,
                                t_uchar * new_level);
extern int arch_mirror_revision_ready (t_uchar ** errstr, struct arch_archive *a,
				       struct arch_archive *from_archive,
				       t_uchar * version,
				       t_uchar * prev_level,
				       t_uchar * uid,
				       t_uchar * txn_id,
				       t_uchar * new_level);
extern int arch_archive_finish_revision (t_uchar ** errstr, struct arch_archive * a,
                                         t_uchar * version,
                                         t_uchar * prev_level,
                                         t_uchar * uid,
                                         t_uchar * txn_id,
                                         t_uchar * new_level);
extern enum arch_revision_lock_state arch_archive_revision_lock_state (t_uchar ** prev_level_ret,
                                                                       t_uchar ** uid_ret,
                                                                       t_uchar ** txn_id_ret,
                                                                       struct arch_archive * a,
                                                                       t_uchar * version);
extern int arch_archive_break_revision_lock (t_uchar ** errstr, struct arch_archive * a,
                                             t_uchar * version,
                                             t_uchar * prev_level,
                                             t_uchar * uid,
                                             t_uchar * txn_id);
extern int arch_archive_put_log (t_uchar ** errstr, struct arch_archive * a,
                                 t_uchar * version,
                                 t_uchar * prev_level,
                                 t_uchar * uid,
                                 t_uchar * txn_id,
                                 t_uchar * log_text);
extern int arch_archive_put_continuation (t_uchar ** errstr, struct arch_archive * a,
                                          t_uchar * version,
                                          t_uchar * prev_level,
                                          t_uchar * uid,
                                          t_uchar * txn_id,
                                          t_uchar * continuation);
extern int arch_archive_put_changeset_targz (t_uchar ** errstr, struct arch_archive * a,
                                             t_uchar * version,
                                             t_uchar * prev_level,
                                             t_uchar * uid,
                                             t_uchar * txn_id,
                                             t_uchar * level,
                                             int in_fd);
extern int arch_archive_put_changeset (t_uchar ** errstr, struct arch_archive * a,
                                       t_uchar * version,
                                       t_uchar * prev_level,
                                       t_uchar * uid,
                                       t_uchar * txn_id,
                                       t_uchar * level,
                                       t_uchar * dir);
extern int arch_archive_put_import_targz (t_uchar ** errstr, struct arch_archive * a,
                                          t_uchar * version,
                                          t_uchar * prev_level,
                                          t_uchar * uid,
                                          t_uchar * txn_id,
                                          t_uchar * level,
                                          int in_fd);
extern int arch_archive_put_import (t_uchar ** errstr, struct arch_archive * a,
                                    t_uchar * version,
                                    t_uchar * prev_level,
                                    t_uchar * uid,
                                    t_uchar * txn_id,
                                    t_uchar * level,
                                    t_uchar * dir);
extern int arch_archive_put_cached_targz (t_uchar ** errstr, struct arch_archive * a, t_uchar * revision, int in_fd);
extern int arch_archive_mirror_cached_targz (t_uchar ** errstr, struct arch_archive * a, struct arch_archive *from_archive, t_uchar * revision, int in_fd);
extern int arch_archive_put_cached (t_uchar ** errstr, struct arch_archive * a, t_uchar * revision, t_uchar * dir);
extern int arch_archive_delete_cached (t_uchar ** errstr, struct arch_archive * a, t_uchar * revision);
extern void arch_archive_repair_non_txnal (int chatter_fd, struct arch_archive * a);
extern void arch_archive_get_ancestry (int out_fd, struct arch_archive * arch, t_uchar * revision);
extern int arch_archive_put_ancestry (t_uchar ** errstr, struct arch_archive * a, t_uchar * revision, int in_fd);
extern int arch_archive_mirror_ancestry (t_uchar ** errstr, struct arch_archive * a, struct arch_archive *from_archive, t_uchar * revision, int in_fd);
extern int arch_archive_delete_ancestry (t_uchar ** errstr, struct arch_archive * a, t_uchar * revision);
extern t_uchar * arch_generate_txn_id (void);
extern arch_patch_id * arch_previous_revision (struct arch_archive * arch, arch_patch_id * revision);
extern t_uchar * arch_ancestor_revision (struct arch_archive * arch, t_uchar * revision);
extern t_uchar * archive_tmp_file_name (t_uchar * dir, t_uchar * basename);
extern t_uchar * make_tmp_tar_archive (t_uchar * dir);
extern int arch_get_meta_int_info(struct arch_archive * arch, t_uchar * key);
extern void arch_maybe_cache_commit (struct arch_archive *arch, 
                                     t_uchar * revision, 
                                     t_uchar * anc_archive,
                                     t_uchar * anc_revision,
                                     t_uchar * changeset_path);
extern int arch_archive_has_registry_entry (struct arch_archive * arch);
extern t_uchar * arch_fs_archive_category_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar * category);
extern t_uchar * arch_fs_archive_branch_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar * package);
extern t_uchar * arch_fs_archive_version_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar * version);
extern t_uchar * arch_fs_archive_revision_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar const * revision);
extern t_uchar * arch_fs_archive_revision_log_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar const * revision);
extern t_uchar * arch_fs_archive_changeset_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar const * revision);
extern t_uchar * arch_fs_archive_import_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar const * revision);
extern t_uchar * arch_fs_archive_cached_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar const * revision);
extern t_uchar * arch_fs_archive_cached_checksum_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar const * revision);
extern t_uchar * arch_fs_archive_ancestry_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar const * revision);
extern t_uchar * arch_fs_archive_ancestry_checksum_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar const * revision);
extern t_uchar * arch_fs_archive_continuation_path (struct arch_archive * arch, t_uchar * archive_path, t_uchar const * revision);
extern t_uchar * arch_fs_archive_revision_lock_unlocked_path (struct arch_archive * arch, t_uchar * archive_path,
                                                              t_uchar * version,
                                                              t_uchar * prev_level);
extern t_uchar * arch_fs_archive_revision_lock_locked_path (struct arch_archive * arch, t_uchar * archive_path,
                                                            t_uchar * version,
                                                            t_uchar * prev_level,
                                                            t_uchar * arch_user_id,
                                                            t_uchar * txn_id);
extern t_uchar * arch_fs_archive_revision_lock_locked_contents_path (struct arch_archive * arch, t_uchar * archive_path,
                                                                     t_uchar * version,
                                                                     t_uchar * prev_level,
                                                                     t_uchar * arch_user_id,
                                                                     t_uchar * txn_id);
extern t_uchar * arch_fs_archive_revision_lock_broken_path (struct arch_archive * arch, t_uchar * archive_path,
                                                            t_uchar * version,
                                                            t_uchar * prev_level);
#endif  /* INCLUDE__LIBARCH__ARCHIVE_H */


/* tag: Tom Lord Sat Jan  5 15:26:10 2002 (archive.h)
 */
