/* cmd-library-config.c
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */


#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/cmd/main.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/libraries.h"
#include "commands/library-config.h"
#include "commands/version.h"



static t_uchar * usage = N_("[options] library-dir");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_greedy, 0, "greedy", 0, \
      N_("make the library greedy")) \
  OP (opt_non_greedy, 0, "non-greedy", 0, \
      N_("make the library not greedy")) \
  OP (opt_sparse, 0, "sparse", 0, \
      N_("make the library sparse")) \
  OP (opt_non_sparse, 0, "non-sparse", 0, \
      N_("make the library not sparse"))


t_uchar arch_cmd_library_config_help[] = N_("configure parameters of a revision library\n"
                                            "Set/show various parameters for a revision library.\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_library_config (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  int make_greedy = 0;
  int make_non_greedy = 0;
  int make_sparse = 0;
  int make_non_sparse = 0;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_library_config_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_greedy:
          {
            make_greedy = 1;
            make_non_greedy = 0;
            break;
          }

        case opt_non_greedy:
          {
            make_non_greedy = 1;
            make_greedy = 0;
            break;
          }

        case opt_sparse:
          {
            make_sparse = 1;
            make_non_sparse = 0;
            break;
          }

        case opt_non_sparse:
          {
            make_non_sparse = 1;
            make_sparse = 0;
            break;
          }
        }
    }

  if (argc != 2)
    goto usage_error;

  {
    t_uchar * lib = argv[1];

    arch_verify_is_library (lib);

    if (make_greedy)
      arch_set_library_greediness (lib, 1);

    if (make_non_greedy)
      arch_set_library_greediness (lib, 0);

    if (make_sparse)
      arch_set_library_sparseness (lib, 1);

    if (make_non_sparse)
      arch_set_library_sparseness (lib, 0);

    safe_printfmt (1, "* library %s\n", lib);

    if (arch_library_is_greedy (lib))
      safe_printfmt (1, "greedy? yes\n");
    else
      safe_printfmt (1, "greedy? no\n");

    if (arch_library_is_sparse (lib))
      safe_printfmt (1, "sparse? yes\n");
    else
      safe_printfmt (1, "sparse? no\n");
  }

   return 0;
}




/* tag: Tom Lord Fri Dec  5 16:35:15 2003 (cmd-library-config.c)
 */
