/* cmd-delta.c:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */




#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/fs/cwd.h"
#include "libfsutils/tmp-files.h"
#include "libfsutils/rmrf.h"
#include "libfsutils/dir-as-cwd.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/my.h"
#include "libarch/archive.h"
#include "libarch/pristines.h"
#include "libarch/make-changeset.h"
#include "libarch/changeset-report.h"
#include "libarch/inode-sig.h"
#include "commands/cmd.h"
#include "commands/cmdutils.h"
#include "commands/delta.h"
#include "commands/version.h"


/* __STDC__ prototypes for static functions */
static void delta_callback (void * vfd, char * fmt, va_list ap);



static t_uchar * usage = N_("[options] (REVISION|TREE)-A (REVISION|TREE)-B [DESTDIR]");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_archive, "A", "archive", 1, \
      N_("Override `my-default-archive'")) \
  OP (opt_quiet, "q" , "quiet", 0, \
      N_("quiet down progress reports while computing changeset")) \
  OP (opt_no_changeset, "n" , "no-changeset", 0, \
      N_("do not generate a changeset")) \
  OP (opt_diff, 0, "diffs", 0, \
      N_("print changeset report with diffs (implies -n)")) \
  OP (opt_p, "p", "show-c-function", 0, \
      N_("Show which C function each change is in.")) \
  OP (opt_unescaped, 0, "unescaped", 0, \
      N_("show filenames in unescaped form"))


t_uchar arch_cmd_delta_help[] = N_("compute a changeset (or diff) between any two trees or revisions\n"
                                 "\n"
                                 "Given (REVISION|TREE)-A and (REVISION|TREE)-B, baz will build a changeset\n"
                                 "that comprises the changes between REVISION-A and REVISION-B\n"
                                 "\n"
                                 "Example:\n"
                                 "  baz delta baz--devo--1.1--patch-6 \\\n"
                                 "            baz--devo--1.1--patch-8 ,,changes\n"
                                 "\n"
                                 "  Will pull patch-6 and patch-8 from baz--devo--1.1 and compute\n"
                                 "  a changeset, which will be saved in a newly created ,,changes\n"
                                 "  directory. If you would like a report instead,\n"
                                 "  append the --diffs option");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};



int
arch_cmd_delta (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  t_uchar * default_archive = 0;
  int no_changeset = 0;
  int report = 0;
  int quiet = 0;
  int escape_classes = arch_escape_classes;
  t_uchar ** diff_opts = NULL;
  t_uchar * p_opts[] = {"-p", NULL};

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_delta_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            lim_free (0, default_archive);
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_diff:
          {
            report = 1;
            no_changeset = 1;
            break;
          }

        case opt_no_changeset:
          {
            no_changeset = 1;
            break;
          }

        case opt_quiet:
          {
            quiet = 1;
            break;
          }

        case opt_unescaped:
          {
            escape_classes = 0;
            break;
          }
        case opt_p:
          {
            diff_opts = p_opts;
          }
        }
    }

  if (argc < 3 || argc > 4)
    goto usage_error;

  if (default_archive && !arch_valid_archive_name (default_archive))
    {
      safe_printfmt (2, "%s: invalid archive name (%s)\n",
                     argv[0], default_archive);
      exit (1);
    }

  default_archive = arch_my_default_archive (default_archive);

  {
    t_uchar * a_spec;
    t_uchar * b_spec;
    t_uchar * dest = 0;
    t_uchar * scratch_dir = 0;
    struct arch_make_changeset_report make_report = {0, };
    t_uchar * orig_archive = 0;
    t_uchar * orig_revision = 0;
    assoc_table inode_shortcut = 0;
    arch_project_tree_t * orig_tree;
    arch_project_tree_t * mod_tree;

    a_spec = argv[1];
    b_spec = argv[2];

    if (argc == 4)
      {
        dest = talloc_strdup (talloc_context, argv[3]);
      }
    else
      {
        dest = talloc_tmp_file_name (talloc_context, ".", ",,delta");
        rmrf_file (dest);
      }

    scratch_dir = talloc_tmp_file_name (dest, ".", ",,delta-scratch");

    orig_tree = arch_interpret_delta_path (&orig_archive, &orig_revision, scratch_dir, a_spec, NULL);
    mod_tree = arch_interpret_delta_path (0, 0, scratch_dir, b_spec, NULL);

    if (! quiet)
      {
	safe_printfmt (1, "* computing changeset\n");
	make_report.callback = delta_callback;
	make_report.thunk = (void *)1;
      }

    if (orig_archive)
      {
        arch_read_inode_sig_ids (0, &inode_shortcut, mod_tree->root, orig_archive, orig_revision);
      }

    arch_make_changeset (&make_report, orig_tree, mod_tree, dest, arch_unspecified_id_tagging, arch_inventory_unrecognized, 0, inode_shortcut, 0, escape_classes);

    if (report)
      {
        struct arch_changeset_report r = {0, };

        safe_printfmt (1, "* changeset report\n");
        arch_evaluate_changeset (&r, dest);
        arch_print_changeset_custom_diffs(&r, orig_tree->root, mod_tree->root, diff_opts, escape_classes);

        arch_free_changeset_report_data (&r);
      }

    rmrf_file (scratch_dir);

    if (no_changeset)
      rmrf_file (dest);
    else if (! quiet)
      safe_printfmt (1, "* changeset: %s\n", dest);

    talloc_free (dest);
    lim_free (0, orig_archive);
    lim_free (0, orig_revision);
    free_assoc_table (inode_shortcut);
    arch_free_make_changeset_report_data (&make_report);
    arch_project_tree_delete (orig_tree);
    arch_project_tree_delete (mod_tree);
  }

  lim_free (0, default_archive);

  return 0;
}





static void
delta_callback (void * vfd, char * fmt, va_list ap)
{
  int fd;

  fd = (int)(t_ulong)vfd;
  safe_printfmt_va_list (fd, fmt, ap);
  safe_flush (fd);
}





/* tag: Tom Lord Wed Jun  4 13:44:58 2003 (cmd-revdelta.c)
 */
