/* cmd-apply-delta.c:
 *
 ****************************************************************
 * Copyright (C) 2003 Tom Lord
 *
 * See the file "COPYING" for further information about
 * the copyright and warranty status of this work.
 */




#include "config-options.h"
#include "po/gettext.h"
#include "hackerlab/bugs/exception.h"
#include "hackerlab/cmd/main.h"
#include "hackerlab/fs/file-names.h"
#include "hackerlab/fs/cwd.h"
#include "libfsutils/tmp-files.h"
#include "libfsutils/rmrf.h"
#include "libfsutils/dir-as-cwd.h"
#include "libarch/namespace.h"
#include "libarch/project-tree.h"
#include "libarch/proj-tree-lint.h"
#include "libarch/my.h"
#include "libarch/archive.h"
#include "libarch/pristines.h"
#include "libarch/local-cache.h"
#include "libarch/make-changeset.h"
#include "libarch/copy-project-tree.h"
#include "libarch/conflict-handling.h"
#include "libarch/apply-changeset.h"
#include "libarch/merge-three-way.h"
#include "commands/cmd.h"
#include "commands/get.h"
#include "commands/delta.h"
#include "commands/apply-delta.h"
#include "commands/version.h"
#include "commands/cmdutils.h"


/* __STDC__ prototypes for static functions */


static t_uchar * usage = N_("[options] FROM(REVISION|DIR) TO(REVISION|DIR)");

#define OPTS(OP) \
  OP (opt_help_msg, "h", "help", 0, \
      N_("Display a help message and exit.")) \
  OP (opt_long_help, "H", 0, 0, \
      N_("Display a verbose help message and exit.")) \
  OP (opt_version, "V", "version", 0, \
      N_("Display a release identifier string\n" \
      "and exit.")) \
  OP (opt_archive, "A", "archive", 1, \
      N_("Override `my-default-archive'")) \
  OP (opt_quiet, "q", "quiet", 0, \
      N_("instruct command to be quieter when applying delta")) \
  OP (opt_dir, "d", "dir DIR", 1, \
      N_("Operate on project tree in DIR (default `.')")) \
  OP (opt_three_way, "t", "three-way", 0, \
      N_("Perform a 3-way (diff3-style) merge.")) \
  OP (opt_dest, 0, "dest DEST", 1, \
      N_("Instead of modifying the project tree in-place,\n" \
      "make a copy of it to DEST and apply the result to that")) \
  OP (opt_unescaped, 0, "unescaped", 0, \
      N_("show filenames in unescaped form"))


t_uchar arch_cmd_apply_delta_help[] =
  ("compute a changeset between any two trees or revisions and apply it to a project tree\n"

   "A delta between A and B (both of which may be either a full revision or\n"
   "a project tree) is computed, and then applied to the project tree.\n"
   "\n"
   "Exit Status Codes:\n"
   "\n"
   "    0  No conflict during patch\n"
   "    1  Conflicts occurred during patch\n"
   "    3  Internal Error\n");

enum options
{
  OPTS (OPT_ENUM)
};

static struct opt_desc opts[] =
{
  OPTS (OPT_DESC)
    {-1, 0, 0, 0, 0}
};

static int do_three_way(int chatter_fd, arch_project_tree_t * upon_tree, t_uchar * from_spec, t_uchar * to_spec, int escape_classes);



int
arch_cmd_apply_delta (t_uchar * program_name, int argc, char * argv[])
{
  int o;
  struct opt_parsed * option;
  int three_way = 0;
  t_uchar * default_archive = 0;
  t_uchar * target = 0;
  t_uchar * dest = 0;
  int quiet = 0;
  int exit_status = 0;
  int escape_classes = arch_escape_classes;

  safe_buffer_fd (1, 0, O_WRONLY, 0);

  option = 0;

  while (1)
    {
      o = opt_standard (lim_use_must_malloc, &option, opts, &argc, argv, program_name, usage, libarch_version_string, arch_cmd_apply_delta_help, opt_help_msg, opt_long_help, opt_version);
      if (o == opt_none)
        break;
      switch (o)
        {
        default:
          safe_printfmt (2, "unhandled option `%s'\n", option->opt_string);
          panic ("internal error parsing arguments");

        usage_error:
          opt_usage (2, argv[0], program_name, usage, 1);
          exit (1);

          /* bogus_arg: */
          safe_printfmt (2, "ill-formed argument for `%s' (`%s')\n", option->opt_string, option->arg_string);
          goto usage_error;

        case opt_archive:
          {
            lim_free (0, default_archive);
            default_archive = str_save (0, option->arg_string);
            break;
          }

        case opt_quiet:
          {
            quiet = 1;
            break;
          }

        case opt_three_way:
          {
            three_way = 1;
            break;
          }


        case opt_dir:
          {
            lim_free (0, target);
            target = str_save (0, option->arg_string);
            break;
          }

        case opt_dest:
          {
            lim_free (0, dest);
            dest = str_save (0, option->arg_string);
            break;
          }

	case opt_unescaped:
	  {
	    escape_classes = 0;
	    break;
	  }
        }
    }

  if (argc != 3)
    goto usage_error;

  if (default_archive && !arch_valid_archive_name (default_archive))
    {
      safe_printfmt (2, "%s: invalid archive name (%s)\n",
                     argv[0], default_archive);
      exit (1);
    }

  default_archive = arch_my_default_archive (default_archive);

  {
    t_uchar * from_spec;
    t_uchar * to_spec;
    t_uchar * changeset = 0;
    arch_project_tree_t * upon_tree;

    from_spec = argv[1];
    to_spec = argv[2];

    if (target)
	upon_tree = arch_project_tree_new (talloc_context, target);
    else
	upon_tree = arch_project_tree_new (talloc_context, ".");

    arch_tree_ensure_no_conflicts(upon_tree);

    if (dest)
      {
        t_uchar * dest_dir = file_name_directory_file (0, dest);

        if (! dest_dir)
          dest_dir = str_save (0, ".");

        changeset = talloc_tmp_file_name (talloc_context, dest_dir, ",,apply-delta-changeset");

        safe_printfmt (1, "* copying %s to %s\n", upon_tree->root, dest);
        arch_copy_project_tree (upon_tree, dest, 1, 1);

        lim_free (0, dest_dir);
      }
    else
      {
        dest = str_save (0, upon_tree->root);
        changeset = talloc_tmp_file_name (talloc_context, dest, ",,apply-delta-changeset");
      }

    /* FIXME: reinstate quiet/non quiet for three way. */
    if (three_way)
      {
        exit_status = do_three_way(1, upon_tree, from_spec, to_spec, escape_classes);
      }
    else
      {
	if (quiet)
	  {
	    if (default_archive)
	      {
		arch_call_cmd (arch_cmd_delta, argv[0], "--quiet",
			       "-A", default_archive, from_spec, to_spec, changeset,
			       ((escape_classes == 0) ? "--unescaped" : 0), 0);
	      }
	    else
	      {
		arch_call_cmd (arch_cmd_delta, argv[0], "--quiet",
			       from_spec, to_spec, changeset,
			       ((escape_classes == 0) ? "--unescaped" : 0), 0);
	      }
	  }
	else
	  {
	    if (default_archive)
	      {
		arch_call_cmd (arch_cmd_delta, argv[0], "-A", default_archive,
			       from_spec, to_spec, changeset,
			       ((escape_classes == 0) ? "--unescaped" : 0), 0);
	      }
	    else
	      {
		arch_call_cmd (arch_cmd_delta, argv[0],
			       from_spec, to_spec, changeset,
			       ((escape_classes == 0) ? "--unescaped" : 0), 0);
	      }
	  }
          
          {
            
            /* This is the end of the ! three_way else */
            struct arch_apply_changeset_report * r;
	    arch_project_tree_t * dest_tree;
	    
	    dest_tree = arch_project_tree_new (talloc_context, dest);
            
            safe_printfmt (1, "* applying changeset\n");
            
            r = arch_apply_changeset (changeset, talloc_context, dest_tree,
                                  arch_unspecified_id_tagging,
                                  arch_inventory_unrecognized,
                                  0, 0, escape_classes, 
				  apply_print_callback, (void *)1);
            
            exit_status = arch_conflicts_occurred (r);
            
	    talloc_free (r);
	    arch_project_tree_delete (dest_tree);
          
          }
        
        rmrf_file (changeset);
      
      }
    
    talloc_free (changeset);
    arch_project_tree_delete (upon_tree);
  
  } 
  lim_free (0, dest);
  lim_free (0, default_archive);

  return exit_status;
}




static int
do_three_way(int chatter_fd, arch_project_tree_t * upon_tree, t_uchar * from_spec, t_uchar * to_spec, int escape_classes)
{
  t_uchar * scratch_dir = talloc_tmp_file_name (talloc_context, ".", ",,apply-delta-scratch");
  arch_project_tree_t *from_tree = arch_interpret_delta_path (0, 0, scratch_dir, from_spec, upon_tree);
  arch_project_tree_t * to_tree = arch_interpret_delta_path (0, 0, scratch_dir, to_spec, upon_tree);
  int result = arch_merge3 (chatter_fd, upon_tree, from_tree, to_tree, escape_classes, 1);
  rmrf_file (scratch_dir);
  talloc_free (scratch_dir);
  arch_project_tree_delete (from_tree);
  arch_project_tree_delete (to_tree);
  return result;
}



/* tag: Tom Lord Wed Jun  4 15:40:44 2003 (cmd-deltapatch.c)
 */
