// ===========================================================================
// File: "aidaPrint.c"
//                        Created: 2010-08-09 21:56:18
//              Last modification: 2011-06-06 21:56:07
// Author: Bernard Desgraupes
// e-mail: <bdesgraupes@users.sourceforge.net>
// (c) Copyright: Bernard Desgraupes 2010-2011
// All rights reserved.
// ===========================================================================

#include "aidaMain.h"



// ------------------------------------------------------------------------
// 
// "aida_writeByte" --
// 
// Handle bytes read by the lexer. Accumulate enough bytes until it is
// possible to convert from the InputEncoding to Utf-8 and to
// build a Tcl_Obj.
// 
// ------------------------------------------------------------------------
void aida_writeByte(char inChar, aida_parse_t * inScanParam)
{
	if (inScanParam->to == to_dstr) {
		Tcl_DStringAppend(inScanParam->out.dstr, &inChar, 1);
	} else {
		Tcl_Obj *	resObj;
		int			err;
		
		gInChars[gInLen++] = inChar;
		resObj = aida_externalCharsToTclObj(gInChars, gInLen, &err);
		
		if (err == TCL_OK) {
			if (resObj != NULL) {
				aida_writeTclObj(resObj, inScanParam);
			} 
			gInLen = 0;
		} else if (err == TCL_CONVERT_MULTIBYTE) {
			// This is OK, expecting more bytes
		} else {
			// This must be TCL_CONVERT_NOSPACE
			aida_abort("failed to convert input data\n");
		} 
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_writeString" --
// 
// Write a string (which is in the input encoding) either to the output
// channel or to the in-memory buffer depending on the aida_parse_t
// settings. When writing to the output channel, the string is converted to
// the output encoding if necessary. The in-memory buffer is always in the
// input encoding.
// 
// ------------------------------------------------------------------------
void aida_writeString(char * inStr, aida_parse_t * inScanParam)
{
	if (inScanParam->to == to_dstr) {
		Tcl_DStringAppend(inScanParam->out.dstr, inStr, -1);
	} else {
		aida_writeTclObj(aida_externalToTclObj(inStr), inScanParam);
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_writeTclObj" --
// 
// Write the contents of a Tcl object to the output channel or to the
// in-memory buffer depending on the aida_parse_t settings. When writing to
// the in-memory buffer, the string is converted back to the input
// encoding because this buffer is going to be used again as an input string. 
// 
// ------------------------------------------------------------------------
void aida_writeTclObj(Tcl_Obj * inObj, aida_parse_t * inScanParam)
{	
	Tcl_DString		ds;
	Tcl_Encoding	enc;

	Tcl_DStringInit(&ds);	
	if (inScanParam->to == to_dstr) {
		// Convert back to the input encoding
		enc = gEncodings->ienc;
	} else {
		enc = inScanParam->enc;
	} 
	aida_TclObjToExternalDString(inObj, enc, &ds);
	aida_writeDString(&ds, inScanParam);
	Tcl_DStringFree(&ds);
}


// ------------------------------------------------------------------------
// 
// "aida_writeDString" --
// 
// Write a string (which is in the input encoding) either to the output
// channel or to the in-memory buffer depending on the aida_parse_t
// settings. When writing to the output channel, the string is converted to
// the output encoding if necessary. The in-memory buffer is always in the
// input encoding.
// 
// ------------------------------------------------------------------------
void aida_writeDString(Tcl_DString * inDStr, aida_parse_t * inScanParam)
{
	if (inScanParam->to == to_dstr) {
		Tcl_DStringAppend(inScanParam->out.dstr, Tcl_DStringValue(inDStr), Tcl_DStringLength(inDStr));
	} else {
		if (!gPrescan) {
			fwrite(Tcl_DStringValue(inDStr), Tcl_DStringLength(inDStr), 1, inScanParam->out.fptr);
		} 
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_writeResult" --
// 
// If result is TCL_OK, write the contents of the Tcl result object to the
// output channel or to the in-memory buffer depending on the aida_parse_t
// settings.
// 
// ------------------------------------------------------------------------
void aida_writeResult(int inCode, aida_parse_t * inScanParam)
{
	if (inCode == TCL_OK) {
		aida_writeTclObj(Tcl_GetObjResult(gInterp), inScanParam);
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_result_to_std" --
// 
// Print out the contents of the Tcl result object to the specified stream
// using the specified output encoding.
// 
// ------------------------------------------------------------------------
void aida_result_to_std(FILE * stream, Tcl_Encoding inEncoding)
{
	int				len = 0;
	Tcl_Obj *		resObj;
	Tcl_DString		ds;
	
	resObj = Tcl_GetObjResult(gInterp);
	Tcl_DStringInit(&ds);	
	Tcl_UtfToExternalDString(inEncoding, Tcl_GetStringFromObj(resObj, &len), -1, &ds);
	if (Tcl_DStringLength(&ds) > 0) {
		fprintf(stream, "%s\n", Tcl_DStringValue(&ds));
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_result_to_console" --
// 
// Print out the contents of the Tcl result object to stdout or stderr
// depending on the result code specified by the inCode argument.
// 
// ------------------------------------------------------------------------
void aida_result_to_console(int inCode)
{	
	if (inCode == TCL_OK) {
		aida_result_to_std(stdout, NULL);
	} else {
		aida_result_to_std(stderr, NULL);
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_assert_result" --
// 
// Check the Tcl result returned by a function and abort if it is not TCL_OK.
// 
// ------------------------------------------------------------------------
void aida_assert_result(int inCode)
{
	if (inCode != TCL_OK) {
		aida_result_to_std(stderr, NULL);
		aida_abort("assert failure\n");
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_print_err" --
// 
// Print formatted strings to the stderr channel.
// 
// ------------------------------------------------------------------------
void aida_print_err(const char * frmt,...)
{
	va_list		ap;

	va_start(ap, frmt);
	vfprintf(stderr, frmt, ap);
	va_end(ap);
}


// ------------------------------------------------------------------------
// 
// "aida_print_out" --
// 
// Print formatted strings to the standard output channel.
// 
// ------------------------------------------------------------------------
void aida_print_out(const char * frmt,...)
{
	va_list		ap;

	va_start(ap, frmt);
	vfprintf(stdout, frmt, ap);
	va_end(ap);
}


// ------------------------------------------------------------------------
// 
// "aida_verbose" --
// 
// Print information to stderr depending on the verbosity level.
// 
// ------------------------------------------------------------------------
void aida_verbose(int inLevel, const char * frmt,...)
{
	if (inLevel <= gVerbosity) {
		va_list		ap;

		va_start(ap, frmt);
		vfprintf(stderr, frmt, ap);
		va_end(ap);
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_warn" --
// 
// Print information to stderr if verbosity level is >= 1. This is synonym
// of aida_verbose(1,...).
// 
// ------------------------------------------------------------------------
void aida_warn(const char * frmt,...)
{
	if (gVerbosity >= 1) {
		va_list		ap;

		va_start(ap, frmt);
		vfprintf(stderr, frmt, ap);
		va_end(ap);
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_transferContents" --
// 
// BUFSIZ is defined in stdio.h.
// 
// ------------------------------------------------------------------------
int aida_transferContents(FILE * inSrc, FILE * inDest) {
	int		result = TCL_OK;
	char	str[BUFSIZ];

	while (fgets(str, BUFSIZ, inSrc) != NULL) {
		if (fputs(str, inDest) == EOF) {
			//perror("fputs");
			result = TCL_ERROR;
			break;
		} 
	}
	if (ferror(inSrc)) {
		result = TCL_ERROR;
	} 
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_print_array" --
// 
// Print a comma-separated array of strings.
// 
// ------------------------------------------------------------------------
void
aida_print_array(FILE * stream, const char * inTable[], bool inLF)
{
	int		i = 0, count = 0;
	
	while (inTable[i++] != NULL) {
		count++;
	}
	
	for (i = 0; i < count; i++) {
		fprintf(stream, "%s", inTable[i]);
		if (count > 0 && i < count-1) {
			fprintf(stream, ", ");
		} else if (inLF) {
			fprintf(stream, "\n");
		}
	}
}



