// ===========================================================================
// File: "aidaInterp.c"
//                        Created: 2010-08-30 08:47:26
//              Last modification: 2013-12-09 16:11:15
// Author: Bernard Desgraupes
// e-mail: <bdesgraupes@users.sourceforge.net>
// (c) Copyright: Bernard Desgraupes 2010-2013
// All rights reserved.
// ===========================================================================

#include "aidaMain.h"

#define MIN_TCL_MAJOR   8
#define MIN_TCL_MINOR   4


static bool		sTclInitialized = false;


// ------------------------------------------------------------------------
// 
// "aida_initTcl" --
// 
// Initialize the Tcl interpreter. TCL_MAJOR_VERSION and TCL_MINOR_VERSION
// are defined in tcl.h.
// 
// ------------------------------------------------------------------------

int aida_initTcl(const char * inName)
{
	if (sTclInitialized) {
		return TCL_OK;
	} 
	
	// Require at least version 8.4 of Tcl
	if (TCL_MAJOR_VERSION < MIN_TCL_MAJOR || 
	    (TCL_MAJOR_VERSION == MIN_TCL_MAJOR && TCL_MINOR_VERSION < MIN_TCL_MINOR)) {
	    aida_print_err("error: %s was compiled with Tcl %d.%d but requires at least Tcl %d.%d\n",
		    gPrgName, TCL_MAJOR_VERSION, TCL_MINOR_VERSION, MIN_TCL_MAJOR, MIN_TCL_MINOR);
	    return TCL_ERROR;
	}
	
	// Setting up Tcl_FindExecutable to ensure that [info nameofexecutable]
	// returns the correct information
	Tcl_FindExecutable(inName);
	
	// Create the interpreter and initialize memory routines
	gInterp = Tcl_CreateInterp();
	if (gInterp == NULL) {
		aida_print_err("failed to create the Tcl interpreter\n");
		return TCL_ERROR;
	} 
    Tcl_InitMemory(gInterp);

	// Create some global variables and core commands
	aida_createTclVars();
	aida_createTclCommands();
	
	// Call an init.tcl script if any
	if (Tcl_Init(gInterp) != TCL_OK) {
		aida_print_err("Tcl_Init failed: ");
		aida_result_to_console(TCL_ERROR);
		return TCL_ERROR;
	}
	
	// Source the configuration files if any
	if (aida_sourceRcFiles("config.tcl") != TCL_OK) {
		return TCL_ERROR;
	} 

	// Initialize the encodings
	if (aida_init_encodings() != TCL_OK) {
		return TCL_ERROR;
	} 
	
	sTclInitialized = true;
	
	return TCL_OK;
}


// ------------------------------------------------------------------------
// 
// "aida_createTclCommands" --
// 
// Declare core commands defined in the Tcl interpreter.
// 
// ------------------------------------------------------------------------
int aida_createTclCommands()
{	
    Tcl_CreateObjCommand(gInterp, "aida::getDepth", aidaTcl_getDepth,
			     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateObjCommand(gInterp, "aida::getDirname", aidaTcl_getDirname,
			     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateObjCommand(gInterp, "aida::splitting", aidaTcl_splitting,
			     (ClientData) NULL, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateObjCommand(gInterp, "aida::inputEncoding", aidaTcl_encoding,
			     (ClientData) 1, (Tcl_CmdDeleteProc *) NULL);
    Tcl_CreateObjCommand(gInterp, "aida::outputEncoding", aidaTcl_encoding,
			     (ClientData) 0, (Tcl_CmdDeleteProc *) NULL);

	return TCL_OK;
}


// ------------------------------------------------------------------------
// 
// "aida_createTclVars" --
// 
// Set some variables in the Tcl interpreter.
// 
// NB: it is important that gLibDir be allocated via Tcl_Alloc in order to
// be declared as a linked string. It is defined by aida_initLibDir().
// 
// ------------------------------------------------------------------------
int aida_createTclVars()
{
	int			result = TCL_OK;
	Tcl_Obj *	objv;
	
	Tcl_SetVar(gInterp, "aida_version", gVersionStr, TCL_GLOBAL_ONLY);
	Tcl_SetVar(gInterp, "aida_library", gLibDir, TCL_GLOBAL_ONLY);
	Tcl_SetVar2Ex(gInterp, "aida_verbosity", NULL, Tcl_NewIntObj(gVerbosity), TCL_GLOBAL_ONLY);
	Tcl_SetVar2Ex(gInterp, "aida_cwd", NULL, Tcl_FSGetCwd(gInterp), TCL_GLOBAL_ONLY);
	Tcl_SetVar2Ex(gInterp, "aida_unwrap", NULL, Tcl_NewIntObj(gUnwrapText), TCL_GLOBAL_ONLY);
	Tcl_SetVar2Ex(gInterp, "aida_mapping", NULL, Tcl_NewIntObj(gMapChars), TCL_GLOBAL_ONLY);
	Tcl_SetVar2Ex(gInterp, "aida_temp", NULL, Tcl_NewStringObj(P_tmpdir,-1), TCL_GLOBAL_ONLY);

	// Make linked variables
	Tcl_LinkVar(gInterp, "aida_library", (char*)&gLibDir, TCL_LINK_STRING);
	Tcl_LinkVar(gInterp, "aida_verbosity", (char*)&gVerbosity, TCL_LINK_INT);
	Tcl_LinkVar(gInterp, "aida_unwrap", (char*)&gUnwrapText, TCL_LINK_INT);
	Tcl_LinkVar(gInterp, "aida_mapping", (char*)&gMapChars, TCL_LINK_INT);
	
	objv = Tcl_NewStringObj(gLibDir, -1);
	Tcl_SetVar2Ex(gInterp, "aida_path", NULL, Tcl_NewListObj(1, &objv), TCL_GLOBAL_ONLY);

	// Initialize the aida_target and aida_name variables. Their value is
	// adjusted latter by aida_adjustTclVars().
	Tcl_SetVar(gInterp, "aida_target", "", TCL_GLOBAL_ONLY);
	Tcl_SetVar(gInterp, "aida_name", "", TCL_GLOBAL_ONLY);
	Tcl_SetVar(gInterp, "aida_output", "", TCL_GLOBAL_ONLY);

	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_adjustTclVars" --
// 
// Reset some variables in the Tcl interpreter
// 
// ------------------------------------------------------------------------
int aida_adjustTclVars()
{
	Tcl_SetVar(gInterp, "aida_target", gTarget, TCL_GLOBAL_ONLY);
	if (gInName != NULL) {
		Tcl_SetVar2Ex(gInterp, "aida_name", NULL, Tcl_NewStringObj(gInName,-1), TCL_GLOBAL_ONLY);
	} 
	// Store the value of the -output option
	if (gSplitting) {
		if (gSplitFrmt != NULL) {
			Tcl_SetVar2Ex(gInterp, "aida_output", NULL, Tcl_NewStringObj(gSplitFrmt,-1), TCL_GLOBAL_ONLY);
		} 
	} else {
		if (gOutName != NULL) {
			Tcl_SetVar2Ex(gInterp, "aida_output", NULL, Tcl_NewStringObj(gOutName,-1), TCL_GLOBAL_ONLY);
		} 
	} 
	return TCL_OK;
}


// ------------------------------------------------------------------------
// 
// "aida_installCoreLib" --
// 
// Source the core library procs.
// 
// ------------------------------------------------------------------------
int aida_installCoreLib()
{
	int 	result = TCL_OK;

	aida_verbose(2, "installing core library procs\n");
	aida_verbose(2, "library dir: %s\n", gLibDir);

	// Source the core Aida Tcl commands
	result = aida_sourceLibFile("core.tcl", NULL);
	AIDA_CHECK(result, done, "");
	
done:	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_installTclCode" --
// 
// Source the necessary code in this order:
//     base settings and procs
//     target settings and procs
//     configuration files
// 
// ------------------------------------------------------------------------
int aida_installTclCode(char * inTarget)
{
	int 	result = TCL_OK;
	
	// Source the global code
	result = aida_sourceLibFile("default.tcl", "base");
	AIDA_CHECK(result, done, "");
	result = aida_sourceLibFile("convert.tcl", "base");
	AIDA_CHECK(result, done, "");
	
	if (inTarget != NULL) {
		// Source the target specific code
		result = aida_sourceLibFile("default.tcl", inTarget);
		AIDA_CHECK(result, done, "");
		result = aida_sourceLibFile("convert.tcl", inTarget);
		AIDA_CHECK(result, done, "");		
	}
	
	// Source local or user domain files if any
	result = aida_sourceRcFiles("default.tcl");
	AIDA_CHECK(result, done, "");
	result = aida_sourceRcFiles("convert.tcl");
	AIDA_CHECK(result, done, "");
	
	result = aida_installCharsMap();
	AIDA_CHECK(result, done, "");
	
done:	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_sourceRcFiles" --
// 
// Source the configuration files if they exist. They are located in the
// following directories:
// 		site configuration		$AIDA_SITE_CONFIG
// 		user configuration		~/.aidarc/
// The site-wide dir is visited first, then the per-user dir.
// For the site-wide dir to be taken into account, one must define the
// AIDA_SITE_CONFIG environment variable. For instance, invoke the aida
// command like this:
// 		AIDA_SITE_CONFIG=/path/to/dir aida subcmd...
// or
// 		export AIDA_SITE_CONFIG=/path/to/dir 
// 		aida subcmd...
// 
// ------------------------------------------------------------------------
int aida_sourceRcFiles(char * inName)
{
	int			result = TCL_OK;
	char		*siteDir;
	bool		ignErr;
	Tcl_Obj 	*listObj, *pathObj, *homeObj;

	aida_verbose(2, "trying to source rc file '%s'...\n",inName);
				
	// Site configuration
	siteDir = getenv("AIDA_SITE_CONFIG");
	if (siteDir != NULL) {
		aida_verbose(2, "looking in site configuration dir '%s'\n", siteDir);
		listObj = Tcl_NewListObj(0, NULL);
		Tcl_IncrRefCount(listObj);
		Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(siteDir, -1));
		Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(inName, -1));
		pathObj = Tcl_FSJoinPath(listObj, 2);
		result = aida_evalFile(pathObj, &ignErr);
		Tcl_DecrRefCount(listObj);
	}
	
	// User configuration
	if (result == TCL_OK) {
		homeObj = Tcl_FSGetNormalizedPath(gInterp, Tcl_NewStringObj("~", -1));
		if (homeObj != NULL) {
			aida_verbose(2, "looking in user's configuration dir '~/.aidarc'\n");
			listObj = Tcl_NewListObj(0, NULL);
			Tcl_IncrRefCount(listObj);
			Tcl_ListObjAppendElement(gInterp, listObj, homeObj);
			Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(".aidarc", -1));
			Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(inName, -1));
			pathObj = Tcl_FSJoinPath(listObj, 3);
			result = aida_evalFile(pathObj, &ignErr);
			Tcl_DecrRefCount(listObj);
		} 
	} 
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_installCharsMap" --
// 
// Convert the mappings defined in the aida_map array to a C table of
// pointers on the replacement Tcl string objects. 
// 
// ------------------------------------------------------------------------
int aida_installCharsMap()
{
	int			i, result = TCL_OK;
    Tcl_UniChar uch;
		
	for (i = 0; i < 256; i++) {
		uch = i;
		gMap[i] = Tcl_ObjGetVar2(gInterp, Tcl_NewStringObj("aida_map", -1), 
								 Tcl_NewUnicodeObj(&uch, 1), TCL_GLOBAL_ONLY);
		if (gMap[i] != NULL) {
			Tcl_IncrRefCount(gMap[i]);
		} 
	}
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_evalFile" --
// 
// ------------------------------------------------------------------------
int aida_evalFile(Tcl_Obj * inFile, bool * openError)
{
	int			result = TCL_OK;
	char *		fname = Tcl_GetString(inFile);
	
	Tcl_IncrRefCount(inFile);
	aida_verbose(2, "looking for file '%s'\n", fname);
	if (Tcl_FSAccess(inFile, F_OK) == 0) {
		*openError = false;
		aida_verbose(2, "found file '%s'\n", fname);
		result = Tcl_FSEvalFile(gInterp, inFile);
		if (result != TCL_OK) {
			aida_print_err("error sourcing file '%s': ", fname);
			aida_result_to_console(TCL_ERROR);
		} else {
			aida_verbose(3, "sourced file '%s'\n", fname);
		}	
	} else {
		aida_verbose(3, "can't find file '%s'\n", fname);
		*openError = true;
	}
	Tcl_DecrRefCount(inFile);
		
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_sourceLibFile" --
// 
// Source the Tcl code associated with the current target. 
// 
// ------------------------------------------------------------------------
int aida_sourceLibFile(char * inName, char * inTarget)
{
	int			i, result = TCL_OK, objc;
	Tcl_Obj *	objv[3];
	Tcl_Obj *	pathObj;
	Tcl_Obj *	listObj;
	
	if (inTarget != NULL) {
		aida_verbose(2, "sourcing library file '%s/%s'\n", inTarget, inName);
	} else {
		aida_verbose(2, "sourcing library file '%s'\n", inName);
	} 
	
	listObj = Tcl_NewListObj(0, NULL);
	Tcl_IncrRefCount(listObj);
		
	// Build the path
	objc = 2;
	if (inTarget != NULL) {
		// Ask the library to locate the target dir
		objv[0] = Tcl_NewStringObj("aida::findTargetDir", -1);
		objv[1] = Tcl_NewStringObj(inTarget, -1);
		
		for (i = 0; i < objc; i++) Tcl_IncrRefCount(objv[i]);
		result = Tcl_EvalObjv(gInterp, objc, objv, TCL_EVAL_GLOBAL);
		for (i = 0; i < objc; i++) Tcl_DecrRefCount(objv[i]);
		aida_assert_result(result);
		Tcl_ListObjAppendElement(gInterp, listObj, Tcl_GetObjResult(gInterp));
	} else {
		// Look in the library
		Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(gLibDir, -1));
	} 
	// Append the name of the file
	Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(inName, -1));
    pathObj = Tcl_FSJoinPath(listObj, objc);
	
	// Build the source command
	Tcl_IncrRefCount(pathObj);
	Tcl_ResetResult(gInterp);
	result = Tcl_FSEvalFile(gInterp, pathObj);
	Tcl_DecrRefCount(pathObj);
	Tcl_DecrRefCount(listObj);
	
	if (result != TCL_OK) {
		if (inTarget != NULL) {
			aida_print_err("error sourcing file '%s' for target %s: ", inName, inTarget);
		} else {
			aida_print_err("error sourcing file '%s': ", inName);
		} 
		aida_result_to_console(TCL_ERROR);
	} 
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_setHeaderValue" --
// 
// Store the key/value pairs in the aida_head array. 
// 
// ------------------------------------------------------------------------
int aida_setHeaderValue(char * inKey, char * inValue)
{
	int			i, result = TCL_OK, objc;
	Tcl_Obj *	valObj;
	Tcl_Obj *	objv[3];
	
	if (gSplitting && !gPrescan) {
		// The header has already been parsed during the prescan phase
		result = TCL_OK;
	} else {
		valObj = aida_externalToTclObj(inValue);
		Tcl_IncrRefCount(valObj);
		
		objv[0] = Tcl_NewStringObj("aida::registerParam", -1);
		objv[1] = aida_externalToTclObj(inKey);
		objv[2] = valObj;
		objc = 3;
		
		for (i = 0; i < objc; i++) Tcl_IncrRefCount(objv[i]);
		result = Tcl_EvalObjv(gInterp, objc, objv, TCL_EVAL_GLOBAL);
		for (i = 0; i < objc; i++) Tcl_DecrRefCount(objv[i]);
		if (result != TCL_OK) {
			aida_print_err("error: can't set '%s' header parameter\n", inKey);
			result = TCL_ERROR;
		} 
		Tcl_DecrRefCount(valObj);
		aida_assert_result(result);
	}
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_executeHook" --
// 
// ------------------------------------------------------------------------
int 
aida_executeHook(char * inHook, int inArgc, ...)
{
	int			result = TCL_OK, cnt = inArgc;
	Tcl_Obj 	*listObj;
	va_list		ap;
	Tcl_Obj *	vp;
	
	if (!gPrescan) {
		aida_verbose(3, "executing hook [%s]\n", inHook);
		listObj = Tcl_NewListObj(0, NULL);
		Tcl_IncrRefCount(listObj);
		Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(inHook, -1));
		
		// The target is always the first argument
		Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(gTarget, -1));
		
		// Append any additional arguments specified as Tcl_Obj*
		va_start(ap, inArgc);
		for( ; cnt > 0; cnt-- ) {
			vp = va_arg(ap, Tcl_Obj *);
			Tcl_ListObjAppendElement(gInterp, listObj, vp);
		} 
		va_end(ap);
		
		// Evaluate the built object
		result = Tcl_EvalObjEx(gInterp, listObj, TCL_EVAL_GLOBAL);
		Tcl_DecrRefCount(listObj);
		
		if (result != TCL_OK) {
			aida_result_to_console(result);
		} 
	} 
			
	return result;	
}


// ------------------------------------------------------------------------
// 
// "aida_executeProc" --
// 
// Build a callback command and evaluate it in the Tcl interpreter. 
// The target is always passed as the first argument.
// The proc is called as:
//     inProc $target $args
// 'inArgc' is the number of additional arguments. All the additional
// arguments are expected to be Tcl_Obj*.
// 
// ------------------------------------------------------------------------
int 
aida_executeProc(char * inProc, aida_parse_t * inScanParam, int inArgc, ...)
{
	int			result, cnt = inArgc;
	Tcl_Obj 	*listObj;
	va_list		ap;
	Tcl_Obj *	vp;
	
	if (gPrescan) {
		result = TCL_OK;
	} else {
		aida_verbose(3, "executing proc [%s]\n", inProc);
		listObj = Tcl_NewListObj(0, NULL);
		Tcl_IncrRefCount(listObj);
		Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(inProc, -1));
		
		// The target is always the first argument
		Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(gTarget, -1));
		
		// Append any additional arguments specified as Tcl_Obj*
		va_start(ap, inArgc);
		for( ; cnt > 0; cnt-- ) {
			vp = va_arg(ap, Tcl_Obj *);
			Tcl_ListObjAppendElement(gInterp, listObj, vp);
		} 
		va_end(ap);
		
		// Evaluate the built object
		result = Tcl_EvalObjEx(gInterp, listObj, TCL_EVAL_GLOBAL);
		Tcl_DecrRefCount(listObj);

		// Print the result
		aida_writeResult(result, inScanParam);
	}

	return result;	
}


// ------------------------------------------------------------------------
// 
// "aida_executeTextProc" --
// 
// Convenience function to execute a proc taking a single text argument.
// 
// ------------------------------------------------------------------------
int 
aida_executeTextProc(char * inProc, aida_parse_t * inScanParam, Tcl_DString * inDst)
{
	int			result;
	Tcl_Obj 	*txtObj;
	
	if (gPrescan) {
		result = TCL_OK;
	} else {
		txtObj = aida_externalToTclObj(Tcl_DStringValue(inDst));
		Tcl_IncrRefCount(txtObj);
		result = aida_executeProc(inProc, inScanParam, 1, txtObj);
		Tcl_DecrRefCount(txtObj);
	}
	
	// Release the Tcl_DString
	Tcl_DStringFree(inDst);
	
	return result;
}

								  
// ------------------------------------------------------------------------
// 
// "aida_substString" --
// 
// Make Tcl substitution on the given string.
// Flags are composed of the following bits:
//     TCL_SUBST_COMMANDS, 
//     TCL_SUBST_VARIABLES, 
//     TCL_SUBST_BACKSLASHES
// or TCL_SUBST_ALL.
// ------------------------------------------------------------------------
Tcl_Obj * 
aida_substString(char * inStr, int inFlags)
{
	int			result;
	Tcl_Obj 	*strObj, *substObj;
	
	if (gPrescan) {
		result = TCL_OK;
	} else {
		strObj = aida_externalToTclObj(inStr);
		Tcl_IncrRefCount(strObj);
		substObj = Tcl_SubstObj(gInterp, strObj, inFlags);
		Tcl_DecrRefCount(strObj);
	}
		
	return substObj;
}

								  
// ------------------------------------------------------------------------
// 
// "aida_evalIfCondition" --
// 
// Evaluate the condition of an if tag.
// 
// ------------------------------------------------------------------------
int 
aida_evalIfCondition(Tcl_DString * inDst)
{
	int			i, objc, cond, result;
	Tcl_Obj *	objv[2];

	if (Tcl_DStringLength(inDst) == 0) {
		aida_abort("empty if condition");
	} 
	
	aida_verbose(3, "evaluating condition '%s'\n", Tcl_DStringValue(inDst));
	objv[0] = Tcl_NewStringObj("expr", -1);
	objv[1] = aida_externalToTclObj(Tcl_DStringValue(inDst));
	objc = 2;
								  
	for (i = 0; i < objc; i++) Tcl_IncrRefCount(objv[i]);
	result = Tcl_EvalObjv(gInterp, objc, objv, TCL_EVAL_GLOBAL);
	for (i = 0; i < objc; i++) Tcl_DecrRefCount(objv[i]);
	
	// Release the Tcl_DString
	Tcl_DStringFree(inDst);
	
	// Check there was no error
	aida_assert_result(result);
	
	result = Tcl_GetBooleanFromObj(gInterp, Tcl_GetObjResult(gInterp), &cond);
	aida_assert_result(result);
	if (gIfDepth < 0) {
		aida_abort("negative if depth (too many closing 'if' tags?)");
	} 
	gIfConds[gIfDepth] = cond;
	aida_verbose(3, "condition evaluates to '%s'\n", cond? "true":"false");
	
	return result;
}
								  

// ------------------------------------------------------------------------
// 
// "aida_procToFile" --
// 
// ------------------------------------------------------------------------
int 
aida_procToFile(Tcl_Obj * inProc, FILE * inDest, int inArgc, ...)
{
	int				result = TCL_OK, cnt = inArgc;
	Tcl_Obj 		*listObj, *vp;
	Tcl_DString		ds;
	va_list			ap;
	
	aida_verbose(3, "processing [%s]\n", Tcl_GetString(inProc));
	listObj = Tcl_NewListObj(0, NULL);
	Tcl_IncrRefCount(listObj);
	Tcl_ListObjAppendElement(gInterp, listObj, inProc);
	
	// The target is always the first argument
	Tcl_ListObjAppendElement(gInterp, listObj, Tcl_NewStringObj(gTarget, -1));
	
	// Append any additional arguments specified as Tcl_Obj*
	va_start(ap, inArgc);
	for( ; cnt > 0; cnt-- ) {
		vp = va_arg(ap, Tcl_Obj *);
		Tcl_ListObjAppendElement(gInterp, listObj, vp);
	} 
	va_end(ap);
	
	// Evaluate the built object
	result = Tcl_EvalObjEx(gInterp, listObj, TCL_EVAL_GLOBAL);
	Tcl_DecrRefCount(listObj);

	// Print the result
	if (result == TCL_OK) {
		Tcl_DStringInit(&ds);	
		aida_TclObjToExternalDString(Tcl_GetObjResult(gInterp), gEncodings->oenc, &ds);
		fwrite(Tcl_DStringValue(&ds), Tcl_DStringLength(&ds), 1, inDest);
		Tcl_DStringFree(&ds);
	} else {
		aida_result_to_std(stderr, NULL);
		fprintf(stderr, "\n");
	} 
	
	return result;	
}

