// ===========================================================================
// File: "aidaCommands.c"
//                        Created: 2010-08-09 21:56:18
//              Last modification: 2015-06-27 06:33:39
// Author: Bernard Desgraupes
// e-mail: <bdesgraupes@users.sourceforge.net>
// Copyright (c) 2010-2015 Bernard Desgraupes
// All rights reserved.
// ===========================================================================

#include "aidaMain.h"
#include "aidaConstants.h"


// ------------------------------------------------------------------------
// 
// "aidaCmd_convert" --
// 
// Implement the 'aida convert' subcommand.
// 
// ------------------------------------------------------------------------
int aidaCmd_convert(int argc, char *argv[])
{
	int			i, index, result = TCL_ERROR;
	bool		failed = false, finished = false;
	Tcl_Obj *	optObj;
	
	if (argc < 2) {
		aida_wrongNumArgs("convert");
	} 

	// Parse the options
	i = 2;
	while (i < argc && !failed && !finished) {
		optObj = Tcl_NewStringObj(argv[i], -1);
		
		if (Tcl_GetIndexFromObj(gInterp, optObj, gAidaConvertOptions, "option", 0, &index) != TCL_OK) {
			if (i < argc - 1) {
				aida_result_to_console(TCL_ERROR);
				failed = true;
			} 
			break;
		} else {
			if (index == CONVERT_OPTION_DASH) {
				// Get out of the while loop
				finished = true;
			} else {
				if (i > argc-2) {
					aida_abort("missing value for option '%s'\n", argv[i]);
				} 
				i++;
				switch (index) {
					case CONVERT_OPTION_COLLAPSE:
					gCollapse = atoi(argv[i++]);
					if (gCollapse <= 0) {
						aida_print_err("error: -collapse value must be a positive integer.\n");
						failed = true;
					} 
					break;
					
					case CONVERT_OPTION_FROM:
					if (aida_encoding_name(argv[i++], true) != TCL_OK) {
						failed = true;
					} 
					break;
					
					case CONVERT_OPTION_EVAL:
					if (Tcl_EvalEx(gInterp,argv[i++],-1,TCL_EVAL_GLOBAL) != TCL_OK) {
						failed = true;
					} 
					break;
					
					case CONVERT_OPTION_OUTPUT:
					gOutName = strdup(argv[i++]);
					break;
					
					case CONVERT_OPTION_PREFIX:
					gPrefixFile = strdup(argv[i++]);
					break;
					
					case CONVERT_OPTION_TARGET:
					gTarget = strdup(argv[i++]);
					break;
					
					case CONVERT_OPTION_TO:
					if (aida_encoding_name(argv[i++], false) != TCL_OK) {
						failed = true;
					} 
					break;
					
				}
			} 
		}
	}

	if (!failed) {
		result = aida_process(argc, argv, i);
		if (result == TCL_OK) {
			result = aida_doConvert();
		} 
	} 
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aidaCmd_help" --
// 
// Implement the 'aida help' subcommand.
// 
// ------------------------------------------------------------------------
int aidaCmd_help(int argc, char *argv[])
{
	int			index, result = TCL_OK;
	
	if (argc < 2 || argc > 3) {
		aida_wrongNumArgs("help");
	} 

	if (argc == 2) {
		aida_usage(gPrgName);
	} else {
		Tcl_Obj *		subCmd;
		subCmd = Tcl_NewStringObj(argv[2], -1);
		
		if (Tcl_GetIndexFromObj(gInterp, subCmd, gAidaHelpKeywords, "keyword", 0, &index) == TCL_OK) {
			switch (index) {
				case AIDA_HELP_HEADER:
				result = aida_installCoreLib();
				AIDA_CHECK(result, done, "error: can't install core library\n")

				result = aida_installTclCode(NULL);
				AIDA_CHECK(result, done, "");
				
				result = aida_help_header();
				break;
				
				case AIDA_HELP_SYNTAX:
#				include "aidaRefSyntax.i"
				break;
				
			}

		} else {
			result = Tcl_GetIndexFromObj(gInterp, subCmd, gAidaSubcommands, "argument", 0, &index);
			if (result != TCL_OK) {
				aida_result_to_console(result);
				aida_print_err("can also be: ");
				aida_print_array(stderr, gAidaHelpKeywords, true);
			} else {
				switch (index) {
					case AIDA_CMD_CONVERT:
					result = aida_convert_usage();
					break;
					
					case AIDA_CMD_HELP:
					result = aida_help_usage();
					break;
					
					case AIDA_CMD_INFO:
					result = aida_info_usage();
					break;
					
					case AIDA_CMD_SPLIT:
					result = aida_split_usage();
					break;
					
				}
			}
		}
	} 

done:          
	return result;
}


// ------------------------------------------------------------------------
// 
// "aidaCmd_info" --
// 
// Implement the 'aida info' subcommand.
// 
// Some infos do not need to load the library:
//     AIDA_INFO_LIBRARY
//     AIDA_INFO_FROM
//     AIDA_INFO_TEMP
//     AIDA_INFO_TO
//     AIDA_INFO_VERSION
// 
// ------------------------------------------------------------------------
int aidaCmd_info(int argc, char *argv[])
{
	int			index, result = TCL_OK;
	Tcl_Obj *	subCmd;

	if (argc < 3) {
		aida_wrongNumArgs("info");
	} 

	subCmd = Tcl_NewStringObj(argv[2], -1);
	result = Tcl_GetIndexFromObj(gInterp, subCmd, gAidaInfos, "option", 0, &index);
	if (result != TCL_OK) {
		aida_result_to_console(result);
	} else {
		if (index == AIDA_INFO_LIBRARY) {
			aida_print_out("%s\n", gLibDir);
		} else if (index == AIDA_INFO_FROM) {
			aida_print_out("%s\n", Tcl_GetEncodingName(gEncodings->ienc));
		} else if (index == AIDA_INFO_TEMP) {
			aida_print_out("%s\n", P_tmpdir);
		} else if (index == AIDA_INFO_TO) {
			aida_print_out("%s\n", Tcl_GetEncodingName(gEncodings->oenc));
		} else if (index == AIDA_INFO_VERSION) {
			aida_print_out("%s\n", gVersionStr);
		} else {
			result = aida_installCoreLib();
			AIDA_CHECK(result, done, "");

			result = aida_installTclCode(NULL);
			AIDA_CHECK(result, done, "");

			switch (index) {
				case AIDA_INFO_ATTRIBUTES:
				if (argc == 4) {
					result = aida_info_attributes(argv[3]);
				} else {
					result = aida_info_attributes(NULL);
				} 
				break;
							
				case AIDA_INFO_CONFIGURATION:
				result = aida_info_configuration();
				break;
							
				case AIDA_INFO_ENCODINGS:
				result = aida_info_encodings();
				break;
				
				case AIDA_INFO_EXTENSIONS:
				if (argc == 4) {
					result = aida_info_extensions(argv[3]);
				} else {
					result = aida_info_extensions(NULL);
				} 
				break;
				
				case AIDA_INFO_LICENSE:
				result = aida_info_license();
				break;
				
				case AIDA_INFO_MAPPING:
				if (argc == 4) {
					result = aida_info_mapping(argv[3]);
				} else {
					result = aida_info_mapping(NULL);
				} 
				break;
				
				case AIDA_INFO_PARAMETERS:
				if (argc == 4) {
					result = aida_info_parameters(argv[3]);
				} else {
					result = aida_info_parameters(NULL);
				} 
				break;
							
				case AIDA_INFO_PATH:
				result = aida_info_path();
				break;
							
				case AIDA_INFO_SYSENC:
				result = aida_info_sysenc();
				break;
							
				case AIDA_INFO_TARGETS:
				result = aida_info_targets();
				break;
							
			}
		} 
	}

done:
	return result;
}
	

// ------------------------------------------------------------------------
// 
// "aidaCmd_split" --
// 
// Implement the 'aida split' subcommand.
// 
// ------------------------------------------------------------------------
int aidaCmd_split(int argc, char *argv[])
{
	int			i, index, result = TCL_ERROR;
	bool		failed = false, finished = false;
	Tcl_Obj *	optObj;
	
	if (argc < 2) {
		aida_wrongNumArgs("split");
	} 

	gSplitting = true;
	
	// Parse the options
	i = 2;
	while (i < argc && !failed && !finished) {
		optObj = Tcl_NewStringObj(argv[i], -1);
		
		if (Tcl_GetIndexFromObj(gInterp, optObj, gAidaSplitOptions, "option", 0, &index) != TCL_OK) {
			if (i < argc - 1) {
				aida_result_to_console(TCL_ERROR);
				failed = true;
			} 
			break;
		} else {
			if (index == SPLIT_OPTION_DASH) {
				// Get out of the while loop
				finished = true;
			} else {
				if (i > argc-2) {
					aida_abort("missing value for option '%s'\n", argv[i]);
				} 
				i++;
				switch (index) {
					case SPLIT_OPTION_COLLAPSE:
					gCollapse = atoi(argv[i++]);
					if (gCollapse <= 0) {
						aida_print_err("error: -collapse value must be a positive integer.\n");
						failed = true;
					} 
					break;
					
					case SPLIT_OPTION_EVAL:
					if (Tcl_EvalEx(gInterp,argv[i++],-1,TCL_EVAL_GLOBAL) != TCL_OK) {
						failed = true;
					} 
					break;
					
					case SPLIT_OPTION_FROM:
					if (aida_encoding_name(argv[i++], true) != TCL_OK) {
						failed = true;
					} 
					break;
					
					case SPLIT_OPTION_LEVEL:
					gSplitLevel = atoi(argv[i++]);
					break;
					
					case SPLIT_OPTION_OUTPUT:
					gSplitFrmt = strdup(argv[i++]);
					break;
					
					case SPLIT_OPTION_PREFIX:
					gPrefixFile = strdup(argv[i++]);
					break;
					
					case SPLIT_OPTION_TARGET:
					gTarget = strdup(argv[i++]);
					break;
					
					case SPLIT_OPTION_TO:
					if (aida_encoding_name(argv[i++], false) != TCL_OK) {
						failed = true;
					} 
					break;
					
				}
			}
		}
	}
	
	if (!failed) {
		result = aida_process(argc, argv, i);
		if (result == TCL_OK) {
			aida_checkSplitFormat();
			result = aida_doSplit();
		} 
	} 
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_process" --
// 
// Process the data.
// 
// ------------------------------------------------------------------------
int aida_process(int argc, char *argv[], int index)
{
	int			result = TCL_OK;
	bool		ignErr;	
	char *		encodedFile;

	result = aida_installCoreLib();
	AIDA_CHECK(result, done, "error: can't install core library\n")

	// Install the encodings
	result = aida_install_encodings();
	aida_assert_result(result);
	
	// Allocate top gCurrInclude
	aida_newInclude();
	
	if (index == argc) {
		// Reading from stdin
		gInputFile = stdin;
		gCurrInclude->std = true;
		gCurrInclude->fptr = gInputFile;
	} else if (index == argc-1) {
		// Last argument is the input file
		gInName = strdup(argv[argc-1]);
		aida_verbose(2, "opening input file '%s'\n", gInName);
		
		// Construct new base
		aida_setIncludeBase(gInName);
		
		// Transcode the input file if it is in a multibyte encoding
		result = aida_transcodeInput(gInName, &(gCurrInclude->tmp), &encodedFile);
		AIDA_CHECK(result, done, "error: can't transcode file\n")

		gCurrInclude->name = encodedFile;
		result = aida_open(encodedFile, "r", &gInputFile);
		AIDA_CHECK(result, done, "error: can't open input file\n")

		gCurrInclude->fptr = gInputFile;
		
	} else {
		result = TCL_ERROR;
		AIDA_CHECK(result, done, "error: too many arguments after last option\n")
	} 
	
	result = aida_checkRequired();
	AIDA_CHECK(result, done, "error: can't get required parameters\n")
	
	// Read the target source files
	result = aida_installTclCode(gTarget);
	AIDA_CHECK(result, done, "error: can't install target library code\n")
	
	if (gPrefixFile != NULL) {
		result = aida_evalFile(Tcl_NewStringObj(gPrefixFile, -1), &ignErr);
		AIDA_CHECK(result, done, "error: can't source prefix file\n")
	} 

done:
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_checkRequired" --
// 
// Ensure that all the required variables have been set.
// 
// ------------------------------------------------------------------------
int aida_checkRequired()
{
	int			result = TCL_OK;
	char *		name = "html", *str;
	Tcl_Obj *	targObj;
	
	if (gTarget == NULL) {
		// Look for an AIDA_TARGET environment variable
		str = getenv("AIDA_TARGET");
		if (str != NULL) {
			gTarget = strdup(str);
		} else {
			// Look for a default setting via the aida::DefaultTarget global Tcl variable
			targObj = Tcl_GetVar2Ex(gInterp, "::aida::DefaultTarget", NULL, TCL_NAMESPACE_ONLY);
			if (targObj != NULL) {
				name = Tcl_GetString(targObj);
				aida_verbose(3, "using default target %s\n", name);
			} else {
				aida_print_err("can't get default target\n");
				result = TCL_ERROR;
				goto done;
			}
			gTarget = strdup(name);
		}
	} 
	
	if (gSplitting) {
		if (gSplitLevel < 0) {
			gSplitLevel = 1;
		} 
	} 
	
	// Update global Tcl variables
	aida_adjustTclVars();
	
done:
	return result;
}

		
// ------------------------------------------------------------------------
// 
// "aida_doConvert" --
// 
// Execute the conversion process.
// 
// ------------------------------------------------------------------------
int aida_doConvert()
{
	int 	result = TCL_OK;
		
	// Execute the preConvert hook if any
	result = aida_executeHook("aida::preConvertHook", 0);
	AIDA_CHECK(result, done, "error: preConvertHook failed\n")

	// Create a preamble fragment: it will be invoked during the
	// reconstruction of the output
	aida_newFragment(kind_proc, "aida::preambleProc");	
	
	// Open a temp output channel
	gCurrOutput = aida_switchOutput();

	// Start parsing
	result = aida_parse(gInputFile);
	AIDA_CHECK(result, done, "error: parsing failed\n")

	// Create a postamble fragment: it will be invoked during the
	// reconstruction of the output
	aida_newFragment(kind_proc, "aida::postambleProc");	

	// Assemble the fragments
	result = aida_reconstructOutput();
	
	// Execute the postConvert hook if any
	result = aida_executeHook("aida::postConvertHook", 0);
	AIDA_CHECK(result, done, "error: postConvertHook failed\n")
	
done:	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_doSplit" --
// 
// Execute the splitting process.
// 
// ------------------------------------------------------------------------
int aida_doSplit()
{
	int 	result = TCL_OK;
	
	// Run the scanner in prescan mode
	result = aida_preScanData();
	AIDA_CHECK(result, done, "error: prescanning failed\n")

	// Create an initial split fragment
	aida_newFragment(kind_split, NULL);	
	
	// Open a temp output channel
	gCurrOutput = aida_switchOutput();
	
	// Execute the preConvert hook if any
	result = aida_executeHook("aida::preConvertHook", 0);
	AIDA_CHECK(result, done, "error: preConvertHook failed\n")

	// Start parsing
	result = aida_parse(gInputFile);
	AIDA_CHECK(result, done, "error: parsing failed\n")

	// Assemble the fragments
	result = aida_reconstructOutput();
	
	// Execute the postConvert hook if any
	result = aida_executeHook("aida::postConvertHook", 0);
	AIDA_CHECK(result, done, "error: postConvertHook failed\n")
	
done:	
	return result;
}

// ------------------------------------------------------------------------
// 
// "aida_preScanData" --
// 
// Make a first pass with the scanner in prescan mode to collect
// information about anchors and sections.
// 
// ------------------------------------------------------------------------
int aida_preScanData()
{
	int 	result = TCL_OK;
	
	aida_verbose(3, "start prescanning the data\n");

	// Initialize the split num
	gSplitNum++;
	
	// Run the scanner
	gPrescan = true;
	result = aida_parse(gInputFile);
	gPrescan = false;	
	
	// Reset the file position indicator
	fseek(gInputFile, 0, SEEK_SET);
	
	// Just in case, no fragment should have been created during the
	// prescan phase
	aida_free_fragments();
	gFragmentHead = NULL;
	
	// Reset globals
	gSplitNum = 0;
	gIfDepth = 0;
	gIfLevel = 0;

	aida_verbose(3, "end of prescanning\n");

	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_reconstructOutput" --
// 
// After execution of the conversion, collect and assemble the results from
// the fragments.
// 
// ------------------------------------------------------------------------
int aida_reconstructOutput()
{
	int 			result = TCL_OK, splitIndex = 0;
	FILE			*tmpin, *tmpout = NULL;
	char *			name = NULL;
	aida_file_t *	aft;
	aida_frag_t		*curr = gFragmentHead;
	
	// Allocate an aida_file_t struct
	aft = (aida_file_t*)malloc(sizeof(aida_file_t));
	if (aft == NULL) {
		aida_abort("can't allocate memory for new temporary file");
	} 
	
	aida_verbose(3, "reconstructing output\n");
	if (!gSplitting) {
		if (gOutName != NULL) {
			tmpout = aida_newTempFile(aft, ".");
			name = aft->name;
		} else {
			tmpout = stdout;
		} 	
	} 
	
	if (gFragmentHead != NULL) {
		// Close the last opened file fragment
		aida_closeFragmentFile(gFragmentCurr);
		
		curr = gFragmentHead;
		do {
			switch (curr->kind) {
				case kind_file:
				// Open the file 
				result = aida_open(curr->u.file.name, "r", &tmpin);
				AIDA_CHECK(result, done, "error: reopen fragment failed\n")
				
				// Copy its contents to tmpout 
				aida_verbose(3, "transfer contents from '%s' to '%s'\n", curr->u.file.name, name? name:"stdout");	
				result = aida_transferContents(tmpin, tmpout);
				AIDA_CHECK(result, done, "error: transfer of contents failed\n")
				fclose(tmpin);
				break;
				
				case kind_proc:
				// Execute the proc and write the result in tmpout
				result = aida_procToFile(curr->u.proc, tmpout, 0);
				AIDA_CHECK(result, done, "error: fragment proc failed\n")
				break;
				
				case kind_split:
				// Open a new temporary file for output. This updates the
				// aft struct.
				splitIndex = curr->u.split.index;
				tmpout = aida_nextSplitFile(splitIndex, aft);
				name = aft->name;
				break;
				
			}
			curr = curr->next;
		} while (curr != NULL);
	} 
	
	// Rename the temporary file to its final destination
	if (gSplitting) {
		aida_procToFile(Tcl_NewStringObj("aida::postambleProc", -1), tmpout, 0);
		result = aida_renameWithNum(name, splitIndex);
	} else {
		if (gOutName != NULL) {
			if (strlen(gOutName) == 0) {
				// If an -output option was specified but the value is an
				// empty string, build a default name
				gOutName = aida_buildOutFileName();
			} 
			result = aida_renameFile(name, gOutName);
		} 
	} 
	
done:	
	if (tmpout != NULL && tmpout != stdout) {
		fclose(tmpout);
	} 
	free(aft);
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_openInclude" --
// 
// Open a new input/include file.
// 
// ------------------------------------------------------------------------
int aida_openInclude(Tcl_DString * inDst, aida_incl_t ** outIncl)
{
	int				result = TCL_OK;
	aida_incl_t *	ait = NULL;
	Tcl_Obj			*pathObj, *substObj;
	char *			encodedFile;
	Tcl_DString 	*txtds;

	
	// Allocate an aida_incl_t struct
	ait = aida_newInclude();
	
	// Parse the string and construct the new base
	txtds = aida_parseDString(inDst, START_TR_FRAG);
	pathObj = aida_setIncludeBase(Tcl_DStringValue(txtds));

	// Transcode the include file if necessary
	result = aida_transcodeInput(Tcl_GetString(pathObj), &(ait->tmp), &encodedFile);
	aida_assert_result(result);
	ait->name = encodedFile;
	
	// Open the file
	result = aida_open(encodedFile, "r", &(ait->fptr));

	Tcl_DecrRefCount(pathObj);

	*outIncl = ait;
	
	return result;
}


// ------------------------------------------------------------------------
// 
// "aida_closeInclude" --
// 
// Close the input/include file and deallocate the corresponding struct.
// 
// ------------------------------------------------------------------------
void aida_closeInclude(aida_incl_t * inData)
{
	if (inData != NULL) {
		if (inData->fptr != NULL) {
			fclose(inData->fptr);
		} 
		if (!gDontUnlink && inData->tmp == true && inData->name != NULL && inData->std == false) {
			aida_verbose(3, "unlink '%s'\n", inData->name);
			unlink(inData->name);
		} 
		if (inData->name != NULL) {
			free(inData->name);
		} 
		Tcl_DecrRefCount(inData->base);
		gCurrInclude = inData->caller;
		free(inData);
	} 
}


// ------------------------------------------------------------------------
// 
// "aida_newInclude" --
// 
// Allocate a new input/include struct.
// 
// ------------------------------------------------------------------------
aida_incl_t * aida_newInclude()
{
	aida_incl_t *	ait = NULL;

	// Allocate an aida_incl_t struct
	ait = (aida_incl_t*)malloc(sizeof(aida_incl_t));
	if (ait == NULL) {
		aida_abort("can't allocate memory for new include");
	} 
	memset(ait, 0, sizeof(aida_incl_t));

	ait->caller = gCurrInclude;
	gCurrInclude = ait;

	return ait;
}

