/*
SRSearchField.m

Author: Makoto Kinoshita

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "SRSearchField.h"

// Constants
static int  _SRSearcFieldTrackingWidth = 4;

@implementation SRSearchField

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

+ (Class)cellClass
{
    return [SRSearchFieldCell class];
}

- (id)initWithFrame:(NSRect)frame
{
    self = [super initWithFrame:frame];
    if (!self) {
        return nil;
    }
    
    // Initialize instance variables
    _isFlexible = NO;
    _minWidth = 80;
    _maxWidth = 480;
    _leftTrackingRectTag = -1;
    _rightTrackingRectTag = -1;
    
    return self;
}

//--------------------------------------------------------------//
#pragma mark -- Width management --
//--------------------------------------------------------------//

- (BOOL)isFlexible
{
    return _isFlexible;
}

- (void)setFlexible:(BOOL)flag
{
    _isFlexible = flag;
}

- (int)minWidth
{
    return _minWidth;
}

- (void)setMinWidth:(int)minWidth
{
    _minWidth = minWidth;
}

- (int)maxWidth
{
    return _maxWidth;
}

- (void)setMaxWidth:(int)maxWidth
{
    _maxWidth = maxWidth;
}

//--------------------------------------------------------------//
#pragma mark -- Tracking area --
//--------------------------------------------------------------//

- (NSRect)leftTrackingArea
{
    NSRect  frame;
    frame = [self frame];
    return NSMakeRect(
            0, 0, _SRSearcFieldTrackingWidth, frame.size.height);
}

- (NSRect)rightTrackingArea
{
    NSRect  frame;
    frame = [self frame];
    return NSMakeRect(
            frame.size.width - _SRSearcFieldTrackingWidth, 0, _SRSearcFieldTrackingWidth, frame.size.height);
}

- (void)updateTrackingArea
{
    if (!_isFlexible) {
        return;
    }
    
    // Remove old tracking rect
    if (_leftTrackingRectTag > 0) {
        [self removeTrackingRect:_leftTrackingRectTag];
    }
    if (_rightTrackingRectTag > 0) {
        [self removeTrackingRect:_rightTrackingRectTag];
    }
    
    // Add tracking rect
    _leftTrackingRectTag = [self addTrackingRect:[self leftTrackingArea] 
            owner:self 
            userData:&_leftTrackingRectTag 
            assumeInside:NO];
    _rightTrackingRectTag = [self addTrackingRect:[self rightTrackingArea] 
            owner:self 
            userData:&_rightTrackingRectTag 
            assumeInside:NO];
}

//--------------------------------------------------------------//
#pragma mark -- Dragging --
//--------------------------------------------------------------//

- (void)resetCursorRects
{
    if (!_isFlexible) {
        return;
    }
    
    // Discard cursor rects
    [self discardCursorRects];
    
    // Add cursor rects
    [self addCursorRect:[self leftTrackingArea] cursor:[NSCursor resizeLeftRightCursor]];
    [self addCursorRect:[self rightTrackingArea] cursor:[NSCursor resizeLeftRightCursor]];
}

- (void)mouseDown:(NSEvent*)event
{
    if (!_isFlexible) {
        [super mouseDown:event];
        return;
    }
    
    // Get frame
    NSRect  frame;
    frame = [self frame];
    
    // Get mouse location
    NSPoint mouseLoc;
    mouseLoc = [self convertPoint:[event locationInWindow] fromView:nil];
    
    // Track mouse
    NSRect  leftTrackingArea, rightTrackingArea;
    BOOL    inLeft, inRight;
    leftTrackingArea = [self leftTrackingArea];
    rightTrackingArea = [self rightTrackingArea];
    inLeft = [self mouse:mouseLoc inRect:leftTrackingArea];
    inRight = [self mouse:mouseLoc inRect:rightTrackingArea];
    if (inLeft || inRight) {
        // Event loop
        while (YES) {
            // Wait next event
            NSEvent*    waitingEvent;
            waitingEvent = [NSApp nextEventMatchingMask:(NSLeftMouseDraggedMask | NSLeftMouseUpMask) 
                    untilDate:[NSDate distantFuture] 
                    inMode:NSEventTrackingRunLoopMode 
                    dequeue:YES];
            if (!waitingEvent) {
                return;
            }
            
            // For dragging
            if ([waitingEvent type] == NSLeftMouseDragged) {
                // Calc dragging distant
                NSPoint newMouseLoc;
                int     delta = 0;
                int     newWidth;
                newMouseLoc = [self convertPoint:[waitingEvent locationInWindow] fromView:nil];
                if (inLeft) {
                    delta = [event locationInWindow].x - [waitingEvent locationInWindow].x;
                }
                if (inRight) {
                    delta = [waitingEvent locationInWindow].x - [event locationInWindow].x;
                }
                newWidth = frame.size.width + delta;
                
                if (delta != 0) {
                    if (newWidth < _minWidth) {
                        newWidth = _minWidth;
                    }
                    if (newWidth > _maxWidth) {
                        newWidth = _maxWidth;
                    }
                    
                    // Notify to delegate
                    id  delegate;
                    delegate = [self delegate];
                    if (delegate && [delegate respondsToSelector:@selector(searchField:resizeToNewSize:)]) {
                        [delegate searchField:self 
                                resizeToNewSize:NSMakeSize(newWidth, frame.size.height)];
                    }
                    
                    // Update tracking area
                    [self updateTrackingArea];
                }
                
                continue;
            }
            
            // For mouse up
            if ([waitingEvent type] == NSLeftMouseUp) {
                return;
            }
        }
    }
    
    [super mouseDown:event];
}

@end

#pragma mark -

@interface NSSearchFieldCell (private)
- (void)_trackButton:(id)button forEvent:(id)event inRect:(NSRect)rect ofView:(id)view;
@end

@implementation SRSearchFieldCell

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (NSRect)searchButtonRectForBounds:(NSRect)rect
{
    // Move 2 pixel to right
    NSRect  searchButtonRect;
    searchButtonRect = [super searchButtonRectForBounds:rect];
    //searchButtonRect.origin.x += 2;
    
    return searchButtonRect;
}

//--------------------------------------------------------------//
#pragma mark -- Search menu --
//--------------------------------------------------------------//

- (void)_trackButton:(id)button forEvent:(id)event inRect:(NSRect)rect ofView:(id)view
{
    // Notify to delegate
    id  delegate;
    delegate = [(SRSearchField*)[self controlView] delegate];
    if ([delegate respondsToSelector:@selector(searchFieldWillShowMenu:)]) {
        [delegate searchFieldWillShowMenu:(SRSearchField*)[self controlView]];
    }
    
    // Invoke super
    [super _trackButton:button forEvent:event inRect:rect ofView:view];
}

@end
