/*
HMPopUpButtonCell.m

Author: MIURA Kazki

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMPopUpButtonCell.h"
#import "HMAppKitEx.h"

#define DEBUG_DRAW 0

static const float HMSolidPopUpHeight = 18.0;


@implementation HMPopUpButtonCell

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (void)dealloc
{
	[_arrowImage release], _arrowImage = nil;
	[_bezelImage release], _bezelImage = nil;
	[_alternateBezelImage release], _alternateBezelImage = nil;
	[_titleAttributes release], _titleAttributes = nil;
	
	[super dealloc];
}

- (id)copyWithZone:(NSZone*)zone
{
	HMPopUpButtonCell *cell;
	cell = (HMPopUpButtonCell*)[super copyWithZone:zone];
	cell->_arrowImage = [_arrowImage copyWithZone:zone];
	cell->_bezelImage = [_bezelImage copyWithZone:zone];
	cell->_alternateBezelImage = [_alternateBezelImage copyWithZone:zone];
	cell->_popUpType = _popUpType;
	cell->_titleAttributes = [_titleAttributes copyWithZone:zone];
	
	return cell;
}

//--------------------------------------------------------------//
#pragma mark -- Arrow --
//--------------------------------------------------------------//

- (NSImage*)arrowImage
{
	return _arrowImage;
}

- (void)setArrowImage:(NSImage*)image
{
	[_arrowImage release];
	_arrowImage = [image retain];
}

//--------------------------------------------------------------//
#pragma mark -- PopUpType --
//--------------------------------------------------------------//

- (HMPopUpType)popUpType
{
	return _popUpType;
}

- (void)_configureForHMSolidPopUp
{
	// Set image position
	[self setImagePosition:NSImageLeft];
	
	// Set arrow image
    NSBundle*   bundle;
    bundle = [NSBundle bundleForClass:[self class]];
    
    NSImage*    image;
    image = [[[NSImage alloc] initWithContentsOfFile:
            [bundle pathForImageResource:@"solidPopupSelector"]] autorelease];
	[self setArrowImage:image];
	
	// Set title attributes ...
	NSMutableDictionary *attributes;
	NSMutableParagraphStyle *paragraph;
	NSButton *button;
	button = [[[NSButton alloc] initWithFrame:NSMakeRect(0, 0, 0, 0)] autorelease];
	attributes = [[[[button attributedTitle] attributesAtIndex:0 effectiveRange:nil] mutableCopy] autorelease];
	paragraph = [[[attributes valueForKey:NSParagraphStyleAttributeName] mutableCopy] autorelease];
	
	// paragraph
	[paragraph setAlignment:NSLeftTextAlignment];
	[paragraph setLineBreakMode:[self lineBreakMode]];
	[attributes setValue:paragraph forKey:NSParagraphStyleAttributeName];
	
	// font, color, baseline offset
	NSDictionary *newAttributes;
	newAttributes = [NSDictionary dictionaryWithObjectsAndKeys:
            [self font], NSFontAttributeName, 
			[NSColor colorWithCalibratedWhite:0.357 alpha:1.0], NSForegroundColorAttributeName,
			[NSNumber numberWithFloat:1.0], NSBaselineOffsetAttributeName,
			paragraph, NSParagraphStyleAttributeName,
			nil];
	[attributes addEntriesFromDictionary:newAttributes];
	
	// set
	[self setTitleAttributes:attributes];
}

- (void)setPopUpType:(HMPopUpType)type
{
	// Filter
	if (type >= HMPopUpTypeCount || type == _popUpType) {
		return;
	}
	
	// Set type
	_popUpType = type;
	
	// Reset
	[self setArrowImage:nil];
	[_bezelImage release], _bezelImage = nil;
	[_alternateBezelImage release], _alternateBezelImage = nil;
	[self setTitleAttributes:nil];
	
	// Configure for type
	switch ([self popUpType]) {
	case HMSolidPopUp: {
		[self _configureForHMSolidPopUp];
		break;
	}
	}
	
	// Update UI
	[[self controlView] setNeedsDisplay:YES];
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (void)drawWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	switch ([self popUpType]) {
	case HMNormalPopUp: {
		[super drawWithFrame:cellFrame inView:controlView];
		break;
	}
	default: {
		if ([self isBordered] || [self isBezeled]) {
			[self drawBezelWithFrame:cellFrame inView:controlView];
		}
		[self drawInteriorWithFrame:cellFrame inView:controlView];
		break;
	}
	}
}

- (void)_drawBezelForSolidPopUpWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// Filter
	if (![self isBordered] && ![self isBezeled]) {
		return;
	}
	
	// Get drawing rect
	NSRect drawingRect;
	drawingRect = [self drawingRectForBounds:cellFrame];
	if (NSIsEmptyRect(drawingRect)) {
		return;
	}
	
	// Prepare images
	if (!_bezelImage || drawingRect.size.width != [_bezelImage size].width) {
		
		// Release old images
		[_bezelImage release];
		[_alternateBezelImage release];
		
        // Get bundle
        NSBundle*   bundle;
        bundle = [NSBundle bundleForClass:[self class]];
        
		// Create images ...
		NSImage*    leftImage;
		NSImage*    middleImage;
		NSImage*    rightImage;
        
        leftImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidPopupL"]] autorelease];
        middleImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidPopupM"]] autorelease];
        rightImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidPopupR"]] autorelease];
        
        _bezelImage = [[NSImage imageWithSize:drawingRect.size
				leftImage:leftImage
				middleImage:middleImage
				rightImage:rightImage
				middleRect:nil] retain];
		
        leftImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidPopupPressedL"]] autorelease];
        middleImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidPopupPressedM"]] autorelease];
        rightImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidPopupPressedR"]] autorelease];
        
		_alternateBezelImage = [[NSImage imageWithSize:drawingRect.size
				leftImage:leftImage
				middleImage:middleImage
				rightImage:rightImage
				middleRect:nil] retain];
	}
	
	// Get image
	NSImage *img;
	img = [self isHighlighted] ? _alternateBezelImage : _bezelImage;
	if (!img) {
		return;
	}
	
	// Draw bezel
	float fraction;
	fraction = [self isEnabled] ? 1.0 : 0.5;
	[img drawInRect:drawingRect
			fromRect:HMMakeRect(NSZeroPoint, [img size])
			operation:NSCompositeSourceOver
			fraction:fraction
			contextRect:cellFrame
			isContextFlipped:[controlView isFlipped]];
}

- (void)drawBezelWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	switch ([self popUpType]) {
	case HMSolidPopUp: {
		[self _drawBezelForSolidPopUpWithFrame:cellFrame inView:controlView];
		[self drawArrowWithFrame:cellFrame inView:controlView];
		break;
	}
	default: {
		[super drawBezelWithFrame:cellFrame inView:controlView];
		break;
	}
	}
}

- (void)drawArrowWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// Get arrow image
	NSImage *arrowImage;
	arrowImage = [self arrowImage];
	if (!arrowImage) {
		return;
	}
	
	// Get arrow rect
	NSRect arrowRect;
	arrowRect = [self arrowRectForBounds:cellFrame];
	if (NSIsEmptyRect(arrowRect)) {
		return;
	}
	
	// Draw arrow
	float fraction;
	fraction = [self isEnabled] ? 1.0 : 0.5;
	[arrowImage drawInRect:arrowRect
			fromRect:HMMakeRect(NSZeroPoint, [arrowImage size])
			operation:NSCompositeSourceOver
			fraction:fraction
			contextRect:cellFrame
			isContextFlipped:[controlView isFlipped]];
}

- (void)drawImageWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	switch ([self popUpType]) {
	case HMNormalPopUp: {
		[super drawImageWithFrame:cellFrame inView:controlView];
		break;
	}
	default: {
		// Get image
		NSImage *img;
		img = [self image];
		if (!img) {
			return;
		}
		
		// Get image rect
		NSRect imgRect;
		imgRect = [self imageRectForBounds:cellFrame];
		
		// Set image size
		NSSize imgSize;
		imgSize = [img size];
		imgSize = HMScaleSizeInSize([img size], imgRect.size);
		
		// Calc image rect
		imgRect = HMCenterSize(imgSize, imgRect);
		
		// Draw image
		float fraction;
		fraction = [self isEnabled] ? 1.0 : 0.5;
		[img drawInRect:imgRect
				fromRect:HMMakeRect(NSZeroPoint, [img size])
				operation:NSCompositeSourceOver
				fraction:fraction
				contextRect:cellFrame
				isContextFlipped:[controlView isFlipped]];
		
#if DEBUG_DRAW
[[[NSColor blueColor] colorWithAlphaComponent:0.3] set];
NSRectFillUsingOperation(imgRect, NSCompositeSourceOver);
#endif
		break;
	}
	}
}

- (void)drawTitleWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	switch ([self popUpType]) {
	case HMNormalPopUp: {
		[super drawTitleWithFrame:cellFrame inView:controlView];
		break;
	}
	default: {
		
		// Get title rect
		NSRect titleRect;
		titleRect = [self titleRectForBounds:cellFrame];
		if (NSIsEmptyRect(titleRect)) {
			return;
		}
		if (![controlView isFlipped]) {
			titleRect = HMFlipRect(titleRect, cellFrame);
		}
		
		// Get current attributed title
		NSMutableAttributedString *title;
		title = [[[self attributedTitle] mutableCopy] autorelease];
		if (!title) {
			return;
		}
		if (![self isEnabled]) {
			NSMutableDictionary *attributes;
			NSColor *color;
			attributes = [[[title attributesAtIndex:0 effectiveRange:NULL] mutableCopy] autorelease];
			color = [attributes objectForKey:NSForegroundColorAttributeName];
			color = [color colorWithAlphaComponent:([color alphaComponent] / 2.0)];
			[attributes setObject:color forKey:NSForegroundColorAttributeName];
			[title setAttributes:attributes range:NSMakeRange(0, [title length])];
		}
		
		// Refine title rect for vertical center
		titleRect = NSInsetRect(titleRect, 0, (NSHeight(titleRect) - [title size].height) / 2.0);
		
		// Draw title
		[title drawInRect:titleRect];
		
#if DEBUG_DRAW
[[[NSColor blueColor] colorWithAlphaComponent:0.3] set];
NSRectFillUsingOperation(titleRect, NSCompositeSourceOver);
#endif
		break;
	}
	}
}

//--------------------------------------------------------------//
#pragma mark -- Size --
//--------------------------------------------------------------//

/**
	ToDo: Get width from menu and then add arrow width to it >>>
*/
- (NSSize)_cellSizeOfHMSolidPopUp
{
	float keyWidth;
	keyWidth = [super keyEquivalentWidth];
	
	// Make big rect
	NSRect bigRect;
	bigRect = NSMakeRect(0, 0, 6e7, 6e7);
	
	// Get image width
	float imgWidth;
	imgWidth = [self imageRectForBounds:bigRect].size.width;
	if (imgWidth) {
		imgWidth += 4.0;
	}
	
	// Get title width
	float titleWidth;
	titleWidth = [[self attributedTitle] size].width;
	if (titleWidth) {
		titleWidth += 4.0;
	}
	
	// Get cell width
	float width;
	width = 10.0 + imgWidth + titleWidth + 18.0;
	
	return NSMakeSize(width, HMSolidPopUpHeight);
}

- (NSSize)cellSize
{
	switch ([self popUpType]) {
	case HMSolidPopUp: {
		return [self _cellSizeOfHMSolidPopUp];
	}
	default: {
		return [super cellSize];
	}
	}
}

- (NSSize)cellSizeForBounds:(NSRect)aRect
{
	switch ([self popUpType]) {
	case HMSolidPopUp: {
		float height;
		height = fminf(HMSolidPopUpHeight, aRect.size.height);
		return NSMakeSize(aRect.size.width, height);
	}
	default: {
		return [super cellSizeForBounds:aRect];
	}
	}
}

- (NSRect)drawingRectForBounds:(NSRect)theRect
{
	switch ([self popUpType]) {
	case HMNormalPopUp: {
		return [super drawingRectForBounds:theRect];
	}
	default: {
		NSSize cellSize;
		cellSize = [self cellSizeForBounds:theRect];
		
		return HMCenterSize(cellSize, theRect);
	}
	}
}

- (NSRect)_arrowRectOfHMSolidPopUpForBounds:(NSRect)theRect
{
	// Filter
	if (![self arrowImage] || [self arrowPosition] == NSPopUpNoArrow) {
		return NSZeroRect;
	}
	
	// Get drawing rect
	NSRect innerRect;
	innerRect = [self drawingRectForBounds:theRect];
	innerRect = NSInsetRect(innerRect, 0, 1);
	if (NSIsEmptyRect(innerRect)) {
		return NSZeroRect;
	}
	
	// Get arrow size
	NSSize arrowSize;
	arrowSize = [[self arrowImage] size];
	if (HMIsEmptySize(arrowSize)) {
		return NSZeroRect;
	}
	
	// Calc origin
	NSPoint arrowOrigin;
	arrowOrigin.x = NSMaxX(innerRect) - 7.0 - arrowSize.width;
	arrowOrigin.y = NSMidY(innerRect) - arrowSize.height / 2.0;
	
	// Make arrow rect
	NSRect arrowRect;
	arrowRect = HMMakeRect(arrowOrigin, arrowSize);
	arrowRect = NSIntersectionRect(arrowRect, innerRect);
	arrowRect = NSIntegralRect(arrowRect);
	arrowRect.size = arrowSize;
	
	return arrowRect;
}

- (NSRect)arrowRectForBounds:(NSRect)theRect
{
	switch ([self popUpType]) {
	case HMSolidPopUp: {
		return [self _arrowRectOfHMSolidPopUpForBounds:theRect];
	}
	default: {
		return NSZeroRect;
	}
	}
}

- (NSRect)_imageRectOfHMSolidPopUpForBounds:(NSRect)theRect
{
	// Get inner rect
	NSRect innerRect;
	innerRect = NSInsetRect([self drawingRectForBounds:theRect], 0, [self cellAttribute:NSCellIsInsetButton]);
	
	// Get image rect
	NSRect imgRect;
	imgRect = [super imageRectForBounds:theRect];
	
	// Refine image rect
	imgRect.size = HMScaleSizeInSize(imgRect.size, innerRect.size);
	imgRect.origin.y = NSMidY(innerRect) - imgRect.size.height / 2.0;
	
	return imgRect;
}

- (NSRect)imageRectForBounds:(NSRect)theRect
{
	switch ([self popUpType]) {
	case HMSolidPopUp: {
		return [self _imageRectOfHMSolidPopUpForBounds:theRect];
	}
	default: {
		return [super imageRectForBounds:theRect];
	}
	}
}

- (NSRect)_titleRectOfHMSolidPopUpForBounds:(NSRect)theRect
{
	// Get drawing rect
	NSRect drawingRect;
	drawingRect = [self drawingRectForBounds:theRect];
	if (NSIsEmptyRect(drawingRect)) {
		return NSZeroRect;
	}
	
	// Get arrow rect
	NSRect arrowRect;
	arrowRect = [self arrowRectForBounds:theRect];
	
	// Get margin
	float margin;
	margin = NSMaxX(theRect) - NSMaxX(arrowRect);
	
	// Get title rect
	NSRect titleRect;
	titleRect = NSInsetRect(drawingRect, margin, [self cellAttribute:NSCellIsInsetButton]);
	titleRect.size.width -= arrowRect.size.width + margin + 4.0;
	titleRect.origin.x += 4.0;
	
	return titleRect;
}

- (NSRect)titleRectForBounds:(NSRect)theRect
{
	switch ([self popUpType]) {
	case HMSolidPopUp: {
		return [self _titleRectOfHMSolidPopUpForBounds:theRect];
	}
	default: {
		return [super titleRectForBounds:theRect];
	}
	}
}

//--------------------------------------------------------------//
#pragma mark -- Title --
//--------------------------------------------------------------//

- (NSDictionary*)titleAttributes
{
	return _titleAttributes;
}

- (void)setTitleAttributes:(NSDictionary*)attributes
{
	[_titleAttributes release];
	_titleAttributes = [attributes retain];
}

- (NSAttributedString*)attributedTitle
{
	// Filter
	if ([self popUpType] == HMNormalPopUp) {
		return [super attributedTitle];
	}
	
	// Get title
	NSString *title;
	title = [self title];
	if (!title) {
		return nil;
	}
	
	// Get attributes
	NSDictionary *attributes;
	attributes = [self titleAttributes];
	
	return [[[NSAttributedString alloc]
			initWithString:title attributes:attributes] autorelease];
}

- (void)setLineBreakMode:(NSLineBreakMode)mode
{
	// Super
	[super setLineBreakMode:mode];
	
	// Filter
	if ([self popUpType] == HMNormalPopUp) {
		return;
	}
	
	// Set line break mode for title attributes ...
	NSMutableDictionary *attributes;
	NSMutableParagraphStyle *paragraph;
	
	attributes = [[[self titleAttributes] mutableCopy] autorelease];
	paragraph = [[[attributes valueForKey:NSParagraphStyleAttributeName] mutableCopy] autorelease];
	if (!paragraph) {
		return;
	}
	[paragraph setLineBreakMode:mode];
	[attributes setValue:paragraph forKey:NSParagraphStyleAttributeName];
	
	[self setTitleAttributes:attributes];
}

@end
