//
//  HMLinkTextFieldCell.m
//  iTunesLikeTable
//
//  Created by MIURA Kazki on 2/22/07.
//  Copyright 2007 MIURA Kazki. All rights reserved.
//

#define DEBUG_DRAW 0

#import "HMAppKitEx.h"
#import "HMLinkTextFieldCell.h"

@interface HMImageTextFieldCell (Private)
- (NSRect)_focusRingFrameForFrame:(NSRect)frame cellFrame:(NSRect)cellFrame;
@end

@interface HMLinkTextFieldCell (Private)
- (float)_buttonWidthWithSpace;
@end

#pragma mark -


@implementation HMLinkTextFieldCell

//--------------------------------------------------------------//
#pragma mark -- Initialization --
//--------------------------------------------------------------//

- (id)init
{
	if (!(self = [super init])) {
		return nil;
	}
	
    // Get bundle
    NSBundle*   bundle;
    bundle = [NSBundle bundleForClass:[self class]];
    
	// Button cell
	_buttonCell = [[HMButtonCell alloc] init];
	[_buttonCell setHMBezelStyle:HMCustomImageBezelStyle];
    
    NSImage*    image;
    image = [[[NSImage alloc] initWithContentsOfFile:
            [bundle pathForImageResource:@"tableLinkSelected"]] autorelease];
	[_buttonCell setImage:image];
    image = [[[NSImage alloc] initWithContentsOfFile:
            [bundle pathForImageResource:@"tableLinkSelected"]] autorelease];
	[_buttonCell setAlternateImage:image];		// Change >>>
	
	// Button image
	[self setSelectedButtonImage:[_buttonCell image]];
    image = [[[NSImage alloc] initWithContentsOfFile:
            [bundle pathForImageResource:@"tableLink"]] autorelease];
	[self setSecondarySelectedButtonImage:image];
	
	// Focus ring
	[self setFocusOnlyTitle:YES];
	
	return self;
}

- (void)dealloc
{
	[_buttonCell release], _buttonCell = nil;
	[_selectedButtonImage release], _selectedButtonImage = nil;
	[_secondarySelectedButtonImage release], _secondarySelectedButtonImage = nil;
	
	[super dealloc];
}

- (id)copyWithZone:(NSZone*)zone
{
	HMLinkTextFieldCell *cell;
	cell = (HMLinkTextFieldCell*)[super copyWithZone:zone];
	cell->_buttonCell = [_buttonCell copyWithZone:zone];
	cell->_selectedButtonImage = [_selectedButtonImage copyWithZone:zone];
	cell->_secondarySelectedButtonImage = [_secondarySelectedButtonImage copyWithZone:zone];
	
	return cell;
}

//--------------------------------------------------------------//
#pragma mark -- Button Cell --
//--------------------------------------------------------------//

- (HMButtonCell*)buttonCell
{
	return _buttonCell;
}

- (NSImage*)selectedButtonImage
{
	return _selectedButtonImage;
}

- (NSImage*)secondarySelectedButtonImage
{
	return _secondarySelectedButtonImage;
}

- (void)setSelectedButtonImage:(NSImage*)image
{
	if (image && _selectedButtonImage != image) {
		[_selectedButtonImage release];
		_selectedButtonImage = [image retain];
	}
}

- (void)setSecondarySelectedButtonImage:(NSImage*)image
{
	if (_secondarySelectedButtonImage != image) {
		[_secondarySelectedButtonImage release];
		_secondarySelectedButtonImage = [image retain];
	}
}

//--------------------------------------------------------------//
#pragma mark -- Cell Attributes --
//--------------------------------------------------------------//

- (void)setEnabled:(BOOL)flag
{
	[super setEnabled:flag];
	[[self buttonCell] setEnabled:flag];
}

//--------------------------------------------------------------//
#pragma mark -- Mouse --
//--------------------------------------------------------------//

- (BOOL)trackMouse:(NSEvent *)theEvent
		inRect:(NSRect)cellFrame
		ofView:(NSView *)controlView
		untilMouseUp:(BOOL)untilMouseUp
{
	// Button track until mouse up ...
	// (ignoring button's isContinuous & sendActionOn)
	NSPoint point;
	NSRect buttonRect;
	point = [controlView convertPoint:[theEvent locationInWindow] fromView:nil];
	buttonRect = [self buttonRectForBounds:cellFrame];
	if (![controlView isFlipped]) {
		buttonRect = HMFlipRect(buttonRect, cellFrame);
	}
	if (NSPointInRect(point, buttonRect)) {
		HMButtonCell *button;
		NSEvent *currentEvent;
		NSPoint currentPoint;
		button = [self buttonCell];
		[button highlight:YES withFrame:buttonRect inView:controlView];
		while (currentEvent = [[controlView window] nextEventMatchingMask:(NSLeftMouseDraggedMask | NSLeftMouseUpMask)]) {
			currentPoint = [controlView convertPoint:[currentEvent locationInWindow] fromView:nil];
			switch ([currentEvent type]) {
			case NSLeftMouseDragged: {
				[button highlight:NSPointInRect(currentPoint, buttonRect) withFrame:buttonRect inView:controlView];
				break;
			}
			case NSLeftMouseUp: {
				[button highlight:NO withFrame:buttonRect inView:controlView];
				if (NSPointInRect(currentPoint, buttonRect)) {
					[[button target] performSelector:[button action] withObject:button];
				}
				return YES;
			}
			}
		}		// END of wating event
	}		// END of button track
	
	return [super trackMouse:theEvent inRect:cellFrame ofView:controlView untilMouseUp:untilMouseUp];
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (NSRect)_focusRingFrameForFrame:(NSRect)frame
		cellFrame:(NSRect)cellFrame
{
	NSRect ringRect;
	ringRect = [super _focusRingFrameForFrame:frame cellFrame:cellFrame];
	if (![self focusOnlyTitle]) {
		ringRect.size.width += [self _buttonWidthWithSpace];
	}
	return ringRect;
}

- (void)drawInteriorWithFrame:(NSRect)cellFrame 
		inView:(NSView*)controlView
{
	// Draw button
	if ([self isHighlighted]) {
		[self drawButtonWithFrame:cellFrame inView:controlView];
	}
	
	// Super
	[super drawInteriorWithFrame:cellFrame inView:controlView];
}

- (void)drawButtonWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// Get button rect
	NSRect buttonRect;
	buttonRect = [self buttonRectForBounds:cellFrame];
	if (![controlView isFlipped]) {
		buttonRect = HMFlipRect(buttonRect, cellFrame);
	}
	
#if DEBUG_DRAW
NSColor *color;
color = [[NSColor blueColor] colorWithAlphaComponent:0.5];
NSRectFillListWithColors(&buttonRect, &color, 1);
#endif
	
	// Set image
	if ([[controlView window] isKeyWindow] && [[controlView window] firstResponder] == controlView) {
		[[self buttonCell] setImage:[self selectedButtonImage]];
	}
	else {
		[[self buttonCell] setImage:[self secondarySelectedButtonImage]];
	}
	
	// Draw button
	[[self buttonCell] drawWithFrame:buttonRect inView:controlView];
}

//--------------------------------------------------------------//
#pragma mark -- Size --
//--------------------------------------------------------------//

- (float)_buttonWidthWithSpace
{
	float padding, buttonWidth;
	padding = [self paddingBetweenImageAndText];
	buttonWidth = [[self buttonCell] cellSize].width;
	if (!buttonWidth) {
		return 0;
	}
	
	return (padding + buttonWidth + padding);
}

- (NSSize)cellSize
{
	// Get cell size
	NSSize cellSize;
	cellSize = [super cellSize];
	
	// Add button space width
	cellSize.width += [self _buttonWidthWithSpace];
	
	return cellSize;
}

- (NSSize)cellSizeForBounds:(NSRect)theRect
{
	// Super
	NSSize cellSize;
	cellSize = [super cellSizeForBounds:theRect];
	
	// Add button space width
	cellSize.width += [self _buttonWidthWithSpace];
	
	// Limite width
	cellSize.width = MIN(NSWidth(theRect), cellSize.width);
	
	return cellSize;
}

// * Prior to title rect
- (NSRect)buttonRectForBounds:(NSRect)theRect
{
	// Filter
	if (![self isHighlighted]) {
		return NSZeroRect;
	}
	
	NSRect space, slice;
	space = theRect;
	
	// Subtract image space
	float imageWidth;
	imageWidth = [self imageWidthWithSpace];
	NSDivideRect(space, &slice, &space, imageWidth, NSMinXEdge);
	
	// Get button size
	NSSize buttonSize;
	buttonSize = [[self buttonCell] cellSize];
	
	// Make button rect
	NSRect buttonRect;
	float padding;
	padding = [self paddingBetweenImageAndText];
	buttonRect.size.width = MIN(buttonSize.width, space.size.width);
	buttonRect.origin.x = NSMaxX(space) - padding - buttonRect.size.width;
	buttonRect.size.height = buttonSize.height;
	buttonRect.origin.y = HMCenterSize(buttonSize, space).origin.y;
	
	return buttonRect;
}

// * Depend on image & button rect
- (NSRect)titleRectForBounds:(NSRect)theRect
{
	// Get title rect ...
	NSRect titleRect;
	titleRect = [super titleRectForBounds:theRect];
	if (![self isHighlighted]) {
		return titleRect;
	}
	
	// subtract button space
	NSRect slice;
	NSDivideRect(titleRect, &slice, &titleRect, [self _buttonWidthWithSpace], NSMaxXEdge);
	
	return titleRect;
}

@end
