//
//  HMInfoBarButtonCell.m
//  HMInfoBar
//
//  Created by MIURA Kazki on 10.04.06.
//  Copyright 2006 MIURA Kazki. All rights reserved.
//

#import "HMAppKitEx.h"
#import "HMInfoBarButtonCell.h"


static const int HMInfoBarButtonCellMarginX = 2;

@interface HMInfoBarButtonCell (Private)
- (NSImage*)_backgroundImageWithSize:(NSSize)size;
- (NSString*)_horizontalEllipsis;
@end



@implementation HMInfoBarButtonCell


//--------------------------------------------------------------//
#pragma mark	-- Creating --
//--------------------------------------------------------------//

- (id)init
{
	if (self = [super init]) {
		// Initialize ...
		[self setBezelStyle:NSRoundedBezelStyle];
		[self setButtonType:NSToggleButton];
		[self setControlSize:NSMiniControlSize];
		[self setFont:[NSFont boldSystemFontOfSize:11.0]];
		[self setAlignment:NSLeftTextAlignment];
		[self setShowsFirstResponder:YES];
		_isMouseOver = NO;
		
		// Image
        NSBundle*   bundle;
        bundle = [NSBundle bundleForClass:[self class]];
		_backgroundImage = nil;
		_leftImage = [[NSImage alloc] initWithContentsOfFile:[bundle pathForImageResource:@"infoBarSelectedL"]];
        _middleImage = [[NSImage alloc] initWithContentsOfFile:[bundle pathForImageResource:@"infoBarSelectedM"]];
		_rightImage = [[NSImage alloc] initWithContentsOfFile:[bundle pathForImageResource:@"infoBarSelectedR"]];
	}
	
	return self;
}

- (void)dealloc
{
	// Release
	[_backgroundImage release], _backgroundImage = nil;
	[_leftImage release], _leftImage = nil;
	[_middleImage release], _middleImage = nil;
	[_rightImage release], _rightImage = nil;
	
	// dealloc super
	[super dealloc];
}

- (id)copyWithZone:(NSZone*)zone
{
	HMInfoBarButtonCell *cell;
	cell = (HMInfoBarButtonCell*)[super copyWithZone:zone];
	cell->_backgroundImage = [_backgroundImage retain];
	cell->_leftImage = [_leftImage retain];
	cell->_middleImage = [_middleImage retain];
	cell->_rightImage = [_rightImage retain];
	
	return cell;
}

//--------------------------------------------------------------//
#pragma mark	-- Drawing --
//--------------------------------------------------------------//

- (void)setBackgroundLeftImage:(NSImage*)leftImage
		middleImage:(NSImage*)middleImage
		rightImage:(NSImage*)rightImage
{
	// Filter
	if (!leftImage || !middleImage || !rightImage) {
		return;
	}
	
	// Left
	[leftImage retain];
	[_leftImage release];
	_leftImage = leftImage;
	
	// Middle
	[middleImage retain];
	[_middleImage release];
	_middleImage = middleImage;
	
	// Right
	[rightImage retain];
	[_rightImage release];
	_rightImage = rightImage;
	
	// Background
	[self _backgroundImageWithSize:[self cellSize]];
}

- (void)drawBezelWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// Get state
	int state;
	state = [self state];
	
	// Get isInKeyWindow
	BOOL isInKeyWindow;
	isInKeyWindow = ([controlView window] == [NSApp keyWindow]);
	
	// Get acceptsFirstMouse
	BOOL acceptsFirstMouse;
	acceptsFirstMouse = [[self controlView] acceptsFirstMouse:nil];
	
	// Get isEnabled
	BOOL isEnabled;
	isEnabled = [self isEnabled];
	
	// Filter
	if (state == NSOffState && !_isMouseOver) {
		// Draw interior only
		[self drawInteriorWithFrame:cellFrame inView:controlView];
		return;
	}
	
	// Use cartesian coordinate system
	NSAffineTransform *transform = nil;
	if ([controlView isFlipped]) {
		transform = [NSAffineTransform verticalFlipTransformForRect:cellFrame];
		[transform concat];
	}
	
	// Get background image
	NSImage *bgImage;
	NSSize bgImageSize;
	bgImageSize.width = cellFrame.size.width;
	bgImageSize.height = [[self class] heightForControlSize:[self controlSize]];
	bgImage = [self _backgroundImageWithSize:bgImageSize];
	
	// Draw background image
	NSPoint drawPoint;
	drawPoint.x = 0;
	drawPoint.y = rintf((cellFrame.size.height - bgImageSize.height) / 2.0);
	if (state == NSOffState && _isMouseOver && (isInKeyWindow || acceptsFirstMouse) && isEnabled) {
		// Mouse Over Image
		[bgImage drawAtPoint:drawPoint
				fromRect:HMMakeRect(NSZeroPoint, [bgImage size])
				operation:NSCompositeSourceOver
				fraction:0.5];
	}
	else if (state == NSOnState) {
		if (isEnabled) {
			// Selected Image
			[bgImage drawAtPoint:drawPoint
					fromRect:HMMakeRect(NSZeroPoint, [bgImage size])
					operation:NSCompositeSourceOver
					fraction:1.0];
		}
		else {
			// Selected Image (disabled)
			[bgImage drawAtPoint:drawPoint
					fromRect:HMMakeRect(NSZeroPoint, [bgImage size])
					operation:NSCompositeSourceOver
					fraction:0.3];
		}
	}
	
	// Revert coordinate system
	if (transform) {
		[transform invert];
		[transform concat];
	}
}

- (void)drawInteriorWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// Draw title ...
	
	// get title rect
	NSRect titleRect;
	titleRect = [self titleRectForBounds:cellFrame];
	if (NSIsEmptyRect(titleRect)) {
		return;
	}
	
	// get title
	NSAttributedString *title;
	if ([self state] == NSOffState && !_isMouseOver) {
		title = [self attributedTitle];
	}
	else {
		title = [self attributedAlternateTitle];
	}
	if (!title) {
		return;
	}
	title = [title truncatedAttributedStringForWidth:NSWidth(titleRect)];
	
	// draw title
	[self drawTitle:title withFrame:titleRect inView:controlView];
}

//--------------------------------------------------------------//
#pragma mark	-- Size --
//--------------------------------------------------------------//

+ (float)heightForControlSize:(int)controlSize
{
	switch (controlSize) {
	case NSRegularControlSize :
		return 20.0;
	case NSSmallControlSize :
		return 17.0;
	case NSMiniControlSize :
		return 15.0;
	}
	
	return 0.0;
}

- (NSSize)cellSize
{
	// Get height
	int height;
	height = [[self class] heightForControlSize:[self controlSize]];
	
	// Get left & right width
	int leftWidth, rightWidth;
	leftWidth = rintf(height / [_leftImage size].height * [_leftImage size].width);
	rightWidth = rintf(height / [_rightImage size].height * [_rightImage size].width);
	
	// Get middle width
	int middleWidth = 0;
	NSAttributedString *attrStr;
	attrStr = [self attributedTitle];
	if (attrStr) {
		middleWidth += ceil([attrStr size].width);
	}
	// Add margin
	if (middleWidth > 0) {
		middleWidth += HMInfoBarButtonCellMarginX * 2;
	}
	
	// Get cell size
	NSSize cellSize;
	cellSize.width = leftWidth + middleWidth + rightWidth;
	cellSize.height = height;
	
	return cellSize;
}

- (NSSize)maxSize
{
	return [self cellSize];
}

- (NSSize)minSize
{
	// Get height
	int height;
	height = [[self class] heightForControlSize:[self controlSize]];
	
	// Get left & right width
	int leftWidth, rightWidth;
	leftWidth = rintf(height / [_leftImage size].height * [_leftImage size].width);
	rightWidth = rintf(height / [_rightImage size].height * [_rightImage size].width);
	
	// Get middleWidth
	int middleWidth = 0;
	NSAttributedString *attrStr;
	unsigned length;
	attrStr = [self attributedTitle];
	length = [attrStr length];
	if (length == 0) {
		// do nothing
	}
	else if (length <= 2) {
		// Get attributed string's width
		middleWidth = ceil([attrStr size].width);
	}
	else {
		// Get width of first character & _horizontalEllipsis ...
		NSString *minTitle;
		minTitle = [NSString stringWithFormat:@"%@%@",
				[[attrStr string] substringToIndex:1], [self _horizontalEllipsis]];
		
		// Make minimum title
		NSAttributedString *attrMinTitle;
		attrMinTitle = [[NSAttributedString alloc]
				initWithString:minTitle
				attributes:[attrStr attributesAtIndex:0 effectiveRange:NULL]];
		
		// Get attributed minimum title width
		middleWidth = ceil([attrMinTitle size].width);
	}
	// Add margin
	if (middleWidth > 0) {
		middleWidth += HMInfoBarButtonCellMarginX * 2;
	}
	
	// Get minSize
	NSSize minSize;
	minSize.height = height;
	minSize.width = leftWidth + middleWidth + rightWidth;
	
	return minSize;
}

- (NSRect)titleRectForBounds:(NSRect)theRect
{
	// Get middle rect (Gimmic)
	NSSize bgImageSize;
	bgImageSize = NSMakeSize(NSWidth(theRect), [[self class] heightForControlSize:[self controlSize]]);
	if (!_backgroundImage || !NSEqualSizes([_backgroundImage size], bgImageSize)) {
		[self _backgroundImageWithSize:bgImageSize];
	}
	
	// Get title rect
	NSRect titleRect;
	float inset;
	inset = [self cellAttribute:NSCellIsInsetButton];
	titleRect = NSOffsetRect(_middleRect, theRect.origin.x, theRect.origin.y);
	titleRect = NSInsetRect(titleRect, HMInfoBarButtonCellMarginX, 0);
	
	return titleRect;
}

//--------------------------------------------------------------//
#pragma mark	-- Title --
//--------------------------------------------------------------//

- (void)setTitle:(NSString*)title
{
	// set title & alternate title
	[super setTitle:title];
	[super setAlternateTitle:title];
}

- (NSAttributedString*)attributedTitle
{
	// Return customized attributed title
	NSMutableDictionary *attributes;
	attributes = [[[[super attributedTitle] attributesAtIndex:0 effectiveRange:NULL] mutableCopy] autorelease];
	
	// Foreground Color
	NSColor *color;
	color = [NSColor colorWithCalibratedWhite:0.357 alpha:1.0];
	if (![self isEnabled]) {
		color = [color colorWithAlphaComponent:0.5];
	}
	[attributes setValue:color forKey:NSForegroundColorAttributeName];
	
	return [[[NSAttributedString alloc] initWithString:[self title] attributes:attributes] autorelease];
}

- (NSAttributedString*)attributedAlternateTitle
{
	// Return customized attributed alternate title
	NSMutableDictionary *attributes;
	attributes = [[[[self attributedTitle] attributesAtIndex:0 effectiveRange:NULL] mutableCopy] autorelease];
	
	// Foreground Color
	NSColor *color;
	color = [NSColor whiteColor];
	if (![self isEnabled]) {
		color = [color colorWithAlphaComponent:0.5];
	}
	[attributes setValue:[NSColor whiteColor] forKey:NSForegroundColorAttributeName];
	
	// Shadow
	NSShadow *shadow;
	shadow = [[[NSShadow alloc] init] autorelease];
	[shadow setShadowOffset:NSMakeSize(0.0, -1.0)];
	[shadow setShadowColor:[NSColor darkGrayColor]];
	[shadow setShadowBlurRadius:0.5];
	[attributes setValue:shadow forKey:NSShadowAttributeName];
	
	return [[[NSAttributedString alloc] initWithString:[self alternateTitle] attributes:attributes] autorelease];
}

//--------------------------------------------------------------//
#pragma mark	-- Util --
//--------------------------------------------------------------//

- (NSImage*)_backgroundImageWithSize:(NSSize)size
{
	// Get background image
	if (!_backgroundImage || !NSEqualSizes(size, [_backgroundImage size])) {
		
		// Release old background image
		[_backgroundImage release];
		
		// Get new background image
		_backgroundImage = [[NSImage imageWithSize:size
				leftImage:_leftImage
				middleImage:_middleImage
				rightImage:_rightImage
				middleRect:&_middleRect] retain];
#if DEBUG
		NSAssert(_backgroundImage, @"Can't create background image, cos' cell frame is too small.");
#endif
	}
	
	return _backgroundImage;
}

- (NSString*)_horizontalEllipsis
{
	static NSString *_horizontalEllipsis = nil;
	if (!_horizontalEllipsis) {
		unichar uni[1];
		uni[0] = 0x2026;
		_horizontalEllipsis = [[NSString stringWithCharacters:uni length:1] retain];
	}
	
	return _horizontalEllipsis;
}

- (void)mouseEntered:(NSEvent*)event
{
	_isMouseOver = YES;
	
	// Update UI
	if ([[self controlView] respondsToSelector:@selector(updateCell:)]) {
		[[self controlView] performSelector:@selector(updateCell:) withObject:self];
	}
	else {
		[[self controlView] setNeedsDisplay:YES];
	}
}

- (void)mouseExited:(NSEvent*)event
{
	_isMouseOver = NO;
	
	// Update UI
	if ([[self controlView] respondsToSelector:@selector(updateCell:)]) {
		[[self controlView] performSelector:@selector(updateCell:) withObject:self];
	}
	else {
		[[self controlView] setNeedsDisplay:YES];
	}
}

- (void)setAlternateTitle:(NSString*)altTitle
{
	// Deny
	[self doesNotRecognizeSelector:_cmd];
}

- (void)setAttributedAlternateTitle:(NSAttributedString*)attrStr
{
	// Deny
	[self doesNotRecognizeSelector:_cmd];
}

@end
