/*
HMButtonCell.m

Author: Makoto Kinoshita & MIURA Kazki

Copyright 2004-2006 The Shiira Project. All rights reserved.

Redistribution and use in source and binary forms, with or without modification, are permitted 
provided that the following conditions are met:

  1. Redistributions of source code must retain the above copyright notice, this list of conditions 
  and the following disclaimer.

  2. Redistributions in binary form must reproduce the above copyright notice, this list of 
  conditions and the following disclaimer in the documentation and/or other materials provided 
  with the distribution.

THIS SOFTWARE IS PROVIDED BY THE SHIIRA PROJECT ``AS IS'' AND ANY EXPRESS OR IMPLIED WARRANTIES, 
INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR 
PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE SHIIRA PROJECT OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, 
INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED TO, 
PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION) 
HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING 
NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE 
POSSIBILITY OF SUCH DAMAGE.
*/

#import "HMButtonCell.h"
#import "HMAppKitEx.h"

static const float HMSolidBezelStyleHeight = 15.0;
static const float HMSolidBezelStyleSideWidth = 7.0;

@implementation HMButtonCell

//--------------------------------------------------------------//
#pragma mark -- Initialize --
//--------------------------------------------------------------//

- (void)dealloc
{
	[_bezelImage release], _bezelImage = nil;
	[_alternateBezelImage release], _alternateBezelImage = nil;
    [_selectedImage release], _selectedImage = nil;
	
	[super dealloc];
}

- (id)copyWithZone:(NSZone*)zone
{
	HMButtonCell *cell;
	cell = (HMButtonCell*)[super copyWithZone:zone];
	cell->_bezelImage = nil;
	cell->_alternateBezelImage = nil;
	cell->_selectedImage = nil;
	
	return cell;
}

//--------------------------------------------------------------//
#pragma mark -- ButtonType --
//--------------------------------------------------------------//

- (void)_configureForHMSeparatorButtonType
{
	[self setBordered:NO];
}

- (void)setHMButtonType:(HMButtonType)type
{
	// Filter
	if (type >= HMButtonTypeCount || _HMButtonType == type) {
		return;
	}
	
	// Set button type
	_HMButtonType = type;
	
	// Configure for type
	switch (_HMButtonType) {
	case HMSeparatorButtonType: {
		[self _configureForHMSeparatorButtonType];
		break;
	}
	}
	
	// Update UI
	[[self controlView] setNeedsDisplay:YES];
}

- (void)setButtonType:(NSButtonType)type
{
	// Set HMButtonType to Normal
	[self setHMButtonType:HMNormalButtonType];
	
	// Super
	[super setButtonType:type];
}

- (BOOL)isSeparator
{
	return _HMButtonType == HMSeparatorButtonType;
}

- (void)setSeparator:(BOOL)isSeparator
{
	[self setHMButtonType:(isSeparator ? HMSeparatorButtonType : HMNormalButtonType)];
}

//--------------------------------------------------------------//
#pragma mark -- BezelStyle --
//--------------------------------------------------------------//

- (HMBezelStyle)HMBezelStyle
{
	return _HMBezelStyle;
}

- (void)_configureForHMCustomImageBezelStyle
{
	[self setBordered:NO];
	[super setButtonType:NSMomentaryChangeButton];
	[super setBezelStyle:NSRegularSquareBezelStyle];
}

- (void)_configureForHMSolidBezelStyle
{
	[self setBordered:NO];
	[self setHighlightsBy:NSNoCellMask];
}

- (void)setHMBezelStyle:(HMBezelStyle)style
{
	// Filter
	if (style >= HMBezelStyleCount || _HMBezelStyle == style) {
		return;
	}
	
	// Nullify bezel images
	[_bezelImage release], _bezelImage = nil;
	[_alternateBezelImage release], _alternateBezelImage = nil;
	
	// Set bezel style
	_HMBezelStyle = style;
	
	// Configure for the style
	switch (_HMBezelStyle) {
	case HMCustomImageBezelStyle: {
		[self _configureForHMCustomImageBezelStyle];
		break;
	}
	case HMSolidBezelStyle: {
		[self _configureForHMSolidBezelStyle];
		break;
	}
	}
	
	// Update UI
	[[self controlView] setNeedsDisplay:YES];
}

- (void)setBezelStyle:(NSBezelStyle)style
{
	// Set HMBezelStyle to Normal
	[self setHMBezelStyle:HMNormalBezelStyle];
	
	// Super
	[super setBezelStyle:style];
}

- (NSImage*)bezelImage
{
	return _bezelImage;
}

- (void)setBezelImage:(NSImage*)image
{
	[_bezelImage release];
	_bezelImage = [image retain];
}

- (NSImage*)alternateBezelImage
{
	return _alternateBezelImage;
}

- (void)setAlternateBezelImage:(NSImage*)image
{
	[_alternateBezelImage release];
	_alternateBezelImage = [image retain];
}

//--------------------------------------------------------------//
#pragma mark -- Title --
//--------------------------------------------------------------//

- (NSAttributedString*)attributedTitle
{
	// Switch by HMBezelStyle
	switch (_HMBezelStyle) {
	case HMSolidBezelStyle: {
	
		// Return customized attributed title...
		NSMutableDictionary *attributes;
		attributes = [[[[super attributedTitle] attributesAtIndex:0 effectiveRange:NULL] mutableCopy] autorelease];
		
		// font
		[attributes setValue:[NSFont boldSystemFontOfSize:10.0] forKey:NSFontAttributeName];
		
		// foreground color
		float alpha;
		alpha = [self isEnabled] ? 1.0 : 0.5;
		[attributes setValue:[NSColor colorWithCalibratedWhite:0.357 alpha:alpha] forKey:NSForegroundColorAttributeName];
		
		return [[[NSAttributedString alloc] initWithString:[self title] attributes:attributes] autorelease];
	}
	}
	
	return [super attributedTitle];
}

//--------------------------------------------------------------//
#pragma mark -- Selection --
//--------------------------------------------------------------//

- (BOOL)isSelected
{
    return _isSelected;
}

- (void)setSelected:(BOOL)isSelected
{
    _isSelected = isSelected;
}

- (NSImage*)selectedImage
{
    return _selectedImage;
}

- (void)setSelectedImage:(NSImage*)image
{
    _selectedImage = [image retain];
}

//--------------------------------------------------------------//
#pragma mark -- Drawing --
//--------------------------------------------------------------//

- (void)_drawSeparatorWithFrame:(NSRect)cellFrame 
		inView:(NSView*)controlView
{
	// Draw separator
	NSRect	rect;
	
#if 1
	[[NSColor colorWithCalibratedWhite:0.36f alpha:1.0f] set];
	rect.origin.x = cellFrame.origin.x;
	rect.origin.y = cellFrame.origin.y + ceil(cellFrame.size.height / 2) - 1;
	rect.size.width = cellFrame.size.width;
	rect.size.height = 1;
	NSRectFill(rect);
#else
	[[NSColor colorWithCalibratedWhite:0.6f alpha:0.8f] set];
	rect.origin.x = cellFrame.origin.x;
	rect.origin.y = cellFrame.origin.y + ceil(cellFrame.size.height / 2) - 1;
	rect.size.width = cellFrame.size.width;
	rect.size.height = 1;
	NSRectFill(rect);
	
	[[NSColor colorWithCalibratedWhite:0.9f alpha:0.8f] set];
	rect.origin.y += 1;
	NSRectFill(rect);
#endif
}

- (void)_drawCustomImageBezelWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// Get bezel image
	NSImage *bezel;
	if ([self state] == NSOffState || ![self alternateBezelImage]) {
		bezel = [self bezelImage];
	}
	else {
		bezel = [self alternateBezelImage];
	}
	if (!bezel) {
		return;
	}
	
	// Coordinate
	if ([bezel isFlipped] != [controlView isFlipped]) {
		[bezel setFlipped:[controlView isFlipped]];
	}
	
	// Draw
	float fraction;
	fraction = [self isEnabled] ? 1.0 : 0.5;
	[bezel drawInRect:cellFrame
			fromRect:HMMakeRect(NSZeroPoint, [bezel size])
			operation:NSCompositeSourceOver
			fraction:fraction];
}

- (void)_drawSolidBezelWithFrame:(NSRect)cellFrame 
		inView:(NSView*)controlView
{
	// Get width
	float width;
	width = cellFrame.size.width;
	
	// Prepare images
	if (!_bezelImage || width != [_bezelImage size].width) {
		// Release old images
		[_bezelImage release];
		[_alternateBezelImage release];
		
		// Create images ...
        NSBundle*   bundle;
        bundle = [NSBundle bundleForClass:[self class]];
        
        NSImage*    leftImage;
        NSImage*    middleImage;
        NSImage*    rightImage;
        
		NSSize imgSize;
		imgSize.width = width;
		imgSize.height = HMSolidBezelStyleHeight;
		
        leftImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidButtonL"]] autorelease];
        middleImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidButtonM"]] autorelease];
        rightImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidButtonR"]] autorelease];
        
		_bezelImage = [[NSImage imageWithSize:imgSize
				leftImage:leftImage
				middleImage:middleImage
				rightImage:rightImage
				middleRect:nil] retain];
		
        leftImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidButtonPressedL"]] autorelease];
        middleImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidButtonPressedM"]] autorelease];
        rightImage = [[[NSImage alloc] initWithContentsOfFile:
                [bundle pathForImageResource:@"solidButtonPressedR"]] autorelease];
		
        _alternateBezelImage = [[NSImage imageWithSize:imgSize
				leftImage:leftImage
				middleImage:middleImage
				rightImage:rightImage
				middleRect:nil] retain];
	}
	
	// Get image
	NSImage *img;
	img = [self isHighlighted] ? _alternateBezelImage : _bezelImage;
	if (!img) {
		return;
	}
	
	// Draw
	NSPoint drawPoint;
	float fraction;
	drawPoint.x = cellFrame.origin.x;
	drawPoint.y = HMCenterSize([img size], cellFrame).origin.y;
	fraction = [self isEnabled] ? 1.0 : 0.5;
	[img drawInRect:cellFrame fromRect:HMMakeRect(NSZeroPoint, [img size]) operation:NSCompositeSourceOver fraction:fraction contextRect:cellFrame isContextFlipped:[controlView isFlipped]];
}

- (NSRect)_drawTitleForSolidBezelWithFrame:(NSRect)cellFrame
		inView:(NSView*)controlView
{
	// Get title
	NSAttributedString *title;
	BOOL alternate = NO;
	if ([self showsStateBy] & NSContentsCellMask && [self state] != NSOffState) {
		alternate = YES;
	}
	if ([self highlightsBy] & NSContentsCellMask && [self isHighlighted]) {
		alternate = YES;
	}
	if (alternate && [self attributedAlternateTitle]) {
		title = [self attributedAlternateTitle];
	}
	else {
		title = [self attributedTitle];
	}
	
	// Get title rect
	NSRect titleRect;
	titleRect = [self titleRectForBounds:cellFrame];
	if (![controlView isFlipped]) {
		titleRect = HMFlipRect(titleRect, cellFrame);
	}
	
	// Draw title
	return [super drawTitle:title withFrame:titleRect inView:controlView];
}

- (void)_drawSelectedImageWithFrame:(NSRect)cellFrame 
        inView:(NSView*)controlView
{
    if (!_selectedImage || !_isSelected) {
        return;
    }
    
    if ([_selectedImage isFlipped] != [controlView isFlipped]) {
        [_selectedImage setFlipped:[controlView isFlipped]];
    }
    
    // Draw image
    NSRect  srcRect, destRect;
    srcRect.origin = NSZeroPoint;
    srcRect.size = [_selectedImage size];
    destRect.origin = cellFrame.origin;
    destRect.size = [_selectedImage size];
    [_selectedImage drawInRect:destRect fromRect:srcRect 
            operation:NSCompositeSourceOver fraction:1.0f];
}

- (void)drawInteriorWithFrame:(NSRect)cellFrame
		inView:(NSView *)controlView
{
    // For selected
    if (_isSelected) {
        [self _drawSelectedImageWithFrame:cellFrame inView:controlView];
    }
    else {
        // HMButtonType
        switch (_HMButtonType) {
        case HMSeparatorButtonType: {
            [self _drawSeparatorWithFrame:cellFrame inView:controlView];
            return;
        }
        }
        
        // HMBezelStyle
        switch (_HMBezelStyle) {
        case HMCustomImageBezelStyle: {
            [self _drawCustomImageBezelWithFrame:cellFrame inView:controlView];
            break;
        }
        case HMSolidBezelStyle: {
            [self _drawSolidBezelWithFrame:cellFrame inView:controlView];
            [self _drawTitleForSolidBezelWithFrame:cellFrame inView:controlView];
            return;
        }
        }
	}
    
	[super drawInteriorWithFrame:cellFrame inView:controlView];
}

//--------------------------------------------------------------//
#pragma mark -- Cell Size --
//--------------------------------------------------------------//

- (NSSize)_cellSizeOfHMSolidBezelStyle
{
	// Get width
	float width = 0;
	width += HMSolidBezelStyleSideWidth * 2;
	width += [[self attributedTitle] size].width;
	
	return NSMakeSize(width, HMSolidBezelStyleHeight);
}

- (NSSize)cellSize
{
	// HMButtonType
	switch (_HMButtonType) {
	case HMSeparatorButtonType: {
		return NSMakeSize(1.0f, 1.0f);
	}
	}
	
	// HMBezelStyle
	switch (_HMBezelStyle) {
	case HMCustomImageBezelStyle: {
		return [[self image] size];
	}
	case HMSolidBezelStyle: {
		return [self _cellSizeOfHMSolidBezelStyle];
	}
	}
	
	return [super cellSize];
}

- (NSRect)imageRectForBounds:(NSRect)theRect
{
	// HMBezelStyle
	switch ([self HMBezelStyle]) {
	case HMCustomImageBezelStyle: {
		return theRect;
	}
	default: {
		return [super imageRectForBounds:theRect];
	}
	}
}

- (NSRect)titleRectForBounds:(NSRect)theRect
{
	// HMBezelStyle
	switch ([self HMBezelStyle]) {
	case HMCustomImageBezelStyle: {
		return theRect;
	}
	case HMSolidBezelStyle: {
		return NSMakeRect(
				NSMinX(theRect) + HMSolidBezelStyleSideWidth, NSMinY(theRect),
				theRect.size.width - HMSolidBezelStyleSideWidth * 2, HMSolidBezelStyleHeight);
	}
	default: {
		return [super titleRectForBounds:theRect];
	}
	}
}

@end
