# stri_enc_list: List Known Character Encodings

## Description

Gives the list of encodings that are supported by <span class="pkg">ICU</span>.

## Usage

``` r
stri_enc_list(simplify = TRUE)
```

## Arguments

|            |                                                                                 |
|------------|---------------------------------------------------------------------------------|
| `simplify` | single logical value; return a character vector or a list of character vectors? |

## Details

Apart from given encoding identifiers and their aliases, some other specifiers might additionally be available. This is due to the fact that <span class="pkg">ICU</span> tries to normalize converter names. For instance, `'UTF8'` is also valid, see [stringi-encoding](about_encoding.md) for more information.

## Value

If `simplify` is `FALSE`, a list of character vectors is returned. Each list element represents a unique character encoding. The `name` attribute gives the <span class="pkg">ICU</span> Canonical Name of an encoding family. The elements (character vectors) are its aliases.

If `simplify` is `TRUE` (the default), then the resulting list is coerced to a character vector and sorted, and returned with removed duplicated entries.

## Author(s)

[Marek Gagolewski](https://www.gagolewski.com/) and other contributors

## See Also

The official online manual of <span class="pkg">stringi</span> at <https://stringi.gagolewski.com/>

Gagolewski M., <span class="pkg">stringi</span>: Fast and portable character string processing in R, *Journal of Statistical Software* 103(2), 2022, 1-59, [doi:10.18637/jss.v103.i02](https://doi.org/10.18637/jss.v103.i02)

Other encoding_management: [`about_encoding`](about_encoding.md), [`stri_enc_info()`](stri_enc_info.md), [`stri_enc_mark()`](stri_enc_mark.md), [`stri_enc_set()`](stri_enc_set.md)

## Examples




```r
stri_enc_list()
##    [1] "037"                                          
##    [2] "273"                                          
##    [3] "277"                                          
##    [4] "278"                                          
##    [5] "280"                                          
##    [6] "284"                                          
##    [7] "285"                                          
##    [8] "297"                                          
##    [9] "420"                                          
##   [10] "424"                                          
##   [11] "437"                                          
##   [12] "500"                                          
##   [13] "646"                                          
##   [14] "737"                                          
##   [15] "775"                                          
##   [16] "813"                                          
##   [17] "819"                                          
##   [18] "838"                                          
##   [19] "850"                                          
##   [20] "851"                                          
##   [21] "852"                                          
##   [22] "855"                                          
##   [23] "856"                                          
##   [24] "857"                                          
##   [25] "860"                                          
##   [26] "861"                                          
##   [27] "862"                                          
##   [28] "863"                                          
##   [29] "865"                                          
##   [30] "866"                                          
##   [31] "868"                                          
##   [32] "869"                                          
##   [33] "871"                                          
##   [34] "875"                                          
##   [35] "912"                                          
##   [36] "913"                                          
##   [37] "914"                                          
##   [38] "915"                                          
##   [39] "916"                                          
##   [40] "920"                                          
##   [41] "921"                                          
##   [42] "922"                                          
##   [43] "923"                                          
##   [44] "930"                                          
##   [45] "933"                                          
##   [46] "935"                                          
##   [47] "937"                                          
##   [48] "939"                                          
##   [49] "943"                                          
##   [50] "949"                                          
##   [51] "950"                                          
##   [52] "964"                                          
##   [53] "970"                                          
##   [54] "1006"                                         
##   [55] "1025"                                         
##   [56] "1026"                                         
##   [57] "1047"                                         
##   [58] "1089"                                         
##   [59] "1097"                                         
##   [60] "1098"                                         
##   [61] "1112"                                         
##   [62] "1122"                                         
##   [63] "1123"                                         
##   [64] "1124"                                         
##   [65] "1383"                                         
##   [66] "5601"                                         
##   [67] "8859_1"                                       
##   [68] "8859_2"                                       
##   [69] "8859_3"                                       
##   [70] "8859_4"                                       
##   [71] "8859_5"                                       
##   [72] "8859_6"                                       
##   [73] "8859_7"                                       
##   [74] "8859_8"                                       
##   [75] "8859_9"                                       
##   [76] "8859_13"                                      
##   [77] "8859_15"                                      
##   [78] "33722"                                        
##   [79] "Adobe-Standard-Encoding"                      
##   [80] "ANSI_X3.4-1968"                               
##   [81] "ANSI_X3.4-1986"                               
##   [82] "ANSI1251"                                     
##   [83] "arabic"                                       
##   [84] "ASCII"                                        
##   [85] "ascii7"                                       
##   [86] "ASMO-708"                                     
##   [87] "Big5"                                         
##   [88] "Big5-HKSCS"                                   
##   [89] "big5-hkscs:unicode3.0"                        
##   [90] "big5hk"                                       
##   [91] "BOCU-1"                                       
##   [92] "CCSID00858"                                   
##   [93] "CCSID01140"                                   
##   [94] "CCSID01141"                                   
##   [95] "CCSID01142"                                   
##   [96] "CCSID01143"                                   
##   [97] "CCSID01144"                                   
##   [98] "CCSID01145"                                   
##   [99] "CCSID01146"                                   
##  [100] "CCSID01147"                                   
##  [101] "CCSID01148"                                   
##  [102] "CCSID01149"                                   
##  [103] "CESU-8"                                       
##  [104] "chinese"                                      
##  [105] "cns11643"                                     
##  [106] "COMPOUND_TEXT"                                
##  [107] "cp-ar"                                        
##  [108] "cp-gr"                                        
##  [109] "cp-is"                                        
##  [110] "cp037"                                        
##  [111] "cp37"                                         
##  [112] "CP273"                                        
##  [113] "cp277"                                        
##  [114] "cp278"                                        
##  [115] "CP280"                                        
##  [116] "CP284"                                        
##  [117] "CP285"                                        
##  [118] "cp290"                                        
##  [119] "cp297"                                        
##  [120] "cp367"                                        
##  [121] "cp420"                                        
##  [122] "cp424"                                        
##  [123] "cp437"                                        
##  [124] "CP500"                                        
##  [125] "cp737"                                        
##  [126] "cp775"                                        
##  [127] "cp803"                                        
##  [128] "cp813"                                        
##  [129] "cp819"                                        
##  [130] "cp838"                                        
##  [131] "cp850"                                        
##  [132] "cp851"                                        
##  [133] "cp852"                                        
##  [134] "cp855"                                        
##  [135] "cp856"                                        
##  [136] "cp857"                                        
##  [137] "CP00858"                                      
##  [138] "cp858"                                        
##  [139] "cp860"                                        
##  [140] "cp861"                                        
##  [141] "cp862"                                        
##  [142] "cp863"                                        
##  [143] "cp864"                                        
##  [144] "cp865"                                        
##  [145] "cp866"                                        
##  [146] "CP868"                                        
##  [147] "cp869"                                        
##  [148] "CP870"                                        
##  [149] "CP871"                                        
##  [150] "cp874"                                        
##  [151] "cp875"                                        
##  [152] "cp878"                                        
##  [153] "cp912"                                        
##  [154] "cp913"                                        
##  [155] "cp914"                                        
##  [156] "cp915"                                        
##  [157] "cp916"                                        
##  [158] "CP918"                                        
##  [159] "cp920"                                        
##  [160] "cp921"                                        
##  [161] "cp922"                                        
##  [162] "cp923"                                        
##  [163] "cp930"                                        
##  [164] "cp932"                                        
##  [165] "cp933"                                        
##  [166] "cp935"                                        
##  [167] "CP936"                                        
##  [168] "cp937"                                        
##  [169] "cp939"                                        
##  [170] "cp943"                                        
##  [171] "cp943c"                                       
##  [172] "cp949"                                        
##  [173] "cp949c"                                       
##  [174] "cp950"                                        
##  [175] "cp964"                                        
##  [176] "cp970"                                        
##  [177] "cp1006"                                       
##  [178] "cp1025"                                       
##  [179] "CP1026"                                       
##  [180] "cp1047"                                       
##  [181] "cp1089"                                       
##  [182] "cp1097"                                       
##  [183] "cp1098"                                       
##  [184] "cp1112"                                       
##  [185] "cp1122"                                       
##  [186] "cp1123"                                       
##  [187] "cp1124"                                       
##  [188] "cp1125"                                       
##  [189] "cp1131"                                       
##  [190] "CP01140"                                      
##  [191] "cp1140"                                       
##  [192] "CP01141"                                      
##  [193] "cp1141"                                       
##  [194] "CP01142"                                      
##  [195] "cp1142"                                       
##  [196] "CP01143"                                      
##  [197] "cp1143"                                       
##  [198] "CP01144"                                      
##  [199] "cp1144"                                       
##  [200] "CP01145"                                      
##  [201] "cp1145"                                       
##  [202] "CP01146"                                      
##  [203] "cp1146"                                       
##  [204] "CP01147"                                      
##  [205] "cp1147"                                       
##  [206] "CP01148"                                      
##  [207] "cp1148"                                       
##  [208] "CP01149"                                      
##  [209] "cp1149"                                       
##  [210] "cp1200"                                       
##  [211] "cp1201"                                       
##  [212] "cp1208"                                       
##  [213] "cp1250"                                       
##  [214] "cp1251"                                       
##  [215] "cp1252"                                       
##  [216] "cp1253"                                       
##  [217] "cp1254"                                       
##  [218] "cp1255"                                       
##  [219] "cp1256"                                       
##  [220] "cp1257"                                       
##  [221] "cp1258"                                       
##  [222] "cp1363"                                       
##  [223] "cp1383"                                       
##  [224] "cp1386"                                       
##  [225] "cp33722"                                      
##  [226] "cpibm37"                                      
##  [227] "cpibm284"                                     
##  [228] "cpibm285"                                     
##  [229] "cpibm297"                                     
##  [230] "csAdobeStandardEncoding"                      
##  [231] "csASCII"                                      
##  [232] "csBig5"                                       
##  [233] "csBOCU-1"                                     
##  [234] "csEUCKR"                                      
##  [235] "csEUCPkdFmtJapanese"                          
##  [236] "csGB2312"                                     
##  [237] "csHPRoman8"                                   
##  [238] "csIBM037"                                     
##  [239] "csIBM273"                                     
##  [240] "csIBM277"                                     
##  [241] "csIBM278"                                     
##  [242] "csIBM280"                                     
##  [243] "csIBM284"                                     
##  [244] "csIBM285"                                     
##  [245] "csIBM290"                                     
##  [246] "csIBM297"                                     
##  [247] "csIBM420"                                     
##  [248] "csIBM424"                                     
##  [249] "csIBM500"                                     
##  [250] "csIBM855"                                     
##  [251] "csIBM857"                                     
##  [252] "csIBM860"                                     
##  [253] "csIBM861"                                     
##  [254] "csIBM863"                                     
##  [255] "csIBM864"                                     
##  [256] "csIBM865"                                     
##  [257] "csIBM866"                                     
##  [258] "csIBM868"                                     
##  [259] "csIBM869"                                     
##  [260] "csIBM870"                                     
##  [261] "csIBM871"                                     
##  [262] "csIBM918"                                     
##  [263] "csIBM1026"                                    
##  [264] "csIBMThai"                                    
##  [265] "csISO58GB231280"                              
##  [266] "csISO2022CN"                                  
##  [267] "csISO2022JP"                                  
##  [268] "csISO2022JP2"                                 
##  [269] "csISO2022KR"                                  
##  [270] "csisolatin0"                                  
##  [271] "csISOLatin1"                                  
##  [272] "csISOLatin2"                                  
##  [273] "csISOLatin3"                                  
##  [274] "csISOLatin4"                                  
##  [275] "csISOLatin5"                                  
##  [276] "csISOLatin6"                                  
##  [277] "csisolatin9"                                  
##  [278] "csISOLatinArabic"                             
##  [279] "csISOLatinCyrillic"                           
##  [280] "csISOLatinGreek"                              
##  [281] "csISOLatinHebrew"                             
##  [282] "csJISEncoding"                                
##  [283] "csKOI8R"                                      
##  [284] "csKSC56011987"                                
##  [285] "csMacintosh"                                  
##  [286] "csPC8CodePage437"                             
##  [287] "csPC775Baltic"                                
##  [288] "csPC850Multilingual"                          
##  [289] "csPC851"                                      
##  [290] "csPC862LatinHebrew"                           
##  [291] "csPCp852"                                     
##  [292] "csPCp855"                                     
##  [293] "csShiftJIS"                                   
##  [294] "csUCS4"                                       
##  [295] "csUnicode"                                    
##  [296] "csWindows31J"                                 
##  [297] "cyrillic"                                     
##  [298] "DOS-720"                                      
##  [299] "DOS-862"                                      
##  [300] "ebcdic-ar"                                    
##  [301] "ebcdic-cp-ar1"                                
##  [302] "ebcdic-cp-ar2"                                
##  [303] "ebcdic-cp-be"                                 
##  [304] "ebcdic-cp-ca"                                 
##  [305] "ebcdic-cp-ch"                                 
##  [306] "EBCDIC-CP-DK"                                 
##  [307] "ebcdic-cp-es"                                 
##  [308] "ebcdic-cp-fi"                                 
##  [309] "ebcdic-cp-fr"                                 
##  [310] "ebcdic-cp-gb"                                 
##  [311] "ebcdic-cp-he"                                 
##  [312] "ebcdic-cp-is"                                 
##  [313] "ebcdic-cp-it"                                 
##  [314] "ebcdic-cp-nl"                                 
##  [315] "EBCDIC-CP-NO"                                 
##  [316] "ebcdic-cp-roece"                              
##  [317] "ebcdic-cp-se"                                 
##  [318] "ebcdic-cp-us"                                 
##  [319] "ebcdic-cp-wt"                                 
##  [320] "ebcdic-cp-yu"                                 
##  [321] "ebcdic-de"                                    
##  [322] "ebcdic-de-273+euro"                           
##  [323] "ebcdic-dk"                                    
##  [324] "ebcdic-dk-277+euro"                           
##  [325] "ebcdic-es-284+euro"                           
##  [326] "ebcdic-fi-278+euro"                           
##  [327] "ebcdic-fr-297+euro"                           
##  [328] "ebcdic-gb"                                    
##  [329] "ebcdic-gb-285+euro"                           
##  [330] "ebcdic-he"                                    
##  [331] "ebcdic-international-500+euro"                
##  [332] "ebcdic-is"                                    
##  [333] "ebcdic-is-871+euro"                           
##  [334] "ebcdic-it-280+euro"                           
##  [335] "EBCDIC-JP-kana"                               
##  [336] "ebcdic-no-277+euro"                           
##  [337] "ebcdic-se-278+euro"                           
##  [338] "ebcdic-sv"                                    
##  [339] "ebcdic-us-37+euro"                            
##  [340] "ebcdic-xml-us"                                
##  [341] "ECMA-114"                                     
##  [342] "ECMA-118"                                     
##  [343] "ECMA-128"                                     
##  [344] "ELOT_928"                                     
##  [345] "EUC-CN"                                       
##  [346] "EUC-JP"                                       
##  [347] "euc-jp-2007"                                  
##  [348] "EUC-KR"                                       
##  [349] "EUC-TW"                                       
##  [350] "euc-tw-2014"                                  
##  [351] "eucjis"                                       
##  [352] "eucTH"                                        
##  [353] "Extended_UNIX_Code_Packed_Format_for_Japanese"
##  [354] "GB_2312-80"                                   
##  [355] "GB2312"                                       
##  [356] "gb2312-1980"                                  
##  [357] "GB2312.1980-0"                                
##  [358] "gb18030"                                      
##  [359] "GB18030"                                      
##  [360] "GBK"                                          
##  [361] "greek"                                        
##  [362] "greek8"                                       
##  [363] "gsm-03.38-2009"                               
##  [364] "GSM0338"                                      
##  [365] "hebrew"                                       
##  [366] "hebrew8"                                      
##  [367] "hkbig5"                                       
##  [368] "HKSCS-BIG5"                                   
##  [369] "hp-roman8"                                    
##  [370] "hp15CN"                                       
##  [371] "HZ"                                           
##  [372] "HZ-GB-2312"                                   
##  [373] "ibm-37"                                       
##  [374] "ibm-037"                                      
##  [375] "ibm-37_P100-1995"                             
##  [376] "ibm-37_P100-1995,swaplfnl"                    
##  [377] "ibm-37-s390"                                  
##  [378] "ibm-273"                                      
##  [379] "ibm-273_P100-1995"                            
##  [380] "ibm-277"                                      
##  [381] "ibm-277_P100-1995"                            
##  [382] "ibm-278"                                      
##  [383] "ibm-278_P100-1995"                            
##  [384] "ibm-280"                                      
##  [385] "ibm-280_P100-1995"                            
##  [386] "ibm-284"                                      
##  [387] "ibm-284_P100-1995"                            
##  [388] "ibm-285"                                      
##  [389] "ibm-285_P100-1995"                            
##  [390] "ibm-290"                                      
##  [391] "ibm-290_P100-1995"                            
##  [392] "ibm-297"                                      
##  [393] "ibm-297_P100-1995"                            
##  [394] "ibm-367"                                      
##  [395] "ibm-420"                                      
##  [396] "ibm-420_X120-1999"                            
##  [397] "ibm-424"                                      
##  [398] "ibm-424_P100-1995"                            
##  [399] "ibm-437"                                      
##  [400] "ibm-437_P100-1995"                            
##  [401] "ibm-500"                                      
##  [402] "ibm-500_P100-1995"                            
##  [403] "ibm-720"                                      
##  [404] "ibm-720_P100-1997"                            
##  [405] "ibm-737"                                      
##  [406] "ibm-737_P100-1997"                            
##  [407] "ibm-775"                                      
##  [408] "ibm-775_P100-1996"                            
##  [409] "ibm-803"                                      
##  [410] "ibm-803_P100-1999"                            
##  [411] "ibm-813"                                      
##  [412] "ibm-813_P100-1995"                            
##  [413] "ibm-819"                                      
##  [414] "ibm-838"                                      
##  [415] "ibm-838_P100-1995"                            
##  [416] "ibm-850"                                      
##  [417] "ibm-850_P100-1995"                            
##  [418] "ibm-851"                                      
##  [419] "ibm-851_P100-1995"                            
##  [420] "ibm-852"                                      
##  [421] "ibm-852_P100-1995"                            
##  [422] "ibm-855"                                      
##  [423] "ibm-855_P100-1995"                            
##  [424] "ibm-856"                                      
##  [425] "ibm-856_P100-1995"                            
##  [426] "ibm-857"                                      
##  [427] "ibm-857_P100-1995"                            
##  [428] "ibm-858"                                      
##  [429] "ibm-858_P100-1997"                            
##  [430] "ibm-860"                                      
##  [431] "ibm-860_P100-1995"                            
##  [432] "ibm-861"                                      
##  [433] "ibm-861_P100-1995"                            
##  [434] "ibm-862"                                      
##  [435] "ibm-862_P100-1995"                            
##  [436] "ibm-863"                                      
##  [437] "ibm-863_P100-1995"                            
##  [438] "ibm-864"                                      
##  [439] "ibm-864_X110-1999"                            
##  [440] "ibm-865"                                      
##  [441] "ibm-865_P100-1995"                            
##  [442] "ibm-866"                                      
##  [443] "ibm-866_P100-1995"                            
##  [444] "ibm-867"                                      
##  [445] "ibm-867_P100-1998"                            
##  [446] "ibm-868"                                      
##  [447] "ibm-868_P100-1995"                            
##  [448] "ibm-869"                                      
##  [449] "ibm-869_P100-1995"                            
##  [450] "ibm-870"                                      
##  [451] "ibm-870_P100-1995"                            
##  [452] "ibm-871"                                      
##  [453] "ibm-871_P100-1995"                            
##  [454] "ibm-874"                                      
##  [455] "ibm-874_P100-1995"                            
##  [456] "ibm-875"                                      
##  [457] "ibm-875_P100-1995"                            
##  [458] "ibm-878"                                      
##  [459] "ibm-878_P100-1996"                            
##  [460] "ibm-901"                                      
##  [461] "ibm-901_P100-1999"                            
##  [462] "ibm-902"                                      
##  [463] "ibm-902_P100-1999"                            
##  [464] "ibm-912"                                      
##  [465] "ibm-912_P100-1995"                            
##  [466] "ibm-913"                                      
##  [467] "ibm-913_P100-2000"                            
##  [468] "ibm-914"                                      
##  [469] "ibm-914_P100-1995"                            
##  [470] "ibm-915"                                      
##  [471] "ibm-915_P100-1995"                            
##  [472] "ibm-916"                                      
##  [473] "ibm-916_P100-1995"                            
##  [474] "ibm-918"                                      
##  [475] "ibm-918_P100-1995"                            
##  [476] "ibm-920"                                      
##  [477] "ibm-920_P100-1995"                            
##  [478] "ibm-921"                                      
##  [479] "ibm-921_P100-1995"                            
##  [480] "ibm-922"                                      
##  [481] "ibm-922_P100-1999"                            
##  [482] "ibm-923"                                      
##  [483] "ibm-923_P100-1998"                            
##  [484] "ibm-930"                                      
##  [485] "ibm-930_P120-1999"                            
##  [486] "ibm-931"                                      
##  [487] "ibm-932"                                      
##  [488] "ibm-932_VSUB_VPUA"                            
##  [489] "ibm-933"                                      
##  [490] "ibm-933_P110-1995"                            
##  [491] "ibm-935"                                      
##  [492] "ibm-935_P110-1999"                            
##  [493] "ibm-937"                                      
##  [494] "ibm-937_P110-1999"                            
##  [495] "ibm-939"                                      
##  [496] "ibm-939_P120-1999"                            
##  [497] "ibm-942"                                      
##  [498] "ibm-942_P12A-1999"                            
##  [499] "ibm-942_VSUB_VPUA"                            
##  [500] "ibm-943"                                      
##  [501] "ibm-943_P15A-2003"                            
##  [502] "ibm-943_P130-1999"                            
##  [503] "ibm-943_VASCII_VSUB_VPUA"                     
##  [504] "ibm-943_VSUB_VPUA"                            
##  [505] "IBM-943C"                                     
##  [506] "ibm-949"                                      
##  [507] "ibm-949_P11A-1999"                            
##  [508] "ibm-949_P110-1999"                            
##  [509] "ibm-949_VASCII_VSUB_VPUA"                     
##  [510] "ibm-949_VSUB_VPUA"                            
##  [511] "IBM-949C"                                     
##  [512] "ibm-950"                                      
##  [513] "ibm-950_P110-1999"                            
##  [514] "ibm-954"                                      
##  [515] "ibm-954_P101-2007"                            
##  [516] "ibm-964"                                      
##  [517] "ibm-964_P110-1999"                            
##  [518] "ibm-964_VPUA"                                 
##  [519] "ibm-970"                                      
##  [520] "ibm-970_P110_P110-2006_U2"                    
##  [521] "ibm-970_VPUA"                                 
##  [522] "ibm-971"                                      
##  [523] "ibm-971_P100-1995"                            
##  [524] "ibm-971_VPUA"                                 
##  [525] "ibm-1006"                                     
##  [526] "ibm-1006_P100-1995"                           
##  [527] "ibm-1025"                                     
##  [528] "ibm-1025_P100-1995"                           
##  [529] "ibm-1026"                                     
##  [530] "ibm-1026_P100-1995"                           
##  [531] "ibm-1047"                                     
##  [532] "ibm-1047_P100-1995"                           
##  [533] "ibm-1047_P100-1995,swaplfnl"                  
##  [534] "ibm-1047-s390"                                
##  [535] "ibm-1051"                                     
##  [536] "ibm-1051_P100-1995"                           
##  [537] "ibm-1089"                                     
##  [538] "ibm-1089_P100-1995"                           
##  [539] "ibm-1097"                                     
##  [540] "ibm-1097_P100-1995"                           
##  [541] "ibm-1098"                                     
##  [542] "ibm-1098_P100-1995"                           
##  [543] "ibm-1112"                                     
##  [544] "ibm-1112_P100-1995"                           
##  [545] "ibm-1122"                                     
##  [546] "ibm-1122_P100-1999"                           
##  [547] "ibm-1123"                                     
##  [548] "ibm-1123_P100-1995"                           
##  [549] "ibm-1124"                                     
##  [550] "ibm-1124_P100-1996"                           
##  [551] "ibm-1125"                                     
##  [552] "ibm-1125_P100-1997"                           
##  [553] "ibm-1129"                                     
##  [554] "ibm-1129_P100-1997"                           
##  [555] "ibm-1130"                                     
##  [556] "ibm-1130_P100-1997"                           
##  [557] "ibm-1131"                                     
##  [558] "ibm-1131_P100-1997"                           
##  [559] "ibm-1132"                                     
##  [560] "ibm-1132_P100-1998"                           
##  [561] "ibm-1133"                                     
##  [562] "ibm-1133_P100-1997"                           
##  [563] "ibm-1137"                                     
##  [564] "ibm-1137_P100-1999"                           
##  [565] "ibm-1140"                                     
##  [566] "ibm-1140_P100-1997"                           
##  [567] "ibm-1140_P100-1997,swaplfnl"                  
##  [568] "ibm-1140-s390"                                
##  [569] "ibm-1141"                                     
##  [570] "ibm-1141_P100-1997"                           
##  [571] "ibm-1141_P100-1997,swaplfnl"                  
##  [572] "ibm-1141-s390"                                
##  [573] "ibm-1142"                                     
##  [574] "ibm-1142_P100-1997"                           
##  [575] "ibm-1142_P100-1997,swaplfnl"                  
##  [576] "ibm-1142-s390"                                
##  [577] "ibm-1143"                                     
##  [578] "ibm-1143_P100-1997"                           
##  [579] "ibm-1143_P100-1997,swaplfnl"                  
##  [580] "ibm-1143-s390"                                
##  [581] "ibm-1144"                                     
##  [582] "ibm-1144_P100-1997"                           
##  [583] "ibm-1144_P100-1997,swaplfnl"                  
##  [584] "ibm-1144-s390"                                
##  [585] "ibm-1145"                                     
##  [586] "ibm-1145_P100-1997"                           
##  [587] "ibm-1145_P100-1997,swaplfnl"                  
##  [588] "ibm-1145-s390"                                
##  [589] "ibm-1146"                                     
##  [590] "ibm-1146_P100-1997"                           
##  [591] "ibm-1146_P100-1997,swaplfnl"                  
##  [592] "ibm-1146-s390"                                
##  [593] "ibm-1147"                                     
##  [594] "ibm-1147_P100-1997"                           
##  [595] "ibm-1147_P100-1997,swaplfnl"                  
##  [596] "ibm-1147-s390"                                
##  [597] "ibm-1148"                                     
##  [598] "ibm-1148_P100-1997"                           
##  [599] "ibm-1148_P100-1997,swaplfnl"                  
##  [600] "ibm-1148-s390"                                
##  [601] "ibm-1149"                                     
##  [602] "ibm-1149_P100-1997"                           
##  [603] "ibm-1149_P100-1997,swaplfnl"                  
##  [604] "ibm-1149-s390"                                
##  [605] "ibm-1153"                                     
##  [606] "ibm-1153_P100-1999"                           
##  [607] "ibm-1153_P100-1999,swaplfnl"                  
##  [608] "ibm-1153-s390"                                
##  [609] "ibm-1154"                                     
##  [610] "ibm-1154_P100-1999"                           
##  [611] "ibm-1155"                                     
##  [612] "ibm-1155_P100-1999"                           
##  [613] "ibm-1156"                                     
##  [614] "ibm-1156_P100-1999"                           
##  [615] "ibm-1157"                                     
##  [616] "ibm-1157_P100-1999"                           
##  [617] "ibm-1158"                                     
##  [618] "ibm-1158_P100-1999"                           
##  [619] "ibm-1160"                                     
##  [620] "ibm-1160_P100-1999"                           
##  [621] "ibm-1162"                                     
##  [622] "ibm-1162_P100-1999"                           
##  [623] "ibm-1164"                                     
##  [624] "ibm-1164_P100-1999"                           
##  [625] "ibm-1168"                                     
##  [626] "ibm-1168_P100-2002"                           
##  [627] "ibm-1200"                                     
##  [628] "ibm-1201"                                     
##  [629] "ibm-1202"                                     
##  [630] "ibm-1203"                                     
##  [631] "ibm-1204"                                     
##  [632] "ibm-1205"                                     
##  [633] "ibm-1208"                                     
##  [634] "ibm-1209"                                     
##  [635] "ibm-1212"                                     
##  [636] "ibm-1213"                                     
##  [637] "ibm-1214"                                     
##  [638] "ibm-1215"                                     
##  [639] "ibm-1232"                                     
##  [640] "ibm-1233"                                     
##  [641] "ibm-1234"                                     
##  [642] "ibm-1235"                                     
##  [643] "ibm-1236"                                     
##  [644] "ibm-1237"                                     
##  [645] "ibm-1250"                                     
##  [646] "ibm-1250_P100-1995"                           
##  [647] "ibm-1251"                                     
##  [648] "ibm-1251_P100-1995"                           
##  [649] "ibm-1252"                                     
##  [650] "ibm-1252_P100-2000"                           
##  [651] "ibm-1253"                                     
##  [652] "ibm-1253_P100-1995"                           
##  [653] "ibm-1254"                                     
##  [654] "ibm-1254_P100-1995"                           
##  [655] "ibm-1255"                                     
##  [656] "ibm-1255_P100-1995"                           
##  [657] "ibm-1256"                                     
##  [658] "ibm-1256_P110-1997"                           
##  [659] "ibm-1257"                                     
##  [660] "ibm-1257_P100-1995"                           
##  [661] "ibm-1258"                                     
##  [662] "ibm-1258_P100-1997"                           
##  [663] "ibm-1276"                                     
##  [664] "ibm-1276_P100-1995"                           
##  [665] "ibm-1363"                                     
##  [666] "ibm-1363_P11B-1998"                           
##  [667] "ibm-1363_P110-1997"                           
##  [668] "ibm-1363_VASCII_VSUB_VPUA"                    
##  [669] "ibm-1363_VSUB_VPUA"                           
##  [670] "ibm-1364"                                     
##  [671] "ibm-1364_P110-2007"                           
##  [672] "ibm-1371"                                     
##  [673] "ibm-1371_P100-1999"                           
##  [674] "ibm-1373"                                     
##  [675] "ibm-1373_P100-2002"                           
##  [676] "ibm-1375"                                     
##  [677] "ibm-1375_P100-2008"                           
##  [678] "ibm-1383"                                     
##  [679] "ibm-1383_P110-1999"                           
##  [680] "ibm-1383_VPUA"                                
##  [681] "ibm-1386"                                     
##  [682] "ibm-1386_P100-2001"                           
##  [683] "ibm-1386_VSUB_VPUA"                           
##  [684] "ibm-1388"                                     
##  [685] "ibm-1388_P103-2001"                           
##  [686] "ibm-1390"                                     
##  [687] "ibm-1390_P110-2003"                           
##  [688] "ibm-1392"                                     
##  [689] "ibm-1399"                                     
##  [690] "ibm-1399_P110-2003"                           
##  [691] "ibm-4517"                                     
##  [692] "ibm-4517_P100-2005"                           
##  [693] "ibm-4899"                                     
##  [694] "ibm-4899_P100-1998"                           
##  [695] "ibm-4902"                                     
##  [696] "ibm-4909"                                     
##  [697] "ibm-4909_P100-1999"                           
##  [698] "ibm-4971"                                     
##  [699] "ibm-4971_P100-1999"                           
##  [700] "ibm-5012"                                     
##  [701] "ibm-5012_P100-1999"                           
##  [702] "ibm-5026"                                     
##  [703] "ibm-5035"                                     
##  [704] "ibm-5050"                                     
##  [705] "ibm-5054"                                     
##  [706] "ibm-5123"                                     
##  [707] "ibm-5123_P100-1999"                           
##  [708] "ibm-5304"                                     
##  [709] "ibm-5305"                                     
##  [710] "ibm-5346"                                     
##  [711] "ibm-5346_P100-1998"                           
##  [712] "ibm-5347"                                     
##  [713] "ibm-5347_P100-1998"                           
##  [714] "ibm-5348"                                     
##  [715] "ibm-5348_P100-1997"                           
##  [716] "ibm-5349"                                     
##  [717] "ibm-5349_P100-1998"                           
##  [718] "ibm-5350"                                     
##  [719] "ibm-5350_P100-1998"                           
##  [720] "ibm-5351"                                     
##  [721] "ibm-5351_P100-1998"                           
##  [722] "ibm-5352"                                     
##  [723] "ibm-5352_P100-1998"                           
##  [724] "ibm-5353"                                     
##  [725] "ibm-5353_P100-1998"                           
##  [726] "ibm-5354"                                     
##  [727] "ibm-5354_P100-1998"                           
##  [728] "ibm-5471"                                     
##  [729] "ibm-5471_P100-2006"                           
##  [730] "ibm-5478"                                     
##  [731] "ibm-5478_P100-1995"                           
##  [732] "ibm-8482"                                     
##  [733] "ibm-8482_P100-1999"                           
##  [734] "ibm-9005"                                     
##  [735] "ibm-9005_X110-2007"                           
##  [736] "ibm-9030"                                     
##  [737] "ibm-9066"                                     
##  [738] "ibm-9067"                                     
##  [739] "ibm-9067_X100-2005"                           
##  [740] "ibm-9400"                                     
##  [741] "ibm-9424"                                     
##  [742] "ibm-9447"                                     
##  [743] "ibm-9447_P100-2002"                           
##  [744] "ibm-9448"                                     
##  [745] "ibm-9448_X100-2005"                           
##  [746] "ibm-9449"                                     
##  [747] "ibm-9449_P100-2002"                           
##  [748] "ibm-9580"                                     
##  [749] "ibm-12712"                                    
##  [750] "ibm-12712_P100-1998"                          
##  [751] "ibm-12712_P100-1998,swaplfnl"                 
##  [752] "ibm-12712-s390"                               
##  [753] "ibm-13488"                                    
##  [754] "ibm-13489"                                    
##  [755] "ibm-13490"                                    
##  [756] "ibm-13491"                                    
##  [757] "ibm-13496"                                    
##  [758] "ibm-13497"                                    
##  [759] "ibm-16684"                                    
##  [760] "ibm-16684_P110-2003"                          
##  [761] "ibm-16804"                                    
##  [762] "ibm-16804_X110-1999"                          
##  [763] "ibm-16804_X110-1999,swaplfnl"                 
##  [764] "ibm-16804-s390"                               
##  [765] "ibm-17584"                                    
##  [766] "ibm-17585"                                    
##  [767] "ibm-17586"                                    
##  [768] "ibm-17587"                                    
##  [769] "ibm-17592"                                    
##  [770] "ibm-17593"                                    
##  [771] "ibm-20780"                                    
##  [772] "ibm-21680"                                    
##  [773] "ibm-21681"                                    
##  [774] "ibm-21682"                                    
##  [775] "ibm-21683"                                    
##  [776] "ibm-25546"                                    
##  [777] "ibm-25776"                                    
##  [778] "ibm-25777"                                    
##  [779] "ibm-25778"                                    
##  [780] "ibm-25779"                                    
##  [781] "ibm-29872"                                    
##  [782] "ibm-29873"                                    
##  [783] "ibm-29874"                                    
##  [784] "ibm-29875"                                    
##  [785] "ibm-33722"                                    
##  [786] "ibm-33722_P12A_P12A-2009_U2"                  
##  [787] "ibm-33722_P120-1999"                          
##  [788] "ibm-33722_VASCII_VPUA"                        
##  [789] "ibm-33722_VPUA"                               
##  [790] "ibm-61955"                                    
##  [791] "ibm-61956"                                    
##  [792] "ibm-65025"                                    
##  [793] "ibm-eucCN"                                    
##  [794] "IBM-eucJP"                                    
##  [795] "ibm-eucKR"                                    
##  [796] "ibm-eucTW"                                    
##  [797] "IBM-Thai"                                     
##  [798] "IBM037"                                       
##  [799] "IBM273"                                       
##  [800] "IBM277"                                       
##  [801] "IBM278"                                       
##  [802] "IBM280"                                       
##  [803] "IBM284"                                       
##  [804] "IBM285"                                       
##  [805] "IBM290"                                       
##  [806] "IBM297"                                       
##  [807] "IBM367"                                       
##  [808] "IBM420"                                       
##  [809] "IBM424"                                       
##  [810] "IBM437"                                       
##  [811] "IBM500"                                       
##  [812] "IBM737"                                       
##  [813] "IBM775"                                       
##  [814] "IBM819"                                       
##  [815] "IBM838"                                       
##  [816] "IBM850"                                       
##  [817] "IBM851"                                       
##  [818] "IBM852"                                       
##  [819] "IBM855"                                       
##  [820] "IBM856"                                       
##  [821] "IBM857"                                       
##  [822] "IBM00858"                                     
##  [823] "IBM860"                                       
##  [824] "IBM861"                                       
##  [825] "IBM862"                                       
##  [826] "IBM863"                                       
##  [827] "IBM864"                                       
##  [828] "IBM865"                                       
##  [829] "IBM866"                                       
##  [830] "IBM868"                                       
##  [831] "IBM869"                                       
##  [832] "IBM870"                                       
##  [833] "IBM871"                                       
##  [834] "IBM875"                                       
##  [835] "IBM918"                                       
##  [836] "IBM922"                                       
##  [837] "IBM930"                                       
##  [838] "IBM939"                                       
##  [839] "IBM1006"                                      
##  [840] "IBM1026"                                      
##  [841] "IBM1047"                                      
##  [842] "IBM1047_LF"                                   
##  [843] "IBM1098"                                      
##  [844] "IBM01140"                                     
##  [845] "IBM01141"                                     
##  [846] "IBM1141_LF"                                   
##  [847] "IBM01142"                                     
##  [848] "IBM01143"                                     
##  [849] "IBM01144"                                     
##  [850] "IBM01145"                                     
##  [851] "IBM01146"                                     
##  [852] "IBM01147"                                     
##  [853] "IBM01148"                                     
##  [854] "IBM01149"                                     
##  [855] "IBM1153"                                      
##  [856] "IMAP-mailbox-name"                            
##  [857] "iscii-bng"                                    
##  [858] "iscii-dev"                                    
##  [859] "iscii-guj"                                    
##  [860] "iscii-gur"                                    
##  [861] "iscii-knd"                                    
##  [862] "iscii-mlm"                                    
##  [863] "iscii-ori"                                    
##  [864] "iscii-tlg"                                    
##  [865] "iscii-tml"                                    
##  [866] "ISCII,version=0"                              
##  [867] "ISCII,version=1"                              
##  [868] "ISCII,version=2"                              
##  [869] "ISCII,version=3"                              
##  [870] "ISCII,version=4"                              
##  [871] "ISCII,version=5"                              
##  [872] "ISCII,version=6"                              
##  [873] "ISCII,version=7"                              
##  [874] "ISCII,version=8"                              
##  [875] "iso_646.irv:1983"                             
##  [876] "ISO_646.irv:1991"                             
##  [877] "ISO_2022,locale=ja,version=0"                 
##  [878] "ISO_2022,locale=ja,version=1"                 
##  [879] "ISO_2022,locale=ja,version=2"                 
##  [880] "ISO_2022,locale=ja,version=3"                 
##  [881] "ISO_2022,locale=ja,version=4"                 
##  [882] "ISO_2022,locale=ko,version=0"                 
##  [883] "ISO_2022,locale=ko,version=1"                 
##  [884] "ISO_2022,locale=zh,version=0"                 
##  [885] "ISO_2022,locale=zh,version=1"                 
##  [886] "ISO_2022,locale=zh,version=2"                 
##  [887] "ISO_8859-1:1987"                              
##  [888] "ISO_8859-2:1987"                              
##  [889] "ISO_8859-3:1988"                              
##  [890] "ISO_8859-4:1988"                              
##  [891] "ISO_8859-5:1988"                              
##  [892] "ISO_8859-6:1987"                              
##  [893] "ISO_8859-7:1987"                              
##  [894] "ISO_8859-8:1988"                              
##  [895] "ISO_8859-9:1989"                              
##  [896] "ISO_8859-10:1992"                             
##  [897] "ISO_8859-14:1998"                             
##  [898] "ISO-2022-CN"                                  
##  [899] "ISO-2022-CN-CNS"                              
##  [900] "ISO-2022-CN-EXT"                              
##  [901] "ISO-2022-JP"                                  
##  [902] "ISO-2022-JP-1"                                
##  [903] "ISO-2022-JP-2"                                
##  [904] "ISO-2022-KR"                                  
##  [905] "iso-8859_10-1998"                             
##  [906] "iso-8859_11-2001"                             
##  [907] "iso-8859_14-1998"                             
##  [908] "ISO-8859-1"                                   
##  [909] "ISO-8859-2"                                   
##  [910] "ISO-8859-3"                                   
##  [911] "ISO-8859-4"                                   
##  [912] "ISO-8859-5"                                   
##  [913] "ISO-8859-6"                                   
##  [914] "ISO-8859-6-E"                                 
##  [915] "ISO-8859-6-I"                                 
##  [916] "ISO-8859-7"                                   
##  [917] "ISO-8859-8"                                   
##  [918] "ISO-8859-8-E"                                 
##  [919] "ISO-8859-8-I"                                 
##  [920] "ISO-8859-9"                                   
##  [921] "ISO-8859-10"                                  
##  [922] "ISO-8859-11"                                  
##  [923] "ISO-8859-13"                                  
##  [924] "ISO-8859-14"                                  
##  [925] "ISO-8859-15"                                  
##  [926] "ISO-10646-UCS-2"                              
##  [927] "ISO-10646-UCS-4"                              
##  [928] "iso-celtic"                                   
##  [929] "iso-ir-6"                                     
##  [930] "iso-ir-58"                                    
##  [931] "iso-ir-100"                                   
##  [932] "iso-ir-101"                                   
##  [933] "iso-ir-109"                                   
##  [934] "iso-ir-110"                                   
##  [935] "iso-ir-126"                                   
##  [936] "iso-ir-127"                                   
##  [937] "iso-ir-138"                                   
##  [938] "iso-ir-144"                                   
##  [939] "iso-ir-148"                                   
##  [940] "iso-ir-149"                                   
##  [941] "iso-ir-157"                                   
##  [942] "iso-ir-199"                                   
##  [943] "ISO646-US"                                    
##  [944] "iso8859_15_fdis"                              
##  [945] "JIS"                                          
##  [946] "JIS_Encoding"                                 
##  [947] "JIS7"                                         
##  [948] "JIS8"                                         
##  [949] "koi8"                                         
##  [950] "KOI8-R"                                       
##  [951] "KOI8-U"                                       
##  [952] "korean"                                       
##  [953] "KS_C_5601-1987"                               
##  [954] "KS_C_5601-1989"                               
##  [955] "ksc"                                          
##  [956] "KSC_5601"                                     
##  [957] "l1"                                           
##  [958] "l2"                                           
##  [959] "l3"                                           
##  [960] "l4"                                           
##  [961] "l5"                                           
##  [962] "l6"                                           
##  [963] "l8"                                           
##  [964] "l9"                                           
##  [965] "Latin-9"                                      
##  [966] "latin0"                                       
##  [967] "latin1"                                       
##  [968] "latin2"                                       
##  [969] "latin3"                                       
##  [970] "latin4"                                       
##  [971] "latin5"                                       
##  [972] "latin6"                                       
##  [973] "latin8"                                       
##  [974] "lmbcs"                                        
##  [975] "LMBCS-1"                                      
##  [976] "mac"                                          
##  [977] "mac-cyrillic"                                 
##  [978] "macce"                                        
##  [979] "maccentraleurope"                             
##  [980] "maccy"                                        
##  [981] "macgr"                                        
##  [982] "macintosh"                                    
##  [983] "macos-0_2-10.2"                               
##  [984] "macos-6_2-10.4"                               
##  [985] "macos-7_3-10.2"                               
##  [986] "macos-29-10.2"                                
##  [987] "macos-35-10.2"                                
##  [988] "macroman"                                     
##  [989] "mactr"                                        
##  [990] "MS_Kanji"                                     
##  [991] "MS874"                                        
##  [992] "ms932"                                        
##  [993] "MS936"                                        
##  [994] "ms949"                                        
##  [995] "ms950"                                        
##  [996] "MS950_HKSCS"                                  
##  [997] "PC-Multilingual-850+euro"                     
##  [998] "pck"                                          
##  [999] "r8"                                           
## [1000] "roman8"                                       
## [1001] "SCSU"                                         
## [1002] "Shift_JIS"                                    
## [1003] "shift_jis78"                                  
## [1004] "sjis"                                         
## [1005] "sjis78"                                       
## [1006] "sun_eu_greek"                                 
## [1007] "thai8"                                        
## [1008] "TIS-620"                                      
## [1009] "tis620.2533"                                  
## [1010] "turkish"                                      
## [1011] "turkish8"                                     
## [1012] "ucs-2"                                        
## [1013] "ucs-4"                                        
## [1014] "ujis"                                         
## [1015] "unicode"                                      
## [1016] "unicode-1-1-utf-7"                            
## [1017] "unicode-1-1-utf-8"                            
## [1018] "unicode-2-0-utf-7"                            
## [1019] "unicode-2-0-utf-8"                            
## [1020] "UnicodeBig"                                   
## [1021] "UnicodeBigUnmarked"                           
## [1022] "UnicodeLittle"                                
## [1023] "UnicodeLittleUnmarked"                        
## [1024] "us"                                           
## [1025] "US-ASCII"                                     
## [1026] "UTF-7"                                        
## [1027] "UTF-8"                                        
## [1028] "UTF-16"                                       
## [1029] "UTF-16,version=1"                             
## [1030] "UTF-16,version=2"                             
## [1031] "UTF-16BE"                                     
## [1032] "UTF-16BE,version=1"                           
## [1033] "UTF-16LE"                                     
## [1034] "UTF-16LE,version=1"                           
## [1035] "UTF-32"                                       
## [1036] "UTF-32BE"                                     
## [1037] "UTF-32LE"                                     
## [1038] "UTF16_BigEndian"                              
## [1039] "UTF16_LittleEndian"                           
## [1040] "UTF16_OppositeEndian"                         
## [1041] "UTF16_PlatformEndian"                         
## [1042] "UTF32_BigEndian"                              
## [1043] "UTF32_LittleEndian"                           
## [1044] "UTF32_OppositeEndian"                         
## [1045] "UTF32_PlatformEndian"                         
## [1046] "windows-31j"                                  
## [1047] "windows-437"                                  
## [1048] "windows-720"                                  
## [1049] "windows-737"                                  
## [1050] "windows-775"                                  
## [1051] "windows-850"                                  
## [1052] "windows-852"                                  
## [1053] "windows-855"                                  
## [1054] "windows-857"                                  
## [1055] "windows-858"                                  
## [1056] "windows-861"                                  
## [1057] "windows-862"                                  
## [1058] "windows-866"                                  
## [1059] "windows-869"                                  
## [1060] "windows-874"                                  
## [1061] "windows-874-2000"                             
## [1062] "windows-932"                                  
## [1063] "windows-936"                                  
## [1064] "windows-936-2000"                             
## [1065] "windows-949"                                  
## [1066] "windows-949-2000"                             
## [1067] "windows-950"                                  
## [1068] "windows-950-2000"                             
## [1069] "windows-1200"                                 
## [1070] "windows-1201"                                 
## [1071] "windows-1250"                                 
## [1072] "windows-1251"                                 
## [1073] "windows-1252"                                 
## [1074] "windows-1253"                                 
## [1075] "windows-1254"                                 
## [1076] "windows-1255"                                 
## [1077] "windows-1256"                                 
## [1078] "windows-1257"                                 
## [1079] "windows-1258"                                 
## [1080] "windows-10000"                                
## [1081] "windows-10006"                                
## [1082] "windows-10007"                                
## [1083] "windows-10029"                                
## [1084] "windows-10081"                                
## [1085] "windows-20127"                                
## [1086] "windows-20866"                                
## [1087] "windows-21866"                                
## [1088] "windows-28592"                                
## [1089] "windows-28593"                                
## [1090] "windows-28594"                                
## [1091] "windows-28595"                                
## [1092] "windows-28596"                                
## [1093] "windows-28597"                                
## [1094] "windows-28598"                                
## [1095] "windows-28599"                                
## [1096] "windows-28603"                                
## [1097] "windows-28605"                                
## [1098] "windows-51949"                                
## [1099] "windows-54936"                                
## [1100] "windows-57002"                                
## [1101] "windows-57003"                                
## [1102] "windows-57004"                                
## [1103] "windows-57005"                                
## [1104] "windows-57006"                                
## [1105] "windows-57007"                                
## [1106] "windows-57008"                                
## [1107] "windows-57009"                                
## [1108] "windows-57010"                                
## [1109] "windows-57011"                                
## [1110] "windows-65000"                                
## [1111] "windows-65001"                                
## [1112] "x-big5"                                       
## [1113] "x-compound-text"                              
## [1114] "X-EUC-JP"                                     
## [1115] "x-IBM720"                                     
## [1116] "x-IBM737"                                     
## [1117] "x-IBM856"                                     
## [1118] "x-IBM867"                                     
## [1119] "x-IBM874"                                     
## [1120] "x-IBM875"                                     
## [1121] "x-IBM921"                                     
## [1122] "x-IBM922"                                     
## [1123] "x-IBM930"                                     
## [1124] "x-IBM930A"                                    
## [1125] "x-IBM933"                                     
## [1126] "x-IBM935"                                     
## [1127] "x-IBM937"                                     
## [1128] "x-IBM939"                                     
## [1129] "x-IBM939A"                                    
## [1130] "x-IBM942"                                     
## [1131] "x-IBM942C"                                    
## [1132] "x-IBM943"                                     
## [1133] "x-IBM949"                                     
## [1134] "x-IBM949C"                                    
## [1135] "x-IBM950"                                     
## [1136] "x-IBM954"                                     
## [1137] "x-IBM954C"                                    
## [1138] "x-IBM964"                                     
## [1139] "x-IBM970"                                     
## [1140] "x-IBM971"                                     
## [1141] "x-IBM1006"                                    
## [1142] "x-IBM1025"                                    
## [1143] "x-IBM1097"                                    
## [1144] "x-IBM1098"                                    
## [1145] "x-IBM1112"                                    
## [1146] "x-IBM1122"                                    
## [1147] "x-IBM1123"                                    
## [1148] "x-IBM1124"                                    
## [1149] "x-IBM1153"                                    
## [1150] "x-IBM1363"                                    
## [1151] "x-IBM1363C"                                   
## [1152] "x-IBM1364"                                    
## [1153] "x-IBM1371"                                    
## [1154] "x-IBM1388"                                    
## [1155] "x-IBM1390"                                    
## [1156] "x-IBM1399"                                    
## [1157] "x-IBM33722"                                   
## [1158] "x-IBM33722A"                                  
## [1159] "x-IBM33722C"                                  
## [1160] "x-iscii-as"                                   
## [1161] "x-iscii-be"                                   
## [1162] "x-iscii-de"                                   
## [1163] "x-iscii-gu"                                   
## [1164] "x-iscii-ka"                                   
## [1165] "x-iscii-ma"                                   
## [1166] "x-iscii-or"                                   
## [1167] "x-iscii-pa"                                   
## [1168] "x-iscii-ta"                                   
## [1169] "x-iscii-te"                                   
## [1170] "x-ISCII91"                                    
## [1171] "x-ISO-2022-CN-CNS"                            
## [1172] "x-ISO-2022-CN-GB"                             
## [1173] "x-ISO-8859-6S"                                
## [1174] "x-iso-8859-11"                                
## [1175] "x-JISAutoDetect"                              
## [1176] "x-KSC5601"                                    
## [1177] "x-mac-ce"                                     
## [1178] "x-mac-centraleurroman"                        
## [1179] "x-mac-cyrillic"                               
## [1180] "x-mac-greek"                                  
## [1181] "x-mac-turkish"                                
## [1182] "x-MacCentralEurope"                           
## [1183] "x-MacCyrillic"                                
## [1184] "x-MacGreek"                                   
## [1185] "x-macroman"                                   
## [1186] "x-MacTurkish"                                 
## [1187] "x-MacUkraine"                                 
## [1188] "x-ms-cp932"                                   
## [1189] "x-MS932_0213"                                 
## [1190] "x-MS950-HKSCS"                                
## [1191] "x-roman8"                                     
## [1192] "x-sjis"                                       
## [1193] "x-UTF_8J"                                     
## [1194] "x-utf-16be"                                   
## [1195] "x-utf-16le"                                   
## [1196] "x-UTF-16LE-BOM"                               
## [1197] "x-windows-874"                                
## [1198] "x-windows-950"                                
## [1199] "x-windows-1256S"                              
## [1200] "x-windows-50220"                              
## [1201] "x-windows-50221"                              
## [1202] "x-windows-iso2022jp"                          
## [1203] "x11-compound-text"
stri_enc_list(FALSE)
## $`BOCU-1`
## [1] "BOCU-1"   "csBOCU-1" "ibm-1214" "ibm-1215"
## 
## $`CESU-8`
## [1] "CESU-8"   "ibm-9400"
## 
## $`ebcdic-xml-us`
## [1] "ebcdic-xml-us"
## 
## $`euc-jp-2007`
## [1] "csEUCPkdFmtJapanese"                          
## [2] "EUC-JP"                                       
## [3] "euc-jp-2007"                                  
## [4] "eucjis"                                       
## [5] "Extended_UNIX_Code_Packed_Format_for_Japanese"
## [6] "ujis"                                         
## [7] "X-EUC-JP"                                     
## 
## $`euc-tw-2014`
## [1] "EUC-TW"      "euc-tw-2014"
## 
## $gb18030
## [1] "gb18030"       "GB18030"       "ibm-1392"      "windows-54936"
## 
## $`gsm-03.38-2009`
## [1] "gsm-03.38-2009" "GSM0338"       
## 
## $HZ
## [1] "HZ"         "HZ-GB-2312"
## 
## $`ibm-37_P100-1995`
##  [1] "037"              "cp037"            "cp37"             "cpibm37"         
##  [5] "csIBM037"         "ebcdic-cp-ca"     "ebcdic-cp-nl"     "ebcdic-cp-us"    
##  [9] "ebcdic-cp-wt"     "ibm-37"           "ibm-037"          "ibm-37_P100-1995"
## [13] "IBM037"          
## 
## $`ibm-37_P100-1995,swaplfnl`
## [1] "ibm-37_P100-1995,swaplfnl" "ibm-37-s390"              
## 
## $`ibm-273_P100-1995`
## [1] "273"               "CP273"             "csIBM273"         
## [4] "ebcdic-de"         "ibm-273"           "ibm-273_P100-1995"
## [7] "IBM273"           
## 
## $`ibm-277_P100-1995`
## [1] "277"               "cp277"             "csIBM277"         
## [4] "EBCDIC-CP-DK"      "EBCDIC-CP-NO"      "ebcdic-dk"        
## [7] "ibm-277"           "ibm-277_P100-1995" "IBM277"           
## 
## $`ibm-278_P100-1995`
## [1] "278"               "cp278"             "csIBM278"         
## [4] "ebcdic-cp-fi"      "ebcdic-cp-se"      "ebcdic-sv"        
## [7] "ibm-278"           "ibm-278_P100-1995" "IBM278"           
## 
## $`ibm-280_P100-1995`
## [1] "280"               "CP280"             "csIBM280"         
## [4] "ebcdic-cp-it"      "ibm-280"           "ibm-280_P100-1995"
## [7] "IBM280"           
## 
## $`ibm-284_P100-1995`
## [1] "284"               "CP284"             "cpibm284"         
## [4] "csIBM284"          "ebcdic-cp-es"      "ibm-284"          
## [7] "ibm-284_P100-1995" "IBM284"           
## 
## $`ibm-285_P100-1995`
## [1] "285"               "CP285"             "cpibm285"         
## [4] "csIBM285"          "ebcdic-cp-gb"      "ebcdic-gb"        
## [7] "ibm-285"           "ibm-285_P100-1995" "IBM285"           
## 
## $`ibm-290_P100-1995`
## [1] "cp290"             "csIBM290"          "EBCDIC-JP-kana"   
## [4] "ibm-290"           "ibm-290_P100-1995" "IBM290"           
## 
## $`ibm-297_P100-1995`
## [1] "297"               "cp297"             "cpibm297"         
## [4] "csIBM297"          "ebcdic-cp-fr"      "ibm-297"          
## [7] "ibm-297_P100-1995" "IBM297"           
## 
## $`ibm-420_X120-1999`
## [1] "420"               "cp420"             "csIBM420"         
## [4] "ebcdic-cp-ar1"     "ibm-420"           "ibm-420_X120-1999"
## [7] "IBM420"           
## 
## $`ibm-424_P100-1995`
## [1] "424"               "cp424"             "csIBM424"         
## [4] "ebcdic-cp-he"      "ibm-424"           "ibm-424_P100-1995"
## [7] "IBM424"           
## 
## $`ibm-437_P100-1995`
## [1] "437"               "cp437"             "csPC8CodePage437" 
## [4] "ibm-437"           "ibm-437_P100-1995" "IBM437"           
## [7] "windows-437"      
## 
## $`ibm-500_P100-1995`
## [1] "500"               "CP500"             "csIBM500"         
## [4] "ebcdic-cp-be"      "ebcdic-cp-ch"      "ibm-500"          
## [7] "ibm-500_P100-1995" "IBM500"           
## 
## $`ibm-720_P100-1997`
## [1] "DOS-720"           "ibm-720"           "ibm-720_P100-1997"
## [4] "windows-720"       "x-IBM720"         
## 
## $`ibm-737_P100-1997`
## [1] "737"               "cp737"             "ibm-737"          
## [4] "ibm-737_P100-1997" "IBM737"            "windows-737"      
## [7] "x-IBM737"         
## 
## $`ibm-775_P100-1996`
## [1] "775"               "cp775"             "csPC775Baltic"    
## [4] "ibm-775"           "ibm-775_P100-1996" "IBM775"           
## [7] "windows-775"      
## 
## $`ibm-803_P100-1999`
## [1] "cp803"             "ibm-803"           "ibm-803_P100-1999"
## 
## $`ibm-813_P100-1995`
## [1] "813"               "cp813"             "ibm-813"          
## [4] "ibm-813_P100-1995"
## 
## $`ibm-838_P100-1995`
## [1] "838"               "cp838"             "csIBMThai"        
## [4] "ibm-838"           "ibm-838_P100-1995" "ibm-9030"         
## [7] "IBM-Thai"          "IBM838"           
## 
## $`ibm-850_P100-1995`
## [1] "850"                 "cp850"               "csPC850Multilingual"
## [4] "ibm-850"             "ibm-850_P100-1995"   "IBM850"             
## [7] "windows-850"        
## 
## $`ibm-851_P100-1995`
## [1] "851"               "cp851"             "csPC851"          
## [4] "ibm-851"           "ibm-851_P100-1995" "IBM851"           
## 
## $`ibm-852_P100-1995`
## [1] "852"               "cp852"             "csPCp852"         
## [4] "ibm-852"           "ibm-852_P100-1995" "IBM852"           
## [7] "windows-852"      
## 
## $`ibm-855_P100-1995`
## [1] "855"               "cp855"             "csIBM855"         
## [4] "csPCp855"          "ibm-855"           "ibm-855_P100-1995"
## [7] "IBM855"            "windows-855"      
## 
## $`ibm-856_P100-1995`
## [1] "856"               "cp856"             "ibm-856"          
## [4] "ibm-856_P100-1995" "IBM856"            "x-IBM856"         
## 
## $`ibm-857_P100-1995`
## [1] "857"               "cp857"             "csIBM857"         
## [4] "ibm-857"           "ibm-857_P100-1995" "IBM857"           
## [7] "windows-857"      
## 
## $`ibm-858_P100-1997`
## [1] "CCSID00858"               "CP00858"                 
## [3] "cp858"                    "ibm-858"                 
## [5] "ibm-858_P100-1997"        "IBM00858"                
## [7] "PC-Multilingual-850+euro" "windows-858"             
## 
## $`ibm-860_P100-1995`
## [1] "860"               "cp860"             "csIBM860"         
## [4] "ibm-860"           "ibm-860_P100-1995" "IBM860"           
## 
## $`ibm-861_P100-1995`
## [1] "861"               "cp-is"             "cp861"            
## [4] "csIBM861"          "ibm-861"           "ibm-861_P100-1995"
## [7] "IBM861"            "windows-861"      
## 
## $`ibm-862_P100-1995`
## [1] "862"                "cp862"              "csPC862LatinHebrew"
## [4] "DOS-862"            "ibm-862"            "ibm-862_P100-1995" 
## [7] "IBM862"             "windows-862"       
## 
## $`ibm-863_P100-1995`
## [1] "863"               "cp863"             "csIBM863"         
## [4] "ibm-863"           "ibm-863_P100-1995" "IBM863"           
## 
## $`ibm-864_X110-1999`
## [1] "cp864"             "csIBM864"          "ibm-864"          
## [4] "ibm-864_X110-1999" "IBM864"           
## 
## $`ibm-865_P100-1995`
## [1] "865"               "cp865"             "csIBM865"         
## [4] "ibm-865"           "ibm-865_P100-1995" "IBM865"           
## 
## $`ibm-866_P100-1995`
## [1] "866"               "cp866"             "csIBM866"         
## [4] "ibm-866"           "ibm-866_P100-1995" "IBM866"           
## [7] "windows-866"      
## 
## $`ibm-867_P100-1998`
## [1] "ibm-867"           "ibm-867_P100-1998" "x-IBM867"         
## 
## $`ibm-868_P100-1995`
## [1] "868"               "cp-ar"             "CP868"            
## [4] "csIBM868"          "ibm-868"           "ibm-868_P100-1995"
## [7] "IBM868"           
## 
## $`ibm-869_P100-1995`
## [1] "869"               "cp-gr"             "cp869"            
## [4] "csIBM869"          "ibm-869"           "ibm-869_P100-1995"
## [7] "IBM869"            "windows-869"      
## 
## $`ibm-870_P100-1995`
## [1] "CP870"             "csIBM870"          "ebcdic-cp-roece"  
## [4] "ebcdic-cp-yu"      "ibm-870"           "ibm-870_P100-1995"
## [7] "IBM870"           
## 
## $`ibm-871_P100-1995`
## [1] "871"               "CP871"             "csIBM871"         
## [4] "ebcdic-cp-is"      "ebcdic-is"         "ibm-871"          
## [7] "ibm-871_P100-1995" "IBM871"           
## 
## $`ibm-874_P100-1995`
## [1] "cp874"             "eucTH"             "ibm-874"          
## [4] "ibm-874_P100-1995" "ibm-9066"          "TIS-620"          
## [7] "tis620.2533"       "x-IBM874"         
## 
## $`ibm-875_P100-1995`
## [1] "875"               "cp875"             "ibm-875"          
## [4] "ibm-875_P100-1995" "IBM875"            "x-IBM875"         
## 
## $`ibm-878_P100-1996`
## [1] "cp878"             "csKOI8R"           "ibm-878"          
## [4] "ibm-878_P100-1996" "koi8"              "KOI8-R"           
## [7] "windows-20866"    
## 
## $`ibm-901_P100-1999`
## [1] "ibm-901"           "ibm-901_P100-1999"
## 
## $`ibm-902_P100-1999`
## [1] "ibm-902"           "ibm-902_P100-1999"
## 
## $`ibm-912_P100-1995`
##  [1] "912"               "8859_2"            "cp912"            
##  [4] "csISOLatin2"       "ibm-912"           "ibm-912_P100-1995"
##  [7] "ISO_8859-2:1987"   "ISO-8859-2"        "iso-ir-101"       
## [10] "l2"                "latin2"            "windows-28592"    
## 
## $`ibm-913_P100-2000`
##  [1] "913"               "8859_3"            "cp913"            
##  [4] "csISOLatin3"       "ibm-913"           "ibm-913_P100-2000"
##  [7] "ISO_8859-3:1988"   "ISO-8859-3"        "iso-ir-109"       
## [10] "l3"                "latin3"            "windows-28593"    
## 
## $`ibm-914_P100-1995`
##  [1] "914"               "8859_4"            "cp914"            
##  [4] "csISOLatin4"       "ibm-914"           "ibm-914_P100-1995"
##  [7] "ISO_8859-4:1988"   "ISO-8859-4"        "iso-ir-110"       
## [10] "l4"                "latin4"            "windows-28594"    
## 
## $`ibm-915_P100-1995`
##  [1] "915"                "8859_5"             "cp915"             
##  [4] "csISOLatinCyrillic" "cyrillic"           "ibm-915"           
##  [7] "ibm-915_P100-1995"  "ISO_8859-5:1988"    "ISO-8859-5"        
## [10] "iso-ir-144"         "windows-28595"     
## 
## $`ibm-916_P100-1995`
## [1] "916"               "cp916"             "ibm-916"          
## [4] "ibm-916_P100-1995"
## 
## $`ibm-918_P100-1995`
## [1] "CP918"             "csIBM918"          "ebcdic-cp-ar2"    
## [4] "ibm-918"           "ibm-918_P100-1995" "IBM918"           
## 
## $`ibm-920_P100-1995`
##  [1] "920"               "8859_9"            "cp920"            
##  [4] "csISOLatin5"       "ECMA-128"          "ibm-920"          
##  [7] "ibm-920_P100-1995" "ISO_8859-9:1989"   "ISO-8859-9"       
## [10] "iso-ir-148"        "l5"                "latin5"           
## [13] "turkish"           "turkish8"          "windows-28599"    
## 
## $`ibm-921_P100-1995`
## [1] "921"               "8859_13"           "cp921"            
## [4] "ibm-921"           "ibm-921_P100-1995" "ISO-8859-13"      
## [7] "windows-28603"     "x-IBM921"         
## 
## $`ibm-922_P100-1999`
## [1] "922"               "cp922"             "ibm-922"          
## [4] "ibm-922_P100-1999" "IBM922"            "x-IBM922"         
## 
## $`ibm-923_P100-1998`
##  [1] "923"               "8859_15"           "cp923"            
##  [4] "csisolatin0"       "csisolatin9"       "ibm-923"          
##  [7] "ibm-923_P100-1998" "ISO-8859-15"       "iso8859_15_fdis"  
## [10] "l9"                "Latin-9"           "latin0"           
## [13] "windows-28605"    
## 
## $`ibm-930_P120-1999`
## [1] "930"               "cp930"             "ibm-930"          
## [4] "ibm-930_P120-1999" "ibm-5026"          "IBM930"           
## [7] "x-IBM930"          "x-IBM930A"        
## 
## $`ibm-933_P110-1995`
## [1] "933"               "cp933"             "ibm-933"          
## [4] "ibm-933_P110-1995" "x-IBM933"         
## 
## $`ibm-935_P110-1999`
## [1] "935"               "cp935"             "ibm-935"          
## [4] "ibm-935_P110-1999" "x-IBM935"         
## 
## $`ibm-937_P110-1999`
## [1] "937"               "cp937"             "ibm-937"          
## [4] "ibm-937_P110-1999" "x-IBM937"         
## 
## $`ibm-939_P120-1999`
## [1] "939"               "cp939"             "ibm-931"          
## [4] "ibm-939"           "ibm-939_P120-1999" "ibm-5035"         
## [7] "IBM939"            "x-IBM939"          "x-IBM939A"        
## 
## $`ibm-942_P12A-1999`
##  [1] "cp932"             "ibm-932"           "ibm-932_VSUB_VPUA"
##  [4] "ibm-942"           "ibm-942_P12A-1999" "ibm-942_VSUB_VPUA"
##  [7] "shift_jis78"       "sjis78"            "x-IBM942"         
## [10] "x-IBM942C"        
## 
## $`ibm-943_P15A-2003`
##  [1] "cp932"             "cp943c"            "csShiftJIS"       
##  [4] "csWindows31J"      "ibm-943"           "ibm-943_P15A-2003"
##  [7] "ibm-943_VSUB_VPUA" "IBM-943C"          "MS_Kanji"         
## [10] "ms932"             "pck"               "Shift_JIS"        
## [13] "sjis"              "windows-31j"       "windows-932"      
## [16] "x-JISAutoDetect"   "x-ms-cp932"        "x-MS932_0213"     
## [19] "x-sjis"           
## 
## $`ibm-943_P130-1999`
## [1] "943"                      "cp943"                   
## [3] "ibm-943"                  "ibm-943_P130-1999"       
## [5] "ibm-943_VASCII_VSUB_VPUA" "Shift_JIS"               
## [7] "x-IBM943"                
## 
## $`ibm-949_P11A-1999`
## [1] "cp949c"            "ibm-949"           "ibm-949_P11A-1999"
## [4] "ibm-949_VSUB_VPUA" "IBM-949C"          "x-IBM949C"        
## 
## $`ibm-949_P110-1999`
## [1] "949"                      "cp949"                   
## [3] "ibm-949"                  "ibm-949_P110-1999"       
## [5] "ibm-949_VASCII_VSUB_VPUA" "x-IBM949"                
## 
## $`ibm-950_P110-1999`
## [1] "950"               "cp950"             "ibm-950"          
## [4] "ibm-950_P110-1999" "x-IBM950"         
## 
## $`ibm-954_P101-2007`
## [1] "ibm-954"           "ibm-954_P101-2007" "x-IBM954"         
## [4] "x-IBM954C"        
## 
## $`ibm-964_P110-1999`
## [1] "964"               "cns11643"          "cp964"            
## [4] "ibm-964"           "ibm-964_P110-1999" "ibm-964_VPUA"     
## [7] "ibm-eucTW"         "x-IBM964"         
## 
## $`ibm-970_P110_P110-2006_U2`
##  [1] "970"                       "5601"                     
##  [3] "cp970"                     "csEUCKR"                  
##  [5] "EUC-KR"                    "ibm-970"                  
##  [7] "ibm-970_P110_P110-2006_U2" "ibm-970_VPUA"             
##  [9] "ibm-eucKR"                 "KS_C_5601-1987"           
## [11] "KSC_5601"                  "windows-51949"            
## [13] "x-IBM970"                 
## 
## $`ibm-971_P100-1995`
## [1] "ibm-971"           "ibm-971_P100-1995" "ibm-971_VPUA"     
## [4] "x-IBM971"         
## 
## $`ibm-1006_P100-1995`
## [1] "1006"               "cp1006"             "ibm-1006"          
## [4] "ibm-1006_P100-1995" "IBM1006"            "x-IBM1006"         
## 
## $`ibm-1025_P100-1995`
## [1] "1025"               "cp1025"             "ibm-1025"          
## [4] "ibm-1025_P100-1995" "x-IBM1025"         
## 
## $`ibm-1026_P100-1995`
## [1] "1026"               "CP1026"             "csIBM1026"         
## [4] "ibm-1026"           "ibm-1026_P100-1995" "IBM1026"           
## 
## $`ibm-1047_P100-1995`
## [1] "1047"               "cp1047"             "ibm-1047"          
## [4] "ibm-1047_P100-1995" "IBM1047"           
## 
## $`ibm-1047_P100-1995,swaplfnl`
## [1] "ibm-1047_P100-1995,swaplfnl" "ibm-1047-s390"              
## [3] "IBM1047_LF"                 
## 
## $`ibm-1051_P100-1995`
## [1] "csHPRoman8"         "hp-roman8"          "ibm-1051"          
## [4] "ibm-1051_P100-1995" "r8"                 "roman8"            
## [7] "x-roman8"          
## 
## $`ibm-1089_P100-1995`
##  [1] "1089"               "8859_6"             "arabic"            
##  [4] "ASMO-708"           "cp1089"             "csISOLatinArabic"  
##  [7] "ECMA-114"           "ibm-1089"           "ibm-1089_P100-1995"
## [10] "ISO_8859-6:1987"    "ISO-8859-6"         "ISO-8859-6-E"      
## [13] "ISO-8859-6-I"       "iso-ir-127"         "windows-28596"     
## [16] "x-ISO-8859-6S"     
## 
## $`ibm-1097_P100-1995`
## [1] "1097"               "cp1097"             "ibm-1097"          
## [4] "ibm-1097_P100-1995" "x-IBM1097"         
## 
## $`ibm-1098_P100-1995`
## [1] "1098"               "cp1098"             "ibm-1098"          
## [4] "ibm-1098_P100-1995" "IBM1098"            "x-IBM1098"         
## 
## $`ibm-1112_P100-1995`
## [1] "1112"               "cp1112"             "ibm-1112"          
## [4] "ibm-1112_P100-1995" "x-IBM1112"         
## 
## $`ibm-1122_P100-1999`
## [1] "1122"               "cp1122"             "ibm-1122"          
## [4] "ibm-1122_P100-1999" "x-IBM1122"         
## 
## $`ibm-1123_P100-1995`
## [1] "1123"               "cp1123"             "ibm-1123"          
## [4] "ibm-1123_P100-1995" "x-IBM1123"         
## 
## $`ibm-1124_P100-1996`
## [1] "1124"               "cp1124"             "ibm-1124"          
## [4] "ibm-1124_P100-1996" "x-IBM1124"         
## 
## $`ibm-1125_P100-1997`
## [1] "cp1125"             "ibm-1125"           "ibm-1125_P100-1997"
## 
## $`ibm-1129_P100-1997`
## [1] "ibm-1129"           "ibm-1129_P100-1997"
## 
## $`ibm-1130_P100-1997`
## [1] "ibm-1130"           "ibm-1130_P100-1997"
## 
## $`ibm-1131_P100-1997`
## [1] "cp1131"             "ibm-1131"           "ibm-1131_P100-1997"
## 
## $`ibm-1132_P100-1998`
## [1] "ibm-1132"           "ibm-1132_P100-1998"
## 
## $`ibm-1133_P100-1997`
## [1] "ibm-1133"           "ibm-1133_P100-1997"
## 
## $`ibm-1137_P100-1999`
## [1] "ibm-1137"           "ibm-1137_P100-1999"
## 
## $`ibm-1140_P100-1997`
## [1] "CCSID01140"         "CP01140"            "cp1140"            
## [4] "ebcdic-us-37+euro"  "ibm-1140"           "ibm-1140_P100-1997"
## [7] "IBM01140"          
## 
## $`ibm-1140_P100-1997,swaplfnl`
## [1] "ibm-1140_P100-1997,swaplfnl" "ibm-1140-s390"              
## 
## $`ibm-1141_P100-1997`
## [1] "CCSID01141"         "CP01141"            "cp1141"            
## [4] "ebcdic-de-273+euro" "ibm-1141"           "ibm-1141_P100-1997"
## [7] "IBM01141"          
## 
## $`ibm-1141_P100-1997,swaplfnl`
## [1] "ibm-1141_P100-1997,swaplfnl" "ibm-1141-s390"              
## [3] "IBM1141_LF"                 
## 
## $`ibm-1142_P100-1997`
## [1] "CCSID01142"         "CP01142"            "cp1142"            
## [4] "ebcdic-dk-277+euro" "ebcdic-no-277+euro" "ibm-1142"          
## [7] "ibm-1142_P100-1997" "IBM01142"          
## 
## $`ibm-1142_P100-1997,swaplfnl`
## [1] "ibm-1142_P100-1997,swaplfnl" "ibm-1142-s390"              
## 
## $`ibm-1143_P100-1997`
## [1] "CCSID01143"         "CP01143"            "cp1143"            
## [4] "ebcdic-fi-278+euro" "ebcdic-se-278+euro" "ibm-1143"          
## [7] "ibm-1143_P100-1997" "IBM01143"          
## 
## $`ibm-1143_P100-1997,swaplfnl`
## [1] "ibm-1143_P100-1997,swaplfnl" "ibm-1143-s390"              
## 
## $`ibm-1144_P100-1997`
## [1] "CCSID01144"         "CP01144"            "cp1144"            
## [4] "ebcdic-it-280+euro" "ibm-1144"           "ibm-1144_P100-1997"
## [7] "IBM01144"          
## 
## $`ibm-1144_P100-1997,swaplfnl`
## [1] "ibm-1144_P100-1997,swaplfnl" "ibm-1144-s390"              
## 
## $`ibm-1145_P100-1997`
## [1] "CCSID01145"         "CP01145"            "cp1145"            
## [4] "ebcdic-es-284+euro" "ibm-1145"           "ibm-1145_P100-1997"
## [7] "IBM01145"          
## 
## $`ibm-1145_P100-1997,swaplfnl`
## [1] "ibm-1145_P100-1997,swaplfnl" "ibm-1145-s390"              
## 
## $`ibm-1146_P100-1997`
## [1] "CCSID01146"         "CP01146"            "cp1146"            
## [4] "ebcdic-gb-285+euro" "ibm-1146"           "ibm-1146_P100-1997"
## [7] "IBM01146"          
## 
## $`ibm-1146_P100-1997,swaplfnl`
## [1] "ibm-1146_P100-1997,swaplfnl" "ibm-1146-s390"              
## 
## $`ibm-1147_P100-1997`
## [1] "CCSID01147"         "CP01147"            "cp1147"            
## [4] "ebcdic-fr-297+euro" "ibm-1147"           "ibm-1147_P100-1997"
## [7] "IBM01147"          
## 
## $`ibm-1147_P100-1997,swaplfnl`
## [1] "ibm-1147_P100-1997,swaplfnl" "ibm-1147-s390"              
## 
## $`ibm-1148_P100-1997`
## [1] "CCSID01148"                    "CP01148"                      
## [3] "cp1148"                        "ebcdic-international-500+euro"
## [5] "ibm-1148"                      "ibm-1148_P100-1997"           
## [7] "IBM01148"                     
## 
## $`ibm-1148_P100-1997,swaplfnl`
## [1] "ibm-1148_P100-1997,swaplfnl" "ibm-1148-s390"              
## 
## $`ibm-1149_P100-1997`
## [1] "CCSID01149"         "CP01149"            "cp1149"            
## [4] "ebcdic-is-871+euro" "ibm-1149"           "ibm-1149_P100-1997"
## [7] "IBM01149"          
## 
## $`ibm-1149_P100-1997,swaplfnl`
## [1] "ibm-1149_P100-1997,swaplfnl" "ibm-1149-s390"              
## 
## $`ibm-1153_P100-1999`
## [1] "ibm-1153"           "ibm-1153_P100-1999" "IBM1153"           
## [4] "x-IBM1153"         
## 
## $`ibm-1153_P100-1999,swaplfnl`
## [1] "ibm-1153_P100-1999,swaplfnl" "ibm-1153-s390"              
## 
## $`ibm-1154_P100-1999`
## [1] "ibm-1154"           "ibm-1154_P100-1999"
## 
## $`ibm-1155_P100-1999`
## [1] "ibm-1155"           "ibm-1155_P100-1999"
## 
## $`ibm-1156_P100-1999`
## [1] "ibm-1156"           "ibm-1156_P100-1999"
## 
## $`ibm-1157_P100-1999`
## [1] "ibm-1157"           "ibm-1157_P100-1999"
## 
## $`ibm-1158_P100-1999`
## [1] "ibm-1158"           "ibm-1158_P100-1999"
## 
## $`ibm-1160_P100-1999`
## [1] "ibm-1160"           "ibm-1160_P100-1999"
## 
## $`ibm-1162_P100-1999`
## [1] "ibm-1162"           "ibm-1162_P100-1999"
## 
## $`ibm-1164_P100-1999`
## [1] "ibm-1164"           "ibm-1164_P100-1999"
## 
## $`ibm-1168_P100-2002`
## [1] "ibm-1168"           "ibm-1168_P100-2002" "KOI8-U"            
## [4] "windows-21866"     
## 
## $`ibm-1250_P100-1995`
## [1] "ibm-1250"           "ibm-1250_P100-1995" "windows-1250"      
## 
## $`ibm-1251_P100-1995`
## [1] "ibm-1251"           "ibm-1251_P100-1995" "windows-1251"      
## 
## $`ibm-1252_P100-2000`
## [1] "ibm-1252"           "ibm-1252_P100-2000" "windows-1252"      
## 
## $`ibm-1253_P100-1995`
## [1] "ibm-1253"           "ibm-1253_P100-1995" "windows-1253"      
## 
## $`ibm-1254_P100-1995`
## [1] "ibm-1254"           "ibm-1254_P100-1995" "windows-1254"      
## 
## $`ibm-1255_P100-1995`
## [1] "ibm-1255"           "ibm-1255_P100-1995"
## 
## $`ibm-1256_P110-1997`
## [1] "ibm-1256"           "ibm-1256_P110-1997"
## 
## $`ibm-1257_P100-1995`
## [1] "ibm-1257"           "ibm-1257_P100-1995"
## 
## $`ibm-1258_P100-1997`
## [1] "ibm-1258"           "ibm-1258_P100-1997" "windows-1258"      
## 
## $`ibm-1276_P100-1995`
## [1] "Adobe-Standard-Encoding" "csAdobeStandardEncoding"
## [3] "ibm-1276"                "ibm-1276_P100-1995"     
## 
## $`ibm-1363_P11B-1998`
##  [1] "5601"               "cp1363"             "csKSC56011987"     
##  [4] "ibm-1363"           "ibm-1363_P11B-1998" "ibm-1363_VSUB_VPUA"
##  [7] "iso-ir-149"         "korean"             "KS_C_5601-1987"    
## [10] "KS_C_5601-1989"     "ksc"                "KSC_5601"          
## [13] "windows-949"        "x-IBM1363C"        
## 
## $`ibm-1363_P110-1997`
## [1] "ibm-1363"                  "ibm-1363_P110-1997"       
## [3] "ibm-1363_VASCII_VSUB_VPUA" "x-IBM1363"                
## 
## $`ibm-1364_P110-2007`
## [1] "ibm-1364"           "ibm-1364_P110-2007" "x-IBM1364"         
## 
## $`ibm-1371_P100-1999`
## [1] "ibm-1371"           "ibm-1371_P100-1999" "x-IBM1371"         
## 
## $`ibm-1373_P100-2002`
## [1] "ibm-1373"           "ibm-1373_P100-2002" "windows-950"       
## 
## $`ibm-1375_P100-2008`
## [1] "Big5-HKSCS"         "big5hk"             "HKSCS-BIG5"        
## [4] "ibm-1375"           "ibm-1375_P100-2008"
## 
## $`ibm-1383_P110-1999`
##  [1] "1383"               "cp1383"             "csGB2312"          
##  [4] "EUC-CN"             "GB2312"             "hp15CN"            
##  [7] "ibm-1383"           "ibm-1383_P110-1999" "ibm-1383_VPUA"     
## [10] "ibm-eucCN"         
## 
## $`ibm-1386_P100-2001`
## [1] "cp1386"             "ibm-1386"           "ibm-1386_P100-2001"
## [4] "ibm-1386_VSUB_VPUA" "windows-936"       
## 
## $`ibm-1388_P103-2001`
## [1] "ibm-1388"           "ibm-1388_P103-2001" "ibm-9580"          
## [4] "x-IBM1388"         
## 
## $`ibm-1390_P110-2003`
## [1] "ibm-1390"           "ibm-1390_P110-2003" "x-IBM1390"         
## 
## $`ibm-1399_P110-2003`
## [1] "ibm-1399"           "ibm-1399_P110-2003" "x-IBM1399"         
## 
## $`ibm-4517_P100-2005`
## [1] "ibm-4517"           "ibm-4517_P100-2005"
## 
## $`ibm-4899_P100-1998`
## [1] "ibm-4899"           "ibm-4899_P100-1998"
## 
## $`ibm-4909_P100-1999`
## [1] "ibm-4909"           "ibm-4909_P100-1999"
## 
## $`ibm-4971_P100-1999`
## [1] "ibm-4971"           "ibm-4971_P100-1999"
## 
## $`ibm-5012_P100-1999`
##  [1] "8859_8"             "csISOLatinHebrew"   "hebrew"            
##  [4] "hebrew8"            "ibm-5012"           "ibm-5012_P100-1999"
##  [7] "ISO_8859-8:1988"    "ISO-8859-8"         "ISO-8859-8-E"      
## [10] "ISO-8859-8-I"       "iso-ir-138"         "windows-28598"     
## 
## $`ibm-5123_P100-1999`
## [1] "ibm-5123"           "ibm-5123_P100-1999"
## 
## $`ibm-5346_P100-1998`
## [1] "cp1250"             "ibm-5346"           "ibm-5346_P100-1998"
## [4] "windows-1250"      
## 
## $`ibm-5347_P100-1998`
## [1] "ANSI1251"           "cp1251"             "ibm-5347"          
## [4] "ibm-5347_P100-1998" "windows-1251"      
## 
## $`ibm-5348_P100-1997`
## [1] "cp1252"             "ibm-5348"           "ibm-5348_P100-1997"
## [4] "windows-1252"      
## 
## $`ibm-5349_P100-1998`
## [1] "cp1253"             "ibm-5349"           "ibm-5349_P100-1998"
## [4] "windows-1253"      
## 
## $`ibm-5350_P100-1998`
## [1] "cp1254"             "ibm-5350"           "ibm-5350_P100-1998"
## [4] "windows-1254"      
## 
## $`ibm-5351_P100-1998`
## [1] "ibm-5351"           "ibm-5351_P100-1998" "windows-1255"      
## 
## $`ibm-5352_P100-1998`
## [1] "ibm-5352"           "ibm-5352_P100-1998" "windows-1256"      
## 
## $`ibm-5353_P100-1998`
## [1] "ibm-5353"           "ibm-5353_P100-1998" "windows-1257"      
## 
## $`ibm-5354_P100-1998`
## [1] "cp1258"             "ibm-5354"           "ibm-5354_P100-1998"
## [4] "windows-1258"      
## 
## $`ibm-5471_P100-2006`
## [1] "Big5-HKSCS"            "big5-hkscs:unicode3.0" "hkbig5"               
## [4] "ibm-5471"              "ibm-5471_P100-2006"    "MS950_HKSCS"          
## [7] "x-MS950-HKSCS"        
## 
## $`ibm-5478_P100-1995`
## [1] "chinese"            "csISO58GB231280"    "GB_2312-80"        
## [4] "gb2312-1980"        "GB2312.1980-0"      "ibm-5478"          
## [7] "ibm-5478_P100-1995" "iso-ir-58"         
## 
## $`ibm-8482_P100-1999`
## [1] "ibm-8482"           "ibm-8482_P100-1999"
## 
## $`ibm-9005_X110-2007`
##  [1] "8859_7"             "csISOLatinGreek"    "ECMA-118"          
##  [4] "ELOT_928"           "greek"              "greek8"            
##  [7] "ibm-9005"           "ibm-9005_X110-2007" "ISO_8859-7:1987"   
## [10] "ISO-8859-7"         "iso-ir-126"         "sun_eu_greek"      
## [13] "windows-28597"     
## 
## $`ibm-9067_X100-2005`
## [1] "ibm-9067"           "ibm-9067_X100-2005"
## 
## $`ibm-9447_P100-2002`
## [1] "cp1255"             "ibm-9447"           "ibm-9447_P100-2002"
## [4] "windows-1255"      
## 
## $`ibm-9448_X100-2005`
## [1] "cp1256"             "ibm-9448"           "ibm-9448_X100-2005"
## [4] "windows-1256"       "x-windows-1256S"   
## 
## $`ibm-9449_P100-2002`
## [1] "cp1257"             "ibm-9449"           "ibm-9449_P100-2002"
## [4] "windows-1257"      
## 
## $`ibm-12712_P100-1998`
## [1] "ebcdic-he"           "ibm-12712"           "ibm-12712_P100-1998"
## 
## $`ibm-12712_P100-1998,swaplfnl`
## [1] "ibm-12712_P100-1998,swaplfnl" "ibm-12712-s390"              
## 
## $`ibm-16684_P110-2003`
## [1] "ibm-16684"           "ibm-16684_P110-2003" "ibm-20780"          
## 
## $`ibm-16804_X110-1999`
## [1] "ebcdic-ar"           "ibm-16804"           "ibm-16804_X110-1999"
## 
## $`ibm-16804_X110-1999,swaplfnl`
## [1] "ibm-16804_X110-1999,swaplfnl" "ibm-16804-s390"              
## 
## $`ibm-33722_P12A_P12A-2009_U2`
## [1] "ibm-5050"                    "ibm-33722"                  
## [3] "ibm-33722_P12A_P12A-2009_U2" "ibm-33722_VPUA"             
## [5] "IBM-eucJP"                  
## 
## $`ibm-33722_P120-1999`
## [1] "33722"                 "cp33722"               "ibm-5050"             
## [4] "ibm-33722"             "ibm-33722_P120-1999"   "ibm-33722_VASCII_VPUA"
## [7] "x-IBM33722"            "x-IBM33722A"           "x-IBM33722C"          
## 
## $`IMAP-mailbox-name`
## [1] "IMAP-mailbox-name"
## 
## $`ISCII,version=0`
## [1] "ibm-4902"        "iscii-dev"       "ISCII,version=0" "windows-57002"  
## [5] "x-iscii-de"      "x-ISCII91"      
## 
## $`ISCII,version=1`
## [1] "iscii-bng"       "ISCII,version=1" "windows-57003"   "windows-57006"  
## [5] "x-iscii-as"      "x-iscii-be"     
## 
## $`ISCII,version=2`
## [1] "iscii-gur"       "ISCII,version=2" "windows-57011"   "x-iscii-pa"     
## 
## $`ISCII,version=3`
## [1] "iscii-guj"       "ISCII,version=3" "windows-57010"   "x-iscii-gu"     
## 
## $`ISCII,version=4`
## [1] "iscii-ori"       "ISCII,version=4" "windows-57007"   "x-iscii-or"     
## 
## $`ISCII,version=5`
## [1] "iscii-tml"       "ISCII,version=5" "windows-57004"   "x-iscii-ta"     
## 
## $`ISCII,version=6`
## [1] "iscii-tlg"       "ISCII,version=6" "windows-57005"   "x-iscii-te"     
## 
## $`ISCII,version=7`
## [1] "iscii-knd"       "ISCII,version=7" "windows-57008"   "x-iscii-ka"     
## 
## $`ISCII,version=8`
## [1] "iscii-mlm"       "ISCII,version=8" "windows-57009"   "x-iscii-ma"     
## 
## $`ISO_2022,locale=ja,version=0`
## [1] "csISO2022JP"                  "ISO_2022,locale=ja,version=0"
## [3] "ISO-2022-JP"                  "x-windows-50220"             
## [5] "x-windows-iso2022jp"         
## 
## $`ISO_2022,locale=ja,version=1`
## [1] "csJISEncoding"                "ibm-5054"                    
## [3] "ISO_2022,locale=ja,version=1" "ISO-2022-JP-1"               
## [5] "JIS"                          "JIS_Encoding"                
## [7] "x-windows-50221"             
## 
## $`ISO_2022,locale=ja,version=2`
## [1] "csISO2022JP2"                 "ISO_2022,locale=ja,version=2"
## [3] "ISO-2022-JP-2"               
## 
## $`ISO_2022,locale=ja,version=3`
## [1] "ISO_2022,locale=ja,version=3" "JIS7"                        
## 
## $`ISO_2022,locale=ja,version=4`
## [1] "ISO_2022,locale=ja,version=4" "JIS8"                        
## 
## $`ISO_2022,locale=ko,version=0`
## [1] "csISO2022KR"                  "ISO_2022,locale=ko,version=0"
## [3] "ISO-2022-KR"                 
## 
## $`ISO_2022,locale=ko,version=1`
## [1] "ibm-25546"                    "ISO_2022,locale=ko,version=1"
## 
## $`ISO_2022,locale=zh,version=0`
## [1] "csISO2022CN"                  "ISO_2022,locale=zh,version=0"
## [3] "ISO-2022-CN"                  "x-ISO-2022-CN-GB"            
## 
## $`ISO_2022,locale=zh,version=1`
## [1] "ISO_2022,locale=zh,version=1" "ISO-2022-CN-EXT"             
## 
## $`ISO_2022,locale=zh,version=2`
## [1] "ISO_2022,locale=zh,version=2" "ISO-2022-CN-CNS"             
## [3] "x-ISO-2022-CN-CNS"           
## 
## $`iso-8859_10-1998`
## [1] "csISOLatin6"      "ISO_8859-10:1992" "iso-8859_10-1998" "ISO-8859-10"     
## [5] "iso-ir-157"       "l6"               "latin6"          
## 
## $`iso-8859_11-2001`
## [1] "iso-8859_11-2001" "ISO-8859-11"      "thai8"            "x-iso-8859-11"   
## 
## $`iso-8859_14-1998`
## [1] "ISO_8859-14:1998" "iso-8859_14-1998" "ISO-8859-14"      "iso-celtic"      
## [5] "iso-ir-199"       "l8"               "latin8"          
## 
## $`ISO-8859-1`
##  [1] "819"             "8859_1"          "cp819"           "csISOLatin1"    
##  [5] "ibm-819"         "IBM819"          "ISO_8859-1:1987" "ISO-8859-1"     
##  [9] "iso-ir-100"      "l1"              "latin1"         
## 
## $`LMBCS-1`
## [1] "ibm-65025" "lmbcs"     "LMBCS-1"  
## 
## $`macos-0_2-10.2`
## [1] "csMacintosh"    "mac"            "macintosh"      "macos-0_2-10.2"
## [5] "macroman"       "windows-10000"  "x-macroman"    
## 
## $`macos-6_2-10.4`
## [1] "macgr"          "macos-6_2-10.4" "windows-10006"  "x-mac-greek"   
## [5] "x-MacGreek"    
## 
## $`macos-7_3-10.2`
## [1] "mac-cyrillic"   "maccy"          "macos-7_3-10.2" "windows-10007" 
## [5] "x-mac-cyrillic" "x-MacCyrillic"  "x-MacUkraine"  
## 
## $`macos-29-10.2`
## [1] "macce"                 "maccentraleurope"      "macos-29-10.2"        
## [4] "windows-10029"         "x-mac-ce"              "x-mac-centraleurroman"
## [7] "x-MacCentralEurope"   
## 
## $`macos-35-10.2`
## [1] "macos-35-10.2" "mactr"         "windows-10081" "x-mac-turkish"
## [5] "x-MacTurkish" 
## 
## $SCSU
## [1] "ibm-1212" "ibm-1213" "SCSU"    
## 
## $`US-ASCII`
##  [1] "646"              "ANSI_X3.4-1968"   "ANSI_X3.4-1986"   "ASCII"           
##  [5] "ascii7"           "cp367"            "csASCII"          "ibm-367"         
##  [9] "IBM367"           "iso_646.irv:1983" "ISO_646.irv:1991" "iso-ir-6"        
## [13] "ISO646-US"        "us"               "US-ASCII"         "windows-20127"   
## 
## $`UTF-7`
## [1] "unicode-1-1-utf-7" "unicode-2-0-utf-7" "UTF-7"            
## [4] "windows-65000"    
## 
## $`UTF-8`
##  [1] "cp1208"            "ibm-1208"          "ibm-1209"         
##  [4] "ibm-5304"          "ibm-5305"          "ibm-13496"        
##  [7] "ibm-13497"         "ibm-17592"         "ibm-17593"        
## [10] "unicode-1-1-utf-8" "unicode-2-0-utf-8" "UTF-8"            
## [13] "windows-65001"     "x-UTF_8J"         
## 
## $`UTF-16`
## [1] "csUnicode"       "ibm-1204"        "ibm-1205"        "ISO-10646-UCS-2"
## [5] "ucs-2"           "unicode"         "UTF-16"         
## 
## $`UTF-16,version=1`
## [1] "UTF-16,version=1"
## 
## $`UTF-16,version=2`
## [1] "UTF-16,version=2"
## 
## $`UTF-16BE`
##  [1] "cp1200"             "cp1201"             "ibm-1200"          
##  [4] "ibm-1201"           "ibm-13488"          "ibm-13489"         
##  [7] "ibm-17584"          "ibm-17585"          "ibm-21680"         
## [10] "ibm-21681"          "ibm-25776"          "ibm-25777"         
## [13] "ibm-29872"          "ibm-29873"          "ibm-61955"         
## [16] "ibm-61956"          "UnicodeBigUnmarked" "UTF-16BE"          
## [19] "UTF16_BigEndian"    "windows-1201"       "x-utf-16be"        
## 
## $`UTF-16BE,version=1`
## [1] "UnicodeBig"         "UTF-16BE,version=1"
## 
## $`UTF-16LE`
##  [1] "ibm-1202"              "ibm-1203"              "ibm-13490"            
##  [4] "ibm-13491"             "ibm-17586"             "ibm-17587"            
##  [7] "ibm-21682"             "ibm-21683"             "ibm-25778"            
## [10] "ibm-25779"             "ibm-29874"             "ibm-29875"            
## [13] "UnicodeLittleUnmarked" "UTF-16LE"              "UTF16_LittleEndian"   
## [16] "windows-1200"          "x-utf-16le"           
## 
## $`UTF-16LE,version=1`
## [1] "UnicodeLittle"      "UTF-16LE,version=1" "x-UTF-16LE-BOM"    
## 
## $`UTF-32`
## [1] "csUCS4"          "ibm-1236"        "ibm-1237"        "ISO-10646-UCS-4"
## [5] "ucs-4"           "UTF-32"         
## 
## $`UTF-32BE`
## [1] "ibm-1232"        "ibm-1233"        "ibm-9424"        "UTF-32BE"       
## [5] "UTF32_BigEndian"
## 
## $`UTF-32LE`
## [1] "ibm-1234"           "ibm-1235"           "UTF-32LE"          
## [4] "UTF32_LittleEndian"
## 
## $UTF16_OppositeEndian
## [1] "UTF16_OppositeEndian"
## 
## $UTF16_PlatformEndian
## [1] "UTF16_PlatformEndian"
## 
## $UTF32_OppositeEndian
## [1] "UTF32_OppositeEndian"
## 
## $UTF32_PlatformEndian
## [1] "UTF32_PlatformEndian"
## 
## $`windows-874-2000`
## [1] "MS874"            "TIS-620"          "windows-874"      "windows-874-2000"
## [5] "x-windows-874"   
## 
## $`windows-936-2000`
## [1] "CP936"            "GBK"              "MS936"            "windows-936"     
## [5] "windows-936-2000"
## 
## $`windows-949-2000`
##  [1] "csKSC56011987"    "iso-ir-149"       "korean"           "KS_C_5601-1987"  
##  [5] "KS_C_5601-1989"   "KSC_5601"         "ms949"            "windows-949"     
##  [9] "windows-949-2000" "x-KSC5601"       
## 
## $`windows-950-2000`
## [1] "Big5"             "csBig5"           "ms950"            "windows-950"     
## [5] "windows-950-2000" "x-big5"           "x-windows-950"   
## 
## $`x11-compound-text`
## [1] "COMPOUND_TEXT"     "x-compound-text"   "x11-compound-text"
```
