c######################################################################
c
c STEADY-STATE SOLVER - banded or full Jacobian
c
c FINDS THE ROOT OF A SET OF NONLINEAR EQUATIONS               
c implementation: karline Soetaert, NIOZ, the Netherlands
c
c######################################################################

c                !<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<!
c                !<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<!
c                !       SOLVING STEADY-STATE         !
c                !>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>!
c                !>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>!

c**********************************************************************
     
       SUBROUTINE dSteady(xmodel,N,numabd,time,Svar,beta,alpha,                &
     &                    ImplicitMethod,Bandup,BandDown,                      &
     &                    MaxIter,TolChange,atol,rtol,                         &
     &                    itol,jac,PositivityInt,Pos, ipos,                    &
     &                    SteadyStateReachedInt,delt,copyvar,ewt,              &
     &                    xindx,precis,niter,out,nout)
c------------------------------------------------------------------------------*
c Solves a system of nonlinear equations using the Newton-Raphson method       * 
c assumes a full or banded Jacobian                                            *
c------------------------------------------------------------------------------*  
       IMPLICIT NONE
       
c number of equations, num rows in jacobian, structure of jacobian and
c upper and lower half-bandwidth
       INTEGER N, NumABD, ImplicitMethod, Bandup,BandDown    

c max and actual iterations
       INTEGER MaxIter, niter

c state variables 
       DOUBLE PRECISION Svar(N) 

c Beta : the negative of the rate of change,
       DOUBLE PRECISION BETA(N)
       
c tolerances, precision
       INTEGER          itol
       DOUBLE PRECISION TolChange, atol(*),rtol(*) 
       DOUBLE PRECISION ewt(N), maxewt, RelativeChange,precis(MaxIter)

c false if failed - true if variables must be positive
c positivity either enforced at once (positivity=TRUE) or as a vector of elements
       LOGICAL SteadyStateReached, positivity
       INTEGER SteadyStateReachedInt, positivityInt
       INTEGER  Ipos, Pos(Ipos)

c model and jacobian function
       EXTERNAL xmodel, jac  
       DOUBLE PRECISION out(*)      
       INTEGER          nout(*) 
       DOUBLE PRECISION time
       
c 
       INTEGER          I, K, Info, idum(1)
       INTEGER          xIndx(N)

c Two types of jacobians used in the newton-Raphson method:
c (1) Diagonal structure: non-zero elements are stored in an array alpha
c (2) Not banded        : full matrix alpha; maximal dimensions NumSVar 

       DOUBLE PRECISION Alpha (NumABD,N)

c working arrays for solver
       DOUBLE PRECISION Delt(N), copyvar(N)

c Internal acronyms for type of jacobian
       INTEGER FullJacUser,FullJacIntern,BandJacUser,BandJacIntern

       PARAMETER (FullJacUser = 21,  FullJacIntern   = 22,                     &   
     &            BandJacUser = 24,  BandJacIntern   = 25)
c-------------------------------------------------------------------------------

       SteadyStateReached = .FALSE.     ! Steady state not yet reached
       SteadyStateReachedInt = 0
       Positivity = .FALSE.
       IF (positivityInt > 0.1) Positivity = .TRUE.

       DO K=1,MaxIter
          niter = K
c the error weight vector EWT 
            CALL errSet(N,itol,rtol,atol,Svar,ewt)
            
c Generate jacobian, several types   
c full or banded, generated by user-defined jac or internally by perturbation, 
          IF (ImplicitMethod .EQ. FullJacIntern) THEN
            CALL XFULLJACOB  (N,BETA,Alpha,Svar,copyvar,ewt,xmodel,            &
     &                        time, out, nout)

          ELSE IF (ImplicitMethod .EQ. BandJacIntern) THEN
            CALL XBandJACOB  (BandUp, BandDown, N,                             &
     &                        NumABD, BETA, alpha,copyvar, Delt,               &
     &                        Svar,ewt, xmodel,time, out, nout)
       
          ELSE IF (ImplicitMethod .EQ. FullJacUser) THEN
            CALL JAC(N,time,Svar,0,0,Alpha,N,out,nout)
            CALL XMODEL(N,time,Svar,beta,out,nout)
            Beta = -beta

          ELSE IF (ImplicitMethod .EQ. BandJacUser) THEN
            CALL JAC(N,time,Svar,0,0,Alpha,numabd,out,nout)
            CALL XMODEL(N,time,Svar,beta,out,nout)
            Beta = -beta
           
          END IF

c Check convergence 
          precis(K) = 0.d0
          maxewt    = 0.d0
          DO I=1, N
            precis(K) = precis(K)+abs(BETA(I))
            maxewt = MAX(maxewt, abs(BETA(I)/ewt(i)))
          ENDDO
          precis(K) = precis(K)/N
          IF(maxewt .LE. 1) THEN
              SteadyStateReached = .TRUE.
              SteadyStateReachedInt = 1
              EXIT 
          ENDIF

c Invert jacobian 

          IF (ImplicitMethod .EQ. FullJacIntern .OR.                           &
     &        ImplicitMethod .EQ. FullJacUser) THEN
              CALL DGEFA(ALPHA,N,N,xIndx,info)
              IF (info .NE. 0) THEN
               call rwarn("error during factorisation of matrix (dgefa);       &
     &         singular matrix")
                 idum(1) = info
                 call intpr("diagonal element is zero ",-1,idum,1)
                 EXIT
              ENDIF
              CALL DGESL(ALPHA,N,N,xIndx,Beta,info)

          ELSE IF (ImplicitMethod .EQ. BandJacIntern .OR.                      &
     &             ImplicitMethod .EQ. BandJacUser)  THEN
     
              CALL dgbfa(alpha,NumABD,N,BandDown,BandUp,xindx,                 &
     &                   info)
              IF (info .ne. 0) THEN
               call rwarn("error during factorisation of matrix (dgbfa);       &
     &           singular matrix")
                 idum(1) = info
                 call intpr ("diagonal element is zero", -1, idum, 1)
                 EXIT
              ENDIF  
              CALL dgbsl(alpha,NumABD,N,BandDown,BandUp,xindx,                 &
     &                   beta,info)
     
          END IF


c Test convergence + new value of state variables
          RelativeChange=0.d0

          DO I=1, N
            RelativeChange  = MAX(RelativeChange,ABS(BETA(I)))
            Svar(I)         = Svar(I)+BETA(I)
            IF (Positivity) Svar(I)=MAX(0.D0,Svar(I))
          ENDDO
          if (.not. positivity .and. ipos .GT. 1) THEN
              DO I = 1, ipos
                Svar(Pos(I)) = MAX(0.D0,Svar(Pos(I)))
              ENDDO
          ENDIF

          IF(RelativeChange<=TolChange)THEN
            SteadyStateReached = .TRUE.
            SteadyStateReachedInt = 1
c last precision reached
            IF (K .LT. maxiter) THEN
             precis(K+1) = 0.d0
             DO i = 1, N
              beta(i) = 0.D0
             ENDDO
             CALL XMODEL(N,time,Svar,beta,out,nout)

             DO I=1, N
               precis(K+1) = precis(K+1)+abs(beta(I))
             ENDDO
             precis(K+1) = precis(K+1)/N
             niter = K+1
            ENDIF
            EXIT 
          ENDIF


       ENDDO 

       END SUBROUTINE dSteady
     
c                !<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<!
c                !<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<!
c                !             FUNCTIONS              !
c                !>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>!
c                !>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>!

c****************************************************************
c Perturbing a certain value, to calculate numerical differences
c****************************************************************

       DOUBLE PRECISION FUNCTION Perturb (Value)

c---------------------------------------------------------------------
c Calculates the numerical difference value and
c adds this small amount to the original value
c---------------------------------------------------------------------

       DOUBLE PRECISION Value

c---------------------------------------------------------------------
c   ! A small, positive value 
       Perturb = ABS(Value) * 1.D-8

c   ! but not too small
       IF (Perturb < 1.D-8)  Perturb = 1.D-8

       Value = Value + Perturb

       END FUNCTION Perturb

c                !<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<!
c                !<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<<!
c                !        CREATING JACOBIANS          !
c                !>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>!
c                !>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>>!

c********************************************************************
c Full Jacobian
c********************************************************************

       SUBROUTINE xFullJacob(N,Beta, Alpha, Svar, CopyVar, ewt,                &
     &                       xmodel, time, out, nout)
c-------------------------------------------------------------------*
c Fills the matrices alpha and beta by finite differences           *
c Uses the model subroutine that estimates the rate of change dC/dt *
c The first call is used to fill the array beta (=-dC(I)/dt)        *
c Subsequent calls are used to fill the array alpha                 *
c (partial derivates [d(dC(I)/dt)/dC(j)])                           *
c For N state variables there are N+1 calls to the model* 
c-------------------------------------------------------------------*

       IMPLICIT NONE

       INTEGER           N   

       DOUBLE PRECISION  Beta (N)
       DOUBLE PRECISION  Alpha(N,N)
       DOUBLE PRECISION  Svar (N),ewt(N)
       DOUBLE PRECISION  time, out(*)
       INTEGER           nout(*)
       EXTERNAL          xmodel
     
       INTEGER           I, J
       DOUBLE PRECISION  DivDelt,CopyVar(N),dSvar(N)
       DOUBLE PRECISION  perturb
     
c--------------------------------------------------------------------
c     ! Call model-specific subroutines; input is Svar; output is dSvar
       DO i = 1, N
         dSvar(i) = 0.D0
       ENDDO  
       CALL XMODEL(N,time,Svar,dSvar,out,nout)
     
       DO i = 1, N
         Beta(i) = -dSvar(i)
       ENDDO

c     ! Jacobian: Perturb each state variable, one by one

       DO i = 1, N
         DO j = 1, N
          Alpha(i,j) = 0.D0  
         ENDDO
         CopyVar(i) = Svar(i)   
       ENDDO
       
       DO I= 1, N

         DivDelt  =  Perturb(CopyVar(I)) 
c alternative is not so efficient!
c        DivDelt = sign(ewt(I),CopyVar(I))
c       CopyVar(I) = CopyVar(I) + DivDelt

         DO J = 1, N
           dSvar(j) = 0.D0
         ENDDO 
         CALL XMODEL(N,time,CopyVar,dSvar,out,nout)  
       
         DO J = 1,N
           alpha (j,i) = (dSvar(J) + Beta(j))/DivDelt
         ENDDO
         
         CopyVar(I) = SVar(I)                        
       ENDDO

       RETURN

       END SUBROUTINE xFullJacob

c*******************************************************************
c Simple banded jacobian
c*******************************************************************

       SUBROUTINE xBandJacob(MU, ML, N, LDA, Beta, alpha,OldSvar,              &
     &                        Delt,Svar,ewt,xmodel,time,out,nout)
c-------------------------------------------------------------------*
c Fills the matrices alpha and beta, used in the implicit routines  *
c Uses the model subroutine that estimates the rate of change dC/dt *
c Assumes that variables are sorted per box, per species            *
c The first call is used to fill the array beta (=-dC(I)/dt)        *
c Subsequent calls are used to fill the array alpha                 *
c (partial derivates [d(dC(I)/dt)/dC(j)])                           *
c MU and ML are bands above and below the diagonal                  *
c-------------------------------------------------------------------*
    
       IMPLICIT NONE
     
       INTEGER           MU, ML, LDA, N 
       DOUBLE PRECISION  time, out(*)
       INTEGER           nout(*)
     
       DOUBLE PRECISION Beta(N), OldSvar(N), Delt(*)
       DOUBLE PRECISION alpha(LDA,N), Svar(N), ewt(N)
       EXTERNAL         xmodel
     
     
       INTEGER  NIter, Nband, I, J, K, Istart, Istop
       DOUBLE PRECISION Val, divDelta, dSvar(N), perturb
c--------------------------------------------------------------------
c Number of iterations and total width of the 'band'
       NIter  = MIN(N,ML+MU+1)        
       Nband  = ML+ MU + 1            
     
c Initialise
       DO I = 1, LDA
         DO J = 1, N
           alpha(I,J)   = 0.d0
         ENDDO
       ENDDO
       DO I = 1, N           
         OldSvar(I) = Svar(I)
         dSvar(I) = 0.D0
       ENDDO
       
c Call model-specific subroutines; input is Svar; output is dSvar
       CALL XMODEL(N,time,Svar,dSvar,out,nout)
     
c Beta = -rate of change
       DO I = 1, N           
         Beta(I)  = -dSvar(I) 
       ENDDO    
c Jacobian, estimated numerically: 
c the value of Xj is increased with a quantity delt, the user-defined
c routines called (xModel), the changes in fi used to estimate dfi/dXj
     
       DO J =1,NIter      

c         ! Add perturbation to every Nband elements, save value of perturbation
         DO K = J, N, Nband

c          ! perturbed value
           DELT(K) = Perturb(Svar(K))
c alternative - not so efficient
c          DELT(K) = sign(ewt(K),Svar(K))
c          Svar(K)  = Svar(K) + DELT(K) 

         ENDDO

c Calculate the corresponding rate of change
         DO I = 1, N           
           dSvar(I) = 0.D0
         ENDDO
         CALL XMODEL(N,time,Svar,dSvar,out,nout)
     
c The partial derivates
         DO K = J, N, Nband

           DivDelta  = 1.D0 / DELT(K)

c          ! These are the positions of the rates of change affected
           Istart = MAX(K-MU, 1)
           Istop  = MIN(K+ML, N)
          
           DO I=Istart, IStop
c            ! The gradient alfa(i,k) stored in alpha(I-k+MU+ML,k)
             Val = (dSvar(I) + Beta(I)) * DivDelta
             alpha (I-K+MU+ML+1, K) = Val
           ENDDO
           
         ENDDO
         DO I = 1, N           
           Svar(I) = oldSvar(I)
           dSvar(I) = 0.D0
         ENDDO

       ENDDO   
     
       RETURN
     
       END SUBROUTINE xBandJacob
