% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/Eqn.R
\name{Eqn}
\alias{Eqn}
\alias{Eqn_newline}
\alias{Eqn_text}
\alias{Eqn_hspace}
\alias{Eqn_vspace}
\alias{Eqn_size}
\alias{ref}
\title{Create a LaTeX Equation Wrapper}
\usage{
Eqn(
  ...,
  label = NULL,
  align = FALSE,
  preview = getOption("previewEqn"),
  html_output = knitr::is_html_output(),
  quarto = getOption("quartoEqn"),
  mat_args = list(),
  preview.pdf = FALSE,
  preview.packages = NULL
)

Eqn_newline(space = 0)

Eqn_text(text)

Eqn_hspace(lhs = 5, mid = "", rhs = NULL, times = 1)

Eqn_vspace(space)

Eqn_size(string, size = 0)

ref(
  label,
  parentheses = TRUE,
  html_output = knitr::is_html_output(),
  quarto = getOption("quartoEqn")
)
}
\arguments{
\item{...}{comma separated LaTeX expressions that are either a) a \code{character} vector,
  which will be automatically wrapped the
  expression inside a call to \code{\link{cat}}, b) a \code{matrix} object
  containing character or numeric information, which will be passed \code{\link{latexMatrix}},
  along with the information in \code{mat_args}, or
  c) an object that was explicitly created via \code{\link{latexMatrix}}, which
  provides greater specificity.

  Note that user defined functions that use \code{\link{cat}} within
  their body should return an empty character vector to avoid printing the
  returned object}

\item{label}{character vector specifying the label to use (e.g., \code{eq:myeqn}), which
  for LaTeX can be reference via \code{\\ref{eq:myeqn}} or via the inline function
  \code{\link{ref}}. Including a label will also include an equation number automatically.

  For compiled documents if an HTML output is detected (see \code{html_output})
  then the equations will be labelled  via \code{(\#eq:myeqn)} and references via \code{\@ref(eq:myeqn)},
  or again via \code{\link{ref}} for convenience. For Quarto documents the label must be of the
  form \code{eq-LABEL}}

\item{align}{logical; use the \code{align} environment with explicit \code{&} representing alignment
points. Default: \code{FALSE}}

\item{preview}{logical; render an HTML version of the equation and display? This is intended for
testing purposes and is only applicable to interactive R sessions, though
for code testing purposes can be set globally
via \code{\link{options}} (e.g., \code{options('previewEqn' = FALSE)}).
Disabled whenever \code{quarto} or \code{html_output} are \code{TRUE}}

\item{html_output}{logical; use labels for HTML outputs instead of the LaTeX? Automatically
changed for compiled documents that support \code{knitr}. Generally not
required or recommended for the user to modify, except to view the generated syntax}

\item{quarto}{logical; use Quarto referencing syntax? When \code{TRUE}
the \code{html_output} will be irrelevant. Generally not recommended for the
user to modify, except to view the generated syntax}

\item{mat_args}{list of arguments to be passed to \code{\link{latexMatrix}} to change the
properties of the \code{matrix} input object(s). Note that these inputs are used globally, and apply to
each \code{matrix} object supplied. If further specificity is required create
\code{\link{latexMatrix}} objects directly.}

\item{preview.pdf}{logical; build a PDF of the preview equation? Generally
not require unless additional LaTeX packages are required that are not supported
by MathJax}

\item{preview.packages}{character vector for adding additional LaTeX package information to the
equation preview. Only used when \code{preview.pdf = TRUE}}

\item{space}{includes extra vertical space. Metric of the vertical space
must be 'ex', 'pt', 'mm', 'cm', 'em', 'bp', 'dd', 'pc', or 'in'}

\item{text}{argument to be used within \code{\\text{}}}

\item{lhs}{spacing size. Can be a number between -1 and 6. -1 provides negative
spaces and 0 gives no spacing. Input can also be a character vector, which will be
passed to \code{\\hspace{}} (e.g., \code{'1cm'}; see \code{space} argument
for supported metrics).
Default is 5, resulting in a \code{\\quad} space.}

\item{mid}{character vector to place in the middle of the space specification. Most
commonly this will be operators like \code{'='}}

\item{rhs}{see lhs for details. If left as \code{NULL} and \code{mid} is specified
the this will be set to \code{rhs} to create symmetric spaces around \code{mid}}

\item{times}{number of times to repeat the spacings}

\item{string}{a string that should have its text size modified. If missing
the size modifier is returned, which applies the size modifier
to the remainder of the text until reset with \code{Eqn_size()}}

\item{size}{numeric size of LaTeX text modifier,
ranging from -3 (\code{\\tiny}) to 5 (\code{\\HUGE}), with 0 defining the
normal test size (\code{\\normalsize}; default)}

\item{parentheses}{logical; include parentheses around the referenced equation?}
}
\description{
The \code{Eqn} function is designed to produce LaTeX expressions of mathematical
equations for writing.
The output can be copied/pasted into documents or
used directly in chunks in \code{.Rmd}, \code{.Rnw}, or \code{.qmd}
documents to compile to equations.
It wraps the equations generated by its arguments
in either a \code{\\begin{equation} ...\\end{equation}} or
\code{\\begin{align} ...\\end{align}} LaTeX environment. See also
\code{\link{ref}} for consistent inline referencing of numbered equations.

In a code chunk, use the chunk options \code{results='asis', echo=FALSE} to show only
the result of compiling the LaTeX expressions.

\code{Eqn_newline()} emits a newline (\code{\\}) in an equation, with
an optional increase to the padding following the newline.

\code{Eqn_text()} inserts a literal string to be rendered in a text font in an equation.

\code{Eqn_hspace()} is used to create (symmetric) equation spaces, most typically around
\code{=} signs
Input to \code{lhs}, \code{rhs} can be a
numeric to increase the size of the space or a
character vector to be passed to the LaTeX macro \code{\\hspace{}}.

\code{Eqn_vspace()} inserts vertical space between lines in an equation.
Typically used for aligned, multiline equations.

\code{Eqn_size()} is used to increase or decrease the size of LaTeX text and equations. Can be applied
to a specific string or applied to all subsequent text until overwritten.

\code{ref{}} provides inline references to equations in R
markdown and Quarto documents.
Depending on the output type this function will provide the correct
inline wrapper for MathJax or LaTeX equations. This provides more
consistent referencing when switching between HTML and PDF outputs as
well as documentation types (e.g., \code{.Rmd} vs \code{.qmd}).
}
\examples{

# character input
Eqn('e=mc^2')

# show only the LaTeX code
Eqn('e=mc^2', preview=FALSE)

# Equation numbers & labels
Eqn('e=mc^2', label = 'eq:einstein')
Eqn("X=U \\\\lambda V", label='eq:svd')

# html_output and quarto outputs only show code
#   (both auto detected in compiled documents)
Eqn('e=mc^2', label = 'eq:einstein', html_output = TRUE)

# Quarto output
Eqn('e=mc^2', label = 'eq-einstein', quarto = TRUE)

\dontrun{
# The following requires LaTeX compilers to be pre-installed

# View PDF instead of HTML
Eqn('e=mc^2', preview.pdf=TRUE)

# Add extra LaTeX dependencies for PDF build
Eqn('\\\\bm{e}=mc^2', preview.pdf=TRUE,
    preview.packages=c('amsmath', 'bm'))

}

# Multiple expressions
Eqn("e=mc^2",
    Eqn_newline(),
    "X=U \\\\lambda V", label='eq:svd')

# expressions that use cat() within their calls
Eqn('SVD = ',
    latexMatrix("u", "n", "k"),
    latexMatrix("\\\\lambda", "k", "k", diag=TRUE),
    latexMatrix("v", "k", "p", transpose = TRUE),
    label='eq:svd')

# align equations using & operator
Eqn("X &= U \\\\lambda V", Eqn_newline(),
    "& = ", latexMatrix("u", "n", "k"),
    latexMatrix("\\\\lambda", "k", "k", diag=TRUE),
    latexMatrix("v", "k", "p", transpose = TRUE),
    align=TRUE)

#  numeric/character matrix example
A <- matrix(c(2, 1, -1,
              -3, -1, 2,
              -2,  1, 2), 3, 3, byrow=TRUE)
b <- matrix(c(8, -11, -3))

# numeric matrix wrapped internally
cbind(A,b) |> Eqn()
cbind(A,b) |> latexMatrix() |> Eqn()

# change numeric matrix brackets globally
cbind(A,b) |> Eqn(mat_args=list(matrix='bmatrix'))

# greater flexibility when using latexMatrix()
cbind(A, b) |> latexMatrix() |> partition(columns=3) |> Eqn()

# with showEqn()
showEqn(A, b, latex=TRUE) |> Eqn()


Eqn_newline()
Eqn_newline('10ex')


Eqn_hspace()
Eqn_hspace(3) # smaller
Eqn_hspace(3, times=2)
Eqn_hspace('1cm')

# symmetric spacing around mid
Eqn_hspace(mid='=')
Eqn_hspace(mid='=', times=2)


Eqn_vspace('1.5ex')
Eqn_vspace('1cm')



# set size globally
Eqn_size(size=3)
Eqn_size() # reset

# locally for defined string
string <- 'e = mc^2'
Eqn_size(string, size=1)



# used inside of Eqn() or manually defined labels in the document
Eqn('e = mc^2', label='eq:einstein')

# use within inline block via `r ref()`
ref('eq:einstein')
ref('eq:einstein', parentheses=FALSE)
ref('eq:einstein', html_output=TRUE)

# With Quarto
Eqn('e = mc^2', label='eq-einstein', quarto=TRUE)
ref('eq:einstein', quarto=TRUE)
ref('eq:einstein', quarto=TRUE, parentheses=FALSE)

}
\seealso{
\code{\link{latexMatrix}}, \code{\link{matrix2latex}}, \code{\link{ref}}
}
\author{
Phil Chalmers
}
