% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data.R
\name{cooks.distance.gnm}
\alias{cooks.distance.gnm}
\title{Cook's Distance for Generalized Nonlinear Models}
\usage{
\method{cooks.distance}{gnm}(model, plot.it = FALSE, dispersion = NULL, coefs, identify, ...)
}
\arguments{
\item{model}{an object of class \emph{gnm}.}

\item{plot.it}{an (optional) logical indicating if the plot is required or just the data matrix in which that
plot is based. As default, \code{plot.it} is set to \code{FALSE}.}

\item{dispersion}{an (optional) value indicating the estimate of the dispersion parameter. As default, \code{dispersion} is set to \code{summary(object)$dispersion}.}

\item{coefs}{an (optional) character string that matches (partially) some of the model parameter names.}

\item{identify}{an (optional) integer indicating the number of individuals to identify on the plot of the Cook's
distance. This is only appropriate if \code{plot.it=TRUE}.}

\item{...}{further arguments passed to or from other methods. If \code{plot.it=TRUE} then \code{...} may be used
to include graphical parameters to customize the plot. For example, \code{col}, \code{pch}, \code{cex}, \code{main},
\code{sub}, \code{xlab}, \code{ylab}.}
}
\value{
A matrix as many rows as individuals in the sample and one column with the values of the Cook's distance.
}
\description{
Produces an approximation of the Cook's distance, better known as the \emph{one-step approximation},
for measuring the effect of deleting each observation in turn on the estimates of the parameters in a linear
predictor. Additionally, this function can produce an index plot of Cook's distance for all or a subset of the
parameters in the linear predictor (via the argument \code{coefs}).
}
\details{
The Cook's distance consists of the \emph{distance} between two estimates of the parameters in the linear
predictor using a metric based on the (estimate of the) variance-covariance matrix. The first one set of estimates
is computed from a dataset including all individuals, and the second one is computed from a dataset in which the
\emph{i}-th individual is excluded. To avoid computational burden, the second set of estimates is replaced by its
\emph{one-step approximation}. See the \link{dfbeta.overglm} documentation.
}
\examples{
###### Example 1: The effects of fertilizers on coastal Bermuda grass
data(Grass)
fit1 <- gnm(Yield ~ b0 + b1/(Nitrogen + a1) + b2/(Phosphorus + a2) + b3/(Potassium + a3),
            family=gaussian(inverse), start=c(b0=0.1,b1=13,b2=1,b3=1,a1=45,a2=15,a3=30), data=Grass)

cooks.distance(fit1, plot.it=TRUE, col="red", lty=1, lwd=1,
  col.lab="blue", col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 2: Assay of an Insecticide with a Synergist
data(Melanopus)
fit2 <- gnm(Killed/Exposed ~ b0 + b1*log(Insecticide-a1) + b2*Synergist/(a2 + Synergist),
            family=binomial(logit), weights=Exposed, start=c(b0=-3,b1=1.2,a1=1.7,b2=1.7,a2=2),
		   data=Melanopus)

### Cook's distance just for the parameter "b1"
cooks.distance(fit2, plot.it=TRUE, coef="b1", col="red", lty=1, lwd=1,
  col.lab="blue", col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 3: Developmental rate of Drosophila melanogaster
data(Drosophila)
fit3 <- gnm(Duration ~ b0 + b1*Temp + b2/(Temp-a), family=Gamma(log),
            start=c(b0=3,b1=-0.25,b2=-210,a=55), weights=Size, data=Drosophila)

cooks.distance(fit3, plot.it=TRUE, col="red", lty=1, lwd=1,
  col.lab="blue", col.axis="blue", col.main="black", family="mono", cex=0.8)

###### Example 4: Radioimmunological Assay of Cortisol
data(Cortisol)
fit4 <- gnm(Y ~ b0 + (b1-b0)/(1 + exp(b2+ b3*lDose))^b4, family=Gamma(identity),
            start=c(b0=130,b1=2800,b2=3,b3=3,b4=0.5), data=Cortisol)

cooks.distance(fit4, plot.it=TRUE, col="red", lty=1, lwd=1,
  col.lab="blue", col.axis="blue", col.main="black", family="mono", cex=0.8)

}
