% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/data.R
\name{dfbeta.gnm}
\alias{dfbeta.gnm}
\title{Dfbeta statistic for Generalized Nonlinear Models}
\usage{
\method{dfbeta}{gnm}(model, coefs, identify, ...)
}
\arguments{
\item{model}{an object of class \emph{gnm}.}

\item{coefs}{an (optional) character string which (partially) match with the names of some model parameters.}

\item{identify}{an (optional) integer indicating the number of individuals to identify on the plot of the Dfbeta statistic.
This is only appropriate if \code{coefs} is specified.}

\item{...}{further arguments passed to or from other methods. If \code{plot.it=TRUE} then \code{...} may be used
to include graphical parameters to customize the plot. For example, \code{col}, \code{pch}, \code{cex}, \code{main},
\code{sub}, \code{xlab}, \code{ylab}.}
}
\value{
A matrix with as many rows as cases in the sample and as many columns as parameters in the linear predictor. The
\eqn{i}-th row in that matrix corresponds to the difference between the parameters estimates obtained using all cases
and the \emph{one-step approximation} of those estimates when excluding the \eqn{i}-th case from the dataset.
}
\description{
Calculates an approximation of the parameter estimates that would be produced by deleting each case in turn,
which is known as the \emph{one-step approximation}. Additionally, the function can produce an index plot of the Dfbeta statistic
for some parameter specified by the argument \code{coefs}.
}
\details{
The \emph{one-step approximation} of the parameters estimates when the \eqn{i}-th case
is excluded from the dataset consists of the vector obtained as a result of the first iteration of the Fisher Scoring
algorithm when it is performed using: (1) a dataset in which the \eqn{i}-th case is excluded; and (2)
a starting value that is the estimate of the same model but based on the dataset including all cases.
}
\examples{
###### Example 1: The effects of fertilizers on coastal Bermuda grass
data(Grass)
fit1 <- gnm(Yield ~ b0 + b1/(Nitrogen + a1) + b2/(Phosphorus + a2) + b3/(Potassium + a3),
            family=gaussian(inverse), start=c(b0=0.1,b1=13,b2=1,b3=1,a1=45,a2=15,a3=30), data=Grass)

fit1a <- update(fit1, subset=-c(1), start=coef(fit1), maxit=1)
coef(fit1) - coef(fit1a)

dfbetas <- dfbeta(fit1)
round(dfbetas[1,],5)

###### Example 2: Assay of an Insecticide with a Synergist
data(Melanopus)
fit2 <- gnm(Killed/Exposed ~ b0 + b1*log(Insecticide-a1) + b2*Synergist/(a2 + Synergist),
            family=binomial(logit), weights=Exposed, start=c(b0=-3,b1=1.2,a1=1.7,b2=1.7,a2=2),
		   data=Melanopus)

fit2a <- update(fit2, subset=-c(2), start=coef(fit2), maxit=1)
coef(fit2) - coef(fit2a)

dfbetas <- dfbeta(fit2)
round(dfbetas[2,],5)

###### Example 3: Developmental rate of Drosophila melanogaster
data(Drosophila)
fit3 <- gnm(Duration ~ b0 + b1*Temp + b2/(Temp-a), family=Gamma(log),
            start=c(b0=3,b1=-0.25,b2=-210,a=55), weights=Size, data=Drosophila)

fit3a <- update(fit3, subset=-c(3), start=coef(fit3), maxit=1)
coef(fit3) - coef(fit3a)

dfbetas <- dfbeta(fit3)
round(dfbetas[3,],5)

###### Example 4: Radioimmunological Assay of Cortisol
data(Cortisol)
fit4 <- gnm(Y ~ b0 + (b1-b0)/(1 + exp(b2+ b3*lDose))^b4, family=Gamma(identity),
            start=c(b0=130,b1=2800,b2=3,b3=3,b4=0.5), data=Cortisol)

fit4a <- update(fit4, subset=-c(4), start=coef(fit4), maxit=1)
coef(fit4) - coef(fit4a)

dfbetas <- dfbeta(fit4)
round(dfbetas[4,],5)

}
\references{
Pregibon D. (1981). Logistic regression diagnostics. \emph{The Annals of Statistics}, 9, 705-724.

Wei B.C. (1998). \emph{Exponential Family Nonlinear Models}. Springer, Singapore.
}
