; Beginning of Licence
;
; This software is licensed only for personal and educational use and
; not for the production of commercial software.  Modifications to this
; program are allowed but the resulting source must be annotated to
; indicate the nature of and the author of these changes.  
;
; Any modified source is bound by this licence and must remain available 
; as open source under the same conditions it was supplied and with this 
; licence at the top.

; This software is supplied AS IS without any warranty.  In no way shall 
; Mark Tarver or Lambda Associates be held liable for any damages resulting 
; from the use of this program.

; The terms of these conditions remain binding unless the individual 
; holds a valid license to use Qi commercially.  This license is found 
; in the final page of 'Functional Programming in Qi'.  In that event 
; the terms of that license apply to the license holder. 
;
; (c) copyright Mark Tarver, 2008
; End of Licence

(IN-PACKAGE :qi)

(DEFMACRO if (X Y Z)
 `(LET ((*C* ,X))
   (COND ((EQ *C* 'true) ,Y) ((EQ *C* 'false) ,Z)
    (T (error "~S is not a boolean~%" *C*)))))

(DEFMACRO and (X Y) `(if ,X (if ,Y 'true 'false) 'false))

(DEFMACRO or (X Y) `(if ,X 'true (if ,Y 'true 'false)))

(DEFUN not (X)
 (COND ((EQ X 'true) 'false) 
       ((EQ X 'false) 'true)
       (T (error "~S is not a boolean~%" X))))

(DEFUN element? (x y) (IF (MEMBER x y :TEST 'ABSEQUAL) 'true 'false))

(DEFUN subst (X Y Z) (SUBST X Y Z :TEST 'ABSEQUAL))

(DEFUN subst (X Y Z) 
  (COND ((ABSEQUAL Y Z) X)
        ((TUPLE-P Z) (@p (subst X Y (fst Z)) (subst X Y (snd Z))))
        ((CONSP Z) (CONS (subst X Y (CAR Z)) (subst X Y (CDR Z))))
        (T Z)))        

(DEFUN remove (x y) (REMOVE x y :TEST 'ABSEQUAL))

(DEFUN difference (x y) (SET-DIFFERENCE x y :TEST 'ABSEQUAL))

(DEFUN union (x y) 
  (IF (NULL x) y
      (IF (MEMBER (CAR x) y :TEST 'ABSEQUAL)
          (union (CDR x) y)
          (CONS (CAR x) (union (CDR x) y)))))

(DEFUN assoc (x y) (ASSOC x y :TEST 'ABSEQUAL))

(DEFMACRO let (VAR VAL EXPR) (LIST 'LET (LIST (LIST VAR VAL)) EXPR))

(DEFMACRO list (&REST X) (CONS 'LIST X))

(DEFMACRO do (&REST X) (CONS 'PROGN X))

(DEFUN y-or-n? (X) (IF (Y-OR-N-P X) 'true 'false))

(DEFUN empty? (X) (IF (NULL X) 'true 'false))

(DEFUN value (X) (SYMBOL-VALUE X))

(DEFUN length (X) (LIST-LENGTH X))

(DEFUN nth (N L) 
  (IF (= N 1) (IF (NULL L) (error "nth expects a longer list.~%") (CAR L))
      (nth (1- N) (CDR L))))

(DEFUN concat (X Y) (READ-FROM-STRING (FORMAT NIL "~A~A" X Y)))

(DEFUN append (X Y) (APPEND X Y))

(DEFUN reverse (X) (REVERSE X))

(DEFUN set (X Y) (SET X Y))

(DEFUN cons (X Y) (CONS X Y))

(DEFUN @c (X Y) (CONS X Y))

(DEFUN cons? (X) (IF (CONSP X) 'true 'false))

(DEFMACRO time (X) (LIST 'TIME X))

(DEFUN implementation_error (Func)
 (ERROR "Qi implementation error in ~A: report to dr.mtarver@ukonline.co.uk~%" Func))

(DEFUN explode (X) (COERCE (FORMAT NIL "~S" X) 'LIST))

(DEFUN head (X)
 (IF (CONSP X) (CAR X) (ERROR "head expects a non-empty list.~% ")))

(DEFUN tail (X)
 (IF (CONSP X) (CDR X) (ERROR "tail expects a non-empty list.~% ")))

(DEFSTRUCT
 (TUPLE
  (:PRINT-FUNCTION
   (LAMBDA (Struct Stream Depth) (DECLARE (IGNORE Depth))
    (print_tuple Struct Stream)))
  (:CONC-NAME NIL) (:CONSTRUCTOR @p (fst snd)))
 fst snd)

(DEFUN print_tuple (Tuple Stream)
 (FORMAT Stream "(@p ~S ~S)" (fst Tuple) (snd Tuple)))

(DEFUN tuple? (X) (IF (TUPLE-P X) 'true 'false))

(DEFUN ABSEQUAL (X Y)
 (COND ((AND (CONSP X) (CONSP Y)) 
        (AND (ABSEQUAL (CAR X) (CAR Y))
             (ABSEQUAL (CDR X) (CDR Y))))
       ((AND (TUPLE-P X) (TUPLE-P Y))
       (AND (ABSEQUAL (fst X) (fst Y)) (ABSEQUAL (snd X) (snd Y))))
       (T (EQUAL X Y))))

(DEFUN variable? (X)
 (IF (var? X)
     'true 
     'false))

(DEFUN var? (X) (AND (SYMBOLP X) (NOT (NULL X)) (UPPER-CASE-P (CHAR (SYMBOL-NAME X) 0))))

(DEFUN symbol? (X)
  (IF (AND (SYMBOLP X) 
	       (NOT (MEMBER X '(true false NIL)))
           (NOT (place_holder? X))
           (NOT (UPPER-CASE-P (CHAR (SYMBOL-NAME X) 0))))
      'true
      'false))

(DEFUN place_holder? (X)
  (LET ((NAMESTRING (SYMBOL-NAME X))) 
         (AND (> (LENGTH (THE STRING NAMESTRING)) 2)
	    (CHAR-EQUAL #\& (CHAR NAMESTRING 0))
                (CHAR-EQUAL #\& (CHAR NAMESTRING 1)))))  

(DEFUN number? (X) (IF (NUMBERP X) 'true 'false))

(DEFUN string? (X) (IF (STRINGP X) 'true 'false))

(DEFUN character? (X) (IF (CHARACTERP X) 'true 'false))

(DEFUN boolean? (X) (IF (MEMBER X '(true false)) 'true 'false))

(DEFUN integer? (X) (IF (INTEGERP X) 'true 'false))

(DEFUN complex? (X) (IF (COMPLEXP X) 'true 'false))

(DEFUN float? (X) (IF (FLOATP X) 'true 'false))

(DEFUN real? (X) (IF (REALP X) 'true 'false))

(DEFUN rational? (X) (IF (RATIONALP X) 'true 'false))

(DEFUN sqrt (X) (SQRT X))

(DEFUN random (X) (RANDOM X))

(DEFUN round (X) (ROUND X))

(DEFUN congruent? (X Y) (IF (EQUALP X Y) 'true 'false))

(DEFUN qi_= (X Y) (IF (ABSEQUAL X Y) 'true 'false))

(DEFUN == (X Y) (qi_= X Y))

(DEFUN qi_> (X Y) (IF (> X Y) 'true 'false))

(DEFUN qi_< (X Y) (IF (< X Y) 'true 'false))

(DEFUN qi_>= (X Y) (IF (>= X Y) 'true 'false))

(DEFUN qi_<= (X Y) (IF (<= X Y) 'true 'false))

(DEFUN gensym (X) (GENTEMP X))

(DEFUN newvar (X) (GENTEMP (FORMAT NIL "~A" X)))
(DEFUN newsym (X) (GENTEMP (FORMAT NIL "~A" X)))

(DEFUN source_code (F) (get-prop F 'source NIL))

(DEFUN map (V32 V33)
 (COND ((NULL V33) NIL)
  ((CONSP V33) (CONS (apply V32 (CAR V33)) (map V32 (CDR V33))))
  (T (ERROR "map requires a list; not ~S~%" V33))))

(DEFUN read-file-as-charlist (File)
 (LET ((AbsFile (FORMAT NIL "~A~A" *qi_home_directory* File)))
  (IF (NOT (PROBE-FILE AbsFile)) (ERROR "~%~A does not exist~%" AbsFile))
  (WITH-OPEN-FILE (In AbsFile :DIRECTION :INPUT)
   (DO ((Letter T) (Letters NIL)) ((NULL Letter) (NREVERSE (CDR Letters)))
    (SETQ Letter (READ-CHAR In NIL NIL)) (PUSH Letter Letters)))))

(DEFVAR *qi_home_directory* "")

(DEFUN cd (String)
 (IF (EQUAL String "") (SETQ *qi_home_directory* String)
  (SETQ *qi_home_directory* (FORMAT NIL "~A/" String))))

(DEFVAR *tc* 'false)

(DEFUN write-to-file (Filename Output)
 (LET ((AbsFilename (FORMAT NIL "~A~A" *qi_home_directory* Filename)))
  (WITH-OPEN-FILE (OUTSTREAM AbsFilename
                               :DIRECTION :OUTPUT 
                               :IF-EXISTS :APPEND 
                               :IF-DOES-NOT-EXIST :CREATE)
    (FORMAT OUTSTREAM "~%")
    (COND ((STRINGP Output) (WRITE-STRING Output OUTSTREAM)) 
          (T (PPRINT Output OUTSTREAM)))  )
  AbsFilename))

(DEFMACRO freeze (X) `(FUNCTION (LAMBDA () ,X)))

(DEFUN thaw (X) (FUNCALL X))

(DEFUN ps (X) (PPRINT (source_code X)))

(DEFUN abort () (ERROR ""))

(DEFUN read-char (X) (DECLARE (IGNORE X)) (READ-CHAR))

(DEFUN input () (eval (CAR (lineread))))

(DEFMACRO input+ (Colon Type) `(input+help (curry-type (QUOTE ,Type))))

(DEFUN input+help (Type)
  (IF (NOT (monomorphic? Type)) (ERROR "error: ~A should be a monotype.~%" Type))
  (LET ((I (CAR (lineread))))       
       (COND  ((EQ (statictypecheck NIL I Type) 'false)
              (FORMAT T "this is not a ~A, please re-enter: " Type) 
              (input+help Type))
             (T (eval I)))))

(DEFUN monomorphic? (Type)
  (IF (CONSP Type)
      (AND (monomorphic? (CAR Type)) (monomorphic? (CDR Type)))
      (NOT (var? Type))))

(DEFUN if-without-checking (String) 
   (IF (EQ *tc* 'false) (ERROR String)))

(DEFUN if-with-checking (String) 
   (IF (EQ *tc* 'true) (ERROR String)))

(DEFUN make-array (dims) (MAKE-ARRAY dims :INITIAL-ELEMENT #\Escape))

(DEFUN get-array (array dims default)
 (LET ((array_element (APPLY #'AREF (CONS array dims))))
  (IF (EQ array_element #\Escape) default array_element)))

(DEFUN put-array (array dims value)
 (SETF (APPLY #'AREF (CONS array dims)) value))

(DEFUN debug (X)
  (DECLARE (IGNORE X)) 
  (IF (PROBE-FILE "debug.txt") (DELETE-FILE "debug.txt")) 
  (DRIBBLE (FORMAT NIL "~A~A" *qi_home_directory* "debug.txt"))
  "done")

(DEFUN undebug (X) (DECLARE (IGNORE X)) (DRIBBLE) "done")

(DEFUN version (X) (SETQ *version* X))

(DEFUN type (X) (typecheck NIL X 'A))

(DEFUN typecheck (Hyps X Type) 
   (statictypecheck (MAPCAR 'cons_form_hyp Hyps) (cons_form X) Type))

(DEFUN cons_form_hyp (Hyp)
  (IF (typing? Hyp)
      (CONS (cons_form (CAR Hyp)) (CDR Hyp))
      Hyp))

(DEFUN typing? (Hyp) (AND (LISTP Hyp) (= (LENGTH Hyp) 3) (EQ (CADR Hyp) '|:|))) 