/*
 * Decompiled with CFR 0.152.
 */
package org.dict.zip;

import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.OutputStream;
import java.nio.ByteBuffer;
import java.nio.ByteOrder;
import java.nio.charset.Charset;
import java.util.BitSet;
import java.util.zip.CRC32;
import java.util.zip.CheckedInputStream;
import org.dict.zip.DictZipFileUtils;

public class DictZipHeader {
    protected int[] chunks;
    private int headerLength;
    private static final int GZIPFLAG_SIZE = 16;
    private final BitSet gzipFlag = new BitSet(16);
    private OperatingSystem headerOS = OperatingSystem.FAT;
    private CompressionLevel extraFlag;
    private long[] offsets;
    private int extraLength;
    private byte subfieldID1;
    private byte subfieldID2;
    private int subfieldLength;
    private int subfieldVersion;
    private int chunkLength;
    private int chunkCount;
    private long mtime;
    private String filename;
    private String comment;
    private static final Charset CHARSET = Charset.forName("ISO-8859-1");
    private static final int GZIP_MAGIC = 35615;
    private static final int FTEXT = 0;
    private static final int FHCRC = 1;
    private static final int FEXTRA = 2;
    private static final int FNAME = 3;
    private static final int FCOMMENT = 4;
    private static final int INT32_LEN = 4;
    private static final int EOS_LEN = 2;
    private static final int GZIP_HEADER_LEN = 10;

    private DictZipHeader() {
    }

    public DictZipHeader(long dataSize, int bufferSize) {
        if (bufferSize <= 0) {
            throw new IllegalArgumentException("Buffer size is zero or minus.");
        }
        long tmpCount = dataSize / (long)bufferSize;
        if (dataSize % (long)bufferSize > 0L) {
            ++tmpCount;
        }
        if (tmpCount > Integer.MAX_VALUE) {
            throw new IllegalArgumentException("data size is out of DictZip range.");
        }
        this.gzipFlag.set(2);
        this.extraFlag = CompressionLevel.DEFAULT_COMPRESSION;
        this.subfieldID1 = (byte)82;
        this.subfieldID2 = (byte)65;
        this.subfieldLength = 6 + (int)tmpCount * 2;
        this.subfieldVersion = 1;
        this.chunkLength = bufferSize;
        this.chunkCount = (int)tmpCount;
        this.chunks = new int[this.chunkCount];
        this.extraLength = this.subfieldLength + 4;
        this.headerLength = 10 + this.extraLength;
        this.filename = "";
        this.comment = "";
    }

    private void initOffsets() {
        this.offsets = new long[this.chunks.length];
        this.offsets[0] = this.headerLength;
        for (int i = 1; i < this.chunks.length; ++i) {
            this.offsets[i] = this.offsets[i - 1] + (long)this.chunks[i - 1];
        }
    }

    public static DictZipHeader readHeader(String s) throws IOException {
        DictZipHeader h = new DictZipHeader();
        CRC32 crc = new CRC32();
        FileInputStream in = new FileInputStream(s);
        DictZipHeader.readHeader(h, in, crc);
        ((InputStream)in).close();
        return h;
    }

    public static DictZipHeader readHeader(InputStream is, CRC32 crc) throws IOException {
        DictZipHeader h = new DictZipHeader();
        DictZipHeader.readHeader(h, is, crc);
        return h;
    }

    private static void readHeader(DictZipHeader h, InputStream is, CRC32 crc) throws IOException {
        CheckedInputStream in = new CheckedInputStream(is, crc);
        crc.reset();
        if (DictZipFileUtils.readUShort(in) != 35615) {
            throw new IOException("Not in GZIP format");
        }
        if (DictZipFileUtils.readUByte(in) != 8) {
            throw new IOException("Unsupported compression method");
        }
        int flg = DictZipFileUtils.readUByte(in);
        for (int i = 0; i < 16; ++i) {
            int testbit = 1 << i;
            if ((flg & testbit) != testbit) continue;
            h.gzipFlag.set(i);
        }
        h.mtime = DictZipFileUtils.readUInt(in);
        int compFlg = DictZipFileUtils.readUByte(in);
        if (compFlg == 2) {
            h.extraFlag = CompressionLevel.BEST_COMPRESSION;
        } else if (compFlg == 4) {
            h.extraFlag = CompressionLevel.BEST_SPEED;
        } else if (compFlg == 0) {
            h.extraFlag = CompressionLevel.DEFAULT_COMPRESSION;
        } else {
            throw new IOException("Corrupt GZIP header");
        }
        int hos = DictZipFileUtils.readUByte(in);
        h.headerOS = OperatingSystem.UNKNOWN;
        for (OperatingSystem os : OperatingSystem.values()) {
            if (hos != os.value) continue;
            h.headerOS = os;
            break;
        }
        h.headerLength = 10;
        if (h.gzipFlag.get(2)) {
            h.extraLength = DictZipFileUtils.readUShort(in);
            h.headerLength += h.extraLength + 2;
            h.subfieldID1 = (byte)DictZipFileUtils.readUByte(in);
            h.subfieldID2 = (byte)DictZipFileUtils.readUByte(in);
            h.subfieldLength = DictZipFileUtils.readUShort(in);
            h.subfieldVersion = DictZipFileUtils.readUShort(in);
            h.chunkLength = DictZipFileUtils.readUShort(in);
            h.chunkCount = DictZipFileUtils.readUShort(in);
            h.chunks = new int[h.chunkCount];
            for (int i = 0; i < h.chunkCount; ++i) {
                h.chunks[i] = DictZipFileUtils.readUShort(in);
            }
        }
        if (h.gzipFlag.get(3)) {
            int ubyte;
            StringBuilder sb = new StringBuilder();
            while ((ubyte = DictZipFileUtils.readUByte(in)) != 0) {
                sb.append((char)(ubyte & 0xFF));
                ++h.headerLength;
            }
            h.filename = sb.toString();
            ++h.headerLength;
        }
        if (h.gzipFlag.get(4)) {
            while (DictZipFileUtils.readUByte(in) != 0) {
                ++h.headerLength;
            }
            ++h.headerLength;
        }
        if (h.gzipFlag.get(1)) {
            int v = (int)crc.getValue() & 0xFFFF;
            if (DictZipFileUtils.readUShort(in) != v) {
                throw new IOException("Corrupt GZIP header");
            }
            h.headerLength += 2;
        }
        h.initOffsets();
    }

    public final String toString() {
        StringBuilder sb = new StringBuilder();
        sb.append("\nHeader length = ").append(this.headerLength);
        sb.append("\nSubfield ID = ").append((char)this.subfieldID1).append((char)this.subfieldID2);
        sb.append("\nSubfield length = ").append(this.subfieldLength);
        sb.append("\nSubfield version = ").append(this.subfieldVersion);
        sb.append("\nChunk length = ").append(this.chunkLength);
        sb.append("\nNumber of chunks = ").append(this.chunkCount);
        sb.append("\nLength of member = ").append(this.getMemberLength());
        return sb.toString();
    }

    public static void writeHeader(DictZipHeader h, OutputStream out) throws IOException {
        int i;
        CRC32 headerCrc = new CRC32();
        headerCrc.reset();
        ByteBuffer bb = ByteBuffer.allocate(22).order(ByteOrder.LITTLE_ENDIAN);
        bb.putShort((short)-29921);
        bb.put((byte)8);
        bb.put(h.gzipFlag.toByteArray()[0]);
        bb.putInt((int)h.mtime);
        bb.put((byte)h.extraFlag.value);
        bb.put((byte)h.headerOS.value);
        bb.putShort((short)h.extraLength);
        bb.put(h.subfieldID1);
        bb.put(h.subfieldID2);
        bb.putShort((short)h.subfieldLength);
        bb.putShort((short)h.subfieldVersion);
        bb.putShort((short)h.chunkLength);
        bb.putShort((short)h.chunkCount);
        out.write(bb.array());
        if (h.gzipFlag.get(1)) {
            headerCrc.update(bb.array());
        }
        for (i = 0; i < h.chunkCount; ++i) {
            DictZipFileUtils.writeShort(out, h.chunks[i]);
        }
        if (h.gzipFlag.get(1)) {
            for (i = 0; i < h.chunkCount; ++i) {
                headerCrc.update(ByteBuffer.allocate(2).putShort((short)h.chunks[i]).array());
            }
        }
        if (h.gzipFlag.get(3)) {
            if (h.filename != null) {
                out.write(h.filename.getBytes(CHARSET));
                if (h.gzipFlag.get(1)) {
                    headerCrc.update(h.filename.getBytes(CHARSET));
                }
            }
            out.write(0);
        }
        if (h.gzipFlag.get(4)) {
            if (h.comment != null) {
                out.write(h.comment.getBytes(CHARSET));
                if (h.gzipFlag.get(1)) {
                    headerCrc.update(h.comment.getBytes(CHARSET));
                }
            }
            out.write(0);
        }
        if (h.gzipFlag.get(1)) {
            DictZipFileUtils.writeShort(out, (int)headerCrc.getValue());
        }
    }

    public final int getOffset(long start) throws IllegalArgumentException {
        long off = start % (long)this.chunkLength;
        if (off < Integer.MAX_VALUE) {
            return (int)off;
        }
        throw new IllegalArgumentException("Index is out of boundary.");
    }

    public final long getPosition(long start) throws IllegalArgumentException {
        long idx = start / (long)this.chunkLength;
        if (idx < Integer.MAX_VALUE) {
            return this.offsets[(int)idx];
        }
        throw new IllegalArgumentException("Index is out of boundary.");
    }

    private void setGzipFlag(int flag, boolean val) {
        this.gzipFlag.set(flag, val);
    }

    public void setHeaderCRC(boolean val) {
        this.setGzipFlag(1, val);
    }

    public BitSet getGzipFlag() {
        return this.gzipFlag;
    }

    public final String getType() {
        if (this.subfieldID1 == 82 && this.subfieldID2 == 65) {
            return "dzip";
        }
        return "gzip";
    }

    public int getChunkLength() {
        return this.chunkLength;
    }

    public int getChunkCount() {
        return this.chunkCount;
    }

    public long getMtime() {
        return this.mtime;
    }

    public String getFilename() {
        if (this.gzipFlag.get(3)) {
            return this.filename;
        }
        return null;
    }

    public void setHeaderOS(OperatingSystem os) {
        this.headerOS = os;
    }

    public OperatingSystem getHeaderOS() {
        return this.headerOS;
    }

    public void setExtraFlag(CompressionLevel flag) {
        this.extraFlag = flag;
    }

    public CompressionLevel getExtraFlag() {
        return this.extraFlag;
    }

    public void setMtime(long mtime) {
        this.mtime = mtime;
    }

    public void setFilename(String filename) {
        if (filename != null) {
            this.filename = filename;
            this.gzipFlag.set(3);
        }
    }

    public void setComment(String comment) {
        if (comment != null) {
            this.comment = comment;
            this.gzipFlag.set(4);
        }
    }

    public long getMemberLength() {
        return this.offsets[this.chunkCount - 1] + (long)this.chunks[this.chunkCount - 1] + 2L + 8L;
    }

    public int getHeaderLength() {
        return this.headerLength;
    }

    public static enum OperatingSystem {
        FAT(0),
        AMIGA(1),
        VMS(2),
        UNIX(3),
        VMCMS(4),
        ATARI(5),
        HPFS(6),
        MAC(7),
        ZSYS(8),
        CPM(9),
        TOPS(10),
        NTFS(11),
        QDOS(12),
        ACORN(13),
        UNKNOWN(255);

        private int value;

        private OperatingSystem(int value) {
            this.value = value;
        }
    }

    public static enum CompressionLevel {
        DEFAULT_COMPRESSION(0),
        BEST_COMPRESSION(2),
        BEST_SPEED(4);

        private int value;

        private CompressionLevel(int value) {
            this.value = value;
        }
    }
}

