/*
 * Decompiled with CFR 0.152.
 */
package org.omegat.util;

import java.io.BufferedReader;
import java.io.BufferedWriter;
import java.io.File;
import java.io.FileFilter;
import java.io.IOException;
import java.nio.charset.Charset;
import java.nio.file.FileVisitOption;
import java.nio.file.Files;
import java.nio.file.OpenOption;
import java.nio.file.Path;
import java.nio.file.attribute.BasicFileAttributes;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.Comparator;
import java.util.Date;
import java.util.HashSet;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.Optional;
import java.util.Set;
import java.util.function.BiPredicate;
import java.util.function.Function;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import java.util.stream.Collectors;
import java.util.stream.Stream;
import org.apache.commons.io.FileUtils;
import org.apache.commons.io.FilenameUtils;
import org.omegat.util.Log;
import org.omegat.util.Platform;
import org.omegat.util.StreamUtil;
import org.omegat.util.StringUtil;

public final class FileUtil {
    public static final long RENAME_RETRY_TIMEOUT = 3000L;
    private static final Pattern RE_ABSOLUTE_WINDOWS = Pattern.compile("[A-Za-z]\\:(/.*)");
    private static final Pattern RE_ABSOLUTE_LINUX = Pattern.compile("/.*");
    private static final Pattern[] NO_PATTERNS = new Pattern[0];

    private FileUtil() {
    }

    public static void removeOldBackups(final File originalFile, int maxBackups) {
        try {
            File[] bakFiles = originalFile.getParentFile().listFiles(new FileFilter(){

                @Override
                public boolean accept(File f) {
                    return !f.isDirectory() && f.getName().startsWith(originalFile.getName()) && f.getName().endsWith(".bak");
                }
            });
            if (bakFiles != null && bakFiles.length > maxBackups) {
                Arrays.sort(bakFiles, new Comparator<File>(){

                    @Override
                    public int compare(File f1, File f2) {
                        if (f2.lastModified() < f1.lastModified()) {
                            return -1;
                        }
                        if (f2.lastModified() > f1.lastModified()) {
                            return 1;
                        }
                        return 0;
                    }
                });
                for (int i = maxBackups; i < bakFiles.length; ++i) {
                    bakFiles[i].delete();
                }
            }
        }
        catch (Exception exception) {
            // empty catch block
        }
    }

    public static File backupFile(File original) throws IOException {
        long fileMillis = original.lastModified();
        String str = new SimpleDateFormat("yyyyMMddHHmm").format(new Date(fileMillis));
        File backup = new File(original.getPath() + "." + str + ".bak");
        FileUtils.copyFile(original, backup);
        return backup;
    }

    public static void rename(File from, File to) throws IOException {
        if (!from.exists()) {
            throw new IOException("Source file to rename (" + from + ") doesn't exist");
        }
        if (to.exists()) {
            throw new IOException("Target file to rename (" + to + ") already exists");
        }
        long b = System.currentTimeMillis();
        while (!from.renameTo(to)) {
            long e = System.currentTimeMillis();
            if (e - b <= 3000L) continue;
            throw new IOException("Error renaming " + from + " to " + to);
        }
    }

    public static void copyFileWithEolConversion(File inFile, File outFile, Charset charset) throws IOException {
        File dir = outFile.getParentFile();
        if (!dir.exists()) {
            dir.mkdirs();
        }
        String eol = null;
        if (outFile.exists()) {
            eol = FileUtil.getEOL(outFile, charset);
        }
        if (eol == null) {
            eol = System.lineSeparator();
        }
        try (BufferedReader in = Files.newBufferedReader(inFile.toPath(), charset);
             BufferedWriter out = Files.newBufferedWriter(outFile.toPath(), charset, new OpenOption[0]);){
            String s;
            while ((s = in.readLine()) != null) {
                out.write(s);
                out.write(eol);
            }
        }
    }

    public static String getEOL(File file, Charset charset) throws IOException {
        String r;
        block14: {
            r = null;
            try (BufferedReader in = Files.newBufferedReader(file.toPath(), charset);){
                int ch;
                do {
                    if ((ch = in.read()) >= 0) continue;
                    break block14;
                } while (ch != 10 && ch != 13);
                r = Character.toString((char)ch);
                int ch2 = in.read();
                if (ch2 == 10 || ch2 == 13) {
                    r = r + Character.toString((char)ch2);
                }
            }
        }
        return r;
    }

    public static List<File> findFiles(File dir, FileFilter filter) {
        ArrayList<File> result = new ArrayList<File>();
        HashSet<String> knownDirs = new HashSet<String>();
        FileUtil.findFiles(dir, filter, result, knownDirs);
        return result;
    }

    private static void findFiles(File dir, FileFilter filter, List<File> result, Set<String> knownDirs) {
        try {
            String currDir = dir.getCanonicalPath();
            if (!knownDirs.add(currDir)) {
                return;
            }
        }
        catch (IOException ex) {
            Log.log(ex);
            return;
        }
        File[] list = dir.listFiles();
        if (list != null) {
            for (File f : list) {
                if (f.isDirectory()) {
                    FileUtil.findFiles(f, filter, result, knownDirs);
                    continue;
                }
                if (!filter.accept(f)) continue;
                result.add(f);
            }
        }
    }

    public static String computeRelativePath(File rootDir, File file) throws IOException {
        String rootAbs = rootDir.getAbsolutePath().replace('\\', '/') + '/';
        String fileAbs = file.getAbsolutePath().replace('\\', '/');
        switch (Platform.getOsType()) {
            case WIN32: 
            case WIN64: {
                if (fileAbs.toUpperCase().startsWith(rootAbs.toUpperCase())) break;
                throw new IOException("File '" + file + "' is not under dir '" + rootDir + "'");
            }
            default: {
                if (fileAbs.startsWith(rootAbs)) break;
                throw new IOException("File '" + file + "' is not under dir '" + rootDir + "'");
            }
        }
        return fileAbs.substring(rootAbs.length());
    }

    public static boolean isInPath(File path, File tmxFile) {
        try {
            FileUtil.computeRelativePath(path, tmxFile);
            return true;
        }
        catch (IOException ex) {
            return false;
        }
    }

    public static void copyFilesTo(File destination, File[] toCopy, ICollisionCallback onCollision) throws IOException {
        if (destination.exists() && !destination.isDirectory()) {
            throw new IOException("Copy-to destination exists and is not a directory.");
        }
        Map<File, File> collisions = FileUtil.copyFilesTo(destination, toCopy, (File)null);
        if (collisions.isEmpty()) {
            return;
        }
        ArrayList<File> toReplace = new ArrayList<File>();
        ArrayList<File> toDelete = new ArrayList<File>();
        int count = 0;
        for (Map.Entry<File, File> e : collisions.entrySet()) {
            if (onCollision != null && onCollision.isCanceled()) break;
            if (onCollision == null || onCollision.shouldReplace(e.getValue(), count, collisions.size())) {
                toReplace.add(e.getKey());
                toDelete.add(e.getValue());
            }
            ++count;
        }
        if (onCollision == null || !onCollision.isCanceled()) {
            for (File file : toDelete) {
                FileUtils.forceDelete(file);
            }
            FileUtil.copyFilesTo(destination, toReplace.toArray(new File[toReplace.size()]), (File)null);
        }
    }

    private static Map<File, File> copyFilesTo(File destination, File[] toCopy, File root) throws IOException {
        LinkedHashMap<File, File> collisions = new LinkedHashMap<File, File>();
        for (File file : toCopy) {
            if (destination.getPath().startsWith(file.getPath())) continue;
            File thisRoot = root == null ? file.getParentFile() : root;
            String filePath = file.getPath();
            String relPath = filePath.substring(thisRoot.getPath().length(), filePath.length());
            File dest = new File(destination, relPath);
            if (file.equals(dest)) continue;
            if (dest.exists()) {
                collisions.put(file, dest);
                continue;
            }
            if (file.isDirectory()) {
                FileUtil.copyFilesTo(destination, file.listFiles(), thisRoot);
                continue;
            }
            FileUtils.copyFile(file, dest);
        }
        return collisions;
    }

    public static boolean isRelative(String path) {
        return !RE_ABSOLUTE_LINUX.matcher(path = path.replace('\\', '/')).matches() && !RE_ABSOLUTE_WINDOWS.matcher(path).matches();
    }

    public static String absoluteForSystem(String path, Platform.OsType currentOsType) {
        Matcher m = RE_ABSOLUTE_WINDOWS.matcher(path = path.replace('\\', '/'));
        if (m.matches() && currentOsType != Platform.OsType.WIN32 && currentOsType != Platform.OsType.WIN64) {
            return m.group(1);
        }
        return path;
    }

    public static List<File> buildFileList(File rootDir, boolean recursive) throws IOException {
        int depth = recursive ? Integer.MAX_VALUE : 0;
        try (Stream<Path> stream = Files.find(rootDir.toPath(), depth, (p, attr) -> p.toFile().isFile(), FileVisitOption.FOLLOW_LINKS);){
            List<File> list = stream.map(Path::toFile).sorted(StreamUtil.localeComparator(File::getPath)).collect(Collectors.toList());
            return list;
        }
    }

    public static List<String> buildRelativeFilesList(File rootDir, List<String> includes, List<String> excludes) throws IOException {
        Path root = rootDir.toPath();
        Pattern[] includeMasks = FileUtil.compileFileMasks(includes);
        Pattern[] excludeMasks = FileUtil.compileFileMasks(excludes);
        BiPredicate<Path, BasicFileAttributes> pred = (p, attr) -> p.toFile().isFile() && FileUtil.checkFileInclude(root.relativize((Path)p).toString(), includeMasks, excludeMasks);
        try (Stream<Path> stream = Files.find(root, Integer.MAX_VALUE, pred, FileVisitOption.FOLLOW_LINKS);){
            List<String> list = stream.map(p -> root.relativize((Path)p).toString().replace('\\', '/')).sorted(StreamUtil.localeComparator(Function.identity())).collect(Collectors.toList());
            return list;
        }
    }

    public static boolean checkFileInclude(String filePath, Pattern[] includes, Pattern[] excludes) {
        String normalized = filePath.replace('\\', '/');
        String checkPath = normalized.startsWith("/") ? normalized : '/' + normalized;
        boolean included = Stream.of(includes).map(p -> p.matcher(checkPath)).anyMatch(Matcher::matches);
        boolean excluded = false;
        if (!included) {
            excluded = Stream.of(excludes).map(p -> p.matcher(checkPath)).anyMatch(Matcher::matches);
        }
        return included || !excluded;
    }

    static Pattern[] compileFileMasks(List<String> masks) {
        if (masks == null) {
            return NO_PATTERNS;
        }
        return (Pattern[])masks.stream().map(FileUtil::compileFileMask).toArray(Pattern[]::new);
    }

    static Pattern compileFileMask(String mask) {
        int cp;
        StringBuilder m = new StringBuilder();
        if (!mask.startsWith("/")) {
            mask = "**/" + mask;
        }
        if (mask.endsWith("/")) {
            mask = mask + "**";
        }
        for (int i = 0; i < mask.length(); i += Character.charCount(cp)) {
            cp = mask.codePointAt(i);
            if (cp >= 65 && cp <= 90) {
                m.appendCodePoint(cp);
                continue;
            }
            if (cp >= 97 && cp <= 122) {
                m.appendCodePoint(cp);
                continue;
            }
            if (cp >= 48 && cp <= 57) {
                m.appendCodePoint(cp);
                continue;
            }
            if (cp == 47) {
                if (mask.regionMatches(i, "/**/", 0, 4)) {
                    m.append("(?:/|/.*/)");
                    i += 3;
                    continue;
                }
                if (mask.regionMatches(i, "/**", 0, 3)) {
                    m.append("(?:|/.*)");
                    i += 2;
                    continue;
                }
                m.appendCodePoint(cp);
                continue;
            }
            if (cp == 63) {
                m.append("[^/]");
                continue;
            }
            if (cp == 42) {
                if (mask.regionMatches(i, "**/", 0, 3)) {
                    m.append("(?:|.*/)");
                    i += 2;
                    continue;
                }
                if (mask.regionMatches(i, "**", 0, 2)) {
                    m.append(".*");
                    ++i;
                    continue;
                }
                m.append("[^/]*");
                continue;
            }
            m.append('\\').appendCodePoint(cp);
        }
        return Pattern.compile(m.toString());
    }

    public static List<String> getUniqueNames(List<String> paths) {
        boolean didTrim;
        ArrayList<String> fullPaths = new ArrayList<String>(paths);
        fullPaths.replaceAll(p -> Optional.ofNullable(FilenameUtils.normalizeNoEndSeparator(p, true)).orElse(""));
        ArrayList<String> working = new ArrayList<String>(fullPaths);
        working.replaceAll(p -> StringUtil.getTailSegments(p, 47, 1));
        if (working.size() == 1 && !((String)working.get(0)).isEmpty()) {
            return working;
        }
        int[] segments = new int[fullPaths.size()];
        Arrays.fill(segments, 1);
        do {
            didTrim = false;
            int[] counts = new int[working.size()];
            Arrays.setAll(counts, i -> Collections.frequency(working, working.get(i)));
            for (int i2 = 0; i2 < counts.length; ++i2) {
                if (counts[i2] <= 1) continue;
                String curr = (String)working.get(i2);
                int n = i2;
                String trimmed = StringUtil.getTailSegments((String)fullPaths.get(i2), 47, segments[n] = segments[n] + 1);
                if (curr.equals(trimmed)) continue;
                working.set(i2, trimmed);
                didTrim = true;
            }
        } while (didTrim);
        for (int i3 = 0; i3 < working.size(); ++i3) {
            if (!((String)working.get(i3)).isEmpty()) continue;
            working.set(i3, paths.get(i3));
        }
        return working;
    }

    public static class TmFileComparator
    implements Comparator<String> {
        private static final String AUTO_PREFIX = "auto/";
        private static final String ENFORCE_PREFIX = "enforce/";
        private final File tmRoot;

        public TmFileComparator(File tmRoot) {
            this.tmRoot = tmRoot;
        }

        @Override
        public int compare(String n1, String n2) {
            String r2;
            String r1;
            try {
                r1 = FileUtil.computeRelativePath(this.tmRoot, new File(n1));
                r2 = FileUtil.computeRelativePath(this.tmRoot, new File(n2));
            }
            catch (IOException e) {
                return n1.compareTo(n2);
            }
            int c = r1.startsWith(ENFORCE_PREFIX) && r2.startsWith(ENFORCE_PREFIX) ? n1.compareTo(n2) : (r1.startsWith(ENFORCE_PREFIX) ? -1 : (r2.startsWith(ENFORCE_PREFIX) ? 1 : (r1.startsWith(AUTO_PREFIX) && r2.startsWith(AUTO_PREFIX) ? n1.compareTo(n2) : (r1.startsWith(AUTO_PREFIX) ? -1 : (r2.startsWith(AUTO_PREFIX) ? 1 : n1.compareTo(n2))))));
            return c;
        }
    }

    public static interface ICollisionCallback {
        public boolean isCanceled();

        public boolean shouldReplace(File var1, int var2, int var3);
    }
}

