/* GrandPerspective, Version 2.6.3 
 *   A utility for Mac OS X that graphically shows disk usage. 
 * Copyright (C) 2005-2022, Erwin Bonsma 
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version. 
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
 * more details. 
 * 
 * You should have received a copy of the GNU General Public License along 
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. 
 */

#import <Cocoa/Cocoa.h>


@class FilterTestRef;
@class FilterTestRepository;
@class FileItemTest;

/* A file item filter. It consists of one or more filter tests. The filter test succeeds when any of
 * its subtest succeed (i.e. the subtests are combine using the OR operator). Each filter subtest
 * can optionally be inverted.
 *
 * The subtests are referenced by name, which means that the FileItemTest that represents this
 * filter is affected by any changes to used filter tests. The current file item test can be
 * obtained using -createFileItemTestFromRepository:.
 */
@interface Filter : NSObject {
 }

+ (instancetype) filter;
+ (instancetype) filterWithFilterTests:(NSArray *)filterTests;
+ (instancetype) filterWithFilter:(Filter *)filter;

/* Creates a filter from a dictionary as generated by -dictionaryForObject.
 */
+ (Filter *)filterFromDictionary:(NSDictionary *)dict;

/* Initialises an empty filter with an automatically generated name.
 */
- (instancetype) init;

/* Initialises the filter with the given filter tests. The tests should be instances of
 * FilterTestRef.
 */
- (instancetype) initWithFilterTests:(NSArray *)filterTests NS_DESIGNATED_INITIALIZER;

/* Initialises the filter based on the provided one. The newly created filter will, however, not yet
 * have an instantiated file item test. When the test is (eventually) created using
 * -createFileItemTestFromRepository:, it will be based on the tests as then defined in the
 * repository.
 */
- (instancetype) initWithFilter:(Filter *)filter;

@property (nonatomic, readonly) NSUInteger numFilterTests;
@property (nonatomic, readonly, copy) NSArray *filterTests;
- (FilterTestRef *)filterTestAtIndex:(NSUInteger)index;
- (FilterTestRef *)filterTestWithName:(NSString *)name;
- (NSUInteger) indexOfFilterTest:(FilterTestRef *)test;


/* Creates and returns the test object that represents the filter given the tests currently in the
 * default test repository.
 *
 * If any test cannot be found in the repository its name will be added to "unboundTests".
 */
- (FileItemTest *)createFileItemTestUnboundTests:(NSMutableArray *)unboundTests;

/* Creates and returns the test object that represents the filter given the tests currently in the
 * test repository.
 *
 * If any test cannot be found in the repository its name will be added to "unboundTests".
 */
- (FileItemTest *)createFileItemTestFromRepository:(FilterTestRepository *)repository
                                      unboundTests:(NSMutableArray *)unboundTests;

/* Returns a dictionary that represents the object. It can be used for storing the object to
 * preferences.
 */
- (NSDictionary *)dictionaryForObject;

@end // @interface Filter
