/* GrandPerspective, Version 2.6.0 
 *   A utility for Mac OS X that graphically shows disk usage. 
 * Copyright (C) 2005-2020, Erwin Bonsma 
 * 
 * This program is free software; you can redistribute it and/or modify it 
 * under the terms of the GNU General Public License as published by the Free 
 * Software Foundation; either version 2 of the License, or (at your option) 
 * any later version. 
 * 
 * This program is distributed in the hope that it will be useful, but WITHOUT 
 * ANY WARRANTY; without even the implied warranty of MERCHANTABILITY or 
 * FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General Public License for 
 * more details. 
 * 
 * You should have received a copy of the GNU General Public License along 
 * with this program; if not, write to the Free Software Foundation, Inc., 
 * 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA. 
 */

#import <Cocoa/Cocoa.h>

@class FileItem;
@class PlainFileItem;
@protocol FileItemMappingScheme;

/* An implementation of a particular file item mapping scheme. It can map file items to hash values.
 *
 * Implementations are not (necessarily) thread-safe. Each thread should get an instance it can
 * safely use by invoking -fileItemMapping on the file item mapping scheme.
 */
@protocol FileItemMapping

@property (nonatomic, readonly, strong) NSObject<FileItemMappingScheme> *fileItemMappingScheme;

/* Calculates a hash value for a file item in a tree, when the item is encountered while traversing
 * the tree. The calculation may use the "depth" of the file item relative to the root of the tree,
 * as provided by the TreeLayoutBuilder to the TreeLayoutTraverser.
 *
 * For calculating the hash value when not traversing a tree, use -hashForFileItem:inTree:.
 */
- (NSUInteger) hashForFileItem:(PlainFileItem *)item atDepth:(NSUInteger)depth;

/* Calculates a hash value for a given file item in a tree. It performs the same calculation as
 * -hashForFileItem:depth:. Unlike the latter method, this one can be used when a tree is not being
 * traversed (and the "depth" of the item is not easily available). The depth will be calculated
 * relative to the provided tree root.
 */
- (NSUInteger) hashForFileItem:(PlainFileItem *)item inTree:(FileItem *)treeRoot;

/* Returns "YES" iff there are meaningful descriptions for each hash value. In this case, the range
 * of hash values is expected to be the consecutive numbers from zero upwards, as many as are
 * needed. For each these values, the method -descriptionForHash will provide a short descriptive
 * string.
 */
@property (nonatomic, readonly) BOOL canProvideLegend;

@end


/* Informal protocol to be implemented by FileItemMapping schemes for which -canProvideLegend
 * returns "YES".
 */
@interface LegendProvidingFileItemMapping

/* Short descriptive string for the given hash value. Returns "nil" if no description can be given
 * (i.e. when -canProvideLegend returns "NO"), or if the hash value is outside of the valid range.
 */
- (NSString *)descriptionForHash:(NSUInteger)hash;

@property (nonatomic, readonly, copy) NSString *descriptionForRemainingHashes;

@end
