RCS_ID("$Id: FFImageWindow.m 595 2006-09-30 19:11:14Z ravemax $")

#import "FFImageWindow.h"
#import "FFImageList.h"
#import "FFPreferences.h"
#import "FFPageView.h"
#import "NSWindow_Additions.h"

@implementation FFImageWindow

static const long STAY_SNAPPED_TIMEOUT	= 1; // seconds

- (id)initWithContentRect:(NSRect)rect andStyleMask:(unsigned int)style {
	self = [super initWithContentRect:rect styleMask:style backing:NSBackingStoreBuffered defer:FALSE];
	if (self) {
		m_nc				= [NSNotificationCenter defaultCenter];
		m_prevViewSize		= NSMakeSize(0.0f, 0.0f);
		m_prevFrameSize		= NSMakeSize(0.0f, 0.0f);
		m_prevKeepARatio	= FALSE;
		m_snapTolerance		= [[FFPreferences instance] imageWinSnapTolerance];
		m_alreadySnapping	= FALSE;
		
		memset(m_snapAllowed, TRUE, NUM_SIDES);
		memset(m_staySnapped, FALSE, NUM_SIDES);
		
		[m_nc addObserver:self selector:@selector(windowDidMove:)
					 name:NSWindowDidMoveNotification object:self];
	}
	return self;
}

- (void)dealloc {
    [m_nc removeObserver:self name:NSWindowDidMoveNotification object:self];
    [super dealloc];
}

- (void)setFrame:(NSRect)frameRect display:(BOOL)flag {
	int ya = (int)(frameRect.size.height - [self widgetHeight]) % 2; // 0 or 1 if odd

	frameRect.size.width	+= (int)frameRect.size.width % 2; // Never odd
	frameRect.size.height	+= ya;
	frameRect.origin.y		-= ya; // No clue why, but w/o the frame goes up

	FFLOG(7, @"setFrame: %@ (widgetHt=%4.1f)", NSStringFromRect(frameRect), [self widgetHeight])

	[super setFrame:frameRect display:flag];
}

- (void)adjustViewSize:(NSSize)size minWidth:(float)minWd 
			   refresh:(FFRefresh)refresh
	   keepAspectRatio:(BOOL)aratio andCenter:(BOOL)center {

	FFLOG(8, @"adjustView:%@ minWd:%4.1f keepARatio:%d",
		  NSStringFromSize(size), minWd, aratio);

	BOOL newViewSize = !NSEqualSizes(size, m_prevViewSize) || (aratio != m_prevKeepARatio);

	if (newViewSize || !NSEqualSizes([self frame].size, m_prevFrameSize)) {
		NSRect	newFrame;
		NSRect	screenFrame	= [[self screen] visibleFrame];
		NSSize	screenSize	= screenFrame.size;
		
		if (newViewSize) {
			float widgetHt = [self widgetHeight];
			
			// Store the new view size & keep aspect ratio
			m_prevViewSize		= size;
			m_prevKeepARatio	= aratio;

			// Never odd view-width and -height (managed by setFrame: if resized)
			size.width	+= (int)size.width % 2;			
			size.height	+= (int)size.height % 2;

			// Height 
			if (size.height < (screenSize.height - widgetHt))
				newFrame.size.height = size.height + widgetHt;
			else
				newFrame.size.height = screenSize.height - ((int)widgetHt % 2);
			
			// Width
			if (size.width < screenSize.width)
				newFrame.size.width	= MAX(minWd, size.width);
			else
				newFrame.size.width = screenSize.width;

			// Frame size didn't change - relayout, display and return
			if (NSEqualSizes([self frame].size, newFrame.size)) {
				FFLOG(7, @"- frame size didn't change");
				if (refresh == REFRESH_ALWAYS)
					[(FFPageView*)[self contentView] doLayoutAndDisplay];
				return;
			}			
			
			// Aspect ratio?
			if (aratio) {
				float wv = newFrame.size.width / size.width; // max. 1.0
				float hv = (newFrame.size.height - widgetHt) / size.height;
				
				if (wv > hv)
					newFrame.size.width = ceilf(size.width * hv);
				else if (wv < hv)
					newFrame.size.height = ceilf(size.height * wv);
				
				[self setContentAspectRatio:size];
			}
		} else
			newFrame.size = m_prevFrameSize;
		
		// Calc origin
		if (center) {
			newFrame.origin.x   = screenFrame.origin.x + (screenSize.width - newFrame.size.width) / 2;
			newFrame.origin.y   = screenFrame.origin.y + (screenSize.height - newFrame.size.height) / 2;
		} else {
			NSRect	wr = [self frame];
			float	endX;

			// X
			endX = wr.origin.x - screenFrame.origin.x + newFrame.size.width;
			if (endX <= screenSize.width)
				newFrame.origin.x	= wr.origin.x; // Keep X
			else
				newFrame.origin.x	= wr.origin.x - (endX - screenSize.width); // Move to the left

			// Y (try to stick to the top)
			newFrame.origin.y = MAX(screenFrame.origin.y, wr.origin.y + wr.size.height - newFrame.size.height);
		}
	
		// Integral part of x,y - else Cocoa modifies the content size :-/
		newFrame.origin.x = (float)((int)(newFrame.origin.x));
		newFrame.origin.y = (float)((int)(newFrame.origin.y));
		
		FFLOG(8, @"- newFrame: %@", NSStringFromRect(newFrame));

		// Store size and set it		
		m_prevFrameSize = newFrame.size;

		// Workaround for a NSView bug
		if (refresh != REFRESH_NEVER) {
			[(FFPageView*)[self contentView] setNeedsLayout:TRUE];
			[self setFrame:newFrame display:TRUE];
			FFLOG(8, @"- needsLayout: %d", [(FFPageView*)[self contentView] needsLayout])
			[(FFPageView*)[self contentView] doLayoutAndDisplayIfNeeded];
		} else
			[self setFrame:newFrame display:FALSE];
		
	// Wasn't resized
	} else if (refresh == REFRESH_ALWAYS)
		[(FFPageView*)[self contentView] doLayoutAndDisplay]; // _contentView also works but it's "private"

}

- (void)adjustViewSize:(NSSize)size refresh:(FFRefresh)refresh
	   keepAspectRatio:(BOOL)aratio andCenter:(BOOL)center {

	[NSException raise:[self className] format:@"adjustViewSize must be implemented by the subclasses"];
}

- (float)widgetHeight {
	[NSException raise:[self className] format:@"adjustViewSize must be implemented by the subclasses"];
	return 0.0f;
}


- (BOOL)canBecomeKeyWindow { return TRUE; }

- (void)mouseDragged:(NSEvent*)event {
	NSRect  fr		= [self frame];
	NSPoint newOrg;
	
	newOrg.x = fr.origin.x + [event deltaX];
	newOrg.y = fr.origin.y - [event deltaY];
	
	[self setFrameOrigin:newOrg];
}

- (void)orderOut:(id)sender {
	// Remove list window
	if ([[self childWindows] count] == 1)
		[self removeChildWindow:(NSWindow*)[[self childWindows] objectAtIndex:0]];
	
	[super orderOut:sender];
}

#pragma mark -
#pragma mark Mini image (minimized)

- (void)_generateMiniWindow {
	FFPageView*			pageView;
	int					wdAndHt;
	NSImage*			img;
	NSBitmapImageRep*   rep;
	
	pageView = (FFPageView*)[self contentView];
	wdAndHt = MIN(pageView->width, pageView->height);

	// Matching representation
	rep = [[[NSBitmapImageRep alloc] 
			initWithBitmapDataPlanes:NULL
						  pixelsWide:wdAndHt
						  pixelsHigh:wdAndHt
					   bitsPerSample:8
					 samplesPerPixel:4
							hasAlpha:TRUE
							isPlanar:FALSE
					  colorSpaceName:NSCalibratedRGBColorSpace
						 bytesPerRow:wdAndHt*4
						bitsPerPixel:32] autorelease];
	
	// Read the buffer
	[pageView readBufferWithWidth:wdAndHt andHeight:wdAndHt intoBuffer:[rep bitmapData]];
	
	// Create and set mini image
	img= [[[NSImage alloc] initWithSize:NSMakeSize(wdAndHt, wdAndHt)] autorelease];
	[img addRepresentation:rep];
	[img setFlipped:TRUE];
	
	[self setMiniwindowImage:img];	
}

- (void)miniaturize:(id)sender {
	[m_nc postNotificationName:StopSlideshowNotificaton object:nil];
	[self _generateMiniWindow];
	[super miniaturize:sender];
}

#pragma mark -
#pragma mark Snapping

- (BOOL)_snapToSide:(int)side coordinate:(float*)coord to:(float)dest inPoint:(NSPoint*)pnt {
	if (m_snapAllowed[side]) {
		m_snapAllowed[side] = FALSE;
		m_staySnapped[side] = TRUE;
		gettimeofday(&m_timeOnSnap[side], NULL);
		
	} else if (m_staySnapped[side]) {
		struct timeval ct;
		gettimeofday(&ct, NULL);
		if ((ct.tv_sec - m_timeOnSnap[side].tv_sec) > STAY_SNAPPED_TIMEOUT) {
			m_staySnapped[side] = FALSE;
			return FALSE;
		}
	} else
		return FALSE;
	
	*coord = dest;
	return TRUE;
}

- (void)windowDidMove:(NSNotification*)_unused {
	NSRect	winFrame	= [self frame];
	NSPoint	aPoint		= winFrame.origin;
	NSRect	screenFrame	= [[self screen] frame];
	float	menuHt		= [self isPrimaryScreen] ? [NSMenuView menuBarHeight] : 0.0f;
	BOOL	orgModified	= FALSE;
    
    // Don't snap if Option/Alt key is held down during drag.
	if ([[NSApp currentEvent] modifierFlags] & NSAlternateKeyMask)
		return;
	
    if ((m_snapTolerance > 0) && !m_alreadySnapping) {
		m_alreadySnapping = TRUE; // so we don't keep getting NSWindowDidMoveNotifications whilst we are snapping the window

	// Left hand side of the screen
	if (winFrame.origin.x - screenFrame.origin.x < m_snapTolerance)
		orgModified |= [self _snapToSide:SIDE_LEFT coordinate:&(aPoint.x)
									  to:screenFrame.origin.x inPoint:&aPoint];

		else {
			m_snapAllowed[SIDE_LEFT] = TRUE;
			
			// Right hand side of the screen
			if (screenFrame.size.width - (winFrame.size.width + winFrame.origin.x - screenFrame.origin.x) < m_snapTolerance)
				orgModified |= [self _snapToSide:SIDE_RIGHT coordinate:&(aPoint.x) 
											  to:(screenFrame.size.width - winFrame.size.width + screenFrame.origin.x)
										 inPoint:&aPoint];
			else
				m_snapAllowed[SIDE_RIGHT] = TRUE;
		}
		
		// Bottom of the screen
		if (winFrame.origin.y - screenFrame.origin.y < m_snapTolerance)		
			orgModified |= [self _snapToSide:SIDE_BOTTOM coordinate:&(aPoint.y)
										to:screenFrame.origin.y inPoint:&aPoint];
		else {
			m_snapAllowed[SIDE_BOTTOM] = TRUE;
			
			// Top of the screen
			if (screenFrame.size.height - menuHt - (winFrame.size.height + winFrame.origin.y - screenFrame.origin.y) < m_snapTolerance)
				orgModified |= [self _snapToSide:SIDE_TOP coordinate:&(aPoint.y)
											  to:(screenFrame.size.height - menuHt - winFrame.size.height + screenFrame.origin.y)
										 inPoint:&aPoint];
			else
				m_snapAllowed[SIDE_TOP] = TRUE;
		}
		
		if (orgModified)
			[self setFrameOrigin:aPoint];	
		
		m_alreadySnapping = FALSE;
    }
	
}

@end

