// The glue to put it alltogether

RCS_ID("$Id: FFMainController.m 610 2007-08-26 19:40:40Z ravemax $")

#import "FFMainController.h"
#import "FFPrefsController.h"
#import "FFImageListController.h"
#import "FFPreferences.h"
#import "FFOptions.h"
#import "FFImageList.h"
#import "FFActionHandler.h"
#import "FFDockIcon.h"
#import "FFSupport.h"
#import "FFApplication.h"
#import "FFInputEvent.h"
#import "FFEXIFController.h"
#import "FFRecentList.h"
#import "FFOptions_Scriptable.h"
#import "FFImageList_Scriptable.h"
#import "FFScriptCommands.h"
#import "FFScriptMenu.h"
#import "FFDirectory.h"
#import "FFSmartFolder.h"

@implementation FFMainController

// Quicksave constant
static const NSString*	QuickSaveFilename = @"QuickSave"; // w/o extension

#pragma mark -
#pragma mark Initialisation and cleanup

- (void)_setupPrefs {
	m_prefs = [FFPreferences instance];
	m_prefsCtrl = nil;
}

- (void)_setupScriptMenu {
	NSMenuItem*		mitem, *newItem;
	NSMenu*			menu;
	NSEnumerator*	en;
	NSString*		name;
	unsigned		numEv, idxEv;
	FFInputEvent*	ev;

	// Load the script
	m_scriptMenu	= [[FFScriptMenu alloc] initWithAppDirectory:[NSApp supportDirectory]
													  andOptions:m_opts];

	// Add the funky script icon
	mitem = (NSMenuItem*)[[NSApp mainMenu] itemWithTag:MENU_SCRIPTS];
	[mitem setImage:[NSImage imageNamed:@"script_menu"]];
	
	// Add the script menu items
	menu	= [mitem submenu];
	en		= [m_scriptMenu	nameEnumerator];
	while (name = [en nextObject]) {
		newItem = [[NSMenuItem alloc] initWithTitle:name action:@selector(executeScriptAttachedToMenuItem:)
									  keyEquivalent:@""];
		[newItem setTarget:m_scriptMenu];
		[newItem setTag:MENU_SCRIPTS]; // Important. Used to detect scripts
		[menu addItem:[newItem autorelease]];
	}
	
	// Remove the shortcuts of the non existing scripts
	numEv	= [m_prefs numberOfInputEvents];
	idxEv	= 0;

	while (idxEv < numEv) {
		ev = [m_prefs inputEventAtIndex:idxEv];
		if ([ev isAssignedToScript]) {
			if ([m_scriptMenu isKnownScript:[ev script]])
				idxEv++;
			else {
				[m_prefs removeInputEvent:ev];
				numEv--;
			}

		// Not an event assigned to a script
		} else
			idxEv++;
	}
}

- (void)_setupMenuKeyEquivAndSpeechWithInputEvents:(NSNotificationCenter*)nc {
	NSEnumerator*	en = [m_prefs inputEventEnumerator];
	FFInputEvent*	ev;
	NSMenu*			menu = [NSApp mainMenu]; 
	FFMenuAction	lastAction = 0;

	// We need something to store the speech commands
	m_speechCommands = [[NSMutableDictionary alloc] init];

	// Set key equiv. or add speech command
	while (ev = [en nextObject]) {
		// Add speech command
		if ([ev type] == EVENT_SPEECH)
			[m_speechCommands setObject:ev forKey:[ev command]];
		
		// Nothing set yet
		else if (([ev action] != lastAction) && 
			[self _addInputEvent:ev toMenu:menu])
			lastAction = [ev action];
	}

	// Speech recognition en/disabling is done by the options code
	[self _updateSpeechCommands];
}

- (void)_setupOptions:(NSNotificationCenter*)nc {
	m_opts = [[FFOptions alloc] initWithPrefs:m_prefs];
	
	m_rotationItems[OPT_NO_ROTATION]	= m_menuNoRotation;
	m_rotationItems[OPT_ROTATE_LEFT]	= m_menuRotateLeft;
	m_rotationItems[OPT_ROTATE_RIGHT]   = m_menuRotateRight;
	m_rotationItems[OPT_ROTATE_180]		= m_menuRotate180;		
	m_scalingItems[OPT_NO_SCALING]		= m_menuNoScaling;
	m_scalingItems[OPT_FIT_TO_WINDOW]	= m_menuFitToWindow;
	m_scalingItems[OPT_PAGEWIDTH]		= m_menuPagewidth;
	m_pageModeItems[OPT_MODE_SINGLE]	= m_menuSingle;
	m_pageModeItems[OPT_MODE_WESTERN]	= m_menuWestern;
	m_pageModeItems[OPT_MODE_MANGA]		= m_menuManga;
		
	[self updateRotationMenuItems:nil];
	[self updateScalingMenuItems:nil];
	[self updatePageModeMenuItems:nil];
	[self updateNoBlowUpMenuItem:nil];
	[self updateAntialiasingMenuItem:nil];
	[self updateSpeechOnMenuItem:nil];
  	
	[nc addObserver:self selector:@selector(updateRotationMenuItems:)
			   name:RotationChangedNotification object:nil];
	[nc addObserver:self selector:@selector(updateScalingMenuItems:)
			   name:ScalingChangedNotification object:nil];
	[nc addObserver:self selector:@selector(updatePageModeMenuItems:)
			   name:PageModeChangedNotification object:nil];
	[nc addObserver:self selector:@selector(updateNoBlowUpMenuItem:)
			   name:NoBlowUpChangedNotification object:nil];
	[nc addObserver:self selector:@selector(updateAntialiasingMenuItem:) 
			   name:AntialiasingChangedNotification object:nil];
	[nc addObserver:self selector:@selector(updateSpeechOnMenuItem:)
			   name:SpeechOnChangedNotification object:nil];
	[nc addObserver:self selector:@selector(updateZoomLevelMenuItem:)
			   name:ZoomMultiplierChangedNotification object:nil];
}

- (void)_setupRecentListAndImageList:(NSNotificationCenter*)nc {	
	// Recent list
	m_recentList = [[FFRecentList alloc] initWithUserDefaults];
	[self updateRecentMenu:nil];
	[nc addObserver:self selector:@selector(updateRecentMenu:)
			   name:RecentListChangedNotification object:nil];

	// Image list
	m_imgList = [[FFImageList alloc] initWithPreferences:m_prefs options:m_opts
										   andRecentList:m_recentList];
	[m_imgListCtrl setImageList:m_imgList];
}

- (void)_setupScripting {
	[m_opts registerProperties];
	[m_imgList registerPropertiesAndElements];
	[FFScriptCommands setMainController:self andImageList:m_imgList];
}

- (void)_tryToLoadQuickSave {
	NSFileManager*	fm		= [NSFileManager defaultManager];
	NSString*		path	= [[NSApp supportDirectory] stringByAppendingPathComponent:
								[QuickSaveFilename stringByAppendingPathExtension:[FFImageList fileExtension]]];

	if ([fm fileExistsAtPath:path]) {
		[m_imgList addImageList:path restoreSelection:TRUE addToRecent:FALSE postNotifications:TRUE];
		[fm removeFileAtPath:path handler:nil];
	}
}

- (void)awakeFromNib {
	NSNotificationCenter*   nc = [NSNotificationCenter defaultCenter];
	
	[self _setupPrefs];
	[NSApp setDelegate:self];
	[NSApp setupInputEvents]; // must before _setupOptions
	[NSApp setMainController:self];
	[self _setupOptions:nc];
	[self _setupScriptMenu];
	[self _setupMenuKeyEquivAndSpeechWithInputEvents:nc];
	[self _setupRecentListAndImageList:nc];
		
	// One liners
	[self updateZoomLevelMenuItem:nil];
	
	m_actionHandler = [[FFActionHandler alloc] initWithOptions:m_opts
					  preferences:m_prefs andImageList:m_imgList];
	[m_imgListCtrl setWindowToSnap:[m_actionHandler imageWindow] 
					  andTolerance:[m_prefs listWinSnapTolerance]];
	
	m_dockIcon  = [[FFDockIcon alloc] init];
		
	[self _setupScripting];
	[self _tryToLoadQuickSave];
}

- (void)dealloc {
	[m_actionHandler release];
	[m_dockIcon release];
	if (m_speechRecog != nil) {
		[m_speechRecog stopListening];
		[m_speechRecog release];
	}
	[m_speechCommands release];
	[m_imgList release];
	[m_recentList release];
	[m_opts release];
	if (m_prefsCtrl != nil)
		[m_prefsCtrl release];
	[m_scriptMenu release];
	[m_prefs release];

	[[NSNotificationCenter defaultCenter] removeObserver:self];
	
	[super dealloc];
}

#pragma mark -
#pragma mark IB Action & menu handling (action, validation)

- (void)_showPreferences {
	m_prefsCtrl = [FFPrefsController showWithExitingController:m_prefsCtrl andPreferences:m_prefs];
}

- (void)_quickSaveAndQuit {
	if ([m_imgList numberOfImages] > 0) {
		[m_imgList save:[[NSApp supportDirectory] stringByAppendingPathComponent:
			[QuickSaveFilename stringByAppendingPathExtension:[FFImageList fileExtension]]]];
	}

	[NSApp terminate:self];
}


- (void)_showWithBrowser {
	NSString*   helpFolder  = [[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleHelpBookFolder"];
	NSString*   rsrcPath	= [[NSBundle mainBundle] resourcePath];
	NSString*   indexPath   = [NSString stringWithFormat:@"%@/English.lproj/%@/%@.html",
								rsrcPath, helpFolder, helpFolder];
	
	(void)[[NSWorkspace sharedWorkspace] openFile:indexPath];	
}
					
- (void)executeMenuAction:(FFMenuAction)action {
	switch (action) {
		#define MENU_CASE(ITEM, ACTION) \
			case (MENU_##ITEM) : \
				ACTION; \
				break
		
		#define ZOOM_ACTION(DIR) \
			if ([m_opts scaling] == OPT_NO_SCALING) \
				[m_opts zoom##DIR]; \
			else \
				[m_opts setScaling:OPT_NO_SCALING]
				
		// App
		MENU_CASE(APP_FEEDBACK, [FFSupport sendFeedbackToMailaddress:@"info@feedface.com"]);
		MENU_CASE(APP_DONATE_POSTCARD, [FFSupport donatePostcard]);
		MENU_CASE(APP_CHECK_UPDATES, [FFSupport updatesCheckForProject:@"ffview"]);
		MENU_CASE(APP_PREFERENCES, [self _showPreferences]);
		MENU_CASE(APP_HIDE_FFVIEW, [NSApp hide:self]);
		MENU_CASE(APP_HIDE_OTHERS, [NSApp hideOtherApplications:self]);
		MENU_CASE(APP_SHOW_ALL, [NSApp unhideAllApplications:self]);
		MENU_CASE(APP_QUICKSAVE_AND_QUIT, [self _quickSaveAndQuit]);
		MENU_CASE(APP_QUIT_FFVIEW, [NSApp terminate:self]);
		
		// File		
		MENU_CASE(FILE_OPEN, [m_imgListCtrl openAddSheet:FALSE]);
		MENU_CASE(FILE_SAVE_IMGLIST, [m_imgListCtrl openSaveSheet]);
		MENU_CASE(FILE_SHUFFLE_IMGLIST, [m_imgList shuffle]);
		MENU_CASE(FILE_CLEAR_IMGLIST, [m_imgList clear]);
		MENU_CASE(FILE_REMOVE, [m_imgList removeCurrentImage]);
		MENU_CASE(FILE_REVEAL_IN_FINDER, [m_imgList revealInFinder]);
		MENU_CASE(FILE_MOVE_TO_TRASH, [m_imgList removeAndTrashCurrentImage]);
		MENU_CASE(FILE_COPY_SELECTED, [m_imgListCtrl copySelected]);
		MENU_CASE(FILE_CLEAR_RECENT, [m_recentList clearAndSendNotification:TRUE]);
		MENU_CASE(FILE_FINDER_ICON, [m_imgList useImageAtIndexAsFinderIcon:[m_imgListCtrl indexOfSingleSelectedImage]]);
				
		// View
		MENU_CASE(VIEW_FULLSCREEN, if ([m_imgList numberOfImages] > 0) [m_opts toggleScreenMode]);
		MENU_CASE(VIEW_PREV_IMG, [m_imgList selectPreviousImage]);
		MENU_CASE(VIEW_NEXT_IMG, [m_imgList selectNextImage]);
		MENU_CASE(VIEW_FIRST_IMG, [m_imgList selectFirstImage]);
		MENU_CASE(VIEW_LAST_IMG, [m_imgList selectLastImage]);
		MENU_CASE(VIEW_PAGE_UP, [m_actionHandler pageUp]);
		MENU_CASE(VIEW_PAGE_DOWN, [m_actionHandler pageDown]);
		MENU_CASE(VIEW_FOLDER_DRAWER, [m_imgListCtrl toggleDirDrawer]);
		MENU_CASE(VIEW_PREV_FOLDER, [m_imgListCtrl openPrevDir]);
		MENU_CASE(VIEW_PREV_NEXT, [m_imgListCtrl openNextDir]);
		MENU_CASE(VIEW_ACTUAL_SIZE, ZOOM_ACTION(Reset));
		MENU_CASE(VIEW_ZOOM_IN, ZOOM_ACTION(In));
		MENU_CASE(VIEW_ZOOM_OUT, ZOOM_ACTION(Out));
		MENU_CASE(VIEW_START_SLIDESHOW, [m_imgList startSlideshowWithInterval:[m_prefs slideshowTime] endless:FALSE]);
		MENU_CASE(VIEW_ENDLESS_SLIDESHOW, [m_imgList startSlideshowWithInterval:[m_prefs slideshowTime] endless:TRUE]);
		MENU_CASE(VIEW_STOP_SLIDESHOW, [m_imgList stopSlideshow]);
		MENU_CASE(VIEW_OSD, [m_opts toggleOSDVisibility]);
		MENU_CASE(VIEW_MAG_LENSE, [m_opts toggleMagnifyingLensVisibility]);
		
		// Options
		MENU_CASE(OPT_NO_ROTATION, [m_opts setRotation:OPT_NO_ROTATION]);
		MENU_CASE(OPT_ROTATE_90, [m_opts setRotation:OPT_ROTATE_RIGHT]);
		MENU_CASE(OPT_ROTATE_180, [m_opts setRotation:OPT_ROTATE_180]);
		MENU_CASE(OPT_ROTATE_270, [m_opts setRotation:OPT_ROTATE_LEFT]);
		MENU_CASE(OPT_NO_SCALING, [m_opts setScaling:OPT_NO_SCALING]);
		MENU_CASE(OPT_FIT_WIN, [m_opts setScaling:OPT_FIT_TO_WINDOW]);
		MENU_CASE(OPT_PAGEWIDTH, [m_opts setScaling:OPT_PAGEWIDTH]);						
		MENU_CASE(OPT_SINGLE, [m_opts setPageMode:OPT_MODE_SINGLE]);
		MENU_CASE(OPT_WESTERN, [m_opts setPageMode:OPT_MODE_WESTERN]);
		MENU_CASE(OPT_MANGA, [m_opts setPageMode:OPT_MODE_MANGA]);
		MENU_CASE(OPT_NO_BLOW_UP, [m_opts setNoBlowUp:([m_opts noBlowUp] ^ TRUE)]);
		MENU_CASE(OPT_ANTIALIASING, [m_opts setAntialiasing:([m_opts antialiasing] ^ TRUE)]);
		MENU_CASE(OPT_SPEECH_ON, [m_opts setSpeechOn:([m_opts speechOn] ^ TRUE)]);
	
		// Navigation
		MENU_CASE(NAVI_PAN_LEFT, [m_actionHandler panLeft]);
		MENU_CASE(NAVI_PAN_LEFT_ACCEL, [m_actionHandler panLeftAccel]);
		MENU_CASE(NAVI_PAN_RIGHT, [m_actionHandler panRight]);
		MENU_CASE(NAVI_PAN_RIGHT_ACCEL, [m_actionHandler panRightAccel]);
		MENU_CASE(NAVI_PAN_UP, [m_actionHandler panUp]);
		MENU_CASE(NAVI_PAN_UP_ACCEL, [m_actionHandler panUpAccel]);
		MENU_CASE(NAVI_PAN_DOWN, [m_actionHandler panDown]);
		MENU_CASE(NAVI_PAN_DOWN_ACCEL, [m_actionHandler panDownAccel]);
		
		// Window
		MENU_CASE(WIN_CLOSE_WINDOW, if (![m_opts fullscreen]) [[NSApp keyWindow] performClose:self]);
		MENU_CASE(WIN_ZOOM_WINDOW, if (![m_opts fullscreen]) [[NSApp keyWindow] zoom:self]);
		MENU_CASE(WIN_MINIMIZE_WINDOW, if (![m_opts fullscreen]) [[NSApp keyWindow] miniaturize:self]);
		MENU_CASE(WIN_IMAGE_LIST, [m_imgListCtrl show]);
		MENU_CASE(WIN_IMAGE, [m_actionHandler showImageWindow]);
		MENU_CASE(WIN_EXIF, [FFEXIFController showPanelWithImageList:m_imgList]);
		MENU_CASE(WIN_BRING_ALL_TO_FRONT, [NSApp arrangeInFront:self]);
		MENU_CASE(WIN_HIDE_TOOLBAR, [[NSApp keyWindow] toggleToolbarShown:self]);
		MENU_CASE(WIN_CUSTOMIZE_TOOLBAR, [[NSApp keyWindow] runToolbarCustomizationPalette:self]);

		// Help
		MENU_CASE(HELP_HELP, [NSApp showHelp:self]);
		MENU_CASE(HELP_BROWSER, [self _showWithBrowser]);
		MENU_CASE(HELP_DEBUG_STATUS, [m_actionHandler showInternalStatus]);
		
		// Scripts
		MENU_CASE(SCRIPTS_OPEN_FOLDER, [m_scriptMenu openScriptFolder]);
				
		default:
			; // do nothing
	}
}

- (void)executeEvent:(FFInputEvent*)event {
	if ([event isAssignedToScript])
		[m_scriptMenu executeScriptWithName:[event script]];
	else
		[self executeMenuAction:[event action]];
}

- (IBAction)menuAction:(id)sender {
	[self executeMenuAction:(FFMenuAction)[sender tag]];
}

- (BOOL)validateMenuItem:(id<NSMenuItem>)mitem {
	switch ([mitem tag]) {
		// Only if images 
		case MENU_FILE_CLEAR_IMGLIST :
		case MENU_FILE_REMOVE :
		case MENU_VIEW_FULLSCREEN :
		case MENU_VIEW_OSD :
		case MENU_VIEW_MAG_LENSE :
			return ([m_imgList numberOfImages] > 0);
		
		// Not in fullscreen (only condition)
		case MENU_APP_CHECK_UPDATES :
		case MENU_APP_PREFERENCES :
		case MENU_FILE_OPEN	:
		case MENU_FILE_SAVE_IMGLIST :
		case MENU_FILE_REVEAL_IN_FINDER :
		case MENU_WIN_IMAGE_LIST :
		case MENU_WIN_EXIF :
		case MENU_SCRIPTS_OPEN_FOLDER :
			return ![m_opts fullscreen];
			
		// File
		case MENU_FILE_SHUFFLE_IMGLIST :
			return ([m_imgList numberOfImages] > 1);
		case MENU_FILE_MOVE_TO_TRASH :
			return [m_prefs trashDelete] && ([m_imgList numberOfImages] > 0);
		case MENU_FILE_COPY_SELECTED :
			return [m_imgListCtrl listHasSelectedImages] && ![m_opts fullscreen];
		case MENU_FILE_FINDER_ICON :
			return [m_imgList isWritableContainerOfImageAtIndex:[m_imgListCtrl indexOfSingleSelectedImage]];
			
		// View
		case MENU_VIEW_PREV_IMG :
			return ([m_imgList numberOfImages] > 1) &&
				(([m_prefs listWrap] != PREF_WRAP_NEVER) || ([m_imgList selectedImageIndex] > 0));
		case MENU_VIEW_NEXT_IMG :
			return ([m_imgList numberOfImages] > 1) &&
				(([m_prefs listWrap] != PREF_WRAP_NEVER) || ([m_imgList selectedImageIndex] < [m_imgList numberOfImages] - 1));
		case MENU_VIEW_FIRST_IMG :
			return ([m_imgList selectedImageIndex] > 0);
		case MENU_VIEW_LAST_IMG :
			return ([m_imgList selectedImageIndex] < [m_imgList numberOfImages] - 1);
		case MENU_VIEW_PAGE_UP :
			return [m_actionHandler canPageUp] ||
				([m_imgList numberOfImages] > 1) && (([m_prefs listWrap] != PREF_WRAP_NEVER) || ([m_imgList selectedImageIndex] > 0));
		case MENU_VIEW_PAGE_DOWN :
			return [m_actionHandler canPageDown] ||
				([m_imgList numberOfImages] > 1) && (([m_prefs listWrap] != PREF_WRAP_NEVER) || ([m_imgList selectedImageIndex] < [m_imgList numberOfImages] - 1));
		case MENU_VIEW_FOLDER_DRAWER :
			return [m_imgListCtrl imageListWindowIsVisible];
		case MENU_VIEW_PREV_FOLDER :
			return [m_imgListCtrl aPreviousDirExists];
		case MENU_VIEW_PREV_NEXT :
			return [m_imgListCtrl aNextDirExists];
		case MENU_VIEW_ACTUAL_SIZE :
			return (([m_opts scaling] != OPT_NO_SCALING) || ([m_opts zoomMultiplier] != 1.0f));
		case MENU_VIEW_ZOOM_OUT :
			return (([m_opts scaling] != OPT_NO_SCALING) || ([m_opts zoomMultiplier] - [m_prefs zoomFactor] > 0.0f));
		case MENU_VIEW_START_SLIDESHOW :
			return ![m_imgList slideshowIsRunning] && ([m_imgList selectedImageIndex]+1 < [m_imgList numberOfImages]);
		case MENU_VIEW_ENDLESS_SLIDESHOW :
			return ![m_imgList slideshowIsRunning] && ([m_imgList numberOfImages] > 1);
		case MENU_VIEW_STOP_SLIDESHOW :
			return [m_imgList slideshowIsRunning];
			
		// Options
		case MENU_OPT_NO_BLOW_UP :
			return ([m_opts scaling] == OPT_FIT_TO_WINDOW);

		// Navigation (maybe change to real methods later)
		case MENU_NAVI_PAN_LEFT :
		case MENU_NAVI_PAN_LEFT_ACCEL :
			return [m_actionHandler canMoveLeft];
		case MENU_NAVI_PAN_RIGHT :
		case MENU_NAVI_PAN_RIGHT_ACCEL :
			return [m_actionHandler canMoveRight];
		case MENU_NAVI_PAN_UP :
		case MENU_NAVI_PAN_UP_ACCEL	:
			return [m_actionHandler canPageUp];
		case MENU_NAVI_PAN_DOWN	:
		case MENU_NAVI_PAN_DOWN_ACCEL :
			return [m_actionHandler canPageDown];
			
		// Window
		case MENU_WIN_CLOSE_WINDOW :
		case MENU_WIN_ZOOM_WINDOW :
		case MENU_WIN_MINIMIZE_WINDOW :
			return (([NSApp keyWindow] != nil) && ![m_opts fullscreen]);
		case MENU_WIN_IMAGE	:
			return (([m_imgList numberOfImages] > 0) && ![m_opts fullscreen]);
		case MENU_WIN_HIDE_TOOLBAR :
			return ([NSApp keyWindow] != nil) && ([[NSApp keyWindow] toolbar] != nil);
		case MENU_WIN_CUSTOMIZE_TOOLBAR	: {
				NSWindow* win = [NSApp keyWindow];
				if (win == nil)
					return FALSE;
				NSToolbar* tb = [win toolbar];
				return (tb != nil) && [tb allowsUserCustomization];
			}
	}
	
	return TRUE;
}

#pragma mark -
#pragma mark Input events - update menu items

// Only keys allowed, no numeric flag
static inline BOOL isInvalidInputEventForMenu(FFInputEvent* ev) {
	return (([ev type] != EVENT_KEY) ||	
			([ev modifier] & NSNumericPadKeyMask));
}

- (id)_menuItemForEvent:(FFInputEvent*)ev ofMenu:(NSMenu*)menu {
	int			tag = [ev action];
	NSMenuItem* sub;
	
	// Script
	if ([ev isAssignedToScript]) {
		return [[[menu itemWithTag:MENU_SCRIPTS] submenu] itemWithTitle:[ev script]];
		
	// Normal menu action
	} else {
		sub = (NSMenuItem*)[menu itemWithTag:(int)(tag / 100)];
		if ((sub != nil) && [sub hasSubmenu])
			return [[sub submenu] itemWithTag:tag];
	}
	return nil;
}

- (BOOL)_addInputEvent:(FFInputEvent*)ev toMenu:(NSMenu*)menu {
	id	item;
	
	if (isInvalidInputEventForMenu(ev))
		return FALSE;

	// Update the menuitem
	item = [self _menuItemForEvent:ev ofMenu:menu];
	if (item != nil) {
		[item setKeyEquivalent:[NSString stringWithFormat:@"%C", [ev key]]];
		[item setKeyEquivalentModifierMask:[ev modifier]];
	}
	
	return TRUE;
}

- (FFInputEvent*)_firstValidInputEventWithAction:(FFMenuAction)action 
								   dstEnumerator:(NSEnumerator**)eventEn {
	FFInputEvent*	ev;

	*eventEn = [m_prefs inputEventEnumerator];
	while (ev = [*eventEn nextObject]) {
		if (([ev action] == action) && !isInvalidInputEventForMenu(ev))
			return ev;
	}
	return nil;
}

- (void)addInputEvent:(FFInputEvent*)newEv toMenu:(NSMenu*)menu {
	NSEnumerator*	en;
	FFInputEvent*	ev;
	
	// Nothing assigned yet
	if ([self _firstValidInputEventWithAction:[newEv action] dstEnumerator:&en] == nil)
		(void)[self _addInputEvent:newEv toMenu:menu];

	// Maybe the new one is before (= not after) the found event
	else {
		while (ev = [en nextObject]) {
			if (ev == newEv)
				break;
		}
		
		// Its before it
		if (ev == nil)
			(void)[self _addInputEvent:newEv toMenu:menu];
	}
}

- (void)removeInputEvent:(FFInputEvent*)oldEv fromMenu:(NSMenu*)menu {
	NSEnumerator*	en;
	FFInputEvent*	ev;
	
	// Old event wasn't added
	if (isInvalidInputEventForMenu(oldEv))
		return;

	// Search the event that is currently "assigned" to the menu item
	ev = [self _firstValidInputEventWithAction:[oldEv action] dstEnumerator:&en];
	
	// The old event is the assigned one
	if (ev == oldEv) {
		// Search the next matching one
		while (ev = [en nextObject]) {
			if (([ev action] == [oldEv action]) &&
				[self _addInputEvent:ev toMenu:menu])
				break;
		}

		// Remove equiv if there wasn't a matching one
		if (ev == nil) {
			NSMenuItem* item = [self _menuItemForEvent:oldEv ofMenu:menu];
			if (item != nil)
				[item setKeyEquivalent:@""];
		}
	}
}

- (void)replaceInputEvent:(FFInputEvent*)oldEv inMenu:(NSMenu*)menu
					 with:(FFInputEvent*)newEv {

	// New event is invalid - just try to remove the old one
	if (isInvalidInputEventForMenu(newEv))
		[self removeInputEvent:oldEv fromMenu:menu];

	// Old event is the assigned one or nothing assigned yet
	else {
		FFInputEvent*	ev;
		NSEnumerator*	en;

		ev = [self _firstValidInputEventWithAction:[oldEv action] dstEnumerator:&en];
		if ((ev == nil) || (ev == oldEv))
			(void)[self _addInputEvent:newEv toMenu:menu];
	}
}

#pragma mark -
#pragma mark NSApplication delegate methods (except speech)

/*
 *  NSApplication delegate method - drop on the dock icon (only 10.3+) & saving prefs
 */
- (void)application:(NSApplication*)theApplication openFiles:(NSArray*)filenames {
	if ([m_prefs clearListBeforeDrop])
		[m_imgList clear];

	// Add all files of the directory of the single passed file (ACDSee like)
	if ([m_prefs addAllOnDrop] && ([filenames count] == 1) && 
		[FFImageList isImageExtension:[[filenames objectAtIndex:0] pathExtension]]) {	
			NSString*		path	= [[filenames objectAtIndex:0] stringByDeletingLastPathComponent];
			NSEnumerator*	en		= [[[NSFileManager defaultManager] directoryContentsAtPath:path] objectEnumerator];
			NSMutableArray*	ifiles	= [NSMutableArray arrayWithCapacity:1];
			NSString*		fn;

			while (fn = [en nextObject]) 
				if ([FFImageList isImageExtension:[fn pathExtension]])
					[ifiles addObject:[path stringByAppendingPathComponent:fn]];
					[m_imgList addFiles:ifiles tillLevel:0  restoreSelectionIfPossible:FALSE];
	}
	else
		[m_imgList addFiles:filenames tillLevel:[m_prefs maxDirectorySearchDepth] restoreSelectionIfPossible:TRUE];
	
	// Switch to fullscreen - if the image window is hidden & wanted
	if ([m_prefs autoFullscreen])
		[m_actionHandler autoFullScreenIfPossible];
}

- (void)applicationWillTerminate:(NSNotification*)notification {
	[m_actionHandler cleanUp];
	[m_prefs save];
	if ([m_prefs saveOpts])
		[m_opts save];
	if ([m_prefs clearRecentOnQuit])
		[m_recentList clearAndSendNotification:FALSE];
	[m_recentList save];
	[m_imgList cleanUp];
	[m_dockIcon setIconWithoutBadge];
	[NSApp cleanUp];
}

- (void)applicationDidResignActive:(NSNotification *)aNotification {
	if ([m_opts fullscreen])
		[self executeMenuAction:MENU_VIEW_FULLSCREEN]; // leave fullscreen
}

#pragma mark -
#pragma mark Speech recognition

- (void)_updateSpeechCommands {
	if ((m_speechRecog != nil) && (m_speechCommands != nil))
		[m_speechRecog setCommands:[m_speechCommands allKeys]];
}

- (void)_enableOrDisableSpeech:(BOOL)enable {
	if (enable) {
		if (m_speechRecog == nil) {
			m_speechRecog = [[NSSpeechRecognizer alloc] init];
			[m_speechRecog setDelegate:self];
			[self _updateSpeechCommands];
		}
		[m_speechRecog startListening];
	} else { 
		if (m_speechRecog != nil)
			[m_speechRecog stopListening];
	}
}

- (void)addSpeechEvent:(FFInputEvent*)ev {
	[m_speechCommands setObject:ev forKey:[ev command]];
	[self _updateSpeechCommands];
}

- (void)removeSpeechEvent:(FFInputEvent*)ev {
	[m_speechCommands removeObjectForKey:[ev command]];
	[self _updateSpeechCommands];
}

// NSApplication delegate method
- (void)speechRecognizer:(NSSpeechRecognizer*)sender didRecognizeCommand:(id)command {
	[self executeEvent:(FFInputEvent*)[m_speechCommands objectForKey:command]];
}

#pragma mark -
#pragma mark Notifications from FFOptions

#define SET_ALL_ITEMS_TO_OFFSTATE(ITEMS, NUM) \
	int i; \
	for (i = 0; i < (NUM); i++) \
		[(ITEMS)[i] setState:NSOffState];

- (void)updateRotationMenuItems:(NSNotification*)notification {
	SET_ALL_ITEMS_TO_OFFSTATE(m_rotationItems, OPT_NUM_ROTATION);
	[m_rotationItems[[m_opts rotation]] setState:NSOnState];
}

- (void)updateScalingMenuItems:(NSNotification*)notification {
	SET_ALL_ITEMS_TO_OFFSTATE(m_scalingItems, OPT_NUM_SCALING);
	[m_scalingItems[[m_opts scaling]] setState:NSOnState];
}

- (void)updatePageModeMenuItems:(NSNotification*)notification {
	SET_ALL_ITEMS_TO_OFFSTATE(m_pageModeItems, OPT_NUM_PAGE_MODE);
	[m_pageModeItems[[m_opts pageMode]] setState:NSOnState];
}

- (void)updateNoBlowUpMenuItem:(NSNotification*)notification {
	[m_menuNoBlowUp setState:[m_opts noBlowUp]];
}

- (void)updateAntialiasingMenuItem:(NSNotification*)notification {
	[m_menuAntialiasing setState:[m_opts antialiasing]];
}

- (void)updateSpeechOnMenuItem:(NSNotification*)notification{
	BOOL on = [m_opts speechOn];
	[m_menuSpeechOn setState:on];
	[self _enableOrDisableSpeech:on];
}

- (void)updateZoomLevelMenuItem:(NSNotification*)notification {
	[m_menuZoomLevel setTitle:[NSString stringWithFormat:@"[Zoom %.0f%%]", [m_opts zoomMultiplier]*100]];
}

#pragma mark -
#pragma mark Recent menu handling

- (void)updateRecentMenu:(NSNotification*)_unused {
	id<NSMenuItem>	mitem;
	NSEnumerator*	ren;
	NSString*		name;
	unsigned		nidx = 0;

	// Remove all entries
	for (;;) {
		mitem = [m_recentSubMenu itemAtIndex:0];
		if ([mitem isSeparatorItem])
			break;
		[m_recentSubMenu removeItemAtIndex:0];
	}

	// Add the recent names
	ren = [m_recentList itemEnumerator];
	while (name = [ren nextObject]) {
		NSString* title = [name lastPathComponent];
		if ([FFSmartFolder isSmartFolderExtension:[title pathExtension]])
			title = [[title stringByDeletingPathExtension] stringByAppendingString:@" [SmartFolder]"];
				
		[m_recentSubMenu insertItemWithTitle:title
									  action:@selector(openRecent:)
							   keyEquivalent:@""
									 atIndex:nidx++];
	}
}

- (void)openRecent:(id)menuItem {
	unsigned	idx		= [m_recentSubMenu indexOfItem:menuItem];
	NSString*	name	= [m_recentList itemAtIndex:idx];
	BOOL		isDir;

	if ([[NSFileManager defaultManager] fileExistsAtPath:name isDirectory:&isDir]) {
		// Imagelist
		if ([[name pathExtension] caseInsensitiveCompare:[FFImageList fileExtension]] == NSOrderedSame) {
			[m_imgList replaceWithImageList:name restoreSelection:TRUE];

		// Something that can appear in the folder drawer..
		} else {
			FFDirNode* dir;
			if (isDir)
				dir = [FFDirNode directoryWithPath:name];
			else if ([FFSmartFolder isSmartFolderExtension:[name pathExtension]])
				dir = [FFDirNode smartFolderWithPath:name];
			else
				dir = [FFDirNode archiveWithPath:name];
		
			[m_imgList replace:dir onlyImages:FALSE];
		}
		
	} else {
		NSRunAlertPanel(FFTR(@"Recent"), 
						FFTRC(@"The file/directory '%@' doesn't exist anymore.", @"1=Filename/path"),
						FFTR(@"OK"), nil, nil,
						name);

		[m_recentList removeAtIndex:idx];
		[m_recentSubMenu removeItemAtIndex:idx];
	}
}

#pragma mark -
#pragma mark Mainmenu delegate - Archive encoding

static NSComparisonResult _encItemCompare(id left, id right, void* context) {
	return [[left title] compare:[right title]];
}

- (void)changeEncoding:(id)sender {
	NSStringEncoding enc = [sender tag];
	[[m_archivEncodingMenu itemWithTag:[m_opts archiveEncoding]] setState:NSOffState];
	[m_opts setArchiveEncoding:enc];
	[[m_archivEncodingMenu itemWithTag:enc] setState:NSOnState];
}

- (void)menuNeedsUpdate:(NSMenu*)menu {
	if (menu != m_archivEncodingMenu)
		return;
	
	if ([m_archivEncodingMenu numberOfItems] == 0) {
		// 1. Create items for all encodings
		const NSStringEncoding*	encIDs		= [NSString availableStringEncodings];
		NSMutableArray*			commonItems	= [[NSMutableArray alloc] initWithCapacity:17];
		NSMutableArray*			propItems	= [[NSMutableArray alloc] init];
		NSMenuItem*				encItem;
		NSEnumerator*			it;		
		
		for (; *encIDs != NULL; encIDs++) {
			encItem =  [[NSMenuItem alloc] initWithTitle:[NSString localizedNameOfStringEncoding:*encIDs]
												  action:@selector(changeEncoding:) keyEquivalent:@""];
			[encItem setTarget:self];
			[encItem setTag:*encIDs];
			
			if (*encIDs < NSProprietaryStringEncoding)
				[commonItems addObject:encItem];
			else
				[propItems addObject:encItem];	
		}
		
		// 2. Sort
		[commonItems sortUsingFunction:_encItemCompare context:NULL];
		[propItems sortUsingFunction:_encItemCompare context:NULL];
		
		// 3. Add to menu
		for (it = [commonItems objectEnumerator]; encItem = [it nextObject]; )
			[m_archivEncodingMenu addItem:[encItem autorelease]];
		
		[m_archivEncodingMenu addItem:[NSMenuItem separatorItem]];
		
		for (it = [propItems objectEnumerator]; encItem = [it nextObject]; )
			[m_archivEncodingMenu addItem:[encItem autorelease]];		
		
		// 4. Cleanup
		[commonItems release];
		[propItems release];
		
		// 5. Select user encoding
		[[m_archivEncodingMenu itemWithTag:[m_opts archiveEncoding]] setState:NSOnState];
	}
}

@end
