RCS_ID("$Id: FFImageListController.m 596 2006-09-30 19:12:25Z ravemax $")

#import "FFImageListController.h"
#import "FFImageList.h"
#import "FFImageList_Thumbs.h"
#import "FFDropWindow.h"
#import "FFDirectory.h"
#import "FFImageAndTextCell.h"
#import "FFPreferences.h"
#import "FFApplication.h"
#import "FFSegmentedControl.h"
#import "FFToolbarUtil.h"
#import "FFThumbView.h"
#import "FFToolTip.h"
#import "NSMenu_Additions.h"

// Notifications (sends)
NSString*	ImageListWinActivatedNotification	= @"image_list_activated";

// Notifications (receive)
NSString*	ShowOrHideImageListWinNotification	= @"show_or_hide_imglist";
NSString*	StartProgressNotification			= @"start_progress_imglist";
NSString*	StopProgressNotification			= @"stop_progress_imglist";

// Constants for the table
static NSString* EyeImage				= @"eye";
static NSString* SelectedColumnIdent	= @"selected_col";
static NSString* NameColumnIdent		= @"name_col";

// Constants for the directory outlineview
static NSString* DirColumnIdent			= @"dir_col";

// The toolbar identifiers
static NSString* AddIdent				= @"tb_add";
static NSString* RemoveIdent			= @"tb_remove";
static NSString* ClearIdent				= @"tb_clear";
static NSString* DisplayModeIdent		= @"tb_display_mode";
static NSString* FoldersIdent			= @"tb_folders";
//static NSString* FilterIdent			= @"tb_filter";

// The thumbview identifiers
static NSString* ListTabIdent			= @"list_tab";
static NSString* ThumbTabIdent			= @"thumb_tab";

// The column indexes
#define SELECTED_COL_INDEX  0
#define NAME_COL_INDEX		1

// Display mode constans
#define DISPLAY_MODE_LIST	0 

// Global toolbar delegate variables (required only onces) - initialize takes care of them
static NSArray* AllowedToolbarItemIdents = nil;
static NSArray* DefaultToolbarItemIdents = nil;

// Class
@implementation FFImageListController

#pragma mark -
#pragma mark Initialisation and cleanup

+ (void)initialize {
	if (AllowedToolbarItemIdents == nil) {
		// Toolbar
		AllowedToolbarItemIdents = [[NSArray alloc] initWithObjects:
			AddIdent,
			RemoveIdent,
			ClearIdent,
			DisplayModeIdent,
			NSToolbarFlexibleSpaceItemIdentifier,
			NSToolbarSeparatorItemIdentifier,
			FoldersIdent,
//			FilterIdent,
			nil];
		
		DefaultToolbarItemIdents = AllowedToolbarItemIdents; // may differ if more icons
	}
}

- (void)_setupHeaderIndicatorImages {
	static NSString* HIFileFormat	= @"%d.tiff";
	
	NSString*		hiPath;
	NSFileManager*	fm = [NSFileManager defaultManager];

	hiPath	= [[NSApp supportDirectory] stringByAppendingPathComponent:
				[NSString stringWithFormat:@"HeaderIndicator_%@", FFTR(@"LocalizationName")]];
	
	// Create all images
	if (![fm fileExistsAtPath:hiPath]) {
		[fm createDirectoryAtPath:hiPath attributes:nil];
		
		// Text attributes
		NSDictionary* fontAttrib = [[NSDictionary alloc] initWithObjectsAndKeys:
			[[[m_table tableColumnWithIdentifier:NameColumnIdent] headerCell] font], NSFontAttributeName,
			[NSColor colorWithDeviceWhite:0.0f alpha:0.55], NSForegroundColorAttributeName,
			nil];		
		
		// All strings
		#define	NUM_HI_TEXTS (IMGLIST_NUM_ORDER / 2)
		NSString* texts[NUM_HI_TEXTS];
		texts[IMGLIST_USER_ASC		/ 2]	= FFTR(@"User");
		texts[IMGLIST_NUMERIC_ASC	/ 2]	= FFTR(@"Num.");
		texts[IMGLIST_ALPHA_ASC		/ 2]	= FFTR(@"Alpha.");
		texts[IMGLIST_PATH_ASC		/ 2]	= FFTR(@"Path");
		texts[IMGLIST_DATE_ASC		/ 2]	= FFTR(@"Date");
		texts[IMGLIST_FILESIZE_ASC	/ 2]	= FFTR(@"Size");
		
		// Determine the max. width
		int		i, j;
		NSSize	textSize, imgSize = NSMakeSize(0.0f, 0.0f);
		
		for (i = 0; i < NUM_HI_TEXTS; i++) {
			textSize		= [texts[i] sizeWithAttributes:fontAttrib];
			imgSize.width	= MAX(imgSize.width, textSize.width);
			imgSize.height	= MAX(imgSize.height, textSize.height);
		}
		
		// Load the arrows + calc. all necessary parameters
		NSImage*	directionImg[2];
		NSSize		directionSize;
		NSPoint		directionPos, textPos;
		
		directionImg[0]	= [NSImage imageNamed:@"NSAscendingSortIndicator"];
		directionImg[1]	= [NSImage imageNamed:@"NSDescendingSortIndicator"];
		directionSize	= [directionImg[0] size];
		imgSize.height	= MAX(directionSize.height, imgSize.height);
		directionPos	= NSMakePoint(0.0f, ceilf((imgSize.height - directionSize.height) / 2));
		textPos			= NSMakePoint(directionSize.width + 1.0f, 0.0f);
		imgSize.width	= ceilf(textPos.x + imgSize.width);
		
		// Create all images
		for (i = 0; i < NUM_HI_TEXTS; i++) {
			for (j = 0; j < 2; j++) {
				int	idx = i * 2 + j;
				
				// Draw the image
				m_headerIndicatorImgs[idx] = [[NSImage alloc] initWithSize:imgSize];
				[m_headerIndicatorImgs[idx] lockFocus];
			
				[[NSColor clearColor] set];
				NSRectFill(NSMakeRect(0.0f, 0.0f, imgSize.width, imgSize.height));

				[directionImg[j] compositeToPoint:directionPos operation:NSCompositeSourceOver];
				[texts[i] drawAtPoint:textPos withAttributes:fontAttrib];
			
				[m_headerIndicatorImgs[idx] unlockFocus];
				
				// Save it
				[[m_headerIndicatorImgs[idx] TIFFRepresentation] writeToFile:
					[hiPath stringByAppendingPathComponent:[NSString stringWithFormat:HIFileFormat, idx]]
																  atomically:FALSE];
			}
		}
		
		[fontAttrib release];
	} else {
		int	i;
		for (i = 0; i < IMGLIST_NUM_ORDER; i++)
			m_headerIndicatorImgs[i] = [[NSImage alloc] initWithContentsOfFile:
				[hiPath stringByAppendingPathComponent:[NSString stringWithFormat:HIFileFormat, i]]];
	}
}

- (void)_setupSort {
	FFImageListSortOrder	order = [[FFPreferences instance] imageListSortOrder];
	NSTableColumn*			col	  = [m_table tableColumnWithIdentifier:NameColumnIdent];
	
	[m_table setIndicatorImage:m_headerIndicatorImgs[order] inTableColumn:col];
	[m_table setHighlightedTableColumn:col];
	
	m_sortMethodMenu = [[NSMenu alloc] init];
	[m_sortMethodMenu addItemWithTitle:FFTR(@"Sort by:") action:NULL keyEquivalent:@""];
	
	[m_sortMethodMenu addItemWithTitle:FFTR(@"User") target:self 
								action:@selector(changeSortMethod:) tag:IMGLIST_USER_ASC];
	[m_sortMethodMenu addItemWithTitle:FFTR(@"Filename (Numerical)") target:self 
								action:@selector(changeSortMethod:) tag:IMGLIST_NUMERIC_ASC];
	[m_sortMethodMenu addItemWithTitle:FFTR(@"Filename (Alphabetical)") target:self 
								action:@selector(changeSortMethod:) tag:IMGLIST_ALPHA_ASC];
	[m_sortMethodMenu addItemWithTitle:FFTR(@"Path") target:self 
								action:@selector(changeSortMethod:) tag:IMGLIST_PATH_ASC];
	[m_sortMethodMenu addItemWithTitle:FFTR(@"Date") target:self 
								action:@selector(changeSortMethod:) tag:IMGLIST_DATE_ASC];
	[m_sortMethodMenu addItemWithTitle:FFTR(@"Filesize") target:self 
								action:@selector(changeSortMethod:) tag:IMGLIST_FILESIZE_ASC];

	[m_table setMenu:m_sortMethodMenu];
	[m_thumbView setMenu:m_sortMethodMenu];
}

- (void)_setupToolbarAndDisplayMode {
	#define DISPLAY_MODE_IMG_NUM 2
	static const NSString* DisplayModeImages[DISPLAY_MODE_IMG_NUM] = {
		@"tbl_list", @"tbl_thumb" };
	
	// Toolbar w/o labels
	NSToolbar*	tb = [[NSToolbar alloc] initWithIdentifier:@"list_win_toolbar"];
	[tb setAllowsUserCustomization:TRUE];
	
	[tb setDelegate:self];
	[tb setDisplayMode:NSToolbarDisplayModeIconOnly];
	[tb setAutosavesConfiguration:TRUE]; // Save if the toolbar is visible or not
	
	m_toolbarItems = [[NSMutableDictionary alloc] init];

	// The display mode segment control
	m_displayModeControl = createSegmentedControl(DISPLAY_MODE_IMG_NUM, DisplayModeImages,
												  NSSegmentSwitchTrackingSelectOne,
												  self, @selector(displayModeChanged));

	addViewToToolbar(DisplayModeIdent, m_toolbarItems, m_displayModeControl,
					 FFTR(@"Mode"), FFTR(@"Sets the display mode"));
	
	m_displayMode = 0;
	[m_displayModeControl setSelectedSegment:m_displayMode]; // list mode
	[m_tabView selectTabViewItemWithIdentifier:ListTabIdent];
	
	// All other toolbar buttons
	createToolbarItem(AddIdent, m_toolbarItems,
					  @"tbl_add",
					  self, @selector(addClicked),
					  FFTR(@"Add"), FFTR(@"Adds file(s) to the list"));
	createToolbarItem(RemoveIdent, m_toolbarItems,
					  @"tbl_remove",
					  self, @selector(removeClicked),
					  FFTR(@"Remove"), FFTR(@"Removes file(s) from the list"));
	createToolbarItem(ClearIdent, m_toolbarItems,
					  @"tbl_clear",
					  self, @selector(clearClicked),
					  FFTR(@"Clear"), FFTR(@"Clears the list"));
	createToolbarItem(FoldersIdent, m_toolbarItems,
					  @"tbl_folders",
					  self, @selector(foldersClicked),
					  FFTR(@"Folders"), FFTR(@"Shows or hides the folders drawer"));
	
#if 0 // FILTER	
	// The filter
	tbi = [[NSToolbarItem alloc] initWithItemIdentifier:FilterIdent];
	[m_toolbarItems setObject:tbi forKey:FilterIdent];
	[tbi setToolTip:FFTR(@"Filter the current list")];
	[tbi setView:m_filterView];	
	[tbi setMinSize:NSMakeSize(NSWidth([m_filterView frame]), NSHeight([m_filterView frame]))];
	[tbi setMaxSize:NSMakeSize(250, NSHeight([m_filterView frame]))]; // Hardcoded!!
#endif
	[m_window setToolbar:tb];
}

- (id)init {
	if ([super init]) {
		[NSBundle loadNibNamed:@"ImageList" owner:self];
		
		// i18n
		[m_window setTitle:FFTR(@"Image list")];
		NSTableColumn* nameCol = [m_table tableColumnWithIdentifier:NameColumnIdent];
		[[nameCol headerCell] setStringValue:FFTR(@"Filename")];
		[m_appendFilenames setTitle:FFTR(@"Append to image list instead of replacing")];
		[m_restorePosition setTitle:FFTR(@"Restore image list position")];
		
		// awakeFrom : Set the info label to an empty string - no flicker
		[m_infoLabel setStringValue:@""];
		
		// awakeFromNib : load the selected image indicator
		m_eyeImage  = [[NSImage alloc] initWithContentsOfFile:
			[[NSBundle mainBundle] pathForImageResource:EyeImage]];

		// awakeFromNib : sort stuff & table
		[self _setupHeaderIndicatorImages];
		[self _setupSort];
		[[nameCol dataCell] setWraps:TRUE];
		[m_table sizeToFit];

		// awakeFromNib : toolbar
		[self _setupToolbarAndDisplayMode];
		
		// awakeFromNib : Set the image & text data cell
		NSTableColumn*		col = [m_dirView tableColumnWithIdentifier:DirColumnIdent];
		FFImageAndTextCell*	cell = [[[FFImageAndTextCell alloc] init] autorelease];
		[cell setFont:[[col dataCell] font]];
		[col setDataCell:cell];

		// awakeFromNib : single clicking now only expands/collapses the directory
		[m_dirView setDoubleAction:@selector(directoryDoubleClicked:)];
		[m_dirView setTarget:self];
		if ([m_directory containsRemovableMedia])
			[m_dirView expandItem:[m_directory outlineView:m_dirView child:0 ofItem:nil]];
		
		// awakeFromNib : Thumbview & tooltip
		[m_scrollView setBackgroundColor:[m_thumbView backgroundColor]];
		[m_thumbView setSelectionImage:m_eyeImage];
		m_toolTip	= [[FFToolTip alloc] initWithDelegate:self];
		
		// Simple member vars
		m_progressCounter	= 0;
		m_filterIsActive	= FALSE;
		m_nameColor[0]		= [NSColor controlTextColor];
		m_nameColor[1]		= [NSColor disabledControlTextColor];
		m_clickedNodeParent	= nil;
		
		// Delegate
		[m_window setDelegate:self]; 
		
		// Notifications
		m_nc = [NSNotificationCenter defaultCenter];
		[m_nc addObserver:self selector:@selector(showOrHideImageList:)
					 name:ShowOrHideImageListWinNotification object:nil];
		[m_nc addObserver:self selector:@selector(dirContentChanged:) 
					 name:DirectoryContentChangedNotification object:nil];
		[m_nc addObserver:self selector:@selector(startProgress:)
					 name:StartProgressNotification object:nil];
		[m_nc addObserver:self selector:@selector(stopProgress:)
					 name:StopProgressNotification object:nil];
		
		// Ok show if - unless pref option set
		FFPreferences* prefs = [FFPreferences instance];
		if ([prefs dirDrawerVisibleOnStartup]) {
//			[m_foldersBtn setState:NSOnState];
			[m_dirDrawer open];
		}
	
		if (![prefs listWinHiddenOnStartup]) 
			[m_window makeKeyAndOrderFront:self];		
	}
	return self;
}

- (void)dealloc {
//	int i;
//	for (i = 0; i < IMGLIST_NUM_ORDER; i++)
//		[m_headerIndicatorImgs[i] release];
	
	[m_eyeImage release];
	[[m_window toolbar] release];
	[m_toolbarItems release];
	[m_toolTip release];
	[m_sortMethodMenu release];
	
	[[NSNotificationCenter defaultCenter] removeObserver:self];
	
	[super dealloc];
}

- (void)setImageList:(FFImageList*)imgList {
	// Table data
	m_imgList = imgList;
	[m_table setDataSource:self];
	[m_table setDelegate:self];
	[m_table registerForDraggedTypes:[NSArray arrayWithObject:ImageListDragType]];
	
	[m_nc addObserver:self selector:@selector(reloadTheImageTableData:)
				 name:ImageListChangedNotification object:nil];
	[m_nc addObserver:self selector:@selector(reloadTheImageTableData:)
				 name:ImageListRearrangedNotification object:nil];
	[m_nc addObserver:self selector:@selector(reloadTheImageTableData:)
				 name:NewImageSelectedNotification object:nil];
	[m_nc addObserver:self selector:@selector(singleThumbLoaded:)
				 name:SingleThumbLoadedNotification object:nil];
	
	// Doubleclick on a image
	[m_table setTarget:self];
	[m_table setDoubleAction:@selector(selectDoubleClickedImage)];

	// Set the title & info label
	[self _updateInfoLabel];
	
	// Thumbview
	[m_thumbView setDataSource:m_imgList];
	
	// Now its safe to enable drop operations
	[m_window setDropReceiver:self andSelector:@selector(addDroppedFiles:)];
	
	// File copy 
	[m_dirView shouldCollapseAutoExpandedItemsForDeposited:TRUE];
	[m_directory setImageList:imgList];
	[m_dirView registerForDraggedTypes:[NSArray arrayWithObject:ImageListDragType]];
//	[m_dirView registerForDraggedTypes:[NSArray arrayWithObject:@"jpg"]];
//	[m_dirView registerForDraggedTypes:[NSArray arrayWithObject:NSFilesPromisePboardType]];
}

#pragma mark -
#pragma mark IB actions

- (IBAction)directorySelectedInDrawer:(id)sender {
	id item = [sender itemAtRow:[sender selectedRow]];
	
	if (item != nil) { // Click on an emtpy cell
		if ([sender isItemExpanded:item])
			[sender collapseItem:item];
		else
			[sender expandItem:item];
	}
}

- (IBAction)filter:(id)sender {
	m_filterIsActive = [m_imgList filter:[(NSSearchField*)sender stringValue]];
	[m_table reloadData];
}

#pragma mark -
#pragma mark Exported methods

- (void)show {
	[m_window makeKeyAndOrderFront:self];
}

- (void)openAddSheet:(BOOL)append {
	NSOpenPanel*	op;
	int				ret;
	
	// Various panel options
	op = [NSOpenPanel openPanel];
	[op setCanChooseDirectories:TRUE];
	[op setResolvesAliases:TRUE];
	[op setAllowsMultipleSelection:TRUE];
	[op setDelegate:self];
	
	// Add view
	[m_appendFilenames setState:(int)append];
	[m_restorePosition setState:NSOnState];
	[m_restorePosition setEnabled:FALSE];
	[op setAccessoryView:m_openPanelView];
	
	// Here we go
	[NSApp setForwardAll:TRUE];
	ret = [op runModalForTypes:KnownFileExtensions];
	[NSApp setForwardAll:FALSE];
	if (ret == NSCancelButton)
		return;
	
	if ([m_appendFilenames state] == NSOffState)
		[m_imgList clear];

	BOOL tryToRestore = [m_restorePosition isEnabled] && ([m_restorePosition state] == NSOnState);
	[m_imgList addFiles:[op filenames] tillLevel:1 restoreSelectionIfPossible:tryToRestore];
}

- (void)panelSelectionDidChange:(id)sender {
	if (([[sender filenames] count] == 1) &&
		[[[sender filename] pathExtension] isEqualToString:ImageListFileExtension])
		[m_restorePosition setEnabled:TRUE];
	else
		[m_restorePosition setEnabled:FALSE];
}

- (void)openSaveSheet {
	NSSavePanel*	sp;
	int				ret;
	
	sp = [NSSavePanel savePanel];
	
	[sp setRequiredFileType:ImageListFileExtension];
	[sp setCanSelectHiddenExtension:TRUE];

	NSString* fn = [[NSDate date] descriptionWithCalendarFormat:@"%Y-%m-%d_%H-%M" timeZone:nil locale:nil];
	
	[NSApp setForwardAll:TRUE];
	ret = [sp runModalForDirectory:nil file:fn];
	[NSApp setForwardAll:FALSE];
	
	if (ret == NSFileHandlingPanelOKButton)
		[m_imgList save:[sp filename]];
}


- (BOOL)listHasSelectedImages {
	return ([m_table selectedRow] != -1);
}

- (int)indexOfSingleSelectedImage {
	return ([m_table numberOfSelectedRows] != 1) ? -1 : [m_table selectedRow];
}

- (BOOL)copySelected {
	NSOpenPanel*	openPanel;
	NSIndexSet*		idxSel;
	unsigned		idx;
	
	// No selection
	if (![self listHasSelectedImages])
		return FALSE;

	// Get the dst. directory from the user
	openPanel = [NSOpenPanel openPanel];
	[openPanel setCanChooseDirectories:TRUE];
	[openPanel setCanChooseFiles:FALSE];
	[openPanel setPrompt:FFTR(@"Choose")];
	[openPanel setCanCreateDirectories:TRUE];
	
	if ([openPanel runModalForTypes:nil] == NSCancelButton)
		return FALSE;
	
	// Ok go through all em
	idxSel = [m_table selectedRowIndexes];
	for (idx = [idxSel firstIndex]; idx <= [idxSel lastIndex]; idx++)
		if ([idxSel containsIndex:idx]) {
			if (![m_imgList copyImageAtIndex:idx toDirectory:[openPanel filename]]) // Abort
				return FALSE;
			
		}

	return TRUE;
}	 

- (void)setWindowToSnap:(NSWindow*)win andTolerance:(float)tolerance {
	[m_window setWindowToSnap:win andTolerance:tolerance];
}

- (BOOL)imageListWindowIsVisible {
	return [m_window isVisible];
}

- (void)toggleDirDrawer  {
	[m_dirDrawer toggle:self];
}

- (BOOL)aPreviousDirExists {
	return ((m_clickedNodeParent != nil) && (m_clickedNodeIndex > 0));
}

- (BOOL)aNextDirExists {
	return ((m_clickedNodeParent != nil) && 
			(m_clickedNodeIndex+1 < [m_clickedNodeParent numberOfSubDirectories]));
}

- (void)openPrevDir {
	if ([self aPreviousDirExists]) {
		m_clickedNodeIndex--;
		[m_imgList replace:[m_clickedNodeParent subDirectoryAtIndex:m_clickedNodeIndex]
				onlyImages:TRUE];
	}
}

- (void)openNextDir {
	if ([self aNextDirExists]) {
		m_clickedNodeIndex++;
		[m_imgList replace:[m_clickedNodeParent subDirectoryAtIndex:m_clickedNodeIndex] 
				onlyImages:TRUE];
	}
}

#pragma mark -
#pragma mark [Internal] drop, notification & action methods

- (void)_updateInfoLabel {
	unsigned	numImg	= [m_imgList numberOfImages];
	
	if (numImg == 0)
		[m_infoLabel setStringValue:FFTR(@"Empty")];
	else {
		int	imgNo	= [m_imgList selectedImageIndex]+1,
			selRows	= (m_displayMode == DISPLAY_MODE_LIST) ? 
						[m_table numberOfSelectedRows] : 0;
	
		if (selRows > 0)
			[m_infoLabel setStringValue:[NSString stringWithFormat:
				FFTRC(@"%d of %u selected, displays no. %d", @"%1=number of selected images, 2=number of images, 3=current image"),
				selRows, numImg, imgNo]];
		else if (numImg > 1)
			[m_infoLabel setStringValue:[NSString stringWithFormat:
				FFTRC(@"%u images, displays no. %d", @"1=Number of images, 2=current image"),
				numImg, imgNo]];
		else
			[m_infoLabel setStringValue:FFTR(@"1 image")];
	}
}

- (void)addDroppedFiles:(NSArray*)files {
	if ([[FFPreferences instance] clearListBeforeDrop])
		[m_imgList clear];

	[m_imgList addFiles:files tillLevel:[[FFPreferences instance] maxDirectorySearchDepth]
		restoreSelectionIfPossible:TRUE];
}

- (void)reloadTheImageTableData:(NSNotification*)not {
	[m_table reloadData];
	[self _updateInfoLabel];
	
	if (m_displayMode != DISPLAY_MODE_LIST) {
		if ([[not name] isEqualToString:NewImageSelectedNotification])
			[m_thumbView updateSelection];
		else {
			[m_imgList loadThumbs:FALSE];
			[m_thumbView reload];
		}
	} else
		[m_table scrollRowToVisible:[m_imgList selectedImageIndex]];
}

- (void)selectDoubleClickedImage {
	[m_imgList selectImageAtIndex:(unsigned)[m_table clickedRow]];
}

- (void)directoryDoubleClicked:(id)sender {
	FFDirNode*	dir	= [sender itemAtRow:[sender selectedRow]];

	if (dir != nil) {
		m_clickedNodeParent = [dir parentNode];
		m_clickedNodeIndex	= [m_clickedNodeParent indexOfSubDirectory:dir];
		[m_imgList replace:dir onlyImages:TRUE];
	}
}

- (void)changeSortMethod:(id)sender {
	FFImageListSortOrder so = [sender tag];
	[m_table setIndicatorImage:m_headerIndicatorImgs[so] 
				 inTableColumn:[m_table tableColumnWithIdentifier:NameColumnIdent]];
	
	(void)[m_imgList sort:so];
}

#pragma mark -
#pragma mark FFDirectory delegate methods

- (BOOL)directory:(FFDirectory*)dir shouldCollapseNode:(FFDirNode*)node {
	if ((m_clickedNodeParent != nil) && [m_clickedNodeParent isSelfOrParentNode:node])
		m_clickedNodeParent = nil;
			
	return TRUE;
}

#pragma mark -
#pragma mark Window delegate methods

- (void)windowDidBecomeKey:(NSNotification*)notification {
	if ([m_imgList numberOfImages] > 0)
		[m_nc postNotificationName:ImageListWinActivatedNotification
							object:[NSNumber numberWithInt:[m_window windowNumber]]];
}

#pragma mark -
#pragma mark Filename table - datasource methods

- (int)numberOfRowsInTableView:(NSTableView*)tv {
	return [m_imgList numberOfImages];
}

- (id)tableView:(NSTableView*)tv objectValueForTableColumn:(NSTableColumn*)col row:(int)rowIndex {
	if ([[col identifier] isEqualTo:SelectedColumnIdent]) {
		if ([m_imgList selectedImageIndex] != rowIndex)
			return nil;
		return m_eyeImage;
	}
	return [m_imgList displayNameAtIndex:(unsigned)rowIndex];
}

- (BOOL)tableView:(NSTableView*)tv writeRows:(NSArray*)rows toPasteboard:(NSPasteboard*)pboard {
/*	// Only allowed in user mode
	if (![m_imgList isUserSortOrder])
		return FALSE;

    [pboard declareTypes:[NSArray arrayWithObject:ImageListDragType] owner: self];
	[pboard setPropertyList:rows forType:ImageListDragType];
	
	return TRUE;*/
	
	[pboard declareTypes:[NSArray arrayWithObjects:ImageListDragType, NSFilesPromisePboardType, nil] owner: self];
	
	[pboard setPropertyList:rows
					forType:ImageListDragType];

	NSMutableArray*	extArray = [NSMutableArray arrayWithCapacity:1];
	NSEnumerator*	en;
	NSNumber*		rowIndex;
	NSString*		ext;
	
	en = [rows objectEnumerator];
	while (rowIndex = [en nextObject]) {
		ext	= [m_imgList extensionAtIndex:[rowIndex unsignedIntValue]];
		if (![extArray containsObject:ext])
			[extArray addObject:ext];
	}
		
	[pboard setPropertyList:extArray forType:NSFilesPromisePboardType];

	return TRUE;
}


#if 0
- (NSArray*)namesOfPromisedFilesDroppedAtDestination:(NSURL*)dropDestination {
	NSLog(@"1-- %@", dropDestination);
	return [NSArray arrayWithObject:@"test.jpg"];
}

- (NSArray *)tableView:(NSTableView *)tv namesOfPromisedFilesDroppedAtDestination:(NSURL *)dropDestination forDraggedRowsWithIndexes:(NSIndexSet *)indexSet {
	NSLog(@"2-- %@", dropDestination);
	return [NSArray arrayWithObject:@"test.jpg"];

}
#endif

- (NSDragOperation)tableView:(NSTableView*)tv validateDrop:(id <NSDraggingInfo>)info proposedRow:(int)row 
	   proposedDropOperation:(NSTableViewDropOperation)operation {

	return [m_imgList isUserSortOrder] ? NSDragOperationGeneric : NSDragOperationNone;
//	return NSDragOperationGeneric;
}

- (BOOL)tableView:(NSTableView *)tableView acceptDrop:(id <NSDraggingInfo>)info row:(int)row 
	dropOperation:(NSTableViewDropOperation)operation {
	
	NSPasteboard*	pboard = [info draggingPasteboard];
    if ([pboard availableTypeFromArray:[NSArray arrayWithObject:ImageListDragType]]) {
		[m_imgList moveImagesWithIndexes:[pboard propertyListForType:ImageListDragType] toIndex:row];		
		[m_table deselectAll:self];
	}
	return TRUE;
}

#pragma mark -
#pragma mark Handlers for received notifications

- (void)showOrHideImageList:(NSNotification*)notification {
	if ([m_window isVisible])
		[m_window orderOut:self];
	else
		[m_window makeKeyAndOrderFront:self];
}
	
- (void)dirContentChanged:(NSNotification*)not {
	[m_dirView reloadData];
	
	// Root changed
	if ([m_directory outlineView:m_dirView child:0 ofItem:nil] == [not object])
		[m_dirView expandItem:[not object]];
}

- (void)singleThumbLoaded:(NSNotification*)not {
	[m_thumbView thumbLoaded:(FFThumb*)[[not object] unsignedIntValue]]; // not a 100% solution
}

- (void)startProgress:(NSNotification*)not {
	m_progressCounter++;
	[m_progress startAnimation:self];
}

- (void)stopProgress:(NSNotification*)not {
	m_progressCounter--;
	if (m_progressCounter == 0)
		[m_progress stopAnimation:self];
}

#pragma mark -
#pragma mark Filename table - delegate methods

- (void)tableView:(NSTableView*)tv didClickTableColumn:(NSTableColumn*)col {
	// Ignore "selected" column
	if ([[col identifier] isEqualToString:SelectedColumnIdent])
		return;
	
	FFImageListSortOrder order = ([[FFPreferences instance] imageListSortOrder] + 1) % IMGLIST_NUM_ORDER;
	[m_table setIndicatorImage:m_headerIndicatorImgs[order] inTableColumn:col];
	
	(void)[m_imgList sort:order];
}

#if 0
- (void)tableView:(NSTableView*)tv willDisplayCell:(id)cell forTableColumn:(NSTableColumn*)col row:(int)rowIndex {
	if ([[col identifier] isEqualToString:NameColumnIdent]) {
		if (m_filterIsActive)
			[cell setTextColor:[m_imgList filteredAtIndex:(unsigned)rowIndex] ? m_nameColor[1] : m_nameColor[0]];
		else
			[cell setTextColor:m_nameColor[0]];
	}
}
#endif

- (void)tableViewSelectionDidChange:(NSNotification*)_unused {
	[self _updateInfoLabel];
}

#pragma mark -
#pragma mark toolbar buttons, display mode & toolbar delegate methods

- (NSToolbarItem*)toolbar:(NSToolbar*)tb itemForItemIdentifier:(NSString*)ident
		willBeInsertedIntoToolbar:(BOOL)flag {
	return [m_toolbarItems objectForKey:ident];
}

- (NSArray*)toolbarAllowedItemIdentifiers:(NSToolbar*)tb { return AllowedToolbarItemIdents; }
- (NSArray*)toolbarDefaultItemIdentifiers:(NSToolbar*)tb { return DefaultToolbarItemIdents; }

- (void)addClicked {
	[self openAddSheet:TRUE];
}

- (void)removeClicked {
	// Something selected ?
	if ([m_table selectedRow] == -1)
		return;
	
	[m_imgList removeInIndexSet:[m_table selectedRowIndexes]];
	[m_table deselectAll:nil];
}

- (void)clearClicked {
	[m_imgList clear];
}

- (void)foldersClicked {
	[[FFPreferences instance] setDirDrawerVisibleOnStartup:([m_dirDrawer state] == NSDrawerClosedState)];
	[m_dirDrawer toggle:self];
	//	[m_dirDrawer toggle:m_foldersBtn];
}

- (void)displayModeChanged {
	if (m_displayMode != [m_displayModeControl selectedSegment]) {
		m_displayMode = [m_displayModeControl selectedSegment];
		
		// list mode
		if (m_displayMode == DISPLAY_MODE_LIST) {
			[m_tabView selectTabViewItemWithIdentifier:ListTabIdent];
			[m_imgList loadThumbs:TRUE]; // stop loading
			[m_table scrollRowToVisible:[m_imgList selectedImageIndex]];
			
		// thumb mode
		} else {
			[m_tabView selectTabViewItemWithIdentifier:ThumbTabIdent];
			[m_imgList loadThumbs:FALSE]; // start or continue loading
			[m_window makeFirstResponder:m_thumbView];
			[m_thumbView reload];
		}
		
		[self _updateInfoLabel];
	}
}

#pragma mark -
#pragma mark Thumbview & Tooltip delegate methods

- (void)thumbViewDidClickThumbWithIndex:(unsigned)index {
	[m_imgList selectImageAtIndex:index];
}

- (void)thumbViewDidMarkThumbWithIndex:(unsigned)index withCenter:(NSPoint)center {
	m_markedThumb = index;
	NSString* desc = [m_imgList thumbDescriptionAtIndex:index];
	if ([desc length] > 0)
		[m_toolTip showWithLinesAtLocation:center,
			[m_imgList thumbNameAtIndex:index], desc, nil];
}

- (void)thumbViewDidUnmarkThumb {
	[m_toolTip hide];
}

- (BOOL)thumbViewWillBeginDrag:(unsigned)index {
    NSPasteboard *pboard = [NSPasteboard pasteboardWithName:NSDragPboard];

	[pboard declareTypes:[NSArray arrayWithObjects:ImageListDragType, NSFilesPromisePboardType, 
						NSStringPboardType, nil] owner: self];

	[pboard setPropertyList:[NSArray arrayWithObject:[NSNumber numberWithUnsignedInt:index]] 
					forType:ImageListDragType];
	[pboard setPropertyList:[NSArray arrayWithObject:[m_imgList extensionAtIndex:index]]
					forType:NSFilesPromisePboardType];
	[pboard setString:[m_imgList thumbDescriptionAtIndex:index]
			  forType:NSStringPboardType];
	
	return TRUE;
}

- (NSArray*)thumbViewNamesOfFilesDroppedAtDestination:(NSURL*)url {
    NSPasteboard*	pboard;
	NSArray*		imgIndexes;
	unsigned		index;

	pboard		= [NSPasteboard pasteboardWithName:NSDragPboard];
	imgIndexes	= [pboard propertyListForType:ImageListDragType];
	
	if ([imgIndexes count] != 1)
		return nil;
	
	index	= [[imgIndexes objectAtIndex:0] unsignedIntValue];

	if (![m_imgList copyImageAtIndex:index toDirectory:[url path]])
		return nil;
		
	return [NSArray arrayWithObject:[m_imgList fileNameAtIndex:index]];
}

- (void)toolTipClicked {
	[m_imgList selectImageAtIndex:m_markedThumb];
}

@end
