RCS_ID("$Id: FFDirectory.m 597 2006-09-30 19:12:57Z ravemax $")

#import "FFDirectory.h"
#import "FFPreferences.h"
#import "FFImageList.h"
#import "FFImageAndTextCell.h"
#import "FFSmartFolder.h"
#import "FFImageListController.h"

@implementation FFDirNode

#pragma mark -
#pragma mark Initialisation and cleanup

static NSImage* dirImages[NUM_DIR_TYPES];
static NSColor*	finderColors[8]; // kColor >> kIsOnDesk + 1

+ (void)initialize {
	dirImages[DIR_NORMAL]		= [[NSImage imageNamed:@"folder_small"] retain];
	dirImages[DIR_ARCHIVE]		= [[NSImage imageNamed:@"archive_small"] retain];
	dirImages[DIR_SMARTFOLDER]	= [[NSImage imageNamed:@"smartfolder_small"] retain];
	
	finderColors[0]	= nil;
	finderColors[1]	= [[NSColor colorWithDeviceRed:(207.0/255.0) green:(207.0/255.0) blue:(207.0/255.0) alpha:1.0] retain]; // gray
	finderColors[2]	= [[NSColor colorWithDeviceRed:(206.0/255.0) green:(230.0/255.0) blue:(133.0/255.0) alpha:1.0] retain]; // green
	finderColors[3]	= [[NSColor colorWithDeviceRed:(217.0/255.0) green:(182.0/255.0) blue:(231.0/255.0) alpha:1.0] retain]; // violett
	finderColors[4]	= [[NSColor colorWithDeviceRed:(149.0/255.0) green:(202.0/255.0) blue:(252.0/255.0) alpha:1.0] retain]; // blue
	finderColors[5]	= [[NSColor colorWithDeviceRed:(248.0/255.0) green:(237.0/255.0) blue:(133.0/255.0) alpha:1.0] retain]; // yellow
	finderColors[6]	= [[NSColor colorWithDeviceRed:(243.0/255.0) green:(147.0/255.0) blue:(140.0/255.0) alpha:1.0] retain]; // red
	finderColors[7]	= [[NSColor colorWithDeviceRed:(249.0/255.0) green:(203.0/255.0) blue:(135.0/255.0) alpha:1.0] retain]; // orange
}

- (int)_finderLabelColorOfPath:(NSString*)path {
	CFURLRef		url;
	FSRef			fsRef;
	BOOL			ret;
	FSCatalogInfo	cinfo;
	
	// Get FSRef
	url = CFURLCreateWithFileSystemPath(NULL, (CFStringRef)path, kCFURLPOSIXPathStyle, FALSE);
	if (!url)
		return 0;
		
	ret = CFURLGetFSRef(url, &fsRef);
	CFRelease(url);
	
	// Get Finder flags
	if (ret && 
		(FSGetCatalogInfo(&fsRef, kFSCatInfoFinderInfo, &cinfo, NULL, NULL, NULL) == noErr))
		return (((FileInfo*)&cinfo.finderInfo)->finderFlags & kColor) >> kIsOnDesk;
			
	return 0;
}

- (id)initWithPath:(NSString*)path type:(FFDirType)type andImage:(NSImage*)img {
	self = [super init];
	if (self) {
		m_completePath  = [path retain];
	
		if (type == DIR_SMARTFOLDER)
			m_displayName = [[[m_completePath lastPathComponent] stringByDeletingPathExtension] retain];
		else
			m_displayName = [[m_completePath lastPathComponent] retain];
	
		m_type			= type;
		m_subDirs		= nil;
		m_smartFolder	= nil;
		m_parentNode	= nil;
		m_img			= (img == nil) ? nil : [img retain];
		m_bgColor		= finderColors[ [self _finderLabelColorOfPath:path] ];
	}
	return self;
}

+ (FFDirNode*)directoryWithPath:(NSString*)path {
	return [[[FFDirNode alloc] initWithPath:path type:DIR_NORMAL andImage:nil] autorelease];
}

+ (FFDirNode*)archiveWithPath:(NSString*)path {
	return [[[FFDirNode alloc] initWithPath:path type:DIR_ARCHIVE andImage:nil] autorelease];
}

+ (FFDirNode*)smartFolderWithPath:(NSString*)path {
	return [[[FFDirNode alloc] initWithPath:path type:DIR_SMARTFOLDER andImage:nil] autorelease];
}

+ (FFDirNode*)removeableMediaWithPath:(NSString*)path {
	NSImage* img = [[NSWorkspace sharedWorkspace] iconForFile:path];
	if (img != nil)
		[img setSize:NSMakeSize(16.0f, 16.0f)]; // 50%

	return [[[FFDirNode alloc] initWithPath:path type:DIR_NORMAL andImage:img] autorelease];
}

- (void)dealloc {
	if (m_subDirs != nil)
		[m_subDirs release];
	if (m_img != nil)
		[m_img release];
	[m_displayName release];
	[m_completePath release];
	
	[super dealloc];
}

#pragma mark -
#pragma mark Getters

- (NSString*)completePath	{ return m_completePath; }
- (NSString*)displayName	{ return m_displayName; }
- (FFDirType)type			{ return m_type; }
- (NSColor*)bgColor			{ return m_bgColor; }
- (NSImage*)image			{ return (m_img == nil) ? dirImages[m_type] : m_img; }

- (void)_setParentNode:(FFDirNode*)pn {
	m_parentNode = pn; // weak
}

- (FFDirNode*)parentNode {
	return m_parentNode;
}

- (FFSmartFolder*)smartFolder { 
	if ((m_smartFolder == nil) && (m_type == DIR_SMARTFOLDER))
		m_smartFolder = [[FFSmartFolder alloc] initWithSavedSearch:m_completePath];
	
	return m_smartFolder;
}

- (void)_searchSubDirectories {
	NSFileManager*  fm;	
	NSArray*		dc;
	NSEnumerator*   en;
	NSString*		name, *cpath;
	BOOL			sfolder, isDir;
	FFDirNode*		newNode;
	
	m_subDirs   = [[NSMutableArray alloc] init];	
	fm			= [NSFileManager defaultManager];
	
	if (m_type == DIR_NORMAL) {
		sfolder	= FALSE;		
		dc		= [fm directoryContentsAtPath:m_completePath];
	} else { // DIR_SMARTFOLDER
		sfolder	= TRUE;
		dc		= ([self smartFolder] == nil) ? nil : [[self smartFolder] contents];
	}

	if (dc == nil)
		NSRunAlertPanel(FFTR(@"Failed to get the directory content"),
						FFTRC(@"The directory '%@' used for the image list directory drawer couldn't be read. Try to change it in the preferences.", @"1=Path"),
						FFTR(@"OK"), nil, nil,
						m_completePath);
	else {			
		en  = [dc objectEnumerator];
		while (name = [en nextObject]) {
			if ([name characterAtIndex:0] == '.') // Skip hidden
				continue;
		
			cpath = sfolder ? name : [m_completePath stringByAppendingPathComponent:name];
			if ([fm fileExistsAtPath:cpath isDirectory:&isDir]) {
				newNode = nil;
				if (isDir)
					newNode = [FFDirNode directoryWithPath:cpath];
				else {
					NSString* ext = [name pathExtension];
					if ([FFImageList isArchiveExtension:ext])
						newNode = [FFDirNode archiveWithPath:cpath];
					else if ([FFSmartFolder isSmartFolderExtension:ext])
						newNode = [FFDirNode smartFolderWithPath:cpath];
				}
				if (newNode != nil) {
					[newNode _setParentNode:self];
					[m_subDirs addObject:newNode];
				}
			}
		}
	}
}

- (unsigned)numberOfSubDirectories {
	if (m_type == DIR_ARCHIVE)
		return 0;
	
	if (m_subDirs == nil) {
		NSNotificationCenter* nc = [NSNotificationCenter defaultCenter];
		[nc postNotificationName:StartProgressNotification object:nil];
		[self _searchSubDirectories];
		[nc postNotificationName:StopProgressNotification object:nil];
	}
	
	return [m_subDirs count];
}

- (FFDirNode*)subDirectoryAtIndex:(unsigned)idx {
	return [m_subDirs objectAtIndex:idx];
}

- (unsigned)indexOfSubDirectory:(FFDirNode*)node {
	return [m_subDirs indexOfObjectIdenticalTo:node];
}

- (BOOL)isSelfOrParentNode:(FFDirNode*)node {
	if (self == node)
		return TRUE;
	if (m_parentNode == nil)
		return FALSE;
	
	return [m_parentNode isSelfOrParentNode:node];
}

#pragma mark -
#pragma mark Misc.

- (void)forgetSubDirectories {
	[m_subDirs release];
	m_subDirs = nil;
	if (m_smartFolder != nil) {
		[m_smartFolder release];
		m_smartFolder = nil;
	}
}

@end

#pragma mark -

NSString* DirectoryContentChangedNotification = @"dir_cnt_changed";

@implementation FFDirectory

- (id)init {
	self = [super init];
	if (self != nil) {
		// User node (= old root)
		m_nodes	= [[NSMutableArray alloc] initWithCapacity:1];
		[m_nodes addObject:[FFDirNode directoryWithPath:[[FFPreferences instance] directoryRoot]]];
		
		// Available removeable medis
		NSArray* mrm = [[NSWorkspace sharedWorkspace] mountedRemovableMedia];
		if ([mrm count] > 0) {
			NSEnumerator*	en = [mrm objectEnumerator];
			NSString*		rmpath;
			
			while (rmpath = [en nextObject])
				[m_nodes addObject:[FFDirNode removeableMediaWithPath:rmpath]];
		}
			
		// Notifications
		NSNotificationCenter* nc = [NSNotificationCenter defaultCenter];
		[nc addObserver:self selector:@selector(directoryRootChanged:)
					 name:DirectoryRootChangedNotification object:nil];

		NSNotificationCenter* wnc = [[NSWorkspace sharedWorkspace] notificationCenter];
			
		[wnc addObserver:self selector:@selector(workspaceDidMount:)
				   name:NSWorkspaceDidMountNotification object:nil];
		[wnc addObserver:self selector:@selector(workspaceDidUnmount:)
				   name:NSWorkspaceDidUnmountNotification object:nil];
	}
	return self;
}

- (void)dealloc {
	[m_nodes release];
	[super dealloc];
}

#pragma mark -
#pragma mark Exported methods

- (void)setImageList:(FFImageList*)list {
	m_imgList = list; // weak
}

- (BOOL)containsRemovableMedia {
	return ([m_nodes count] > 1);
}

#pragma mark -
#pragma mark Outlineview data source methods

- (id)outlineView:(NSOutlineView *)outlineView child:(int)index ofItem:(id)item {
	if (item == nil) {
		if ([self containsRemovableMedia])
			return [m_nodes objectAtIndex:index];
		return [[m_nodes objectAtIndex:0] subDirectoryAtIndex:index];		
	}
	
	return [item subDirectoryAtIndex:index];
}

- (id)outlineView:(NSOutlineView*)view objectValueForTableColumn:(NSTableColumn*)col byItem:(id)item {
	return [item displayName];
}

- (BOOL)outlineView:(NSOutlineView *)outlineView isItemExpandable:(id)item {
	return ([(FFDirNode*)item type] != DIR_ARCHIVE);
}

- (int)outlineView:(NSOutlineView *)outlineView numberOfChildrenOfItem:(id)item {
	if (item == nil) {
		if ([self containsRemovableMedia])
			return [m_nodes count];
		return [[m_nodes objectAtIndex:0] numberOfSubDirectories];
	} 
	
	return [item numberOfSubDirectories];
}

- (NSDragOperation)outlineView:(NSOutlineView*)outlineView validateDrop:(id<NSDraggingInfo>)info
				  proposedItem:(id)item proposedChildIndex:(int)index {
	
	if ([(FFDirNode*)item type] == DIR_NORMAL) {
		NSPasteboard* pboard = [info draggingPasteboard];
		if ([pboard availableTypeFromArray:[NSArray arrayWithObject:ImageListDragType]])
			return NSDragOperationCopy;
	}

	return NSDragOperationNone;
}

- (BOOL)outlineView:(NSOutlineView *)ov acceptDrop:(id<NSDraggingInfo>)info
			   item:(id)item childIndex:(int)index {

	NSPasteboard*	pboard	= [info draggingPasteboard];
	NSEnumerator*	en		= [[pboard propertyListForType:ImageListDragType] objectEnumerator];
	NSNumber*		row;
	
	while (row = [en nextObject]) {
		if (![m_imgList copyImageAtIndex:[row unsignedIntValue] toDirectory:[item completePath]])
			break;
	}
	
	return TRUE;
}

#pragma mark -
#pragma mark Outlineview delegate methods

- (BOOL)outlineView:(NSOutlineView*)ov shouldCollapseItem:(id)item {
	if (m_delegate == nil)
		return TRUE;
	
	return [m_delegate directory:self shouldCollapseNode:(FFDirNode*)item];
}

- (void)outlineViewItemDidCollapse:(NSNotification*)not {	
	[[[not userInfo] objectForKey:@"NSObject"] forgetSubDirectories];
}

- (void)outlineView:(NSOutlineView*)ov willDisplayCell:(id)cell 
	 forTableColumn:(NSTableColumn*)col item:(id)item {

	NSColor* c = [(FFDirNode*)item bgColor];
	if (c == nil)
		[cell setDrawsBackground:FALSE];
	else {
		[cell setDrawsBackground:TRUE];
		[cell setBackgroundColor:c];
	}
	
	[(FFImageAndTextCell*)cell setImage:[(FFDirNode*)item image]];
}

#pragma mark -
#pragma mark Notification handlers

#define DIR_CONTENT_CHANGED(OBJ) \
	[[NSNotificationCenter defaultCenter] \
		postNotificationName:DirectoryContentChangedNotification object:OBJ]

- (void)directoryRootChanged:(NSNotification*)not {
	FFDirNode* rootNode = [FFDirNode directoryWithPath:[[FFPreferences instance] directoryRoot]];
	[m_nodes replaceObjectAtIndex:0 withObject:rootNode];	
	DIR_CONTENT_CHANGED(rootNode);
}

- (void)workspaceDidMount:(NSNotification*)not {
	NSString* dpath = [[not userInfo] objectForKey:@"NSDevicePath"];
	if ([[[NSWorkspace sharedWorkspace] mountedRemovableMedia] containsObject:dpath]) {
		FFDirNode* rmNode = [FFDirNode removeableMediaWithPath:dpath];
		[m_nodes addObject:rmNode];
		DIR_CONTENT_CHANGED(rmNode);
	}
}

- (void)workspaceDidUnmount:(NSNotification*)not {
	NSString*	devPath	= [[not userInfo] objectForKey:@"NSDevicePath"];
	BOOL		found	= FALSE;
	unsigned	i;

	for (i = 1; i < [m_nodes count]; i++)
		if ([[[m_nodes objectAtIndex:i] completePath] isEqualToString:devPath]) {
			[m_nodes removeObjectAtIndex:i];
			found = TRUE;
			break;
		}
	
	if (found) {
		DIR_CONTENT_CHANGED(nil);
		[m_imgList removeFilesWithPrefix:devPath];
	}
}

@end
