RCS_ID("$Id: FFImageList.m 610 2007-08-26 19:40:40Z ravemax $")

#import "FFImageList.h"
#import <stdlib.h>
#import <FFArchive/FFArchive.h>
//#import <FFArchive/FFArchiveZIP.h>
//#import <FFArchive/FFArchiveRAR.h>
#import "NSArray_Additions.h"
#import "FFImageFile.h"
#import "FFPreferences.h"
#import "FFOptions.h"
#import "FFRecentList.h"
#import "FFInfoPanel.h"
#import "FFDirectory.h"
#import "FFSmartFolder.h"
#import "FFArchivePDF.h"
#import "FFImageListController.h"
#import "FFImageLoader.h"

// Notifications (sends)
NSString*	ImageListChangedNotification		= @"image_list_changed";
NSString*	NewImageSelectedNotification		= @"new_image_selected";
NSString*	SlideshowStartedNotification		= @"slideshow_start";
NSString*	SlideshowStoppedNotification		= @"slideshow_stop";
NSString*   ImageListRearrangedNotification		= @"image_list_rearranged";

#define IMAGELIST_CHANGED_NOTIFICATION() \
	[m_nc postNotificationName:ImageListChangedNotification object:self]

#define NEW_IMAGE_NOTIFICATION() \
	[m_nc postNotificationName:NewImageSelectedNotification object:self]

#define SLIDESHOW_START_NOTIFICATION() \
	[m_nc postNotificationName:SlideshowStartedNotification object:nil]

#define SLIDESHOW_STOP_NOTIFICATION() \
	[m_nc postNotificationName:SlideshowStoppedNotification object:nil]

#define IMAGELIST_REARRANGED_NOTIFCATION() \
	[m_nc postNotificationName:ImageListRearrangedNotification object:self]

// Notifications (receives)
NSString*	SelectPreviousImageNotification		= @"sel_prev_img";
NSString*	SelectNextImageNotification			= @"sel_next_img";
NSString*	SelectFirstImageNotification		= @"sel_first_img";
NSString*	SelectLastImageNotification			= @"sel_last_img";
NSString*	SelectSpecifiedImageNotification	= @"sel_spec_img";
NSString*	RemoveCurrentImageNotification		= @"remove_img";
NSString*	StopSlideshowNotificaton			= @"stop_slideshow";

// Constants for drag'n'drop in the table
NSString*	ImageListDragType	= @"ImageListDragType";

// The types
static NSString* TypeKeyImages		= @"images";
static NSString* TypeKeyZIP			= @"ZIP";
static NSString* TypeKeyRAR			= @"RAR";
static NSString* TypeKeyPDF			= @"PDF";
static NSString* TypeKeyDirectory   = @"directory";
static NSString* TypeKeyImageList   = @"FFView list";
static NSString* ImageTypeSuffix	= @"image";

// Storage keys
static NSString* StoreKeyFiles = @"files";
static NSString* StoreKeyIndex = @"index";

// Ignore resource fork files (e.g. in ZIP files)
static NSString* ResourceForkFilenamePrefix	= @"._";

// Static (private) global variables
static SEL SortSelectors[IMGLIST_NUM_ORDER];

// Global vars - types & extensions (comes from Info.plist)
NSMutableDictionary*	KnownFileTypes		= nil;
NSMutableArray*			KnownFileExtensions = nil;
NSString*				ImageListFileExtension;

// Useful macros
#define FILTER_IMAGEFILE(FILE) \
	[FILE setFiltered:![[[FILE displayName] string] hasPrefix:m_filterString]];

@implementation FFImageList

#pragma mark -
#pragma mark Init and cleanup

+ (void)initialize {
	if (KnownFileTypes == nil) {
	// File types (= document types)
		NSArray*		docTypes, *exts;
		NSEnumerator*   en;
		NSDictionary*   type;
		
		// Get all file/doc types
		KnownFileTypes = [[NSMutableDictionary alloc] initWithCapacity:3];
		docTypes = [[[NSBundle mainBundle] infoDictionary] objectForKey:@"CFBundleDocumentTypes"];
		en = [docTypes objectEnumerator];
		while (type = [en nextObject]) {
			NSString* name = [type objectForKey:@"CFBundleTypeName"];
			if ([name isEqualToString:TypeKeyDirectory])
				continue;
			if ([name hasSuffix:ImageTypeSuffix]) {
				NSMutableArray* e = [KnownFileTypes objectForKey:TypeKeyImages];
				if (e == nil)
					[KnownFileTypes setObject:[NSMutableArray arrayWithArray:[type objectForKey:@"CFBundleTypeExtensions"]] forKey:TypeKeyImages];
				else
					[e addObjectsFromArray:[type objectForKey:@"CFBundleTypeExtensions"]];
			} else
				[KnownFileTypes setObject:[type objectForKey:@"CFBundleTypeExtensions"] forKey:name];
		}
		
		// Combine all extensions into one array (e.g. for the openpanel)
		KnownFileExtensions = [[NSMutableArray alloc] initWithCapacity:[KnownFileTypes count]];
		en = [[KnownFileTypes allValues] objectEnumerator];
		while (exts = [en nextObject])
			[KnownFileExtensions addObjectsFromArray:exts];		
		
		// Get the list extension
		ImageListFileExtension = [[KnownFileTypes objectForKey:TypeKeyImageList] objectAtIndex:0];
		
		// Sort selectors
		SortSelectors[IMGLIST_USER_ASC]		= @selector(compareFileIndexAsc:);
		SortSelectors[IMGLIST_USER_DESC]	= @selector(compareFileIndexDesc:);
		SortSelectors[IMGLIST_NUMERIC_ASC]	= @selector(compareNumericAsc:);
		SortSelectors[IMGLIST_NUMERIC_DESC]	= @selector(compareNumericDesc:);
		SortSelectors[IMGLIST_ALPHA_ASC]	= @selector(compareAlphaAsc:);
		SortSelectors[IMGLIST_ALPHA_DESC]	= @selector(compareAlphaDesc:);
		SortSelectors[IMGLIST_PATH_ASC]		= @selector(comparePathAsc:);
		SortSelectors[IMGLIST_PATH_DESC]	= @selector(comparePathDesc:);
		SortSelectors[IMGLIST_DATE_ASC]		= @selector(compareDateAsc:);
		SortSelectors[IMGLIST_DATE_DESC]	= @selector(compareDateDesc:);
		SortSelectors[IMGLIST_FILESIZE_ASC]	= @selector(compareFileSizeAsc:);
		SortSelectors[IMGLIST_FILESIZE_DESC]= @selector(compareFileSizeDesc:);
	}
}

- (id)initWithPreferences:(FFPreferences*)prefs options:(FFOptions*)opts 
			andRecentList:(FFRecentList*)rlist {
	if ([super init]) {
		m_prefs			= [prefs retain];
		m_opts			= [opts retain];
		m_recentList	= [rlist retain];
		
		m_list			= [[NSMutableArray alloc] init];
		m_fm			= [NSFileManager defaultManager];
		m_nc			= [NSNotificationCenter defaultCenter];
		m_filterString	= nil;
		m_thumbLock		= [[NSLock alloc] init];
		m_thumbWidth	= [prefs thumbWidth];
		m_thumbHeight	= [prefs thumbHeight];		
		
		m_selectedImage			= -1;		
		m_archiveFilesInTemp	= [[NSMutableArray alloc] init];
		m_slideshowTimer		= nil;
		m_sortOrder				= [prefs imageListSortOrder];
		
		// Register for notifications
		[m_nc addObserver:self selector:@selector(selectPreviousImage:)
					 name:SelectPreviousImageNotification object:nil];
		[m_nc addObserver:self selector:@selector(selectNextImage:)
					 name:SelectNextImageNotification object:nil];
		[m_nc addObserver:self selector:@selector(selectFirstImage:)
					name:SelectFirstImageNotification object:nil];
		[m_nc addObserver:self selector:@selector(selectLastImage:)
					name:SelectLastImageNotification object:nil];
		[m_nc addObserver:self selector:@selector(selectSpecifiedImage:)
					 name:SelectSpecifiedImageNotification object:nil];
		[m_nc addObserver:self selector:@selector(removeCurrentImage:)
					 name:RemoveCurrentImageNotification object:nil];
		[m_nc addObserver:self selector:@selector(stopSlideshow:)
					 name:StopSlideshowNotificaton object:nil];
		
		// Archive helper
		[FFArchive setHelper:self];
	}
	return self;
}

- (void)dealloc {	
	[m_nc removeObserver:self];
	[self stopSlideshow];

	[FFArchive setHelper:nil];	
	[m_archiveFilesInTemp release];

	[m_thumbLock release];
	[m_list release];
	[m_recentList release];
	[m_opts release];
	[m_prefs release];
	
	
	[super dealloc];
}

- (void)cleanUp {
	// Remove temporary extracted files
	if ([m_archiveFilesInTemp count] > 0) {
		NSEnumerator*	en = [m_archiveFilesInTemp objectEnumerator];
		NSString*		afname;
		
		while (afname = [en nextObject])
			[m_fm removeFileAtPath:afname handler:nil];
	
		[m_archiveFilesInTemp removeAllObjects];
	}
}

#pragma mark -
#pragma mark Manipulation of the list

- (FFArchive*)_createArchiveWithFilepath:(NSString*)fpath 
						  usingExtension:(NSString*)ext {

	FFArchive* arch = [FFArchive archiveWithFile:fpath fallbackEncoding:[m_opts archiveEncoding]];
	if (arch == nil) {
		NSString* ext = [fpath pathExtension];
		if ([[KnownFileTypes objectForKey:TypeKeyPDF] containsCaseInsensitiveString:ext])
			arch = [[FFArchivePDF alloc] initWithFile:fpath fallbackEncoding:[m_opts archiveEncoding]];
	}		
	return arch;
}

- (BOOL)_addFile:(NSString*)fpath addToRecent:(BOOL)recent {
	NSString*		fext;
	FFArchive*		arch;
	NSArray*		afiles, *imgExts;
	NSEnumerator*   en;
	NSDictionary*	file;
	NSString*		fn;
	
	// Valid extension ?
	fext = [fpath pathExtension];
	if (![KnownFileExtensions containsCaseInsensitiveString:fext])
		return FALSE;

	// Is a known archive ?
	arch = [self _createArchiveWithFilepath:fpath usingExtension:fext];
	
	// Get all archive image files
	if (arch != nil) {
		NS_DURING
			imgExts = [KnownFileTypes objectForKey:TypeKeyImages];
			afiles = [arch filesInArchive];
			en = [afiles objectEnumerator];
			while (file = [en nextObject]) {
				fn = [file objectForKey:FFFilename];
				if (![[fn lastPathComponent] hasPrefix:ResourceForkFilenamePrefix]
					&& [imgExts containsCaseInsensitiveString:[fn pathExtension]])
					[m_list addObject:[FFImageFile file:fn fromArchive:arch
										   withFileSize:[[file objectForKey:FFUncompressedFileSize] unsignedLongLongValue]
												andDate:[file objectForKey:FFFileCreationDate]]];
			}
		NS_HANDLER
			NSRunInformationalAlertPanel(FFTR(@"Archive trouble"),
										 FFTRC(@"Filename: %@\nMessage: \"%@\"", @"0=Path, 1=Errormessage"),
										 FFTR(@"OK"), nil, nil,
										 fpath, [localException reason]);
			return FALSE;
		NS_ENDHANDLER
	
		[arch release];
		
		// Remember this non-corrupt & valid archive unless its root dir is already stored
		if (recent)
			[m_recentList add:fpath];

	// Image list
	} else if ([[KnownFileTypes objectForKey:TypeKeyImageList] containsCaseInsensitiveString:fext])
		[self addImageList:fpath restoreSelection:FALSE addToRecent:recent postNotifications:FALSE];
	else
		[m_list addObject:[FFImageFile normalFile:fpath]];
	
	return TRUE;
}

- (void)_addFilesWithEnumerator:(NSEnumerator*)fe directory:(NSString*)dir 
						  level:(int)level {
	NSString*   curFPath;
	BOOL		isDir;

	while (curFPath = [fe nextObject]) {
		// Only filename
		if (dir != nil)
			curFPath = [dir stringByAppendingPathComponent:curFPath];
		
		// Directory..
		if (![m_fm fileExistsAtPath:curFPath isDirectory:&isDir]) // this should never happen
			continue; 
		if (isDir) {
			if (level > 0) {
				if (dir == nil)
					[m_recentList add:curFPath];

				if (![m_prefs isBlacklistedDirectory:[curFPath lastPathComponent]])
					[self _addFilesWithEnumerator:[[m_fm directoryContentsAtPath:curFPath] objectEnumerator]
										directory:curFPath level:(level - 1)];
			}
		// .. Smart folder
		} else if ([FFSmartFolder isSmartFolderExtension:[curFPath pathExtension]]) {
			FFSmartFolder* sf = [[FFSmartFolder alloc] initWithSavedSearch:curFPath];
			if (sf != nil) {
				[self _addFilesWithEnumerator:[[sf contents] objectEnumerator]
									directory:nil level:(level - 1)];
				[sf release];
			}
			
		// .. or File
		} else
			(void)[self _addFile:curFPath addToRecent:(dir == nil)];
	}
}

- (void)_sortAcordingToCurrentOrder {
	if (m_sortOrder != IMGLIST_USER_ASC)
		[m_list sortUsingSelector:SortSelectors[m_sortOrder]];
}

- (void)addFiles:(NSArray*)files tillLevel:(int)level 
		restoreSelectionIfPossible:(BOOL)resSel {
	unsigned cb = [m_list count];

	[m_nc postNotificationName:StartProgressNotification object:nil];
	
	// If an .ffviewlist (1) add the files (2) restore the index
	if (resSel && (cb == 0) && ([files count] == 1) && 
		([[[files objectAtIndex:0] pathExtension] caseInsensitiveCompare:ImageListFileExtension] == NSOrderedSame))
		[self addImageList:[files objectAtIndex:0] restoreSelection:TRUE addToRecent:TRUE postNotifications:FALSE];
	else
		[self _addFilesWithEnumerator:[files objectEnumerator] directory:nil level:level];

	if ([m_list count] > cb) {
		// Store current index if any
		FFImageFile* selFile;
		if (m_selectedImage != -1)
			selFile = [m_list objectAtIndex:m_selectedImage];
		
		[self _sortAcordingToCurrentOrder];
		
		if (cb == 0) {
			if (m_selectedImage == -1) // No position was restored
				m_selectedImage = 0;
			NEW_IMAGE_NOTIFICATION();
			IMAGELIST_CHANGED_NOTIFICATION();
		} else if (m_sortOrder == IMGLIST_USER_ASC)
			IMAGELIST_CHANGED_NOTIFICATION();
		else {
			m_selectedImage = [m_list indexOfObjectIdenticalTo:selFile];
			IMAGELIST_REARRANGED_NOTIFCATION();
		}
	}
	
	[m_nc postNotificationName:StopProgressNotification object:nil];
}

- (void)replace:(FFDirNode*)dir onlyImages:(BOOL)onlyImgs {
	[m_list removeAllObjects];

	if ([dir type] == DIR_ARCHIVE)
		(void)[self _addFile:[dir completePath] addToRecent:FALSE];
	else {
		NSEnumerator*	fen;
		NSString*		dpath, *fname;
		
		if ([dir type] == DIR_NORMAL) {
			dpath	= [dir completePath];
			fen		= [[m_fm directoryContentsAtPath:dpath] objectEnumerator];
		} else if ([dir smartFolder] != nil) {
			dpath	= nil;
			NSArray* dudu = [[dir smartFolder] contents];
			fen		= [dudu objectEnumerator];
		}
	
		while (fname = [fen nextObject]) {
			if (!onlyImgs || [[KnownFileTypes objectForKey:TypeKeyImages] containsCaseInsensitiveString:[fname pathExtension]])
				(void)[self _addFile:(dpath == nil ? fname :  [dpath stringByAppendingPathComponent:fname])
						 addToRecent:FALSE];
		}
	}
	
	if ([m_list count] > 0) {
		[self _sortAcordingToCurrentOrder];
		m_selectedImage = 0;
		NEW_IMAGE_NOTIFICATION();
		
		// Only remember the dir/archive if it contained any files & was valid
		[m_recentList add:[dir completePath]];
	} else
		m_selectedImage = -1;
	
	IMAGELIST_CHANGED_NOTIFICATION();
}

- (void)replaceWithImageList:(NSString*)filename restoreSelection:(BOOL)resSel {
	[m_list removeAllObjects];
	
	if (![self addImageList:filename restoreSelection:resSel addToRecent:TRUE postNotifications:TRUE]) {
		m_selectedImage = -1;
//		NEW_IMAGE_NOTIFICATION();
		IMAGELIST_CHANGED_NOTIFICATION();
	}
}

- (void)clear {
	[m_list removeAllObjects];
	m_selectedImage = -1;
	IMAGELIST_CHANGED_NOTIFICATION();
	[self cleanUp];
}

- (void)removeInIndexSet:(NSIndexSet*)iset {
	unsigned	cnt;
	
	cnt = [iset count];	
	if (cnt == [m_list count])
		[self clear];
	else {		
		unsigned*   indices;
		NSRange		rg;
		unsigned	sis;
		BOOL		newImg;
		
		indices = (unsigned*)malloc(sizeof(unsigned)*cnt);
		rg.location = [iset firstIndex];
		rg.length = ([iset lastIndex]+1)-rg.location;
		[iset getIndexes:indices maxCount:cnt inIndexRange:&rg];
	
		// Get the number of selected entries before the currently selected image
		for (sis = 0; (indices[sis] < (unsigned)m_selectedImage) && (sis < cnt); sis++)
			;
				
		// Remember if the currently selected image is removed
		newImg = [iset containsIndex:(unsigned)m_selectedImage];
		
		// Now remove the selected ones
		[m_list removeObjectsFromIndices:indices numIndices:cnt];	
		free(indices);

		// Correct image selection
		m_selectedImage -= sis;		
		if (newImg) {
			if (m_selectedImage >= [m_list count])
				m_selectedImage--;
			NEW_IMAGE_NOTIFICATION();
		}
		IMAGELIST_CHANGED_NOTIFICATION();		
	}
}

- (void)removeFilesWithPrefix:(NSString*)prefix {
	BOOL	changed = FALSE, newImg = FALSE;
	int		i = 0;

	while (i < [m_list count]) {
		if ([[[m_list objectAtIndex:i] fullPath] hasPrefix:prefix]) {
			changed = TRUE;
			[m_list removeObjectAtIndex:i];
			if (m_selectedImage > i)
				m_selectedImage--;
			else if (m_selectedImage == i) {
				newImg = TRUE;
				if (m_selectedImage >= [m_list count])
					m_selectedImage--;
			}
		} else
			i++;
	}
	
	if (changed) {
		if (newImg && (m_selectedImage > -1))
			NEW_IMAGE_NOTIFICATION();
		IMAGELIST_CHANGED_NOTIFICATION();		
	}
}

#pragma mark
#pragma mark Storage

- (BOOL)save:(NSString*)filename {
	NSMutableArray* fa = [NSMutableArray arrayWithCapacity:[self numberOfImages]];
	NSEnumerator*   en = [m_list objectEnumerator];
	NSString*		ilpath;
	FFImageFile*	imf;
	NSDictionary*	ldict;
	BOOL			ret;
	
	// Create the file dictionary
	ilpath = [filename stringByDeletingLastPathComponent];
	while (imf = [en nextObject])
		[fa addObject:[imf asDictRelativeToPath:ilpath]];
	
	// Create the list dictionary & save it
	ldict = [NSDictionary dictionaryWithObjectsAndKeys:
		fa, StoreKeyFiles,
		[NSNumber numberWithInt:m_selectedImage], StoreKeyIndex,
		nil];

	ret = [ldict writeToFile:filename atomically:FALSE];

	if (ret)
		[m_recentList add:filename];

	return ret;
}

#if 0
- (BOOL)addImageList:(NSString*)filename restoreSelection:(BOOL)resSel addToRecent:(BOOL)recent {
	// Load the list
	NSDictionary*   list	= [NSDictionary dictionaryWithContentsOfFile:filename];
	NSArray*		fa		= [list objectForKey:StoreKeyFiles];
	
	if (fa == nil) {
		NSRunAlertPanel(FFTR(@"Corrupt list"), FFTR(@"The file doesn't contain filenames."),
						FFTR(@"OK"), nil, nil);
		return FALSE;
	}
	
	// Add all files
	NSString*				ilpath  = [filename stringByDeletingLastPathComponent];
	NSEnumerator*			en		= [fa objectEnumerator];
	NSDictionary*			fd;
	BOOL					warn	= TRUE;
	NSFileManager*			fm		= [NSFileManager defaultManager];
	NSMutableDictionary*	archs   = [NSMutableDictionary dictionary];
	
	while (fd = [en nextObject]) {		
		// File exists ?
		NSString*   abspath = [FFImageFile absolutFullpathFromDict:fd relativeToPath:ilpath];

		if (![fm isReadableFileAtPath:abspath]) {
			if (warn) {
				int ret = NSRunAlertPanel(FFTR(@"File error"),
										  FFTRC(@"The file '%@' doesn't exist or isn't readable.", @"1=Absolute path"),
										  FFTR(@"Continue"), FFTR(@"Continue - no warnings"), FFTR(@"Stop processing"),
										  abspath);
				if (ret == NSAlertOtherReturn)
					return FALSE;
				if (ret == NSAlertAlternateReturn)
					warn = FALSE;
			}
			continue;
		}
		
		// Type
		FFImageFileType type = [FFImageFile typeFromDict:fd];
		FFImageFile*	imf;
		
		if (type == IMG_PART_OF_ARCHIVE) {
			FFArchive*  ar = [archs objectForKey:[abspath lastPathComponent]];
			// Archive not read yet
			if (ar == nil) {
				ar = [self _createArchiveWithFilepath:abspath usingExtension:nil];
				[archs setObject:ar forKey:[abspath lastPathComponent]];
			}
			imf = [FFImageFile fileFromDict:fd fromArchive:ar];
		} else
			imf = [FFImageFile normalFileFromDict:fd relativeToPath:ilpath];

		[m_list addObject:imf];
//		FILTER_IMAGEFILE(imf);
	}

	// Pretty stupid
	if (resSel) {
		NSNumber* idx = [list objectForKey:StoreKeyIndex];
		if (idx != nil) {
			m_selectedImage = [idx intValue];
			if (m_selectedImage >= [m_list count]) // Some files couldn't be loaded
				m_selectedImage = [m_list count] - 1;
		
			NEW_IMAGE_NOTIFICATION();
		}		
		IMAGELIST_CHANGED_NOTIFICATION();
	}
	
	// Open Recent
	if (recent)
		[m_recentList add:filename];
	
	return TRUE;
}
#endif

- (BOOL)addImageList:(NSString*)filename restoreSelection:(BOOL)resSel 
		 addToRecent:(BOOL)recent postNotifications:(BOOL)postNot {
	// Load the list
	NSDictionary*   list			= [NSDictionary dictionaryWithContentsOfFile:filename];
	NSArray*		fa				= [list objectForKey:StoreKeyFiles];
	int				selIdx			= -1;
	
	if (fa == nil) {
		NSRunAlertPanel(FFTR(@"Corrupt list"), FFTR(@"The file doesn't contain filenames."),
						FFTR(@"OK"), nil, nil);
		return FALSE;
	}
	
	// Get the stored list position (index)
	if (resSel) {
		NSNumber* idx = [list objectForKey:StoreKeyIndex];
		if (idx != nil)
			selIdx = [idx intValue];
	}
	
	// Add all files
	NSString*				ilpath  = [filename stringByDeletingLastPathComponent];
	NSEnumerator*			en		= [fa objectEnumerator];
	NSDictionary*			fd;
	BOOL					warn	= TRUE;
	NSFileManager*			fm		= [NSFileManager defaultManager];
	NSMutableDictionary*	archs   = [NSMutableDictionary dictionary];
	int						curIdx	= 0;
	FFImageFile*			selImf	= nil;
	
	for (; fd = [en nextObject]; curIdx++) {		
		// File exists ?
		NSString*   abspath = [FFImageFile absolutFullpathFromDict:fd relativeToPath:ilpath];
		
		if (![fm isReadableFileAtPath:abspath]) {
			if (warn) {
				int ret = NSRunAlertPanel(FFTR(@"File error"),
										  FFTRC(@"The file '%@' doesn't exist or isn't readable.", @"1=Absolute path"),
										  FFTR(@"Continue"), FFTR(@"Continue - no warnings"), FFTR(@"Stop processing"),
										  abspath);
				if (ret == NSAlertOtherReturn)
					return FALSE;
				if (ret == NSAlertAlternateReturn)
					warn = FALSE;
			}
			continue;
		}
		
		// Type
		FFImageFileType type = [FFImageFile typeFromDict:fd];
		FFImageFile*	imf;
		
		if (type == IMG_PART_OF_ARCHIVE) {
			FFArchive*  ar = [archs objectForKey:[abspath lastPathComponent]];
			// Archive not read yet
			if (ar == nil) {
				ar = [self _createArchiveWithFilepath:abspath usingExtension:nil];
				[archs setObject:ar forKey:[abspath lastPathComponent]];
			}
			imf = [FFImageFile fileFromDict:fd fromArchive:ar];
		} else
			imf = [FFImageFile normalFileFromDict:fd relativeToPath:ilpath];
		
		[m_list addObject:imf];
		//		FILTER_IMAGEFILE(imf);
		
		// Was this the selected image?
		if (curIdx == selIdx)
			selImf	= imf;
	}

	// Sort the files
	if (postNot && ([m_list count] > 0))
		[self _sortAcordingToCurrentOrder];
		
		
	// Restore the selection
	if (resSel) {
		if ([m_list count] > 0)
			m_selectedImage = (selImf != nil) ? [m_list indexOfObjectIdenticalTo:selImf] : 0;
		else
			m_selectedImage	= -1;
	}

	// Notifications senden
	if (postNot) {
		if (!resSel)
			m_selectedImage	= ([m_list count] > 0) ? 0 : -1;
						
		NEW_IMAGE_NOTIFICATION();
		IMAGELIST_CHANGED_NOTIFICATION();
	}
	
	// Open Recent
	if (recent)
		[m_recentList add:filename];
	
	return TRUE;
}

#pragma mark -
#pragma mark Information and values of the list

- (unsigned)numberOfImages { 
	return [m_list count];
}

- (NSAttributedString*)displayNameAtIndex:(unsigned)index {
	return [(FFImageFile*)[m_list objectAtIndex:index] displayName];
}

#define ARCHIVE_FILE_TO_TEMP(IMGFILE) \
	[NSTemporaryDirectory() stringByAppendingFormat:@"/ffview_af_%u.%@", \
		[IMGFILE fileIndex], [[IMGFILE fullPath] pathExtension]]

- (NSString*)fullPathOfImageFile:(FFImageFile*)imf {
	NSString*	fp, *ap;

	fp = [imf fullPath];
	if ([imf fileType] == IMG_NORMAL_FILE)
		return fp;
	
	// Extract if not im temp dir
	ap = ARCHIVE_FILE_TO_TEMP(imf);
	if (![m_archiveFilesInTemp containsObject:ap]) {
		FFLOG(1, @"extract '%@'", ap)
		
		NS_DURING
			[[imf archive] extractFile:fp toFilePath:ap];			
		NS_HANDLER
			NSRunInformationalAlertPanel(FFTR(@"Archive trouble"),
										 FFTRC(@"Filename: %@\nMessage: \"%@\"", @"0=Path, 1=Errormessage"),
										 FFTR(@"OK"), nil, nil,
										 [[imf displayName] string], [localException reason]);
			return nil;
		NS_ENDHANDLER
		
		[m_archiveFilesInTemp addObject:ap];
	}
	
	return ap;
}

- (NSString*)fullPathAtIndex:(unsigned)index {
	if (index < [m_list count])
		return [self fullPathOfImageFile:[m_list objectAtIndex:index]];
	return nil;
}

- (BOOL)filteredAtIndex:(unsigned)index {
	return [[m_list objectAtIndex:index] filtered];
}

- (unsigned)fileIndexAtIndex:(unsigned)index {
	if (index < [m_list count])
		return [[m_list objectAtIndex:index] fileIndex];
	return UINT32_MAX;
}

- (NSString*)extensionAtIndex:(unsigned)index {
	return [[[self displayNameAtIndex:index] string] pathExtension];
}

- (NSString*)fileNameAtIndex:(unsigned)index {
	return [[[self displayNameAtIndex:index] string] lastPathComponent];	
}

#pragma mark -
#pragma mark Image selection

- (int)selectedImageIndex {
	return m_selectedImage;
}

- (NSAttributedString*)displayNameOfSelectedImage {
	return [(FFImageFile*)[m_list objectAtIndex:m_selectedImage] displayName];
}

- (NSAttributedString*)displayNameOfNextImage {
	if (m_selectedImage+1 < [m_list count])
		return [(FFImageFile*)[m_list objectAtIndex:(m_selectedImage+1)] displayName];
	return nil;
}

- (BOOL)_wrapCheckWithTitle:(NSString*)title andMessage:(NSString*)msg {
	if (([m_list count] < 1) || ([m_prefs listWrap] == PREF_WRAP_NEVER))
		return FALSE;

	if ([m_prefs listWrap] == PREF_WRAP_ASK)
		return ([FFInfoPanel runInFullscreen:[m_opts fullscreen] withTitle:title message:msg
									okButton:FFTR(@"Yes") andCancelButton:FFTR(@"No")] == NSOKButton);
	
	// PREF_WRAP_ALWAYS
	NSBeep();
	return TRUE; 
}

- (void)setSelectionDecrementer:(int)dec {
	FFLOG(8, @"dec: %d", dec)
	m_selDec = dec;
}

- (BOOL)selectPreviousImage {
	// No previous image
	if ([m_list count] == 1)
		return FALSE;

	// Normal previous image(s)
	if (m_selectedImage >= m_selDec)
		m_selectedImage -= m_selDec;

	// Wrap around
	else if (m_selectedImage - m_selDec <= -m_selDec) {
		if ([self _wrapCheckWithTitle:FFTR(@"Beginning of the list reached")
						   andMessage:FFTR(@"Go to the end of the list?")])
			m_selectedImage = [m_list count] - m_selDec;
		else
			return FALSE;
		
	} else // First image (only double page)
		m_selectedImage = 0;
	
	NEW_IMAGE_NOTIFICATION();
	return TRUE;
}

- (void)selectPreviousImage:(NSNotification*)notification {
	(void)[self selectPreviousImage];
}

- (void)setSelectionIncrementer:(int)inc {
	FFLOG(8, @"inc: %d", inc)
	m_selInc = inc;
}

- (BOOL)selectNextImage {
	if (m_selectedImage + m_selInc < [m_list count])
		m_selectedImage += m_selInc;
	
	// No image left - wrap check
	else if (m_selectedImage + m_selInc >= [m_list count]) {
		if ((m_slideshowTimer == nil) &&
			[self _wrapCheckWithTitle:FFTR(@"End of the list reached")
						   andMessage:FFTR(@"Go to the beginning of the list ?")]) {
			m_selectedImage = 0;
		} else
			return FALSE;
		
	// 1 image left (only possible if double page mode)
	} else
		m_selectedImage++;

	NEW_IMAGE_NOTIFICATION();
	return TRUE;
}

- (void)selectNextImage:(NSNotification*)notification {
	(void)[self selectNextImage];
}

- (BOOL)selectFirstImage {
	if (m_selectedImage <= 0) // no images or first already selected
		return FALSE;
	
	m_selectedImage = 0;
	NEW_IMAGE_NOTIFICATION();
	return TRUE;
}

- (void)selectFirstImage:(NSNotification*)notification {
	(void)[self selectFirstImage];
}

- (BOOL)selectLastImage {
	int nidx = [m_list count]-1;
	if ((nidx < 0) || (m_selectedImage == nidx))
		return FALSE;
	
	m_selectedImage = nidx;
	NEW_IMAGE_NOTIFICATION();
	return TRUE;
}

- (void)selectLastImage:(NSNotification*)notification {
	(void)[self selectLastImage];
}

- (BOOL)selectImageAtIndex:(unsigned)index {
	if (index == m_selectedImage)
		return FALSE;
	
	if (index >= [m_list count])
		return FALSE;
	
	m_selectedImage = index;
	NEW_IMAGE_NOTIFICATION();
	return TRUE;
}

- (void)selectSpecifiedImage:(NSNotification*)not {
	FFImageFile* file = (FFImageFile*)[not object];
	if (file != nil) {
		int	idx = [m_list indexOfObject:file];
		if (idx != NSNotFound) {
			m_selectedImage = idx;
			NEW_IMAGE_NOTIFICATION();
		}
	}
}

- (BOOL)removeCurrentImage {
	if (m_selectedImage == -1)
		return FALSE;
	
	if ([m_list count] == 1)
		[self clear];
	else {
		[m_list removeObjectAtIndex:m_selectedImage];
	
		// Was the last image
		if (m_selectedImage == [m_list count])
				m_selectedImage--;
		
		NEW_IMAGE_NOTIFICATION();
		IMAGELIST_CHANGED_NOTIFICATION();
	}
	return TRUE;
}

- (void)removeCurrentImage:(NSNotification*)notification {
	(void)[self removeCurrentImage];
}

#pragma mark -
#pragma mark Slideshow

- (BOOL)slideshowIsRunning  { return (BOOL)(m_slideshowTimer != nil); }

- (void)_slideshowImageSelect:(NSTimer*)timer {
	BOOL		endless	= [[timer userInfo] boolValue];
	unsigned	imgLeft	= [self numberOfImages] - [self selectedImageIndex] - 1;
	
	if ((imgLeft == 0) && endless)
		[self selectFirstImage];
	else {
		[self selectNextImage];
		if ((imgLeft == 1) && !endless)
			[self stopSlideshow];
	}
}

- (void)startSlideshowWithInterval:(NSTimeInterval)interval endless:(BOOL)endless {
	if (([self selectedImageIndex]+1 < [self numberOfImages]) 
		|| (endless && ([self numberOfImages] > 1))) {
		m_slideshowTimer = [NSTimer scheduledTimerWithTimeInterval:[m_prefs slideshowTime]
															target:self
														  selector:@selector(_slideshowImageSelect:)
														  userInfo:[NSNumber numberWithBool:endless]
														   repeats:TRUE];
		SLIDESHOW_START_NOTIFICATION();
	}
}

- (void)stopSlideshow {
	if (m_slideshowTimer != nil) {
		[m_slideshowTimer invalidate];
		m_slideshowTimer = nil;
		
		SLIDESHOW_STOP_NOTIFICATION();
	}
}

- (void)stopSlideshow:(NSNotification*)notification {
	[self stopSlideshow];
}

#pragma mark -
#pragma mark Sorting & rearranging

- (BOOL)isUserSortOrder {
	return ((m_sortOrder == IMGLIST_USER_ASC) || (m_sortOrder == IMGLIST_USER_DESC));
}

- (void)_reverseList { // May be a category method also
	int i = 0, j = [m_list count]-1;
	
	// Fix the selected image index
	m_selectedImage = j - m_selectedImage;
	
	// Exchange upper and lower item
	while (i < j) {
		[m_list exchangeObjectAtIndex:i withObjectAtIndex:j];
		i++;
		j--;
	}
}

- (void)_sortAndFixSelectedImageUsingComparator:(SEL)compr {
	FFImageFile* selFile = [m_list objectAtIndex:m_selectedImage];
	[m_list sortUsingSelector:compr];
	m_selectedImage = [m_list indexOfObjectIdenticalTo:selFile];
}

- (BOOL)sort:(FFImageListSortOrder)order {
	FFImageListSortOrder oldOrder = [m_prefs imageListSortOrder];
	
	// Set new sort order
	m_sortOrder = order;
	[m_prefs setImageListSortOrder:order];
	
	// List is empty
	if (m_selectedImage == -1)
		return FALSE;

	// Sort or simply reverse the list
	if (((order % 2 == 1) && (order - 1 == oldOrder)) || // desc->asc
		((order % 2 == 0) && (order + 1 == oldOrder)))	 // asc->desc
		[self _reverseList];
	else
		[self _sortAndFixSelectedImageUsingComparator:SortSelectors[order]];
	
	IMAGELIST_REARRANGED_NOTIFCATION();
	
	return TRUE;
}

- (void)moveImagesWithIndexes:(NSArray*)imgIndexes toIndex:(int)index {
	// imgIndexes is already sorted	
	FFImageFile*	selFile;
	NSMutableArray*	tmp;
	NSEnumerator*	en;
	NSNumber*		imgIdx;
	FFImageFile*	imgFile;
	unsigned		fidx;

	// Remember the selected image
	selFile = [m_list objectAtIndex:m_selectedImage];
	
	// Store in temporary array
	tmp	= [[NSMutableArray alloc] initWithCapacity:[imgIndexes count]];
	en	= [imgIndexes objectEnumerator];
	while (imgIdx = [en nextObject]) {
		fidx = [imgIdx unsignedIntValue];
		[tmp addObject:[m_list objectAtIndex:fidx]];
		if (fidx < index)
			index--;
	}
	
	// Remove from the list
	[m_list removeObjectsInArray:tmp];
	
	// Insert
	en	= [tmp objectEnumerator];
	while (imgFile = [en nextObject]) {
		[m_list insertObject:imgFile atIndex:index];
		index++;
	}
			
	// Release temporary array
	[tmp release];
			
	// Fix the file-indexes
	en		= [m_list objectEnumerator];
	fidx	= (m_sortOrder == IMGLIST_USER_ASC) ? 0 : [m_list count]-1;
	while (imgFile = [en nextObject]) {
		[imgFile setUserSortIndex:fidx];
		fidx++;
	}

	// Fix the selected image position
	m_selectedImage = [m_list indexOfObjectIdenticalTo:selFile];
	
	IMAGELIST_REARRANGED_NOTIFCATION();	
}

- (BOOL)shuffle {
	FFImageFile*	selFile;
	int				n, loop;
	BOOL*			used;
	unsigned		ei, ei2;
	
	if (m_selectedImage == -1)
		return FALSE;
	
	selFile = [m_list objectAtIndex:m_selectedImage];
	n		= [m_list count];
	loop	= n / 2;
	used	= (BOOL*)malloc(n*sizeof(BOOL));
	
	// No index used yet
	memset(used, FALSE, n);
	
	// Exchange them...
	for (; loop > 0; loop--) {
		#define RAND_INDEX(VAR) \
			do \
				VAR = rand() % n; \
			while (used[VAR]); \
			used[VAR] = TRUE
				
			RAND_INDEX(ei);
			RAND_INDEX(ei2);
			
			[m_list exchangeObjectAtIndex:ei withObjectAtIndex:ei2];
	}
	free(used);
	
	// We are done
	m_selectedImage = [m_list indexOfObjectIdenticalTo:selFile];
	IMAGELIST_REARRANGED_NOTIFCATION();
	return TRUE;
}

#pragma mark -
#pragma mark Filtering

- (BOOL)filter:(NSString*)text {
	NSEnumerator*	en	= [m_list objectEnumerator];
	FFImageFile*	ifile;
	
	if (m_filterString != nil)
		[m_filterString release];
	m_filterString = ([text length] > 0) ? [text retain] : nil;
	
	// Reset
	if (m_filterString == nil) {
		while (ifile = [en nextObject])
			[ifile setFiltered:FALSE];
		return FALSE;
	}
	
	// Filtering
	while (ifile = [en nextObject])
		FILTER_IMAGEFILE(ifile);

	return TRUE;
}

#pragma mark -
#pragma mark Misc

- (void)revealInFinder {
	if (m_selectedImage == -1)
		return;

	FFImageFile*	f = [m_list objectAtIndex:m_selectedImage];
	NSString*		p;
	if ([f fileType] == IMG_NORMAL_FILE)
		p = [f fullPath];
	else
		p = [[f archive] filePath];

	(void)[[NSWorkspace sharedWorkspace] selectFile:p inFileViewerRootedAtPath:@""];
}

- (BOOL)removeAndTrashCurrentImage {
	if (m_selectedImage == -1)
		return FALSE;
	
	// Is it allowed ?
	if (![m_prefs trashDelete]) {
		[FFInfoPanel runInFullscreen:[m_opts fullscreen] withTitle:FFTR(@"Move to Trash is forbidden")
							 message:FFTR(@"Enable it through the preferences.")
						 andOkButton:FFTR(@"OK")];
		return FALSE;
	}
	
	// Move to trash
	FFImageFile* f = [m_list objectAtIndex:m_selectedImage];
	if ([f fileType] == IMG_PART_OF_ARCHIVE) {
		[FFInfoPanel runInFullscreen:[m_opts fullscreen] withTitle:FFTR(@"Archive file")
							 message:FFTR(@"Its impossible to move an  archive file to trash.")
						 andOkButton:FFTR(@"OK")];
		return FALSE;
		
	}
	
	int		tag;
	BOOL	ret;
	ret = [[NSWorkspace sharedWorkspace] 
		performFileOperation:NSWorkspaceRecycleOperation
					  source:[[f fullPath] stringByDeletingLastPathComponent] destination:@""
					   files:[NSArray arrayWithObject:[[f fullPath] lastPathComponent]] 
						 tag:&tag];
	
	if (ret)
		return [self removeCurrentImage]; // Always true

	[FFInfoPanel runInFullscreen:[m_opts fullscreen] withTitle:FFTR(@"Failed to move to Trash")
						 message:FFTR(@"Maybe the wrong access rights?")
					 andOkButton:FFTR(@"OK")];
	
	return FALSE;
}

- (BOOL)copyImageAtIndex:(unsigned)index toDirectory:(NSString*)directory {
	FFImageFile*	imf;
	NSString*		fp, *newPath;
	NSFileManager*	fm;
	int				ret;
	
	if (index >= [m_list count])
		return FALSE;
	
	imf		= [m_list objectAtIndex:index];
	fp		= [imf fullPath];
	newPath	= [directory stringByAppendingPathComponent:[fp lastPathComponent]];
	
	// Already a file w/ the same name ?
	fm	= [NSFileManager defaultManager];
	if ([fm fileExistsAtPath:newPath]) {
		ret = NSRunInformationalAlertPanel(FFTR(@"File exists"),
										   FFTRC(@"There is already a file '%@' - overwrite it?", @"1=Filename"),
										   FFTR(@"No"), FFTR(@"Yes"), nil,
										   newPath);
		// Continue, even when the user choose not to overwrite the file
		if (ret == NSOKButton) // No
			return TRUE;
		
		[fm removeFileAtPath:newPath handler:nil]; // remove the existing file
	}
	
	// Extract
	if ([imf fileType] == IMG_PART_OF_ARCHIVE) {
		NS_DURING
			[[imf archive] extractFile:fp toFilePath:newPath];
		NS_HANDLER
			ret = NSRunInformationalAlertPanel(FFTR(@"Archive trouble"),
											   FFTRC(@"Filename: %@\nMessage: \"%@\"", @"0=Path, 1=Errormessage"),
											   FFTR(@"Continue"), FFTR(@"Abort"), nil,
											   [imf displayName], [localException reason]);
			if (ret == NSCancelButton) // Abort
				return FALSE;
		NS_ENDHANDLER
	
	// Copy
	} else {
		if (![fm copyPath:fp toPath:newPath handler:nil]) {
			ret = NSRunInformationalAlertPanel(FFTR(@"Copying failed"),
											   FFTRC(@"Failed to copy '%@' to '%@'", @"1=Filename, 2=Path"),
											   FFTR(@"Continue"), FFTR(@"Abort"), nil,
											   fp, newPath);
			if (ret == NSCancelButton) // Abort
				return FALSE;
		}
		
	}
	return TRUE; // continue
}

+ (NSString*)fileExtension {
	return ImageListFileExtension;
}

+ (BOOL)isArchiveExtension:(NSString*)ext {
	if ([[KnownFileTypes objectForKey:TypeKeyZIP] containsCaseInsensitiveString:ext] ||
		[[KnownFileTypes objectForKey:TypeKeyRAR] containsCaseInsensitiveString:ext] ||
		[[KnownFileTypes objectForKey:TypeKeyPDF] containsCaseInsensitiveString:ext])
		return TRUE;
	return FALSE;
}

+ (BOOL)isImageExtension:(NSString*)ext {
	return [[KnownFileTypes objectForKey:TypeKeyImages] containsCaseInsensitiveString:ext];
}

- (NSString*)_getContainerOfImageAtIndex:(int)index {
	FFImageFile* imf = [m_list objectAtIndex:index];
	
	if ([imf fileType] == IMG_NORMAL_FILE)
		return [[imf fullPath] stringByDeletingLastPathComponent];

	return [[imf archive] filePath];
}

// 10.4+
static inline BOOL _isSetIconDefined() {
	return [NSWorkspace instancesRespondToSelector:@selector(setIcon:forFile:options:)];
}

- (BOOL)isWritableContainerOfImageAtIndex:(int)index {
	if ((index == -1) || !_isSetIconDefined())
		return FALSE;
	
	return [m_fm isWritableFileAtPath:[self _getContainerOfImageAtIndex:index]];
}

- (BOOL)useImageAtIndexAsFinderIcon:(int)index {
	NSString*	cname;
	GLuint		owd, oht, od;
	NSImage*	timg;
	
	if ((index == -1) || !_isSetIconDefined())
		return FALSE;

	// Writable?
	cname = [self _getContainerOfImageAtIndex:index];
	if (![m_fm isWritableFileAtPath:cname])
		return FALSE;
	
	// Load the downsized image
	if (!loadThumbnail([self fullPathAtIndex:index],
					   128, 128,
					   &timg,
					   &owd, &oht, &od))
		return FALSE;

	
	// Set the icon
	[[NSWorkspace sharedWorkspace] setIcon:timg
								   forFile:cname
								   options:NSExclude10_4ElementsIconCreationOption];		
	
	// Everything worked out as we planned.. sweet
	[timg release];

	return TRUE;
}

#pragma mark -
#pragma mark FFArchivehelper methods

- (NSString*)getPasswordForFilename:(NSString*)filename {
	return [FFInfoPanel runWithInputInFullscreen:[m_opts fullscreen]
									   withTitle:FFTR(@"Password protected archive")
										 message:[NSString stringWithFormat:FFTR(@"Please enter the password for '%@'"), filename]
										okButton:FFTR(@"OK") andCancelButton:FFTR(@"Cancel")];
}

@end
