package Onis::Parser;

use strict;
use warnings;

use Exporter;
use Onis::Config qw#get_config#;
use Onis::Data::Core qw#nick_rename store#;
use Onis::Parser::Persistent qw/set_absolute_time get_absolute_time add_relative_time get_state %MONTHNAMES @MONTHNUMS/;

@Onis::Parser::EXPORT_OK = qw/parse last_date/;
@Onis::Parser::ISA = ('Exporter');

our $WORD_LENGTH = 5;

if (get_config ('min_word_length'))
{
	my $tmp = get_config ('min_word_length');
	$tmp =~ s/\D//g;
	$WORD_LENGTH = $tmp if ($tmp);
}

my $VERSION = '$Id: Eggdrop.pm,v 1.6 2003/12/16 09:22:28 octo Exp $';
print STDERR $/, __FILE__, ": $VERSION" if ($::DEBUG);

return (1);

sub parse
{
	my $line = shift;
	my $state;

	if ($line =~ m/^\[(\d\d):(\d\d)\] --- \w\w\w (\w\w\w)  ?(\d\d?) (\d{4})/)
	{
		if (!defined ($MONTHNAMES{$3})) { return (4); }
		set_absolute_time ($5, $MONTHNAMES{$3}, $4, $1, $2, 0);
	}
	elsif ($line =~ m/^\[(\d\d):(\d\d)\] /)
	{
		add_relative_time ($1, $2);
	}

	$state = get_state ();
	if ($state != 1)
	{
		return ($state);
	}
	
	if ($line =~ /^\[(\d\d):(\d\d)\] <([^>]+)> (.+)/)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			nick	=> $3,
			text	=> $4,
			type	=> 'TEXT'
		};
		
		my @words = grep { length ($_) >= $WORD_LENGTH } (split (m/\W+/, $4));
		$data->{'words'} = \@words;
		
		store ($data);
	}

	elsif ($line =~ /^\[(\d\d):(\d\d)\] Action: (\S+) (.+)/)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			nick	=> $3,
			text	=> "* $3 $4",
			type	=> 'ACTION'
		};
		
		my @words = grep { length ($_) >= $WORD_LENGTH } (split (m/\W+/, $4));
		$data->{'words'} = \@words;
		
		store ($data);
	}

	elsif ($line =~ /^\[(\d\d):(\d\d)\] (\S+) \(([^\)]+)\) joined ([#!+&]\S+)\./)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			nick	=> $3,
			host	=> $4,
			channel	=> $5,
			type	=> 'JOIN'
		};
		store ($data);
	}

	elsif ($line =~ /^\[(\d\d):(\d\d)\] ([#!+&]\S+): mode change '([-+][^']+)' by ([^!]+)!(.+)/)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			channel	=> $3,
			mode	=> $4,
			nick	=> $5,
			host	=> $6,
			type	=> 'MODE'
		};
		store ($data);
	}
	
	elsif ($line =~ /^\[\d\d:\d\d\] Nick change: (\S+) -> (\S+)$/)
	{
		nick_rename ($1, $2);
	}

	elsif ($line =~ /^\[(\d\d):(\d\d)\] Topic changed on ([#!+&]\S+) by ([^!]+)!(\S+): (.+)/)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			channel	=> $3,
			nick	=> $4,
			host	=> $5,
			text	=> $6,
			type	=> 'TOPIC'
		};
		store ($data);
	}

	elsif ($line =~ /^\[(\d\d):(\d\d)\] (\S+) \(([^\)]+)\) left irc: (.*)/)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			nick	=> $3,
			host	=> $4,
			text	=> $5,
			type	=> 'QUIT'
		};
		store ($data);
	}

	elsif ($line =~ /^\[(\d\d):(\d\d)\] (\S+) \(([^\)]+)\) left ([#!+&]\S+) \((.*)\)/)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			nick	=> $3,
			host	=> $4,
			channel	=> $5,
			text	=> $6,
			type	=> 'LEAVE'
		};
		store ($data);
	}

	elsif ($line =~ /^\[(\d\d):(\d\d)\] (\S+) kicked from ([#!+&]\S+) by ([^:]+): (.+)/)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			channel	=> $4,
			nick_received	=> $3,
			nick	=> $5,
			text	=> $6,
			type	=> 'KICK'
		};
		store ($data);
	}

	elsif ($line =~ m/^\[(\d\d):(\d\d)\] (\S+) \(([^\)]+)\) got netsplit/)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			nick	=> $3,
			host	=> $4,
			type	=> 'NETSPLIT_LEAVE'
		};
		store ($data);
	}

	elsif ($line =~ m/^\[(\d\d):(\d\d)\] (\S+) \(([^\)]+)\) returned to ([#!+&]\S+)\./)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			nick	=> $3,
			host	=> $4,
			channel	=> $5,
			type	=> 'NETSPLIT_JOIN'
		};
		store ($data);
	}

	elsif ($line =~ m/^\[(\d\d):(\d\d)\] (\S+) \(([^\)]+)\) got lost in the net-split./)
	{
		my $data =
		{
			hour	=> $1,
			minute	=> $2,
			nick	=> $3,
			host	=> $4,
			type	=> 'NETSPLIT_QUIT'
		};
		store ($data);
	}

	else
	{
		print STDERR $/, __FILE__, ": Not parsed: ``$line''" if ($::DEBUG & 0x20);
		return (2);
	}

	return (1);
}

sub last_date
{
	# $line =~ m/^\[(\d\d):(\d\d)\] --- \w\w\w (\w\w\w)  ?(\d\d?) (\d{4})/
	my $time = get_absolute_time ();
	my ($min, $hour, $day, $month_num, $year) = (localtime ($time))[1 .. 5];
	my $month_name = $MONTHNUMS[$month_num];

	$year += 1900;

	my $retval = sprintf ("[%02u:%02u] --- TIM %s %02u %04u\n", $hour, $min, $month_name, $day, $year);

	return ($retval);
}
