/*
 * unity-webapps-dpkg-url-index.c
 * Copyright (C) Canonical LTD 2011
 *
 * Author: Robert Carr <racarr@canonical.com>
 * 
 unity-webapps is free software: you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published
 * by the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 * 
 * unity-webapps is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 * See the GNU Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.";
 */

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <glib/gstdio.h>
#include <gio/gio.h>

#include "unity-webapps-dpkg-url-index.h"
#include "unity-webapps-url-db.h"
#include "unity-webapps-apt-package-mechanism.h"

#include "unity-webapps-dpkg-available-application.h"

#include "../unity-webapps-debug.h"

struct _UnityWebappsDpkgUrlIndexPrivate {
  UnityWebappsUrlDB *url_db;
  UnityWebappsPackageMechanism *package_mechanism;
  
  // Package name -> Available Application. Available applications are owned by this table.
  GHashTable *available_applications_by_name;
};

enum {
  PROP_0,
  PROP_PACKAGE_MECHANISM
};

G_DEFINE_TYPE(UnityWebappsDpkgUrlIndex, unity_webapps_dpkg_url_index, UNITY_WEBAPPS_TYPE_URL_INDEX)

#define UNITY_WEBAPPS_DPKG_URL_INDEX_GET_PRIVATE(object) (G_TYPE_INSTANCE_GET_PRIVATE ((object), UNITY_WEBAPPS_TYPE_DPKG_URL_INDEX, UnityWebappsDpkgUrlIndexPrivate))

static UnityWebappsAvailableApplication *
dpkg_url_index_get_application_record (UnityWebappsDpkgUrlIndex *self,
				       const gchar *name,
				       const gchar *application_name,
				       const gchar *application_domain)
{
  UnityWebappsAvailableApplication *app;
  
  app = g_hash_table_lookup (self->priv->available_applications_by_name, name);
  if (app != NULL)
    {
      return app;
    }
  
  app = unity_webapps_dpkg_available_application_new (name, application_name,
						      application_domain,
						      self->priv->package_mechanism);
  g_hash_table_insert (self->priv->available_applications_by_name, g_strdup (name), app);
  
  return app;
}

static GList *
dpkg_url_index_lookup_url (UnityWebappsUrlIndex *url_index,
			     const gchar *url)
{
  UnityWebappsDpkgUrlIndex *self;
  GList *applications, *records, *w;
  
  self = UNITY_WEBAPPS_DPKG_URL_INDEX (url_index);

  if (self->priv->url_db == NULL)
    {
      return NULL;
    }
  
  unity_webapps_url_db_lookup_urls (self->priv->url_db, url, &records);
  
  applications = NULL;
  for (w = records; w != NULL; w = w->next)
    {
      UnityWebappsUrlDBRecord *record;
      UnityWebappsAvailableApplication *app;
      
      record = (UnityWebappsUrlDBRecord *)w->data;
      
      app = dpkg_url_index_get_application_record (self, record->package_name, record->application_name, record->domain);
      
      g_assert (app);
      applications = g_list_append (applications, app);
    }
  
  unity_webapps_url_db_record_list_free (records);

  return applications;
}

static void
unity_webapps_dpkg_url_index_get_property (GObject *object,
					   guint prop_id,
					   GValue *value,
					   GParamSpec *pspec)
{
  UnityWebappsDpkgUrlIndex *self;
  
  self = UNITY_WEBAPPS_DPKG_URL_INDEX (object);
  
  switch (prop_id)
    {
    case PROP_PACKAGE_MECHANISM:
      g_value_set_boxed (value, self->priv->package_mechanism);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}

static void
unity_webapps_dpkg_url_index_set_property (GObject *object,
						 guint prop_id,
						 const GValue *value,
						 GParamSpec *pspec)
{
  UnityWebappsDpkgUrlIndex *self;;
  
  self = UNITY_WEBAPPS_DPKG_URL_INDEX (object);
  
  switch (prop_id)
    {
    case PROP_PACKAGE_MECHANISM:
      g_assert (self->priv->package_mechanism == NULL);
      self->priv->package_mechanism = (UnityWebappsPackageMechanism *)g_value_get_object (value);
      break;
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID (object, prop_id, pspec);
    }
}



static void
unity_webapps_dpkg_url_index_finalize (GObject *object)
{
  UnityWebappsDpkgUrlIndex *self;
  
  self = UNITY_WEBAPPS_DPKG_URL_INDEX (object);

  if (self->priv->url_db)
    {
      unity_webapps_url_db_free (self->priv->url_db);
    }
  if (self->priv->available_applications_by_name)
    {
      g_hash_table_destroy (self->priv->available_applications_by_name);
    }
  
  g_object_unref (G_OBJECT (self->priv->package_mechanism));
}

static void
unity_webapps_dpkg_url_index_class_init (UnityWebappsDpkgUrlIndexClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);
  UnityWebappsUrlIndexClass *index_class = UNITY_WEBAPPS_URL_INDEX_CLASS (klass);
  
  object_class->finalize = unity_webapps_dpkg_url_index_finalize;
  object_class->get_property = unity_webapps_dpkg_url_index_get_property;
  object_class->set_property = unity_webapps_dpkg_url_index_set_property;

  index_class->lookup_url = dpkg_url_index_lookup_url;
  
  g_type_class_add_private (object_class, sizeof(UnityWebappsDpkgUrlIndexPrivate));
  
  g_object_class_install_property (object_class, PROP_PACKAGE_MECHANISM,
				   g_param_spec_object("package-mechanism",
						       "Package Mechanism",
						       "The Mechanism to use for external package functionality",
						       UNITY_WEBAPPS_TYPE_PACKAGE_MECHANISM,
						       G_PARAM_READWRITE | G_PARAM_CONSTRUCT_ONLY | G_PARAM_STATIC_STRINGS));
 
}

static void
unity_webapps_dpkg_url_index_init (UnityWebappsDpkgUrlIndex *self)
{
  self->priv = UNITY_WEBAPPS_DPKG_URL_INDEX_GET_PRIVATE (self);
  
  self->priv->url_db = unity_webapps_url_db_open_default (TRUE);
  self->priv->available_applications_by_name = g_hash_table_new_full (g_str_hash, g_str_equal, g_free, g_object_unref);
  
  self->priv->package_mechanism = NULL;
}



UnityWebappsUrlIndex *
unity_webapps_dpkg_url_index_new (UnityWebappsPackageMechanism *package_mechanism)
{
  return UNITY_WEBAPPS_URL_INDEX (g_object_new (UNITY_WEBAPPS_TYPE_DPKG_URL_INDEX, 
						"package-mechanism", package_mechanism, 
						NULL));
}

UnityWebappsUrlIndex *
unity_webapps_dpkg_url_index_new_default ()
{
  UnityWebappsPackageMechanism *mechanism;
  
  mechanism = unity_webapps_apt_package_mechanism_new ();
  return unity_webapps_dpkg_url_index_new (mechanism);
}
