(ns puppetlabs.puppetdb.import
  "Import utility

   This is a command-line tool for importing data into PuppetDB. It expects
   as input a tarball generated by the PuppetDB `export` command-line tool."
  (:import [org.apache.commons.compress.archivers.tar TarArchiveEntry]
           [puppetlabs.puppetdb.archive TarGzReader])
  (:require [clojure.java.io :as io]
            [clojure.tools.logging :as log]
            [puppetlabs.puppetdb.archive :as archive]
            [puppetlabs.puppetdb.reports :as reports]
            [puppetlabs.puppetdb.utils :as utils]
            [puppetlabs.puppetdb.cheshire :as json]
            [puppetlabs.puppetdb.schema :refer [defn-validated]]))

(defn file-pattern
  [entity]
  (re-pattern (str "^" (.getPath (io/file utils/export-root-dir entity ".*\\.json")) "$")))

(defn-validated process-tar-entry
  "Determine the type of an entry from the exported archive, and process it
  accordingly."
  [command-fn
   tar-reader :- TarGzReader
   tar-entry :- TarArchiveEntry
   command-versions]
  (let [path (.getName tar-entry)]
    (condp re-find path
      (file-pattern "catalogs")
      (do (log/infof "Importing catalog from archive entry '%s'" path)
          ;; NOTE: these submissions are async and we have no guarantee that they
          ;;   will succeed. We might want to add something at the end of the import
          ;;   that polls puppetdb until the command queue is empty, then does a
          ;;   query to the /nodes endpoint and shows the set difference between
          ;;   the list of nodes that we submitted and the output of that query
          (command-fn :replace-catalog
                      (:replace_catalog command-versions)
                      (utils/read-json-content tar-reader)))
      (file-pattern "reports")
      (do (log/infof "Importing report from archive entry '%s'" path)
          (command-fn :store-report
                      (:store_report command-versions)
                      (utils/read-json-content tar-reader true)))
      (file-pattern "facts")
      (do (log/infof "Importing facts from archive entry '%s'" path)
          (command-fn :replace-facts
                      (:replace_facts command-versions)
                      (utils/read-json-content tar-reader)))
      nil)))

(defn import!
  [infile command-versions command-fn]
  (with-open [tar-reader (archive/tarball-reader infile)]
    (doseq [tar-entry (archive/all-entries tar-reader)]
      (process-tar-entry command-fn tar-reader tar-entry command-versions))))
