/* tag: properties window (implementation)
 *
 * Copyright (C) 2000  Andrew Chatham
 * Copyright (C) 2003  Charles Duffy
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 */

#include <string.h>

#include "properties.h"
#include "ticket_applet.h"

static GtkWidget *selected_child;

static void
properties_help_cb (GnomePropertyBox *widget,
                    int        id,
                    TicketApplet *ad)
{
	GError *error = NULL;

	egg_help_display_desktop_on_screen (
		        NULL, "ticket-applet-2-manual",
			"ticket-applet-2-manual", "ticket-settings",
		        gtk_widget_get_screen (GTK_WIDGET(ad)),
		        &error);

	if (error) {
		GtkWidget *dialog;
		dialog = gtk_message_dialog_new (GTK_WINDOW (widget),
		                                 GTK_DIALOG_DESTROY_WITH_PARENT,
		                                 GTK_MESSAGE_ERROR,
		                                 GTK_BUTTONS_CLOSE,
		                                  _("There was an error displaying help: %s"),
		                                 error->message);

		g_signal_connect (G_OBJECT (dialog), "response",
		                  G_CALLBACK (gtk_widget_destroy),
		                  NULL);

		gtk_window_set_resizable (GTK_WINDOW (dialog), FALSE);
		gtk_window_set_screen (GTK_WINDOW (dialog),
		                       gtk_widget_get_screen (GTK_WIDGET(ad)));
		gtk_widget_show (dialog);
		g_error_free (error);
	}
}


static void
set_values(TicketApplet *ad)
{
  gchar keybase[255], key[255];
  gchar *val;
  gint i;
  GList *head = gtk_container_children(GTK_CONTAINER(ad->cmd_list));
  GList *cur = head;
  GList *tmp;
  GtkListItem *li;
  GError *error;

  // TODO: actually check error status
  panel_applet_gconf_set_string(PANEL_APPLET(ad), USER_CONFIG_KEY,
		  ad->user, &error);
  panel_applet_gconf_set_string(PANEL_APPLET(ad), REALM_CONFIG_KEY,
		  ad->realm, &error);
  panel_applet_gconf_set_int(PANEL_APPLET(ad), REFRESH_CONFIG_KEY,
		  ad->refresh, &error);
  
  //don't want extra commands lingering
  g_snprintf(keybase, 255, panel_applet_gconf_get_full_key(PANEL_APPLET(ad), COMMANDS_CONFIG_KEY));
  gnome_config_clean_section(keybase); 
  for (i = 0; cur != NULL; cur = g_list_next(cur), i++) {
    g_snprintf(key, 255, "%s/cmd%d", keybase, i);

    li = GTK_LIST_ITEM(cur->data);
    tmp = gtk_container_children(GTK_CONTAINER(li));
    gtk_label_get(GTK_LABEL(tmp->data), &val);
    g_list_free(tmp);

    /* We want to change this list to a list of strings so we can use 
       it as ad->cmds when we recreat the command list in the future.
    */
    cur->data = g_strdup(val); 

    gnome_config_set_string(key, val);
  }
  change_timeout(ad);
  for (cur = ad->cmds; cur != NULL; cur = g_list_next(cur)) {
    g_free(cur->data);
  }
  g_list_free(ad->cmds);
  ad->cmds = head;
  gnome_config_sync();
}

static gint 
destroy_cb(GtkWidget *widget, gpointer data)
{
  TicketApplet *ad = data;
  ad->p_box = NULL;
  return FALSE;
}

static void
apply_cb (GtkWidget *w, void *nodata, gpointer data)
{
  TicketApplet *ad = data;

  g_free(ad->user);
  g_free(ad->realm);

  ad->user = gtk_editable_get_chars(GTK_EDITABLE(ad->user_entry),
				    0, 100);
  ad->realm = gtk_editable_get_chars(GTK_EDITABLE(ad->realm_entry),
				     0, 100);

  ad->refresh = ad->p_refresh * 1000 * 60;

  set_values(ad);
  /* TODO: has equivalent? needs one? applet_widget_sync_config(APPLET_WIDGET(ad->applet)); */
}

static void
refresh_change_cb(GtkObject *adj, gpointer data)
{
  TicketApplet *ad = data;
  if (GTK_IS_ADJUSTMENT(adj)) {
    ad->p_refresh = (guint)GTK_ADJUSTMENT(adj)->value;
  } else
    ad->p_refresh = (guint)GTK_ADJUSTMENT(GTK_SPIN_BUTTON(adj)->adjustment)->value;
  gnome_property_box_changed(GNOME_PROPERTY_BOX(ad->p_box));
}

static void
entry_change_cb(GtkObject *obj, gpointer data)
{
  TicketApplet *ad = data;

  gnome_property_box_changed(GNOME_PROPERTY_BOX(ad->p_box));
}


/* Initialize all of the widgets for the "settings" page" */
static void
add_settings(TicketApplet *ad, GtkWidget *table)
{
  GtkWidget *spin;
  GtkWidget *spin_label;
  GtkWidget *realm_entry;
  GtkWidget *realm_label;
  GtkWidget *user_entry;
  GtkWidget *user_label;

  GtkObject *adj;
  

  user_label = gtk_label_new("Username: ");
  realm_label = gtk_label_new("Realm: ");
  spin_label = gtk_label_new("Refresh: ");

  user_entry = gtk_entry_new();
  ad->user_entry = user_entry;
  if (ad->user)
    gtk_entry_set_text(GTK_ENTRY(user_entry), ad->user);

  realm_entry = gtk_entry_new();
  ad->realm_entry = realm_entry;
  if (ad->realm)
    gtk_entry_set_text(GTK_ENTRY(realm_entry), ad->realm);


  adj = gtk_adjustment_new( ad->p_refresh, 1.0, 100.0 , 1, 1, 10);
  spin = gtk_spin_button_new( GTK_ADJUSTMENT(adj), 1, 0 );

  gtk_signal_connect( GTK_OBJECT(adj),"value_changed",
		      GTK_SIGNAL_FUNC(refresh_change_cb), ad);

  gtk_signal_connect( GTK_OBJECT(user_entry), "changed", 
		      GTK_SIGNAL_FUNC(entry_change_cb),
		      ad);
  gtk_signal_connect( GTK_OBJECT(realm_entry), "changed", 
		      GTK_SIGNAL_FUNC(entry_change_cb),
		      ad);

  gtk_table_attach_defaults(GTK_TABLE(table), user_label, 0, 1, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), realm_label, 0, 1, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), spin_label, 0, 1, 2, 3);
  gtk_table_attach_defaults(GTK_TABLE(table), user_entry, 1, 2, 0, 1);
  gtk_table_attach_defaults(GTK_TABLE(table), realm_entry, 1, 2, 1, 2);
  gtk_table_attach_defaults(GTK_TABLE(table), spin, 1, 2, 2, 3);

  gtk_signal_connect( GTK_OBJECT(ad->p_box),"apply", 
		      GTK_SIGNAL_FUNC(apply_cb), ad);
  gtk_signal_connect( GTK_OBJECT(ad->p_box),"destroy", 
		      GTK_SIGNAL_FUNC(destroy_cb), ad );
  g_signal_connect( G_OBJECT(ad->p_box),"help",
		    G_CALLBACK(properties_help_cb), ad);

}

static void
selection_cb(GtkWidget *list, GtkWidget *child, gpointer data)
{
  selected_child = child;
}

static void
unselection_cb(GtkWidget *list, GtkWidget *child)
{
  selected_child = NULL;
}

static void
remove_cb(GtkWidget *w, gpointer data)
{
  TicketApplet *ad = data;
  gint pos;

  if (selected_child == NULL) return;

  pos = gtk_list_child_position(GTK_LIST(ad->cmd_list), 
				selected_child);
  gtk_list_clear_items(GTK_LIST(ad->cmd_list), pos, pos+1);
  gnome_property_box_changed(GNOME_PROPERTY_BOX(ad->p_box));
}

static void
add_cmd_cb(GtkWidget *entry, gpointer data)
{
  GtkWidget *li;
  TicketApplet *ad = data;
  gchar *cmd;
  
  cmd = gtk_editable_get_chars(GTK_EDITABLE(entry), 0, 100);

  li = gtk_list_item_new_with_label(cmd);
  gtk_widget_show(li);

  gtk_container_add(GTK_CONTAINER(ad->cmd_list), li);
  gtk_entry_set_text(GTK_ENTRY(entry), "");
  gnome_property_box_changed(GNOME_PROPERTY_BOX(ad->p_box));
}


/* Initialize all the widgets for the "Commands" page */
static void
add_commands(TicketApplet *ad, GtkWidget *vbox)
{
  GtkWidget *sw;
  GtkWidget *list;
  GtkWidget *hbox;
  GtkWidget *li;
  GtkWidget *label;
  GtkWidget *cmd_entry;
  GtkWidget *remove_button;
  GList *cur;

  sw = gtk_scrolled_window_new(NULL, NULL);

  list = gtk_list_new();
  gtk_list_set_selection_mode(GTK_LIST(list), GTK_SELECTION_SINGLE);

  ad->cmd_list = list;
  selected_child = NULL;
  gtk_scrolled_window_add_with_viewport(GTK_SCROLLED_WINDOW(sw), list);

  for (cur = ad->cmds; cur != NULL; cur = g_list_next(cur)) {
    li = gtk_list_item_new_with_label(cur->data);
    gtk_container_add(GTK_CONTAINER(list), li);
  }

  hbox = gtk_hbox_new(FALSE, 5);

  remove_button = gtk_button_new_with_label("Remove");
  gtk_box_pack_start(GTK_BOX(hbox), remove_button, FALSE, FALSE, 5);

  label = gtk_label_new("Command: ");
  gtk_box_pack_start(GTK_BOX(hbox), label, FALSE, FALSE, 5);

  cmd_entry = gtk_entry_new();
  gtk_box_pack_start(GTK_BOX(hbox), cmd_entry, FALSE, FALSE, 5);

  gtk_signal_connect(GTK_OBJECT(cmd_entry), "activate", 
		     GTK_SIGNAL_FUNC(add_cmd_cb), ad);
  gtk_signal_connect(GTK_OBJECT(list), "select-child",
		     GTK_SIGNAL_FUNC(selection_cb), ad);
  gtk_signal_connect(GTK_OBJECT(list), "unselect-child",
		     GTK_SIGNAL_FUNC(unselection_cb), ad);
  gtk_signal_connect(GTK_OBJECT(remove_button), "clicked",
		     GTK_SIGNAL_FUNC(remove_cb), ad);

  gtk_box_pack_start(GTK_BOX(vbox), sw, FALSE, FALSE, 5);
  gtk_box_pack_start(GTK_BOX(vbox), hbox, FALSE, FALSE, 5);
}

void
properties_verb(BonoboUIComponent *uih, gpointer data, const char* path)
{
  TicketApplet *ad = data;
  
  GtkWidget *table;
  GtkWidget *label;

  GtkWidget *cmd_vbox;

  if (ad->p_box) {
    gdk_window_raise(ad->p_box->window);
    return;
  }

  ad->p_refresh = ad->refresh / 1000 / 60;

  ad->p_box = gnome_property_box_new();

  gtk_window_set_title(GTK_WINDOW(&GNOME_PROPERTY_BOX(ad->p_box)->dialog.window), 
		       "TicketApplet Settings");

  table = gtk_table_new(2, 3, FALSE);
  label = gtk_label_new("Settings");
  gnome_property_box_append_page( GNOME_PROPERTY_BOX(ad->p_box),
				  table, label);

  cmd_vbox = gtk_vbox_new(FALSE, 5);
  label = gtk_label_new("Commands");
  gnome_property_box_append_page( GNOME_PROPERTY_BOX(ad->p_box),
				  cmd_vbox, label);
  
  add_settings(ad, table);
  add_commands(ad, cmd_vbox);

  gtk_widget_show_all(GTK_WIDGET(ad->p_box));

}

void
property_save(gchar *path, TicketApplet *ad)
{
  //nothing yet
  gnome_config_sync();
}
