#!/usr/bin/perl -w # -*- mode: perl -*-
=pod

=head1 NAME

XML::Handler::EasyTree::Visitor - Visit one node of an EasyTree

=head1 SYNOPSIS

  use XML::Parser;
  use XML::Handler::EasyTree;
  use XML::Handler::EasyTree::Traversal;
  use XML::Handler::EasyTree::Visitor;

  #make an EasyTree
  my $p=         new XML::Parser(Style=>'EasyTree');
  my $easy_tree= $p->parsefile('something.xml');

  #traverse the EasyTree
  my $traversal= new XML::Handler::EasyTree::Traversal;
  my $visitor=   new XML::Handler::EasyTree::Visitor;

  #visit each node of easytree $tree with $visitor
  $lTraversal->traverse($tree,$visitor);

=head1 DESCRIPTION

XML::Handler::EasyTree::Traversal and XML::Handler::EasyTree::Visitor
permit simple visitation of all nodes of an EasyTree. The current application 
of that is the XML::Handler::EasyTree::Writer;

(Yes, so far this manpage has been identical to that of
XML::Handler::EasyTree::Traversal)

Any visitor to be used with traversal should subclass this class and
overload the methods C<start_visit> and C<end_visit>. 

=head1 METHODS

=head2 start_visit

   C<$self->start_visit( $easy_tree_node )>

receives as sole parameter a hash that contains a node of an
XML::Handler::EasyTree. Do with this easy tree as you
please. Returnvalues

=over 2

=item different from zero 

if you want the traversal to go deeper into the tree

=item 0 

if you want the to end at this node

=back

=head1 EXAMPLE

Parse a tree and as XML and write it out to a file "xml-test-outfile.xml"

  # !perl -w
  use strict;
  use XML::Parser;
  use XML::Handler::EasyTree::WriterVisitor;
  use IO::File;
  
  $XML::Handler::EasyTree::Noempty=1;
  my $xml=<<'EOF';
  <foo>
    <head id="a">Hello <em>there</em>
    </head>
    <bar>Howdy<ref/>
    </bar>
    do
  </foo>
  EOF
  my $p=new XML::Parser(Style=>'EasyTree');
  my $tree=$p->parse($xml);


  $visitor=new XML::Handler::EasyTree::WriteVisitor(new IO::Stream('xml-test-outfile.xml','w');

  my $traversal=new XML::Handler::EasyTree::Traversal();

  $traversal->traverse($tree,$visitor);

  
=head1 KNOWN WEAKNESSES

None (yet)

=head1 AUTHOR

Wolfgang M&uuml;ller <wolfgang.mueller@cui.unige.ch>

Copyright (c) 2001 Wolfgang Mller. All rights reserved. This program
is free software; you can redistribute it and/or modify it under the same
terms as Perl itself.

=head1 SEE ALSO

  XML::Handler::EasyTree, XML::Handler::EasyTree::Traversal

=cut

package XML::Handler::EasyTree::Visitor;
require Exporter;
use strict;
use vars qw(@ISA @EXPORT_OK);

@ISA= qw(Exporter);
@EXPORT_OK= qw(new
	       start_visit
	       end_visit
	      );
use strict;

sub new(){
  my $class = shift;
  my $self = {};
  bless $self, $class;
  $self->initialize();
  return $self;
}

sub initialize(){
  my$self=shift;
}

#########################################
#
# start_visit
#
# this start_visit function is intended as example, and prints the node
# it gets as a parameter.
#
# PARAMETERS: The start_visit function takes a tree node (i.e. a hash reference)
# as parameter. The parameter is provided by CGIFTLink::traverse 
#
# RETURNS:    nothing
#
sub start_visit( $ ){
  my $self=shift;
  my $lRoot=shift;

  if($lRoot->{type} eq 'e'){

    print "<Node: ",$lRoot->{name}, "\n";

    my $i;
    for $i (keys(%{$lRoot->{attrib}})){

      print "    ",$i,"=\"",$lRoot->{attrib}->{$i},"\"\n";
      
    }
    print ">\n";
  }elsif($lRoot->{type} eq 't'){
    print "TEXT NODE:{",$lRoot->{content},"}\n\n";
  }
}
#########################################
#
# end_visit
#
# this end_visit function does plain nothing
#
# PARAMETERS: The end_visit function takes a tree node (i.e. a hash reference)
#    as parameter. The parameter is provided by CGIFTLink::traverse 
#
# RETURNS:    nothing
#
sub end_visit( $ ){
  my $self=shift;
  my $lRoot=shift;
  if($lRoot->{type} eq 'e'){
    print "</Node: ",$lRoot->{name}, ">\n";
  }
}


