/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                      PPPP    AAA   IIIII  N   N  TTTTT                      %
%                      P   P  A   A    I    NN  N    T                        %
%                      PPPP   AAAAA    I    N N N    T                        %
%                      P      A   A    I    N  NN    T                        %
%                      P      A   A  IIIII  N   N    T                        %
%                                                                             %
%                                                                             %
%                        Methods to Paint on an Image                         %
%                                                                             %
%                              Software Design                                %
%                                John Cristy                                  %
%                                 July 1998                                   %
%                                                                             %
%                                                                             %
%  Copyright 1999-2004 ImageMagick Studio LLC, a non-profit organization      %
%  dedicated to making software imaging solutions freely available.           %
%                                                                             %
%  You may not use this file except in compliance with the License.  You may  %
%  obtain a copy of the License at                                            %
%                                                                             %
%    http://www.imagemagick.org/www/Copyright.html                            %
%                                                                             %
%  Unless required by applicable law or agreed to in writing, software        %
%  distributed under the License is distributed on an "AS IS" BASIS,          %
%  WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.   %
%  See the License for the specific language governing permissions and        %
%  limitations under the License.                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%
*/

/*
 Include declarations.
*/
#include "magick/studio.h"
#include "magick/color.h"
#include "magick/composite.h"
#include "magick/composite_private.h"
#include "magick/draw.h"
#include "magick/draw_private.h"
#include "magick/error.h"
#include "magick/monitor.h"
#include "magick/paint.h"
#include "magick/string_.h"

/*
  Define declarations.
*/
#define MaxStacksize  (1 << 15)
#define Push(up,left,right,delta) \
  if ((s < (segment_stack+MaxStacksize)) && (((up)+(delta)) >= 0) && \
      (((up)+(delta)) < (long) image->rows)) \
    { \
      s->y1=(MagickRealType) (up); \
      s->x1=(MagickRealType) (left); \
      s->x2=(MagickRealType) (right); \
      s->y2=(MagickRealType) (delta); \
      s++; \
    }

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   C o l o r F l o o d f i l l I m a g e                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  ColorFloodfill() changes the color value of any pixel that matches
%  target and is an immediate neighbor.  If the method FillToBorderMethod is
%  specified, the color value is changed for any neighbor pixel that does not
%  match the bordercolor member of image.
%
%  By default target must match a particular pixel color exactly.
%  However, in many cases two colors may differ by a small amount.  The
%  fuzz member of image defines how much tolerance is acceptable to
%  consider two colors as the same.  For example, set fuzz to 10 and the
%  color red at intensities of 100 and 102 respectively are now
%  interpreted as the same color for the purposes of the floodfill.
%
%  The format of the ColorFloodfillImage method is:
%
%      MagickBooleanType ColorFloodfillImage(Image *image,
%        const DrawInfo *draw_info,const PixelPacket target,
%        const long x_offset,const long y_offset,const PaintMethod method)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o draw_info: The draw info.
%
%    o target: The RGB value of the target color.
%
%    o x,y: The starting location of the operation.
%
%    o method: Choose either FloodfillMethod or FillToBorderMethod.
%
%
*/
MagickExport MagickBooleanType ColorFloodfillImage(Image *image,
  const DrawInfo *draw_info,const PixelPacket target,const long x_offset,
  const long y_offset,const PaintMethod method)
{
  long
    offset,
    start,
    x1,
    x2,
    y;

  MagickBooleanType
    skip;

  MagickSizeType
    number_pixels;

  PixelPacket
    fill_color;

  register const PixelPacket
    *p;

  register long
    x;

  register PixelPacket
    *q;

  register SegmentInfo
    *s;

  SegmentInfo
    *segment_stack;

  unsigned char
    *floodplane;

  /*
    Check boundary conditions.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  assert(draw_info != (DrawInfo *) NULL);
  assert(draw_info->signature == MagickSignature);
  if ((x_offset < 0) || (x_offset >= (long) image->columns))
    return(MagickFalse);
  if ((y_offset < 0) || (y_offset >= (long) image->rows))
    return(MagickFalse);
  /*
    Set floodfill color.
  */
  fill_color=GetFillColor(draw_info,x_offset,y_offset);
  if (FuzzyColorCompare(image,&fill_color,&target) != MagickFalse)
    return(MagickTrue);
  number_pixels=(MagickSizeType) image->columns*image->rows;
  if (number_pixels != (MagickSizeType) ((size_t) number_pixels))
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      image->filename);
  floodplane=(unsigned char *) AcquireMagickMemory((size_t) number_pixels);
  segment_stack=(SegmentInfo *)
    AcquireMagickMemory(MaxStacksize*sizeof(SegmentInfo));
  if ((floodplane== (unsigned char *) NULL) ||
      (segment_stack == (SegmentInfo *) NULL))
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      image->filename);
  (void) ResetMagickMemory(floodplane,MagickFalse,(size_t)
    image->columns*image->rows);
  floodplane[y_offset*image->columns+x_offset]=(unsigned char) MagickTrue;
  /*
    Push initial segment on stack.
  */
  image->storage_class=DirectClass;
  if (image->matte == MagickFalse)
    SetImageOpacity(image,OpaqueOpacity);
  x=x_offset;
  y=y_offset;
  start=0;
  s=segment_stack;
  Push(y,x,x,1);
  Push(y+1,x,x,-1);
  while (s > segment_stack)
  {
    /*
      Pop segment off stack.
    */
    s--;
    x1=(long) s->x1;
    x2=(long) s->x2;
    offset=(long) s->y2;
    y=(long) s->y1+offset;
    /*
      Recolor neighboring pixels.
    */
    p=AcquireImagePixels(image,0,y,(unsigned long) (x1+1),1,&image->exception);
    if (p == (const PixelPacket *) NULL)
      break;
    p+=x1;
    for (x=x1; x >= 0; x--)
    {
      if ((int) floodplane[y*image->columns+x] != MagickFalse)
        {
          p--;
          break;
        }
      if (method == FloodfillMethod)
        {
          if (FuzzyColorCompare(image,p,&target) == MagickFalse)
            break;
        }
      else
        {
          if (FuzzyColorCompare(image,p,&target) != MagickFalse)
            break;
        }
      floodplane[y*image->columns+x]=(unsigned char) MagickTrue;
      p--;
    }
    skip=(MagickBooleanType) (x >= x1);
    if (skip == MagickFalse)
      {
        start=x+1;
        if (start < x1)
          Push(y,start,x1-1,-offset);
        x=x1+1;
      }
    do
    {
      if (skip == MagickFalse)
        {
          if (x < (long) image->columns)
            {
              p=AcquireImagePixels(image,x,y,image->columns-x,1,
                &image->exception);
              if (p == (const PixelPacket *) NULL)
                break;
              for ( ; x < (long) image->columns; x++)
              {
                if ((int) floodplane[y*image->columns+x] != MagickFalse)
                  {
                    p++;
                    continue;
                  }
                if (method == FloodfillMethod)
                  {
                    if (FuzzyColorCompare(image,p,&target) == MagickFalse)
                      break;
                  }
                else
                  {
                    if (FuzzyColorCompare(image,p,&target) != MagickFalse)
                      break;
                  }
                floodplane[y*image->columns+x]=(unsigned char) MagickTrue;
                p++;
              }
            }
          Push(y,start,x-1,offset);
          if (x > (x2+1))
            Push(y,x2+1,x-1,-offset);
        }
      skip=MagickFalse;
      x++;
      if (x <= x2)
        {
          p=AcquireImagePixels(image,x,y,(unsigned long) (x2-x+1),1,
            &image->exception);
          if (p == (const PixelPacket *) NULL)
            break;
          for ( ; x <= x2; x++)
          {
            if ((int) floodplane[y*image->columns+x] != MagickFalse)
              {
                p++;
                break;
              }
            if (method == FloodfillMethod)
              {
                if (FuzzyColorCompare(image,p,&target) != MagickFalse)
                  break;
              }
            else
              {
                if (FuzzyColorCompare(image,p,&target) == MagickFalse)
                  break;
              }
            p++;
          }
        }
      start=x;
    } while (x <= x2);
  }
  for (y=0; y < (long) image->rows; y++)
  {
    /*
      Tile fill color onto floodplane.
    */
    q=GetImagePixels(image,0,y,image->columns,1);
    if (q == (PixelPacket *) NULL)
      break;
    for (x=0; x < (long) image->columns; x++)
    {
      if ((int) floodplane[y*image->columns+x] != MagickFalse)
        {
          fill_color=GetFillColor(draw_info,x,y);
          *q=MagickCompositeOver(&fill_color,(MagickRealType)
            fill_color.opacity,q,(MagickRealType) q->opacity);
        }
      q++;
    }
    if (SyncImagePixels(image) == MagickFalse)
      break;
  }
  segment_stack=(SegmentInfo *) RelinquishMagickMemory(segment_stack);
  floodplane=(unsigned char *) RelinquishMagickMemory(floodplane);
  return((MagickBooleanType) (y == (long) image->rows));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%                                                                             %
%   M a t t e F l o o d f i l l I m a g e                                     %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  MatteFloodfill() changes the transparency value of any pixel that matches
%  target and is an immediate neighbor.  If the method FillToBorderMethod
%  is specified, the transparency value is changed for any neighbor pixel
%  that does not match the bordercolor member of image.
%
%  By default target must match a particular pixel transparency exactly.
%  However, in many cases two transparency values may differ by a
%  small amount.  The fuzz member of image defines how much tolerance is
%  acceptable to consider two transparency values as the same.  For example,
%  set fuzz to 10 and the opacity values of 100 and 102 respectively are
%  now interpreted as the same value for the purposes of the floodfill.
%
%  The format of the MatteFloodfillImage method is:
%
%      MagickBooleanType MatteFloodfillImage(Image *image,
%        const PixelPacket target,const Quantum opacity,const long x_offset,
%        const long y_offset,const PaintMethod method)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o target: The RGB value of the target color.
%
%    o opacity: The level of transparency: 0 is fully opaque and MaxRGB is
%      fully transparent.
%
%    o x,y: The starting location of the operation.
%
%    o method:  Choose either FloodfillMethod or FillToBorderMethod.
%
%
*/
MagickExport MagickBooleanType MatteFloodfillImage(Image *image,
  const PixelPacket target,const Quantum opacity,const long x_offset,
  const long y_offset,const PaintMethod method)
{
  long
    offset,
    start,
    x1,
    x2,
    y;

  MagickBooleanType
    skip;

  MagickSizeType
    number_pixels;

  PixelPacket
    fill_color;

  register const PixelPacket
    *p;

  register long
    x;

  register PixelPacket
    *q;

  register SegmentInfo
    *s;

  SegmentInfo
    *segment_stack;

  unsigned char
    *floodplane;

  /*
    Check boundary conditions.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  if ((x_offset < 0) || (x_offset >= (long) image->columns))
    return(MagickFalse);
  if ((y_offset < 0) || (y_offset >= (long) image->rows))
    return(MagickFalse);
  /*
    Set floodfill color.
  */
  fill_color.opacity=opacity;
  if (FuzzyOpacityCompare(image,&fill_color,&target) != MagickFalse)
    return(MagickTrue);
  number_pixels=(MagickSizeType) image->columns*image->rows;
  if (number_pixels != (MagickSizeType) ((size_t) number_pixels))
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      image->filename);
  floodplane=(unsigned char *) AcquireMagickMemory((size_t) number_pixels);
  segment_stack=(SegmentInfo *)
    AcquireMagickMemory(MaxStacksize*sizeof(SegmentInfo));
  if ((floodplane== (unsigned char *) NULL) ||
      (segment_stack == (SegmentInfo *) NULL))
    ThrowBinaryException(ResourceLimitError,"MemoryAllocationFailed",
      image->filename);
  (void) ResetMagickMemory(floodplane,MagickFalse,(size_t)
    image->columns*image->rows);
  floodplane[y_offset*image->columns+x_offset]=(unsigned char) MagickTrue;
  /*
    Push initial segment on stack.
  */
  image->storage_class=DirectClass;
  if (image->matte == MagickFalse)
    SetImageOpacity(image,OpaqueOpacity);
  x=x_offset;
  y=y_offset;
  start=0;
  s=segment_stack;
  Push(y,x,x,1);
  Push(y+1,x,x,-1);
  while (s > segment_stack)
  {
    /*
      Pop segment off stack.
    */
    s--;
    x1=(long) s->x1;
    x2=(long) s->x2;
    offset=(long) s->y2;
    y=(long) s->y1+offset;
    /*
      Recolor neighboring pixels.
    */
    p=AcquireImagePixels(image,0,y,(unsigned long) (x1+1),1,&image->exception);
    if (p == (const PixelPacket *) NULL)
      break;
    p+=x1;
    for (x=x1; x >= 0; x--)
    {
      if ((int) floodplane[y*image->columns+x] != MagickFalse)
        {
          p--;
          break;
        }
      if (method == FloodfillMethod)
        {
          if (FuzzyColorCompare(image,p,&target) == MagickFalse)
            break;
        }
      else
        {
          if (FuzzyColorCompare(image,p,&target) != MagickFalse)
            break;
        }
      floodplane[y*image->columns+x]=(unsigned char) MagickTrue;
      p--;
    }
    skip=(MagickBooleanType) (x >= x1);
    if (skip == MagickFalse)
      {
        start=x+1;
        if (start < x1)
          Push(y,start,x1-1,-offset);
        x=x1+1;
      }
    do
    {
      if (skip == MagickFalse)
        {
          if (x < (long) image->columns)
            {
              p=AcquireImagePixels(image,x,y,image->columns-x,1,
                &image->exception);
              if (p == (const PixelPacket *) NULL)
                break;
              for ( ; x < (long) image->columns; x++)
              {
                if ((int) floodplane[y*image->columns+x] != MagickFalse)
                  {
                    p++;
                    continue;
                  }
                if (method == FloodfillMethod)
                  {
                    if (FuzzyColorCompare(image,p,&target) == MagickFalse)
                      break;
                  }
                else
                  {
                    if (FuzzyColorCompare(image,p,&target) != MagickFalse)
                      break;
                  }
                floodplane[y*image->columns+x]=(unsigned char) MagickTrue;
                p++;
              }
            }
          Push(y,start,x-1,offset);
          if (x > (x2+1))
            Push(y,x2+1,x-1,-offset);
        }
      skip=MagickFalse;
      x++;
      if (x <= x2)
        {
          p=AcquireImagePixels(image,x,y,(unsigned long) (x2-x+1),1,
            &image->exception);
          if (p == (const PixelPacket *) NULL)
            break;
          for ( ; x <= x2; x++)
          {
            if ((int) floodplane[y*image->columns+x] != MagickFalse)
              {
                p++;
                break;
              }
            if (method == FloodfillMethod)
              {
                if (FuzzyColorCompare(image,p,&target) != MagickFalse)
                  break;
              }
            else
              {
                if (FuzzyColorCompare(image,p,&target) == MagickFalse)
                  break;
              }
            p++;
          }
        }
      start=x;
    } while (x <= x2);
  }
  for (y=0; y < (long) image->rows; y++)
  {
    /*
      Tile fill color onto floodplane.
    */
    q=GetImagePixels(image,0,y,image->columns,1);
    if (q == (PixelPacket *) NULL)
      break;
    for (x=0; x < (long) image->columns; x++)
    {
      if ((int) floodplane[y*image->columns+x] != MagickFalse)
        q->opacity=opacity;
      q++;
    }
    if (SyncImagePixels(image) == MagickFalse)
      break;
  }
  segment_stack=(SegmentInfo *) RelinquishMagickMemory(segment_stack);
  floodplane=(unsigned char *) RelinquishMagickMemory(floodplane);
  return((MagickBooleanType) (y == (long) image->rows));
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     O p a q u e I m a g e                                                   %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  OpaqueImage() changes any pixel that matches color with the color
%  defined by fill.
%
%  By default color must match a particular pixel color exactly.  However,
%  in many cases two colors may differ by a small amount.  Fuzz defines
%  how much tolerance is acceptable to consider two colors as the same.
%  For example, set fuzz to 10 and the color red at intensities of 100 and
%  102 respectively are now interpreted as the same color.
%
%  The format of the OpaqueImage method is:
%
%      MagickBooleanType OpaqueImage(Image *image,const PixelPacket target,
%        const PixelPacket fill)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o target: The RGB value of the target color.
%
%    o fill: The replacement color.
%
%
*/
MagickExport MagickBooleanType OpaqueImage(Image *image,
  const PixelPacket target,const PixelPacket fill)
{
#define OpaqueImageTag  "Opaque/Image"

  long
    y;

  MagickBooleanType
    status;

  register long
    x;

  register PixelPacket
    *q;

  register long
    i;

  /*
    Make image color opaque.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  switch (image->storage_class)
  {
    case DirectClass:
    default:
    {
      /*
        Make DirectClass image opaque.
      */
      for (y=0; y < (long) image->rows; y++)
      {
        q=GetImagePixels(image,0,y,image->columns,1);
        if (q == (PixelPacket *) NULL)
          break;
        for (x=0; x < (long) image->columns; x++)
        {
          if (FuzzyColorCompare(image,q,&target) != MagickFalse)
            *q=fill;
          q++;
        }
        if (SyncImagePixels(image) == MagickFalse)
          break;
        if (QuantumTick(y,image->rows) != 0)
          {
            status=MagickMonitor(OpaqueImageTag,y,image->rows,
              &image->exception);
            if (status == MagickFalse)
              break;
          }
      }
      break;
    }
    case PseudoClass:
    {
      /*
        Make PseudoClass image opaque.
      */
      for (i=0; i < (long) image->colors; i++)
      {
        if (FuzzyColorCompare(image,&image->colormap[i],&target) != MagickFalse)
          image->colormap[i]=fill;
        if (QuantumTick(i,image->colors))
          {
            status=MagickMonitor(OpaqueImageTag,i,image->colors,
              &image->exception);
            if (status == MagickFalse)
              break;
          }
      }
      if (fill.opacity != OpaqueOpacity)
        {
          for (y=0; y < (long) image->rows; y++)
          {
            q=GetImagePixels(image,0,y,image->columns,1);
            if (q == (PixelPacket *) NULL)
              break;
            for (x=0; x < (long) image->columns; x++)
            {
              if (FuzzyColorCompare(image,q,&target) != MagickFalse)
                q->opacity=fill.opacity;
              q++;
            }
            if (SyncImagePixels(image) == MagickFalse)
              break;
          }
        }
      (void) SyncImage(image);
      break;
    }
  }
  if (fill.opacity != OpaqueOpacity)
    image->matte=MagickTrue;
  return(MagickTrue);
}

/*
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%                                                                             %
%                                                                             %
%     T r a n s p a r e n t I m a g e                                         %
%                                                                             %
%                                                                             %
%                                                                             %
%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
%
%  TransparentImage() changes the opacity value associated with any pixel
%  that matches color to the value defined by opacity.
%
%  By default color must match a particular pixel color exactly.  However,
%  in many cases two colors may differ by a small amount.  Fuzz defines
%  how much tolerance is acceptable to consider two colors as the same.
%  For example, set fuzz to 10 and the color red at intensities of 100 and
%  102 respectively are now interpreted as the same color.
%
%  The format of the TransparentImage method is:
%
%      MagickBooleanType TransparentImage(Image *image,const PixelPacket target,
%        const Quantum opacity)
%
%  A description of each parameter follows:
%
%    o image: The image.
%
%    o target: The RGB value of the target color.
%
%    o opacity: The replacement opacity value.
%
%
*/
MagickExport MagickBooleanType TransparentImage(Image *image,
  const PixelPacket target,const Quantum opacity)
{
#define TransparentImageTag  "Transparent/Image"

  long
    y;

  MagickBooleanType
    status;

  register long
    x;

  register PixelPacket
    *q;

  /*
    Make image color transparent.
  */
  assert(image != (Image *) NULL);
  assert(image->signature == MagickSignature);
  if (image->debug != MagickFalse)
    (void) LogMagickEvent(TraceEvent,GetMagickModule(),image->filename);
  if (image->matte == MagickFalse)
    SetImageOpacity(image,OpaqueOpacity);
  for (y=0; y < (long) image->rows; y++)
  {
    q=GetImagePixels(image,0,y,image->columns,1);
    if (q == (PixelPacket *) NULL)
      break;
    for (x=(long) image->columns-1; x >= 0; x--)
    {
      if (FuzzyColorCompare(image,q,&target) != MagickFalse)
        q->opacity=opacity;
      q++;
    }
    if (SyncImagePixels(image) == MagickFalse)
      break;
    if (QuantumTick(y,image->rows) != 0)
      {
        status=MagickMonitor(TransparentImageTag,y,image->rows,
          &image->exception);
        if (status == MagickFalse)
          break;
      }
  }
  return(MagickTrue);
}
